import json
import os
from tempfile import NamedTemporaryFile
from unittest import mock

from galaxy.tool_util.verify.script import (
    _arg_parser,
    build_case_references,
    Results,
    test_tools as run,
    TestReference,
)

VT_PATH = 'galaxy.tool_util.verify.script.verify_tool'
NEW_HISTORY = object()


def test_arg_parse():
    parser = _arg_parser()

    # defaults
    args = parser.parse_args([])
    assert not args.with_reference_data
    assert not args.history_per_test_case
    assert args.page_size == 0
    assert args.page_number == 0

    # skip flags
    args = parser.parse_args(["--skip-with-reference-data", "--history-per-test-case"])
    assert not args.with_reference_data
    assert args.history_per_test_case

    # enable flags
    args = parser.parse_args(["--with-reference-data", "--history-per-suite"])
    assert args.with_reference_data
    assert not args.history_per_test_case

    # pagination
    args = parser.parse_args(["--page-size", "5", "--page-number", "40"])
    assert args.page_size == 5
    assert args.page_number == 40


def test_test_tools():
    interactor = MockGalaxyInteractor()
    f = NamedTemporaryFile()
    results = Results("my suite", f.name)
    test_references = [
        TestReference("cat", "0.1.0", 0),
        TestReference("cat", "0.1.0", 1),
        TestReference("cat", "0.2.0", 0),
    ]
    with mock.patch(VT_PATH) as mock_verify:
        assert_results_not_written(results)
        run(
            interactor,
            test_references,
            results,
        )
        calls = mock_verify.call_args_list
        assert len(calls) == 3
        assert len(results.test_exceptions) == 0
        assert_results_written(results)
        assert interactor.history_created
        assert interactor.history_deleted


def test_test_tools_no_history_cleanup():
    interactor = MockGalaxyInteractor()
    f = NamedTemporaryFile()
    results = Results("my suite", f.name)
    test_references = [
        TestReference("cat", "0.1.0", 0),
    ]
    with mock.patch(VT_PATH) as mock_verify:
        assert_results_not_written(results)
        run(
            interactor,
            test_references,
            results,
            no_history_cleanup=True,
        )
        calls = mock_verify.call_args_list
        assert len(calls) == 1
        assert len(results.test_exceptions) == 0
        assert_results_written(results)
        assert interactor.history_created
        assert not interactor.history_deleted


def test_test_tool_per_test_history():
    interactor = MockGalaxyInteractor()
    f = NamedTemporaryFile()
    results = Results("my suite", f.name)
    test_references = [
        TestReference("cat", "0.1.0", 0),
        TestReference("cat", "0.1.0", 1),
    ]
    with mock.patch(VT_PATH) as mock_verify:
        assert_results_not_written(results)
        run(
            interactor,
            test_references,
            results,
            history_per_test_case=True,
        )
        calls = mock_verify.call_args_list
        assert len(calls) == 2
        assert len(results.test_exceptions) == 0
        assert_results_written(results)
        assert not interactor.history_created
        assert not interactor.history_deleted


def test_test_tools_records_exception():
    interactor = MockGalaxyInteractor()
    f = NamedTemporaryFile()
    results = Results("my suite", f.name)
    test_references = [
        TestReference("bad", "0.1.0", 0),
    ]
    with mock.patch(VT_PATH) as mock_verify:
        assert_results_not_written(results)

        def side_effect(*args, **kwd):
            raise Exception("Cow")

        mock_verify.side_effect = side_effect
        run(
            interactor,
            test_references,
            results,
        )
        calls = mock_verify.call_args_list
        assert len(calls) == 1
        assert len(results.test_exceptions) == 1
        assert_results_written(results)


def test_test_tools_records_retry_exception():
    interactor = MockGalaxyInteractor()
    f = NamedTemporaryFile()
    results = Results("my suite", f.name)
    test_references = [
        TestReference("bad", "0.1.0", 0),
    ]
    with mock.patch(VT_PATH) as mock_verify:
        assert_results_not_written(results)

        count = 0

        def side_effect(*args, **kwd):
            nonlocal count
            raise_exception = count == 0
            count += 1
            if raise_exception:
                raise Exception("Cow")

        mock_verify.side_effect = side_effect
        run(
            interactor,
            test_references,
            results,
            retries=1,
        )
        calls = mock_verify.call_args_list
        assert len(calls) == 2
        assert len(results.test_exceptions) == 0
        assert_results_written(results)


def test_results():
    f = NamedTemporaryFile()
    results = Results("my suite", f.name)
    results.register_result({"id": "foo", "has_data": True, "data": {"status": "success"}})
    results.write()
    message = results.info_message()

    with open(f.name, "r") as f:
        report_obj = json.load(f)
    assert "tests" in report_obj
    assert len(report_obj["tests"]) == 1
    assert report_obj["results"]["total"] == 1, report_obj["results"]
    assert report_obj["results"]["errors"] == 0
    assert report_obj["results"]["skips"] == 0

    assert "Passed tool tests (1)" in message
    assert "Skipped tool tests (0)" in message

    results.register_result({"id": "bar", "has_data": True, "data": {"status": "skip"}})
    results.write()
    message = results.info_message()

    with open(f.name, "r") as f:
        report_obj = json.load(f)
    assert len(report_obj["tests"]) == 2
    assert report_obj["results"]["skips"] == 1
    assert "Passed tool tests (1)" in message
    assert "Skipped tool tests (1)" in message


def test_build_references():
    interactor = MockGalaxyInteractor()
    test_references = build_case_references(interactor)
    assert len(test_references) == 6

    test_references = build_case_references(interactor, 'cat1', tool_version="*")
    assert len(test_references) == 6

    test_references = build_case_references(interactor, 'cat1', tool_version="*", page_size=3, page_number=1)
    assert len(test_references) == 3

    test_references = build_case_references(interactor, 'cat1', tool_version="*", page_size=3, page_number=2)
    assert len(test_references) == 0

    test_references = build_case_references(interactor, 'cat1', tool_version="*", page_size=3, page_number=3)
    assert len(test_references) == 0

    test_references = build_case_references(interactor, 'cat1', tool_version=None)
    assert len(test_references) == 4

    test_references = build_case_references(interactor, 'cat1', tool_version="0.2.0")
    assert len(test_references) == 4

    test_references = build_case_references(interactor, 'cat1', tool_version="0.1.0")
    assert len(test_references) == 2

    test_references = build_case_references(interactor, 'cat1', tool_version="0.1.0", test_index=1)
    assert len(test_references) == 1

    # Specifying an index but not a version, grabs latest version and fills it in.
    test_references = build_case_references(interactor, 'cat1', test_index=2)
    assert len(test_references) == 1
    test_reference = test_references[0]
    assert test_reference.tool_id == "cat1"
    assert test_reference.tool_version == "0.2.0"
    assert test_reference.test_index == 2


def assert_results_not_written(results):
    assert os.stat(results.test_json).st_size == 0


def assert_results_written(results):
    assert os.stat(results.test_json).st_size > 0
    with open(results.test_json, "r") as f:
        json.load(f)


class MockGalaxyInteractor:

    def __init__(self):
        self.history_deleted = False
        self.history_created = False

    def new_history(self, history_name=""):
        self.history_created = True
        return NEW_HISTORY

    def delete_history(self, history):
        self.history_deleted = True

    def get_tests_summary(self):
        return {
            'cat1': {
                '0.2.0': {
                    'count': 4,
                },
                '0.1.0': {
                    'count': 2,
                },
            },
        }

    def get_tool_tests(self, tool_id, tool_version=None):
        tool_dict = self.get_tests_summary().get(tool_id)
        test_defs = []
        for this_tool_version, version_defs in tool_dict.items():
            if tool_version is not None and tool_version != "*" and this_tool_version != tool_version:
                continue

            count = version_defs['count']
            for i in range(count):
                test_def = {
                    'tool_id': tool_id,
                    'tool_version': this_tool_version or '0.1.1-default',
                }
                test_defs.append(test_def)

            if tool_version is None or tool_version != "*":
                break

        return test_defs
