"""This module defines the fundamental logic for defining RL-Environment IO.

This module defines:
    - The key Types/ Datastructures concerning Agent-Environment Input-Output.
    - An interface for defining Environment Logic
    - An interface for composing Environment Logic
    - Helper functions to initialize Types communicated to the Agent.
"""
from __future__ import annotations
import abc

from typing import Any, TYPE_CHECKING, TypeVar, Generic, Sequence, Protocol

if TYPE_CHECKING:  # https://github.com/python/mypy/issues/6239
    from dataclasses import dataclass
else:
    from chex import dataclass

from jaxtyping import Bool, Array, PyTree, Num, Int8

import jax
import jax.numpy as jnp

from jit_env import specs


# Define Environment IO Types.

class StateProtocol(Protocol):
    """Environment states should always carry a PRNGKey for predictable IO.

    For example, if we were to save states in a replay buffer (for whatever
    reason) we require all dependent data to be contained in State.
    This would not be the case if `key` were part of the Environment
    `step` function, as this could be changed arbitrarily by the caller.
    In turn, this leads to un-reproducible behaviour.
    """
    key: jax.random.KeyArray


# The following should all be valid Jax types
Observation = TypeVar("Observation")
State = TypeVar("State", bound="StateProtocol")
Action = TypeVar("Action")


class StepType:
    """Defines the status of a `TimeStep` within a sequence."""
    FIRST: Int8[Array, ''] = jnp.array(0, jnp.int8)
    MID: Int8[Array, ''] = jnp.array(1, jnp.int8)
    LAST: Int8[Array, ''] = jnp.array(2, jnp.int8)


@dataclass
class TimeStep(Generic[Observation]):
    """Defines the datastructure that is communicated to the Agent

    Like in dm_env, `env.reset` will generate a `TimeStep` with step_type set
    to `StepType.FIRST`. If `env.step` terminates the episode, step_type is
    set to `StepType.LAST`. Otherwise, step_type is set to `StepType.MID`.

    Attributes:
        step_type:
            A StepType value that indicates episode boundaries.
        reward:
            A (tree of) numerical value(s) to be optimized for by an Agent.
        discount:
            A (tree of) float(s) in [0., 1.] to scale cumulative returns.
        observation:
            Generic Jax-Compatible data-structure that the agent observes to
            compute new actions.
        extras:
            Optional data that is typically not communicated to the Agent
            but allows the user to track certain loss-metrics, accuracies,
            or other information. This can be stored in a Replay Buffer for
            training or for generally monitorring Agent behaviour.
    """
    step_type: Int8[Array, '']
    reward: PyTree[Num[Array, '...']]
    discount: PyTree[Num[Array, '...']]
    observation: Observation
    extras: dict[str, Any] | None = None

    def first(self) -> Bool[Array, '']:
        """Proxy function to check if step was generated by `reset`."""
        return self.step_type == StepType.FIRST

    def mid(self) -> Bool[Array, '']:
        """Proxy function to check if step was generated during an episode."""
        return self.step_type == StepType.MID

    def last(self) -> Bool[Array, '']:
        """Proxy function to check if step was terminated."""
        return self.step_type == StepType.LAST


# Define Environment and Wrapper

class Environment(Generic[State, Action, Observation], metaclass=abc.ABCMeta):
    """Interface for defining Environment logic for RL-Agents. """

    def __str__(self) -> str:
        """Returns a minimal representation of the Environment Structure."""
        return self.__class__.__name__

    def __repr__(self) -> str:
        """Returns a complete informative representation of self."""
        return super().__repr__()

    @property
    def unwrapped(self) -> Environment:
        """Helper function to support unpacking Composite Environments."""
        return self

    @abc.abstractmethod
    def reset(self, key: jax.random.KeyArray) -> tuple[State, TimeStep]:
        """Starts a new episode as a functionally pure transformation.

        Args:
            key: Pseudo RNG Key to initialize `State` with.

        Returns:
            A tuple of `State` and `TimeStep` at indices;

            1) The `State` object is expected to carry a chain of `key` values
               in order to internally maintain the random state.
            2) The TimeStep object will have step_type set to `StepType.FIRST`
               and discount casted to a structure of `1.0` values.
        """

    @abc.abstractmethod
    def step(self, state: State, action: Action) -> tuple[State, TimeStep]:
        """Updates the environment according to the given state and action.

        If the environment already returned a `TimeStep` with `StepType.LAST`
        then subsequent calls to `step` will still execute but remain stuck
        inside the absorbing terminal state.

        Args:
            state:
                A State carried through `step` and created by `reset`.
            action:
                A Jax-compatible data-structure adhering to self.action_spec().

        Returns:
            A tuple of `State` and `TimeStep` at indices;

            1)  The `State` object is expected to carry a chain of `key` values
                in order to internally maintain the random state.
            2)  The TimeStep object will have step_type set to `StepType.MID`
                if the environment continues, otherwise it is `StepType.LAST`.
                Depending on termination the discount will be set to either
                step.discount or zero.
        """

    @abc.abstractmethod
    def reward_spec(self) -> specs.Spec:
        """Describes the reward returned by the environment before it exists.

        Typically, this can be a single float.

        Returns:
            A specs.Spec type indicating the reward data-structure.
        """

    @abc.abstractmethod
    def discount_spec(self) -> specs.Spec:
        """Describes the discount returned by the environment before it exists.

        Typically, this can be a single float between [0.0, 1.0]. It is also
        typical that the spec has the same dimensionality as the rewards, or
        at least be 'broadcastable'.

        Returns:
            A specs.Spec type indicating the discount data-structure.
        """

    @abc.abstractmethod
    def observation_spec(self) -> specs.Spec:
        """Describes the observations of this environment before it exists.

        The observation can be a general (composite) data-structure of Jax
        compatible leaf types.

        Returns:
            A specs.Spec type indicating the observation data-structure.
        """

    @abc.abstractmethod
    def action_spec(self) -> specs.Spec:
        """Describes the action modality of this environment before it exists.

        The action can be a general (composite) data-structure of Jax
        compatible leaf types.

        Returns:
            A specs.Spec type indicating the action data-structure.
        """

    def close(self):
        """Frees any resources used by the environment.

        Implement this method for an environment backed by an external process.

        This method can be used directly::

            env = Env(...)
            # Use env.
            env.close()

        or via a context manager::

            with Env(...) as env:
              # Use env.
        """

    def render(self, state: State) -> Any:
        """Generate a pixel-observation based on the given state. """
        raise NotImplementedError("Render Function not Implemented")

    def __enter__(self) -> Environment:
        """Allows the environment to be used in a with-statement context."""
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        """Allows the environment to be used in a with-statement context."""
        del exc_type, exc_value, traceback  # Unused.
        self.close()


class Wrapper(
    Environment[State, Action, Observation],
    Generic[State, Action, Observation],
    metaclass=abc.ABCMeta
):
    """Interface for Composing Environment logic for RL-Agents. """

    def __init__(self, env: Environment):
        """Initializes the Composite Environment with a base Environment.

        Args:
            env: The base environment to extend with functionality.
        """
        super().__init__()
        self.env = env

    def __str__(self) -> str:
        """Returns a recursive composition structure of all Wrappers."""
        return f"{self.__class__.__name__}({str(self.env)})"

    def __getattr__(self, item: str) -> Any:
        if item == "__setstate__":
            raise AttributeError(item)
        return getattr(self.env, item)

    @property
    def unwrapped(self) -> Environment:
        """Helper function to unpack Composite Environments to the base."""
        return self.env.unwrapped

    def reset(self, key: jax.random.KeyArray) -> tuple[State, TimeStep]:
        return self.env.reset(key)

    def step(self, state: State, action: Action) -> tuple[State, TimeStep]:
        return self.env.step(state, action)

    def reward_spec(self) -> specs.Spec:
        return self.env.reward_spec()

    def discount_spec(self) -> specs.Spec:
        return self.env.discount_spec()

    def observation_spec(self) -> specs.Spec:
        return self.env.observation_spec()

    def action_spec(self) -> specs.Spec:
        return self.env.action_spec()

    def render(self, state: State) -> Any:
        return self.env.render(state)


# Define helpers to instantiate TimeStep objects at Environment boundaries

def restart(
    observation: Observation,
    extras: dict | None = None,
    shape: int | Sequence[int] = ()
) -> TimeStep:
    """Returns a `TimeStep` with `step_type` set to `StepType.FIRST`. """
    return TimeStep(
        step_type=StepType.FIRST,
        reward=jnp.zeros(shape, dtype=float),
        discount=jnp.ones(shape, dtype=float),
        observation=observation,
        extras=extras,
    )


def transition(
    reward: PyTree[Num[Array, '...']],
    observation: Observation,
    discount: PyTree[Num[Array, '...']] | None = None,
    extras: dict | None = None,
    shape: int | Sequence[int] = ()
) -> TimeStep:
    """Returns a `TimeStep` with `step_type` set to `StepType.MID`. """
    discount = jnp.ones(shape, jnp.float32) if discount is None else discount
    return TimeStep(
        step_type=StepType.MID,
        reward=reward,
        discount=discount,
        observation=observation,
        extras=extras,
    )


def termination(
    reward: PyTree[Num[Array, '...']],
    observation: Observation,
    extras: dict | None = None,
    shape: int | Sequence[int] = ()
) -> TimeStep:
    """Returns a `TimeStep` with `step_type` set to `StepType.LAST`. """
    return TimeStep(
        step_type=StepType.LAST,
        reward=reward,
        discount=jnp.zeros(shape, dtype=float),
        observation=observation,
        extras=extras,
    )


def truncation(
    reward: PyTree[Num[Array, '...']],
    observation: Observation,
    discount: PyTree[Num[Array, '...']] | None = None,
    extras: dict | None = None,
    shape: int | Sequence[int] = ()
) -> TimeStep:
    """Alternative to `termination` that does not set `discount` to zero. """
    discount = jnp.ones(shape, jnp.float32) if discount is None else discount
    return TimeStep(
        step_type=StepType.LAST,
        reward=reward,
        discount=discount,
        observation=observation,
        extras=extras,
    )
