import ast
import builtins
import logging
import time
import token
from io import StringIO
from tokenize import TokenInfo, generate_tokens
from typing import Any, Iterable, List, Optional, Tuple

from pegen.tokenizer import Tokenizer

from .parser import PythonParser

log = logging.getLogger(__name__)


def tokenize(s: str) -> Iterable[TokenInfo]:
    for toknum, tokval, (srow, scol), (erow, ecol), linenum in generate_tokens(
        StringIO(s).readline
    ):
        yield TokenInfo(toknum, tokval, (srow, scol), (erow, ecol), linenum)


def merge_operators(tokens: Iterable[TokenInfo]) -> List[TokenInfo]:
    result = []
    for toknum, tokval, (srow, scol), (erow, ecol), linenum in tokens:
        if tokval == ">" and result[-1].string == "|":  # |>
            token_info = TokenInfo(token.OP, "|>", result[-1][2], (erow, ecol), linenum)
            del result[-1]
            result.append(token_info)
            continue
        elif tokval == "?":
            if result[-1].string == "?":  # ??
                token_info = TokenInfo(token.OP, "??", result[-1][2], (erow, ecol), linenum)
                del result[-1]
                result.append(token_info)
                continue
            else:
                token_info = TokenInfo(token.OP, "?", (srow, scol), (erow, ecol), linenum)
                result.append(token_info)
                continue
        elif tokval == ">" and result[-1].string == "=":  # =>
            token_info = TokenInfo(token.OP, "=>", result[-1][2], (erow, ecol), linenum)
            del result[-1]
            result.append(token_info)
            continue

        result.append(TokenInfo(toknum, tokval, (srow, scol), (erow, ecol), linenum))
    return result


def compile(
    s: str,
    filename: str = "<unknown>",
    verbose_tokenizer: bool = False,
    verbose_parser: bool = False,
    py_version: Optional[Tuple[int, int]] = None
) -> ast.Module:
    start_time = time.time_ns()
    tokengen = iter(merge_operators(tokenize(s)))
    tokenizer = Tokenizer(tokengen, verbose=verbose_tokenizer)
    parser = PythonParser(tokenizer, filename=filename, verbose=verbose_parser, py_version=py_version)
    try:
        return parser.parse("file")
    except SyntaxError as syntax_error:
        if parser._exception is None and str(syntax_error) == "invalid syntax":
            raise parser.make_syntax_error("unknown syntax error") from None
        else:
            raise
    finally:
        end_time = time.time_ns()
        log.debug(f"Compile {filename} took {(end_time - start_time) / 1e6:.2f} ms")


def exec(
    s: str,
    filename: str = "<unknown>",
    verbose_tokenizer: bool = False,
    verbose_parser: bool = False,
) -> Any:
    return builtins.exec(
        builtins.compile(
            compile(s, filename, verbose_tokenizer, verbose_parser), filename, "exec"
        )
    )
