"""
Python versions of the 256-color colormaps provided in
http://peterkovesi.com/projects/colourmaps/CETperceptual_csv_0_1.zip

Each of these colormaps can be accessed as a Bokeh palette or
Matplotlib colormap, either by string name:

  palette['name']
  cm['name']

or as Python attributes:

  palette.name
  cm.name

or as individually importable Python attributes:

  m_name
  b_name

All colormaps are named using Peter Kovesi's naming scheme:

<category>_<huesequence>_<lightnessrange>_c<meanchroma>[_s<colorshift>_[r<ifreversed>]]

but some have shorter, more convenient aliases, some of which are
named for the color ranges included and others
based on the qualitative appearance.  The colormaps with
shorter names tend to be the most useful subset, and for
cases like automatic population of a GUI widget these
colormaps are provided as a separate subset:

  palette_n['name'] or palette_n.name
  cm_n['name'] or cm_n.name

Also included are some sets of 256 Glasbey colors. These are available via the
same methods described above and are named:

  glasbey_<starting_palette>[_<min|max>c_<chroma_value>][_<min|max>l_<lightness_value>][_hue_<start>_<end>]

Some of the Glasbey sets are aliased to short names as explained in the User Guide.
"""

__version__ = '1.0.0'

from collections import OrderedDict
from itertools import chain


class AttrODict(OrderedDict):
    """Ordered dictionary with attribute access (e.g. for tab completion)"""
    def __dir__(self): return self.keys()
    def __delattr__(self, name): del self[name]
    def __getattr__(self, name):
        return self[name] if not name.startswith('_') else super(AttrODict, self).__getattr__(name)
    def __setattr__(self, name, value):
        if (name.startswith('_')): return super(AttrODict, self).__setattr__(name, value)
        self[name] = value


try:
    from matplotlib.colors import LinearSegmentedColormap
    from matplotlib.cm import register_cmap
except:
    def LinearSegmentedColormap(colorlist,name): pass
    def register_cmap(name,cmap): pass
    LinearSegmentedColormap.from_list=lambda n,c,N: None


def rgb_to_hex(r,g,b):
    return '#%02x%02x%02x' % (r,g,b)


def bokeh_palette(name,colorlist):
    palette[name] = [rgb_to_hex(int(r*255),int(g*255),int(b*255)) for r,g,b in colorlist]
    return palette[name]


def mpl_cm(name,colorlist):
    cm[name]      = LinearSegmentedColormap.from_list(name, colorlist, N=len(colorlist))
    register_cmap("cet_"+name, cmap=cm[name])
    return cm[name]


def get_aliases(name):
    """Get the aliases for a given colormap name"""
    names = [name]

    def check_aliases(names, d,  k_position=-1, v_position=0):
        for name in [n for n in names]:
            for k, v in d.items():
                v = [v] if not isinstance(v, list) else v
                for vname in v:
                    if name == vname and k not in names:
                        if k_position == -2:
                            names.append(k)
                        else:
                            names.insert(k_position, k)
                    if name == k and vname not in names:
                        if v_position == -2:
                            names.append(vname)
                        else:
                            names.insert(v_position, vname)
        return names

    n_names = len(names)
    while True:
        names = check_aliases(names, aliases, k_position=-2, v_position=0)
        names = check_aliases(names, cetnames_flipped, k_position=-2, v_position=-1)
        if len(names) == n_names:
            break
        n_names = len(names)
    # sort names as 1or0_underscores, CET, multiple_under_scores (alias, cetname, algorithmicname)
    def name_sortfn(name):
        if name.count("_") > 1:
            return 2
        if "CET" in name:
            return 1
        return 0
    return ',  '.join(sorted(names, key=name_sortfn))

def all_original_names(group=None, not_group=None, only_aliased=False, only_CET=False):
    """Get all original names - optionally in a particular group - or only those with aliases"""
    names = palette.keys()
    if group:
        names = filter(lambda x: group in x, names)
    if not_group:
        names = filter(lambda x: not_group not in x, names)
    if only_aliased:
        names = filter(lambda x: x in aliases.keys(), names)
    else:
        names = filter(lambda x: x not in chain.from_iterable(aliases.values()), names)
    if only_CET:
        names = filter(lambda x: x in cetnames_flipped.values(), names)
    else:
        names = filter(lambda x: x not in cetnames_flipped.values(), names)
    return sorted(list(names))


palette = AttrODict()
cm = AttrODict()
palette_n = AttrODict()
cm_n = AttrODict()

aliases = {
    'circle_mgbm_67_c31': ['cyclic_isoluminant'],
    'cyclic_mygbm_30_95_c78_s25': ['colorwheel'],
    'diverging_bkr_55_10_c35': ['bkr'],
    'diverging_bky_60_10_c30': ['bky'],
    'diverging_bwr_40_95_c42': ['coolwarm'],
    'diverging_gwv_55_95_c39': ['gwv'],
    'diverging_linear_bjy_30_90_c45': ['bjy'],
    'diverging_protanopic_deuteranopic_bwy_60_95_c32': ['bwy'],
    'diverging_tritanopic_cwr_75_98_c20': ['cwr'],
    'glasbey_bw_minc_20': ['glasbey'],
    'glasbey_bw_minc_20_hue_150_280': ['glasbey_cool'],
    'glasbey_bw_minc_20_hue_330_100': ['glasbey_warm'],
    'glasbey_bw_minc_20_maxl_70': ['glasbey_dark'],
    'glasbey_bw_minc_20_minl_30': ['glasbey_light'],
    'isoluminant_cgo_80_c38': ['isolum'],
    'linear_bgy_10_95_c74': ['bgy'],
    'linear_bgyw_15_100_c68': ['bgyw'],
    'linear_blue_95_50_c20': ['blues'],
    'linear_bmw_5_95_c89': ['bmw'],
    'linear_bmy_10_95_c78': ['bmy'],
    'linear_grey_0_100_c0': ['gray'],
    'linear_grey_10_95_c0': ['dimgray'],
    'linear_kbc_5_95_c73': ['kbc', 'linear_blue_5_95_c73'],
    'linear_kbgoy_20_95_c57': ['gouldian'],
    'linear_kbgyw_10_98_c63': ['kbgyw'],
    'linear_kgy_5_95_c69': ['kgy', 'linear_green_5_95_c69'],
    'linear_kryw_0_100_c71': ['fire'],
    'linear_ternary_blue_0_44_c57': ['kb'],
    'linear_ternary_green_0_46_c42': ['kg'],
    'linear_ternary_red_0_50_c52': ['kr'],
    'rainbow_bgyr_10_90_c83': ['rainbow4'],
    'rainbow_bgyr_35_85_c73': ['rainbow'],
}
cetnames_flipped = {
    'cyclic_mrybm_35_75_c68': 'CET_C1',
    'cyclic_mrybm_35_75_c68_s25': 'CET_C1s',
    'cyclic_mygbm_30_95_c78': 'CET_C2',
    'cyclic_mygbm_30_95_c78_s25': 'CET_C2s',
    'cyclic_wrkbw_10_90_c43': 'CET_C3',
    'cyclic_wrkbw_10_90_c43_s25': 'CET_C3s',
    'cyclic_wrwbw_40_90_c42': 'CET_C4',
    'cyclic_wrwbw_40_90_c42_s25': 'CET_C4s',
    'cyclic_grey_15_85_c0': 'CET_C5',
    'cyclic_grey_15_85_c0_s25': 'CET_C5s',
    'cyclic_rygcbmr_50_90_c64': 'CET_C6',
    'cyclic_rygcbmr_50_90_c64_s25': 'CET_C6s',
    'cyclic_ymcgy_60_90_c67': 'CET_C7',
    'cyclic_ymcgy_60_90_c67_s25': 'CET_C7s',
    'cyclic_mygbm_50_90_c46': 'CET_C8',
    'cyclic_mygbm_50_90_c46_s25': 'CET_C8s',
    'cyclic_mybm_20_100_c48': 'CET_C9',
    'cyclic_mybm_20_100_c48_s25': 'CET_C9s',
    'circle_mgbm_67_c31': 'CET_C10',
    'circle_mgbm_67_c31_s25': 'CET_C10s',
    'cyclic_bgrmb_35_70_c75': 'CET_C11',
    'cyclic_bgrmb_35_70_c75_s25': 'CET_C11s',
    'cyclic_protanopic_deuteranopic_bwyk_16_96_c31': 'CET_CBC1',
    'cyclic_protanopic_deuteranopic_wywb_55_96_c33': 'CET_CBC2',
    'diverging_protanopic_deuteranopic_bwy_60_95_c32': 'CET_CBD1',
    'diverging_linear_protanopic_deuteranopic_bjy_57_89_c34': 'CET_CBD2',
    'linear_protanopic_deuteranopic_kbjyw_5_95_c25': 'CET_CBL1',
    'linear_protanopic_deuteranopic_kbw_5_98_c40': 'CET_CBL2',
    'linear_protanopic_deuteranopic_kbw_5_95_c34': 'CET_CBL3',
    'linear_protanopic_deuteranopic_kyw_5_95_c49': 'CET_CBL4',
    'cyclic_tritanopic_cwrk_40_100_c20': 'CET_CBTC1',
    'cyclic_tritanopic_wrwc_70_100_c20': 'CET_CBTC2',
    'diverging_tritanopic_cwr_75_98_c20': 'CET_CBTD1',
    'linear_tritanopic_krjcw_5_98_c46': 'CET_CBTL1',
    'linear_tritanopic_krjcw_5_95_c24': 'CET_CBTL2',
    'linear_tritanopic_kcw_5_95_c22': 'CET_CBTL3',
    'linear_tritanopic_krw_5_95_c46': 'CET_CBTL4',
    'diverging_bwr_40_95_c42': 'CET_D1',
    'diverging_bwr_20_95_c54': 'CET_D1A',
    'diverging_gwv_55_95_c39': 'CET_D2',
    'diverging_gwr_55_95_c38': 'CET_D3',
    'diverging_bkr_55_10_c35': 'CET_D4',
    'diverging_bky_60_10_c30': 'CET_D6',
    'diverging_linear_bjy_30_90_c45': 'CET_D7',
    'diverging_linear_bjr_30_55_c53': 'CET_D8',
    'diverging_bwr_55_98_c37': 'CET_D9',
    'diverging_cwm_80_100_c22': 'CET_D10',
    'diverging_isoluminant_cjo_70_c25': 'CET_D11',
    'diverging_isoluminant_cjm_75_c23': 'CET_D12',
    'diverging_bwg_20_95_c41': 'CET_D13',
    'isoluminant_cgo_70_c39': 'CET_I1',
    'isoluminant_cgo_80_c38': 'CET_I2',
    'isoluminant_cm_70_c39': 'CET_I3',
    'linear_grey_0_100_c0': 'CET_L1',
    'linear_grey_10_95_c0': 'CET_L2',
    'linear_kryw_0_100_c71': 'CET_L3',
    'linear_kry_0_97_c73': 'CET_L4',
    'linear_kgy_5_95_c69': 'CET_L5',
    'linear_kbc_5_95_c73': 'CET_L6',
    'linear_bmw_5_95_c86': 'CET_L7',
    'linear_bmy_10_95_c71': 'CET_L8',
    'linear_bgyw_20_98_c66': 'CET_L9',
    'linear_gow_60_85_c27': 'CET_L10',
    'linear_gow_65_90_c35': 'CET_L11',
    'linear_blue_95_50_c20': 'CET_L12',
    'linear_ternary_red_0_50_c52': 'CET_L13',
    'linear_ternary_green_0_46_c42': 'CET_L14',
    'linear_ternary_blue_0_44_c57': 'CET_L15',
    'linear_kbgyw_5_98_c62': 'CET_L16',
    'linear_worb_100_25_c53': 'CET_L17',
    'linear_wyor_100_45_c55': 'CET_L18',
    'linear_wcmr_100_45_c42': 'CET_L19',
    'linear_kbgoy_20_95_c57': 'CET_L20',
    'rainbow_bgyrm_35_85_c69': 'CET_R1',
    'rainbow_bgyr_35_85_c72': 'CET_R2',
    'diverging_rainbow_bgymr_45_85_c67': 'CET_R3',
    'rainbow_bgyr_10_90_c83': 'CET_R4',
}


diverging_isoluminant_cjm_75_c23 = [  # cmap_def
[0, 0.78663, 1],
[0, 0.78616, 0.99976],
[0, 0.78569, 0.99755],
[0, 0.78523, 0.99535],
[0, 0.78477, 0.99315],
[0, 0.7843, 0.99095],
[0, 0.78383, 0.98875],
[0.051263, 0.78336, 0.98655],
[0.088344, 0.78289, 0.98435],
[0.11524, 0.78243, 0.98215],
[0.13723, 0.78197, 0.97995],
[0.15612, 0.7815, 0.97775],
[0.17285, 0.78103, 0.97555],
[0.18805, 0.78056, 0.97335],
[0.20194, 0.78009, 0.97116],
[0.21488, 0.77962, 0.96896],
[0.227, 0.77915, 0.96676],
[0.23838, 0.77868, 0.96457],
[0.24919, 0.77821, 0.96237],
[0.25948, 0.77774, 0.96018],
[0.26929, 0.77727, 0.95799],
[0.27869, 0.7768, 0.9558],
[0.28771, 0.77632, 0.9536],
[0.29642, 0.77585, 0.95141],
[0.30482, 0.77538, 0.94921],
[0.31294, 0.7749, 0.94702],
[0.32083, 0.77443, 0.94483],
[0.32846, 0.77395, 0.94264],
[0.33588, 0.77348, 0.94045],
[0.3431, 0.77301, 0.93826],
[0.35013, 0.77253, 0.93607],
[0.357, 0.77206, 0.93388],
[0.36367, 0.77159, 0.93169],
[0.37021, 0.77111, 0.9295],
[0.37662, 0.77063, 0.92732],
[0.38287, 0.77015, 0.92513],
[0.389, 0.76967, 0.92295],
[0.395, 0.76919, 0.92075],
[0.40089, 0.76872, 0.91857],
[0.40666, 0.76824, 0.91639],
[0.41233, 0.76776, 0.91421],
[0.41789, 0.76728, 0.91202],
[0.42335, 0.7668, 0.90984],
[0.42874, 0.76632, 0.90765],
[0.43402, 0.76584, 0.90547],
[0.43922, 0.76537, 0.90329],
[0.44436, 0.76488, 0.9011],
[0.4494, 0.7644, 0.89892],
[0.45436, 0.76391, 0.89674],
[0.45926, 0.76343, 0.89456],
[0.46409, 0.76295, 0.89238],
[0.46884, 0.76246, 0.8902],
[0.47352, 0.76198, 0.88802],
[0.47815, 0.7615, 0.88584],
[0.48272, 0.76101, 0.88366],
[0.48724, 0.76053, 0.88148],
[0.49169, 0.76004, 0.8793],
[0.49607, 0.75955, 0.87712],
[0.50042, 0.75907, 0.87495],
[0.50472, 0.75859, 0.87277],
[0.50896, 0.7581, 0.87059],
[0.51314, 0.75761, 0.86842],
[0.51729, 0.75712, 0.86624],
[0.52138, 0.75663, 0.86407],
[0.52543, 0.75614, 0.8619],
[0.52945, 0.75565, 0.85972],
[0.5334, 0.75517, 0.85754],
[0.53734, 0.75468, 0.85537],
[0.54121, 0.75418, 0.8532],
[0.54506, 0.75369, 0.85102],
[0.54887, 0.75319, 0.84885],
[0.55263, 0.7527, 0.84668],
[0.55635, 0.75221, 0.84451],
[0.56005, 0.75172, 0.84234],
[0.56372, 0.75123, 0.84017],
[0.56734, 0.75074, 0.838],
[0.57092, 0.75024, 0.83582],
[0.57448, 0.74974, 0.83366],
[0.57801, 0.74924, 0.83149],
[0.58151, 0.74875, 0.82932],
[0.58498, 0.74826, 0.82715],
[0.5884, 0.74776, 0.82498],
[0.59182, 0.74726, 0.82281],
[0.59519, 0.74676, 0.82064],
[0.59854, 0.74626, 0.81847],
[0.60186, 0.74576, 0.81631],
[0.60516, 0.74526, 0.81414],
[0.60843, 0.74476, 0.81198],
[0.61167, 0.74426, 0.80981],
[0.61488, 0.74376, 0.80764],
[0.61807, 0.74326, 0.80547],
[0.62124, 0.74276, 0.80331],
[0.62439, 0.74226, 0.80115],
[0.6275, 0.74175, 0.79899],
[0.6306, 0.74125, 0.79682],
[0.63367, 0.74074, 0.79466],
[0.63672, 0.74024, 0.79249],
[0.63975, 0.73974, 0.79033],
[0.64276, 0.73924, 0.78816],
[0.64574, 0.73873, 0.78601],
[0.6487, 0.73822, 0.78384],
[0.65165, 0.73771, 0.78168],
[0.65457, 0.73721, 0.77952],
[0.65748, 0.7367, 0.77736],
[0.66036, 0.73618, 0.77519],
[0.66323, 0.73568, 0.77303],
[0.66607, 0.73517, 0.77087],
[0.6689, 0.73466, 0.76871],
[0.6717, 0.73415, 0.76655],
[0.67449, 0.73364, 0.7644],
[0.67727, 0.73312, 0.76225],
[0.68001, 0.73261, 0.76012],
[0.68275, 0.7321, 0.758],
[0.68547, 0.73158, 0.75589],
[0.68816, 0.73106, 0.7538],
[0.69083, 0.73054, 0.75175],
[0.69348, 0.73001, 0.74973],
[0.69611, 0.72948, 0.74778],
[0.69871, 0.72895, 0.74588],
[0.70129, 0.72841, 0.74406],
[0.70383, 0.72787, 0.74233],
[0.70635, 0.72732, 0.74072],
[0.70883, 0.72676, 0.73924],
[0.71127, 0.7262, 0.73789],
[0.71366, 0.72563, 0.73671],
[0.71602, 0.72504, 0.73571],
[0.71832, 0.72445, 0.73489],
[0.72058, 0.72384, 0.73427],
[0.72279, 0.72323, 0.73384],
[0.72496, 0.72261, 0.73361],
[0.72707, 0.72198, 0.73357],
[0.72914, 0.72134, 0.73372],
[0.73116, 0.72069, 0.73405],
[0.73315, 0.72005, 0.73455],
[0.73509, 0.71939, 0.73518],
[0.73699, 0.71872, 0.73594],
[0.73887, 0.71805, 0.73683],
[0.74071, 0.71739, 0.7378],
[0.74254, 0.71671, 0.73887],
[0.74434, 0.71603, 0.73999],
[0.74612, 0.71534, 0.74115],
[0.74789, 0.71465, 0.74238],
[0.74964, 0.71397, 0.74362],
[0.7514, 0.71328, 0.74489],
[0.75313, 0.71258, 0.74618],
[0.75487, 0.71189, 0.74749],
[0.7566, 0.7112, 0.74879],
[0.75832, 0.7105, 0.75011],
[0.76003, 0.7098, 0.75143],
[0.76175, 0.7091, 0.75275],
[0.76346, 0.70839, 0.75408],
[0.76517, 0.70768, 0.7554],
[0.76687, 0.70697, 0.75672],
[0.76857, 0.70626, 0.75805],
[0.77028, 0.70555, 0.75937],
[0.77198, 0.70483, 0.7607],
[0.77367, 0.70412, 0.76202],
[0.77536, 0.7034, 0.76334],
[0.77706, 0.70268, 0.76467],
[0.77874, 0.70195, 0.76599],
[0.78042, 0.70123, 0.76731],
[0.78211, 0.70051, 0.76863],
[0.78379, 0.69977, 0.76996],
[0.78546, 0.69904, 0.77128],
[0.78714, 0.69831, 0.7726],
[0.78881, 0.69758, 0.77392],
[0.79048, 0.69684, 0.77525],
[0.79215, 0.69611, 0.77657],
[0.79382, 0.69536, 0.7779],
[0.79547, 0.69461, 0.77922],
[0.79713, 0.69387, 0.78054],
[0.79879, 0.69313, 0.78187],
[0.80045, 0.69238, 0.78318],
[0.8021, 0.69162, 0.78451],
[0.80375, 0.69086, 0.78583],
[0.8054, 0.69011, 0.78715],
[0.80705, 0.68936, 0.78847],
[0.80869, 0.6886, 0.7898],
[0.81033, 0.68783, 0.79112],
[0.81197, 0.68706, 0.79244],
[0.81361, 0.6863, 0.79377],
[0.81524, 0.68553, 0.79508],
[0.81688, 0.68475, 0.7964],
[0.8185, 0.68398, 0.79773],
[0.82013, 0.68321, 0.79905],
[0.82176, 0.68242, 0.80037],
[0.82338, 0.68164, 0.80169],
[0.82501, 0.68086, 0.80301],
[0.82663, 0.68007, 0.80434],
[0.82824, 0.67928, 0.80565],
[0.82986, 0.6785, 0.80698],
[0.83148, 0.67771, 0.8083],
[0.83309, 0.67691, 0.80962],
[0.8347, 0.67611, 0.81094],
[0.83631, 0.67531, 0.81226],
[0.83791, 0.67451, 0.81358],
[0.83952, 0.6737, 0.81491],
[0.84112, 0.6729, 0.81623],
[0.84272, 0.67209, 0.81755],
[0.84432, 0.67128, 0.81887],
[0.84591, 0.67047, 0.82019],
[0.8475, 0.66965, 0.82151],
[0.8491, 0.66883, 0.82284],
[0.85069, 0.66801, 0.82415],
[0.85228, 0.66718, 0.82548],
[0.85386, 0.66635, 0.8268],
[0.85545, 0.66552, 0.82812],
[0.85703, 0.66469, 0.82944],
[0.85861, 0.66386, 0.83076],
[0.8602, 0.66302, 0.83209],
[0.86177, 0.66218, 0.83341],
[0.86335, 0.66134, 0.83473],
[0.86492, 0.66049, 0.83605],
[0.86649, 0.65964, 0.83737],
[0.86807, 0.65879, 0.83869],
[0.86963, 0.65794, 0.84001],
[0.8712, 0.65709, 0.84133],
[0.87277, 0.65623, 0.84265],
[0.87433, 0.65536, 0.84397],
[0.87589, 0.6545, 0.84529],
[0.87745, 0.65363, 0.84661],
[0.87901, 0.65276, 0.84793],
[0.88056, 0.6519, 0.84926],
[0.88212, 0.65102, 0.85058],
[0.88368, 0.65014, 0.8519],
[0.88523, 0.64927, 0.85322],
[0.88678, 0.64838, 0.85454],
[0.88833, 0.64749, 0.85587],
[0.88987, 0.64661, 0.85718],
[0.89142, 0.64571, 0.8585],
[0.89296, 0.64482, 0.85983],
[0.89451, 0.64392, 0.86115],
[0.89605, 0.64302, 0.86248],
[0.89759, 0.64212, 0.8638],
[0.89912, 0.64121, 0.86511],
[0.90066, 0.64031, 0.86643],
[0.90219, 0.63939, 0.86776],
[0.90372, 0.63848, 0.86908],
[0.90526, 0.63756, 0.8704],
[0.90679, 0.63664, 0.87172],
[0.90832, 0.63572, 0.87304],
[0.90984, 0.63479, 0.87436],
[0.91137, 0.63386, 0.87569],
[0.91289, 0.63293, 0.877],
[0.91441, 0.63199, 0.87833],
[0.91593, 0.63105, 0.87965],
[0.91746, 0.63011, 0.88097],
[0.91897, 0.62916, 0.88229],
[0.92048, 0.62822, 0.88362],
[0.922, 0.62726, 0.88494],
[0.92352, 0.62631, 0.88626],
[0.92503, 0.62535, 0.88758],
[0.92654, 0.62439, 0.88891],
[0.92805, 0.62343, 0.89022],
[0.92956, 0.62245, 0.89155],
[0.93106, 0.62148, 0.89287],
]
b_diverging_isoluminant_cjm_75_c23 = bokeh_palette('diverging_isoluminant_cjm_75_c23',diverging_isoluminant_cjm_75_c23)
m_diverging_isoluminant_cjm_75_c23 = mpl_cm('diverging_isoluminant_cjm_75_c23',diverging_isoluminant_cjm_75_c23)
m_diverging_isoluminant_cjm_75_c23_r = mpl_cm('diverging_isoluminant_cjm_75_c23_r',list(reversed(diverging_isoluminant_cjm_75_c23)))
CET_D12 = b_diverging_isoluminant_cjm_75_c23
m_CET_D12 = m_diverging_isoluminant_cjm_75_c23
m_CET_D12_r = m_diverging_isoluminant_cjm_75_c23_r
palette['CET_D12'] = b_diverging_isoluminant_cjm_75_c23
cm['CET_D12'] = m_diverging_isoluminant_cjm_75_c23
cm['CET_D12_r'] = m_diverging_isoluminant_cjm_75_c23_r
register_cmap('cet_CET_D12',m_diverging_isoluminant_cjm_75_c23)
register_cmap('cet_CET_D12_r',m_diverging_isoluminant_cjm_75_c23_r)




linear_grey_10_95_c0 = [  # cmap_def
[0.10767, 0.1077, 0.1077],
[0.11032, 0.11035, 0.11035],
[0.11295, 0.11298, 0.11297],
[0.11554, 0.11558, 0.11557],
[0.1182, 0.11824, 0.11823],
[0.12079, 0.12083, 0.12082],
[0.12344, 0.12348, 0.12347],
[0.12615, 0.12618, 0.12618],
[0.12879, 0.12882, 0.12881],
[0.13149, 0.13152, 0.13151],
[0.13418, 0.13421, 0.1342],
[0.13684, 0.13688, 0.13687],
[0.13951, 0.13955, 0.13954],
[0.14226, 0.1423, 0.14229],
[0.14499, 0.14503, 0.14502],
[0.1477, 0.14774, 0.14773],
[0.15042, 0.15046, 0.15045],
[0.15313, 0.15317, 0.15316],
[0.15591, 0.15595, 0.15594],
[0.15866, 0.1587, 0.15869],
[0.16142, 0.16147, 0.16145],
[0.16418, 0.16423, 0.16422],
[0.16695, 0.16699, 0.16698],
[0.16973, 0.16977, 0.16976],
[0.17248, 0.17253, 0.17252],
[0.17529, 0.17533, 0.17532],
[0.17811, 0.17815, 0.17814],
[0.18087, 0.18092, 0.1809],
[0.18369, 0.18374, 0.18372],
[0.18652, 0.18656, 0.18655],
[0.18934, 0.18939, 0.18938],
[0.19217, 0.19221, 0.1922],
[0.19502, 0.19506, 0.19505],
[0.19785, 0.1979, 0.19788],
[0.20068, 0.20073, 0.20072],
[0.20357, 0.20362, 0.20361],
[0.20645, 0.2065, 0.20649],
[0.20929, 0.20934, 0.20933],
[0.21219, 0.21224, 0.21222],
[0.21504, 0.21509, 0.21508],
[0.21795, 0.218, 0.21799],
[0.22086, 0.22091, 0.2209],
[0.22374, 0.22379, 0.22377],
[0.22666, 0.22671, 0.22669],
[0.22954, 0.2296, 0.22958],
[0.23248, 0.23253, 0.23252],
[0.23542, 0.23547, 0.23546],
[0.23832, 0.23838, 0.23836],
[0.24127, 0.24133, 0.24131],
[0.24419, 0.24425, 0.24424],
[0.24716, 0.24722, 0.2472],
[0.25009, 0.25015, 0.25014],
[0.25308, 0.25313, 0.25312],
[0.25603, 0.25608, 0.25607],
[0.25902, 0.25908, 0.25906],
[0.26198, 0.26204, 0.26203],
[0.26496, 0.26502, 0.265],
[0.26794, 0.268, 0.26798],
[0.27095, 0.27101, 0.271],
[0.27395, 0.27401, 0.274],
[0.27695, 0.27702, 0.277],
[0.27996, 0.28002, 0.28],
[0.28298, 0.28304, 0.28303],
[0.28598, 0.28604, 0.28603],
[0.28902, 0.28909, 0.28907],
[0.29205, 0.29212, 0.2921],
[0.29508, 0.29514, 0.29513],
[0.29812, 0.29818, 0.29817],
[0.30116, 0.30123, 0.30121],
[0.30422, 0.30429, 0.30427],
[0.30728, 0.30734, 0.30733],
[0.31036, 0.31043, 0.31041],
[0.31342, 0.31349, 0.31347],
[0.31649, 0.31656, 0.31654],
[0.31957, 0.31964, 0.31962],
[0.32266, 0.32273, 0.32271],
[0.32572, 0.3258, 0.32578],
[0.32883, 0.32891, 0.32889],
[0.33193, 0.332, 0.33198],
[0.33504, 0.33512, 0.3351],
[0.33813, 0.3382, 0.33818],
[0.34125, 0.34133, 0.34131],
[0.34436, 0.34444, 0.34442],
[0.3475, 0.34757, 0.34755],
[0.35063, 0.3507, 0.35068],
[0.35374, 0.35382, 0.3538],
[0.35689, 0.35697, 0.35695],
[0.36002, 0.3601, 0.36008],
[0.36317, 0.36325, 0.36323],
[0.36633, 0.36641, 0.36639],
[0.36948, 0.36956, 0.36954],
[0.37263, 0.37272, 0.3727],
[0.3758, 0.37589, 0.37587],
[0.37897, 0.37906, 0.37904],
[0.38214, 0.38223, 0.38221],
[0.38532, 0.3854, 0.38538],
[0.38852, 0.3886, 0.38858],
[0.3917, 0.39179, 0.39177],
[0.39489, 0.39498, 0.39496],
[0.3981, 0.39818, 0.39816],
[0.4013, 0.40138, 0.40136],
[0.40449, 0.40458, 0.40456],
[0.40771, 0.4078, 0.40778],
[0.41093, 0.41102, 0.411],
[0.41415, 0.41423, 0.41421],
[0.41738, 0.41747, 0.41744],
[0.4206, 0.42068, 0.42066],
[0.42383, 0.42392, 0.4239],
[0.42708, 0.42717, 0.42715],
[0.43031, 0.43041, 0.43038],
[0.43355, 0.43364, 0.43362],
[0.43681, 0.43691, 0.43688],
[0.44007, 0.44016, 0.44014],
[0.44333, 0.44342, 0.4434],
[0.44659, 0.44668, 0.44666],
[0.44986, 0.44995, 0.44993],
[0.45313, 0.45322, 0.4532],
[0.4564, 0.4565, 0.45647],
[0.45968, 0.45978, 0.45976],
[0.46296, 0.46306, 0.46303],
[0.46625, 0.46635, 0.46633],
[0.46956, 0.46966, 0.46963],
[0.47284, 0.47294, 0.47292],
[0.47615, 0.47625, 0.47623],
[0.47946, 0.47956, 0.47953],
[0.48276, 0.48286, 0.48284],
[0.48607, 0.48618, 0.48615],
[0.48939, 0.48949, 0.48947],
[0.49271, 0.49281, 0.49279],
[0.49603, 0.49614, 0.49611],
[0.49936, 0.49947, 0.49944],
[0.5027, 0.5028, 0.50278],
[0.50603, 0.50614, 0.50612],
[0.50938, 0.50949, 0.50946],
[0.51273, 0.51284, 0.51281],
[0.51607, 0.51618, 0.51615],
[0.51943, 0.51954, 0.51951],
[0.52279, 0.5229, 0.52287],
[0.52615, 0.52626, 0.52624],
[0.52952, 0.52963, 0.5296],
[0.53289, 0.533, 0.53297],
[0.53626, 0.53637, 0.53634],
[0.53963, 0.53974, 0.53971],
[0.54302, 0.54313, 0.5431],
[0.54641, 0.54652, 0.54649],
[0.54979, 0.5499, 0.54987],
[0.55317, 0.55329, 0.55326],
[0.55657, 0.55669, 0.55666],
[0.55998, 0.56009, 0.56007],
[0.56338, 0.5635, 0.56347],
[0.56679, 0.56691, 0.56688],
[0.57019, 0.57031, 0.57028],
[0.57361, 0.57373, 0.5737],
[0.57703, 0.57715, 0.57712],
[0.58045, 0.58057, 0.58054],
[0.58387, 0.58399, 0.58396],
[0.5873, 0.58742, 0.58739],
[0.59073, 0.59086, 0.59083],
[0.59417, 0.5943, 0.59427],
[0.59761, 0.59774, 0.59771],
[0.60106, 0.60118, 0.60115],
[0.60451, 0.60463, 0.6046],
[0.60796, 0.60809, 0.60806],
[0.61141, 0.61154, 0.61151],
[0.61486, 0.61499, 0.61496],
[0.61833, 0.61846, 0.61843],
[0.62179, 0.62192, 0.62189],
[0.62527, 0.62539, 0.62536],
[0.62874, 0.62887, 0.62884],
[0.63222, 0.63235, 0.63231],
[0.63569, 0.63583, 0.63579],
[0.63918, 0.63931, 0.63928],
[0.64267, 0.6428, 0.64276],
[0.64615, 0.64629, 0.64625],
[0.64965, 0.64978, 0.64975],
[0.65314, 0.65327, 0.65324],
[0.65665, 0.65678, 0.65675],
[0.66015, 0.66028, 0.66025],
[0.66366, 0.6638, 0.66376],
[0.66717, 0.6673, 0.66727],
[0.67069, 0.67082, 0.67079],
[0.6742, 0.67434, 0.6743],
[0.67772, 0.67786, 0.67783],
[0.68124, 0.68138, 0.68134],
[0.68477, 0.68491, 0.68488],
[0.68831, 0.68845, 0.68841],
[0.69184, 0.69198, 0.69195],
[0.69538, 0.69552, 0.69548],
[0.69891, 0.69906, 0.69902],
[0.70246, 0.7026, 0.70257],
[0.70601, 0.70616, 0.70612],
[0.70956, 0.70971, 0.70967],
[0.71311, 0.71326, 0.71322],
[0.71668, 0.71682, 0.71679],
[0.72023, 0.72038, 0.72034],
[0.7238, 0.72394, 0.72391],
[0.72736, 0.72751, 0.72748],
[0.73093, 0.73108, 0.73105],
[0.73451, 0.73466, 0.73463],
[0.73808, 0.73823, 0.7382],
[0.74167, 0.74182, 0.74178],
[0.74525, 0.7454, 0.74536],
[0.74884, 0.74899, 0.74895],
[0.75243, 0.75258, 0.75254],
[0.75602, 0.75618, 0.75614],
[0.75961, 0.75977, 0.75973],
[0.76322, 0.76337, 0.76333],
[0.76682, 0.76697, 0.76694],
[0.77043, 0.77058, 0.77055],
[0.77403, 0.77419, 0.77415],
[0.77765, 0.77781, 0.77777],
[0.78126, 0.78142, 0.78138],
[0.78489, 0.78504, 0.785],
[0.78851, 0.78867, 0.78863],
[0.79213, 0.79229, 0.79225],
[0.79576, 0.79592, 0.79588],
[0.79939, 0.79955, 0.79951],
[0.80302, 0.80318, 0.80314],
[0.80666, 0.80683, 0.80679],
[0.8103, 0.81046, 0.81042],
[0.81394, 0.81411, 0.81406],
[0.81759, 0.81775, 0.81771],
[0.82124, 0.82141, 0.82136],
[0.82489, 0.82506, 0.82501],
[0.82855, 0.82872, 0.82867],
[0.83221, 0.83237, 0.83233],
[0.83587, 0.83604, 0.83599],
[0.83953, 0.8397, 0.83966],
[0.8432, 0.84337, 0.84333],
[0.84687, 0.84704, 0.847],
[0.85054, 0.85071, 0.85067],
[0.85422, 0.85439, 0.85435],
[0.8579, 0.85807, 0.85803],
[0.86158, 0.86176, 0.86171],
[0.86527, 0.86544, 0.8654],
[0.86896, 0.86913, 0.86909],
[0.87265, 0.87282, 0.87278],
[0.87634, 0.87652, 0.87647],
[0.88004, 0.88022, 0.88017],
[0.88374, 0.88392, 0.88388],
[0.88744, 0.88762, 0.88758],
[0.89115, 0.89133, 0.89128],
[0.89486, 0.89504, 0.895],
[0.89857, 0.89875, 0.89871],
[0.90229, 0.90247, 0.90242],
[0.90601, 0.90619, 0.90614],
[0.90972, 0.90991, 0.90986],
[0.91345, 0.91363, 0.91359],
[0.91718, 0.91736, 0.91732],
[0.9209, 0.92108, 0.92104],
[0.92463, 0.92482, 0.92477],
[0.92837, 0.92855, 0.92851],
[0.9321, 0.93229, 0.93224],
[0.93585, 0.93604, 0.93599],
[0.93959, 0.93978, 0.93973],
[0.94334, 0.94353, 0.94348],
]
b_linear_grey_10_95_c0 = bokeh_palette('linear_grey_10_95_c0',linear_grey_10_95_c0)
m_linear_grey_10_95_c0 = mpl_cm('linear_grey_10_95_c0',linear_grey_10_95_c0)
m_linear_grey_10_95_c0_r = mpl_cm('linear_grey_10_95_c0_r',list(reversed(linear_grey_10_95_c0)))
dimgray = b_linear_grey_10_95_c0
m_dimgray = m_linear_grey_10_95_c0
m_dimgray_r = m_linear_grey_10_95_c0_r
palette['dimgray'] = b_linear_grey_10_95_c0
palette_n['dimgray'] = b_linear_grey_10_95_c0
cm['dimgray'] = m_linear_grey_10_95_c0
cm['dimgray_r'] = m_linear_grey_10_95_c0_r
cm_n['dimgray'] = mpl_cm('dimgray',linear_grey_10_95_c0)
cm_n['dimgray_r'] = mpl_cm('dimgray_r',list(reversed(linear_grey_10_95_c0)))
CET_L2 = b_linear_grey_10_95_c0
m_CET_L2 = m_linear_grey_10_95_c0
m_CET_L2_r = m_linear_grey_10_95_c0_r
palette['CET_L2'] = b_linear_grey_10_95_c0
cm['CET_L2'] = m_linear_grey_10_95_c0
cm['CET_L2_r'] = m_linear_grey_10_95_c0_r
register_cmap('cet_CET_L2',m_linear_grey_10_95_c0)
register_cmap('cet_CET_L2_r',m_linear_grey_10_95_c0_r)




isoluminant_cgo_70_c39 = [  # cmap_def
[0.21566, 0.71777, 0.92594],
[0.21805, 0.71808, 0.92254],
[0.2204, 0.71839, 0.91913],
[0.22272, 0.7187, 0.91573],
[0.22499, 0.71901, 0.91232],
[0.22727, 0.71931, 0.90891],
[0.2295, 0.71962, 0.9055],
[0.23174, 0.71992, 0.90208],
[0.23392, 0.72022, 0.89866],
[0.23611, 0.72051, 0.89524],
[0.23825, 0.72081, 0.89181],
[0.24038, 0.7211, 0.88838],
[0.2425, 0.72139, 0.88495],
[0.2446, 0.72168, 0.88151],
[0.24668, 0.72196, 0.87807],
[0.24877, 0.72225, 0.87462],
[0.25081, 0.72253, 0.87117],
[0.25284, 0.72281, 0.86772],
[0.25488, 0.72309, 0.86426],
[0.25687, 0.72336, 0.8608],
[0.25887, 0.72363, 0.85734],
[0.26085, 0.7239, 0.85387],
[0.26281, 0.72417, 0.8504],
[0.26477, 0.72444, 0.84692],
[0.26672, 0.7247, 0.84344],
[0.26866, 0.72496, 0.83996],
[0.27061, 0.72522, 0.83647],
[0.2725, 0.72548, 0.83297],
[0.27442, 0.72573, 0.82948],
[0.27635, 0.72598, 0.82598],
[0.27824, 0.72622, 0.82247],
[0.28012, 0.72647, 0.81896],
[0.28203, 0.72671, 0.81545],
[0.28389, 0.72694, 0.81193],
[0.28576, 0.72718, 0.8084],
[0.28764, 0.72741, 0.80487],
[0.28953, 0.72764, 0.80134],
[0.2914, 0.72787, 0.7978],
[0.29325, 0.72809, 0.79426],
[0.29511, 0.72831, 0.79071],
[0.29699, 0.72853, 0.78716],
[0.29885, 0.72875, 0.7836],
[0.3007, 0.72896, 0.78003],
[0.30255, 0.72917, 0.77647],
[0.30443, 0.72937, 0.77289],
[0.30631, 0.72958, 0.76931],
[0.30815, 0.72978, 0.76573],
[0.31004, 0.72997, 0.76213],
[0.31189, 0.73016, 0.75854],
[0.31378, 0.73035, 0.75494],
[0.31565, 0.73053, 0.75133],
[0.31753, 0.73071, 0.74772],
[0.31943, 0.73089, 0.74409],
[0.32131, 0.73106, 0.74047],
[0.32321, 0.73123, 0.73684],
[0.32511, 0.7314, 0.7332],
[0.32703, 0.73156, 0.72956],
[0.32897, 0.73172, 0.72591],
[0.33088, 0.73187, 0.72225],
[0.33284, 0.73202, 0.71859],
[0.33479, 0.73217, 0.71492],
[0.33674, 0.73231, 0.71125],
[0.33871, 0.73244, 0.70756],
[0.34069, 0.73257, 0.70388],
[0.34269, 0.7327, 0.70018],
[0.34469, 0.73282, 0.69648],
[0.34672, 0.73293, 0.69278],
[0.34875, 0.73304, 0.68906],
[0.35081, 0.73315, 0.68534],
[0.35286, 0.73325, 0.68161],
[0.35496, 0.73334, 0.67788],
[0.35706, 0.73343, 0.67413],
[0.35916, 0.73351, 0.67039],
[0.3613, 0.73359, 0.66663],
[0.36345, 0.73366, 0.66287],
[0.36562, 0.73373, 0.6591],
[0.36781, 0.73379, 0.65532],
[0.37003, 0.73384, 0.65155],
[0.37227, 0.73388, 0.64775],
[0.37454, 0.73392, 0.64396],
[0.37683, 0.73395, 0.64016],
[0.37914, 0.73398, 0.63635],
[0.38148, 0.73399, 0.63254],
[0.38384, 0.734, 0.62872],
[0.38622, 0.734, 0.62489],
[0.38865, 0.734, 0.62106],
[0.3911, 0.73398, 0.61722],
[0.39357, 0.73396, 0.61337],
[0.39609, 0.73392, 0.60953],
[0.39863, 0.73388, 0.60567],
[0.40121, 0.73383, 0.60181],
[0.40382, 0.73377, 0.59795],
[0.40646, 0.7337, 0.59409],
[0.40914, 0.73361, 0.59021],
[0.41186, 0.73352, 0.58635],
[0.41461, 0.73342, 0.58247],
[0.41741, 0.7333, 0.5786],
[0.42024, 0.73318, 0.57472],
[0.4231, 0.73304, 0.57084],
[0.42602, 0.73289, 0.56698],
[0.42898, 0.73273, 0.5631],
[0.43199, 0.73255, 0.55923],
[0.43504, 0.73236, 0.55536],
[0.43813, 0.73216, 0.5515],
[0.44125, 0.73194, 0.54764],
[0.44444, 0.73171, 0.54379],
[0.44765, 0.73146, 0.53995],
[0.45093, 0.73119, 0.53612],
[0.45426, 0.73092, 0.53231],
[0.45762, 0.73062, 0.5285],
[0.46104, 0.73032, 0.52471],
[0.46451, 0.72999, 0.52094],
[0.468, 0.72965, 0.51719],
[0.47157, 0.72928, 0.51346],
[0.47518, 0.7289, 0.50975],
[0.47882, 0.7285, 0.50606],
[0.48251, 0.72809, 0.50241],
[0.48625, 0.72766, 0.49879],
[0.49004, 0.7272, 0.4952],
[0.49386, 0.72673, 0.49165],
[0.49773, 0.72625, 0.48813],
[0.50164, 0.72574, 0.48464],
[0.50557, 0.72521, 0.48121],
[0.50955, 0.72466, 0.47782],
[0.51357, 0.72409, 0.47449],
[0.51761, 0.72351, 0.4712],
[0.52167, 0.7229, 0.46795],
[0.52578, 0.72228, 0.46477],
[0.5299, 0.72164, 0.46164],
[0.53404, 0.72098, 0.45857],
[0.53822, 0.7203, 0.45556],
[0.5424, 0.71961, 0.45262],
[0.5466, 0.7189, 0.44973],
[0.55081, 0.71817, 0.4469],
[0.55503, 0.71743, 0.44415],
[0.55926, 0.71667, 0.44145],
[0.5635, 0.71589, 0.43882],
[0.56773, 0.71509, 0.43627],
[0.57197, 0.71428, 0.43376],
[0.57622, 0.71346, 0.43134],
[0.58045, 0.71262, 0.42898],
[0.58468, 0.71178, 0.42669],
[0.5889, 0.71092, 0.42445],
[0.59311, 0.71004, 0.42224],
[0.5973, 0.70917, 0.42009],
[0.60146, 0.70828, 0.41796],
[0.60561, 0.70738, 0.41587],
[0.60975, 0.70647, 0.41382],
[0.61386, 0.70556, 0.4118],
[0.61796, 0.70463, 0.40983],
[0.62204, 0.7037, 0.40789],
[0.62612, 0.70276, 0.406],
[0.63017, 0.70181, 0.40413],
[0.63421, 0.70085, 0.4023],
[0.63822, 0.69988, 0.40055],
[0.64223, 0.69891, 0.3988],
[0.64621, 0.69792, 0.39711],
[0.65019, 0.69693, 0.39547],
[0.65415, 0.69593, 0.39385],
[0.65809, 0.69492, 0.39229],
[0.66202, 0.6939, 0.39078],
[0.66593, 0.69288, 0.3893],
[0.66983, 0.69184, 0.38787],
[0.67371, 0.69079, 0.38648],
[0.67758, 0.68974, 0.38515],
[0.68143, 0.68868, 0.38386],
[0.68527, 0.6876, 0.38261],
[0.6891, 0.68652, 0.38142],
[0.69291, 0.68544, 0.38026],
[0.6967, 0.68433, 0.37915],
[0.70047, 0.68324, 0.37809],
[0.70424, 0.68212, 0.37708],
[0.70798, 0.681, 0.37611],
[0.71172, 0.67987, 0.37518],
[0.71544, 0.67874, 0.37432],
[0.71914, 0.6776, 0.37349],
[0.72282, 0.67644, 0.37271],
[0.7265, 0.67528, 0.37197],
[0.73016, 0.67411, 0.37128],
[0.7338, 0.67294, 0.37065],
[0.73742, 0.67175, 0.37006],
[0.74103, 0.67057, 0.36951],
[0.74462, 0.66937, 0.36902],
[0.74821, 0.66816, 0.36856],
[0.75177, 0.66694, 0.36815],
[0.75531, 0.66573, 0.36778],
[0.75884, 0.6645, 0.36746],
[0.76236, 0.66327, 0.36719],
[0.76586, 0.66203, 0.36696],
[0.76934, 0.66077, 0.36678],
[0.77281, 0.65952, 0.36664],
[0.77626, 0.65826, 0.36654],
[0.7797, 0.65699, 0.36649],
[0.78312, 0.65572, 0.36647],
[0.78653, 0.65443, 0.3665],
[0.78991, 0.65314, 0.36657],
[0.79329, 0.65185, 0.36668],
[0.79664, 0.65055, 0.36682],
[0.79999, 0.64925, 0.36701],
[0.80331, 0.64792, 0.36723],
[0.80662, 0.64661, 0.3675],
[0.80991, 0.64528, 0.36781],
[0.81318, 0.64395, 0.36816],
[0.81645, 0.64261, 0.36854],
[0.81969, 0.64126, 0.36896],
[0.82293, 0.63992, 0.36941],
[0.82614, 0.63856, 0.36989],
[0.82935, 0.6372, 0.37041],
[0.83253, 0.63583, 0.37097],
[0.8357, 0.63446, 0.37156],
[0.83885, 0.63308, 0.37219],
[0.84199, 0.63169, 0.37285],
[0.84511, 0.6303, 0.37355],
[0.84822, 0.6289, 0.37427],
[0.85131, 0.6275, 0.37502],
[0.85439, 0.62609, 0.3758],
[0.85745, 0.62468, 0.37663],
[0.86051, 0.62326, 0.37746],
[0.86354, 0.62183, 0.37833],
[0.86656, 0.62041, 0.37923],
[0.86956, 0.61897, 0.38016],
[0.87256, 0.61752, 0.38112],
[0.87554, 0.61607, 0.38209],
[0.8785, 0.61462, 0.38312],
[0.88145, 0.61316, 0.38414],
[0.88438, 0.6117, 0.3852],
[0.8873, 0.61023, 0.38628],
[0.89021, 0.60876, 0.3874],
[0.8931, 0.60728, 0.38854],
[0.89598, 0.60578, 0.38969],
[0.89885, 0.60429, 0.39088],
[0.9017, 0.60279, 0.39208],
[0.90454, 0.60128, 0.3933],
[0.90737, 0.59978, 0.39455],
[0.91018, 0.59826, 0.39583],
[0.91298, 0.59674, 0.3971],
[0.91577, 0.59521, 0.39842],
[0.91854, 0.59368, 0.39974],
[0.92131, 0.59215, 0.40111],
[0.92406, 0.59059, 0.40246],
[0.92679, 0.58904, 0.40386],
[0.92952, 0.58748, 0.40527],
[0.93223, 0.58593, 0.40669],
[0.93493, 0.58436, 0.40813],
[0.93763, 0.58278, 0.4096],
[0.9403, 0.5812, 0.41108],
[0.94297, 0.57962, 0.41258],
[0.94562, 0.57802, 0.41408],
[0.94826, 0.57644, 0.41561],
[0.95089, 0.57482, 0.41716],
[0.95351, 0.57322, 0.41871],
[0.95612, 0.57159, 0.42029],
[0.95872, 0.56997, 0.42188],
[0.9613, 0.56834, 0.42348],
[0.96388, 0.56671, 0.42511],
[0.96644, 0.56505, 0.42674],
]
b_isoluminant_cgo_70_c39 = bokeh_palette('isoluminant_cgo_70_c39',isoluminant_cgo_70_c39)
m_isoluminant_cgo_70_c39 = mpl_cm('isoluminant_cgo_70_c39',isoluminant_cgo_70_c39)
m_isoluminant_cgo_70_c39_r = mpl_cm('isoluminant_cgo_70_c39_r',list(reversed(isoluminant_cgo_70_c39)))
CET_I1 = b_isoluminant_cgo_70_c39
m_CET_I1 = m_isoluminant_cgo_70_c39
m_CET_I1_r = m_isoluminant_cgo_70_c39_r
palette['CET_I1'] = b_isoluminant_cgo_70_c39
cm['CET_I1'] = m_isoluminant_cgo_70_c39
cm['CET_I1_r'] = m_isoluminant_cgo_70_c39_r
register_cmap('cet_CET_I1',m_isoluminant_cgo_70_c39)
register_cmap('cet_CET_I1_r',m_isoluminant_cgo_70_c39_r)




diverging_bky_60_10_c30 = [  # cmap_def
[0.056674, 0.57959, 0.98121],
[0.066107, 0.57547, 0.97345],
[0.074463, 0.57135, 0.9657],
[0.081994, 0.56724, 0.95796],
[0.088742, 0.56314, 0.95024],
[0.09499, 0.55904, 0.94252],
[0.10067, 0.55495, 0.93482],
[0.10603, 0.55086, 0.92713],
[0.11098, 0.54678, 0.91945],
[0.1156, 0.54271, 0.91178],
[0.11993, 0.53864, 0.90413],
[0.12401, 0.53457, 0.89649],
[0.1279, 0.53052, 0.88886],
[0.13155, 0.52647, 0.88124],
[0.135, 0.52241, 0.87363],
[0.13827, 0.51838, 0.86604],
[0.14137, 0.51435, 0.85845],
[0.1443, 0.51031, 0.85088],
[0.14707, 0.50628, 0.84333],
[0.14975, 0.50227, 0.83578],
[0.15224, 0.49827, 0.82825],
[0.15465, 0.49426, 0.82073],
[0.15694, 0.49026, 0.81323],
[0.15911, 0.48626, 0.80573],
[0.1612, 0.48228, 0.79826],
[0.16316, 0.4783, 0.79079],
[0.16499, 0.47433, 0.78334],
[0.1668, 0.47036, 0.77589],
[0.16848, 0.46639, 0.76846],
[0.17009, 0.46243, 0.76105],
[0.1716, 0.45849, 0.75365],
[0.17304, 0.45455, 0.74626],
[0.17442, 0.45061, 0.73889],
[0.17569, 0.44668, 0.73153],
[0.17693, 0.44275, 0.72418],
[0.17811, 0.43883, 0.71685],
[0.17917, 0.43493, 0.70953],
[0.18019, 0.43102, 0.70222],
[0.18114, 0.42712, 0.69493],
[0.18204, 0.42322, 0.68765],
[0.18288, 0.41934, 0.68038],
[0.18367, 0.41546, 0.67314],
[0.1844, 0.41158, 0.6659],
[0.18509, 0.40772, 0.65868],
[0.18571, 0.40387, 0.65148],
[0.18626, 0.40001, 0.64428],
[0.18676, 0.39617, 0.6371],
[0.18724, 0.39232, 0.62994],
[0.18767, 0.3885, 0.62279],
[0.18804, 0.38466, 0.61565],
[0.18835, 0.38086, 0.60854],
[0.18861, 0.37705, 0.60143],
[0.18884, 0.37323, 0.59434],
[0.18902, 0.36944, 0.58727],
[0.18916, 0.36565, 0.58021],
[0.18925, 0.36187, 0.57317],
[0.18931, 0.3581, 0.56614],
[0.18932, 0.35433, 0.55913],
[0.18929, 0.35058, 0.55214],
[0.18923, 0.34682, 0.54515],
[0.18912, 0.34307, 0.53819],
[0.18898, 0.33934, 0.53124],
[0.1888, 0.3356, 0.52431],
[0.18859, 0.33187, 0.51739],
[0.18833, 0.32815, 0.51049],
[0.18805, 0.32444, 0.5036],
[0.18772, 0.32075, 0.49673],
[0.18734, 0.31705, 0.48988],
[0.18692, 0.31337, 0.48305],
[0.18649, 0.3097, 0.47624],
[0.18603, 0.30603, 0.46945],
[0.18554, 0.30234, 0.46265],
[0.18499, 0.2987, 0.45589],
[0.1844, 0.29505, 0.44915],
[0.18379, 0.29142, 0.44242],
[0.18315, 0.28778, 0.43571],
[0.18248, 0.28416, 0.429],
[0.18178, 0.28053, 0.42232],
[0.18105, 0.27695, 0.41567],
[0.18029, 0.27335, 0.40903],
[0.1795, 0.26977, 0.4024],
[0.1787, 0.26618, 0.39581],
[0.17786, 0.26261, 0.38922],
[0.17694, 0.25905, 0.38265],
[0.176, 0.25549, 0.37611],
[0.17509, 0.25195, 0.36958],
[0.17411, 0.24842, 0.36307],
[0.1731, 0.24486, 0.35659],
[0.17207, 0.24136, 0.35012],
[0.17101, 0.23785, 0.34366],
[0.16994, 0.23435, 0.33724],
[0.16882, 0.23085, 0.33082],
[0.16766, 0.22737, 0.32443],
[0.16649, 0.22389, 0.31807],
[0.16526, 0.22045, 0.31171],
[0.16408, 0.21698, 0.3054],
[0.16281, 0.21352, 0.29909],
[0.16154, 0.21009, 0.2928],
[0.16017, 0.20669, 0.28653],
[0.15887, 0.20325, 0.28029],
[0.15748, 0.19982, 0.27408],
[0.15612, 0.19646, 0.26786],
[0.15467, 0.19306, 0.26171],
[0.15321, 0.18968, 0.25556],
[0.15174, 0.18632, 0.24942],
[0.15026, 0.18294, 0.2433],
[0.14871, 0.17961, 0.23725],
[0.14718, 0.17625, 0.23118],
[0.14564, 0.17295, 0.22512],
[0.14403, 0.16968, 0.21913],
[0.14243, 0.16638, 0.21314],
[0.14084, 0.16314, 0.20723],
[0.13919, 0.15988, 0.2013],
[0.13765, 0.15674, 0.19547],
[0.13602, 0.15358, 0.18968],
[0.13455, 0.15054, 0.18396],
[0.13306, 0.14756, 0.17836],
[0.13167, 0.14469, 0.1728],
[0.13042, 0.14192, 0.16741],
[0.12922, 0.13927, 0.16217],
[0.12825, 0.13686, 0.15706],
[0.12743, 0.13466, 0.15217],
[0.12685, 0.13265, 0.14756],
[0.12655, 0.13093, 0.14317],
[0.12652, 0.12946, 0.13905],
[0.12678, 0.12834, 0.1353],
[0.12737, 0.12752, 0.13186],
[0.12833, 0.12704, 0.12874],
[0.12961, 0.12694, 0.12604],
[0.13123, 0.12718, 0.12362],
[0.13318, 0.1278, 0.12159],
[0.13545, 0.1287, 0.11992],
[0.13807, 0.12997, 0.11861],
[0.14093, 0.13147, 0.11751],
[0.14404, 0.13325, 0.11674],
[0.14739, 0.13526, 0.11622],
[0.15092, 0.1375, 0.11588],
[0.15463, 0.13984, 0.11572],
[0.15851, 0.14242, 0.11571],
[0.16247, 0.14511, 0.11584],
[0.16652, 0.14785, 0.11606],
[0.17065, 0.15069, 0.11635],
[0.17485, 0.15359, 0.11668],
[0.17908, 0.15658, 0.11705],
[0.18333, 0.15955, 0.11747],
[0.18766, 0.16261, 0.11793],
[0.19194, 0.16563, 0.11839],
[0.19628, 0.16874, 0.11884],
[0.20058, 0.17182, 0.11927],
[0.20495, 0.17495, 0.11971],
[0.2093, 0.17808, 0.12015],
[0.21365, 0.18116, 0.12058],
[0.21802, 0.18431, 0.12101],
[0.22237, 0.18748, 0.12142],
[0.22675, 0.19061, 0.12183],
[0.23112, 0.19379, 0.12224],
[0.23551, 0.19696, 0.12264],
[0.23987, 0.20011, 0.12303],
[0.24425, 0.20332, 0.12341],
[0.24867, 0.20654, 0.12378],
[0.25306, 0.20972, 0.12415],
[0.25746, 0.21292, 0.12451],
[0.26188, 0.21617, 0.12486],
[0.2663, 0.21939, 0.12521],
[0.27074, 0.22263, 0.12555],
[0.27516, 0.22588, 0.12587],
[0.2796, 0.22912, 0.12618],
[0.28403, 0.23239, 0.12646],
[0.28849, 0.23566, 0.12673],
[0.29294, 0.23892, 0.12701],
[0.29741, 0.24221, 0.12728],
[0.30187, 0.24549, 0.12756],
[0.30637, 0.24881, 0.12782],
[0.31086, 0.25212, 0.12807],
[0.31534, 0.25543, 0.1283],
[0.31985, 0.25875, 0.12852],
[0.32434, 0.26208, 0.12872],
[0.32887, 0.26541, 0.12892],
[0.33339, 0.26877, 0.12911],
[0.3379, 0.27209, 0.12931],
[0.34245, 0.27548, 0.12949],
[0.34699, 0.27884, 0.12967],
[0.35153, 0.28221, 0.12983],
[0.35609, 0.28557, 0.12998],
[0.36064, 0.28897, 0.13012],
[0.36522, 0.29236, 0.13024],
[0.36979, 0.29575, 0.13035],
[0.37439, 0.29917, 0.13045],
[0.37897, 0.30257, 0.13053],
[0.38358, 0.30601, 0.13061],
[0.38818, 0.30944, 0.13067],
[0.39279, 0.31285, 0.13073],
[0.39741, 0.31631, 0.13077],
[0.40203, 0.31975, 0.1308],
[0.40668, 0.3232, 0.13082],
[0.41131, 0.32665, 0.13083],
[0.41596, 0.33013, 0.13082],
[0.42062, 0.3336, 0.13081],
[0.42529, 0.33708, 0.13078],
[0.42996, 0.34056, 0.13075],
[0.43464, 0.34404, 0.1307],
[0.43932, 0.34756, 0.13063],
[0.44402, 0.35106, 0.13056],
[0.44872, 0.35457, 0.13047],
[0.45341, 0.35808, 0.13037],
[0.45813, 0.3616, 0.13025],
[0.46284, 0.36513, 0.13012],
[0.46758, 0.36867, 0.12998],
[0.47231, 0.3722, 0.12981],
[0.47705, 0.37575, 0.12963],
[0.4818, 0.3793, 0.12943],
[0.48655, 0.38286, 0.12922],
[0.49133, 0.38642, 0.129],
[0.49608, 0.38999, 0.12877],
[0.50086, 0.39357, 0.12854],
[0.50564, 0.39715, 0.1283],
[0.51043, 0.40075, 0.12803],
[0.51523, 0.40433, 0.12773],
[0.52003, 0.40793, 0.1274],
[0.52484, 0.41154, 0.12707],
[0.52966, 0.41515, 0.12674],
[0.53448, 0.41876, 0.12641],
[0.53931, 0.42238, 0.12606],
[0.54415, 0.42601, 0.12567],
[0.549, 0.42965, 0.12524],
[0.55385, 0.43328, 0.12481],
[0.5587, 0.43694, 0.12436],
[0.56357, 0.44058, 0.1239],
[0.56844, 0.44424, 0.12341],
[0.57332, 0.44789, 0.12291],
[0.5782, 0.45157, 0.1224],
[0.58309, 0.45524, 0.12186],
[0.58798, 0.45892, 0.12131],
[0.5929, 0.46259, 0.12074],
[0.5978, 0.46628, 0.12015],
[0.60272, 0.46998, 0.11954],
[0.60765, 0.47367, 0.11893],
[0.61257, 0.47738, 0.11828],
[0.61751, 0.48108, 0.11755],
[0.62245, 0.48479, 0.11685],
[0.6274, 0.48852, 0.11615],
[0.63236, 0.49224, 0.11536],
[0.63732, 0.49596, 0.11454],
[0.64229, 0.4997, 0.11378],
[0.64726, 0.50344, 0.11298],
[0.65224, 0.50719, 0.1121],
[0.65724, 0.51093, 0.11121],
[0.66223, 0.5147, 0.11031],
[0.66723, 0.51846, 0.10936],
[0.67223, 0.52221, 0.10832],
[0.67725, 0.52599, 0.10736],
[0.68226, 0.52976, 0.10634],
[0.68729, 0.53353, 0.10521],
[0.69232, 0.53733, 0.10412],
[0.69735, 0.54111, 0.103],
[0.7024, 0.54491, 0.1018],
]
b_diverging_bky_60_10_c30 = bokeh_palette('diverging_bky_60_10_c30',diverging_bky_60_10_c30)
m_diverging_bky_60_10_c30 = mpl_cm('diverging_bky_60_10_c30',diverging_bky_60_10_c30)
m_diverging_bky_60_10_c30_r = mpl_cm('diverging_bky_60_10_c30_r',list(reversed(diverging_bky_60_10_c30)))
bky = b_diverging_bky_60_10_c30
m_bky = m_diverging_bky_60_10_c30
m_bky_r = m_diverging_bky_60_10_c30_r
palette['bky'] = b_diverging_bky_60_10_c30
palette_n['bky'] = b_diverging_bky_60_10_c30
cm['bky'] = m_diverging_bky_60_10_c30
cm['bky_r'] = m_diverging_bky_60_10_c30_r
cm_n['bky'] = mpl_cm('bky',diverging_bky_60_10_c30)
cm_n['bky_r'] = mpl_cm('bky_r',list(reversed(diverging_bky_60_10_c30)))
CET_D6 = b_diverging_bky_60_10_c30
m_CET_D6 = m_diverging_bky_60_10_c30
m_CET_D6_r = m_diverging_bky_60_10_c30_r
palette['CET_D6'] = b_diverging_bky_60_10_c30
cm['CET_D6'] = m_diverging_bky_60_10_c30
cm['CET_D6_r'] = m_diverging_bky_60_10_c30_r
register_cmap('cet_CET_D6',m_diverging_bky_60_10_c30)
register_cmap('cet_CET_D6_r',m_diverging_bky_60_10_c30_r)




diverging_bwr_40_95_c42 = [  # cmap_def
[0.12757, 0.31619, 0.85836],
[0.14675, 0.32048, 0.85923],
[0.16386, 0.32476, 0.8601],
[0.17936, 0.3291, 0.86096],
[0.1937, 0.33343, 0.86182],
[0.20708, 0.33776, 0.86268],
[0.21967, 0.34213, 0.86354],
[0.23162, 0.34649, 0.86439],
[0.24297, 0.35089, 0.86524],
[0.25389, 0.35529, 0.86609],
[0.26436, 0.35969, 0.86693],
[0.27445, 0.36411, 0.86778],
[0.28422, 0.36855, 0.86862],
[0.29369, 0.37299, 0.86945],
[0.30289, 0.37746, 0.87028],
[0.31184, 0.38192, 0.87112],
[0.32061, 0.3864, 0.87195],
[0.32914, 0.39091, 0.87277],
[0.33747, 0.39542, 0.8736],
[0.34563, 0.39992, 0.87442],
[0.35364, 0.40445, 0.87523],
[0.36151, 0.40899, 0.87605],
[0.36923, 0.41355, 0.87686],
[0.37682, 0.4181, 0.87767],
[0.38428, 0.42267, 0.87847],
[0.39164, 0.42727, 0.87927],
[0.39887, 0.43186, 0.88007],
[0.40603, 0.43645, 0.88087],
[0.41307, 0.44106, 0.88167],
[0.42002, 0.44568, 0.88245],
[0.4269, 0.45032, 0.88324],
[0.43367, 0.45496, 0.88403],
[0.44039, 0.45961, 0.88481],
[0.44703, 0.46428, 0.88559],
[0.45359, 0.46894, 0.88636],
[0.46009, 0.47361, 0.88713],
[0.46653, 0.4783, 0.8879],
[0.4729, 0.483, 0.88867],
[0.47922, 0.48771, 0.88943],
[0.48547, 0.49242, 0.89019],
[0.49169, 0.49715, 0.89094],
[0.49784, 0.50188, 0.8917],
[0.50394, 0.50662, 0.89245],
[0.50999, 0.51137, 0.89319],
[0.516, 0.51612, 0.89394],
[0.52196, 0.5209, 0.89468],
[0.52789, 0.52567, 0.89541],
[0.53377, 0.53046, 0.89615],
[0.53962, 0.53525, 0.89688],
[0.54543, 0.54005, 0.8976],
[0.5512, 0.54486, 0.89832],
[0.55693, 0.54968, 0.89905],
[0.56263, 0.5545, 0.89976],
[0.56829, 0.55933, 0.90048],
[0.57392, 0.56418, 0.90119],
[0.57953, 0.56902, 0.90189],
[0.5851, 0.57388, 0.9026],
[0.59064, 0.57874, 0.9033],
[0.59616, 0.58361, 0.90399],
[0.60164, 0.58849, 0.90468],
[0.60711, 0.59339, 0.90538],
[0.61254, 0.59828, 0.90606],
[0.61795, 0.60318, 0.90674],
[0.62334, 0.6081, 0.90742],
[0.6287, 0.613, 0.9081],
[0.63404, 0.61793, 0.90877],
[0.63935, 0.62287, 0.90943],
[0.64464, 0.62781, 0.9101],
[0.64992, 0.63275, 0.91076],
[0.65517, 0.63771, 0.91142],
[0.66041, 0.64267, 0.91207],
[0.66562, 0.64763, 0.91272],
[0.67082, 0.65261, 0.91337],
[0.67599, 0.6576, 0.91401],
[0.68115, 0.66259, 0.91465],
[0.68629, 0.66758, 0.91528],
[0.69141, 0.67259, 0.91591],
[0.69652, 0.6776, 0.91655],
[0.70161, 0.68261, 0.91717],
[0.70669, 0.68763, 0.91779],
[0.71175, 0.69267, 0.91841],
[0.71679, 0.6977, 0.91902],
[0.72181, 0.70275, 0.91963],
[0.72683, 0.70779, 0.92023],
[0.73183, 0.71285, 0.92083],
[0.73682, 0.71792, 0.92143],
[0.74179, 0.72298, 0.92203],
[0.74675, 0.72806, 0.92262],
[0.7517, 0.73315, 0.92321],
[0.75663, 0.73823, 0.92379],
[0.76155, 0.74334, 0.92436],
[0.76647, 0.74844, 0.92494],
[0.77137, 0.75354, 0.92551],
[0.77625, 0.75866, 0.92608],
[0.78113, 0.76378, 0.92664],
[0.78599, 0.76891, 0.9272],
[0.79084, 0.77404, 0.92775],
[0.79569, 0.77918, 0.9283],
[0.80052, 0.78434, 0.92885],
[0.80534, 0.78949, 0.92939],
[0.81015, 0.79464, 0.92993],
[0.81496, 0.79981, 0.93047],
[0.81975, 0.80498, 0.931],
[0.82454, 0.81015, 0.93152],
[0.82932, 0.81534, 0.93204],
[0.83408, 0.82053, 0.93256],
[0.83884, 0.82573, 0.93308],
[0.84359, 0.83093, 0.93358],
[0.84833, 0.83612, 0.93407],
[0.85306, 0.84131, 0.93455],
[0.85779, 0.84648, 0.935],
[0.8625, 0.85164, 0.93542],
[0.86719, 0.85675, 0.9358],
[0.87187, 0.86182, 0.93612],
[0.87652, 0.86682, 0.93638],
[0.88114, 0.87172, 0.93653],
[0.88573, 0.87649, 0.93657],
[0.89027, 0.8811, 0.93646],
[0.89475, 0.8855, 0.93617],
[0.89915, 0.88964, 0.93565],
[0.90346, 0.89347, 0.93488],
[0.90766, 0.89694, 0.9338],
[0.91172, 0.89998, 0.93239],
[0.91562, 0.90254, 0.9306],
[0.91935, 0.90457, 0.9284],
[0.92288, 0.90602, 0.92576],
[0.92619, 0.90686, 0.92266],
[0.92926, 0.90706, 0.91909],
[0.93208, 0.9066, 0.91506],
[0.93464, 0.90549, 0.91056],
[0.93695, 0.90374, 0.90562],
[0.93899, 0.90139, 0.90027],
[0.94078, 0.89845, 0.89453],
[0.94232, 0.89499, 0.88845],
[0.94363, 0.89105, 0.88206],
[0.94474, 0.8867, 0.87541],
[0.94565, 0.88198, 0.86855],
[0.94638, 0.87695, 0.86149],
[0.94697, 0.87168, 0.85429],
[0.94743, 0.86619, 0.84698],
[0.94776, 0.86054, 0.83959],
[0.948, 0.85477, 0.83212],
[0.94815, 0.8489, 0.82461],
[0.94822, 0.84295, 0.81708],
[0.94823, 0.83695, 0.80952],
[0.94818, 0.83092, 0.80196],
[0.94808, 0.82486, 0.79439],
[0.94793, 0.81878, 0.78683],
[0.94774, 0.81269, 0.77926],
[0.9475, 0.8066, 0.77171],
[0.94722, 0.8005, 0.76417],
[0.9469, 0.7944, 0.75665],
[0.94655, 0.78831, 0.74913],
[0.94616, 0.78221, 0.74163],
[0.94574, 0.77612, 0.73414],
[0.94528, 0.77003, 0.72666],
[0.94479, 0.76394, 0.7192],
[0.94425, 0.75785, 0.71174],
[0.94369, 0.75176, 0.7043],
[0.94309, 0.74567, 0.69687],
[0.94245, 0.73959, 0.68946],
[0.94178, 0.73349, 0.68205],
[0.94108, 0.7274, 0.67466],
[0.94034, 0.72131, 0.66728],
[0.93957, 0.71523, 0.65991],
[0.93876, 0.70914, 0.65256],
[0.93793, 0.70305, 0.64521],
[0.93706, 0.69695, 0.63789],
[0.93616, 0.69086, 0.63058],
[0.93522, 0.68477, 0.62327],
[0.93425, 0.67867, 0.61598],
[0.93325, 0.67258, 0.60871],
[0.93222, 0.66648, 0.60143],
[0.93116, 0.66038, 0.59418],
[0.93006, 0.65428, 0.58694],
[0.92893, 0.64818, 0.57973],
[0.92777, 0.64208, 0.57251],
[0.92659, 0.63597, 0.5653],
[0.92537, 0.62986, 0.55812],
[0.92412, 0.62375, 0.55095],
[0.92285, 0.61762, 0.54379],
[0.92154, 0.61151, 0.53665],
[0.9202, 0.60538, 0.52951],
[0.91883, 0.59925, 0.52238],
[0.91743, 0.59312, 0.51527],
[0.916, 0.58697, 0.50819],
[0.91455, 0.58083, 0.5011],
[0.91306, 0.57468, 0.49403],
[0.91155, 0.56853, 0.48698],
[0.91001, 0.56237, 0.47994],
[0.90844, 0.55619, 0.4729],
[0.90684, 0.55002, 0.46589],
[0.90521, 0.54384, 0.45889],
[0.90355, 0.53765, 0.45191],
[0.90187, 0.53145, 0.44493],
[0.90016, 0.52524, 0.43797],
[0.89842, 0.51903, 0.43102],
[0.89665, 0.51279, 0.42408],
[0.89486, 0.50655, 0.41717],
[0.89304, 0.5003, 0.41026],
[0.89119, 0.49404, 0.40336],
[0.88932, 0.48777, 0.39647],
[0.88741, 0.48147, 0.3896],
[0.88549, 0.47518, 0.38274],
[0.88354, 0.46885, 0.3759],
[0.88156, 0.46251, 0.36907],
[0.87955, 0.45616, 0.36225],
[0.87752, 0.4498, 0.35546],
[0.87547, 0.44341, 0.34865],
[0.87338, 0.437, 0.34187],
[0.87128, 0.43057, 0.33511],
[0.86914, 0.42411, 0.32834],
[0.86698, 0.41764, 0.32159],
[0.8648, 0.41114, 0.31485],
[0.8626, 0.40461, 0.30813],
[0.86036, 0.39806, 0.30141],
[0.8581, 0.39148, 0.29471],
[0.85582, 0.38485, 0.28802],
[0.85351, 0.37821, 0.28133],
[0.85118, 0.37151, 0.27466],
[0.84883, 0.36479, 0.26799],
[0.84645, 0.35804, 0.26135],
[0.84405, 0.35123, 0.25471],
[0.84163, 0.34436, 0.24807],
[0.83918, 0.33747, 0.24143],
[0.83671, 0.3305, 0.23481],
[0.83422, 0.32349, 0.22818],
[0.8317, 0.31643, 0.22155],
[0.82916, 0.30929, 0.21492],
[0.82659, 0.30206, 0.20831],
[0.824, 0.29477, 0.20168],
[0.8214, 0.2874, 0.19508],
[0.81876, 0.27996, 0.18847],
[0.81611, 0.27239, 0.1818],
[0.81343, 0.26474, 0.17519],
[0.81074, 0.25697, 0.16854],
[0.80802, 0.24907, 0.16191],
[0.80528, 0.24103, 0.15521],
[0.80252, 0.23283, 0.1485],
[0.79974, 0.22446, 0.1418],
[0.79693, 0.21594, 0.13506],
[0.79411, 0.20718, 0.12827],
[0.79126, 0.19815, 0.12139],
[0.78839, 0.18889, 0.1145],
[0.78551, 0.1793, 0.10759],
[0.7826, 0.16938, 0.10056],
[0.77967, 0.15898, 0.093435],
[0.77672, 0.14808, 0.086208],
[0.77375, 0.13656, 0.078844],
[0.77076, 0.12429, 0.07141],
[0.76775, 0.11107, 0.063657],
[0.76473, 0.096422, 0.055653],
[0.76168, 0.080002, 0.047336],
[0.75862, 0.060659, 0.038618],
[0.75553, 0.03608, 0.029827],
[0.75243, 0.0084181, 0.021805],
]
b_diverging_bwr_40_95_c42 = bokeh_palette('diverging_bwr_40_95_c42',diverging_bwr_40_95_c42)
m_diverging_bwr_40_95_c42 = mpl_cm('diverging_bwr_40_95_c42',diverging_bwr_40_95_c42)
m_diverging_bwr_40_95_c42_r = mpl_cm('diverging_bwr_40_95_c42_r',list(reversed(diverging_bwr_40_95_c42)))
coolwarm = b_diverging_bwr_40_95_c42
m_coolwarm = m_diverging_bwr_40_95_c42
m_coolwarm_r = m_diverging_bwr_40_95_c42_r
palette['coolwarm'] = b_diverging_bwr_40_95_c42
palette_n['coolwarm'] = b_diverging_bwr_40_95_c42
cm['coolwarm'] = m_diverging_bwr_40_95_c42
cm['coolwarm_r'] = m_diverging_bwr_40_95_c42_r
cm_n['coolwarm'] = mpl_cm('coolwarm',diverging_bwr_40_95_c42)
cm_n['coolwarm_r'] = mpl_cm('coolwarm_r',list(reversed(diverging_bwr_40_95_c42)))
CET_D1 = b_diverging_bwr_40_95_c42
m_CET_D1 = m_diverging_bwr_40_95_c42
m_CET_D1_r = m_diverging_bwr_40_95_c42_r
palette['CET_D1'] = b_diverging_bwr_40_95_c42
cm['CET_D1'] = m_diverging_bwr_40_95_c42
cm['CET_D1_r'] = m_diverging_bwr_40_95_c42_r
register_cmap('cet_CET_D1',m_diverging_bwr_40_95_c42)
register_cmap('cet_CET_D1_r',m_diverging_bwr_40_95_c42_r)




cyclic_mrybm_35_75_c68 = [  # cmap_def
[0.97561, 0.52012, 0.97096],
[0.98033, 0.51722, 0.96319],
[0.9842, 0.51357, 0.95448],
[0.98725, 0.50922, 0.94488],
[0.98954, 0.5042, 0.93442],
[0.99112, 0.49856, 0.9232],
[0.99204, 0.49233, 0.91126],
[0.99239, 0.48561, 0.89871],
[0.99221, 0.47844, 0.88563],
[0.99158, 0.47088, 0.87209],
[0.99055, 0.46298, 0.85819],
[0.98919, 0.45483, 0.84399],
[0.98754, 0.44642, 0.82958],
[0.98565, 0.43783, 0.81499],
[0.98354, 0.42907, 0.80029],
[0.98125, 0.42019, 0.78551],
[0.97879, 0.4112, 0.77068],
[0.97618, 0.40212, 0.75583],
[0.97344, 0.39297, 0.74098],
[0.97057, 0.38376, 0.72614],
[0.96757, 0.3745, 0.71131],
[0.96444, 0.36519, 0.6965],
[0.96118, 0.35586, 0.6817],
[0.95779, 0.3465, 0.66694],
[0.95425, 0.33716, 0.65219],
[0.95055, 0.3278, 0.63745],
[0.94668, 0.31849, 0.62271],
[0.94264, 0.30923, 0.60798],
[0.93839, 0.3, 0.59322],
[0.93394, 0.29087, 0.57844],
[0.92929, 0.28185, 0.56366],
[0.92441, 0.27294, 0.54883],
[0.91931, 0.26418, 0.53396],
[0.91398, 0.25557, 0.51908],
[0.90842, 0.24712, 0.50415],
[0.90264, 0.23883, 0.48918],
[0.89665, 0.23073, 0.47419],
[0.89044, 0.22281, 0.45918],
[0.88405, 0.21501, 0.44415],
[0.87747, 0.2074, 0.42911],
[0.87072, 0.19987, 0.41407],
[0.86383, 0.1925, 0.39903],
[0.85679, 0.18523, 0.38401],
[0.84962, 0.17801, 0.369],
[0.84235, 0.17082, 0.35401],
[0.83498, 0.16373, 0.33907],
[0.82753, 0.15664, 0.32416],
[0.82001, 0.14962, 0.30933],
[0.81245, 0.14261, 0.29452],
[0.80486, 0.13567, 0.27982],
[0.79725, 0.12885, 0.26518],
[0.78966, 0.12217, 0.25066],
[0.78212, 0.11578, 0.23628],
[0.77464, 0.1097, 0.22201],
[0.7673, 0.104, 0.20795],
[0.76012, 0.098945, 0.19412],
[0.75317, 0.094721, 0.18049],
[0.74652, 0.091421, 0.16723],
[0.74022, 0.089258, 0.15426],
[0.73434, 0.088445, 0.14174],
[0.72894, 0.089069, 0.12963],
[0.7241, 0.091195, 0.118],
[0.71988, 0.094767, 0.10689],
[0.71631, 0.099636, 0.096236],
[0.71343, 0.1058, 0.086225],
[0.71128, 0.11302, 0.076774],
[0.70983, 0.12104, 0.068001],
[0.7091, 0.12989, 0.05981],
[0.70904, 0.1392, 0.052084],
[0.70963, 0.14902, 0.04517],
[0.71081, 0.15914, 0.038855],
[0.71251, 0.1695, 0.033292],
[0.7147, 0.17995, 0.028947],
[0.71729, 0.1905, 0.02547],
[0.72021, 0.20107, 0.022733],
[0.72342, 0.21166, 0.020622],
[0.72685, 0.22214, 0.019034],
[0.73046, 0.23256, 0.017876],
[0.73419, 0.24288, 0.017071],
[0.73802, 0.25311, 0.016547],
[0.74191, 0.26323, 0.016249],
[0.74585, 0.27322, 0.016125],
[0.74979, 0.28311, 0.016137],
[0.75375, 0.29288, 0.01625],
[0.75769, 0.30257, 0.01644],
[0.76161, 0.31217, 0.016684],
[0.7655, 0.32171, 0.016989],
[0.76932, 0.33117, 0.017296],
[0.7731, 0.34058, 0.0176],
[0.77681, 0.34994, 0.017896],
[0.78044, 0.35925, 0.018177],
[0.784, 0.36855, 0.018435],
[0.78746, 0.37783, 0.018665],
[0.79082, 0.38709, 0.018861],
[0.79409, 0.39636, 0.019014],
[0.79724, 0.40564, 0.019122],
[0.80029, 0.4149, 0.019179],
[0.80322, 0.42418, 0.019183],
[0.80605, 0.43346, 0.019132],
[0.80876, 0.44276, 0.019026],
[0.81136, 0.45205, 0.018866],
[0.81386, 0.46134, 0.018654],
[0.81626, 0.47063, 0.018394],
[0.81857, 0.47992, 0.01809],
[0.8208, 0.48918, 0.017747],
[0.82295, 0.49846, 0.017369],
[0.82503, 0.50771, 0.016963],
[0.82704, 0.51694, 0.016575],
[0.82898, 0.52615, 0.016225],
[0.83084, 0.53532, 0.015949],
[0.83262, 0.54446, 0.015792],
[0.83431, 0.55354, 0.01582],
[0.8359, 0.56257, 0.016118],
[0.83738, 0.5715, 0.016798],
[0.8387, 0.58035, 0.018004],
[0.83985, 0.58906, 0.019918],
[0.84077, 0.59761, 0.022763],
[0.84143, 0.60595, 0.02681],
[0.84177, 0.61404, 0.032376],
[0.84173, 0.62182, 0.040011],
[0.84125, 0.62922, 0.04912],
[0.84026, 0.63619, 0.059305],
[0.83868, 0.64264, 0.070471],
[0.83647, 0.6485, 0.08264],
[0.83356, 0.65372, 0.095596],
[0.82989, 0.65824, 0.10926],
[0.82544, 0.66201, 0.12343],
[0.82017, 0.66498, 0.13819],
[0.81408, 0.66715, 0.15321],
[0.80717, 0.66852, 0.16856],
[0.79947, 0.66908, 0.18405],
[0.79098, 0.66887, 0.19961],
[0.78178, 0.66795, 0.2152],
[0.7719, 0.66635, 0.23072],
[0.76139, 0.66416, 0.2461],
[0.75032, 0.66143, 0.26133],
[0.73873, 0.65825, 0.27636],
[0.72668, 0.65467, 0.29113],
[0.71421, 0.65077, 0.30567],
[0.70137, 0.64662, 0.31995],
[0.68818, 0.64226, 0.33397],
[0.67466, 0.63773, 0.34774],
[0.66084, 0.6331, 0.36127],
[0.64671, 0.62838, 0.37458],
[0.63228, 0.6236, 0.38765],
[0.61753, 0.61878, 0.40056],
[0.60245, 0.61393, 0.41326],
[0.58703, 0.60909, 0.42578],
[0.57126, 0.60423, 0.43817],
[0.55511, 0.59936, 0.4504],
[0.53858, 0.59449, 0.4625],
[0.5216, 0.58962, 0.47451],
[0.5042, 0.58475, 0.4864],
[0.48632, 0.57986, 0.49821],
[0.46796, 0.57493, 0.50993],
[0.44912, 0.56996, 0.5216],
[0.42978, 0.56493, 0.53322],
[0.40998, 0.55983, 0.5448],
[0.38972, 0.55461, 0.55635],
[0.36908, 0.54926, 0.56789],
[0.3481, 0.54375, 0.57943],
[0.32688, 0.53805, 0.59098],
[0.30556, 0.53211, 0.60256],
[0.28424, 0.52592, 0.61416],
[0.26316, 0.51945, 0.62583],
[0.2425, 0.51265, 0.63755],
[0.22257, 0.50551, 0.64934],
[0.20368, 0.49802, 0.66119],
[0.18619, 0.49012, 0.67313],
[0.17051, 0.48183, 0.68514],
[0.15712, 0.47312, 0.69723],
[0.14638, 0.46402, 0.70941],
[0.13859, 0.4545, 0.72164],
[0.1338, 0.44458, 0.73395],
[0.1319, 0.43426, 0.7463],
[0.13253, 0.42359, 0.75869],
[0.13519, 0.4126, 0.77109],
[0.1393, 0.40128, 0.7835],
[0.1445, 0.38969, 0.79588],
[0.15024, 0.3779, 0.80821],
[0.1563, 0.36594, 0.82045],
[0.16246, 0.35387, 0.83257],
[0.16864, 0.34179, 0.84452],
[0.17484, 0.32979, 0.85623],
[0.18105, 0.31794, 0.86767],
[0.18746, 0.30639, 0.87875],
[0.19408, 0.29524, 0.88943],
[0.20101, 0.28465, 0.89962],
[0.20848, 0.27478, 0.90926],
[0.2165, 0.26579, 0.9183],
[0.22506, 0.25778, 0.92668],
[0.2343, 0.25098, 0.93436],
[0.24409, 0.24544, 0.94131],
[0.25448, 0.24134, 0.94753],
[0.26529, 0.23866, 0.95301],
[0.27652, 0.2375, 0.95778],
[0.28798, 0.23773, 0.96185],
[0.29964, 0.23933, 0.9653],
[0.31135, 0.24218, 0.96816],
[0.32305, 0.24613, 0.9705],
[0.33464, 0.25106, 0.97238],
[0.34606, 0.25676, 0.97386],
[0.35733, 0.26313, 0.97502],
[0.36835, 0.27, 0.9759],
[0.37916, 0.27724, 0.97657],
[0.38973, 0.28476, 0.97706],
[0.40009, 0.2925, 0.97742],
[0.41026, 0.30036, 0.97768],
[0.42023, 0.3083, 0.97787],
[0.43006, 0.31627, 0.978],
[0.43977, 0.32421, 0.97811],
[0.4494, 0.33213, 0.9782],
[0.459, 0.33997, 0.97829],
[0.46859, 0.34772, 0.97838],
[0.47825, 0.35536, 0.97849],
[0.48802, 0.36282, 0.97864],
[0.49795, 0.37015, 0.97883],
[0.50807, 0.3773, 0.97906],
[0.51845, 0.38426, 0.97935],
[0.52912, 0.39102, 0.97969],
[0.5401, 0.39755, 0.9801],
[0.55143, 0.40386, 0.98057],
[0.56311, 0.40995, 0.98111],
[0.57513, 0.41579, 0.98173],
[0.58751, 0.4214, 0.98242],
[0.60022, 0.42681, 0.98317],
[0.61322, 0.43201, 0.984],
[0.6265, 0.437, 0.98489],
[0.64002, 0.44182, 0.98583],
[0.65371, 0.44647, 0.98682],
[0.66757, 0.45099, 0.98786],
[0.68153, 0.4554, 0.98894],
[0.69558, 0.4597, 0.99005],
[0.70967, 0.46392, 0.99118],
[0.72376, 0.46807, 0.99233],
[0.73785, 0.47218, 0.9935],
[0.7519, 0.47624, 0.99467],
[0.76589, 0.48026, 0.99582],
[0.77979, 0.48425, 0.99696],
[0.79361, 0.48821, 0.99807],
[0.80729, 0.49211, 0.99913],
[0.82083, 0.49595, 1],
[0.8342, 0.49972, 1],
[0.84736, 0.50339, 1],
[0.86029, 0.50693, 1],
[0.87292, 0.5103, 1],
[0.8852, 0.51345, 1],
[0.89709, 0.51632, 1],
[0.90851, 0.51889, 1],
[0.91939, 0.52104, 1],
[0.92968, 0.52273, 0.99852],
[0.9393, 0.52391, 0.996],
[0.94819, 0.5245, 0.99274],
[0.95631, 0.52446, 0.98864],
[0.9636, 0.52374, 0.98368],
[0.97003, 0.52229, 0.97779],
]
b_cyclic_mrybm_35_75_c68 = bokeh_palette('cyclic_mrybm_35_75_c68',cyclic_mrybm_35_75_c68)
m_cyclic_mrybm_35_75_c68 = mpl_cm('cyclic_mrybm_35_75_c68',cyclic_mrybm_35_75_c68)
m_cyclic_mrybm_35_75_c68_r = mpl_cm('cyclic_mrybm_35_75_c68_r',list(reversed(cyclic_mrybm_35_75_c68)))
CET_C1 = b_cyclic_mrybm_35_75_c68
m_CET_C1 = m_cyclic_mrybm_35_75_c68
m_CET_C1_r = m_cyclic_mrybm_35_75_c68_r
palette['CET_C1'] = b_cyclic_mrybm_35_75_c68
cm['CET_C1'] = m_cyclic_mrybm_35_75_c68
cm['CET_C1_r'] = m_cyclic_mrybm_35_75_c68_r
register_cmap('cet_CET_C1',m_cyclic_mrybm_35_75_c68)
register_cmap('cet_CET_C1_r',m_cyclic_mrybm_35_75_c68_r)




diverging_gkr_60_10_c40 = [  # cmap_def
[0.2109, 0.65114, 0.08795],
[0.21113, 0.64645, 0.089674],
[0.21134, 0.64175, 0.091371],
[0.21153, 0.63707, 0.092907],
[0.21169, 0.63239, 0.09454],
[0.21184, 0.62772, 0.095984],
[0.21196, 0.62306, 0.097463],
[0.21206, 0.6184, 0.098858],
[0.21214, 0.61374, 0.1002],
[0.2122, 0.6091, 0.10151],
[0.21223, 0.60446, 0.1028],
[0.21225, 0.59982, 0.10401],
[0.21225, 0.59519, 0.10517],
[0.21224, 0.59056, 0.10638],
[0.2122, 0.58596, 0.10746],
[0.21215, 0.58134, 0.10849],
[0.21207, 0.57674, 0.1096],
[0.21198, 0.57214, 0.11058],
[0.21187, 0.56755, 0.11155],
[0.21174, 0.56297, 0.11249],
[0.21159, 0.55838, 0.1134],
[0.21142, 0.55382, 0.11421],
[0.21123, 0.54926, 0.11506],
[0.21103, 0.5447, 0.11592],
[0.21081, 0.54014, 0.11669],
[0.21057, 0.5356, 0.11742],
[0.21032, 0.53106, 0.1182],
[0.21005, 0.52654, 0.1189],
[0.20977, 0.522, 0.11955],
[0.20946, 0.5175, 0.1202],
[0.20915, 0.51298, 0.12082],
[0.20881, 0.50848, 0.12142],
[0.20846, 0.50398, 0.122],
[0.2081, 0.49948, 0.12257],
[0.20772, 0.49501, 0.12311],
[0.20732, 0.49053, 0.12363],
[0.20692, 0.48605, 0.12413],
[0.20649, 0.48159, 0.12461],
[0.20604, 0.47713, 0.12507],
[0.20556, 0.47268, 0.12553],
[0.20508, 0.46824, 0.12595],
[0.2046, 0.46381, 0.12633],
[0.2041, 0.45938, 0.12668],
[0.20357, 0.45497, 0.12703],
[0.20302, 0.45055, 0.12737],
[0.20246, 0.44614, 0.12771],
[0.20189, 0.44175, 0.12802],
[0.2013, 0.43736, 0.1283],
[0.2007, 0.43297, 0.12855],
[0.20009, 0.42859, 0.12879],
[0.19947, 0.42421, 0.12902],
[0.19883, 0.41986, 0.12924],
[0.19818, 0.4155, 0.12945],
[0.19752, 0.41115, 0.12965],
[0.19686, 0.40681, 0.12982],
[0.19616, 0.40247, 0.12998],
[0.19543, 0.39816, 0.13012],
[0.19472, 0.39383, 0.13024],
[0.19399, 0.38952, 0.13034],
[0.19321, 0.38522, 0.13043],
[0.19244, 0.38094, 0.13051],
[0.19166, 0.37665, 0.13056],
[0.19086, 0.37236, 0.13061],
[0.19005, 0.36809, 0.13064],
[0.18924, 0.36383, 0.13066],
[0.18842, 0.35958, 0.13066],
[0.18757, 0.35534, 0.13065],
[0.18668, 0.3511, 0.13063],
[0.18583, 0.34687, 0.13059],
[0.18493, 0.34265, 0.13054],
[0.184, 0.33842, 0.13048],
[0.18308, 0.33422, 0.13041],
[0.18214, 0.33002, 0.13032],
[0.1812, 0.32581, 0.13022],
[0.18025, 0.32164, 0.1301],
[0.17928, 0.31747, 0.12997],
[0.17832, 0.31331, 0.12982],
[0.17731, 0.30916, 0.12966],
[0.17626, 0.305, 0.12949],
[0.17526, 0.30085, 0.1293],
[0.17423, 0.29673, 0.1291],
[0.17316, 0.29259, 0.1289],
[0.1721, 0.28848, 0.1287],
[0.17102, 0.28437, 0.12849],
[0.16995, 0.28027, 0.12827],
[0.16885, 0.27621, 0.12802],
[0.16771, 0.27209, 0.12776],
[0.16659, 0.26803, 0.12748],
[0.16541, 0.26399, 0.12719],
[0.1643, 0.25993, 0.1269],
[0.16312, 0.25587, 0.12661],
[0.16194, 0.25185, 0.12632],
[0.16072, 0.24783, 0.12601],
[0.1595, 0.24378, 0.12567],
[0.1583, 0.23979, 0.12532],
[0.15704, 0.2358, 0.12495],
[0.15581, 0.23181, 0.12458],
[0.15452, 0.22783, 0.1242],
[0.15324, 0.22384, 0.12382],
[0.15195, 0.21989, 0.12342],
[0.15065, 0.21594, 0.12301],
[0.14935, 0.212, 0.1226],
[0.148, 0.20805, 0.12217],
[0.14664, 0.20414, 0.12174],
[0.14533, 0.20018, 0.1213],
[0.14393, 0.19632, 0.12085],
[0.14254, 0.1924, 0.1204],
[0.14116, 0.18853, 0.11993],
[0.1397, 0.18465, 0.11946],
[0.13835, 0.18077, 0.11899],
[0.13693, 0.17695, 0.11852],
[0.13553, 0.17313, 0.118],
[0.1342, 0.16938, 0.11747],
[0.13286, 0.16558, 0.11697],
[0.13158, 0.16193, 0.1165],
[0.13041, 0.15828, 0.116],
[0.12925, 0.1547, 0.11549],
[0.12828, 0.15123, 0.11499],
[0.12744, 0.14789, 0.11453],
[0.1268, 0.14468, 0.11413],
[0.12643, 0.14163, 0.11378],
[0.12631, 0.13876, 0.11346],
[0.12649, 0.13609, 0.11319],
[0.12701, 0.13374, 0.11297],
[0.12798, 0.13162, 0.11282],
[0.12929, 0.12981, 0.11275],
[0.13109, 0.12828, 0.11276],
[0.1333, 0.12705, 0.11287],
[0.13593, 0.12624, 0.11307],
[0.13905, 0.12571, 0.11335],
[0.14261, 0.1255, 0.11371],
[0.14653, 0.12561, 0.11414],
[0.15086, 0.12602, 0.11466],
[0.15553, 0.12666, 0.11531],
[0.16047, 0.12758, 0.11603],
[0.1657, 0.12871, 0.11675],
[0.17119, 0.13008, 0.11752],
[0.17685, 0.13154, 0.11839],
[0.18266, 0.13315, 0.11923],
[0.18863, 0.1349, 0.1201],
[0.1947, 0.13667, 0.121],
[0.20079, 0.13857, 0.12192],
[0.20703, 0.1405, 0.12286],
[0.21324, 0.14246, 0.12382],
[0.21954, 0.14446, 0.12478],
[0.22586, 0.14644, 0.12577],
[0.23219, 0.14845, 0.12669],
[0.23851, 0.1505, 0.12766],
[0.24487, 0.1525, 0.12861],
[0.25127, 0.15454, 0.12959],
[0.25763, 0.15658, 0.13056],
[0.26403, 0.1586, 0.13149],
[0.27043, 0.16058, 0.13244],
[0.27682, 0.16261, 0.13339],
[0.28323, 0.1646, 0.13435],
[0.28965, 0.1666, 0.13528],
[0.29607, 0.16858, 0.13618],
[0.3025, 0.17056, 0.13717],
[0.30898, 0.17253, 0.13812],
[0.31543, 0.17453, 0.13902],
[0.3219, 0.17646, 0.13994],
[0.32839, 0.17847, 0.14093],
[0.33489, 0.18039, 0.14184],
[0.34139, 0.18234, 0.14277],
[0.34791, 0.1843, 0.1437],
[0.35444, 0.18626, 0.14463],
[0.36096, 0.18821, 0.14557],
[0.36752, 0.19011, 0.14645],
[0.3741, 0.19204, 0.14739],
[0.38068, 0.19399, 0.14829],
[0.38725, 0.19589, 0.14922],
[0.39386, 0.1978, 0.15015],
[0.40049, 0.19969, 0.15104],
[0.40711, 0.2016, 0.15194],
[0.41375, 0.20353, 0.15285],
[0.4204, 0.20541, 0.15376],
[0.42707, 0.20731, 0.15468],
[0.43374, 0.20919, 0.1556],
[0.44044, 0.21108, 0.15651],
[0.44714, 0.21294, 0.15738],
[0.45386, 0.21481, 0.15831],
[0.46058, 0.21671, 0.15919],
[0.46734, 0.21856, 0.16006],
[0.47409, 0.22044, 0.16099],
[0.48085, 0.22227, 0.1619],
[0.48764, 0.22411, 0.16278],
[0.49444, 0.22599, 0.16368],
[0.50124, 0.22782, 0.16455],
[0.50806, 0.22963, 0.1654],
[0.51489, 0.2315, 0.16631],
[0.52173, 0.2333, 0.16721],
[0.5286, 0.23516, 0.16807],
[0.53546, 0.23698, 0.16898],
[0.54234, 0.23877, 0.16985],
[0.54924, 0.24058, 0.17071],
[0.55614, 0.2424, 0.17158],
[0.56307, 0.2442, 0.17245],
[0.56999, 0.246, 0.17332],
[0.57695, 0.24782, 0.17421],
[0.5839, 0.24959, 0.17508],
[0.59087, 0.25141, 0.17591],
[0.59785, 0.25318, 0.17679],
[0.60485, 0.25498, 0.17769],
[0.61185, 0.25674, 0.17854],
[0.61888, 0.25851, 0.17939],
[0.6259, 0.26028, 0.18024],
[0.63295, 0.26205, 0.18109],
[0.64001, 0.26383, 0.18194],
[0.64708, 0.26557, 0.1828],
[0.65415, 0.26731, 0.18366],
[0.66125, 0.26909, 0.18452],
[0.66835, 0.27084, 0.18539],
[0.67547, 0.27256, 0.18622],
[0.68259, 0.27431, 0.18707],
[0.68974, 0.27606, 0.18793],
[0.69689, 0.27778, 0.18876],
[0.70405, 0.27952, 0.18959],
[0.71123, 0.28123, 0.19042],
[0.71841, 0.28297, 0.19126],
[0.72562, 0.28467, 0.1921],
[0.73282, 0.28638, 0.19294],
[0.74005, 0.2881, 0.19379],
[0.74728, 0.28981, 0.19463],
[0.75453, 0.29152, 0.19544],
[0.76178, 0.29321, 0.19629],
[0.76904, 0.2949, 0.19711],
[0.77633, 0.29661, 0.19792],
[0.78362, 0.29829, 0.19874],
[0.79092, 0.29998, 0.19955],
[0.79823, 0.30166, 0.20037],
[0.80555, 0.30334, 0.2012],
[0.81289, 0.30503, 0.20202],
[0.82024, 0.3067, 0.20285],
[0.82759, 0.30837, 0.20368],
[0.83497, 0.31005, 0.2045],
[0.84235, 0.31169, 0.20529],
[0.84974, 0.31336, 0.20613],
[0.85714, 0.31501, 0.20694],
[0.86455, 0.31667, 0.20773],
[0.87198, 0.31832, 0.20854],
[0.87941, 0.31997, 0.20934],
[0.88686, 0.3216, 0.21015],
[0.89431, 0.32325, 0.21095],
[0.90178, 0.32488, 0.21177],
[0.90925, 0.32651, 0.21256],
[0.91675, 0.32815, 0.21334],
[0.92424, 0.32979, 0.21416],
[0.93175, 0.33139, 0.21494],
[0.93927, 0.33303, 0.21575],
[0.9468, 0.33465, 0.21655],
[0.95434, 0.33625, 0.21733],
[0.96189, 0.33786, 0.21812],
[0.96945, 0.33948, 0.21891],
[0.97703, 0.34108, 0.2197],
[0.98461, 0.34269, 0.2205],
[0.9922, 0.34427, 0.22128],
]
b_diverging_gkr_60_10_c40 = bokeh_palette('diverging_gkr_60_10_c40',diverging_gkr_60_10_c40)
m_diverging_gkr_60_10_c40 = mpl_cm('diverging_gkr_60_10_c40',diverging_gkr_60_10_c40)
m_diverging_gkr_60_10_c40_r = mpl_cm('diverging_gkr_60_10_c40_r',list(reversed(diverging_gkr_60_10_c40)))




diverging_gwv_55_95_c39 = [  # cmap_def
[0.22157, 0.59107, 0.055073],
[0.22965, 0.59389, 0.068993],
[0.2376, 0.59671, 0.081291],
[0.24531, 0.59952, 0.092565],
[0.25292, 0.60233, 0.10315],
[0.26037, 0.60514, 0.11313],
[0.26768, 0.60796, 0.12253],
[0.27491, 0.61076, 0.13168],
[0.28202, 0.61356, 0.14045],
[0.28902, 0.61637, 0.14894],
[0.29592, 0.61918, 0.15721],
[0.30273, 0.62197, 0.16528],
[0.3095, 0.62478, 0.1732],
[0.31616, 0.62758, 0.18095],
[0.32276, 0.63038, 0.18859],
[0.32929, 0.63317, 0.1961],
[0.33575, 0.63597, 0.20349],
[0.34216, 0.63876, 0.21079],
[0.34851, 0.64155, 0.21802],
[0.35482, 0.64434, 0.22514],
[0.36105, 0.64714, 0.23224],
[0.36726, 0.64993, 0.23924],
[0.37343, 0.65271, 0.24619],
[0.37954, 0.6555, 0.25311],
[0.38562, 0.65828, 0.25997],
[0.39167, 0.66106, 0.26678],
[0.39767, 0.66385, 0.27357],
[0.40364, 0.66662, 0.28029],
[0.40958, 0.66941, 0.287],
[0.41548, 0.67219, 0.29368],
[0.42135, 0.67496, 0.30034],
[0.42721, 0.67774, 0.30697],
[0.43302, 0.68051, 0.31358],
[0.43881, 0.68328, 0.32017],
[0.44459, 0.68606, 0.32672],
[0.45032, 0.68883, 0.33328],
[0.45604, 0.6916, 0.33982],
[0.46174, 0.69436, 0.34632],
[0.46742, 0.69713, 0.35283],
[0.47307, 0.6999, 0.35932],
[0.47871, 0.70267, 0.3658],
[0.48432, 0.70543, 0.37227],
[0.48992, 0.70819, 0.37873],
[0.49551, 0.71096, 0.38518],
[0.50108, 0.71371, 0.39164],
[0.50662, 0.71648, 0.39808],
[0.51215, 0.71923, 0.40451],
[0.51768, 0.72199, 0.41095],
[0.52318, 0.72475, 0.41738],
[0.52867, 0.7275, 0.42379],
[0.53414, 0.73026, 0.43022],
[0.5396, 0.73301, 0.43664],
[0.54506, 0.73576, 0.44305],
[0.5505, 0.73851, 0.44946],
[0.55592, 0.74126, 0.45587],
[0.56134, 0.74401, 0.46228],
[0.56675, 0.74676, 0.4687],
[0.57214, 0.7495, 0.47512],
[0.57752, 0.75225, 0.48152],
[0.58289, 0.755, 0.48794],
[0.58826, 0.75775, 0.49435],
[0.59362, 0.76049, 0.50076],
[0.59896, 0.76323, 0.50717],
[0.60431, 0.76597, 0.5136],
[0.60963, 0.76871, 0.52001],
[0.61495, 0.77146, 0.52644],
[0.62027, 0.77419, 0.53286],
[0.62557, 0.77693, 0.53928],
[0.63088, 0.77967, 0.54572],
[0.63617, 0.7824, 0.55215],
[0.64145, 0.78514, 0.55858],
[0.64673, 0.78787, 0.56502],
[0.652, 0.79061, 0.57147],
[0.65727, 0.79334, 0.57791],
[0.66253, 0.79607, 0.58437],
[0.66778, 0.7988, 0.59082],
[0.67303, 0.80153, 0.59729],
[0.67827, 0.80426, 0.60376],
[0.68351, 0.80699, 0.61022],
[0.68875, 0.80972, 0.6167],
[0.69397, 0.81244, 0.62318],
[0.69919, 0.81517, 0.62965],
[0.70441, 0.8179, 0.63615],
[0.70963, 0.82062, 0.64264],
[0.71483, 0.82335, 0.64914],
[0.72004, 0.82607, 0.65564],
[0.72525, 0.82879, 0.66215],
[0.73044, 0.83152, 0.66866],
[0.73563, 0.83424, 0.67518],
[0.74082, 0.83696, 0.6817],
[0.74601, 0.83968, 0.68824],
[0.7512, 0.84239, 0.69477],
[0.75637, 0.84511, 0.70131],
[0.76155, 0.84782, 0.70786],
[0.76672, 0.85054, 0.71441],
[0.77189, 0.85326, 0.72097],
[0.77706, 0.85597, 0.72754],
[0.78223, 0.85869, 0.73411],
[0.78739, 0.8614, 0.74068],
[0.79255, 0.86412, 0.74727],
[0.7977, 0.86682, 0.75386],
[0.80286, 0.86954, 0.76045],
[0.80801, 0.87225, 0.76705],
[0.81316, 0.87496, 0.77366],
[0.81831, 0.87767, 0.78027],
[0.82345, 0.88037, 0.7869],
[0.8286, 0.88308, 0.79353],
[0.83375, 0.88579, 0.80016],
[0.83888, 0.88849, 0.8068],
[0.84401, 0.89118, 0.81343],
[0.84914, 0.89385, 0.82007],
[0.85424, 0.89651, 0.82671],
[0.85933, 0.89914, 0.83333],
[0.8644, 0.90172, 0.83995],
[0.86942, 0.90426, 0.84652],
[0.8744, 0.90673, 0.85306],
[0.87932, 0.90912, 0.85955],
[0.88415, 0.91139, 0.86596],
[0.88888, 0.91352, 0.87228],
[0.89348, 0.91548, 0.87846],
[0.89792, 0.91724, 0.8845],
[0.90218, 0.91875, 0.89034],
[0.90621, 0.92, 0.89596],
[0.90998, 0.92093, 0.90131],
[0.91347, 0.92152, 0.90637],
[0.91664, 0.92174, 0.91109],
[0.91948, 0.92157, 0.91546],
[0.92195, 0.92098, 0.91944],
[0.92406, 0.91999, 0.92303],
[0.92579, 0.91859, 0.92621],
[0.92715, 0.91679, 0.92898],
[0.92815, 0.9146, 0.93137],
[0.92881, 0.91206, 0.93337],
[0.92915, 0.90919, 0.93503],
[0.9292, 0.90603, 0.93636],
[0.92898, 0.9026, 0.9374],
[0.92854, 0.89895, 0.93818],
[0.9279, 0.89512, 0.93875],
[0.9271, 0.89113, 0.93913],
[0.92616, 0.88701, 0.93935],
[0.92511, 0.8828, 0.93945],
[0.92397, 0.87851, 0.93946],
[0.92276, 0.87416, 0.93938],
[0.92149, 0.86976, 0.93925],
[0.92018, 0.86533, 0.93907],
[0.91883, 0.86089, 0.93886],
[0.91747, 0.85642, 0.93862],
[0.91607, 0.85195, 0.93837],
[0.91467, 0.84746, 0.93811],
[0.91325, 0.84298, 0.93783],
[0.91182, 0.83849, 0.93754],
[0.91039, 0.834, 0.93726],
[0.90894, 0.82952, 0.93697],
[0.9075, 0.82502, 0.93668],
[0.90604, 0.82053, 0.93639],
[0.90457, 0.81605, 0.93609],
[0.9031, 0.81156, 0.93579],
[0.90162, 0.80708, 0.93549],
[0.90014, 0.80258, 0.93519],
[0.89864, 0.7981, 0.93488],
[0.89715, 0.79361, 0.93457],
[0.89564, 0.78912, 0.93426],
[0.89412, 0.78464, 0.93395],
[0.8926, 0.78014, 0.93363],
[0.89107, 0.77565, 0.93332],
[0.88953, 0.77117, 0.93299],
[0.88799, 0.76667, 0.93267],
[0.88644, 0.76218, 0.93235],
[0.88488, 0.75769, 0.93202],
[0.88332, 0.75319, 0.93169],
[0.88174, 0.7487, 0.93136],
[0.88016, 0.74421, 0.93103],
[0.87857, 0.73972, 0.93069],
[0.87698, 0.73522, 0.93035],
[0.87538, 0.73072, 0.93001],
[0.87377, 0.72622, 0.92966],
[0.87215, 0.72172, 0.92931],
[0.87052, 0.71723, 0.92896],
[0.8689, 0.71271, 0.92861],
[0.86725, 0.70822, 0.92826],
[0.86561, 0.70371, 0.9279],
[0.86396, 0.6992, 0.92754],
[0.8623, 0.69469, 0.92718],
[0.86063, 0.69018, 0.92682],
[0.85895, 0.68568, 0.92645],
[0.85727, 0.68116, 0.92609],
[0.85558, 0.67665, 0.92572],
[0.85388, 0.67213, 0.92534],
[0.85218, 0.66761, 0.92497],
[0.85046, 0.66309, 0.92459],
[0.84874, 0.65856, 0.92421],
[0.84701, 0.65404, 0.92383],
[0.84528, 0.64951, 0.92345],
[0.84353, 0.64497, 0.92306],
[0.84178, 0.64044, 0.92267],
[0.84003, 0.6359, 0.92228],
[0.83826, 0.63136, 0.92189],
[0.83649, 0.62682, 0.92149],
[0.83471, 0.62227, 0.92109],
[0.83291, 0.61772, 0.92069],
[0.83112, 0.61316, 0.92029],
[0.82932, 0.60861, 0.91989],
[0.8275, 0.60405, 0.91948],
[0.82568, 0.59948, 0.91907],
[0.82386, 0.59491, 0.91866],
[0.82203, 0.59033, 0.91825],
[0.82018, 0.58576, 0.91784],
[0.81833, 0.58117, 0.91743],
[0.81648, 0.57659, 0.91701],
[0.81461, 0.57199, 0.91658],
[0.81273, 0.56739, 0.91615],
[0.81085, 0.56279, 0.91573],
[0.80897, 0.55817, 0.9153],
[0.80707, 0.55356, 0.91487],
[0.80516, 0.54895, 0.91444],
[0.80325, 0.54431, 0.91401],
[0.80133, 0.53967, 0.91357],
[0.7994, 0.53504, 0.91313],
[0.79746, 0.53039, 0.91269],
[0.79552, 0.52573, 0.91225],
[0.79357, 0.52107, 0.9118],
[0.7916, 0.51639, 0.91136],
[0.78964, 0.51172, 0.91091],
[0.78766, 0.50703, 0.91046],
[0.78568, 0.50233, 0.91001],
[0.78368, 0.49764, 0.90955],
[0.78168, 0.49291, 0.90909],
[0.77967, 0.48819, 0.90863],
[0.77766, 0.48345, 0.90817],
[0.77563, 0.47871, 0.90771],
[0.77359, 0.47395, 0.90724],
[0.77155, 0.46919, 0.90678],
[0.7695, 0.46441, 0.90631],
[0.76744, 0.45961, 0.90584],
[0.76538, 0.45481, 0.90536],
[0.76329, 0.44999, 0.90489],
[0.76121, 0.44516, 0.90441],
[0.75912, 0.44031, 0.90393],
[0.75702, 0.43546, 0.90345],
[0.75491, 0.43057, 0.90297],
[0.75278, 0.42568, 0.90248],
[0.75066, 0.42076, 0.90199],
[0.74852, 0.41584, 0.9015],
[0.74638, 0.4109, 0.90101],
[0.74423, 0.40594, 0.90052],
[0.74206, 0.40096, 0.90002],
[0.7399, 0.39595, 0.89953],
[0.73771, 0.39092, 0.89903],
[0.73552, 0.38586, 0.89853],
[0.73333, 0.3808, 0.89802],
[0.73112, 0.37569, 0.89752],
[0.7289, 0.37056, 0.89702],
[0.72668, 0.36542, 0.89651],
[0.72444, 0.36023, 0.896],
[0.72219, 0.35504, 0.89548],
[0.71995, 0.34979, 0.89497],
]
b_diverging_gwv_55_95_c39 = bokeh_palette('diverging_gwv_55_95_c39',diverging_gwv_55_95_c39)
m_diverging_gwv_55_95_c39 = mpl_cm('diverging_gwv_55_95_c39',diverging_gwv_55_95_c39)
m_diverging_gwv_55_95_c39_r = mpl_cm('diverging_gwv_55_95_c39_r',list(reversed(diverging_gwv_55_95_c39)))
gwv = b_diverging_gwv_55_95_c39
m_gwv = m_diverging_gwv_55_95_c39
m_gwv_r = m_diverging_gwv_55_95_c39_r
palette['gwv'] = b_diverging_gwv_55_95_c39
palette_n['gwv'] = b_diverging_gwv_55_95_c39
cm['gwv'] = m_diverging_gwv_55_95_c39
cm['gwv_r'] = m_diverging_gwv_55_95_c39_r
cm_n['gwv'] = mpl_cm('gwv',diverging_gwv_55_95_c39)
cm_n['gwv_r'] = mpl_cm('gwv_r',list(reversed(diverging_gwv_55_95_c39)))
CET_D2 = b_diverging_gwv_55_95_c39
m_CET_D2 = m_diverging_gwv_55_95_c39
m_CET_D2_r = m_diverging_gwv_55_95_c39_r
palette['CET_D2'] = b_diverging_gwv_55_95_c39
cm['CET_D2'] = m_diverging_gwv_55_95_c39
cm['CET_D2_r'] = m_diverging_gwv_55_95_c39_r
register_cmap('cet_CET_D2',m_diverging_gwv_55_95_c39)
register_cmap('cet_CET_D2_r',m_diverging_gwv_55_95_c39_r)




rainbow_bgyrm_35_85_c71 = [  # cmap_def
[0, 0.20755, 0.97632],
[0, 0.22113, 0.96201],
[0, 0.23397, 0.94773],
[0, 0.24623, 0.93346],
[0, 0.25798, 0.91922],
[0, 0.26932, 0.905],
[0, 0.28022, 0.89081],
[0, 0.29079, 0.87663],
[0, 0.30104, 0.8625],
[0, 0.31103, 0.84838],
[0, 0.32074, 0.83429],
[0, 0.3302, 0.82023],
[0, 0.33945, 0.80621],
[0, 0.34846, 0.79222],
[0, 0.35727, 0.77828],
[0, 0.36586, 0.76439],
[0, 0.37426, 0.75056],
[0, 0.38245, 0.7368],
[0, 0.39044, 0.72311],
[0, 0.39823, 0.70952],
[0, 0.4058, 0.69603],
[0, 0.41317, 0.68264],
[0.0064797, 0.42031, 0.66938],
[0.016533, 0.42727, 0.65626],
[0.028131, 0.43398, 0.64326],
[0.041954, 0.44052, 0.63041],
[0.055821, 0.44686, 0.6177],
[0.069351, 0.45301, 0.60514],
[0.082392, 0.45898, 0.59273],
[0.095132, 0.46479, 0.58043],
[0.10747, 0.47045, 0.56826],
[0.11942, 0.47598, 0.55619],
[0.13095, 0.48139, 0.54423],
[0.14195, 0.48671, 0.53234],
[0.15241, 0.49195, 0.52051],
[0.16235, 0.49712, 0.50873],
[0.17166, 0.50225, 0.49697],
[0.18039, 0.50733, 0.48522],
[0.18854, 0.51239, 0.47347],
[0.19606, 0.51744, 0.46171],
[0.20297, 0.52246, 0.4499],
[0.20931, 0.5275, 0.43806],
[0.21508, 0.53254, 0.42613],
[0.22033, 0.53758, 0.41415],
[0.225, 0.54263, 0.40206],
[0.22922, 0.54768, 0.38989],
[0.23295, 0.55275, 0.37763],
[0.23626, 0.55783, 0.36524],
[0.23913, 0.56292, 0.35274],
[0.24164, 0.56801, 0.3401],
[0.24376, 0.57311, 0.32732],
[0.24561, 0.57819, 0.31441],
[0.24723, 0.58328, 0.30137],
[0.24866, 0.58836, 0.28822],
[0.24993, 0.59342, 0.27496],
[0.25123, 0.59845, 0.2616],
[0.25256, 0.60344, 0.2482],
[0.25407, 0.60839, 0.23474],
[0.25584, 0.61326, 0.22129],
[0.25802, 0.61807, 0.2079],
[0.26074, 0.6228, 0.19465],
[0.26408, 0.62744, 0.18153],
[0.26813, 0.63197, 0.16873],
[0.273, 0.63639, 0.15625],
[0.27873, 0.64068, 0.14417],
[0.28531, 0.64485, 0.13262],
[0.29279, 0.6489, 0.12165],
[0.30109, 0.65281, 0.1115],
[0.31019, 0.65661, 0.10209],
[0.31996, 0.66028, 0.093634],
[0.33034, 0.66383, 0.08618],
[0.34123, 0.66728, 0.079836],
[0.35254, 0.67063, 0.074617],
[0.36416, 0.67388, 0.07048],
[0.37603, 0.67707, 0.067441],
[0.38805, 0.68017, 0.065341],
[0.40017, 0.68323, 0.064135],
[0.41235, 0.68622, 0.063669],
[0.42451, 0.68918, 0.063793],
[0.43665, 0.6921, 0.064391],
[0.44873, 0.69498, 0.065355],
[0.46073, 0.69783, 0.066603],
[0.47266, 0.70067, 0.068066],
[0.4845, 0.70348, 0.06967],
[0.49625, 0.70628, 0.071332],
[0.50793, 0.70905, 0.07306],
[0.5195, 0.71181, 0.074792],
[0.53099, 0.71455, 0.076552],
[0.5424, 0.71728, 0.078345],
[0.55372, 0.72, 0.08017],
[0.56497, 0.72269, 0.082034],
[0.57617, 0.72539, 0.083799],
[0.58729, 0.72805, 0.085523],
[0.59835, 0.73071, 0.087403],
[0.60937, 0.73336, 0.089162],
[0.62033, 0.73599, 0.090964],
[0.63124, 0.73861, 0.092702],
[0.64211, 0.74121, 0.094561],
[0.65293, 0.7438, 0.096272],
[0.66373, 0.74637, 0.09813],
[0.67447, 0.74893, 0.099882],
[0.68519, 0.75149, 0.10167],
[0.69588, 0.75402, 0.1035],
[0.70654, 0.75653, 0.10524],
[0.71717, 0.75904, 0.10708],
[0.72776, 0.76153, 0.10884],
[0.73834, 0.764, 0.11066],
[0.7489, 0.76647, 0.11246],
[0.75944, 0.76891, 0.1142],
[0.76996, 0.77134, 0.11604],
[0.78045, 0.77373, 0.1178],
[0.79092, 0.7761, 0.11957],
[0.80136, 0.77843, 0.12132],
[0.81178, 0.78071, 0.12308],
[0.82215, 0.78294, 0.12485],
[0.83247, 0.78509, 0.12657],
[0.84272, 0.78715, 0.12828],
[0.85289, 0.78908, 0.12995],
[0.86295, 0.79087, 0.13152],
[0.87287, 0.79248, 0.13305],
[0.88262, 0.79388, 0.13451],
[0.89216, 0.79501, 0.13581],
[0.90143, 0.79585, 0.13707],
[0.9104, 0.79634, 0.13819],
[0.919, 0.79646, 0.1391],
[0.9272, 0.79616, 0.13987],
[0.93494, 0.79542, 0.1405],
[0.94218, 0.7942, 0.14092],
[0.94889, 0.79248, 0.14112],
[0.95503, 0.79026, 0.14111],
[0.9606, 0.78755, 0.14089],
[0.96559, 0.78436, 0.14045],
[0.97002, 0.7807, 0.1398],
[0.9739, 0.77661, 0.13901],
[0.97726, 0.77212, 0.13808],
[0.98015, 0.76727, 0.13693],
[0.9826, 0.76211, 0.13566],
[0.98467, 0.75668, 0.13433],
[0.9864, 0.75102, 0.13284],
[0.98784, 0.74515, 0.13129],
[0.98904, 0.73914, 0.12968],
[0.99004, 0.73298, 0.128],
[0.99087, 0.72672, 0.1263],
[0.99158, 0.72038, 0.12453],
[0.99217, 0.71398, 0.12276],
[0.99268, 0.70752, 0.121],
[0.99312, 0.70102, 0.11926],
[0.9935, 0.69447, 0.11747],
[0.99383, 0.68791, 0.11573],
[0.99412, 0.68132, 0.11394],
[0.99438, 0.6747, 0.1122],
[0.99461, 0.66807, 0.11045],
[0.99481, 0.66142, 0.10865],
[0.995, 0.65473, 0.10697],
[0.99516, 0.64804, 0.10517],
[0.99529, 0.64132, 0.1035],
[0.99541, 0.63459, 0.10176],
[0.9955, 0.62783, 0.10005],
[0.99557, 0.62104, 0.098389],
[0.99561, 0.61422, 0.096641],
[0.99564, 0.60739, 0.095059],
[0.99564, 0.60052, 0.093326],
[0.99562, 0.59363, 0.091753],
[0.99558, 0.5867, 0.090085],
[0.99551, 0.57975, 0.08846],
[0.99543, 0.57276, 0.086866],
[0.99532, 0.56573, 0.085214],
[0.9952, 0.55867, 0.083665],
[0.99505, 0.55159, 0.082128],
[0.99488, 0.54446, 0.080509],
[0.99469, 0.53729, 0.078934],
[0.99448, 0.53007, 0.077397],
[0.99425, 0.52281, 0.0759],
[0.994, 0.5155, 0.074438],
[0.99373, 0.50816, 0.07298],
[0.99343, 0.50075, 0.071525],
[0.99312, 0.49329, 0.070045],
[0.99279, 0.48577, 0.068661],
[0.99244, 0.4782, 0.067207],
[0.99206, 0.47057, 0.0658],
[0.99167, 0.46285, 0.064435],
[0.99126, 0.45509, 0.063129],
[0.99083, 0.44724, 0.061707],
[0.99038, 0.43931, 0.060477],
[0.98991, 0.43131, 0.059288],
[0.98943, 0.42321, 0.058031],
[0.98894, 0.41505, 0.05706],
[0.98843, 0.40681, 0.056202],
[0.98792, 0.3985, 0.055473],
[0.98741, 0.39011, 0.055034],
[0.9869, 0.38168, 0.054941],
[0.9864, 0.37321, 0.055283],
[0.98593, 0.36474, 0.056189],
[0.98549, 0.3563, 0.057588],
[0.9851, 0.34792, 0.059933],
[0.98478, 0.33968, 0.062969],
[0.98453, 0.33161, 0.066854],
[0.98439, 0.32384, 0.071751],
[0.98437, 0.31645, 0.077527],
[0.98449, 0.30952, 0.084428],
[0.98477, 0.30314, 0.09217],
[0.98521, 0.29747, 0.10081],
[0.98584, 0.29256, 0.1104],
[0.98667, 0.28854, 0.12072],
[0.98768, 0.28545, 0.1319],
[0.98888, 0.2834, 0.14376],
[0.99026, 0.28237, 0.15632],
[0.9918, 0.28237, 0.16948],
[0.9935, 0.28336, 0.18318],
[0.99531, 0.28528, 0.19743],
[0.99722, 0.2881, 0.21214],
[0.99921, 0.29171, 0.22722],
[1, 0.29596, 0.24264],
[1, 0.3008, 0.25836],
[1, 0.30615, 0.27432],
[1, 0.31182, 0.29047],
[1, 0.31783, 0.30679],
[1, 0.32406, 0.32321],
[1, 0.33045, 0.33974],
[1, 0.33697, 0.35632],
[1, 0.34355, 0.37295],
[1, 0.3502, 0.38964],
[1, 0.35686, 0.40637],
[1, 0.36352, 0.42312],
[1, 0.37019, 0.43991],
[1, 0.37687, 0.45672],
[1, 0.38352, 0.47355],
[1, 0.39014, 0.49042],
[1, 0.39675, 0.50731],
[1, 0.40335, 0.52423],
[1, 0.40993, 0.54118],
[1, 0.41648, 0.55818],
[1, 0.42301, 0.57522],
[1, 0.42955, 0.5923],
[1, 0.43607, 0.60942],
[1, 0.44258, 0.62659],
[1, 0.44907, 0.6438],
[1, 0.45556, 0.66106],
[1, 0.46203, 0.67837],
[1, 0.4685, 0.69574],
[1, 0.47498, 0.71314],
[1, 0.48143, 0.7306],
[1, 0.48789, 0.74812],
[1, 0.49434, 0.76568],
[1, 0.50078, 0.78329],
[1, 0.50723, 0.80096],
[1, 0.51367, 0.81867],
[1, 0.5201, 0.83645],
[1, 0.52655, 0.85427],
[1, 0.53299, 0.87215],
[1, 0.53942, 0.89007],
[1, 0.54587, 0.90805],
[1, 0.55231, 0.92608],
[0.99945, 0.55874, 0.94416],
[0.99597, 0.56518, 0.96229],
[0.99228, 0.57163, 0.98048],
]
b_rainbow_bgyrm_35_85_c71 = bokeh_palette('rainbow_bgyrm_35_85_c71',rainbow_bgyrm_35_85_c71)
m_rainbow_bgyrm_35_85_c71 = mpl_cm('rainbow_bgyrm_35_85_c71',rainbow_bgyrm_35_85_c71)
m_rainbow_bgyrm_35_85_c71_r = mpl_cm('rainbow_bgyrm_35_85_c71_r',list(reversed(rainbow_bgyrm_35_85_c71)))




linear_ternary_green_0_46_c42 = [  # cmap_def
[0, 0, 0],
[0, 0.0035927, 0],
[0, 0.0071431, 1.452e-17],
[2.0473e-17, 0.010757, 3.0748e-17],
[8.1864e-21, 0.014524, 2.5379e-17],
[1.7446e-17, 0.018105, 6.1007e-18],
[4.169e-17, 0.021687, 5.0758e-17],
[0, 0.025284, 2.5867e-17],
[6.0593e-17, 0.028821, 1.8546e-17],
[0, 0.032444, 2.5867e-17],
[4.253e-17, 0.036221, 5.8567e-18],
[0, 0.039766, 1.269e-17],
[2.4692e-17, 0.043161, 2.6355e-17],
[1.4341e-16, 0.046597, 3.7093e-17],
[0, 0.049768, 1.9522e-17],
[5.3227e-17, 0.052749, 4.0997e-17],
[1.6012e-18, 0.055759, 0],
[0, 0.058542, 7.1257e-17],
[5.6429e-17, 0.061269, 3.1236e-17],
[6.2098e-17, 0.063951, 4.783e-17],
[1.3454e-16, 0.066548, 3.026e-17],
[1.454e-17, 0.069095, 2.3427e-17],
[0, 0.071498, 4.6854e-17],
[3.2058e-17, 0.07387, 3.9045e-18],
[1.6029e-16, 0.076086, 1.9522e-17],
[1.1606e-16, 0.07832, 5.271e-17],
[6.2194e-17, 0.080566, 1.3666e-17],
[0, 0.082735, 0],
[0, 0.084825, 0],
[0, 0.086923, 0],
[8.7753e-18, 0.088913, 4.0997e-17],
[4.5733e-17, 0.090908, 0],
[5.8543e-17, 0.092798, 0],
[7.9487e-17, 0.094766, 0],
[0, 0.096572, 0],
[0, 0.098465, 1.9522e-18],
[0, 0.10025, 0],
[4.9576e-17, 0.10204, 0],
[1.1817e-16, 0.10382, 0],
[1.1049e-16, 0.10552, 1.9522e-18],
[0, 0.10725, 0],
[0, 0.10892, 0],
[8.1348e-18, 0.11059, 4.2949e-17],
[1.5174e-16, 0.11222, 1.1713e-17],
[4.2384e-06, 0.11381, 5.6236e-07],
[1.619e-05, 0.11539, 2.1481e-06],
[1.6699e-05, 0.11698, 2.2156e-06],
[1.2756e-05, 0.11863, 1.6925e-06],
[9.7295e-06, 0.12018, 1.2909e-06],
[1.3177e-05, 0.12176, 1.7484e-06],
[1.6418e-05, 0.12338, 2.1783e-06],
[1.7338e-05, 0.12502, 2.3004e-06],
[1.6062e-05, 0.12662, 2.1312e-06],
[1.1099e-05, 0.12826, 1.4726e-06],
[8.2989e-06, 0.1299, 1.1011e-06],
[1.4157e-05, 0.1315, 1.8783e-06],
[1.7475e-05, 0.13312, 2.3186e-06],
[1.7885e-05, 0.13477, 2.373e-06],
[1.5761e-05, 0.13635, 2.0912e-06],
[1.1631e-05, 0.13803, 1.5432e-06],
[8.4045e-06, 0.13962, 1.1151e-06],
[1.4146e-05, 0.1413, 1.8769e-06],
[1.7579e-05, 0.14293, 2.3324e-06],
[1.8299e-05, 0.14458, 2.428e-06],
[1.7087e-05, 0.14621, 2.2671e-06],
[1.2461e-05, 0.14787, 1.6533e-06],
[9.2002e-06, 0.14954, 1.2207e-06],
[1.3439e-05, 0.15116, 1.7831e-06],
[1.7166e-05, 0.15281, 2.2777e-06],
[1.8547e-05, 0.15448, 2.4608e-06],
[1.8043e-05, 0.15618, 2.394e-06],
[1.5436e-05, 0.15782, 2.048e-06],
[1.1288e-05, 0.15947, 1.4977e-06],
[9.1088e-06, 0.16118, 1.2086e-06],
[1.463e-05, 0.16284, 1.9412e-06],
[1.7571e-05, 0.16452, 2.3725e-06],
[9.5709e-06, 0.16618, 2.3585e-06],
[0, 0.16786, 2.12e-06],
[0, 0.16957, 1.7933e-06],
[0, 0.17123, 1.2872e-06],
[0, 0.17292, 1.0023e-06],
[0, 0.17462, 1.66e-06],
[0, 0.17629, 2.0398e-06],
[0, 0.17804, 2.1855e-06],
[0, 0.1797, 2.2541e-06],
[0, 0.18139, 1.9666e-06],
[0, 0.18309, 1.524e-06],
[0, 0.18482, 1.1233e-06],
[0, 0.18653, 1.2478e-06],
[0, 0.18826, 1.8149e-06],
[0, 0.18994, 2.1666e-06],
[0, 0.19165, 2.2543e-06],
[0, 0.19339, 2.1978e-06],
[0, 0.1951, 2.0405e-06],
[0, 0.19684, 1.6279e-06],
[0, 0.19854, 1.1674e-06],
[0, 0.20025, 1.2082e-06],
[0, 0.20199, 1.6696e-06],
[0, 0.20375, 2.0125e-06],
[0, 0.20547, 2.2769e-06],
[0, 0.20722, 2.3292e-06],
[0, 0.20894, 2.1993e-06],
[0, 0.21069, 1.9215e-06],
[0, 0.21244, 1.5648e-06],
[0, 0.21419, 1.1031e-06],
[0, 0.21594, 1.1824e-06],
[0, 0.21768, 1.6796e-06],
[0, 0.21944, 2.0449e-06],
[0, 0.2212, 2.2667e-06],
[0, 0.22296, 2.3344e-06],
[0, 0.22468, 2.2953e-06],
[0, 0.22648, 2.1282e-06],
[0, 0.22824, 1.8455e-06],
[0, 0.22999, 1.3868e-06],
[0, 0.23177, 1.0492e-06],
[0, 0.23352, 1.2152e-06],
[0, 0.23533, 1.6192e-06],
[0, 0.2371, 2.0168e-06],
[0, 0.23885, 2.2741e-06],
[0, 0.24062, 2.3737e-06],
[0, 0.24241, 2.3904e-06],
[0, 0.24419, 2.2583e-06],
[0, 0.24598, 2.0443e-06],
[0, 0.24779, 1.7786e-06],
[0, 0.24956, 1.4126e-06],
[0, 0.25137, 1.0243e-06],
[0, 0.25316, 1.0702e-06],
[0, 0.25496, 1.5288e-06],
[0, 0.25675, 1.8855e-06],
[0, 0.25854, 2.1333e-06],
[0, 0.26035, 2.3139e-06],
[0, 0.26215, 2.3948e-06],
[0, 0.26397, 2.4069e-06],
[0, 0.26577, 2.301e-06],
[0, 0.26756, 2.1598e-06],
[0, 0.26941, 1.9587e-06],
[0, 0.2712, 1.6921e-06],
[0, 0.27302, 1.2675e-06],
[0, 0.27484, 9.345e-07],
[0, 0.27667, 1.0391e-06],
[0, 0.27848, 1.4548e-06],
[0, 0.2803, 1.7503e-06],
[0, 0.28214, 1.9921e-06],
[0, 0.28396, 2.2429e-06],
[0, 0.28578, 2.3368e-06],
[0, 0.28761, 2.387e-06],
[0, 0.28946, 2.4145e-06],
[0, 0.2913, 2.3634e-06],
[0, 0.29313, 2.3021e-06],
[0, 0.29497, 2.1385e-06],
[0, 0.29682, 1.9682e-06],
[0, 0.29866, 1.7236e-06],
[0, 0.3005, 1.4728e-06],
[0, 0.30234, 1.137e-06],
[0, 0.30421, 8.5235e-07],
[0, 0.30606, 9.2407e-07],
[0, 0.30791, 1.2303e-06],
[0, 0.30977, 1.5259e-06],
[0, 0.31161, 1.7664e-06],
[0, 0.31349, 1.8547e-06],
[0, 0.31534, 8.3145e-07],
[6.8048e-16, 0.31721, 9.3707e-17],
[3.4536e-16, 0.31907, 3.1236e-17],
[3.3512e-16, 0.32095, 6.2472e-17],
[1.0248e-17, 0.32282, 0],
[0, 0.32467, 6.2472e-17],
[0, 0.32654, 1.0933e-16],
[6.7536e-16, 0.32843, 1.0933e-16],
[7.4709e-16, 0.3303, 0],
[3.3512e-16, 0.33219, 6.2472e-17],
[6.9585e-16, 0.33407, 4.6854e-17],
[1.6756e-16, 0.33594, 3.1236e-17],
[8.4804e-16, 0.33782, 1.2494e-16],
[1.7781e-16, 0.33972, 0],
[0, 0.3416, 6.2472e-17],
[0, 0.34348, 0],
[7.2147e-16, 0.34538, 0],
[0, 0.34727, 0],
[0, 0.34917, 0],
[1.5731e-16, 0.35106, 6.2472e-17],
[6.4974e-16, 0.35296, 1.8741e-16],
[3.8636e-16, 0.35487, 0],
[2.0855e-16, 0.35676, 0],
[0, 0.35865, 0],
[3.1462e-16, 0.36055, 1.2494e-16],
[4.4119e-16, 0.36247, 2.8112e-16],
[5.1292e-16, 0.36437, 6.2472e-17],
[0, 0.3663, 0],
[6.8048e-16, 0.3682, 9.3707e-17],
[7.0098e-16, 0.3701, 3.1236e-17],
[2.9412e-16, 0.37202, 1.8741e-16],
[5.0268e-16, 0.37395, 9.3707e-17],
[0, 0.37586, 3.1236e-17],
[6.7023e-16, 0.37778, 1.2494e-16],
[0, 0.3797, 3.1236e-17],
[6.7023e-16, 0.38163, 1.2494e-16],
[8.5829e-16, 0.38356, 9.3707e-17],
[5.2317e-16, 0.38547, 3.1236e-17],
[1.0566e-15, 0.3874, 3.1236e-17],
[4.9243e-16, 0.38934, 1.2494e-16],
[0, 0.39128, 6.2472e-17],
[3.5561e-16, 0.3932, 0],
[1.0607e-16, 0.39514, 2.1865e-16],
[0, 0.39707, 9.3707e-17],
[2.0855e-16, 0.39901, 0],
[1.2037e-15, 0.40097, 1.2494e-16],
[9.951e-16, 0.4029, 2.1865e-16],
[1.1934e-15, 0.40484, 1.5618e-16],
[0, 0.40679, 1.2494e-16],
[0, 0.40874, 1.2494e-16],
[0, 0.41069, 0],
[1.3507e-15, 0.41264, 2.1865e-16],
[1.0258e-15, 0.41458, 1.2494e-16],
[3.1462e-16, 0.41654, 1.2494e-16],
[1.3815e-15, 0.41849, 1.2494e-16],
[6.9073e-16, 0.42045, 6.2472e-17],
[7.0098e-16, 0.4224, 3.1236e-17],
[6.0875e-16, 0.42437, 3.1236e-16],
[3.4536e-16, 0.42633, 3.1236e-17],
[9.3361e-16, 0.4283, 4.0607e-16],
[7.0098e-16, 0.43026, 3.1236e-17],
[1.3507e-15, 0.43223, 2.1865e-16],
[1.0258e-15, 0.43419, 1.2494e-16],
[1.3712e-15, 0.43617, 1.5618e-16],
[5.4726e-16, 0.43814, 4.9977e-16],
[7.1123e-16, 0.44011, 0],
[1.7063e-15, 0.44209, 2.1865e-16],
[6.3949e-16, 0.44406, 2.1865e-16],
[3.4536e-16, 0.44603, 3.1236e-17],
[6.1488e-17, 0.44801, 0],
[4.4784e-16, 0.45, 0],
[1.0566e-15, 0.45199, 3.1236e-17],
[1.0463e-15, 0.45396, 6.2472e-17],
[6.7023e-16, 0.45594, 1.2494e-16],
[1.0668e-15, 0.45793, 0],
[9.951e-16, 0.45992, 2.1865e-16],
[7.4197e-16, 0.46191, 0],
[1.1181e-15, 0.46391, 0],
[9.6436e-16, 0.46589, 3.1236e-16],
[1.2585e-15, 0.46789, 4.9977e-16],
[1.3815e-15, 0.46989, 1.2494e-16],
[6.7023e-16, 0.47189, 1.2494e-16],
[3.5561e-16, 0.47388, 0],
[2.0722e-15, 0.47589, 1.8741e-16],
[1.1078e-15, 0.47788, 0],
[1.6346e-15, 0.4799, 4.373e-16],
[6.2924e-16, 0.4819, 2.4989e-16],
[1.6756e-15, 0.48389, 3.1236e-16],
[1.402e-15, 0.48591, 6.2472e-17],
[1.0668e-15, 0.48793, 0],
[0, 0.48992, 0],
[0, 0.49195, 2.4989e-16],
[6.7023e-16, 0.49395, 1.2494e-16],
[1.4429e-15, 0.49597, 0],
[2.7363e-16, 0.498, 2.4989e-16],
[7.1123e-16, 0.5, 0],
]
b_linear_ternary_green_0_46_c42 = bokeh_palette('linear_ternary_green_0_46_c42',linear_ternary_green_0_46_c42)
m_linear_ternary_green_0_46_c42 = mpl_cm('linear_ternary_green_0_46_c42',linear_ternary_green_0_46_c42)
m_linear_ternary_green_0_46_c42_r = mpl_cm('linear_ternary_green_0_46_c42_r',list(reversed(linear_ternary_green_0_46_c42)))
kg = b_linear_ternary_green_0_46_c42
m_kg = m_linear_ternary_green_0_46_c42
m_kg_r = m_linear_ternary_green_0_46_c42_r
palette['kg'] = b_linear_ternary_green_0_46_c42
palette_n['kg'] = b_linear_ternary_green_0_46_c42
cm['kg'] = m_linear_ternary_green_0_46_c42
cm['kg_r'] = m_linear_ternary_green_0_46_c42_r
cm_n['kg'] = mpl_cm('kg',linear_ternary_green_0_46_c42)
cm_n['kg_r'] = mpl_cm('kg_r',list(reversed(linear_ternary_green_0_46_c42)))
CET_L14 = b_linear_ternary_green_0_46_c42
m_CET_L14 = m_linear_ternary_green_0_46_c42
m_CET_L14_r = m_linear_ternary_green_0_46_c42_r
palette['CET_L14'] = b_linear_ternary_green_0_46_c42
cm['CET_L14'] = m_linear_ternary_green_0_46_c42
cm['CET_L14_r'] = m_linear_ternary_green_0_46_c42_r
register_cmap('cet_CET_L14',m_linear_ternary_green_0_46_c42)
register_cmap('cet_CET_L14_r',m_linear_ternary_green_0_46_c42_r)




linear_kryw_5_100_c64 = [  # cmap_def
[0.066001, 0.066024, 0.066019],
[0.080369, 0.066784, 0.064674],
[0.092822, 0.067552, 0.063341],
[0.10397, 0.06834, 0.06191],
[0.11402, 0.069166, 0.060555],
[0.12335, 0.069862, 0.059198],
[0.13261, 0.070292, 0.057702],
[0.14179, 0.070442, 0.056438],
[0.15095, 0.070268, 0.055015],
[0.16001, 0.069905, 0.053644],
[0.16905, 0.069432, 0.052281],
[0.17797, 0.068765, 0.050929],
[0.18676, 0.067963, 0.049608],
[0.19551, 0.06705, 0.048145],
[0.20423, 0.066, 0.046785],
[0.21283, 0.064845, 0.045348],
[0.22141, 0.063566, 0.043893],
[0.22991, 0.062103, 0.042492],
[0.23838, 0.060521, 0.04109],
[0.24679, 0.058793, 0.039622],
[0.25516, 0.056921, 0.038174],
[0.26346, 0.054829, 0.03674],
[0.27169, 0.052601, 0.035359],
[0.27989, 0.050287, 0.033739],
[0.288, 0.047729, 0.032416],
[0.29607, 0.045108, 0.031085],
[0.30408, 0.042249, 0.02978],
[0.31201, 0.03918, 0.028501],
[0.3199, 0.03606, 0.027253],
[0.32767, 0.032742, 0.026037],
[0.33538, 0.029627, 0.024852],
[0.34298, 0.026563, 0.023698],
[0.3505, 0.023574, 0.022579],
[0.3579, 0.0207, 0.021495],
[0.36518, 0.01798, 0.020449],
[0.37234, 0.015464, 0.019443],
[0.37938, 0.013196, 0.018481],
[0.38624, 0.011135, 0.017565],
[0.39295, 0.0095033, 0.016699],
[0.39946, 0.0083814, 0.015886],
[0.4058, 0.0078194, 0.015132],
[0.41198, 0.0076635, 0.014439],
[0.41811, 0.0076226, 0.013789],
[0.42426, 0.0075941, 0.013171],
[0.43042, 0.0075784, 0.012613],
[0.43658, 0.0075771, 0.012105],
[0.44275, 0.0075916, 0.01156],
[0.44893, 0.0076225, 0.01099],
[0.45511, 0.0076698, 0.010514],
[0.46129, 0.0077346, 0.010144],
[0.46749, 0.0078179, 0.0098304],
[0.47368, 0.0079218, 0.0095481],
[0.4799, 0.0080473, 0.0092968],
[0.4861, 0.0081943, 0.0090788],
[0.49232, 0.0083639, 0.0088967],
[0.49856, 0.0085577, 0.0087515],
[0.50479, 0.008779, 0.0086465],
[0.51102, 0.0090297, 0.0085815],
[0.51728, 0.0093076, 0.0085522],
[0.52353, 0.0096083, 0.0085576],
[0.52979, 0.0099253, 0.0085974],
[0.53605, 0.010269, 0.0086738],
[0.54232, 0.010684, 0.0087889],
[0.5486, 0.011214, 0.0089401],
[0.55488, 0.01179, 0.0091252],
[0.56117, 0.012305, 0.0093423],
[0.56747, 0.012813, 0.0095893],
[0.57376, 0.013386, 0.0098594],
[0.58008, 0.013995, 0.010134],
[0.5864, 0.014611, 0.010408],
[0.59274, 0.015241, 0.010687],
[0.59909, 0.015887, 0.010978],
[0.60544, 0.016546, 0.011263],
[0.61182, 0.017218, 0.011525],
[0.61821, 0.017905, 0.011749],
[0.62462, 0.018606, 0.011933],
[0.63103, 0.019321, 0.012084],
[0.63746, 0.020051, 0.012209],
[0.64389, 0.020795, 0.01231],
[0.65035, 0.021554, 0.012391],
[0.65681, 0.022328, 0.01245],
[0.66329, 0.023117, 0.01249],
[0.66978, 0.023921, 0.012509],
[0.67628, 0.024741, 0.012509],
[0.68279, 0.025575, 0.012488],
[0.68932, 0.026426, 0.012446],
[0.69585, 0.027292, 0.012384],
[0.7024, 0.028175, 0.012321],
[0.70896, 0.029075, 0.012266],
[0.71553, 0.029992, 0.01222],
[0.7221, 0.030928, 0.012183],
[0.7287, 0.031878, 0.012154],
[0.7353, 0.032858, 0.012135],
[0.74191, 0.033811, 0.012128],
[0.74853, 0.035044, 0.012132],
[0.75514, 0.036234, 0.01215],
[0.76173, 0.037634, 0.012181],
[0.76828, 0.039279, 0.012225],
[0.77482, 0.041197, 0.012285],
[0.78133, 0.043182, 0.01236],
[0.7878, 0.045551, 0.01245],
[0.79425, 0.048054, 0.012555],
[0.80067, 0.050754, 0.012678],
[0.80707, 0.053571, 0.012821],
[0.81342, 0.056666, 0.012987],
[0.81976, 0.059852, 0.013177],
[0.82606, 0.063125, 0.01339],
[0.83234, 0.066486, 0.013626],
[0.83859, 0.070019, 0.013884],
[0.84481, 0.073701, 0.014163],
[0.85098, 0.077343, 0.014466],
[0.85714, 0.081244, 0.014794],
[0.86327, 0.085127, 0.015149],
[0.86936, 0.089149, 0.015532],
[0.87543, 0.093173, 0.015943],
[0.88146, 0.097361, 0.016384],
[0.88746, 0.10157, 0.016855],
[0.89343, 0.10587, 0.017358],
[0.89937, 0.11022, 0.017892],
[0.90527, 0.11454, 0.01846],
[0.91114, 0.11904, 0.019064],
[0.91695, 0.12359, 0.019699],
[0.92264, 0.12853, 0.020354],
[0.92813, 0.13395, 0.021017],
[0.93344, 0.13977, 0.021689],
[0.93854, 0.14611, 0.02237],
[0.94344, 0.1528, 0.023063],
[0.94814, 0.15984, 0.023768],
[0.95265, 0.16722, 0.024487],
[0.95695, 0.17483, 0.025222],
[0.96106, 0.18267, 0.025974],
[0.96494, 0.19079, 0.026746],
[0.96862, 0.19909, 0.02754],
[0.97211, 0.20758, 0.028357],
[0.97538, 0.21622, 0.029202],
[0.97845, 0.22495, 0.030075],
[0.98129, 0.23387, 0.030982],
[0.98393, 0.24289, 0.03192],
[0.98635, 0.25203, 0.032909],
[0.98856, 0.26123, 0.033896],
[0.99056, 0.27053, 0.035185],
[0.99233, 0.27989, 0.036297],
[0.99387, 0.28935, 0.037474],
[0.99519, 0.29884, 0.038698],
[0.9963, 0.30839, 0.039989],
[0.99718, 0.31799, 0.041353],
[0.99783, 0.32763, 0.042645],
[0.99825, 0.33734, 0.044084],
[0.99848, 0.34702, 0.045528],
[0.99862, 0.35658, 0.047065],
[0.99872, 0.36596, 0.048549],
[0.99881, 0.37516, 0.05001],
[0.99886, 0.38422, 0.051453],
[0.99888, 0.39313, 0.052916],
[0.99888, 0.40191, 0.054408],
[0.99885, 0.41058, 0.05593],
[0.9988, 0.41912, 0.057352],
[0.99871, 0.42755, 0.058876],
[0.9986, 0.43588, 0.060349],
[0.99846, 0.44412, 0.061737],
[0.99829, 0.45227, 0.063269],
[0.99809, 0.46032, 0.064678],
[0.99787, 0.46831, 0.066112],
[0.99761, 0.47622, 0.067563],
[0.99733, 0.48405, 0.069046],
[0.99701, 0.49183, 0.070378],
[0.99667, 0.49952, 0.071816],
[0.9963, 0.50718, 0.073256],
[0.9959, 0.51477, 0.074592],
[0.99546, 0.52229, 0.075945],
[0.995, 0.52978, 0.077303],
[0.99451, 0.53722, 0.078672],
[0.99398, 0.5446, 0.080052],
[0.99343, 0.55194, 0.081445],
[0.99284, 0.55923, 0.08275],
[0.99221, 0.5665, 0.084159],
[0.99153, 0.57372, 0.085535],
[0.99079, 0.58093, 0.087222],
[0.98999, 0.58811, 0.088926],
[0.98913, 0.59528, 0.090807],
[0.9882, 0.60242, 0.092757],
[0.98722, 0.60955, 0.09496],
[0.98617, 0.61665, 0.097148],
[0.98506, 0.62374, 0.099536],
[0.98388, 0.63081, 0.10204],
[0.98264, 0.63786, 0.10462],
[0.98133, 0.6449, 0.1074],
[0.97995, 0.65193, 0.11026],
[0.9785, 0.65893, 0.11321],
[0.97698, 0.66593, 0.11625],
[0.97539, 0.67291, 0.11938],
[0.97373, 0.67987, 0.1226],
[0.972, 0.68684, 0.126],
[0.97019, 0.69378, 0.12941],
[0.96831, 0.70072, 0.13295],
[0.96635, 0.70764, 0.13654],
[0.96431, 0.71455, 0.14025],
[0.9622, 0.72146, 0.14405],
[0.96001, 0.72836, 0.1479],
[0.95774, 0.73524, 0.15182],
[0.95538, 0.7421, 0.15586],
[0.95297, 0.74897, 0.16005],
[0.95072, 0.75572, 0.16525],
[0.94897, 0.76218, 0.17245],
[0.94775, 0.76837, 0.1816],
[0.94697, 0.77433, 0.19225],
[0.94657, 0.7801, 0.204],
[0.94649, 0.78571, 0.21653],
[0.94667, 0.79118, 0.22961],
[0.94708, 0.79653, 0.24318],
[0.94773, 0.80176, 0.2572],
[0.94857, 0.80689, 0.2715],
[0.94954, 0.81193, 0.28598],
[0.95065, 0.81691, 0.30062],
[0.95189, 0.82181, 0.31549],
[0.95324, 0.82665, 0.33046],
[0.95466, 0.83144, 0.3455],
[0.95617, 0.83616, 0.3607],
[0.95775, 0.84084, 0.37601],
[0.95937, 0.84548, 0.39137],
[0.96106, 0.85008, 0.40685],
[0.96275, 0.85465, 0.42235],
[0.9645, 0.85917, 0.43802],
[0.96626, 0.86367, 0.45372],
[0.96804, 0.86814, 0.46954],
[0.96981, 0.87258, 0.4854],
[0.97158, 0.87699, 0.50136],
[0.97335, 0.88139, 0.5174],
[0.9751, 0.88577, 0.53349],
[0.97684, 0.89012, 0.54971],
[0.97854, 0.89447, 0.56596],
[0.98023, 0.89878, 0.58234],
[0.98186, 0.90309, 0.59876],
[0.98347, 0.90738, 0.61531],
[0.98503, 0.91166, 0.6319],
[0.98655, 0.91593, 0.64861],
[0.98799, 0.92019, 0.66537],
[0.9894, 0.92443, 0.68224],
[0.99073, 0.92867, 0.69917],
[0.99199, 0.9329, 0.71621],
[0.99319, 0.93713, 0.73332],
[0.9943, 0.94135, 0.7505],
[0.99535, 0.94556, 0.76779],
[0.99629, 0.94977, 0.78512],
[0.99718, 0.95397, 0.8026],
[0.99795, 0.95817, 0.8201],
[0.99865, 0.96236, 0.83775],
[0.99923, 0.96655, 0.85544],
[0.99971, 0.97075, 0.87321],
[1, 0.97493, 0.8911],
[1, 0.97913, 0.90901],
[1, 0.98331, 0.92711],
[1, 0.9875, 0.94522],
[1, 0.99168, 0.96339],
[1, 0.99586, 0.98169],
[0.99986, 1, 1],
]
b_linear_kryw_5_100_c64 = bokeh_palette('linear_kryw_5_100_c64',linear_kryw_5_100_c64)
m_linear_kryw_5_100_c64 = mpl_cm('linear_kryw_5_100_c64',linear_kryw_5_100_c64)
m_linear_kryw_5_100_c64_r = mpl_cm('linear_kryw_5_100_c64_r',list(reversed(linear_kryw_5_100_c64)))




diverging_bwr_55_98_c37 = [  # cmap_def
[0.14124, 0.50092, 0.99811],
[0.16238, 0.50425, 0.99811],
[0.18106, 0.50758, 0.99811],
[0.19801, 0.51092, 0.9981],
[0.21361, 0.51428, 0.9981],
[0.22813, 0.51763, 0.99809],
[0.24174, 0.52099, 0.99808],
[0.25463, 0.52437, 0.99807],
[0.26685, 0.52775, 0.99806],
[0.27856, 0.53114, 0.99805],
[0.28978, 0.53454, 0.99803],
[0.30057, 0.53795, 0.99801],
[0.31101, 0.54135, 0.99799],
[0.3211, 0.54477, 0.99796],
[0.33087, 0.5482, 0.99794],
[0.34039, 0.55164, 0.99791],
[0.34964, 0.55507, 0.99788],
[0.35865, 0.55852, 0.99784],
[0.36746, 0.56199, 0.99781],
[0.37609, 0.56544, 0.99777],
[0.38451, 0.56891, 0.99773],
[0.39277, 0.57239, 0.99769],
[0.40089, 0.57588, 0.99764],
[0.40883, 0.57937, 0.9976],
[0.41665, 0.58286, 0.99755],
[0.42433, 0.58637, 0.9975],
[0.4319, 0.58987, 0.99744],
[0.43933, 0.59339, 0.99739],
[0.44667, 0.59692, 0.99733],
[0.45389, 0.60045, 0.99727],
[0.46103, 0.60399, 0.99721],
[0.46806, 0.60753, 0.99714],
[0.47502, 0.61108, 0.99707],
[0.48188, 0.61463, 0.997],
[0.48866, 0.6182, 0.99693],
[0.49537, 0.62176, 0.99685],
[0.502, 0.62534, 0.99677],
[0.50857, 0.62892, 0.99669],
[0.51505, 0.63251, 0.99661],
[0.52148, 0.6361, 0.99653],
[0.52785, 0.6397, 0.99644],
[0.53415, 0.6433, 0.99635],
[0.5404, 0.64691, 0.99626],
[0.5466, 0.65053, 0.99616],
[0.55274, 0.65415, 0.99606],
[0.55883, 0.65778, 0.99596],
[0.56487, 0.66141, 0.99586],
[0.57086, 0.66505, 0.99575],
[0.57682, 0.6687, 0.99565],
[0.58271, 0.67235, 0.99554],
[0.58857, 0.676, 0.99542],
[0.59439, 0.67966, 0.99531],
[0.60017, 0.68333, 0.99519],
[0.60591, 0.687, 0.99507],
[0.61162, 0.69067, 0.99494],
[0.61728, 0.69436, 0.99482],
[0.62292, 0.69805, 0.99469],
[0.62852, 0.70174, 0.99455],
[0.63408, 0.70544, 0.99442],
[0.63961, 0.70915, 0.99428],
[0.64511, 0.71285, 0.99414],
[0.65058, 0.71657, 0.994],
[0.65603, 0.72029, 0.99386],
[0.66144, 0.72401, 0.99371],
[0.66681, 0.72774, 0.99356],
[0.67218, 0.73148, 0.9934],
[0.67751, 0.73522, 0.99325],
[0.68281, 0.73897, 0.99309],
[0.6881, 0.74272, 0.99292],
[0.69335, 0.74647, 0.99276],
[0.69858, 0.75023, 0.99259],
[0.70379, 0.754, 0.99242],
[0.70899, 0.75776, 0.99224],
[0.71414, 0.76154, 0.99207],
[0.71929, 0.76532, 0.99189],
[0.72442, 0.7691, 0.9917],
[0.72952, 0.77289, 0.99152],
[0.7346, 0.77668, 0.99133],
[0.73967, 0.78048, 0.99114],
[0.74471, 0.78429, 0.99094],
[0.74973, 0.78809, 0.99074],
[0.75475, 0.7919, 0.99054],
[0.75973, 0.79572, 0.99034],
[0.76471, 0.79954, 0.99013],
[0.76967, 0.80336, 0.98992],
[0.77461, 0.8072, 0.98971],
[0.77953, 0.81103, 0.9895],
[0.78445, 0.81486, 0.98928],
[0.78934, 0.81871, 0.98905],
[0.79422, 0.82256, 0.98883],
[0.79908, 0.82641, 0.9886],
[0.80393, 0.83027, 0.98836],
[0.80877, 0.83413, 0.98813],
[0.81358, 0.83799, 0.98789],
[0.81839, 0.84186, 0.98765],
[0.82319, 0.84573, 0.98741],
[0.82797, 0.84962, 0.98716],
[0.83274, 0.85349, 0.98691],
[0.8375, 0.85738, 0.98666],
[0.84224, 0.86127, 0.9864],
[0.84697, 0.86516, 0.98614],
[0.85169, 0.86907, 0.98588],
[0.8564, 0.87297, 0.98561],
[0.8611, 0.87687, 0.98534],
[0.86579, 0.88079, 0.98507],
[0.87046, 0.88471, 0.98479],
[0.87513, 0.88863, 0.98451],
[0.87979, 0.89255, 0.98422],
[0.88444, 0.89648, 0.98394],
[0.88907, 0.90041, 0.98364],
[0.8937, 0.90434, 0.98335],
[0.89831, 0.90829, 0.98305],
[0.90292, 0.91223, 0.98275],
[0.90752, 0.91617, 0.98245],
[0.91211, 0.92012, 0.98214],
[0.91669, 0.92408, 0.98183],
[0.92126, 0.92804, 0.98152],
[0.92582, 0.932, 0.9812],
[0.93037, 0.93597, 0.98088],
[0.93492, 0.93994, 0.98055],
[0.93946, 0.94392, 0.98022],
[0.94399, 0.9479, 0.97989],
[0.94853, 0.95188, 0.97953],
[0.95311, 0.95584, 0.97909],
[0.95776, 0.95967, 0.97843],
[0.9625, 0.96312, 0.97726],
[0.9672, 0.9657, 0.97517],
[0.97158, 0.96685, 0.97187],
[0.97525, 0.96617, 0.96729],
[0.97801, 0.96372, 0.96172],
[0.97994, 0.95996, 0.9556],
[0.9813, 0.95546, 0.94924],
[0.98234, 0.95064, 0.94283],
[0.98324, 0.94571, 0.93643],
[0.98408, 0.94075, 0.93004],
[0.98488, 0.93579, 0.92366],
[0.98565, 0.93083, 0.9173],
[0.98638, 0.92587, 0.91093],
[0.98709, 0.92091, 0.90457],
[0.98776, 0.91596, 0.89822],
[0.98839, 0.91101, 0.89189],
[0.98901, 0.90605, 0.88555],
[0.98959, 0.9011, 0.87922],
[0.99014, 0.89615, 0.87291],
[0.99066, 0.89119, 0.8666],
[0.99115, 0.88624, 0.8603],
[0.99161, 0.88129, 0.85399],
[0.99205, 0.87633, 0.84771],
[0.99245, 0.87139, 0.84143],
[0.99283, 0.86643, 0.83516],
[0.99317, 0.86148, 0.82889],
[0.99349, 0.85653, 0.82263],
[0.99378, 0.85158, 0.81638],
[0.99404, 0.84663, 0.81014],
[0.99428, 0.84168, 0.80391],
[0.99448, 0.83673, 0.79768],
[0.99467, 0.83179, 0.79146],
[0.99482, 0.82683, 0.78525],
[0.99495, 0.82188, 0.77904],
[0.99505, 0.81693, 0.77284],
[0.99512, 0.81198, 0.76666],
[0.99517, 0.80703, 0.76048],
[0.99519, 0.80207, 0.75431],
[0.99519, 0.79712, 0.74814],
[0.99516, 0.79217, 0.74198],
[0.9951, 0.78722, 0.73583],
[0.99503, 0.78226, 0.72969],
[0.99492, 0.77731, 0.72355],
[0.99479, 0.77235, 0.71744],
[0.99464, 0.76739, 0.71132],
[0.99446, 0.76243, 0.7052],
[0.99425, 0.75748, 0.6991],
[0.99403, 0.75251, 0.69301],
[0.99378, 0.74756, 0.68692],
[0.9935, 0.7426, 0.68084],
[0.9932, 0.73763, 0.67478],
[0.99288, 0.73267, 0.66872],
[0.99253, 0.72769, 0.66267],
[0.99216, 0.72272, 0.65662],
[0.99177, 0.71776, 0.65058],
[0.99135, 0.71278, 0.64455],
[0.99091, 0.7078, 0.63853],
[0.99045, 0.70283, 0.63252],
[0.98996, 0.69784, 0.62651],
[0.98946, 0.69287, 0.62052],
[0.98893, 0.68788, 0.61452],
[0.98837, 0.68289, 0.60856],
[0.9878, 0.6779, 0.60258],
[0.98721, 0.67291, 0.59662],
[0.98659, 0.66791, 0.59066],
[0.98595, 0.66291, 0.58471],
[0.98529, 0.65791, 0.57877],
[0.98461, 0.65289, 0.57285],
[0.9839, 0.64788, 0.56693],
[0.98317, 0.64287, 0.561],
[0.98243, 0.63785, 0.5551],
[0.98166, 0.63283, 0.54921],
[0.98087, 0.6278, 0.54332],
[0.98006, 0.62277, 0.53744],
[0.97923, 0.61772, 0.53156],
[0.97838, 0.61268, 0.5257],
[0.97751, 0.60764, 0.51984],
[0.97662, 0.60258, 0.514],
[0.97571, 0.59752, 0.50816],
[0.97478, 0.59246, 0.50233],
[0.97383, 0.58738, 0.4965],
[0.97286, 0.5823, 0.4907],
[0.97187, 0.57722, 0.48488],
[0.97086, 0.57213, 0.47909],
[0.96983, 0.56703, 0.47329],
[0.96878, 0.56193, 0.46752],
[0.96771, 0.55681, 0.46174],
[0.96662, 0.55168, 0.45597],
[0.96552, 0.54655, 0.45022],
[0.96439, 0.5414, 0.44448],
[0.96325, 0.53626, 0.43872],
[0.96208, 0.53109, 0.433],
[0.9609, 0.52593, 0.42728],
[0.9597, 0.52074, 0.42155],
[0.95849, 0.51554, 0.41584],
[0.95724, 0.51034, 0.41015],
[0.95599, 0.50513, 0.40444],
[0.95472, 0.49989, 0.39876],
[0.95343, 0.49466, 0.39308],
[0.95211, 0.4894, 0.38741],
[0.95079, 0.48413, 0.38175],
[0.94944, 0.47885, 0.3761],
[0.94808, 0.47355, 0.37044],
[0.94669, 0.46823, 0.3648],
[0.9453, 0.4629, 0.35917],
[0.94388, 0.45756, 0.35354],
[0.94245, 0.4522, 0.34793],
[0.94099, 0.44681, 0.34231],
[0.93953, 0.4414, 0.3367],
[0.93804, 0.43598, 0.33109],
[0.93654, 0.43053, 0.32549],
[0.93502, 0.42506, 0.31992],
[0.93348, 0.41957, 0.31433],
[0.93192, 0.41405, 0.30876],
[0.93036, 0.4085, 0.30317],
[0.92877, 0.40293, 0.29761],
[0.92717, 0.39733, 0.29205],
[0.92555, 0.39171, 0.28648],
[0.92391, 0.38604, 0.28092],
[0.92226, 0.38036, 0.2754],
[0.92059, 0.37463, 0.26985],
[0.91891, 0.36887, 0.2643],
[0.91721, 0.36306, 0.25876],
[0.91549, 0.35724, 0.25322],
[0.91376, 0.35135, 0.24768],
[0.91201, 0.34542, 0.24214],
[0.91025, 0.33945, 0.23661],
[0.90847, 0.33343, 0.23107],
[0.90668, 0.32733, 0.22552],
[0.90487, 0.32121, 0.21999],
[0.90304, 0.31501, 0.21443],
]
b_diverging_bwr_55_98_c37 = bokeh_palette('diverging_bwr_55_98_c37',diverging_bwr_55_98_c37)
m_diverging_bwr_55_98_c37 = mpl_cm('diverging_bwr_55_98_c37',diverging_bwr_55_98_c37)
m_diverging_bwr_55_98_c37_r = mpl_cm('diverging_bwr_55_98_c37_r',list(reversed(diverging_bwr_55_98_c37)))
CET_D9 = b_diverging_bwr_55_98_c37
m_CET_D9 = m_diverging_bwr_55_98_c37
m_CET_D9_r = m_diverging_bwr_55_98_c37_r
palette['CET_D9'] = b_diverging_bwr_55_98_c37
cm['CET_D9'] = m_diverging_bwr_55_98_c37
cm['CET_D9_r'] = m_diverging_bwr_55_98_c37_r
register_cmap('cet_CET_D9',m_diverging_bwr_55_98_c37)
register_cmap('cet_CET_D9_r',m_diverging_bwr_55_98_c37_r)




cyclic_mygbm_30_95_c78_s25 = [  # cmap_def
[0.18062, 0.13244, 0.91856],
[0.19173, 0.12446, 0.92396],
[0.20389, 0.11765, 0.92889],
[0.21681, 0.11214, 0.93335],
[0.23028, 0.10794, 0.93739],
[0.24417, 0.10525, 0.94101],
[0.25828, 0.10403, 0.94425],
[0.27246, 0.10417, 0.94716],
[0.28659, 0.1056, 0.94977],
[0.30059, 0.10807, 0.95212],
[0.3144, 0.11153, 0.95426],
[0.32796, 0.11565, 0.95622],
[0.34126, 0.12031, 0.95803],
[0.35426, 0.12544, 0.95973],
[0.36698, 0.13084, 0.96134],
[0.37942, 0.13637, 0.96288],
[0.3916, 0.1421, 0.96436],
[0.40352, 0.14786, 0.96581],
[0.4152, 0.15363, 0.96722],
[0.42669, 0.15941, 0.9686],
[0.43799, 0.16515, 0.96996],
[0.44913, 0.17088, 0.97129],
[0.46013, 0.1765, 0.97261],
[0.47106, 0.18204, 0.97389],
[0.48191, 0.18751, 0.97514],
[0.49273, 0.1928, 0.97635],
[0.50357, 0.19798, 0.97752],
[0.51446, 0.20302, 0.97863],
[0.52541, 0.2079, 0.97969],
[0.53647, 0.21259, 0.98067],
[0.54767, 0.21709, 0.98157],
[0.55903, 0.22137, 0.98238],
[0.57057, 0.22542, 0.9831],
[0.5823, 0.22925, 0.98372],
[0.59423, 0.23284, 0.98424],
[0.60636, 0.23623, 0.98465],
[0.61868, 0.23935, 0.98497],
[0.63116, 0.24227, 0.98518],
[0.64379, 0.24496, 0.9853],
[0.65656, 0.2475, 0.98533],
[0.66943, 0.24982, 0.98527],
[0.68237, 0.25203, 0.98515],
[0.69537, 0.25408, 0.98496],
[0.70839, 0.25601, 0.98471],
[0.72139, 0.25786, 0.98441],
[0.73437, 0.25968, 0.98405],
[0.74728, 0.26143, 0.98365],
[0.76012, 0.26319, 0.98321],
[0.77286, 0.26494, 0.98271],
[0.78548, 0.26676, 0.98216],
[0.79794, 0.26867, 0.98153],
[0.81023, 0.27069, 0.98082],
[0.82233, 0.27285, 0.98001],
[0.83418, 0.27525, 0.97905],
[0.84576, 0.27791, 0.97793],
[0.85702, 0.28089, 0.97661],
[0.86794, 0.2843, 0.97504],
[0.87845, 0.28819, 0.97317],
[0.88852, 0.29263, 0.97097],
[0.8981, 0.29768, 0.96837],
[0.90717, 0.3034, 0.96533],
[0.91567, 0.30984, 0.96182],
[0.92361, 0.317, 0.9578],
[0.93095, 0.32489, 0.95323],
[0.93769, 0.33352, 0.94809],
[0.94383, 0.34283, 0.94239],
[0.94939, 0.35275, 0.93613],
[0.95439, 0.36323, 0.92931],
[0.95886, 0.37422, 0.92198],
[0.96283, 0.38558, 0.91416],
[0.96634, 0.39727, 0.90588],
[0.96944, 0.40921, 0.89721],
[0.97216, 0.4213, 0.88817],
[0.97454, 0.4335, 0.87883],
[0.97663, 0.44573, 0.86923],
[0.97845, 0.45797, 0.8594],
[0.98004, 0.47017, 0.84939],
[0.98142, 0.48228, 0.83922],
[0.98262, 0.4943, 0.82894],
[0.98364, 0.5062, 0.81854],
[0.98453, 0.51799, 0.80808],
[0.98527, 0.52964, 0.79753],
[0.98588, 0.54116, 0.78694],
[0.98637, 0.55255, 0.77629],
[0.98676, 0.56382, 0.76559],
[0.98704, 0.57494, 0.75486],
[0.98722, 0.58595, 0.74407],
[0.98733, 0.59683, 0.73326],
[0.98736, 0.60758, 0.72241],
[0.98732, 0.61821, 0.71153],
[0.98723, 0.62871, 0.7006],
[0.9871, 0.63909, 0.68965],
[0.98694, 0.64936, 0.67864],
[0.98677, 0.65949, 0.66762],
[0.9866, 0.6695, 0.65654],
[0.98644, 0.67938, 0.64543],
[0.9863, 0.68916, 0.63428],
[0.98621, 0.69879, 0.62309],
[0.98616, 0.70831, 0.61184],
[0.98617, 0.71772, 0.60055],
[0.98624, 0.727, 0.5892],
[0.98636, 0.73618, 0.57779],
[0.98653, 0.74526, 0.56632],
[0.98675, 0.75426, 0.55476],
[0.98702, 0.76315, 0.54311],
[0.98731, 0.77198, 0.53134],
[0.98761, 0.78074, 0.51946],
[0.98792, 0.78944, 0.50744],
[0.9882, 0.79807, 0.49526],
[0.98845, 0.80666, 0.48291],
[0.98864, 0.81518, 0.47039],
[0.98875, 0.82365, 0.45765],
[0.98875, 0.83206, 0.4447],
[0.98863, 0.84039, 0.43151],
[0.98834, 0.84863, 0.41804],
[0.98786, 0.85677, 0.40433],
[0.98715, 0.86476, 0.39033],
[0.98616, 0.87259, 0.37607],
[0.98485, 0.8802, 0.36152],
[0.98316, 0.88754, 0.3467],
[0.98105, 0.89458, 0.33163],
[0.97845, 0.90123, 0.31637],
[0.97534, 0.90744, 0.30092],
[0.97166, 0.91315, 0.28537],
[0.96737, 0.91831, 0.26983],
[0.96244, 0.92285, 0.25432],
[0.95686, 0.92672, 0.23895],
[0.95064, 0.9299, 0.22388],
[0.94377, 0.93236, 0.2092],
[0.93628, 0.9341, 0.19506],
[0.9282, 0.93512, 0.18153],
[0.91959, 0.93544, 0.16884],
[0.91047, 0.9351, 0.15697],
[0.90091, 0.93414, 0.1461],
[0.89097, 0.93263, 0.13623],
[0.8807, 0.93061, 0.12747],
[0.87014, 0.92815, 0.11977],
[0.85937, 0.92531, 0.11315],
[0.8484, 0.92216, 0.10742],
[0.83729, 0.91874, 0.10257],
[0.82605, 0.91511, 0.098502],
[0.81472, 0.91132, 0.095091],
[0.80332, 0.90739, 0.092162],
[0.79187, 0.90336, 0.089659],
[0.78036, 0.89925, 0.087518],
[0.76881, 0.89508, 0.08551],
[0.75723, 0.89086, 0.083837],
[0.74562, 0.88662, 0.082243],
[0.73398, 0.88235, 0.080673],
[0.7223, 0.87807, 0.079194],
[0.7106, 0.87376, 0.077792],
[0.69886, 0.86946, 0.076415],
[0.6871, 0.86514, 0.075063],
[0.67529, 0.86082, 0.073757],
[0.66346, 0.85649, 0.072319],
[0.65156, 0.85215, 0.071005],
[0.63963, 0.8478, 0.069678],
[0.62764, 0.84345, 0.068313],
[0.6156, 0.83909, 0.066946],
[0.60351, 0.83473, 0.065602],
[0.59134, 0.83035, 0.064284],
[0.57911, 0.82597, 0.063016],
[0.56681, 0.82158, 0.061599],
[0.55443, 0.81718, 0.060374],
[0.54195, 0.81278, 0.059088],
[0.5294, 0.80837, 0.057695],
[0.51672, 0.80395, 0.056522],
[0.50395, 0.79952, 0.055189],
[0.49106, 0.79508, 0.053903],
[0.47801, 0.79064, 0.052644],
[0.46484, 0.78619, 0.051424],
[0.45151, 0.78173, 0.050257],
[0.43803, 0.77726, 0.04922],
[0.42437, 0.77277, 0.04812],
[0.41056, 0.76827, 0.047322],
[0.39656, 0.76375, 0.04674],
[0.38239, 0.75922, 0.046427],
[0.36808, 0.75466, 0.046596],
[0.35361, 0.75006, 0.047299],
[0.33906, 0.74543, 0.04874],
[0.32443, 0.74075, 0.050897],
[0.30984, 0.73602, 0.054069],
[0.29532, 0.73122, 0.058336],
[0.28105, 0.72634, 0.063783],
[0.26717, 0.72137, 0.070322],
[0.25387, 0.71631, 0.077992],
[0.24134, 0.71112, 0.08687],
[0.22981, 0.7058, 0.096608],
[0.21961, 0.70035, 0.10741],
[0.21092, 0.69475, 0.11899],
[0.204, 0.68901, 0.13129],
[0.19894, 0.6831, 0.14422],
[0.19593, 0.67703, 0.15768],
[0.19486, 0.67081, 0.17161],
[0.1956, 0.66443, 0.18594],
[0.19795, 0.65791, 0.2005],
[0.20163, 0.65125, 0.21533],
[0.20639, 0.64446, 0.2303],
[0.21183, 0.63756, 0.24538],
[0.21771, 0.63057, 0.26052],
[0.22381, 0.62348, 0.27565],
[0.22992, 0.61632, 0.29071],
[0.23593, 0.6091, 0.30574],
[0.24162, 0.60182, 0.32064],
[0.24693, 0.5945, 0.33543],
[0.25184, 0.58716, 0.35008],
[0.25622, 0.5798, 0.36459],
[0.26011, 0.57242, 0.37897],
[0.26346, 0.56501, 0.3932],
[0.26624, 0.55762, 0.4073],
[0.26846, 0.55021, 0.42127],
[0.27013, 0.5428, 0.43513],
[0.27122, 0.53539, 0.44886],
[0.27173, 0.52798, 0.46247],
[0.2717, 0.52057, 0.476],
[0.27112, 0.51317, 0.48942],
[0.26997, 0.50576, 0.50275],
[0.26823, 0.49837, 0.516],
[0.26598, 0.49096, 0.52919],
[0.26316, 0.48352, 0.54228],
[0.25982, 0.4761, 0.55529],
[0.25594, 0.46864, 0.56822],
[0.25162, 0.46117, 0.58105],
[0.2468, 0.45365, 0.5938],
[0.24161, 0.44609, 0.60643],
[0.23605, 0.43849, 0.61894],
[0.23017, 0.43081, 0.63131],
[0.2241, 0.42303, 0.64353],
[0.21793, 0.41517, 0.6556],
[0.2117, 0.40718, 0.66749],
[0.2055, 0.39906, 0.67919],
[0.19945, 0.39079, 0.6907],
[0.19367, 0.38234, 0.70201],
[0.18818, 0.37372, 0.71312],
[0.183, 0.36488, 0.72404],
[0.17829, 0.35585, 0.73477],
[0.17392, 0.34657, 0.7453],
[0.16999, 0.33707, 0.75566],
[0.16639, 0.32732, 0.76586],
[0.16312, 0.31735, 0.7759],
[0.16005, 0.30712, 0.78581],
[0.15724, 0.29667, 0.79557],
[0.15457, 0.28595, 0.80522],
[0.15202, 0.27505, 0.81474],
[0.14966, 0.26395, 0.82414],
[0.14744, 0.25264, 0.8334],
[0.14554, 0.24118, 0.84252],
[0.14402, 0.2296, 0.85145],
[0.14312, 0.218, 0.86019],
[0.14305, 0.20639, 0.86869],
[0.14404, 0.19481, 0.87691],
[0.1463, 0.18336, 0.88484],
[0.15007, 0.17219, 0.89241],
[0.15537, 0.1614, 0.8996],
[0.1623, 0.15103, 0.90637],
[0.17075, 0.14136, 0.9127],
]
b_cyclic_mygbm_30_95_c78_s25 = bokeh_palette('cyclic_mygbm_30_95_c78_s25',cyclic_mygbm_30_95_c78_s25)
m_cyclic_mygbm_30_95_c78_s25 = mpl_cm('cyclic_mygbm_30_95_c78_s25',cyclic_mygbm_30_95_c78_s25)
m_cyclic_mygbm_30_95_c78_s25_r = mpl_cm('cyclic_mygbm_30_95_c78_s25_r',list(reversed(cyclic_mygbm_30_95_c78_s25)))
colorwheel = b_cyclic_mygbm_30_95_c78_s25
m_colorwheel = m_cyclic_mygbm_30_95_c78_s25
m_colorwheel_r = m_cyclic_mygbm_30_95_c78_s25_r
palette['colorwheel'] = b_cyclic_mygbm_30_95_c78_s25
palette_n['colorwheel'] = b_cyclic_mygbm_30_95_c78_s25
cm['colorwheel'] = m_cyclic_mygbm_30_95_c78_s25
cm['colorwheel_r'] = m_cyclic_mygbm_30_95_c78_s25_r
cm_n['colorwheel'] = mpl_cm('colorwheel',cyclic_mygbm_30_95_c78_s25)
cm_n['colorwheel_r'] = mpl_cm('colorwheel_r',list(reversed(cyclic_mygbm_30_95_c78_s25)))
CET_C2s = b_cyclic_mygbm_30_95_c78_s25
m_CET_C2s = m_cyclic_mygbm_30_95_c78_s25
m_CET_C2s_r = m_cyclic_mygbm_30_95_c78_s25_r
palette['CET_C2s'] = b_cyclic_mygbm_30_95_c78_s25
cm['CET_C2s'] = m_cyclic_mygbm_30_95_c78_s25
cm['CET_C2s_r'] = m_cyclic_mygbm_30_95_c78_s25_r
register_cmap('cet_CET_C2s',m_cyclic_mygbm_30_95_c78_s25)
register_cmap('cet_CET_C2s_r',m_cyclic_mygbm_30_95_c78_s25_r)




diverging_rainbow_bgymr_45_85_c67 = [  # cmap_def
[0.032081, 0.36182, 0.97348],
[0.05836, 0.37092, 0.95845],
[0.075961, 0.37984, 0.94345],
[0.088784, 0.38858, 0.92847],
[0.098335, 0.39714, 0.91352],
[0.10538, 0.40555, 0.89858],
[0.1105, 0.4138, 0.88367],
[0.1138, 0.42191, 0.86877],
[0.11557, 0.4299, 0.85388],
[0.11592, 0.43777, 0.83902],
[0.11487, 0.44549, 0.82416],
[0.11266, 0.45311, 0.80932],
[0.10918, 0.46059, 0.79447],
[0.10455, 0.46796, 0.77961],
[0.099069, 0.47521, 0.76474],
[0.092835, 0.4823, 0.74982],
[0.08633, 0.48923, 0.73486],
[0.080053, 0.496, 0.71982],
[0.074782, 0.50259, 0.70469],
[0.071457, 0.50899, 0.68944],
[0.07085, 0.51517, 0.67403],
[0.073695, 0.52113, 0.65846],
[0.079835, 0.52688, 0.64271],
[0.089012, 0.5324, 0.62676],
[0.10026, 0.53772, 0.61062],
[0.11288, 0.54284, 0.59428],
[0.126, 0.54779, 0.57774],
[0.1391, 0.55261, 0.56104],
[0.15183, 0.55732, 0.54417],
[0.16388, 0.56193, 0.52714],
[0.17497, 0.56648, 0.50997],
[0.18508, 0.57098, 0.49266],
[0.19412, 0.57547, 0.47524],
[0.20204, 0.57994, 0.45766],
[0.20895, 0.58441, 0.43995],
[0.21483, 0.58888, 0.4221],
[0.21979, 0.59337, 0.40411],
[0.22384, 0.59785, 0.38597],
[0.22711, 0.60234, 0.36769],
[0.22964, 0.60683, 0.3493],
[0.23163, 0.6113, 0.33081],
[0.23311, 0.61574, 0.3123],
[0.23437, 0.62014, 0.29385],
[0.23553, 0.62448, 0.27559],
[0.23686, 0.62873, 0.25761],
[0.23856, 0.63287, 0.24012],
[0.24094, 0.63689, 0.22333],
[0.24418, 0.64076, 0.2074],
[0.2485, 0.64447, 0.19256],
[0.25393, 0.64801, 0.17902],
[0.26052, 0.65139, 0.16694],
[0.26821, 0.65459, 0.15642],
[0.2769, 0.65765, 0.14749],
[0.28638, 0.66054, 0.14012],
[0.29654, 0.66333, 0.13429],
[0.30712, 0.666, 0.12971],
[0.318, 0.6686, 0.12623],
[0.32904, 0.67112, 0.12356],
[0.34011, 0.6736, 0.12159],
[0.35116, 0.67603, 0.1201],
[0.36212, 0.67844, 0.11895],
[0.37296, 0.68081, 0.11798],
[0.38369, 0.68318, 0.11711],
[0.39426, 0.68554, 0.11638],
[0.40469, 0.68788, 0.11563],
[0.415, 0.69021, 0.11486],
[0.42518, 0.69254, 0.11414],
[0.43523, 0.69484, 0.11344],
[0.44516, 0.69715, 0.11268],
[0.455, 0.69945, 0.1119],
[0.46473, 0.70173, 0.11112],
[0.47437, 0.70402, 0.11034],
[0.48392, 0.7063, 0.10954],
[0.4934, 0.70856, 0.10866],
[0.50281, 0.71081, 0.10783],
[0.51215, 0.71306, 0.10703],
[0.52142, 0.7153, 0.10617],
[0.53064, 0.71754, 0.10523],
[0.53978, 0.71976, 0.10434],
[0.5489, 0.72197, 0.10349],
[0.55794, 0.72417, 0.10255],
[0.56694, 0.72637, 0.1016],
[0.5759, 0.72856, 0.10065],
[0.58481, 0.73074, 0.099684],
[0.59368, 0.73292, 0.098715],
[0.60252, 0.73509, 0.097735],
[0.61132, 0.73724, 0.096645],
[0.62009, 0.73939, 0.095664],
[0.62882, 0.74152, 0.094646],
[0.63752, 0.74366, 0.09351],
[0.64619, 0.74578, 0.092435],
[0.65483, 0.7479, 0.091367],
[0.66346, 0.75, 0.090209],
[0.67205, 0.7521, 0.089055],
[0.68061, 0.75419, 0.087892],
[0.68917, 0.75627, 0.086685],
[0.69769, 0.75835, 0.085411],
[0.7062, 0.76041, 0.084247],
[0.71468, 0.76246, 0.082921],
[0.72315, 0.76451, 0.081681],
[0.73161, 0.76655, 0.0803],
[0.74005, 0.76858, 0.078922],
[0.74847, 0.77061, 0.077534],
[0.75687, 0.77262, 0.076136],
[0.76527, 0.77462, 0.074724],
[0.77364, 0.77663, 0.07331],
[0.78201, 0.77861, 0.071765],
[0.79036, 0.78059, 0.070194],
[0.7987, 0.78257, 0.068669],
[0.80704, 0.78453, 0.067],
[0.81535, 0.78649, 0.065324],
[0.82366, 0.78843, 0.063634],
[0.83198, 0.79037, 0.061832],
[0.84029, 0.7923, 0.060168],
[0.84859, 0.7942, 0.058382],
[0.85689, 0.79607, 0.056834],
[0.8652, 0.79789, 0.055378],
[0.87351, 0.79966, 0.054292],
[0.88182, 0.80132, 0.053775],
[0.89012, 0.80285, 0.054093],
[0.8984, 0.8042, 0.055566],
[0.90662, 0.8053, 0.058415],
[0.91475, 0.80608, 0.063031],
[0.92272, 0.80648, 0.069366],
[0.93047, 0.80641, 0.077301],
[0.9379, 0.80582, 0.08703],
[0.94493, 0.80465, 0.097983],
[0.95147, 0.80288, 0.10998],
[0.95744, 0.80051, 0.12259],
[0.9628, 0.79755, 0.1357],
[0.9675, 0.79406, 0.14898],
[0.97155, 0.79007, 0.16223],
[0.97499, 0.78567, 0.17523],
[0.97786, 0.78094, 0.18795],
[0.98025, 0.77593, 0.20021],
[0.98222, 0.77073, 0.21212],
[0.98385, 0.76537, 0.22355],
[0.98522, 0.75991, 0.23457],
[0.98638, 0.75438, 0.24514],
[0.9874, 0.74879, 0.25538],
[0.98831, 0.74317, 0.26524],
[0.98914, 0.73751, 0.27479],
[0.98991, 0.73185, 0.28406],
[0.99062, 0.72616, 0.29307],
[0.9913, 0.72045, 0.30183],
[0.99194, 0.71474, 0.3104],
[0.99255, 0.70902, 0.31874],
[0.99314, 0.70327, 0.32691],
[0.99369, 0.6975, 0.33494],
[0.99421, 0.69172, 0.34279],
[0.9947, 0.68593, 0.35051],
[0.99516, 0.68011, 0.35809],
[0.9956, 0.67428, 0.36554],
[0.996, 0.66843, 0.37289],
[0.99638, 0.66256, 0.38013],
[0.99672, 0.65667, 0.38727],
[0.99704, 0.65075, 0.39432],
[0.99732, 0.64482, 0.40128],
[0.99758, 0.63886, 0.40815],
[0.99781, 0.63289, 0.41495],
[0.99801, 0.62688, 0.42166],
[0.99818, 0.62086, 0.42832],
[0.99832, 0.6148, 0.43492],
[0.99843, 0.60874, 0.44143],
[0.99851, 0.60262, 0.44789],
[0.99857, 0.59649, 0.45431],
[0.99859, 0.59033, 0.46065],
[0.99858, 0.58414, 0.46696],
[0.99855, 0.57791, 0.4732],
[0.99848, 0.57166, 0.47942],
[0.99839, 0.56537, 0.48557],
[0.99827, 0.55905, 0.4917],
[0.99812, 0.5527, 0.49778],
[0.99793, 0.54631, 0.50381],
[0.99772, 0.53987, 0.50981],
[0.99748, 0.53339, 0.51577],
[0.99721, 0.52689, 0.52171],
[0.99692, 0.52032, 0.52761],
[0.99659, 0.51372, 0.53348],
[0.99623, 0.50706, 0.53932],
[0.99585, 0.50036, 0.54514],
[0.99543, 0.4936, 0.55092],
[0.99499, 0.4868, 0.55668],
[0.99451, 0.47994, 0.56242],
[0.99401, 0.473, 0.56813],
[0.99348, 0.46601, 0.57381],
[0.99292, 0.45897, 0.57948],
[0.99232, 0.45185, 0.58513],
[0.9917, 0.44466, 0.59074],
[0.99105, 0.43739, 0.59635],
[0.99036, 0.43003, 0.60193],
[0.98965, 0.42258, 0.60751],
[0.98891, 0.41506, 0.61304],
[0.98813, 0.40745, 0.61855],
[0.98733, 0.39973, 0.62399],
[0.98649, 0.39194, 0.62934],
[0.98562, 0.38405, 0.63457],
[0.98472, 0.37608, 0.63961],
[0.98378, 0.36804, 0.64438],
[0.98281, 0.35995, 0.64878],
[0.98181, 0.35183, 0.65266],
[0.98078, 0.34373, 0.65588],
[0.97973, 0.3357, 0.65825],
[0.97864, 0.32775, 0.65959],
[0.97754, 0.31999, 0.65974],
[0.97642, 0.3124, 0.65853],
[0.97528, 0.30511, 0.65589],
[0.97411, 0.29808, 0.65173],
[0.97292, 0.2914, 0.64606],
[0.97169, 0.285, 0.63895],
[0.97041, 0.27897, 0.63052],
[0.96907, 0.2732, 0.62092],
[0.96766, 0.26768, 0.61031],
[0.96618, 0.26243, 0.5989],
[0.96461, 0.25735, 0.58686],
[0.96297, 0.2524, 0.57435],
[0.96124, 0.24756, 0.56152],
[0.95942, 0.24278, 0.54846],
[0.95753, 0.23806, 0.53526],
[0.95556, 0.23335, 0.52198],
[0.95351, 0.22869, 0.50869],
[0.95138, 0.22398, 0.49537],
[0.94918, 0.2193, 0.48207],
[0.94691, 0.21459, 0.4688],
[0.94457, 0.20987, 0.45555],
[0.94216, 0.20512, 0.44234],
[0.93968, 0.20033, 0.42915],
[0.93714, 0.19555, 0.416],
[0.93453, 0.19073, 0.40287],
[0.93185, 0.1859, 0.38976],
[0.92911, 0.18098, 0.3767],
[0.92631, 0.17605, 0.36362],
[0.92345, 0.17111, 0.35059],
[0.92052, 0.16609, 0.33755],
[0.91755, 0.16106, 0.32451],
[0.9145, 0.15595, 0.31148],
[0.9114, 0.15076, 0.29845],
[0.90825, 0.14556, 0.2854],
[0.90503, 0.1402, 0.27233],
[0.90176, 0.13484, 0.25926],
[0.89844, 0.12932, 0.24609],
[0.89507, 0.12371, 0.23288],
[0.89164, 0.11805, 0.21961],
[0.88816, 0.11221, 0.20623],
[0.88463, 0.10623, 0.19268],
[0.88106, 0.10002, 0.17898],
[0.87743, 0.093639, 0.16503],
[0.87376, 0.087085, 0.15083],
[0.87004, 0.08013, 0.13619],
[0.86629, 0.072935, 0.12107],
[0.86249, 0.065249, 0.10524],
[0.85864, 0.057063, 0.088412],
[0.85476, 0.048151, 0.069991],
[0.85083, 0.038404, 0.048987],
[0.84687, 0.028128, 0.024389],
[0.84287, 0.018301, 0.0015175],
]
b_diverging_rainbow_bgymr_45_85_c67 = bokeh_palette('diverging_rainbow_bgymr_45_85_c67',diverging_rainbow_bgymr_45_85_c67)
m_diverging_rainbow_bgymr_45_85_c67 = mpl_cm('diverging_rainbow_bgymr_45_85_c67',diverging_rainbow_bgymr_45_85_c67)
m_diverging_rainbow_bgymr_45_85_c67_r = mpl_cm('diverging_rainbow_bgymr_45_85_c67_r',list(reversed(diverging_rainbow_bgymr_45_85_c67)))
CET_R3 = b_diverging_rainbow_bgymr_45_85_c67
m_CET_R3 = m_diverging_rainbow_bgymr_45_85_c67
m_CET_R3_r = m_diverging_rainbow_bgymr_45_85_c67_r
palette['CET_R3'] = b_diverging_rainbow_bgymr_45_85_c67
cm['CET_R3'] = m_diverging_rainbow_bgymr_45_85_c67
cm['CET_R3_r'] = m_diverging_rainbow_bgymr_45_85_c67_r
register_cmap('cet_CET_R3',m_diverging_rainbow_bgymr_45_85_c67)
register_cmap('cet_CET_R3_r',m_diverging_rainbow_bgymr_45_85_c67_r)




linear_bgyw_15_100_c68 = [  # cmap_def
[0.10261, 0, 0.52493],
[0.10283, 0, 0.53197],
[0.10301, 0, 0.53902],
[0.10318, 0.0009426, 0.54604],
[0.10333, 0.0032938, 0.55302],
[0.10346, 0.0057773, 0.56],
[0.10358, 0.0084209, 0.56694],
[0.10367, 0.01129, 0.57386],
[0.10377, 0.014338, 0.58073],
[0.10383, 0.017402, 0.5876],
[0.1039, 0.020685, 0.59441],
[0.10395, 0.024094, 0.60121],
[0.10399, 0.027709, 0.60797],
[0.10402, 0.031481, 0.61469],
[0.10405, 0.035677, 0.62137],
[0.10407, 0.039812, 0.62802],
[0.10409, 0.043972, 0.63463],
[0.1041, 0.048145, 0.64118],
[0.10411, 0.052249, 0.64771],
[0.10412, 0.056438, 0.65417],
[0.10412, 0.060474, 0.66061],
[0.10414, 0.064518, 0.66698],
[0.10414, 0.068586, 0.67332],
[0.10416, 0.072559, 0.67957],
[0.10417, 0.076535, 0.68581],
[0.1042, 0.080591, 0.69196],
[0.10423, 0.084636, 0.69806],
[0.10427, 0.088643, 0.7041],
[0.10432, 0.092635, 0.71007],
[0.10438, 0.096648, 0.71599],
[0.10445, 0.10074, 0.72181],
[0.10454, 0.10477, 0.72759],
[0.10466, 0.1089, 0.73327],
[0.10478, 0.11303, 0.73891],
[0.10495, 0.11711, 0.74441],
[0.10512, 0.12123, 0.74987],
[0.10534, 0.12546, 0.75522],
[0.10558, 0.12966, 0.76048],
[0.10584, 0.13388, 0.76566],
[0.10615, 0.13816, 0.77071],
[0.10646, 0.14242, 0.77569],
[0.10681, 0.14674, 0.78053],
[0.10717, 0.15111, 0.78527],
[0.10755, 0.15551, 0.7899],
[0.10798, 0.15993, 0.79438],
[0.10845, 0.16445, 0.79875],
[0.10901, 0.16898, 0.80298],
[0.10961, 0.17353, 0.80705],
[0.1102, 0.17818, 0.81099],
[0.11082, 0.1828, 0.81477],
[0.1115, 0.18758, 0.81836],
[0.11222, 0.19233, 0.82179],
[0.11298, 0.19716, 0.82505],
[0.11375, 0.20204, 0.82808],
[0.11454, 0.20704, 0.83091],
[0.11544, 0.21207, 0.83352],
[0.11636, 0.21715, 0.83593],
[0.11726, 0.22231, 0.83805],
[0.11827, 0.22758, 0.83989],
[0.11923, 0.23291, 0.84144],
[0.12022, 0.23833, 0.84271],
[0.12122, 0.24384, 0.84364],
[0.12223, 0.24946, 0.84423],
[0.12325, 0.2552, 0.84442],
[0.12427, 0.26103, 0.84418],
[0.12527, 0.26698, 0.84347],
[0.12622, 0.2731, 0.84224],
[0.12705, 0.27937, 0.84043],
[0.12787, 0.28578, 0.83797],
[0.12853, 0.2924, 0.83479],
[0.129, 0.29923, 0.8308],
[0.12928, 0.30627, 0.82589],
[0.12926, 0.31358, 0.81986],
[0.12882, 0.3212, 0.8125],
[0.12785, 0.32922, 0.80355],
[0.12621, 0.33765, 0.79274],
[0.1245, 0.34613, 0.78084],
[0.12367, 0.35431, 0.76889],
[0.12366, 0.36214, 0.75705],
[0.12444, 0.36966, 0.74532],
[0.12594, 0.37693, 0.7337],
[0.12792, 0.38394, 0.72217],
[0.13044, 0.39074, 0.71074],
[0.13327, 0.39733, 0.69939],
[0.13642, 0.40376, 0.68814],
[0.13984, 0.41001, 0.67697],
[0.14348, 0.41611, 0.66586],
[0.14722, 0.42207, 0.65483],
[0.15108, 0.42792, 0.64387],
[0.15499, 0.43363, 0.63298],
[0.15896, 0.43925, 0.62214],
[0.16291, 0.44478, 0.61137],
[0.16687, 0.4502, 0.60064],
[0.17081, 0.45554, 0.58996],
[0.17471, 0.4608, 0.57934],
[0.17858, 0.46599, 0.56874],
[0.18239, 0.47111, 0.55819],
[0.18617, 0.47617, 0.54768],
[0.18985, 0.48116, 0.53721],
[0.19353, 0.4861, 0.52675],
[0.19713, 0.491, 0.51631],
[0.20061, 0.49582, 0.50589],
[0.20411, 0.50061, 0.4955],
[0.20751, 0.50535, 0.48511],
[0.21087, 0.51006, 0.47474],
[0.21416, 0.51473, 0.46436],
[0.21738, 0.51936, 0.45396],
[0.22058, 0.52395, 0.44356],
[0.2237, 0.52851, 0.43315],
[0.22681, 0.53303, 0.42271],
[0.22982, 0.53753, 0.41226],
[0.23283, 0.54198, 0.40175],
[0.23581, 0.54643, 0.39121],
[0.23872, 0.55083, 0.38071],
[0.24169, 0.55518, 0.37044],
[0.24469, 0.55949, 0.3605],
[0.24781, 0.56374, 0.35089],
[0.25097, 0.56793, 0.34154],
[0.25425, 0.57208, 0.33247],
[0.25761, 0.5762, 0.32363],
[0.26111, 0.58025, 0.31505],
[0.26472, 0.58427, 0.3067],
[0.26848, 0.58825, 0.29856],
[0.27234, 0.5922, 0.29062],
[0.27639, 0.5961, 0.28291],
[0.28053, 0.59996, 0.27535],
[0.28484, 0.6038, 0.26796],
[0.28931, 0.60759, 0.26077],
[0.29388, 0.61135, 0.25373],
[0.29862, 0.61508, 0.24682],
[0.3035, 0.61878, 0.24008],
[0.30855, 0.62243, 0.23349],
[0.31372, 0.62607, 0.22707],
[0.31901, 0.62967, 0.22077],
[0.32444, 0.63324, 0.21455],
[0.33002, 0.63679, 0.20847],
[0.3357, 0.6403, 0.20249],
[0.3415, 0.64378, 0.19666],
[0.34744, 0.64724, 0.1909],
[0.35349, 0.65066, 0.18533],
[0.35968, 0.65406, 0.17981],
[0.36597, 0.65744, 0.17443],
[0.37234, 0.66078, 0.16914],
[0.37883, 0.6641, 0.16394],
[0.38543, 0.6674, 0.15885],
[0.39215, 0.67066, 0.15389],
[0.39895, 0.6739, 0.14907],
[0.40585, 0.67712, 0.14434],
[0.41283, 0.6803, 0.13966],
[0.41989, 0.68346, 0.13522],
[0.42707, 0.6866, 0.1309],
[0.43431, 0.68971, 0.12666],
[0.44163, 0.6928, 0.12254],
[0.44902, 0.69586, 0.11865],
[0.45651, 0.69888, 0.11488],
[0.46408, 0.70188, 0.11136],
[0.4717, 0.70487, 0.1079],
[0.47938, 0.70783, 0.10464],
[0.48716, 0.71076, 0.10172],
[0.495, 0.71367, 0.098964],
[0.50288, 0.71655, 0.096355],
[0.51084, 0.71941, 0.094134],
[0.51888, 0.72223, 0.092156],
[0.52696, 0.72504, 0.090393],
[0.53508, 0.72782, 0.088892],
[0.54329, 0.73057, 0.087819],
[0.55155, 0.73329, 0.087005],
[0.55985, 0.73599, 0.086399],
[0.56821, 0.73867, 0.08621],
[0.57664, 0.74131, 0.086387],
[0.5851, 0.74394, 0.086866],
[0.59361, 0.74654, 0.087651],
[0.60218, 0.74911, 0.088758],
[0.61075, 0.75167, 0.090091],
[0.6193, 0.75421, 0.091662],
[0.62776, 0.75676, 0.093291],
[0.63618, 0.7593, 0.095179],
[0.64452, 0.76186, 0.097055],
[0.65279, 0.76442, 0.0992],
[0.66101, 0.76698, 0.10144],
[0.66917, 0.76954, 0.10382],
[0.67726, 0.77211, 0.10635],
[0.68528, 0.77468, 0.10898],
[0.69324, 0.77727, 0.11176],
[0.70115, 0.77985, 0.1146],
[0.70899, 0.78245, 0.11768],
[0.71675, 0.78505, 0.12084],
[0.72446, 0.78766, 0.12415],
[0.73211, 0.79027, 0.12759],
[0.73969, 0.7929, 0.13116],
[0.74719, 0.79554, 0.13488],
[0.75463, 0.79819, 0.13869],
[0.762, 0.80085, 0.14265],
[0.76931, 0.80352, 0.1467],
[0.77655, 0.80621, 0.15091],
[0.78371, 0.80891, 0.15527],
[0.79078, 0.81163, 0.15974],
[0.79778, 0.81436, 0.1644],
[0.80471, 0.81711, 0.16917],
[0.81155, 0.81987, 0.17404],
[0.8183, 0.82266, 0.17908],
[0.82496, 0.82547, 0.18425],
[0.83154, 0.8283, 0.18958],
[0.83802, 0.83116, 0.19506],
[0.84439, 0.83404, 0.20068],
[0.85066, 0.83694, 0.20653],
[0.85681, 0.83988, 0.21248],
[0.86285, 0.84285, 0.21864],
[0.86877, 0.84585, 0.22495],
[0.87454, 0.84889, 0.23152],
[0.88018, 0.85196, 0.23823],
[0.88567, 0.85508, 0.24516],
[0.89101, 0.85823, 0.25235],
[0.89617, 0.86144, 0.25975],
[0.90116, 0.86469, 0.26737],
[0.90596, 0.868, 0.27529],
[0.91052, 0.87137, 0.2835],
[0.91485, 0.87481, 0.29203],
[0.91894, 0.87831, 0.30086],
[0.92276, 0.88189, 0.31007],
[0.92625, 0.88556, 0.3197],
[0.92941, 0.88931, 0.32973],
[0.93217, 0.89317, 0.34027],
[0.93451, 0.89714, 0.3513],
[0.93634, 0.90125, 0.36298],
[0.93763, 0.9055, 0.37534],
[0.9387, 0.90978, 0.38837],
[0.93989, 0.91399, 0.40197],
[0.94123, 0.91811, 0.41613],
[0.94272, 0.92216, 0.43083],
[0.94436, 0.92612, 0.44608],
[0.94613, 0.93, 0.46186],
[0.94803, 0.9338, 0.47813],
[0.95005, 0.93753, 0.49497],
[0.95218, 0.94117, 0.5123],
[0.95441, 0.94474, 0.53014],
[0.95674, 0.94822, 0.54852],
[0.95915, 0.95163, 0.56741],
[0.96162, 0.95496, 0.58679],
[0.96415, 0.95822, 0.60671],
[0.96673, 0.96139, 0.62717],
[0.96934, 0.96449, 0.64813],
[0.97196, 0.96752, 0.66962],
[0.97459, 0.97046, 0.69168],
[0.97722, 0.97333, 0.71425],
[0.9798, 0.97613, 0.73736],
[0.98235, 0.97885, 0.76105],
[0.98483, 0.98151, 0.78529],
[0.98722, 0.98408, 0.81007],
[0.98952, 0.98658, 0.83544],
[0.99169, 0.98901, 0.8614],
[0.99371, 0.99136, 0.88792],
[0.99556, 0.99365, 0.91503],
[0.99723, 0.99585, 0.94278],
[0.99867, 0.99799, 0.97109],
[0.99986, 1, 1],
]
b_linear_bgyw_15_100_c68 = bokeh_palette('linear_bgyw_15_100_c68',linear_bgyw_15_100_c68)
m_linear_bgyw_15_100_c68 = mpl_cm('linear_bgyw_15_100_c68',linear_bgyw_15_100_c68)
m_linear_bgyw_15_100_c68_r = mpl_cm('linear_bgyw_15_100_c68_r',list(reversed(linear_bgyw_15_100_c68)))
bgyw = b_linear_bgyw_15_100_c68
m_bgyw = m_linear_bgyw_15_100_c68
m_bgyw_r = m_linear_bgyw_15_100_c68_r
palette['bgyw'] = b_linear_bgyw_15_100_c68
palette_n['bgyw'] = b_linear_bgyw_15_100_c68
cm['bgyw'] = m_linear_bgyw_15_100_c68
cm['bgyw_r'] = m_linear_bgyw_15_100_c68_r
cm_n['bgyw'] = mpl_cm('bgyw',linear_bgyw_15_100_c68)
cm_n['bgyw_r'] = mpl_cm('bgyw_r',list(reversed(linear_bgyw_15_100_c68)))




diverging_isoluminant_cjo_70_c25 = [  # cmap_def
[0, 0.71527, 1],
[0, 0.7149, 1],
[0, 0.71453, 1],
[0, 0.71417, 1],
[0.024434, 0.7138, 1],
[0.068274, 0.71344, 0.99912],
[0.097664, 0.71307, 0.99639],
[0.12062, 0.71271, 0.99368],
[0.13997, 0.71234, 0.99096],
[0.15693, 0.71199, 0.98824],
[0.1721, 0.71163, 0.98553],
[0.18598, 0.71126, 0.98281],
[0.19872, 0.7109, 0.9801],
[0.21064, 0.71054, 0.97739],
[0.2218, 0.71018, 0.97467],
[0.23235, 0.70981, 0.97196],
[0.24234, 0.70945, 0.96925],
[0.25188, 0.7091, 0.96654],
[0.26097, 0.70874, 0.96383],
[0.26971, 0.70837, 0.96112],
[0.27808, 0.70801, 0.95841],
[0.28614, 0.70765, 0.95571],
[0.29394, 0.70729, 0.953],
[0.30148, 0.70693, 0.95029],
[0.3088, 0.70658, 0.94759],
[0.31588, 0.70622, 0.94488],
[0.32276, 0.70586, 0.94218],
[0.32945, 0.7055, 0.93948],
[0.33595, 0.70514, 0.93678],
[0.3423, 0.70478, 0.93407],
[0.34848, 0.70443, 0.93137],
[0.35453, 0.70407, 0.92867],
[0.36042, 0.70371, 0.92597],
[0.3662, 0.70336, 0.92327],
[0.37184, 0.703, 0.92057],
[0.37737, 0.70265, 0.91788],
[0.38279, 0.70229, 0.91518],
[0.38809, 0.70193, 0.91249],
[0.3933, 0.70158, 0.90979],
[0.39841, 0.70122, 0.9071],
[0.40343, 0.70087, 0.9044],
[0.40835, 0.70052, 0.90171],
[0.4132, 0.70016, 0.89902],
[0.41795, 0.69981, 0.89633],
[0.42262, 0.69945, 0.89363],
[0.42723, 0.6991, 0.89094],
[0.43176, 0.69875, 0.88826],
[0.43621, 0.69839, 0.88557],
[0.44059, 0.69804, 0.88288],
[0.44493, 0.69769, 0.88019],
[0.44919, 0.69734, 0.8775],
[0.45337, 0.69699, 0.87482],
[0.45751, 0.69664, 0.87213],
[0.46159, 0.69629, 0.86945],
[0.46561, 0.69594, 0.86676],
[0.46959, 0.69559, 0.86408],
[0.47349, 0.69524, 0.86139],
[0.47736, 0.69489, 0.85871],
[0.48117, 0.69453, 0.85603],
[0.48494, 0.69418, 0.85335],
[0.48866, 0.69384, 0.85067],
[0.49233, 0.69349, 0.84799],
[0.49596, 0.69315, 0.84531],
[0.49954, 0.6928, 0.84264],
[0.50308, 0.69245, 0.83996],
[0.50659, 0.6921, 0.83728],
[0.51005, 0.69175, 0.8346],
[0.51348, 0.6914, 0.83193],
[0.51686, 0.69105, 0.82925],
[0.5202, 0.69071, 0.82658],
[0.52352, 0.69036, 0.8239],
[0.52679, 0.69002, 0.82123],
[0.53003, 0.68967, 0.81855],
[0.53323, 0.68933, 0.81588],
[0.5364, 0.68898, 0.81321],
[0.53953, 0.68864, 0.81053],
[0.54265, 0.68829, 0.80787],
[0.54572, 0.68795, 0.80519],
[0.54877, 0.6876, 0.80252],
[0.55178, 0.68726, 0.79986],
[0.55476, 0.68691, 0.79718],
[0.55772, 0.68657, 0.79452],
[0.56064, 0.68622, 0.79185],
[0.56355, 0.68588, 0.78919],
[0.56642, 0.68554, 0.78652],
[0.56925, 0.6852, 0.78385],
[0.57208, 0.68485, 0.78118],
[0.57487, 0.68451, 0.77852],
[0.57763, 0.68416, 0.77585],
[0.58038, 0.68382, 0.77318],
[0.5831, 0.68348, 0.77052],
[0.5858, 0.68314, 0.76786],
[0.58846, 0.6828, 0.7652],
[0.59111, 0.68246, 0.76253],
[0.59374, 0.68212, 0.75987],
[0.59634, 0.68177, 0.75721],
[0.59892, 0.68143, 0.75455],
[0.60148, 0.68109, 0.75189],
[0.60402, 0.68075, 0.74922],
[0.60653, 0.68041, 0.74656],
[0.60904, 0.68007, 0.7439],
[0.61151, 0.67974, 0.74124],
[0.61395, 0.6794, 0.73859],
[0.6164, 0.67906, 0.73592],
[0.61882, 0.67872, 0.73327],
[0.62121, 0.67839, 0.73061],
[0.62359, 0.67805, 0.72795],
[0.62595, 0.67771, 0.7253],
[0.62829, 0.67738, 0.72263],
[0.63062, 0.67704, 0.71998],
[0.63293, 0.6767, 0.71733],
[0.63522, 0.67636, 0.71467],
[0.6375, 0.67601, 0.71202],
[0.63977, 0.67567, 0.70938],
[0.64203, 0.67533, 0.70674],
[0.64428, 0.67498, 0.7041],
[0.64653, 0.67463, 0.70147],
[0.64878, 0.67427, 0.69886],
[0.65103, 0.67391, 0.69626],
[0.65329, 0.67353, 0.69367],
[0.65556, 0.67315, 0.6911],
[0.65785, 0.67276, 0.68857],
[0.66016, 0.67235, 0.68605],
[0.6625, 0.67193, 0.68357],
[0.66485, 0.67148, 0.68111],
[0.66726, 0.67103, 0.6787],
[0.6697, 0.67055, 0.67633],
[0.67218, 0.67005, 0.67399],
[0.6747, 0.66953, 0.6717],
[0.67726, 0.66898, 0.66945],
[0.67985, 0.66842, 0.66724],
[0.68249, 0.66784, 0.66507],
[0.68516, 0.66723, 0.66295],
[0.68786, 0.6666, 0.66084],
[0.69058, 0.66596, 0.65878],
[0.69332, 0.66531, 0.65675],
[0.69608, 0.66464, 0.65473],
[0.69884, 0.66396, 0.65274],
[0.70162, 0.66328, 0.65077],
[0.7044, 0.66258, 0.64882],
[0.70718, 0.66187, 0.64687],
[0.70995, 0.66115, 0.64493],
[0.71271, 0.66044, 0.643],
[0.71548, 0.65972, 0.64107],
[0.71822, 0.65899, 0.63915],
[0.72096, 0.65827, 0.63724],
[0.72369, 0.65754, 0.63532],
[0.72641, 0.65681, 0.63341],
[0.72911, 0.65607, 0.6315],
[0.7318, 0.65533, 0.62958],
[0.73448, 0.65459, 0.62767],
[0.73715, 0.65385, 0.62576],
[0.7398, 0.6531, 0.62385],
[0.74244, 0.65236, 0.62193],
[0.74507, 0.65162, 0.62003],
[0.74769, 0.65087, 0.61811],
[0.75029, 0.65012, 0.61621],
[0.75287, 0.64937, 0.61429],
[0.75546, 0.64862, 0.61238],
[0.75803, 0.64786, 0.61047],
[0.76059, 0.64711, 0.60857],
[0.76313, 0.64634, 0.60666],
[0.76567, 0.64558, 0.60475],
[0.76819, 0.64482, 0.60283],
[0.77071, 0.64405, 0.60092],
[0.77321, 0.64329, 0.59902],
[0.7757, 0.64252, 0.59711],
[0.77819, 0.64175, 0.5952],
[0.78066, 0.64098, 0.59329],
[0.78312, 0.64021, 0.59138],
[0.78557, 0.63943, 0.58947],
[0.78801, 0.63865, 0.58756],
[0.79044, 0.63787, 0.58566],
[0.79287, 0.63709, 0.58374],
[0.79528, 0.63631, 0.58183],
[0.79768, 0.63552, 0.57993],
[0.80008, 0.63473, 0.57801],
[0.80246, 0.63395, 0.57612],
[0.80484, 0.63315, 0.5742],
[0.80721, 0.63236, 0.5723],
[0.80956, 0.63156, 0.57038],
[0.81191, 0.63077, 0.56848],
[0.81425, 0.62996, 0.56657],
[0.81658, 0.62916, 0.56466],
[0.8189, 0.62836, 0.56276],
[0.82122, 0.62755, 0.56084],
[0.82352, 0.62674, 0.55893],
[0.82582, 0.62593, 0.55704],
[0.82811, 0.62511, 0.55512],
[0.83039, 0.6243, 0.55321],
[0.83266, 0.62349, 0.55131],
[0.83493, 0.62266, 0.5494],
[0.83719, 0.62184, 0.54749],
[0.83944, 0.62102, 0.54559],
[0.84168, 0.6202, 0.54367],
[0.84391, 0.61937, 0.54176],
[0.84614, 0.61854, 0.53985],
[0.84836, 0.6177, 0.53795],
[0.85057, 0.61687, 0.53604],
[0.85278, 0.61603, 0.53413],
[0.85497, 0.61518, 0.53223],
[0.85716, 0.61434, 0.53031],
[0.85935, 0.6135, 0.52841],
[0.86152, 0.61265, 0.5265],
[0.86369, 0.6118, 0.52458],
[0.86585, 0.61096, 0.52268],
[0.86801, 0.6101, 0.52076],
[0.87015, 0.60925, 0.51886],
[0.8723, 0.60839, 0.51695],
[0.87443, 0.60753, 0.51503],
[0.87656, 0.60666, 0.51312],
[0.87868, 0.60579, 0.51121],
[0.8808, 0.60493, 0.5093],
[0.88291, 0.60406, 0.50739],
[0.88501, 0.60319, 0.50547],
[0.88711, 0.60231, 0.50356],
[0.8892, 0.60143, 0.50166],
[0.89128, 0.60055, 0.49974],
[0.89336, 0.59967, 0.49784],
[0.89544, 0.59878, 0.49591],
[0.8975, 0.59789, 0.49401],
[0.89956, 0.59701, 0.49209],
[0.90162, 0.59611, 0.49018],
[0.90367, 0.59521, 0.48827],
[0.90571, 0.59431, 0.48635],
[0.90775, 0.59341, 0.48443],
[0.90978, 0.59252, 0.48252],
[0.9118, 0.5916, 0.4806],
[0.91383, 0.59069, 0.47869],
[0.91583, 0.58978, 0.47677],
[0.91785, 0.58886, 0.47486],
[0.91985, 0.58794, 0.47293],
[0.92185, 0.58703, 0.47102],
[0.92385, 0.58611, 0.46911],
[0.92583, 0.58518, 0.46719],
[0.92781, 0.58425, 0.46526],
[0.92979, 0.58332, 0.46333],
[0.93176, 0.58238, 0.46142],
[0.93373, 0.58145, 0.4595],
[0.9357, 0.58051, 0.45758],
[0.93765, 0.57957, 0.45566],
[0.93961, 0.57862, 0.45373],
[0.94155, 0.57767, 0.45181],
[0.9435, 0.57673, 0.44989],
[0.94543, 0.57577, 0.44796],
[0.94737, 0.5748, 0.44603],
[0.94929, 0.57384, 0.44412],
[0.95122, 0.57289, 0.44219],
[0.95314, 0.57192, 0.44025],
[0.95505, 0.57094, 0.43833],
[0.95696, 0.56997, 0.4364],
[0.95887, 0.56899, 0.43447],
[0.96077, 0.56802, 0.43254],
[0.96266, 0.56704, 0.43061],
[0.96455, 0.56605, 0.42867],
[0.96644, 0.56505, 0.42674],
]
b_diverging_isoluminant_cjo_70_c25 = bokeh_palette('diverging_isoluminant_cjo_70_c25',diverging_isoluminant_cjo_70_c25)
m_diverging_isoluminant_cjo_70_c25 = mpl_cm('diverging_isoluminant_cjo_70_c25',diverging_isoluminant_cjo_70_c25)
m_diverging_isoluminant_cjo_70_c25_r = mpl_cm('diverging_isoluminant_cjo_70_c25_r',list(reversed(diverging_isoluminant_cjo_70_c25)))
CET_D11 = b_diverging_isoluminant_cjo_70_c25
m_CET_D11 = m_diverging_isoluminant_cjo_70_c25
m_CET_D11_r = m_diverging_isoluminant_cjo_70_c25_r
palette['CET_D11'] = b_diverging_isoluminant_cjo_70_c25
cm['CET_D11'] = m_diverging_isoluminant_cjo_70_c25
cm['CET_D11_r'] = m_diverging_isoluminant_cjo_70_c25_r
register_cmap('cet_CET_D11',m_diverging_isoluminant_cjo_70_c25)
register_cmap('cet_CET_D11_r',m_diverging_isoluminant_cjo_70_c25_r)




cyclic_mygbm_30_95_c78 = [  # cmap_def
[0.93769, 0.33352, 0.94809],
[0.94383, 0.34283, 0.94239],
[0.94939, 0.35275, 0.93613],
[0.95439, 0.36323, 0.92931],
[0.95886, 0.37422, 0.92198],
[0.96283, 0.38558, 0.91416],
[0.96634, 0.39727, 0.90588],
[0.96944, 0.40921, 0.89721],
[0.97216, 0.4213, 0.88817],
[0.97454, 0.4335, 0.87883],
[0.97663, 0.44573, 0.86923],
[0.97845, 0.45797, 0.8594],
[0.98004, 0.47017, 0.84939],
[0.98142, 0.48228, 0.83922],
[0.98262, 0.4943, 0.82894],
[0.98364, 0.5062, 0.81854],
[0.98453, 0.51799, 0.80808],
[0.98527, 0.52964, 0.79753],
[0.98588, 0.54116, 0.78694],
[0.98637, 0.55255, 0.77629],
[0.98676, 0.56382, 0.76559],
[0.98704, 0.57494, 0.75486],
[0.98722, 0.58595, 0.74407],
[0.98733, 0.59683, 0.73326],
[0.98736, 0.60758, 0.72241],
[0.98732, 0.61821, 0.71153],
[0.98723, 0.62871, 0.7006],
[0.9871, 0.63909, 0.68965],
[0.98694, 0.64936, 0.67864],
[0.98677, 0.65949, 0.66762],
[0.9866, 0.6695, 0.65654],
[0.98644, 0.67938, 0.64543],
[0.9863, 0.68916, 0.63428],
[0.98621, 0.69879, 0.62309],
[0.98616, 0.70831, 0.61184],
[0.98617, 0.71772, 0.60055],
[0.98624, 0.727, 0.5892],
[0.98636, 0.73618, 0.57779],
[0.98653, 0.74526, 0.56632],
[0.98675, 0.75426, 0.55476],
[0.98702, 0.76315, 0.54311],
[0.98731, 0.77198, 0.53134],
[0.98761, 0.78074, 0.51946],
[0.98792, 0.78944, 0.50744],
[0.9882, 0.79807, 0.49526],
[0.98845, 0.80666, 0.48291],
[0.98864, 0.81518, 0.47039],
[0.98875, 0.82365, 0.45765],
[0.98875, 0.83206, 0.4447],
[0.98863, 0.84039, 0.43151],
[0.98834, 0.84863, 0.41804],
[0.98786, 0.85677, 0.40433],
[0.98715, 0.86476, 0.39033],
[0.98616, 0.87259, 0.37607],
[0.98485, 0.8802, 0.36152],
[0.98316, 0.88754, 0.3467],
[0.98105, 0.89458, 0.33163],
[0.97845, 0.90123, 0.31637],
[0.97534, 0.90744, 0.30092],
[0.97166, 0.91315, 0.28537],
[0.96737, 0.91831, 0.26983],
[0.96244, 0.92285, 0.25432],
[0.95686, 0.92672, 0.23895],
[0.95064, 0.9299, 0.22388],
[0.94377, 0.93236, 0.2092],
[0.93628, 0.9341, 0.19506],
[0.9282, 0.93512, 0.18153],
[0.91959, 0.93544, 0.16884],
[0.91047, 0.9351, 0.15697],
[0.90091, 0.93414, 0.1461],
[0.89097, 0.93263, 0.13623],
[0.8807, 0.93061, 0.12747],
[0.87014, 0.92815, 0.11977],
[0.85937, 0.92531, 0.11315],
[0.8484, 0.92216, 0.10742],
[0.83729, 0.91874, 0.10257],
[0.82605, 0.91511, 0.098502],
[0.81472, 0.91132, 0.095091],
[0.80332, 0.90739, 0.092162],
[0.79187, 0.90336, 0.089659],
[0.78036, 0.89925, 0.087518],
[0.76881, 0.89508, 0.08551],
[0.75723, 0.89086, 0.083837],
[0.74562, 0.88662, 0.082243],
[0.73398, 0.88235, 0.080673],
[0.7223, 0.87807, 0.079194],
[0.7106, 0.87376, 0.077792],
[0.69886, 0.86946, 0.076415],
[0.6871, 0.86514, 0.075063],
[0.67529, 0.86082, 0.073757],
[0.66346, 0.85649, 0.072319],
[0.65156, 0.85215, 0.071005],
[0.63963, 0.8478, 0.069678],
[0.62764, 0.84345, 0.068313],
[0.6156, 0.83909, 0.066946],
[0.60351, 0.83473, 0.065602],
[0.59134, 0.83035, 0.064284],
[0.57911, 0.82597, 0.063016],
[0.56681, 0.82158, 0.061599],
[0.55443, 0.81718, 0.060374],
[0.54195, 0.81278, 0.059088],
[0.5294, 0.80837, 0.057695],
[0.51672, 0.80395, 0.056522],
[0.50395, 0.79952, 0.055189],
[0.49106, 0.79508, 0.053903],
[0.47801, 0.79064, 0.052644],
[0.46484, 0.78619, 0.051424],
[0.45151, 0.78173, 0.050257],
[0.43803, 0.77726, 0.04922],
[0.42437, 0.77277, 0.04812],
[0.41056, 0.76827, 0.047322],
[0.39656, 0.76375, 0.04674],
[0.38239, 0.75922, 0.046427],
[0.36808, 0.75466, 0.046596],
[0.35361, 0.75006, 0.047299],
[0.33906, 0.74543, 0.04874],
[0.32443, 0.74075, 0.050897],
[0.30984, 0.73602, 0.054069],
[0.29532, 0.73122, 0.058336],
[0.28105, 0.72634, 0.063783],
[0.26717, 0.72137, 0.070322],
[0.25387, 0.71631, 0.077992],
[0.24134, 0.71112, 0.08687],
[0.22981, 0.7058, 0.096608],
[0.21961, 0.70035, 0.10741],
[0.21092, 0.69475, 0.11899],
[0.204, 0.68901, 0.13129],
[0.19894, 0.6831, 0.14422],
[0.19593, 0.67703, 0.15768],
[0.19486, 0.67081, 0.17161],
[0.1956, 0.66443, 0.18594],
[0.19795, 0.65791, 0.2005],
[0.20163, 0.65125, 0.21533],
[0.20639, 0.64446, 0.2303],
[0.21183, 0.63756, 0.24538],
[0.21771, 0.63057, 0.26052],
[0.22381, 0.62348, 0.27565],
[0.22992, 0.61632, 0.29071],
[0.23593, 0.6091, 0.30574],
[0.24162, 0.60182, 0.32064],
[0.24693, 0.5945, 0.33543],
[0.25184, 0.58716, 0.35008],
[0.25622, 0.5798, 0.36459],
[0.26011, 0.57242, 0.37897],
[0.26346, 0.56501, 0.3932],
[0.26624, 0.55762, 0.4073],
[0.26846, 0.55021, 0.42127],
[0.27013, 0.5428, 0.43513],
[0.27122, 0.53539, 0.44886],
[0.27173, 0.52798, 0.46247],
[0.2717, 0.52057, 0.476],
[0.27112, 0.51317, 0.48942],
[0.26997, 0.50576, 0.50275],
[0.26823, 0.49837, 0.516],
[0.26598, 0.49096, 0.52919],
[0.26316, 0.48352, 0.54228],
[0.25982, 0.4761, 0.55529],
[0.25594, 0.46864, 0.56822],
[0.25162, 0.46117, 0.58105],
[0.2468, 0.45365, 0.5938],
[0.24161, 0.44609, 0.60643],
[0.23605, 0.43849, 0.61894],
[0.23017, 0.43081, 0.63131],
[0.2241, 0.42303, 0.64353],
[0.21793, 0.41517, 0.6556],
[0.2117, 0.40718, 0.66749],
[0.2055, 0.39906, 0.67919],
[0.19945, 0.39079, 0.6907],
[0.19367, 0.38234, 0.70201],
[0.18818, 0.37372, 0.71312],
[0.183, 0.36488, 0.72404],
[0.17829, 0.35585, 0.73477],
[0.17392, 0.34657, 0.7453],
[0.16999, 0.33707, 0.75566],
[0.16639, 0.32732, 0.76586],
[0.16312, 0.31735, 0.7759],
[0.16005, 0.30712, 0.78581],
[0.15724, 0.29667, 0.79557],
[0.15457, 0.28595, 0.80522],
[0.15202, 0.27505, 0.81474],
[0.14966, 0.26395, 0.82414],
[0.14744, 0.25264, 0.8334],
[0.14554, 0.24118, 0.84252],
[0.14402, 0.2296, 0.85145],
[0.14312, 0.218, 0.86019],
[0.14305, 0.20639, 0.86869],
[0.14404, 0.19481, 0.87691],
[0.1463, 0.18336, 0.88484],
[0.15007, 0.17219, 0.89241],
[0.15537, 0.1614, 0.8996],
[0.1623, 0.15103, 0.90637],
[0.17075, 0.14136, 0.9127],
[0.18062, 0.13244, 0.91856],
[0.19173, 0.12446, 0.92396],
[0.20389, 0.11765, 0.92889],
[0.21681, 0.11214, 0.93335],
[0.23028, 0.10794, 0.93739],
[0.24417, 0.10525, 0.94101],
[0.25828, 0.10403, 0.94425],
[0.27246, 0.10417, 0.94716],
[0.28659, 0.1056, 0.94977],
[0.30059, 0.10807, 0.95212],
[0.3144, 0.11153, 0.95426],
[0.32796, 0.11565, 0.95622],
[0.34126, 0.12031, 0.95803],
[0.35426, 0.12544, 0.95973],
[0.36698, 0.13084, 0.96134],
[0.37942, 0.13637, 0.96288],
[0.3916, 0.1421, 0.96436],
[0.40352, 0.14786, 0.96581],
[0.4152, 0.15363, 0.96722],
[0.42669, 0.15941, 0.9686],
[0.43799, 0.16515, 0.96996],
[0.44913, 0.17088, 0.97129],
[0.46013, 0.1765, 0.97261],
[0.47106, 0.18204, 0.97389],
[0.48191, 0.18751, 0.97514],
[0.49273, 0.1928, 0.97635],
[0.50357, 0.19798, 0.97752],
[0.51446, 0.20302, 0.97863],
[0.52541, 0.2079, 0.97969],
[0.53647, 0.21259, 0.98067],
[0.54767, 0.21709, 0.98157],
[0.55903, 0.22137, 0.98238],
[0.57057, 0.22542, 0.9831],
[0.5823, 0.22925, 0.98372],
[0.59423, 0.23284, 0.98424],
[0.60636, 0.23623, 0.98465],
[0.61868, 0.23935, 0.98497],
[0.63116, 0.24227, 0.98518],
[0.64379, 0.24496, 0.9853],
[0.65656, 0.2475, 0.98533],
[0.66943, 0.24982, 0.98527],
[0.68237, 0.25203, 0.98515],
[0.69537, 0.25408, 0.98496],
[0.70839, 0.25601, 0.98471],
[0.72139, 0.25786, 0.98441],
[0.73437, 0.25968, 0.98405],
[0.74728, 0.26143, 0.98365],
[0.76012, 0.26319, 0.98321],
[0.77286, 0.26494, 0.98271],
[0.78548, 0.26676, 0.98216],
[0.79794, 0.26867, 0.98153],
[0.81023, 0.27069, 0.98082],
[0.82233, 0.27285, 0.98001],
[0.83418, 0.27525, 0.97905],
[0.84576, 0.27791, 0.97793],
[0.85702, 0.28089, 0.97661],
[0.86794, 0.2843, 0.97504],
[0.87845, 0.28819, 0.97317],
[0.88852, 0.29263, 0.97097],
[0.8981, 0.29768, 0.96837],
[0.90717, 0.3034, 0.96533],
[0.91567, 0.30984, 0.96182],
[0.92361, 0.317, 0.9578],
[0.93095, 0.32489, 0.95323],
]
b_cyclic_mygbm_30_95_c78 = bokeh_palette('cyclic_mygbm_30_95_c78',cyclic_mygbm_30_95_c78)
m_cyclic_mygbm_30_95_c78 = mpl_cm('cyclic_mygbm_30_95_c78',cyclic_mygbm_30_95_c78)
m_cyclic_mygbm_30_95_c78_r = mpl_cm('cyclic_mygbm_30_95_c78_r',list(reversed(cyclic_mygbm_30_95_c78)))
CET_C2 = b_cyclic_mygbm_30_95_c78
m_CET_C2 = m_cyclic_mygbm_30_95_c78
m_CET_C2_r = m_cyclic_mygbm_30_95_c78_r
palette['CET_C2'] = b_cyclic_mygbm_30_95_c78
cm['CET_C2'] = m_cyclic_mygbm_30_95_c78
cm['CET_C2_r'] = m_cyclic_mygbm_30_95_c78_r
register_cmap('cet_CET_C2',m_cyclic_mygbm_30_95_c78)
register_cmap('cet_CET_C2_r',m_cyclic_mygbm_30_95_c78_r)




cyclic_mrybm_35_75_c68_s25 = [  # cmap_def
[0.24409, 0.24544, 0.94131],
[0.25448, 0.24134, 0.94753],
[0.26529, 0.23866, 0.95301],
[0.27652, 0.2375, 0.95778],
[0.28798, 0.23773, 0.96185],
[0.29964, 0.23933, 0.9653],
[0.31135, 0.24218, 0.96816],
[0.32305, 0.24613, 0.9705],
[0.33464, 0.25106, 0.97238],
[0.34606, 0.25676, 0.97386],
[0.35733, 0.26313, 0.97502],
[0.36835, 0.27, 0.9759],
[0.37916, 0.27724, 0.97657],
[0.38973, 0.28476, 0.97706],
[0.40009, 0.2925, 0.97742],
[0.41026, 0.30036, 0.97768],
[0.42023, 0.3083, 0.97787],
[0.43006, 0.31627, 0.978],
[0.43977, 0.32421, 0.97811],
[0.4494, 0.33213, 0.9782],
[0.459, 0.33997, 0.97829],
[0.46859, 0.34772, 0.97838],
[0.47825, 0.35536, 0.97849],
[0.48802, 0.36282, 0.97864],
[0.49795, 0.37015, 0.97883],
[0.50807, 0.3773, 0.97906],
[0.51845, 0.38426, 0.97935],
[0.52912, 0.39102, 0.97969],
[0.5401, 0.39755, 0.9801],
[0.55143, 0.40386, 0.98057],
[0.56311, 0.40995, 0.98111],
[0.57513, 0.41579, 0.98173],
[0.58751, 0.4214, 0.98242],
[0.60022, 0.42681, 0.98317],
[0.61322, 0.43201, 0.984],
[0.6265, 0.437, 0.98489],
[0.64002, 0.44182, 0.98583],
[0.65371, 0.44647, 0.98682],
[0.66757, 0.45099, 0.98786],
[0.68153, 0.4554, 0.98894],
[0.69558, 0.4597, 0.99005],
[0.70967, 0.46392, 0.99118],
[0.72376, 0.46807, 0.99233],
[0.73785, 0.47218, 0.9935],
[0.7519, 0.47624, 0.99467],
[0.76589, 0.48026, 0.99582],
[0.77979, 0.48425, 0.99696],
[0.79361, 0.48821, 0.99807],
[0.80729, 0.49211, 0.99913],
[0.82083, 0.49595, 1],
[0.8342, 0.49972, 1],
[0.84736, 0.50339, 1],
[0.86029, 0.50693, 1],
[0.87292, 0.5103, 1],
[0.8852, 0.51345, 1],
[0.89709, 0.51632, 1],
[0.90851, 0.51889, 1],
[0.91939, 0.52104, 1],
[0.92968, 0.52273, 0.99852],
[0.9393, 0.52391, 0.996],
[0.94819, 0.5245, 0.99274],
[0.95631, 0.52446, 0.98864],
[0.9636, 0.52374, 0.98368],
[0.97003, 0.52229, 0.97779],
[0.97561, 0.52012, 0.97096],
[0.98033, 0.51722, 0.96319],
[0.9842, 0.51357, 0.95448],
[0.98725, 0.50922, 0.94488],
[0.98954, 0.5042, 0.93442],
[0.99112, 0.49856, 0.9232],
[0.99204, 0.49233, 0.91126],
[0.99239, 0.48561, 0.89871],
[0.99221, 0.47844, 0.88563],
[0.99158, 0.47088, 0.87209],
[0.99055, 0.46298, 0.85819],
[0.98919, 0.45483, 0.84399],
[0.98754, 0.44642, 0.82958],
[0.98565, 0.43783, 0.81499],
[0.98354, 0.42907, 0.80029],
[0.98125, 0.42019, 0.78551],
[0.97879, 0.4112, 0.77068],
[0.97618, 0.40212, 0.75583],
[0.97344, 0.39297, 0.74098],
[0.97057, 0.38376, 0.72614],
[0.96757, 0.3745, 0.71131],
[0.96444, 0.36519, 0.6965],
[0.96118, 0.35586, 0.6817],
[0.95779, 0.3465, 0.66694],
[0.95425, 0.33716, 0.65219],
[0.95055, 0.3278, 0.63745],
[0.94668, 0.31849, 0.62271],
[0.94264, 0.30923, 0.60798],
[0.93839, 0.3, 0.59322],
[0.93394, 0.29087, 0.57844],
[0.92929, 0.28185, 0.56366],
[0.92441, 0.27294, 0.54883],
[0.91931, 0.26418, 0.53396],
[0.91398, 0.25557, 0.51908],
[0.90842, 0.24712, 0.50415],
[0.90264, 0.23883, 0.48918],
[0.89665, 0.23073, 0.47419],
[0.89044, 0.22281, 0.45918],
[0.88405, 0.21501, 0.44415],
[0.87747, 0.2074, 0.42911],
[0.87072, 0.19987, 0.41407],
[0.86383, 0.1925, 0.39903],
[0.85679, 0.18523, 0.38401],
[0.84962, 0.17801, 0.369],
[0.84235, 0.17082, 0.35401],
[0.83498, 0.16373, 0.33907],
[0.82753, 0.15664, 0.32416],
[0.82001, 0.14962, 0.30933],
[0.81245, 0.14261, 0.29452],
[0.80486, 0.13567, 0.27982],
[0.79725, 0.12885, 0.26518],
[0.78966, 0.12217, 0.25066],
[0.78212, 0.11578, 0.23628],
[0.77464, 0.1097, 0.22201],
[0.7673, 0.104, 0.20795],
[0.76012, 0.098945, 0.19412],
[0.75317, 0.094721, 0.18049],
[0.74652, 0.091421, 0.16723],
[0.74022, 0.089258, 0.15426],
[0.73434, 0.088445, 0.14174],
[0.72894, 0.089069, 0.12963],
[0.7241, 0.091195, 0.118],
[0.71988, 0.094767, 0.10689],
[0.71631, 0.099636, 0.096236],
[0.71343, 0.1058, 0.086225],
[0.71128, 0.11302, 0.076774],
[0.70983, 0.12104, 0.068001],
[0.7091, 0.12989, 0.05981],
[0.70904, 0.1392, 0.052084],
[0.70963, 0.14902, 0.04517],
[0.71081, 0.15914, 0.038855],
[0.71251, 0.1695, 0.033292],
[0.7147, 0.17995, 0.028947],
[0.71729, 0.1905, 0.02547],
[0.72021, 0.20107, 0.022733],
[0.72342, 0.21166, 0.020622],
[0.72685, 0.22214, 0.019034],
[0.73046, 0.23256, 0.017876],
[0.73419, 0.24288, 0.017071],
[0.73802, 0.25311, 0.016547],
[0.74191, 0.26323, 0.016249],
[0.74585, 0.27322, 0.016125],
[0.74979, 0.28311, 0.016137],
[0.75375, 0.29288, 0.01625],
[0.75769, 0.30257, 0.01644],
[0.76161, 0.31217, 0.016684],
[0.7655, 0.32171, 0.016989],
[0.76932, 0.33117, 0.017296],
[0.7731, 0.34058, 0.0176],
[0.77681, 0.34994, 0.017896],
[0.78044, 0.35925, 0.018177],
[0.784, 0.36855, 0.018435],
[0.78746, 0.37783, 0.018665],
[0.79082, 0.38709, 0.018861],
[0.79409, 0.39636, 0.019014],
[0.79724, 0.40564, 0.019122],
[0.80029, 0.4149, 0.019179],
[0.80322, 0.42418, 0.019183],
[0.80605, 0.43346, 0.019132],
[0.80876, 0.44276, 0.019026],
[0.81136, 0.45205, 0.018866],
[0.81386, 0.46134, 0.018654],
[0.81626, 0.47063, 0.018394],
[0.81857, 0.47992, 0.01809],
[0.8208, 0.48918, 0.017747],
[0.82295, 0.49846, 0.017369],
[0.82503, 0.50771, 0.016963],
[0.82704, 0.51694, 0.016575],
[0.82898, 0.52615, 0.016225],
[0.83084, 0.53532, 0.015949],
[0.83262, 0.54446, 0.015792],
[0.83431, 0.55354, 0.01582],
[0.8359, 0.56257, 0.016118],
[0.83738, 0.5715, 0.016798],
[0.8387, 0.58035, 0.018004],
[0.83985, 0.58906, 0.019918],
[0.84077, 0.59761, 0.022763],
[0.84143, 0.60595, 0.02681],
[0.84177, 0.61404, 0.032376],
[0.84173, 0.62182, 0.040011],
[0.84125, 0.62922, 0.04912],
[0.84026, 0.63619, 0.059305],
[0.83868, 0.64264, 0.070471],
[0.83647, 0.6485, 0.08264],
[0.83356, 0.65372, 0.095596],
[0.82989, 0.65824, 0.10926],
[0.82544, 0.66201, 0.12343],
[0.82017, 0.66498, 0.13819],
[0.81408, 0.66715, 0.15321],
[0.80717, 0.66852, 0.16856],
[0.79947, 0.66908, 0.18405],
[0.79098, 0.66887, 0.19961],
[0.78178, 0.66795, 0.2152],
[0.7719, 0.66635, 0.23072],
[0.76139, 0.66416, 0.2461],
[0.75032, 0.66143, 0.26133],
[0.73873, 0.65825, 0.27636],
[0.72668, 0.65467, 0.29113],
[0.71421, 0.65077, 0.30567],
[0.70137, 0.64662, 0.31995],
[0.68818, 0.64226, 0.33397],
[0.67466, 0.63773, 0.34774],
[0.66084, 0.6331, 0.36127],
[0.64671, 0.62838, 0.37458],
[0.63228, 0.6236, 0.38765],
[0.61753, 0.61878, 0.40056],
[0.60245, 0.61393, 0.41326],
[0.58703, 0.60909, 0.42578],
[0.57126, 0.60423, 0.43817],
[0.55511, 0.59936, 0.4504],
[0.53858, 0.59449, 0.4625],
[0.5216, 0.58962, 0.47451],
[0.5042, 0.58475, 0.4864],
[0.48632, 0.57986, 0.49821],
[0.46796, 0.57493, 0.50993],
[0.44912, 0.56996, 0.5216],
[0.42978, 0.56493, 0.53322],
[0.40998, 0.55983, 0.5448],
[0.38972, 0.55461, 0.55635],
[0.36908, 0.54926, 0.56789],
[0.3481, 0.54375, 0.57943],
[0.32688, 0.53805, 0.59098],
[0.30556, 0.53211, 0.60256],
[0.28424, 0.52592, 0.61416],
[0.26316, 0.51945, 0.62583],
[0.2425, 0.51265, 0.63755],
[0.22257, 0.50551, 0.64934],
[0.20368, 0.49802, 0.66119],
[0.18619, 0.49012, 0.67313],
[0.17051, 0.48183, 0.68514],
[0.15712, 0.47312, 0.69723],
[0.14638, 0.46402, 0.70941],
[0.13859, 0.4545, 0.72164],
[0.1338, 0.44458, 0.73395],
[0.1319, 0.43426, 0.7463],
[0.13253, 0.42359, 0.75869],
[0.13519, 0.4126, 0.77109],
[0.1393, 0.40128, 0.7835],
[0.1445, 0.38969, 0.79588],
[0.15024, 0.3779, 0.80821],
[0.1563, 0.36594, 0.82045],
[0.16246, 0.35387, 0.83257],
[0.16864, 0.34179, 0.84452],
[0.17484, 0.32979, 0.85623],
[0.18105, 0.31794, 0.86767],
[0.18746, 0.30639, 0.87875],
[0.19408, 0.29524, 0.88943],
[0.20101, 0.28465, 0.89962],
[0.20848, 0.27478, 0.90926],
[0.2165, 0.26579, 0.9183],
[0.22506, 0.25778, 0.92668],
[0.2343, 0.25098, 0.93436],
]
b_cyclic_mrybm_35_75_c68_s25 = bokeh_palette('cyclic_mrybm_35_75_c68_s25',cyclic_mrybm_35_75_c68_s25)
m_cyclic_mrybm_35_75_c68_s25 = mpl_cm('cyclic_mrybm_35_75_c68_s25',cyclic_mrybm_35_75_c68_s25)
m_cyclic_mrybm_35_75_c68_s25_r = mpl_cm('cyclic_mrybm_35_75_c68_s25_r',list(reversed(cyclic_mrybm_35_75_c68_s25)))
CET_C1s = b_cyclic_mrybm_35_75_c68_s25
m_CET_C1s = m_cyclic_mrybm_35_75_c68_s25
m_CET_C1s_r = m_cyclic_mrybm_35_75_c68_s25_r
palette['CET_C1s'] = b_cyclic_mrybm_35_75_c68_s25
cm['CET_C1s'] = m_cyclic_mrybm_35_75_c68_s25
cm['CET_C1s_r'] = m_cyclic_mrybm_35_75_c68_s25_r
register_cmap('cet_CET_C1s',m_cyclic_mrybm_35_75_c68_s25)
register_cmap('cet_CET_C1s_r',m_cyclic_mrybm_35_75_c68_s25_r)




linear_bmw_5_95_c86 = [  # cmap_def
[0, 0.0078689, 0.29563],
[0, 0.01018, 0.30347],
[0, 0.01249, 0.31136],
[0, 0.014296, 0.31929],
[0, 0.015847, 0.32724],
[0, 0.01713, 0.33526],
[0, 0.018143, 0.3433],
[0, 0.018877, 0.35137],
[0, 0.019332, 0.35948],
[0, 0.019537, 0.36762],
[0, 0.019696, 0.37581],
[0, 0.019848, 0.38402],
[0, 0.019989, 0.39226],
[0.0002784, 0.020121, 0.40055],
[0.0013175, 0.020246, 0.40884],
[0.0023914, 0.020361, 0.41718],
[0.0034883, 0.020469, 0.42554],
[0.004603, 0.020572, 0.43392],
[0.0057432, 0.020668, 0.44236],
[0.0068995, 0.020759, 0.4508],
[0.0080586, 0.020848, 0.45928],
[0.0092385, 0.020933, 0.46778],
[0.010387, 0.021014, 0.47632],
[0.011786, 0.021099, 0.48487],
[0.012932, 0.021185, 0.49346],
[0.014093, 0.021272, 0.50208],
[0.015228, 0.021363, 0.51072],
[0.016327, 0.021464, 0.51939],
[0.017385, 0.021575, 0.52808],
[0.018402, 0.021697, 0.53681],
[0.019367, 0.021832, 0.54555],
[0.020271, 0.021984, 0.55432],
[0.021093, 0.022157, 0.56312],
[0.021816, 0.022358, 0.57193],
[0.022431, 0.022589, 0.58078],
[0.022925, 0.022854, 0.58964],
[0.02328, 0.023157, 0.59854],
[0.023474, 0.023505, 0.60747],
[0.023484, 0.023905, 0.6164],
[0.023283, 0.024365, 0.62537],
[0.022843, 0.024892, 0.63435],
[0.02213, 0.025497, 0.64335],
[0.021108, 0.026192, 0.6524],
[0.019919, 0.026976, 0.66144],
[0.018888, 0.027836, 0.67043],
[0.018212, 0.028767, 0.67933],
[0.017924, 0.029768, 0.68816],
[0.018038, 0.030843, 0.69692],
[0.018576, 0.03199, 0.7056],
[0.019563, 0.033222, 0.7142],
[0.021022, 0.034609, 0.72273],
[0.022976, 0.036126, 0.73118],
[0.02546, 0.037612, 0.73955],
[0.028511, 0.039192, 0.74783],
[0.032139, 0.040867, 0.75603],
[0.036548, 0.042546, 0.76415],
[0.0414, 0.04435, 0.77219],
[0.046471, 0.046119, 0.78014],
[0.051815, 0.048019, 0.78801],
[0.057373, 0.050007, 0.7958],
[0.063189, 0.051954, 0.8035],
[0.069106, 0.054002, 0.81111],
[0.075029, 0.056167, 0.81863],
[0.081171, 0.05824, 0.82607],
[0.087403, 0.060492, 0.83342],
[0.093627, 0.062774, 0.84068],
[0.099985, 0.064997, 0.84785],
[0.10645, 0.067338, 0.85494],
[0.11293, 0.069742, 0.86194],
[0.11943, 0.072103, 0.86884],
[0.12606, 0.074573, 0.87564],
[0.13279, 0.077004, 0.88234],
[0.13992, 0.079441, 0.88889],
[0.1477, 0.081844, 0.89524],
[0.15601, 0.084044, 0.90139],
[0.16477, 0.08607, 0.90734],
[0.17399, 0.088131, 0.91308],
[0.18361, 0.090027, 0.9186],
[0.19359, 0.091859, 0.92392],
[0.20383, 0.093506, 0.92903],
[0.21429, 0.095185, 0.93394],
[0.22495, 0.096634, 0.93864],
[0.23584, 0.098158, 0.94313],
[0.24685, 0.099485, 0.94742],
[0.25802, 0.10076, 0.95149],
[0.26936, 0.10196, 0.95534],
[0.28076, 0.1031, 0.95898],
[0.2923, 0.1041, 0.9624],
[0.3039, 0.10503, 0.96562],
[0.31558, 0.10597, 0.96862],
[0.3273, 0.10679, 0.97142],
[0.3391, 0.10749, 0.974],
[0.35094, 0.10811, 0.97636],
[0.36282, 0.1087, 0.9785],
[0.37476, 0.10925, 0.98043],
[0.38671, 0.1097, 0.98214],
[0.3987, 0.11006, 0.98364],
[0.41071, 0.11034, 0.98492],
[0.42271, 0.11056, 0.98599],
[0.43474, 0.11072, 0.98685],
[0.44667, 0.11084, 0.98757],
[0.45843, 0.11094, 0.98824],
[0.47001, 0.11104, 0.98888],
[0.48139, 0.11114, 0.9895],
[0.49262, 0.11123, 0.99008],
[0.50371, 0.11133, 0.99064],
[0.51465, 0.11141, 0.99117],
[0.52546, 0.11149, 0.99168],
[0.53615, 0.11157, 0.99216],
[0.54673, 0.11164, 0.99261],
[0.55721, 0.1117, 0.99304],
[0.56758, 0.11176, 0.99343],
[0.57785, 0.11181, 0.9938],
[0.58804, 0.11186, 0.99415],
[0.59816, 0.1119, 0.99446],
[0.6082, 0.11193, 0.99475],
[0.61815, 0.11196, 0.99501],
[0.62805, 0.11197, 0.99525],
[0.63787, 0.11198, 0.99545],
[0.64764, 0.11198, 0.99563],
[0.65735, 0.11197, 0.99579],
[0.66699, 0.11195, 0.99591],
[0.6766, 0.11192, 0.99601],
[0.68614, 0.11188, 0.99608],
[0.69564, 0.11183, 0.99613],
[0.7051, 0.11177, 0.99614],
[0.71451, 0.1117, 0.99613],
[0.72388, 0.11162, 0.9961],
[0.73317, 0.11164, 0.99604],
[0.7423, 0.11194, 0.99598],
[0.75124, 0.11262, 0.99593],
[0.75998, 0.11368, 0.99588],
[0.76854, 0.1151, 0.99583],
[0.77691, 0.11698, 0.9958],
[0.7851, 0.11925, 0.99576],
[0.79312, 0.12184, 0.99574],
[0.80096, 0.12485, 0.99572],
[0.80864, 0.12819, 0.9957],
[0.81615, 0.13184, 0.99569],
[0.8235, 0.13575, 0.99568],
[0.83069, 0.14, 0.99568],
[0.83772, 0.14456, 0.99569],
[0.84458, 0.14935, 0.9957],
[0.85129, 0.15434, 0.99571],
[0.85785, 0.15956, 0.99573],
[0.86427, 0.16499, 0.99576],
[0.87053, 0.17062, 0.99578],
[0.87664, 0.17639, 0.99582],
[0.88261, 0.18234, 0.99586],
[0.88844, 0.18847, 0.9959],
[0.89411, 0.19472, 0.99595],
[0.89964, 0.20104, 0.996],
[0.90503, 0.20754, 0.99606],
[0.91028, 0.21413, 0.99612],
[0.91538, 0.22082, 0.99619],
[0.92035, 0.22757, 0.99626],
[0.92518, 0.23444, 0.99634],
[0.92985, 0.24141, 0.9964],
[0.93435, 0.24851, 0.99646],
[0.9387, 0.25569, 0.9965],
[0.9429, 0.26299, 0.99653],
[0.94693, 0.27039, 0.99654],
[0.9508, 0.27784, 0.99654],
[0.95452, 0.28537, 0.99653],
[0.95808, 0.29301, 0.9965],
[0.96146, 0.30071, 0.99646],
[0.96469, 0.30849, 0.9964],
[0.96776, 0.31632, 0.99633],
[0.97067, 0.32419, 0.99624],
[0.97343, 0.33214, 0.99614],
[0.97602, 0.34012, 0.99603],
[0.97845, 0.34814, 0.9959],
[0.98073, 0.35621, 0.99576],
[0.98283, 0.36432, 0.9956],
[0.98477, 0.37248, 0.99543],
[0.98655, 0.38069, 0.99525],
[0.98816, 0.38891, 0.99505],
[0.98962, 0.39715, 0.99483],
[0.9909, 0.40545, 0.9946],
[0.99203, 0.41375, 0.99435],
[0.99299, 0.42208, 0.99409],
[0.99377, 0.43046, 0.99382],
[0.99438, 0.43884, 0.99353],
[0.99482, 0.44726, 0.99322],
[0.9951, 0.45569, 0.99291],
[0.99529, 0.46407, 0.99259],
[0.99544, 0.47235, 0.99229],
[0.99558, 0.48052, 0.99201],
[0.99572, 0.48859, 0.99175],
[0.99585, 0.49654, 0.99152],
[0.99597, 0.50442, 0.99132],
[0.99608, 0.51219, 0.99113],
[0.99619, 0.51988, 0.99097],
[0.99629, 0.52749, 0.99084],
[0.99638, 0.53502, 0.99072],
[0.99647, 0.54248, 0.99063],
[0.99655, 0.54987, 0.99057],
[0.99662, 0.55719, 0.99053],
[0.99669, 0.56443, 0.99051],
[0.99675, 0.57162, 0.99052],
[0.9968, 0.57875, 0.99055],
[0.99685, 0.58582, 0.9906],
[0.99689, 0.59284, 0.99068],
[0.99693, 0.5998, 0.99078],
[0.99696, 0.60672, 0.9909],
[0.99698, 0.61357, 0.99105],
[0.997, 0.6204, 0.99122],
[0.99701, 0.62716, 0.99142],
[0.99702, 0.6339, 0.99164],
[0.99702, 0.64058, 0.99189],
[0.99701, 0.64722, 0.99216],
[0.997, 0.65381, 0.99245],
[0.99699, 0.66038, 0.99277],
[0.99697, 0.66692, 0.9931],
[0.99697, 0.67341, 0.99343],
[0.99699, 0.67985, 0.99373],
[0.99703, 0.68627, 0.99402],
[0.99708, 0.69265, 0.99429],
[0.99714, 0.69899, 0.99455],
[0.9972, 0.70531, 0.99479],
[0.99728, 0.7116, 0.99502],
[0.99736, 0.71786, 0.99524],
[0.99745, 0.72409, 0.99544],
[0.99754, 0.7303, 0.99563],
[0.99762, 0.73648, 0.99582],
[0.99771, 0.74266, 0.99599],
[0.9978, 0.7488, 0.99615],
[0.99789, 0.75492, 0.9963],
[0.99798, 0.76103, 0.99645],
[0.99806, 0.76711, 0.99658],
[0.99814, 0.77318, 0.99671],
[0.99821, 0.77923, 0.99682],
[0.99828, 0.78527, 0.99693],
[0.99835, 0.79129, 0.99703],
[0.99841, 0.79729, 0.99712],
[0.99846, 0.80328, 0.9972],
[0.99851, 0.80926, 0.99727],
[0.99854, 0.81522, 0.99734],
[0.99857, 0.82117, 0.9974],
[0.9986, 0.82711, 0.99745],
[0.99861, 0.83304, 0.99749],
[0.99862, 0.83895, 0.99753],
[0.99861, 0.84486, 0.99756],
[0.9986, 0.85075, 0.99758],
[0.99858, 0.85663, 0.9976],
[0.99854, 0.86251, 0.99761],
[0.9985, 0.86838, 0.99761],
[0.99845, 0.87423, 0.9976],
[0.99838, 0.88007, 0.99759],
[0.99831, 0.88591, 0.99757],
[0.99822, 0.89174, 0.99755],
[0.99812, 0.89756, 0.99752],
[0.99801, 0.90339, 0.99748],
[0.99789, 0.90918, 0.99744],
[0.99775, 0.91498, 0.99739],
[0.9976, 0.92077, 0.99733],
]
b_linear_bmw_5_95_c86 = bokeh_palette('linear_bmw_5_95_c86',linear_bmw_5_95_c86)
m_linear_bmw_5_95_c86 = mpl_cm('linear_bmw_5_95_c86',linear_bmw_5_95_c86)
m_linear_bmw_5_95_c86_r = mpl_cm('linear_bmw_5_95_c86_r',list(reversed(linear_bmw_5_95_c86)))
CET_L7 = b_linear_bmw_5_95_c86
m_CET_L7 = m_linear_bmw_5_95_c86
m_CET_L7_r = m_linear_bmw_5_95_c86_r
palette['CET_L7'] = b_linear_bmw_5_95_c86
cm['CET_L7'] = m_linear_bmw_5_95_c86
cm['CET_L7_r'] = m_linear_bmw_5_95_c86_r
register_cmap('cet_CET_L7',m_linear_bmw_5_95_c86)
register_cmap('cet_CET_L7_r',m_linear_bmw_5_95_c86_r)




linear_grey_0_100_c0 = [  # cmap_def
[0, 0, 0],
[0.0055606, 0.0055631, 0.0055625],
[0.011212, 0.011219, 0.011217],
[0.016877, 0.016885, 0.016883],
[0.022438, 0.022448, 0.022445],
[0.027998, 0.028011, 0.028008],
[0.03354, 0.033554, 0.033551],
[0.039316, 0.039333, 0.039329],
[0.0447, 0.044719, 0.044714],
[0.049695, 0.049713, 0.049709],
[0.054322, 0.054343, 0.054338],
[0.058797, 0.058821, 0.058815],
[0.063041, 0.063062, 0.063057],
[0.066996, 0.067019, 0.067014],
[0.070827, 0.070853, 0.070846],
[0.074487, 0.074511, 0.074505],
[0.077967, 0.077992, 0.077986],
[0.081449, 0.081477, 0.08147],
[0.084728, 0.084752, 0.084746],
[0.087913, 0.087939, 0.087933],
[0.091003, 0.091031, 0.091024],
[0.094004, 0.094034, 0.094027],
[0.096981, 0.097012, 0.097005],
[0.10004, 0.10007, 0.10006],
[0.10311, 0.10314, 0.10314],
[0.10618, 0.10621, 0.1062],
[0.10924, 0.10928, 0.10927],
[0.11232, 0.11235, 0.11234],
[0.11538, 0.11541, 0.1154],
[0.11852, 0.11855, 0.11854],
[0.12157, 0.1216, 0.12159],
[0.12471, 0.12474, 0.12473],
[0.12787, 0.1279, 0.12789],
[0.13102, 0.13105, 0.13105],
[0.13418, 0.13421, 0.1342],
[0.13734, 0.13738, 0.13737],
[0.14051, 0.14055, 0.14054],
[0.1437, 0.14373, 0.14372],
[0.14687, 0.14691, 0.1469],
[0.15011, 0.15015, 0.15014],
[0.15329, 0.15333, 0.15332],
[0.15656, 0.15659, 0.15659],
[0.15975, 0.15979, 0.15978],
[0.16304, 0.16308, 0.16307],
[0.16628, 0.16632, 0.16631],
[0.16957, 0.16961, 0.1696],
[0.17281, 0.17286, 0.17285],
[0.17608, 0.17612, 0.17611],
[0.1794, 0.17944, 0.17943],
[0.18269, 0.18273, 0.18272],
[0.18604, 0.18608, 0.18607],
[0.18934, 0.18939, 0.18938],
[0.19267, 0.19272, 0.19271],
[0.19602, 0.19607, 0.19606],
[0.19934, 0.19939, 0.19938],
[0.20271, 0.20276, 0.20275],
[0.20611, 0.20616, 0.20615],
[0.20946, 0.20951, 0.2095],
[0.21284, 0.21289, 0.21288],
[0.21626, 0.21632, 0.2163],
[0.21965, 0.2197, 0.21969],
[0.22307, 0.22312, 0.22311],
[0.22649, 0.22654, 0.22653],
[0.22988, 0.22994, 0.22992],
[0.23332, 0.23337, 0.23336],
[0.2368, 0.23685, 0.23684],
[0.24022, 0.24027, 0.24026],
[0.24368, 0.24373, 0.24372],
[0.24716, 0.24722, 0.2472],
[0.25063, 0.25069, 0.25067],
[0.25412, 0.25418, 0.25417],
[0.2576, 0.25766, 0.25765],
[0.26111, 0.26117, 0.26115],
[0.26461, 0.26467, 0.26465],
[0.26812, 0.26818, 0.26817],
[0.27164, 0.2717, 0.27168],
[0.27518, 0.27525, 0.27523],
[0.27872, 0.27879, 0.27877],
[0.28227, 0.28234, 0.28232],
[0.2858, 0.28587, 0.28585],
[0.28938, 0.28945, 0.28943],
[0.29294, 0.293, 0.29298],
[0.29652, 0.29659, 0.29657],
[0.30009, 0.30016, 0.30014],
[0.30368, 0.30375, 0.30373],
[0.30728, 0.30734, 0.30733],
[0.3109, 0.31096, 0.31095],
[0.31449, 0.31456, 0.31454],
[0.31811, 0.31819, 0.31817],
[0.32173, 0.32181, 0.32179],
[0.32536, 0.32543, 0.32542],
[0.32902, 0.32909, 0.32907],
[0.33267, 0.33274, 0.33272],
[0.3363, 0.33638, 0.33636],
[0.33997, 0.34005, 0.34003],
[0.34363, 0.3437, 0.34369],
[0.34731, 0.34739, 0.34737],
[0.35099, 0.35107, 0.35105],
[0.35468, 0.35476, 0.35474],
[0.35836, 0.35844, 0.35842],
[0.36207, 0.36214, 0.36212],
[0.36577, 0.36585, 0.36583],
[0.36948, 0.36956, 0.36954],
[0.37319, 0.37328, 0.37326],
[0.37693, 0.37701, 0.37699],
[0.38066, 0.38074, 0.38072],
[0.38439, 0.38447, 0.38445],
[0.38814, 0.38822, 0.3882],
[0.39189, 0.39197, 0.39195],
[0.39565, 0.39574, 0.39572],
[0.3994, 0.39949, 0.39946],
[0.40318, 0.40327, 0.40325],
[0.40695, 0.40704, 0.40702],
[0.41074, 0.41083, 0.41081],
[0.41452, 0.41461, 0.41459],
[0.41831, 0.4184, 0.41838],
[0.42211, 0.4222, 0.42218],
[0.42592, 0.42601, 0.42599],
[0.42974, 0.42983, 0.42981],
[0.43355, 0.43364, 0.43362],
[0.43739, 0.43748, 0.43746],
[0.44121, 0.44131, 0.44128],
[0.44505, 0.44515, 0.44512],
[0.4489, 0.449, 0.44897],
[0.45275, 0.45284, 0.45282],
[0.45659, 0.45669, 0.45667],
[0.46045, 0.46055, 0.46052],
[0.46433, 0.46443, 0.46441],
[0.46819, 0.46829, 0.46826],
[0.47208, 0.47218, 0.47215],
[0.47596, 0.47606, 0.47603],
[0.47985, 0.47995, 0.47992],
[0.48373, 0.48383, 0.48381],
[0.48764, 0.48775, 0.48772],
[0.49155, 0.49165, 0.49163],
[0.49545, 0.49556, 0.49553],
[0.49936, 0.49947, 0.49944],
[0.50329, 0.50339, 0.50337],
[0.50722, 0.50733, 0.5073],
[0.51115, 0.51126, 0.51123],
[0.51509, 0.5152, 0.51517],
[0.51904, 0.51915, 0.51912],
[0.52299, 0.5231, 0.52307],
[0.52695, 0.52706, 0.52703],
[0.5309, 0.53101, 0.53099],
[0.53487, 0.53498, 0.53495],
[0.53884, 0.53895, 0.53893],
[0.54282, 0.54293, 0.5429],
[0.5468, 0.54691, 0.54688],
[0.55078, 0.5509, 0.55087],
[0.55478, 0.55489, 0.55486],
[0.55877, 0.55888, 0.55886],
[0.56278, 0.5629, 0.56287],
[0.56679, 0.56691, 0.56688],
[0.57079, 0.57091, 0.57088],
[0.57481, 0.57493, 0.5749],
[0.57884, 0.57896, 0.57893],
[0.58286, 0.58298, 0.58295],
[0.5869, 0.58702, 0.58699],
[0.59094, 0.59106, 0.59103],
[0.59498, 0.5951, 0.59507],
[0.59903, 0.59916, 0.59913],
[0.60309, 0.60321, 0.60318],
[0.60715, 0.60728, 0.60724],
[0.61121, 0.61134, 0.61131],
[0.61527, 0.6154, 0.61537],
[0.61936, 0.61948, 0.61945],
[0.62343, 0.62356, 0.62353],
[0.62751, 0.62764, 0.62761],
[0.6316, 0.63173, 0.6317],
[0.63569, 0.63583, 0.63579],
[0.6398, 0.63993, 0.63989],
[0.64389, 0.64402, 0.64399],
[0.648, 0.64813, 0.6481],
[0.65212, 0.65225, 0.65222],
[0.65624, 0.65637, 0.65634],
[0.66035, 0.66049, 0.66045],
[0.66448, 0.66461, 0.66458],
[0.66862, 0.66875, 0.66872],
[0.67275, 0.67289, 0.67286],
[0.6769, 0.67703, 0.677],
[0.68103, 0.68117, 0.68114],
[0.68519, 0.68533, 0.6853],
[0.68935, 0.68949, 0.68945],
[0.6935, 0.69364, 0.69361],
[0.69766, 0.69781, 0.69777],
[0.70183, 0.70197, 0.70194],
[0.70601, 0.70616, 0.70612],
[0.71019, 0.71033, 0.7103],
[0.71437, 0.71451, 0.71448],
[0.71855, 0.7187, 0.71867],
[0.72274, 0.72289, 0.72286],
[0.72694, 0.72709, 0.72706],
[0.73114, 0.73129, 0.73126],
[0.73535, 0.7355, 0.73546],
[0.73957, 0.73972, 0.73968],
[0.74378, 0.74393, 0.74389],
[0.748, 0.74815, 0.74811],
[0.75222, 0.75237, 0.75233],
[0.75644, 0.7566, 0.75656],
[0.76068, 0.76083, 0.76079],
[0.76492, 0.76507, 0.76503],
[0.76915, 0.76931, 0.76927],
[0.7734, 0.77355, 0.77351],
[0.77765, 0.77781, 0.77777],
[0.78191, 0.78206, 0.78202],
[0.78616, 0.78632, 0.78628],
[0.79042, 0.79058, 0.79054],
[0.79469, 0.79485, 0.79481],
[0.79896, 0.79913, 0.79909],
[0.80323, 0.8034, 0.80336],
[0.80752, 0.80768, 0.80764],
[0.8118, 0.81196, 0.81192],
[0.81609, 0.81626, 0.81621],
[0.82038, 0.82054, 0.8205],
[0.82467, 0.82484, 0.8248],
[0.82898, 0.82915, 0.82911],
[0.83328, 0.83345, 0.83341],
[0.83759, 0.83776, 0.83772],
[0.84191, 0.84208, 0.84204],
[0.84622, 0.84639, 0.84635],
[0.85054, 0.85071, 0.85067],
[0.85487, 0.85504, 0.855],
[0.8592, 0.85937, 0.85933],
[0.86354, 0.86371, 0.86367],
[0.86787, 0.86805, 0.86801],
[0.87222, 0.87239, 0.87235],
[0.87656, 0.87673, 0.87669],
[0.88091, 0.88109, 0.88104],
[0.88527, 0.88544, 0.8854],
[0.88962, 0.8898, 0.88976],
[0.89399, 0.89417, 0.89412],
[0.89835, 0.89853, 0.89849],
[0.90272, 0.90291, 0.90286],
[0.9071, 0.90728, 0.90724],
[0.91148, 0.91166, 0.91161],
[0.91585, 0.91604, 0.91599],
[0.92024, 0.92043, 0.92038],
[0.92463, 0.92482, 0.92477],
[0.92903, 0.92921, 0.92917],
[0.93342, 0.93361, 0.93356],
[0.93783, 0.93802, 0.93797],
[0.94224, 0.94243, 0.94238],
[0.94664, 0.94683, 0.94679],
[0.95106, 0.95125, 0.9512],
[0.95548, 0.95567, 0.95562],
[0.9599, 0.96009, 0.96004],
[0.96432, 0.96451, 0.96446],
[0.96875, 0.96894, 0.9689],
[0.97318, 0.97338, 0.97333],
[0.97762, 0.97781, 0.97777],
[0.98206, 0.98226, 0.98221],
[0.98651, 0.9867, 0.98666],
[0.99095, 0.99115, 0.9911],
[0.9954, 0.9956, 0.99555],
[0.99986, 1, 1],
]
b_linear_grey_0_100_c0 = bokeh_palette('linear_grey_0_100_c0',linear_grey_0_100_c0)
m_linear_grey_0_100_c0 = mpl_cm('linear_grey_0_100_c0',linear_grey_0_100_c0)
m_linear_grey_0_100_c0_r = mpl_cm('linear_grey_0_100_c0_r',list(reversed(linear_grey_0_100_c0)))
gray = b_linear_grey_0_100_c0
m_gray = m_linear_grey_0_100_c0
m_gray_r = m_linear_grey_0_100_c0_r
palette['gray'] = b_linear_grey_0_100_c0
palette_n['gray'] = b_linear_grey_0_100_c0
cm['gray'] = m_linear_grey_0_100_c0
cm['gray_r'] = m_linear_grey_0_100_c0_r
cm_n['gray'] = mpl_cm('gray',linear_grey_0_100_c0)
cm_n['gray_r'] = mpl_cm('gray_r',list(reversed(linear_grey_0_100_c0)))
CET_L1 = b_linear_grey_0_100_c0
m_CET_L1 = m_linear_grey_0_100_c0
m_CET_L1_r = m_linear_grey_0_100_c0_r
palette['CET_L1'] = b_linear_grey_0_100_c0
cm['CET_L1'] = m_linear_grey_0_100_c0
cm['CET_L1_r'] = m_linear_grey_0_100_c0_r
register_cmap('cet_CET_L1',m_linear_grey_0_100_c0)
register_cmap('cet_CET_L1_r',m_linear_grey_0_100_c0_r)




linear_bmy_10_95_c71 = [  # cmap_def
[0.0018068, 0.058001, 0.36399],
[0.0024107, 0.059964, 0.37048],
[0.0031987, 0.061692, 0.37697],
[0.0042843, 0.063571, 0.38341],
[0.0057638, 0.065341, 0.38979],
[0.0075093, 0.067136, 0.39617],
[0.0097084, 0.068953, 0.40248],
[0.012532, 0.070615, 0.40875],
[0.0155, 0.07232, 0.41499],
[0.019067, 0.074062, 0.42116],
[0.023134, 0.075673, 0.42729],
[0.027668, 0.077301, 0.43336],
[0.032903, 0.078925, 0.43937],
[0.038954, 0.080546, 0.44532],
[0.045181, 0.082169, 0.45122],
[0.051521, 0.083668, 0.45704],
[0.058053, 0.085134, 0.46278],
[0.06473, 0.086625, 0.46847],
[0.071444, 0.088071, 0.47409],
[0.078236, 0.089456, 0.4796],
[0.085243, 0.090814, 0.48502],
[0.092333, 0.092133, 0.49038],
[0.099463, 0.093353, 0.49564],
[0.10684, 0.094646, 0.50078],
[0.11423, 0.095771, 0.50581],
[0.12178, 0.096854, 0.51075],
[0.12948, 0.097993, 0.51558],
[0.13728, 0.098973, 0.52029],
[0.14525, 0.099911, 0.52487],
[0.1533, 0.10078, 0.5293],
[0.16159, 0.10159, 0.53358],
[0.16995, 0.10233, 0.53773],
[0.17849, 0.103, 0.5417],
[0.18715, 0.10358, 0.54552],
[0.19599, 0.10402, 0.54915],
[0.20499, 0.10436, 0.55259],
[0.21418, 0.10461, 0.55582],
[0.22355, 0.10475, 0.55884],
[0.2331, 0.10475, 0.56162],
[0.24286, 0.10462, 0.56415],
[0.25285, 0.10434, 0.5664],
[0.26305, 0.10391, 0.56836],
[0.27348, 0.10327, 0.57],
[0.28411, 0.10237, 0.5713],
[0.29502, 0.10125, 0.57224],
[0.30619, 0.099883, 0.57281],
[0.31733, 0.098383, 0.57312],
[0.32829, 0.096731, 0.57334],
[0.33902, 0.095191, 0.57351],
[0.34952, 0.093473, 0.57361],
[0.35984, 0.09185, 0.57367],
[0.36998, 0.090102, 0.57366],
[0.37996, 0.088354, 0.5736],
[0.38979, 0.086567, 0.57349],
[0.39947, 0.084774, 0.57331],
[0.40903, 0.082914, 0.57309],
[0.41845, 0.081085, 0.5728],
[0.42777, 0.079164, 0.57245],
[0.43696, 0.077258, 0.57205],
[0.44604, 0.075373, 0.57159],
[0.45505, 0.073529, 0.57107],
[0.46394, 0.071572, 0.57049],
[0.47273, 0.069648, 0.56986],
[0.48145, 0.067661, 0.56917],
[0.49008, 0.06569, 0.56843],
[0.49863, 0.063767, 0.56763],
[0.5071, 0.061792, 0.56677],
[0.51549, 0.059971, 0.56585],
[0.52382, 0.057948, 0.56487],
[0.53207, 0.056175, 0.56385],
[0.54025, 0.054286, 0.56276],
[0.54838, 0.052493, 0.5616],
[0.55642, 0.05079, 0.5604],
[0.56441, 0.049208, 0.55913],
[0.57234, 0.04752, 0.55781],
[0.58021, 0.045977, 0.55643],
[0.588, 0.044652, 0.55502],
[0.59575, 0.043191, 0.55358],
[0.60343, 0.042094, 0.55214],
[0.61104, 0.040933, 0.55067],
[0.6186, 0.039904, 0.5492],
[0.62609, 0.039006, 0.5477],
[0.63354, 0.038249, 0.5462],
[0.64092, 0.037647, 0.54466],
[0.64826, 0.037202, 0.54313],
[0.65554, 0.036908, 0.54156],
[0.66278, 0.036776, 0.53999],
[0.66996, 0.036816, 0.53841],
[0.6771, 0.037039, 0.53681],
[0.68417, 0.037443, 0.53518],
[0.69122, 0.038023, 0.53354],
[0.69821, 0.038789, 0.53189],
[0.70517, 0.039756, 0.53022],
[0.71207, 0.040926, 0.52854],
[0.71894, 0.042269, 0.52684],
[0.72576, 0.043681, 0.52511],
[0.73255, 0.045402, 0.52338],
[0.73929, 0.047268, 0.52162],
[0.74598, 0.049331, 0.51985],
[0.75264, 0.051385, 0.51807],
[0.75926, 0.053656, 0.51626],
[0.76585, 0.056145, 0.51445],
[0.77239, 0.058663, 0.5126],
[0.7789, 0.061273, 0.51075],
[0.78537, 0.064105, 0.50888],
[0.79179, 0.067003, 0.50698],
[0.79815, 0.07017, 0.50503],
[0.8044, 0.073831, 0.503],
[0.81054, 0.077734, 0.50091],
[0.81658, 0.082182, 0.49873],
[0.82251, 0.086838, 0.49646],
[0.82832, 0.09184, 0.49413],
[0.83402, 0.097033, 0.49171],
[0.83961, 0.10255, 0.48919],
[0.8451, 0.10821, 0.48661],
[0.85047, 0.1141, 0.48394],
[0.85573, 0.12016, 0.48118],
[0.86088, 0.12639, 0.47835],
[0.86592, 0.13272, 0.47543],
[0.87085, 0.13913, 0.47242],
[0.87567, 0.14574, 0.46934],
[0.88036, 0.15236, 0.46614],
[0.88495, 0.15912, 0.46287],
[0.88942, 0.16592, 0.45953],
[0.89379, 0.17282, 0.45608],
[0.89803, 0.17978, 0.45256],
[0.90216, 0.18681, 0.44893],
[0.90617, 0.1939, 0.44521],
[0.91008, 0.20098, 0.44141],
[0.91386, 0.20817, 0.43752],
[0.91752, 0.21538, 0.43351],
[0.92106, 0.22265, 0.42942],
[0.92449, 0.22991, 0.42524],
[0.9278, 0.23727, 0.42094],
[0.93099, 0.24459, 0.41657],
[0.93405, 0.252, 0.41209],
[0.93701, 0.25939, 0.4076],
[0.93986, 0.26675, 0.40316],
[0.9426, 0.27411, 0.39879],
[0.94522, 0.28144, 0.39449],
[0.94774, 0.28878, 0.39026],
[0.95015, 0.29609, 0.3861],
[0.95245, 0.30338, 0.382],
[0.95466, 0.31069, 0.37798],
[0.95674, 0.31797, 0.37403],
[0.95873, 0.32523, 0.37012],
[0.96061, 0.33251, 0.36631],
[0.96239, 0.33976, 0.36253],
[0.96406, 0.34701, 0.35883],
[0.96562, 0.35425, 0.35522],
[0.96708, 0.36148, 0.35164],
[0.96843, 0.36871, 0.34814],
[0.96968, 0.37593, 0.3447],
[0.97082, 0.38315, 0.34134],
[0.97186, 0.39036, 0.33803],
[0.9728, 0.39757, 0.33481],
[0.97362, 0.40476, 0.33161],
[0.97434, 0.41197, 0.32852],
[0.97495, 0.41917, 0.32547],
[0.97546, 0.42636, 0.32252],
[0.97586, 0.43355, 0.31962],
[0.97615, 0.44074, 0.31678],
[0.97634, 0.44792, 0.31402],
[0.97641, 0.45512, 0.31133],
[0.97637, 0.46229, 0.30872],
[0.97624, 0.46947, 0.30617],
[0.97607, 0.47658, 0.30363],
[0.97591, 0.4836, 0.30112],
[0.97579, 0.49054, 0.29861],
[0.97569, 0.49738, 0.2961],
[0.97563, 0.50413, 0.29358],
[0.97559, 0.5108, 0.29108],
[0.97559, 0.51739, 0.28856],
[0.97562, 0.5239, 0.28603],
[0.97568, 0.53035, 0.28352],
[0.97578, 0.53672, 0.28099],
[0.9759, 0.54302, 0.27848],
[0.97607, 0.54926, 0.27597],
[0.97626, 0.55543, 0.27343],
[0.97649, 0.56155, 0.2709],
[0.97675, 0.56761, 0.26835],
[0.97705, 0.57361, 0.26582],
[0.97738, 0.57956, 0.26328],
[0.97775, 0.58545, 0.26073],
[0.97815, 0.59129, 0.25816],
[0.97859, 0.59708, 0.25562],
[0.97907, 0.60282, 0.25307],
[0.97958, 0.60852, 0.25049],
[0.98013, 0.61416, 0.24794],
[0.98072, 0.61978, 0.24534],
[0.98135, 0.62534, 0.24277],
[0.98202, 0.63087, 0.24018],
[0.98273, 0.63635, 0.23762],
[0.98347, 0.64177, 0.23503],
[0.98425, 0.64718, 0.23241],
[0.98507, 0.65256, 0.22982],
[0.98586, 0.65793, 0.22748],
[0.98656, 0.66331, 0.22545],
[0.98719, 0.6687, 0.22375],
[0.98774, 0.6741, 0.22234],
[0.98823, 0.67951, 0.22119],
[0.98867, 0.68493, 0.22026],
[0.98906, 0.69035, 0.21952],
[0.98939, 0.69578, 0.21896],
[0.98968, 0.7012, 0.2186],
[0.98992, 0.70664, 0.2184],
[0.99012, 0.71207, 0.21835],
[0.99028, 0.7175, 0.21841],
[0.9904, 0.72292, 0.21862],
[0.99048, 0.72836, 0.21896],
[0.99054, 0.73379, 0.2194],
[0.99055, 0.73922, 0.21995],
[0.99053, 0.74463, 0.22062],
[0.99048, 0.75005, 0.22135],
[0.9904, 0.75548, 0.22218],
[0.99028, 0.7609, 0.22313],
[0.99014, 0.76631, 0.22411],
[0.98996, 0.77172, 0.22522],
[0.98976, 0.77714, 0.22642],
[0.98952, 0.78254, 0.22765],
[0.98925, 0.78794, 0.22898],
[0.98896, 0.79335, 0.23033],
[0.98863, 0.79874, 0.23182],
[0.98828, 0.80414, 0.2333],
[0.9879, 0.80953, 0.23491],
[0.98749, 0.81492, 0.23654],
[0.98705, 0.82031, 0.23821],
[0.98658, 0.82569, 0.23995],
[0.98608, 0.83108, 0.24175],
[0.98556, 0.83646, 0.24358],
[0.98501, 0.84184, 0.24547],
[0.98443, 0.84721, 0.24743],
[0.98382, 0.85258, 0.2494],
[0.98318, 0.85795, 0.25145],
[0.98251, 0.86332, 0.25351],
[0.98182, 0.86869, 0.25562],
[0.98109, 0.87405, 0.25776],
[0.98034, 0.87941, 0.25997],
[0.97957, 0.88477, 0.26219],
[0.97875, 0.89012, 0.26445],
[0.97792, 0.89548, 0.26675],
[0.97705, 0.90083, 0.26909],
[0.97615, 0.90619, 0.27144],
[0.97523, 0.91153, 0.27384],
[0.97428, 0.91688, 0.27628],
[0.9733, 0.92223, 0.27873],
[0.97229, 0.92756, 0.2812],
[0.97124, 0.93291, 0.28373],
[0.97017, 0.93825, 0.28625],
[0.96907, 0.94359, 0.28885],
[0.96794, 0.94893, 0.29144],
[0.96677, 0.95427, 0.29405],
[0.96558, 0.9596, 0.29671],
[0.96436, 0.96493, 0.29938],
[0.9631, 0.97026, 0.30206],
[0.96182, 0.97559, 0.30477],
]
b_linear_bmy_10_95_c71 = bokeh_palette('linear_bmy_10_95_c71',linear_bmy_10_95_c71)
m_linear_bmy_10_95_c71 = mpl_cm('linear_bmy_10_95_c71',linear_bmy_10_95_c71)
m_linear_bmy_10_95_c71_r = mpl_cm('linear_bmy_10_95_c71_r',list(reversed(linear_bmy_10_95_c71)))
CET_L8 = b_linear_bmy_10_95_c71
m_CET_L8 = m_linear_bmy_10_95_c71
m_CET_L8_r = m_linear_bmy_10_95_c71_r
palette['CET_L8'] = b_linear_bmy_10_95_c71
cm['CET_L8'] = m_linear_bmy_10_95_c71
cm['CET_L8_r'] = m_linear_bmy_10_95_c71_r
register_cmap('cet_CET_L8',m_linear_bmy_10_95_c71)
register_cmap('cet_CET_L8_r',m_linear_bmy_10_95_c71_r)




diverging_isoluminant_cjm_75_c24 = [  # cmap_def
[0, 0.79435, 0.99671],
[0, 0.79384, 0.99455],
[0, 0.79333, 0.99239],
[0, 0.79282, 0.99023],
[0, 0.79231, 0.98806],
[0, 0.7918, 0.9859],
[0, 0.79128, 0.98374],
[0, 0.79077, 0.98158],
[0, 0.79026, 0.97942],
[0, 0.78975, 0.97726],
[0, 0.78924, 0.9751],
[0, 0.78872, 0.97295],
[0, 0.7882, 0.97079],
[0, 0.78769, 0.96863],
[0, 0.78717, 0.96647],
[0.030516, 0.78666, 0.96431],
[0.077099, 0.78614, 0.96216],
[0.10792, 0.78562, 0.96001],
[0.13219, 0.7851, 0.95785],
[0.15262, 0.78458, 0.9557],
[0.17059, 0.78406, 0.95354],
[0.18672, 0.78354, 0.95139],
[0.20145, 0.78302, 0.94923],
[0.21508, 0.7825, 0.94708],
[0.22782, 0.78198, 0.94493],
[0.23976, 0.78145, 0.94278],
[0.25106, 0.78093, 0.94063],
[0.26179, 0.7804, 0.93848],
[0.272, 0.77988, 0.93633],
[0.28182, 0.77935, 0.93417],
[0.29123, 0.77882, 0.93203],
[0.30027, 0.7783, 0.92988],
[0.30902, 0.77777, 0.92773],
[0.31745, 0.77725, 0.92558],
[0.32562, 0.77672, 0.92344],
[0.33356, 0.77618, 0.92129],
[0.34126, 0.77565, 0.91914],
[0.34875, 0.77512, 0.917],
[0.35605, 0.77459, 0.91485],
[0.36315, 0.77406, 0.91271],
[0.37009, 0.77352, 0.91056],
[0.37688, 0.77299, 0.90842],
[0.3835, 0.77246, 0.90627],
[0.38997, 0.77193, 0.90413],
[0.39633, 0.77139, 0.90199],
[0.40254, 0.77086, 0.89985],
[0.40865, 0.77032, 0.8977],
[0.41463, 0.76978, 0.89556],
[0.4205, 0.76924, 0.89342],
[0.42627, 0.7687, 0.89128],
[0.43196, 0.76816, 0.88914],
[0.43753, 0.76762, 0.887],
[0.443, 0.76708, 0.88486],
[0.44839, 0.76654, 0.88272],
[0.4537, 0.766, 0.88058],
[0.45893, 0.76546, 0.87844],
[0.46409, 0.76492, 0.8763],
[0.46917, 0.76437, 0.87417],
[0.47416, 0.76382, 0.87203],
[0.4791, 0.76328, 0.86989],
[0.48396, 0.76273, 0.86776],
[0.48876, 0.76219, 0.86562],
[0.4935, 0.76164, 0.86349],
[0.49819, 0.76109, 0.86135],
[0.5028, 0.76054, 0.85922],
[0.50737, 0.75999, 0.85708],
[0.51187, 0.75944, 0.85495],
[0.51632, 0.75889, 0.85282],
[0.52073, 0.75834, 0.85068],
[0.52508, 0.75779, 0.84855],
[0.52939, 0.75723, 0.84641],
[0.53364, 0.75668, 0.84429],
[0.53786, 0.75612, 0.84216],
[0.54202, 0.75557, 0.84003],
[0.54615, 0.75502, 0.83789],
[0.55022, 0.75446, 0.83576],
[0.55426, 0.7539, 0.83363],
[0.55825, 0.75334, 0.83151],
[0.56222, 0.75278, 0.82938],
[0.56614, 0.75222, 0.82725],
[0.57001, 0.75166, 0.82511],
[0.57386, 0.7511, 0.82299],
[0.57767, 0.75054, 0.82086],
[0.58145, 0.74997, 0.81873],
[0.5852, 0.7494, 0.81661],
[0.58889, 0.74884, 0.81448],
[0.59258, 0.74828, 0.81235],
[0.59622, 0.74772, 0.81023],
[0.59983, 0.74715, 0.8081],
[0.60342, 0.74658, 0.80598],
[0.60697, 0.74601, 0.80385],
[0.61049, 0.74544, 0.80173],
[0.61398, 0.74487, 0.79961],
[0.61745, 0.7443, 0.79748],
[0.62089, 0.74373, 0.79536],
[0.6243, 0.74316, 0.79324],
[0.62769, 0.74258, 0.79111],
[0.63105, 0.742, 0.78899],
[0.63438, 0.74143, 0.78687],
[0.63769, 0.74085, 0.78475],
[0.64097, 0.74028, 0.78262],
[0.64423, 0.73971, 0.7805],
[0.64747, 0.73913, 0.77838],
[0.65069, 0.73854, 0.77626],
[0.65388, 0.73796, 0.77414],
[0.65705, 0.73738, 0.77202],
[0.66019, 0.7368, 0.7699],
[0.66332, 0.73621, 0.76778],
[0.66642, 0.73563, 0.76567],
[0.66951, 0.73505, 0.76355],
[0.67257, 0.73446, 0.76145],
[0.6756, 0.73388, 0.75935],
[0.67862, 0.73329, 0.75727],
[0.68161, 0.7327, 0.75521],
[0.68458, 0.73211, 0.75316],
[0.68753, 0.73151, 0.75116],
[0.69045, 0.73091, 0.74918],
[0.69334, 0.73031, 0.74727],
[0.6962, 0.72971, 0.74541],
[0.69902, 0.7291, 0.74364],
[0.70181, 0.72848, 0.74196],
[0.70456, 0.72786, 0.74039],
[0.70726, 0.72724, 0.73895],
[0.70992, 0.72661, 0.73766],
[0.71252, 0.72597, 0.73652],
[0.71507, 0.72533, 0.73557],
[0.71757, 0.72468, 0.73481],
[0.72, 0.72402, 0.73423],
[0.72237, 0.72335, 0.73386],
[0.7247, 0.72268, 0.73368],
[0.72695, 0.722, 0.7337],
[0.72915, 0.72132, 0.73391],
[0.7313, 0.72063, 0.73429],
[0.7334, 0.71994, 0.73484],
[0.73545, 0.71924, 0.73553],
[0.73745, 0.71854, 0.73635],
[0.73943, 0.71784, 0.73729],
[0.74136, 0.71713, 0.73833],
[0.74327, 0.71642, 0.73945],
[0.74515, 0.71571, 0.74062],
[0.74702, 0.71498, 0.74185],
[0.74887, 0.71426, 0.74313],
[0.75071, 0.71355, 0.74443],
[0.75252, 0.71282, 0.74576],
[0.75434, 0.7121, 0.74711],
[0.75615, 0.71138, 0.74847],
[0.75795, 0.71064, 0.74983],
[0.75974, 0.70991, 0.75121],
[0.76153, 0.70918, 0.75258],
[0.76332, 0.70845, 0.75397],
[0.76511, 0.7077, 0.75535],
[0.76688, 0.70697, 0.75673],
[0.76866, 0.70623, 0.75812],
[0.77044, 0.70548, 0.75949],
[0.77221, 0.70474, 0.76088],
[0.77397, 0.70399, 0.76225],
[0.77574, 0.70324, 0.76364],
[0.77751, 0.70249, 0.76502],
[0.77926, 0.70173, 0.7664],
[0.78102, 0.70098, 0.76778],
[0.78278, 0.70021, 0.76916],
[0.78453, 0.69945, 0.77055],
[0.78628, 0.69869, 0.77193],
[0.78802, 0.69792, 0.7733],
[0.78977, 0.69715, 0.77468],
[0.79151, 0.69639, 0.77607],
[0.79325, 0.69561, 0.77745],
[0.79498, 0.69483, 0.77883],
[0.79672, 0.69405, 0.78021],
[0.79845, 0.69328, 0.78159],
[0.80018, 0.6925, 0.78297],
[0.8019, 0.69171, 0.78435],
[0.80363, 0.69092, 0.78573],
[0.80534, 0.69014, 0.78711],
[0.80707, 0.68935, 0.78849],
[0.80878, 0.68855, 0.78987],
[0.81049, 0.68775, 0.79125],
[0.8122, 0.68695, 0.79263],
[0.81391, 0.68615, 0.79401],
[0.81562, 0.68535, 0.79539],
[0.81732, 0.68454, 0.79677],
[0.81902, 0.68373, 0.79815],
[0.82072, 0.68292, 0.79953],
[0.82242, 0.68211, 0.80091],
[0.82411, 0.68129, 0.80228],
[0.82581, 0.68047, 0.80367],
[0.8275, 0.67965, 0.80504],
[0.82919, 0.67883, 0.80643],
[0.83087, 0.67801, 0.8078],
[0.83255, 0.67718, 0.80919],
[0.83424, 0.67634, 0.81056],
[0.83591, 0.67551, 0.81194],
[0.83759, 0.67467, 0.81332],
[0.83927, 0.67383, 0.8147],
[0.84094, 0.67299, 0.81608],
[0.84261, 0.67215, 0.81746],
[0.84427, 0.6713, 0.81883],
[0.84594, 0.67045, 0.82021],
[0.8476, 0.6696, 0.8216],
[0.84927, 0.66874, 0.82297],
[0.85092, 0.66788, 0.82435],
[0.85259, 0.66702, 0.82573],
[0.85424, 0.66615, 0.82711],
[0.8559, 0.66529, 0.82849],
[0.85754, 0.66442, 0.82987],
[0.85919, 0.66355, 0.83125],
[0.86084, 0.66268, 0.83263],
[0.86249, 0.6618, 0.83401],
[0.86414, 0.66091, 0.83538],
[0.86577, 0.66003, 0.83676],
[0.86741, 0.65914, 0.83814],
[0.86906, 0.65826, 0.83952],
[0.87069, 0.65737, 0.8409],
[0.87233, 0.65647, 0.84228],
[0.87395, 0.65557, 0.84366],
[0.87559, 0.65467, 0.84504],
[0.87721, 0.65377, 0.84641],
[0.87884, 0.65285, 0.84779],
[0.88047, 0.65195, 0.84918],
[0.88209, 0.65104, 0.85055],
[0.88371, 0.65012, 0.85193],
[0.88533, 0.64921, 0.85331],
[0.88695, 0.64828, 0.85469],
[0.88857, 0.64736, 0.85607],
[0.89018, 0.64643, 0.85745],
[0.89179, 0.64549, 0.85883],
[0.8934, 0.64456, 0.86021],
[0.89502, 0.64362, 0.86159],
[0.89662, 0.64269, 0.86297],
[0.89822, 0.64174, 0.86435],
[0.89983, 0.64079, 0.86572],
[0.90143, 0.63985, 0.8671],
[0.90304, 0.63889, 0.86849],
[0.90463, 0.63793, 0.86986],
[0.90623, 0.63698, 0.87124],
[0.90783, 0.63601, 0.87262],
[0.90942, 0.63504, 0.874],
[0.91101, 0.63408, 0.87538],
[0.9126, 0.6331, 0.87675],
[0.91419, 0.63213, 0.87814],
[0.91578, 0.63115, 0.87951],
[0.91737, 0.63016, 0.8809],
[0.91895, 0.62917, 0.88227],
[0.92053, 0.62819, 0.88366],
[0.92212, 0.62719, 0.88504],
[0.9237, 0.62619, 0.88642],
[0.92527, 0.62519, 0.88779],
[0.92685, 0.62419, 0.88918],
[0.92842, 0.62319, 0.89055],
[0.93, 0.62217, 0.89194],
[0.93157, 0.62116, 0.89331],
[0.93314, 0.62014, 0.8947],
[0.93471, 0.61912, 0.89607],
[0.93628, 0.61809, 0.89746],
[0.93785, 0.61706, 0.89883],
[0.93941, 0.61603, 0.90021],
]
b_diverging_isoluminant_cjm_75_c24 = bokeh_palette('diverging_isoluminant_cjm_75_c24',diverging_isoluminant_cjm_75_c24)
m_diverging_isoluminant_cjm_75_c24 = mpl_cm('diverging_isoluminant_cjm_75_c24',diverging_isoluminant_cjm_75_c24)
m_diverging_isoluminant_cjm_75_c24_r = mpl_cm('diverging_isoluminant_cjm_75_c24_r',list(reversed(diverging_isoluminant_cjm_75_c24)))




rainbow_bgyr_35_85_c73 = [  # cmap_def
[0, 0.20755, 0.97632],
[0, 0.21857, 0.96476],
[0, 0.2291, 0.95322],
[0, 0.23923, 0.94169],
[0, 0.24899, 0.93018],
[0, 0.25842, 0.91868],
[0, 0.26756, 0.9072],
[0, 0.27649, 0.89573],
[0, 0.28513, 0.88428],
[0, 0.29358, 0.87284],
[0, 0.30184, 0.86141],
[0, 0.30994, 0.85],
[0, 0.31785, 0.83859],
[0, 0.3256, 0.82721],
[0, 0.33324, 0.81583],
[0, 0.34073, 0.80447],
[0, 0.3481, 0.79312],
[0, 0.35537, 0.78178],
[0, 0.3625, 0.77045],
[0, 0.36954, 0.75913],
[0, 0.37649, 0.74783],
[0, 0.38333, 0.73654],
[0, 0.39005, 0.72528],
[0, 0.39667, 0.71405],
[0, 0.40319, 0.70286],
[0, 0.40957, 0.69171],
[0, 0.4158, 0.68063],
[0, 0.42188, 0.66965],
[0, 0.4278, 0.65875],
[0, 0.43352, 0.64799],
[0, 0.43905, 0.63737],
[0, 0.44438, 0.62689],
[0, 0.4495, 0.61659],
[0, 0.45441, 0.60644],
[0, 0.45914, 0.59646],
[0.0019224, 0.46368, 0.58662],
[0.027932, 0.46808, 0.57693],
[0.054843, 0.47236, 0.56735],
[0.076598, 0.47653, 0.55785],
[0.095053, 0.48062, 0.54844],
[0.11106, 0.48465, 0.53907],
[0.12523, 0.48865, 0.52974],
[0.13795, 0.49262, 0.52042],
[0.14935, 0.49658, 0.5111],
[0.15962, 0.50055, 0.50179],
[0.169, 0.50452, 0.49245],
[0.17747, 0.50849, 0.48309],
[0.18517, 0.51246, 0.4737],
[0.19217, 0.51645, 0.46429],
[0.19856, 0.52046, 0.45483],
[0.20443, 0.52448, 0.44531],
[0.20974, 0.52851, 0.43577],
[0.21461, 0.53255, 0.42616],
[0.21905, 0.53661, 0.41651],
[0.22309, 0.54066, 0.40679],
[0.22674, 0.54474, 0.397],
[0.23002, 0.54883, 0.38713],
[0.233, 0.55292, 0.3772],
[0.23568, 0.55703, 0.36716],
[0.23802, 0.56114, 0.35704],
[0.24006, 0.56526, 0.34678],
[0.24185, 0.56939, 0.3364],
[0.24334, 0.57354, 0.32588],
[0.24458, 0.57769, 0.31523],
[0.24556, 0.58185, 0.30439],
[0.2463, 0.58603, 0.29336],
[0.2468, 0.59019, 0.28214],
[0.24707, 0.59438, 0.27067],
[0.24714, 0.59856, 0.25895],
[0.24704, 0.60275, 0.24696],
[0.24679, 0.60693, 0.23472],
[0.24648, 0.61109, 0.22215],
[0.24616, 0.61523, 0.20933],
[0.24597, 0.61936, 0.19628],
[0.24602, 0.62342, 0.18297],
[0.24648, 0.62742, 0.1696],
[0.24753, 0.63135, 0.15615],
[0.24931, 0.63518, 0.14276],
[0.25204, 0.63889, 0.12965],
[0.25579, 0.64249, 0.11693],
[0.2607, 0.64593, 0.10484],
[0.26674, 0.64925, 0.093668],
[0.2739, 0.65241, 0.083583],
[0.28205, 0.65544, 0.074764],
[0.29104, 0.65834, 0.067449],
[0.30071, 0.66112, 0.061598],
[0.31093, 0.66382, 0.057362],
[0.32147, 0.66641, 0.054542],
[0.33226, 0.66895, 0.052918],
[0.34314, 0.67142, 0.052293],
[0.35402, 0.67386, 0.052401],
[0.36487, 0.67627, 0.05303],
[0.37564, 0.67864, 0.054018],
[0.38629, 0.681, 0.055243],
[0.39683, 0.68335, 0.05664],
[0.40725, 0.68569, 0.057955],
[0.41755, 0.68801, 0.059542],
[0.42772, 0.69031, 0.060911],
[0.43777, 0.69262, 0.062454],
[0.4477, 0.69491, 0.063883],
[0.45755, 0.69719, 0.065334],
[0.46731, 0.69947, 0.066802],
[0.47698, 0.70173, 0.068291],
[0.48656, 0.70399, 0.069758],
[0.49607, 0.70625, 0.071211],
[0.50551, 0.70848, 0.072621],
[0.5149, 0.71071, 0.074107],
[0.52421, 0.71293, 0.07551],
[0.53346, 0.71514, 0.076938],
[0.54268, 0.71735, 0.078389],
[0.55183, 0.71954, 0.079862],
[0.56093, 0.72172, 0.08136],
[0.56998, 0.7239, 0.082779],
[0.57901, 0.72607, 0.084273],
[0.58797, 0.72822, 0.08563],
[0.59692, 0.73037, 0.08717],
[0.60581, 0.73251, 0.088583],
[0.61468, 0.73464, 0.09003],
[0.62353, 0.73676, 0.091506],
[0.63233, 0.73887, 0.092872],
[0.6411, 0.74097, 0.094388],
[0.64986, 0.74307, 0.095796],
[0.65858, 0.74515, 0.097234],
[0.66728, 0.74722, 0.098697],
[0.67595, 0.74928, 0.10013],
[0.6846, 0.75135, 0.10157],
[0.69324, 0.75339, 0.10305],
[0.70185, 0.75543, 0.10444],
[0.71045, 0.75746, 0.10594],
[0.71902, 0.75947, 0.10738],
[0.72758, 0.76149, 0.10881],
[0.73613, 0.76349, 0.1103],
[0.74466, 0.76548, 0.11173],
[0.75317, 0.76746, 0.1132],
[0.76167, 0.76943, 0.11457],
[0.77016, 0.7714, 0.11608],
[0.77864, 0.77335, 0.11749],
[0.78711, 0.7753, 0.11897],
[0.79556, 0.77724, 0.12038],
[0.80401, 0.77916, 0.12182],
[0.81244, 0.78108, 0.12327],
[0.82086, 0.78299, 0.12474],
[0.82929, 0.78489, 0.12623],
[0.83769, 0.78678, 0.12766],
[0.84609, 0.78866, 0.1291],
[0.8545, 0.79053, 0.1306],
[0.86289, 0.79238, 0.13203],
[0.87127, 0.7942, 0.13346],
[0.87964, 0.79598, 0.13491],
[0.88799, 0.79771, 0.13626],
[0.8963, 0.79937, 0.13769],
[0.90456, 0.80091, 0.13896],
[0.91273, 0.80229, 0.14021],
[0.92077, 0.80347, 0.14138],
[0.92863, 0.80438, 0.14238],
[0.93625, 0.80495, 0.14322],
[0.94355, 0.80511, 0.14385],
[0.95044, 0.80481, 0.14425],
[0.95686, 0.80399, 0.14437],
[0.96274, 0.80262, 0.14419],
[0.96803, 0.80067, 0.14372],
[0.9727, 0.79816, 0.14294],
[0.97674, 0.79512, 0.14188],
[0.9802, 0.79159, 0.14055],
[0.9831, 0.78764, 0.13895],
[0.98551, 0.78333, 0.13721],
[0.98751, 0.77872, 0.13526],
[0.98917, 0.77389, 0.13319],
[0.99056, 0.76889, 0.13103],
[0.99175, 0.76376, 0.12875],
[0.99279, 0.75855, 0.12647],
[0.99371, 0.75327, 0.1241],
[0.99455, 0.74796, 0.12173],
[0.99534, 0.74261, 0.11937],
[0.99608, 0.73724, 0.11697],
[0.99679, 0.73185, 0.11453],
[0.99748, 0.72646, 0.11217],
[0.99814, 0.72104, 0.10976],
[0.99879, 0.71563, 0.1073],
[0.99942, 0.71019, 0.1048],
[1, 0.70475, 0.10238],
[1, 0.69929, 0.099908],
[1, 0.69382, 0.09745],
[1, 0.68834, 0.095],
[1, 0.68284, 0.092452],
[1, 0.67734, 0.08996],
[1, 0.6718, 0.087465],
[1, 0.66626, 0.08489],
[1, 0.66071, 0.082364],
[1, 0.65514, 0.079729],
[1, 0.64955, 0.077118],
[1, 0.64394, 0.074554],
[1, 0.63832, 0.071927],
[1, 0.63268, 0.069347],
[1, 0.62702, 0.066593],
[1, 0.62135, 0.063895],
[1, 0.61565, 0.061104],
[1, 0.60993, 0.058355],
[1, 0.6042, 0.055584],
[1, 0.59843, 0.052708],
[1, 0.59265, 0.049893],
[1, 0.58684, 0.046988],
[1, 0.58101, 0.043966],
[1, 0.57515, 0.041014],
[1, 0.56926, 0.037943],
[1, 0.56336, 0.034887],
[1, 0.55742, 0.031822],
[1, 0.55145, 0.028972],
[1, 0.54546, 0.026194],
[1, 0.53942, 0.023487],
[1, 0.53336, 0.020851],
[1, 0.52727, 0.018287],
[1, 0.52114, 0.015792],
[1, 0.51497, 0.013362],
[1, 0.50878, 0.010828],
[1, 0.50252, 0.0085264],
[1, 0.49623, 0.0063082],
[1, 0.48991, 0.0041556],
[1, 0.48353, 0.0020694],
[1, 0.47711, 4.915e-05],
[1, 0.47065, 0],
[1, 0.46413, 0],
[1, 0.45754, 0],
[1, 0.45092, 0],
[1, 0.44423, 0],
[1, 0.43748, 0],
[1, 0.43065, 0],
[1, 0.42375, 0],
[1, 0.4168, 0],
[1, 0.40976, 0],
[1, 0.40263, 0],
[1, 0.39544, 0],
[1, 0.38813, 0],
[1, 0.38075, 0],
[1, 0.37324, 0],
[1, 0.36564, 0],
[1, 0.35792, 0],
[1, 0.35006, 0],
[1, 0.34208, 0],
[1, 0.33395, 0],
[1, 0.32564, 0],
[1, 0.31719, 0],
[1, 0.30856, 0],
[1, 0.29971, 0],
[1, 0.29063, 0],
[1, 0.28132, 0],
[1, 0.27173, 0],
[1, 0.26187, 0],
[1, 0.25167, 0],
[1, 0.24106, 0],
[1, 0.23001, 0],
[1, 0.21851, 0],
[1, 0.20642, 0],
[1, 0.19361, 0],
[1, 0.17996, 0],
[1, 0.16528, 0],
]
b_rainbow_bgyr_35_85_c73 = bokeh_palette('rainbow_bgyr_35_85_c73',rainbow_bgyr_35_85_c73)
m_rainbow_bgyr_35_85_c73 = mpl_cm('rainbow_bgyr_35_85_c73',rainbow_bgyr_35_85_c73)
m_rainbow_bgyr_35_85_c73_r = mpl_cm('rainbow_bgyr_35_85_c73_r',list(reversed(rainbow_bgyr_35_85_c73)))
rainbow = b_rainbow_bgyr_35_85_c73
m_rainbow = m_rainbow_bgyr_35_85_c73
m_rainbow_r = m_rainbow_bgyr_35_85_c73_r
palette['rainbow'] = b_rainbow_bgyr_35_85_c73
palette_n['rainbow'] = b_rainbow_bgyr_35_85_c73
cm['rainbow'] = m_rainbow_bgyr_35_85_c73
cm['rainbow_r'] = m_rainbow_bgyr_35_85_c73_r
cm_n['rainbow'] = mpl_cm('rainbow',rainbow_bgyr_35_85_c73)
cm_n['rainbow_r'] = mpl_cm('rainbow_r',list(reversed(rainbow_bgyr_35_85_c73)))




rainbow_bgyr_35_85_c72 = [  # cmap_def
[0, 0.20387, 0.96251],
[0, 0.21524, 0.9514],
[0, 0.22613, 0.94031],
[0, 0.23654, 0.92923],
[0, 0.24654, 0.91817],
[0, 0.2562, 0.90712],
[0, 0.26557, 0.89608],
[0, 0.27465, 0.88506],
[0, 0.28348, 0.87405],
[0, 0.29209, 0.86305],
[0, 0.30047, 0.85206],
[0, 0.3087, 0.84109],
[0, 0.31672, 0.83013],
[0, 0.32458, 0.81917],
[0, 0.33232, 0.80823],
[0, 0.3399, 0.7973],
[0, 0.34736, 0.78638],
[0, 0.3547, 0.77546],
[0, 0.36191, 0.76456],
[0, 0.36902, 0.75367],
[0, 0.37602, 0.7428],
[0, 0.38291, 0.73193],
[0, 0.38969, 0.72108],
[0, 0.39636, 0.71026],
[0, 0.40292, 0.69946],
[0, 0.40934, 0.68872],
[0, 0.41561, 0.67802],
[0, 0.42172, 0.66738],
[0, 0.42768, 0.65684],
[0, 0.43342, 0.64639],
[0, 0.43896, 0.63605],
[0, 0.44432, 0.62583],
[0, 0.44945, 0.61575],
[0, 0.45438, 0.60579],
[0, 0.45911, 0.59597],
[0.0043377, 0.46367, 0.58627],
[0.029615, 0.46807, 0.57668],
[0.055795, 0.47235, 0.56717],
[0.077065, 0.47652, 0.55774],
[0.095292, 0.48061, 0.54837],
[0.11119, 0.48465, 0.53903],
[0.1253, 0.48865, 0.52971],
[0.13799, 0.49262, 0.5204],
[0.14937, 0.49658, 0.5111],
[0.15963, 0.50055, 0.50179],
[0.169, 0.50452, 0.49244],
[0.17747, 0.50849, 0.48309],
[0.18517, 0.51246, 0.4737],
[0.19217, 0.51645, 0.46429],
[0.19856, 0.52046, 0.45483],
[0.20443, 0.52448, 0.44531],
[0.20974, 0.52851, 0.43577],
[0.21461, 0.53255, 0.42616],
[0.21905, 0.53661, 0.41651],
[0.22309, 0.54066, 0.40679],
[0.22674, 0.54474, 0.397],
[0.23002, 0.54883, 0.38713],
[0.233, 0.55292, 0.3772],
[0.23568, 0.55703, 0.36716],
[0.23802, 0.56114, 0.35704],
[0.24006, 0.56526, 0.34678],
[0.24185, 0.56939, 0.3364],
[0.24334, 0.57354, 0.32588],
[0.24458, 0.57769, 0.31523],
[0.24556, 0.58185, 0.30439],
[0.2463, 0.58603, 0.29336],
[0.2468, 0.59019, 0.28214],
[0.24707, 0.59438, 0.27067],
[0.24714, 0.59856, 0.25896],
[0.24703, 0.60275, 0.24696],
[0.24679, 0.60693, 0.23473],
[0.24647, 0.61109, 0.22216],
[0.24615, 0.61523, 0.20936],
[0.24595, 0.61936, 0.19632],
[0.246, 0.62342, 0.18304],
[0.24644, 0.62742, 0.16969],
[0.24748, 0.63135, 0.1563],
[0.24925, 0.63518, 0.14299],
[0.25196, 0.6389, 0.13001],
[0.2557, 0.64249, 0.11741],
[0.26057, 0.64594, 0.10557],
[0.26659, 0.64926, 0.094696],
[0.27372, 0.65242, 0.084904],
[0.28182, 0.65545, 0.076489],
[0.29078, 0.65835, 0.069753],
[0.30043, 0.66113, 0.064513],
[0.31061, 0.66383, 0.060865],
[0.32112, 0.66642, 0.058721],
[0.33186, 0.66896, 0.057692],
[0.34272, 0.67144, 0.057693],
[0.35356, 0.67388, 0.058443],
[0.36439, 0.67628, 0.059738],
[0.37512, 0.67866, 0.061142],
[0.38575, 0.68102, 0.062974],
[0.39627, 0.68337, 0.064759],
[0.40666, 0.68571, 0.066664],
[0.41692, 0.68803, 0.068644],
[0.42707, 0.69034, 0.070512],
[0.43709, 0.69266, 0.072423],
[0.44701, 0.69494, 0.074359],
[0.45683, 0.69723, 0.076211],
[0.46657, 0.6995, 0.07809],
[0.47621, 0.70177, 0.079998],
[0.48577, 0.70403, 0.081943],
[0.49527, 0.70629, 0.083778],
[0.5047, 0.70853, 0.085565],
[0.51405, 0.71076, 0.087502],
[0.52335, 0.71298, 0.089316],
[0.53259, 0.7152, 0.091171],
[0.54176, 0.7174, 0.092931],
[0.5509, 0.7196, 0.094839],
[0.55999, 0.72178, 0.096566],
[0.56902, 0.72396, 0.098445],
[0.57802, 0.72613, 0.10023],
[0.58698, 0.72828, 0.10204],
[0.5959, 0.73044, 0.10385],
[0.60479, 0.73258, 0.10564],
[0.61363, 0.73471, 0.10744],
[0.62246, 0.73683, 0.10925],
[0.63125, 0.73895, 0.11102],
[0.64001, 0.74104, 0.11282],
[0.64874, 0.74315, 0.11452],
[0.65745, 0.74523, 0.11636],
[0.66613, 0.74731, 0.11813],
[0.67479, 0.74937, 0.11986],
[0.68343, 0.75144, 0.12161],
[0.69205, 0.75348, 0.12338],
[0.70065, 0.75552, 0.12517],
[0.70923, 0.75755, 0.12691],
[0.71779, 0.75957, 0.12868],
[0.72633, 0.76158, 0.13048],
[0.73487, 0.76359, 0.13221],
[0.74338, 0.76559, 0.13396],
[0.75188, 0.76756, 0.13568],
[0.76037, 0.76954, 0.13747],
[0.76884, 0.77151, 0.13917],
[0.77731, 0.77346, 0.14097],
[0.78576, 0.77541, 0.14269],
[0.7942, 0.77736, 0.14444],
[0.80262, 0.77928, 0.14617],
[0.81105, 0.7812, 0.14791],
[0.81945, 0.78311, 0.14967],
[0.82786, 0.78502, 0.15138],
[0.83626, 0.78691, 0.15311],
[0.84465, 0.7888, 0.15486],
[0.85304, 0.79066, 0.15662],
[0.86141, 0.79251, 0.15835],
[0.86978, 0.79434, 0.16002],
[0.87814, 0.79612, 0.16178],
[0.88647, 0.79786, 0.16346],
[0.89477, 0.79952, 0.16507],
[0.90301, 0.80106, 0.1667],
[0.91115, 0.80245, 0.16819],
[0.91917, 0.80364, 0.16964],
[0.92701, 0.80456, 0.1709],
[0.93459, 0.80514, 0.172],
[0.94185, 0.80532, 0.17289],
[0.94869, 0.80504, 0.17355],
[0.95506, 0.80424, 0.17392],
[0.96088, 0.80289, 0.17399],
[0.96609, 0.80097, 0.17375],
[0.97069, 0.7985, 0.17319],
[0.97465, 0.79549, 0.17234],
[0.97801, 0.79201, 0.17121],
[0.98082, 0.7881, 0.16986],
[0.98314, 0.78384, 0.16825],
[0.98504, 0.77928, 0.16652],
[0.9866, 0.7745, 0.16463],
[0.98789, 0.76955, 0.16265],
[0.98897, 0.76449, 0.16056],
[0.9899, 0.75932, 0.15848],
[0.99072, 0.75411, 0.15634],
[0.99146, 0.74885, 0.15414],
[0.99214, 0.74356, 0.15196],
[0.99279, 0.73825, 0.14981],
[0.9934, 0.73293, 0.1476],
[0.99398, 0.72759, 0.14543],
[0.99454, 0.72224, 0.1432],
[0.99509, 0.71689, 0.14103],
[0.99562, 0.71152, 0.1388],
[0.99613, 0.70614, 0.13659],
[0.99662, 0.70075, 0.13444],
[0.9971, 0.69534, 0.13223],
[0.99755, 0.68993, 0.13006],
[0.998, 0.6845, 0.12783],
[0.99842, 0.67906, 0.12564],
[0.99883, 0.67361, 0.1234],
[0.99922, 0.66815, 0.12119],
[0.99959, 0.66267, 0.11904],
[0.99994, 0.65717, 0.11682],
[1, 0.65166, 0.11458],
[1, 0.64613, 0.11244],
[1, 0.64059, 0.11024],
[1, 0.63503, 0.10797],
[1, 0.62945, 0.1058],
[1, 0.62386, 0.1036],
[1, 0.61825, 0.10135],
[1, 0.61261, 0.099135],
[1, 0.60697, 0.096882],
[1, 0.6013, 0.094743],
[1, 0.59561, 0.092465],
[1, 0.58989, 0.090257],
[1, 0.58416, 0.088032],
[1, 0.5784, 0.085726],
[1, 0.57263, 0.083542],
[1, 0.56682, 0.081316],
[1, 0.56098, 0.079004],
[1, 0.55513, 0.076745],
[1, 0.54925, 0.07453],
[1, 0.54333, 0.072245],
[1, 0.53739, 0.070004],
[1, 0.53141, 0.067732],
[1, 0.52541, 0.065424],
[1, 0.51937, 0.06318],
[1, 0.5133, 0.06081],
[1, 0.50718, 0.058502],
[1, 0.50104, 0.056232],
[1, 0.49486, 0.053826],
[1, 0.48863, 0.051494],
[1, 0.48236, 0.049242],
[1, 0.47605, 0.046828],
[1, 0.46969, 0.044447],
[1, 0.46327, 0.042093],
[1, 0.45681, 0.039648],
[1, 0.45031, 0.037261],
[1, 0.44374, 0.034882],
[1, 0.43712, 0.032495],
[1, 0.43043, 0.030303],
[1, 0.42367, 0.02818],
[1, 0.41686, 0.026121],
[1, 0.40997, 0.024126],
[1, 0.40299, 0.022194],
[1, 0.39595, 0.020325],
[1, 0.38882, 0.018517],
[0.99994, 0.38159, 0.016771],
[0.99961, 0.37428, 0.015085],
[0.99927, 0.36685, 0.013457],
[0.99892, 0.35932, 0.011916],
[0.99855, 0.35167, 0.010169],
[0.99817, 0.3439, 0.0087437],
[0.99778, 0.336, 0.0073541],
[0.99738, 0.32796, 0.0060199],
[0.99696, 0.31976, 0.0047429],
[0.99653, 0.31138, 0.0035217],
[0.99609, 0.30282, 0.0023557],
[0.99563, 0.29407, 0.0012445],
[0.99517, 0.2851, 0.00018742],
[0.99469, 0.27591, 0],
[0.9942, 0.26642, 0],
[0.99369, 0.25664, 0],
[0.99318, 0.24652, 0],
[0.99265, 0.23605, 0],
[0.99211, 0.22511, 0],
[0.99155, 0.2137, 0],
[0.99099, 0.20169, 0],
[0.99041, 0.18903, 0],
]
b_rainbow_bgyr_35_85_c72 = bokeh_palette('rainbow_bgyr_35_85_c72',rainbow_bgyr_35_85_c72)
m_rainbow_bgyr_35_85_c72 = mpl_cm('rainbow_bgyr_35_85_c72',rainbow_bgyr_35_85_c72)
m_rainbow_bgyr_35_85_c72_r = mpl_cm('rainbow_bgyr_35_85_c72_r',list(reversed(rainbow_bgyr_35_85_c72)))
CET_R2 = b_rainbow_bgyr_35_85_c72
m_CET_R2 = m_rainbow_bgyr_35_85_c72
m_CET_R2_r = m_rainbow_bgyr_35_85_c72_r
palette['CET_R2'] = b_rainbow_bgyr_35_85_c72
cm['CET_R2'] = m_rainbow_bgyr_35_85_c72
cm['CET_R2_r'] = m_rainbow_bgyr_35_85_c72_r
register_cmap('cet_CET_R2',m_rainbow_bgyr_35_85_c72)
register_cmap('cet_CET_R2_r',m_rainbow_bgyr_35_85_c72_r)




linear_kry_5_98_c75 = [  # cmap_def
[0.066001, 0.066024, 0.066019],
[0.080538, 0.066611, 0.064712],
[0.093113, 0.067208, 0.063412],
[0.10433, 0.067823, 0.062042],
[0.11447, 0.068466, 0.060693],
[0.12392, 0.069079, 0.059406],
[0.13324, 0.069387, 0.057942],
[0.1425, 0.069367, 0.056714],
[0.1517, 0.069013, 0.055346],
[0.16087, 0.068361, 0.05402],
[0.16995, 0.067575, 0.052699],
[0.17892, 0.066658, 0.051395],
[0.18783, 0.065607, 0.050099],
[0.19664, 0.064412, 0.04882],
[0.20537, 0.063105, 0.047405],
[0.21407, 0.061489, 0.046005],
[0.2227, 0.059924, 0.044746],
[0.23128, 0.057935, 0.043254],
[0.23978, 0.055984, 0.042035],
[0.24826, 0.053694, 0.040618],
[0.25665, 0.051268, 0.039228],
[0.265, 0.048696, 0.03786],
[0.2733, 0.045729, 0.036506],
[0.28153, 0.042641, 0.03519],
[0.28972, 0.039309, 0.033667],
[0.29783, 0.035771, 0.032419],
[0.3059, 0.031918, 0.031169],
[0.31386, 0.028215, 0.029947],
[0.32176, 0.024452, 0.028752],
[0.32961, 0.020653, 0.027583],
[0.33736, 0.016844, 0.026442],
[0.34502, 0.013048, 0.025331],
[0.35258, 0.0091504, 0.024255],
[0.36003, 0.005529, 0.023213],
[0.36736, 0.0020418, 0.022204],
[0.37459, 0, 0.021232],
[0.38167, 0, 0.020297],
[0.38862, 0, 0.019403],
[0.39539, 0, 0.018552],
[0.40197, 0, 0.017747],
[0.40837, 0, 0.016994],
[0.41456, 0, 0.016295],
[0.42062, 0, 0.015634],
[0.42666, 0, 0.014992],
[0.4327, 0, 0.014366],
[0.43874, 0, 0.013752],
[0.44481, 0, 0.013144],
[0.45086, 0, 0.012574],
[0.45693, 0, 0.012021],
[0.463, 0, 0.011365],
[0.4691, 0, 0.010683],
[0.47519, 0, 0.010135],
[0.48128, 0, 0.0096387],
[0.48739, 0, 0.0091289],
[0.49349, 0, 0.0086244],
[0.49961, 0, 0.0081373],
[0.50573, 0, 0.0076619],
[0.51187, 0, 0.0071948],
[0.51802, 0, 0.0067375],
[0.52416, 0, 0.0062922],
[0.53031, 0, 0.0058577],
[0.53647, 0, 0.0054319],
[0.54263, 0, 0.0050145],
[0.54881, 0, 0.0046053],
[0.55498, 0, 0.0042044],
[0.56116, 0, 0.0038134],
[0.56735, 0, 0.0034314],
[0.57355, 0, 0.0030565],
[0.57975, 0, 0.0026882],
[0.58596, 0, 0.0023271],
[0.59219, 0, 0.0019742],
[0.59842, 0, 0.0016322],
[0.60468, 0, 0.001301],
[0.61096, 0, 0.00097818],
[0.61724, 0, 0.00066352],
[0.62356, 0, 0.00035687],
[0.62988, 0, 5.8077e-05],
[0.63622, 0, 0],
[0.64258, 0, 0],
[0.64896, 0, 0],
[0.65535, 0, 0],
[0.66176, 0, 0],
[0.66819, 0, 0],
[0.67463, 0, 0],
[0.68108, 0, 0],
[0.68756, 0, 0],
[0.69405, 0, 0],
[0.70057, 0, 0],
[0.7071, 0, 0],
[0.71364, 0, 0],
[0.7202, 0, 0],
[0.72678, 0, 0],
[0.73337, 0, 0],
[0.73999, 0, 0],
[0.74661, 0, 0],
[0.75325, 0, 0],
[0.75991, 0, 0],
[0.76657, 0, 0],
[0.77322, 0, 0],
[0.77985, 0, 0],
[0.78648, 0, 0],
[0.79309, 0, 0],
[0.79969, 0, 0],
[0.80627, 0, 0],
[0.81283, 0, 0],
[0.81939, 0, 0],
[0.82593, 0, 0],
[0.83246, 0, 0],
[0.83897, 0, 0],
[0.84547, 0, 0],
[0.85195, 0, 0],
[0.85842, 0, 0],
[0.86487, 0, 0],
[0.87132, 0, 0],
[0.87774, 0, 0],
[0.88415, 0, 0],
[0.89054, 0, 0],
[0.89692, 0, 0],
[0.90328, 0, 0],
[0.90963, 0, 0],
[0.91595, 0, 0],
[0.92227, 0.0017832, 0],
[0.92856, 0.0061612, 0],
[0.93484, 0.010887, 0],
[0.94104, 0.016478, 0],
[0.9471, 0.023267, 0],
[0.95296, 0.031841, 0],
[0.95862, 0.042415, 0],
[0.96409, 0.053281, 0],
[0.96937, 0.064278, 0],
[0.97446, 0.075188, 0],
[0.97936, 0.086066, 0],
[0.98405, 0.096997, 0],
[0.98853, 0.10794, 0],
[0.99282, 0.11888, 0],
[0.9969, 0.12973, 0],
[1, 0.14054, 0],
[1, 0.1513, 0],
[1, 0.1621, 0],
[1, 0.17286, 0],
[1, 0.18361, 0],
[1, 0.19437, 0],
[1, 0.20505, 0],
[1, 0.21574, 0],
[1, 0.22642, 0],
[1, 0.23711, 0],
[1, 0.24776, 0],
[1, 0.25838, 0],
[1, 0.26903, 0],
[1, 0.27963, 0],
[1, 0.2902, 0],
[1, 0.30076, 0],
[1, 0.31115, 0],
[1, 0.32126, 0],
[1, 0.33113, 0],
[1, 0.34081, 0],
[1, 0.35027, 0],
[1, 0.35954, 0],
[1, 0.36866, 0.00010807],
[1, 0.3776, 0.0016338],
[1, 0.3864, 0.0031726],
[1, 0.39508, 0.0047225],
[1, 0.40364, 0.0062803],
[1, 0.41207, 0.0078416],
[1, 0.42039, 0.0094129],
[1, 0.42862, 0.011021],
[1, 0.43675, 0.012734],
[1, 0.44479, 0.014303],
[1, 0.45274, 0.015864],
[1, 0.4606, 0.017417],
[1, 0.4684, 0.018959],
[1, 0.47614, 0.020491],
[1, 0.48379, 0.022011],
[1, 0.4914, 0.023517],
[1, 0.49892, 0.025006],
[1, 0.5064, 0.026472],
[1, 0.51383, 0.027913],
[1, 0.52119, 0.02933],
[1, 0.52852, 0.030724],
[1, 0.53577, 0.032109],
[1, 0.54298, 0.033495],
[1, 0.55012, 0.035116],
[1, 0.55721, 0.036562],
[1, 0.56423, 0.038032],
[1, 0.57119, 0.039515],
[1, 0.57811, 0.041015],
[1, 0.58499, 0.042457],
[1, 0.59181, 0.043871],
[1, 0.59857, 0.045349],
[1, 0.6053, 0.046804],
[1, 0.61199, 0.048174],
[1, 0.61864, 0.049638],
[1, 0.62524, 0.050967],
[1, 0.6318, 0.052314],
[1, 0.63833, 0.053669],
[1, 0.64482, 0.05503],
[1, 0.65128, 0.056416],
[1, 0.6577, 0.057622],
[1, 0.66409, 0.05901],
[1, 0.67045, 0.060291],
[1, 0.67678, 0.061456],
[1, 0.68308, 0.062815],
[1, 0.68935, 0.063988],
[1, 0.69558, 0.065188],
[1, 0.70178, 0.06638],
[1, 0.70798, 0.067574],
[1, 0.71415, 0.068771],
[1, 0.72028, 0.069853],
[1, 0.72637, 0.07091],
[1, 0.73244, 0.071928],
[1, 0.73847, 0.072954],
[1, 0.74448, 0.073945],
[1, 0.75047, 0.074848],
[1, 0.75643, 0.075762],
[1, 0.76237, 0.076661],
[1, 0.7683, 0.077556],
[1, 0.77421, 0.078448],
[1, 0.78011, 0.079336],
[1, 0.786, 0.080219],
[1, 0.79186, 0.081101],
[1, 0.79771, 0.081992],
[1, 0.80356, 0.082776],
[1, 0.80939, 0.083622],
[1, 0.81521, 0.084477],
[1, 0.82102, 0.085237],
[1, 0.82682, 0.086035],
[1, 0.83261, 0.086929],
[1, 0.83839, 0.087733],
[1, 0.84416, 0.088519],
[1, 0.84993, 0.089324],
[1, 0.85569, 0.090123],
[1, 0.86143, 0.090933],
[1, 0.86717, 0.091749],
[1, 0.87291, 0.092489],
[1, 0.87864, 0.093256],
[1, 0.88437, 0.094098],
[1, 0.89009, 0.094916],
[1, 0.8958, 0.095666],
[1, 0.90151, 0.096403],
[1, 0.90721, 0.097221],
[1, 0.91291, 0.098051],
[1, 0.9186, 0.098807],
[1, 0.92429, 0.099584],
[1, 0.92997, 0.10036],
[1, 0.93565, 0.10115],
[1, 0.94133, 0.10194],
[1, 0.947, 0.10274],
[1, 0.95267, 0.10354],
[1, 0.95834, 0.10428],
[1, 0.964, 0.10506],
[1, 0.96966, 0.10589],
[1, 0.97532, 0.1067],
[1, 0.98098, 0.10746],
[1, 0.98663, 0.10821],
[1, 0.99228, 0.10903],
[1, 0.99792, 0.10986],
]
b_linear_kry_5_98_c75 = bokeh_palette('linear_kry_5_98_c75',linear_kry_5_98_c75)
m_linear_kry_5_98_c75 = mpl_cm('linear_kry_5_98_c75',linear_kry_5_98_c75)
m_linear_kry_5_98_c75_r = mpl_cm('linear_kry_5_98_c75_r',list(reversed(linear_kry_5_98_c75)))




linear_blue_95_50_c20 = [  # cmap_def
[0.94334, 0.94353, 0.94348],
[0.94036, 0.94166, 0.94325],
[0.93738, 0.9398, 0.94302],
[0.93439, 0.93793, 0.94278],
[0.93141, 0.93607, 0.94255],
[0.92843, 0.9342, 0.94231],
[0.92545, 0.93234, 0.94206],
[0.92247, 0.93048, 0.94182],
[0.91948, 0.92862, 0.94157],
[0.9165, 0.92676, 0.94131],
[0.91352, 0.9249, 0.94106],
[0.91054, 0.92304, 0.9408],
[0.90755, 0.92118, 0.94054],
[0.90456, 0.91932, 0.94028],
[0.90157, 0.91747, 0.94001],
[0.89858, 0.91561, 0.93974],
[0.8956, 0.91376, 0.93947],
[0.89261, 0.9119, 0.93919],
[0.88962, 0.91005, 0.93891],
[0.88663, 0.9082, 0.93863],
[0.88364, 0.90635, 0.93834],
[0.88065, 0.90449, 0.93805],
[0.87766, 0.90264, 0.93776],
[0.87466, 0.90079, 0.93746],
[0.87167, 0.89894, 0.93716],
[0.86868, 0.8971, 0.93686],
[0.86569, 0.89525, 0.93655],
[0.8627, 0.89339, 0.93624],
[0.8597, 0.89155, 0.93593],
[0.8567, 0.8897, 0.93561],
[0.8537, 0.88785, 0.93528],
[0.85071, 0.88601, 0.93496],
[0.84771, 0.88417, 0.93463],
[0.84472, 0.88232, 0.9343],
[0.84172, 0.88048, 0.93396],
[0.83872, 0.87864, 0.93362],
[0.83572, 0.87679, 0.93327],
[0.83272, 0.87495, 0.93292],
[0.82973, 0.87311, 0.93257],
[0.82673, 0.87127, 0.93222],
[0.82373, 0.86943, 0.93186],
[0.82073, 0.86759, 0.93149],
[0.81773, 0.86575, 0.93112],
[0.81473, 0.86392, 0.93075],
[0.81173, 0.86208, 0.93037],
[0.80873, 0.86025, 0.92999],
[0.80573, 0.85841, 0.9296],
[0.80273, 0.85657, 0.92921],
[0.79973, 0.85474, 0.92881],
[0.79673, 0.85291, 0.92841],
[0.79374, 0.85107, 0.928],
[0.79073, 0.84924, 0.92759],
[0.78773, 0.8474, 0.92717],
[0.78474, 0.84557, 0.92675],
[0.78174, 0.84374, 0.92633],
[0.77873, 0.84191, 0.9259],
[0.77573, 0.84009, 0.92546],
[0.77274, 0.83825, 0.92502],
[0.76974, 0.83642, 0.92457],
[0.76674, 0.8346, 0.92412],
[0.76375, 0.83277, 0.92366],
[0.76075, 0.83094, 0.92319],
[0.75776, 0.82912, 0.92272],
[0.75477, 0.82729, 0.92224],
[0.75178, 0.82546, 0.92176],
[0.74878, 0.82363, 0.92127],
[0.74579, 0.82181, 0.92077],
[0.7428, 0.81998, 0.92027],
[0.73982, 0.81816, 0.91976],
[0.73683, 0.81634, 0.91924],
[0.73384, 0.81451, 0.91871],
[0.73086, 0.81269, 0.91818],
[0.72788, 0.81087, 0.91765],
[0.72491, 0.80905, 0.9171],
[0.72192, 0.80723, 0.91654],
[0.71895, 0.8054, 0.91598],
[0.71599, 0.80358, 0.91541],
[0.71301, 0.80176, 0.91483],
[0.71005, 0.79995, 0.91424],
[0.70709, 0.79812, 0.91365],
[0.70413, 0.7963, 0.91304],
[0.70117, 0.79449, 0.91242],
[0.69822, 0.79266, 0.91179],
[0.69528, 0.79084, 0.91116],
[0.69234, 0.78903, 0.91051],
[0.6894, 0.78721, 0.90986],
[0.68646, 0.78539, 0.90919],
[0.68354, 0.78357, 0.90851],
[0.68061, 0.78175, 0.90782],
[0.67771, 0.77993, 0.90711],
[0.6748, 0.77812, 0.9064],
[0.6719, 0.7763, 0.90567],
[0.66901, 0.77447, 0.90492],
[0.66612, 0.77266, 0.90416],
[0.66325, 0.77084, 0.90339],
[0.66038, 0.76902, 0.9026],
[0.65753, 0.7672, 0.90179],
[0.65467, 0.76538, 0.90097],
[0.65185, 0.76356, 0.90013],
[0.64903, 0.76174, 0.89928],
[0.64622, 0.75992, 0.89839],
[0.64342, 0.7581, 0.8975],
[0.64065, 0.75627, 0.89658],
[0.63789, 0.75445, 0.89563],
[0.63514, 0.75262, 0.89466],
[0.63243, 0.7508, 0.89366],
[0.62972, 0.74897, 0.89264],
[0.62705, 0.74715, 0.89159],
[0.6244, 0.74531, 0.8905],
[0.62177, 0.74349, 0.88938],
[0.61919, 0.74165, 0.88823],
[0.61663, 0.73982, 0.88703],
[0.6141, 0.73798, 0.88579],
[0.61162, 0.73613, 0.8845],
[0.60918, 0.73429, 0.88317],
[0.60674, 0.73245, 0.88182],
[0.60432, 0.73061, 0.88046],
[0.6019, 0.72876, 0.87909],
[0.5995, 0.72692, 0.87772],
[0.59711, 0.72508, 0.87632],
[0.59471, 0.72323, 0.87492],
[0.59235, 0.72139, 0.87351],
[0.58997, 0.71955, 0.87209],
[0.58761, 0.71771, 0.87065],
[0.58527, 0.71586, 0.86921],
[0.58292, 0.71402, 0.86776],
[0.58059, 0.71217, 0.86629],
[0.57827, 0.71033, 0.86481],
[0.57597, 0.70849, 0.86333],
[0.57366, 0.70665, 0.86183],
[0.57136, 0.7048, 0.86033],
[0.56908, 0.70296, 0.8588],
[0.56682, 0.70112, 0.85727],
[0.56455, 0.69927, 0.85574],
[0.5623, 0.69743, 0.85418],
[0.56006, 0.69559, 0.85263],
[0.55782, 0.69375, 0.85105],
[0.55559, 0.69191, 0.84947],
[0.55338, 0.69007, 0.84788],
[0.55118, 0.68823, 0.84627],
[0.549, 0.68638, 0.84466],
[0.54681, 0.68454, 0.84304],
[0.54464, 0.6827, 0.8414],
[0.54248, 0.68085, 0.83976],
[0.54033, 0.67901, 0.8381],
[0.53819, 0.67718, 0.83644],
[0.53605, 0.67533, 0.83476],
[0.53393, 0.67349, 0.83307],
[0.53183, 0.67165, 0.83138],
[0.52973, 0.66981, 0.82967],
[0.52764, 0.66797, 0.82795],
[0.52557, 0.66613, 0.82622],
[0.5235, 0.66429, 0.82448],
[0.52144, 0.66245, 0.82274],
[0.5194, 0.6606, 0.82097],
[0.51737, 0.65876, 0.8192],
[0.51534, 0.65693, 0.81742],
[0.51333, 0.65508, 0.81564],
[0.51133, 0.65324, 0.81383],
[0.50934, 0.65141, 0.81203],
[0.50737, 0.64957, 0.8102],
[0.50539, 0.64772, 0.80838],
[0.50344, 0.64588, 0.80654],
[0.5015, 0.64404, 0.80469],
[0.49956, 0.64221, 0.80282],
[0.49765, 0.64037, 0.80095],
[0.49573, 0.63853, 0.79907],
[0.49383, 0.6367, 0.79718],
[0.49195, 0.63486, 0.79528],
[0.49007, 0.63302, 0.79337],
[0.4882, 0.63118, 0.79145],
[0.48631, 0.62934, 0.78955],
[0.48439, 0.62751, 0.78769],
[0.48243, 0.62568, 0.78587],
[0.48043, 0.62386, 0.78409],
[0.4784, 0.62203, 0.78233],
[0.47634, 0.62022, 0.78062],
[0.47423, 0.61841, 0.77894],
[0.47211, 0.6166, 0.77729],
[0.46995, 0.61479, 0.77567],
[0.46775, 0.61298, 0.77408],
[0.46552, 0.61119, 0.77252],
[0.46327, 0.60939, 0.77099],
[0.461, 0.6076, 0.76947],
[0.4587, 0.6058, 0.76799],
[0.45636, 0.60403, 0.76653],
[0.45401, 0.60223, 0.7651],
[0.45164, 0.60045, 0.76367],
[0.44923, 0.59867, 0.76228],
[0.4468, 0.5969, 0.76091],
[0.44436, 0.59512, 0.75956],
[0.44188, 0.59336, 0.75823],
[0.43937, 0.59159, 0.75691],
[0.43686, 0.58981, 0.75562],
[0.43431, 0.58805, 0.75434],
[0.43176, 0.5863, 0.75308],
[0.42917, 0.58454, 0.75184],
[0.42655, 0.58278, 0.75061],
[0.42392, 0.58102, 0.74939],
[0.42127, 0.57928, 0.74821],
[0.4186, 0.57752, 0.74702],
[0.41591, 0.57578, 0.74586],
[0.4132, 0.57403, 0.7447],
[0.41046, 0.57229, 0.74357],
[0.4077, 0.57054, 0.74245],
[0.40491, 0.56881, 0.74133],
[0.40211, 0.56708, 0.74024],
[0.39929, 0.56533, 0.73915],
[0.39645, 0.56361, 0.73807],
[0.39358, 0.56188, 0.73701],
[0.3907, 0.56015, 0.73596],
[0.38779, 0.55842, 0.73493],
[0.38485, 0.5567, 0.7339],
[0.3819, 0.55498, 0.73288],
[0.37893, 0.55325, 0.73187],
[0.37593, 0.55154, 0.73088],
[0.3729, 0.54983, 0.72989],
[0.36986, 0.54812, 0.72892],
[0.36679, 0.54641, 0.72795],
[0.36369, 0.54469, 0.72699],
[0.36057, 0.54299, 0.72605],
[0.35745, 0.54127, 0.72511],
[0.35426, 0.53957, 0.72418],
[0.35108, 0.53788, 0.72326],
[0.34785, 0.53617, 0.72235],
[0.3446, 0.53447, 0.72144],
[0.34133, 0.53278, 0.72055],
[0.33801, 0.53108, 0.71967],
[0.3347, 0.52939, 0.71879],
[0.33131, 0.5277, 0.71792],
[0.32793, 0.52601, 0.71706],
[0.3245, 0.52432, 0.7162],
[0.32105, 0.52263, 0.71535],
[0.31756, 0.52095, 0.7145],
[0.31404, 0.51927, 0.71368],
[0.31049, 0.5176, 0.71284],
[0.30689, 0.5159, 0.71203],
[0.30325, 0.51424, 0.71122],
[0.2996, 0.51256, 0.71041],
[0.29588, 0.51089, 0.70961],
[0.29213, 0.50922, 0.70882],
[0.28834, 0.50755, 0.70803],
[0.2845, 0.50587, 0.70724],
[0.28061, 0.50422, 0.70647],
[0.27671, 0.50255, 0.7057],
[0.27272, 0.50089, 0.70494],
[0.2687, 0.49923, 0.70418],
[0.26461, 0.49758, 0.70343],
[0.26047, 0.49591, 0.70268],
[0.25627, 0.49427, 0.70194],
[0.25203, 0.4926, 0.70121],
[0.24771, 0.49097, 0.70048],
[0.24329, 0.4893, 0.69975],
[0.23883, 0.48767, 0.69903],
[0.23431, 0.48601, 0.69831],
[0.22967, 0.48437, 0.6976],
]
b_linear_blue_95_50_c20 = bokeh_palette('linear_blue_95_50_c20',linear_blue_95_50_c20)
m_linear_blue_95_50_c20 = mpl_cm('linear_blue_95_50_c20',linear_blue_95_50_c20)
m_linear_blue_95_50_c20_r = mpl_cm('linear_blue_95_50_c20_r',list(reversed(linear_blue_95_50_c20)))
blues = b_linear_blue_95_50_c20
m_blues = m_linear_blue_95_50_c20
m_blues_r = m_linear_blue_95_50_c20_r
palette['blues'] = b_linear_blue_95_50_c20
palette_n['blues'] = b_linear_blue_95_50_c20
cm['blues'] = m_linear_blue_95_50_c20
cm['blues_r'] = m_linear_blue_95_50_c20_r
cm_n['blues'] = mpl_cm('blues',linear_blue_95_50_c20)
cm_n['blues_r'] = mpl_cm('blues_r',list(reversed(linear_blue_95_50_c20)))
CET_L12 = b_linear_blue_95_50_c20
m_CET_L12 = m_linear_blue_95_50_c20
m_CET_L12_r = m_linear_blue_95_50_c20_r
palette['CET_L12'] = b_linear_blue_95_50_c20
cm['CET_L12'] = m_linear_blue_95_50_c20
cm['CET_L12_r'] = m_linear_blue_95_50_c20_r
register_cmap('cet_CET_L12',m_linear_blue_95_50_c20)
register_cmap('cet_CET_L12_r',m_linear_blue_95_50_c20_r)




linear_ternary_red_0_50_c52 = [  # cmap_def
[0, 0, 0],
[0.0080248, 0.0015152, 4.5084e-17],
[0.016166, 0.0030536, 0],
[0.024126, 0.0045572, 6.0397e-18],
[0.032164, 0.0060809, 1.6594e-17],
[0.040345, 0.0076161, 0],
[0.047917, 0.0090568, 0],
[0.055209, 0.010404, 3.9777e-17],
[0.06236, 0.011803, 0],
[0.068888, 0.012988, 0],
[0.074917, 0.014153, 0],
[0.080792, 0.015256, 5.8811e-17],
[0.086386, 0.016319, 8.7851e-18],
[0.091832, 0.017341, 0],
[0.096947, 0.018317, 0],
[0.10199, 0.019262, 2.1475e-17],
[0.10688, 0.020182, 0],
[0.11156, 0.021071, 4.2461e-17],
[0.11612, 0.021929, 0],
[0.12051, 0.022761, 0],
[0.12482, 0.023572, 0],
[0.12901, 0.024369, 0],
[0.13312, 0.025144, 0],
[0.13712, 0.025899, 2.9284e-17],
[0.14104, 0.026637, 0],
[0.14484, 0.027362, 0],
[0.14852, 0.028076, 0],
[0.15218, 0.028765, 0],
[0.15579, 0.029444, 0],
[0.15929, 0.03011, 0],
[0.16274, 0.03076, 0],
[0.16612, 0.031401, 0],
[0.16947, 0.032031, 0],
[0.17271, 0.032655, 0],
[0.1759, 0.033257, 0],
[0.17909, 0.033824, 0],
[0.18223, 0.034436, 0],
[0.18542, 0.035067, 0],
[0.18847, 0.03563, 0],
[0.19142, 0.03617, 0],
[0.19437, 0.036729, 0],
[0.19725, 0.037267, 5.4019e-07],
[0.20012, 0.037802, 1.2837e-06],
[0.20303, 0.038349, 9.9959e-07],
[0.20594, 0.038898, 7.7892e-07],
[0.20884, 0.039449, 9.7112e-07],
[0.21176, 0.039999, 1.2788e-06],
[0.21466, 0.040546, 1.373e-06],
[0.21761, 0.0411, 1.2556e-06],
[0.22057, 0.041669, 9.5959e-07],
[0.22346, 0.042219, 8.1372e-07],
[0.22635, 0.042741, 1.1485e-06],
[0.22925, 0.043284, 1.3628e-06],
[0.23223, 0.04386, 1.3699e-06],
[0.23522, 0.044439, 1.2685e-06],
[0.23815, 0.045002, 1.0271e-06],
[0.24106, 0.045522, 7.9582e-07],
[0.24401, 0.046076, 1.0449e-06],
[0.24701, 0.046672, 1.3237e-06],
[0.24995, 0.047232, 1.4871e-06],
[0.25293, 0.047749, 1.4007e-06],
[0.25591, 0.048339, 1.1191e-06],
[0.25894, 0.048928, 8.5433e-07],
[0.26191, 0.049479, 9.2045e-07],
[0.26486, 0.050025, 1.3084e-06],
[0.26783, 0.05059, 1.4958e-06],
[0.27084, 0.051156, 1.5067e-06],
[0.27383, 0.051721, 1.3409e-06],
[0.27682, 0.052289, 1.0257e-06],
[0.27983, 0.052858, 7.8803e-07],
[0.28284, 0.053424, 1.0814e-06],
[0.28584, 0.053993, 1.3794e-06],
[0.28888, 0.054565, 1.5209e-06],
[0.2919, 0.055134, 1.522e-06],
[0.29494, 0.055708, 1.3561e-06],
[0.29798, 0.056296, 1.0594e-06],
[0.30101, 0.056866, 8.5564e-07],
[0.30401, 0.057414, 1.0969e-06],
[0.30705, 0.057986, 1.3643e-06],
[0.31013, 0.05858, 1.5001e-06],
[0.3132, 0.05917, 1.5634e-06],
[0.31627, 0.059754, 1.4993e-06],
[0.31931, 0.06032, 1.2688e-06],
[0.32234, 0.060876, 9.3009e-07],
[0.32538, 0.061448, 8.4705e-07],
[0.32848, 0.062046, 1.1943e-06],
[0.33157, 0.06265, 1.4594e-06],
[0.33467, 0.06322, 1.6339e-06],
[0.33772, 0.063788, 1.6369e-06],
[0.3408, 0.064374, 1.4847e-06],
[0.34388, 0.064956, 1.2226e-06],
[0.34698, 0.065538, 8.9134e-07],
[0.35008, 0.066125, 8.5068e-07],
[0.35318, 0.066712, 1.2269e-06],
[0.35629, 0.067299, 1.4894e-06],
[0.35939, 0.067883, 1.6295e-06],
[0.36252, 0.068473, 1.6472e-06],
[0.36565, 0.069079, 1.5852e-06],
[0.36876, 0.069658, 1.4248e-06],
[0.37185, 0.070223, 1.0684e-06],
[0.37498, 0.070835, 7.9019e-07],
[0.37813, 0.071435, 1.022e-06],
[0.38125, 0.072005, 1.3647e-06],
[0.38438, 0.072592, 1.5517e-06],
[0.38754, 0.073217, 1.6548e-06],
[0.39069, 0.073807, 1.7253e-06],
[0.39382, 0.074384, 1.6463e-06],
[0.39695, 0.074981, 1.4452e-06],
[0.40011, 0.075576, 1.1513e-06],
[0.40327, 0.076172, 8.5042e-07],
[0.40643, 0.076769, 8.941e-07],
[0.40959, 0.077367, 1.2367e-06],
[0.41277, 0.077967, 1.4894e-06],
[0.41594, 0.078566, 1.6548e-06],
[0.41912, 0.079167, 1.728e-06],
[0.4223, 0.079769, 1.7184e-06],
[0.4255, 0.080371, 1.6249e-06],
[0.4287, 0.080974, 1.4674e-06],
[0.43192, 0.081588, 1.2356e-06],
[0.43512, 0.082195, 9.4259e-07],
[0.43828, 0.08278, 7.201e-07],
[0.4415, 0.083388, 1.052e-06],
[0.44472, 0.084012, 1.3338e-06],
[0.44791, 0.084617, 1.593e-06],
[0.45112, 0.085205, 1.7373e-06],
[0.45433, 0.085801, 1.7608e-06],
[0.45756, 0.086427, 1.7212e-06],
[0.4608, 0.087056, 1.6842e-06],
[0.46404, 0.087652, 1.5738e-06],
[0.46727, 0.088257, 1.3578e-06],
[0.47049, 0.088871, 1.1045e-06],
[0.47373, 0.089482, 8.1745e-07],
[0.47697, 0.090095, 7.8044e-07],
[0.48022, 0.090707, 1.0884e-06],
[0.48348, 0.091327, 1.3399e-06],
[0.48673, 0.091949, 1.5673e-06],
[0.48996, 0.092545, 1.7249e-06],
[0.49321, 0.093154, 1.7614e-06],
[0.49649, 0.093784, 1.7613e-06],
[0.49977, 0.094411, 1.763e-06],
[0.50305, 0.095031, 1.7236e-06],
[0.50631, 0.09564, 1.6147e-06],
[0.50957, 0.096244, 1.4163e-06],
[0.51285, 0.096865, 1.1666e-06],
[0.51615, 0.097503, 9.3008e-07],
[0.51945, 0.098123, 7.3324e-07],
[0.52274, 0.098737, 8.3616e-07],
[0.52603, 0.099359, 1.0957e-06],
[0.52932, 0.099983, 1.3165e-06],
[0.53262, 0.10061, 1.4935e-06],
[0.53592, 0.10123, 1.6285e-06],
[0.53923, 0.10185, 1.7225e-06],
[0.54255, 0.10248, 1.7716e-06],
[0.54588, 0.10311, 1.8035e-06],
[0.54919, 0.10374, 1.8308e-06],
[0.55249, 0.10435, 1.7782e-06],
[0.5558, 0.10498, 1.6557e-06],
[0.55915, 0.10562, 1.5548e-06],
[0.56251, 0.10626, 1.4434e-06],
[0.56583, 0.10688, 1.3031e-06],
[0.56915, 0.10751, 1.1028e-06],
[0.57249, 0.10813, 8.5642e-07],
[0.57584, 0.10877, 6.5276e-07],
[0.57921, 0.10941, 6.6729e-07],
[0.58255, 0.11004, 8.9887e-07],
[0.58591, 0.11067, 1.1025e-06],
[0.58925, 0.1113, 1.264e-06],
[0.59262, 0.11194, 1.3991e-06],
[0.59599, 0.11257, 1.5204e-06],
[0.59936, 0.11322, 1.6369e-06],
[0.60271, 0.11385, 1.7384e-06],
[0.60606, 0.11447, 1.7667e-06],
[0.60945, 0.11512, 1.7698e-06],
[0.61284, 0.11576, 1.7921e-06],
[0.61623, 0.11641, 1.8189e-06],
[0.6196, 0.11703, 1.7927e-06],
[0.62299, 0.11767, 1.7298e-06],
[0.62639, 0.11833, 1.7048e-06],
[0.62978, 0.11896, 1.6692e-06],
[0.63317, 0.1196, 1.5768e-06],
[0.63657, 0.12024, 1.4791e-06],
[0.63997, 0.12088, 1.3801e-06],
[0.64337, 0.12153, 1.2686e-06],
[0.64679, 0.12217, 1.1466e-06],
[0.65019, 0.12281, 1.0261e-06],
[0.65362, 0.12346, 8.8816e-07],
[0.65704, 0.12411, 7.6247e-07],
[0.66046, 0.12475, 6.252e-07],
[0.6639, 0.1254, 5.0813e-07],
[0.66734, 0.12605, 5.4766e-07],
[0.67075, 0.12669, 4.2573e-07],
[0.67419, 0.12734, 2.3855e-08],
[0.67764, 0.128, 0],
[0.68106, 0.12864, 0],
[0.68451, 0.12929, 0],
[0.68797, 0.12996, 0],
[0.69142, 0.13061, 0],
[0.69486, 0.13125, 0],
[0.69832, 0.1319, 0],
[0.70177, 0.13256, 0],
[0.70524, 0.13321, 0],
[0.70871, 0.13387, 0],
[0.71218, 0.13453, 0],
[0.71565, 0.13518, 0],
[0.7191, 0.13582, 0],
[0.72258, 0.13648, 0],
[0.72607, 0.13715, 0],
[0.72956, 0.13781, 0],
[0.73304, 0.13847, 0],
[0.73651, 0.13912, 0],
[0.74001, 0.13977, 0],
[0.74351, 0.14044, 0],
[0.74701, 0.14111, 0],
[0.75051, 0.14176, 0],
[0.75401, 0.14242, 0],
[0.75751, 0.14309, 0],
[0.76102, 0.14375, 0],
[0.76454, 0.14441, 0],
[0.76805, 0.14508, 0],
[0.77157, 0.14575, 0],
[0.77507, 0.1464, 0],
[0.7786, 0.14707, 0],
[0.78213, 0.14774, 0],
[0.78565, 0.14839, 0],
[0.78919, 0.14907, 0],
[0.79273, 0.14975, 0],
[0.79625, 0.1504, 0],
[0.7998, 0.15107, 0],
[0.80333, 0.15174, 0],
[0.80688, 0.15241, 0],
[0.81042, 0.15308, 0],
[0.81397, 0.15375, 0],
[0.81752, 0.15442, 0],
[0.82108, 0.15509, 0],
[0.82465, 0.15577, 0],
[0.82821, 0.15645, 0],
[0.83177, 0.15711, 0],
[0.83534, 0.15778, 0],
[0.83891, 0.15847, 0],
[0.84248, 0.15914, 0],
[0.84604, 0.1598, 0],
[0.84963, 0.16048, 0],
[0.85322, 0.16117, 0],
[0.8568, 0.16184, 0],
[0.86038, 0.16251, 0],
[0.86398, 0.16319, 0],
[0.86757, 0.16388, 0],
[0.87115, 0.16455, 0],
[0.87473, 0.16523, 0],
[0.87834, 0.16591, 0],
[0.88196, 0.16659, 0],
[0.88556, 0.16727, 0],
[0.88915, 0.16795, 0],
[0.89277, 0.16863, 0],
[0.89639, 0.16932, 0],
[0.9, 0.17, 2.0303e-16],
]
b_linear_ternary_red_0_50_c52 = bokeh_palette('linear_ternary_red_0_50_c52',linear_ternary_red_0_50_c52)
m_linear_ternary_red_0_50_c52 = mpl_cm('linear_ternary_red_0_50_c52',linear_ternary_red_0_50_c52)
m_linear_ternary_red_0_50_c52_r = mpl_cm('linear_ternary_red_0_50_c52_r',list(reversed(linear_ternary_red_0_50_c52)))
kr = b_linear_ternary_red_0_50_c52
m_kr = m_linear_ternary_red_0_50_c52
m_kr_r = m_linear_ternary_red_0_50_c52_r
palette['kr'] = b_linear_ternary_red_0_50_c52
palette_n['kr'] = b_linear_ternary_red_0_50_c52
cm['kr'] = m_linear_ternary_red_0_50_c52
cm['kr_r'] = m_linear_ternary_red_0_50_c52_r
cm_n['kr'] = mpl_cm('kr',linear_ternary_red_0_50_c52)
cm_n['kr_r'] = mpl_cm('kr_r',list(reversed(linear_ternary_red_0_50_c52)))
CET_L13 = b_linear_ternary_red_0_50_c52
m_CET_L13 = m_linear_ternary_red_0_50_c52
m_CET_L13_r = m_linear_ternary_red_0_50_c52_r
palette['CET_L13'] = b_linear_ternary_red_0_50_c52
cm['CET_L13'] = m_linear_ternary_red_0_50_c52
cm['CET_L13_r'] = m_linear_ternary_red_0_50_c52_r
register_cmap('cet_CET_L13',m_linear_ternary_red_0_50_c52)
register_cmap('cet_CET_L13_r',m_linear_ternary_red_0_50_c52_r)




linear_bgyw_15_100_c67 = [  # cmap_def
[0.10477, 0.0011197, 0.51875],
[0.105, 0.0031206, 0.52587],
[0.10521, 0.005201, 0.53301],
[0.10541, 0.0074297, 0.5401],
[0.10559, 0.0097752, 0.54718],
[0.10576, 0.012459, 0.55425],
[0.10592, 0.015073, 0.56126],
[0.10605, 0.017812, 0.56829],
[0.10619, 0.020743, 0.57524],
[0.10631, 0.023778, 0.5822],
[0.10643, 0.027021, 0.58909],
[0.10653, 0.030386, 0.59599],
[0.10663, 0.033922, 0.60282],
[0.10672, 0.037861, 0.60963],
[0.10682, 0.041783, 0.6164],
[0.10691, 0.045535, 0.62314],
[0.107, 0.049457, 0.62982],
[0.10709, 0.053176, 0.63647],
[0.10718, 0.057025, 0.64308],
[0.10728, 0.060795, 0.64963],
[0.10738, 0.064591, 0.65615],
[0.1075, 0.068417, 0.6626],
[0.10762, 0.072133, 0.66902],
[0.10776, 0.075909, 0.67536],
[0.10791, 0.079695, 0.68166],
[0.10808, 0.08353, 0.6879],
[0.10826, 0.087375, 0.69408],
[0.10848, 0.091188, 0.7002],
[0.10873, 0.095035, 0.70626],
[0.10901, 0.098847, 0.71224],
[0.10933, 0.10273, 0.71815],
[0.10965, 0.10666, 0.724],
[0.11, 0.11057, 0.72975],
[0.11035, 0.11443, 0.73545],
[0.11077, 0.11848, 0.74103],
[0.11121, 0.1224, 0.74656],
[0.11171, 0.12647, 0.75198],
[0.11224, 0.13056, 0.75731],
[0.11282, 0.13464, 0.76255],
[0.11344, 0.13874, 0.76767],
[0.11404, 0.14289, 0.77271],
[0.11474, 0.14709, 0.77762],
[0.11555, 0.15131, 0.78241],
[0.11637, 0.1556, 0.78711],
[0.11721, 0.15989, 0.79164],
[0.11817, 0.16429, 0.79606],
[0.11913, 0.16868, 0.80035],
[0.12014, 0.17313, 0.80446],
[0.12122, 0.17766, 0.80845],
[0.12236, 0.18216, 0.81227],
[0.12358, 0.1868, 0.8159],
[0.12487, 0.19146, 0.81936],
[0.12621, 0.19619, 0.82267],
[0.12758, 0.20095, 0.82573],
[0.12902, 0.20585, 0.82858],
[0.13057, 0.21075, 0.83123],
[0.13209, 0.21574, 0.83365],
[0.1337, 0.22083, 0.83579],
[0.13536, 0.22598, 0.83764],
[0.13708, 0.23121, 0.83921],
[0.13882, 0.23653, 0.84047],
[0.14063, 0.24194, 0.84141],
[0.14244, 0.24745, 0.84198],
[0.1443, 0.25307, 0.84216],
[0.14616, 0.25882, 0.84191],
[0.14804, 0.26468, 0.84117],
[0.14995, 0.27071, 0.8399],
[0.15179, 0.27687, 0.83804],
[0.15363, 0.2832, 0.83552],
[0.15544, 0.28972, 0.83227],
[0.15711, 0.29645, 0.82819],
[0.1587, 0.30339, 0.82319],
[0.16004, 0.31064, 0.81705],
[0.16121, 0.31816, 0.80955],
[0.16198, 0.32608, 0.80043],
[0.16232, 0.33446, 0.78944],
[0.16261, 0.34288, 0.77734],
[0.16339, 0.351, 0.7652],
[0.16458, 0.35878, 0.75318],
[0.16618, 0.3663, 0.7413],
[0.16813, 0.37354, 0.72954],
[0.17035, 0.38056, 0.71789],
[0.1728, 0.38735, 0.70636],
[0.17542, 0.39397, 0.69491],
[0.17821, 0.40042, 0.68357],
[0.18104, 0.40669, 0.67233],
[0.18398, 0.41284, 0.66117],
[0.18698, 0.41883, 0.65009],
[0.19, 0.42472, 0.6391],
[0.19303, 0.43049, 0.62818],
[0.19608, 0.43615, 0.61732],
[0.19905, 0.44173, 0.60653],
[0.20205, 0.4472, 0.59581],
[0.20503, 0.45261, 0.58515],
[0.20793, 0.45792, 0.57453],
[0.2108, 0.46316, 0.56398],
[0.21364, 0.46835, 0.55345],
[0.21643, 0.47347, 0.54299],
[0.21913, 0.47854, 0.53255],
[0.22179, 0.48354, 0.52214],
[0.22439, 0.4885, 0.51178],
[0.22697, 0.4934, 0.50144],
[0.22945, 0.49828, 0.49112],
[0.23194, 0.50308, 0.48079],
[0.23434, 0.50787, 0.47051],
[0.23669, 0.5126, 0.4602],
[0.23896, 0.51731, 0.44992],
[0.24122, 0.52197, 0.43962],
[0.24342, 0.52661, 0.42932],
[0.2456, 0.53121, 0.419],
[0.24776, 0.53578, 0.40865],
[0.24982, 0.54033, 0.39827],
[0.2519, 0.54485, 0.38784],
[0.25394, 0.54933, 0.37747],
[0.25603, 0.55376, 0.36733],
[0.25825, 0.55813, 0.35754],
[0.26059, 0.56246, 0.34801],
[0.26305, 0.56672, 0.33878],
[0.26563, 0.57092, 0.32981],
[0.26835, 0.57509, 0.32108],
[0.27123, 0.57921, 0.31258],
[0.27426, 0.58328, 0.30433],
[0.27745, 0.58731, 0.29629],
[0.28078, 0.5913, 0.28844],
[0.28431, 0.59525, 0.28077],
[0.28799, 0.59916, 0.27333],
[0.29185, 0.60303, 0.26603],
[0.29585, 0.60687, 0.2589],
[0.30003, 0.61067, 0.25192],
[0.30437, 0.61442, 0.24507],
[0.3089, 0.61816, 0.23841],
[0.31357, 0.62185, 0.23192],
[0.3184, 0.62552, 0.22553],
[0.32339, 0.62915, 0.21927],
[0.32852, 0.63275, 0.21311],
[0.3338, 0.63632, 0.20712],
[0.33922, 0.63986, 0.20117],
[0.34475, 0.64337, 0.19538],
[0.35045, 0.64686, 0.1897],
[0.35629, 0.6503, 0.18415],
[0.36225, 0.65372, 0.17872],
[0.36834, 0.65712, 0.17336],
[0.37454, 0.66048, 0.1681],
[0.38085, 0.66383, 0.16297],
[0.38726, 0.66714, 0.15793],
[0.39382, 0.67043, 0.15302],
[0.40049, 0.67368, 0.14824],
[0.40723, 0.67691, 0.14356],
[0.41408, 0.68011, 0.13895],
[0.42102, 0.68329, 0.13455],
[0.42809, 0.68644, 0.13028],
[0.43524, 0.68956, 0.12609],
[0.44246, 0.69267, 0.12198],
[0.44975, 0.69574, 0.11813],
[0.45716, 0.69877, 0.1144],
[0.46465, 0.70179, 0.11092],
[0.4722, 0.70479, 0.10752],
[0.47982, 0.70776, 0.1043],
[0.48754, 0.7107, 0.1014],
[0.49532, 0.71361, 0.098689],
[0.50314, 0.7165, 0.096121],
[0.51106, 0.71936, 0.093905],
[0.51907, 0.72219, 0.091988],
[0.52711, 0.72501, 0.090242],
[0.53519, 0.72779, 0.08877],
[0.54339, 0.73055, 0.087727],
[0.55162, 0.73328, 0.086928],
[0.5599, 0.73598, 0.086341],
[0.56824, 0.73866, 0.086171],
[0.57666, 0.74131, 0.086363],
[0.58511, 0.74394, 0.086854],
[0.59362, 0.74654, 0.087647],
[0.60218, 0.74911, 0.088756],
[0.61075, 0.75167, 0.090091],
[0.6193, 0.75421, 0.091662],
[0.62776, 0.75676, 0.093291],
[0.63618, 0.7593, 0.095179],
[0.64452, 0.76186, 0.097055],
[0.65279, 0.76442, 0.0992],
[0.66101, 0.76698, 0.10144],
[0.66917, 0.76954, 0.10382],
[0.67726, 0.77211, 0.10635],
[0.68528, 0.77468, 0.10898],
[0.69324, 0.77727, 0.11176],
[0.70115, 0.77985, 0.1146],
[0.70899, 0.78245, 0.11768],
[0.71675, 0.78505, 0.12084],
[0.72446, 0.78766, 0.12415],
[0.73211, 0.79027, 0.12759],
[0.73969, 0.7929, 0.13116],
[0.74719, 0.79554, 0.13488],
[0.75463, 0.79819, 0.13869],
[0.762, 0.80085, 0.14265],
[0.76931, 0.80352, 0.1467],
[0.77655, 0.80621, 0.15091],
[0.78371, 0.80891, 0.15527],
[0.79078, 0.81163, 0.15974],
[0.79778, 0.81436, 0.1644],
[0.80471, 0.81711, 0.16917],
[0.81155, 0.81987, 0.17404],
[0.8183, 0.82266, 0.17908],
[0.82496, 0.82547, 0.18425],
[0.83154, 0.8283, 0.18958],
[0.83802, 0.83116, 0.19506],
[0.84439, 0.83404, 0.20068],
[0.85066, 0.83694, 0.20653],
[0.85681, 0.83988, 0.21248],
[0.86285, 0.84285, 0.21864],
[0.86877, 0.84585, 0.22495],
[0.87454, 0.84889, 0.23152],
[0.88018, 0.85196, 0.23823],
[0.88567, 0.85508, 0.24516],
[0.89101, 0.85823, 0.25235],
[0.89617, 0.86144, 0.25975],
[0.90116, 0.86469, 0.26737],
[0.90596, 0.868, 0.27529],
[0.91052, 0.87137, 0.2835],
[0.91485, 0.87481, 0.29203],
[0.91894, 0.87831, 0.30086],
[0.92276, 0.88189, 0.31007],
[0.92625, 0.88556, 0.3197],
[0.92941, 0.88931, 0.32973],
[0.93217, 0.89317, 0.34027],
[0.93451, 0.89714, 0.3513],
[0.93634, 0.90125, 0.36298],
[0.93763, 0.9055, 0.37534],
[0.9387, 0.90978, 0.38837],
[0.93989, 0.91399, 0.40197],
[0.94123, 0.91811, 0.41613],
[0.94272, 0.92216, 0.43083],
[0.94436, 0.92612, 0.44608],
[0.94613, 0.93, 0.46186],
[0.94803, 0.9338, 0.47813],
[0.95005, 0.93753, 0.49497],
[0.95218, 0.94117, 0.5123],
[0.95441, 0.94474, 0.53014],
[0.95674, 0.94822, 0.54852],
[0.95915, 0.95163, 0.56741],
[0.96162, 0.95496, 0.58679],
[0.96415, 0.95822, 0.60671],
[0.96673, 0.96139, 0.62717],
[0.96934, 0.96449, 0.64813],
[0.97196, 0.96752, 0.66962],
[0.97459, 0.97046, 0.69168],
[0.97722, 0.97333, 0.71425],
[0.9798, 0.97613, 0.73736],
[0.98235, 0.97885, 0.76105],
[0.98483, 0.98151, 0.78529],
[0.98722, 0.98408, 0.81007],
[0.98952, 0.98658, 0.83544],
[0.99169, 0.98901, 0.8614],
[0.99371, 0.99136, 0.88792],
[0.99556, 0.99365, 0.91503],
[0.99723, 0.99585, 0.94278],
[0.99867, 0.99799, 0.97109],
[0.99986, 1, 1],
]
b_linear_bgyw_15_100_c67 = bokeh_palette('linear_bgyw_15_100_c67',linear_bgyw_15_100_c67)
m_linear_bgyw_15_100_c67 = mpl_cm('linear_bgyw_15_100_c67',linear_bgyw_15_100_c67)
m_linear_bgyw_15_100_c67_r = mpl_cm('linear_bgyw_15_100_c67_r',list(reversed(linear_bgyw_15_100_c67)))




linear_kry_5_95_c72 = [  # cmap_def
[0.066001, 0.066024, 0.066019],
[0.07965, 0.066744, 0.064745],
[0.091664, 0.06747, 0.063475],
[0.10228, 0.068214, 0.06216],
[0.112, 0.068994, 0.060816],
[0.12092, 0.069706, 0.059585],
[0.12971, 0.070185, 0.058154],
[0.13843, 0.07042, 0.05692],
[0.14709, 0.070382, 0.055604],
[0.15576, 0.07009, 0.054295],
[0.16433, 0.069694, 0.052994],
[0.17278, 0.069179, 0.051712],
[0.18118, 0.068484, 0.050432],
[0.18954, 0.067685, 0.049199],
[0.1978, 0.066788, 0.047758],
[0.20605, 0.065769, 0.046454],
[0.2142, 0.064648, 0.045151],
[0.2223, 0.063428, 0.043725],
[0.23036, 0.062012, 0.042429],
[0.23838, 0.060521, 0.04109],
[0.24634, 0.058898, 0.039699],
[0.25428, 0.057101, 0.038324],
[0.26215, 0.055158, 0.036964],
[0.26998, 0.053082, 0.035663],
[0.27773, 0.050917, 0.034117],
[0.28544, 0.048622, 0.032846],
[0.29312, 0.046013, 0.031565],
[0.30072, 0.043344, 0.030323],
[0.30827, 0.040658, 0.029105],
[0.31576, 0.037722, 0.02791],
[0.32318, 0.034637, 0.02674],
[0.33052, 0.031576, 0.025595],
[0.33779, 0.028628, 0.024479],
[0.34497, 0.025745, 0.023396],
[0.35205, 0.022943, 0.022345],
[0.35905, 0.020248, 0.021325],
[0.36595, 0.017697, 0.02034],
[0.37272, 0.015331, 0.01939],
[0.37938, 0.01319, 0.01848],
[0.38589, 0.01125, 0.017611],
[0.39225, 0.0096476, 0.016788],
[0.39845, 0.0085386, 0.016013],
[0.40447, 0.0079151, 0.015288],
[0.41036, 0.0076915, 0.014615],
[0.41618, 0.0076329, 0.013987],
[0.42199, 0.007602, 0.01339],
[0.42783, 0.0075831, 0.012837],
[0.43365, 0.0075764, 0.012347],
[0.4395, 0.0075827, 0.011859],
[0.44534, 0.0076027, 0.011316],
[0.4512, 0.0076373, 0.010796],
[0.45705, 0.0076872, 0.010385],
[0.46291, 0.0077536, 0.010055],
[0.46879, 0.0078379, 0.0097685],
[0.47467, 0.0079404, 0.0095068],
[0.48054, 0.0080616, 0.0092717],
[0.48643, 0.0082023, 0.0090674],
[0.49232, 0.0083637, 0.0088963],
[0.49823, 0.0085472, 0.008759],
[0.50414, 0.0087542, 0.0086553],
[0.51004, 0.008987, 0.0085858],
[0.51595, 0.0092456, 0.0085516],
[0.52187, 0.0095265, 0.0085507],
[0.5278, 0.0098238, 0.0085815],
[0.53373, 0.010138, 0.0086438],
[0.53967, 0.010498, 0.0087375],
[0.54563, 0.010949, 0.0088628],
[0.55158, 0.011488, 0.0090195],
[0.55753, 0.012012, 0.0092082],
[0.56349, 0.012487, 0.0094284],
[0.56945, 0.012988, 0.0096743],
[0.57543, 0.013545, 0.009933],
[0.5814, 0.014124, 0.010192],
[0.5874, 0.014709, 0.010451],
[0.59341, 0.015309, 0.010717],
[0.59942, 0.015921, 0.010993],
[0.60544, 0.016546, 0.011263],
[0.61149, 0.017183, 0.011511],
[0.61754, 0.017832, 0.011726],
[0.62361, 0.018494, 0.011906],
[0.62967, 0.019169, 0.012055],
[0.63576, 0.019858, 0.012179],
[0.64186, 0.020559, 0.01228],
[0.64796, 0.021273, 0.012363],
[0.65409, 0.022, 0.012427],
[0.66022, 0.022741, 0.012473],
[0.66635, 0.023496, 0.012501],
[0.67251, 0.024264, 0.012512],
[0.67867, 0.025046, 0.012503],
[0.68485, 0.025842, 0.012477],
[0.69103, 0.026652, 0.012431],
[0.69722, 0.027476, 0.012369],
[0.70343, 0.028316, 0.012312],
[0.70965, 0.02917, 0.012261],
[0.71588, 0.030041, 0.012218],
[0.7221, 0.030928, 0.012182],
[0.72835, 0.031827, 0.012155],
[0.73461, 0.032755, 0.012137],
[0.74086, 0.033657, 0.012129],
[0.74714, 0.034746, 0.01213],
[0.7534, 0.035909, 0.012143],
[0.75965, 0.037152, 0.012168],
[0.76588, 0.038639, 0.012207],
[0.77207, 0.040359, 0.012259],
[0.77825, 0.042283, 0.012323],
[0.7844, 0.044355, 0.0124],
[0.79052, 0.046616, 0.012491],
[0.79662, 0.049121, 0.012597],
[0.80269, 0.051607, 0.01272],
[0.80874, 0.054357, 0.012862],
[0.81476, 0.057256, 0.013025],
[0.82076, 0.060328, 0.013209],
[0.82673, 0.063454, 0.013414],
[0.83267, 0.066671, 0.01364],
[0.83859, 0.070023, 0.013884],
[0.84448, 0.073512, 0.014147],
[0.85034, 0.076937, 0.014432],
[0.85618, 0.080598, 0.014739],
[0.86199, 0.084349, 0.015072],
[0.86776, 0.088098, 0.015429],
[0.87351, 0.09196, 0.015811],
[0.87924, 0.095821, 0.016218],
[0.88494, 0.099784, 0.016653],
[0.89061, 0.10383, 0.017114],
[0.89625, 0.10787, 0.017605],
[0.90186, 0.11205, 0.018127],
[0.90744, 0.11624, 0.018679],
[0.91299, 0.12043, 0.019262],
[0.91846, 0.12485, 0.019871],
[0.92381, 0.12964, 0.020493],
[0.92898, 0.13485, 0.021122],
[0.93399, 0.14044, 0.02176],
[0.93881, 0.14642, 0.022407],
[0.94345, 0.15278, 0.023063],
[0.94791, 0.15946, 0.023731],
[0.95218, 0.16642, 0.024411],
[0.95628, 0.17361, 0.025105],
[0.9602, 0.18101, 0.025814],
[0.96394, 0.18866, 0.026541],
[0.9675, 0.19646, 0.027287],
[0.97086, 0.20443, 0.028053],
[0.97403, 0.21254, 0.028843],
[0.97702, 0.22083, 0.029658],
[0.97982, 0.22918, 0.0305],
[0.98243, 0.23769, 0.031373],
[0.98485, 0.24624, 0.032279],
[0.98708, 0.25493, 0.03322],
[0.98911, 0.26367, 0.034197],
[0.99095, 0.27247, 0.035441],
[0.99259, 0.28137, 0.036475],
[0.99402, 0.29032, 0.0376],
[0.99526, 0.29935, 0.038765],
[0.99629, 0.3084, 0.039989],
[0.99713, 0.31749, 0.041278],
[0.99777, 0.32662, 0.042522],
[0.9982, 0.3358, 0.043828],
[0.99844, 0.34498, 0.045261],
[0.99859, 0.35407, 0.04666],
[0.99869, 0.36301, 0.048013],
[0.99878, 0.37178, 0.049508],
[0.99884, 0.38043, 0.050843],
[0.99888, 0.38894, 0.052219],
[0.99889, 0.39731, 0.05362],
[0.99887, 0.40558, 0.055042],
[0.99884, 0.41373, 0.056502],
[0.99878, 0.42178, 0.057786],
[0.99869, 0.42975, 0.059296],
[0.99857, 0.43763, 0.060625],
[0.99844, 0.4454, 0.061996],
[0.99827, 0.45312, 0.063411],
[0.99808, 0.46074, 0.064752],
[0.99787, 0.46831, 0.066112],
[0.99763, 0.47581, 0.067489],
[0.99736, 0.48323, 0.068891],
[0.99707, 0.49061, 0.070164],
[0.99675, 0.49792, 0.071543],
[0.9964, 0.50517, 0.072849],
[0.99603, 0.51237, 0.074177],
[0.99563, 0.51953, 0.075448],
[0.9952, 0.52664, 0.076731],
[0.99474, 0.53369, 0.078024],
[0.99426, 0.54072, 0.079325],
[0.99375, 0.54769, 0.08063],
[0.99322, 0.55463, 0.081958],
[0.99265, 0.56153, 0.083158],
[0.99204, 0.5684, 0.084524],
[0.99138, 0.57524, 0.085856],
[0.99067, 0.58206, 0.087485],
[0.98991, 0.58886, 0.089115],
[0.98909, 0.59566, 0.090908],
[0.98821, 0.60242, 0.092755],
[0.98728, 0.60918, 0.094848],
[0.98629, 0.6159, 0.096892],
[0.98524, 0.62262, 0.099162],
[0.98414, 0.62932, 0.1015],
[0.98297, 0.63601, 0.10396],
[0.98175, 0.64268, 0.10656],
[0.98046, 0.64934, 0.10919],
[0.97912, 0.65599, 0.11195],
[0.97771, 0.66262, 0.11474],
[0.97623, 0.66924, 0.11771],
[0.9747, 0.67585, 0.12072],
[0.9731, 0.68244, 0.12383],
[0.97144, 0.68904, 0.12703],
[0.9697, 0.69561, 0.13037],
[0.96791, 0.70217, 0.1337],
[0.96603, 0.70874, 0.13715],
[0.96409, 0.71529, 0.14067],
[0.96208, 0.72182, 0.14425],
[0.96, 0.72836, 0.14791],
[0.95785, 0.73488, 0.15162],
[0.95563, 0.74139, 0.15542],
[0.95333, 0.74791, 0.15928],
[0.95104, 0.75439, 0.16306],
[0.94892, 0.76075, 0.16643],
[0.94705, 0.767, 0.16936],
[0.9454, 0.77315, 0.1718],
[0.94394, 0.7792, 0.1739],
[0.94268, 0.78516, 0.17562],
[0.94159, 0.79104, 0.17704],
[0.94067, 0.79684, 0.17816],
[0.93991, 0.80258, 0.17893],
[0.93929, 0.80825, 0.17943],
[0.93881, 0.81387, 0.1797],
[0.93845, 0.81943, 0.17972],
[0.9382, 0.82494, 0.17952],
[0.93807, 0.83042, 0.17911],
[0.93806, 0.83583, 0.17845],
[0.93816, 0.84121, 0.17756],
[0.93835, 0.84655, 0.17643],
[0.93863, 0.85186, 0.17517],
[0.93901, 0.85713, 0.17363],
[0.93948, 0.86238, 0.17189],
[0.94002, 0.86758, 0.16998],
[0.94064, 0.87277, 0.16783],
[0.94135, 0.87792, 0.16542],
[0.94214, 0.88305, 0.16286],
[0.94298, 0.88816, 0.16003],
[0.94391, 0.89324, 0.15697],
[0.9449, 0.8983, 0.15365],
[0.94594, 0.90334, 0.15015],
[0.94707, 0.90836, 0.14627],
[0.94825, 0.91336, 0.14216],
[0.94948, 0.91834, 0.13776],
[0.95079, 0.92331, 0.13295],
[0.95212, 0.92826, 0.12782],
[0.95354, 0.93319, 0.12222],
[0.955, 0.93811, 0.11627],
[0.95649, 0.94302, 0.1098],
[0.95807, 0.94791, 0.10259],
[0.95966, 0.95279, 0.094821],
[0.96132, 0.95766, 0.086007],
[0.96302, 0.96252, 0.076202],
[0.96476, 0.96736, 0.065035],
[0.96656, 0.97219, 0.051675],
[0.96837, 0.97701, 0.035491],
]
b_linear_kry_5_95_c72 = bokeh_palette('linear_kry_5_95_c72',linear_kry_5_95_c72)
m_linear_kry_5_95_c72 = mpl_cm('linear_kry_5_95_c72',linear_kry_5_95_c72)
m_linear_kry_5_95_c72_r = mpl_cm('linear_kry_5_95_c72_r',list(reversed(linear_kry_5_95_c72)))




rainbow_bgyrm_35_85_c69 = [  # cmap_def
[0, 0.18656, 0.96265],
[0, 0.20247, 0.94889],
[0, 0.21731, 0.93516],
[0, 0.23125, 0.92145],
[0.011192, 0.24442, 0.90777],
[0.03614, 0.257, 0.8941],
[0.055678, 0.26904, 0.88045],
[0.069388, 0.28056, 0.86682],
[0.079257, 0.29172, 0.85322],
[0.086705, 0.30246, 0.83965],
[0.092118, 0.31289, 0.82609],
[0.095864, 0.32302, 0.81256],
[0.098322, 0.33286, 0.79906],
[0.099569, 0.34241, 0.78559],
[0.099878, 0.3517, 0.77216],
[0.099398, 0.36075, 0.75877],
[0.098334, 0.36957, 0.74542],
[0.096762, 0.37814, 0.73214],
[0.09515, 0.38647, 0.71891],
[0.09345, 0.39458, 0.70576],
[0.092187, 0.40244, 0.6927],
[0.091506, 0.41008, 0.67971],
[0.091673, 0.41747, 0.66683],
[0.092844, 0.42463, 0.65405],
[0.095394, 0.43156, 0.64138],
[0.099143, 0.43827, 0.62883],
[0.10417, 0.44476, 0.6164],
[0.11037, 0.45105, 0.60407],
[0.11743, 0.45714, 0.59186],
[0.12529, 0.46306, 0.57975],
[0.13367, 0.46884, 0.56772],
[0.14235, 0.47446, 0.55578],
[0.15115, 0.47996, 0.54393],
[0.15993, 0.48534, 0.53213],
[0.16859, 0.49067, 0.52036],
[0.17696, 0.49589, 0.50865],
[0.185, 0.50108, 0.49692],
[0.1926, 0.50621, 0.48521],
[0.19974, 0.51133, 0.47348],
[0.20647, 0.51642, 0.46174],
[0.21263, 0.5215, 0.44995],
[0.21832, 0.52659, 0.43811],
[0.2235, 0.53167, 0.42619],
[0.2282, 0.53676, 0.41421],
[0.23239, 0.54184, 0.40212],
[0.23614, 0.54695, 0.38996],
[0.23944, 0.55206, 0.37769],
[0.24232, 0.55719, 0.3653],
[0.24481, 0.56231, 0.3528],
[0.24695, 0.56744, 0.34016],
[0.24876, 0.57258, 0.32738],
[0.25023, 0.5777, 0.31447],
[0.25154, 0.58283, 0.30143],
[0.25262, 0.58794, 0.28828],
[0.25361, 0.59304, 0.27502],
[0.25458, 0.5981, 0.26166],
[0.2556, 0.60312, 0.24825],
[0.25681, 0.6081, 0.2348],
[0.2583, 0.61301, 0.22134],
[0.26022, 0.61785, 0.20794],
[0.26266, 0.6226, 0.19469],
[0.26575, 0.62726, 0.18157],
[0.26958, 0.63182, 0.16877],
[0.27422, 0.63626, 0.15629],
[0.27974, 0.64057, 0.1442],
[0.28615, 0.64476, 0.13265],
[0.29347, 0.64883, 0.12168],
[0.30163, 0.65275, 0.11152],
[0.31062, 0.65656, 0.10212],
[0.3203, 0.66024, 0.093657],
[0.33059, 0.6638, 0.086199],
[0.34143, 0.66725, 0.079852],
[0.35268, 0.67061, 0.074629],
[0.36426, 0.67387, 0.07049],
[0.37611, 0.67706, 0.067449],
[0.3881, 0.68016, 0.065346],
[0.40021, 0.68322, 0.064139],
[0.41237, 0.68622, 0.063672],
[0.42452, 0.68918, 0.063795],
[0.43666, 0.6921, 0.064392],
[0.44874, 0.69498, 0.065356],
[0.46073, 0.69783, 0.066603],
[0.47267, 0.70067, 0.068066],
[0.4845, 0.70348, 0.06967],
[0.49625, 0.70628, 0.071332],
[0.50793, 0.70905, 0.07306],
[0.5195, 0.71181, 0.074792],
[0.53099, 0.71455, 0.076552],
[0.5424, 0.71728, 0.078345],
[0.55372, 0.72, 0.08017],
[0.56497, 0.72269, 0.082034],
[0.57617, 0.72539, 0.083799],
[0.58729, 0.72805, 0.085523],
[0.59835, 0.73071, 0.087403],
[0.60937, 0.73336, 0.089162],
[0.62033, 0.73599, 0.090964],
[0.63124, 0.73861, 0.092702],
[0.64211, 0.74121, 0.094561],
[0.65293, 0.7438, 0.096272],
[0.66373, 0.74637, 0.09813],
[0.67447, 0.74893, 0.099882],
[0.68519, 0.75149, 0.10167],
[0.69588, 0.75402, 0.1035],
[0.70654, 0.75653, 0.10524],
[0.71717, 0.75904, 0.10708],
[0.72776, 0.76153, 0.10884],
[0.73834, 0.764, 0.11066],
[0.7489, 0.76647, 0.11246],
[0.75944, 0.76891, 0.1142],
[0.76995, 0.77134, 0.11604],
[0.78045, 0.77373, 0.11781],
[0.79091, 0.7761, 0.11958],
[0.80135, 0.77844, 0.12133],
[0.81175, 0.78072, 0.1231],
[0.8221, 0.78295, 0.12488],
[0.83239, 0.78511, 0.12661],
[0.84262, 0.78718, 0.12834],
[0.85274, 0.78913, 0.13004],
[0.86274, 0.79094, 0.13165],
[0.87259, 0.79257, 0.13322],
[0.88223, 0.79401, 0.13475],
[0.89164, 0.79518, 0.13612],
[0.90076, 0.79607, 0.1375],
[0.90954, 0.79664, 0.13869],
[0.91793, 0.79684, 0.13974],
[0.92586, 0.79664, 0.14073],
[0.93328, 0.79601, 0.14151],
[0.94018, 0.79492, 0.14212],
[0.94648, 0.79336, 0.14256],
[0.95218, 0.79132, 0.14282],
[0.95727, 0.78881, 0.14289],
[0.96174, 0.78582, 0.1428],
[0.9656, 0.7824, 0.14253],
[0.96888, 0.77856, 0.1421],
[0.97162, 0.77434, 0.14153],
[0.97385, 0.76979, 0.14082],
[0.97563, 0.76494, 0.13994],
[0.977, 0.75981, 0.13902],
[0.97803, 0.75448, 0.13804],
[0.97875, 0.74895, 0.13692],
[0.97924, 0.74329, 0.13575],
[0.97951, 0.73749, 0.13461],
[0.97962, 0.7316, 0.13337],
[0.97959, 0.72564, 0.13212],
[0.97946, 0.71962, 0.13087],
[0.97924, 0.71355, 0.12958],
[0.97896, 0.70744, 0.1283],
[0.97862, 0.70131, 0.12698],
[0.97824, 0.69516, 0.12574],
[0.97784, 0.68899, 0.12442],
[0.9774, 0.6828, 0.12313],
[0.97693, 0.6766, 0.12186],
[0.97644, 0.67039, 0.12061],
[0.97594, 0.66416, 0.11937],
[0.97542, 0.65793, 0.11815],
[0.97489, 0.65168, 0.11688],
[0.97433, 0.64541, 0.11567],
[0.97376, 0.63914, 0.1144],
[0.97318, 0.63285, 0.11325],
[0.97257, 0.62654, 0.11203],
[0.97195, 0.62023, 0.11083],
[0.97131, 0.61388, 0.10966],
[0.97065, 0.60754, 0.10841],
[0.96997, 0.60116, 0.10728],
[0.96928, 0.59477, 0.10613],
[0.96858, 0.58836, 0.1049],
[0.96785, 0.58193, 0.10381],
[0.96711, 0.57549, 0.10266],
[0.96635, 0.56901, 0.10152],
[0.96557, 0.56253, 0.1004],
[0.96478, 0.55599, 0.099296],
[0.96398, 0.54946, 0.098231],
[0.96315, 0.5429, 0.097093],
[0.96231, 0.5363, 0.096016],
[0.96146, 0.52968, 0.095014],
[0.96059, 0.52303, 0.093919],
[0.9597, 0.51634, 0.092832],
[0.95879, 0.50964, 0.091874],
[0.95787, 0.50288, 0.090828],
[0.95693, 0.4961, 0.089807],
[0.95599, 0.48929, 0.088805],
[0.95502, 0.48244, 0.08783],
[0.95403, 0.47555, 0.086854],
[0.95304, 0.4686, 0.085824],
[0.95202, 0.46162, 0.084968],
[0.95101, 0.4546, 0.084163],
[0.94997, 0.44753, 0.083344],
[0.94894, 0.44044, 0.082697],
[0.94791, 0.43332, 0.082226],
[0.94688, 0.42618, 0.081913],
[0.94587, 0.41904, 0.081847],
[0.94489, 0.41191, 0.082109],
[0.94394, 0.40481, 0.082728],
[0.94305, 0.39778, 0.083928],
[0.94223, 0.39086, 0.085608],
[0.94149, 0.38408, 0.08812],
[0.94088, 0.3775, 0.091363],
[0.94041, 0.37118, 0.095416],
[0.9401, 0.36521, 0.10035],
[0.93998, 0.35964, 0.1063],
[0.94007, 0.35454, 0.11316],
[0.9404, 0.34999, 0.12089],
[0.94096, 0.34604, 0.12965],
[0.94179, 0.3428, 0.13922],
[0.94288, 0.34026, 0.14972],
[0.94422, 0.33847, 0.16095],
[0.94582, 0.33747, 0.17292],
[0.94764, 0.33723, 0.18561],
[0.94967, 0.33772, 0.19885],
[0.95189, 0.33894, 0.21269],
[0.95427, 0.34081, 0.22702],
[0.95677, 0.34328, 0.24176],
[0.95937, 0.34627, 0.25686],
[0.96202, 0.34974, 0.27224],
[0.96472, 0.35359, 0.2879],
[0.96743, 0.35778, 0.30376],
[0.97012, 0.36222, 0.31979],
[0.97279, 0.36688, 0.33592],
[0.97541, 0.3717, 0.35214],
[0.97796, 0.37668, 0.36845],
[0.98045, 0.38173, 0.3848],
[0.98286, 0.38685, 0.40121],
[0.98518, 0.39205, 0.41764],
[0.9874, 0.39727, 0.43409],
[0.98953, 0.40253, 0.45059],
[0.99155, 0.40782, 0.46711],
[0.99348, 0.41314, 0.48364],
[0.99528, 0.41844, 0.5002],
[0.99698, 0.42377, 0.51679],
[0.99858, 0.42911, 0.5334],
[1, 0.43445, 0.55005],
[1, 0.43981, 0.56674],
[1, 0.44517, 0.58345],
[1, 0.45054, 0.60021],
[1, 0.45593, 0.61701],
[1, 0.46132, 0.63385],
[1, 0.46673, 0.65073],
[1, 0.47215, 0.66766],
[1, 0.47757, 0.68462],
[1, 0.48301, 0.70164],
[1, 0.48847, 0.71871],
[1, 0.49393, 0.73582],
[1, 0.49939, 0.75297],
[1, 0.50489, 0.77018],
[1, 0.51038, 0.78744],
[1, 0.51588, 0.80475],
[1, 0.52141, 0.8221],
[1, 0.52695, 0.8395],
[1, 0.53249, 0.85695],
[1, 0.53805, 0.87445],
[1, 0.54361, 0.892],
[1, 0.5492, 0.9096],
[0.99848, 0.55478, 0.92724],
[0.9966, 0.56039, 0.94494],
[0.99453, 0.566, 0.96269],
[0.99228, 0.57163, 0.98048],
]
b_rainbow_bgyrm_35_85_c69 = bokeh_palette('rainbow_bgyrm_35_85_c69',rainbow_bgyrm_35_85_c69)
m_rainbow_bgyrm_35_85_c69 = mpl_cm('rainbow_bgyrm_35_85_c69',rainbow_bgyrm_35_85_c69)
m_rainbow_bgyrm_35_85_c69_r = mpl_cm('rainbow_bgyrm_35_85_c69_r',list(reversed(rainbow_bgyrm_35_85_c69)))
CET_R1 = b_rainbow_bgyrm_35_85_c69
m_CET_R1 = m_rainbow_bgyrm_35_85_c69
m_CET_R1_r = m_rainbow_bgyrm_35_85_c69_r
palette['CET_R1'] = b_rainbow_bgyrm_35_85_c69
cm['CET_R1'] = m_rainbow_bgyrm_35_85_c69
cm['CET_R1_r'] = m_rainbow_bgyrm_35_85_c69_r
register_cmap('cet_CET_R1',m_rainbow_bgyrm_35_85_c69)
register_cmap('cet_CET_R1_r',m_rainbow_bgyrm_35_85_c69_r)




linear_kryw_0_100_c71 = [  # cmap_def
[0, 0, 0],
[0.027065, 2.143e-05, 0],
[0.052054, 7.4728e-05, 0],
[0.071511, 0.00013914, 0],
[0.08742, 0.0002088, 0],
[0.10109, 0.00028141, 0],
[0.11337, 0.000356, 2.4266e-17],
[0.12439, 0.00043134, 3.3615e-17],
[0.13463, 0.00050796, 2.1604e-17],
[0.14411, 0.0005856, 0],
[0.15292, 0.00070304, 0],
[0.16073, 0.0013432, 0],
[0.16871, 0.0014516, 0],
[0.17657, 0.0012408, 0],
[0.18364, 0.0015336, 0],
[0.19052, 0.0017515, 0],
[0.19751, 0.0015146, 0],
[0.20401, 0.0015249, 0],
[0.20994, 0.0019639, 0],
[0.21605, 0.002031, 0],
[0.22215, 0.0017559, 0],
[0.22808, 0.001546, 1.8755e-05],
[0.23378, 0.0016315, 3.5012e-05],
[0.23955, 0.0017194, 3.3352e-05],
[0.24531, 0.0018097, 1.8559e-05],
[0.25113, 0.0019038, 1.9139e-05],
[0.25694, 0.0020015, 3.5308e-05],
[0.26278, 0.0021017, 3.2613e-05],
[0.26864, 0.0022048, 2.0338e-05],
[0.27451, 0.0023119, 2.2453e-05],
[0.28041, 0.0024227, 3.6003e-05],
[0.28633, 0.0025363, 2.9817e-05],
[0.29229, 0.0026532, 1.9559e-05],
[0.29824, 0.0027747, 2.7666e-05],
[0.30423, 0.0028999, 3.5752e-05],
[0.31026, 0.0030279, 2.3231e-05],
[0.31628, 0.0031599, 1.2902e-05],
[0.32232, 0.0032974, 3.2915e-05],
[0.32838, 0.0034379, 3.2803e-05],
[0.33447, 0.0035819, 2.0757e-05],
[0.34057, 0.003731, 2.3831e-05],
[0.34668, 0.0038848, 3.502e-05],
[0.35283, 0.0040418, 2.4468e-05],
[0.35897, 0.0042032, 1.1444e-05],
[0.36515, 0.0043708, 3.2793e-05],
[0.37134, 0.0045418, 3.012e-05],
[0.37756, 0.0047169, 1.4846e-05],
[0.38379, 0.0048986, 2.796e-05],
[0.39003, 0.0050848, 3.2782e-05],
[0.3963, 0.0052751, 1.9244e-05],
[0.40258, 0.0054715, 2.2667e-05],
[0.40888, 0.0056736, 3.3223e-05],
[0.41519, 0.0058798, 2.159e-05],
[0.42152, 0.0060922, 1.8214e-05],
[0.42788, 0.0063116, 3.2525e-05],
[0.43424, 0.0065353, 2.2247e-05],
[0.44062, 0.006765, 1.5852e-05],
[0.44702, 0.0070024, 3.1769e-05],
[0.45344, 0.0072442, 2.1245e-05],
[0.45987, 0.0074929, 1.5726e-05],
[0.46631, 0.0077499, 3.0976e-05],
[0.47277, 0.0080108, 1.8722e-05],
[0.47926, 0.0082789, 1.9285e-05],
[0.48574, 0.0085553, 3.0063e-05],
[0.49225, 0.0088392, 1.4313e-05],
[0.49878, 0.0091356, 2.3404e-05],
[0.50531, 0.0094374, 2.8099e-05],
[0.51187, 0.0097365, 6.4695e-06],
[0.51844, 0.010039, 2.5791e-05],
[0.52501, 0.010354, 2.4393e-05],
[0.53162, 0.010689, 1.6037e-05],
[0.53825, 0.011031, 2.7295e-05],
[0.54489, 0.011393, 1.5848e-05],
[0.55154, 0.011789, 2.3111e-05],
[0.55818, 0.012159, 2.5416e-05],
[0.56485, 0.012508, 1.5064e-05],
[0.57154, 0.012881, 2.541e-05],
[0.57823, 0.013283, 1.6166e-05],
[0.58494, 0.013701, 2.263e-05],
[0.59166, 0.014122, 2.3316e-05],
[0.59839, 0.014551, 1.9432e-05],
[0.60514, 0.014994, 2.4323e-05],
[0.6119, 0.01545, 1.3929e-05],
[0.61868, 0.01592, 2.1615e-05],
[0.62546, 0.016401, 1.5846e-05],
[0.63226, 0.016897, 2.0838e-05],
[0.63907, 0.017407, 1.9549e-05],
[0.64589, 0.017931, 2.0961e-05],
[0.65273, 0.018471, 2.0737e-05],
[0.65958, 0.019026, 2.0621e-05],
[0.66644, 0.019598, 2.0675e-05],
[0.67332, 0.020187, 2.0301e-05],
[0.68019, 0.020793, 2.0029e-05],
[0.68709, 0.021418, 2.0088e-05],
[0.69399, 0.022062, 1.9102e-05],
[0.70092, 0.022727, 1.9662e-05],
[0.70784, 0.023412, 1.7757e-05],
[0.71478, 0.024121, 1.8236e-05],
[0.72173, 0.024852, 1.4944e-05],
[0.7287, 0.025608, 2.0245e-06],
[0.73567, 0.02639, 1.5013e-07],
[0.74266, 0.027199, 0],
[0.74964, 0.028038, 0],
[0.75665, 0.028906, 0],
[0.76365, 0.029806, 0],
[0.77068, 0.030743, 0],
[0.77771, 0.031711, 0],
[0.78474, 0.032732, 0],
[0.79179, 0.033741, 0],
[0.79886, 0.034936, 0],
[0.80593, 0.036031, 0],
[0.81299, 0.03723, 0],
[0.82007, 0.038493, 0],
[0.82715, 0.039819, 0],
[0.83423, 0.041236, 0],
[0.84131, 0.042647, 0],
[0.84838, 0.044235, 0],
[0.85545, 0.045857, 0],
[0.86252, 0.047645, 0],
[0.86958, 0.049578, 0],
[0.87661, 0.051541, 0],
[0.88365, 0.053735, 0],
[0.89064, 0.056168, 0],
[0.89761, 0.058852, 0],
[0.90451, 0.061777, 0],
[0.91131, 0.065281, 0],
[0.91796, 0.069448, 0],
[0.92445, 0.074684, 0],
[0.93061, 0.08131, 0],
[0.93648, 0.088878, 0],
[0.94205, 0.097336, 0],
[0.9473, 0.10665, 0],
[0.9522, 0.1166, 0],
[0.95674, 0.12716, 0],
[0.96094, 0.13824, 0],
[0.96479, 0.14963, 0],
[0.96829, 0.16128, 0],
[0.97147, 0.17303, 0],
[0.97436, 0.18489, 0],
[0.97698, 0.19672, 0],
[0.97934, 0.20846, 0],
[0.98148, 0.22013, 0],
[0.9834, 0.23167, 0],
[0.98515, 0.24301, 0],
[0.98672, 0.25425, 0],
[0.98815, 0.26525, 0],
[0.98944, 0.27614, 0],
[0.99061, 0.28679, 0],
[0.99167, 0.29731, 0],
[0.99263, 0.30764, 0],
[0.9935, 0.31781, 0],
[0.99428, 0.3278, 0],
[0.995, 0.33764, 0],
[0.99564, 0.34735, 0],
[0.99623, 0.35689, 0],
[0.99675, 0.3663, 0],
[0.99722, 0.37556, 0],
[0.99765, 0.38471, 0],
[0.99803, 0.39374, 0],
[0.99836, 0.40265, 0],
[0.99866, 0.41145, 0],
[0.99892, 0.42015, 0],
[0.99915, 0.42874, 0],
[0.99935, 0.43724, 0],
[0.99952, 0.44563, 0],
[0.99966, 0.45395, 0],
[0.99977, 0.46217, 0],
[0.99986, 0.47032, 0],
[0.99993, 0.47838, 0],
[0.99997, 0.48638, 0],
[1, 0.4943, 0],
[1, 0.50214, 0],
[1, 0.50991, 1.2756e-05],
[1, 0.51761, 4.5388e-05],
[1, 0.52523, 9.6977e-05],
[1, 0.5328, 0.00016858],
[1, 0.54028, 0.0002582],
[1, 0.54771, 0.00036528],
[1, 0.55508, 0.00049276],
[1, 0.5624, 0.00063955],
[1, 0.56965, 0.00080443],
[1, 0.57687, 0.00098902],
[1, 0.58402, 0.0011943],
[1, 0.59113, 0.0014189],
[1, 0.59819, 0.0016626],
[1, 0.60521, 0.0019281],
[1, 0.61219, 0.0022145],
[1, 0.61914, 0.0025213],
[1, 0.62603, 0.0028496],
[1, 0.6329, 0.0032006],
[1, 0.63972, 0.0035741],
[1, 0.64651, 0.0039701],
[1, 0.65327, 0.0043898],
[1, 0.66, 0.0048341],
[1, 0.66669, 0.005303],
[1, 0.67336, 0.0057969],
[1, 0.67999, 0.006317],
[1, 0.68661, 0.0068648],
[1, 0.69319, 0.0074406],
[1, 0.69974, 0.0080433],
[1, 0.70628, 0.0086756],
[1, 0.71278, 0.0093486],
[1, 0.71927, 0.010023],
[1, 0.72573, 0.010724],
[1, 0.73217, 0.011565],
[1, 0.73859, 0.012339],
[1, 0.74499, 0.01316],
[1, 0.75137, 0.014042],
[1, 0.75772, 0.014955],
[1, 0.76406, 0.015913],
[1, 0.77039, 0.016915],
[1, 0.77669, 0.017964],
[1, 0.78298, 0.019062],
[1, 0.78925, 0.020212],
[1, 0.7955, 0.021417],
[1, 0.80174, 0.02268],
[1, 0.80797, 0.024005],
[1, 0.81418, 0.025396],
[1, 0.82038, 0.026858],
[1, 0.82656, 0.028394],
[1, 0.83273, 0.030013],
[1, 0.83889, 0.031717],
[1, 0.84503, 0.03348],
[1, 0.85116, 0.035488],
[1, 0.85728, 0.037452],
[1, 0.8634, 0.039592],
[1, 0.86949, 0.041898],
[1, 0.87557, 0.044392],
[1, 0.88165, 0.046958],
[1, 0.88771, 0.04977],
[1, 0.89376, 0.052828],
[1, 0.8998, 0.056209],
[1, 0.90584, 0.059919],
[1, 0.91185, 0.063925],
[1, 0.91783, 0.068579],
[1, 0.92384, 0.073948],
[1, 0.92981, 0.080899],
[1, 0.93576, 0.090648],
[1, 0.94166, 0.10377],
[1, 0.94752, 0.12051],
[1, 0.9533, 0.14149],
[1, 0.959, 0.1672],
[1, 0.96456, 0.19823],
[1, 0.96995, 0.23514],
[1, 0.9751, 0.2786],
[1, 0.97992, 0.32883],
[1, 0.98432, 0.38571],
[1, 0.9882, 0.44866],
[1, 0.9915, 0.51653],
[1, 0.99417, 0.58754],
[1, 0.99625, 0.65985],
[1, 0.99778, 0.73194],
[1, 0.99885, 0.80259],
[1, 0.99953, 0.87115],
[1, 0.99989, 0.93683],
[1, 1, 1],
]
b_linear_kryw_0_100_c71 = bokeh_palette('linear_kryw_0_100_c71',linear_kryw_0_100_c71)
m_linear_kryw_0_100_c71 = mpl_cm('linear_kryw_0_100_c71',linear_kryw_0_100_c71)
m_linear_kryw_0_100_c71_r = mpl_cm('linear_kryw_0_100_c71_r',list(reversed(linear_kryw_0_100_c71)))
fire = b_linear_kryw_0_100_c71
m_fire = m_linear_kryw_0_100_c71
m_fire_r = m_linear_kryw_0_100_c71_r
palette['fire'] = b_linear_kryw_0_100_c71
palette_n['fire'] = b_linear_kryw_0_100_c71
cm['fire'] = m_linear_kryw_0_100_c71
cm['fire_r'] = m_linear_kryw_0_100_c71_r
cm_n['fire'] = mpl_cm('fire',linear_kryw_0_100_c71)
cm_n['fire_r'] = mpl_cm('fire_r',list(reversed(linear_kryw_0_100_c71)))
CET_L3 = b_linear_kryw_0_100_c71
m_CET_L3 = m_linear_kryw_0_100_c71
m_CET_L3_r = m_linear_kryw_0_100_c71_r
palette['CET_L3'] = b_linear_kryw_0_100_c71
cm['CET_L3'] = m_linear_kryw_0_100_c71
cm['CET_L3_r'] = m_linear_kryw_0_100_c71_r
register_cmap('cet_CET_L3',m_linear_kryw_0_100_c71)
register_cmap('cet_CET_L3_r',m_linear_kryw_0_100_c71_r)




diverging_bkr_55_10_c35 = [  # cmap_def
[0.097481, 0.50736, 0.98205],
[0.1033, 0.50389, 0.97428],
[0.10863, 0.50041, 0.96653],
[0.11369, 0.49694, 0.9588],
[0.11841, 0.49348, 0.95107],
[0.12275, 0.49002, 0.94336],
[0.12692, 0.48657, 0.93566],
[0.13089, 0.48312, 0.92796],
[0.1346, 0.47968, 0.92028],
[0.13811, 0.47624, 0.91262],
[0.14143, 0.4728, 0.90497],
[0.14458, 0.46938, 0.89732],
[0.14758, 0.46594, 0.88969],
[0.15043, 0.46252, 0.88207],
[0.15312, 0.45912, 0.87446],
[0.15574, 0.4557, 0.86687],
[0.1582, 0.4523, 0.85929],
[0.16051, 0.4489, 0.85172],
[0.16277, 0.44549, 0.84416],
[0.16488, 0.44211, 0.83662],
[0.16693, 0.43871, 0.82909],
[0.16886, 0.43534, 0.82157],
[0.17068, 0.43197, 0.81406],
[0.17242, 0.42859, 0.80657],
[0.17409, 0.42522, 0.79909],
[0.17566, 0.42185, 0.79162],
[0.17718, 0.41849, 0.78417],
[0.17862, 0.41515, 0.77672],
[0.17995, 0.4118, 0.76929],
[0.18122, 0.40846, 0.76187],
[0.18243, 0.40512, 0.75448],
[0.18359, 0.40179, 0.74709],
[0.18468, 0.39847, 0.73972],
[0.18572, 0.39515, 0.73235],
[0.18665, 0.39183, 0.725],
[0.18758, 0.38852, 0.71767],
[0.18841, 0.38521, 0.71034],
[0.18918, 0.38191, 0.70304],
[0.1899, 0.37862, 0.69574],
[0.19058, 0.37533, 0.68847],
[0.1912, 0.37204, 0.68119],
[0.19177, 0.36878, 0.67395],
[0.1923, 0.3655, 0.6667],
[0.19277, 0.36224, 0.65948],
[0.1932, 0.35896, 0.65228],
[0.19358, 0.35572, 0.64508],
[0.19393, 0.35246, 0.6379],
[0.19422, 0.34922, 0.63074],
[0.19446, 0.34597, 0.62359],
[0.19466, 0.34276, 0.61645],
[0.19481, 0.33953, 0.60933],
[0.19492, 0.33629, 0.60222],
[0.19499, 0.33309, 0.59513],
[0.19502, 0.32988, 0.58805],
[0.19502, 0.32665, 0.58099],
[0.19497, 0.32346, 0.57394],
[0.19489, 0.32028, 0.56692],
[0.19477, 0.31708, 0.55991],
[0.19462, 0.31391, 0.5529],
[0.19442, 0.31074, 0.54593],
[0.19419, 0.30755, 0.53895],
[0.19391, 0.30439, 0.532],
[0.1936, 0.30123, 0.52507],
[0.19325, 0.29808, 0.51815],
[0.19288, 0.29493, 0.51124],
[0.19247, 0.29181, 0.50436],
[0.19202, 0.28867, 0.49749],
[0.19154, 0.28553, 0.49064],
[0.19103, 0.28244, 0.48379],
[0.19049, 0.27932, 0.47697],
[0.18992, 0.27622, 0.47017],
[0.18932, 0.27311, 0.46338],
[0.18869, 0.27002, 0.45661],
[0.18804, 0.26691, 0.44986],
[0.18733, 0.26385, 0.44313],
[0.18658, 0.26076, 0.43641],
[0.18585, 0.25769, 0.42971],
[0.18505, 0.25465, 0.42302],
[0.18421, 0.25159, 0.41637],
[0.18335, 0.24855, 0.40972],
[0.18247, 0.24548, 0.40309],
[0.18156, 0.24245, 0.39648],
[0.18062, 0.23944, 0.38989],
[0.17967, 0.23642, 0.38333],
[0.17869, 0.23338, 0.37678],
[0.17768, 0.23038, 0.37023],
[0.17659, 0.2274, 0.36372],
[0.17551, 0.22438, 0.35724],
[0.17442, 0.22142, 0.35076],
[0.17327, 0.21845, 0.34429],
[0.17211, 0.21547, 0.33785],
[0.17093, 0.21251, 0.33143],
[0.16973, 0.20955, 0.32503],
[0.16847, 0.20663, 0.31867],
[0.16721, 0.20368, 0.3123],
[0.16588, 0.20071, 0.30599],
[0.16458, 0.1978, 0.29967],
[0.16324, 0.1949, 0.29336],
[0.16187, 0.19198, 0.28708],
[0.16042, 0.18908, 0.28083],
[0.15902, 0.1862, 0.27461],
[0.15755, 0.18329, 0.26841],
[0.1561, 0.18041, 0.26222],
[0.15456, 0.17758, 0.25605],
[0.15302, 0.17471, 0.2499],
[0.15147, 0.17184, 0.24379],
[0.14991, 0.16902, 0.23772],
[0.14827, 0.16616, 0.23164],
[0.14665, 0.16336, 0.22559],
[0.14505, 0.16051, 0.21957],
[0.14339, 0.15773, 0.21357],
[0.14175, 0.15496, 0.20763],
[0.14009, 0.1522, 0.20171],
[0.13851, 0.14952, 0.19588],
[0.13692, 0.1468, 0.19008],
[0.13541, 0.14419, 0.18437],
[0.134, 0.14163, 0.17876],
[0.13268, 0.1391, 0.17324],
[0.1315, 0.1367, 0.16787],
[0.13053, 0.13444, 0.16268],
[0.12973, 0.13226, 0.15764],
[0.12916, 0.13028, 0.15284],
[0.12891, 0.12842, 0.1483],
[0.12899, 0.12676, 0.14407],
[0.12945, 0.12536, 0.14011],
[0.13031, 0.12415, 0.13652],
[0.13151, 0.12321, 0.13332],
[0.13315, 0.12253, 0.13051],
[0.13523, 0.12211, 0.12804],
[0.13775, 0.12196, 0.126],
[0.14065, 0.12206, 0.12429],
[0.14394, 0.12241, 0.12299],
[0.14762, 0.12299, 0.12205],
[0.15161, 0.12379, 0.12145],
[0.15595, 0.12478, 0.12115],
[0.1605, 0.12596, 0.12111],
[0.16531, 0.1272, 0.12131],
[0.17036, 0.12863, 0.12171],
[0.17553, 0.13018, 0.12228],
[0.18085, 0.13174, 0.12299],
[0.1863, 0.13339, 0.12382],
[0.1918, 0.1351, 0.12474],
[0.19738, 0.13681, 0.12575],
[0.20301, 0.13858, 0.12675],
[0.20867, 0.14035, 0.12784],
[0.21437, 0.14214, 0.12891],
[0.2201, 0.14394, 0.13008],
[0.22584, 0.14575, 0.13119],
[0.23158, 0.14753, 0.13232],
[0.23735, 0.14933, 0.13347],
[0.24308, 0.15109, 0.13464],
[0.24887, 0.15287, 0.13574],
[0.25465, 0.15465, 0.13692],
[0.26041, 0.15645, 0.1381],
[0.26621, 0.15821, 0.1392],
[0.27198, 0.15993, 0.14038],
[0.27781, 0.16174, 0.14154],
[0.28361, 0.16349, 0.14269],
[0.28944, 0.16518, 0.14384],
[0.29526, 0.16697, 0.14501],
[0.30109, 0.16869, 0.14614],
[0.30695, 0.17041, 0.1473],
[0.3128, 0.17212, 0.14842],
[0.31867, 0.17385, 0.14962],
[0.32454, 0.17555, 0.15074],
[0.33043, 0.17728, 0.15189],
[0.33632, 0.17898, 0.15304],
[0.34224, 0.18066, 0.1542],
[0.34815, 0.18235, 0.15536],
[0.35407, 0.18405, 0.15654],
[0.36001, 0.18576, 0.15766],
[0.36596, 0.18743, 0.15884],
[0.3719, 0.18909, 0.15995],
[0.37787, 0.19075, 0.16115],
[0.38385, 0.19241, 0.16231],
[0.38983, 0.1941, 0.16346],
[0.39584, 0.19573, 0.1646],
[0.40184, 0.19738, 0.16573],
[0.40786, 0.19901, 0.16693],
[0.41389, 0.20064, 0.16806],
[0.41994, 0.20228, 0.16925],
[0.42598, 0.20394, 0.17038],
[0.43206, 0.20555, 0.17153],
[0.43813, 0.20718, 0.17269],
[0.44421, 0.20879, 0.17385],
[0.4503, 0.2104, 0.17502],
[0.4564, 0.21202, 0.17614],
[0.46251, 0.21361, 0.17734],
[0.46864, 0.2152, 0.1785],
[0.47479, 0.21681, 0.17964],
[0.48092, 0.21839, 0.18078],
[0.48709, 0.21998, 0.18194],
[0.49325, 0.22155, 0.1831],
[0.49943, 0.22314, 0.18427],
[0.50562, 0.22468, 0.18546],
[0.51182, 0.22628, 0.18659],
[0.51804, 0.22784, 0.18779],
[0.52426, 0.22938, 0.18892],
[0.53049, 0.23094, 0.19008],
[0.53674, 0.23249, 0.19123],
[0.54299, 0.23403, 0.1924],
[0.54925, 0.23559, 0.19357],
[0.55552, 0.23713, 0.19474],
[0.56181, 0.23863, 0.19589],
[0.5681, 0.24016, 0.19706],
[0.57441, 0.2417, 0.1982],
[0.58073, 0.2432, 0.19936],
[0.58705, 0.24472, 0.20052],
[0.59339, 0.24623, 0.20169],
[0.59974, 0.24776, 0.20286],
[0.60609, 0.24925, 0.20405],
[0.61246, 0.25075, 0.20519],
[0.61885, 0.25225, 0.20639],
[0.62523, 0.25374, 0.20753],
[0.63163, 0.25524, 0.20869],
[0.63803, 0.25671, 0.20986],
[0.64445, 0.25817, 0.21103],
[0.65088, 0.25967, 0.21221],
[0.65732, 0.26113, 0.21335],
[0.66377, 0.2626, 0.21453],
[0.67023, 0.26407, 0.21571],
[0.6767, 0.26552, 0.21688],
[0.68317, 0.26696, 0.21804],
[0.68966, 0.26842, 0.21921],
[0.69616, 0.26988, 0.22039],
[0.70267, 0.27131, 0.22154],
[0.70919, 0.27274, 0.22272],
[0.71571, 0.27418, 0.22387],
[0.72224, 0.27563, 0.22504],
[0.72879, 0.27705, 0.22624],
[0.73535, 0.27847, 0.2274],
[0.74191, 0.27988, 0.22858],
[0.74849, 0.28129, 0.22972],
[0.75508, 0.28272, 0.23091],
[0.76167, 0.28411, 0.23209],
[0.76827, 0.28551, 0.23324],
[0.77488, 0.28691, 0.23444],
[0.78151, 0.28831, 0.23561],
[0.78814, 0.28971, 0.23679],
[0.79478, 0.2911, 0.23795],
[0.80143, 0.29248, 0.23912],
[0.80809, 0.29385, 0.24028],
[0.81476, 0.29523, 0.24148],
[0.82144, 0.29662, 0.24264],
[0.82812, 0.29797, 0.24381],
[0.83482, 0.29935, 0.24499],
[0.84153, 0.3007, 0.24617],
[0.84824, 0.30205, 0.24736],
[0.85497, 0.30341, 0.24855],
[0.8617, 0.30477, 0.2497],
[0.86845, 0.30613, 0.25089],
[0.8752, 0.30745, 0.25208],
[0.88196, 0.30881, 0.25325],
[0.88873, 0.31015, 0.25444],
[0.8955, 0.31147, 0.25561],
[0.90229, 0.31279, 0.25679],
]
b_diverging_bkr_55_10_c35 = bokeh_palette('diverging_bkr_55_10_c35',diverging_bkr_55_10_c35)
m_diverging_bkr_55_10_c35 = mpl_cm('diverging_bkr_55_10_c35',diverging_bkr_55_10_c35)
m_diverging_bkr_55_10_c35_r = mpl_cm('diverging_bkr_55_10_c35_r',list(reversed(diverging_bkr_55_10_c35)))
bkr = b_diverging_bkr_55_10_c35
m_bkr = m_diverging_bkr_55_10_c35
m_bkr_r = m_diverging_bkr_55_10_c35_r
palette['bkr'] = b_diverging_bkr_55_10_c35
palette_n['bkr'] = b_diverging_bkr_55_10_c35
cm['bkr'] = m_diverging_bkr_55_10_c35
cm['bkr_r'] = m_diverging_bkr_55_10_c35_r
cm_n['bkr'] = mpl_cm('bkr',diverging_bkr_55_10_c35)
cm_n['bkr_r'] = mpl_cm('bkr_r',list(reversed(diverging_bkr_55_10_c35)))
CET_D4 = b_diverging_bkr_55_10_c35
m_CET_D4 = m_diverging_bkr_55_10_c35
m_CET_D4_r = m_diverging_bkr_55_10_c35_r
palette['CET_D4'] = b_diverging_bkr_55_10_c35
cm['CET_D4'] = m_diverging_bkr_55_10_c35
cm['CET_D4_r'] = m_diverging_bkr_55_10_c35_r
register_cmap('cet_CET_D4',m_diverging_bkr_55_10_c35)
register_cmap('cet_CET_D4_r',m_diverging_bkr_55_10_c35_r)




linear_bmw_5_95_c89 = [  # cmap_def
[0, 0.017709, 0.34641],
[0, 0.020672, 0.3555],
[0, 0.023402, 0.3646],
[0, 0.025899, 0.37376],
[0, 0.028159, 0.38295],
[0, 0.030176, 0.39217],
[0, 0.031948, 0.40143],
[0, 0.03346, 0.41074],
[0, 0.034886, 0.42006],
[0, 0.036005, 0.42943],
[0, 0.037031, 0.43882],
[0, 0.038092, 0.44825],
[0, 0.039171, 0.45769],
[0, 0.04027, 0.46718],
[0, 0.041408, 0.47668],
[0, 0.042463, 0.4862],
[0, 0.043503, 0.49576],
[0, 0.044741, 0.50533],
[0, 0.045774, 0.51494],
[0, 0.047001, 0.52456],
[0, 0.048092, 0.5342],
[0, 0.049352, 0.54387],
[0, 0.050447, 0.55355],
[0, 0.051609, 0.56326],
[0, 0.052795, 0.57298],
[0, 0.054014, 0.58271],
[0, 0.055261, 0.59247],
[0, 0.056563, 0.60223],
[0, 0.057716, 0.61201],
[0, 0.059101, 0.6218],
[0, 0.060386, 0.63161],
[0, 0.061629, 0.64141],
[0, 0.063072, 0.65124],
[0, 0.064391, 0.66106],
[0, 0.065778, 0.6709],
[0, 0.067217, 0.68073],
[0, 0.068717, 0.69056],
[0, 0.070145, 0.7004],
[0, 0.071697, 0.71023],
[0, 0.073287, 0.72005],
[0, 0.074826, 0.72985],
[0, 0.076443, 0.73965],
[0, 0.078139, 0.74942],
[0, 0.079895, 0.75915],
[0, 0.08168, 0.76872],
[0, 0.083307, 0.77811],
[0, 0.084962, 0.78731],
[0, 0.086573, 0.79632],
[0, 0.088156, 0.80515],
[0, 0.089689, 0.81379],
[0, 0.091208, 0.82225],
[0, 0.092616, 0.83051],
[0, 0.09408, 0.83857],
[0, 0.095474, 0.84644],
[0, 0.096762, 0.8541],
[0, 0.098154, 0.86158],
[0, 0.099403, 0.86886],
[0, 0.10065, 0.87593],
[0, 0.10187, 0.88281],
[0, 0.10309, 0.88949],
[0, 0.1042, 0.89597],
[0, 0.10532, 0.90224],
[0, 0.10647, 0.9083],
[0, 0.10749, 0.91415],
[0, 0.10849, 0.9198],
[0, 0.10956, 0.92525],
[0, 0.11052, 0.9305],
[0, 0.11147, 0.93553],
[0, 0.1124, 0.94036],
[0, 0.11331, 0.94497],
[0, 0.11413, 0.94937],
[0, 0.11498, 0.95357],
[0, 0.11585, 0.9576],
[0, 0.11662, 0.96148],
[0, 0.11733, 0.96522],
[0, 0.11806, 0.96883],
[0, 0.11873, 0.97228],
[0, 0.11931, 0.97558],
[0, 0.11987, 0.97874],
[0, 0.12039, 0.98176],
[0, 0.12087, 0.98464],
[0, 0.12132, 0.98737],
[0, 0.12173, 0.98995],
[0.042484, 0.1221, 0.99239],
[0.096239, 0.12243, 0.99468],
[0.13272, 0.12272, 0.99682],
[0.16225, 0.12297, 0.99881],
[0.18788, 0.12318, 1],
[0.21091, 0.12336, 1],
[0.23216, 0.1235, 1],
[0.25202, 0.1236, 1],
[0.27081, 0.12366, 1],
[0.28872, 0.12369, 1],
[0.30596, 0.12367, 1],
[0.32259, 0.12362, 1],
[0.33869, 0.12354, 1],
[0.35436, 0.12342, 1],
[0.36964, 0.12326, 1],
[0.38457, 0.12307, 1],
[0.39919, 0.12286, 1],
[0.41348, 0.12262, 1],
[0.42736, 0.12239, 1],
[0.44084, 0.12216, 1],
[0.454, 0.12195, 1],
[0.46684, 0.12174, 1],
[0.47939, 0.12155, 1],
[0.49169, 0.12136, 1],
[0.50374, 0.12117, 1],
[0.51557, 0.121, 1],
[0.52721, 0.12083, 1],
[0.53866, 0.12067, 1],
[0.54992, 0.12051, 1],
[0.56102, 0.12037, 1],
[0.57198, 0.12022, 1],
[0.58279, 0.12009, 1],
[0.59347, 0.11996, 1],
[0.60403, 0.11983, 1],
[0.61446, 0.11971, 1],
[0.62479, 0.11959, 1],
[0.63501, 0.11948, 1],
[0.64513, 0.11938, 1],
[0.65516, 0.11927, 1],
[0.66511, 0.11918, 1],
[0.67497, 0.11908, 1],
[0.68475, 0.11899, 1],
[0.69445, 0.1189, 1],
[0.7041, 0.11881, 1],
[0.71366, 0.11873, 1],
[0.72312, 0.11873, 1],
[0.73244, 0.11896, 1],
[0.74156, 0.11949, 1],
[0.75051, 0.12036, 1],
[0.75927, 0.12156, 1],
[0.76785, 0.12312, 1],
[0.77627, 0.12504, 1],
[0.78452, 0.12724, 1],
[0.79261, 0.12981, 1],
[0.80054, 0.13262, 1],
[0.80832, 0.1357, 1],
[0.81594, 0.1391, 1],
[0.82341, 0.14279, 1],
[0.83074, 0.14669, 1],
[0.83791, 0.15087, 1],
[0.84494, 0.15527, 1],
[0.85183, 0.15984, 1],
[0.85859, 0.16467, 1],
[0.8652, 0.16967, 1],
[0.87169, 0.17481, 1],
[0.87803, 0.1801, 1],
[0.88424, 0.18562, 1],
[0.89031, 0.1912, 1],
[0.89626, 0.19697, 1],
[0.90207, 0.20282, 1],
[0.90776, 0.2088, 1],
[0.91332, 0.21488, 1],
[0.91875, 0.22109, 1],
[0.92405, 0.22737, 1],
[0.92919, 0.23379, 1],
[0.93415, 0.24041, 1],
[0.93894, 0.2472, 1],
[0.94355, 0.25413, 1],
[0.94798, 0.26118, 1],
[0.95224, 0.26837, 1],
[0.95633, 0.2757, 1],
[0.96024, 0.28311, 1],
[0.96396, 0.29064, 1],
[0.96751, 0.29827, 1],
[0.97089, 0.30602, 1],
[0.97409, 0.31382, 1],
[0.97712, 0.3217, 1],
[0.97997, 0.32968, 1],
[0.98265, 0.33769, 1],
[0.98515, 0.34577, 1],
[0.98746, 0.35394, 1],
[0.9896, 0.36216, 1],
[0.99155, 0.37042, 1],
[0.99333, 0.37874, 1],
[0.99493, 0.3871, 1],
[0.99635, 0.39551, 1],
[0.9976, 0.40394, 1],
[0.99866, 0.41243, 1],
[0.99953, 0.42094, 1],
[1, 0.4295, 1],
[1, 0.43809, 1],
[1, 0.44668, 1],
[1, 0.45524, 1],
[1, 0.46369, 1],
[1, 0.47203, 1],
[1, 0.48024, 1],
[1, 0.48835, 1],
[1, 0.49634, 1],
[1, 0.50426, 1],
[1, 0.51206, 1],
[1, 0.51978, 1],
[1, 0.52742, 1],
[1, 0.53498, 1],
[1, 0.54247, 1],
[1, 0.54988, 1],
[1, 0.55722, 1],
[1, 0.56449, 1],
[1, 0.57169, 1],
[1, 0.57884, 1],
[1, 0.58594, 1],
[1, 0.59297, 1],
[1, 0.59995, 1],
[1, 0.60688, 1],
[1, 0.61374, 1],
[1, 0.62058, 1],
[1, 0.62736, 1],
[1, 0.6341, 1],
[1, 0.64079, 1],
[1, 0.64743, 1],
[1, 0.65405, 1],
[1, 0.66063, 1],
[1, 0.66718, 1],
[1, 0.6737, 1],
[1, 0.68019, 1],
[1, 0.68667, 1],
[1, 0.69312, 1],
[1, 0.69954, 1],
[1, 0.70595, 1],
[1, 0.71233, 1],
[1, 0.71869, 1],
[1, 0.72504, 1],
[1, 0.73136, 1],
[1, 0.73767, 1],
[1, 0.74396, 1],
[1, 0.75023, 1],
[1, 0.75649, 1],
[1, 0.76273, 1],
[1, 0.76895, 1],
[1, 0.77517, 1],
[1, 0.78137, 1],
[1, 0.78755, 1],
[1, 0.79372, 1],
[1, 0.79988, 1],
[1, 0.80602, 1],
[1, 0.81215, 1],
[1, 0.81826, 1],
[1, 0.82437, 1],
[1, 0.83047, 1],
[1, 0.83656, 1],
[1, 0.84263, 1],
[1, 0.84869, 1],
[1, 0.85475, 1],
[1, 0.86079, 1],
[1, 0.86683, 1],
[1, 0.87286, 0.99978],
[1, 0.87887, 0.9995],
[1, 0.88488, 0.99921],
[1, 0.89088, 0.99892],
[0.99975, 0.89688, 0.99861],
[0.99925, 0.90287, 0.9983],
[0.99872, 0.90884, 0.99799],
[0.99817, 0.91481, 0.99766],
[0.9976, 0.92077, 0.99733],
]
b_linear_bmw_5_95_c89 = bokeh_palette('linear_bmw_5_95_c89',linear_bmw_5_95_c89)
m_linear_bmw_5_95_c89 = mpl_cm('linear_bmw_5_95_c89',linear_bmw_5_95_c89)
m_linear_bmw_5_95_c89_r = mpl_cm('linear_bmw_5_95_c89_r',list(reversed(linear_bmw_5_95_c89)))
bmw = b_linear_bmw_5_95_c89
m_bmw = m_linear_bmw_5_95_c89
m_bmw_r = m_linear_bmw_5_95_c89_r
palette['bmw'] = b_linear_bmw_5_95_c89
palette_n['bmw'] = b_linear_bmw_5_95_c89
cm['bmw'] = m_linear_bmw_5_95_c89
cm['bmw_r'] = m_linear_bmw_5_95_c89_r
cm_n['bmw'] = mpl_cm('bmw',linear_bmw_5_95_c89)
cm_n['bmw_r'] = mpl_cm('bmw_r',list(reversed(linear_bmw_5_95_c89)))




cyclic_wrwbw_40_90_c42_s25 = [  # cmap_def
[0.10246, 0.39014, 0.89689],
[0.11205, 0.39152, 0.89693],
[0.12596, 0.39371, 0.897],
[0.14277, 0.39668, 0.89709],
[0.16137, 0.40041, 0.8972],
[0.18085, 0.40478, 0.89732],
[0.20072, 0.4098, 0.89746],
[0.22066, 0.41534, 0.8976],
[0.24027, 0.42135, 0.89775],
[0.25956, 0.42779, 0.89791],
[0.27833, 0.43456, 0.89806],
[0.29659, 0.44162, 0.8982],
[0.31428, 0.44891, 0.89835],
[0.33142, 0.45639, 0.89848],
[0.34803, 0.46404, 0.89861],
[0.3641, 0.4718, 0.89873],
[0.37969, 0.47966, 0.89883],
[0.39481, 0.48761, 0.89893],
[0.40949, 0.49562, 0.89901],
[0.42377, 0.5037, 0.89909],
[0.4377, 0.51182, 0.89914],
[0.45127, 0.51998, 0.89919],
[0.46453, 0.52819, 0.89922],
[0.47747, 0.53643, 0.89924],
[0.49016, 0.54469, 0.89925],
[0.50259, 0.55298, 0.89924],
[0.51481, 0.56131, 0.89922],
[0.52682, 0.56967, 0.89919],
[0.53863, 0.57806, 0.89914],
[0.55025, 0.58648, 0.89908],
[0.56172, 0.59493, 0.89901],
[0.57302, 0.60341, 0.89892],
[0.58417, 0.61191, 0.89881],
[0.59519, 0.62045, 0.89869],
[0.60608, 0.629, 0.89856],
[0.61685, 0.63759, 0.89841],
[0.62751, 0.6462, 0.89825],
[0.63805, 0.65483, 0.89807],
[0.6485, 0.66351, 0.89788],
[0.65884, 0.67219, 0.89767],
[0.6691, 0.6809, 0.89745],
[0.67927, 0.68965, 0.89721],
[0.68936, 0.6984, 0.89695],
[0.69935, 0.70719, 0.89668],
[0.70928, 0.71598, 0.89638],
[0.71911, 0.72477, 0.89605],
[0.72887, 0.73355, 0.89569],
[0.73854, 0.7423, 0.89528],
[0.74812, 0.75101, 0.89481],
[0.75759, 0.75963, 0.89427],
[0.76695, 0.76815, 0.89362],
[0.7762, 0.77652, 0.89286],
[0.78529, 0.78468, 0.89193],
[0.79422, 0.79255, 0.8908],
[0.80295, 0.80008, 0.88943],
[0.81147, 0.80717, 0.88775],
[0.81973, 0.81373, 0.88571],
[0.82771, 0.81967, 0.88323],
[0.83537, 0.82487, 0.88026],
[0.84268, 0.82926, 0.87672],
[0.84961, 0.83271, 0.87256],
[0.85612, 0.83516, 0.86771],
[0.8622, 0.83654, 0.86213],
[0.86783, 0.83681, 0.85578],
[0.873, 0.83593, 0.84864],
[0.8777, 0.8339, 0.84071],
[0.88194, 0.83075, 0.832],
[0.88572, 0.82651, 0.82253],
[0.88906, 0.82126, 0.81236],
[0.89199, 0.81507, 0.80154],
[0.89451, 0.80802, 0.79013],
[0.89666, 0.80023, 0.77819],
[0.89847, 0.79178, 0.76581],
[0.89996, 0.78278, 0.75305],
[0.90116, 0.77331, 0.73999],
[0.9021, 0.76347, 0.72667],
[0.9028, 0.75332, 0.71316],
[0.90327, 0.74294, 0.69952],
[0.90355, 0.73237, 0.68579],
[0.90363, 0.72166, 0.67199],
[0.90355, 0.71085, 0.65815],
[0.9033, 0.69996, 0.64429],
[0.90289, 0.68903, 0.63046],
[0.90234, 0.67804, 0.61663],
[0.90165, 0.66702, 0.60282],
[0.90082, 0.65598, 0.58904],
[0.89986, 0.64492, 0.57531],
[0.89877, 0.63386, 0.56162],
[0.89757, 0.62278, 0.54797],
[0.89623, 0.61169, 0.53437],
[0.89478, 0.60058, 0.52081],
[0.8932, 0.58945, 0.5073],
[0.89151, 0.57831, 0.49383],
[0.8897, 0.56715, 0.48041],
[0.88777, 0.55595, 0.46704],
[0.88574, 0.54474, 0.4537],
[0.88359, 0.53349, 0.44042],
[0.88134, 0.52221, 0.42718],
[0.87897, 0.5109, 0.41398],
[0.87649, 0.49954, 0.40083],
[0.87391, 0.48816, 0.3877],
[0.87123, 0.4767, 0.37464],
[0.86845, 0.4652, 0.3616],
[0.86556, 0.45364, 0.34861],
[0.86257, 0.44202, 0.33566],
[0.85948, 0.43032, 0.32274],
[0.8563, 0.41853, 0.30985],
[0.85302, 0.40666, 0.29699],
[0.84965, 0.3947, 0.28415],
[0.84618, 0.38265, 0.27137],
[0.84264, 0.3705, 0.25865],
[0.83901, 0.35827, 0.24597],
[0.83532, 0.34594, 0.23336],
[0.83157, 0.33356, 0.22089],
[0.82777, 0.3211, 0.20848],
[0.82395, 0.30862, 0.19626],
[0.82013, 0.29614, 0.1842],
[0.81634, 0.28375, 0.17242],
[0.81259, 0.2715, 0.16099],
[0.80896, 0.25953, 0.14996],
[0.80546, 0.24792, 0.1394],
[0.80217, 0.23683, 0.12955],
[0.79914, 0.22645, 0.1204],
[0.7964, 0.21697, 0.11224],
[0.79405, 0.20862, 0.10504],
[0.79209, 0.20162, 0.099146],
[0.79061, 0.19625, 0.094653],
[0.78962, 0.19261, 0.091622],
[0.78916, 0.19089, 0.090163],
[0.78923, 0.19116, 0.09039],
[0.78984, 0.19342, 0.092261],
[0.79096, 0.19754, 0.095704],
[0.79258, 0.20339, 0.10062],
[0.79465, 0.21076, 0.10693],
[0.79711, 0.21943, 0.11431],
[0.79994, 0.22917, 0.12278],
[0.80304, 0.23975, 0.13216],
[0.8064, 0.251, 0.14222],
[0.80993, 0.26273, 0.15286],
[0.81359, 0.27479, 0.16406],
[0.81735, 0.28707, 0.17556],
[0.82116, 0.29951, 0.18745],
[0.82498, 0.31196, 0.1995],
[0.8288, 0.32444, 0.21181],
[0.83258, 0.33689, 0.2242],
[0.83632, 0.34927, 0.23677],
[0.84, 0.36157, 0.24937],
[0.84359, 0.37378, 0.26206],
[0.84712, 0.38589, 0.27479],
[0.85056, 0.39792, 0.28758],
[0.8539, 0.40986, 0.30042],
[0.85716, 0.42169, 0.3133],
[0.86032, 0.43345, 0.32618],
[0.86338, 0.44514, 0.33913],
[0.86634, 0.45674, 0.35208],
[0.8692, 0.46829, 0.36509],
[0.87196, 0.47978, 0.37813],
[0.87462, 0.49122, 0.39122],
[0.87716, 0.50259, 0.40434],
[0.87961, 0.51395, 0.41751],
[0.88195, 0.52525, 0.43072],
[0.88418, 0.53652, 0.44398],
[0.8863, 0.54775, 0.45727],
[0.8883, 0.55896, 0.47062],
[0.89019, 0.57014, 0.484],
[0.89197, 0.5813, 0.49745],
[0.89363, 0.59245, 0.51092],
[0.89518, 0.60356, 0.52444],
[0.8966, 0.61466, 0.53802],
[0.8979, 0.62575, 0.55163],
[0.89908, 0.63683, 0.56528],
[0.90012, 0.64788, 0.579],
[0.90103, 0.65892, 0.59275],
[0.9018, 0.66994, 0.60654],
[0.90242, 0.6809, 0.62037],
[0.90287, 0.69183, 0.63422],
[0.90316, 0.70268, 0.64808],
[0.90325, 0.71344, 0.66197],
[0.90314, 0.72408, 0.67583],
[0.9028, 0.73456, 0.68967],
[0.9022, 0.74481, 0.70344],
[0.90131, 0.7548, 0.71713],
[0.90011, 0.76443, 0.73066],
[0.89855, 0.77364, 0.74401],
[0.8966, 0.78234, 0.75711],
[0.89423, 0.79041, 0.7699],
[0.8914, 0.79778, 0.7823],
[0.88809, 0.80433, 0.79423],
[0.88427, 0.80997, 0.80561],
[0.87992, 0.81463, 0.8164],
[0.87505, 0.81822, 0.8265],
[0.86966, 0.82071, 0.83587],
[0.86376, 0.82205, 0.84446],
[0.85737, 0.82224, 0.85225],
[0.85052, 0.8213, 0.85923],
[0.84325, 0.81927, 0.8654],
[0.8356, 0.81621, 0.87079],
[0.8276, 0.81219, 0.87544],
[0.81931, 0.80731, 0.8794],
[0.81075, 0.80165, 0.88274],
[0.80198, 0.79534, 0.88552],
[0.79302, 0.78846, 0.88781],
[0.7839, 0.78111, 0.88969],
[0.77463, 0.77338, 0.89122],
[0.76527, 0.76536, 0.89246],
[0.75579, 0.75709, 0.89345],
[0.74622, 0.74866, 0.89427],
[0.73657, 0.7401, 0.89494],
[0.72685, 0.73145, 0.89549],
[0.71705, 0.72274, 0.89595],
[0.70717, 0.71401, 0.89635],
[0.69722, 0.70526, 0.89669],
[0.68719, 0.69651, 0.89699],
[0.67709, 0.68776, 0.89726],
[0.66689, 0.67902, 0.8975],
[0.65663, 0.67032, 0.89772],
[0.64625, 0.66163, 0.89792],
[0.63579, 0.65297, 0.89811],
[0.62522, 0.64434, 0.89828],
[0.61453, 0.63573, 0.89844],
[0.60375, 0.62715, 0.89859],
[0.59283, 0.6186, 0.89872],
[0.58178, 0.61007, 0.89884],
[0.57059, 0.60157, 0.89894],
[0.55925, 0.59311, 0.89902],
[0.54776, 0.58467, 0.8991],
[0.53609, 0.57626, 0.89915],
[0.52424, 0.56787, 0.8992],
[0.51219, 0.55952, 0.89923],
[0.49993, 0.5512, 0.89925],
[0.48745, 0.54291, 0.89925],
[0.47471, 0.53465, 0.89924],
[0.46169, 0.52643, 0.89922],
[0.44837, 0.51823, 0.89918],
[0.43473, 0.51007, 0.89913],
[0.42073, 0.50196, 0.89907],
[0.40637, 0.49389, 0.899],
[0.39159, 0.48588, 0.89891],
[0.37638, 0.47795, 0.89881],
[0.36067, 0.47012, 0.8987],
[0.34449, 0.46237, 0.89858],
[0.32777, 0.45477, 0.89845],
[0.31053, 0.44731, 0.89832],
[0.29269, 0.44007, 0.89817],
[0.27432, 0.43307, 0.89802],
[0.25544, 0.42636, 0.89787],
[0.23609, 0.42002, 0.89772],
[0.21638, 0.4141, 0.89757],
[0.19647, 0.40866, 0.89743],
[0.17659, 0.40379, 0.89729],
[0.15722, 0.39953, 0.89717],
[0.13894, 0.39598, 0.89707],
[0.1226, 0.39316, 0.89698],
[0.10959, 0.39116, 0.89692],
[0.1011, 0.38996, 0.89689],
[0.098617, 0.38962, 0.89687],
]
b_cyclic_wrwbw_40_90_c42_s25 = bokeh_palette('cyclic_wrwbw_40_90_c42_s25',cyclic_wrwbw_40_90_c42_s25)
m_cyclic_wrwbw_40_90_c42_s25 = mpl_cm('cyclic_wrwbw_40_90_c42_s25',cyclic_wrwbw_40_90_c42_s25)
m_cyclic_wrwbw_40_90_c42_s25_r = mpl_cm('cyclic_wrwbw_40_90_c42_s25_r',list(reversed(cyclic_wrwbw_40_90_c42_s25)))
CET_C4s = b_cyclic_wrwbw_40_90_c42_s25
m_CET_C4s = m_cyclic_wrwbw_40_90_c42_s25
m_CET_C4s_r = m_cyclic_wrwbw_40_90_c42_s25_r
palette['CET_C4s'] = b_cyclic_wrwbw_40_90_c42_s25
cm['CET_C4s'] = m_cyclic_wrwbw_40_90_c42_s25
cm['CET_C4s_r'] = m_cyclic_wrwbw_40_90_c42_s25_r
register_cmap('cet_CET_C4s',m_cyclic_wrwbw_40_90_c42_s25)
register_cmap('cet_CET_C4s_r',m_cyclic_wrwbw_40_90_c42_s25_r)




isoluminant_cgo_80_c38 = [  # cmap_def
[0.43773, 0.82141, 1],
[0.438, 0.82178, 1],
[0.43825, 0.82216, 1],
[0.43853, 0.82252, 1],
[0.4388, 0.82289, 1],
[0.43909, 0.82325, 0.99787],
[0.43939, 0.82362, 0.99477],
[0.43969, 0.82398, 0.99168],
[0.44, 0.82433, 0.98857],
[0.44031, 0.82469, 0.98546],
[0.44063, 0.82504, 0.98235],
[0.44095, 0.8254, 0.97923],
[0.44129, 0.82575, 0.97611],
[0.44163, 0.82609, 0.97299],
[0.44199, 0.82644, 0.96986],
[0.44234, 0.82678, 0.96673],
[0.4427, 0.82712, 0.96359],
[0.44307, 0.82746, 0.96045],
[0.44345, 0.82779, 0.95731],
[0.44384, 0.82812, 0.95416],
[0.44424, 0.82845, 0.95101],
[0.44464, 0.82878, 0.94785],
[0.44505, 0.82911, 0.94469],
[0.44546, 0.82943, 0.94152],
[0.44589, 0.82975, 0.93835],
[0.44634, 0.83007, 0.93517],
[0.44679, 0.83038, 0.93199],
[0.44725, 0.83069, 0.9288],
[0.4477, 0.831, 0.92562],
[0.4482, 0.83131, 0.92243],
[0.44869, 0.83161, 0.91922],
[0.44919, 0.83191, 0.91602],
[0.4497, 0.83221, 0.91281],
[0.45022, 0.8325, 0.9096],
[0.45076, 0.83279, 0.90638],
[0.4513, 0.83308, 0.90316],
[0.45186, 0.83337, 0.89993],
[0.45244, 0.83365, 0.8967],
[0.45301, 0.83394, 0.89345],
[0.4536, 0.83421, 0.89021],
[0.45421, 0.83448, 0.88696],
[0.45483, 0.83475, 0.8837],
[0.45546, 0.83502, 0.88044],
[0.45611, 0.83528, 0.87717],
[0.45676, 0.83555, 0.8739],
[0.45744, 0.8358, 0.87062],
[0.45813, 0.83606, 0.86733],
[0.45884, 0.83631, 0.86405],
[0.45956, 0.83656, 0.86075],
[0.46029, 0.8368, 0.85744],
[0.46104, 0.83704, 0.85414],
[0.46181, 0.83728, 0.85082],
[0.46259, 0.83751, 0.8475],
[0.46339, 0.83774, 0.84417],
[0.46422, 0.83796, 0.84084],
[0.46505, 0.83818, 0.8375],
[0.46591, 0.8384, 0.83416],
[0.46679, 0.83861, 0.8308],
[0.46767, 0.83882, 0.82744],
[0.46859, 0.83903, 0.82407],
[0.46953, 0.83923, 0.8207],
[0.47047, 0.83943, 0.81732],
[0.47145, 0.83962, 0.81393],
[0.47244, 0.83981, 0.81053],
[0.47345, 0.83999, 0.80714],
[0.47449, 0.84017, 0.80373],
[0.47556, 0.84034, 0.80031],
[0.47664, 0.84051, 0.79688],
[0.47774, 0.84067, 0.79346],
[0.47888, 0.84083, 0.79002],
[0.48003, 0.84098, 0.78658],
[0.48121, 0.84113, 0.78312],
[0.48242, 0.84127, 0.77966],
[0.48365, 0.8414, 0.77619],
[0.48491, 0.84153, 0.77272],
[0.4862, 0.84166, 0.76924],
[0.48753, 0.84178, 0.76575],
[0.48887, 0.84189, 0.76225],
[0.49025, 0.842, 0.75875],
[0.49167, 0.8421, 0.75524],
[0.49309, 0.84219, 0.75172],
[0.49457, 0.84228, 0.74819],
[0.49605, 0.84236, 0.74465],
[0.4976, 0.84243, 0.74111],
[0.49917, 0.84249, 0.73756],
[0.50078, 0.84255, 0.73401],
[0.50242, 0.8426, 0.73045],
[0.5041, 0.84264, 0.72687],
[0.5058, 0.84267, 0.7233],
[0.50756, 0.8427, 0.71972],
[0.50935, 0.84271, 0.71613],
[0.51117, 0.84272, 0.71253],
[0.51304, 0.84272, 0.70893],
[0.51496, 0.84271, 0.70532],
[0.51691, 0.84269, 0.70171],
[0.51891, 0.84266, 0.69809],
[0.52094, 0.84262, 0.69447],
[0.52303, 0.84257, 0.69084],
[0.52516, 0.84251, 0.68722],
[0.52734, 0.84243, 0.68359],
[0.52957, 0.84235, 0.67995],
[0.53184, 0.84226, 0.67632],
[0.53416, 0.84215, 0.67268],
[0.53654, 0.84203, 0.66904],
[0.53896, 0.84189, 0.66541],
[0.54142, 0.84175, 0.66178],
[0.54396, 0.84159, 0.65814],
[0.54654, 0.84142, 0.65451],
[0.54917, 0.84123, 0.65088],
[0.55185, 0.84103, 0.64727],
[0.55458, 0.84082, 0.64365],
[0.55737, 0.84059, 0.64005],
[0.56021, 0.84035, 0.63646],
[0.56311, 0.84009, 0.63287],
[0.56606, 0.83981, 0.6293],
[0.56906, 0.83952, 0.62575],
[0.57213, 0.8392, 0.6222],
[0.57524, 0.83888, 0.61869],
[0.5784, 0.83854, 0.61518],
[0.58162, 0.83818, 0.61171],
[0.5849, 0.8378, 0.60826],
[0.58821, 0.83741, 0.60483],
[0.59159, 0.837, 0.60142],
[0.595, 0.83656, 0.59805],
[0.59846, 0.83612, 0.59472],
[0.60197, 0.83565, 0.59142],
[0.60552, 0.83517, 0.58814],
[0.60913, 0.83467, 0.58492],
[0.61275, 0.83415, 0.58173],
[0.61642, 0.83362, 0.57858],
[0.62014, 0.83306, 0.57549],
[0.62388, 0.83249, 0.57243],
[0.62765, 0.83191, 0.56942],
[0.63145, 0.8313, 0.56646],
[0.63527, 0.83068, 0.56356],
[0.63912, 0.83004, 0.56069],
[0.64299, 0.82939, 0.55789],
[0.64689, 0.82872, 0.55514],
[0.6508, 0.82803, 0.55245],
[0.65472, 0.82733, 0.5498],
[0.65865, 0.82662, 0.54721],
[0.66261, 0.82589, 0.54468],
[0.66656, 0.82515, 0.54221],
[0.67053, 0.82439, 0.53979],
[0.6745, 0.82363, 0.53744],
[0.67847, 0.82285, 0.53513],
[0.68244, 0.82206, 0.53288],
[0.68642, 0.82125, 0.53068],
[0.69038, 0.82043, 0.52855],
[0.69435, 0.8196, 0.52647],
[0.6983, 0.81877, 0.5244],
[0.70224, 0.81793, 0.52237],
[0.70617, 0.81709, 0.52036],
[0.71008, 0.81624, 0.51839],
[0.71397, 0.81537, 0.51642],
[0.71785, 0.81451, 0.51449],
[0.72171, 0.81363, 0.51258],
[0.72557, 0.81275, 0.5107],
[0.7294, 0.81187, 0.50885],
[0.73323, 0.81098, 0.50702],
[0.73704, 0.81008, 0.50522],
[0.74083, 0.80918, 0.50344],
[0.74462, 0.80826, 0.50171],
[0.7484, 0.80735, 0.49999],
[0.75216, 0.80642, 0.49833],
[0.75591, 0.80548, 0.49666],
[0.75965, 0.80455, 0.49505],
[0.76337, 0.8036, 0.49346],
[0.76709, 0.80265, 0.49191],
[0.7708, 0.80169, 0.49039],
[0.77448, 0.80072, 0.4889],
[0.77817, 0.79975, 0.48745],
[0.78184, 0.79876, 0.48602],
[0.7855, 0.79777, 0.48463],
[0.78915, 0.79678, 0.48328],
[0.79278, 0.79578, 0.48198],
[0.7964, 0.79477, 0.4807],
[0.80002, 0.79376, 0.47946],
[0.80362, 0.79273, 0.47826],
[0.80722, 0.7917, 0.47709],
[0.8108, 0.79066, 0.47597],
[0.81437, 0.78962, 0.47489],
[0.81792, 0.78857, 0.47383],
[0.82147, 0.78751, 0.47283],
[0.82501, 0.78645, 0.47187],
[0.82853, 0.78538, 0.47094],
[0.83205, 0.7843, 0.47006],
[0.83555, 0.78321, 0.46922],
[0.83903, 0.78212, 0.46841],
[0.84251, 0.78102, 0.46765],
[0.84597, 0.77992, 0.46694],
[0.84943, 0.77881, 0.46626],
[0.85287, 0.77769, 0.46563],
[0.85629, 0.77657, 0.46504],
[0.85971, 0.77544, 0.46451],
[0.86311, 0.7743, 0.46401],
[0.86649, 0.77316, 0.46354],
[0.86986, 0.77201, 0.46313],
[0.87323, 0.77086, 0.46276],
[0.87657, 0.7697, 0.46244],
[0.87991, 0.76853, 0.46216],
[0.88323, 0.76736, 0.46193],
[0.88653, 0.76619, 0.46174],
[0.88982, 0.76501, 0.46159],
[0.89309, 0.76382, 0.46149],
[0.89636, 0.76263, 0.46143],
[0.8996, 0.76143, 0.46142],
[0.90283, 0.76023, 0.46145],
[0.90605, 0.75903, 0.46152],
[0.90924, 0.75782, 0.46164],
[0.91242, 0.7566, 0.4618],
[0.91559, 0.75539, 0.462],
[0.91874, 0.75416, 0.46225],
[0.92188, 0.75293, 0.46253],
[0.92499, 0.75171, 0.46286],
[0.92809, 0.75047, 0.46324],
[0.93118, 0.74922, 0.46366],
[0.93424, 0.74799, 0.46413],
[0.9373, 0.74674, 0.46462],
[0.94033, 0.74549, 0.46515],
[0.94335, 0.74424, 0.46573],
[0.94635, 0.74299, 0.46635],
[0.94933, 0.74172, 0.46702],
[0.95229, 0.74046, 0.46771],
[0.95524, 0.7392, 0.46844],
[0.95817, 0.73792, 0.46923],
[0.96108, 0.73666, 0.47004],
[0.96398, 0.73539, 0.47088],
[0.96685, 0.73411, 0.47177],
[0.96971, 0.73284, 0.47268],
[0.97256, 0.73155, 0.47363],
[0.97537, 0.73027, 0.47464],
[0.97818, 0.72899, 0.47566],
[0.98097, 0.7277, 0.47671],
[0.98374, 0.72642, 0.47781],
[0.98649, 0.72513, 0.47893],
[0.98923, 0.72383, 0.4801],
[0.99194, 0.72254, 0.48128],
[0.99464, 0.72125, 0.4825],
[0.99732, 0.71995, 0.48375],
[0.99999, 0.71865, 0.48503],
[1, 0.71736, 0.48634],
[1, 0.71606, 0.48769],
[1, 0.71476, 0.48905],
[1, 0.71346, 0.49046],
[1, 0.71215, 0.49188],
[1, 0.71085, 0.49332],
[1, 0.70955, 0.49481],
[1, 0.70824, 0.4963],
[1, 0.70694, 0.49785],
[1, 0.70563, 0.49939],
[1, 0.70432, 0.50098],
[1, 0.70301, 0.50257],
[1, 0.70169, 0.50421],
[1, 0.70039, 0.50584],
[1, 0.69907, 0.50753],
]
b_isoluminant_cgo_80_c38 = bokeh_palette('isoluminant_cgo_80_c38',isoluminant_cgo_80_c38)
m_isoluminant_cgo_80_c38 = mpl_cm('isoluminant_cgo_80_c38',isoluminant_cgo_80_c38)
m_isoluminant_cgo_80_c38_r = mpl_cm('isoluminant_cgo_80_c38_r',list(reversed(isoluminant_cgo_80_c38)))
isolum = b_isoluminant_cgo_80_c38
m_isolum = m_isoluminant_cgo_80_c38
m_isolum_r = m_isoluminant_cgo_80_c38_r
palette['isolum'] = b_isoluminant_cgo_80_c38
palette_n['isolum'] = b_isoluminant_cgo_80_c38
cm['isolum'] = m_isoluminant_cgo_80_c38
cm['isolum_r'] = m_isoluminant_cgo_80_c38_r
cm_n['isolum'] = mpl_cm('isolum',isoluminant_cgo_80_c38)
cm_n['isolum_r'] = mpl_cm('isolum_r',list(reversed(isoluminant_cgo_80_c38)))
CET_I2 = b_isoluminant_cgo_80_c38
m_CET_I2 = m_isoluminant_cgo_80_c38
m_CET_I2_r = m_isoluminant_cgo_80_c38_r
palette['CET_I2'] = b_isoluminant_cgo_80_c38
cm['CET_I2'] = m_isoluminant_cgo_80_c38
cm['CET_I2_r'] = m_isoluminant_cgo_80_c38_r
register_cmap('cet_CET_I2',m_isoluminant_cgo_80_c38)
register_cmap('cet_CET_I2_r',m_isoluminant_cgo_80_c38_r)




cyclic_grey_15_85_c0_s25 = [  # cmap_def
[0.17676, 0.17681, 0.1768],
[0.1775, 0.17755, 0.17754],
[0.17871, 0.17875, 0.17874],
[0.18037, 0.18041, 0.1804],
[0.1825, 0.18255, 0.18254],
[0.18509, 0.18514, 0.18512],
[0.18802, 0.18807, 0.18806],
[0.19124, 0.19129, 0.19128],
[0.19481, 0.19485, 0.19484],
[0.19857, 0.19862, 0.19861],
[0.20257, 0.20262, 0.20261],
[0.20676, 0.20681, 0.2068],
[0.21105, 0.2111, 0.21109],
[0.21547, 0.21552, 0.21551],
[0.21999, 0.22004, 0.22003],
[0.22453, 0.22458, 0.22457],
[0.22919, 0.22924, 0.22923],
[0.23386, 0.23392, 0.2339],
[0.23857, 0.23863, 0.23862],
[0.24332, 0.24337, 0.24336],
[0.24812, 0.24817, 0.24816],
[0.2529, 0.25295, 0.25294],
[0.25769, 0.25775, 0.25774],
[0.26252, 0.26258, 0.26257],
[0.26734, 0.2674, 0.26739],
[0.27219, 0.27226, 0.27224],
[0.27708, 0.27714, 0.27713],
[0.28196, 0.28203, 0.28201],
[0.28684, 0.2869, 0.28688],
[0.29176, 0.29183, 0.29181],
[0.29668, 0.29675, 0.29674],
[0.3016, 0.30167, 0.30165],
[0.30657, 0.30664, 0.30662],
[0.31152, 0.31159, 0.31157],
[0.31651, 0.31658, 0.31656],
[0.32149, 0.32156, 0.32154],
[0.32649, 0.32656, 0.32654],
[0.33151, 0.33158, 0.33156],
[0.33654, 0.33662, 0.3366],
[0.34159, 0.34166, 0.34165],
[0.34664, 0.34672, 0.3467],
[0.3517, 0.35178, 0.35176],
[0.3568, 0.35688, 0.35686],
[0.36188, 0.36196, 0.36194],
[0.36698, 0.36706, 0.36704],
[0.37208, 0.37216, 0.37215],
[0.37721, 0.37729, 0.37727],
[0.38233, 0.38241, 0.38239],
[0.38746, 0.38755, 0.38753],
[0.3926, 0.39269, 0.39267],
[0.39775, 0.39784, 0.39781],
[0.40288, 0.40297, 0.40295],
[0.40802, 0.4081, 0.40808],
[0.41315, 0.41324, 0.41322],
[0.41826, 0.41834, 0.41832],
[0.42336, 0.42345, 0.42343],
[0.42846, 0.42855, 0.42853],
[0.43353, 0.43362, 0.4336],
[0.43859, 0.43868, 0.43866],
[0.44365, 0.44374, 0.44372],
[0.44868, 0.44878, 0.44875],
[0.4537, 0.4538, 0.45377],
[0.45872, 0.45882, 0.4588],
[0.46374, 0.46385, 0.46382],
[0.46877, 0.46888, 0.46885],
[0.47381, 0.47391, 0.47389],
[0.47887, 0.47897, 0.47895],
[0.48395, 0.48405, 0.48403],
[0.48907, 0.48917, 0.48914],
[0.49422, 0.49432, 0.4943],
[0.49939, 0.49949, 0.49947],
[0.50462, 0.50472, 0.5047],
[0.50986, 0.50997, 0.50994],
[0.51515, 0.51526, 0.51523],
[0.52047, 0.52058, 0.52056],
[0.52583, 0.52594, 0.52592],
[0.53121, 0.53132, 0.5313],
[0.53663, 0.53674, 0.53671],
[0.54205, 0.54217, 0.54214],
[0.5475, 0.54762, 0.54759],
[0.55297, 0.55309, 0.55306],
[0.55846, 0.55857, 0.55855],
[0.56397, 0.56409, 0.56406],
[0.56948, 0.5696, 0.56957],
[0.57501, 0.57513, 0.5751],
[0.58055, 0.58067, 0.58064],
[0.58611, 0.58623, 0.5862],
[0.59167, 0.59179, 0.59176],
[0.59724, 0.59737, 0.59734],
[0.60282, 0.60295, 0.60292],
[0.60842, 0.60855, 0.60852],
[0.61401, 0.61414, 0.6141],
[0.61963, 0.61976, 0.61973],
[0.62524, 0.62537, 0.62534],
[0.63088, 0.63101, 0.63098],
[0.63652, 0.63665, 0.63662],
[0.64216, 0.64229, 0.64226],
[0.64781, 0.64795, 0.64791],
[0.65348, 0.65361, 0.65358],
[0.65915, 0.65929, 0.65926],
[0.66484, 0.66497, 0.66494],
[0.67054, 0.67067, 0.67064],
[0.67624, 0.67637, 0.67634],
[0.68194, 0.68208, 0.68205],
[0.68766, 0.68781, 0.68777],
[0.69339, 0.69353, 0.6935],
[0.69913, 0.69927, 0.69923],
[0.70487, 0.70502, 0.70498],
[0.71062, 0.71076, 0.71073],
[0.71636, 0.71651, 0.71647],
[0.72208, 0.72223, 0.72219],
[0.72779, 0.72794, 0.72791],
[0.73347, 0.73362, 0.73359],
[0.73911, 0.73926, 0.73922],
[0.74467, 0.74482, 0.74478],
[0.75014, 0.75029, 0.75026],
[0.75549, 0.75565, 0.75561],
[0.76068, 0.76083, 0.7608],
[0.76567, 0.76582, 0.76578],
[0.77039, 0.77055, 0.77051],
[0.77481, 0.77497, 0.77493],
[0.77887, 0.77903, 0.77899],
[0.7825, 0.78266, 0.78262],
[0.78566, 0.78581, 0.78578],
[0.78827, 0.78843, 0.78839],
[0.79029, 0.79045, 0.79041],
[0.79169, 0.79185, 0.79182],
[0.79244, 0.7926, 0.79257],
[0.79252, 0.79269, 0.79265],
[0.79193, 0.79209, 0.79205],
[0.79069, 0.79085, 0.79081],
[0.78881, 0.78897, 0.78893],
[0.78633, 0.78649, 0.78645],
[0.7833, 0.78346, 0.78342],
[0.77978, 0.77994, 0.7799],
[0.77582, 0.77598, 0.77594],
[0.77149, 0.77164, 0.7716],
[0.76682, 0.76698, 0.76694],
[0.76189, 0.76205, 0.76201],
[0.75675, 0.75691, 0.75687],
[0.75145, 0.7516, 0.75156],
[0.746, 0.74615, 0.74611],
[0.74045, 0.7406, 0.74056],
[0.73484, 0.73499, 0.73495],
[0.72917, 0.72932, 0.72928],
[0.72346, 0.72361, 0.72357],
[0.71774, 0.71789, 0.71785],
[0.712, 0.71215, 0.71211],
[0.70626, 0.70641, 0.70637],
[0.70052, 0.70066, 0.70062],
[0.69477, 0.69492, 0.69488],
[0.68905, 0.68919, 0.68916],
[0.68333, 0.68346, 0.68343],
[0.67762, 0.67776, 0.67772],
[0.67191, 0.67205, 0.67201],
[0.66621, 0.66635, 0.66631],
[0.66052, 0.66066, 0.66063],
[0.65485, 0.65498, 0.65495],
[0.64919, 0.64932, 0.64929],
[0.64352, 0.64365, 0.64362],
[0.63787, 0.63801, 0.63797],
[0.63223, 0.63236, 0.63233],
[0.6266, 0.62673, 0.6267],
[0.62098, 0.62111, 0.62108],
[0.61536, 0.61549, 0.61546],
[0.60976, 0.60989, 0.60986],
[0.60418, 0.6043, 0.60427],
[0.59858, 0.59871, 0.59868],
[0.59302, 0.59314, 0.59311],
[0.58744, 0.58756, 0.58753],
[0.58189, 0.58201, 0.58198],
[0.57635, 0.57647, 0.57644],
[0.57081, 0.57092, 0.5709],
[0.56528, 0.56539, 0.56537],
[0.55977, 0.55989, 0.55986],
[0.55426, 0.55438, 0.55435],
[0.54877, 0.54888, 0.54886],
[0.54328, 0.54339, 0.54336],
[0.53781, 0.53792, 0.53789],
[0.53234, 0.53245, 0.53242],
[0.52688, 0.52699, 0.52697],
[0.52143, 0.52154, 0.52151],
[0.51599, 0.5161, 0.51607],
[0.51057, 0.51067, 0.51065],
[0.50515, 0.50526, 0.50523],
[0.49974, 0.49985, 0.49982],
[0.49436, 0.49446, 0.49444],
[0.48897, 0.48907, 0.48905],
[0.48359, 0.4837, 0.48367],
[0.47824, 0.47834, 0.47831],
[0.47288, 0.47298, 0.47295],
[0.46755, 0.46764, 0.46762],
[0.46221, 0.46231, 0.46229],
[0.45689, 0.45699, 0.45697],
[0.45159, 0.45169, 0.45166],
[0.44629, 0.44638, 0.44636],
[0.441, 0.44109, 0.44107],
[0.43574, 0.43583, 0.43581],
[0.43047, 0.43056, 0.43054],
[0.42522, 0.42531, 0.42529],
[0.41998, 0.42007, 0.42004],
[0.41475, 0.41483, 0.41481],
[0.40953, 0.40962, 0.4096],
[0.40432, 0.40441, 0.40439],
[0.39912, 0.39921, 0.39919],
[0.39395, 0.39403, 0.39401],
[0.38878, 0.38887, 0.38885],
[0.38363, 0.38371, 0.38369],
[0.37848, 0.37856, 0.37854],
[0.37334, 0.37342, 0.3734],
[0.36822, 0.36831, 0.36829],
[0.36311, 0.36319, 0.36317],
[0.35802, 0.3581, 0.35808],
[0.35294, 0.35302, 0.353],
[0.34787, 0.34794, 0.34793],
[0.34282, 0.34289, 0.34287],
[0.33776, 0.33783, 0.33781],
[0.33274, 0.33281, 0.3328],
[0.3277, 0.32778, 0.32776],
[0.32271, 0.32278, 0.32276],
[0.31771, 0.31778, 0.31776],
[0.31272, 0.31279, 0.31277],
[0.30775, 0.30782, 0.3078],
[0.3028, 0.30286, 0.30285],
[0.29786, 0.29793, 0.29791],
[0.29294, 0.293, 0.29299],
[0.28802, 0.28809, 0.28807],
[0.28314, 0.2832, 0.28319],
[0.27826, 0.27832, 0.2783],
[0.27339, 0.27345, 0.27344],
[0.26853, 0.2686, 0.26858],
[0.2637, 0.26376, 0.26375],
[0.25887, 0.25893, 0.25892],
[0.25406, 0.25412, 0.2541],
[0.24925, 0.24931, 0.2493],
[0.24446, 0.24452, 0.24451],
[0.23973, 0.23978, 0.23977],
[0.23502, 0.23508, 0.23506],
[0.2303, 0.23036, 0.23034],
[0.22568, 0.22573, 0.22572],
[0.22109, 0.22114, 0.22113],
[0.21656, 0.21662, 0.2166],
[0.21212, 0.21217, 0.21216],
[0.20777, 0.20782, 0.20781],
[0.20358, 0.20363, 0.20361],
[0.19951, 0.19955, 0.19954],
[0.19568, 0.19573, 0.19572],
[0.19207, 0.19212, 0.19211],
[0.18876, 0.1888, 0.18879],
[0.18577, 0.18581, 0.1858],
[0.18308, 0.18313, 0.18312],
[0.18084, 0.18089, 0.18087],
[0.17906, 0.1791, 0.17909],
[0.17775, 0.1778, 0.17779],
[0.17689, 0.17694, 0.17693],
[0.17656, 0.1766, 0.17659],
]
b_cyclic_grey_15_85_c0_s25 = bokeh_palette('cyclic_grey_15_85_c0_s25',cyclic_grey_15_85_c0_s25)
m_cyclic_grey_15_85_c0_s25 = mpl_cm('cyclic_grey_15_85_c0_s25',cyclic_grey_15_85_c0_s25)
m_cyclic_grey_15_85_c0_s25_r = mpl_cm('cyclic_grey_15_85_c0_s25_r',list(reversed(cyclic_grey_15_85_c0_s25)))
CET_C5s = b_cyclic_grey_15_85_c0_s25
m_CET_C5s = m_cyclic_grey_15_85_c0_s25
m_CET_C5s_r = m_cyclic_grey_15_85_c0_s25_r
palette['CET_C5s'] = b_cyclic_grey_15_85_c0_s25
cm['CET_C5s'] = m_cyclic_grey_15_85_c0_s25
cm['CET_C5s_r'] = m_cyclic_grey_15_85_c0_s25_r
register_cmap('cet_CET_C5s',m_cyclic_grey_15_85_c0_s25)
register_cmap('cet_CET_C5s_r',m_cyclic_grey_15_85_c0_s25_r)




diverging_cwm_80_100_c22 = [  # cmap_def
[0, 0.85071, 1],
[0, 0.85194, 1],
[0.056211, 0.85316, 1],
[0.10029, 0.85438, 1],
[0.13168, 0.8556, 1],
[0.15712, 0.85682, 1],
[0.17902, 0.85804, 1],
[0.19846, 0.85925, 1],
[0.21615, 0.86047, 1],
[0.23241, 0.86169, 1],
[0.24758, 0.8629, 1],
[0.26183, 0.86412, 1],
[0.27532, 0.86533, 1],
[0.28813, 0.86654, 1],
[0.30039, 0.86775, 1],
[0.31215, 0.86897, 1],
[0.32349, 0.87017, 1],
[0.33443, 0.87138, 1],
[0.34499, 0.87259, 1],
[0.35526, 0.87379, 1],
[0.36521, 0.875, 1],
[0.37491, 0.87621, 1],
[0.38437, 0.87741, 1],
[0.3936, 0.87861, 1],
[0.40262, 0.87982, 1],
[0.41146, 0.88102, 1],
[0.42011, 0.88222, 1],
[0.4286, 0.88342, 1],
[0.43693, 0.88462, 1],
[0.44511, 0.88582, 1],
[0.45315, 0.88702, 1],
[0.46106, 0.88821, 1],
[0.46885, 0.88941, 1],
[0.47652, 0.8906, 1],
[0.48407, 0.8918, 1],
[0.49154, 0.89299, 1],
[0.49889, 0.89419, 1],
[0.50614, 0.89538, 1],
[0.51332, 0.89657, 1],
[0.5204, 0.89776, 1],
[0.5274, 0.89895, 1],
[0.53432, 0.90014, 1],
[0.54117, 0.90132, 1],
[0.54795, 0.90251, 1],
[0.55466, 0.9037, 1],
[0.56129, 0.90488, 1],
[0.56787, 0.90607, 1],
[0.57438, 0.90725, 1],
[0.58084, 0.90844, 1],
[0.58724, 0.90962, 1],
[0.59358, 0.9108, 1],
[0.59988, 0.91198, 1],
[0.60611, 0.91316, 1],
[0.6123, 0.91434, 1],
[0.61845, 0.91551, 1],
[0.62455, 0.91669, 1],
[0.6306, 0.91787, 1],
[0.63661, 0.91904, 1],
[0.64258, 0.92021, 1],
[0.6485, 0.92139, 1],
[0.65439, 0.92256, 1],
[0.66024, 0.92373, 1],
[0.66606, 0.9249, 1],
[0.67183, 0.92607, 1],
[0.67758, 0.92724, 1],
[0.68329, 0.92841, 1],
[0.68897, 0.92958, 1],
[0.69461, 0.93075, 1],
[0.70023, 0.93191, 1],
[0.70581, 0.93307, 1],
[0.71137, 0.93424, 1],
[0.7169, 0.9354, 1],
[0.72239, 0.93657, 1],
[0.72786, 0.93773, 1],
[0.73331, 0.93889, 1],
[0.73873, 0.94005, 1],
[0.74412, 0.94121, 1],
[0.74949, 0.94237, 1],
[0.75485, 0.94352, 1],
[0.76017, 0.94468, 1],
[0.76547, 0.94583, 1],
[0.77075, 0.94699, 1],
[0.776, 0.94814, 1],
[0.78124, 0.9493, 1],
[0.78646, 0.95045, 1],
[0.79165, 0.9516, 1],
[0.79683, 0.95275, 1],
[0.80199, 0.9539, 1],
[0.80713, 0.95505, 1],
[0.81224, 0.9562, 1],
[0.81735, 0.95735, 1],
[0.82244, 0.9585, 1],
[0.8275, 0.95964, 1],
[0.83255, 0.96079, 1],
[0.83759, 0.96192, 1],
[0.84261, 0.96307, 1],
[0.8476, 0.96421, 1],
[0.8526, 0.96535, 1],
[0.85756, 0.9665, 1],
[0.86253, 0.96764, 1],
[0.86746, 0.96878, 1],
[0.8724, 0.96991, 1],
[0.87731, 0.97105, 1],
[0.88221, 0.97219, 1],
[0.8871, 0.97332, 1],
[0.89197, 0.97446, 1],
[0.89684, 0.97559, 1],
[0.90168, 0.97672, 1],
[0.90652, 0.97786, 1],
[0.91134, 0.97899, 1],
[0.91615, 0.98012, 1],
[0.92095, 0.98125, 1],
[0.92575, 0.98238, 1],
[0.93053, 0.98351, 1],
[0.93529, 0.98464, 1],
[0.94005, 0.98576, 1],
[0.9448, 0.98689, 1],
[0.94953, 0.98801, 1],
[0.95426, 0.98914, 1],
[0.95897, 0.99026, 1],
[0.96367, 0.99138, 1],
[0.96837, 0.99251, 1],
[0.97306, 0.99363, 1],
[0.97773, 0.99474, 1],
[0.9824, 0.99586, 1],
[0.98725, 0.99702, 1],
[0.9912, 0.99798, 1],
[0.99685, 0.99934, 1],
[0.99988, 0.9982, 0.99969],
[0.99991, 0.99575, 0.99926],
[0.99993, 0.9933, 0.99882],
[0.99994, 0.99084, 0.99839],
[0.99996, 0.98838, 0.99796],
[0.99997, 0.98593, 0.99752],
[0.99998, 0.98347, 0.99709],
[0.99998, 0.98102, 0.99665],
[0.99998, 0.97856, 0.99622],
[0.99998, 0.97611, 0.99578],
[0.99998, 0.97365, 0.99535],
[0.99997, 0.97119, 0.99491],
[0.99996, 0.96874, 0.99448],
[0.99995, 0.96628, 0.99404],
[0.99993, 0.96382, 0.99361],
[0.99991, 0.96136, 0.99317],
[0.99989, 0.9589, 0.99273],
[0.99986, 0.95644, 0.99229],
[0.99984, 0.95398, 0.99185],
[0.9998, 0.95152, 0.99141],
[0.99977, 0.94906, 0.99097],
[0.99973, 0.9466, 0.99053],
[0.99969, 0.94414, 0.99009],
[0.99965, 0.94167, 0.98965],
[0.9996, 0.93921, 0.98921],
[0.99955, 0.93675, 0.98877],
[0.9995, 0.93428, 0.98832],
[0.99945, 0.93182, 0.98788],
[0.99939, 0.92935, 0.98744],
[0.99933, 0.92689, 0.987],
[0.99926, 0.92442, 0.98656],
[0.9992, 0.92196, 0.98611],
[0.99913, 0.91949, 0.98567],
[0.99905, 0.91702, 0.98522],
[0.99898, 0.91455, 0.98478],
[0.9989, 0.91208, 0.98433],
[0.99882, 0.90961, 0.98388],
[0.99873, 0.90714, 0.98343],
[0.99865, 0.90467, 0.98299],
[0.99855, 0.9022, 0.98254],
[0.99846, 0.89973, 0.9821],
[0.99837, 0.89726, 0.98165],
[0.99827, 0.89478, 0.9812],
[0.99816, 0.89231, 0.98075],
[0.99806, 0.88983, 0.9803],
[0.99795, 0.88735, 0.97985],
[0.99784, 0.88488, 0.9794],
[0.99773, 0.8824, 0.97894],
[0.99761, 0.87992, 0.97849],
[0.99749, 0.87744, 0.97804],
[0.99737, 0.87496, 0.97759],
[0.99725, 0.87248, 0.97714],
[0.99712, 0.87, 0.97668],
[0.99699, 0.86752, 0.97623],
[0.99686, 0.86503, 0.97578],
[0.99672, 0.86256, 0.97532],
[0.99658, 0.86007, 0.97487],
[0.99644, 0.85758, 0.97441],
[0.9963, 0.8551, 0.97396],
[0.99615, 0.85261, 0.9735],
[0.996, 0.85012, 0.97305],
[0.99585, 0.84763, 0.97259],
[0.9957, 0.84514, 0.97214],
[0.99554, 0.84265, 0.97168],
[0.99538, 0.84016, 0.97122],
[0.99522, 0.83767, 0.97076],
[0.99505, 0.83517, 0.9703],
[0.99488, 0.83267, 0.96984],
[0.99471, 0.83018, 0.96938],
[0.99454, 0.82768, 0.96892],
[0.99436, 0.82518, 0.96846],
[0.99418, 0.82268, 0.96801],
[0.994, 0.82018, 0.96754],
[0.99382, 0.81768, 0.96708],
[0.99363, 0.81518, 0.96662],
[0.99344, 0.81267, 0.96615],
[0.99325, 0.81017, 0.96569],
[0.99305, 0.80766, 0.96523],
[0.99285, 0.80516, 0.96476],
[0.99265, 0.80265, 0.9643],
[0.99245, 0.80014, 0.96384],
[0.99224, 0.79763, 0.96337],
[0.99203, 0.79512, 0.96291],
[0.99182, 0.7926, 0.96244],
[0.99161, 0.79009, 0.96197],
[0.99139, 0.78757, 0.96151],
[0.99117, 0.78506, 0.96105],
[0.99095, 0.78254, 0.96058],
[0.99072, 0.78001, 0.96011],
[0.9905, 0.7775, 0.95964],
[0.99027, 0.77497, 0.95917],
[0.99003, 0.77245, 0.95871],
[0.9898, 0.76992, 0.95824],
[0.98956, 0.76739, 0.95777],
[0.98932, 0.76487, 0.9573],
[0.98908, 0.76233, 0.95683],
[0.98883, 0.7598, 0.95636],
[0.98858, 0.75727, 0.95589],
[0.98833, 0.75474, 0.95542],
[0.98808, 0.7522, 0.95494],
[0.98782, 0.74966, 0.95447],
[0.98756, 0.74712, 0.954],
[0.9873, 0.74458, 0.95353],
[0.98704, 0.74204, 0.95305],
[0.98677, 0.7395, 0.95258],
[0.9865, 0.73694, 0.9521],
[0.98623, 0.7344, 0.95163],
[0.98596, 0.73185, 0.95115],
[0.98568, 0.72929, 0.95068],
[0.9854, 0.72674, 0.9502],
[0.98512, 0.72418, 0.94972],
[0.98484, 0.72162, 0.94925],
[0.98455, 0.71907, 0.94877],
[0.98426, 0.71651, 0.94829],
[0.98397, 0.71394, 0.94782],
[0.98367, 0.71138, 0.94734],
[0.98338, 0.70881, 0.94686],
[0.98308, 0.70624, 0.94638],
[0.98278, 0.70366, 0.9459],
[0.98247, 0.70109, 0.94542],
[0.98217, 0.69851, 0.94494],
[0.98186, 0.69594, 0.94446],
[0.98154, 0.69335, 0.94398],
[0.98123, 0.69077, 0.9435],
[0.98091, 0.68819, 0.94302],
[0.98059, 0.6856, 0.94253],
[0.98027, 0.68301, 0.94205],
[0.97995, 0.68041, 0.94156],
]
b_diverging_cwm_80_100_c22 = bokeh_palette('diverging_cwm_80_100_c22',diverging_cwm_80_100_c22)
m_diverging_cwm_80_100_c22 = mpl_cm('diverging_cwm_80_100_c22',diverging_cwm_80_100_c22)
m_diverging_cwm_80_100_c22_r = mpl_cm('diverging_cwm_80_100_c22_r',list(reversed(diverging_cwm_80_100_c22)))
CET_D10 = b_diverging_cwm_80_100_c22
m_CET_D10 = m_diverging_cwm_80_100_c22
m_CET_D10_r = m_diverging_cwm_80_100_c22_r
palette['CET_D10'] = b_diverging_cwm_80_100_c22
cm['CET_D10'] = m_diverging_cwm_80_100_c22
cm['CET_D10_r'] = m_diverging_cwm_80_100_c22_r
register_cmap('cet_CET_D10',m_diverging_cwm_80_100_c22)
register_cmap('cet_CET_D10_r',m_diverging_cwm_80_100_c22_r)




linear_gow_60_85_c27 = [  # cmap_def
[0.39872, 0.60624, 0.56451],
[0.40156, 0.60721, 0.56266],
[0.4044, 0.60816, 0.5608],
[0.40724, 0.60911, 0.55895],
[0.41009, 0.61005, 0.55712],
[0.41293, 0.61099, 0.55528],
[0.41577, 0.61192, 0.55344],
[0.41863, 0.61285, 0.55162],
[0.42148, 0.61378, 0.5498],
[0.42435, 0.6147, 0.54798],
[0.42723, 0.61561, 0.54618],
[0.4301, 0.61653, 0.54436],
[0.43298, 0.61743, 0.54256],
[0.43588, 0.61833, 0.54076],
[0.43876, 0.61923, 0.53897],
[0.44169, 0.62012, 0.53719],
[0.44461, 0.621, 0.5354],
[0.44752, 0.62186, 0.53362],
[0.45047, 0.62274, 0.53186],
[0.45342, 0.62361, 0.5301],
[0.45639, 0.62446, 0.52835],
[0.45937, 0.62531, 0.5266],
[0.46235, 0.62615, 0.52485],
[0.46536, 0.62698, 0.52312],
[0.46838, 0.62781, 0.52138],
[0.47142, 0.62864, 0.51966],
[0.47447, 0.62944, 0.51795],
[0.47754, 0.63025, 0.51623],
[0.48061, 0.63105, 0.51454],
[0.48372, 0.63184, 0.51285],
[0.48685, 0.63262, 0.51116],
[0.48999, 0.63339, 0.50949],
[0.49314, 0.63416, 0.50783],
[0.49633, 0.63491, 0.50616],
[0.49954, 0.63565, 0.50452],
[0.50277, 0.63639, 0.50287],
[0.50602, 0.63711, 0.50125],
[0.50931, 0.63782, 0.49961],
[0.51261, 0.63852, 0.49802],
[0.51594, 0.63922, 0.4964],
[0.5193, 0.6399, 0.49483],
[0.52268, 0.64057, 0.49323],
[0.5261, 0.64122, 0.49168],
[0.52955, 0.64187, 0.49012],
[0.53302, 0.6425, 0.48858],
[0.53654, 0.64312, 0.48705],
[0.54007, 0.64372, 0.48552],
[0.54365, 0.64431, 0.48402],
[0.54726, 0.64489, 0.48253],
[0.55092, 0.64545, 0.48105],
[0.55461, 0.64599, 0.4796],
[0.55833, 0.64652, 0.47815],
[0.56212, 0.64704, 0.47672],
[0.56592, 0.64753, 0.47532],
[0.56978, 0.648, 0.47391],
[0.57369, 0.64846, 0.47253],
[0.57764, 0.6489, 0.47118],
[0.58164, 0.64932, 0.46984],
[0.5857, 0.64971, 0.46851],
[0.5898, 0.65009, 0.46722],
[0.59397, 0.65044, 0.46593],
[0.59819, 0.65076, 0.46469],
[0.60248, 0.65106, 0.46344],
[0.60683, 0.65134, 0.46224],
[0.61124, 0.65159, 0.46106],
[0.61572, 0.6518, 0.45991],
[0.62028, 0.65199, 0.45879],
[0.62491, 0.65214, 0.45769],
[0.62961, 0.65226, 0.45663],
[0.63441, 0.65235, 0.4556],
[0.63928, 0.6524, 0.45461],
[0.64424, 0.65241, 0.45364],
[0.6493, 0.65237, 0.45273],
[0.65445, 0.6523, 0.45184],
[0.65971, 0.65218, 0.451],
[0.66508, 0.65201, 0.45021],
[0.67058, 0.65178, 0.44946],
[0.67612, 0.65152, 0.44876],
[0.68161, 0.65126, 0.44805],
[0.68699, 0.65102, 0.44736],
[0.69226, 0.65081, 0.44667],
[0.69741, 0.65061, 0.44598],
[0.70245, 0.65044, 0.44529],
[0.70738, 0.65029, 0.44461],
[0.71221, 0.65017, 0.44392],
[0.71694, 0.65007, 0.44324],
[0.72155, 0.65, 0.44256],
[0.72607, 0.64995, 0.44187],
[0.73048, 0.64993, 0.44118],
[0.7348, 0.64994, 0.4405],
[0.73901, 0.64997, 0.43982],
[0.74312, 0.65003, 0.43914],
[0.74713, 0.65011, 0.43846],
[0.75106, 0.65023, 0.43779],
[0.75488, 0.65037, 0.43711],
[0.7586, 0.65054, 0.43643],
[0.76223, 0.65074, 0.43576],
[0.76578, 0.65097, 0.43508],
[0.76922, 0.65123, 0.4344],
[0.77257, 0.65152, 0.43372],
[0.77583, 0.65185, 0.43305],
[0.779, 0.65219, 0.43239],
[0.78209, 0.65257, 0.43171],
[0.78507, 0.65298, 0.43103],
[0.78797, 0.65343, 0.43036],
[0.79078, 0.65391, 0.42968],
[0.7935, 0.65441, 0.42901],
[0.79612, 0.65495, 0.42833],
[0.79867, 0.65553, 0.42766],
[0.80112, 0.65614, 0.42699],
[0.80349, 0.65678, 0.4263],
[0.80576, 0.65745, 0.42564],
[0.80795, 0.65816, 0.42497],
[0.81006, 0.65889, 0.42428],
[0.81207, 0.65966, 0.42361],
[0.814, 0.66047, 0.42293],
[0.81584, 0.66131, 0.42225],
[0.8176, 0.66219, 0.42158],
[0.81927, 0.66309, 0.4209],
[0.82085, 0.66403, 0.42023],
[0.82235, 0.665, 0.41956],
[0.82376, 0.66601, 0.41887],
[0.82508, 0.66705, 0.41819],
[0.82632, 0.66813, 0.41752],
[0.82747, 0.66924, 0.41684],
[0.82853, 0.67039, 0.41616],
[0.82952, 0.67156, 0.41547],
[0.83041, 0.67278, 0.41479],
[0.83123, 0.67401, 0.41414],
[0.83202, 0.67526, 0.41355],
[0.83279, 0.67651, 0.41307],
[0.83355, 0.67776, 0.41268],
[0.83429, 0.67901, 0.41239],
[0.83501, 0.68025, 0.41219],
[0.83572, 0.68151, 0.41208],
[0.83641, 0.68277, 0.41209],
[0.83708, 0.68402, 0.41219],
[0.83774, 0.68528, 0.41239],
[0.83838, 0.68653, 0.41269],
[0.83901, 0.68779, 0.41309],
[0.83962, 0.68906, 0.41358],
[0.84021, 0.69031, 0.41417],
[0.84079, 0.69157, 0.41487],
[0.84135, 0.69284, 0.41567],
[0.84189, 0.6941, 0.41657],
[0.84242, 0.69537, 0.41757],
[0.84293, 0.69663, 0.41865],
[0.84342, 0.6979, 0.41986],
[0.8439, 0.69916, 0.42115],
[0.84436, 0.70044, 0.42254],
[0.8448, 0.7017, 0.42404],
[0.84523, 0.70298, 0.42564],
[0.84563, 0.70425, 0.42735],
[0.84602, 0.70552, 0.42913],
[0.84639, 0.7068, 0.43103],
[0.84674, 0.70808, 0.43303],
[0.84707, 0.70936, 0.43513],
[0.84739, 0.71063, 0.43732],
[0.84768, 0.71191, 0.4396],
[0.84796, 0.71319, 0.44201],
[0.84821, 0.71447, 0.4445],
[0.84845, 0.71577, 0.44707],
[0.84866, 0.71705, 0.44976],
[0.84885, 0.71833, 0.45256],
[0.84903, 0.71963, 0.45543],
[0.84918, 0.72091, 0.45841],
[0.8493, 0.7222, 0.46149],
[0.84941, 0.7235, 0.46467],
[0.84949, 0.7248, 0.46793],
[0.84954, 0.72609, 0.4713],
[0.84958, 0.72739, 0.47477],
[0.84958, 0.72869, 0.47832],
[0.84956, 0.72999, 0.48199],
[0.84952, 0.73129, 0.48573],
[0.84945, 0.7326, 0.48957],
[0.84935, 0.7339, 0.49352],
[0.84922, 0.73521, 0.49757],
[0.84907, 0.73652, 0.5017],
[0.84889, 0.73783, 0.50591],
[0.84868, 0.73916, 0.51022],
[0.84848, 0.74046, 0.51454],
[0.84828, 0.74176, 0.51885],
[0.8481, 0.74306, 0.52316],
[0.84792, 0.74435, 0.52745],
[0.84775, 0.74564, 0.53175],
[0.84759, 0.74692, 0.53603],
[0.84743, 0.74821, 0.54032],
[0.84727, 0.74947, 0.5446],
[0.84712, 0.75075, 0.54888],
[0.84698, 0.75202, 0.55314],
[0.84684, 0.75328, 0.55742],
[0.8467, 0.75455, 0.56168],
[0.84656, 0.7558, 0.56594],
[0.84643, 0.75706, 0.57019],
[0.84629, 0.75831, 0.57444],
[0.84616, 0.75955, 0.5787],
[0.84603, 0.76081, 0.58295],
[0.8459, 0.76205, 0.58719],
[0.84577, 0.76329, 0.59144],
[0.84564, 0.76453, 0.59568],
[0.84551, 0.76577, 0.59993],
[0.84537, 0.767, 0.60417],
[0.84524, 0.76824, 0.6084],
[0.84511, 0.76947, 0.61263],
[0.84497, 0.77071, 0.61687],
[0.84484, 0.77194, 0.6211],
[0.8447, 0.77316, 0.62533],
[0.84455, 0.77439, 0.62955],
[0.84441, 0.77561, 0.63379],
[0.84426, 0.77684, 0.63801],
[0.84411, 0.77806, 0.64224],
[0.84396, 0.77928, 0.64646],
[0.8438, 0.7805, 0.65068],
[0.84364, 0.78172, 0.6549],
[0.84348, 0.78294, 0.65912],
[0.84331, 0.78416, 0.66335],
[0.84314, 0.78537, 0.66757],
[0.84297, 0.78659, 0.67179],
[0.84279, 0.7878, 0.67601],
[0.84261, 0.78901, 0.68022],
[0.84243, 0.79022, 0.68444],
[0.84224, 0.79143, 0.68867],
[0.84204, 0.79265, 0.69289],
[0.84184, 0.79386, 0.6971],
[0.84163, 0.79507, 0.70132],
[0.84142, 0.79627, 0.70554],
[0.8412, 0.79748, 0.70976],
[0.84098, 0.79869, 0.71398],
[0.84076, 0.7999, 0.7182],
[0.84053, 0.8011, 0.72241],
[0.84029, 0.80231, 0.72663],
[0.84004, 0.80352, 0.73085],
[0.8398, 0.80472, 0.73508],
[0.83954, 0.80593, 0.7393],
[0.83928, 0.80713, 0.74352],
[0.83901, 0.80834, 0.74774],
[0.83873, 0.80954, 0.75196],
[0.83845, 0.81074, 0.75618],
[0.83817, 0.81195, 0.76041],
[0.83788, 0.81315, 0.76463],
[0.83758, 0.81435, 0.76885],
[0.83727, 0.81555, 0.77308],
[0.83695, 0.81676, 0.77731],
[0.83663, 0.81795, 0.78154],
[0.8363, 0.81916, 0.78577],
[0.83597, 0.82036, 0.78999],
[0.83563, 0.82156, 0.79422],
[0.83528, 0.82277, 0.79845],
[0.83492, 0.82396, 0.80268],
[0.83456, 0.82516, 0.80692],
[0.83419, 0.82637, 0.81115],
[0.83381, 0.82757, 0.81538],
[0.83342, 0.82877, 0.81962],
[0.83302, 0.82997, 0.82386],
[0.83262, 0.83118, 0.82809],
[0.83221, 0.83237, 0.83233],
]
b_linear_gow_60_85_c27 = bokeh_palette('linear_gow_60_85_c27',linear_gow_60_85_c27)
m_linear_gow_60_85_c27 = mpl_cm('linear_gow_60_85_c27',linear_gow_60_85_c27)
m_linear_gow_60_85_c27_r = mpl_cm('linear_gow_60_85_c27_r',list(reversed(linear_gow_60_85_c27)))
CET_L10 = b_linear_gow_60_85_c27
m_CET_L10 = m_linear_gow_60_85_c27
m_CET_L10_r = m_linear_gow_60_85_c27_r
palette['CET_L10'] = b_linear_gow_60_85_c27
cm['CET_L10'] = m_linear_gow_60_85_c27
cm['CET_L10_r'] = m_linear_gow_60_85_c27_r
register_cmap('cet_CET_L10',m_linear_gow_60_85_c27)
register_cmap('cet_CET_L10_r',m_linear_gow_60_85_c27_r)




linear_bmy_10_95_c78 = [  # cmap_def
[0, 0.047803, 0.4883],
[0, 0.049705, 0.49553],
[0, 0.051451, 0.50272],
[0, 0.053214, 0.50983],
[0, 0.054977, 0.51684],
[0, 0.056756, 0.52379],
[0, 0.058368, 0.53062],
[0, 0.060099, 0.53736],
[0, 0.061618, 0.54402],
[0, 0.063279, 0.55055],
[0, 0.06478, 0.55698],
[0, 0.066288, 0.56332],
[0, 0.067769, 0.5695],
[0, 0.069247, 0.57558],
[0, 0.070546, 0.58154],
[0, 0.071898, 0.58736],
[0, 0.073227, 0.59303],
[0, 0.074425, 0.59856],
[0, 0.075597, 0.60398],
[0, 0.076712, 0.60919],
[0, 0.077789, 0.61423],
[0, 0.078828, 0.61913],
[0, 0.079825, 0.62386],
[0, 0.080751, 0.62837],
[0, 0.081634, 0.63267],
[0, 0.082399, 0.63679],
[0, 0.083056, 0.64069],
[0, 0.083728, 0.64438],
[0, 0.084298, 0.64781],
[0, 0.084735, 0.65098],
[0, 0.085075, 0.65389],
[0, 0.085333, 0.65653],
[0, 0.085503, 0.65887],
[0, 0.085575, 0.66091],
[0, 0.085538, 0.66263],
[0, 0.085387, 0.664],
[0.041146, 0.085117, 0.66498],
[0.082348, 0.084719, 0.66559],
[0.11295, 0.084153, 0.66576],
[0.13887, 0.083348, 0.66549],
[0.16224, 0.08245, 0.66472],
[0.18389, 0.081312, 0.66345],
[0.20444, 0.079918, 0.6616],
[0.2241, 0.078308, 0.65916],
[0.24323, 0.076464, 0.65608],
[0.26189, 0.074387, 0.65234],
[0.27969, 0.072084, 0.64827],
[0.29634, 0.069818, 0.64421],
[0.31196, 0.067486, 0.64024],
[0.32675, 0.065157, 0.63634],
[0.34081, 0.062906, 0.63252],
[0.35421, 0.060546, 0.6288],
[0.36705, 0.058179, 0.62515],
[0.3794, 0.055931, 0.62158],
[0.39131, 0.05356, 0.6181],
[0.40281, 0.051267, 0.6147],
[0.41397, 0.049057, 0.6114],
[0.42479, 0.046684, 0.60817],
[0.43533, 0.044376, 0.60501],
[0.44558, 0.042108, 0.60193],
[0.4556, 0.039763, 0.59895],
[0.46537, 0.037486, 0.59604],
[0.47496, 0.035294, 0.59322],
[0.48432, 0.032963, 0.59047],
[0.49352, 0.030903, 0.5878],
[0.50254, 0.028926, 0.58523],
[0.5114, 0.027022, 0.58272],
[0.52012, 0.025191, 0.5803],
[0.52872, 0.02343, 0.57794],
[0.53717, 0.021744, 0.57569],
[0.54549, 0.020131, 0.57351],
[0.5537, 0.018589, 0.5714],
[0.56182, 0.017116, 0.56937],
[0.56982, 0.015713, 0.56743],
[0.57772, 0.014383, 0.56556],
[0.58556, 0.013108, 0.56378],
[0.59331, 0.011906, 0.56202],
[0.60102, 0.010448, 0.56026],
[0.60871, 0.0092728, 0.55849],
[0.61634, 0.0080608, 0.55673],
[0.62394, 0.0068636, 0.55496],
[0.6315, 0.0056715, 0.55317],
[0.63904, 0.0044851, 0.5514],
[0.64654, 0.0033027, 0.5496],
[0.65401, 0.0021237, 0.54781],
[0.66146, 0.00094944, 0.54601],
[0.66888, 0, 0.54419],
[0.67628, 0, 0.54238],
[0.68364, 0, 0.54055],
[0.69099, 0, 0.53873],
[0.69832, 0, 0.53689],
[0.70562, 0, 0.53504],
[0.71291, 0, 0.53319],
[0.72018, 0, 0.53134],
[0.72742, 0, 0.52948],
[0.73466, 0, 0.52761],
[0.74187, 0, 0.52573],
[0.74906, 0, 0.52385],
[0.75625, 0, 0.52195],
[0.76342, 0, 0.52005],
[0.77058, 0, 0.51816],
[0.77773, 0, 0.51624],
[0.78486, 0, 0.51433],
[0.79198, 0, 0.51239],
[0.79909, 0, 0.51046],
[0.80618, 0, 0.50853],
[0.81322, 0, 0.50656],
[0.82017, 0, 0.50459],
[0.82703, 0, 0.50258],
[0.83381, 0, 0.50056],
[0.84048, 0, 0.49853],
[0.84706, 0, 0.49645],
[0.85355, 0, 0.49437],
[0.85996, 0, 0.49225],
[0.86628, 0, 0.49011],
[0.8725, 0, 0.48796],
[0.87863, 0.0025964, 0.48577],
[0.88468, 0.0090591, 0.48356],
[0.89063, 0.016464, 0.48133],
[0.8965, 0.024452, 0.47908],
[0.90228, 0.033265, 0.47679],
[0.90796, 0.042948, 0.47449],
[0.91356, 0.052352, 0.47216],
[0.91907, 0.061335, 0.4698],
[0.92449, 0.070106, 0.46741],
[0.92982, 0.07857, 0.46499],
[0.93506, 0.086993, 0.46255],
[0.94022, 0.095176, 0.46008],
[0.94529, 0.10322, 0.45759],
[0.95027, 0.11119, 0.45507],
[0.95516, 0.11907, 0.45252],
[0.95996, 0.12687, 0.44993],
[0.96467, 0.13467, 0.44731],
[0.9693, 0.14235, 0.44468],
[0.97383, 0.15004, 0.44201],
[0.97827, 0.15766, 0.43929],
[0.9826, 0.16532, 0.43656],
[0.9868, 0.17308, 0.4338],
[0.99086, 0.18089, 0.43103],
[0.99477, 0.18882, 0.42822],
[0.99854, 0.19681, 0.42539],
[1, 0.20481, 0.42252],
[1, 0.21285, 0.41965],
[1, 0.22099, 0.41674],
[1, 0.22914, 0.4138],
[1, 0.23736, 0.41083],
[1, 0.24555, 0.40783],
[1, 0.25382, 0.4048],
[1, 0.2621, 0.40173],
[1, 0.27043, 0.39864],
[1, 0.27875, 0.39552],
[1, 0.28708, 0.39235],
[1, 0.29545, 0.38916],
[1, 0.30385, 0.38591],
[1, 0.31225, 0.38264],
[1, 0.3207, 0.37933],
[1, 0.32913, 0.37598],
[1, 0.33756, 0.37257],
[1, 0.34602, 0.36914],
[1, 0.35451, 0.36565],
[1, 0.36297, 0.36212],
[1, 0.37145, 0.35853],
[1, 0.37995, 0.35492],
[1, 0.38846, 0.35122],
[1, 0.39696, 0.34749],
[1, 0.40547, 0.34369],
[1, 0.4139, 0.33986],
[1, 0.4222, 0.33597],
[1, 0.4304, 0.33207],
[1, 0.43847, 0.3281],
[1, 0.44643, 0.3241],
[1, 0.45429, 0.32007],
[1, 0.46204, 0.31598],
[1, 0.46971, 0.31182],
[1, 0.47727, 0.30763],
[1, 0.48476, 0.30339],
[1, 0.49217, 0.2991],
[1, 0.4995, 0.29473],
[1, 0.50677, 0.2903],
[1, 0.51397, 0.28581],
[1, 0.5211, 0.28126],
[1, 0.52816, 0.27665],
[1, 0.53517, 0.27191],
[1, 0.54212, 0.26713],
[1, 0.54902, 0.26226],
[1, 0.55585, 0.2573],
[1, 0.56265, 0.25223],
[1, 0.56938, 0.24705],
[1, 0.57609, 0.24177],
[1, 0.58273, 0.23637],
[1, 0.58934, 0.23082],
[1, 0.59591, 0.22513],
[1, 0.60244, 0.21933],
[1, 0.60893, 0.21333],
[1, 0.61536, 0.20719],
[1, 0.62179, 0.20085],
[1, 0.62814, 0.19473],
[1, 0.63438, 0.18898],
[1, 0.64052, 0.18363],
[1, 0.64658, 0.17866],
[1, 0.65258, 0.17395],
[1, 0.65851, 0.16956],
[1, 0.66439, 0.16532],
[1, 0.67023, 0.1614],
[1, 0.67601, 0.15763],
[1, 0.68174, 0.1541],
[1, 0.68745, 0.15075],
[1, 0.69312, 0.14754],
[1, 0.69875, 0.14449],
[1, 0.70435, 0.14163],
[1, 0.70993, 0.13886],
[1, 0.71548, 0.13623],
[1, 0.72099, 0.13382],
[1, 0.72649, 0.1315],
[1, 0.73197, 0.12928],
[1, 0.73742, 0.12722],
[1, 0.74286, 0.1253],
[1, 0.74827, 0.12347],
[1, 0.75367, 0.1218],
[1, 0.75905, 0.12025],
[1, 0.76442, 0.11888],
[1, 0.76976, 0.11751],
[1, 0.7751, 0.11639],
[1, 0.78042, 0.11528],
[1, 0.78573, 0.11434],
[1, 0.79102, 0.11357],
[1, 0.79631, 0.11289],
[1, 0.80158, 0.1123],
[1, 0.80685, 0.11183],
[1, 0.81209, 0.1115],
[1, 0.81733, 0.11127],
[1, 0.82257, 0.11116],
[1, 0.82778, 0.11116],
[1, 0.833, 0.11128],
[1, 0.8382, 0.1115],
[1, 0.8434, 0.11183],
[1, 0.84859, 0.11227],
[1, 0.85377, 0.11282],
[1, 0.85894, 0.11347],
[1, 0.86412, 0.11414],
[1, 0.86928, 0.115],
[1, 0.87443, 0.11595],
[1, 0.87958, 0.11697],
[1, 0.88472, 0.11809],
[1, 0.88986, 0.11929],
[1, 0.89499, 0.12054],
[1, 0.90012, 0.1219],
[1, 0.90524, 0.12335],
[1, 0.91036, 0.12486],
[1, 0.91547, 0.12649],
[1, 0.92058, 0.12812],
[1, 0.92569, 0.1299],
[1, 0.93079, 0.13165],
[1, 0.93588, 0.13349],
[1, 0.94097, 0.13541],
[1, 0.94606, 0.13735],
]
b_linear_bmy_10_95_c78 = bokeh_palette('linear_bmy_10_95_c78',linear_bmy_10_95_c78)
m_linear_bmy_10_95_c78 = mpl_cm('linear_bmy_10_95_c78',linear_bmy_10_95_c78)
m_linear_bmy_10_95_c78_r = mpl_cm('linear_bmy_10_95_c78_r',list(reversed(linear_bmy_10_95_c78)))
bmy = b_linear_bmy_10_95_c78
m_bmy = m_linear_bmy_10_95_c78
m_bmy_r = m_linear_bmy_10_95_c78_r
palette['bmy'] = b_linear_bmy_10_95_c78
palette_n['bmy'] = b_linear_bmy_10_95_c78
cm['bmy'] = m_linear_bmy_10_95_c78
cm['bmy_r'] = m_linear_bmy_10_95_c78_r
cm_n['bmy'] = mpl_cm('bmy',linear_bmy_10_95_c78)
cm_n['bmy_r'] = mpl_cm('bmy_r',list(reversed(linear_bmy_10_95_c78)))




linear_gow_65_90_c35 = [  # cmap_def
[0.43765, 0.67814, 0.36054],
[0.44314, 0.67844, 0.361],
[0.4486, 0.67874, 0.36146],
[0.45399, 0.67903, 0.36192],
[0.45934, 0.67932, 0.36237],
[0.46465, 0.67961, 0.36282],
[0.46991, 0.6799, 0.36327],
[0.47513, 0.68018, 0.36373],
[0.48029, 0.68047, 0.36419],
[0.48542, 0.68075, 0.36465],
[0.49052, 0.68102, 0.36511],
[0.49557, 0.6813, 0.36557],
[0.50059, 0.68157, 0.36603],
[0.50556, 0.68184, 0.3665],
[0.51051, 0.68211, 0.36696],
[0.51542, 0.68237, 0.36741],
[0.5203, 0.68264, 0.36788],
[0.52515, 0.6829, 0.36835],
[0.52996, 0.68316, 0.36882],
[0.53474, 0.68341, 0.36928],
[0.53949, 0.68366, 0.36975],
[0.54422, 0.68391, 0.37021],
[0.54891, 0.68416, 0.37068],
[0.55357, 0.6844, 0.37115],
[0.55821, 0.68465, 0.37162],
[0.56283, 0.68489, 0.3721],
[0.56741, 0.68514, 0.37257],
[0.57197, 0.68538, 0.37305],
[0.57651, 0.68561, 0.37353],
[0.58101, 0.68585, 0.37401],
[0.5855, 0.68608, 0.37449],
[0.58995, 0.68631, 0.37496],
[0.59438, 0.68654, 0.37543],
[0.5988, 0.68676, 0.37592],
[0.60319, 0.68699, 0.37641],
[0.60756, 0.68721, 0.37689],
[0.6119, 0.68744, 0.37737],
[0.61622, 0.68766, 0.37785],
[0.62053, 0.68788, 0.37833],
[0.6248, 0.6881, 0.37882],
[0.62906, 0.68831, 0.37931],
[0.6333, 0.68853, 0.37979],
[0.63752, 0.68875, 0.38029],
[0.64171, 0.68896, 0.38078],
[0.64589, 0.68917, 0.38127],
[0.65005, 0.68938, 0.38175],
[0.65418, 0.68959, 0.38224],
[0.6583, 0.6898, 0.38274],
[0.6624, 0.69001, 0.38324],
[0.66647, 0.69021, 0.38374],
[0.67054, 0.69042, 0.38423],
[0.67457, 0.69062, 0.38472],
[0.67859, 0.69083, 0.38522],
[0.68259, 0.69103, 0.38572],
[0.68657, 0.69124, 0.38622],
[0.69054, 0.69145, 0.38672],
[0.69448, 0.69166, 0.38723],
[0.69841, 0.69186, 0.38773],
[0.70231, 0.69207, 0.38824],
[0.70621, 0.69228, 0.38876],
[0.71007, 0.69248, 0.38926],
[0.71392, 0.69269, 0.38976],
[0.71776, 0.69289, 0.39027],
[0.72157, 0.6931, 0.39079],
[0.72537, 0.6933, 0.39131],
[0.72914, 0.69351, 0.39182],
[0.7329, 0.69372, 0.39233],
[0.73663, 0.69393, 0.39284],
[0.74035, 0.69414, 0.39336],
[0.74405, 0.69435, 0.39388],
[0.74774, 0.69457, 0.3944],
[0.7514, 0.69479, 0.39492],
[0.75504, 0.69501, 0.39545],
[0.75865, 0.69523, 0.39597],
[0.76225, 0.69545, 0.39649],
[0.76584, 0.69568, 0.39701],
[0.76939, 0.69591, 0.39755],
[0.77293, 0.69614, 0.39808],
[0.77645, 0.69638, 0.39861],
[0.77995, 0.69661, 0.39913],
[0.78342, 0.69685, 0.39966],
[0.78688, 0.69709, 0.40021],
[0.7903, 0.69734, 0.40076],
[0.79372, 0.69759, 0.40129],
[0.79709, 0.69785, 0.40182],
[0.80046, 0.69811, 0.40236],
[0.80379, 0.69838, 0.40291],
[0.80711, 0.69865, 0.40346],
[0.81039, 0.69893, 0.404],
[0.81365, 0.69921, 0.40454],
[0.81689, 0.6995, 0.40509],
[0.82009, 0.6998, 0.40565],
[0.82328, 0.70011, 0.40621],
[0.82643, 0.70042, 0.40676],
[0.82956, 0.70074, 0.40731],
[0.83264, 0.70107, 0.40787],
[0.83571, 0.7014, 0.40843],
[0.83874, 0.70174, 0.409],
[0.84174, 0.7021, 0.40956],
[0.8447, 0.70247, 0.41014],
[0.84763, 0.70285, 0.41071],
[0.85052, 0.70324, 0.41127],
[0.85337, 0.70364, 0.41185],
[0.85619, 0.70405, 0.41243],
[0.85896, 0.70448, 0.41301],
[0.86169, 0.70492, 0.41359],
[0.86438, 0.70538, 0.41418],
[0.86701, 0.70586, 0.41476],
[0.8696, 0.70636, 0.41536],
[0.87215, 0.70687, 0.41595],
[0.87463, 0.7074, 0.41656],
[0.87705, 0.70795, 0.41716],
[0.87942, 0.70853, 0.41776],
[0.88172, 0.70913, 0.41837],
[0.88395, 0.70976, 0.41899],
[0.88611, 0.71042, 0.41961],
[0.88818, 0.71111, 0.42023],
[0.89017, 0.71183, 0.42086],
[0.89208, 0.71258, 0.4215],
[0.89388, 0.71338, 0.42214],
[0.89558, 0.71421, 0.42279],
[0.89716, 0.7151, 0.42344],
[0.8986, 0.71605, 0.42411],
[0.8999, 0.71704, 0.42479],
[0.90104, 0.7181, 0.42548],
[0.902, 0.71924, 0.42616],
[0.90274, 0.72045, 0.42688],
[0.90325, 0.72176, 0.42761],
[0.90355, 0.72316, 0.42836],
[0.90378, 0.72458, 0.42917],
[0.90399, 0.726, 0.43006],
[0.9042, 0.72741, 0.43102],
[0.90441, 0.72883, 0.43206],
[0.90462, 0.73024, 0.43315],
[0.90481, 0.73165, 0.43433],
[0.90501, 0.73306, 0.43559],
[0.90519, 0.73447, 0.43691],
[0.90537, 0.73586, 0.4383],
[0.90555, 0.73727, 0.43978],
[0.90572, 0.73867, 0.44132],
[0.90588, 0.74007, 0.44294],
[0.90604, 0.74145, 0.44463],
[0.90619, 0.74285, 0.44638],
[0.90633, 0.74424, 0.44821],
[0.90647, 0.74563, 0.45013],
[0.9066, 0.74702, 0.45211],
[0.90673, 0.7484, 0.45416],
[0.90684, 0.74978, 0.45628],
[0.90696, 0.75117, 0.45847],
[0.90706, 0.75254, 0.46074],
[0.90715, 0.75392, 0.46307],
[0.90724, 0.75529, 0.46549],
[0.90732, 0.75666, 0.46797],
[0.90739, 0.75804, 0.47054],
[0.90746, 0.7594, 0.47315],
[0.90751, 0.76077, 0.47586],
[0.90756, 0.76213, 0.47862],
[0.90759, 0.7635, 0.48146],
[0.90762, 0.76487, 0.48437],
[0.90764, 0.76622, 0.48736],
[0.90764, 0.76758, 0.49041],
[0.90764, 0.76894, 0.49353],
[0.90763, 0.77029, 0.49672],
[0.9076, 0.77165, 0.49998],
[0.90756, 0.773, 0.50331],
[0.90752, 0.77435, 0.50672],
[0.90746, 0.7757, 0.51019],
[0.90738, 0.77706, 0.51373],
[0.9073, 0.7784, 0.51735],
[0.9072, 0.77974, 0.52102],
[0.90709, 0.78109, 0.52477],
[0.90697, 0.78243, 0.52859],
[0.90683, 0.78378, 0.53248],
[0.90667, 0.78512, 0.53643],
[0.90651, 0.78646, 0.54046],
[0.90632, 0.78779, 0.54455],
[0.90612, 0.78913, 0.54872],
[0.90591, 0.79046, 0.55293],
[0.90568, 0.7918, 0.55724],
[0.90544, 0.79314, 0.56159],
[0.9052, 0.79447, 0.56594],
[0.90497, 0.79578, 0.5703],
[0.90475, 0.7971, 0.57465],
[0.90454, 0.79841, 0.579],
[0.90433, 0.79973, 0.58333],
[0.90413, 0.80102, 0.58767],
[0.90394, 0.80232, 0.59201],
[0.90376, 0.80361, 0.59633],
[0.90358, 0.8049, 0.60065],
[0.90341, 0.80619, 0.60497],
[0.90323, 0.80747, 0.60929],
[0.90307, 0.80875, 0.6136],
[0.9029, 0.81003, 0.61791],
[0.90273, 0.8113, 0.62221],
[0.90257, 0.81257, 0.62652],
[0.90241, 0.81384, 0.63083],
[0.90225, 0.8151, 0.63512],
[0.90209, 0.81637, 0.63942],
[0.90194, 0.81763, 0.64371],
[0.90178, 0.81888, 0.648],
[0.90162, 0.82014, 0.6523],
[0.90146, 0.8214, 0.65659],
[0.9013, 0.82265, 0.66087],
[0.90114, 0.8239, 0.66516],
[0.90098, 0.82514, 0.66945],
[0.90082, 0.82639, 0.67373],
[0.90066, 0.82764, 0.67801],
[0.90049, 0.82888, 0.68229],
[0.90032, 0.83012, 0.68657],
[0.90015, 0.83137, 0.69084],
[0.89998, 0.8326, 0.69512],
[0.89981, 0.83384, 0.6994],
[0.89963, 0.83508, 0.70368],
[0.89945, 0.83631, 0.70795],
[0.89927, 0.83755, 0.71222],
[0.89908, 0.83878, 0.7165],
[0.89889, 0.84002, 0.72077],
[0.89869, 0.84124, 0.72505],
[0.89849, 0.84248, 0.72932],
[0.89829, 0.8437, 0.73359],
[0.89809, 0.84493, 0.73786],
[0.89788, 0.84616, 0.74213],
[0.89766, 0.84738, 0.7464],
[0.89745, 0.84861, 0.75068],
[0.89722, 0.84984, 0.75495],
[0.897, 0.85106, 0.75921],
[0.89676, 0.85229, 0.76349],
[0.89652, 0.85351, 0.76776],
[0.89628, 0.85474, 0.77203],
[0.89603, 0.85596, 0.7763],
[0.89578, 0.85718, 0.78057],
[0.89552, 0.8584, 0.78485],
[0.89526, 0.85962, 0.78912],
[0.89499, 0.86084, 0.7934],
[0.89471, 0.86207, 0.79767],
[0.89443, 0.86328, 0.80194],
[0.89414, 0.8645, 0.80622],
[0.89385, 0.86572, 0.81049],
[0.89355, 0.86694, 0.81477],
[0.89324, 0.86816, 0.81904],
[0.89293, 0.86938, 0.82332],
[0.89262, 0.87059, 0.8276],
[0.89229, 0.87181, 0.83188],
[0.89196, 0.87303, 0.83616],
[0.89162, 0.87424, 0.84044],
[0.89128, 0.87546, 0.84472],
[0.89093, 0.87667, 0.849],
[0.89057, 0.87789, 0.85328],
[0.8902, 0.87911, 0.85756],
[0.88983, 0.88032, 0.86185],
[0.88945, 0.88154, 0.86614],
[0.88907, 0.88276, 0.87042],
[0.88867, 0.88398, 0.87471],
[0.88827, 0.88519, 0.879],
[0.88786, 0.88641, 0.88329],
[0.88744, 0.88762, 0.88758],
]
b_linear_gow_65_90_c35 = bokeh_palette('linear_gow_65_90_c35',linear_gow_65_90_c35)
m_linear_gow_65_90_c35 = mpl_cm('linear_gow_65_90_c35',linear_gow_65_90_c35)
m_linear_gow_65_90_c35_r = mpl_cm('linear_gow_65_90_c35_r',list(reversed(linear_gow_65_90_c35)))
CET_L11 = b_linear_gow_65_90_c35
m_CET_L11 = m_linear_gow_65_90_c35
m_CET_L11_r = m_linear_gow_65_90_c35_r
palette['CET_L11'] = b_linear_gow_65_90_c35
cm['CET_L11'] = m_linear_gow_65_90_c35
cm['CET_L11_r'] = m_linear_gow_65_90_c35_r
register_cmap('cet_CET_L11',m_linear_gow_65_90_c35)
register_cmap('cet_CET_L11_r',m_linear_gow_65_90_c35_r)




linear_ternary_blue_0_44_c57 = [  # cmap_def
[0, 0, 0],
[0.00074336, 0.0024531, 0.0074342],
[0.0014922, 0.0049243, 0.014921],
[0.0022238, 0.0073388, 0.022238],
[0.0029727, 0.0098191, 0.029727],
[0.003759, 0.01243, 0.037588],
[0.0044965, 0.014837, 0.045061],
[0.0052171, 0.017216, 0.052174],
[0.0059233, 0.019547, 0.05928],
[0.0066117, 0.021819, 0.066122],
[0.0072841, 0.024037, 0.07289],
[0.0079425, 0.026211, 0.079443],
[0.008587, 0.02834, 0.085862],
[0.0092271, 0.03044, 0.092265],
[0.009844, 0.03246, 0.098364],
[0.010477, 0.034486, 0.10458],
[0.011109, 0.036556, 0.11078],
[0.01171, 0.038495, 0.11667],
[0.012273, 0.040403, 0.12242],
[0.012818, 0.042245, 0.1279],
[0.013348, 0.043976, 0.13326],
[0.013879, 0.04566, 0.1385],
[0.014395, 0.047394, 0.14358],
[0.014889, 0.049082, 0.14858],
[0.015355, 0.050605, 0.15335],
[0.015827, 0.052141, 0.15801],
[0.016294, 0.053656, 0.16261],
[0.016752, 0.05516, 0.16717],
[0.017191, 0.056648, 0.17153],
[0.017621, 0.057987, 0.17584],
[0.018044, 0.059476, 0.18012],
[0.018452, 0.060774, 0.18421],
[0.018859, 0.062143, 0.1883],
[0.019258, 0.063465, 0.19229],
[0.019638, 0.064737, 0.19618],
[0.019991, 0.066008, 0.19999],
[0.020369, 0.06727, 0.20383],
[0.020743, 0.068517, 0.20757],
[0.021108, 0.069731, 0.21123],
[0.021466, 0.07089, 0.21479],
[0.021817, 0.072042, 0.21833],
[0.022172, 0.073262, 0.22184],
[0.022518, 0.074345, 0.22528],
[0.022855, 0.075456, 0.22864],
[0.023186, 0.076553, 0.23196],
[0.023512, 0.077642, 0.23526],
[0.02384, 0.078716, 0.2385],
[0.024178, 0.079786, 0.24179],
[0.024507, 0.080865, 0.24506],
[0.024831, 0.08197, 0.24832],
[0.025156, 0.08299, 0.25156],
[0.02548, 0.084106, 0.2548],
[0.025801, 0.085132, 0.25798],
[0.026127, 0.086192, 0.26127],
[0.026456, 0.087333, 0.26456],
[0.026783, 0.088377, 0.26781],
[0.027109, 0.089458, 0.27108],
[0.027438, 0.090542, 0.27438],
[0.027766, 0.091648, 0.27766],
[0.02809, 0.092669, 0.28088],
[0.028421, 0.093789, 0.2842],
[0.028753, 0.094903, 0.28753],
[0.029077, 0.095942, 0.29078],
[0.029408, 0.097031, 0.29407],
[0.029741, 0.09817, 0.29742],
[0.030072, 0.099229, 0.30072],
[0.030403, 0.10033, 0.30402],
[0.030735, 0.10143, 0.30733],
[0.031072, 0.10254, 0.31072],
[0.031403, 0.10364, 0.31403],
[0.031734, 0.1047, 0.31734],
[0.032072, 0.10585, 0.32074],
[0.032408, 0.10696, 0.32408],
[0.032743, 0.10802, 0.32739],
[0.033082, 0.10918, 0.33077],
[0.033414, 0.11028, 0.33418],
[0.033742, 0.11138, 0.33753],
[0.034077, 0.1125, 0.3409],
[0.03442, 0.11362, 0.34424],
[0.034765, 0.11469, 0.34763],
[0.035114, 0.11586, 0.35104],
[0.035456, 0.11695, 0.35445],
[0.035791, 0.1181, 0.35784],
[0.036121, 0.1192, 0.36122],
[0.036456, 0.12031, 0.36459],
[0.036796, 0.12143, 0.36797],
[0.037139, 0.12255, 0.37137],
[0.037483, 0.12369, 0.37481],
[0.037828, 0.12482, 0.37826],
[0.038169, 0.12598, 0.38168],
[0.038511, 0.12707, 0.3851],
[0.038856, 0.12823, 0.38856],
[0.0392, 0.12935, 0.392],
[0.039546, 0.13052, 0.39546],
[0.03989, 0.13163, 0.39889],
[0.040235, 0.13277, 0.40233],
[0.040581, 0.13392, 0.40582],
[0.040922, 0.13506, 0.40923],
[0.041266, 0.13617, 0.41269],
[0.041619, 0.13735, 0.4162],
[0.04197, 0.13849, 0.41964],
[0.042315, 0.1396, 0.42308],
[0.042655, 0.14079, 0.42658],
[0.042998, 0.14192, 0.43006],
[0.043353, 0.14307, 0.43353],
[0.043707, 0.14422, 0.43705],
[0.044054, 0.1454, 0.44054],
[0.044401, 0.14652, 0.44404],
[0.044755, 0.14769, 0.44752],
[0.045109, 0.14883, 0.45103],
[0.045452, 0.15001, 0.45455],
[0.045792, 0.15115, 0.45803],
[0.046147, 0.15231, 0.46154],
[0.046509, 0.15347, 0.46506],
[0.046871, 0.15464, 0.4686],
[0.047221, 0.15582, 0.47216],
[0.047563, 0.15696, 0.47568],
[0.047916, 0.15815, 0.47921],
[0.048273, 0.1593, 0.48272],
[0.048631, 0.16046, 0.48628],
[0.048997, 0.16167, 0.48986],
[0.049352, 0.16283, 0.49342],
[0.049698, 0.16401, 0.49696],
[0.050046, 0.16514, 0.50048],
[0.050404, 0.16634, 0.50406],
[0.050762, 0.16751, 0.50763],
[0.051121, 0.1687, 0.5112],
[0.051479, 0.16988, 0.51479],
[0.051833, 0.17105, 0.51834],
[0.05219, 0.17223, 0.5219],
[0.052551, 0.17342, 0.52551],
[0.052909, 0.17461, 0.5291],
[0.053266, 0.17577, 0.53267],
[0.053628, 0.17697, 0.53628],
[0.05399, 0.17818, 0.53989],
[0.054349, 0.17935, 0.54349],
[0.054708, 0.18053, 0.54707],
[0.055067, 0.18172, 0.55067],
[0.055429, 0.18292, 0.55429],
[0.055793, 0.18412, 0.55793],
[0.056155, 0.18533, 0.56157],
[0.056521, 0.18651, 0.5652],
[0.056888, 0.18772, 0.56884],
[0.057244, 0.18891, 0.57245],
[0.0576, 0.1901, 0.57607],
[0.057967, 0.1913, 0.57969],
[0.058336, 0.1925, 0.58335],
[0.058702, 0.19372, 0.58701],
[0.059067, 0.19492, 0.59066],
[0.059436, 0.19613, 0.59432],
[0.059804, 0.19733, 0.59797],
[0.060167, 0.19853, 0.60161],
[0.060527, 0.19973, 0.60525],
[0.060887, 0.20094, 0.60892],
[0.061253, 0.20216, 0.61259],
[0.061628, 0.20338, 0.6163],
[0.062, 0.2046, 0.61999],
[0.06237, 0.20581, 0.62368],
[0.062743, 0.20703, 0.62736],
[0.063109, 0.20824, 0.63104],
[0.063471, 0.20946, 0.63472],
[0.06384, 0.21068, 0.63842],
[0.064212, 0.21191, 0.64213],
[0.064582, 0.2131, 0.64581],
[0.064951, 0.21435, 0.64951],
[0.065324, 0.21557, 0.65323],
[0.065695, 0.21679, 0.65696],
[0.066066, 0.21802, 0.66066],
[0.066438, 0.21924, 0.66438],
[0.066811, 0.22049, 0.66812],
[0.067184, 0.22169, 0.67183],
[0.067556, 0.22294, 0.67556],
[0.067927, 0.22415, 0.67926],
[0.068301, 0.2254, 0.68302],
[0.068677, 0.22664, 0.68678],
[0.069052, 0.22788, 0.69052],
[0.069427, 0.2291, 0.69424],
[0.0698, 0.23033, 0.69799],
[0.07017, 0.23158, 0.70175],
[0.070545, 0.23281, 0.70549],
[0.070927, 0.23405, 0.70927],
[0.07131, 0.23531, 0.71304],
[0.071685, 0.23655, 0.71681],
[0.072052, 0.23779, 0.72056],
[0.072426, 0.23902, 0.72432],
[0.072809, 0.24026, 0.72809],
[0.073193, 0.24153, 0.73188],
[0.073573, 0.24276, 0.73565],
[0.073946, 0.24401, 0.73944],
[0.074319, 0.24526, 0.74322],
[0.0747, 0.24651, 0.74702],
[0.075082, 0.24778, 0.75083],
[0.075461, 0.24902, 0.75462],
[0.075841, 0.25026, 0.75841],
[0.076222, 0.25154, 0.76222],
[0.076602, 0.25279, 0.76602],
[0.076984, 0.25405, 0.76985],
[0.077365, 0.25531, 0.77365],
[0.077744, 0.25656, 0.77745],
[0.078128, 0.25781, 0.78128],
[0.078511, 0.25909, 0.78511],
[0.078893, 0.26035, 0.78893],
[0.079275, 0.26161, 0.79275],
[0.07966, 0.26288, 0.7966],
[0.080044, 0.26414, 0.80043],
[0.080427, 0.26541, 0.80427],
[0.08081, 0.26667, 0.80809],
[0.081193, 0.26794, 0.81194],
[0.081582, 0.26923, 0.81583],
[0.081967, 0.27049, 0.81966],
[0.082351, 0.27175, 0.8235],
[0.082734, 0.27303, 0.82736],
[0.08312, 0.2743, 0.83122],
[0.08351, 0.27558, 0.8351],
[0.083897, 0.27686, 0.83896],
[0.084288, 0.27814, 0.84284],
[0.084673, 0.27941, 0.84669],
[0.085055, 0.28068, 0.85055],
[0.085441, 0.28197, 0.85445],
[0.085828, 0.28325, 0.85832],
[0.086218, 0.28452, 0.8622],
[0.086607, 0.2858, 0.86608],
[0.087001, 0.28709, 0.86997],
[0.087391, 0.28838, 0.87388],
[0.087778, 0.28967, 0.87777],
[0.088167, 0.29095, 0.88168],
[0.088557, 0.29224, 0.88557],
[0.088947, 0.29352, 0.88947],
[0.089338, 0.29481, 0.89338],
[0.08973, 0.29611, 0.8973],
[0.090121, 0.29741, 0.90121],
[0.090513, 0.29869, 0.90514],
[0.090904, 0.29999, 0.90904],
[0.091296, 0.30128, 0.91296],
[0.09169, 0.30257, 0.91688],
[0.092085, 0.30387, 0.92081],
[0.092476, 0.30517, 0.92477],
[0.092866, 0.30647, 0.92868],
[0.093261, 0.30776, 0.93261],
[0.093659, 0.30908, 0.93659],
[0.094052, 0.31037, 0.94052],
[0.094445, 0.31167, 0.94445],
[0.094842, 0.31297, 0.9484],
[0.095238, 0.31428, 0.95235],
[0.095633, 0.31559, 0.95632],
[0.096025, 0.31689, 0.96026],
[0.096419, 0.31819, 0.96422],
[0.096817, 0.3195, 0.96819],
[0.097216, 0.32082, 0.97216],
[0.097613, 0.32212, 0.97613],
[0.098012, 0.32343, 0.9801],
[0.098407, 0.32474, 0.98406],
[0.098802, 0.32605, 0.98803],
[0.099201, 0.32737, 0.99202],
[0.099601, 0.32868, 0.99601],
[0.1, 0.33, 1],
]
b_linear_ternary_blue_0_44_c57 = bokeh_palette('linear_ternary_blue_0_44_c57',linear_ternary_blue_0_44_c57)
m_linear_ternary_blue_0_44_c57 = mpl_cm('linear_ternary_blue_0_44_c57',linear_ternary_blue_0_44_c57)
m_linear_ternary_blue_0_44_c57_r = mpl_cm('linear_ternary_blue_0_44_c57_r',list(reversed(linear_ternary_blue_0_44_c57)))
kb = b_linear_ternary_blue_0_44_c57
m_kb = m_linear_ternary_blue_0_44_c57
m_kb_r = m_linear_ternary_blue_0_44_c57_r
palette['kb'] = b_linear_ternary_blue_0_44_c57
palette_n['kb'] = b_linear_ternary_blue_0_44_c57
cm['kb'] = m_linear_ternary_blue_0_44_c57
cm['kb_r'] = m_linear_ternary_blue_0_44_c57_r
cm_n['kb'] = mpl_cm('kb',linear_ternary_blue_0_44_c57)
cm_n['kb_r'] = mpl_cm('kb_r',list(reversed(linear_ternary_blue_0_44_c57)))
CET_L15 = b_linear_ternary_blue_0_44_c57
m_CET_L15 = m_linear_ternary_blue_0_44_c57
m_CET_L15_r = m_linear_ternary_blue_0_44_c57_r
palette['CET_L15'] = b_linear_ternary_blue_0_44_c57
cm['CET_L15'] = m_linear_ternary_blue_0_44_c57
cm['CET_L15_r'] = m_linear_ternary_blue_0_44_c57_r
register_cmap('cet_CET_L15',m_linear_ternary_blue_0_44_c57)
register_cmap('cet_CET_L15_r',m_linear_ternary_blue_0_44_c57_r)




isoluminant_cm_70_c39 = [  # cmap_def
[0.078311, 0.7275, 0.90055],
[0.096546, 0.72692, 0.9011],
[0.11226, 0.72635, 0.90165],
[0.12614, 0.72577, 0.9022],
[0.13869, 0.72519, 0.90275],
[0.15028, 0.72461, 0.90329],
[0.16105, 0.72402, 0.90382],
[0.17113, 0.72343, 0.90436],
[0.18067, 0.72284, 0.90489],
[0.18976, 0.72225, 0.90541],
[0.19841, 0.72165, 0.90593],
[0.20674, 0.72106, 0.90644],
[0.2147, 0.72046, 0.90695],
[0.2224, 0.71986, 0.90745],
[0.22982, 0.71926, 0.90795],
[0.23706, 0.71865, 0.90845],
[0.24402, 0.71804, 0.90894],
[0.25084, 0.71744, 0.90942],
[0.25746, 0.71682, 0.9099],
[0.26394, 0.71621, 0.91038],
[0.27024, 0.71559, 0.91085],
[0.27641, 0.71496, 0.91131],
[0.28246, 0.71434, 0.91177],
[0.28836, 0.71372, 0.91222],
[0.29415, 0.71309, 0.91267],
[0.29986, 0.71246, 0.91312],
[0.30546, 0.71183, 0.91356],
[0.31096, 0.71119, 0.91399],
[0.31637, 0.71055, 0.91441],
[0.32169, 0.70991, 0.91483],
[0.32693, 0.70927, 0.91524],
[0.33212, 0.70863, 0.91565],
[0.33721, 0.70797, 0.91605],
[0.34224, 0.70732, 0.91645],
[0.34721, 0.70668, 0.91684],
[0.3521, 0.70602, 0.91723],
[0.35696, 0.70536, 0.91761],
[0.36174, 0.70469, 0.91797],
[0.36648, 0.70403, 0.91834],
[0.37114, 0.70336, 0.91869],
[0.37578, 0.70269, 0.91904],
[0.38037, 0.70202, 0.91938],
[0.3849, 0.70134, 0.91972],
[0.3894, 0.70067, 0.92005],
[0.39385, 0.69998, 0.92037],
[0.39827, 0.6993, 0.92068],
[0.40264, 0.69861, 0.92099],
[0.40698, 0.69792, 0.92129],
[0.41129, 0.69723, 0.92158],
[0.41555, 0.69654, 0.92187],
[0.41979, 0.69584, 0.92214],
[0.42399, 0.69514, 0.92241],
[0.42817, 0.69443, 0.92267],
[0.43232, 0.69372, 0.92292],
[0.43643, 0.69302, 0.92316],
[0.44051, 0.69231, 0.92339],
[0.44458, 0.69158, 0.92362],
[0.44861, 0.69086, 0.92383],
[0.45263, 0.69014, 0.92403],
[0.4566, 0.68942, 0.92423],
[0.46056, 0.68869, 0.92441],
[0.46452, 0.68796, 0.92459],
[0.46843, 0.68722, 0.92475],
[0.47233, 0.68648, 0.92491],
[0.47621, 0.68574, 0.92505],
[0.48007, 0.685, 0.92519],
[0.4839, 0.68424, 0.92531],
[0.48774, 0.6835, 0.92543],
[0.49155, 0.68274, 0.92553],
[0.49532, 0.68198, 0.92562],
[0.49909, 0.68122, 0.9257],
[0.50285, 0.68045, 0.92576],
[0.50659, 0.67969, 0.92582],
[0.51032, 0.67892, 0.92586],
[0.51404, 0.67815, 0.92589],
[0.51773, 0.67737, 0.9259],
[0.52141, 0.67659, 0.92591],
[0.52509, 0.6758, 0.92589],
[0.52875, 0.67501, 0.92587],
[0.5324, 0.67422, 0.92583],
[0.53603, 0.67343, 0.92578],
[0.53965, 0.67264, 0.92571],
[0.54327, 0.67183, 0.92562],
[0.54687, 0.67103, 0.92552],
[0.55046, 0.67022, 0.92541],
[0.55405, 0.66941, 0.92527],
[0.55762, 0.66859, 0.92513],
[0.56118, 0.66778, 0.92496],
[0.56474, 0.66695, 0.92478],
[0.56828, 0.66613, 0.92458],
[0.57183, 0.6653, 0.92436],
[0.57536, 0.66447, 0.92412],
[0.57888, 0.66364, 0.92387],
[0.58239, 0.6628, 0.92359],
[0.58591, 0.66196, 0.92329],
[0.5894, 0.66111, 0.92298],
[0.5929, 0.66026, 0.92264],
[0.59638, 0.65941, 0.92228],
[0.59986, 0.65856, 0.92189],
[0.60333, 0.65771, 0.92149],
[0.6068, 0.65685, 0.92106],
[0.61025, 0.65599, 0.9206],
[0.6137, 0.65512, 0.92013],
[0.61716, 0.65425, 0.91962],
[0.6206, 0.65338, 0.91909],
[0.62403, 0.65251, 0.91854],
[0.62745, 0.65164, 0.91796],
[0.63088, 0.65076, 0.91735],
[0.63429, 0.64989, 0.91671],
[0.63769, 0.64901, 0.91604],
[0.64108, 0.64812, 0.91534],
[0.64447, 0.64724, 0.91461],
[0.64785, 0.64636, 0.91385],
[0.65123, 0.64547, 0.91306],
[0.65459, 0.64458, 0.91223],
[0.65795, 0.6437, 0.91137],
[0.66129, 0.64282, 0.91048],
[0.66462, 0.64193, 0.90955],
[0.66796, 0.64104, 0.90858],
[0.67126, 0.64017, 0.90758],
[0.67456, 0.63928, 0.90654],
[0.67785, 0.6384, 0.90546],
[0.68111, 0.63753, 0.90434],
[0.68437, 0.63666, 0.90319],
[0.68761, 0.63578, 0.90199],
[0.69083, 0.63491, 0.90077],
[0.69404, 0.63405, 0.89953],
[0.69723, 0.63317, 0.89826],
[0.7004, 0.63231, 0.89698],
[0.70355, 0.63144, 0.89567],
[0.70668, 0.63058, 0.89435],
[0.70979, 0.62971, 0.89301],
[0.71288, 0.62885, 0.89165],
[0.71597, 0.62799, 0.89028],
[0.71902, 0.62713, 0.88889],
[0.72206, 0.62627, 0.88748],
[0.72509, 0.62541, 0.88607],
[0.72809, 0.62455, 0.88463],
[0.73108, 0.6237, 0.88318],
[0.73405, 0.62284, 0.88173],
[0.737, 0.62198, 0.88025],
[0.73994, 0.62113, 0.87877],
[0.74285, 0.62028, 0.87728],
[0.74575, 0.61942, 0.87577],
[0.74863, 0.61857, 0.87426],
[0.7515, 0.61771, 0.87273],
[0.75435, 0.61686, 0.8712],
[0.75718, 0.61601, 0.86965],
[0.75999, 0.61515, 0.8681],
[0.76279, 0.6143, 0.86653],
[0.76558, 0.61345, 0.86496],
[0.76834, 0.6126, 0.86339],
[0.7711, 0.61175, 0.8618],
[0.77383, 0.6109, 0.8602],
[0.77656, 0.61005, 0.8586],
[0.77926, 0.6092, 0.85699],
[0.78195, 0.60835, 0.85538],
[0.78463, 0.6075, 0.85375],
[0.78729, 0.60665, 0.85212],
[0.78993, 0.60579, 0.85048],
[0.79257, 0.60495, 0.84884],
[0.79518, 0.6041, 0.84719],
[0.79779, 0.60325, 0.84554],
[0.80038, 0.6024, 0.84387],
[0.80295, 0.60155, 0.84221],
[0.80551, 0.6007, 0.84054],
[0.80806, 0.59985, 0.83886],
[0.8106, 0.599, 0.83718],
[0.81312, 0.59814, 0.83549],
[0.81563, 0.5973, 0.83381],
[0.81812, 0.59644, 0.83211],
[0.82061, 0.59559, 0.83041],
[0.82308, 0.59474, 0.8287],
[0.82554, 0.59389, 0.82699],
[0.82798, 0.59304, 0.82528],
[0.83042, 0.59219, 0.82356],
[0.83284, 0.59133, 0.82184],
[0.83525, 0.59047, 0.82011],
[0.83765, 0.58962, 0.81838],
[0.84003, 0.58876, 0.81665],
[0.84241, 0.58791, 0.81491],
[0.84477, 0.58705, 0.81317],
[0.84712, 0.5862, 0.81143],
[0.84946, 0.58535, 0.80968],
[0.85179, 0.58449, 0.80793],
[0.8541, 0.58363, 0.80617],
[0.85641, 0.58277, 0.80442],
[0.85871, 0.58191, 0.80265],
[0.86099, 0.58105, 0.80089],
[0.86327, 0.5802, 0.79913],
[0.86553, 0.57934, 0.79735],
[0.86779, 0.57847, 0.79558],
[0.87002, 0.57761, 0.79381],
[0.87226, 0.57675, 0.79203],
[0.87448, 0.57589, 0.79025],
[0.87669, 0.57502, 0.78847],
[0.8789, 0.57415, 0.78668],
[0.88109, 0.57329, 0.78489],
[0.88328, 0.57243, 0.7831],
[0.88545, 0.57155, 0.78131],
[0.88761, 0.57068, 0.77951],
[0.88977, 0.56981, 0.77772],
[0.89192, 0.56895, 0.77591],
[0.89405, 0.56808, 0.77411],
[0.89618, 0.56721, 0.77231],
[0.8983, 0.56634, 0.7705],
[0.90041, 0.56546, 0.76869],
[0.90251, 0.56459, 0.76688],
[0.9046, 0.56372, 0.76507],
[0.90668, 0.56284, 0.76325],
[0.90876, 0.56197, 0.76143],
[0.91082, 0.56108, 0.75961],
[0.91288, 0.56021, 0.7578],
[0.91493, 0.55932, 0.75597],
[0.91697, 0.55844, 0.75415],
[0.91899, 0.55756, 0.75232],
[0.92102, 0.55668, 0.7505],
[0.92304, 0.55579, 0.74866],
[0.92504, 0.55491, 0.74683],
[0.92704, 0.55402, 0.745],
[0.92903, 0.55313, 0.74317],
[0.93102, 0.55225, 0.74133],
[0.93299, 0.55136, 0.7395],
[0.93496, 0.55047, 0.73765],
[0.93692, 0.54957, 0.73581],
[0.93887, 0.54869, 0.73397],
[0.94082, 0.54779, 0.73213],
[0.94276, 0.54689, 0.73028],
[0.94469, 0.546, 0.72843],
[0.94661, 0.5451, 0.72659],
[0.94852, 0.54419, 0.72474],
[0.95043, 0.5433, 0.72288],
[0.95233, 0.54239, 0.72103],
[0.95423, 0.54148, 0.71918],
[0.95611, 0.54058, 0.71733],
[0.95799, 0.53967, 0.71548],
[0.95986, 0.53878, 0.71361],
[0.96173, 0.53787, 0.71176],
[0.96359, 0.53696, 0.7099],
[0.96544, 0.53604, 0.70804],
[0.96729, 0.53513, 0.70618],
[0.96912, 0.53421, 0.70431],
[0.97096, 0.53329, 0.70245],
[0.97278, 0.53238, 0.70058],
[0.9746, 0.53146, 0.69871],
[0.97641, 0.53054, 0.69685],
[0.97822, 0.52962, 0.69498],
[0.98002, 0.5287, 0.69311],
[0.98181, 0.52777, 0.69124],
[0.9836, 0.52685, 0.68937],
[0.98538, 0.52592, 0.6875],
[0.98716, 0.52499, 0.68563],
[0.98892, 0.52406, 0.68375],
[0.99069, 0.52313, 0.68187],
[0.99244, 0.52218, 0.67999],
[0.99419, 0.52125, 0.67812],
]
b_isoluminant_cm_70_c39 = bokeh_palette('isoluminant_cm_70_c39',isoluminant_cm_70_c39)
m_isoluminant_cm_70_c39 = mpl_cm('isoluminant_cm_70_c39',isoluminant_cm_70_c39)
m_isoluminant_cm_70_c39_r = mpl_cm('isoluminant_cm_70_c39_r',list(reversed(isoluminant_cm_70_c39)))
CET_I3 = b_isoluminant_cm_70_c39
m_CET_I3 = m_isoluminant_cm_70_c39
m_CET_I3_r = m_isoluminant_cm_70_c39_r
palette['CET_I3'] = b_isoluminant_cm_70_c39
cm['CET_I3'] = m_isoluminant_cm_70_c39
cm['CET_I3_r'] = m_isoluminant_cm_70_c39_r
register_cmap('cet_CET_I3',m_isoluminant_cm_70_c39)
register_cmap('cet_CET_I3_r',m_isoluminant_cm_70_c39_r)




diverging_linear_bjr_30_55_c53 = [  # cmap_def
[0, 0.1654, 0.84384],
[0, 0.16745, 0.84024],
[0, 0.16948, 0.83663],
[0, 0.17145, 0.83302],
[0, 0.17344, 0.82943],
[0, 0.17543, 0.82582],
[0, 0.17744, 0.82223],
[0, 0.1794, 0.81863],
[0.025922, 0.18135, 0.81503],
[0.052828, 0.18332, 0.81145],
[0.073384, 0.18532, 0.80785],
[0.089868, 0.18726, 0.80427],
[0.10397, 0.1892, 0.80068],
[0.11641, 0.19113, 0.79709],
[0.12754, 0.19308, 0.79352],
[0.13776, 0.19502, 0.78993],
[0.14712, 0.19696, 0.78636],
[0.15586, 0.19885, 0.78277],
[0.16403, 0.20076, 0.7792],
[0.17168, 0.20269, 0.77562],
[0.17895, 0.20462, 0.77205],
[0.18585, 0.20653, 0.76848],
[0.19237, 0.20841, 0.76492],
[0.19861, 0.2103, 0.76135],
[0.20461, 0.21221, 0.75779],
[0.21032, 0.21409, 0.75423],
[0.21583, 0.21599, 0.75066],
[0.22112, 0.21785, 0.7471],
[0.22622, 0.21973, 0.74355],
[0.23112, 0.22159, 0.73999],
[0.23587, 0.22347, 0.73643],
[0.24042, 0.22532, 0.73288],
[0.24486, 0.22719, 0.72933],
[0.24917, 0.22905, 0.72578],
[0.25333, 0.2309, 0.72223],
[0.25737, 0.23274, 0.71868],
[0.26128, 0.23461, 0.71514],
[0.26508, 0.23645, 0.7116],
[0.2688, 0.23827, 0.70806],
[0.27236, 0.2401, 0.70452],
[0.27589, 0.24194, 0.70098],
[0.27929, 0.24376, 0.69744],
[0.2826, 0.24558, 0.6939],
[0.2858, 0.24742, 0.69037],
[0.28896, 0.24924, 0.68684],
[0.29202, 0.25106, 0.68331],
[0.29499, 0.25287, 0.67978],
[0.2979, 0.2547, 0.67625],
[0.30074, 0.25648, 0.67273],
[0.3035, 0.25829, 0.6692],
[0.30623, 0.2601, 0.66568],
[0.30886, 0.2619, 0.66216],
[0.31142, 0.26371, 0.65863],
[0.31394, 0.26549, 0.65511],
[0.31639, 0.26726, 0.6516],
[0.31878, 0.26908, 0.64807],
[0.32112, 0.27087, 0.64456],
[0.3234, 0.27263, 0.64104],
[0.32562, 0.27442, 0.63753],
[0.3278, 0.27621, 0.63402],
[0.32994, 0.27798, 0.63051],
[0.33201, 0.27975, 0.62699],
[0.33404, 0.28152, 0.62349],
[0.33602, 0.28329, 0.61998],
[0.33795, 0.28504, 0.61647],
[0.33985, 0.28681, 0.61296],
[0.34169, 0.28858, 0.60946],
[0.34349, 0.29033, 0.60594],
[0.34525, 0.2921, 0.60244],
[0.34698, 0.29385, 0.59894],
[0.34865, 0.2956, 0.59544],
[0.3503, 0.29736, 0.59194],
[0.35188, 0.29912, 0.58843],
[0.35345, 0.30085, 0.58494],
[0.35499, 0.30259, 0.58143],
[0.35647, 0.30434, 0.57793],
[0.35792, 0.3061, 0.57443],
[0.35932, 0.30781, 0.57094],
[0.36069, 0.30957, 0.56744],
[0.36205, 0.31129, 0.56395],
[0.36335, 0.31302, 0.56045],
[0.36463, 0.31475, 0.55696],
[0.36588, 0.31649, 0.55345],
[0.36709, 0.31821, 0.54996],
[0.36827, 0.31994, 0.54647],
[0.36941, 0.32165, 0.54297],
[0.37053, 0.32338, 0.53947],
[0.37161, 0.32509, 0.53597],
[0.37267, 0.32681, 0.53248],
[0.3737, 0.32854, 0.52899],
[0.37469, 0.33025, 0.52549],
[0.37565, 0.33196, 0.52198],
[0.3766, 0.33368, 0.5185],
[0.3775, 0.33539, 0.51499],
[0.37838, 0.33709, 0.51149],
[0.37924, 0.33879, 0.508],
[0.38006, 0.3405, 0.50451],
[0.38087, 0.3422, 0.501],
[0.38163, 0.34389, 0.4975],
[0.38237, 0.34559, 0.494],
[0.3831, 0.3473, 0.4905],
[0.38379, 0.34899, 0.48699],
[0.38445, 0.35069, 0.48348],
[0.38508, 0.35237, 0.47998],
[0.3857, 0.35406, 0.47647],
[0.38629, 0.35576, 0.47296],
[0.38685, 0.35745, 0.46946],
[0.38739, 0.35912, 0.46593],
[0.38791, 0.3608, 0.46242],
[0.3884, 0.3625, 0.45891],
[0.38888, 0.36417, 0.45539],
[0.38931, 0.36586, 0.45188],
[0.38973, 0.36753, 0.44835],
[0.39012, 0.36922, 0.44483],
[0.3905, 0.37088, 0.4413],
[0.39085, 0.37256, 0.43778],
[0.39118, 0.37425, 0.43423],
[0.39148, 0.37591, 0.4307],
[0.39176, 0.37758, 0.42717],
[0.39201, 0.37925, 0.42362],
[0.39224, 0.38092, 0.42008],
[0.39246, 0.38258, 0.41653],
[0.39264, 0.38425, 0.41298],
[0.39281, 0.3859, 0.40942],
[0.39296, 0.38757, 0.40587],
[0.39308, 0.38924, 0.40228],
[0.39318, 0.3909, 0.39872],
[0.39326, 0.39255, 0.39515],
[0.39669, 0.39314, 0.39214],
[0.40343, 0.39266, 0.38969],
[0.41007, 0.39217, 0.38724],
[0.41662, 0.39167, 0.3848],
[0.42308, 0.39116, 0.38235],
[0.42947, 0.39062, 0.37991],
[0.4358, 0.39007, 0.37746],
[0.44205, 0.38952, 0.37501],
[0.44822, 0.38896, 0.37255],
[0.45434, 0.38838, 0.3701],
[0.46038, 0.38778, 0.36765],
[0.46638, 0.38717, 0.3652],
[0.47233, 0.38655, 0.36274],
[0.47821, 0.38592, 0.36028],
[0.48404, 0.38527, 0.35784],
[0.48983, 0.38461, 0.35538],
[0.49557, 0.38394, 0.35291],
[0.50127, 0.38326, 0.35045],
[0.50692, 0.38255, 0.34798],
[0.51253, 0.38183, 0.34551],
[0.5181, 0.38111, 0.34305],
[0.52363, 0.38036, 0.34058],
[0.52913, 0.3796, 0.33809],
[0.53458, 0.37882, 0.33563],
[0.54, 0.37803, 0.33316],
[0.54539, 0.37723, 0.33067],
[0.55075, 0.37641, 0.32819],
[0.55607, 0.37556, 0.3257],
[0.56137, 0.37471, 0.32323],
[0.56664, 0.37384, 0.32075],
[0.57188, 0.37295, 0.31825],
[0.57709, 0.37204, 0.31576],
[0.58227, 0.37112, 0.31326],
[0.58743, 0.37018, 0.31077],
[0.59258, 0.36924, 0.30826],
[0.59768, 0.36827, 0.30577],
[0.60277, 0.36727, 0.30324],
[0.60785, 0.36627, 0.30073],
[0.61288, 0.36524, 0.29821],
[0.61791, 0.36419, 0.2957],
[0.62292, 0.36313, 0.29318],
[0.6279, 0.36205, 0.29066],
[0.63287, 0.36094, 0.28813],
[0.63782, 0.35983, 0.28559],
[0.64275, 0.35868, 0.28307],
[0.64766, 0.35754, 0.28051],
[0.65255, 0.35635, 0.27798],
[0.65744, 0.35515, 0.27544],
[0.6623, 0.35391, 0.27288],
[0.66714, 0.35267, 0.27033],
[0.67198, 0.3514, 0.26774],
[0.6768, 0.35012, 0.26518],
[0.6816, 0.3488, 0.26261],
[0.68639, 0.34747, 0.26004],
[0.69116, 0.3461, 0.25745],
[0.69594, 0.34472, 0.25487],
[0.70069, 0.34332, 0.25226],
[0.70542, 0.34189, 0.24964],
[0.71015, 0.34043, 0.24704],
[0.71486, 0.33896, 0.24441],
[0.71957, 0.33745, 0.24181],
[0.72426, 0.33591, 0.23917],
[0.72895, 0.33436, 0.23654],
[0.73362, 0.33277, 0.23387],
[0.73828, 0.33113, 0.23124],
[0.74294, 0.32951, 0.22858],
[0.74758, 0.32782, 0.2259],
[0.75221, 0.32611, 0.22322],
[0.75684, 0.32438, 0.22053],
[0.76145, 0.32263, 0.21781],
[0.76606, 0.32083, 0.21509],
[0.77066, 0.31899, 0.21238],
[0.77525, 0.31713, 0.20964],
[0.77984, 0.31523, 0.20691],
[0.78442, 0.31331, 0.20416],
[0.78899, 0.31134, 0.20135],
[0.79355, 0.30935, 0.19857],
[0.7981, 0.30729, 0.19578],
[0.80264, 0.30522, 0.19297],
[0.80719, 0.30309, 0.19014],
[0.81172, 0.30093, 0.18732],
[0.81625, 0.29874, 0.18444],
[0.82077, 0.2965, 0.18155],
[0.82529, 0.2942, 0.17869],
[0.8298, 0.29189, 0.17574],
[0.83431, 0.28951, 0.17282],
[0.8388, 0.28707, 0.16989],
[0.84329, 0.2846, 0.16692],
[0.84778, 0.28209, 0.16393],
[0.85227, 0.27951, 0.16088],
[0.85675, 0.27688, 0.15783],
[0.86122, 0.27419, 0.15475],
[0.86569, 0.27144, 0.15165],
[0.87015, 0.26865, 0.14851],
[0.87462, 0.26578, 0.1454],
[0.87907, 0.26285, 0.14218],
[0.88353, 0.25985, 0.13893],
[0.88797, 0.25678, 0.13565],
[0.89242, 0.25364, 0.13237],
[0.89686, 0.25041, 0.12899],
[0.9013, 0.24713, 0.12563],
[0.90573, 0.24374, 0.12213],
[0.91016, 0.24028, 0.11869],
[0.91459, 0.23676, 0.11509],
[0.91901, 0.23308, 0.11151],
[0.92343, 0.22934, 0.10779],
[0.92784, 0.22549, 0.10405],
[0.93226, 0.22153, 0.1002],
[0.93668, 0.21745, 0.096258],
[0.94108, 0.21323, 0.092292],
[0.94549, 0.2089, 0.088179],
[0.94989, 0.20443, 0.083966],
[0.95429, 0.19976, 0.079537],
[0.95869, 0.19499, 0.075045],
[0.96309, 0.19001, 0.070373],
[0.96749, 0.18486, 0.065518],
[0.97188, 0.17948, 0.06046],
[0.97626, 0.17389, 0.055081],
[0.98065, 0.16803, 0.049511],
[0.98504, 0.16195, 0.043373],
[0.98943, 0.15551, 0.037065],
[0.99381, 0.14872, 0.03044],
[0.99819, 0.14159, 0.024101],
[1, 0.13398, 0.017854],
[1, 0.12585, 0.011691],
[1, 0.11706, 0.0054342],
[1, 0.10757, 0],
[1, 0.097091, 0],
]
b_diverging_linear_bjr_30_55_c53 = bokeh_palette('diverging_linear_bjr_30_55_c53',diverging_linear_bjr_30_55_c53)
m_diverging_linear_bjr_30_55_c53 = mpl_cm('diverging_linear_bjr_30_55_c53',diverging_linear_bjr_30_55_c53)
m_diverging_linear_bjr_30_55_c53_r = mpl_cm('diverging_linear_bjr_30_55_c53_r',list(reversed(diverging_linear_bjr_30_55_c53)))
CET_D8 = b_diverging_linear_bjr_30_55_c53
m_CET_D8 = m_diverging_linear_bjr_30_55_c53
m_CET_D8_r = m_diverging_linear_bjr_30_55_c53_r
palette['CET_D8'] = b_diverging_linear_bjr_30_55_c53
cm['CET_D8'] = m_diverging_linear_bjr_30_55_c53
cm['CET_D8_r'] = m_diverging_linear_bjr_30_55_c53_r
register_cmap('cet_CET_D8',m_diverging_linear_bjr_30_55_c53)
register_cmap('cet_CET_D8_r',m_diverging_linear_bjr_30_55_c53_r)




diverging_gwr_55_95_c38 = [  # cmap_def
[0.22157, 0.59107, 0.055073],
[0.22965, 0.59389, 0.068993],
[0.2376, 0.59671, 0.081291],
[0.24531, 0.59952, 0.092565],
[0.25292, 0.60233, 0.10315],
[0.26037, 0.60514, 0.11313],
[0.26768, 0.60796, 0.12253],
[0.27491, 0.61076, 0.13168],
[0.28202, 0.61356, 0.14045],
[0.28902, 0.61637, 0.14894],
[0.29592, 0.61918, 0.15721],
[0.30273, 0.62197, 0.16528],
[0.3095, 0.62478, 0.1732],
[0.31616, 0.62758, 0.18095],
[0.32276, 0.63038, 0.18859],
[0.32929, 0.63317, 0.1961],
[0.33575, 0.63597, 0.20349],
[0.34216, 0.63876, 0.21079],
[0.34851, 0.64155, 0.21802],
[0.35482, 0.64434, 0.22514],
[0.36105, 0.64714, 0.23224],
[0.36726, 0.64993, 0.23924],
[0.37343, 0.65271, 0.24619],
[0.37954, 0.6555, 0.25311],
[0.38562, 0.65828, 0.25997],
[0.39167, 0.66106, 0.26678],
[0.39767, 0.66385, 0.27357],
[0.40364, 0.66662, 0.28029],
[0.40958, 0.66941, 0.287],
[0.41548, 0.67219, 0.29368],
[0.42135, 0.67496, 0.30034],
[0.42721, 0.67774, 0.30697],
[0.43302, 0.68051, 0.31358],
[0.43881, 0.68328, 0.32017],
[0.44459, 0.68606, 0.32672],
[0.45032, 0.68883, 0.33328],
[0.45604, 0.6916, 0.33982],
[0.46174, 0.69436, 0.34632],
[0.46742, 0.69713, 0.35283],
[0.47307, 0.6999, 0.35932],
[0.47871, 0.70267, 0.3658],
[0.48432, 0.70543, 0.37227],
[0.48992, 0.70819, 0.37873],
[0.49551, 0.71096, 0.38518],
[0.50108, 0.71371, 0.39164],
[0.50662, 0.71648, 0.39808],
[0.51215, 0.71923, 0.40451],
[0.51768, 0.72199, 0.41095],
[0.52318, 0.72475, 0.41738],
[0.52867, 0.7275, 0.42379],
[0.53414, 0.73026, 0.43022],
[0.5396, 0.73301, 0.43664],
[0.54506, 0.73576, 0.44305],
[0.5505, 0.73851, 0.44946],
[0.55592, 0.74126, 0.45587],
[0.56134, 0.74401, 0.46228],
[0.56675, 0.74676, 0.4687],
[0.57214, 0.7495, 0.47512],
[0.57752, 0.75225, 0.48152],
[0.58289, 0.755, 0.48794],
[0.58826, 0.75775, 0.49435],
[0.59362, 0.76049, 0.50076],
[0.59896, 0.76323, 0.50717],
[0.60431, 0.76597, 0.5136],
[0.60963, 0.76871, 0.52001],
[0.61495, 0.77146, 0.52644],
[0.62027, 0.77419, 0.53286],
[0.62557, 0.77693, 0.53928],
[0.63088, 0.77967, 0.54572],
[0.63617, 0.7824, 0.55215],
[0.64145, 0.78514, 0.55858],
[0.64673, 0.78787, 0.56502],
[0.652, 0.79061, 0.57147],
[0.65727, 0.79334, 0.57791],
[0.66253, 0.79607, 0.58437],
[0.66778, 0.7988, 0.59082],
[0.67303, 0.80153, 0.59729],
[0.67827, 0.80426, 0.60376],
[0.68351, 0.80699, 0.61022],
[0.68875, 0.80972, 0.6167],
[0.69397, 0.81244, 0.62318],
[0.69919, 0.81517, 0.62965],
[0.70441, 0.8179, 0.63615],
[0.70963, 0.82062, 0.64264],
[0.71483, 0.82335, 0.64914],
[0.72004, 0.82607, 0.65564],
[0.72525, 0.82879, 0.66215],
[0.73044, 0.83152, 0.66866],
[0.73563, 0.83424, 0.67518],
[0.74082, 0.83696, 0.6817],
[0.74601, 0.83968, 0.68824],
[0.7512, 0.84239, 0.69477],
[0.75637, 0.84511, 0.70131],
[0.76155, 0.84782, 0.70786],
[0.76672, 0.85054, 0.71441],
[0.77189, 0.85326, 0.72097],
[0.77706, 0.85597, 0.72754],
[0.78223, 0.85869, 0.73411],
[0.78739, 0.8614, 0.74068],
[0.79255, 0.86412, 0.74727],
[0.7977, 0.86682, 0.75386],
[0.80286, 0.86954, 0.76045],
[0.80801, 0.87225, 0.76705],
[0.81316, 0.87496, 0.77366],
[0.81831, 0.87767, 0.78027],
[0.82345, 0.88037, 0.7869],
[0.8286, 0.88308, 0.79353],
[0.83375, 0.88579, 0.80016],
[0.83889, 0.88849, 0.80678],
[0.84403, 0.89117, 0.8134],
[0.84917, 0.89385, 0.82001],
[0.8543, 0.8965, 0.82659],
[0.85943, 0.89913, 0.83314],
[0.86454, 0.90171, 0.83964],
[0.86965, 0.90424, 0.84606],
[0.87473, 0.9067, 0.85239],
[0.87979, 0.90907, 0.85857],
[0.88483, 0.91132, 0.8646],
[0.88981, 0.91343, 0.87039],
[0.89474, 0.91536, 0.87592],
[0.8996, 0.91708, 0.88112],
[0.90436, 0.91854, 0.88593],
[0.90902, 0.91973, 0.89029],
[0.91355, 0.92058, 0.89413],
[0.91792, 0.92108, 0.8974],
[0.92213, 0.9212, 0.90005],
[0.92614, 0.92091, 0.90203],
[0.92995, 0.9202, 0.90332],
[0.93353, 0.91905, 0.9039],
[0.9369, 0.91749, 0.90377],
[0.94002, 0.91549, 0.90294],
[0.94291, 0.91311, 0.90144],
[0.94556, 0.91036, 0.89932],
[0.94799, 0.90726, 0.89661],
[0.95021, 0.90386, 0.89338],
[0.95224, 0.90019, 0.88969],
[0.95409, 0.89629, 0.8856],
[0.95578, 0.89219, 0.88117],
[0.95733, 0.88793, 0.87646],
[0.95876, 0.88354, 0.87154],
[0.96008, 0.87904, 0.86642],
[0.96131, 0.87447, 0.86117],
[0.96246, 0.86983, 0.85582],
[0.96355, 0.86515, 0.85038],
[0.96458, 0.86044, 0.8449],
[0.96556, 0.8557, 0.83937],
[0.9665, 0.85093, 0.83383],
[0.9674, 0.84616, 0.82826],
[0.96826, 0.84138, 0.82269],
[0.96908, 0.83659, 0.81711],
[0.96988, 0.83181, 0.81154],
[0.97064, 0.82702, 0.80597],
[0.97138, 0.82223, 0.80041],
[0.9721, 0.81743, 0.79486],
[0.97278, 0.81263, 0.78932],
[0.97344, 0.80784, 0.78378],
[0.97407, 0.80303, 0.77824],
[0.97467, 0.79823, 0.77271],
[0.97524, 0.79344, 0.76719],
[0.97579, 0.78863, 0.76168],
[0.97632, 0.78382, 0.75618],
[0.97682, 0.77901, 0.75068],
[0.97729, 0.7742, 0.74518],
[0.97774, 0.76938, 0.7397],
[0.97816, 0.76457, 0.73422],
[0.97855, 0.75974, 0.72875],
[0.97893, 0.75493, 0.72328],
[0.97928, 0.7501, 0.71783],
[0.9796, 0.74527, 0.71237],
[0.9799, 0.74044, 0.70693],
[0.98017, 0.7356, 0.70149],
[0.98042, 0.73077, 0.69606],
[0.98064, 0.72593, 0.69063],
[0.98084, 0.72108, 0.68522],
[0.98102, 0.71624, 0.6798],
[0.98118, 0.71138, 0.6744],
[0.98131, 0.70653, 0.66901],
[0.98142, 0.70166, 0.66363],
[0.9815, 0.69679, 0.65824],
[0.98157, 0.69193, 0.65286],
[0.98161, 0.68705, 0.6475],
[0.98162, 0.68217, 0.64214],
[0.98162, 0.67729, 0.63679],
[0.98159, 0.6724, 0.63144],
[0.98154, 0.6675, 0.6261],
[0.98147, 0.6626, 0.62077],
[0.98138, 0.6577, 0.61544],
[0.98126, 0.65278, 0.61013],
[0.98113, 0.64786, 0.60482],
[0.98097, 0.64294, 0.59952],
[0.98079, 0.63801, 0.59422],
[0.98059, 0.63307, 0.58893],
[0.98037, 0.62813, 0.58366],
[0.98013, 0.62318, 0.57838],
[0.97987, 0.61821, 0.57313],
[0.97959, 0.61324, 0.56786],
[0.97929, 0.60827, 0.56262],
[0.97896, 0.60329, 0.55738],
[0.97862, 0.59829, 0.55214],
[0.97826, 0.59329, 0.54691],
[0.97788, 0.58827, 0.54169],
[0.97747, 0.58326, 0.53649],
[0.97705, 0.57822, 0.53128],
[0.97661, 0.57319, 0.52609],
[0.97615, 0.56814, 0.5209],
[0.97567, 0.56308, 0.51571],
[0.97517, 0.558, 0.51054],
[0.97465, 0.55292, 0.50537],
[0.97411, 0.54782, 0.50022],
[0.97355, 0.54272, 0.49508],
[0.97298, 0.5376, 0.48993],
[0.97239, 0.53246, 0.48479],
[0.97177, 0.52731, 0.47968],
[0.97114, 0.52213, 0.47456],
[0.97049, 0.51696, 0.46946],
[0.96982, 0.51176, 0.46436],
[0.96914, 0.50655, 0.45926],
[0.96843, 0.50133, 0.45417],
[0.96771, 0.49607, 0.4491],
[0.96697, 0.49082, 0.44403],
[0.96621, 0.48553, 0.43896],
[0.96544, 0.48023, 0.4339],
[0.96464, 0.47491, 0.42887],
[0.96383, 0.46957, 0.42382],
[0.96301, 0.4642, 0.4188],
[0.96216, 0.45881, 0.41378],
[0.9613, 0.45339, 0.40877],
[0.96042, 0.44795, 0.40377],
[0.95952, 0.4425, 0.39877],
[0.95861, 0.43701, 0.39379],
[0.95768, 0.43149, 0.38882],
[0.95673, 0.42593, 0.38384],
[0.95577, 0.42035, 0.37888],
[0.95479, 0.41474, 0.37393],
[0.9538, 0.4091, 0.36899],
[0.95278, 0.40342, 0.36405],
[0.95175, 0.3977, 0.35912],
[0.95071, 0.39194, 0.3542],
[0.94964, 0.38614, 0.3493],
[0.94857, 0.38031, 0.34439],
[0.94747, 0.37443, 0.33951],
[0.94636, 0.3685, 0.33463],
[0.94524, 0.36251, 0.32975],
[0.9441, 0.35648, 0.32487],
[0.94294, 0.3504, 0.32002],
[0.94177, 0.34423, 0.31516],
[0.94058, 0.33802, 0.31033],
[0.93938, 0.33175, 0.3055],
[0.93816, 0.3254, 0.30066],
[0.93693, 0.31899, 0.29584],
[0.93568, 0.31248, 0.29104],
[0.93441, 0.30592, 0.28623],
[0.93313, 0.29924, 0.28145],
[0.93184, 0.29246, 0.27668],
[0.93053, 0.28558, 0.27188],
[0.9292, 0.27862, 0.26712],
]
b_diverging_gwr_55_95_c38 = bokeh_palette('diverging_gwr_55_95_c38',diverging_gwr_55_95_c38)
m_diverging_gwr_55_95_c38 = mpl_cm('diverging_gwr_55_95_c38',diverging_gwr_55_95_c38)
m_diverging_gwr_55_95_c38_r = mpl_cm('diverging_gwr_55_95_c38_r',list(reversed(diverging_gwr_55_95_c38)))
CET_D3 = b_diverging_gwr_55_95_c38
m_CET_D3 = m_diverging_gwr_55_95_c38
m_CET_D3_r = m_diverging_gwr_55_95_c38_r
palette['CET_D3'] = b_diverging_gwr_55_95_c38
cm['CET_D3'] = m_diverging_gwr_55_95_c38
cm['CET_D3_r'] = m_diverging_gwr_55_95_c38_r
register_cmap('cet_CET_D3',m_diverging_gwr_55_95_c38)
register_cmap('cet_CET_D3_r',m_diverging_gwr_55_95_c38_r)




linear_bgy_10_95_c74 = [  # cmap_def
[0, 0.047803, 0.4883],
[0, 0.049756, 0.49556],
[0, 0.051605, 0.50281],
[0, 0.053545, 0.51004],
[0, 0.055585, 0.51721],
[0, 0.057566, 0.52435],
[0, 0.05978, 0.53144],
[0, 0.061812, 0.53849],
[0, 0.064016, 0.5455],
[0, 0.066232, 0.55245],
[0, 0.068551, 0.55934],
[0, 0.070824, 0.5662],
[0, 0.073229, 0.57299],
[0, 0.07557, 0.57971],
[0, 0.078003, 0.58638],
[0, 0.080554, 0.59299],
[0, 0.083114, 0.59951],
[0, 0.085697, 0.60596],
[0, 0.08842, 0.61236],
[0, 0.091168, 0.61866],
[0, 0.093925, 0.62487],
[0, 0.096707, 0.63101],
[0, 0.09963, 0.63705],
[0, 0.1026, 0.64298],
[0, 0.10561, 0.64881],
[0, 0.10866, 0.65454],
[0, 0.11183, 0.66016],
[0, 0.11497, 0.66567],
[0, 0.11829, 0.67103],
[0, 0.12156, 0.67626],
[0, 0.12498, 0.68134],
[0, 0.12846, 0.68629],
[0, 0.13201, 0.69107],
[0, 0.13559, 0.6957],
[0, 0.13927, 0.70014],
[0, 0.14307, 0.70439],
[0, 0.1469, 0.70845],
[0, 0.15085, 0.71227],
[0, 0.15487, 0.71588],
[0, 0.159, 0.71923],
[0, 0.16323, 0.7223],
[0, 0.16754, 0.7251],
[0, 0.17195, 0.72757],
[0, 0.17647, 0.72972],
[0, 0.18113, 0.73149],
[0, 0.18594, 0.73289],
[0, 0.19074, 0.73398],
[0, 0.19556, 0.73486],
[0, 0.20033, 0.73556],
[0, 0.20512, 0.73608],
[0, 0.20987, 0.73643],
[0, 0.21461, 0.73659],
[0, 0.21934, 0.73657],
[0, 0.22402, 0.73637],
[0, 0.22875, 0.73599],
[0, 0.2334, 0.73544],
[0, 0.23809, 0.73469],
[0, 0.24275, 0.73376],
[0, 0.24743, 0.73266],
[0, 0.25208, 0.73137],
[0, 0.25673, 0.72991],
[0, 0.26137, 0.72825],
[0, 0.26603, 0.72642],
[0, 0.27068, 0.72441],
[0, 0.27531, 0.72221],
[0, 0.27995, 0.71983],
[0, 0.28458, 0.71727],
[0, 0.28924, 0.71452],
[0, 0.29387, 0.71161],
[0, 0.29852, 0.70851],
[0, 0.30317, 0.70521],
[0, 0.30782, 0.70174],
[0, 0.31248, 0.69809],
[0, 0.31716, 0.69426],
[0, 0.32182, 0.69025],
[0, 0.32649, 0.68607],
[0, 0.33116, 0.68178],
[0, 0.33582, 0.67746],
[0, 0.34046, 0.6731],
[0, 0.34509, 0.66871],
[0, 0.3497, 0.66429],
[0, 0.3543, 0.65984],
[0, 0.35888, 0.65536],
[0, 0.36346, 0.65085],
[0, 0.36803, 0.6463],
[0, 0.37258, 0.64173],
[0, 0.37713, 0.63713],
[0, 0.38167, 0.6325],
[0, 0.38618, 0.62783],
[0, 0.39071, 0.62313],
[0, 0.39523, 0.6184],
[0, 0.39972, 0.61363],
[0, 0.40423, 0.60885],
[0, 0.40872, 0.60402],
[0, 0.41321, 0.59915],
[0, 0.41769, 0.59426],
[0, 0.42215, 0.58932],
[0, 0.42663, 0.58437],
[0, 0.4311, 0.57937],
[0, 0.43556, 0.57433],
[0, 0.44001, 0.56926],
[0, 0.44446, 0.56416],
[0, 0.44891, 0.55902],
[0, 0.45334, 0.55384],
[0, 0.45778, 0.54863],
[0, 0.46222, 0.54336],
[0, 0.46665, 0.53802],
[0, 0.47105, 0.53253],
[0, 0.47545, 0.52691],
[0, 0.47982, 0.52115],
[0, 0.48417, 0.51525],
[0, 0.48852, 0.50921],
[0, 0.49284, 0.50301],
[0, 0.49717, 0.49668],
[0, 0.50147, 0.49022],
[0, 0.50575, 0.48359],
[0, 0.51003, 0.47682],
[0, 0.51431, 0.4699],
[0, 0.51856, 0.4628],
[0.0097866, 0.52281, 0.45558],
[0.023896, 0.52704, 0.44818],
[0.038383, 0.53126, 0.44061],
[0.051763, 0.53547, 0.43289],
[0.063442, 0.53968, 0.42499],
[0.073828, 0.54388, 0.41692],
[0.083244, 0.54807, 0.40866],
[0.092062, 0.55225, 0.40022],
[0.10019, 0.55642, 0.39159],
[0.10786, 0.56059, 0.38276],
[0.11513, 0.56474, 0.37372],
[0.12206, 0.56889, 0.36445],
[0.12871, 0.57304, 0.35498],
[0.13507, 0.57718, 0.34524],
[0.14115, 0.58131, 0.33527],
[0.14697, 0.58544, 0.32499],
[0.15257, 0.58954, 0.31449],
[0.15773, 0.59367, 0.30393],
[0.16231, 0.59779, 0.29352],
[0.16631, 0.60191, 0.28332],
[0.16984, 0.60603, 0.27332],
[0.17292, 0.61015, 0.26352],
[0.17565, 0.61427, 0.25387],
[0.17811, 0.6184, 0.24439],
[0.18021, 0.62252, 0.23514],
[0.18207, 0.62664, 0.22606],
[0.18374, 0.63076, 0.21715],
[0.18522, 0.63487, 0.2084],
[0.18649, 0.63898, 0.19982],
[0.18765, 0.64309, 0.19148],
[0.18863, 0.6472, 0.18334],
[0.18951, 0.6513, 0.1754],
[0.19029, 0.65539, 0.16767],
[0.19099, 0.65948, 0.16013],
[0.19162, 0.66357, 0.15293],
[0.19219, 0.66765, 0.14604],
[0.19272, 0.67172, 0.13937],
[0.1932, 0.67579, 0.13311],
[0.19366, 0.67985, 0.1272],
[0.1941, 0.68391, 0.1218],
[0.19451, 0.68797, 0.11692],
[0.1949, 0.69202, 0.11259],
[0.19529, 0.69606, 0.10881],
[0.19569, 0.70009, 0.10581],
[0.19611, 0.70412, 0.10358],
[0.19653, 0.70814, 0.10209],
[0.19694, 0.71215, 0.10139],
[0.19736, 0.71617, 0.10116],
[0.19779, 0.72018, 0.10101],
[0.19823, 0.7242, 0.10087],
[0.19868, 0.72822, 0.10073],
[0.19914, 0.73225, 0.1006],
[0.19961, 0.73627, 0.10048],
[0.20009, 0.74031, 0.10036],
[0.20058, 0.74434, 0.10025],
[0.20108, 0.74838, 0.10015],
[0.20159, 0.75242, 0.10006],
[0.20211, 0.75647, 0.099977],
[0.20265, 0.76052, 0.099902],
[0.2032, 0.76457, 0.099835],
[0.20376, 0.76862, 0.099777],
[0.20433, 0.77267, 0.099729],
[0.20488, 0.77674, 0.099691],
[0.20546, 0.7808, 0.099663],
[0.20608, 0.78487, 0.099645],
[0.20669, 0.78894, 0.099637],
[0.20729, 0.79301, 0.099641],
[0.20791, 0.79708, 0.099656],
[0.20855, 0.80116, 0.099683],
[0.2092, 0.80523, 0.09972],
[0.20987, 0.80932, 0.09977],
[0.21055, 0.8134, 0.099833],
[0.21125, 0.81749, 0.099908],
[0.21196, 0.82159, 0.099996],
[0.21266, 0.82567, 0.1001],
[0.2134, 0.82977, 0.10021],
[0.21454, 0.83386, 0.10034],
[0.21746, 0.83784, 0.10049],
[0.22334, 0.84166, 0.10065],
[0.23182, 0.84531, 0.10083],
[0.24228, 0.84884, 0.10102],
[0.25428, 0.85224, 0.10123],
[0.26735, 0.85553, 0.10145],
[0.28125, 0.85872, 0.10168],
[0.29571, 0.86184, 0.10193],
[0.31067, 0.86485, 0.10219],
[0.32594, 0.86779, 0.10246],
[0.34137, 0.87065, 0.10275],
[0.35684, 0.87346, 0.10306],
[0.3724, 0.87619, 0.10337],
[0.38805, 0.87886, 0.10369],
[0.40366, 0.88147, 0.10401],
[0.41921, 0.88402, 0.10433],
[0.43478, 0.88651, 0.10468],
[0.45028, 0.88894, 0.10507],
[0.4657, 0.89133, 0.10549],
[0.48111, 0.89365, 0.10592],
[0.49641, 0.89593, 0.10637],
[0.51169, 0.89815, 0.10681],
[0.52691, 0.90032, 0.10725],
[0.54202, 0.90245, 0.1077],
[0.55713, 0.90452, 0.10817],
[0.57212, 0.90655, 0.10869],
[0.5871, 0.90852, 0.10927],
[0.60201, 0.91045, 0.10983],
[0.61689, 0.91232, 0.11037],
[0.63169, 0.91415, 0.11095],
[0.64646, 0.91593, 0.11155],
[0.66118, 0.91767, 0.11216],
[0.67585, 0.91934, 0.11281],
[0.69049, 0.92098, 0.11347],
[0.70508, 0.92257, 0.11409],
[0.71966, 0.92411, 0.11477],
[0.73418, 0.9256, 0.11552],
[0.74868, 0.92704, 0.11627],
[0.76314, 0.92844, 0.11699],
[0.77759, 0.92979, 0.11777],
[0.79201, 0.93109, 0.11859],
[0.8064, 0.93233, 0.11937],
[0.82077, 0.93353, 0.12019],
[0.83511, 0.93468, 0.12102],
[0.84946, 0.93578, 0.12189],
[0.86375, 0.93684, 0.12278],
[0.87808, 0.93783, 0.1237],
[0.89234, 0.93878, 0.12464],
[0.90664, 0.93968, 0.12562],
[0.92088, 0.94052, 0.12657],
[0.93514, 0.94131, 0.12755],
[0.94939, 0.94206, 0.12857],
[0.9636, 0.94275, 0.12961],
[0.97785, 0.94338, 0.13068],
[0.99205, 0.94397, 0.13172],
[1, 0.94449, 0.13281],
[1, 0.94497, 0.13392],
[1, 0.94539, 0.13505],
[1, 0.94574, 0.13614],
[1, 0.94606, 0.13735],
]
b_linear_bgy_10_95_c74 = bokeh_palette('linear_bgy_10_95_c74',linear_bgy_10_95_c74)
m_linear_bgy_10_95_c74 = mpl_cm('linear_bgy_10_95_c74',linear_bgy_10_95_c74)
m_linear_bgy_10_95_c74_r = mpl_cm('linear_bgy_10_95_c74_r',list(reversed(linear_bgy_10_95_c74)))
bgy = b_linear_bgy_10_95_c74
m_bgy = m_linear_bgy_10_95_c74
m_bgy_r = m_linear_bgy_10_95_c74_r
palette['bgy'] = b_linear_bgy_10_95_c74
palette_n['bgy'] = b_linear_bgy_10_95_c74
cm['bgy'] = m_linear_bgy_10_95_c74
cm['bgy_r'] = m_linear_bgy_10_95_c74_r
cm_n['bgy'] = mpl_cm('bgy',linear_bgy_10_95_c74)
cm_n['bgy_r'] = mpl_cm('bgy_r',list(reversed(linear_bgy_10_95_c74)))




cyclic_grey_15_85_c0 = [  # cmap_def
[0.46877, 0.46888, 0.46885],
[0.47381, 0.47391, 0.47389],
[0.47887, 0.47897, 0.47895],
[0.48395, 0.48405, 0.48403],
[0.48907, 0.48917, 0.48914],
[0.49422, 0.49432, 0.4943],
[0.49939, 0.49949, 0.49947],
[0.50462, 0.50472, 0.5047],
[0.50986, 0.50997, 0.50994],
[0.51515, 0.51526, 0.51523],
[0.52047, 0.52058, 0.52056],
[0.52583, 0.52594, 0.52592],
[0.53121, 0.53132, 0.5313],
[0.53663, 0.53674, 0.53671],
[0.54205, 0.54217, 0.54214],
[0.5475, 0.54762, 0.54759],
[0.55297, 0.55309, 0.55306],
[0.55846, 0.55857, 0.55855],
[0.56397, 0.56409, 0.56406],
[0.56948, 0.5696, 0.56957],
[0.57501, 0.57513, 0.5751],
[0.58055, 0.58067, 0.58064],
[0.58611, 0.58623, 0.5862],
[0.59167, 0.59179, 0.59176],
[0.59724, 0.59737, 0.59734],
[0.60282, 0.60295, 0.60292],
[0.60842, 0.60855, 0.60852],
[0.61401, 0.61414, 0.6141],
[0.61963, 0.61976, 0.61973],
[0.62524, 0.62537, 0.62534],
[0.63088, 0.63101, 0.63098],
[0.63652, 0.63665, 0.63662],
[0.64216, 0.64229, 0.64226],
[0.64781, 0.64795, 0.64791],
[0.65348, 0.65361, 0.65358],
[0.65915, 0.65929, 0.65926],
[0.66484, 0.66497, 0.66494],
[0.67054, 0.67067, 0.67064],
[0.67624, 0.67637, 0.67634],
[0.68194, 0.68208, 0.68205],
[0.68766, 0.68781, 0.68777],
[0.69339, 0.69353, 0.6935],
[0.69913, 0.69927, 0.69923],
[0.70487, 0.70502, 0.70498],
[0.71062, 0.71076, 0.71073],
[0.71636, 0.71651, 0.71647],
[0.72208, 0.72223, 0.72219],
[0.72779, 0.72794, 0.72791],
[0.73347, 0.73362, 0.73359],
[0.73911, 0.73926, 0.73922],
[0.74467, 0.74482, 0.74478],
[0.75014, 0.75029, 0.75026],
[0.75549, 0.75565, 0.75561],
[0.76068, 0.76083, 0.7608],
[0.76567, 0.76582, 0.76578],
[0.77039, 0.77055, 0.77051],
[0.77481, 0.77497, 0.77493],
[0.77887, 0.77903, 0.77899],
[0.7825, 0.78266, 0.78262],
[0.78566, 0.78581, 0.78578],
[0.78827, 0.78843, 0.78839],
[0.79029, 0.79045, 0.79041],
[0.79169, 0.79185, 0.79182],
[0.79244, 0.7926, 0.79257],
[0.79252, 0.79269, 0.79265],
[0.79193, 0.79209, 0.79205],
[0.79069, 0.79085, 0.79081],
[0.78881, 0.78897, 0.78893],
[0.78633, 0.78649, 0.78645],
[0.7833, 0.78346, 0.78342],
[0.77978, 0.77994, 0.7799],
[0.77582, 0.77598, 0.77594],
[0.77149, 0.77164, 0.7716],
[0.76682, 0.76698, 0.76694],
[0.76189, 0.76205, 0.76201],
[0.75675, 0.75691, 0.75687],
[0.75145, 0.7516, 0.75156],
[0.746, 0.74615, 0.74611],
[0.74045, 0.7406, 0.74056],
[0.73484, 0.73499, 0.73495],
[0.72917, 0.72932, 0.72928],
[0.72346, 0.72361, 0.72357],
[0.71774, 0.71789, 0.71785],
[0.712, 0.71215, 0.71211],
[0.70626, 0.70641, 0.70637],
[0.70052, 0.70066, 0.70062],
[0.69477, 0.69492, 0.69488],
[0.68905, 0.68919, 0.68916],
[0.68333, 0.68346, 0.68343],
[0.67762, 0.67776, 0.67772],
[0.67191, 0.67205, 0.67201],
[0.66621, 0.66635, 0.66631],
[0.66052, 0.66066, 0.66063],
[0.65485, 0.65498, 0.65495],
[0.64919, 0.64932, 0.64929],
[0.64352, 0.64365, 0.64362],
[0.63787, 0.63801, 0.63797],
[0.63223, 0.63236, 0.63233],
[0.6266, 0.62673, 0.6267],
[0.62098, 0.62111, 0.62108],
[0.61536, 0.61549, 0.61546],
[0.60976, 0.60989, 0.60986],
[0.60418, 0.6043, 0.60427],
[0.59858, 0.59871, 0.59868],
[0.59302, 0.59314, 0.59311],
[0.58744, 0.58756, 0.58753],
[0.58189, 0.58201, 0.58198],
[0.57635, 0.57647, 0.57644],
[0.57081, 0.57092, 0.5709],
[0.56528, 0.56539, 0.56537],
[0.55977, 0.55989, 0.55986],
[0.55426, 0.55438, 0.55435],
[0.54877, 0.54888, 0.54886],
[0.54328, 0.54339, 0.54336],
[0.53781, 0.53792, 0.53789],
[0.53234, 0.53245, 0.53242],
[0.52688, 0.52699, 0.52697],
[0.52143, 0.52154, 0.52151],
[0.51599, 0.5161, 0.51607],
[0.51057, 0.51067, 0.51065],
[0.50515, 0.50526, 0.50523],
[0.49974, 0.49985, 0.49982],
[0.49436, 0.49446, 0.49444],
[0.48897, 0.48907, 0.48905],
[0.48359, 0.4837, 0.48367],
[0.47824, 0.47834, 0.47831],
[0.47288, 0.47298, 0.47295],
[0.46755, 0.46764, 0.46762],
[0.46221, 0.46231, 0.46229],
[0.45689, 0.45699, 0.45697],
[0.45159, 0.45169, 0.45166],
[0.44629, 0.44638, 0.44636],
[0.441, 0.44109, 0.44107],
[0.43574, 0.43583, 0.43581],
[0.43047, 0.43056, 0.43054],
[0.42522, 0.42531, 0.42529],
[0.41998, 0.42007, 0.42004],
[0.41475, 0.41483, 0.41481],
[0.40953, 0.40962, 0.4096],
[0.40432, 0.40441, 0.40439],
[0.39912, 0.39921, 0.39919],
[0.39395, 0.39403, 0.39401],
[0.38878, 0.38887, 0.38885],
[0.38363, 0.38371, 0.38369],
[0.37848, 0.37856, 0.37854],
[0.37334, 0.37342, 0.3734],
[0.36822, 0.36831, 0.36829],
[0.36311, 0.36319, 0.36317],
[0.35802, 0.3581, 0.35808],
[0.35294, 0.35302, 0.353],
[0.34787, 0.34794, 0.34793],
[0.34282, 0.34289, 0.34287],
[0.33776, 0.33783, 0.33781],
[0.33274, 0.33281, 0.3328],
[0.3277, 0.32778, 0.32776],
[0.32271, 0.32278, 0.32276],
[0.31771, 0.31778, 0.31776],
[0.31272, 0.31279, 0.31277],
[0.30775, 0.30782, 0.3078],
[0.3028, 0.30286, 0.30285],
[0.29786, 0.29793, 0.29791],
[0.29294, 0.293, 0.29299],
[0.28802, 0.28809, 0.28807],
[0.28314, 0.2832, 0.28319],
[0.27826, 0.27832, 0.2783],
[0.27339, 0.27345, 0.27344],
[0.26853, 0.2686, 0.26858],
[0.2637, 0.26376, 0.26375],
[0.25887, 0.25893, 0.25892],
[0.25406, 0.25412, 0.2541],
[0.24925, 0.24931, 0.2493],
[0.24446, 0.24452, 0.24451],
[0.23973, 0.23978, 0.23977],
[0.23502, 0.23508, 0.23506],
[0.2303, 0.23036, 0.23034],
[0.22568, 0.22573, 0.22572],
[0.22109, 0.22114, 0.22113],
[0.21656, 0.21662, 0.2166],
[0.21212, 0.21217, 0.21216],
[0.20777, 0.20782, 0.20781],
[0.20358, 0.20363, 0.20361],
[0.19951, 0.19955, 0.19954],
[0.19568, 0.19573, 0.19572],
[0.19207, 0.19212, 0.19211],
[0.18876, 0.1888, 0.18879],
[0.18577, 0.18581, 0.1858],
[0.18308, 0.18313, 0.18312],
[0.18084, 0.18089, 0.18087],
[0.17906, 0.1791, 0.17909],
[0.17775, 0.1778, 0.17779],
[0.17689, 0.17694, 0.17693],
[0.17656, 0.1766, 0.17659],
[0.17676, 0.17681, 0.1768],
[0.1775, 0.17755, 0.17754],
[0.17871, 0.17875, 0.17874],
[0.18037, 0.18041, 0.1804],
[0.1825, 0.18255, 0.18254],
[0.18509, 0.18514, 0.18512],
[0.18802, 0.18807, 0.18806],
[0.19124, 0.19129, 0.19128],
[0.19481, 0.19485, 0.19484],
[0.19857, 0.19862, 0.19861],
[0.20257, 0.20262, 0.20261],
[0.20676, 0.20681, 0.2068],
[0.21105, 0.2111, 0.21109],
[0.21547, 0.21552, 0.21551],
[0.21999, 0.22004, 0.22003],
[0.22453, 0.22458, 0.22457],
[0.22919, 0.22924, 0.22923],
[0.23386, 0.23392, 0.2339],
[0.23857, 0.23863, 0.23862],
[0.24332, 0.24337, 0.24336],
[0.24812, 0.24817, 0.24816],
[0.2529, 0.25295, 0.25294],
[0.25769, 0.25775, 0.25774],
[0.26252, 0.26258, 0.26257],
[0.26734, 0.2674, 0.26739],
[0.27219, 0.27226, 0.27224],
[0.27708, 0.27714, 0.27713],
[0.28196, 0.28203, 0.28201],
[0.28684, 0.2869, 0.28688],
[0.29176, 0.29183, 0.29181],
[0.29668, 0.29675, 0.29674],
[0.3016, 0.30167, 0.30165],
[0.30657, 0.30664, 0.30662],
[0.31152, 0.31159, 0.31157],
[0.31651, 0.31658, 0.31656],
[0.32149, 0.32156, 0.32154],
[0.32649, 0.32656, 0.32654],
[0.33151, 0.33158, 0.33156],
[0.33654, 0.33662, 0.3366],
[0.34159, 0.34166, 0.34165],
[0.34664, 0.34672, 0.3467],
[0.3517, 0.35178, 0.35176],
[0.3568, 0.35688, 0.35686],
[0.36188, 0.36196, 0.36194],
[0.36698, 0.36706, 0.36704],
[0.37208, 0.37216, 0.37215],
[0.37721, 0.37729, 0.37727],
[0.38233, 0.38241, 0.38239],
[0.38746, 0.38755, 0.38753],
[0.3926, 0.39269, 0.39267],
[0.39775, 0.39784, 0.39781],
[0.40288, 0.40297, 0.40295],
[0.40802, 0.4081, 0.40808],
[0.41315, 0.41324, 0.41322],
[0.41826, 0.41834, 0.41832],
[0.42336, 0.42345, 0.42343],
[0.42846, 0.42855, 0.42853],
[0.43353, 0.43362, 0.4336],
[0.43859, 0.43868, 0.43866],
[0.44365, 0.44374, 0.44372],
[0.44868, 0.44878, 0.44875],
[0.4537, 0.4538, 0.45377],
[0.45872, 0.45882, 0.4588],
[0.46374, 0.46385, 0.46382],
]
b_cyclic_grey_15_85_c0 = bokeh_palette('cyclic_grey_15_85_c0',cyclic_grey_15_85_c0)
m_cyclic_grey_15_85_c0 = mpl_cm('cyclic_grey_15_85_c0',cyclic_grey_15_85_c0)
m_cyclic_grey_15_85_c0_r = mpl_cm('cyclic_grey_15_85_c0_r',list(reversed(cyclic_grey_15_85_c0)))
CET_C5 = b_cyclic_grey_15_85_c0
m_CET_C5 = m_cyclic_grey_15_85_c0
m_CET_C5_r = m_cyclic_grey_15_85_c0_r
palette['CET_C5'] = b_cyclic_grey_15_85_c0
cm['CET_C5'] = m_cyclic_grey_15_85_c0
cm['CET_C5_r'] = m_cyclic_grey_15_85_c0_r
register_cmap('cet_CET_C5',m_cyclic_grey_15_85_c0)
register_cmap('cet_CET_C5_r',m_cyclic_grey_15_85_c0_r)




cyclic_wrwbw_40_90_c42 = [  # cmap_def
[0.873, 0.83593, 0.84864],
[0.8777, 0.8339, 0.84071],
[0.88194, 0.83075, 0.832],
[0.88572, 0.82651, 0.82253],
[0.88906, 0.82126, 0.81236],
[0.89199, 0.81507, 0.80154],
[0.89451, 0.80802, 0.79013],
[0.89666, 0.80023, 0.77819],
[0.89847, 0.79178, 0.76581],
[0.89996, 0.78278, 0.75305],
[0.90116, 0.77331, 0.73999],
[0.9021, 0.76347, 0.72667],
[0.9028, 0.75332, 0.71316],
[0.90327, 0.74294, 0.69952],
[0.90355, 0.73237, 0.68579],
[0.90363, 0.72166, 0.67199],
[0.90355, 0.71085, 0.65815],
[0.9033, 0.69996, 0.64429],
[0.90289, 0.68903, 0.63046],
[0.90234, 0.67804, 0.61663],
[0.90165, 0.66702, 0.60282],
[0.90082, 0.65598, 0.58904],
[0.89986, 0.64492, 0.57531],
[0.89877, 0.63386, 0.56162],
[0.89757, 0.62278, 0.54797],
[0.89623, 0.61169, 0.53437],
[0.89478, 0.60058, 0.52081],
[0.8932, 0.58945, 0.5073],
[0.89151, 0.57831, 0.49383],
[0.8897, 0.56715, 0.48041],
[0.88777, 0.55595, 0.46704],
[0.88574, 0.54474, 0.4537],
[0.88359, 0.53349, 0.44042],
[0.88134, 0.52221, 0.42718],
[0.87897, 0.5109, 0.41398],
[0.87649, 0.49954, 0.40083],
[0.87391, 0.48816, 0.3877],
[0.87123, 0.4767, 0.37464],
[0.86845, 0.4652, 0.3616],
[0.86556, 0.45364, 0.34861],
[0.86257, 0.44202, 0.33566],
[0.85948, 0.43032, 0.32274],
[0.8563, 0.41853, 0.30985],
[0.85302, 0.40666, 0.29699],
[0.84965, 0.3947, 0.28415],
[0.84618, 0.38265, 0.27137],
[0.84264, 0.3705, 0.25865],
[0.83901, 0.35827, 0.24597],
[0.83532, 0.34594, 0.23336],
[0.83157, 0.33356, 0.22089],
[0.82777, 0.3211, 0.20848],
[0.82395, 0.30862, 0.19626],
[0.82013, 0.29614, 0.1842],
[0.81634, 0.28375, 0.17242],
[0.81259, 0.2715, 0.16099],
[0.80896, 0.25953, 0.14996],
[0.80546, 0.24792, 0.1394],
[0.80217, 0.23683, 0.12955],
[0.79914, 0.22645, 0.1204],
[0.7964, 0.21697, 0.11224],
[0.79405, 0.20862, 0.10504],
[0.79209, 0.20162, 0.099146],
[0.79061, 0.19625, 0.094653],
[0.78962, 0.19261, 0.091622],
[0.78916, 0.19089, 0.090163],
[0.78923, 0.19116, 0.09039],
[0.78984, 0.19342, 0.092261],
[0.79096, 0.19754, 0.095704],
[0.79258, 0.20339, 0.10062],
[0.79465, 0.21076, 0.10693],
[0.79711, 0.21943, 0.11431],
[0.79994, 0.22917, 0.12278],
[0.80304, 0.23975, 0.13216],
[0.8064, 0.251, 0.14222],
[0.80993, 0.26273, 0.15286],
[0.81359, 0.27479, 0.16406],
[0.81735, 0.28707, 0.17556],
[0.82116, 0.29951, 0.18745],
[0.82498, 0.31196, 0.1995],
[0.8288, 0.32444, 0.21181],
[0.83258, 0.33689, 0.2242],
[0.83632, 0.34927, 0.23677],
[0.84, 0.36157, 0.24937],
[0.84359, 0.37378, 0.26206],
[0.84712, 0.38589, 0.27479],
[0.85056, 0.39792, 0.28758],
[0.8539, 0.40986, 0.30042],
[0.85716, 0.42169, 0.3133],
[0.86032, 0.43345, 0.32618],
[0.86338, 0.44514, 0.33913],
[0.86634, 0.45674, 0.35208],
[0.8692, 0.46829, 0.36509],
[0.87196, 0.47978, 0.37813],
[0.87462, 0.49122, 0.39122],
[0.87716, 0.50259, 0.40434],
[0.87961, 0.51395, 0.41751],
[0.88195, 0.52525, 0.43072],
[0.88418, 0.53652, 0.44398],
[0.8863, 0.54775, 0.45727],
[0.8883, 0.55896, 0.47062],
[0.89019, 0.57014, 0.484],
[0.89197, 0.5813, 0.49745],
[0.89363, 0.59245, 0.51092],
[0.89518, 0.60356, 0.52444],
[0.8966, 0.61466, 0.53802],
[0.8979, 0.62575, 0.55163],
[0.89908, 0.63683, 0.56528],
[0.90012, 0.64788, 0.579],
[0.90103, 0.65892, 0.59275],
[0.9018, 0.66994, 0.60654],
[0.90242, 0.6809, 0.62037],
[0.90287, 0.69183, 0.63422],
[0.90316, 0.70268, 0.64808],
[0.90325, 0.71344, 0.66197],
[0.90314, 0.72408, 0.67583],
[0.9028, 0.73456, 0.68967],
[0.9022, 0.74481, 0.70344],
[0.90131, 0.7548, 0.71713],
[0.90011, 0.76443, 0.73066],
[0.89855, 0.77364, 0.74401],
[0.8966, 0.78234, 0.75711],
[0.89423, 0.79041, 0.7699],
[0.8914, 0.79778, 0.7823],
[0.88809, 0.80433, 0.79423],
[0.88427, 0.80997, 0.80561],
[0.87992, 0.81463, 0.8164],
[0.87505, 0.81822, 0.8265],
[0.86966, 0.82071, 0.83587],
[0.86376, 0.82205, 0.84446],
[0.85737, 0.82224, 0.85225],
[0.85052, 0.8213, 0.85923],
[0.84325, 0.81927, 0.8654],
[0.8356, 0.81621, 0.87079],
[0.8276, 0.81219, 0.87544],
[0.81931, 0.80731, 0.8794],
[0.81075, 0.80165, 0.88274],
[0.80198, 0.79534, 0.88552],
[0.79302, 0.78846, 0.88781],
[0.7839, 0.78111, 0.88969],
[0.77463, 0.77338, 0.89122],
[0.76527, 0.76536, 0.89246],
[0.75579, 0.75709, 0.89345],
[0.74622, 0.74866, 0.89427],
[0.73657, 0.7401, 0.89494],
[0.72685, 0.73145, 0.89549],
[0.71705, 0.72274, 0.89595],
[0.70717, 0.71401, 0.89635],
[0.69722, 0.70526, 0.89669],
[0.68719, 0.69651, 0.89699],
[0.67709, 0.68776, 0.89726],
[0.66689, 0.67902, 0.8975],
[0.65663, 0.67032, 0.89772],
[0.64625, 0.66163, 0.89792],
[0.63579, 0.65297, 0.89811],
[0.62522, 0.64434, 0.89828],
[0.61453, 0.63573, 0.89844],
[0.60375, 0.62715, 0.89859],
[0.59283, 0.6186, 0.89872],
[0.58178, 0.61007, 0.89884],
[0.57059, 0.60157, 0.89894],
[0.55925, 0.59311, 0.89902],
[0.54776, 0.58467, 0.8991],
[0.53609, 0.57626, 0.89915],
[0.52424, 0.56787, 0.8992],
[0.51219, 0.55952, 0.89923],
[0.49993, 0.5512, 0.89925],
[0.48745, 0.54291, 0.89925],
[0.47471, 0.53465, 0.89924],
[0.46169, 0.52643, 0.89922],
[0.44837, 0.51823, 0.89918],
[0.43473, 0.51007, 0.89913],
[0.42073, 0.50196, 0.89907],
[0.40637, 0.49389, 0.899],
[0.39159, 0.48588, 0.89891],
[0.37638, 0.47795, 0.89881],
[0.36067, 0.47012, 0.8987],
[0.34449, 0.46237, 0.89858],
[0.32777, 0.45477, 0.89845],
[0.31053, 0.44731, 0.89832],
[0.29269, 0.44007, 0.89817],
[0.27432, 0.43307, 0.89802],
[0.25544, 0.42636, 0.89787],
[0.23609, 0.42002, 0.89772],
[0.21638, 0.4141, 0.89757],
[0.19647, 0.40866, 0.89743],
[0.17659, 0.40379, 0.89729],
[0.15722, 0.39953, 0.89717],
[0.13894, 0.39598, 0.89707],
[0.1226, 0.39316, 0.89698],
[0.10959, 0.39116, 0.89692],
[0.1011, 0.38996, 0.89689],
[0.098617, 0.38962, 0.89687],
[0.10246, 0.39014, 0.89689],
[0.11205, 0.39152, 0.89693],
[0.12596, 0.39371, 0.897],
[0.14277, 0.39668, 0.89709],
[0.16137, 0.40041, 0.8972],
[0.18085, 0.40478, 0.89732],
[0.20072, 0.4098, 0.89746],
[0.22066, 0.41534, 0.8976],
[0.24027, 0.42135, 0.89775],
[0.25956, 0.42779, 0.89791],
[0.27833, 0.43456, 0.89806],
[0.29659, 0.44162, 0.8982],
[0.31428, 0.44891, 0.89835],
[0.33142, 0.45639, 0.89848],
[0.34803, 0.46404, 0.89861],
[0.3641, 0.4718, 0.89873],
[0.37969, 0.47966, 0.89883],
[0.39481, 0.48761, 0.89893],
[0.40949, 0.49562, 0.89901],
[0.42377, 0.5037, 0.89909],
[0.4377, 0.51182, 0.89914],
[0.45127, 0.51998, 0.89919],
[0.46453, 0.52819, 0.89922],
[0.47747, 0.53643, 0.89924],
[0.49016, 0.54469, 0.89925],
[0.50259, 0.55298, 0.89924],
[0.51481, 0.56131, 0.89922],
[0.52682, 0.56967, 0.89919],
[0.53863, 0.57806, 0.89914],
[0.55025, 0.58648, 0.89908],
[0.56172, 0.59493, 0.89901],
[0.57302, 0.60341, 0.89892],
[0.58417, 0.61191, 0.89881],
[0.59519, 0.62045, 0.89869],
[0.60608, 0.629, 0.89856],
[0.61685, 0.63759, 0.89841],
[0.62751, 0.6462, 0.89825],
[0.63805, 0.65483, 0.89807],
[0.6485, 0.66351, 0.89788],
[0.65884, 0.67219, 0.89767],
[0.6691, 0.6809, 0.89745],
[0.67927, 0.68965, 0.89721],
[0.68936, 0.6984, 0.89695],
[0.69935, 0.70719, 0.89668],
[0.70928, 0.71598, 0.89638],
[0.71911, 0.72477, 0.89605],
[0.72887, 0.73355, 0.89569],
[0.73854, 0.7423, 0.89528],
[0.74812, 0.75101, 0.89481],
[0.75759, 0.75963, 0.89427],
[0.76695, 0.76815, 0.89362],
[0.7762, 0.77652, 0.89286],
[0.78529, 0.78468, 0.89193],
[0.79422, 0.79255, 0.8908],
[0.80295, 0.80008, 0.88943],
[0.81147, 0.80717, 0.88775],
[0.81973, 0.81373, 0.88571],
[0.82771, 0.81967, 0.88323],
[0.83537, 0.82487, 0.88026],
[0.84268, 0.82926, 0.87672],
[0.84961, 0.83271, 0.87256],
[0.85612, 0.83516, 0.86771],
[0.8622, 0.83654, 0.86213],
[0.86783, 0.83681, 0.85578],
]
b_cyclic_wrwbw_40_90_c42 = bokeh_palette('cyclic_wrwbw_40_90_c42',cyclic_wrwbw_40_90_c42)
m_cyclic_wrwbw_40_90_c42 = mpl_cm('cyclic_wrwbw_40_90_c42',cyclic_wrwbw_40_90_c42)
m_cyclic_wrwbw_40_90_c42_r = mpl_cm('cyclic_wrwbw_40_90_c42_r',list(reversed(cyclic_wrwbw_40_90_c42)))
CET_C4 = b_cyclic_wrwbw_40_90_c42
m_CET_C4 = m_cyclic_wrwbw_40_90_c42
m_CET_C4_r = m_cyclic_wrwbw_40_90_c42_r
palette['CET_C4'] = b_cyclic_wrwbw_40_90_c42
cm['CET_C4'] = m_cyclic_wrwbw_40_90_c42
cm['CET_C4_r'] = m_cyclic_wrwbw_40_90_c42_r
register_cmap('cet_CET_C4',m_cyclic_wrwbw_40_90_c42)
register_cmap('cet_CET_C4_r',m_cyclic_wrwbw_40_90_c42_r)




linear_kryw_5_100_c67 = [  # cmap_def
[0.066001, 0.066024, 0.066019],
[0.080835, 0.066624, 0.064683],
[0.093644, 0.067235, 0.063358],
[0.10501, 0.067862, 0.061943],
[0.11533, 0.068524, 0.060589],
[0.12493, 0.06913, 0.059252],
[0.13446, 0.069402, 0.057766],
[0.14389, 0.069337, 0.05652],
[0.15329, 0.068912, 0.055114],
[0.16265, 0.068217, 0.053762],
[0.17186, 0.067393, 0.052418],
[0.18101, 0.066415, 0.051087],
[0.19008, 0.065312, 0.049777],
[0.19905, 0.064056, 0.04839],
[0.208, 0.06267, 0.047034],
[0.21686, 0.060979, 0.045586],
[0.22565, 0.059301, 0.04424],
[0.2344, 0.057256, 0.042784],
[0.24306, 0.055098, 0.041497],
[0.2517, 0.05272, 0.040051],
[0.26025, 0.050192, 0.038638],
[0.26877, 0.047379, 0.037249],
[0.27721, 0.044339, 0.035898],
[0.28558, 0.041067, 0.034375],
[0.2939, 0.0375, 0.03304],
[0.30215, 0.033602, 0.031743],
[0.31036, 0.029855, 0.030482],
[0.31847, 0.026015, 0.029246],
[0.32649, 0.022146, 0.028041],
[0.33446, 0.018274, 0.026868],
[0.34231, 0.014418, 0.025725],
[0.35006, 0.010378, 0.024615],
[0.35772, 0.0066727, 0.023538],
[0.36524, 0.0030611, 0.022497],
[0.37265, 0, 0.021492],
[0.37993, 0, 0.020527],
[0.38704, 0, 0.019604],
[0.39401, 0, 0.018725],
[0.40079, 0, 0.017895],
[0.40735, 0, 0.017114],
[0.41371, 0, 0.016389],
[0.41991, 0, 0.01571],
[0.42607, 0, 0.015054],
[0.43225, 0, 0.014413],
[0.43842, 0, 0.013785],
[0.44461, 0, 0.013163],
[0.4508, 0, 0.01258],
[0.45699, 0, 0.012015],
[0.46319, 0, 0.011343],
[0.46943, 0, 0.01065],
[0.47565, 0, 0.010097],
[0.48187, 0, 0.0095904],
[0.48811, 0, 0.0090697],
[0.49436, 0, 0.0085559],
[0.5006, 0, 0.0080597],
[0.50686, 0, 0.0075745],
[0.51313, 0, 0.0071],
[0.5194, 0, 0.0066376],
[0.52568, 0, 0.0061855],
[0.53196, 0, 0.0057427],
[0.53826, 0, 0.0053086],
[0.54455, 0, 0.0048845],
[0.55086, 0, 0.0044712],
[0.55718, 0, 0.004067],
[0.5635, 0, 0.0036703],
[0.56981, 0, 0.003281],
[0.57615, 0, 0.0028993],
[0.58248, 0, 0.002527],
[0.58883, 0, 0.0021643],
[0.5952, 0, 0.0018102],
[0.60158, 0, 0.0014648],
[0.60799, 0, 0.0011281],
[0.6144, 0, 0.0008024],
[0.62084, 0, 0.00048773],
[0.62729, 0, 0.00018111],
[0.63377, 0, 0],
[0.64026, 0, 0],
[0.64676, 0, 0],
[0.65328, 0, 0],
[0.65983, 0, 0],
[0.66638, 0, 0],
[0.67297, 0, 0],
[0.67955, 0, 0],
[0.68617, 0, 0],
[0.6928, 0, 0],
[0.69944, 0, 0],
[0.70611, 0, 0],
[0.71279, 0, 0],
[0.7195, 0, 0],
[0.72621, 0, 0],
[0.73295, 0, 0],
[0.7397, 0, 0],
[0.74647, 0, 0],
[0.75325, 0, 0],
[0.76005, 0, 0],
[0.76685, 0, 0],
[0.77364, 0, 0],
[0.78042, 0, 0],
[0.78719, 0, 0],
[0.79394, 0, 0],
[0.80067, 0, 0],
[0.8074, 0, 0],
[0.8141, 0, 0],
[0.8208, 0, 0],
[0.82748, 0, 0],
[0.83414, 0, 0],
[0.84079, 0, 0],
[0.84742, 0, 0],
[0.85404, 0, 0],
[0.86065, 0, 0],
[0.86723, 0, 0],
[0.8738, 0, 0],
[0.88036, 0, 0],
[0.8869, 0, 0],
[0.89342, 0, 0],
[0.89993, 0, 0],
[0.90642, 0, 0],
[0.91289, 0, 0],
[0.91935, 0, 0],
[0.92579, 0.0041866, 0],
[0.93221, 0.0088411, 0],
[0.93858, 0.014235, 0],
[0.94484, 0.020592, 0],
[0.9509, 0.028663, 0],
[0.95676, 0.038768, 0],
[0.96242, 0.049864, 0],
[0.96787, 0.061034, 0],
[0.97311, 0.072263, 0],
[0.97815, 0.083447, 0],
[0.983, 0.094617, 0],
[0.98764, 0.10571, 0],
[0.99205, 0.11685, 0],
[0.99626, 0.12797, 0],
[1, 0.139, 0],
[1, 0.15006, 0],
[1, 0.16105, 0],
[1, 0.17204, 0],
[1, 0.18303, 0],
[1, 0.19402, 0],
[1, 0.20493, 0],
[1, 0.21586, 0],
[1, 0.22677, 0],
[1, 0.23768, 0],
[1, 0.24857, 0],
[1, 0.25943, 0],
[1, 0.27027, 0],
[1, 0.28108, 0],
[1, 0.29193, 0],
[1, 0.30267, 0],
[1, 0.31323, 0],
[1, 0.32352, 0],
[1, 0.33356, 0],
[1, 0.34337, 0],
[1, 0.35298, 0],
[1, 0.3624, 0],
[1, 0.37164, 0.00061699],
[1, 0.38075, 0.0021825],
[1, 0.38968, 0.0037567],
[1, 0.3985, 0.0053395],
[1, 0.40718, 0.0069314],
[1, 0.41575, 0.0085283],
[1, 0.4242, 0.010112],
[1, 0.43257, 0.011953],
[1, 0.44081, 0.013525],
[1, 0.44899, 0.015122],
[1, 0.45706, 0.016714],
[1, 0.46506, 0.018299],
[1, 0.47298, 0.019872],
[1, 0.48083, 0.021427],
[1, 0.48862, 0.022968],
[1, 0.49633, 0.024493],
[1, 0.504, 0.026001],
[1, 0.51159, 0.027485],
[1, 0.51914, 0.028939],
[1, 0.52664, 0.030367],
[1, 0.53406, 0.03178],
[1, 0.54143, 0.033208],
[1, 0.54875, 0.034768],
[1, 0.55598, 0.036315],
[1, 0.56318, 0.037812],
[1, 0.5703, 0.039323],
[1, 0.57737, 0.040851],
[1, 0.58439, 0.04235],
[1, 0.59136, 0.043767],
[1, 0.59828, 0.045295],
[1, 0.60516, 0.046771],
[1, 0.61199, 0.048174],
[1, 0.61878, 0.049666],
[1, 0.62552, 0.051026],
[1, 0.63222, 0.0524],
[1, 0.63889, 0.053786],
[1, 0.64551, 0.055176],
[1, 0.65211, 0.056589],
[1, 0.65866, 0.057815],
[1, 0.66518, 0.05925],
[1, 0.67167, 0.060511],
[1, 0.67814, 0.061743],
[1, 0.68456, 0.063098],
[1, 0.69095, 0.064301],
[1, 0.69731, 0.065521],
[1, 0.70365, 0.066739],
[1, 0.70997, 0.068288],
[1, 0.71622, 0.071897],
[1, 0.72231, 0.079934],
[1, 0.72829, 0.091953],
[1, 0.73416, 0.10627],
[1, 0.73994, 0.12182],
[1, 0.74564, 0.13819],
[1, 0.75128, 0.1548],
[1, 0.75686, 0.17158],
[1, 0.76238, 0.18858],
[1, 0.76786, 0.20562],
[1, 0.7733, 0.22256],
[1, 0.77871, 0.23945],
[1, 0.7841, 0.2564],
[1, 0.78945, 0.27335],
[1, 0.79478, 0.29018],
[1, 0.80008, 0.30712],
[1, 0.80537, 0.32404],
[1, 0.81064, 0.34096],
[1, 0.81591, 0.35795],
[1, 0.82116, 0.37487],
[1, 0.8264, 0.39194],
[1, 0.83163, 0.40902],
[1, 0.83685, 0.42615],
[1, 0.84207, 0.44336],
[1, 0.84727, 0.4606],
[1, 0.85249, 0.47795],
[1, 0.85769, 0.49533],
[1, 0.8629, 0.51282],
[1, 0.86811, 0.53035],
[1, 0.87331, 0.54803],
[1, 0.87852, 0.56571],
[1, 0.88373, 0.58356],
[1, 0.88894, 0.60144],
[1, 0.89415, 0.61947],
[1, 0.89937, 0.63753],
[1, 0.9046, 0.65574],
[1, 0.90983, 0.67401],
[1, 0.91506, 0.69239],
[1, 0.9203, 0.71087],
[1, 0.92556, 0.72942],
[1, 0.93081, 0.74812],
[1, 0.93608, 0.76684],
[1, 0.94135, 0.78577],
[1, 0.94664, 0.80471],
[1, 0.95193, 0.82381],
[1, 0.95723, 0.84299],
[1, 0.96255, 0.86225],
[1, 0.96787, 0.88166],
[1, 0.97321, 0.90108],
[1, 0.97856, 0.92074],
[1, 0.98392, 0.94042],
[1, 0.98928, 0.96017],
[1, 0.99466, 0.98008],
[0.99986, 1, 1],
]
b_linear_kryw_5_100_c67 = bokeh_palette('linear_kryw_5_100_c67',linear_kryw_5_100_c67)
m_linear_kryw_5_100_c67 = mpl_cm('linear_kryw_5_100_c67',linear_kryw_5_100_c67)
m_linear_kryw_5_100_c67_r = mpl_cm('linear_kryw_5_100_c67_r',list(reversed(linear_kryw_5_100_c67)))




diverging_linear_bjy_30_90_c45 = [  # cmap_def
[0.077854, 0.19343, 0.75827],
[0.095213, 0.19629, 0.7569],
[0.11007, 0.1991, 0.75553],
[0.12317, 0.20194, 0.75416],
[0.13514, 0.20481, 0.75279],
[0.14608, 0.20765, 0.75143],
[0.15626, 0.21049, 0.75004],
[0.16572, 0.21333, 0.74867],
[0.17472, 0.21621, 0.7473],
[0.18318, 0.21905, 0.74592],
[0.19126, 0.22188, 0.74454],
[0.19897, 0.22472, 0.74317],
[0.20641, 0.2276, 0.74178],
[0.21348, 0.23044, 0.7404],
[0.22035, 0.23329, 0.73902],
[0.22695, 0.23617, 0.73763],
[0.23332, 0.23902, 0.73624],
[0.23953, 0.24188, 0.73486],
[0.24552, 0.24472, 0.73347],
[0.25139, 0.24761, 0.73208],
[0.25706, 0.25045, 0.73069],
[0.26259, 0.25333, 0.72929],
[0.26797, 0.25618, 0.7279],
[0.27326, 0.25907, 0.7265],
[0.2784, 0.26192, 0.72511],
[0.28343, 0.26478, 0.7237],
[0.28834, 0.26764, 0.7223],
[0.29316, 0.27055, 0.7209],
[0.29788, 0.27341, 0.7195],
[0.3025, 0.27629, 0.71809],
[0.30706, 0.27916, 0.71668],
[0.31151, 0.28204, 0.71527],
[0.3159, 0.2849, 0.71386],
[0.3202, 0.28778, 0.71244],
[0.32441, 0.29066, 0.71103],
[0.32858, 0.29354, 0.70962],
[0.33268, 0.29643, 0.7082],
[0.33669, 0.29932, 0.70678],
[0.34065, 0.30219, 0.70535],
[0.34455, 0.30509, 0.70393],
[0.34839, 0.30797, 0.7025],
[0.35217, 0.31088, 0.70108],
[0.35592, 0.31377, 0.69964],
[0.35959, 0.31666, 0.69821],
[0.36322, 0.31955, 0.69678],
[0.36681, 0.32245, 0.69535],
[0.37033, 0.32533, 0.6939],
[0.37383, 0.32824, 0.69247],
[0.37727, 0.33113, 0.69102],
[0.38067, 0.33405, 0.68958],
[0.38403, 0.33695, 0.68814],
[0.38734, 0.33986, 0.68669],
[0.39062, 0.34277, 0.68524],
[0.39386, 0.34566, 0.68378],
[0.39705, 0.34858, 0.68232],
[0.40023, 0.35149, 0.68086],
[0.40336, 0.35441, 0.67941],
[0.40645, 0.35733, 0.67795],
[0.40951, 0.36023, 0.67649],
[0.41254, 0.36315, 0.67502],
[0.41553, 0.36607, 0.67355],
[0.41848, 0.36899, 0.67208],
[0.42141, 0.3719, 0.67061],
[0.42432, 0.37484, 0.66913],
[0.4272, 0.37776, 0.66765],
[0.43004, 0.38069, 0.66616],
[0.43285, 0.38362, 0.66468],
[0.43564, 0.38654, 0.6632],
[0.4384, 0.38948, 0.66171],
[0.44113, 0.39241, 0.66021],
[0.44384, 0.39535, 0.65871],
[0.44652, 0.39829, 0.65722],
[0.44918, 0.40123, 0.65572],
[0.45181, 0.40416, 0.65421],
[0.45442, 0.4071, 0.6527],
[0.45699, 0.41005, 0.6512],
[0.45955, 0.413, 0.64969],
[0.46209, 0.41594, 0.64817],
[0.46461, 0.41888, 0.64665],
[0.4671, 0.42183, 0.64513],
[0.46957, 0.42479, 0.6436],
[0.47202, 0.42775, 0.64208],
[0.47444, 0.4307, 0.64055],
[0.47685, 0.43364, 0.63902],
[0.47923, 0.43661, 0.63748],
[0.4816, 0.43957, 0.63594],
[0.48394, 0.44254, 0.6344],
[0.48627, 0.44549, 0.63285],
[0.48858, 0.44846, 0.6313],
[0.49088, 0.45143, 0.62974],
[0.49313, 0.4544, 0.62819],
[0.49539, 0.45736, 0.62663],
[0.49763, 0.46033, 0.62506],
[0.49983, 0.4633, 0.6235],
[0.50204, 0.46628, 0.62192],
[0.50422, 0.46927, 0.62036],
[0.50637, 0.47224, 0.61878],
[0.50853, 0.47522, 0.61719],
[0.51065, 0.47819, 0.6156],
[0.51277, 0.48117, 0.61401],
[0.51487, 0.48416, 0.61242],
[0.51695, 0.48715, 0.61083],
[0.51901, 0.49014, 0.60923],
[0.52106, 0.49312, 0.60762],
[0.52309, 0.49611, 0.606],
[0.52511, 0.49911, 0.6044],
[0.52712, 0.50211, 0.60277],
[0.52911, 0.5051, 0.60115],
[0.53107, 0.50811, 0.59953],
[0.53303, 0.5111, 0.59789],
[0.53497, 0.51411, 0.59626],
[0.53691, 0.51711, 0.59462],
[0.53882, 0.5201, 0.59298],
[0.54072, 0.52312, 0.59133],
[0.54261, 0.52613, 0.58967],
[0.54448, 0.52914, 0.58801],
[0.54635, 0.53215, 0.58635],
[0.54819, 0.53516, 0.58469],
[0.55002, 0.53818, 0.58301],
[0.55184, 0.54119, 0.58133],
[0.55364, 0.54421, 0.57966],
[0.55543, 0.54723, 0.57796],
[0.55722, 0.55025, 0.57628],
[0.55898, 0.55327, 0.57457],
[0.56074, 0.55629, 0.57288],
[0.56249, 0.55933, 0.57116],
[0.56422, 0.56236, 0.56945],
[0.56593, 0.56538, 0.56773],
[0.56883, 0.56809, 0.5657],
[0.57292, 0.57047, 0.56336],
[0.57698, 0.57287, 0.56099],
[0.58101, 0.57525, 0.55863],
[0.58503, 0.57764, 0.55626],
[0.58901, 0.58005, 0.55389],
[0.59299, 0.58244, 0.55151],
[0.59693, 0.58486, 0.54913],
[0.60085, 0.58726, 0.54673],
[0.60476, 0.58967, 0.54433],
[0.60865, 0.5921, 0.54191],
[0.6125, 0.59451, 0.5395],
[0.61635, 0.59694, 0.53708],
[0.62018, 0.59936, 0.53465],
[0.62399, 0.60179, 0.53221],
[0.62777, 0.60423, 0.52976],
[0.63154, 0.60666, 0.52731],
[0.6353, 0.6091, 0.52484],
[0.63904, 0.61154, 0.52237],
[0.64276, 0.61398, 0.51989],
[0.64646, 0.61644, 0.51741],
[0.65015, 0.61889, 0.51491],
[0.65382, 0.62134, 0.5124],
[0.65749, 0.62381, 0.50989],
[0.66112, 0.62626, 0.50736],
[0.66475, 0.62873, 0.50483],
[0.66837, 0.6312, 0.50229],
[0.67197, 0.63367, 0.49973],
[0.67556, 0.63615, 0.49717],
[0.67913, 0.63862, 0.4946],
[0.68269, 0.6411, 0.49202],
[0.68624, 0.64358, 0.48942],
[0.68978, 0.64607, 0.48682],
[0.6933, 0.64856, 0.4842],
[0.69681, 0.65106, 0.48158],
[0.70031, 0.65355, 0.47894],
[0.7038, 0.65605, 0.47629],
[0.70727, 0.65855, 0.47362],
[0.71074, 0.66105, 0.47096],
[0.71419, 0.66357, 0.46826],
[0.71764, 0.66607, 0.46556],
[0.72106, 0.66859, 0.46285],
[0.72449, 0.6711, 0.46013],
[0.7279, 0.67362, 0.45739],
[0.7313, 0.67614, 0.45465],
[0.73469, 0.67866, 0.45188],
[0.73807, 0.68119, 0.4491],
[0.74144, 0.68372, 0.4463],
[0.74481, 0.68625, 0.44349],
[0.74816, 0.68879, 0.44066],
[0.75151, 0.69132, 0.43783],
[0.75485, 0.69386, 0.43497],
[0.75817, 0.69641, 0.4321],
[0.76149, 0.69895, 0.42919],
[0.7648, 0.7015, 0.42628],
[0.7681, 0.70405, 0.42335],
[0.7714, 0.70661, 0.42041],
[0.77468, 0.70917, 0.41745],
[0.77796, 0.71173, 0.41446],
[0.78123, 0.71428, 0.41146],
[0.7845, 0.71685, 0.40844],
[0.78775, 0.71942, 0.4054],
[0.79099, 0.72198, 0.40232],
[0.79424, 0.72456, 0.39924],
[0.79747, 0.72713, 0.39614],
[0.8007, 0.72971, 0.393],
[0.80392, 0.73229, 0.38984],
[0.80713, 0.73487, 0.38666],
[0.81033, 0.73745, 0.38347],
[0.81354, 0.74005, 0.38023],
[0.81673, 0.74263, 0.37698],
[0.81992, 0.74522, 0.37369],
[0.8231, 0.74782, 0.37037],
[0.82628, 0.75041, 0.36704],
[0.82945, 0.75301, 0.36366],
[0.83261, 0.75561, 0.36026],
[0.83577, 0.75822, 0.35684],
[0.83892, 0.76082, 0.35336],
[0.84207, 0.76343, 0.34987],
[0.84521, 0.76604, 0.34632],
[0.84834, 0.76865, 0.34277],
[0.85147, 0.77127, 0.33916],
[0.8546, 0.77388, 0.33551],
[0.85772, 0.77651, 0.33181],
[0.86083, 0.77913, 0.32808],
[0.86395, 0.78175, 0.3243],
[0.86705, 0.78438, 0.3205],
[0.87015, 0.78701, 0.31662],
[0.87324, 0.78964, 0.31269],
[0.87633, 0.79227, 0.30875],
[0.87942, 0.79491, 0.30472],
[0.8825, 0.79754, 0.30064],
[0.88558, 0.80019, 0.29651],
[0.88866, 0.80283, 0.29232],
[0.89172, 0.80547, 0.28806],
[0.89479, 0.80812, 0.28374],
[0.89785, 0.81077, 0.27936],
[0.9009, 0.81342, 0.27488],
[0.90395, 0.81607, 0.27036],
[0.907, 0.81872, 0.26572],
[0.91005, 0.82138, 0.26101],
[0.91309, 0.82404, 0.2562],
[0.91612, 0.82671, 0.25133],
[0.91915, 0.82937, 0.24631],
[0.92219, 0.83204, 0.24121],
[0.92521, 0.8347, 0.236],
[0.92823, 0.83737, 0.23064],
[0.93125, 0.84005, 0.22515],
[0.93426, 0.84272, 0.21954],
[0.93728, 0.84539, 0.21376],
[0.94029, 0.84807, 0.20781],
[0.94329, 0.85075, 0.20166],
[0.94629, 0.85343, 0.19535],
[0.94929, 0.85612, 0.18881],
[0.95229, 0.8588, 0.18199],
[0.95528, 0.86149, 0.17496],
[0.95827, 0.86419, 0.16758],
[0.96125, 0.86687, 0.15984],
[0.96424, 0.86957, 0.15175],
[0.96722, 0.87227, 0.1432],
[0.9702, 0.87497, 0.13412],
[0.97317, 0.87767, 0.12434],
[0.97614, 0.88037, 0.11385],
[0.97911, 0.88307, 0.10233],
[0.98208, 0.88578, 0.089538],
[0.98505, 0.88849, 0.07491],
[0.98801, 0.8912, 0.057534],
[0.99097, 0.89391, 0.035647],
]
b_diverging_linear_bjy_30_90_c45 = bokeh_palette('diverging_linear_bjy_30_90_c45',diverging_linear_bjy_30_90_c45)
m_diverging_linear_bjy_30_90_c45 = mpl_cm('diverging_linear_bjy_30_90_c45',diverging_linear_bjy_30_90_c45)
m_diverging_linear_bjy_30_90_c45_r = mpl_cm('diverging_linear_bjy_30_90_c45_r',list(reversed(diverging_linear_bjy_30_90_c45)))
bjy = b_diverging_linear_bjy_30_90_c45
m_bjy = m_diverging_linear_bjy_30_90_c45
m_bjy_r = m_diverging_linear_bjy_30_90_c45_r
palette['bjy'] = b_diverging_linear_bjy_30_90_c45
palette_n['bjy'] = b_diverging_linear_bjy_30_90_c45
cm['bjy'] = m_diverging_linear_bjy_30_90_c45
cm['bjy_r'] = m_diverging_linear_bjy_30_90_c45_r
cm_n['bjy'] = mpl_cm('bjy',diverging_linear_bjy_30_90_c45)
cm_n['bjy_r'] = mpl_cm('bjy_r',list(reversed(diverging_linear_bjy_30_90_c45)))
CET_D7 = b_diverging_linear_bjy_30_90_c45
m_CET_D7 = m_diverging_linear_bjy_30_90_c45
m_CET_D7_r = m_diverging_linear_bjy_30_90_c45_r
palette['CET_D7'] = b_diverging_linear_bjy_30_90_c45
cm['CET_D7'] = m_diverging_linear_bjy_30_90_c45
cm['CET_D7_r'] = m_diverging_linear_bjy_30_90_c45_r
register_cmap('cet_CET_D7',m_diverging_linear_bjy_30_90_c45)
register_cmap('cet_CET_D7_r',m_diverging_linear_bjy_30_90_c45_r)




linear_bgyw_20_98_c66 = [  # cmap_def
[0.019721, 0, 0.67312],
[0.021792, 0.0093198, 0.67457],
[0.023764, 0.020375, 0.67602],
[0.025723, 0.031533, 0.67744],
[0.027593, 0.042972, 0.67884],
[0.029434, 0.053399, 0.68022],
[0.0312, 0.062887, 0.68158],
[0.032926, 0.071508, 0.68292],
[0.034668, 0.079503, 0.68423],
[0.036359, 0.087188, 0.68553],
[0.037918, 0.09442, 0.68679],
[0.039376, 0.1013, 0.68804],
[0.040814, 0.10797, 0.68926],
[0.042148, 0.11441, 0.69045],
[0.043258, 0.12068, 0.69161],
[0.044531, 0.12678, 0.69276],
[0.045545, 0.13274, 0.69387],
[0.046606, 0.13856, 0.69495],
[0.047504, 0.14425, 0.69602],
[0.048422, 0.14985, 0.69703],
[0.049286, 0.15533, 0.69803],
[0.049957, 0.16073, 0.69899],
[0.050622, 0.16607, 0.69992],
[0.051213, 0.17134, 0.70083],
[0.051756, 0.17652, 0.70168],
[0.052246, 0.18167, 0.70251],
[0.052659, 0.18678, 0.70331],
[0.053041, 0.19182, 0.70406],
[0.053359, 0.19684, 0.70477],
[0.0536, 0.20176, 0.70545],
[0.05381, 0.20672, 0.70608],
[0.053957, 0.2116, 0.70667],
[0.054028, 0.21646, 0.70721],
[0.054057, 0.22128, 0.70771],
[0.054033, 0.22609, 0.70816],
[0.053937, 0.23085, 0.70856],
[0.053775, 0.23561, 0.70892],
[0.05358, 0.24032, 0.7092],
[0.053321, 0.24504, 0.70944],
[0.052992, 0.24974, 0.70962],
[0.052597, 0.25445, 0.70974],
[0.052169, 0.25912, 0.70979],
[0.051684, 0.26379, 0.70978],
[0.051133, 0.26843, 0.70969],
[0.050513, 0.27307, 0.70954],
[0.049841, 0.27771, 0.70932],
[0.049147, 0.28235, 0.709],
[0.04825, 0.28695, 0.70859],
[0.04738, 0.29159, 0.7081],
[0.046417, 0.29621, 0.70751],
[0.045403, 0.30083, 0.70682],
[0.044335, 0.30547, 0.70603],
[0.043076, 0.3101, 0.70511],
[0.041991, 0.31472, 0.70408],
[0.040637, 0.31937, 0.70291],
[0.039245, 0.32401, 0.70159],
[0.037801, 0.32869, 0.70013],
[0.036303, 0.33336, 0.6985],
[0.034665, 0.33804, 0.69669],
[0.033, 0.34276, 0.69467],
[0.031386, 0.34749, 0.69245],
[0.02977, 0.35224, 0.68995],
[0.028127, 0.35704, 0.68719],
[0.026459, 0.36185, 0.68412],
[0.02481, 0.36671, 0.6807],
[0.023175, 0.37161, 0.67686],
[0.022142, 0.37657, 0.67264],
[0.022449, 0.38146, 0.66815],
[0.024367, 0.38628, 0.66347],
[0.027722, 0.39108, 0.6586],
[0.032448, 0.39582, 0.65355],
[0.03852, 0.40052, 0.64833],
[0.045286, 0.40516, 0.64294],
[0.052276, 0.40978, 0.63739],
[0.059484, 0.41435, 0.63169],
[0.066532, 0.41889, 0.62583],
[0.073577, 0.42341, 0.61983],
[0.080364, 0.42791, 0.61368],
[0.08713, 0.43237, 0.60741],
[0.093557, 0.43681, 0.60099],
[0.099853, 0.44122, 0.59445],
[0.10599, 0.44562, 0.58778],
[0.11186, 0.45001, 0.58099],
[0.11745, 0.45438, 0.57408],
[0.12286, 0.45873, 0.56706],
[0.12813, 0.46305, 0.55991],
[0.13311, 0.46739, 0.55265],
[0.13789, 0.47171, 0.54528],
[0.1424, 0.476, 0.53781],
[0.14672, 0.48029, 0.53021],
[0.15086, 0.48457, 0.52251],
[0.15477, 0.48885, 0.51471],
[0.15849, 0.49311, 0.50679],
[0.16197, 0.49739, 0.49878],
[0.16519, 0.50165, 0.49067],
[0.16827, 0.50588, 0.48243],
[0.17113, 0.51014, 0.47409],
[0.17379, 0.51439, 0.46564],
[0.17622, 0.51863, 0.45709],
[0.1785, 0.52286, 0.44843],
[0.1805, 0.52709, 0.43965],
[0.18232, 0.53132, 0.43076],
[0.18395, 0.53555, 0.42174],
[0.18539, 0.53977, 0.41262],
[0.18656, 0.544, 0.40336],
[0.18757, 0.54823, 0.39396],
[0.18834, 0.55246, 0.38443],
[0.18903, 0.55666, 0.37491],
[0.18989, 0.56081, 0.3656],
[0.19095, 0.5649, 0.35655],
[0.19225, 0.56895, 0.34772],
[0.19379, 0.57295, 0.33913],
[0.19553, 0.57689, 0.33073],
[0.19754, 0.58079, 0.32256],
[0.19976, 0.58465, 0.31455],
[0.20225, 0.58845, 0.30674],
[0.20498, 0.59224, 0.2991],
[0.20794, 0.59596, 0.29161],
[0.21114, 0.59966, 0.28426],
[0.21456, 0.60332, 0.27707],
[0.21821, 0.60694, 0.27001],
[0.22206, 0.61053, 0.26306],
[0.22618, 0.61408, 0.25623],
[0.23049, 0.6176, 0.24956],
[0.23505, 0.62109, 0.24299],
[0.23975, 0.62454, 0.23654],
[0.24464, 0.62797, 0.23013],
[0.24971, 0.63137, 0.22383],
[0.25499, 0.63473, 0.21763],
[0.26043, 0.63806, 0.21154],
[0.26605, 0.64137, 0.2055],
[0.27177, 0.64465, 0.19951],
[0.27767, 0.6479, 0.19362],
[0.28367, 0.65113, 0.18778],
[0.28987, 0.65432, 0.18198],
[0.29618, 0.6575, 0.17626],
[0.30259, 0.66063, 0.1706],
[0.30913, 0.66376, 0.16494],
[0.31579, 0.66684, 0.15938],
[0.32256, 0.66991, 0.15386],
[0.32942, 0.67295, 0.14835],
[0.33634, 0.67597, 0.14288],
[0.34341, 0.67895, 0.13748],
[0.35057, 0.68191, 0.13209],
[0.35779, 0.68486, 0.12667],
[0.36507, 0.68777, 0.12127],
[0.37248, 0.69065, 0.11603],
[0.37996, 0.69352, 0.11072],
[0.38748, 0.69636, 0.10536],
[0.3951, 0.69917, 0.1001],
[0.40279, 0.70196, 0.094912],
[0.41054, 0.70473, 0.089608],
[0.41833, 0.70747, 0.084378],
[0.42622, 0.71018, 0.079119],
[0.43414, 0.71287, 0.073963],
[0.44214, 0.71554, 0.068745],
[0.4502, 0.71818, 0.063586],
[0.45829, 0.72079, 0.05835],
[0.46644, 0.72338, 0.053154],
[0.47467, 0.72595, 0.048139],
[0.48291, 0.72849, 0.043005],
[0.49122, 0.731, 0.038082],
[0.49958, 0.73349, 0.033323],
[0.50799, 0.73595, 0.029083],
[0.51642, 0.7384, 0.025269],
[0.52493, 0.7408, 0.021961],
[0.53346, 0.7432, 0.018981],
[0.54204, 0.74556, 0.016456],
[0.55067, 0.74789, 0.014369],
[0.55933, 0.7502, 0.012644],
[0.56805, 0.75248, 0.01132],
[0.5768, 0.75475, 0.010293],
[0.58554, 0.75698, 0.0097626],
[0.59425, 0.7592, 0.0095221],
[0.60291, 0.76143, 0.0093793],
[0.61149, 0.76365, 0.0093837],
[0.62001, 0.76588, 0.0095486],
[0.62846, 0.7681, 0.0098317],
[0.63684, 0.77033, 0.010273],
[0.64515, 0.77256, 0.010973],
[0.6534, 0.77478, 0.011934],
[0.6616, 0.77702, 0.012885],
[0.66973, 0.77925, 0.014059],
[0.6778, 0.78149, 0.015391],
[0.68581, 0.78373, 0.016941],
[0.69375, 0.78598, 0.018714],
[0.70164, 0.78822, 0.02067],
[0.70948, 0.79047, 0.022856],
[0.71724, 0.79273, 0.02532],
[0.72495, 0.79499, 0.027999],
[0.73261, 0.79726, 0.030918],
[0.74019, 0.79954, 0.034117],
[0.74772, 0.80182, 0.037833],
[0.75519, 0.80411, 0.041619],
[0.7626, 0.80641, 0.045386],
[0.76994, 0.80872, 0.049415],
[0.77723, 0.81104, 0.053367],
[0.78444, 0.81337, 0.057479],
[0.79159, 0.81571, 0.061666],
[0.79868, 0.81806, 0.065966],
[0.80569, 0.82043, 0.070321],
[0.81263, 0.82282, 0.074765],
[0.8195, 0.82521, 0.079243],
[0.82631, 0.82763, 0.083887],
[0.83303, 0.83006, 0.088519],
[0.83968, 0.83251, 0.093204],
[0.84624, 0.83498, 0.098097],
[0.85271, 0.83747, 0.10302],
[0.85909, 0.83999, 0.10799],
[0.86538, 0.84253, 0.11317],
[0.87158, 0.84509, 0.11836],
[0.87768, 0.84768, 0.12362],
[0.88367, 0.85031, 0.12904],
[0.88954, 0.85296, 0.13462],
[0.89531, 0.85565, 0.14024],
[0.90094, 0.85837, 0.14604],
[0.90644, 0.86113, 0.15193],
[0.9118, 0.86394, 0.15801],
[0.91702, 0.86679, 0.16424],
[0.92207, 0.86968, 0.17059],
[0.92694, 0.87264, 0.17716],
[0.93162, 0.87565, 0.18392],
[0.9361, 0.87872, 0.19089],
[0.94037, 0.88186, 0.19807],
[0.9444, 0.88507, 0.2055],
[0.94813, 0.88837, 0.21323],
[0.95158, 0.89176, 0.22126],
[0.9547, 0.89524, 0.22957],
[0.95743, 0.89884, 0.23832],
[0.95979, 0.90256, 0.24754],
[0.96199, 0.9063, 0.25771],
[0.96425, 0.90998, 0.26918],
[0.96658, 0.91361, 0.28181],
[0.96896, 0.91718, 0.29575],
[0.9714, 0.92068, 0.31084],
[0.97388, 0.92413, 0.32705],
[0.9764, 0.92751, 0.34453],
[0.97893, 0.93084, 0.36314],
[0.98148, 0.93409, 0.38291],
[0.98401, 0.93727, 0.40392],
[0.98651, 0.94038, 0.42621],
[0.98895, 0.94342, 0.4498],
[0.9913, 0.94637, 0.47473],
[0.99352, 0.94926, 0.50109],
[0.99555, 0.95206, 0.52897],
[0.99736, 0.95478, 0.55845],
[0.99887, 0.95742, 0.58967],
[1, 0.95997, 0.62277],
[1, 0.96243, 0.65786],
[1, 0.96481, 0.6951],
[1, 0.96711, 0.7347],
[0.99841, 0.96931, 0.77694],
[0.99564, 0.97144, 0.82203],
[0.99146, 0.97349, 0.87012],
[0.98544, 0.97545, 0.92187],
[0.97718, 0.97737, 0.97732],
]
b_linear_bgyw_20_98_c66 = bokeh_palette('linear_bgyw_20_98_c66',linear_bgyw_20_98_c66)
m_linear_bgyw_20_98_c66 = mpl_cm('linear_bgyw_20_98_c66',linear_bgyw_20_98_c66)
m_linear_bgyw_20_98_c66_r = mpl_cm('linear_bgyw_20_98_c66_r',list(reversed(linear_bgyw_20_98_c66)))
CET_L9 = b_linear_bgyw_20_98_c66
m_CET_L9 = m_linear_bgyw_20_98_c66
m_CET_L9_r = m_linear_bgyw_20_98_c66_r
palette['CET_L9'] = b_linear_bgyw_20_98_c66
cm['CET_L9'] = m_linear_bgyw_20_98_c66
cm['CET_L9_r'] = m_linear_bgyw_20_98_c66_r
register_cmap('cet_CET_L9',m_linear_bgyw_20_98_c66)
register_cmap('cet_CET_L9_r',m_linear_bgyw_20_98_c66_r)




linear_tritanopic_krjcw_5_95_c24 = [  # cmap_def
[0.066001, 0.066024, 0.066019],
[0.076759, 0.067596, 0.06602],
[0.086416, 0.069198, 0.066054],
[0.095198, 0.070651, 0.066121],
[0.10321, 0.072161, 0.066221],
[0.11063, 0.073753, 0.066367],
[0.1176, 0.075172, 0.066552],
[0.12457, 0.076447, 0.066783],
[0.13159, 0.077536, 0.067062],
[0.13857, 0.078462, 0.067395],
[0.14552, 0.079356, 0.067762],
[0.15233, 0.080233, 0.068159],
[0.15915, 0.081098, 0.068594],
[0.16589, 0.081959, 0.069059],
[0.17259, 0.082719, 0.06953],
[0.17925, 0.083516, 0.069981],
[0.18588, 0.084338, 0.070485],
[0.19242, 0.085059, 0.071081],
[0.19893, 0.085781, 0.07166],
[0.20543, 0.086603, 0.072227],
[0.21189, 0.087401, 0.072907],
[0.21829, 0.088131, 0.073607],
[0.22463, 0.088885, 0.074252],
[0.231, 0.089646, 0.074934],
[0.23731, 0.090422, 0.075663],
[0.24354, 0.091225, 0.076415],
[0.24976, 0.092019, 0.077197],
[0.25596, 0.092759, 0.078008],
[0.26213, 0.093597, 0.078847],
[0.26825, 0.094494, 0.079718],
[0.27433, 0.095356, 0.080621],
[0.28037, 0.09619, 0.081571],
[0.28637, 0.097128, 0.082482],
[0.29236, 0.098132, 0.0834],
[0.29828, 0.099086, 0.084431],
[0.30418, 0.1001, 0.085383],
[0.31004, 0.10117, 0.086488],
[0.31584, 0.1023, 0.087649],
[0.32157, 0.10349, 0.08879],
[0.32728, 0.10464, 0.09],
[0.33296, 0.10596, 0.091278],
[0.33857, 0.10728, 0.09253],
[0.34413, 0.10861, 0.093897],
[0.34966, 0.11011, 0.095321],
[0.35514, 0.11159, 0.096697],
[0.36052, 0.11318, 0.098274],
[0.36589, 0.11474, 0.099798],
[0.37118, 0.1165, 0.10141],
[0.37644, 0.11829, 0.1031],
[0.38162, 0.1201, 0.10476],
[0.38674, 0.12201, 0.10662],
[0.39181, 0.12403, 0.10839],
[0.3968, 0.12616, 0.11034],
[0.40174, 0.12831, 0.11228],
[0.40662, 0.13059, 0.11425],
[0.41142, 0.1329, 0.11638],
[0.41615, 0.13531, 0.11853],
[0.4208, 0.13784, 0.12068],
[0.42539, 0.14042, 0.12295],
[0.42989, 0.1431, 0.12533],
[0.43431, 0.14588, 0.12775],
[0.43865, 0.1487, 0.13027],
[0.44292, 0.15165, 0.13279],
[0.44708, 0.15469, 0.13541],
[0.45117, 0.15782, 0.13815],
[0.45516, 0.16105, 0.14093],
[0.45906, 0.16436, 0.14377],
[0.46285, 0.16774, 0.14669],
[0.46656, 0.17122, 0.14975],
[0.47016, 0.17481, 0.15281],
[0.47364, 0.17847, 0.15604],
[0.47702, 0.18218, 0.15929],
[0.48029, 0.18606, 0.16268],
[0.48344, 0.18996, 0.16612],
[0.48647, 0.194, 0.16969],
[0.48937, 0.19807, 0.17331],
[0.49215, 0.20226, 0.17707],
[0.49478, 0.20657, 0.18089],
[0.49728, 0.21091, 0.18487],
[0.49962, 0.21535, 0.18892],
[0.50184, 0.2199, 0.19307],
[0.50389, 0.22448, 0.19734],
[0.50577, 0.2292, 0.2017],
[0.50751, 0.23398, 0.20623],
[0.50907, 0.23884, 0.21081],
[0.51044, 0.24378, 0.21554],
[0.51162, 0.24884, 0.22041],
[0.51261, 0.25395, 0.22538],
[0.5134, 0.25916, 0.23048],
[0.51398, 0.26442, 0.23575],
[0.51435, 0.26979, 0.24111],
[0.51448, 0.27521, 0.24662],
[0.51437, 0.28071, 0.2523],
[0.51411, 0.28625, 0.25803],
[0.51379, 0.29179, 0.26385],
[0.51341, 0.29727, 0.26966],
[0.51298, 0.3027, 0.2755],
[0.51251, 0.30813, 0.28135],
[0.51199, 0.31353, 0.28724],
[0.51142, 0.31889, 0.29316],
[0.51079, 0.32422, 0.29912],
[0.51011, 0.32956, 0.30508],
[0.50938, 0.33486, 0.31108],
[0.5086, 0.34012, 0.31709],
[0.50775, 0.34537, 0.32314],
[0.50684, 0.35061, 0.32922],
[0.50587, 0.35583, 0.33531],
[0.50486, 0.36102, 0.34142],
[0.50378, 0.36622, 0.34757],
[0.50263, 0.37137, 0.35373],
[0.50143, 0.37655, 0.35993],
[0.50014, 0.38168, 0.36616],
[0.49881, 0.3868, 0.37239],
[0.4974, 0.39193, 0.37867],
[0.49591, 0.39703, 0.38496],
[0.49437, 0.40213, 0.39129],
[0.49274, 0.40722, 0.39763],
[0.49106, 0.41231, 0.404],
[0.48927, 0.41738, 0.4104],
[0.48743, 0.42242, 0.41681],
[0.48548, 0.42749, 0.42325],
[0.48347, 0.43253, 0.42972],
[0.48137, 0.43757, 0.43622],
[0.47919, 0.4426, 0.44274],
[0.47691, 0.44761, 0.44928],
[0.47455, 0.45265, 0.45584],
[0.47208, 0.45765, 0.46242],
[0.46953, 0.46265, 0.46905],
[0.46686, 0.46766, 0.47569],
[0.4641, 0.47266, 0.48235],
[0.46122, 0.47765, 0.48903],
[0.45824, 0.48264, 0.49574],
[0.45516, 0.48764, 0.50247],
[0.45195, 0.49261, 0.50924],
[0.44861, 0.4976, 0.51601],
[0.44515, 0.50256, 0.52283],
[0.44156, 0.50754, 0.52966],
[0.43784, 0.51251, 0.53652],
[0.43396, 0.51748, 0.54339],
[0.42996, 0.52244, 0.55029],
[0.42579, 0.52741, 0.55723],
[0.42146, 0.53237, 0.56417],
[0.41699, 0.53734, 0.57114],
[0.41232, 0.54229, 0.57814],
[0.40747, 0.54724, 0.58517],
[0.40242, 0.55221, 0.59222],
[0.39719, 0.55717, 0.59928],
[0.39174, 0.56212, 0.60637],
[0.38606, 0.56707, 0.61348],
[0.38015, 0.57202, 0.62063],
[0.37399, 0.57698, 0.62779],
[0.36755, 0.58192, 0.63498],
[0.36082, 0.58688, 0.64219],
[0.3538, 0.59184, 0.64943],
[0.34645, 0.59679, 0.65668],
[0.33874, 0.60173, 0.66396],
[0.33063, 0.60669, 0.67126],
[0.32211, 0.61164, 0.67859],
[0.31313, 0.6166, 0.68595],
[0.30362, 0.62155, 0.69332],
[0.29356, 0.62651, 0.70072],
[0.28288, 0.63146, 0.70814],
[0.27144, 0.63642, 0.71558],
[0.2593, 0.64137, 0.72303],
[0.24667, 0.64629, 0.73039],
[0.23389, 0.65116, 0.7376],
[0.22104, 0.65597, 0.74465],
[0.20808, 0.66072, 0.75153],
[0.1951, 0.66542, 0.75824],
[0.182, 0.67007, 0.76479],
[0.16887, 0.67467, 0.7712],
[0.15562, 0.67922, 0.77745],
[0.14229, 0.68372, 0.78356],
[0.12889, 0.68819, 0.78952],
[0.11552, 0.69261, 0.79534],
[0.10223, 0.69697, 0.80102],
[0.089177, 0.7013, 0.80656],
[0.076371, 0.70558, 0.81196],
[0.064198, 0.70983, 0.81723],
[0.052839, 0.71403, 0.82238],
[0.04299, 0.7182, 0.82739],
[0.03571, 0.72233, 0.83229],
[0.031547, 0.72642, 0.83705],
[0.03119, 0.73048, 0.8417],
[0.034486, 0.7345, 0.84622],
[0.04152, 0.73848, 0.85063],
[0.051018, 0.74243, 0.85493],
[0.062365, 0.74634, 0.8591],
[0.074692, 0.75022, 0.86317],
[0.087754, 0.75407, 0.86712],
[0.10114, 0.75788, 0.87097],
[0.11477, 0.76166, 0.87471],
[0.12858, 0.76542, 0.87835],
[0.14243, 0.76913, 0.88188],
[0.15631, 0.77282, 0.88531],
[0.17014, 0.77648, 0.88864],
[0.18394, 0.7801, 0.89187],
[0.19772, 0.78369, 0.895],
[0.21146, 0.78726, 0.89803],
[0.22511, 0.79079, 0.90097],
[0.23873, 0.7943, 0.90381],
[0.25231, 0.79777, 0.90657],
[0.26582, 0.80121, 0.90923],
[0.27928, 0.80463, 0.9118],
[0.29269, 0.80802, 0.91428],
[0.3061, 0.81138, 0.91667],
[0.31942, 0.8147, 0.91897],
[0.33272, 0.818, 0.92118],
[0.34593, 0.82128, 0.92332],
[0.35912, 0.82452, 0.92537],
[0.37227, 0.82774, 0.92733],
[0.38537, 0.83093, 0.92922],
[0.39843, 0.83409, 0.93103],
[0.41145, 0.83723, 0.93276],
[0.42447, 0.84033, 0.9344],
[0.43746, 0.8434, 0.93597],
[0.45041, 0.84645, 0.93746],
[0.46332, 0.84948, 0.93888],
[0.47621, 0.85247, 0.94022],
[0.48906, 0.85544, 0.94148],
[0.50189, 0.85837, 0.94268],
[0.5147, 0.86129, 0.9438],
[0.52751, 0.86417, 0.94485],
[0.54029, 0.86702, 0.94582],
[0.55304, 0.86985, 0.94672],
[0.56577, 0.87266, 0.94757],
[0.57847, 0.87543, 0.94834],
[0.59116, 0.87818, 0.94904],
[0.60387, 0.88089, 0.94967],
[0.61655, 0.88358, 0.95023],
[0.6292, 0.88625, 0.95073],
[0.64184, 0.88888, 0.95117],
[0.65446, 0.89149, 0.95154],
[0.66709, 0.89407, 0.95184],
[0.67971, 0.89662, 0.95208],
[0.69232, 0.89914, 0.95225],
[0.70491, 0.90163, 0.95237],
[0.71749, 0.9041, 0.95243],
[0.73006, 0.90654, 0.95243],
[0.74265, 0.90894, 0.95235],
[0.75522, 0.91133, 0.95222],
[0.76776, 0.91368, 0.95204],
[0.78031, 0.916, 0.9518],
[0.79287, 0.9183, 0.9515],
[0.80542, 0.92056, 0.95113],
[0.81797, 0.9228, 0.95071],
[0.83051, 0.925, 0.95024],
[0.84304, 0.92718, 0.94971],
[0.85559, 0.92933, 0.94911],
[0.86813, 0.93144, 0.94847],
[0.88066, 0.93353, 0.94777],
[0.89318, 0.9356, 0.94703],
[0.90574, 0.93762, 0.94621],
[0.91828, 0.93962, 0.94535],
[0.93081, 0.94158, 0.94444],
[0.94334, 0.94353, 0.94348],
]
b_linear_tritanopic_krjcw_5_95_c24 = bokeh_palette('linear_tritanopic_krjcw_5_95_c24',linear_tritanopic_krjcw_5_95_c24)
m_linear_tritanopic_krjcw_5_95_c24 = mpl_cm('linear_tritanopic_krjcw_5_95_c24',linear_tritanopic_krjcw_5_95_c24)
m_linear_tritanopic_krjcw_5_95_c24_r = mpl_cm('linear_tritanopic_krjcw_5_95_c24_r',list(reversed(linear_tritanopic_krjcw_5_95_c24)))
CET_CBTL2 = b_linear_tritanopic_krjcw_5_95_c24
m_CET_CBTL2 = m_linear_tritanopic_krjcw_5_95_c24
m_CET_CBTL2_r = m_linear_tritanopic_krjcw_5_95_c24_r
palette['CET_CBTL2'] = b_linear_tritanopic_krjcw_5_95_c24
cm['CET_CBTL2'] = m_linear_tritanopic_krjcw_5_95_c24
cm['CET_CBTL2_r'] = m_linear_tritanopic_krjcw_5_95_c24_r
register_cmap('cet_CET_CBTL2',m_linear_tritanopic_krjcw_5_95_c24)
register_cmap('cet_CET_CBTL2_r',m_linear_tritanopic_krjcw_5_95_c24_r)




diverging_bwr_20_95_c54 = [  # cmap_def
[0.091874, 0.16093, 0.4469],
[0.09332, 0.16518, 0.45817],
[0.094888, 0.16952, 0.4695],
[0.096269, 0.17381, 0.48085],
[0.097771, 0.17816, 0.49227],
[0.099148, 0.18246, 0.50373],
[0.10052, 0.18684, 0.51524],
[0.10188, 0.19121, 0.52681],
[0.10324, 0.1956, 0.53841],
[0.10448, 0.19999, 0.55006],
[0.10579, 0.20445, 0.56175],
[0.10705, 0.20887, 0.57349],
[0.10821, 0.21331, 0.58527],
[0.10946, 0.2178, 0.5971],
[0.11061, 0.22227, 0.60896],
[0.11176, 0.22678, 0.62086],
[0.11291, 0.2313, 0.63278],
[0.11401, 0.23582, 0.64473],
[0.11517, 0.24034, 0.65671],
[0.11641, 0.24491, 0.66868],
[0.11766, 0.2495, 0.68064],
[0.11905, 0.25411, 0.69259],
[0.12055, 0.25873, 0.70448],
[0.12227, 0.26339, 0.7163],
[0.12429, 0.26804, 0.72801],
[0.12666, 0.27275, 0.73958],
[0.12946, 0.2775, 0.75097],
[0.13277, 0.28229, 0.76212],
[0.13665, 0.28708, 0.773],
[0.14125, 0.29196, 0.78356],
[0.1465, 0.29689, 0.79375],
[0.15254, 0.30184, 0.80352],
[0.15936, 0.30689, 0.81285],
[0.16694, 0.31196, 0.82169],
[0.17519, 0.31714, 0.83003],
[0.18408, 0.32237, 0.83785],
[0.19356, 0.32765, 0.84516],
[0.2035, 0.33302, 0.85196],
[0.21381, 0.33843, 0.85827],
[0.2244, 0.34391, 0.86413],
[0.23524, 0.34946, 0.86955],
[0.24614, 0.35507, 0.8746],
[0.25713, 0.3607, 0.87932],
[0.26808, 0.36642, 0.88374],
[0.27901, 0.37214, 0.88791],
[0.2898, 0.37793, 0.89188],
[0.30048, 0.38375, 0.89568],
[0.31103, 0.38959, 0.89934],
[0.3214, 0.39547, 0.9029],
[0.33162, 0.40137, 0.90637],
[0.34168, 0.40729, 0.90978],
[0.35157, 0.41325, 0.91315],
[0.36131, 0.41922, 0.91648],
[0.37089, 0.42521, 0.91979],
[0.38034, 0.43122, 0.92309],
[0.38964, 0.43724, 0.92637],
[0.39881, 0.44328, 0.92965],
[0.40787, 0.44934, 0.93291],
[0.41682, 0.45542, 0.93618],
[0.42566, 0.46151, 0.93944],
[0.4344, 0.46762, 0.94269],
[0.44306, 0.47374, 0.94592],
[0.45163, 0.4799, 0.94914],
[0.46012, 0.48605, 0.95233],
[0.46856, 0.49224, 0.9555],
[0.47694, 0.49845, 0.95862],
[0.48526, 0.50466, 0.96169],
[0.49355, 0.5109, 0.9647],
[0.50181, 0.51717, 0.96762],
[0.51003, 0.52346, 0.97044],
[0.51824, 0.52978, 0.97315],
[0.52644, 0.53613, 0.97571],
[0.53462, 0.54252, 0.97812],
[0.54281, 0.54893, 0.98035],
[0.551, 0.55538, 0.98238],
[0.55919, 0.56188, 0.9842],
[0.5674, 0.5684, 0.98579],
[0.5756, 0.57497, 0.98716],
[0.5838, 0.58158, 0.98828],
[0.59201, 0.58822, 0.98917],
[0.60019, 0.59492, 0.98983],
[0.60837, 0.60165, 0.99027],
[0.61652, 0.60842, 0.99051],
[0.62464, 0.61521, 0.99055],
[0.63273, 0.62205, 0.99043],
[0.64077, 0.62892, 0.99015],
[0.64877, 0.63581, 0.98974],
[0.65671, 0.64273, 0.98921],
[0.6646, 0.64968, 0.98859],
[0.67243, 0.65665, 0.98789],
[0.68019, 0.66364, 0.98713],
[0.68791, 0.67064, 0.98631],
[0.69556, 0.67767, 0.98545],
[0.70314, 0.6847, 0.98455],
[0.71067, 0.69176, 0.98362],
[0.71813, 0.69883, 0.98268],
[0.72555, 0.70592, 0.98171],
[0.7329, 0.71301, 0.98073],
[0.7402, 0.72013, 0.97973],
[0.74744, 0.72725, 0.97871],
[0.75463, 0.7344, 0.97769],
[0.76177, 0.74155, 0.97666],
[0.76886, 0.74871, 0.97561],
[0.7759, 0.75589, 0.97456],
[0.7829, 0.76308, 0.97349],
[0.78986, 0.77029, 0.97242],
[0.79677, 0.77751, 0.97133],
[0.80364, 0.78473, 0.97021],
[0.81048, 0.79195, 0.96908],
[0.81728, 0.79917, 0.96791],
[0.82404, 0.80637, 0.9667],
[0.83078, 0.81353, 0.96544],
[0.83748, 0.82065, 0.96411],
[0.84414, 0.8277, 0.96271],
[0.85078, 0.83466, 0.96118],
[0.85739, 0.84147, 0.95952],
[0.86397, 0.8481, 0.95769],
[0.87051, 0.8545, 0.95564],
[0.87701, 0.8606, 0.95332],
[0.88347, 0.86632, 0.9507],
[0.88986, 0.87161, 0.94771],
[0.89618, 0.87636, 0.94431],
[0.9024, 0.88051, 0.94045],
[0.9085, 0.88397, 0.93608],
[0.91447, 0.88666, 0.93116],
[0.92027, 0.88852, 0.92566],
[0.92588, 0.88949, 0.91958],
[0.93128, 0.88955, 0.9129],
[0.93643, 0.88867, 0.90562],
[0.94132, 0.88685, 0.89777],
[0.94593, 0.88412, 0.88939],
[0.95026, 0.8805, 0.8805],
[0.95429, 0.87605, 0.87117],
[0.95803, 0.87085, 0.86143],
[0.96147, 0.86496, 0.85135],
[0.96463, 0.85847, 0.84098],
[0.96754, 0.85145, 0.83037],
[0.97018, 0.84399, 0.81957],
[0.9726, 0.83616, 0.80864],
[0.9748, 0.82802, 0.79758],
[0.97681, 0.81964, 0.78646],
[0.97864, 0.81108, 0.77528],
[0.98031, 0.80236, 0.76407],
[0.98183, 0.79354, 0.75284],
[0.9832, 0.78462, 0.74162],
[0.98446, 0.77564, 0.7304],
[0.98559, 0.76662, 0.7192],
[0.98661, 0.75756, 0.70801],
[0.98752, 0.74847, 0.69685],
[0.98832, 0.73936, 0.68571],
[0.98903, 0.73022, 0.67459],
[0.98964, 0.72108, 0.66352],
[0.99016, 0.71193, 0.65246],
[0.99058, 0.70276, 0.64143],
[0.99092, 0.69357, 0.63043],
[0.99116, 0.68436, 0.61946],
[0.99132, 0.67515, 0.60851],
[0.99139, 0.66591, 0.59759],
[0.99137, 0.65666, 0.5867],
[0.99126, 0.64738, 0.57585],
[0.99106, 0.63808, 0.56501],
[0.99077, 0.62876, 0.55423],
[0.99039, 0.61943, 0.54347],
[0.98991, 0.61006, 0.53275],
[0.98934, 0.60069, 0.52207],
[0.98865, 0.5913, 0.51145],
[0.98786, 0.58189, 0.50088],
[0.98695, 0.57248, 0.49038],
[0.9859, 0.56306, 0.47994],
[0.98472, 0.55364, 0.46959],
[0.98338, 0.54423, 0.45933],
[0.98188, 0.53484, 0.44917],
[0.9802, 0.52548, 0.43912],
[0.97832, 0.51615, 0.42922],
[0.97626, 0.50687, 0.41945],
[0.97398, 0.49766, 0.40983],
[0.97149, 0.48849, 0.40037],
[0.96878, 0.4794, 0.39107],
[0.96585, 0.47039, 0.38193],
[0.96271, 0.46144, 0.37296],
[0.95935, 0.45259, 0.36416],
[0.9558, 0.4438, 0.35553],
[0.95206, 0.43508, 0.34704],
[0.94815, 0.42642, 0.33868],
[0.94407, 0.41783, 0.33046],
[0.93986, 0.40928, 0.32236],
[0.93552, 0.40077, 0.31435],
[0.93108, 0.39227, 0.30646],
[0.92653, 0.38379, 0.29862],
[0.92191, 0.37531, 0.29086],
[0.91723, 0.36682, 0.28317],
[0.91247, 0.35831, 0.27552],
[0.90768, 0.34976, 0.26789],
[0.90284, 0.34117, 0.26033],
[0.89796, 0.33254, 0.25279],
[0.89306, 0.32382, 0.24525],
[0.88813, 0.31504, 0.23778],
[0.88317, 0.3062, 0.23027],
[0.87818, 0.29724, 0.22284],
[0.87318, 0.28817, 0.21537],
[0.86815, 0.27902, 0.20794],
[0.8631, 0.26973, 0.2005],
[0.85801, 0.2603, 0.19312],
[0.85291, 0.25073, 0.18575],
[0.84776, 0.241, 0.17838],
[0.84258, 0.23113, 0.171],
[0.83735, 0.22106, 0.16371],
[0.83206, 0.21078, 0.15643],
[0.8267, 0.2003, 0.14918],
[0.82126, 0.18965, 0.14201],
[0.81571, 0.17878, 0.13493],
[0.81004, 0.16767, 0.12792],
[0.80423, 0.1564, 0.12101],
[0.79825, 0.14491, 0.11428],
[0.79207, 0.13322, 0.10775],
[0.78568, 0.12135, 0.10141],
[0.77903, 0.10946, 0.095344],
[0.77213, 0.097351, 0.089503],
[0.76494, 0.085214, 0.084009],
[0.75744, 0.073154, 0.078743],
[0.74963, 0.060978, 0.073995],
[0.74151, 0.049092, 0.06957],
[0.73309, 0.037259, 0.065437],
[0.72436, 0.026823, 0.061723],
[0.71534, 0.018618, 0.058443],
[0.70605, 0.012283, 0.055532],
[0.69652, 0.007339, 0.052912],
[0.68678, 0.0040371, 0.05068],
[0.67685, 0.0019439, 0.04877],
[0.66675, 0.00085262, 0.047012],
[0.65654, 0.00057092, 0.045448],
[0.64622, 0.00092486, 0.044119],
[0.63582, 0.0017616, 0.042871],
[0.62537, 0.0029503, 0.041932],
[0.61487, 0.0043822, 0.040902],
[0.60438, 0.0059666, 0.039904],
[0.59386, 0.0076341, 0.038882],
[0.58336, 0.0093374, 0.037833],
[0.57288, 0.011077, 0.036734],
[0.56242, 0.012852, 0.03563],
[0.55198, 0.014445, 0.034215],
[0.54157, 0.015957, 0.032974],
[0.5312, 0.017388, 0.031658],
[0.52087, 0.018728, 0.030292],
[0.51058, 0.019973, 0.028861],
[0.50033, 0.021123, 0.027367],
[0.49012, 0.02218, 0.025811],
[0.47996, 0.023147, 0.024193],
[0.46984, 0.024025, 0.022515],
[0.45976, 0.024816, 0.020778],
[0.44973, 0.025523, 0.018981],
[0.43974, 0.026148, 0.017126],
[0.4298, 0.026693, 0.015214],
[0.41991, 0.027159, 0.01324],
[0.41008, 0.027551, 0.011092],
[0.40028, 0.027868, 0.0089508],
]
b_diverging_bwr_20_95_c54 = bokeh_palette('diverging_bwr_20_95_c54',diverging_bwr_20_95_c54)
m_diverging_bwr_20_95_c54 = mpl_cm('diverging_bwr_20_95_c54',diverging_bwr_20_95_c54)
m_diverging_bwr_20_95_c54_r = mpl_cm('diverging_bwr_20_95_c54_r',list(reversed(diverging_bwr_20_95_c54)))
CET_D1A = b_diverging_bwr_20_95_c54
m_CET_D1A = m_diverging_bwr_20_95_c54
m_CET_D1A_r = m_diverging_bwr_20_95_c54_r
palette['CET_D1A'] = b_diverging_bwr_20_95_c54
cm['CET_D1A'] = m_diverging_bwr_20_95_c54
cm['CET_D1A_r'] = m_diverging_bwr_20_95_c54_r
register_cmap('cet_CET_D1A',m_diverging_bwr_20_95_c54)
register_cmap('cet_CET_D1A_r',m_diverging_bwr_20_95_c54_r)




linear_tritanopic_krjcw_5_98_c46 = [  # cmap_def
[0.066001, 0.066024, 0.066019],
[0.078225, 0.067325, 0.065441],
[0.089145, 0.068628, 0.064864],
[0.098951, 0.069855, 0.064281],
[0.10795, 0.071087, 0.063702],
[0.11636, 0.072248, 0.063142],
[0.12443, 0.073387, 0.062556],
[0.13261, 0.074138, 0.061908],
[0.14082, 0.07462, 0.061308],
[0.14899, 0.074903, 0.060792],
[0.15715, 0.075095, 0.060285],
[0.16526, 0.075197, 0.059744],
[0.17337, 0.075206, 0.059135],
[0.18144, 0.075118, 0.058479],
[0.18952, 0.074929, 0.057832],
[0.19757, 0.074639, 0.057263],
[0.20562, 0.074251, 0.056707],
[0.21365, 0.073764, 0.056064],
[0.22168, 0.073113, 0.055395],
[0.2297, 0.072287, 0.054742],
[0.23776, 0.071458, 0.054088],
[0.24577, 0.070377, 0.053435],
[0.25381, 0.069302, 0.052789],
[0.26183, 0.067939, 0.052151],
[0.26987, 0.066459, 0.051527],
[0.27787, 0.064849, 0.050918],
[0.28586, 0.063128, 0.050324],
[0.29385, 0.0611, 0.049773],
[0.30182, 0.059091, 0.049266],
[0.30978, 0.056819, 0.048734],
[0.31768, 0.05432, 0.048193],
[0.32554, 0.051716, 0.047758],
[0.33339, 0.049059, 0.047443],
[0.34115, 0.045994, 0.047203],
[0.34886, 0.0429, 0.047029],
[0.35651, 0.039762, 0.046938],
[0.36406, 0.036455, 0.046949],
[0.37153, 0.033025, 0.047066],
[0.37893, 0.029848, 0.047282],
[0.38622, 0.026784, 0.047594],
[0.39342, 0.023855, 0.048064],
[0.40053, 0.021083, 0.048744],
[0.40752, 0.018491, 0.049455],
[0.41441, 0.016096, 0.050213],
[0.4212, 0.013914, 0.051116],
[0.42791, 0.011974, 0.052114],
[0.43451, 0.010004, 0.053214],
[0.44102, 0.0085084, 0.054407],
[0.44746, 0.0072282, 0.055682],
[0.45383, 0.0061645, 0.056992],
[0.46012, 0.0053117, 0.058278],
[0.46636, 0.0046574, 0.059757],
[0.47255, 0.0041885, 0.061036],
[0.4787, 0.0038901, 0.062535],
[0.4848, 0.003747, 0.063902],
[0.4909, 0.0037436, 0.065296],
[0.49695, 0.0038647, 0.066699],
[0.50299, 0.0040958, 0.068108],
[0.50903, 0.0044251, 0.069543],
[0.51505, 0.004841, 0.070914],
[0.52106, 0.0053325, 0.07231],
[0.52708, 0.00589, 0.073829],
[0.53309, 0.006505, 0.075226],
[0.53911, 0.0071713, 0.076672],
[0.54514, 0.0078814, 0.07815],
[0.55117, 0.0086309, 0.07966],
[0.55722, 0.009427, 0.081199],
[0.56327, 0.010214, 0.082689],
[0.56932, 0.011173, 0.084233],
[0.57539, 0.012194, 0.085652],
[0.58147, 0.013064, 0.087261],
[0.58756, 0.014003, 0.088744],
[0.59367, 0.014954, 0.090266],
[0.59979, 0.015932, 0.091819],
[0.60591, 0.016934, 0.093269],
[0.61205, 0.017961, 0.094878],
[0.6182, 0.019014, 0.096325],
[0.62437, 0.020091, 0.097933],
[0.63054, 0.021195, 0.099434],
[0.63673, 0.022324, 0.10097],
[0.64292, 0.02348, 0.10253],
[0.64913, 0.024662, 0.10407],
[0.65534, 0.02587, 0.10563],
[0.66157, 0.027106, 0.10721],
[0.66781, 0.028369, 0.10873],
[0.67405, 0.02966, 0.11033],
[0.68031, 0.030979, 0.11189],
[0.68658, 0.032325, 0.11346],
[0.69287, 0.033673, 0.11497],
[0.69915, 0.035312, 0.11659],
[0.70546, 0.036723, 0.11817],
[0.71177, 0.03819, 0.11971],
[0.71809, 0.039682, 0.12127],
[0.72443, 0.041205, 0.12285],
[0.73077, 0.042628, 0.12445],
[0.73712, 0.044143, 0.12608],
[0.74349, 0.045602, 0.12765],
[0.74986, 0.047156, 0.12923],
[0.75624, 0.048669, 0.13087],
[0.76264, 0.050114, 0.13245],
[0.76904, 0.051566, 0.13406],
[0.77545, 0.053037, 0.13563],
[0.78188, 0.054536, 0.13727],
[0.78831, 0.05607, 0.13886],
[0.79476, 0.057476, 0.14049],
[0.80121, 0.05904, 0.1421],
[0.80767, 0.060497, 0.14372],
[0.81414, 0.061942, 0.14537],
[0.82063, 0.06346, 0.14695],
[0.82712, 0.064908, 0.14856],
[0.83363, 0.06638, 0.15023],
[0.84014, 0.067875, 0.15183],
[0.84665, 0.069402, 0.15346],
[0.85318, 0.070821, 0.15511],
[0.85972, 0.072295, 0.15677],
[0.86625, 0.073902, 0.15844],
[0.87278, 0.075433, 0.16006],
[0.8793, 0.077071, 0.1618],
[0.88581, 0.078831, 0.1635],
[0.89231, 0.080742, 0.16519],
[0.89877, 0.082767, 0.16701],
[0.9052, 0.084981, 0.16882],
[0.91157, 0.087474, 0.17068],
[0.91789, 0.090182, 0.17261],
[0.92411, 0.093217, 0.17466],
[0.93023, 0.096697, 0.17678],
[0.93623, 0.10069, 0.17905],
[0.94207, 0.10517, 0.18144],
[0.94772, 0.11031, 0.18404],
[0.95316, 0.11603, 0.18683],
[0.95834, 0.12238, 0.18985],
[0.96324, 0.12954, 0.19315],
[0.96781, 0.1374, 0.19676],
[0.97201, 0.14596, 0.20063],
[0.97581, 0.15522, 0.20494],
[0.97918, 0.16516, 0.2096],
[0.98208, 0.17576, 0.21468],
[0.98448, 0.18698, 0.22023],
[0.98634, 0.1987, 0.2262],
[0.98765, 0.21091, 0.23263],
[0.98839, 0.22353, 0.23956],
[0.98856, 0.23648, 0.24696],
[0.98815, 0.24967, 0.25486],
[0.98715, 0.26312, 0.26319],
[0.98557, 0.27669, 0.27197],
[0.98343, 0.29032, 0.28121],
[0.98073, 0.30401, 0.29087],
[0.97751, 0.31768, 0.3009],
[0.97377, 0.33127, 0.31131],
[0.96955, 0.34479, 0.32204],
[0.96488, 0.35817, 0.33309],
[0.95977, 0.37138, 0.34438],
[0.95425, 0.38442, 0.35595],
[0.94834, 0.39727, 0.36771],
[0.94208, 0.40993, 0.37968],
[0.93548, 0.42235, 0.39182],
[0.92854, 0.43458, 0.4041],
[0.92131, 0.44659, 0.41651],
[0.91378, 0.45838, 0.42903],
[0.90595, 0.46998, 0.44166],
[0.89784, 0.48135, 0.45437],
[0.88946, 0.49253, 0.46715],
[0.88079, 0.50352, 0.48],
[0.87184, 0.51435, 0.49289],
[0.86261, 0.52498, 0.50586],
[0.85307, 0.53545, 0.51889],
[0.84324, 0.54577, 0.53194],
[0.83309, 0.55592, 0.54505],
[0.82262, 0.56595, 0.55819],
[0.8118, 0.57585, 0.57138],
[0.80063, 0.58561, 0.58461],
[0.78908, 0.59524, 0.59787],
[0.77712, 0.60477, 0.61117],
[0.76474, 0.61418, 0.62451],
[0.75191, 0.62351, 0.63788],
[0.7386, 0.63271, 0.65128],
[0.7248, 0.64183, 0.6647],
[0.71047, 0.65084, 0.67815],
[0.69557, 0.65977, 0.6916],
[0.68008, 0.6686, 0.70507],
[0.66397, 0.67734, 0.71853],
[0.64719, 0.68598, 0.73198],
[0.6297, 0.69452, 0.74539],
[0.6115, 0.70298, 0.75876],
[0.59254, 0.71132, 0.77207],
[0.57278, 0.71956, 0.78528],
[0.55221, 0.72767, 0.79837],
[0.53079, 0.73566, 0.81131],
[0.50854, 0.74352, 0.82406],
[0.4854, 0.75124, 0.83659],
[0.46143, 0.75879, 0.84885],
[0.43661, 0.76618, 0.8608],
[0.41099, 0.77339, 0.8724],
[0.38459, 0.78041, 0.8836],
[0.35755, 0.78723, 0.89435],
[0.32991, 0.79384, 0.90461],
[0.30184, 0.80022, 0.91435],
[0.27364, 0.80637, 0.92353],
[0.24554, 0.81229, 0.93211],
[0.21813, 0.81797, 0.94008],
[0.19209, 0.82341, 0.94743],
[0.16858, 0.82862, 0.95413],
[0.14919, 0.83358, 0.96019],
[0.13588, 0.83831, 0.96563],
[0.13065, 0.84281, 0.97045],
[0.13412, 0.84709, 0.97469],
[0.14552, 0.85118, 0.97836],
[0.16281, 0.85507, 0.98151],
[0.18396, 0.85878, 0.98417],
[0.20739, 0.86233, 0.98639],
[0.23195, 0.86572, 0.9882],
[0.25692, 0.86898, 0.98965],
[0.28187, 0.87212, 0.99078],
[0.30652, 0.87515, 0.99163],
[0.33068, 0.87809, 0.99225],
[0.35429, 0.88095, 0.99266],
[0.37727, 0.88374, 0.99291],
[0.39958, 0.88647, 0.99301],
[0.42126, 0.88914, 0.99299],
[0.44232, 0.89178, 0.99288],
[0.46272, 0.89438, 0.99269],
[0.48256, 0.89695, 0.99244],
[0.50184, 0.8995, 0.99215],
[0.52057, 0.90202, 0.99181],
[0.53883, 0.90452, 0.99145],
[0.55662, 0.90701, 0.99107],
[0.57397, 0.90948, 0.99067],
[0.59094, 0.91194, 0.99025],
[0.60754, 0.9144, 0.98983],
[0.62378, 0.91684, 0.9894],
[0.63969, 0.91926, 0.98897],
[0.65529, 0.92169, 0.98853],
[0.67063, 0.92411, 0.98809],
[0.6857, 0.92651, 0.98766],
[0.70054, 0.92891, 0.98722],
[0.71516, 0.93131, 0.98677],
[0.72957, 0.93369, 0.98632],
[0.74379, 0.93607, 0.98587],
[0.75783, 0.93843, 0.98542],
[0.77171, 0.94079, 0.98497],
[0.78542, 0.94314, 0.98451],
[0.79899, 0.94548, 0.98405],
[0.81241, 0.94781, 0.98358],
[0.82571, 0.95014, 0.98312],
[0.83888, 0.95245, 0.98265],
[0.85194, 0.95476, 0.98218],
[0.86488, 0.95706, 0.98171],
[0.87771, 0.95935, 0.98123],
[0.89045, 0.96163, 0.98075],
[0.90309, 0.9639, 0.98027],
[0.91564, 0.96617, 0.97979],
[0.9281, 0.96843, 0.9793],
[0.94049, 0.97068, 0.97881],
[0.95279, 0.97292, 0.97831],
[0.96502, 0.97515, 0.97782],
[0.97718, 0.97737, 0.97732],
]
b_linear_tritanopic_krjcw_5_98_c46 = bokeh_palette('linear_tritanopic_krjcw_5_98_c46',linear_tritanopic_krjcw_5_98_c46)
m_linear_tritanopic_krjcw_5_98_c46 = mpl_cm('linear_tritanopic_krjcw_5_98_c46',linear_tritanopic_krjcw_5_98_c46)
m_linear_tritanopic_krjcw_5_98_c46_r = mpl_cm('linear_tritanopic_krjcw_5_98_c46_r',list(reversed(linear_tritanopic_krjcw_5_98_c46)))
CET_CBTL1 = b_linear_tritanopic_krjcw_5_98_c46
m_CET_CBTL1 = m_linear_tritanopic_krjcw_5_98_c46
m_CET_CBTL1_r = m_linear_tritanopic_krjcw_5_98_c46_r
palette['CET_CBTL1'] = b_linear_tritanopic_krjcw_5_98_c46
cm['CET_CBTL1'] = m_linear_tritanopic_krjcw_5_98_c46
cm['CET_CBTL1_r'] = m_linear_tritanopic_krjcw_5_98_c46_r
register_cmap('cet_CET_CBTL1',m_linear_tritanopic_krjcw_5_98_c46)
register_cmap('cet_CET_CBTL1_r',m_linear_tritanopic_krjcw_5_98_c46_r)




cyclic_protanopic_deuteranopic_wywb_55_96_c33 = [  # cmap_def
[0.93344, 0.92965, 0.9265],
[0.93483, 0.9285, 0.91829],
[0.93521, 0.92627, 0.90898],
[0.93467, 0.92304, 0.89864],
[0.9333, 0.9189, 0.88737],
[0.93123, 0.91403, 0.87529],
[0.92857, 0.90852, 0.86255],
[0.92545, 0.90254, 0.84929],
[0.92195, 0.8962, 0.83562],
[0.91818, 0.88959, 0.82167],
[0.9142, 0.88281, 0.80752],
[0.91006, 0.87592, 0.79326],
[0.9058, 0.86896, 0.77891],
[0.90144, 0.86196, 0.76454],
[0.89701, 0.85495, 0.75015],
[0.89251, 0.84793, 0.73577],
[0.88796, 0.84092, 0.7214],
[0.88336, 0.83393, 0.70707],
[0.87871, 0.82695, 0.69276],
[0.87401, 0.81998, 0.67847],
[0.86926, 0.81303, 0.66421],
[0.86446, 0.8061, 0.64997],
[0.85961, 0.79918, 0.63576],
[0.85471, 0.79227, 0.62156],
[0.84977, 0.78539, 0.6074],
[0.84477, 0.77851, 0.59325],
[0.83973, 0.77165, 0.57912],
[0.83463, 0.76481, 0.565],
[0.82949, 0.75798, 0.55091],
[0.8243, 0.75118, 0.53683],
[0.81906, 0.74438, 0.52275],
[0.81377, 0.73759, 0.50869],
[0.80844, 0.73083, 0.49463],
[0.80306, 0.72408, 0.48056],
[0.79763, 0.71736, 0.46651],
[0.79215, 0.71064, 0.45246],
[0.78663, 0.70394, 0.43838],
[0.78106, 0.69725, 0.4243],
[0.77544, 0.69059, 0.4102],
[0.76978, 0.68394, 0.39607],
[0.76407, 0.67731, 0.3819],
[0.75832, 0.6707, 0.36769],
[0.75251, 0.66409, 0.35344],
[0.74667, 0.65752, 0.33912],
[0.74077, 0.65095, 0.3247],
[0.73484, 0.64439, 0.31022],
[0.72885, 0.63786, 0.29559],
[0.72282, 0.63135, 0.28082],
[0.71676, 0.62485, 0.26592],
[0.71064, 0.61837, 0.25077],
[0.70448, 0.61191, 0.23541],
[0.6983, 0.60548, 0.21975],
[0.6921, 0.59911, 0.20379],
[0.6859, 0.5928, 0.18746],
[0.67973, 0.58656, 0.1707],
[0.67364, 0.58047, 0.15351],
[0.66769, 0.57456, 0.13583],
[0.66195, 0.56892, 0.11768],
[0.65653, 0.56364, 0.098985],
[0.65154, 0.55879, 0.079761],
[0.64711, 0.55454, 0.060162],
[0.64338, 0.55098, 0.040145],
[0.6405, 0.54824, 0.022737],
[0.63856, 0.5464, 0.011467],
[0.63764, 0.54552, 0.00612],
[0.63779, 0.54566, 0.0069528],
[0.639, 0.54681, 0.014011],
[0.6412, 0.54891, 0.026881],
[0.64433, 0.55189, 0.045584],
[0.64825, 0.55564, 0.065542],
[0.65284, 0.56007, 0.085072],
[0.65797, 0.56502, 0.10412],
[0.66348, 0.57042, 0.12263],
[0.66928, 0.57615, 0.14071],
[0.67528, 0.5821, 0.15822],
[0.68138, 0.58823, 0.17526],
[0.68757, 0.59448, 0.19187],
[0.69376, 0.60081, 0.2081],
[0.69996, 0.60722, 0.22396],
[0.70614, 0.61364, 0.23954],
[0.71228, 0.62011, 0.25487],
[0.71839, 0.62659, 0.26994],
[0.72445, 0.6331, 0.2848],
[0.73047, 0.63962, 0.29953],
[0.73643, 0.64615, 0.31411],
[0.74236, 0.6527, 0.32858],
[0.74824, 0.65927, 0.34297],
[0.75408, 0.66586, 0.35728],
[0.75986, 0.67247, 0.3715],
[0.76561, 0.67908, 0.38569],
[0.77131, 0.68572, 0.39985],
[0.77696, 0.69238, 0.41398],
[0.78256, 0.69904, 0.42808],
[0.78811, 0.70573, 0.44216],
[0.79363, 0.71243, 0.45622],
[0.79909, 0.71916, 0.47028],
[0.80451, 0.7259, 0.48433],
[0.80987, 0.73265, 0.4984],
[0.81519, 0.73942, 0.51245],
[0.82047, 0.7462, 0.52653],
[0.82569, 0.75299, 0.5406],
[0.83087, 0.75981, 0.55468],
[0.836, 0.76664, 0.56878],
[0.84108, 0.77349, 0.5829],
[0.84611, 0.78035, 0.59704],
[0.85109, 0.78723, 0.6112],
[0.85603, 0.79413, 0.62537],
[0.86092, 0.80103, 0.63957],
[0.86575, 0.80795, 0.65379],
[0.87054, 0.81489, 0.66803],
[0.87528, 0.82185, 0.68229],
[0.87996, 0.82882, 0.69659],
[0.8846, 0.8358, 0.71091],
[0.88918, 0.8428, 0.72527],
[0.8937, 0.84979, 0.73965],
[0.89813, 0.85678, 0.75405],
[0.90247, 0.86374, 0.76846],
[0.90668, 0.87063, 0.78288],
[0.91072, 0.87743, 0.79727],
[0.91454, 0.88408, 0.8116],
[0.91805, 0.89049, 0.82582],
[0.92115, 0.89657, 0.83985],
[0.92376, 0.90221, 0.85359],
[0.92572, 0.90727, 0.86693],
[0.92693, 0.9116, 0.87972],
[0.92726, 0.91509, 0.89181],
[0.92662, 0.9176, 0.90305],
[0.92493, 0.91903, 0.91331],
[0.92216, 0.91934, 0.92247],
[0.91831, 0.91852, 0.93048],
[0.91344, 0.91658, 0.93732],
[0.90763, 0.91362, 0.94303],
[0.90098, 0.90973, 0.94767],
[0.89363, 0.90506, 0.95137],
[0.8857, 0.89973, 0.95427],
[0.87731, 0.89389, 0.95649],
[0.86857, 0.88766, 0.9582],
[0.85956, 0.88115, 0.95951],
[0.85035, 0.87445, 0.96053],
[0.84101, 0.86762, 0.96134],
[0.83155, 0.86072, 0.96201],
[0.822, 0.85377, 0.9626],
[0.81238, 0.8468, 0.96312],
[0.8027, 0.83983, 0.9636],
[0.79297, 0.83286, 0.96404],
[0.78318, 0.82591, 0.96448],
[0.77333, 0.81897, 0.9649],
[0.76342, 0.81205, 0.96531],
[0.75346, 0.80515, 0.96572],
[0.74343, 0.79826, 0.96611],
[0.73332, 0.79138, 0.96649],
[0.72314, 0.78453, 0.96686],
[0.71289, 0.77769, 0.96722],
[0.70257, 0.77086, 0.96758],
[0.69216, 0.76405, 0.96792],
[0.68165, 0.75726, 0.96824],
[0.67106, 0.75048, 0.96856],
[0.66036, 0.74372, 0.96887],
[0.64957, 0.73698, 0.96917],
[0.63866, 0.73026, 0.96946],
[0.62763, 0.72354, 0.96974],
[0.61649, 0.71686, 0.97001],
[0.6052, 0.71018, 0.97027],
[0.59378, 0.70353, 0.97052],
[0.5822, 0.69689, 0.97076],
[0.57047, 0.69028, 0.97099],
[0.55857, 0.68368, 0.97122],
[0.54649, 0.6771, 0.97143],
[0.5342, 0.67054, 0.97163],
[0.52169, 0.664, 0.97183],
[0.50897, 0.65748, 0.97202],
[0.49597, 0.65098, 0.9722],
[0.48272, 0.64449, 0.97236],
[0.46916, 0.63803, 0.97252],
[0.45526, 0.63159, 0.97267],
[0.44099, 0.62517, 0.97282],
[0.42633, 0.61878, 0.97295],
[0.41125, 0.61241, 0.97307],
[0.39571, 0.60609, 0.97319],
[0.37968, 0.59984, 0.9733],
[0.36317, 0.59366, 0.9734],
[0.34619, 0.58759, 0.97348],
[0.32882, 0.58169, 0.97356],
[0.31115, 0.57603, 0.97363],
[0.29341, 0.57065, 0.97369],
[0.27598, 0.56568, 0.97374],
[0.25925, 0.56122, 0.97379],
[0.24386, 0.5574, 0.97382],
[0.23068, 0.55429, 0.97384],
[0.22054, 0.55203, 0.97386],
[0.21423, 0.55068, 0.97387],
[0.2124, 0.5503, 0.97387],
[0.21521, 0.55089, 0.97387],
[0.22242, 0.55244, 0.97386],
[0.23329, 0.55489, 0.97384],
[0.24704, 0.55816, 0.97381],
[0.26275, 0.56215, 0.97378],
[0.27969, 0.56672, 0.97373],
[0.29725, 0.57178, 0.97368],
[0.31497, 0.57722, 0.97362],
[0.3326, 0.58294, 0.97355],
[0.3499, 0.58888, 0.97347],
[0.36678, 0.59498, 0.97338],
[0.38318, 0.60118, 0.97328],
[0.39909, 0.60746, 0.97317],
[0.41454, 0.61378, 0.97305],
[0.42953, 0.62016, 0.97292],
[0.44411, 0.62655, 0.97279],
[0.45828, 0.63298, 0.97264],
[0.47211, 0.63943, 0.97249],
[0.4856, 0.64589, 0.97233],
[0.4988, 0.65237, 0.97216],
[0.51173, 0.65888, 0.97198],
[0.52441, 0.66541, 0.97179],
[0.53687, 0.67195, 0.97159],
[0.54911, 0.67852, 0.97138],
[0.56115, 0.6851, 0.97117],
[0.57302, 0.6917, 0.97094],
[0.58472, 0.69832, 0.97071],
[0.59625, 0.70496, 0.97047],
[0.60765, 0.71162, 0.97021],
[0.6189, 0.71829, 0.96995],
[0.63002, 0.72499, 0.96968],
[0.64101, 0.7317, 0.9694],
[0.6519, 0.73843, 0.96911],
[0.66268, 0.74518, 0.96881],
[0.67335, 0.75194, 0.96849],
[0.68392, 0.75872, 0.96817],
[0.6944, 0.76552, 0.96784],
[0.7048, 0.77233, 0.9675],
[0.71511, 0.77916, 0.96715],
[0.72535, 0.78601, 0.96678],
[0.73551, 0.79287, 0.96641],
[0.74559, 0.79975, 0.96603],
[0.75561, 0.80664, 0.96563],
[0.76557, 0.81354, 0.96523],
[0.77546, 0.82047, 0.96481],
[0.7853, 0.82741, 0.96438],
[0.79507, 0.83437, 0.96395],
[0.80479, 0.84133, 0.9635],
[0.81446, 0.84832, 0.96304],
[0.82408, 0.85531, 0.96257],
[0.83363, 0.86231, 0.96207],
[0.8431, 0.86928, 0.96154],
[0.85248, 0.87621, 0.96097],
[0.86173, 0.88308, 0.96032],
[0.87082, 0.88983, 0.95956],
[0.87969, 0.89641, 0.95863],
[0.88827, 0.90273, 0.95748],
[0.89645, 0.90869, 0.956],
[0.90413, 0.91417, 0.95409],
[0.9112, 0.91902, 0.95162],
[0.91753, 0.92313, 0.94847],
[0.92299, 0.92634, 0.94449],
[0.92751, 0.92854, 0.93956],
[0.93101, 0.92967, 0.93358],
]
b_cyclic_protanopic_deuteranopic_wywb_55_96_c33 = bokeh_palette('cyclic_protanopic_deuteranopic_wywb_55_96_c33',cyclic_protanopic_deuteranopic_wywb_55_96_c33)
m_cyclic_protanopic_deuteranopic_wywb_55_96_c33 = mpl_cm('cyclic_protanopic_deuteranopic_wywb_55_96_c33',cyclic_protanopic_deuteranopic_wywb_55_96_c33)
m_cyclic_protanopic_deuteranopic_wywb_55_96_c33_r = mpl_cm('cyclic_protanopic_deuteranopic_wywb_55_96_c33_r',list(reversed(cyclic_protanopic_deuteranopic_wywb_55_96_c33)))
CET_CBC2 = b_cyclic_protanopic_deuteranopic_wywb_55_96_c33
m_CET_CBC2 = m_cyclic_protanopic_deuteranopic_wywb_55_96_c33
m_CET_CBC2_r = m_cyclic_protanopic_deuteranopic_wywb_55_96_c33_r
palette['CET_CBC2'] = b_cyclic_protanopic_deuteranopic_wywb_55_96_c33
cm['CET_CBC2'] = m_cyclic_protanopic_deuteranopic_wywb_55_96_c33
cm['CET_CBC2_r'] = m_cyclic_protanopic_deuteranopic_wywb_55_96_c33_r
register_cmap('cet_CET_CBC2',m_cyclic_protanopic_deuteranopic_wywb_55_96_c33)
register_cmap('cet_CET_CBC2_r',m_cyclic_protanopic_deuteranopic_wywb_55_96_c33_r)




diverging_bwg_20_95_c41 = [  # cmap_def
[0.065559, 0.17633, 0.40763],
[0.069881, 0.18111, 0.41595],
[0.074129, 0.18595, 0.42428],
[0.078264, 0.19074, 0.43264],
[0.082501, 0.1956, 0.44099],
[0.08661, 0.20046, 0.44937],
[0.090704, 0.20539, 0.45774],
[0.094774, 0.21032, 0.46612],
[0.098735, 0.21529, 0.47452],
[0.1027, 0.22031, 0.4829],
[0.10665, 0.22531, 0.4913],
[0.11051, 0.23036, 0.49967],
[0.11429, 0.23548, 0.50807],
[0.11813, 0.24057, 0.51644],
[0.12182, 0.24573, 0.52482],
[0.12558, 0.25093, 0.53318],
[0.12921, 0.25613, 0.54154],
[0.13285, 0.26139, 0.5499],
[0.13639, 0.26666, 0.55823],
[0.13991, 0.27196, 0.56657],
[0.14342, 0.27733, 0.57488],
[0.14683, 0.28272, 0.58318],
[0.15024, 0.28811, 0.59147],
[0.15353, 0.29356, 0.59974],
[0.15683, 0.29905, 0.608],
[0.16001, 0.30456, 0.61623],
[0.16322, 0.31012, 0.62444],
[0.16631, 0.31571, 0.63263],
[0.16938, 0.32132, 0.64079],
[0.17232, 0.32697, 0.64893],
[0.17525, 0.33268, 0.65705],
[0.17812, 0.33839, 0.66512],
[0.18085, 0.34416, 0.67318],
[0.18355, 0.34998, 0.68119],
[0.18619, 0.35582, 0.68919],
[0.18872, 0.3617, 0.69713],
[0.19116, 0.36761, 0.70504],
[0.19353, 0.37358, 0.7129],
[0.19579, 0.37958, 0.72073],
[0.19795, 0.38561, 0.72851],
[0.2, 0.39171, 0.73623],
[0.20196, 0.39784, 0.74392],
[0.20382, 0.40401, 0.75155],
[0.20551, 0.41023, 0.75911],
[0.2071, 0.41648, 0.76662],
[0.20853, 0.42278, 0.77406],
[0.20982, 0.42914, 0.78144],
[0.21096, 0.43555, 0.78875],
[0.21194, 0.44201, 0.79598],
[0.21271, 0.4485, 0.80314],
[0.2133, 0.45506, 0.81021],
[0.2137, 0.46166, 0.81719],
[0.21388, 0.46832, 0.82408],
[0.21383, 0.47505, 0.83088],
[0.21352, 0.48182, 0.83757],
[0.21296, 0.48864, 0.84414],
[0.21218, 0.49553, 0.85061],
[0.2111, 0.50247, 0.85695],
[0.20977, 0.50946, 0.86317],
[0.20821, 0.51651, 0.86925],
[0.20647, 0.52361, 0.87519],
[0.20454, 0.53074, 0.88099],
[0.20251, 0.53792, 0.88665],
[0.20048, 0.54511, 0.89216],
[0.19857, 0.55233, 0.89751],
[0.19689, 0.55955, 0.90271],
[0.19551, 0.56678, 0.90775],
[0.19465, 0.57399, 0.91264],
[0.19438, 0.58119, 0.91737],
[0.19482, 0.58836, 0.92195],
[0.19609, 0.59551, 0.92637],
[0.19826, 0.60262, 0.93063],
[0.2014, 0.60968, 0.93474],
[0.20555, 0.61671, 0.9387],
[0.21065, 0.62369, 0.9425],
[0.21672, 0.63062, 0.94614],
[0.22367, 0.63749, 0.94963],
[0.2315, 0.64432, 0.95297],
[0.24007, 0.65111, 0.95614],
[0.2494, 0.65785, 0.95916],
[0.2594, 0.66454, 0.96202],
[0.26998, 0.67118, 0.96473],
[0.28107, 0.67778, 0.96729],
[0.29267, 0.68432, 0.96967],
[0.3047, 0.69082, 0.97191],
[0.31713, 0.69727, 0.97398],
[0.32991, 0.70368, 0.97589],
[0.343, 0.71004, 0.97765],
[0.35638, 0.71636, 0.97924],
[0.37, 0.72261, 0.98067],
[0.38389, 0.72883, 0.98194],
[0.39799, 0.735, 0.98305],
[0.41229, 0.74112, 0.984],
[0.42677, 0.7472, 0.98478],
[0.44143, 0.75322, 0.9854],
[0.45624, 0.7592, 0.98586],
[0.4712, 0.76513, 0.98615],
[0.48629, 0.77102, 0.98628],
[0.50151, 0.77685, 0.98625],
[0.51683, 0.78263, 0.98606],
[0.53224, 0.78837, 0.98572],
[0.54774, 0.79407, 0.98522],
[0.5633, 0.79972, 0.98458],
[0.57889, 0.80532, 0.98379],
[0.5945, 0.8109, 0.98287],
[0.61012, 0.81644, 0.98183],
[0.62571, 0.82194, 0.98068],
[0.64124, 0.82742, 0.97942],
[0.65671, 0.83287, 0.97808],
[0.67208, 0.83831, 0.97666],
[0.68733, 0.84374, 0.97517],
[0.70246, 0.84916, 0.97364],
[0.71747, 0.85457, 0.97206],
[0.73231, 0.85999, 0.97043],
[0.74696, 0.86538, 0.96876],
[0.76141, 0.87077, 0.96704],
[0.7756, 0.87614, 0.96527],
[0.78949, 0.88146, 0.96342],
[0.80297, 0.88672, 0.96147],
[0.81594, 0.89187, 0.95939],
[0.82822, 0.89686, 0.95713],
[0.83964, 0.90163, 0.95464],
[0.84994, 0.9061, 0.95186],
[0.85886, 0.91018, 0.94874],
[0.86615, 0.91378, 0.94521],
[0.87156, 0.91681, 0.94123],
[0.87485, 0.91918, 0.93679],
[0.87588, 0.92084, 0.93184],
[0.87458, 0.92175, 0.92642],
[0.87097, 0.9219, 0.92054],
[0.86513, 0.92131, 0.91426],
[0.85725, 0.92003, 0.90762],
[0.84755, 0.91814, 0.90069],
[0.83631, 0.9157, 0.89353],
[0.82379, 0.91284, 0.8862],
[0.81025, 0.90961, 0.87874],
[0.79592, 0.90611, 0.8712],
[0.78101, 0.9024, 0.86361],
[0.76566, 0.89854, 0.85599],
[0.74999, 0.89458, 0.84835],
[0.73409, 0.89052, 0.84071],
[0.71801, 0.88641, 0.83307],
[0.70179, 0.88225, 0.82542],
[0.68547, 0.87805, 0.81778],
[0.66907, 0.87381, 0.81014],
[0.65261, 0.86954, 0.80249],
[0.63612, 0.86522, 0.79482],
[0.6196, 0.86087, 0.78713],
[0.60306, 0.85647, 0.7794],
[0.58654, 0.85202, 0.77164],
[0.57006, 0.84752, 0.76383],
[0.55365, 0.84296, 0.75597],
[0.53733, 0.83834, 0.74804],
[0.5211, 0.83366, 0.74004],
[0.50501, 0.82891, 0.73196],
[0.48906, 0.82409, 0.72381],
[0.47327, 0.81921, 0.71558],
[0.45768, 0.81427, 0.70725],
[0.44227, 0.80927, 0.69884],
[0.42706, 0.80419, 0.69034],
[0.41205, 0.79906, 0.68175],
[0.39726, 0.79387, 0.67309],
[0.38271, 0.78861, 0.66432],
[0.3684, 0.78329, 0.65547],
[0.35433, 0.77792, 0.64653],
[0.34053, 0.77248, 0.6375],
[0.32698, 0.767, 0.62839],
[0.31374, 0.76146, 0.61919],
[0.30077, 0.75587, 0.60989],
[0.28814, 0.75023, 0.60051],
[0.27587, 0.74454, 0.59104],
[0.26392, 0.73881, 0.58149],
[0.25236, 0.73302, 0.57186],
[0.24121, 0.72719, 0.56215],
[0.23049, 0.72131, 0.55234],
[0.22028, 0.7154, 0.54245],
[0.21054, 0.70944, 0.53247],
[0.20135, 0.70343, 0.52241],
[0.19277, 0.69739, 0.51226],
[0.18482, 0.6913, 0.50204],
[0.17753, 0.68519, 0.49174],
[0.1709, 0.67903, 0.48135],
[0.165, 0.67285, 0.4709],
[0.15984, 0.66662, 0.46036],
[0.15546, 0.66036, 0.44978],
[0.15175, 0.65408, 0.43914],
[0.14873, 0.64776, 0.42848],
[0.14636, 0.64142, 0.41779],
[0.14455, 0.63507, 0.4071],
[0.14319, 0.62869, 0.39645],
[0.14222, 0.62228, 0.38584],
[0.14153, 0.61588, 0.37533],
[0.14104, 0.60946, 0.36492],
[0.14064, 0.60303, 0.35466],
[0.14025, 0.5966, 0.34453],
[0.13983, 0.59017, 0.33462],
[0.13936, 0.58374, 0.32487],
[0.13879, 0.57732, 0.31535],
[0.1381, 0.57089, 0.30606],
[0.13721, 0.56448, 0.29697],
[0.13614, 0.55808, 0.28809],
[0.13501, 0.55169, 0.27947],
[0.13366, 0.5453, 0.27104],
[0.13219, 0.53892, 0.26281],
[0.13061, 0.53256, 0.25481],
[0.12883, 0.5262, 0.24697],
[0.12697, 0.51985, 0.23933],
[0.12504, 0.51352, 0.23189],
[0.12297, 0.50719, 0.22457],
[0.12084, 0.50088, 0.21748],
[0.11867, 0.49458, 0.21052],
[0.11637, 0.48829, 0.20373],
[0.11397, 0.48202, 0.19708],
[0.11157, 0.47575, 0.19055],
[0.10907, 0.4695, 0.18418],
[0.10655, 0.46325, 0.17798],
[0.10391, 0.45703, 0.17184],
[0.10124, 0.45081, 0.16585],
[0.098563, 0.44462, 0.15999],
[0.095803, 0.43842, 0.15427],
[0.092982, 0.43226, 0.14864],
[0.090206, 0.42608, 0.14316],
[0.087373, 0.41994, 0.13778],
[0.084463, 0.4138, 0.13247],
[0.081522, 0.40768, 0.12725],
[0.078467, 0.40157, 0.12215],
[0.075481, 0.39549, 0.11717],
[0.072443, 0.38941, 0.11232],
[0.069456, 0.38335, 0.1075],
[0.06627, 0.3773, 0.1028],
[0.063173, 0.37126, 0.098194],
[0.060006, 0.36525, 0.093619],
[0.056764, 0.35924, 0.089201],
[0.053428, 0.35326, 0.084837],
[0.050177, 0.3473, 0.080536],
[0.04687, 0.34134, 0.076306],
[0.043388, 0.33541, 0.072205],
[0.04007, 0.32949, 0.068208],
[0.03665, 0.32357, 0.064214],
[0.033173, 0.31769, 0.060346],
[0.029999, 0.3118, 0.056535],
[0.026949, 0.30598, 0.052698],
[0.024019, 0.30012, 0.049141],
[0.021208, 0.2943, 0.045442],
[0.018516, 0.2885, 0.041997],
[0.015942, 0.28274, 0.038495],
[0.01348, 0.27696, 0.035242],
[0.010993, 0.27121, 0.031988],
[0.0088139, 0.26549, 0.02896],
[0.0069067, 0.25978, 0.025879],
[0.0052216, 0.25409, 0.022744],
[0.0037523, 0.24842, 0.019551],
[0.0024927, 0.24274, 0.016299],
[0.0014373, 0.23713, 0.012975],
[0.0005808, 0.2315, 0.0094103],
[0, 0.2259, 0.0059464],
]
b_diverging_bwg_20_95_c41 = bokeh_palette('diverging_bwg_20_95_c41',diverging_bwg_20_95_c41)
m_diverging_bwg_20_95_c41 = mpl_cm('diverging_bwg_20_95_c41',diverging_bwg_20_95_c41)
m_diverging_bwg_20_95_c41_r = mpl_cm('diverging_bwg_20_95_c41_r',list(reversed(diverging_bwg_20_95_c41)))
CET_D13 = b_diverging_bwg_20_95_c41
m_CET_D13 = m_diverging_bwg_20_95_c41
m_CET_D13_r = m_diverging_bwg_20_95_c41_r
palette['CET_D13'] = b_diverging_bwg_20_95_c41
cm['CET_D13'] = m_diverging_bwg_20_95_c41
cm['CET_D13_r'] = m_diverging_bwg_20_95_c41_r
register_cmap('cet_CET_D13',m_diverging_bwg_20_95_c41)
register_cmap('cet_CET_D13_r',m_diverging_bwg_20_95_c41_r)




cyclic_protanopic_deuteranopic_bwyk_16_96_c31 = [  # cmap_def
[0.24364, 0.52802, 0.91769],
[0.2544, 0.53367, 0.92309],
[0.26662, 0.53935, 0.92754],
[0.28004, 0.54507, 0.93114],
[0.29432, 0.55085, 0.93398],
[0.30919, 0.55669, 0.93621],
[0.32432, 0.56261, 0.93792],
[0.33957, 0.56857, 0.93924],
[0.35472, 0.57461, 0.94027],
[0.36965, 0.58072, 0.94109],
[0.38434, 0.58686, 0.94177],
[0.39872, 0.59306, 0.94235],
[0.41278, 0.59929, 0.94287],
[0.42649, 0.60554, 0.94334],
[0.43991, 0.61182, 0.94378],
[0.45302, 0.61813, 0.94421],
[0.46585, 0.62447, 0.94463],
[0.47842, 0.63083, 0.94504],
[0.49076, 0.6372, 0.94545],
[0.50285, 0.64358, 0.94584],
[0.51476, 0.65, 0.94623],
[0.52647, 0.65643, 0.94661],
[0.53801, 0.66287, 0.94698],
[0.54938, 0.66933, 0.94735],
[0.56059, 0.67581, 0.94771],
[0.57167, 0.68231, 0.94805],
[0.58261, 0.68883, 0.94839],
[0.59344, 0.69536, 0.94872],
[0.60414, 0.70191, 0.94905],
[0.61473, 0.70848, 0.94936],
[0.62522, 0.71506, 0.94967],
[0.63561, 0.72166, 0.94997],
[0.64591, 0.72828, 0.95026],
[0.65613, 0.73492, 0.95055],
[0.66625, 0.74156, 0.95082],
[0.6763, 0.74823, 0.95109],
[0.68627, 0.75491, 0.95134],
[0.69618, 0.7616, 0.95159],
[0.70602, 0.76831, 0.95182],
[0.71579, 0.77503, 0.95205],
[0.7255, 0.78178, 0.95228],
[0.73516, 0.78853, 0.95249],
[0.74475, 0.7953, 0.95269],
[0.7543, 0.80209, 0.95289],
[0.76379, 0.80889, 0.95307],
[0.77324, 0.8157, 0.95325],
[0.78264, 0.82253, 0.95342],
[0.792, 0.82937, 0.95357],
[0.80131, 0.83623, 0.95372],
[0.81059, 0.8431, 0.95384],
[0.81982, 0.84996, 0.95392],
[0.82901, 0.85681, 0.95394],
[0.83813, 0.86364, 0.95388],
[0.84716, 0.87039, 0.95368],
[0.8561, 0.87705, 0.95329],
[0.86488, 0.88356, 0.95262],
[0.87345, 0.88982, 0.95157],
[0.88173, 0.89575, 0.95003],
[0.88962, 0.90122, 0.94785],
[0.89702, 0.90612, 0.94491],
[0.90377, 0.9103, 0.94105],
[0.90977, 0.91362, 0.93618],
[0.91488, 0.91596, 0.9302],
[0.91901, 0.91726, 0.9231],
[0.9221, 0.91744, 0.91486],
[0.9241, 0.9165, 0.90556],
[0.92502, 0.91447, 0.89527],
[0.92494, 0.91145, 0.88414],
[0.92394, 0.90753, 0.8723],
[0.92213, 0.90284, 0.8599],
[0.91963, 0.89752, 0.84706],
[0.91659, 0.89171, 0.83393],
[0.9131, 0.88553, 0.82058],
[0.90928, 0.87908, 0.8071],
[0.90521, 0.87245, 0.79355],
[0.90095, 0.86569, 0.77995],
[0.89657, 0.85886, 0.76636],
[0.89208, 0.852, 0.75276],
[0.88752, 0.84512, 0.73919],
[0.88289, 0.83822, 0.72563],
[0.87822, 0.83134, 0.7121],
[0.8735, 0.82447, 0.69861],
[0.86874, 0.81761, 0.68514],
[0.86394, 0.81077, 0.67169],
[0.8591, 0.80395, 0.65827],
[0.85421, 0.79713, 0.64486],
[0.8493, 0.79033, 0.63149],
[0.84433, 0.78355, 0.61813],
[0.83933, 0.77678, 0.6048],
[0.83429, 0.77002, 0.59149],
[0.8292, 0.76328, 0.57819],
[0.82407, 0.75656, 0.56491],
[0.8189, 0.74984, 0.55167],
[0.8137, 0.74315, 0.53843],
[0.80846, 0.73646, 0.52519],
[0.80317, 0.7298, 0.51197],
[0.79785, 0.72314, 0.49878],
[0.79248, 0.71651, 0.48557],
[0.78708, 0.70989, 0.47239],
[0.78164, 0.70328, 0.4592],
[0.77615, 0.69669, 0.446],
[0.77063, 0.69011, 0.43282],
[0.76507, 0.68355, 0.41962],
[0.75946, 0.67701, 0.40641],
[0.75383, 0.67048, 0.39317],
[0.74815, 0.66397, 0.37993],
[0.74243, 0.65747, 0.36665],
[0.73668, 0.65098, 0.35333],
[0.73088, 0.64451, 0.33998],
[0.72505, 0.63807, 0.32654],
[0.71918, 0.63164, 0.31306],
[0.71327, 0.62522, 0.2995],
[0.70732, 0.61882, 0.28581],
[0.70133, 0.61243, 0.27203],
[0.69531, 0.60606, 0.25813],
[0.68925, 0.59972, 0.24409],
[0.68314, 0.59338, 0.2299],
[0.67699, 0.58706, 0.2156],
[0.67078, 0.58076, 0.20111],
[0.66452, 0.57447, 0.18656],
[0.65822, 0.56821, 0.17192],
[0.65184, 0.56196, 0.1573],
[0.6454, 0.55571, 0.14284],
[0.63889, 0.5495, 0.12862],
[0.63231, 0.54329, 0.11487],
[0.62565, 0.5371, 0.1019],
[0.61893, 0.53091, 0.08997],
[0.61212, 0.52474, 0.079354],
[0.60525, 0.51859, 0.070554],
[0.59832, 0.51243, 0.063809],
[0.59134, 0.50629, 0.059354],
[0.5843, 0.50016, 0.057061],
[0.57723, 0.49405, 0.056893],
[0.57013, 0.48795, 0.058316],
[0.56301, 0.48185, 0.061024],
[0.55586, 0.47577, 0.06459],
[0.54873, 0.46971, 0.068622],
[0.54156, 0.46364, 0.072767],
[0.53442, 0.4576, 0.076933],
[0.52728, 0.45158, 0.081132],
[0.52013, 0.44556, 0.085104],
[0.51301, 0.43957, 0.088952],
[0.50588, 0.43358, 0.09256],
[0.49878, 0.42763, 0.096006],
[0.49169, 0.42167, 0.099287],
[0.48459, 0.41574, 0.10238],
[0.47752, 0.40982, 0.10529],
[0.47046, 0.40392, 0.10807],
[0.4634, 0.39803, 0.11076],
[0.45636, 0.39215, 0.11328],
[0.44934, 0.38629, 0.11565],
[0.44232, 0.38046, 0.11793],
[0.43532, 0.37464, 0.12007],
[0.42831, 0.36884, 0.12212],
[0.42132, 0.36304, 0.12411],
[0.41435, 0.35728, 0.12604],
[0.40739, 0.35151, 0.12783],
[0.40044, 0.34577, 0.12955],
[0.39348, 0.34006, 0.13119],
[0.38654, 0.33436, 0.13275],
[0.37963, 0.32867, 0.13426],
[0.3727, 0.32301, 0.13564],
[0.3658, 0.31735, 0.13703],
[0.35889, 0.31171, 0.13834],
[0.352, 0.30612, 0.13953],
[0.34513, 0.3005, 0.14076],
[0.33824, 0.29493, 0.14187],
[0.33137, 0.28939, 0.14293],
[0.3245, 0.28384, 0.14395],
[0.31765, 0.27833, 0.14493],
[0.31081, 0.27283, 0.14584],
[0.30394, 0.26734, 0.14667],
[0.29711, 0.2619, 0.14752],
[0.29024, 0.25646, 0.14827],
[0.28341, 0.25106, 0.149],
[0.27657, 0.24565, 0.14972],
[0.26972, 0.24028, 0.15036],
[0.26287, 0.23497, 0.15098],
[0.25603, 0.22963, 0.1516],
[0.24923, 0.22437, 0.15223],
[0.24244, 0.2192, 0.1529],
[0.23574, 0.2141, 0.15364],
[0.22909, 0.20912, 0.15452],
[0.2226, 0.20434, 0.1556],
[0.21631, 0.19974, 0.15692],
[0.21024, 0.1955, 0.15863],
[0.20456, 0.19165, 0.16076],
[0.19928, 0.18832, 0.1635],
[0.1946, 0.18556, 0.16685],
[0.19048, 0.18343, 0.17092],
[0.1871, 0.18208, 0.17577],
[0.18444, 0.18154, 0.18145],
[0.18255, 0.18182, 0.18797],
[0.18144, 0.18291, 0.1952],
[0.18108, 0.18479, 0.20316],
[0.18139, 0.18737, 0.21179],
[0.1823, 0.19052, 0.22095],
[0.18372, 0.19427, 0.23054],
[0.18557, 0.19837, 0.24052],
[0.18769, 0.20286, 0.25082],
[0.18999, 0.20762, 0.26134],
[0.19246, 0.21256, 0.27202],
[0.19501, 0.21764, 0.28289],
[0.19756, 0.22284, 0.29382],
[0.20009, 0.2281, 0.30487],
[0.20261, 0.23338, 0.31601],
[0.20507, 0.23874, 0.32718],
[0.20746, 0.24413, 0.33844],
[0.20976, 0.24955, 0.34976],
[0.212, 0.25501, 0.36112],
[0.21413, 0.26045, 0.37255],
[0.2162, 0.26593, 0.38404],
[0.21815, 0.27142, 0.39559],
[0.22003, 0.27695, 0.40718],
[0.2218, 0.28251, 0.41883],
[0.22351, 0.28805, 0.43054],
[0.2251, 0.29363, 0.44231],
[0.22664, 0.29926, 0.45412],
[0.22806, 0.30487, 0.46597],
[0.22936, 0.31053, 0.4779],
[0.2306, 0.3162, 0.48986],
[0.23175, 0.32187, 0.50189],
[0.23276, 0.32757, 0.51396],
[0.23368, 0.33331, 0.52608],
[0.23454, 0.33905, 0.53825],
[0.23526, 0.34481, 0.55045],
[0.23587, 0.3506, 0.56272],
[0.23637, 0.35639, 0.57502],
[0.23677, 0.36221, 0.58738],
[0.23705, 0.36804, 0.59979],
[0.23721, 0.3739, 0.61223],
[0.23725, 0.37976, 0.62473],
[0.23717, 0.38565, 0.63727],
[0.23695, 0.39156, 0.64985],
[0.2366, 0.39748, 0.66248],
[0.23612, 0.40343, 0.67514],
[0.23551, 0.40938, 0.68785],
[0.23476, 0.41536, 0.70061],
[0.23382, 0.42134, 0.7134],
[0.23276, 0.42737, 0.72624],
[0.23156, 0.43338, 0.73913],
[0.23015, 0.43942, 0.75203],
[0.22865, 0.44546, 0.76497],
[0.22696, 0.45152, 0.7779],
[0.22514, 0.45758, 0.79081],
[0.22329, 0.46364, 0.80368],
[0.22139, 0.4697, 0.81645],
[0.21959, 0.47573, 0.82905],
[0.21803, 0.48173, 0.84139],
[0.21692, 0.4877, 0.85338],
[0.21649, 0.4936, 0.86491],
[0.21704, 0.49946, 0.87584],
[0.21888, 0.50526, 0.88605],
[0.2223, 0.51101, 0.89542],
[0.22753, 0.51671, 0.90384],
[0.23467, 0.52237, 0.91128],
]
b_cyclic_protanopic_deuteranopic_bwyk_16_96_c31 = bokeh_palette('cyclic_protanopic_deuteranopic_bwyk_16_96_c31',cyclic_protanopic_deuteranopic_bwyk_16_96_c31)
m_cyclic_protanopic_deuteranopic_bwyk_16_96_c31 = mpl_cm('cyclic_protanopic_deuteranopic_bwyk_16_96_c31',cyclic_protanopic_deuteranopic_bwyk_16_96_c31)
m_cyclic_protanopic_deuteranopic_bwyk_16_96_c31_r = mpl_cm('cyclic_protanopic_deuteranopic_bwyk_16_96_c31_r',list(reversed(cyclic_protanopic_deuteranopic_bwyk_16_96_c31)))
CET_CBC1 = b_cyclic_protanopic_deuteranopic_bwyk_16_96_c31
m_CET_CBC1 = m_cyclic_protanopic_deuteranopic_bwyk_16_96_c31
m_CET_CBC1_r = m_cyclic_protanopic_deuteranopic_bwyk_16_96_c31_r
palette['CET_CBC1'] = b_cyclic_protanopic_deuteranopic_bwyk_16_96_c31
cm['CET_CBC1'] = m_cyclic_protanopic_deuteranopic_bwyk_16_96_c31
cm['CET_CBC1_r'] = m_cyclic_protanopic_deuteranopic_bwyk_16_96_c31_r
register_cmap('cet_CET_CBC1',m_cyclic_protanopic_deuteranopic_bwyk_16_96_c31)
register_cmap('cet_CET_CBC1_r',m_cyclic_protanopic_deuteranopic_bwyk_16_96_c31_r)




linear_wyor_100_45_c55 = [  # cmap_def
[0.99986, 1, 1],
[0.99765, 0.99831, 0.98983],
[0.99541, 0.99656, 0.97966],
[0.99314, 0.99482, 0.96949],
[0.99083, 0.99308, 0.95933],
[0.9885, 0.99134, 0.94917],
[0.98614, 0.9896, 0.93902],
[0.98375, 0.98785, 0.92888],
[0.98133, 0.98612, 0.91874],
[0.97888, 0.98438, 0.9086],
[0.9764, 0.98264, 0.89847],
[0.97389, 0.9809, 0.88835],
[0.97135, 0.97917, 0.87822],
[0.96878, 0.97744, 0.86811],
[0.96618, 0.9757, 0.85798],
[0.96356, 0.97397, 0.84787],
[0.9609, 0.97224, 0.83776],
[0.95822, 0.97051, 0.82765],
[0.9555, 0.96878, 0.81754],
[0.95276, 0.96706, 0.80743],
[0.94999, 0.96532, 0.79732],
[0.94719, 0.9636, 0.78722],
[0.94436, 0.96187, 0.77711],
[0.94149, 0.96015, 0.76699],
[0.94023, 0.95771, 0.76023],
[0.93929, 0.95512, 0.75421],
[0.93835, 0.95252, 0.74819],
[0.9374, 0.94993, 0.74217],
[0.93644, 0.94734, 0.73614],
[0.93546, 0.94475, 0.73014],
[0.93446, 0.94217, 0.72412],
[0.93346, 0.93958, 0.71811],
[0.93245, 0.937, 0.7121],
[0.93142, 0.93441, 0.7061],
[0.93038, 0.93184, 0.70009],
[0.92933, 0.92926, 0.69408],
[0.92826, 0.92669, 0.68809],
[0.92719, 0.92412, 0.68208],
[0.9261, 0.92155, 0.67609],
[0.925, 0.91898, 0.6701],
[0.92389, 0.91641, 0.6641],
[0.92277, 0.91385, 0.6581],
[0.92164, 0.91129, 0.65211],
[0.92048, 0.90873, 0.64611],
[0.91933, 0.90618, 0.64012],
[0.91816, 0.90362, 0.63413],
[0.91698, 0.90107, 0.62813],
[0.91628, 0.89833, 0.62264],
[0.91585, 0.89549, 0.61746],
[0.91542, 0.89266, 0.61227],
[0.91497, 0.88982, 0.60709],
[0.91451, 0.88699, 0.6019],
[0.91404, 0.88416, 0.59672],
[0.91356, 0.88133, 0.59154],
[0.91306, 0.8785, 0.58635],
[0.91255, 0.87568, 0.58117],
[0.91203, 0.87285, 0.576],
[0.9115, 0.87003, 0.5708],
[0.91095, 0.86721, 0.56562],
[0.9104, 0.86439, 0.56044],
[0.90983, 0.86157, 0.55526],
[0.90925, 0.85876, 0.55008],
[0.90866, 0.85594, 0.5449],
[0.90806, 0.85313, 0.53971],
[0.90744, 0.85032, 0.53453],
[0.90682, 0.84751, 0.52935],
[0.90618, 0.84471, 0.52416],
[0.90554, 0.8419, 0.51898],
[0.90487, 0.8391, 0.51379],
[0.9042, 0.8363, 0.5086],
[0.90376, 0.8334, 0.50368],
[0.90359, 0.8304, 0.49912],
[0.90341, 0.82739, 0.49457],
[0.90322, 0.82439, 0.49],
[0.90302, 0.82139, 0.48543],
[0.9028, 0.81839, 0.48087],
[0.90257, 0.81539, 0.47631],
[0.90233, 0.81239, 0.47175],
[0.90208, 0.80939, 0.46718],
[0.90182, 0.8064, 0.4626],
[0.90154, 0.8034, 0.45803],
[0.90126, 0.80041, 0.45346],
[0.90096, 0.79741, 0.4489],
[0.90066, 0.79442, 0.44432],
[0.90034, 0.79143, 0.43974],
[0.90001, 0.78844, 0.43517],
[0.89967, 0.78545, 0.43058],
[0.89932, 0.78246, 0.42599],
[0.89896, 0.77947, 0.42139],
[0.89858, 0.77648, 0.41681],
[0.8982, 0.77349, 0.41222],
[0.89781, 0.77051, 0.40761],
[0.8974, 0.76752, 0.40301],
[0.89709, 0.7645, 0.39857],
[0.89703, 0.76135, 0.3946],
[0.89696, 0.75821, 0.39062],
[0.89688, 0.75507, 0.38663],
[0.89679, 0.75192, 0.38266],
[0.89669, 0.74877, 0.37868],
[0.89657, 0.74562, 0.37469],
[0.89645, 0.74248, 0.3707],
[0.89631, 0.73934, 0.36672],
[0.89617, 0.73618, 0.36271],
[0.89601, 0.73304, 0.35871],
[0.89585, 0.7299, 0.35473],
[0.89567, 0.72675, 0.35072],
[0.89549, 0.7236, 0.3467],
[0.89529, 0.72046, 0.34269],
[0.89509, 0.71732, 0.33867],
[0.89487, 0.71416, 0.33466],
[0.89465, 0.71102, 0.33061],
[0.89441, 0.70787, 0.32657],
[0.89416, 0.70472, 0.32255],
[0.89391, 0.70157, 0.31849],
[0.89364, 0.69842, 0.31443],
[0.89336, 0.69527, 0.31039],
[0.8931, 0.69212, 0.30638],
[0.89306, 0.68883, 0.30298],
[0.893, 0.68555, 0.29962],
[0.89294, 0.68226, 0.29623],
[0.89287, 0.67898, 0.29284],
[0.89278, 0.67569, 0.28946],
[0.89269, 0.67241, 0.28605],
[0.89259, 0.66912, 0.28268],
[0.89248, 0.66582, 0.27928],
[0.89236, 0.66254, 0.27587],
[0.89223, 0.65923, 0.27243],
[0.89209, 0.65594, 0.26904],
[0.89194, 0.65264, 0.26561],
[0.89179, 0.64934, 0.26218],
[0.89162, 0.64603, 0.25875],
[0.89145, 0.64273, 0.25531],
[0.89126, 0.63942, 0.25187],
[0.89107, 0.63612, 0.24842],
[0.89087, 0.6328, 0.24492],
[0.89066, 0.62948, 0.24148],
[0.89044, 0.62617, 0.23799],
[0.89021, 0.62285, 0.23452],
[0.88997, 0.61953, 0.23101],
[0.88973, 0.6162, 0.22751],
[0.88961, 0.61278, 0.22468],
[0.88951, 0.60936, 0.22197],
[0.88939, 0.60592, 0.21926],
[0.88927, 0.60249, 0.21654],
[0.88914, 0.59905, 0.21379],
[0.889, 0.5956, 0.21105],
[0.88885, 0.59216, 0.2083],
[0.88869, 0.58869, 0.20555],
[0.88853, 0.58524, 0.20279],
[0.88835, 0.58177, 0.20002],
[0.88817, 0.5783, 0.19729],
[0.88798, 0.57483, 0.19454],
[0.88779, 0.57136, 0.19174],
[0.88758, 0.56788, 0.18897],
[0.88737, 0.56439, 0.18619],
[0.88715, 0.56089, 0.18338],
[0.88693, 0.5574, 0.18058],
[0.88669, 0.55389, 0.17782],
[0.88645, 0.55039, 0.17499],
[0.8862, 0.54687, 0.17215],
[0.88594, 0.54335, 0.16935],
[0.88567, 0.53982, 0.1665],
[0.8854, 0.53629, 0.16367],
[0.88517, 0.53271, 0.16148],
[0.88495, 0.5291, 0.15949],
[0.88473, 0.52548, 0.15755],
[0.8845, 0.52184, 0.15563],
[0.88427, 0.51822, 0.15366],
[0.88402, 0.51458, 0.15172],
[0.88377, 0.51092, 0.14981],
[0.88351, 0.50726, 0.14784],
[0.88325, 0.50359, 0.14591],
[0.88297, 0.49991, 0.14395],
[0.88269, 0.49622, 0.142],
[0.88241, 0.49252, 0.14002],
[0.88212, 0.48882, 0.13811],
[0.88182, 0.4851, 0.1361],
[0.88152, 0.48138, 0.1342],
[0.8812, 0.47764, 0.13223],
[0.88088, 0.47389, 0.13031],
[0.88056, 0.47014, 0.12832],
[0.88023, 0.46636, 0.12636],
[0.87989, 0.46257, 0.12436],
[0.87954, 0.45878, 0.12238],
[0.87919, 0.45498, 0.12043],
[0.87883, 0.45115, 0.11851],
[0.87847, 0.4473, 0.11705],
[0.8781, 0.44344, 0.11614],
[0.87772, 0.43955, 0.11514],
[0.87733, 0.43565, 0.11419],
[0.87694, 0.43174, 0.11331],
[0.87654, 0.4278, 0.11235],
[0.87614, 0.42385, 0.11142],
[0.87574, 0.41989, 0.11049],
[0.87532, 0.4159, 0.10958],
[0.8749, 0.4119, 0.10858],
[0.87448, 0.40788, 0.10768],
[0.87405, 0.40384, 0.10681],
[0.87361, 0.39977, 0.10587],
[0.87317, 0.39571, 0.10491],
[0.87272, 0.3916, 0.10404],
[0.87227, 0.38747, 0.10316],
[0.87181, 0.38334, 0.10224],
[0.87134, 0.37916, 0.10134],
[0.87087, 0.37496, 0.10045],
[0.87039, 0.37074, 0.099568],
[0.86991, 0.36651, 0.098702],
[0.86942, 0.36223, 0.097851],
[0.86893, 0.35793, 0.096914],
[0.86841, 0.3536, 0.096399],
[0.86784, 0.34928, 0.096516],
[0.86725, 0.34493, 0.096632],
[0.86667, 0.34054, 0.096748],
[0.86609, 0.33612, 0.096864],
[0.86549, 0.33166, 0.096979],
[0.86489, 0.32717, 0.097092],
[0.86429, 0.32265, 0.097204],
[0.86369, 0.31807, 0.097312],
[0.86307, 0.31346, 0.097418],
[0.86246, 0.30881, 0.097519],
[0.86183, 0.30409, 0.097617],
[0.8612, 0.29936, 0.097711],
[0.86057, 0.29454, 0.0978],
[0.85994, 0.28969, 0.097885],
[0.8593, 0.28477, 0.097966],
[0.85865, 0.27981, 0.098043],
[0.858, 0.27477, 0.098117],
[0.85734, 0.26969, 0.098187],
[0.85668, 0.26451, 0.098254],
[0.85602, 0.25928, 0.098319],
[0.85535, 0.25395, 0.098381],
[0.85467, 0.24855, 0.09844],
[0.85397, 0.24306, 0.098636],
[0.85315, 0.23765, 0.09947],
[0.85234, 0.23211, 0.1003],
[0.85151, 0.22648, 0.10112],
[0.85069, 0.22075, 0.10192],
[0.84986, 0.21485, 0.10272],
[0.84902, 0.20887, 0.10351],
[0.84818, 0.20273, 0.10422],
[0.84734, 0.19648, 0.10494],
[0.84649, 0.19, 0.10571],
[0.84564, 0.18336, 0.10646],
[0.84479, 0.17651, 0.10714],
[0.84393, 0.16949, 0.10778],
[0.84307, 0.16216, 0.10844],
[0.8422, 0.15453, 0.10915],
[0.84133, 0.14658, 0.10982],
[0.84045, 0.13829, 0.11043],
[0.83958, 0.1295, 0.11105],
[0.83869, 0.12017, 0.11166],
[0.8378, 0.11027, 0.11225],
[0.83691, 0.099467, 0.11285],
[0.83602, 0.087701, 0.11343],
[0.83512, 0.07445, 0.11395],
[0.83422, 0.059213, 0.11447],
]
b_linear_wyor_100_45_c55 = bokeh_palette('linear_wyor_100_45_c55',linear_wyor_100_45_c55)
m_linear_wyor_100_45_c55 = mpl_cm('linear_wyor_100_45_c55',linear_wyor_100_45_c55)
m_linear_wyor_100_45_c55_r = mpl_cm('linear_wyor_100_45_c55_r',list(reversed(linear_wyor_100_45_c55)))
CET_L18 = b_linear_wyor_100_45_c55
m_CET_L18 = m_linear_wyor_100_45_c55
m_CET_L18_r = m_linear_wyor_100_45_c55_r
palette['CET_L18'] = b_linear_wyor_100_45_c55
cm['CET_L18'] = m_linear_wyor_100_45_c55
cm['CET_L18_r'] = m_linear_wyor_100_45_c55_r
register_cmap('cet_CET_L18',m_linear_wyor_100_45_c55)
register_cmap('cet_CET_L18_r',m_linear_wyor_100_45_c55_r)




linear_wcmr_100_45_c42 = [  # cmap_def
[0.99986, 1, 1],
[0.99494, 0.99813, 0.99962],
[0.99002, 0.9962, 0.99922],
[0.98509, 0.99428, 0.99882],
[0.98015, 0.99236, 0.99842],
[0.9752, 0.99043, 0.99802],
[0.97025, 0.9885, 0.99762],
[0.96528, 0.98658, 0.99722],
[0.96032, 0.98465, 0.99682],
[0.95535, 0.98273, 0.99642],
[0.95036, 0.9808, 0.99602],
[0.94538, 0.97887, 0.99561],
[0.94038, 0.97695, 0.99521],
[0.93537, 0.97502, 0.99481],
[0.93036, 0.9731, 0.9944],
[0.92534, 0.97117, 0.994],
[0.92031, 0.96925, 0.9936],
[0.91527, 0.96733, 0.99319],
[0.91023, 0.9654, 0.99278],
[0.90517, 0.96348, 0.99237],
[0.9001, 0.96155, 0.99196],
[0.89503, 0.95963, 0.99156],
[0.88994, 0.95771, 0.99115],
[0.88485, 0.95579, 0.99073],
[0.88066, 0.95357, 0.99084],
[0.87667, 0.95128, 0.99105],
[0.87268, 0.94899, 0.99126],
[0.86868, 0.9467, 0.99148],
[0.86467, 0.94442, 0.99169],
[0.86065, 0.94214, 0.99189],
[0.85663, 0.93985, 0.9921],
[0.8526, 0.93757, 0.99231],
[0.84855, 0.93529, 0.99251],
[0.84451, 0.93301, 0.99272],
[0.84045, 0.93074, 0.99292],
[0.83638, 0.92846, 0.99312],
[0.83231, 0.92619, 0.99332],
[0.82822, 0.92392, 0.99352],
[0.82413, 0.92165, 0.99372],
[0.82002, 0.91938, 0.99391],
[0.81592, 0.91712, 0.9941],
[0.81179, 0.91485, 0.9943],
[0.80766, 0.91258, 0.99449],
[0.80352, 0.91032, 0.99468],
[0.79937, 0.90806, 0.99487],
[0.79521, 0.9058, 0.99505],
[0.79104, 0.90354, 0.99524],
[0.78796, 0.901, 0.99546],
[0.78551, 0.89831, 0.99569],
[0.78305, 0.89562, 0.99593],
[0.78058, 0.89293, 0.99616],
[0.77812, 0.89024, 0.9964],
[0.77564, 0.88755, 0.99663],
[0.77316, 0.88487, 0.99686],
[0.77068, 0.88218, 0.99708],
[0.76819, 0.8795, 0.99731],
[0.7657, 0.87682, 0.99754],
[0.76319, 0.87414, 0.99776],
[0.76069, 0.87147, 0.99798],
[0.75818, 0.8688, 0.9982],
[0.75567, 0.86612, 0.99842],
[0.75314, 0.86345, 0.99864],
[0.75062, 0.86077, 0.99886],
[0.74809, 0.8581, 0.99908],
[0.74554, 0.85544, 0.99929],
[0.743, 0.85278, 0.9995],
[0.74045, 0.85011, 0.99971],
[0.73789, 0.84745, 0.99992],
[0.73533, 0.84479, 1],
[0.73276, 0.84213, 1],
[0.73115, 0.83926, 1],
[0.73071, 0.83615, 0.99995],
[0.73027, 0.83303, 0.9996],
[0.72981, 0.82992, 0.99926],
[0.72935, 0.8268, 0.99891],
[0.72888, 0.82368, 0.99856],
[0.72841, 0.82056, 0.99821],
[0.72793, 0.81745, 0.99786],
[0.72745, 0.81433, 0.99751],
[0.72696, 0.81122, 0.99715],
[0.72647, 0.8081, 0.9968],
[0.72597, 0.80499, 0.99645],
[0.72547, 0.80187, 0.99609],
[0.72496, 0.79876, 0.99574],
[0.72444, 0.79564, 0.99538],
[0.72392, 0.79253, 0.99502],
[0.72339, 0.78941, 0.99466],
[0.72285, 0.7863, 0.9943],
[0.72231, 0.78318, 0.99395],
[0.72177, 0.78007, 0.99358],
[0.72122, 0.77696, 0.99322],
[0.72067, 0.77383, 0.99286],
[0.72011, 0.77072, 0.99249],
[0.72013, 0.76749, 0.99183],
[0.72168, 0.76394, 0.99037],
[0.72322, 0.76039, 0.98891],
[0.72475, 0.75684, 0.98745],
[0.72625, 0.75328, 0.98598],
[0.72774, 0.74972, 0.98452],
[0.72921, 0.74616, 0.98306],
[0.73066, 0.74259, 0.9816],
[0.7321, 0.73902, 0.98014],
[0.73352, 0.73544, 0.97868],
[0.73492, 0.73187, 0.97722],
[0.7363, 0.72828, 0.97575],
[0.73767, 0.7247, 0.97429],
[0.73903, 0.72111, 0.97284],
[0.74036, 0.71752, 0.97137],
[0.74168, 0.71392, 0.96991],
[0.74299, 0.71032, 0.96846],
[0.74428, 0.70671, 0.967],
[0.74555, 0.7031, 0.96553],
[0.74681, 0.69948, 0.96407],
[0.74806, 0.69587, 0.96261],
[0.74928, 0.69225, 0.96115],
[0.75049, 0.68861, 0.9597],
[0.75185, 0.68495, 0.9581],
[0.75471, 0.68097, 0.95513],
[0.75753, 0.677, 0.95216],
[0.76032, 0.67301, 0.94919],
[0.76307, 0.66902, 0.94623],
[0.7658, 0.66501, 0.94327],
[0.76848, 0.66099, 0.94031],
[0.77114, 0.65698, 0.93735],
[0.77375, 0.65293, 0.93439],
[0.77635, 0.6489, 0.93144],
[0.7789, 0.64484, 0.92848],
[0.78143, 0.64078, 0.92553],
[0.78393, 0.63671, 0.92259],
[0.78639, 0.63263, 0.91964],
[0.78883, 0.62854, 0.9167],
[0.79123, 0.62443, 0.91376],
[0.79361, 0.62031, 0.91081],
[0.79595, 0.61618, 0.90788],
[0.79827, 0.61204, 0.90494],
[0.80056, 0.60789, 0.902],
[0.80282, 0.60373, 0.89907],
[0.80505, 0.59955, 0.89614],
[0.80727, 0.59535, 0.89321],
[0.80945, 0.59115, 0.89029],
[0.81246, 0.58681, 0.88579],
[0.81552, 0.58244, 0.88113],
[0.81853, 0.57806, 0.87648],
[0.8215, 0.57367, 0.87185],
[0.82443, 0.56926, 0.8672],
[0.82731, 0.56483, 0.86258],
[0.83015, 0.56039, 0.85794],
[0.83295, 0.55592, 0.85332],
[0.83571, 0.55144, 0.84871],
[0.83842, 0.54694, 0.84409],
[0.8411, 0.54243, 0.83949],
[0.84374, 0.53789, 0.83488],
[0.84634, 0.53332, 0.83029],
[0.8489, 0.52875, 0.82569],
[0.85142, 0.52414, 0.82111],
[0.85391, 0.51952, 0.81653],
[0.85636, 0.51487, 0.81195],
[0.85878, 0.5102, 0.80738],
[0.86116, 0.5055, 0.80281],
[0.86351, 0.50079, 0.79825],
[0.86581, 0.49604, 0.7937],
[0.86809, 0.49128, 0.78915],
[0.87033, 0.48647, 0.78461],
[0.87265, 0.48181, 0.77881],
[0.87497, 0.47718, 0.77256],
[0.87724, 0.47253, 0.76631],
[0.87946, 0.46786, 0.76007],
[0.88165, 0.46315, 0.75385],
[0.88379, 0.45843, 0.74763],
[0.88589, 0.45368, 0.74141],
[0.88794, 0.44891, 0.73522],
[0.88996, 0.4441, 0.72902],
[0.89193, 0.43926, 0.72284],
[0.89386, 0.43439, 0.71667],
[0.89576, 0.42949, 0.71051],
[0.89761, 0.42456, 0.70435],
[0.89942, 0.4196, 0.6982],
[0.9012, 0.41459, 0.69208],
[0.90294, 0.40955, 0.68595],
[0.90464, 0.40446, 0.67983],
[0.9063, 0.39934, 0.67373],
[0.90793, 0.39418, 0.66764],
[0.90952, 0.38899, 0.66155],
[0.91108, 0.38373, 0.65547],
[0.9126, 0.37843, 0.64941],
[0.91409, 0.37307, 0.64335],
[0.91523, 0.36813, 0.6365],
[0.91607, 0.36353, 0.62897],
[0.91689, 0.3589, 0.62146],
[0.91766, 0.35424, 0.61397],
[0.91838, 0.34955, 0.60649],
[0.91907, 0.34483, 0.59903],
[0.91973, 0.34007, 0.59158],
[0.92035, 0.33528, 0.58414],
[0.92093, 0.33043, 0.57672],
[0.92148, 0.32554, 0.5693],
[0.92199, 0.32064, 0.56191],
[0.92247, 0.31567, 0.55452],
[0.92291, 0.31066, 0.54715],
[0.92331, 0.30559, 0.53979],
[0.92368, 0.30046, 0.53245],
[0.92402, 0.29529, 0.52512],
[0.92432, 0.29005, 0.51781],
[0.92459, 0.28476, 0.5105],
[0.92483, 0.27942, 0.50321],
[0.92504, 0.27399, 0.49593],
[0.92521, 0.26849, 0.48868],
[0.92536, 0.26291, 0.48143],
[0.92547, 0.25725, 0.4742],
[0.92514, 0.2522, 0.4666],
[0.92407, 0.24833, 0.45833],
[0.92297, 0.2444, 0.45009],
[0.92184, 0.24047, 0.44186],
[0.92067, 0.23655, 0.43362],
[0.91949, 0.23256, 0.42542],
[0.91827, 0.22857, 0.41723],
[0.91702, 0.22449, 0.40904],
[0.91574, 0.22046, 0.40088],
[0.91444, 0.21636, 0.39271],
[0.9131, 0.21221, 0.38456],
[0.91174, 0.20801, 0.37643],
[0.91036, 0.2038, 0.3683],
[0.90894, 0.1995, 0.36017],
[0.9075, 0.1952, 0.35206],
[0.90603, 0.19083, 0.34396],
[0.90454, 0.18642, 0.33587],
[0.90302, 0.18193, 0.32777],
[0.90147, 0.17742, 0.3197],
[0.8999, 0.17279, 0.3116],
[0.8983, 0.16811, 0.30352],
[0.89669, 0.16339, 0.29545],
[0.89504, 0.15855, 0.28737],
[0.8931, 0.15427, 0.27921],
[0.88995, 0.15302, 0.2706],
[0.88677, 0.15181, 0.26196],
[0.88358, 0.15061, 0.25331],
[0.88036, 0.14943, 0.24461],
[0.87712, 0.14823, 0.23593],
[0.87387, 0.14707, 0.22719],
[0.8706, 0.14594, 0.2184],
[0.86731, 0.14481, 0.20956],
[0.86401, 0.14368, 0.20065],
[0.86068, 0.14257, 0.1917],
[0.85733, 0.1415, 0.18265],
[0.85397, 0.14041, 0.17353],
[0.85059, 0.13932, 0.1643],
[0.84719, 0.13832, 0.15489],
[0.84378, 0.13728, 0.14538],
[0.84035, 0.13622, 0.13559],
[0.8369, 0.13525, 0.12564],
[0.83344, 0.13428, 0.11535],
[0.82996, 0.13329, 0.10467],
[0.82647, 0.13234, 0.093595],
[0.82295, 0.13139, 0.08195],
[0.81943, 0.13049, 0.069425],
[0.81589, 0.12954, 0.055606],
]
b_linear_wcmr_100_45_c42 = bokeh_palette('linear_wcmr_100_45_c42',linear_wcmr_100_45_c42)
m_linear_wcmr_100_45_c42 = mpl_cm('linear_wcmr_100_45_c42',linear_wcmr_100_45_c42)
m_linear_wcmr_100_45_c42_r = mpl_cm('linear_wcmr_100_45_c42_r',list(reversed(linear_wcmr_100_45_c42)))
CET_L19 = b_linear_wcmr_100_45_c42
m_CET_L19 = m_linear_wcmr_100_45_c42
m_CET_L19_r = m_linear_wcmr_100_45_c42_r
palette['CET_L19'] = b_linear_wcmr_100_45_c42
cm['CET_L19'] = m_linear_wcmr_100_45_c42
cm['CET_L19_r'] = m_linear_wcmr_100_45_c42_r
register_cmap('cet_CET_L19',m_linear_wcmr_100_45_c42)
register_cmap('cet_CET_L19_r',m_linear_wcmr_100_45_c42_r)




diverging_protanopic_deuteranopic_bwy_60_95_c32 = [  # cmap_def
[0.22906, 0.56603, 0.9967],
[0.24157, 0.56872, 0.99641],
[0.25345, 0.57142, 0.99612],
[0.26476, 0.57412, 0.99583],
[0.27564, 0.57683, 0.99553],
[0.28604, 0.57954, 0.99524],
[0.2961, 0.58225, 0.99494],
[0.30584, 0.58497, 0.99464],
[0.31523, 0.58769, 0.99434],
[0.32437, 0.59042, 0.99404],
[0.33327, 0.59316, 0.99374],
[0.34192, 0.5959, 0.99343],
[0.35037, 0.59864, 0.99313],
[0.3586, 0.60139, 0.99282],
[0.36667, 0.60415, 0.99251],
[0.37457, 0.6069, 0.99219],
[0.38229, 0.60966, 0.99188],
[0.38988, 0.61242, 0.99156],
[0.39734, 0.61519, 0.99124],
[0.40465, 0.61797, 0.99092],
[0.41186, 0.62076, 0.9906],
[0.41894, 0.62354, 0.99028],
[0.42591, 0.62632, 0.98996],
[0.43278, 0.62912, 0.98963],
[0.43955, 0.63192, 0.9893],
[0.44622, 0.63472, 0.98897],
[0.45282, 0.63752, 0.98864],
[0.45932, 0.64034, 0.9883],
[0.46573, 0.64315, 0.98797],
[0.47209, 0.64596, 0.98763],
[0.47835, 0.64879, 0.98729],
[0.48455, 0.65162, 0.98695],
[0.4907, 0.65444, 0.98661],
[0.49675, 0.65728, 0.98626],
[0.50276, 0.66012, 0.98592],
[0.50872, 0.66297, 0.98557],
[0.5146, 0.66581, 0.98522],
[0.52043, 0.66866, 0.98487],
[0.52622, 0.67151, 0.98451],
[0.53194, 0.67437, 0.98416],
[0.53763, 0.67724, 0.9838],
[0.54326, 0.6801, 0.98343],
[0.54885, 0.68297, 0.98307],
[0.55438, 0.68585, 0.98271],
[0.55988, 0.68873, 0.98235],
[0.56533, 0.69161, 0.98198],
[0.57075, 0.69449, 0.98161],
[0.57614, 0.69738, 0.98123],
[0.58147, 0.70027, 0.98086],
[0.58677, 0.70317, 0.98049],
[0.59205, 0.70607, 0.98011],
[0.59728, 0.70898, 0.97973],
[0.60247, 0.71188, 0.97935],
[0.60765, 0.71479, 0.97896],
[0.61277, 0.71771, 0.97857],
[0.61788, 0.72063, 0.97819],
[0.62296, 0.72355, 0.9778],
[0.62801, 0.72648, 0.97741],
[0.63302, 0.72941, 0.97701],
[0.63801, 0.73234, 0.97661],
[0.64298, 0.73528, 0.97622],
[0.64791, 0.73821, 0.97581],
[0.65283, 0.74115, 0.97541],
[0.65772, 0.74411, 0.97501],
[0.66259, 0.74706, 0.9746],
[0.66743, 0.75001, 0.97419],
[0.67225, 0.75296, 0.97378],
[0.67704, 0.75593, 0.97337],
[0.68181, 0.75889, 0.97296],
[0.68657, 0.76186, 0.97254],
[0.6913, 0.76483, 0.97212],
[0.69602, 0.7678, 0.9717],
[0.70071, 0.77078, 0.97127],
[0.70538, 0.77376, 0.97084],
[0.71003, 0.77675, 0.97042],
[0.71466, 0.77973, 0.96999],
[0.71929, 0.78272, 0.96955],
[0.72389, 0.78572, 0.96912],
[0.72847, 0.78871, 0.96868],
[0.73304, 0.79171, 0.96824],
[0.73758, 0.79471, 0.9678],
[0.74212, 0.79772, 0.96736],
[0.74664, 0.80073, 0.96691],
[0.75115, 0.80374, 0.96646],
[0.75563, 0.80676, 0.96601],
[0.7601, 0.80977, 0.96555],
[0.76456, 0.81279, 0.9651],
[0.769, 0.81582, 0.96464],
[0.77343, 0.81884, 0.96418],
[0.77785, 0.82188, 0.96372],
[0.78226, 0.82491, 0.96325],
[0.78665, 0.82795, 0.96278],
[0.79102, 0.83099, 0.96231],
[0.79539, 0.83404, 0.96184],
[0.79974, 0.83708, 0.96136],
[0.80408, 0.84013, 0.96089],
[0.80841, 0.84318, 0.96041],
[0.81272, 0.84623, 0.95993],
[0.81703, 0.84929, 0.95944],
[0.82133, 0.85236, 0.95895],
[0.82561, 0.85542, 0.95847],
[0.82988, 0.85848, 0.95797],
[0.83415, 0.86155, 0.95748],
[0.8384, 0.86462, 0.95698],
[0.84264, 0.8677, 0.95648],
[0.84687, 0.87078, 0.95598],
[0.85109, 0.87386, 0.95547],
[0.85531, 0.87694, 0.95497],
[0.85951, 0.88003, 0.95445],
[0.8637, 0.88312, 0.95394],
[0.86789, 0.88621, 0.95343],
[0.87206, 0.88931, 0.95291],
[0.87623, 0.89241, 0.95239],
[0.88039, 0.8955, 0.95185],
[0.88453, 0.8986, 0.95131],
[0.88867, 0.90168, 0.95074],
[0.89277, 0.90475, 0.95013],
[0.89686, 0.90779, 0.94947],
[0.90088, 0.91077, 0.94872],
[0.90485, 0.91368, 0.94786],
[0.90872, 0.91647, 0.94684],
[0.91247, 0.9191, 0.9456],
[0.91603, 0.92152, 0.94409],
[0.91936, 0.92367, 0.94224],
[0.92242, 0.92548, 0.94],
[0.92513, 0.9269, 0.93732],
[0.92745, 0.92788, 0.93415],
[0.92933, 0.92837, 0.93049],
[0.93077, 0.92836, 0.92633],
[0.93173, 0.92785, 0.9217],
[0.93225, 0.92686, 0.91663],
[0.93234, 0.92542, 0.91117],
[0.93205, 0.9236, 0.90539],
[0.93143, 0.92143, 0.89935],
[0.93053, 0.91899, 0.8931],
[0.92941, 0.91635, 0.88671],
[0.92813, 0.91355, 0.88021],
[0.92671, 0.91063, 0.87364],
[0.92521, 0.90764, 0.86702],
[0.92364, 0.90459, 0.86039],
[0.92202, 0.90151, 0.85373],
[0.92036, 0.89841, 0.84708],
[0.91868, 0.89531, 0.84042],
[0.91699, 0.8922, 0.83377],
[0.91527, 0.88909, 0.82712],
[0.91355, 0.88598, 0.82048],
[0.91181, 0.88287, 0.81385],
[0.91007, 0.87977, 0.80722],
[0.90831, 0.87667, 0.8006],
[0.90653, 0.87358, 0.79399],
[0.90475, 0.87048, 0.78738],
[0.90296, 0.8674, 0.78077],
[0.90115, 0.86431, 0.77417],
[0.89933, 0.86122, 0.76758],
[0.8975, 0.85814, 0.761],
[0.89566, 0.85507, 0.75442],
[0.89381, 0.85199, 0.74784],
[0.89195, 0.84892, 0.74126],
[0.89007, 0.84585, 0.73471],
[0.88818, 0.84278, 0.72814],
[0.88629, 0.83972, 0.72159],
[0.88438, 0.83666, 0.71504],
[0.88245, 0.8336, 0.7085],
[0.88052, 0.83055, 0.70196],
[0.87858, 0.82749, 0.69543],
[0.87662, 0.82444, 0.6889],
[0.87466, 0.8214, 0.68237],
[0.87268, 0.81835, 0.67586],
[0.87069, 0.81532, 0.66935],
[0.8687, 0.81228, 0.66284],
[0.86668, 0.80925, 0.65634],
[0.86466, 0.80622, 0.64984],
[0.86263, 0.80319, 0.64334],
[0.86058, 0.80017, 0.63685],
[0.85852, 0.79714, 0.63037],
[0.85646, 0.79413, 0.62389],
[0.85438, 0.7911, 0.6174],
[0.8523, 0.78809, 0.61093],
[0.85019, 0.78509, 0.60446],
[0.84808, 0.78208, 0.59799],
[0.84596, 0.77907, 0.59153],
[0.84383, 0.77607, 0.58507],
[0.84169, 0.77307, 0.57861],
[0.83953, 0.77008, 0.57216],
[0.83737, 0.76709, 0.5657],
[0.83519, 0.7641, 0.55926],
[0.833, 0.76112, 0.55281],
[0.83081, 0.75814, 0.54638],
[0.8286, 0.75516, 0.53993],
[0.82638, 0.75218, 0.53349],
[0.82415, 0.7492, 0.52706],
[0.82191, 0.74624, 0.52061],
[0.81966, 0.74328, 0.51419],
[0.8174, 0.74031, 0.50775],
[0.81513, 0.73735, 0.50132],
[0.81284, 0.73439, 0.49488],
[0.81055, 0.73144, 0.48845],
[0.80825, 0.72849, 0.48201],
[0.80593, 0.72555, 0.47558],
[0.80361, 0.72259, 0.46914],
[0.80127, 0.71966, 0.46269],
[0.79893, 0.71672, 0.45625],
[0.79657, 0.71378, 0.44981],
[0.79421, 0.71086, 0.44337],
[0.79183, 0.70792, 0.43692],
[0.78944, 0.705, 0.43046],
[0.78704, 0.70208, 0.42399],
[0.78464, 0.69916, 0.41754],
[0.78222, 0.69625, 0.41106],
[0.77978, 0.69334, 0.40458],
[0.77735, 0.69042, 0.3981],
[0.7749, 0.68752, 0.39161],
[0.77244, 0.68462, 0.3851],
[0.76997, 0.68172, 0.37859],
[0.76749, 0.67882, 0.37206],
[0.765, 0.67593, 0.36553],
[0.7625, 0.67305, 0.35898],
[0.75999, 0.67017, 0.35242],
[0.75747, 0.66728, 0.34583],
[0.75494, 0.6644, 0.33926],
[0.7524, 0.66153, 0.33264],
[0.74985, 0.65865, 0.32598],
[0.74729, 0.65579, 0.31934],
[0.74472, 0.65291, 0.31264],
[0.74214, 0.65006, 0.30596],
[0.73956, 0.6472, 0.29921],
[0.73695, 0.64434, 0.29243],
[0.73434, 0.64149, 0.28562],
[0.73172, 0.63864, 0.2788],
[0.72909, 0.6358, 0.2719],
[0.72646, 0.63296, 0.26498],
[0.72381, 0.63012, 0.258],
[0.72115, 0.62728, 0.251],
[0.71848, 0.62446, 0.24389],
[0.71581, 0.62162, 0.23678],
[0.71312, 0.61881, 0.22953],
[0.71042, 0.61598, 0.22224],
[0.70771, 0.61316, 0.21485],
[0.705, 0.61035, 0.2074],
[0.70227, 0.60754, 0.19979],
[0.69954, 0.60473, 0.1921],
[0.6968, 0.60193, 0.18428],
[0.69404, 0.59913, 0.17627],
[0.69128, 0.59633, 0.16813],
[0.68852, 0.59354, 0.15977],
[0.68574, 0.59075, 0.15122],
[0.68294, 0.58796, 0.1424],
[0.68014, 0.58518, 0.13326],
[0.67734, 0.5824, 0.12372],
[0.67452, 0.57963, 0.1138],
[0.67169, 0.57686, 0.10332],
[0.66886, 0.57408, 0.092119],
[0.66602, 0.57132, 0.07995],
[0.66317, 0.56855, 0.066579],
[0.6603, 0.5658, 0.051268],
[0.65744, 0.56305, 0.03304],
]
b_diverging_protanopic_deuteranopic_bwy_60_95_c32 = bokeh_palette('diverging_protanopic_deuteranopic_bwy_60_95_c32',diverging_protanopic_deuteranopic_bwy_60_95_c32)
m_diverging_protanopic_deuteranopic_bwy_60_95_c32 = mpl_cm('diverging_protanopic_deuteranopic_bwy_60_95_c32',diverging_protanopic_deuteranopic_bwy_60_95_c32)
m_diverging_protanopic_deuteranopic_bwy_60_95_c32_r = mpl_cm('diverging_protanopic_deuteranopic_bwy_60_95_c32_r',list(reversed(diverging_protanopic_deuteranopic_bwy_60_95_c32)))
bwy = b_diverging_protanopic_deuteranopic_bwy_60_95_c32
m_bwy = m_diverging_protanopic_deuteranopic_bwy_60_95_c32
m_bwy_r = m_diverging_protanopic_deuteranopic_bwy_60_95_c32_r
palette['bwy'] = b_diverging_protanopic_deuteranopic_bwy_60_95_c32
palette_n['bwy'] = b_diverging_protanopic_deuteranopic_bwy_60_95_c32
cm['bwy'] = m_diverging_protanopic_deuteranopic_bwy_60_95_c32
cm['bwy_r'] = m_diverging_protanopic_deuteranopic_bwy_60_95_c32_r
cm_n['bwy'] = mpl_cm('bwy',diverging_protanopic_deuteranopic_bwy_60_95_c32)
cm_n['bwy_r'] = mpl_cm('bwy_r',list(reversed(diverging_protanopic_deuteranopic_bwy_60_95_c32)))
CET_CBD1 = b_diverging_protanopic_deuteranopic_bwy_60_95_c32
m_CET_CBD1 = m_diverging_protanopic_deuteranopic_bwy_60_95_c32
m_CET_CBD1_r = m_diverging_protanopic_deuteranopic_bwy_60_95_c32_r
palette['CET_CBD1'] = b_diverging_protanopic_deuteranopic_bwy_60_95_c32
cm['CET_CBD1'] = m_diverging_protanopic_deuteranopic_bwy_60_95_c32
cm['CET_CBD1_r'] = m_diverging_protanopic_deuteranopic_bwy_60_95_c32_r
register_cmap('cet_CET_CBD1',m_diverging_protanopic_deuteranopic_bwy_60_95_c32)
register_cmap('cet_CET_CBD1_r',m_diverging_protanopic_deuteranopic_bwy_60_95_c32_r)




diverging_tritanopic_cwr_75_98_c20 = [  # cmap_def
[0.16144, 0.79052, 0.90539],
[0.17912, 0.79202, 0.90596],
[0.19524, 0.79352, 0.90654],
[0.21014, 0.79501, 0.90711],
[0.22404, 0.7965, 0.90769],
[0.23717, 0.798, 0.90826],
[0.24955, 0.7995, 0.90883],
[0.26139, 0.80098, 0.90941],
[0.27271, 0.80247, 0.90998],
[0.28359, 0.80397, 0.91056],
[0.29407, 0.80545, 0.91113],
[0.30421, 0.80695, 0.9117],
[0.31403, 0.80844, 0.91227],
[0.32357, 0.80992, 0.91285],
[0.33286, 0.81141, 0.91342],
[0.34189, 0.81289, 0.91399],
[0.35072, 0.81438, 0.91456],
[0.35933, 0.81587, 0.91513],
[0.36776, 0.81735, 0.9157],
[0.37604, 0.81883, 0.91627],
[0.38414, 0.82032, 0.91685],
[0.3921, 0.82181, 0.91742],
[0.39991, 0.82329, 0.91799],
[0.4076, 0.82477, 0.91856],
[0.41517, 0.82625, 0.91913],
[0.42261, 0.82773, 0.9197],
[0.42996, 0.82921, 0.92027],
[0.4372, 0.83069, 0.92083],
[0.44434, 0.83217, 0.92141],
[0.45139, 0.83365, 0.92198],
[0.45834, 0.83512, 0.92254],
[0.46522, 0.8366, 0.92311],
[0.47202, 0.83808, 0.92368],
[0.47874, 0.83955, 0.92425],
[0.48538, 0.84102, 0.92481],
[0.49197, 0.8425, 0.92538],
[0.49849, 0.84397, 0.92595],
[0.50493, 0.84544, 0.92651],
[0.51131, 0.84691, 0.92708],
[0.51766, 0.84838, 0.92765],
[0.52393, 0.84986, 0.92821],
[0.53015, 0.85132, 0.92878],
[0.53632, 0.8528, 0.92934],
[0.54244, 0.85426, 0.92991],
[0.54852, 0.85573, 0.93048],
[0.55454, 0.8572, 0.93104],
[0.56052, 0.85867, 0.93161],
[0.56646, 0.86014, 0.93217],
[0.57236, 0.8616, 0.93273],
[0.57822, 0.86307, 0.9333],
[0.58404, 0.86453, 0.93386],
[0.58982, 0.86599, 0.93442],
[0.59557, 0.86745, 0.93499],
[0.60128, 0.86892, 0.93555],
[0.60697, 0.87037, 0.93612],
[0.61261, 0.87184, 0.93668],
[0.61823, 0.8733, 0.93724],
[0.62382, 0.87476, 0.9378],
[0.62936, 0.87622, 0.93837],
[0.63489, 0.87768, 0.93893],
[0.64039, 0.87913, 0.93949],
[0.64586, 0.88059, 0.94005],
[0.65131, 0.88205, 0.94061],
[0.65673, 0.88351, 0.94117],
[0.66212, 0.88496, 0.94173],
[0.66749, 0.88642, 0.94229],
[0.67284, 0.88787, 0.94285],
[0.67816, 0.88932, 0.94341],
[0.68346, 0.89078, 0.94397],
[0.68874, 0.89223, 0.94453],
[0.69399, 0.89368, 0.94509],
[0.69923, 0.89513, 0.94565],
[0.70445, 0.89658, 0.9462],
[0.70965, 0.89803, 0.94676],
[0.71483, 0.89948, 0.94733],
[0.71999, 0.90093, 0.94788],
[0.72514, 0.90238, 0.94844],
[0.73026, 0.90382, 0.949],
[0.73536, 0.90527, 0.94955],
[0.74045, 0.90672, 0.95011],
[0.74553, 0.90817, 0.95067],
[0.75059, 0.90961, 0.95123],
[0.75563, 0.91105, 0.95178],
[0.76066, 0.9125, 0.95233],
[0.76567, 0.91394, 0.95289],
[0.77067, 0.91538, 0.95345],
[0.77565, 0.91682, 0.954],
[0.78062, 0.91826, 0.95456],
[0.78558, 0.9197, 0.95511],
[0.79052, 0.92114, 0.95567],
[0.79545, 0.92258, 0.95622],
[0.80037, 0.92402, 0.95677],
[0.80527, 0.92546, 0.95733],
[0.81016, 0.9269, 0.95789],
[0.81505, 0.92833, 0.95844],
[0.81991, 0.92977, 0.95899],
[0.82477, 0.93121, 0.95954],
[0.82962, 0.93264, 0.9601],
[0.83446, 0.93407, 0.96065],
[0.83928, 0.93551, 0.9612],
[0.8441, 0.93695, 0.96175],
[0.8489, 0.93838, 0.9623],
[0.85369, 0.93981, 0.96286],
[0.85848, 0.94124, 0.96341],
[0.86326, 0.94267, 0.96396],
[0.86803, 0.9441, 0.96451],
[0.87278, 0.94553, 0.96506],
[0.87753, 0.94696, 0.96561],
[0.88227, 0.94839, 0.96616],
[0.887, 0.94982, 0.96671],
[0.89172, 0.95124, 0.96726],
[0.89644, 0.95267, 0.96781],
[0.90114, 0.9541, 0.96836],
[0.90585, 0.95552, 0.9689],
[0.91054, 0.95694, 0.96944],
[0.91522, 0.95835, 0.96997],
[0.9199, 0.95974, 0.97048],
[0.92455, 0.9611, 0.97097],
[0.92919, 0.96242, 0.97141],
[0.93379, 0.96369, 0.97179],
[0.93836, 0.96486, 0.97208],
[0.94284, 0.96593, 0.97226],
[0.94723, 0.96684, 0.97228],
[0.95148, 0.96757, 0.97212],
[0.95556, 0.96806, 0.97173],
[0.95941, 0.96828, 0.97109],
[0.963, 0.9682, 0.97017],
[0.96629, 0.9678, 0.96895],
[0.96926, 0.96707, 0.96743],
[0.97189, 0.96601, 0.9656],
[0.97417, 0.96464, 0.96351],
[0.97613, 0.96299, 0.96117],
[0.97779, 0.9611, 0.95862],
[0.97918, 0.95899, 0.95589],
[0.98035, 0.95671, 0.95302],
[0.98133, 0.9543, 0.95004],
[0.98218, 0.9518, 0.94699],
[0.9829, 0.94923, 0.94388],
[0.98355, 0.94661, 0.94074],
[0.98415, 0.94395, 0.93757],
[0.9847, 0.94128, 0.93439],
[0.98522, 0.9386, 0.9312],
[0.98572, 0.93591, 0.92801],
[0.9862, 0.93321, 0.92482],
[0.98668, 0.93052, 0.92163],
[0.98714, 0.92782, 0.91844],
[0.98759, 0.92512, 0.91525],
[0.98804, 0.92243, 0.91207],
[0.98847, 0.91973, 0.90889],
[0.9889, 0.91704, 0.90572],
[0.98932, 0.91435, 0.90254],
[0.98973, 0.91165, 0.89936],
[0.99014, 0.90895, 0.89619],
[0.99053, 0.90626, 0.89302],
[0.99091, 0.90356, 0.88985],
[0.99129, 0.90087, 0.88668],
[0.99166, 0.89817, 0.88351],
[0.99202, 0.89548, 0.88035],
[0.99237, 0.89278, 0.87718],
[0.99272, 0.89008, 0.87403],
[0.99305, 0.88739, 0.87087],
[0.99338, 0.88469, 0.86772],
[0.9937, 0.882, 0.86456],
[0.99401, 0.8793, 0.86141],
[0.99431, 0.8766, 0.85826],
[0.9946, 0.87391, 0.85511],
[0.99489, 0.87121, 0.85197],
[0.99516, 0.86852, 0.84882],
[0.99543, 0.86582, 0.84568],
[0.9957, 0.86312, 0.84254],
[0.99595, 0.86043, 0.8394],
[0.9962, 0.85772, 0.83627],
[0.99643, 0.85503, 0.83313],
[0.99667, 0.85234, 0.83],
[0.99689, 0.84964, 0.82687],
[0.9971, 0.84693, 0.82374],
[0.99731, 0.84424, 0.82061],
[0.99751, 0.84154, 0.81749],
[0.9977, 0.83884, 0.81437],
[0.99789, 0.83614, 0.81125],
[0.99807, 0.83344, 0.80813],
[0.99824, 0.83074, 0.80502],
[0.9984, 0.82804, 0.8019],
[0.99855, 0.82534, 0.7988],
[0.9987, 0.82264, 0.79568],
[0.99884, 0.81994, 0.79258],
[0.99897, 0.81724, 0.78947],
[0.9991, 0.81453, 0.78637],
[0.99921, 0.81183, 0.78327],
[0.99932, 0.80913, 0.78017],
[0.99943, 0.80643, 0.77708],
[0.99952, 0.80372, 0.77397],
[0.99961, 0.80102, 0.77089],
[0.99969, 0.79832, 0.76779],
[0.99976, 0.79561, 0.76471],
[0.99983, 0.79291, 0.76162],
[0.99989, 0.7902, 0.75854],
[0.99994, 0.7875, 0.75545],
[0.99998, 0.78479, 0.75237],
[1, 0.78208, 0.74929],
[1, 0.77937, 0.74622],
[1, 0.77667, 0.74315],
[1, 0.77395, 0.74008],
[1, 0.77125, 0.737],
[1, 0.76853, 0.73394],
[1, 0.76583, 0.73087],
[1, 0.76311, 0.72781],
[1, 0.7604, 0.72475],
[1, 0.75769, 0.72169],
[1, 0.75498, 0.71863],
[0.99998, 0.75226, 0.71558],
[0.99994, 0.74954, 0.71252],
[0.99988, 0.74683, 0.70948],
[0.99983, 0.74412, 0.70643],
[0.99976, 0.7414, 0.70338],
[0.99969, 0.73868, 0.70034],
[0.99961, 0.73596, 0.69729],
[0.99953, 0.73324, 0.69425],
[0.99943, 0.73052, 0.69122],
[0.99934, 0.7278, 0.68819],
[0.99923, 0.72508, 0.68515],
[0.99912, 0.72235, 0.68212],
[0.999, 0.71963, 0.67909],
[0.99888, 0.71691, 0.67607],
[0.99875, 0.71418, 0.67305],
[0.99861, 0.71146, 0.67003],
[0.99846, 0.70873, 0.667],
[0.99831, 0.706, 0.66399],
[0.99816, 0.70327, 0.66097],
[0.99799, 0.70054, 0.65796],
[0.99782, 0.6978, 0.65494],
[0.99765, 0.69507, 0.65194],
[0.99746, 0.69234, 0.64894],
[0.99728, 0.6896, 0.64593],
[0.99708, 0.68686, 0.64293],
[0.99688, 0.68412, 0.63993],
[0.99667, 0.68138, 0.63694],
[0.99646, 0.67864, 0.63394],
[0.99624, 0.6759, 0.63095],
[0.99601, 0.67316, 0.62796],
[0.99578, 0.67042, 0.62497],
[0.99554, 0.66767, 0.62198],
[0.9953, 0.66491, 0.619],
[0.99505, 0.66217, 0.61602],
[0.9948, 0.65942, 0.61303],
[0.99453, 0.65667, 0.61006],
[0.99427, 0.65391, 0.60709],
[0.99399, 0.65116, 0.60412],
[0.99371, 0.64839, 0.60114],
[0.99343, 0.64563, 0.59817],
[0.99314, 0.64288, 0.59521],
[0.99284, 0.64012, 0.59226],
[0.99253, 0.63735, 0.58928],
[0.99222, 0.63459, 0.58633],
[0.99191, 0.63182, 0.58337],
[0.99158, 0.62904, 0.58043],
]
b_diverging_tritanopic_cwr_75_98_c20 = bokeh_palette('diverging_tritanopic_cwr_75_98_c20',diverging_tritanopic_cwr_75_98_c20)
m_diverging_tritanopic_cwr_75_98_c20 = mpl_cm('diverging_tritanopic_cwr_75_98_c20',diverging_tritanopic_cwr_75_98_c20)
m_diverging_tritanopic_cwr_75_98_c20_r = mpl_cm('diverging_tritanopic_cwr_75_98_c20_r',list(reversed(diverging_tritanopic_cwr_75_98_c20)))
cwr = b_diverging_tritanopic_cwr_75_98_c20
m_cwr = m_diverging_tritanopic_cwr_75_98_c20
m_cwr_r = m_diverging_tritanopic_cwr_75_98_c20_r
palette['cwr'] = b_diverging_tritanopic_cwr_75_98_c20
palette_n['cwr'] = b_diverging_tritanopic_cwr_75_98_c20
cm['cwr'] = m_diverging_tritanopic_cwr_75_98_c20
cm['cwr_r'] = m_diverging_tritanopic_cwr_75_98_c20_r
cm_n['cwr'] = mpl_cm('cwr',diverging_tritanopic_cwr_75_98_c20)
cm_n['cwr_r'] = mpl_cm('cwr_r',list(reversed(diverging_tritanopic_cwr_75_98_c20)))
CET_CBTD1 = b_diverging_tritanopic_cwr_75_98_c20
m_CET_CBTD1 = m_diverging_tritanopic_cwr_75_98_c20
m_CET_CBTD1_r = m_diverging_tritanopic_cwr_75_98_c20_r
palette['CET_CBTD1'] = b_diverging_tritanopic_cwr_75_98_c20
cm['CET_CBTD1'] = m_diverging_tritanopic_cwr_75_98_c20
cm['CET_CBTD1_r'] = m_diverging_tritanopic_cwr_75_98_c20_r
register_cmap('cet_CET_CBTD1',m_diverging_tritanopic_cwr_75_98_c20)
register_cmap('cet_CET_CBTD1_r',m_diverging_tritanopic_cwr_75_98_c20_r)




linear_protanopic_deuteranopic_kbw_5_98_c40 = [  # cmap_def
[0.066001, 0.066024, 0.066019],
[0.068392, 0.069594, 0.07418],
[0.070639, 0.072929, 0.081677],
[0.072893, 0.076121, 0.088587],
[0.075034, 0.079264, 0.095123],
[0.076966, 0.082389, 0.10148],
[0.078642, 0.085308, 0.10793],
[0.080058, 0.088268, 0.11442],
[0.081238, 0.091132, 0.121],
[0.082365, 0.093864, 0.12764],
[0.083353, 0.096586, 0.13432],
[0.084353, 0.099412, 0.14102],
[0.08516, 0.10221, 0.14774],
[0.085909, 0.105, 0.15449],
[0.086663, 0.10784, 0.16134],
[0.087299, 0.11072, 0.16815],
[0.087793, 0.11357, 0.17506],
[0.088197, 0.11644, 0.18195],
[0.088516, 0.11929, 0.18893],
[0.08874, 0.12215, 0.19591],
[0.088865, 0.12508, 0.20293],
[0.088887, 0.12801, 0.20998],
[0.088806, 0.13094, 0.21707],
[0.088619, 0.13387, 0.22414],
[0.088328, 0.13681, 0.2313],
[0.087938, 0.13974, 0.23842],
[0.087453, 0.14275, 0.24557],
[0.086824, 0.14576, 0.25276],
[0.086036, 0.14871, 0.25993],
[0.085236, 0.15172, 0.26709],
[0.084385, 0.15474, 0.27427],
[0.083325, 0.15777, 0.28142],
[0.082282, 0.16081, 0.28857],
[0.081076, 0.16388, 0.29568],
[0.079801, 0.16693, 0.30276],
[0.078483, 0.16998, 0.30983],
[0.077141, 0.17304, 0.31682],
[0.075789, 0.1761, 0.32377],
[0.074438, 0.17921, 0.33066],
[0.073088, 0.18229, 0.33751],
[0.071707, 0.18544, 0.34427],
[0.070334, 0.18854, 0.35099],
[0.069135, 0.19164, 0.35763],
[0.067852, 0.19479, 0.36417],
[0.066681, 0.19789, 0.37067],
[0.065605, 0.20102, 0.37711],
[0.064634, 0.20421, 0.38347],
[0.063763, 0.20734, 0.38976],
[0.063025, 0.21049, 0.39601],
[0.062318, 0.21366, 0.40219],
[0.061658, 0.21684, 0.40835],
[0.06116, 0.22002, 0.41445],
[0.060787, 0.2232, 0.42051],
[0.060495, 0.22639, 0.42655],
[0.060263, 0.22956, 0.43258],
[0.060082, 0.23277, 0.43856],
[0.059945, 0.236, 0.44455],
[0.059846, 0.2392, 0.45051],
[0.059777, 0.24242, 0.45647],
[0.059733, 0.24564, 0.46243],
[0.059705, 0.2489, 0.46839],
[0.059688, 0.25214, 0.47436],
[0.059676, 0.25539, 0.48032],
[0.059663, 0.25864, 0.48629],
[0.059645, 0.2619, 0.49228],
[0.059619, 0.26516, 0.49828],
[0.05958, 0.26844, 0.50428],
[0.059527, 0.27171, 0.51028],
[0.059457, 0.27501, 0.5163],
[0.059366, 0.27831, 0.52234],
[0.059251, 0.28161, 0.52839],
[0.059111, 0.28491, 0.53445],
[0.058943, 0.28823, 0.54052],
[0.058747, 0.29157, 0.54661],
[0.058523, 0.29488, 0.5527],
[0.058273, 0.29821, 0.5588],
[0.058003, 0.30155, 0.56492],
[0.05772, 0.30491, 0.57106],
[0.057436, 0.30826, 0.57721],
[0.057144, 0.31161, 0.58336],
[0.056825, 0.31498, 0.58953],
[0.056448, 0.31836, 0.59571],
[0.056009, 0.32173, 0.60191],
[0.055535, 0.32511, 0.60812],
[0.055044, 0.32851, 0.61432],
[0.054525, 0.33191, 0.62056],
[0.053973, 0.33532, 0.6268],
[0.05339, 0.33872, 0.63305],
[0.052777, 0.34214, 0.63931],
[0.052132, 0.34555, 0.64558],
[0.051458, 0.34898, 0.65187],
[0.050748, 0.35241, 0.65816],
[0.050004, 0.35585, 0.66447],
[0.049259, 0.35929, 0.67079],
[0.048333, 0.36274, 0.67712],
[0.047426, 0.3662, 0.68345],
[0.046481, 0.36965, 0.68981],
[0.045473, 0.37311, 0.69617],
[0.044473, 0.3766, 0.70254],
[0.043261, 0.38007, 0.70893],
[0.042229, 0.38355, 0.71532],
[0.040999, 0.38703, 0.72172],
[0.039699, 0.39053, 0.72813],
[0.038351, 0.39403, 0.73456],
[0.036958, 0.39754, 0.74099],
[0.035554, 0.40106, 0.74745],
[0.033822, 0.40456, 0.7539],
[0.032325, 0.40808, 0.76037],
[0.030752, 0.41161, 0.76684],
[0.029133, 0.41514, 0.77333],
[0.027467, 0.41867, 0.77983],
[0.025753, 0.42221, 0.78634],
[0.023992, 0.42577, 0.79286],
[0.022182, 0.42932, 0.79939],
[0.020323, 0.43288, 0.80592],
[0.018415, 0.43645, 0.81247],
[0.016457, 0.44002, 0.81902],
[0.014451, 0.44359, 0.82559],
[0.012421, 0.44716, 0.83217],
[0.010137, 0.45075, 0.83875],
[0.0081674, 0.45435, 0.84533],
[0.0062813, 0.45793, 0.8519],
[0.0045555, 0.46153, 0.85845],
[0.003069, 0.46513, 0.86499],
[0.0019224, 0.46874, 0.8715],
[0.0012443, 0.47235, 0.87795],
[0.0011966, 0.47597, 0.88435],
[0.00198, 0.47958, 0.89067],
[0.0038395, 0.48319, 0.89689],
[0.0070702, 0.48682, 0.90298],
[0.012242, 0.49044, 0.9089],
[0.019291, 0.49406, 0.91462],
[0.028957, 0.49768, 0.9201],
[0.041939, 0.5013, 0.92529],
[0.056334, 0.50491, 0.93014],
[0.07148, 0.50852, 0.93458],
[0.087293, 0.51211, 0.93858],
[0.10371, 0.5157, 0.94207],
[0.12071, 0.5193, 0.94497],
[0.13835, 0.52288, 0.94724],
[0.15644, 0.52646, 0.94882],
[0.17494, 0.53001, 0.94965],
[0.19382, 0.53356, 0.94969],
[0.21295, 0.53712, 0.9489],
[0.23232, 0.54065, 0.94726],
[0.25179, 0.54419, 0.94473],
[0.27128, 0.54772, 0.94131],
[0.29072, 0.55126, 0.93701],
[0.31005, 0.5548, 0.93183],
[0.32916, 0.55833, 0.92581],
[0.34799, 0.5619, 0.91897],
[0.36649, 0.56545, 0.91136],
[0.38458, 0.56903, 0.90303],
[0.40225, 0.57264, 0.89403],
[0.41947, 0.57626, 0.88442],
[0.43617, 0.57989, 0.87426],
[0.45236, 0.58355, 0.86363],
[0.46801, 0.58723, 0.85257],
[0.48314, 0.59094, 0.84116],
[0.49776, 0.59467, 0.82944],
[0.51183, 0.59842, 0.81747],
[0.52542, 0.60219, 0.8053],
[0.53853, 0.60597, 0.79299],
[0.55116, 0.60978, 0.78056],
[0.56336, 0.6136, 0.76805],
[0.57514, 0.61743, 0.75551],
[0.58654, 0.62127, 0.74295],
[0.59758, 0.62512, 0.7304],
[0.60828, 0.62896, 0.71787],
[0.61867, 0.63281, 0.70539],
[0.62877, 0.63665, 0.69297],
[0.63861, 0.64049, 0.68059],
[0.64821, 0.64431, 0.66831],
[0.6576, 0.64813, 0.65609],
[0.66677, 0.65195, 0.64394],
[0.67576, 0.65575, 0.63188],
[0.68457, 0.65954, 0.6199],
[0.69322, 0.66332, 0.60798],
[0.70171, 0.66709, 0.5961],
[0.71007, 0.67086, 0.58428],
[0.71829, 0.67461, 0.57251],
[0.72638, 0.67836, 0.56075],
[0.73435, 0.6821, 0.54901],
[0.74219, 0.68585, 0.53726],
[0.74992, 0.68959, 0.5255],
[0.75754, 0.69333, 0.51371],
[0.76504, 0.69707, 0.50188],
[0.77243, 0.70083, 0.48999],
[0.77972, 0.70458, 0.47802],
[0.78691, 0.70834, 0.46597],
[0.79399, 0.71211, 0.45381],
[0.80096, 0.71589, 0.44153],
[0.80784, 0.71968, 0.42911],
[0.81462, 0.72347, 0.41654],
[0.8213, 0.72728, 0.40382],
[0.82788, 0.7311, 0.39092],
[0.83438, 0.73493, 0.37784],
[0.84078, 0.73878, 0.36456],
[0.84708, 0.74263, 0.35109],
[0.85329, 0.74649, 0.33741],
[0.85941, 0.75037, 0.3235],
[0.86544, 0.75426, 0.30941],
[0.87139, 0.75816, 0.29507],
[0.87723, 0.76206, 0.28057],
[0.88299, 0.76598, 0.26594],
[0.88866, 0.76991, 0.25118],
[0.89423, 0.77384, 0.23637],
[0.89971, 0.77779, 0.22159],
[0.9051, 0.78174, 0.207],
[0.91039, 0.7857, 0.19274],
[0.91558, 0.78966, 0.17908],
[0.92068, 0.79363, 0.16631],
[0.9257, 0.79759, 0.15487],
[0.93061, 0.80156, 0.14532],
[0.93542, 0.80553, 0.13816],
[0.94013, 0.8095, 0.13406],
[0.94474, 0.81346, 0.13353],
[0.94923, 0.81743, 0.13683],
[0.95362, 0.8214, 0.14398],
[0.95788, 0.82536, 0.15462],
[0.962, 0.82933, 0.16835],
[0.96598, 0.83329, 0.18464],
[0.96981, 0.83726, 0.20298],
[0.97348, 0.84123, 0.22303],
[0.97696, 0.8452, 0.24438],
[0.98026, 0.84919, 0.26685],
[0.98334, 0.85318, 0.29018],
[0.98621, 0.85718, 0.31424],
[0.98884, 0.8612, 0.33888],
[0.99123, 0.86524, 0.36397],
[0.99337, 0.8693, 0.38944],
[0.99524, 0.87337, 0.41519],
[0.99685, 0.87747, 0.44112],
[0.99819, 0.88159, 0.46718],
[0.99927, 0.88573, 0.49327],
[1, 0.8899, 0.51936],
[1, 0.89409, 0.54534],
[1, 0.89831, 0.57117],
[1, 0.90255, 0.59679],
[1, 0.90681, 0.62212],
[1, 0.9111, 0.64712],
[0.99988, 0.9154, 0.67173],
[0.99916, 0.91973, 0.69589],
[0.99829, 0.92408, 0.71955],
[0.99731, 0.92843, 0.74267],
[0.99626, 0.93281, 0.76519],
[0.99515, 0.93719, 0.78709],
[0.99402, 0.94158, 0.80832],
[0.99289, 0.94598, 0.82887],
[0.9918, 0.95038, 0.84871],
[0.99077, 0.95479, 0.86784],
[0.98983, 0.95919, 0.88625],
[0.98898, 0.96359, 0.90396],
[0.98824, 0.968, 0.92098],
[0.98763, 0.9724, 0.93736],
[0.98715, 0.97679, 0.95311],
]
b_linear_protanopic_deuteranopic_kbw_5_98_c40 = bokeh_palette('linear_protanopic_deuteranopic_kbw_5_98_c40',linear_protanopic_deuteranopic_kbw_5_98_c40)
m_linear_protanopic_deuteranopic_kbw_5_98_c40 = mpl_cm('linear_protanopic_deuteranopic_kbw_5_98_c40',linear_protanopic_deuteranopic_kbw_5_98_c40)
m_linear_protanopic_deuteranopic_kbw_5_98_c40_r = mpl_cm('linear_protanopic_deuteranopic_kbw_5_98_c40_r',list(reversed(linear_protanopic_deuteranopic_kbw_5_98_c40)))
CET_CBL2 = b_linear_protanopic_deuteranopic_kbw_5_98_c40
m_CET_CBL2 = m_linear_protanopic_deuteranopic_kbw_5_98_c40
m_CET_CBL2_r = m_linear_protanopic_deuteranopic_kbw_5_98_c40_r
palette['CET_CBL2'] = b_linear_protanopic_deuteranopic_kbw_5_98_c40
cm['CET_CBL2'] = m_linear_protanopic_deuteranopic_kbw_5_98_c40
cm['CET_CBL2_r'] = m_linear_protanopic_deuteranopic_kbw_5_98_c40_r
register_cmap('cet_CET_CBL2',m_linear_protanopic_deuteranopic_kbw_5_98_c40)
register_cmap('cet_CET_CBL2_r',m_linear_protanopic_deuteranopic_kbw_5_98_c40_r)




linear_protanopic_deuteranopic_kbjyw_5_95_c25 = [  # cmap_def
[0.066001, 0.066024, 0.066019],
[0.067911, 0.069463, 0.075246],
[0.069781, 0.072628, 0.083652],
[0.071601, 0.075736, 0.091369],
[0.073375, 0.078755, 0.098548],
[0.074744, 0.081812, 0.10574],
[0.075822, 0.084689, 0.113],
[0.07659, 0.087551, 0.12018],
[0.077063, 0.090312, 0.12745],
[0.077469, 0.092947, 0.13478],
[0.07779, 0.095675, 0.14203],
[0.077991, 0.098388, 0.14933],
[0.078072, 0.10107, 0.15663],
[0.078044, 0.10383, 0.16393],
[0.077898, 0.1066, 0.1712],
[0.077627, 0.10934, 0.17851],
[0.077244, 0.1121, 0.18581],
[0.076748, 0.11483, 0.19306],
[0.07613, 0.11765, 0.20032],
[0.075395, 0.12044, 0.20761],
[0.074553, 0.12325, 0.21485],
[0.073621, 0.12614, 0.22209],
[0.07242, 0.12894, 0.22932],
[0.071237, 0.13182, 0.23655],
[0.069858, 0.13471, 0.24371],
[0.068359, 0.13758, 0.2509],
[0.066676, 0.14045, 0.25804],
[0.064913, 0.14335, 0.26516],
[0.063061, 0.14624, 0.27225],
[0.06092, 0.14916, 0.27935],
[0.058754, 0.15207, 0.28637],
[0.056459, 0.15501, 0.29338],
[0.05389, 0.15796, 0.30036],
[0.051251, 0.16091, 0.3073],
[0.048462, 0.1639, 0.31422],
[0.045461, 0.16685, 0.32108],
[0.042367, 0.16983, 0.3279],
[0.039038, 0.17278, 0.33469],
[0.035642, 0.17576, 0.34142],
[0.031997, 0.17879, 0.34812],
[0.028613, 0.18176, 0.35476],
[0.025247, 0.1848, 0.36133],
[0.021912, 0.18784, 0.36786],
[0.018626, 0.19082, 0.37436],
[0.015435, 0.19388, 0.38078],
[0.012391, 0.19691, 0.38712],
[0.0092372, 0.19991, 0.39342],
[0.0064174, 0.20298, 0.39965],
[0.0037591, 0.20605, 0.40583],
[0.0012787, 0.20909, 0.41193],
[0, 0.21217, 0.41795],
[0, 0.21521, 0.42389],
[0, 0.2183, 0.42977],
[0, 0.22137, 0.43556],
[0, 0.22443, 0.44127],
[0, 0.22754, 0.44689],
[0, 0.23062, 0.45245],
[0, 0.2337, 0.45788],
[0, 0.23683, 0.46322],
[0, 0.23991, 0.46848],
[0, 0.24301, 0.47362],
[0, 0.24612, 0.47867],
[0, 0.24924, 0.48361],
[0.001718, 0.25236, 0.48845],
[0.0054283, 0.25548, 0.49316],
[0.0098053, 0.2586, 0.49777],
[0.015086, 0.26172, 0.50224],
[0.020916, 0.26484, 0.50658],
[0.02753, 0.26797, 0.5108],
[0.03516, 0.27112, 0.51489],
[0.043256, 0.27425, 0.51884],
[0.051559, 0.2774, 0.52264],
[0.059842, 0.28053, 0.52629],
[0.067933, 0.28369, 0.52979],
[0.075984, 0.28683, 0.53312],
[0.084152, 0.28999, 0.5363],
[0.092218, 0.29315, 0.53929],
[0.10029, 0.29632, 0.54212],
[0.10839, 0.29949, 0.54476],
[0.11658, 0.30264, 0.54721],
[0.12471, 0.30584, 0.54948],
[0.13293, 0.30901, 0.55153],
[0.14118, 0.31216, 0.55337],
[0.14943, 0.31536, 0.55501],
[0.15772, 0.31855, 0.5564],
[0.16609, 0.32174, 0.55756],
[0.17453, 0.32493, 0.55846],
[0.18296, 0.32814, 0.55911],
[0.19147, 0.33134, 0.55951],
[0.19998, 0.33458, 0.55963],
[0.20861, 0.33779, 0.55944],
[0.21727, 0.34102, 0.55894],
[0.22597, 0.34425, 0.55814],
[0.2347, 0.34751, 0.55702],
[0.24346, 0.35077, 0.55551],
[0.2523, 0.35401, 0.55366],
[0.26116, 0.3573, 0.55144],
[0.26997, 0.36057, 0.54894],
[0.27856, 0.36387, 0.54636],
[0.2869, 0.36717, 0.54377],
[0.29504, 0.37047, 0.54116],
[0.30298, 0.3738, 0.53857],
[0.31076, 0.37713, 0.53594],
[0.31835, 0.38046, 0.53331],
[0.32578, 0.3838, 0.53068],
[0.3331, 0.38714, 0.52804],
[0.34026, 0.3905, 0.52538],
[0.3473, 0.39386, 0.52272],
[0.35422, 0.39723, 0.52004],
[0.36104, 0.40062, 0.51737],
[0.36776, 0.40399, 0.51467],
[0.3744, 0.40737, 0.51196],
[0.38094, 0.41078, 0.50925],
[0.38738, 0.41417, 0.50652],
[0.39377, 0.41758, 0.50379],
[0.40007, 0.42098, 0.50105],
[0.40632, 0.4244, 0.49829],
[0.41249, 0.42783, 0.49551],
[0.41859, 0.43126, 0.49272],
[0.42465, 0.43469, 0.48993],
[0.43065, 0.43813, 0.48713],
[0.4366, 0.44157, 0.4843],
[0.4425, 0.44501, 0.48147],
[0.44834, 0.44846, 0.47862],
[0.45415, 0.45192, 0.47577],
[0.45991, 0.45538, 0.47288],
[0.46563, 0.45884, 0.47],
[0.47133, 0.46231, 0.4671],
[0.47697, 0.46578, 0.46418],
[0.48259, 0.46927, 0.46124],
[0.48818, 0.47274, 0.45829],
[0.49373, 0.47623, 0.45533],
[0.49925, 0.47972, 0.45235],
[0.50476, 0.4832, 0.44935],
[0.51022, 0.4867, 0.44632],
[0.51567, 0.49019, 0.44329],
[0.5211, 0.49369, 0.44024],
[0.52651, 0.4972, 0.43717],
[0.53188, 0.50071, 0.43407],
[0.53725, 0.50422, 0.43097],
[0.54259, 0.50773, 0.42784],
[0.54791, 0.51124, 0.42469],
[0.55321, 0.51476, 0.42151],
[0.5585, 0.51829, 0.41832],
[0.56379, 0.5218, 0.41511],
[0.56905, 0.52533, 0.41187],
[0.5743, 0.52887, 0.40861],
[0.57955, 0.53239, 0.40533],
[0.58477, 0.53592, 0.40201],
[0.58998, 0.53945, 0.39868],
[0.59519, 0.54299, 0.39532],
[0.60039, 0.54653, 0.39193],
[0.60557, 0.55007, 0.38851],
[0.61076, 0.55361, 0.38506],
[0.61593, 0.55716, 0.38159],
[0.6211, 0.5607, 0.37809],
[0.62626, 0.56425, 0.37456],
[0.63141, 0.56779, 0.37098],
[0.63657, 0.57134, 0.36737],
[0.64169, 0.57489, 0.36381],
[0.64675, 0.57845, 0.36053],
[0.65172, 0.58202, 0.35767],
[0.6566, 0.58561, 0.35517],
[0.6614, 0.58918, 0.35303],
[0.66612, 0.59279, 0.35128],
[0.67079, 0.59638, 0.34987],
[0.67539, 0.59999, 0.34879],
[0.67993, 0.60361, 0.34803],
[0.68442, 0.60723, 0.34761],
[0.68887, 0.61085, 0.3475],
[0.69327, 0.61448, 0.34767],
[0.69761, 0.61812, 0.34813],
[0.70192, 0.62176, 0.34887],
[0.70621, 0.62541, 0.34988],
[0.71044, 0.62907, 0.35116],
[0.71463, 0.63273, 0.35269],
[0.7188, 0.6364, 0.35449],
[0.72293, 0.64007, 0.35653],
[0.72703, 0.64374, 0.35878],
[0.7311, 0.64742, 0.36128],
[0.73514, 0.65111, 0.364],
[0.73915, 0.6548, 0.36694],
[0.74313, 0.6585, 0.37008],
[0.74708, 0.6622, 0.37343],
[0.75101, 0.66591, 0.37699],
[0.75491, 0.66962, 0.38073],
[0.75878, 0.67334, 0.38465],
[0.76262, 0.67706, 0.38877],
[0.76644, 0.68078, 0.39305],
[0.77023, 0.68451, 0.39752],
[0.77399, 0.68825, 0.40214],
[0.77773, 0.69199, 0.40694],
[0.78145, 0.69574, 0.4119],
[0.78513, 0.69949, 0.41703],
[0.78879, 0.70325, 0.42229],
[0.79241, 0.70701, 0.42772],
[0.79601, 0.71078, 0.43328],
[0.7996, 0.71454, 0.43898],
[0.80314, 0.71832, 0.44483],
[0.80666, 0.7221, 0.4508],
[0.81015, 0.7259, 0.4569],
[0.81362, 0.72969, 0.46313],
[0.81705, 0.73349, 0.46952],
[0.82046, 0.73729, 0.47601],
[0.82383, 0.7411, 0.48262],
[0.82718, 0.74492, 0.48934],
[0.8305, 0.74874, 0.49618],
[0.83379, 0.75256, 0.50312],
[0.83704, 0.7564, 0.5102],
[0.84026, 0.76024, 0.5174],
[0.84344, 0.76408, 0.52468],
[0.8466, 0.76792, 0.53208],
[0.84973, 0.77178, 0.53956],
[0.85282, 0.77564, 0.54716],
[0.85587, 0.77951, 0.55487],
[0.85889, 0.78338, 0.56268],
[0.86188, 0.78726, 0.57056],
[0.86483, 0.79114, 0.57855],
[0.86774, 0.79503, 0.58664],
[0.87061, 0.79893, 0.59483],
[0.87345, 0.80283, 0.6031],
[0.87625, 0.80674, 0.61146],
[0.87901, 0.81065, 0.61992],
[0.88173, 0.81457, 0.62847],
[0.88441, 0.8185, 0.6371],
[0.88705, 0.82244, 0.6458],
[0.88965, 0.82637, 0.6546],
[0.89221, 0.83032, 0.6635],
[0.89472, 0.83427, 0.67247],
[0.89719, 0.83823, 0.68151],
[0.89961, 0.8422, 0.69064],
[0.90199, 0.84616, 0.69987],
[0.90432, 0.85014, 0.70916],
[0.90661, 0.85413, 0.71852],
[0.90884, 0.85812, 0.72798],
[0.91103, 0.86212, 0.73752],
[0.91318, 0.86612, 0.74712],
[0.91526, 0.87013, 0.7568],
[0.91731, 0.87415, 0.76657],
[0.91929, 0.87818, 0.7764],
[0.92123, 0.88221, 0.7863],
[0.92311, 0.88625, 0.79629],
[0.92493, 0.89029, 0.80635],
[0.92671, 0.89435, 0.81646],
[0.92842, 0.8984, 0.82668],
[0.93009, 0.90247, 0.83695],
[0.93169, 0.90654, 0.84728],
[0.93323, 0.91063, 0.8577],
[0.93471, 0.91471, 0.86819],
[0.93614, 0.9188, 0.87873],
[0.9375, 0.92291, 0.88936],
[0.93879, 0.92701, 0.90005],
[0.94003, 0.93113, 0.91081],
[0.94119, 0.93526, 0.92164],
[0.9423, 0.93939, 0.93253],
[0.94334, 0.94353, 0.94348],
]
b_linear_protanopic_deuteranopic_kbjyw_5_95_c25 = bokeh_palette('linear_protanopic_deuteranopic_kbjyw_5_95_c25',linear_protanopic_deuteranopic_kbjyw_5_95_c25)
m_linear_protanopic_deuteranopic_kbjyw_5_95_c25 = mpl_cm('linear_protanopic_deuteranopic_kbjyw_5_95_c25',linear_protanopic_deuteranopic_kbjyw_5_95_c25)
m_linear_protanopic_deuteranopic_kbjyw_5_95_c25_r = mpl_cm('linear_protanopic_deuteranopic_kbjyw_5_95_c25_r',list(reversed(linear_protanopic_deuteranopic_kbjyw_5_95_c25)))
CET_CBL1 = b_linear_protanopic_deuteranopic_kbjyw_5_95_c25
m_CET_CBL1 = m_linear_protanopic_deuteranopic_kbjyw_5_95_c25
m_CET_CBL1_r = m_linear_protanopic_deuteranopic_kbjyw_5_95_c25_r
palette['CET_CBL1'] = b_linear_protanopic_deuteranopic_kbjyw_5_95_c25
cm['CET_CBL1'] = m_linear_protanopic_deuteranopic_kbjyw_5_95_c25
cm['CET_CBL1_r'] = m_linear_protanopic_deuteranopic_kbjyw_5_95_c25_r
register_cmap('cet_CET_CBL1',m_linear_protanopic_deuteranopic_kbjyw_5_95_c25)
register_cmap('cet_CET_CBL1_r',m_linear_protanopic_deuteranopic_kbjyw_5_95_c25_r)




linear_worb_100_25_c53 = [  # cmap_def
[0.99986, 1, 1],
[0.99813, 0.99701, 0.98897],
[0.99635, 0.99397, 0.97793],
[0.99454, 0.99093, 0.9669],
[0.9927, 0.98789, 0.95589],
[0.99081, 0.98487, 0.94488],
[0.98889, 0.98184, 0.93388],
[0.98693, 0.97881, 0.92289],
[0.98493, 0.97579, 0.91191],
[0.9829, 0.97278, 0.90094],
[0.98083, 0.96977, 0.88998],
[0.97872, 0.96676, 0.87903],
[0.97658, 0.96376, 0.86809],
[0.9744, 0.96076, 0.85715],
[0.97219, 0.95777, 0.84621],
[0.96994, 0.95478, 0.83529],
[0.96766, 0.95179, 0.82437],
[0.96534, 0.94881, 0.81346],
[0.96299, 0.94583, 0.80256],
[0.9606, 0.94286, 0.79166],
[0.95818, 0.93989, 0.78076],
[0.95572, 0.93692, 0.76987],
[0.95323, 0.93395, 0.75898],
[0.9507, 0.931, 0.7481],
[0.95041, 0.92708, 0.74072],
[0.95061, 0.92296, 0.73414],
[0.95077, 0.91883, 0.72755],
[0.95091, 0.91472, 0.72097],
[0.95103, 0.9106, 0.71439],
[0.95112, 0.90648, 0.70782],
[0.95118, 0.90237, 0.70126],
[0.95122, 0.89825, 0.6947],
[0.95123, 0.89415, 0.68815],
[0.95122, 0.89004, 0.6816],
[0.95119, 0.88593, 0.67506],
[0.95113, 0.88183, 0.66852],
[0.95104, 0.87773, 0.66199],
[0.95094, 0.87363, 0.65546],
[0.9508, 0.86953, 0.64893],
[0.95065, 0.86544, 0.64241],
[0.95047, 0.86135, 0.6359],
[0.95027, 0.85725, 0.62938],
[0.95004, 0.85317, 0.62288],
[0.94979, 0.84908, 0.61638],
[0.94952, 0.845, 0.60988],
[0.94923, 0.84092, 0.60339],
[0.94891, 0.83684, 0.59689],
[0.94918, 0.83247, 0.59152],
[0.94977, 0.82794, 0.58679],
[0.95035, 0.82341, 0.58207],
[0.9509, 0.81887, 0.57735],
[0.95142, 0.81434, 0.57264],
[0.95192, 0.80981, 0.56793],
[0.95241, 0.80527, 0.56323],
[0.95287, 0.80073, 0.55852],
[0.95331, 0.79619, 0.55383],
[0.95373, 0.79165, 0.54914],
[0.95413, 0.78711, 0.54445],
[0.9545, 0.78256, 0.53977],
[0.95486, 0.77801, 0.53509],
[0.95519, 0.77346, 0.53042],
[0.9555, 0.7689, 0.52576],
[0.9558, 0.76435, 0.52109],
[0.95607, 0.75979, 0.51643],
[0.95632, 0.75524, 0.51177],
[0.95655, 0.75067, 0.50713],
[0.95676, 0.7461, 0.50248],
[0.95695, 0.74153, 0.49785],
[0.95712, 0.73697, 0.4932],
[0.95728, 0.7324, 0.48857],
[0.95753, 0.72771, 0.48478],
[0.9579, 0.72291, 0.48202],
[0.95825, 0.7181, 0.47925],
[0.95859, 0.71329, 0.47649],
[0.95891, 0.70847, 0.47372],
[0.95921, 0.70364, 0.47098],
[0.9595, 0.6988, 0.46822],
[0.95978, 0.69396, 0.46546],
[0.96003, 0.68912, 0.46272],
[0.96028, 0.68425, 0.45998],
[0.9605, 0.67938, 0.45725],
[0.96071, 0.67451, 0.45452],
[0.96091, 0.66964, 0.45179],
[0.96109, 0.66474, 0.44907],
[0.96125, 0.65985, 0.44633],
[0.9614, 0.65493, 0.44362],
[0.96153, 0.65002, 0.4409],
[0.96165, 0.64509, 0.4382],
[0.96176, 0.64016, 0.43549],
[0.96185, 0.63521, 0.43279],
[0.96192, 0.63025, 0.43008],
[0.96198, 0.62528, 0.4274],
[0.96203, 0.62031, 0.4247],
[0.96198, 0.61532, 0.42251],
[0.96172, 0.61035, 0.42167],
[0.96146, 0.60536, 0.42084],
[0.96118, 0.60037, 0.42],
[0.96089, 0.59536, 0.41917],
[0.96058, 0.59034, 0.41832],
[0.96027, 0.58532, 0.41749],
[0.95995, 0.58026, 0.41664],
[0.95961, 0.5752, 0.41579],
[0.95927, 0.57012, 0.41495],
[0.95892, 0.56502, 0.4141],
[0.95855, 0.55992, 0.41326],
[0.95818, 0.55479, 0.41241],
[0.95779, 0.54965, 0.41155],
[0.9574, 0.54449, 0.41071],
[0.95699, 0.53931, 0.40985],
[0.95658, 0.53411, 0.40899],
[0.95615, 0.5289, 0.40814],
[0.95572, 0.52366, 0.40728],
[0.95527, 0.51841, 0.40642],
[0.95482, 0.51312, 0.40557],
[0.95435, 0.50782, 0.40469],
[0.95388, 0.50249, 0.40384],
[0.95333, 0.49718, 0.40312],
[0.95211, 0.49218, 0.40389],
[0.9509, 0.48716, 0.40464],
[0.94967, 0.48212, 0.40541],
[0.94844, 0.47705, 0.40615],
[0.94721, 0.47198, 0.40688],
[0.94596, 0.46688, 0.4076],
[0.94471, 0.46175, 0.40832],
[0.94345, 0.4566, 0.40903],
[0.94219, 0.45143, 0.40974],
[0.94091, 0.44623, 0.41043],
[0.93963, 0.44101, 0.41111],
[0.93835, 0.43577, 0.41178],
[0.93706, 0.43049, 0.41246],
[0.93575, 0.42519, 0.41312],
[0.93444, 0.41985, 0.41377],
[0.93313, 0.41448, 0.41441],
[0.93181, 0.40908, 0.41505],
[0.93048, 0.40365, 0.41568],
[0.92914, 0.39818, 0.4163],
[0.9278, 0.39266, 0.41693],
[0.92645, 0.38711, 0.41753],
[0.9251, 0.38153, 0.41812],
[0.92374, 0.3759, 0.41872],
[0.9214, 0.371, 0.42046],
[0.91895, 0.36617, 0.42229],
[0.91651, 0.3613, 0.42413],
[0.91406, 0.35642, 0.42595],
[0.9116, 0.3515, 0.42776],
[0.90913, 0.34655, 0.42955],
[0.90666, 0.34158, 0.43134],
[0.90418, 0.33656, 0.43311],
[0.90169, 0.33152, 0.43488],
[0.8992, 0.32644, 0.43663],
[0.89671, 0.32133, 0.43837],
[0.8942, 0.3162, 0.44011],
[0.89169, 0.311, 0.44184],
[0.88917, 0.30577, 0.44355],
[0.88665, 0.30047, 0.44525],
[0.88412, 0.29514, 0.44694],
[0.88158, 0.28977, 0.44863],
[0.87903, 0.28434, 0.45031],
[0.87648, 0.27886, 0.45198],
[0.87392, 0.27332, 0.45363],
[0.87136, 0.26768, 0.45529],
[0.86879, 0.26202, 0.45692],
[0.86621, 0.25626, 0.45856],
[0.86267, 0.25165, 0.46077],
[0.85876, 0.24743, 0.4632],
[0.85484, 0.24317, 0.46563],
[0.85091, 0.23891, 0.46805],
[0.84698, 0.23464, 0.47046],
[0.84303, 0.23029, 0.47285],
[0.83908, 0.22597, 0.47526],
[0.83512, 0.22157, 0.47763],
[0.83114, 0.21715, 0.48001],
[0.82716, 0.21269, 0.48238],
[0.82316, 0.2082, 0.48473],
[0.81915, 0.20367, 0.48709],
[0.81514, 0.19907, 0.48943],
[0.81111, 0.19448, 0.49178],
[0.80707, 0.18978, 0.49411],
[0.80302, 0.18507, 0.49642],
[0.79896, 0.18025, 0.49875],
[0.79488, 0.17541, 0.50106],
[0.79079, 0.1705, 0.50336],
[0.7867, 0.16548, 0.50565],
[0.78258, 0.16042, 0.50796],
[0.77846, 0.15529, 0.51024],
[0.77432, 0.15006, 0.51252],
[0.76939, 0.14629, 0.51497],
[0.76379, 0.14388, 0.51756],
[0.75817, 0.14147, 0.52012],
[0.75253, 0.13902, 0.5227],
[0.74688, 0.13662, 0.52527],
[0.7412, 0.13426, 0.52783],
[0.7355, 0.13187, 0.53039],
[0.72978, 0.12948, 0.53295],
[0.72404, 0.12709, 0.5355],
[0.71827, 0.12474, 0.53806],
[0.71247, 0.12237, 0.54059],
[0.70667, 0.12005, 0.54314],
[0.70082, 0.11774, 0.54567],
[0.69495, 0.11543, 0.54821],
[0.68906, 0.11317, 0.55074],
[0.68313, 0.11086, 0.55326],
[0.67718, 0.10854, 0.55578],
[0.67119, 0.10634, 0.5583],
[0.66518, 0.10405, 0.56082],
[0.65913, 0.10182, 0.56334],
[0.65304, 0.099603, 0.56585],
[0.64694, 0.097418, 0.56836],
[0.64078, 0.095268, 0.57086],
[0.6341, 0.094454, 0.57328],
[0.62648, 0.09592, 0.57555],
[0.61882, 0.09742, 0.5778],
[0.61111, 0.098865, 0.58006],
[0.60334, 0.10027, 0.58231],
[0.59551, 0.10166, 0.58457],
[0.58763, 0.10305, 0.58682],
[0.5797, 0.10434, 0.58907],
[0.57169, 0.10567, 0.59133],
[0.56362, 0.10698, 0.59358],
[0.55546, 0.10817, 0.59582],
[0.54725, 0.10947, 0.59807],
[0.53895, 0.11066, 0.60031],
[0.53057, 0.11184, 0.60256],
[0.52209, 0.11302, 0.6048],
[0.51354, 0.1141, 0.60705],
[0.50487, 0.11523, 0.60929],
[0.49609, 0.11637, 0.61152],
[0.48723, 0.11741, 0.61376],
[0.47822, 0.11852, 0.616],
[0.46911, 0.11953, 0.61824],
[0.45984, 0.12055, 0.62048],
[0.45044, 0.12155, 0.6227],
[0.44067, 0.12289, 0.62479],
[0.42985, 0.12576, 0.62622],
[0.41882, 0.12843, 0.62765],
[0.40758, 0.13102, 0.62908],
[0.39611, 0.13348, 0.63051],
[0.38438, 0.1358, 0.63194],
[0.37237, 0.13812, 0.63337],
[0.36005, 0.14027, 0.63479],
[0.3474, 0.14236, 0.63622],
[0.33433, 0.14437, 0.63764],
[0.32083, 0.14628, 0.63907],
[0.30681, 0.14812, 0.64049],
[0.29221, 0.14993, 0.64192],
[0.27695, 0.1516, 0.64334],
[0.26086, 0.15323, 0.64476],
[0.24379, 0.15482, 0.64618],
[0.22557, 0.15637, 0.64761],
[0.2058, 0.1578, 0.64903],
[0.18401, 0.1592, 0.65045],
[0.15944, 0.16053, 0.65187],
[0.13065, 0.16186, 0.65329],
[0.094242, 0.16309, 0.65471],
[0.038143, 0.16429, 0.65613],
[0, 0.16539, 0.65755],
]
b_linear_worb_100_25_c53 = bokeh_palette('linear_worb_100_25_c53',linear_worb_100_25_c53)
m_linear_worb_100_25_c53 = mpl_cm('linear_worb_100_25_c53',linear_worb_100_25_c53)
m_linear_worb_100_25_c53_r = mpl_cm('linear_worb_100_25_c53_r',list(reversed(linear_worb_100_25_c53)))
CET_L17 = b_linear_worb_100_25_c53
m_CET_L17 = m_linear_worb_100_25_c53
m_CET_L17_r = m_linear_worb_100_25_c53_r
palette['CET_L17'] = b_linear_worb_100_25_c53
cm['CET_L17'] = m_linear_worb_100_25_c53
cm['CET_L17_r'] = m_linear_worb_100_25_c53_r
register_cmap('cet_CET_L17',m_linear_worb_100_25_c53)
register_cmap('cet_CET_L17_r',m_linear_worb_100_25_c53_r)




linear_kbgyw_5_98_c62 = [  # cmap_def
[0.066001, 0.066024, 0.066019],
[0.073531, 0.067157, 0.081516],
[0.080354, 0.068289, 0.094843],
[0.08632, 0.069502, 0.10722],
[0.091205, 0.070686, 0.11969],
[0.095166, 0.071969, 0.13223],
[0.098937, 0.073058, 0.14484],
[0.10264, 0.073887, 0.15744],
[0.10625, 0.074435, 0.17015],
[0.1097, 0.074836, 0.18285],
[0.11289, 0.075212, 0.1956],
[0.11587, 0.075547, 0.20839],
[0.11864, 0.075871, 0.22116],
[0.12114, 0.076152, 0.23396],
[0.12347, 0.07644, 0.24672],
[0.12565, 0.076696, 0.25953],
[0.12755, 0.076951, 0.27226],
[0.12927, 0.077199, 0.28502],
[0.13082, 0.077434, 0.29777],
[0.13212, 0.077693, 0.31047],
[0.13324, 0.077937, 0.32315],
[0.13417, 0.078217, 0.33575],
[0.13489, 0.078507, 0.34833],
[0.13539, 0.078813, 0.36086],
[0.1357, 0.079175, 0.3733],
[0.13582, 0.079553, 0.3857],
[0.13576, 0.079992, 0.39802],
[0.13552, 0.080484, 0.41025],
[0.1351, 0.081016, 0.42242],
[0.13446, 0.081668, 0.43445],
[0.13361, 0.082336, 0.44639],
[0.13258, 0.083014, 0.45826],
[0.13136, 0.083921, 0.46995],
[0.12997, 0.084835, 0.48154],
[0.12833, 0.085789, 0.49302],
[0.12653, 0.087046, 0.50432],
[0.12453, 0.088286, 0.51545],
[0.12232, 0.089654, 0.5265],
[0.11998, 0.091206, 0.5373],
[0.11744, 0.092805, 0.54793],
[0.11468, 0.094674, 0.55839],
[0.11181, 0.096524, 0.56866],
[0.10868, 0.098699, 0.57867],
[0.10542, 0.10095, 0.58847],
[0.10197, 0.10342, 0.59806],
[0.098375, 0.106, 0.60741],
[0.094641, 0.10876, 0.61644],
[0.090708, 0.11175, 0.6252],
[0.086641, 0.11483, 0.63368],
[0.082459, 0.11819, 0.64187],
[0.078079, 0.12163, 0.64973],
[0.073814, 0.12537, 0.6572],
[0.069419, 0.12924, 0.66429],
[0.064888, 0.13334, 0.67101],
[0.060456, 0.13763, 0.67733],
[0.056065, 0.14207, 0.68321],
[0.051735, 0.14671, 0.68863],
[0.047663, 0.15158, 0.69354],
[0.043895, 0.15667, 0.69793],
[0.040589, 0.16195, 0.70175],
[0.03776, 0.16742, 0.70497],
[0.035653, 0.17312, 0.70753],
[0.033912, 0.17901, 0.7095],
[0.032749, 0.18494, 0.71116],
[0.031656, 0.1908, 0.71269],
[0.030664, 0.19667, 0.71411],
[0.029761, 0.20244, 0.71541],
[0.028951, 0.20823, 0.71659],
[0.028232, 0.21398, 0.71764],
[0.027604, 0.2197, 0.71856],
[0.027065, 0.22539, 0.71936],
[0.026613, 0.23109, 0.72001],
[0.026245, 0.23676, 0.72053],
[0.025983, 0.2424, 0.72089],
[0.025814, 0.24807, 0.72111],
[0.025721, 0.2537, 0.72117],
[0.025696, 0.25934, 0.72108],
[0.025732, 0.26495, 0.72082],
[0.025825, 0.2706, 0.7204],
[0.026001, 0.27623, 0.7198],
[0.02623, 0.28186, 0.71901],
[0.026481, 0.28748, 0.71804],
[0.026742, 0.29313, 0.71688],
[0.027045, 0.29879, 0.71551],
[0.027353, 0.30445, 0.71392],
[0.027623, 0.31013, 0.71213],
[0.027866, 0.31582, 0.71011],
[0.028066, 0.3215, 0.70785],
[0.028168, 0.32721, 0.70535],
[0.028179, 0.33297, 0.7026],
[0.028061, 0.33871, 0.69957],
[0.027763, 0.34447, 0.69628],
[0.02728, 0.35028, 0.69268],
[0.026539, 0.3561, 0.68879],
[0.025532, 0.36194, 0.68457],
[0.024173, 0.3678, 0.68002],
[0.022675, 0.37371, 0.67511],
[0.022239, 0.37957, 0.66989],
[0.024007, 0.38536, 0.66437],
[0.0278, 0.39108, 0.65859],
[0.033577, 0.39671, 0.65255],
[0.041196, 0.40229, 0.64627],
[0.049455, 0.40782, 0.63975],
[0.057803, 0.4133, 0.63301],
[0.066263, 0.41872, 0.62605],
[0.074632, 0.4241, 0.61889],
[0.082773, 0.42945, 0.61152],
[0.090615, 0.43476, 0.60397],
[0.098247, 0.44004, 0.59622],
[0.10554, 0.44529, 0.58829],
[0.11254, 0.45051, 0.5802],
[0.11917, 0.45572, 0.57193],
[0.12556, 0.46089, 0.56349],
[0.13162, 0.46605, 0.55489],
[0.13737, 0.47121, 0.54613],
[0.14277, 0.47633, 0.53722],
[0.14788, 0.48145, 0.52815],
[0.15268, 0.48655, 0.51893],
[0.15721, 0.49165, 0.50954],
[0.16147, 0.49672, 0.50001],
[0.16533, 0.50181, 0.49034],
[0.16899, 0.50687, 0.48051],
[0.17228, 0.51193, 0.47054],
[0.17532, 0.51699, 0.46039],
[0.17808, 0.52204, 0.45011],
[0.18049, 0.52709, 0.43966],
[0.18264, 0.53214, 0.42904],
[0.18452, 0.53718, 0.41824],
[0.1861, 0.54222, 0.40728],
[0.18737, 0.54725, 0.39614],
[0.18835, 0.55229, 0.38481],
[0.18921, 0.5573, 0.3735],
[0.19029, 0.56224, 0.36248],
[0.19168, 0.56709, 0.35179],
[0.1934, 0.57187, 0.34145],
[0.19542, 0.57659, 0.33138],
[0.19779, 0.58123, 0.32162],
[0.20048, 0.58583, 0.31211],
[0.20356, 0.59035, 0.30285],
[0.20697, 0.59482, 0.29384],
[0.21073, 0.59924, 0.28508],
[0.21484, 0.6036, 0.27654],
[0.21928, 0.60791, 0.26814],
[0.224, 0.61216, 0.25996],
[0.22904, 0.61638, 0.25192],
[0.23435, 0.62055, 0.244],
[0.23991, 0.62468, 0.23627],
[0.24581, 0.62875, 0.2287],
[0.25199, 0.63279, 0.22126],
[0.25837, 0.63678, 0.21392],
[0.26496, 0.64073, 0.2067],
[0.27176, 0.64465, 0.1995],
[0.27883, 0.64852, 0.19249],
[0.28609, 0.65235, 0.18561],
[0.29353, 0.65616, 0.17874],
[0.30111, 0.65991, 0.17192],
[0.30888, 0.66364, 0.16516],
[0.31685, 0.66732, 0.1586],
[0.32493, 0.67096, 0.15199],
[0.33316, 0.67457, 0.14547],
[0.3415, 0.67815, 0.13891],
[0.35004, 0.68168, 0.13253],
[0.35864, 0.68519, 0.12613],
[0.36736, 0.68866, 0.11969],
[0.37623, 0.69209, 0.1134],
[0.38519, 0.69549, 0.10711],
[0.39423, 0.69885, 0.10074],
[0.40339, 0.70217, 0.09449],
[0.41267, 0.70546, 0.088268],
[0.42198, 0.70873, 0.082038],
[0.4314, 0.71195, 0.075694],
[0.44093, 0.71513, 0.069656],
[0.45052, 0.71827, 0.063424],
[0.46016, 0.72139, 0.057182],
[0.46995, 0.72447, 0.05115],
[0.47975, 0.72751, 0.045105],
[0.48961, 0.73052, 0.039074],
[0.49959, 0.73349, 0.033395],
[0.50961, 0.73642, 0.028381],
[0.5197, 0.73933, 0.02399],
[0.52987, 0.74218, 0.020282],
[0.54007, 0.74501, 0.017058],
[0.55036, 0.7478, 0.014496],
[0.56069, 0.75056, 0.012516],
[0.57107, 0.75326, 0.010898],
[0.5815, 0.75595, 0.010052],
[0.59191, 0.75861, 0.0095873],
[0.60224, 0.76126, 0.0093949],
[0.61246, 0.76391, 0.0094303],
[0.62261, 0.76656, 0.0096325],
[0.63265, 0.76922, 0.010052],
[0.64259, 0.77188, 0.010748],
[0.65245, 0.77453, 0.011847],
[0.66223, 0.7772, 0.012989],
[0.6719, 0.77986, 0.014429],
[0.68149, 0.78252, 0.016102],
[0.691, 0.7852, 0.018089],
[0.70042, 0.78787, 0.020399],
[0.70977, 0.79056, 0.02298],
[0.71902, 0.79325, 0.025904],
[0.72818, 0.79595, 0.029244],
[0.73727, 0.79866, 0.032918],
[0.74627, 0.80138, 0.037138],
[0.75519, 0.80411, 0.041627],
[0.764, 0.80686, 0.046159],
[0.77274, 0.80961, 0.05093],
[0.78139, 0.81238, 0.055784],
[0.78995, 0.81517, 0.060722],
[0.7984, 0.81797, 0.065847],
[0.80676, 0.8208, 0.071087],
[0.81501, 0.82365, 0.076312],
[0.82317, 0.82652, 0.081795],
[0.83123, 0.82941, 0.087295],
[0.83917, 0.83232, 0.092849],
[0.84698, 0.83527, 0.098673],
[0.85467, 0.83825, 0.10453],
[0.86225, 0.84126, 0.11062],
[0.86967, 0.8443, 0.11675],
[0.87697, 0.84738, 0.12302],
[0.88411, 0.85051, 0.12951],
[0.8911, 0.85368, 0.1361],
[0.89791, 0.8569, 0.14293],
[0.90454, 0.86018, 0.14993],
[0.91098, 0.86351, 0.15708],
[0.9172, 0.8669, 0.1645],
[0.9232, 0.87036, 0.17211],
[0.92894, 0.87391, 0.17999],
[0.9344, 0.87753, 0.18822],
[0.93955, 0.88126, 0.19673],
[0.94437, 0.88508, 0.20555],
[0.94882, 0.88902, 0.21474],
[0.9528, 0.89309, 0.22442],
[0.95631, 0.8973, 0.23461],
[0.95927, 0.90169, 0.24539],
[0.9619, 0.90615, 0.25733],
[0.96461, 0.91054, 0.27111],
[0.9674, 0.91485, 0.2865],
[0.97027, 0.91907, 0.30377],
[0.97321, 0.92321, 0.32267],
[0.9762, 0.92726, 0.34315],
[0.97923, 0.93121, 0.3654],
[0.98227, 0.93507, 0.38934],
[0.98527, 0.93883, 0.41497],
[0.98821, 0.94249, 0.44241],
[0.99103, 0.94604, 0.47174],
[0.99368, 0.94948, 0.50316],
[0.99606, 0.9528, 0.53675],
[0.9981, 0.95601, 0.57266],
[0.99966, 0.9591, 0.61109],
[1, 0.96206, 0.65228],
[1, 0.9649, 0.69659],
[0.99974, 0.96762, 0.74429],
[0.99739, 0.97022, 0.7957],
[0.99327, 0.97271, 0.85118],
[0.98675, 0.97508, 0.91169],
[0.97718, 0.97737, 0.97732],
]
b_linear_kbgyw_5_98_c62 = bokeh_palette('linear_kbgyw_5_98_c62',linear_kbgyw_5_98_c62)
m_linear_kbgyw_5_98_c62 = mpl_cm('linear_kbgyw_5_98_c62',linear_kbgyw_5_98_c62)
m_linear_kbgyw_5_98_c62_r = mpl_cm('linear_kbgyw_5_98_c62_r',list(reversed(linear_kbgyw_5_98_c62)))
CET_L16 = b_linear_kbgyw_5_98_c62
m_CET_L16 = m_linear_kbgyw_5_98_c62
m_CET_L16_r = m_linear_kbgyw_5_98_c62_r
palette['CET_L16'] = b_linear_kbgyw_5_98_c62
cm['CET_L16'] = m_linear_kbgyw_5_98_c62
cm['CET_L16_r'] = m_linear_kbgyw_5_98_c62_r
register_cmap('cet_CET_L16',m_linear_kbgyw_5_98_c62)
register_cmap('cet_CET_L16_r',m_linear_kbgyw_5_98_c62_r)




linear_kbc_5_95_c73 = [  # cmap_def
[0, 0.0024382, 0.30687],
[0, 0.0044692, 0.31437],
[0, 0.0062324, 0.32191],
[0, 0.0077305, 0.32951],
[0, 0.008958, 0.33713],
[0, 0.0098968, 0.34479],
[0, 0.010555, 0.35249],
[0, 0.011009, 0.36023],
[0, 0.01114, 0.36802],
[0.0018829, 0.010944, 0.37585],
[0.0043414, 0.010697, 0.38371],
[0.0068536, 0.010463, 0.39161],
[0.0094655, 0.010231, 0.39952],
[0.01235, 0.0099983, 0.4075],
[0.015052, 0.0097591, 0.4155],
[0.017851, 0.0094983, 0.42354],
[0.020695, 0.0092222, 0.43163],
[0.02357, 0.0089375, 0.43973],
[0.026496, 0.0086445, 0.44787],
[0.029448, 0.0083501, 0.45605],
[0.032396, 0.0080591, 0.46429],
[0.03558, 0.0077658, 0.47253],
[0.038524, 0.0074701, 0.48082],
[0.041458, 0.0071848, 0.48915],
[0.044141, 0.0069073, 0.49753],
[0.04679, 0.0066377, 0.50593],
[0.04932, 0.0063792, 0.51438],
[0.051527, 0.0061442, 0.52287],
[0.053656, 0.0059329, 0.53139],
[0.055683, 0.0057462, 0.53995],
[0.057455, 0.0055887, 0.54857],
[0.059195, 0.0054661, 0.55722],
[0.060593, 0.0053876, 0.5659],
[0.061766, 0.0053629, 0.57464],
[0.062841, 0.0053961, 0.58343],
[0.063532, 0.0054935, 0.59227],
[0.063966, 0.0056647, 0.60114],
[0.0641, 0.0059201, 0.61007],
[0.063882, 0.0062716, 0.61907],
[0.063281, 0.0067326, 0.6281],
[0.062148, 0.0073175, 0.63719],
[0.060493, 0.0080408, 0.64634],
[0.058092, 0.008929, 0.65557],
[0.055233, 0.0099623, 0.66482],
[0.052105, 0.011266, 0.67402],
[0.049146, 0.012683, 0.68314],
[0.046014, 0.014144, 0.69216],
[0.04309, 0.015741, 0.70108],
[0.040435, 0.017485, 0.70992],
[0.037863, 0.019379, 0.71866],
[0.035616, 0.021429, 0.72731],
[0.033428, 0.02364, 0.73587],
[0.031841, 0.026021, 0.74433],
[0.030635, 0.028579, 0.75268],
[0.029823, 0.031319, 0.76095],
[0.029422, 0.034249, 0.7691],
[0.029449, 0.037557, 0.77717],
[0.029926, 0.040864, 0.78513],
[0.030874, 0.044198, 0.793],
[0.032312, 0.047578, 0.80076],
[0.034281, 0.051036, 0.80842],
[0.036977, 0.054512, 0.81597],
[0.040055, 0.058003, 0.82341],
[0.043483, 0.061561, 0.83076],
[0.047434, 0.065191, 0.83799],
[0.051627, 0.068868, 0.84513],
[0.056188, 0.072421, 0.85216],
[0.060871, 0.076069, 0.85907],
[0.065858, 0.079772, 0.86589],
[0.071059, 0.083524, 0.87259],
[0.07632, 0.087319, 0.87918],
[0.081824, 0.091117, 0.88564],
[0.087161, 0.095093, 0.89195],
[0.092292, 0.099225, 0.89805],
[0.097256, 0.10365, 0.90395],
[0.10208, 0.10819, 0.90965],
[0.10678, 0.11304, 0.91514],
[0.11126, 0.11798, 0.92041],
[0.11561, 0.12306, 0.92547],
[0.11981, 0.12837, 0.93032],
[0.12389, 0.13378, 0.93496],
[0.12789, 0.13928, 0.9394],
[0.13172, 0.14498, 0.94362],
[0.13541, 0.15071, 0.94764],
[0.13898, 0.15659, 0.95143],
[0.14245, 0.16255, 0.95501],
[0.14579, 0.16859, 0.95837],
[0.14895, 0.17474, 0.96151],
[0.15202, 0.18091, 0.96444],
[0.15497, 0.18721, 0.96717],
[0.15778, 0.19355, 0.96967],
[0.16044, 0.19992, 0.97196],
[0.16301, 0.20643, 0.97403],
[0.16537, 0.21292, 0.97587],
[0.16765, 0.21951, 0.9775],
[0.16978, 0.22614, 0.9789],
[0.17171, 0.23278, 0.9801],
[0.17352, 0.2395, 0.98107],
[0.17518, 0.24623, 0.98183],
[0.17664, 0.25301, 0.98238],
[0.17803, 0.25977, 0.98279],
[0.17924, 0.26645, 0.98317],
[0.18037, 0.27304, 0.98356],
[0.18141, 0.27957, 0.98394],
[0.18236, 0.28598, 0.98432],
[0.18323, 0.29237, 0.98469],
[0.18401, 0.29868, 0.98507],
[0.1847, 0.30493, 0.98544],
[0.1853, 0.31112, 0.9858],
[0.1858, 0.31725, 0.98617],
[0.18619, 0.32334, 0.98654],
[0.18649, 0.32939, 0.9869],
[0.18671, 0.33539, 0.98726],
[0.18684, 0.34133, 0.98762],
[0.18688, 0.34725, 0.98797],
[0.18682, 0.35312, 0.98832],
[0.18667, 0.35896, 0.98868],
[0.18643, 0.36477, 0.98903],
[0.1861, 0.37054, 0.98937],
[0.18568, 0.37631, 0.98972],
[0.18513, 0.38201, 0.99006],
[0.18448, 0.38771, 0.9904],
[0.18373, 0.39338, 0.99074],
[0.18287, 0.39902, 0.99107],
[0.18192, 0.40464, 0.9914],
[0.18086, 0.41025, 0.99173],
[0.17969, 0.41582, 0.99206],
[0.17843, 0.42137, 0.99239],
[0.17708, 0.42691, 0.99271],
[0.17576, 0.43242, 0.99301],
[0.17462, 0.43788, 0.9933],
[0.17355, 0.44331, 0.99356],
[0.17263, 0.44871, 0.99381],
[0.17185, 0.45406, 0.99403],
[0.17122, 0.4594, 0.99423],
[0.17072, 0.4647, 0.99442],
[0.17037, 0.46998, 0.99459],
[0.17015, 0.47522, 0.99473],
[0.17008, 0.48043, 0.99486],
[0.17016, 0.48562, 0.99497],
[0.17038, 0.4908, 0.99505],
[0.17077, 0.49593, 0.99512],
[0.17131, 0.50106, 0.99517],
[0.17201, 0.50615, 0.99519],
[0.17286, 0.51122, 0.9952],
[0.17386, 0.51627, 0.99519],
[0.17503, 0.52131, 0.99516],
[0.17629, 0.52633, 0.9951],
[0.17778, 0.53132, 0.99503],
[0.17936, 0.5363, 0.99494],
[0.1811, 0.54125, 0.99482],
[0.18299, 0.5462, 0.99469],
[0.18504, 0.55111, 0.99454],
[0.1872, 0.55601, 0.99436],
[0.18948, 0.5609, 0.99417],
[0.19189, 0.56577, 0.99396],
[0.19443, 0.57063, 0.99373],
[0.19686, 0.5755, 0.99349],
[0.19906, 0.58036, 0.99326],
[0.2011, 0.58524, 0.99303],
[0.20297, 0.59011, 0.99281],
[0.20464, 0.59499, 0.9926],
[0.20612, 0.59989, 0.99239],
[0.2074, 0.60479, 0.99218],
[0.2085, 0.60969, 0.99198],
[0.20942, 0.6146, 0.99179],
[0.21016, 0.61953, 0.9916],
[0.21072, 0.62445, 0.99142],
[0.21111, 0.62938, 0.99124],
[0.21132, 0.63432, 0.99107],
[0.21135, 0.63926, 0.9909],
[0.2112, 0.64421, 0.99074],
[0.21086, 0.64918, 0.99058],
[0.21033, 0.65414, 0.99043],
[0.20961, 0.65911, 0.99029],
[0.2087, 0.66409, 0.99015],
[0.2076, 0.66908, 0.99002],
[0.20631, 0.67406, 0.98989],
[0.20478, 0.67906, 0.98977],
[0.20305, 0.68406, 0.98965],
[0.20108, 0.68908, 0.98954],
[0.1989, 0.69409, 0.98943],
[0.19651, 0.69911, 0.98933],
[0.19382, 0.70414, 0.98924],
[0.19088, 0.70918, 0.98914],
[0.18788, 0.71421, 0.98905],
[0.18488, 0.71923, 0.98895],
[0.18198, 0.72423, 0.98883],
[0.17926, 0.72923, 0.98871],
[0.17665, 0.73421, 0.98857],
[0.17423, 0.73918, 0.98842],
[0.17193, 0.74412, 0.98825],
[0.16983, 0.74906, 0.98808],
[0.16785, 0.75399, 0.98789],
[0.16608, 0.7589, 0.9877],
[0.16452, 0.76381, 0.98749],
[0.16316, 0.76869, 0.98727],
[0.16202, 0.77357, 0.98703],
[0.1611, 0.77844, 0.98679],
[0.16037, 0.7833, 0.98653],
[0.15991, 0.78814, 0.98626],
[0.1597, 0.79298, 0.98597],
[0.15972, 0.7978, 0.98568],
[0.15999, 0.80261, 0.98537],
[0.16053, 0.80742, 0.98505],
[0.16137, 0.81221, 0.98472],
[0.16241, 0.81699, 0.98438],
[0.16373, 0.82176, 0.98402],
[0.16523, 0.82652, 0.98365],
[0.16707, 0.83127, 0.98327],
[0.1691, 0.83601, 0.98287],
[0.17134, 0.84074, 0.98247],
[0.17387, 0.84546, 0.98205],
[0.17729, 0.85015, 0.98162],
[0.18258, 0.85474, 0.98119],
[0.19003, 0.85923, 0.98076],
[0.19916, 0.86363, 0.98033],
[0.20965, 0.86794, 0.97991],
[0.22117, 0.87217, 0.97948],
[0.23338, 0.87632, 0.97905],
[0.24622, 0.88041, 0.97862],
[0.25953, 0.88443, 0.9782],
[0.27309, 0.8884, 0.97777],
[0.28681, 0.89231, 0.97734],
[0.30063, 0.89617, 0.97691],
[0.31448, 0.89998, 0.97648],
[0.32832, 0.90375, 0.97605],
[0.34224, 0.90747, 0.97562],
[0.35613, 0.91115, 0.97518],
[0.36991, 0.91479, 0.97475],
[0.38361, 0.9184, 0.97431],
[0.39728, 0.92197, 0.97387],
[0.4109, 0.9255, 0.97343],
[0.42439, 0.92899, 0.973],
[0.43779, 0.93247, 0.97255],
[0.45116, 0.9359, 0.97211],
[0.46443, 0.93931, 0.97166],
[0.47755, 0.94269, 0.97121],
[0.49069, 0.94603, 0.97076],
[0.5037, 0.94935, 0.97031],
[0.51659, 0.95265, 0.96986],
[0.52949, 0.95592, 0.9694],
[0.54227, 0.95916, 0.96894],
[0.55496, 0.96238, 0.96849],
[0.56763, 0.96557, 0.96803],
[0.58018, 0.96874, 0.96756],
[0.59271, 0.97189, 0.9671],
[0.60515, 0.97501, 0.96663],
[0.61751, 0.97812, 0.96616],
[0.62987, 0.98119, 0.96569],
[0.64213, 0.98425, 0.96521],
[0.65436, 0.98729, 0.96474],
[0.66654, 0.9903, 0.96426],
[0.67864, 0.9933, 0.96378],
[0.69072, 0.99626, 0.9633],
[0.70271, 0.99922, 0.96282],
]
b_linear_kbc_5_95_c73 = bokeh_palette('linear_kbc_5_95_c73',linear_kbc_5_95_c73)
m_linear_kbc_5_95_c73 = mpl_cm('linear_kbc_5_95_c73',linear_kbc_5_95_c73)
m_linear_kbc_5_95_c73_r = mpl_cm('linear_kbc_5_95_c73_r',list(reversed(linear_kbc_5_95_c73)))
kbc = b_linear_kbc_5_95_c73
m_kbc = m_linear_kbc_5_95_c73
m_kbc_r = m_linear_kbc_5_95_c73_r
palette['kbc'] = b_linear_kbc_5_95_c73
palette_n['kbc'] = b_linear_kbc_5_95_c73
cm['kbc'] = m_linear_kbc_5_95_c73
cm['kbc_r'] = m_linear_kbc_5_95_c73_r
cm_n['kbc'] = mpl_cm('kbc',linear_kbc_5_95_c73)
cm_n['kbc_r'] = mpl_cm('kbc_r',list(reversed(linear_kbc_5_95_c73)))
linear_blue_5_95_c73 = b_linear_kbc_5_95_c73
m_linear_blue_5_95_c73 = m_linear_kbc_5_95_c73
m_linear_blue_5_95_c73_r = m_linear_kbc_5_95_c73_r
palette['linear_blue_5_95_c73'] = b_linear_kbc_5_95_c73
palette_n['linear_blue_5_95_c73'] = b_linear_kbc_5_95_c73
cm['linear_blue_5_95_c73'] = m_linear_kbc_5_95_c73
cm['linear_blue_5_95_c73_r'] = m_linear_kbc_5_95_c73_r
cm_n['linear_blue_5_95_c73'] = mpl_cm('linear_blue_5_95_c73',linear_kbc_5_95_c73)
cm_n['linear_blue_5_95_c73_r'] = mpl_cm('linear_blue_5_95_c73_r',list(reversed(linear_kbc_5_95_c73)))
CET_L6 = b_linear_kbc_5_95_c73
m_CET_L6 = m_linear_kbc_5_95_c73
m_CET_L6_r = m_linear_kbc_5_95_c73_r
palette['CET_L6'] = b_linear_kbc_5_95_c73
cm['CET_L6'] = m_linear_kbc_5_95_c73
cm['CET_L6_r'] = m_linear_kbc_5_95_c73_r
register_cmap('cet_CET_L6',m_linear_kbc_5_95_c73)
register_cmap('cet_CET_L6_r',m_linear_kbc_5_95_c73_r)




cyclic_tritanopic_wrwc_70_100_c20 = [  # cmap_def
[0.98457, 0.98115, 0.98169],
[0.98868, 0.9791, 0.97837],
[0.99201, 0.97625, 0.97432],
[0.99463, 0.97265, 0.96956],
[0.99662, 0.96839, 0.96417],
[0.99809, 0.96355, 0.95825],
[0.99913, 0.95825, 0.95187],
[0.99983, 0.95257, 0.94516],
[1, 0.94662, 0.93817],
[1, 0.94046, 0.93099],
[1, 0.93415, 0.92369],
[1, 0.92775, 0.91629],
[1, 0.9213, 0.90884],
[1, 0.9148, 0.90136],
[1, 0.90828, 0.89388],
[0.99982, 0.90175, 0.88639],
[0.99952, 0.89521, 0.87889],
[0.99918, 0.88868, 0.87142],
[0.99881, 0.88215, 0.86396],
[0.9984, 0.87562, 0.8565],
[0.99796, 0.86909, 0.84906],
[0.99748, 0.86257, 0.84163],
[0.99697, 0.85604, 0.83421],
[0.99642, 0.84952, 0.8268],
[0.99584, 0.84299, 0.8194],
[0.99523, 0.83647, 0.81202],
[0.99459, 0.82995, 0.80465],
[0.99391, 0.82343, 0.79729],
[0.9932, 0.81692, 0.78994],
[0.99245, 0.8104, 0.7826],
[0.99167, 0.80388, 0.77527],
[0.99086, 0.79737, 0.76796],
[0.99002, 0.79085, 0.76066],
[0.98914, 0.78434, 0.75337],
[0.98823, 0.77783, 0.74609],
[0.9873, 0.77131, 0.73883],
[0.98633, 0.7648, 0.73158],
[0.98533, 0.75829, 0.72434],
[0.9843, 0.75177, 0.71711],
[0.98323, 0.74525, 0.70989],
[0.98214, 0.73874, 0.70269],
[0.98101, 0.73222, 0.6955],
[0.97986, 0.72571, 0.68832],
[0.97867, 0.71918, 0.68115],
[0.97746, 0.71266, 0.674],
[0.97621, 0.70614, 0.66685],
[0.97494, 0.69961, 0.65973],
[0.97364, 0.69309, 0.65261],
[0.97231, 0.68656, 0.64551],
[0.97094, 0.68002, 0.63842],
[0.96955, 0.6735, 0.63135],
[0.96814, 0.66698, 0.62431],
[0.9667, 0.66048, 0.61731],
[0.96524, 0.65404, 0.61037],
[0.96378, 0.64766, 0.60354],
[0.96231, 0.64139, 0.59684],
[0.96087, 0.6353, 0.59033],
[0.95946, 0.62946, 0.58411],
[0.95811, 0.62398, 0.57828],
[0.95686, 0.61895, 0.57295],
[0.95575, 0.6145, 0.56825],
[0.9548, 0.61078, 0.56432],
[0.95407, 0.6079, 0.56128],
[0.95357, 0.60596, 0.55924],
[0.95334, 0.60505, 0.55828],
[0.95338, 0.60519, 0.55843],
[0.95368, 0.60639, 0.5597],
[0.95425, 0.60861, 0.56203],
[0.95504, 0.61172, 0.56531],
[0.95604, 0.61565, 0.56946],
[0.95719, 0.62027, 0.57434],
[0.95847, 0.62543, 0.57983],
[0.95983, 0.63103, 0.58577],
[0.96125, 0.63694, 0.59207],
[0.96271, 0.64307, 0.59862],
[0.96417, 0.64937, 0.60536],
[0.96563, 0.65577, 0.61223],
[0.96709, 0.66223, 0.61919],
[0.96852, 0.66873, 0.62619],
[0.96993, 0.67525, 0.63325],
[0.97131, 0.68178, 0.64033],
[0.97267, 0.68831, 0.64742],
[0.97399, 0.69484, 0.65452],
[0.97528, 0.70136, 0.66164],
[0.97655, 0.70789, 0.66877],
[0.97779, 0.71441, 0.67592],
[0.97899, 0.72093, 0.68307],
[0.98017, 0.72745, 0.69024],
[0.98132, 0.73397, 0.69742],
[0.98244, 0.74048, 0.70462],
[0.98352, 0.747, 0.71183],
[0.98458, 0.75351, 0.71904],
[0.9856, 0.76003, 0.72628],
[0.98659, 0.76654, 0.73352],
[0.98755, 0.77305, 0.74077],
[0.98848, 0.77957, 0.74805],
[0.98938, 0.78609, 0.75533],
[0.99025, 0.7926, 0.76261],
[0.99108, 0.79912, 0.76992],
[0.99188, 0.80563, 0.77724],
[0.99266, 0.81215, 0.78457],
[0.99339, 0.81866, 0.79191],
[0.99409, 0.82518, 0.79926],
[0.99476, 0.8317, 0.80663],
[0.9954, 0.83822, 0.814],
[0.996, 0.84474, 0.82139],
[0.99657, 0.85126, 0.82879],
[0.99711, 0.85778, 0.8362],
[0.99761, 0.86431, 0.84362],
[0.99808, 0.87084, 0.85105],
[0.99852, 0.87737, 0.8585],
[0.99891, 0.8839, 0.86595],
[0.99928, 0.89043, 0.87342],
[0.99959, 0.89697, 0.88091],
[0.99986, 0.90349, 0.8884],
[1, 0.91001, 0.89588],
[1, 0.9165, 0.90336],
[1, 0.92295, 0.91082],
[1, 0.92932, 0.91823],
[0.99964, 0.93559, 0.92556],
[0.99899, 0.9417, 0.93277],
[0.99797, 0.94758, 0.93981],
[0.9965, 0.95315, 0.94658],
[0.99447, 0.95831, 0.95303],
[0.99178, 0.96297, 0.95903],
[0.98834, 0.96702, 0.96449],
[0.98409, 0.97036, 0.96934],
[0.97898, 0.97293, 0.97348],
[0.97301, 0.97466, 0.97685],
[0.96619, 0.97556, 0.97944],
[0.9586, 0.97562, 0.98125],
[0.95031, 0.97491, 0.98231],
[0.94142, 0.97349, 0.98269],
[0.93202, 0.97146, 0.98248],
[0.92224, 0.96892, 0.98176],
[0.91213, 0.96597, 0.98064],
[0.9018, 0.96271, 0.9792],
[0.89129, 0.95921, 0.97753],
[0.88066, 0.95555, 0.97568],
[0.86993, 0.95177, 0.97373],
[0.85913, 0.94793, 0.9717],
[0.84827, 0.94403, 0.96961],
[0.83736, 0.94011, 0.96751],
[0.82639, 0.93617, 0.96537],
[0.81539, 0.93222, 0.96323],
[0.80434, 0.92826, 0.9611],
[0.79324, 0.92431, 0.95896],
[0.78209, 0.92036, 0.95681],
[0.77088, 0.9164, 0.95468],
[0.75961, 0.91245, 0.95253],
[0.74829, 0.90849, 0.95039],
[0.7369, 0.90453, 0.94825],
[0.72544, 0.90057, 0.94611],
[0.71391, 0.89661, 0.94397],
[0.7023, 0.89264, 0.94182],
[0.69061, 0.88868, 0.93968],
[0.67883, 0.88471, 0.93754],
[0.66696, 0.88074, 0.93539],
[0.655, 0.87677, 0.93324],
[0.64293, 0.8728, 0.9311],
[0.63076, 0.86883, 0.92895],
[0.61846, 0.86485, 0.92681],
[0.60602, 0.86088, 0.92466],
[0.59347, 0.8569, 0.92252],
[0.58076, 0.85293, 0.92037],
[0.5679, 0.84895, 0.91822],
[0.55486, 0.84496, 0.91607],
[0.54164, 0.84098, 0.91393],
[0.52823, 0.837, 0.91178],
[0.5146, 0.83301, 0.90963],
[0.50072, 0.82903, 0.90748],
[0.48659, 0.82503, 0.90534],
[0.47218, 0.82105, 0.90318],
[0.45745, 0.81705, 0.90103],
[0.44238, 0.81306, 0.89888],
[0.4269, 0.80907, 0.89673],
[0.411, 0.80507, 0.89458],
[0.39462, 0.80108, 0.89243],
[0.37775, 0.7971, 0.89029],
[0.36032, 0.79315, 0.88816],
[0.34235, 0.78923, 0.88605],
[0.32379, 0.78537, 0.88397],
[0.30471, 0.7816, 0.88194],
[0.2852, 0.77795, 0.87998],
[0.26548, 0.77449, 0.87813],
[0.24583, 0.77129, 0.8764],
[0.22678, 0.76839, 0.87484],
[0.20899, 0.7659, 0.8735],
[0.19344, 0.76387, 0.87242],
[0.18121, 0.76239, 0.87163],
[0.17355, 0.76151, 0.87115],
[0.1713, 0.76126, 0.87101],
[0.17479, 0.76165, 0.87122],
[0.18351, 0.76266, 0.87177],
[0.19657, 0.76427, 0.87263],
[0.21268, 0.7664, 0.87377],
[0.23081, 0.76898, 0.87516],
[0.25003, 0.77195, 0.87676],
[0.26976, 0.77522, 0.87852],
[0.28945, 0.77872, 0.8804],
[0.30888, 0.7824, 0.88237],
[0.32783, 0.7862, 0.88442],
[0.34626, 0.79007, 0.8865],
[0.36413, 0.794, 0.88862],
[0.38143, 0.79796, 0.89075],
[0.3982, 0.80194, 0.89289],
[0.41446, 0.80593, 0.89504],
[0.43027, 0.80993, 0.8972],
[0.44565, 0.81392, 0.89935],
[0.46065, 0.81791, 0.90149],
[0.47532, 0.82191, 0.90365],
[0.48966, 0.8259, 0.9058],
[0.50373, 0.82989, 0.90795],
[0.51756, 0.83387, 0.9101],
[0.53114, 0.83786, 0.91224],
[0.54451, 0.84184, 0.91439],
[0.55769, 0.84582, 0.91654],
[0.57068, 0.8498, 0.91868],
[0.58351, 0.85378, 0.92083],
[0.59619, 0.85776, 0.92298],
[0.60873, 0.86174, 0.92513],
[0.62112, 0.86571, 0.92727],
[0.63339, 0.86968, 0.92942],
[0.64554, 0.87366, 0.93156],
[0.6576, 0.87763, 0.93371],
[0.66954, 0.8816, 0.93586],
[0.68138, 0.88556, 0.938],
[0.69314, 0.88953, 0.94014],
[0.70481, 0.89349, 0.94229],
[0.7164, 0.89746, 0.94443],
[0.72791, 0.90142, 0.94657],
[0.73937, 0.90539, 0.94871],
[0.75074, 0.90934, 0.95086],
[0.76205, 0.9133, 0.953],
[0.7733, 0.91726, 0.95514],
[0.7845, 0.92121, 0.95728],
[0.79563, 0.92516, 0.95942],
[0.80673, 0.92912, 0.96155],
[0.81776, 0.93307, 0.96369],
[0.82876, 0.93702, 0.96583],
[0.83971, 0.94097, 0.96798],
[0.8506, 0.94492, 0.97011],
[0.86144, 0.94884, 0.97224],
[0.87222, 0.95275, 0.97434],
[0.88291, 0.95663, 0.97642],
[0.8935, 0.96044, 0.97844],
[0.90394, 0.96416, 0.9804],
[0.91419, 0.96776, 0.98223],
[0.92418, 0.97115, 0.9839],
[0.93383, 0.97428, 0.98533],
[0.94306, 0.97707, 0.98644],
[0.95177, 0.9794, 0.98716],
[0.95986, 0.98119, 0.98738],
[0.96725, 0.98233, 0.98702],
[0.97385, 0.98274, 0.98598],
[0.97963, 0.98236, 0.98422],
]
b_cyclic_tritanopic_wrwc_70_100_c20 = bokeh_palette('cyclic_tritanopic_wrwc_70_100_c20',cyclic_tritanopic_wrwc_70_100_c20)
m_cyclic_tritanopic_wrwc_70_100_c20 = mpl_cm('cyclic_tritanopic_wrwc_70_100_c20',cyclic_tritanopic_wrwc_70_100_c20)
m_cyclic_tritanopic_wrwc_70_100_c20_r = mpl_cm('cyclic_tritanopic_wrwc_70_100_c20_r',list(reversed(cyclic_tritanopic_wrwc_70_100_c20)))
CET_CBTC2 = b_cyclic_tritanopic_wrwc_70_100_c20
m_CET_CBTC2 = m_cyclic_tritanopic_wrwc_70_100_c20
m_CET_CBTC2_r = m_cyclic_tritanopic_wrwc_70_100_c20_r
palette['CET_CBTC2'] = b_cyclic_tritanopic_wrwc_70_100_c20
cm['CET_CBTC2'] = m_cyclic_tritanopic_wrwc_70_100_c20
cm['CET_CBTC2_r'] = m_cyclic_tritanopic_wrwc_70_100_c20_r
register_cmap('cet_CET_CBTC2',m_cyclic_tritanopic_wrwc_70_100_c20)
register_cmap('cet_CET_CBTC2_r',m_cyclic_tritanopic_wrwc_70_100_c20_r)




linear_kgy_5_95_c69 = [  # cmap_def
[0.0020082, 0.082527, 0.022201],
[0.004381, 0.08615, 0.022118],
[0.0067598, 0.089747, 0.022041],
[0.0091625, 0.093124, 0.021981],
[0.011768, 0.096433, 0.021933],
[0.014187, 0.099686, 0.021895],
[0.016629, 0.10283, 0.021877],
[0.019082, 0.10587, 0.021867],
[0.021556, 0.1088, 0.021877],
[0.024018, 0.11172, 0.021897],
[0.026293, 0.11459, 0.021902],
[0.028361, 0.1176, 0.0219],
[0.030209, 0.12061, 0.021879],
[0.031836, 0.12368, 0.021843],
[0.033248, 0.12681, 0.021797],
[0.03448, 0.13002, 0.021732],
[0.035611, 0.13318, 0.021656],
[0.036308, 0.13639, 0.021568],
[0.036803, 0.13965, 0.021464],
[0.037067, 0.14299, 0.021348],
[0.037096, 0.14631, 0.021222],
[0.036881, 0.14971, 0.021083],
[0.036566, 0.15305, 0.020931],
[0.036258, 0.15648, 0.020778],
[0.035956, 0.15982, 0.02062],
[0.035648, 0.16326, 0.020454],
[0.03531, 0.16666, 0.020281],
[0.034935, 0.17007, 0.02011],
[0.034549, 0.17347, 0.019938],
[0.034194, 0.17689, 0.019764],
[0.0339, 0.18031, 0.01959],
[0.033658, 0.18374, 0.019418],
[0.03346, 0.18719, 0.019251],
[0.033308, 0.1906, 0.019094],
[0.033202, 0.19407, 0.018947],
[0.033153, 0.19748, 0.018811],
[0.033171, 0.2009, 0.018691],
[0.033269, 0.20438, 0.018588],
[0.033456, 0.2078, 0.018509],
[0.033755, 0.21125, 0.018457],
[0.034229, 0.21467, 0.018437],
[0.03498, 0.21811, 0.018456],
[0.03581, 0.22154, 0.018521],
[0.036701, 0.22495, 0.018622],
[0.037696, 0.22842, 0.018735],
[0.038702, 0.23186, 0.018848],
[0.039729, 0.23531, 0.018957],
[0.040766, 0.23875, 0.019061],
[0.041844, 0.24222, 0.019159],
[0.042758, 0.24568, 0.019252],
[0.043802, 0.24918, 0.019339],
[0.044931, 0.25267, 0.019419],
[0.045873, 0.25616, 0.019493],
[0.047009, 0.25968, 0.019561],
[0.047989, 0.26319, 0.019623],
[0.049157, 0.2667, 0.019676],
[0.050138, 0.27025, 0.019721],
[0.051173, 0.27378, 0.019758],
[0.052215, 0.27732, 0.019785],
[0.053276, 0.28085, 0.019802],
[0.054355, 0.28442, 0.01981],
[0.055446, 0.28799, 0.01981],
[0.056569, 0.29158, 0.019809],
[0.057528, 0.29515, 0.019834],
[0.058624, 0.29874, 0.01989],
[0.059734, 0.30233, 0.019977],
[0.06068, 0.30596, 0.020098],
[0.061651, 0.30957, 0.020253],
[0.062756, 0.31318, 0.020445],
[0.063696, 0.31681, 0.020677],
[0.064656, 0.32045, 0.02095],
[0.065612, 0.32408, 0.021258],
[0.066576, 0.32773, 0.021584],
[0.067549, 0.33138, 0.021916],
[0.068538, 0.33506, 0.022251],
[0.06952, 0.33872, 0.022589],
[0.070402, 0.3424, 0.022929],
[0.071419, 0.34607, 0.023272],
[0.072319, 0.34977, 0.023618],
[0.073361, 0.35346, 0.023966],
[0.074283, 0.35718, 0.024318],
[0.075218, 0.36087, 0.024672],
[0.076165, 0.36459, 0.025028],
[0.077123, 0.36832, 0.025388],
[0.078091, 0.37204, 0.02575],
[0.079071, 0.37578, 0.026116],
[0.080061, 0.37952, 0.026484],
[0.081061, 0.38328, 0.026854],
[0.082077, 0.38702, 0.027228],
[0.082984, 0.39079, 0.027604],
[0.084013, 0.39456, 0.027983],
[0.084952, 0.39834, 0.028365],
[0.085878, 0.40211, 0.02875],
[0.086941, 0.40591, 0.029138],
[0.087908, 0.4097, 0.029528],
[0.088876, 0.4135, 0.029921],
[0.089858, 0.41731, 0.030317],
[0.090847, 0.42111, 0.030716],
[0.091852, 0.42494, 0.031118],
[0.092761, 0.42876, 0.031521],
[0.093773, 0.43259, 0.031927],
[0.094809, 0.43643, 0.032341],
[0.09575, 0.44027, 0.03276],
[0.096692, 0.44412, 0.033171],
[0.097753, 0.44796, 0.033567],
[0.098721, 0.45183, 0.033985],
[0.099698, 0.4557, 0.034483],
[0.10069, 0.45957, 0.035028],
[0.10168, 0.46344, 0.035514],
[0.10269, 0.46733, 0.035934],
[0.1037, 0.47122, 0.036349],
[0.10463, 0.47512, 0.036789],
[0.10568, 0.47902, 0.037237],
[0.10671, 0.48292, 0.037682],
[0.10766, 0.48683, 0.038128],
[0.10865, 0.49076, 0.038578],
[0.10971, 0.49468, 0.039032],
[0.11069, 0.49861, 0.039489],
[0.11169, 0.50254, 0.039948],
[0.1127, 0.50648, 0.040406],
[0.11369, 0.51042, 0.040871],
[0.11464, 0.51438, 0.041351],
[0.11569, 0.51834, 0.041826],
[0.1167, 0.52229, 0.042253],
[0.11769, 0.52627, 0.042639],
[0.11873, 0.53024, 0.043044],
[0.1197, 0.53421, 0.043511],
[0.1207, 0.53821, 0.044017],
[0.1217, 0.54219, 0.044525],
[0.12271, 0.54619, 0.044992],
[0.12373, 0.55018, 0.045398],
[0.12476, 0.55419, 0.045806],
[0.12581, 0.55819, 0.046288],
[0.12679, 0.56222, 0.046801],
[0.12783, 0.56624, 0.047247],
[0.12882, 0.57026, 0.047653],
[0.12988, 0.57429, 0.048118],
[0.13089, 0.57833, 0.048645],
[0.1319, 0.58237, 0.049135],
[0.13292, 0.58642, 0.049563],
[0.13395, 0.59047, 0.04998],
[0.13498, 0.59453, 0.050418],
[0.13595, 0.59859, 0.050866],
[0.13701, 0.60266, 0.051311],
[0.13806, 0.60673, 0.051757],
[0.13904, 0.61081, 0.052205],
[0.14007, 0.61488, 0.052658],
[0.14114, 0.61898, 0.053112],
[0.14214, 0.62307, 0.053569],
[0.14318, 0.62716, 0.054029],
[0.14421, 0.63127, 0.054491],
[0.14526, 0.63537, 0.054953],
[0.14626, 0.63949, 0.055418],
[0.1473, 0.6436, 0.055896],
[0.14831, 0.64772, 0.056378],
[0.14938, 0.65185, 0.056824],
[0.15041, 0.65599, 0.057227],
[0.15143, 0.66012, 0.057637],
[0.15246, 0.66426, 0.058099],
[0.15349, 0.66841, 0.058598],
[0.15454, 0.67256, 0.0591],
[0.1556, 0.67671, 0.059581],
[0.15665, 0.68086, 0.060024],
[0.15766, 0.68503, 0.06044],
[0.15873, 0.68921, 0.060845],
[0.15973, 0.69338, 0.061267],
[0.1608, 0.69755, 0.061737],
[0.16187, 0.70173, 0.06225],
[0.16291, 0.70593, 0.06274],
[0.16396, 0.71012, 0.063178],
[0.16497, 0.71431, 0.063603],
[0.16602, 0.71852, 0.064042],
[0.1671, 0.72272, 0.064491],
[0.16812, 0.72693, 0.064939],
[0.16921, 0.73115, 0.065387],
[0.17024, 0.73537, 0.065837],
[0.17128, 0.7396, 0.066291],
[0.17233, 0.74382, 0.066748],
[0.17339, 0.74806, 0.067206],
[0.17446, 0.75229, 0.067663],
[0.1755, 0.75654, 0.068124],
[0.17654, 0.76078, 0.068597],
[0.17764, 0.76504, 0.069073],
[0.17869, 0.76929, 0.069516],
[0.17973, 0.77355, 0.06992],
[0.18078, 0.77782, 0.070337],
[0.18184, 0.78209, 0.070809],
[0.1829, 0.78636, 0.071291],
[0.18397, 0.79063, 0.071723],
[0.18506, 0.79491, 0.072134],
[0.18612, 0.7992, 0.07259],
[0.18718, 0.80349, 0.073087],
[0.18826, 0.80779, 0.073557],
[0.1893, 0.81208, 0.073983],
[0.19036, 0.81639, 0.074402],
[0.19143, 0.82069, 0.074835],
[0.1925, 0.825, 0.075276],
[0.19359, 0.82933, 0.075716],
[0.19467, 0.83365, 0.076155],
[0.19571, 0.83797, 0.076597],
[0.19681, 0.8423, 0.077042],
[0.19786, 0.84663, 0.077489],
[0.19892, 0.85096, 0.077938],
[0.19999, 0.85531, 0.078388],
[0.20106, 0.85965, 0.078841],
[0.20215, 0.864, 0.079296],
[0.20324, 0.86836, 0.079753],
[0.20434, 0.87271, 0.080211],
[0.20539, 0.87707, 0.080669],
[0.2065, 0.88144, 0.081135],
[0.20756, 0.88581, 0.08161],
[0.20863, 0.89017, 0.082069],
[0.20974, 0.89456, 0.082489],
[0.21167, 0.8989, 0.082885],
[0.21577, 0.90314, 0.083267],
[0.22243, 0.90724, 0.083619],
[0.23129, 0.91122, 0.083917],
[0.24192, 0.91509, 0.084156],
[0.25398, 0.91886, 0.084342],
[0.26714, 0.92253, 0.084483],
[0.28122, 0.92611, 0.084587],
[0.29605, 0.92959, 0.084656],
[0.31138, 0.93299, 0.084696],
[0.32707, 0.93632, 0.08471],
[0.34303, 0.93956, 0.084701],
[0.35913, 0.94274, 0.084672],
[0.37535, 0.94584, 0.084623],
[0.39181, 0.94887, 0.084553],
[0.40832, 0.95183, 0.084465],
[0.42483, 0.95473, 0.084363],
[0.4413, 0.95756, 0.084247],
[0.45785, 0.96033, 0.084113],
[0.47443, 0.96303, 0.083964],
[0.49093, 0.96567, 0.083803],
[0.50737, 0.96826, 0.083635],
[0.52388, 0.97077, 0.083456],
[0.54033, 0.97323, 0.083278],
[0.55668, 0.97564, 0.083106],
[0.57311, 0.97798, 0.082937],
[0.58945, 0.98026, 0.082777],
[0.60571, 0.98249, 0.082624],
[0.62205, 0.98465, 0.082472],
[0.63831, 0.98675, 0.082325],
[0.6545, 0.9888, 0.08218],
[0.67074, 0.99078, 0.082035],
[0.68689, 0.99272, 0.081896],
[0.70305, 0.99458, 0.081762],
[0.71919, 0.99639, 0.081638],
[0.73525, 0.99815, 0.081527],
[0.75139, 0.99984, 0.081428],
[0.76742, 1, 0.081347],
[0.78349, 1, 0.081282],
[0.79956, 1, 0.081236],
[0.81555, 1, 0.08121],
[0.83159, 1, 0.081204],
[0.84754, 1, 0.081223],
]
b_linear_kgy_5_95_c69 = bokeh_palette('linear_kgy_5_95_c69',linear_kgy_5_95_c69)
m_linear_kgy_5_95_c69 = mpl_cm('linear_kgy_5_95_c69',linear_kgy_5_95_c69)
m_linear_kgy_5_95_c69_r = mpl_cm('linear_kgy_5_95_c69_r',list(reversed(linear_kgy_5_95_c69)))
kgy = b_linear_kgy_5_95_c69
m_kgy = m_linear_kgy_5_95_c69
m_kgy_r = m_linear_kgy_5_95_c69_r
palette['kgy'] = b_linear_kgy_5_95_c69
palette_n['kgy'] = b_linear_kgy_5_95_c69
cm['kgy'] = m_linear_kgy_5_95_c69
cm['kgy_r'] = m_linear_kgy_5_95_c69_r
cm_n['kgy'] = mpl_cm('kgy',linear_kgy_5_95_c69)
cm_n['kgy_r'] = mpl_cm('kgy_r',list(reversed(linear_kgy_5_95_c69)))
linear_green_5_95_c69 = b_linear_kgy_5_95_c69
m_linear_green_5_95_c69 = m_linear_kgy_5_95_c69
m_linear_green_5_95_c69_r = m_linear_kgy_5_95_c69_r
palette['linear_green_5_95_c69'] = b_linear_kgy_5_95_c69
palette_n['linear_green_5_95_c69'] = b_linear_kgy_5_95_c69
cm['linear_green_5_95_c69'] = m_linear_kgy_5_95_c69
cm['linear_green_5_95_c69_r'] = m_linear_kgy_5_95_c69_r
cm_n['linear_green_5_95_c69'] = mpl_cm('linear_green_5_95_c69',linear_kgy_5_95_c69)
cm_n['linear_green_5_95_c69_r'] = mpl_cm('linear_green_5_95_c69_r',list(reversed(linear_kgy_5_95_c69)))
CET_L5 = b_linear_kgy_5_95_c69
m_CET_L5 = m_linear_kgy_5_95_c69
m_CET_L5_r = m_linear_kgy_5_95_c69_r
palette['CET_L5'] = b_linear_kgy_5_95_c69
cm['CET_L5'] = m_linear_kgy_5_95_c69
cm['CET_L5_r'] = m_linear_kgy_5_95_c69_r
register_cmap('cet_CET_L5',m_linear_kgy_5_95_c69)
register_cmap('cet_CET_L5_r',m_linear_kgy_5_95_c69_r)




cyclic_tritanopic_cwrk_40_100_c20 = [  # cmap_def
[0.1493, 0.73523, 0.84245],
[0.1629, 0.73969, 0.8467],
[0.17926, 0.74403, 0.85055],
[0.19751, 0.74828, 0.85406],
[0.2169, 0.75246, 0.85726],
[0.23681, 0.7566, 0.86022],
[0.25679, 0.76071, 0.86298],
[0.27658, 0.7648, 0.86558],
[0.29595, 0.76887, 0.8681],
[0.31481, 0.77294, 0.87052],
[0.33311, 0.77702, 0.87291],
[0.35082, 0.78109, 0.87526],
[0.36794, 0.78516, 0.87759],
[0.38453, 0.78924, 0.87991],
[0.40064, 0.79331, 0.88223],
[0.41627, 0.79738, 0.88454],
[0.4315, 0.80146, 0.88686],
[0.44633, 0.80553, 0.88917],
[0.46083, 0.80961, 0.89148],
[0.47503, 0.81368, 0.8938],
[0.48893, 0.81776, 0.89611],
[0.50259, 0.82183, 0.89842],
[0.51601, 0.8259, 0.90073],
[0.52923, 0.82997, 0.90305],
[0.54225, 0.83404, 0.90536],
[0.55508, 0.83811, 0.90767],
[0.56776, 0.84218, 0.90998],
[0.58028, 0.84624, 0.91229],
[0.59266, 0.8503, 0.91461],
[0.6049, 0.85437, 0.91692],
[0.61703, 0.85843, 0.91922],
[0.62903, 0.8625, 0.92154],
[0.64093, 0.86655, 0.92385],
[0.65272, 0.87061, 0.92616],
[0.66443, 0.87467, 0.92847],
[0.67605, 0.87873, 0.93078],
[0.68757, 0.88278, 0.93309],
[0.69902, 0.88684, 0.9354],
[0.7104, 0.8909, 0.93771],
[0.7217, 0.89496, 0.94002],
[0.73295, 0.89901, 0.94233],
[0.74412, 0.90306, 0.94464],
[0.75524, 0.90711, 0.94695],
[0.7663, 0.91116, 0.94926],
[0.77732, 0.91521, 0.95156],
[0.78827, 0.91926, 0.95388],
[0.79918, 0.92331, 0.95618],
[0.81004, 0.92735, 0.95849],
[0.82087, 0.9314, 0.9608],
[0.83166, 0.93544, 0.9631],
[0.84242, 0.93947, 0.96538],
[0.85313, 0.94347, 0.96764],
[0.86381, 0.94744, 0.96985],
[0.87443, 0.95133, 0.972],
[0.88499, 0.95513, 0.97403],
[0.89548, 0.95878, 0.9759],
[0.90584, 0.96222, 0.97756],
[0.91603, 0.96537, 0.97891],
[0.92598, 0.96815, 0.97989],
[0.9356, 0.97044, 0.98037],
[0.94479, 0.97216, 0.98028],
[0.95344, 0.9732, 0.97952],
[0.96142, 0.97348, 0.97802],
[0.96864, 0.97294, 0.97575],
[0.97502, 0.97155, 0.9727],
[0.98052, 0.96933, 0.96886],
[0.98511, 0.9663, 0.9643],
[0.98881, 0.96253, 0.95909],
[0.99169, 0.95812, 0.95332],
[0.99384, 0.95316, 0.94707],
[0.99535, 0.94775, 0.94046],
[0.99633, 0.942, 0.93356],
[0.9969, 0.93598, 0.92646],
[0.99715, 0.92978, 0.91921],
[0.99714, 0.92345, 0.91188],
[0.99697, 0.91704, 0.90448],
[0.99665, 0.91058, 0.89706],
[0.99625, 0.90407, 0.88962],
[0.99577, 0.89756, 0.88218],
[0.99524, 0.89103, 0.87473],
[0.99466, 0.88451, 0.86731],
[0.99405, 0.87799, 0.85989],
[0.9934, 0.87147, 0.85249],
[0.99272, 0.86495, 0.84509],
[0.99201, 0.85843, 0.83771],
[0.99127, 0.85192, 0.83033],
[0.99049, 0.84541, 0.82297],
[0.98968, 0.8389, 0.81562],
[0.98884, 0.83239, 0.80829],
[0.98797, 0.82588, 0.80096],
[0.98707, 0.81938, 0.79365],
[0.98614, 0.81288, 0.78635],
[0.98518, 0.80638, 0.77905],
[0.98418, 0.79988, 0.77178],
[0.98316, 0.79338, 0.76451],
[0.98211, 0.78688, 0.75725],
[0.98102, 0.78038, 0.75001],
[0.9799, 0.77388, 0.74278],
[0.97876, 0.76739, 0.73556],
[0.97758, 0.7609, 0.72835],
[0.97638, 0.75441, 0.72116],
[0.97515, 0.74791, 0.71398],
[0.97389, 0.74141, 0.70681],
[0.9726, 0.73493, 0.69965],
[0.97128, 0.72843, 0.69251],
[0.96993, 0.72193, 0.68538],
[0.96855, 0.71545, 0.67826],
[0.96715, 0.70895, 0.67115],
[0.96571, 0.70245, 0.66406],
[0.96425, 0.69596, 0.65698],
[0.96276, 0.68946, 0.64991],
[0.96125, 0.68296, 0.64285],
[0.9597, 0.67646, 0.63581],
[0.95813, 0.66996, 0.62878],
[0.95651, 0.66347, 0.62176],
[0.95485, 0.65697, 0.61478],
[0.95314, 0.65049, 0.60784],
[0.95135, 0.64402, 0.60092],
[0.94946, 0.63758, 0.59406],
[0.94745, 0.6312, 0.58727],
[0.94527, 0.62486, 0.5806],
[0.94287, 0.61863, 0.57404],
[0.9402, 0.6125, 0.56767],
[0.93718, 0.60653, 0.56149],
[0.93376, 0.60074, 0.55556],
[0.92989, 0.59517, 0.54993],
[0.92551, 0.58984, 0.54461],
[0.92059, 0.58479, 0.53962],
[0.91513, 0.58, 0.53499],
[0.90911, 0.57548, 0.53071],
[0.90259, 0.57121, 0.52676],
[0.89558, 0.56719, 0.5231],
[0.88816, 0.56338, 0.51971],
[0.8804, 0.55974, 0.51654],
[0.87235, 0.55624, 0.51357],
[0.86408, 0.55286, 0.51073],
[0.85564, 0.54957, 0.50802],
[0.84708, 0.54634, 0.50536],
[0.83845, 0.54313, 0.50277],
[0.82977, 0.53996, 0.50022],
[0.82105, 0.53681, 0.49771],
[0.81233, 0.53365, 0.4952],
[0.80359, 0.53052, 0.49269],
[0.79486, 0.52738, 0.49021],
[0.78614, 0.52423, 0.48772],
[0.77743, 0.52109, 0.48522],
[0.76871, 0.51795, 0.48274],
[0.76002, 0.5148, 0.48026],
[0.75133, 0.51165, 0.47778],
[0.74265, 0.50851, 0.47531],
[0.73397, 0.50534, 0.47282],
[0.72531, 0.50219, 0.47035],
[0.71665, 0.49903, 0.46787],
[0.70799, 0.49587, 0.46539],
[0.69935, 0.49271, 0.46292],
[0.69071, 0.48954, 0.46045],
[0.68208, 0.48638, 0.45799],
[0.67346, 0.48321, 0.45553],
[0.66484, 0.48005, 0.45307],
[0.65624, 0.47687, 0.4506],
[0.64763, 0.47369, 0.44814],
[0.63903, 0.47052, 0.44568],
[0.63044, 0.46734, 0.44323],
[0.62185, 0.46416, 0.44076],
[0.61326, 0.46096, 0.43832],
[0.60469, 0.45778, 0.43587],
[0.59611, 0.4546, 0.43341],
[0.58754, 0.4514, 0.43097],
[0.57898, 0.4482, 0.42852],
[0.57041, 0.44501, 0.42607],
[0.56186, 0.44181, 0.42363],
[0.55328, 0.4386, 0.42119],
[0.54473, 0.4354, 0.41875],
[0.53617, 0.4322, 0.41632],
[0.52761, 0.42897, 0.41388],
[0.51906, 0.42576, 0.41145],
[0.51048, 0.42254, 0.40902],
[0.50192, 0.41934, 0.40661],
[0.49334, 0.41614, 0.40421],
[0.48477, 0.41297, 0.40184],
[0.47622, 0.40984, 0.39953],
[0.46769, 0.40676, 0.39731],
[0.45919, 0.40379, 0.39522],
[0.45076, 0.40097, 0.39328],
[0.44244, 0.39833, 0.39159],
[0.43426, 0.39597, 0.39019],
[0.42632, 0.39394, 0.38921],
[0.41868, 0.39235, 0.38871],
[0.41143, 0.39127, 0.38878],
[0.40464, 0.39076, 0.38948],
[0.39842, 0.3909, 0.3909],
[0.3928, 0.3917, 0.39304],
[0.38783, 0.39319, 0.39594],
[0.38356, 0.39537, 0.39954],
[0.37992, 0.39817, 0.40383],
[0.37691, 0.40153, 0.4087],
[0.37444, 0.40541, 0.4141],
[0.37243, 0.4097, 0.41994],
[0.3708, 0.41432, 0.42612],
[0.36946, 0.41921, 0.43258],
[0.36833, 0.4243, 0.43923],
[0.3673, 0.42953, 0.44603],
[0.36636, 0.43486, 0.45295],
[0.36542, 0.44025, 0.45994],
[0.36446, 0.44569, 0.46699],
[0.36346, 0.45118, 0.47406],
[0.3624, 0.45667, 0.48117],
[0.36126, 0.46218, 0.48832],
[0.36003, 0.4677, 0.49547],
[0.35871, 0.47322, 0.50264],
[0.35731, 0.47877, 0.50983],
[0.35579, 0.48431, 0.51705],
[0.35416, 0.48986, 0.52426],
[0.35243, 0.49542, 0.53151],
[0.35061, 0.501, 0.53877],
[0.34866, 0.50656, 0.54605],
[0.34659, 0.51215, 0.55333],
[0.3444, 0.51775, 0.56064],
[0.34211, 0.52335, 0.56797],
[0.33969, 0.52896, 0.57531],
[0.33712, 0.53457, 0.58266],
[0.33442, 0.54019, 0.59004],
[0.33156, 0.54583, 0.59743],
[0.32858, 0.55147, 0.60484],
[0.32543, 0.55712, 0.61226],
[0.32213, 0.56277, 0.61971],
[0.31866, 0.56843, 0.62715],
[0.315, 0.5741, 0.63463],
[0.31118, 0.57979, 0.64211],
[0.30715, 0.58548, 0.64962],
[0.30291, 0.59117, 0.65713],
[0.29847, 0.59687, 0.66466],
[0.29379, 0.60258, 0.67221],
[0.28889, 0.60831, 0.67977],
[0.2837, 0.61402, 0.68735],
[0.27826, 0.61977, 0.69494],
[0.2725, 0.6255, 0.70255],
[0.26644, 0.63126, 0.71017],
[0.26002, 0.63702, 0.71781],
[0.25323, 0.64278, 0.72546],
[0.24601, 0.64855, 0.73313],
[0.23835, 0.65433, 0.7408],
[0.23021, 0.66011, 0.74848],
[0.22159, 0.66589, 0.75615],
[0.21243, 0.67165, 0.76381],
[0.20273, 0.67741, 0.77145],
[0.19256, 0.68314, 0.77902],
[0.18198, 0.68883, 0.78651],
[0.17124, 0.69444, 0.79388],
[0.16061, 0.7, 0.80107],
[0.15065, 0.70545, 0.80804],
[0.14204, 0.71078, 0.81474],
[0.13569, 0.71598, 0.8211],
[0.13268, 0.72101, 0.82708],
[0.13377, 0.72591, 0.83264],
[0.13933, 0.73064, 0.83777],
]
b_cyclic_tritanopic_cwrk_40_100_c20 = bokeh_palette('cyclic_tritanopic_cwrk_40_100_c20',cyclic_tritanopic_cwrk_40_100_c20)
m_cyclic_tritanopic_cwrk_40_100_c20 = mpl_cm('cyclic_tritanopic_cwrk_40_100_c20',cyclic_tritanopic_cwrk_40_100_c20)
m_cyclic_tritanopic_cwrk_40_100_c20_r = mpl_cm('cyclic_tritanopic_cwrk_40_100_c20_r',list(reversed(cyclic_tritanopic_cwrk_40_100_c20)))
CET_CBTC1 = b_cyclic_tritanopic_cwrk_40_100_c20
m_CET_CBTC1 = m_cyclic_tritanopic_cwrk_40_100_c20
m_CET_CBTC1_r = m_cyclic_tritanopic_cwrk_40_100_c20_r
palette['CET_CBTC1'] = b_cyclic_tritanopic_cwrk_40_100_c20
cm['CET_CBTC1'] = m_cyclic_tritanopic_cwrk_40_100_c20
cm['CET_CBTC1_r'] = m_cyclic_tritanopic_cwrk_40_100_c20_r
register_cmap('cet_CET_CBTC1',m_cyclic_tritanopic_cwrk_40_100_c20)
register_cmap('cet_CET_CBTC1_r',m_cyclic_tritanopic_cwrk_40_100_c20_r)




linear_kry_0_97_c73 = [  # cmap_def
[0, 0, 0],
[0.025875, 1.8579e-05, 0],
[0.050536, 6.891e-05, 2.2895e-17],
[0.069748, 0.00013143, 0],
[0.085428, 0.00019917, 0],
[0.099057, 0.00026966, 0],
[0.11118, 0.00034172, 2.0277e-17],
[0.12208, 0.00041489, 4.1874e-17],
[0.13223, 0.00048918, 2.2397e-17],
[0.14161, 0.00056432, 3.363e-18],
[0.15035, 0.00067943, 0],
[0.15851, 0.00088674, 0],
[0.16633, 0.0010025, 0],
[0.1739, 0.0010092, 0],
[0.18103, 0.0011063, 0],
[0.18795, 0.0011839, 0],
[0.19455, 0.0012756, 0],
[0.2009, 0.0013408, 0],
[0.2071, 0.0014347, 0],
[0.21305, 0.00149, 0],
[0.21884, 0.0015929, 0],
[0.2245, 0.0015673, 0],
[0.2301, 0.0015759, 9.1726e-06],
[0.23571, 0.0016594, 7.2964e-06],
[0.24129, 0.0017457, 9.1367e-06],
[0.2469, 0.0018347, 7.4138e-06],
[0.25254, 0.0019267, 9.048e-06],
[0.25818, 0.0020217, 7.7461e-06],
[0.26389, 0.0021199, 9.0408e-06],
[0.26958, 0.0022211, 8.2807e-06],
[0.27529, 0.0023254, 9.1305e-06],
[0.28101, 0.0024328, 8.9188e-06],
[0.28677, 0.0025437, 9.1759e-06],
[0.29255, 0.002658, 9.4883e-06],
[0.29834, 0.0027757, 9.1679e-06],
[0.30416, 0.002897, 9.7948e-06],
[0.31001, 0.0030218, 8.8831e-06],
[0.31586, 0.0031503, 9.5701e-06],
[0.32171, 0.0032825, 8.1271e-06],
[0.3276, 0.0034185, 8.4831e-06],
[0.33352, 0.0035585, 8.1576e-06],
[0.33945, 0.0037024, 7.8335e-06],
[0.34538, 0.0038504, 9.8047e-06],
[0.35134, 0.0040025, 8.7941e-06],
[0.35733, 0.0041588, 1.0059e-05],
[0.3633, 0.0043191, 8.5968e-06],
[0.36932, 0.0044839, 8.3847e-06],
[0.37534, 0.0046534, 9.2457e-06],
[0.3814, 0.0048273, 7.8263e-06],
[0.38745, 0.0050061, 1.0258e-05],
[0.39353, 0.0051895, 8.5681e-06],
[0.39962, 0.0053778, 8.2326e-06],
[0.40575, 0.0055711, 9.7513e-06],
[0.41187, 0.0057692, 7.767e-06],
[0.41801, 0.0059729, 9.806e-06],
[0.42417, 0.0061818, 8.3319e-06],
[0.43035, 0.0063962, 5.875e-06],
[0.43654, 0.0066164, 1.0184e-05],
[0.44274, 0.0068416, 8.184e-06],
[0.44897, 0.0070728, 6.4277e-06],
[0.4552, 0.0073107, 1.0111e-05],
[0.46145, 0.0075542, 7.8984e-06],
[0.46771, 0.0078038, 6.2861e-06],
[0.47399, 0.0080598, 9.9295e-06],
[0.48028, 0.0083217, 7.7295e-06],
[0.4866, 0.0085909, 4.858e-06],
[0.49291, 0.0088684, 9.3925e-06],
[0.49925, 0.0091538, 8.5788e-06],
[0.5056, 0.0094452, 5.3801e-06],
[0.51197, 0.0097424, 7.7889e-06],
[0.51836, 0.01004, 9.2895e-06],
[0.52474, 0.010346, 6.7865e-06],
[0.53116, 0.010674, 3.6172e-06],
[0.5376, 0.011007, 7.8667e-06],
[0.54404, 0.011355, 8.6084e-06],
[0.55051, 0.011709, 6.2993e-06],
[0.55698, 0.012063, 2.7416e-06],
[0.56345, 0.012427, 6.5109e-06],
[0.56992, 0.012791, 8.0473e-06],
[0.57644, 0.013176, 7.2625e-06],
[0.58293, 0.013573, 4.6408e-06],
[0.58946, 0.013979, 2.1857e-06],
[0.596, 0.014395, 5.2887e-06],
[0.60255, 0.014821, 6.7191e-06],
[0.60912, 0.015259, 6.739e-06],
[0.61569, 0.015709, 5.6664e-06],
[0.62227, 0.016172, 3.8093e-06],
[0.62888, 0.016646, 1.4078e-06],
[0.63548, 0.017134, 1.7518e-06],
[0.64211, 0.017635, 3.6381e-06],
[0.64875, 0.018151, 4.8728e-06],
[0.65539, 0.018681, 5.6412e-06],
[0.66205, 0.019227, 6.0862e-06],
[0.66872, 0.019788, 6.2969e-06],
[0.67539, 0.020365, 6.4176e-06],
[0.68208, 0.020959, 6.4663e-06],
[0.68879, 0.021571, 6.5228e-06],
[0.6955, 0.022202, 6.6182e-06],
[0.70222, 0.022851, 6.7011e-06],
[0.70896, 0.023521, 6.7892e-06],
[0.7157, 0.024212, 6.7682e-06],
[0.72245, 0.024926, 6.4978e-06],
[0.72921, 0.025662, 1.5249e-06],
[0.73598, 0.026422, 2.6986e-08],
[0.74277, 0.027209, 0],
[0.74956, 0.028023, 0],
[0.75636, 0.028867, 0],
[0.76317, 0.02974, 0],
[0.76999, 0.030646, 0],
[0.77682, 0.031585, 0],
[0.78365, 0.032567, 0],
[0.79049, 0.033563, 0],
[0.79736, 0.034613, 0],
[0.80423, 0.035779, 0],
[0.8111, 0.036893, 0],
[0.81796, 0.038106, 0],
[0.82484, 0.039371, 0],
[0.83173, 0.040693, 0],
[0.83859, 0.04215, 0],
[0.84547, 0.043514, 0],
[0.85234, 0.045178, 0],
[0.85921, 0.046838, 0],
[0.86608, 0.048579, 0],
[0.87292, 0.050451, 0],
[0.87974, 0.0525, 0],
[0.88656, 0.054719, 0],
[0.89334, 0.057161, 0],
[0.90008, 0.059959, 0],
[0.90677, 0.063006, 0],
[0.91335, 0.066454, 0],
[0.91977, 0.070728, 0],
[0.92596, 0.076324, 0],
[0.93169, 0.083888, 0],
[0.93692, 0.093109, 0],
[0.94164, 0.10384, 0],
[0.94588, 0.1155, 0],
[0.94967, 0.12783, 0],
[0.95306, 0.14045, 0],
[0.95608, 0.15318, 0],
[0.95878, 0.1659, 0],
[0.96121, 0.17852, 0],
[0.96341, 0.19087, 0],
[0.96539, 0.20302, 0],
[0.96721, 0.21491, 0],
[0.96886, 0.22655, 0],
[0.97038, 0.2379, 0],
[0.97178, 0.249, 0],
[0.97308, 0.25985, 0],
[0.97429, 0.27047, 0],
[0.97542, 0.28082, 0],
[0.97647, 0.29102, 0],
[0.97746, 0.30097, 0],
[0.97838, 0.31077, 0],
[0.97926, 0.32037, 0],
[0.98008, 0.32979, 0],
[0.98086, 0.33905, 0],
[0.9816, 0.34817, 0],
[0.9823, 0.35714, 0],
[0.98297, 0.36597, 0],
[0.98361, 0.37467, 0],
[0.98422, 0.38326, 0],
[0.9848, 0.39172, 0],
[0.98535, 0.40008, 0],
[0.98588, 0.40833, 0],
[0.98639, 0.41649, 0],
[0.98688, 0.42455, 0],
[0.98735, 0.43253, 0],
[0.9878, 0.44041, 0],
[0.98823, 0.44821, 0],
[0.98865, 0.45595, 0],
[0.98905, 0.4636, 0],
[0.98944, 0.47119, 0],
[0.98981, 0.47871, 0],
[0.99017, 0.48616, 0],
[0.99052, 0.49355, 0],
[0.99086, 0.50088, 0],
[0.99118, 0.50817, 0],
[0.9915, 0.51538, 0],
[0.9918, 0.52255, 0],
[0.99209, 0.52967, 0],
[0.99238, 0.53674, 0],
[0.99265, 0.54376, 0],
[0.99292, 0.55074, 0],
[0.99318, 0.55768, 0],
[0.99343, 0.56457, 0],
[0.99367, 0.57142, 0],
[0.9939, 0.57823, 0],
[0.99413, 0.58502, 0],
[0.99435, 0.59177, 0],
[0.99457, 0.59847, 0],
[0.99478, 0.60514, 0],
[0.99497, 0.61179, 0],
[0.99517, 0.6184, 0],
[0.99536, 0.62498, 0],
[0.99554, 0.63154, 0],
[0.99572, 0.63806, 0],
[0.9959, 0.64456, 0],
[0.99606, 0.65104, 0],
[0.99623, 0.65749, 0],
[0.99638, 0.66391, 0],
[0.99654, 0.67031, 0],
[0.99669, 0.67668, 0],
[0.99683, 0.68303, 0],
[0.99697, 0.68937, 0],
[0.9971, 0.69568, 0],
[0.99724, 0.70196, 0],
[0.99736, 0.70824, 0],
[0.99749, 0.71448, 0],
[0.99761, 0.72072, 0],
[0.99772, 0.72693, 0],
[0.99783, 0.73313, 0],
[0.99794, 0.73932, 0],
[0.99805, 0.74547, 0],
[0.99814, 0.75163, 0],
[0.99825, 0.75776, 0],
[0.99834, 0.76387, 0],
[0.99843, 0.76997, 0],
[0.99852, 0.77607, 0],
[0.99861, 0.78214, 0],
[0.99869, 0.7882, 0],
[0.99877, 0.79425, 0],
[0.99884, 0.80029, 0],
[0.99892, 0.80631, 0],
[0.99899, 0.81232, 0],
[0.99906, 0.81832, 0],
[0.99912, 0.82431, 0],
[0.99918, 0.83029, 0],
[0.99924, 0.83626, 0],
[0.9993, 0.84222, 0],
[0.99935, 0.84816, 0],
[0.99941, 0.85409, 0],
[0.99945, 0.86003, 0],
[0.99951, 0.86594, 0],
[0.99954, 0.87186, 0],
[0.99959, 0.87775, 0],
[0.99963, 0.88365, 0],
[0.99967, 0.88953, 0],
[0.9997, 0.8954, 0],
[0.99974, 0.90127, 0],
[0.99977, 0.90713, 0],
[0.9998, 0.91298, 0],
[0.99983, 0.91882, 0],
[0.99985, 0.92466, 0],
[0.99988, 0.93049, 0],
[0.9999, 0.93632, 0],
[0.99992, 0.94214, 0],
[0.99993, 0.94795, 0],
[0.99995, 0.95375, 0],
[0.99996, 0.95955, 0],
[0.99998, 0.96534, 0],
[0.99998, 0.97113, 0],
[0.99999, 0.97692, 0],
[1, 0.9827, 0],
[1, 0.98847, 0],
[1, 0.99423, 0],
[1, 1, 1.9991e-15],
]
b_linear_kry_0_97_c73 = bokeh_palette('linear_kry_0_97_c73',linear_kry_0_97_c73)
m_linear_kry_0_97_c73 = mpl_cm('linear_kry_0_97_c73',linear_kry_0_97_c73)
m_linear_kry_0_97_c73_r = mpl_cm('linear_kry_0_97_c73_r',list(reversed(linear_kry_0_97_c73)))
CET_L4 = b_linear_kry_0_97_c73
m_CET_L4 = m_linear_kry_0_97_c73
m_CET_L4_r = m_linear_kry_0_97_c73_r
palette['CET_L4'] = b_linear_kry_0_97_c73
cm['CET_L4'] = m_linear_kry_0_97_c73
cm['CET_L4_r'] = m_linear_kry_0_97_c73_r
register_cmap('cet_CET_L4',m_linear_kry_0_97_c73)
register_cmap('cet_CET_L4_r',m_linear_kry_0_97_c73_r)




glasbey_category10 = [  # cmap_def
[0.121569, 0.466667, 0.705882],
[1.000000, 0.498039, 0.054902],
[0.172549, 0.627451, 0.172549],
[0.839216, 0.152941, 0.156863],
[0.580392, 0.403922, 0.741176],
[0.549020, 0.337255, 0.294118],
[0.890196, 0.466667, 0.760784],
[0.498039, 0.498039, 0.498039],
[0.737255, 0.741176, 0.133333],
[0.090196, 0.745098, 0.811765],
[0.227451, 0.003922, 0.513725],
[0.000000, 0.262745, 0.003922],
[0.058824, 1.000000, 0.662745],
[0.368627, 0.000000, 0.250980],
[0.737255, 0.737255, 1.000000],
[0.847059, 0.686275, 0.635294],
[0.721569, 0.000000, 0.501961],
[0.000000, 0.305882, 0.325490],
[0.419608, 0.396078, 0.000000],
[0.490196, 0.007843, 0.000000],
[0.380392, 0.149020, 1.000000],
[1.000000, 1.000000, 0.603922],
[0.341176, 0.286275, 0.392157],
[0.549020, 0.721569, 0.580392],
[0.580392, 0.988235, 1.000000],
[0.007843, 0.509804, 0.407843],
[0.568627, 1.000000, 0.000000],
[0.513725, 0.000000, 0.627451],
[0.678431, 0.537255, 0.266667],
[0.356863, 0.203922, 0.000000],
[1.000000, 0.752941, 0.952941],
[1.000000, 0.435294, 0.462745],
[0.474510, 0.549020, 1.000000],
[0.866667, 0.000000, 1.000000],
[0.317647, 0.337255, 0.274510],
[0.000000, 0.270588, 0.541176],
[1.000000, 0.749020, 0.376471],
[1.000000, 0.003922, 0.552941],
[0.745098, 0.788235, 0.811765],
[0.686275, 0.596078, 0.709804],
[0.717647, 0.341176, 0.000000],
[0.007843, 0.439216, 0.000000],
[0.803922, 0.533333, 1.000000],
[0.113725, 0.839216, 0.274510],
[0.752941, 0.925490, 0.768627],
[0.478431, 0.596078, 0.709804],
[0.647059, 0.376471, 0.537255],
[0.435294, 0.537255, 0.341176],
[0.741176, 0.490196, 0.462745],
[0.545098, 0.160784, 0.270588],
[0.000000, 0.678431, 1.000000],
[0.560784, 0.831373, 1.000000],
[0.294118, 0.427451, 0.466667],
[0.000000, 0.831373, 0.694118],
[0.576471, 0.000000, 0.952941],
[0.545098, 0.584314, 0.000000],
[0.364706, 0.360784, 0.623529],
[0.996078, 0.874510, 0.733333],
[0.000000, 0.576471, 0.623529],
[1.000000, 0.862745, 0.000000],
[0.000000, 0.670588, 0.474510],
[0.321569, 0.000000, 0.407843],
[0.000000, 0.000000, 0.572549],
[0.043137, 0.364706, 0.243137],
[0.650980, 0.890196, 0.462745],
[0.384314, 0.231373, 0.254902],
[0.776471, 0.780392, 0.541176],
[1.000000, 0.619608, 0.713725],
[0.807843, 0.313725, 0.423529],
[1.000000, 0.027451, 0.839216],
[0.545098, 0.227451, 0.019608],
[0.498039, 0.243137, 0.443137],
[1.000000, 0.286275, 0.007843],
[0.376471, 0.172549, 0.650980],
[0.109804, 0.000000, 1.000000],
[0.905882, 0.874510, 1.000000],
[0.666667, 0.231373, 0.686275],
[0.850980, 0.611765, 0.000000],
[0.639216, 0.639216, 0.623529],
[0.247059, 0.415686, 1.000000],
[0.274510, 0.286275, 0.050980],
[0.482353, 0.411765, 0.521569],
[0.423529, 0.596078, 0.552941],
[1.000000, 0.603922, 0.458824],
[0.517647, 0.360784, 1.000000],
[0.486275, 0.423529, 0.274510],
[0.505882, 0.717647, 0.333333],
[0.741176, 0.000000, 0.290196],
[0.992157, 0.580392, 1.000000],
[0.364706, 0.000000, 0.094118],
[0.541176, 0.819608, 0.819608],
[0.615686, 0.552941, 0.827451],
[0.854902, 0.427451, 0.262745],
[0.545098, 0.345098, 0.000000],
[0.235294, 0.317647, 0.415686],
[0.298039, 0.423529, 0.231373],
[0.933333, 0.815686, 0.847059],
[0.811765, 0.933333, 1.000000],
[0.666667, 0.082353, 0.000000],
[0.874510, 1.000000, 0.309804],
[1.000000, 0.168627, 0.341176],
[0.819608, 0.286275, 0.619608],
[0.439216, 0.490196, 0.721569],
[0.352941, 0.501961, 0.000000],
[0.000000, 0.898039, 0.992157],
[0.466667, 0.294118, 0.584314],
[0.407843, 0.835294, 0.549020],
[0.243137, 0.227451, 0.447059],
[0.674510, 0.254902, 0.247059],
[0.839216, 0.635294, 0.400000],
[0.756863, 0.411765, 0.807843],
[0.415686, 0.349020, 0.368627],
[0.533333, 0.674510, 0.929412],
[0.627451, 0.650980, 0.415686],
[0.823529, 0.666667, 0.901961],
[0.533333, 0.000000, 0.388235],
[0.000000, 0.992157, 0.858824],
[0.407843, 0.156863, 0.098039],
[0.705882, 0.258824, 1.000000],
[0.054902, 0.349020, 0.772549],
[0.090196, 0.529412, 0.262745],
[0.568627, 0.827451, 0.000000],
[0.807843, 0.474510, 0.000000],
[0.976471, 0.352941, 1.000000],
[0.356863, 0.454902, 0.400000],
[0.556863, 0.682353, 0.701961],
[0.611765, 0.490196, 0.549020],
[0.274510, 0.000000, 0.776471],
[0.423529, 0.305882, 0.180392],
[0.650980, 0.427451, 0.274510],
[0.619608, 0.537255, 0.450980],
[0.658824, 0.686275, 0.792157],
[0.807843, 0.552941, 0.654902],
[0.000000, 0.996078, 0.392157],
[0.572549, 0.474510, 0.000000],
[1.000000, 0.388235, 0.631373],
[0.960784, 1.000000, 0.847059],
[0.003922, 0.549020, 0.945098],
[0.078431, 0.674510, 0.627451],
[0.356863, 0.180392, 0.352941],
[0.537255, 0.525490, 0.619608],
[0.815686, 0.800000, 0.733333],
[0.831373, 0.686275, 0.772549],
[0.858824, 0.866667, 0.427451],
[0.815686, 1.000000, 0.956863],
[0.000000, 0.396078, 0.525490],
[0.000000, 0.411765, 0.388235],
[0.658824, 0.254902, 0.407843],
[0.176471, 0.592157, 0.772549],
[0.662745, 0.454902, 1.000000],
[0.152941, 0.733333, 0.368627],
[0.345098, 0.717647, 0.000000],
[0.796078, 1.000000, 0.654902],
[0.643137, 0.478431, 0.670588],
[1.000000, 0.741176, 0.580392],
[0.537255, 0.886275, 0.756863],
[0.058824, 0.788235, 1.000000],
[0.835294, 0.000000, 0.772549],
[0.388235, 0.427451, 0.541176],
[0.411765, 0.521569, 0.560784],
[0.294118, 0.305882, 0.325490],
[0.670588, 0.376471, 0.407843],
[0.478431, 0.713725, 0.835294],
[0.172549, 0.352941, 0.090196],
[0.603922, 0.000000, 0.145098],
[0.745098, 0.819608, 0.952941],
[0.541176, 0.435294, 0.407843],
[0.415686, 0.647059, 0.419608],
[0.525490, 0.329412, 0.407843],
[0.682353, 0.803922, 0.729412],
[0.533333, 0.600000, 0.498039],
[0.796078, 0.862745, 0.000000],
[0.607843, 0.015686, 0.568627],
[0.921569, 0.737255, 0.105882],
[0.921569, 0.611765, 0.823529],
[0.439216, 0.000000, 0.435294],
[0.694118, 0.631373, 0.196078],
[0.792157, 0.423529, 0.576471],
[0.254902, 0.274510, 0.643137],
[0.898039, 0.549020, 0.541176],
[0.835294, 0.270588, 0.000000],
[0.780392, 0.545098, 0.796078],
[0.717647, 0.588235, 0.592157],
[0.831373, 0.125490, 0.462745],
[0.447059, 0.294118, 0.800000],
[0.407843, 0.305882, 0.000000],
[0.407843, 0.133333, 0.219608],
[0.219608, 0.337255, 0.309804],
[0.435294, 0.733333, 0.670588],
[0.525490, 0.227451, 0.192157],
[0.647059, 0.827451, 0.596078],
[0.725490, 0.686275, 0.560784],
[0.847059, 0.894118, 0.874510],
[0.670588, 0.000000, 0.878431],
[0.796078, 0.756863, 0.858824],
[1.000000, 0.874510, 0.549020],
[0.890196, 0.325490, 0.301961],
[0.400000, 0.411765, 0.435294],
[1.000000, 0.000000, 0.109804],
[0.325490, 0.176471, 0.450980],
[0.305882, 0.568627, 0.423529],
[0.658824, 0.427451, 0.066667],
[1.000000, 0.623529, 0.149020],
[0.372549, 0.639216, 0.690196],
[0.784314, 0.521569, 0.341176],
[0.572549, 0.349020, 0.596078],
[0.639216, 0.631373, 1.000000],
[0.996078, 0.729412, 0.729412],
[0.145098, 0.164706, 0.533333],
[0.858824, 0.901961, 0.658824],
[0.592157, 0.949020, 0.654902],
[0.403922, 0.580392, 0.839216],
[0.729412, 0.356863, 0.250980],
[0.227451, 0.364706, 0.572549],
[0.211765, 0.309804, 0.184314],
[0.152941, 0.486275, 0.588235],
[0.541176, 0.584314, 0.607843],
[0.815686, 0.705882, 0.341176],
[0.000000, 0.278431, 0.392157],
[0.372549, 0.364706, 0.184314],
[0.556863, 0.556863, 0.254902],
[0.678431, 0.247059, 0.074510],
[0.415686, 0.588235, 0.235294],
[0.631373, 0.239216, 0.521569],
[0.749020, 0.717647, 0.729412],
[0.674510, 0.776471, 0.403922],
[0.396078, 0.411765, 0.811765],
[0.572549, 0.690196, 0.000000],
[0.172549, 0.890196, 0.854902],
[0.003922, 0.435294, 0.211765],
[1.000000, 0.474510, 0.325490],
[0.258824, 0.505882, 0.498039],
[0.309804, 0.913725, 0.000000],
[0.600000, 0.329412, 0.156863],
[0.364706, 0.039216, 0.000000],
[0.639216, 0.000000, 0.345098],
[0.047059, 0.533333, 0.000000],
[0.352941, 0.513725, 0.654902],
[1.000000, 0.925490, 0.984314],
[0.294118, 0.411765, 0.003922],
[0.533333, 0.462745, 0.831373],
[0.901961, 0.780392, 1.000000],
[0.647059, 1.000000, 0.854902],
[0.847059, 0.435294, 0.470588],
[0.874510, 0.007843, 0.294118],
[0.415686, 0.407843, 0.360784],
[0.470588, 0.419608, 0.635294],
[0.494118, 0.501961, 0.403922],
[0.352941, 0.278431, 0.525490],
[0.000000, 0.000000, 0.792157],
[0.486275, 0.000000, 0.168627],
[0.592157, 1.000000, 0.447059],
[0.713725, 0.890196, 0.882353],
[0.862745, 0.325490, 0.788235],
[0.466667, 0.470588, 0.203922],
[0.345098, 0.745098, 0.556863],
]
b_glasbey_category10 = bokeh_palette('glasbey_category10',glasbey_category10)
m_glasbey_category10 = mpl_cm('glasbey_category10',glasbey_category10)
m_glasbey_category10_r = mpl_cm('glasbey_category10_r',list(reversed(glasbey_category10)))




glasbey_bw_minc_20_minl_30 = [  # cmap_def
[0.843137, 0.000000, 0.000000],
[0.007843, 0.533333, 0.000000],
[0.713725, 0.000000, 1.000000],
[0.023529, 0.674510, 0.776471],
[0.596078, 1.000000, 0.000000],
[1.000000, 0.647059, 0.188235],
[1.000000, 0.560784, 0.784314],
[0.474510, 0.321569, 0.372549],
[0.000000, 0.996078, 0.811765],
[0.690196, 0.647059, 1.000000],
[0.580392, 0.678431, 0.517647],
[0.603922, 0.411765, 0.000000],
[0.215686, 0.415686, 0.384314],
[0.827451, 0.000000, 0.549020],
[0.996078, 0.960784, 0.564706],
[0.784314, 0.435294, 0.400000],
[0.619608, 0.890196, 1.000000],
[0.000000, 0.788235, 0.274510],
[0.662745, 0.466667, 0.678431],
[0.721569, 0.733333, 0.007843],
[0.956863, 0.752941, 0.694118],
[1.000000, 0.156863, 0.992157],
[0.952941, 0.807843, 1.000000],
[0.000000, 0.623529, 0.490196],
[1.000000, 0.384314, 0.000000],
[0.337255, 0.396078, 0.168627],
[0.588235, 0.247059, 0.121569],
[0.568627, 0.192157, 0.560784],
[1.000000, 0.203922, 0.396078],
[0.627451, 0.894118, 0.572549],
[0.552941, 0.607843, 0.698039],
[0.509804, 0.568627, 0.149020],
[0.682353, 0.035294, 0.247059],
[0.470588, 0.780392, 0.733333],
[0.737255, 0.572549, 0.345098],
[0.898039, 0.560784, 1.000000],
[0.447059, 0.725490, 1.000000],
[0.776471, 0.647059, 0.756863],
[1.000000, 0.568627, 0.443137],
[0.827451, 0.764706, 0.490196],
[0.741176, 0.933333, 0.858824],
[0.419608, 0.521569, 0.407843],
[0.572549, 0.431373, 0.337255],
[0.976471, 1.000000, 0.000000],
[0.729412, 0.760784, 0.878431],
[0.678431, 0.341176, 0.490196],
[1.000000, 0.807843, 0.011765],
[1.000000, 0.290196, 0.694118],
[0.760784, 0.341176, 0.011765],
[0.364706, 0.549020, 0.564706],
[0.760784, 0.266667, 0.741176],
[0.000000, 0.458824, 0.250980],
[0.729412, 0.435294, 0.996078],
[0.000000, 0.831373, 0.580392],
[0.000000, 1.000000, 0.462745],
[0.286275, 0.635294, 0.317647],
[0.800000, 0.596078, 0.568627],
[0.000000, 0.921569, 0.933333],
[0.858824, 0.494118, 0.003922],
[0.972549, 0.458824, 0.541176],
[0.725490, 0.588235, 0.000000],
[0.788235, 0.258824, 0.282353],
[0.000000, 0.815686, 0.980392],
[0.462745, 0.345098, 0.152941],
[0.521569, 0.831373, 0.003922],
[0.925490, 1.000000, 0.831373],
[0.654902, 0.482353, 0.533333],
[0.862745, 0.447059, 0.788235],
[0.796078, 0.890196, 0.341176],
[0.545098, 0.749020, 0.368627],
[0.631373, 0.129412, 0.419608],
[0.525490, 0.356863, 0.537255],
[0.541176, 0.733333, 0.815686],
[1.000000, 0.729412, 0.843137],
[0.717647, 0.811765, 0.670588],
[0.592157, 0.254902, 0.305882],
[0.407843, 0.670588, 0.000000],
[0.996078, 0.882353, 0.698039],
[1.000000, 0.215686, 0.160784],
[0.501961, 0.478431, 0.243137],
[0.843137, 0.909804, 1.000000],
[0.654902, 0.584314, 0.776471],
[0.494118, 0.647059, 0.607843],
[0.819608, 0.513725, 0.643137],
[0.329412, 0.509804, 0.231373],
[0.901961, 0.662745, 0.450980],
[0.611765, 1.000000, 1.000000],
[0.854902, 0.333333, 0.505882],
[0.019608, 0.705882, 0.666667],
[1.000000, 0.670588, 0.964706],
[0.819608, 0.686275, 0.937255],
[0.854902, 0.007843, 0.368627],
[0.674510, 0.105882, 0.074510],
[0.376471, 0.701961, 0.521569],
[0.835294, 0.258824, 0.992157],
[0.678431, 0.670588, 0.349020],
[0.984314, 0.615686, 0.654902],
[0.701961, 0.447059, 0.235294],
[0.949020, 0.415686, 0.325490],
[0.682353, 0.823529, 0.835294],
[0.607843, 1.000000, 0.768627],
[0.858824, 0.701961, 0.200000],
[0.925490, 0.007843, 0.764706],
[0.600000, 0.000000, 0.772549],
[0.815686, 1.000000, 0.619608],
[0.650980, 0.352941, 0.290196],
[0.235294, 0.427451, 0.003922],
[0.000000, 0.521569, 0.478431],
[0.584314, 0.572549, 0.403922],
[0.541176, 0.862745, 0.701961],
[0.427451, 0.454902, 0.000000],
[0.666667, 0.368627, 0.792157],
[0.027451, 0.941176, 0.000000],
[0.505882, 0.309804, 0.243137],
[0.850980, 0.505882, 0.321569],
[1.000000, 0.784314, 0.388235],
[0.721569, 0.000000, 0.623529],
[0.600000, 0.674510, 0.870588],
[0.568627, 0.309804, 0.000000],
[0.549020, 0.270588, 0.439216],
[0.309804, 0.431373, 0.321569],
[1.000000, 0.533333, 0.203922],
[0.780392, 0.560784, 0.807843],
[0.835294, 0.886275, 0.619608],
[0.698039, 0.509804, 0.427451],
[0.615686, 0.984314, 0.458824],
[0.341176, 0.870588, 0.466667],
[0.980392, 0.000000, 0.529412],
[0.635294, 0.803922, 1.000000],
[0.078431, 0.796078, 0.823529],
[0.066667, 0.560784, 0.333333],
[0.823529, 0.329412, 0.647059],
[0.000000, 0.874510, 0.764706],
[0.639216, 0.517647, 0.184314],
[0.466667, 0.592157, 0.356863],
[0.733333, 0.670588, 0.501961],
[0.439216, 0.639216, 0.690196],
[0.839216, 0.984314, 1.000000],
[0.909804, 0.007843, 0.227451],
[0.847059, 0.278431, 0.133333],
[1.000000, 0.513725, 0.929412],
[0.717647, 0.219608, 0.388235],
[0.717647, 0.807843, 0.447059],
[0.596078, 0.384314, 0.419608],
[0.541176, 0.454902, 0.568627],
[0.000000, 0.639216, 0.090196],
[0.000000, 0.960784, 0.631373],
[0.752941, 0.568627, 0.949020],
[0.541176, 0.894118, 0.847059],
[0.643137, 0.305882, 0.584314],
[0.431373, 0.368627, 0.000000],
[0.549020, 0.776471, 0.556863],
[0.584314, 0.666667, 0.168627],
[0.780392, 0.450980, 0.866667],
[0.705882, 0.231373, 0.003922],
[0.843137, 0.603922, 0.215686],
[0.874510, 0.678431, 0.717647],
[0.000000, 0.607843, 0.627451],
[0.352941, 0.564706, 0.000000],
[0.592157, 0.737255, 0.658824],
[0.678431, 0.552941, 0.658824],
[0.854902, 0.835294, 1.000000],
[0.333333, 0.490196, 0.447059],
[0.000000, 0.733333, 0.411765],
[1.000000, 0.768627, 0.556863],
[0.725490, 0.000000, 0.831373],
[0.878431, 0.815686, 0.356863],
[0.388235, 0.603922, 0.482353],
[0.752941, 0.933333, 0.737255],
[0.760784, 0.745098, 0.996078],
[0.501961, 0.827451, 0.870588],
[0.886275, 0.521569, 0.494118],
[0.980392, 0.921569, 0.305882],
[0.752941, 0.427451, 0.513725],
[0.796078, 1.000000, 0.313725],
[0.941176, 0.447059, 0.666667],
[0.929412, 0.407843, 1.000000],
[0.600000, 0.278431, 0.682353],
[0.427451, 0.411765, 0.262745],
[0.890196, 0.341176, 0.380392],
[0.866667, 0.400000, 0.176471],
[0.615686, 0.858824, 0.364706],
[0.886275, 0.615686, 0.815686],
[0.725490, 0.462745, 0.000000],
[0.776471, 0.000000, 0.176471],
[0.874510, 0.741176, 0.854902],
[0.352941, 0.713725, 0.874510],
[1.000000, 0.352941, 0.854902],
[0.219608, 0.760784, 0.631373],
[0.619608, 0.415686, 0.549020],
[0.678431, 0.666667, 0.784314],
[0.588235, 0.388235, 0.188235],
[0.713725, 0.337255, 0.384314],
[0.172549, 0.498039, 0.376471],
[0.698039, 0.894118, 0.000000],
[0.933333, 0.647059, 0.568627],
[0.584314, 0.996078, 0.886275],
[1.000000, 0.333333, 0.556863],
[0.745098, 0.435294, 0.631373],
[0.666667, 0.235294, 0.215686],
[0.850980, 0.811765, 0.000000],
[0.670588, 0.501961, 0.807843],
[0.627451, 0.501961, 0.321569],
[0.882353, 0.000000, 0.909804],
[0.764706, 0.360784, 0.243137],
[0.709804, 0.227451, 0.521569],
[0.549020, 0.470588, 0.000000],
[0.858824, 0.737255, 0.588235],
[0.321569, 0.619608, 0.576471],
[0.690196, 0.741176, 0.513725],
[0.572549, 0.713725, 0.717647],
[0.654902, 0.329412, 0.141176],
[1.000000, 0.835294, 0.937255],
[0.474510, 0.682353, 0.419608],
[0.368627, 0.709804, 0.298039],
[0.501961, 0.984314, 0.607843],
[0.282353, 1.000000, 0.937255],
[0.596078, 0.588235, 0.282353],
[0.580392, 0.533333, 0.654902],
[0.196078, 0.835294, 0.000000],
[0.431373, 0.917647, 0.337255],
[0.717647, 0.831373, 0.921569],
[0.439216, 0.333333, 0.439216],
[0.949020, 0.858824, 0.545098],
[0.670588, 0.835294, 0.760784],
[0.498039, 0.803922, 0.949020],
[0.541176, 0.733333, 0.000000],
[0.396078, 0.717647, 0.733333],
[1.000000, 0.713725, 0.000000],
[0.764706, 0.509804, 0.525490],
[0.796078, 0.670588, 0.372549],
[0.396078, 0.470588, 0.282353],
[0.349020, 0.890196, 1.000000],
[0.874510, 0.305882, 0.803922],
[0.917647, 1.000000, 0.474510],
[0.741176, 0.400000, 0.725490],
[0.768627, 0.584314, 0.650980],
[0.392157, 0.776471, 0.454902],
[0.819608, 0.584314, 0.439216],
[0.439216, 0.811765, 0.309804],
[0.670588, 0.431373, 0.400000],
[0.615686, 0.380392, 0.647059],
[0.000000, 0.721569, 0.000000],
[0.890196, 0.600000, 0.705882],
[0.741176, 0.000000, 0.423529],
[0.701961, 0.913725, 0.941176],
[0.807843, 0.749020, 0.894118],
[0.466667, 0.639216, 0.262745],
[0.521569, 0.384314, 0.470588],
[0.341176, 0.560784, 0.360784],
[0.619608, 0.690196, 0.772549],
[0.909804, 0.188235, 0.627451],
[0.145098, 0.486275, 0.164706],
[0.509804, 0.407843, 0.137255],
[0.752941, 0.737255, 0.305882],
[0.866667, 0.827451, 0.647059],
]
b_glasbey_bw_minc_20_minl_30 = bokeh_palette('glasbey_bw_minc_20_minl_30',glasbey_bw_minc_20_minl_30)
m_glasbey_bw_minc_20_minl_30 = mpl_cm('glasbey_bw_minc_20_minl_30',glasbey_bw_minc_20_minl_30)
m_glasbey_bw_minc_20_minl_30_r = mpl_cm('glasbey_bw_minc_20_minl_30_r',list(reversed(glasbey_bw_minc_20_minl_30)))
glasbey_light = b_glasbey_bw_minc_20_minl_30
m_glasbey_light = m_glasbey_bw_minc_20_minl_30
m_glasbey_light_r = m_glasbey_bw_minc_20_minl_30_r
palette['glasbey_light'] = b_glasbey_bw_minc_20_minl_30
palette_n['glasbey_light'] = b_glasbey_bw_minc_20_minl_30
cm['glasbey_light'] = m_glasbey_bw_minc_20_minl_30
cm['glasbey_light_r'] = m_glasbey_bw_minc_20_minl_30_r
cm_n['glasbey_light'] = mpl_cm('glasbey_light',glasbey_bw_minc_20_minl_30)
cm_n['glasbey_light_r'] = mpl_cm('glasbey_light_r',list(reversed(glasbey_bw_minc_20_minl_30)))




glasbey_bw_minc_20_hue_330_100 = [  # cmap_def
[0.843137, 0.000000, 0.000000],
[1.000000, 0.384314, 0.972549],
[0.674510, 0.615686, 0.000000],
[0.454902, 0.000000, 0.443137],
[0.317647, 0.286275, 0.000000],
[1.000000, 0.603922, 0.537255],
[0.568627, 0.443137, 0.541176],
[1.000000, 0.945098, 0.439216],
[0.815686, 0.000000, 0.505882],
[0.509804, 0.227451, 0.196078],
[1.000000, 0.760784, 0.956863],
[0.729412, 0.423529, 0.188235],
[0.776471, 0.749020, 0.572549],
[1.000000, 0.607843, 0.000000],
[1.000000, 0.301961, 0.419608],
[0.470588, 0.450980, 0.298039],
[0.835294, 0.545098, 0.701961],
[0.392157, 0.274510, 0.368627],
[1.000000, 0.352941, 0.000000],
[0.627451, 0.250980, 0.607843],
[0.713725, 0.345098, 0.388235],
[0.576471, 0.000000, 0.301961],
[0.525490, 0.309804, 0.003922],
[0.729412, 0.549020, 0.498039],
[0.603922, 0.000000, 0.000000],
[1.000000, 0.800000, 0.729412],
[0.874510, 0.000000, 0.811765],
[0.894118, 0.752941, 0.000000],
[1.000000, 0.274510, 0.686275],
[0.619608, 0.596078, 0.388235],
[0.592157, 0.458824, 0.000000],
[0.976471, 0.949020, 0.756863],
[1.000000, 0.776471, 0.443137],
[0.749020, 0.011765, 0.250980],
[0.682353, 0.250980, 0.015686],
[0.541176, 0.376471, 0.341176],
[0.768627, 0.364706, 0.607843],
[0.835294, 0.615686, 0.368627],
[0.866667, 0.447059, 0.380392],
[0.564706, 0.266667, 0.400000],
[0.403922, 0.309804, 0.192157],
[0.882353, 0.690196, 0.745098],
[0.780392, 0.521569, 0.000000],
[1.000000, 0.494118, 0.615686],
[1.000000, 0.588235, 0.901961],
[0.972549, 0.525490, 0.270588],
[0.431373, 0.388235, 0.003922],
[0.843137, 0.278431, 0.258824],
[0.607843, 0.000000, 0.490196],
[0.913725, 0.000000, 0.380392],
[1.000000, 0.000000, 0.192157],
[0.756863, 0.713725, 0.337255],
[0.717647, 0.466667, 0.537255],
[0.619608, 0.470588, 0.341176],
[0.823529, 0.450980, 0.788235],
[0.862745, 0.682353, 0.588235],
[0.662745, 0.231373, 0.219608],
[0.486275, 0.227451, 0.462745],
[0.529412, 0.207843, 0.000000],
[0.858824, 0.396078, 0.517647],
[0.827451, 0.325490, 0.000000],
[0.533333, 0.082353, 0.156863],
[0.611765, 0.341176, 0.196078],
[0.682353, 0.549020, 0.650980],
[0.894118, 0.858824, 0.541176],
[0.698039, 0.215686, 0.388235],
[0.858824, 0.541176, 0.564706],
[0.749020, 0.000000, 0.627451],
[0.870588, 0.447059, 0.000000],
[1.000000, 0.662745, 0.427451],
[1.000000, 0.623529, 0.764706],
[0.447059, 0.290196, 0.298039],
[0.847059, 0.635294, 0.039216],
[0.501961, 0.341176, 0.478431],
[0.964706, 0.466667, 0.756863],
[0.713725, 0.439216, 0.384314],
[1.000000, 0.803922, 0.866667],
[0.639216, 0.368627, 0.498039],
[0.478431, 0.133333, 0.349020],
[0.870588, 0.650980, 0.831373],
[1.000000, 0.376471, 0.298039],
[0.560784, 0.407843, 0.176471],
[0.709804, 0.533333, 0.286275],
[0.686275, 0.450980, 0.654902],
[0.956863, 0.874510, 0.000000],
[0.745098, 0.313725, 0.721569],
[1.000000, 0.184314, 0.854902],
[0.823529, 0.509804, 0.317647],
[0.741176, 0.345098, 0.227451],
[0.945098, 0.203922, 0.003922],
[0.588235, 0.301961, 0.317647],
[0.917647, 0.023529, 0.670588],
[0.545098, 0.513725, 0.250980],
[0.890196, 0.247059, 0.529412],
[0.513725, 0.180392, 0.274510],
[0.690196, 0.258824, 0.525490],
[0.917647, 0.819608, 0.647059],
[0.721569, 0.074510, 0.105882],
[1.000000, 0.721569, 0.172549],
[0.866667, 0.011765, 0.215686],
[0.647059, 0.352941, 0.000000],
[0.733333, 0.643137, 0.486275],
[0.392157, 0.372549, 0.188235],
[0.486275, 0.274510, 0.141176],
[0.686275, 0.003922, 0.427451],
[0.733333, 0.611765, 0.254902],
[0.807843, 0.254902, 0.376471],
[0.411765, 0.274510, 0.000000],
[1.000000, 0.486275, 0.466667],
[0.972549, 0.827451, 0.333333],
[1.000000, 0.701961, 0.690196],
[0.862745, 0.584314, 0.482353],
[0.784314, 0.600000, 0.654902],
[0.627451, 0.141176, 0.262745],
[0.858824, 0.717647, 0.435294],
[0.607843, 0.423529, 0.454902],
[0.890196, 0.552941, 0.180392],
[0.894118, 0.415686, 0.211765],
[0.886275, 0.349020, 0.729412],
[0.992157, 0.360784, 0.576471],
[0.486275, 0.380392, 0.258824],
[0.529412, 0.345098, 0.403922],
[0.615686, 0.364706, 0.588235],
[0.580392, 0.176471, 0.105882],
[0.588235, 0.537255, 0.003922],
[0.694118, 0.443137, 0.007843],
[0.792157, 0.239216, 0.094118],
[0.458824, 0.231373, 0.349020],
[0.760784, 0.698039, 0.070588],
[0.635294, 0.000000, 0.631373],
[0.886275, 0.305882, 0.862745],
[0.988235, 0.752941, 0.568627],
[0.756863, 0.313725, 0.482353],
[0.850980, 0.443137, 0.647059],
[0.870588, 0.552941, 0.827451],
[0.933333, 0.662745, 0.298039],
[0.800000, 0.239216, 0.647059],
[0.894118, 0.749020, 0.858824],
[0.580392, 0.160784, 0.396078],
[0.662745, 0.552941, 0.415686],
[0.635294, 0.000000, 0.141176],
[1.000000, 0.172549, 0.486275],
[0.666667, 0.529412, 0.149020],
[0.886275, 0.360784, 0.392157],
[0.894118, 0.631373, 0.619608],
[0.490196, 0.439216, 0.168627],
[0.796078, 0.086275, 0.380392],
[0.470588, 0.356863, 0.105882],
[0.650980, 0.360784, 0.313725],
[1.000000, 0.454902, 0.000000],
[0.803922, 0.454902, 0.478431],
[1.000000, 0.945098, 0.611765],
[0.839216, 0.792157, 0.349020],
[0.596078, 0.282353, 0.027451],
[0.929412, 0.203922, 0.313725],
[0.564706, 0.286275, 0.498039],
[0.756863, 0.125490, 0.749020],
[0.533333, 0.109804, 0.517647],
[0.988235, 0.717647, 0.839216],
[0.768627, 0.192157, 0.227451],
[0.592157, 0.266667, 0.176471],
[0.698039, 0.666667, 0.411765],
[0.662745, 0.149020, 0.000000],
[0.976471, 0.588235, 0.635294],
[0.937255, 0.525490, 0.407843],
[0.874510, 0.713725, 0.270588],
[0.517647, 0.305882, 0.254902],
[0.619608, 0.478431, 0.215686],
[0.992157, 0.000000, 0.592157],
[0.909804, 0.321569, 0.196078],
[0.772549, 0.603922, 0.737255],
[0.721569, 0.501961, 0.368627],
[0.788235, 0.380392, 0.349020],
[0.564706, 0.513725, 0.356863],
[0.725490, 0.341176, 0.086275],
[0.764706, 0.517647, 0.729412],
[0.800000, 0.505882, 0.439216],
[0.784314, 0.431373, 0.294118],
[0.623529, 0.588235, 0.254902],
[0.698039, 0.000000, 0.325490],
[0.784314, 0.490196, 0.180392],
[0.972549, 0.678431, 0.564706],
[1.000000, 0.858824, 0.584314],
[1.000000, 0.000000, 0.996078],
[0.662745, 0.411765, 0.278431],
[1.000000, 0.376471, 0.839216],
[0.956863, 0.419608, 0.498039],
[0.768627, 0.580392, 0.431373],
[0.509804, 0.262745, 0.313725],
[0.647059, 0.301961, 0.407843],
[0.670588, 0.486275, 0.482353],
[0.890196, 0.603922, 0.694118],
[0.811765, 0.780392, 0.486275],
[0.529412, 0.396078, 0.000000],
[1.000000, 0.600000, 0.258824],
[0.419608, 0.164706, 0.403922],
[0.890196, 0.505882, 0.627451],
[0.333333, 0.313725, 0.160784],
[0.941176, 0.564706, 0.780392],
[0.960784, 0.490196, 0.878431],
[0.917647, 0.588235, 0.380392],
[0.898039, 0.168627, 0.156863],
[0.741176, 0.298039, 0.282353],
[0.525490, 0.101961, 0.000000],
[0.525490, 0.341176, 0.168627],
[0.580392, 0.223529, 0.254902],
[0.756863, 0.478431, 0.631373],
[1.000000, 0.447059, 0.254902],
[0.760784, 0.403922, 0.537255],
[0.905882, 0.286275, 0.443137],
[0.898039, 0.678431, 0.478431],
[0.400000, 0.325490, 0.000000],
[0.619608, 0.200000, 0.509804],
[0.615686, 0.431373, 0.380392],
[0.466667, 0.247059, 0.000000],
[1.000000, 0.262745, 0.282353],
[1.000000, 0.835294, 0.960784],
[0.992157, 0.674510, 0.894118],
[0.501961, 0.454902, 0.000000],
[0.788235, 0.392157, 0.003922],
[0.658824, 0.478431, 0.588235],
[0.905882, 0.352941, 0.615686],
[0.733333, 0.572549, 0.003922],
[0.756863, 0.533333, 0.560784],
[0.572549, 0.400000, 0.282353],
[0.835294, 0.607843, 0.227451],
[0.517647, 0.000000, 0.419608],
[1.000000, 0.807843, 0.000000],
[0.690196, 0.262745, 0.325490],
[0.796078, 0.239216, 0.541176],
[1.000000, 0.533333, 0.003922],
[0.866667, 0.741176, 0.580392],
[0.800000, 0.611765, 0.564706],
[0.466667, 0.298039, 0.384314],
[0.709804, 0.266667, 0.168627],
[0.596078, 0.207843, 0.337255],
[0.698039, 0.325490, 0.607843],
[0.611765, 0.400000, 0.007843],
[0.752941, 0.400000, 0.694118],
[0.623529, 0.360784, 0.403922],
[0.737255, 0.603922, 0.364706],
[0.529412, 0.000000, 0.054902],
[0.894118, 0.000000, 0.478431],
[0.690196, 0.415686, 0.443137],
[0.498039, 0.000000, 0.301961],
[0.752941, 0.137255, 0.000000],
[0.729412, 0.125490, 0.521569],
[0.905882, 0.780392, 0.419608],
[0.525490, 0.000000, 0.211765],
[0.643137, 0.407843, 0.164706],
[1.000000, 0.015686, 0.364706],
[0.866667, 0.490196, 0.745098],
[0.678431, 0.466667, 0.262745],
[0.901961, 0.482353, 0.490196],
[0.870588, 0.270588, 0.631373],
[0.803922, 0.349020, 0.423529],
]
b_glasbey_bw_minc_20_hue_330_100 = bokeh_palette('glasbey_bw_minc_20_hue_330_100',glasbey_bw_minc_20_hue_330_100)
m_glasbey_bw_minc_20_hue_330_100 = mpl_cm('glasbey_bw_minc_20_hue_330_100',glasbey_bw_minc_20_hue_330_100)
m_glasbey_bw_minc_20_hue_330_100_r = mpl_cm('glasbey_bw_minc_20_hue_330_100_r',list(reversed(glasbey_bw_minc_20_hue_330_100)))
glasbey_warm = b_glasbey_bw_minc_20_hue_330_100
m_glasbey_warm = m_glasbey_bw_minc_20_hue_330_100
m_glasbey_warm_r = m_glasbey_bw_minc_20_hue_330_100_r
palette['glasbey_warm'] = b_glasbey_bw_minc_20_hue_330_100
palette_n['glasbey_warm'] = b_glasbey_bw_minc_20_hue_330_100
cm['glasbey_warm'] = m_glasbey_bw_minc_20_hue_330_100
cm['glasbey_warm_r'] = m_glasbey_bw_minc_20_hue_330_100_r
cm_n['glasbey_warm'] = mpl_cm('glasbey_warm',glasbey_bw_minc_20_hue_330_100)
cm_n['glasbey_warm_r'] = mpl_cm('glasbey_warm_r',list(reversed(glasbey_bw_minc_20_hue_330_100)))




glasbey_bw_minc_20 = [  # cmap_def
[0.843137, 0.000000, 0.000000],
[0.549020, 0.235294, 1.000000],
[0.007843, 0.533333, 0.000000],
[0.000000, 0.674510, 0.780392],
[0.596078, 1.000000, 0.000000],
[1.000000, 0.498039, 0.819608],
[0.423529, 0.000000, 0.309804],
[1.000000, 0.647059, 0.188235],
[0.345098, 0.231373, 0.000000],
[0.000000, 0.341176, 0.349020],
[0.000000, 0.000000, 0.866667],
[0.000000, 0.992157, 0.811765],
[0.631373, 0.458824, 0.415686],
[0.737255, 0.717647, 1.000000],
[0.584314, 0.709804, 0.470588],
[0.752941, 0.015686, 0.725490],
[0.392157, 0.329412, 0.454902],
[0.474510, 0.000000, 0.000000],
[0.027451, 0.454902, 0.847059],
[0.996078, 0.960784, 0.564706],
[0.000000, 0.294118, 0.000000],
[0.560784, 0.478431, 0.000000],
[1.000000, 0.447059, 0.400000],
[0.933333, 0.725490, 0.725490],
[0.368627, 0.494118, 0.400000],
[0.607843, 0.894118, 1.000000],
[0.925490, 0.000000, 0.466667],
[0.650980, 0.482353, 0.725490],
[0.352941, 0.000000, 0.643137],
[0.015686, 0.776471, 0.000000],
[0.619608, 0.294118, 0.000000],
[0.611765, 0.231373, 0.313725],
[0.796078, 0.768627, 0.000000],
[0.443137, 0.509804, 0.596078],
[0.000000, 0.686275, 0.541176],
[0.513725, 0.533333, 1.000000],
[0.364706, 0.215686, 0.231373],
[0.223529, 0.000000, 0.000000],
[0.992157, 0.752941, 1.000000],
[0.745098, 0.905882, 0.752941],
[0.858824, 0.427451, 0.003922],
[0.576471, 0.721569, 0.713725],
[0.894118, 0.321569, 1.000000],
[0.184314, 0.325490, 0.509804],
[0.768627, 0.400000, 0.564706],
[0.333333, 0.384314, 0.125490],
[0.772549, 0.623529, 0.447059],
[0.015686, 0.509804, 0.529412],
[0.411765, 0.905882, 0.501961],
[0.501961, 0.152941, 0.564706],
[0.427451, 0.705882, 1.000000],
[0.305882, 0.200000, 1.000000],
[0.525490, 0.639216, 0.007843],
[0.996078, 0.011765, 0.796078],
[0.760784, 0.650980, 0.772549],
[0.772549, 0.341176, 0.274510],
[0.462745, 0.345098, 0.239216],
[0.003922, 0.407843, 0.258824],
[0.000000, 0.839216, 0.835294],
[0.854902, 0.878431, 1.000000],
[0.976471, 1.000000, 0.000000],
[0.415686, 0.407843, 0.690196],
[0.764706, 0.596078, 0.000000],
[0.882353, 0.803922, 0.611765],
[0.854902, 0.588235, 1.000000],
[0.733333, 0.011765, 0.992157],
[0.572549, 0.321569, 0.509804],
[0.627451, 0.000000, 0.450980],
[0.341176, 0.607843, 0.333333],
[0.827451, 0.549020, 0.560784],
[0.215686, 0.270588, 0.152941],
[0.592157, 0.647059, 0.764706],
[0.556863, 0.552941, 0.372549],
[1.000000, 0.274510, 0.000000],
[0.784314, 1.000000, 0.980392],
[0.682353, 0.427451, 1.000000],
[0.431373, 0.815686, 0.654902],
[0.749020, 1.000000, 0.549020],
[0.549020, 0.329412, 0.694118],
[0.470588, 0.211765, 0.098039],
[1.000000, 0.627451, 0.474510],
[0.662745, 0.000000, 0.121569],
[1.000000, 0.109804, 0.270588],
[0.372549, 0.066667, 0.137255],
[0.403922, 0.592157, 0.580392],
[1.000000, 0.372549, 0.580392],
[0.298039, 0.403922, 0.454902],
[0.325490, 0.572549, 0.800000],
[0.666667, 0.443137, 0.192157],
[0.007843, 0.811765, 0.996078],
[0.000000, 0.768627, 0.423529],
[0.380392, 0.207843, 0.364706],
[0.564706, 0.831373, 0.184314],
[0.749020, 0.835294, 0.486275],
[0.317647, 0.270588, 0.635294],
[0.305882, 0.137255, 0.047059],
[0.486275, 0.352941, 0.000000],
[1.000000, 0.807843, 0.266667],
[0.513725, 0.007843, 0.811765],
[0.301961, 0.992157, 1.000000],
[0.537255, 0.000000, 0.239216],
[0.482353, 0.321569, 0.360784],
[0.000000, 0.454902, 0.615686],
[0.666667, 0.513725, 0.596078],
[0.505882, 0.443137, 0.560784],
[0.384314, 0.396078, 0.996078],
[0.764706, 0.203922, 0.537255],
[0.803922, 0.160784, 0.278431],
[1.000000, 0.603922, 0.709804],
[0.764706, 0.364706, 0.733333],
[0.129412, 0.407843, 0.007843],
[0.000000, 0.556863, 0.396078],
[0.388235, 0.501961, 0.137255],
[0.537255, 0.529412, 0.749020],
[0.596078, 0.866667, 0.835294],
[0.807843, 0.498039, 0.345098],
[0.823529, 0.717647, 0.356863],
[0.380392, 0.000000, 0.431373],
[0.600000, 0.329412, 0.266667],
[0.690196, 0.780392, 0.858824],
[0.949020, 1.000000, 0.823529],
[0.000000, 0.925490, 0.007843],
[0.803922, 0.525490, 0.741176],
[0.270588, 0.000000, 0.772549],
[0.478431, 0.615686, 0.498039],
[0.447059, 0.443137, 0.278431],
[0.576471, 1.000000, 0.729412],
[0.000000, 0.329412, 0.756863],
[0.678431, 0.580392, 0.925490],
[0.247059, 0.643137, 0.086275],
[0.372549, 0.227451, 0.501961],
[0.000000, 0.298039, 0.200000],
[0.486275, 0.721569, 0.827451],
[0.596078, 0.164706, 0.000000],
[0.223529, 0.431373, 0.392157],
[0.721569, 0.000000, 0.356863],
[1.000000, 0.501961, 0.239216],
[1.000000, 0.823529, 0.913725],
[0.501961, 0.188235, 0.352941],
[0.129412, 0.203922, 0.000000],
[0.631373, 0.368627, 0.435294],
[0.309804, 0.709804, 0.690196],
[0.623529, 0.623529, 0.274510],
[0.200000, 0.486275, 0.239216],
[0.760784, 0.254902, 0.000000],
[0.780392, 0.909804, 0.243137],
[0.423529, 0.019608, 0.909804],
[0.462745, 0.737255, 0.313725],
[0.647059, 0.772549, 0.662745],
[0.854902, 0.329412, 0.431373],
[0.847059, 0.560784, 0.219608],
[0.984314, 0.486275, 1.000000],
[0.294118, 0.392157, 0.286275],
[0.839216, 0.764706, 0.921569],
[0.478431, 0.180392, 0.211765],
[0.298039, 0.560784, 0.647059],
[0.274510, 0.533333, 1.000000],
[0.639216, 0.000000, 0.768627],
[0.917647, 0.639216, 0.835294],
[1.000000, 0.737255, 0.470588],
[0.278431, 0.282353, 0.000000],
[0.635294, 0.780392, 1.000000],
[0.568627, 0.635294, 0.917647],
[0.309804, 0.411765, 0.576471],
[0.905882, 0.368627, 0.698039],
[0.623529, 0.568627, 0.690196],
[0.345098, 0.317647, 0.168627],
[0.690196, 0.368627, 0.831373],
[0.525490, 0.427451, 0.882353],
[0.756863, 0.431373, 0.447059],
[0.894118, 0.000000, 0.890196],
[0.725490, 0.717647, 0.545098],
[0.223529, 0.180392, 0.000000],
[0.890196, 0.494118, 0.643137],
[0.678431, 0.231373, 0.188235],
[0.662745, 0.733333, 0.298039],
[0.411765, 0.709804, 0.513725],
[0.580392, 0.823529, 0.564706],
[0.690196, 0.552941, 0.274510],
[0.027451, 0.372549, 0.470588],
[0.000000, 0.596078, 0.537255],
[0.352941, 0.058824, 0.007843],
[0.356863, 0.490196, 0.501961],
[0.188235, 0.345098, 0.149020],
[0.898039, 0.396078, 0.231373],
[0.372549, 0.250980, 0.160784],
[0.450980, 0.290196, 0.737255],
[0.294118, 0.325490, 0.419608],
[0.788235, 0.478431, 0.866667],
[0.611765, 0.196078, 0.568627],
[0.784314, 0.901961, 0.949020],
[0.019608, 0.670588, 0.921569],
[0.654902, 0.419608, 0.603922],
[0.905882, 0.690196, 0.000000],
[0.376471, 1.000000, 0.388235],
[0.949020, 0.870588, 0.000000],
[0.466667, 0.266667, 0.003922],
[0.376471, 0.145098, 0.254902],
[0.407843, 0.498039, 0.796078],
[0.474510, 0.623529, 0.690196],
[0.050980, 0.913725, 0.635294],
[0.615686, 0.972549, 0.858824],
[0.517647, 0.000000, 0.462745],
[0.556863, 0.427451, 0.286275],
[0.890196, 0.254902, 0.188235],
[0.725490, 0.286275, 0.423529],
[0.478431, 0.290196, 0.525490],
[1.000000, 0.815686, 0.713725],
[0.298039, 0.364706, 0.780392],
[0.886275, 0.701961, 0.572549],
[1.000000, 0.298039, 0.933333],
[0.839216, 0.941176, 0.647059],
[0.749020, 0.376471, 0.149020],
[0.843137, 0.639216, 0.709804],
[0.741176, 0.490196, 0.000000],
[0.533333, 0.435294, 0.698039],
[1.000000, 0.188235, 0.631373],
[1.000000, 0.913725, 0.690196],
[0.200000, 0.345098, 0.298039],
[0.721569, 0.549020, 0.478431],
[0.423529, 0.529412, 0.321569],
[0.737255, 0.580392, 0.823529],
[0.105882, 0.901961, 0.996078],
[0.631373, 0.235294, 0.447059],
[0.639216, 0.317647, 0.658824],
[0.427451, 0.000000, 0.596078],
[0.537255, 0.396078, 0.482353],
[0.352941, 0.345098, 0.545098],
[0.976471, 0.560784, 0.545098],
[0.905882, 0.843137, 0.486275],
[0.439216, 0.423529, 0.003922],
[0.098039, 0.349020, 1.000000],
[0.090196, 0.149020, 1.000000],
[0.000000, 0.847059, 0.337255],
[0.972549, 0.631373, 0.992157],
[0.478431, 0.588235, 0.235294],
[0.694118, 0.654902, 0.835294],
[0.494118, 0.815686, 0.866667],
[0.000000, 0.796078, 0.690196],
[0.478431, 0.278431, 0.235294],
[0.854902, 1.000000, 0.901961],
[0.858824, 0.019608, 0.694118],
[0.952941, 0.866667, 1.000000],
[0.639216, 0.894118, 0.435294],
[0.541176, 0.074510, 0.137255],
[0.400000, 0.407843, 0.513725],
[0.909804, 0.992157, 0.439216],
[0.847059, 0.670588, 0.909804],
[0.878431, 0.729412, 0.835294],
[0.996078, 0.325490, 0.411765],
[0.458824, 0.682353, 0.607843],
[0.596078, 0.200000, 0.878431],
[0.894118, 0.450980, 0.494118],
[0.549020, 0.349020, 0.149020],
[0.466667, 0.278431, 0.411765],
[0.184314, 0.243137, 0.658824],
]
b_glasbey_bw_minc_20 = bokeh_palette('glasbey_bw_minc_20',glasbey_bw_minc_20)
m_glasbey_bw_minc_20 = mpl_cm('glasbey_bw_minc_20',glasbey_bw_minc_20)
m_glasbey_bw_minc_20_r = mpl_cm('glasbey_bw_minc_20_r',list(reversed(glasbey_bw_minc_20)))
glasbey = b_glasbey_bw_minc_20
m_glasbey = m_glasbey_bw_minc_20
m_glasbey_r = m_glasbey_bw_minc_20_r
palette['glasbey'] = b_glasbey_bw_minc_20
palette_n['glasbey'] = b_glasbey_bw_minc_20
cm['glasbey'] = m_glasbey_bw_minc_20
cm['glasbey_r'] = m_glasbey_bw_minc_20_r
cm_n['glasbey'] = mpl_cm('glasbey',glasbey_bw_minc_20)
cm_n['glasbey_r'] = mpl_cm('glasbey_r',list(reversed(glasbey_bw_minc_20)))




glasbey_bw = [  # cmap_def
[0.843137, 0.000000, 0.000000],
[0.549020, 0.235294, 1.000000],
[0.007843, 0.533333, 0.000000],
[0.000000, 0.674510, 0.780392],
[0.596078, 1.000000, 0.000000],
[1.000000, 0.498039, 0.819608],
[0.423529, 0.000000, 0.309804],
[1.000000, 0.647059, 0.188235],
[0.000000, 0.000000, 0.615686],
[0.525490, 0.439216, 0.407843],
[0.000000, 0.286275, 0.258824],
[0.309804, 0.164706, 0.000000],
[0.000000, 0.992157, 0.811765],
[0.737255, 0.717647, 1.000000],
[0.584314, 0.705882, 0.478431],
[0.752941, 0.015686, 0.725490],
[0.145098, 0.400000, 0.635294],
[0.156863, 0.000000, 0.254902],
[0.862745, 0.701961, 0.686275],
[0.996078, 0.960784, 0.564706],
[0.313725, 0.270588, 0.356863],
[0.643137, 0.486275, 0.000000],
[1.000000, 0.443137, 0.400000],
[0.247059, 0.505882, 0.431373],
[0.509804, 0.000000, 0.050980],
[0.639216, 0.482353, 0.701961],
[0.203922, 0.305882, 0.000000],
[0.607843, 0.894118, 1.000000],
[0.921569, 0.000000, 0.466667],
[0.176471, 0.000000, 0.039216],
[0.368627, 0.564706, 1.000000],
[0.000000, 0.780392, 0.125490],
[0.345098, 0.003922, 0.666667],
[0.000000, 0.117647, 0.000000],
[0.603922, 0.278431, 0.000000],
[0.588235, 0.623529, 0.650980],
[0.607843, 0.258824, 0.360784],
[0.000000, 0.121569, 0.196078],
[0.784314, 0.768627, 0.000000],
[1.000000, 0.815686, 1.000000],
[0.000000, 0.745098, 0.603922],
[0.215686, 0.082353, 1.000000],
[0.176471, 0.145098, 0.145098],
[0.874510, 0.345098, 1.000000],
[0.745098, 0.905882, 0.752941],
[0.498039, 0.270588, 0.596078],
[0.321569, 0.309804, 0.235294],
[0.847059, 0.400000, 0.000000],
[0.392157, 0.454902, 0.219608],
[0.756863, 0.450980, 0.533333],
[0.431373, 0.454902, 0.541176],
[0.501961, 0.615686, 0.011765],
[0.745098, 0.545098, 0.396078],
[0.388235, 0.200000, 0.223529],
[0.792157, 0.803922, 0.854902],
[0.423529, 0.921569, 0.513725],
[0.133333, 0.250980, 0.411765],
[0.635294, 0.498039, 1.000000],
[0.996078, 0.011765, 0.796078],
[0.462745, 0.737255, 0.992157],
[0.850980, 0.764706, 0.509804],
[0.807843, 0.639216, 0.807843],
[0.427451, 0.313725, 0.000000],
[0.000000, 0.411765, 0.454902],
[0.278431, 0.623529, 0.368627],
[0.580392, 0.776471, 0.749020],
[0.976471, 1.000000, 0.000000],
[0.752941, 0.329412, 0.270588],
[0.000000, 0.396078, 0.235294],
[0.356863, 0.313725, 0.658824],
[0.325490, 0.125490, 0.392157],
[0.309804, 0.372549, 1.000000],
[0.494118, 0.560784, 0.466667],
[0.725490, 0.031373, 0.980392],
[0.545098, 0.572549, 0.764706],
[0.701961, 0.000000, 0.207843],
[0.533333, 0.376471, 0.494118],
[0.623529, 0.000000, 0.458824],
[1.000000, 0.870588, 0.768627],
[0.317647, 0.031373, 0.000000],
[0.101961, 0.031373, 0.000000],
[0.298039, 0.537255, 0.713725],
[0.000000, 0.874510, 0.874510],
[0.784314, 1.000000, 0.980392],
[0.188235, 0.207843, 0.082353],
[1.000000, 0.152941, 0.278431],
[1.000000, 0.592157, 0.666667],
[0.015686, 0.000000, 0.101961],
[0.788235, 0.376471, 0.694118],
[0.764706, 0.635294, 0.215686],
[0.486275, 0.309804, 0.227451],
[0.976471, 0.619608, 0.466667],
[0.337255, 0.396078, 0.392157],
[0.819608, 0.576471, 1.000000],
[0.176471, 0.121569, 0.411765],
[0.254902, 0.105882, 0.203922],
[0.686275, 0.576471, 0.596078],
[0.384314, 0.619608, 0.600000],
[0.741176, 0.870588, 0.482353],
[1.000000, 0.368627, 0.580392],
[0.058824, 0.160784, 0.137255],
[0.721569, 0.745098, 0.674510],
[0.454902, 0.231373, 0.396078],
[0.062745, 0.000000, 0.050980],
[0.498039, 0.431373, 0.741176],
[0.619608, 0.419608, 0.231373],
[1.000000, 0.274510, 0.000000],
[0.498039, 0.000000, 0.529412],
[1.000000, 0.807843, 0.243137],
[0.188235, 0.231373, 0.262745],
[0.996078, 0.647059, 1.000000],
[0.541176, 0.007843, 0.243137],
[0.462745, 0.172549, 0.003922],
[0.039216, 0.541176, 0.588235],
[0.019608, 0.000000, 0.321569],
[0.556863, 0.839216, 0.196078],
[0.325490, 0.768627, 0.450980],
[0.278431, 0.349020, 0.443137],
[0.345098, 0.007843, 0.133333],
[0.650980, 0.133333, 0.003922],
[0.564706, 0.576471, 0.298039],
[0.000000, 0.262745, 0.117647],
[0.505882, 0.000000, 0.819608],
[0.184314, 0.149020, 0.247059],
[0.749020, 0.223529, 0.517647],
[0.960784, 1.000000, 0.835294],
[0.000000, 0.827451, 1.000000],
[0.415686, 0.000000, 0.972549],
[0.611765, 0.733333, 0.823529],
[0.478431, 0.850980, 0.670588],
[0.411765, 0.341176, 0.364706],
[0.000000, 0.411765, 0.019608],
[0.211765, 0.211765, 0.611765],
[0.003922, 0.513725, 0.278431],
[0.266667, 0.117647, 0.094118],
[0.027451, 0.647059, 0.937255],
[1.000000, 0.505882, 0.188235],
[0.654902, 0.333333, 0.721569],
[0.407843, 0.352941, 0.513725],
[0.450980, 1.000000, 1.000000],
[0.850980, 0.529412, 0.007843],
[0.733333, 0.827451, 1.000000],
[0.556863, 0.215686, 0.184314],
[0.654902, 0.627451, 0.501961],
[0.000000, 0.490196, 0.890196],
[0.556863, 0.494118, 0.560784],
[0.600000, 0.266667, 0.533333],
[0.000000, 0.945098, 0.207843],
[0.682353, 0.666667, 0.788235],
[0.627451, 0.380392, 0.384314],
[0.298039, 0.227451, 0.466667],
[0.423529, 0.509804, 0.513725],
[0.945098, 0.866667, 0.905882],
[1.000000, 0.733333, 0.827451],
[0.219608, 0.647059, 0.137255],
[0.705882, 1.000000, 0.658824],
[0.047059, 0.070588, 0.027451],
[0.843137, 0.321569, 0.431373],
[0.584314, 0.623529, 0.996078],
[0.490196, 0.498039, 0.000000],
[0.462745, 0.623529, 0.725490],
[0.858824, 0.529412, 0.498039],
[0.066667, 0.074510, 0.098039],
[0.831373, 0.509804, 0.831373],
[0.623529, 0.000000, 0.749020],
[0.862745, 0.937255, 1.000000],
[0.556863, 0.670588, 0.603922],
[0.443137, 0.392157, 0.258824],
[0.290196, 0.235294, 0.243137],
[0.031373, 0.305882, 0.372549],
[0.611765, 0.721569, 0.266667],
[0.847059, 0.870588, 0.835294],
[0.796078, 1.000000, 0.423529],
[0.701961, 0.392157, 0.921569],
[0.274510, 0.364706, 0.200000],
[0.000000, 0.619608, 0.490196],
[0.760784, 0.254902, 0.000000],
[0.309804, 0.737255, 0.733333],
[0.850980, 0.545098, 0.694118],
[0.356863, 0.450980, 0.713725],
[0.294118, 0.254902, 0.003922],
[0.584314, 0.513725, 0.368627],
[0.286275, 0.454902, 0.545098],
[1.000000, 0.450980, 1.000000],
[0.513725, 0.415686, 0.113725],
[0.862745, 0.811765, 1.000000],
[0.494118, 0.419608, 0.996078],
[0.388235, 0.462745, 0.376471],
[1.000000, 0.756863, 0.572549],
[0.349020, 0.368627, 0.000000],
[0.894118, 0.035294, 0.901961],
[0.725490, 0.694118, 0.717647],
[0.827451, 0.176471, 0.254902],
[0.196078, 0.258824, 0.215686],
[0.850980, 0.639216, 0.388235],
[0.356863, 0.545098, 0.200000],
[0.184314, 0.121569, 0.000000],
[0.596078, 0.905882, 0.843137],
[0.164706, 0.384314, 0.341176],
[0.807843, 0.447059, 0.301961],
[0.364706, 0.239216, 0.156863],
[0.000000, 0.349020, 0.850980],
[0.678431, 0.580392, 0.839216],
[0.419608, 0.117647, 0.580392],
[0.705882, 0.003922, 0.368627],
[0.254902, 0.000000, 0.274510],
[0.615686, 1.000000, 0.811765],
[0.894118, 0.282353, 0.615686],
[0.890196, 0.890196, 0.278431],
[0.862745, 0.886275, 0.647059],
[0.000000, 0.156863, 0.352941],
[0.666667, 0.356863, 0.509804],
[0.000000, 0.000000, 0.862745],
[0.294118, 0.305882, 0.317647],
[0.854902, 0.749020, 0.835294],
[0.000000, 0.301961, 0.600000],
[0.533333, 0.392157, 0.619608],
[0.415686, 0.117647, 0.113725],
[0.556863, 0.321569, 0.772549],
[0.721569, 0.854902, 0.874510],
[0.866667, 0.701961, 0.992157],
[0.482353, 0.282353, 0.329412],
[0.298039, 0.450980, 0.000000],
[0.270588, 0.000000, 0.466667],
[0.698039, 0.372549, 0.000000],
[0.572549, 0.819608, 0.525490],
[0.333333, 0.200000, 0.298039],
[0.411765, 0.690196, 0.521569],
[0.670588, 0.576471, 0.690196],
[0.905882, 0.329412, 0.258824],
[0.560784, 0.549020, 0.541176],
[0.439216, 0.678431, 0.317647],
[0.670588, 0.486275, 0.454902],
[0.000000, 0.203922, 0.235294],
[0.145098, 0.058824, 0.074510],
[0.905882, 0.690196, 0.000000],
[0.478431, 0.800000, 0.862745],
[0.094118, 0.078431, 0.227451],
[0.615686, 0.321569, 0.223529],
[0.733333, 0.482353, 0.192157],
[0.717647, 0.792157, 0.580392],
[0.192157, 0.031373, 0.000000],
[0.639216, 0.584314, 0.023529],
[0.000000, 0.854902, 0.729412],
[0.454902, 0.627451, 0.870588],
[0.388235, 0.235294, 0.450980],
[1.000000, 0.854902, 0.560784],
[0.466667, 0.721569, 0.000000],
[0.250980, 0.184314, 0.113725],
[0.345098, 0.529412, 0.349020],
[0.176471, 0.000000, 0.129412],
[0.960784, 0.631373, 0.831373],
[0.854902, 0.000000, 0.666667],
[0.462745, 0.160784, 0.286275],
[0.741176, 0.898039, 0.000000],
[0.764706, 0.760784, 0.364706],
]
b_glasbey_bw = bokeh_palette('glasbey_bw',glasbey_bw)
m_glasbey_bw = mpl_cm('glasbey_bw',glasbey_bw)
m_glasbey_bw_r = mpl_cm('glasbey_bw_r',list(reversed(glasbey_bw)))




glasbey_bw_minc_20_maxl_70 = [  # cmap_def
[0.843137, 0.000000, 0.000000],
[0.549020, 0.235294, 1.000000],
[0.007843, 0.533333, 0.000000],
[0.000000, 0.674510, 0.780392],
[0.905882, 0.647059, 0.000000],
[1.000000, 0.498039, 0.819608],
[0.423529, 0.000000, 0.309804],
[0.345098, 0.231373, 0.000000],
[0.000000, 0.341176, 0.349020],
[0.082353, 0.882353, 0.552941],
[0.000000, 0.000000, 0.866667],
[0.635294, 0.462745, 0.415686],
[0.737255, 0.717647, 1.000000],
[0.752941, 0.015686, 0.725490],
[0.392157, 0.329412, 0.450980],
[0.474510, 0.000000, 0.000000],
[0.027451, 0.454902, 0.847059],
[0.450980, 0.607843, 0.490196],
[1.000000, 0.470588, 0.321569],
[0.000000, 0.294118, 0.000000],
[0.560784, 0.482353, 0.003922],
[0.952941, 0.000000, 0.482353],
[0.560784, 0.729412, 0.000000],
[0.650980, 0.482353, 0.721569],
[0.352941, 0.007843, 0.639216],
[0.890196, 0.686275, 0.686275],
[0.627451, 0.227451, 0.321569],
[0.635294, 0.784314, 0.784314],
[0.619608, 0.294118, 0.000000],
[0.329412, 0.403922, 0.270588],
[0.733333, 0.764706, 0.537255],
[0.372549, 0.482353, 0.533333],
[0.376471, 0.219608, 0.235294],
[0.513725, 0.533333, 1.000000],
[0.223529, 0.000000, 0.000000],
[0.890196, 0.325490, 1.000000],
[0.188235, 0.325490, 0.509804],
[0.498039, 0.792157, 1.000000],
[0.772549, 0.400000, 0.560784],
[0.000000, 0.505882, 0.415686],
[0.572549, 0.619608, 0.717647],
[0.800000, 0.454902, 0.027451],
[0.498039, 0.168627, 0.556863],
[0.000000, 0.745098, 0.643137],
[0.176471, 0.694118, 0.321569],
[0.305882, 0.200000, 1.000000],
[0.000000, 0.898039, 0.000000],
[1.000000, 0.000000, 0.807843],
[0.784314, 0.345098, 0.282353],
[0.898039, 0.611765, 1.000000],
[0.113725, 0.631373, 1.000000],
[0.431373, 0.439216, 0.670588],
[0.784314, 0.603922, 0.411765],
[0.470588, 0.341176, 0.231373],
[0.015686, 0.854902, 0.901961],
[0.756863, 0.639216, 0.768627],
[1.000000, 0.415686, 0.541176],
[0.733333, 0.000000, 0.996078],
[0.572549, 0.325490, 0.501961],
[0.623529, 0.007843, 0.454902],
[0.580392, 0.631373, 0.313725],
[0.215686, 0.266667, 0.145098],
[0.686275, 0.427451, 1.000000],
[0.349020, 0.427451, 0.000000],
[1.000000, 0.192157, 0.278431],
[0.513725, 0.501961, 0.341176],
[0.000000, 0.427451, 0.180392],
[0.537255, 0.337255, 0.686275],
[0.352941, 0.290196, 0.639216],
[0.466667, 0.207843, 0.086275],
[0.525490, 0.764706, 0.603922],
[0.372549, 0.066667, 0.137255],
[0.835294, 0.521569, 0.505882],
[0.643137, 0.160784, 0.094118],
[0.000000, 0.533333, 0.694118],
[0.796078, 0.000000, 0.266667],
[1.000000, 0.627451, 0.337255],
[0.921569, 0.305882, 0.000000],
[0.423529, 0.592157, 0.000000],
[0.325490, 0.525490, 0.286275],
[0.458824, 0.352941, 0.000000],
[0.784314, 0.768627, 0.250980],
[0.572549, 0.827451, 0.439216],
[0.294118, 0.596078, 0.580392],
[0.301961, 0.137255, 0.050980],
[0.380392, 0.203922, 0.360784],
[0.517647, 0.000000, 0.811765],
[0.545098, 0.000000, 0.192157],
[0.623529, 0.431373, 0.196078],
[0.674510, 0.517647, 0.600000],
[0.776471, 0.192157, 0.537255],
[0.007843, 0.329412, 0.219608],
[0.031373, 0.419608, 0.517647],
[0.529412, 0.658824, 0.925490],
[0.392157, 0.400000, 0.937255],
[0.768627, 0.364706, 0.729412],
[0.003922, 0.623529, 0.439216],
[0.505882, 0.317647, 0.349020],
[0.513725, 0.435294, 0.549020],
[0.701961, 0.752941, 0.854902],
[0.725490, 0.568627, 0.160784],
[1.000000, 0.592157, 0.698039],
[0.654902, 0.576471, 0.882353],
[0.411765, 0.552941, 0.745098],
[0.298039, 0.313725, 0.003922],
[0.282353, 0.007843, 0.800000],
[0.380392, 0.000000, 0.431373],
[0.270588, 0.415686, 0.400000],
[0.615686, 0.341176, 0.262745],
[0.482353, 0.674510, 0.709804],
[0.803922, 0.517647, 0.741176],
[0.000000, 0.329412, 0.756863],
[0.482353, 0.184314, 0.309804],
[0.984314, 0.486275, 0.000000],
[0.203922, 0.752941, 0.000000],
[1.000000, 0.611765, 0.533333],
[0.882353, 0.717647, 0.411765],
[0.325490, 0.380392, 0.466667],
[0.360784, 0.227451, 0.486275],
[0.929412, 0.647059, 0.854902],
[0.941176, 0.325490, 0.639216],
[0.364706, 0.494118, 0.411765],
[0.768627, 0.466667, 0.313725],
[0.819608, 0.282353, 0.407843],
[0.431373, 0.000000, 0.921569],
[0.121569, 0.203922, 0.000000],
[0.756863, 0.254902, 0.015686],
[0.427451, 0.835294, 0.760784],
[0.274510, 0.439216, 0.623529],
[0.635294, 0.003922, 0.768627],
[0.039216, 0.509804, 0.537255],
[0.686275, 0.650980, 0.003922],
[0.650980, 0.360784, 0.419608],
[0.996078, 0.466667, 1.000000],
[0.545098, 0.521569, 0.682353],
[0.780392, 0.498039, 0.913725],
[0.603922, 0.670588, 0.521569],
[0.529412, 0.423529, 0.850980],
[0.003922, 0.729412, 0.968627],
[0.686275, 0.368627, 0.823529],
[0.349020, 0.317647, 0.168627],
[0.713725, 0.000000, 0.372549],
[0.486275, 0.713725, 0.415686],
[0.286275, 0.521569, 1.000000],
[0.000000, 0.760784, 0.509804],
[0.823529, 0.584314, 0.670588],
[0.639216, 0.294118, 0.658824],
[0.890196, 0.023529, 0.890196],
[0.086275, 0.639216, 0.000000],
[0.223529, 0.180392, 0.000000],
[0.517647, 0.188235, 0.200000],
[0.368627, 0.584314, 0.666667],
[0.352941, 0.062745, 0.000000],
[0.482353, 0.274510, 0.000000],
[0.435294, 0.435294, 0.192157],
[0.200000, 0.345098, 0.149020],
[0.301961, 0.376471, 0.713725],
[0.635294, 0.584314, 0.392157],
[0.384314, 0.250980, 0.156863],
[0.270588, 0.831373, 0.345098],
[0.439216, 0.666667, 0.815686],
[0.180392, 0.419608, 0.305882],
[0.450980, 0.686275, 0.619608],
[0.992157, 0.082353, 0.000000],
[0.847059, 0.705882, 0.572549],
[0.478431, 0.537255, 0.231373],
[0.490196, 0.776471, 0.850980],
[0.862745, 0.568627, 0.215686],
[0.925490, 0.380392, 0.368627],
[0.925490, 0.372549, 0.831373],
[0.898039, 0.482353, 0.654902],
[0.650980, 0.423529, 0.596078],
[0.000000, 0.592157, 0.266667],
[0.729412, 0.372549, 0.133333],
[0.737255, 0.678431, 0.325490],
[0.533333, 0.847059, 0.188235],
[0.529412, 0.207843, 0.450980],
[0.682353, 0.658824, 0.823529],
[0.890196, 0.549020, 0.388235],
[0.819608, 0.694118, 0.925490],
[0.215686, 0.258824, 0.623529],
[0.227451, 0.745098, 0.760784],
[0.400000, 0.615686, 0.301961],
[0.619608, 0.011765, 0.600000],
[0.305882, 0.305882, 0.478431],
[0.482353, 0.298039, 0.525490],
[0.764706, 0.207843, 0.192157],
[0.552941, 0.400000, 0.466667],
[0.666667, 0.000000, 0.176471],
[0.498039, 0.003922, 0.458824],
[0.003922, 0.509804, 0.301961],
[0.450980, 0.290196, 0.403922],
[0.447059, 0.466667, 0.568627],
[0.431373, 0.000000, 0.600000],
[0.627451, 0.729412, 0.321569],
[0.882353, 0.431373, 0.192157],
[0.772549, 0.415686, 0.443137],
[0.427451, 0.356863, 0.588235],
[0.639216, 0.235294, 0.454902],
[0.196078, 0.384314, 0.000000],
[0.533333, 0.000000, 0.313725],
[0.200000, 0.345098, 0.411765],
[0.729412, 0.552941, 0.486275],
[0.098039, 0.349020, 1.000000],
[0.568627, 0.572549, 0.007843],
[0.172549, 0.545098, 0.835294],
[0.090196, 0.149020, 1.000000],
[0.129412, 0.827451, 1.000000],
[0.643137, 0.564706, 0.686275],
[0.545098, 0.427451, 0.309804],
[0.368627, 0.129412, 0.243137],
[0.862745, 0.011765, 0.701961],
[0.435294, 0.341176, 0.792157],
[0.396078, 0.156863, 0.129412],
[0.678431, 0.466667, 0.000000],
[0.639216, 0.749020, 0.968627],
[0.709804, 0.517647, 0.274510],
[0.592157, 0.219608, 0.862745],
[0.698039, 0.317647, 0.580392],
[0.447059, 0.258824, 0.639216],
[0.529412, 0.560784, 0.819608],
[0.541176, 0.439216, 0.694118],
[0.419608, 0.686275, 0.211765],
[0.352941, 0.478431, 0.788235],
[0.780392, 0.623529, 1.000000],
[0.337255, 0.517647, 0.101961],
[0.000000, 0.839216, 0.654902],
[0.509804, 0.278431, 0.223529],
[0.066667, 0.262745, 0.113725],
[0.352941, 0.670588, 0.458824],
[0.568627, 0.356863, 0.003922],
[0.964706, 0.270588, 0.439216],
[1.000000, 0.592157, 0.011765],
[0.882353, 0.258824, 0.192157],
[0.729412, 0.572549, 0.811765],
[0.203922, 0.345098, 0.301961],
[0.972549, 0.501961, 0.490196],
[0.568627, 0.203922, 0.000000],
[0.701961, 0.803922, 0.000000],
[0.180392, 0.623529, 0.827451],
[0.474510, 0.545098, 0.623529],
[0.317647, 0.505882, 0.490196],
[0.756863, 0.211765, 0.843137],
[0.925490, 0.019608, 0.325490],
[0.725490, 0.674510, 0.494118],
[0.282353, 0.439216, 0.196078],
[0.517647, 0.584314, 0.396078],
[0.850980, 0.615686, 0.537255],
[0.000000, 0.392157, 0.639216],
[0.298039, 0.564706, 0.470588],
[0.560784, 0.380392, 0.596078],
[1.000000, 0.325490, 0.219608],
[0.654902, 0.258824, 0.231373],
[0.000000, 0.431373, 0.439216],
[0.596078, 0.517647, 0.243137],
[0.862745, 0.690196, 0.784314],
]
b_glasbey_bw_minc_20_maxl_70 = bokeh_palette('glasbey_bw_minc_20_maxl_70',glasbey_bw_minc_20_maxl_70)
m_glasbey_bw_minc_20_maxl_70 = mpl_cm('glasbey_bw_minc_20_maxl_70',glasbey_bw_minc_20_maxl_70)
m_glasbey_bw_minc_20_maxl_70_r = mpl_cm('glasbey_bw_minc_20_maxl_70_r',list(reversed(glasbey_bw_minc_20_maxl_70)))
glasbey_dark = b_glasbey_bw_minc_20_maxl_70
m_glasbey_dark = m_glasbey_bw_minc_20_maxl_70
m_glasbey_dark_r = m_glasbey_bw_minc_20_maxl_70_r
palette['glasbey_dark'] = b_glasbey_bw_minc_20_maxl_70
palette_n['glasbey_dark'] = b_glasbey_bw_minc_20_maxl_70
cm['glasbey_dark'] = m_glasbey_bw_minc_20_maxl_70
cm['glasbey_dark_r'] = m_glasbey_bw_minc_20_maxl_70_r
cm_n['glasbey_dark'] = mpl_cm('glasbey_dark',glasbey_bw_minc_20_maxl_70)
cm_n['glasbey_dark_r'] = mpl_cm('glasbey_dark_r',list(reversed(glasbey_bw_minc_20_maxl_70)))




glasbey_hv = [  # cmap_def
[0.188235, 0.635294, 0.854902],
[0.988235, 0.309804, 0.188235],
[0.898039, 0.682353, 0.219608],
[0.427451, 0.564706, 0.309804],
[0.545098, 0.545098, 0.545098],
[0.090196, 0.745098, 0.811765],
[0.580392, 0.403922, 0.741176],
[0.839216, 0.152941, 0.156863],
[0.121569, 0.466667, 0.705882],
[0.890196, 0.466667, 0.760784],
[0.549020, 0.337255, 0.294118],
[0.737255, 0.741176, 0.133333],
[0.227451, 0.003922, 0.513725],
[0.000000, 0.262745, 0.000000],
[0.058824, 1.000000, 0.662745],
[0.368627, 0.000000, 0.250980],
[0.776471, 0.741176, 1.000000],
[0.258824, 0.313725, 0.321569],
[0.721569, 0.000000, 0.501961],
[1.000000, 0.717647, 0.701961],
[0.490196, 0.007843, 0.000000],
[0.380392, 0.149020, 1.000000],
[1.000000, 1.000000, 0.603922],
[0.682353, 0.788235, 0.670588],
[0.000000, 0.525490, 0.486275],
[0.333333, 0.227451, 0.000000],
[0.580392, 0.988235, 1.000000],
[0.000000, 0.749020, 0.000000],
[0.490196, 0.000000, 0.627451],
[0.670588, 0.447059, 0.000000],
[0.568627, 1.000000, 0.000000],
[0.003922, 0.745098, 0.541176],
[0.000000, 0.270588, 0.482353],
[0.784314, 0.509804, 0.435294],
[1.000000, 0.121569, 0.513725],
[0.866667, 0.000000, 1.000000],
[0.019608, 0.454902, 0.000000],
[0.392157, 0.266667, 0.380392],
[0.533333, 0.560784, 1.000000],
[1.000000, 0.713725, 0.956863],
[0.325490, 0.384314, 0.215686],
[0.807843, 0.521569, 1.000000],
[0.407843, 0.415686, 0.517647],
[0.745098, 0.705882, 0.745098],
[0.647059, 0.376471, 0.537255],
[0.584314, 0.827451, 1.000000],
[0.003922, 0.000000, 0.972549],
[1.000000, 0.501961, 0.007843],
[0.545098, 0.160784, 0.270588],
[0.678431, 0.627451, 0.427451],
[0.325490, 0.270588, 0.545098],
[0.784314, 1.000000, 0.850980],
[0.666667, 0.274510, 0.000000],
[1.000000, 0.474510, 0.560784],
[0.513725, 0.827451, 0.443137],
[0.564706, 0.619608, 0.749020],
[0.580392, 0.000000, 0.960784],
[0.921569, 0.815686, 0.607843],
[0.678431, 0.545098, 0.694118],
[0.000000, 0.388235, 0.290196],
[1.000000, 0.862745, 0.000000],
[0.533333, 0.466667, 0.317647],
[0.494118, 0.670588, 0.639216],
[0.000000, 0.000000, 0.592157],
[0.960784, 0.000000, 0.776471],
[0.396078, 0.200000, 0.160784],
[0.000000, 0.400000, 0.470588],
[0.015686, 0.890196, 0.784314],
[0.654902, 0.215686, 0.682353],
[0.772549, 0.858824, 0.882353],
[0.301961, 0.431373, 1.000000],
[0.607843, 0.576471, 0.003922],
[0.803922, 0.345098, 0.419608],
[0.937255, 0.870588, 0.996078],
[0.474510, 0.352941, 0.000000],
[0.372549, 0.533333, 0.603922],
[0.705882, 1.000000, 0.572549],
[0.368627, 0.447059, 0.419608],
[0.321569, 0.000000, 0.400000],
[0.019608, 0.529412, 0.317647],
[0.517647, 0.125490, 0.435294],
[0.235294, 0.588235, 0.019608],
[0.396078, 0.450980, 0.000000],
[0.945098, 0.627451, 0.423529],
[0.372549, 0.313725, 0.270588],
[0.741176, 0.000000, 0.290196],
[0.815686, 0.407843, 0.152941],
[0.843137, 0.588235, 0.670588],
[0.537255, 0.364706, 1.000000],
[0.509804, 0.423529, 0.462745],
[0.168627, 0.333333, 0.725490],
[0.431373, 0.486275, 0.733333],
[0.905882, 0.835294, 0.827451],
[0.364706, 0.000000, 0.094118],
[0.486275, 0.231373, 0.003922],
[0.501961, 0.694118, 0.490196],
[0.784314, 0.850980, 0.490196],
[0.000000, 0.909804, 0.231373],
[0.486275, 0.698039, 1.000000],
[1.000000, 0.333333, 1.000000],
[0.643137, 0.152941, 0.129412],
[0.113725, 0.894118, 1.000000],
[0.490196, 0.686275, 0.231373],
[0.482353, 0.294118, 0.568627],
[0.878431, 1.000000, 0.282353],
[0.419608, 0.000000, 0.768627],
[0.803922, 0.658824, 0.592157],
[0.745098, 0.388235, 0.768627],
[0.537255, 0.803922, 0.807843],
[0.274510, 0.011765, 0.784314],
[0.368627, 0.572549, 0.474510],
[0.254902, 0.290196, 0.003922],
[0.019608, 0.654902, 0.615686],
[0.811765, 0.549020, 0.215686],
[1.000000, 0.972549, 0.815686],
[0.262745, 0.329412, 0.443137],
[0.709804, 0.266667, 1.000000],
[0.811765, 0.286275, 0.576471],
[0.811765, 0.643137, 0.874510],
[0.580392, 0.831373, 0.000000],
[0.654902, 0.580392, 0.854902],
[0.176471, 0.647059, 0.345098],
[0.552941, 0.890196, 0.713725],
[0.643137, 0.662745, 0.615686],
[0.423529, 0.360784, 0.717647],
[1.000000, 0.494118, 0.368627],
[0.654902, 0.513725, 0.541176],
[0.686275, 0.745098, 0.847059],
[0.164706, 0.768627, 1.000000],
[0.650980, 0.407843, 0.239216],
[0.964706, 0.568627, 0.996078],
[0.529412, 0.294118, 0.392157],
[1.000000, 0.047059, 0.294118],
[0.129412, 0.368627, 0.137255],
[0.258824, 0.572549, 1.000000],
[0.529412, 0.513725, 0.615686],
[0.403922, 0.176471, 0.270588],
[0.694118, 0.309804, 0.254902],
[0.000000, 0.305882, 0.325490],
[0.372549, 0.105882, 0.000000],
[0.678431, 0.254902, 0.403922],
[0.313725, 0.196078, 0.403922],
[0.839216, 1.000000, 0.992157],
[0.498039, 0.709804, 0.819608],
[0.662745, 0.725490, 0.411765],
[1.000000, 0.588235, 0.796078],
[0.784314, 0.454902, 0.584314],
[0.211765, 0.313725, 0.223529],
[1.000000, 0.815686, 0.388235],
[0.368627, 0.345098, 0.384314],
[0.529412, 0.580392, 0.462745],
[0.662745, 0.470588, 1.000000],
[0.011765, 0.784314, 0.388235],
[0.905882, 0.745098, 0.831373],
[0.831373, 0.890196, 0.815686],
[0.529412, 0.403922, 0.564706],
[0.537255, 0.486275, 0.152941],
[0.803922, 0.862745, 1.000000],
[0.666667, 0.403922, 0.419608],
[0.196078, 0.203922, 0.454902],
[1.000000, 0.368627, 0.662745],
[0.000000, 0.607843, 0.690196],
[0.443137, 1.000000, 0.866667],
[0.470588, 0.360784, 0.219608],
[0.313725, 0.396078, 0.607843],
[0.800000, 0.000000, 0.701961],
[0.341176, 0.482353, 0.333333],
[0.317647, 0.431373, 0.482353],
[0.003922, 0.372549, 0.572549],
[0.666667, 0.741176, 0.745098],
[0.003922, 0.498039, 0.600000],
[0.015686, 0.866667, 0.592157],
[0.529412, 0.227451, 0.172549],
[0.941176, 0.588235, 0.556863],
[0.458824, 0.776471, 0.666667],
[0.439216, 0.411765, 0.364706],
[0.800000, 0.862745, 0.035294],
[0.686275, 0.521569, 0.341176],
[0.847059, 0.000000, 0.458824],
[0.615686, 0.247059, 0.505882],
[0.850980, 0.270588, 0.000000],
[0.866667, 0.403922, 0.329412],
[0.372549, 1.000000, 0.474510],
[0.835294, 0.694118, 0.450980],
[0.384314, 0.149020, 0.368627],
[0.729412, 0.635294, 0.239216],
[0.850980, 0.949020, 0.701961],
[0.341176, 0.007843, 0.560784],
[0.631373, 0.607843, 0.666667],
[0.301961, 0.290196, 0.152941],
[0.643137, 0.662745, 1.000000],
[0.674510, 0.909804, 0.858824],
[0.600000, 0.349020, 0.003922],
[0.674510, 0.000000, 0.886275],
[0.278431, 0.509804, 0.184314],
[0.796078, 0.764706, 0.678431],
[0.000000, 0.772549, 0.713725],
[0.380392, 0.325490, 0.470588],
[0.200000, 0.427451, 0.407843],
[0.647059, 0.572549, 0.501961],
[0.517647, 0.600000, 0.635294],
[0.992157, 0.341176, 0.392157],
[0.439216, 0.588235, 0.823529],
[0.447059, 0.552941, 0.027451],
[0.498039, 0.000000, 0.298039],
[0.082353, 0.188235, 0.627451],
[0.819608, 0.756863, 0.886275],
[0.788235, 0.521569, 0.815686],
[0.423529, 0.270588, 0.294118],
[0.498039, 0.000000, 0.141176],
[0.000000, 0.635294, 0.474510],
[0.698039, 0.662745, 0.811765],
[0.976471, 0.000000, 0.000000],
[0.690196, 0.913725, 1.000000],
[0.576471, 0.619608, 0.313725],
[0.447059, 0.478431, 0.509804],
[0.850980, 0.180392, 0.333333],
[0.278431, 0.380392, 0.003922],
[0.000000, 0.349020, 1.000000],
[0.466667, 0.250980, 0.709804],
[0.674510, 0.894118, 0.376471],
[0.403922, 0.270588, 0.145098],
[0.321569, 0.364706, 0.317647],
[0.584314, 0.450980, 0.407843],
[0.662745, 0.894118, 0.603922],
[0.639216, 0.000000, 0.345098],
[0.850980, 0.384314, 0.964706],
[0.556863, 0.490196, 0.811765],
[1.000000, 0.741176, 0.576471],
[0.639216, 0.000000, 0.572549],
[0.603922, 1.000000, 0.725490],
[0.654902, 0.760784, 1.000000],
[0.956863, 0.384314, 0.000000],
[0.898039, 0.941176, 1.000000],
[0.721569, 0.611765, 0.643137],
[0.376471, 0.588235, 0.580392],
[1.000000, 0.623529, 0.207843],
[0.549020, 0.160784, 0.000000],
[0.447059, 0.419608, 0.196078],
[0.874510, 0.509804, 0.305882],
[0.686275, 0.482353, 0.835294],
[0.737255, 0.176471, 0.000000],
[0.482353, 0.435294, 0.639216],
[0.282353, 0.262745, 0.384314],
[0.780392, 0.639216, 1.000000],
[0.000000, 0.301961, 0.156863],
[0.768627, 0.776471, 0.556863],
[0.878431, 0.282353, 0.843137],
[0.905882, 0.913725, 0.396078],
[0.898039, 0.756863, 0.043137],
[0.000000, 0.956863, 0.945098],
[0.623529, 0.356863, 0.635294],
[0.298039, 0.254902, 0.717647],
[0.396078, 0.200000, 0.556863],
[0.462745, 0.494118, 0.423529],
[0.662745, 0.541176, 0.211765],
]
b_glasbey_hv = bokeh_palette('glasbey_hv',glasbey_hv)
m_glasbey_hv = mpl_cm('glasbey_hv',glasbey_hv)
m_glasbey_hv_r = mpl_cm('glasbey_hv_r',list(reversed(glasbey_hv)))




glasbey_bw_minc_20_hue_150_280 = [  # cmap_def
[0.050980, 0.415686, 1.000000],
[0.035294, 0.584314, 0.301961],
[0.000000, 0.792157, 0.898039],
[0.290196, 1.000000, 0.568627],
[0.000000, 0.345098, 0.376471],
[0.678431, 0.643137, 1.000000],
[0.458824, 0.447059, 0.560784],
[0.298039, 0.000000, 0.745098],
[0.607843, 0.733333, 0.623529],
[0.584314, 1.000000, 0.949020],
[0.258824, 0.560784, 0.568627],
[0.000000, 0.270588, 0.125490],
[0.737255, 0.858824, 1.000000],
[0.254902, 0.643137, 1.000000],
[0.305882, 0.419608, 0.325490],
[0.572549, 0.650980, 0.729412],
[0.027451, 0.313725, 0.588235],
[0.000000, 0.784314, 0.494118],
[0.525490, 0.435294, 0.980392],
[0.003922, 0.478431, 0.666667],
[0.380392, 0.329412, 0.682353],
[0.443137, 0.572549, 0.462745],
[0.780392, 0.913725, 0.792157],
[0.294118, 0.325490, 0.419608],
[0.000000, 0.858824, 0.760784],
[0.113725, 0.000000, 1.000000],
[0.603922, 0.803922, 0.815686],
[0.478431, 0.549020, 0.776471],
[0.329412, 0.690196, 0.603922],
[0.000000, 0.639216, 0.784314],
[0.313725, 0.431373, 0.466667],
[0.745098, 0.729412, 0.862745],
[0.482353, 0.768627, 0.996078],
[0.000000, 0.509804, 0.400000],
[0.513725, 0.870588, 0.662745],
[0.356863, 0.141176, 1.000000],
[0.007843, 0.396078, 0.196078],
[0.000000, 0.945098, 1.000000],
[0.823529, 0.964706, 0.996078],
[0.000000, 1.000000, 0.788235],
[0.439216, 0.537255, 0.611765],
[0.329412, 0.443137, 0.725490],
[0.180392, 0.321569, 0.270588],
[0.356863, 0.517647, 0.992157],
[0.407843, 0.686275, 0.725490],
[0.662745, 1.000000, 0.737255],
[0.352941, 0.662745, 0.439216],
[0.890196, 0.874510, 1.000000],
[0.172549, 0.380392, 0.501961],
[0.568627, 0.556863, 0.670588],
[0.556863, 0.662745, 0.886275],
[0.529412, 0.866667, 0.988235],
[0.015686, 0.545098, 0.866667],
[0.000000, 0.447059, 0.439216],
[0.223529, 0.341176, 0.768627],
[0.380392, 0.360784, 0.949020],
[0.462745, 0.607843, 0.592157],
[0.000000, 0.619608, 0.470588],
[0.349020, 0.494118, 0.454902],
[0.254902, 0.262745, 0.596078],
[0.686275, 0.760784, 1.000000],
[0.478431, 0.447059, 0.733333],
[0.623529, 0.756863, 0.862745],
[0.541176, 0.564706, 0.980392],
[0.286275, 0.360784, 0.568627],
[0.490196, 0.784314, 0.698039],
[0.011765, 0.725490, 0.956863],
[0.349020, 0.454902, 0.580392],
[0.423529, 0.615686, 0.768627],
[0.627451, 0.890196, 0.811765],
[0.015686, 0.403922, 0.317647],
[0.000000, 0.756863, 0.733333],
[0.309804, 0.509804, 0.356863],
[0.000000, 0.886275, 0.466667],
[0.000000, 0.498039, 0.568627],
[0.474510, 0.760784, 0.545098],
[0.647059, 0.631373, 0.811765],
[0.000000, 0.490196, 0.262745],
[0.796078, 1.000000, 0.933333],
[0.552941, 0.701961, 0.686275],
[0.270588, 0.545098, 0.666667],
[0.000000, 0.968627, 0.894118],
[0.000000, 0.654902, 0.662745],
[0.000000, 0.403922, 0.694118],
[0.376471, 0.843137, 0.854902],
[0.647059, 0.909804, 0.937255],
[0.003922, 0.866667, 0.639216],
[0.227451, 0.380392, 0.372549],
[0.380392, 0.368627, 0.517647],
[0.482353, 0.662745, 0.549020],
[0.454902, 0.709804, 0.839216],
[0.000000, 0.270588, 1.000000],
[0.313725, 0.294118, 0.494118],
[0.239216, 0.192157, 0.803922],
[0.000000, 0.776471, 0.623529],
[0.658824, 0.831373, 0.701961],
[0.333333, 0.576471, 0.505882],
[0.576471, 0.984314, 0.827451],
[0.341176, 0.521569, 0.733333],
[0.231373, 0.341176, 0.392157],
[0.380392, 0.600000, 0.658824],
[0.807843, 0.800000, 1.000000],
[0.588235, 0.560784, 0.835294],
[0.000000, 0.705882, 0.368627],
[0.729412, 0.843137, 0.894118],
[0.345098, 0.227451, 0.752941],
[0.000000, 0.705882, 0.501961],
[0.000000, 0.239216, 0.705882],
[0.000000, 0.337255, 0.215686],
[0.474510, 0.635294, 1.000000],
[0.415686, 0.454902, 0.858824],
[0.415686, 0.905882, 0.768627],
[0.494118, 0.784314, 0.862745],
[0.447059, 0.498039, 0.647059],
[0.000000, 0.419608, 0.486275],
[0.341176, 0.396078, 0.478431],
[0.431373, 0.964706, 0.690196],
[0.113725, 0.462745, 0.815686],
[0.364706, 0.505882, 0.533333],
[0.235294, 0.443137, 0.388235],
[0.000000, 0.709804, 0.788235],
[0.368627, 0.572549, 0.878431],
[0.400000, 0.427451, 0.619608],
[0.227451, 0.611765, 0.847059],
[0.211765, 0.368627, 0.250980],
[0.317647, 0.819608, 1.000000],
[0.443137, 0.760784, 0.752941],
[0.000000, 0.564706, 0.501961],
[0.419608, 0.686275, 0.925490],
[0.309804, 0.588235, 0.415686],
[0.752941, 0.905882, 0.886275],
[0.639216, 0.694118, 0.831373],
[0.411765, 0.278431, 0.874510],
[0.000000, 0.588235, 0.662745],
[0.443137, 0.380392, 0.803922],
[0.839216, 0.913725, 1.000000],
[0.549020, 0.603922, 0.756863],
[0.729412, 0.705882, 0.949020],
[0.749020, 0.796078, 0.894118],
[0.184314, 0.443137, 0.305882],
[0.556863, 0.713725, 0.764706],
[0.498039, 0.909804, 0.874510],
[0.227451, 0.447059, 0.549020],
[0.529412, 0.509804, 0.713725],
[0.403922, 0.835294, 0.533333],
[0.043137, 0.505882, 1.000000],
[0.513725, 0.501961, 0.854902],
[0.149020, 0.635294, 0.568627],
[0.631373, 0.784314, 0.733333],
[0.274510, 0.294118, 1.000000],
[0.243137, 0.419608, 0.603922],
[0.780392, 1.000000, 0.839216],
[0.223529, 0.498039, 0.482353],
[0.658824, 0.921569, 0.749020],
[0.003922, 0.345098, 0.313725],
[0.184314, 0.003922, 0.827451],
[0.501961, 0.972549, 1.000000],
[0.607843, 0.823529, 0.996078],
[0.733333, 0.913725, 1.000000],
[0.196078, 0.329412, 0.498039],
[0.000000, 0.874510, 0.992157],
[0.501961, 0.639216, 0.674510],
[0.062745, 0.556863, 0.376471],
[0.341176, 0.380392, 0.768627],
[0.000000, 0.647059, 0.400000],
[0.317647, 0.435294, 1.000000],
[0.305882, 0.501961, 0.843137],
[0.360784, 0.647059, 0.631373],
[0.568627, 0.729412, 1.000000],
[0.376471, 0.713725, 0.549020],
[0.031373, 0.925490, 0.603922],
[0.380392, 0.800000, 0.623529],
[0.333333, 0.643137, 0.509804],
[0.152941, 0.290196, 0.188235],
[0.733333, 1.000000, 1.000000],
[0.360784, 0.654902, 0.760784],
[0.000000, 0.545098, 0.752941],
[0.596078, 0.666667, 1.000000],
[0.486275, 0.600000, 0.686275],
[0.423529, 0.486275, 0.745098],
[0.156863, 0.576471, 0.996078],
[0.000000, 0.392157, 0.839216],
[0.007843, 0.345098, 0.439216],
[0.435294, 0.552941, 0.698039],
[0.000000, 0.301961, 0.784314],
[0.215686, 0.321569, 0.623529],
[0.525490, 0.600000, 0.882353],
[0.552941, 0.717647, 0.890196],
[0.000000, 0.368627, 0.576471],
[0.400000, 0.545098, 0.529412],
[0.341176, 0.000000, 0.890196],
[0.505882, 0.831373, 0.792157],
[0.478431, 0.894118, 0.945098],
[0.513725, 0.717647, 0.635294],
[0.329412, 0.756863, 0.682353],
[0.627451, 0.823529, 0.909804],
[0.247059, 0.152941, 1.000000],
[0.482353, 0.352941, 0.984314],
[0.321569, 0.494118, 0.580392],
[0.250980, 0.392157, 0.682353],
[0.403922, 1.000000, 0.878431],
[0.474510, 0.498039, 1.000000],
[0.270588, 0.423529, 0.847059],
[0.274510, 0.525490, 0.431373],
[0.552941, 0.784314, 0.643137],
[0.243137, 0.384314, 0.325490],
[0.392157, 0.513725, 0.419608],
[0.345098, 0.733333, 0.462745],
[0.207843, 0.454902, 0.490196],
[0.321569, 0.752941, 0.800000],
[0.450980, 0.623529, 0.862745],
[0.301961, 0.305882, 0.701961],
[0.247059, 0.470588, 0.698039],
[0.423529, 0.392157, 0.674510],
[0.031373, 0.698039, 0.647059],
[0.435294, 0.525490, 0.862745],
[0.525490, 0.654902, 0.796078],
[0.219608, 0.396078, 0.435294],
[0.274510, 0.368627, 0.494118],
[0.458824, 0.905882, 0.588235],
[0.419608, 0.478431, 0.560784],
[0.309804, 0.666667, 0.854902],
[0.266667, 0.729412, 0.862745],
[0.352941, 0.345098, 0.576471],
[0.443137, 0.568627, 0.596078],
[0.643137, 0.666667, 0.905882],
[0.682353, 0.788235, 0.945098],
[0.000000, 0.898039, 0.866667],
[0.509804, 0.501961, 0.623529],
[0.000000, 0.423529, 0.588235],
[0.662745, 0.776471, 0.823529],
[0.435294, 0.701961, 0.670588],
[0.501961, 0.549020, 0.654902],
[0.682353, 0.839216, 0.815686],
[0.294118, 0.196078, 0.639216],
[0.807843, 0.847059, 0.984314],
[0.721569, 0.941176, 0.843137],
[0.435294, 0.611765, 0.537255],
[0.329412, 0.286275, 0.596078],
[0.054902, 0.388235, 0.384314],
[0.254902, 0.305882, 0.521569],
[0.227451, 0.639216, 0.705882],
[0.533333, 0.847059, 0.725490],
[0.305882, 0.470588, 0.380392],
[0.572549, 0.501961, 1.000000],
[0.607843, 0.615686, 0.725490],
[0.145098, 1.000000, 0.996078],
[0.007843, 0.823529, 0.796078],
[0.345098, 0.392157, 0.650980],
[0.682353, 0.670588, 0.788235],
[0.662745, 0.937255, 0.901961],
[0.423529, 0.623529, 0.466667],
[0.325490, 0.313725, 0.847059],
[0.000000, 0.529412, 0.545098],
[0.435294, 0.572549, 1.000000],
[0.250980, 0.529412, 0.588235],
]
b_glasbey_bw_minc_20_hue_150_280 = bokeh_palette('glasbey_bw_minc_20_hue_150_280',glasbey_bw_minc_20_hue_150_280)
m_glasbey_bw_minc_20_hue_150_280 = mpl_cm('glasbey_bw_minc_20_hue_150_280',glasbey_bw_minc_20_hue_150_280)
m_glasbey_bw_minc_20_hue_150_280_r = mpl_cm('glasbey_bw_minc_20_hue_150_280_r',list(reversed(glasbey_bw_minc_20_hue_150_280)))
glasbey_cool = b_glasbey_bw_minc_20_hue_150_280
m_glasbey_cool = m_glasbey_bw_minc_20_hue_150_280
m_glasbey_cool_r = m_glasbey_bw_minc_20_hue_150_280_r
palette['glasbey_cool'] = b_glasbey_bw_minc_20_hue_150_280
palette_n['glasbey_cool'] = b_glasbey_bw_minc_20_hue_150_280
cm['glasbey_cool'] = m_glasbey_bw_minc_20_hue_150_280
cm['glasbey_cool_r'] = m_glasbey_bw_minc_20_hue_150_280_r
cm_n['glasbey_cool'] = mpl_cm('glasbey_cool',glasbey_bw_minc_20_hue_150_280)
cm_n['glasbey_cool_r'] = mpl_cm('glasbey_cool_r',list(reversed(glasbey_bw_minc_20_hue_150_280)))




circle_mgbm_67_c31_s25 = [  # cmap_def
[0.67345, 0.60541, 0.82378],
[0.68002, 0.6036, 0.82081],
[0.68649, 0.60181, 0.81769],
[0.69285, 0.60004, 0.81442],
[0.69908, 0.5983, 0.81102],
[0.70519, 0.5966, 0.80747],
[0.71118, 0.59492, 0.80379],
[0.71703, 0.59329, 0.79998],
[0.72274, 0.59168, 0.79606],
[0.72832, 0.59011, 0.79203],
[0.73376, 0.58858, 0.7879],
[0.73907, 0.58709, 0.78369],
[0.74423, 0.58565, 0.77937],
[0.74924, 0.58424, 0.77498],
[0.75413, 0.58287, 0.77052],
[0.75888, 0.58155, 0.76599],
[0.76349, 0.58027, 0.76138],
[0.76797, 0.57902, 0.75673],
[0.77232, 0.57781, 0.75202],
[0.77655, 0.57666, 0.74727],
[0.78064, 0.57553, 0.74247],
[0.78462, 0.57444, 0.73762],
[0.78848, 0.5734, 0.73275],
[0.79223, 0.57239, 0.72783],
[0.79586, 0.57141, 0.72289],
[0.79939, 0.57047, 0.71793],
[0.80281, 0.56956, 0.71293],
[0.80613, 0.56869, 0.70792],
[0.80935, 0.56786, 0.70288],
[0.81246, 0.56706, 0.69782],
[0.81549, 0.56628, 0.69275],
[0.81843, 0.56553, 0.68766],
[0.82128, 0.56482, 0.68256],
[0.82405, 0.56415, 0.67746],
[0.82672, 0.5635, 0.67233],
[0.82931, 0.56288, 0.66719],
[0.83181, 0.5623, 0.66204],
[0.83421, 0.56174, 0.65688],
[0.83652, 0.56122, 0.65171],
[0.83874, 0.56074, 0.64653],
[0.84085, 0.56031, 0.64133],
[0.84286, 0.55992, 0.63614],
[0.84476, 0.55958, 0.63094],
[0.84654, 0.55927, 0.62572],
[0.84823, 0.55902, 0.62051],
[0.84979, 0.55882, 0.61529],
[0.85123, 0.55867, 0.61007],
[0.85256, 0.55857, 0.60486],
[0.85375, 0.55854, 0.59965],
[0.85481, 0.55856, 0.59444],
[0.85574, 0.55865, 0.58925],
[0.85653, 0.5588, 0.58408],
[0.85717, 0.55901, 0.57892],
[0.85767, 0.5593, 0.57378],
[0.85801, 0.55965, 0.56867],
[0.8582, 0.56007, 0.56361],
[0.85822, 0.56057, 0.55856],
[0.85808, 0.56114, 0.55358],
[0.85777, 0.56179, 0.54866],
[0.8573, 0.56251, 0.54377],
[0.85666, 0.5633, 0.53896],
[0.85585, 0.56416, 0.53421],
[0.85487, 0.56509, 0.52955],
[0.85372, 0.5661, 0.52497],
[0.85258, 0.56704, 0.52104],
[0.8511, 0.56817, 0.51664],
[0.84946, 0.56937, 0.51234],
[0.84766, 0.57064, 0.50816],
[0.84571, 0.57196, 0.50407],
[0.8436, 0.57334, 0.50009],
[0.84135, 0.57476, 0.49623],
[0.83896, 0.57624, 0.4925],
[0.83643, 0.57775, 0.48888],
[0.83377, 0.57932, 0.4854],
[0.83098, 0.5809, 0.48204],
[0.82807, 0.58253, 0.47879],
[0.82505, 0.58418, 0.47567],
[0.82193, 0.58587, 0.47266],
[0.8187, 0.58756, 0.46978],
[0.81539, 0.58928, 0.467],
[0.81198, 0.59102, 0.46435],
[0.80848, 0.59278, 0.46179],
[0.8049, 0.59454, 0.45935],
[0.80125, 0.59631, 0.457],
[0.79752, 0.59808, 0.45477],
[0.79374, 0.59987, 0.45264],
[0.78987, 0.60165, 0.45059],
[0.78595, 0.60344, 0.44864],
[0.78197, 0.60522, 0.44677],
[0.77793, 0.60702, 0.44501],
[0.77383, 0.60881, 0.44332],
[0.76968, 0.61058, 0.44172],
[0.76549, 0.61235, 0.44019],
[0.76119, 0.61414, 0.43872],
[0.75677, 0.61597, 0.4373],
[0.75316, 0.61743, 0.4362],
[0.748, 0.61951, 0.43474],
[0.7438, 0.62116, 0.43364],
[0.73934, 0.6229, 0.43255],
[0.73482, 0.62463, 0.43152],
[0.73026, 0.62635, 0.43056],
[0.72565, 0.62807, 0.42968],
[0.72099, 0.62977, 0.4289],
[0.71629, 0.63148, 0.4282],
[0.71153, 0.63317, 0.42759],
[0.70673, 0.63485, 0.42707],
[0.70186, 0.63653, 0.42665],
[0.69695, 0.63819, 0.42633],
[0.69199, 0.63985, 0.42612],
[0.68697, 0.64149, 0.42602],
[0.68189, 0.64313, 0.42604],
[0.67677, 0.64476, 0.42619],
[0.67158, 0.64637, 0.42646],
[0.66634, 0.64797, 0.42688],
[0.66105, 0.64957, 0.42743],
[0.65571, 0.65114, 0.42812],
[0.65031, 0.65269, 0.42895],
[0.64486, 0.65424, 0.42994],
[0.63937, 0.65577, 0.4311],
[0.63382, 0.65728, 0.43241],
[0.62823, 0.65875, 0.43386],
[0.6226, 0.66022, 0.43552],
[0.61693, 0.66167, 0.43733],
[0.61123, 0.66309, 0.43931],
[0.60549, 0.66448, 0.44148],
[0.59974, 0.66585, 0.44382],
[0.59396, 0.66719, 0.44632],
[0.58816, 0.6685, 0.449],
[0.58235, 0.66978, 0.45184],
[0.57655, 0.67102, 0.45485],
[0.57073, 0.67224, 0.45801],
[0.56493, 0.67342, 0.46134],
[0.55914, 0.67456, 0.46481],
[0.55336, 0.67568, 0.46842],
[0.5476, 0.67676, 0.47217],
[0.54185, 0.6778, 0.47604],
[0.53612, 0.67881, 0.48003],
[0.53042, 0.67978, 0.48412],
[0.52475, 0.68072, 0.48834],
[0.5191, 0.68163, 0.49263],
[0.51348, 0.68251, 0.49703],
[0.50789, 0.68336, 0.50152],
[0.50231, 0.68416, 0.50606],
[0.49676, 0.68495, 0.51069],
[0.49125, 0.68571, 0.51537],
[0.48573, 0.68643, 0.52011],
[0.48024, 0.68714, 0.52492],
[0.47477, 0.68781, 0.52977],
[0.46931, 0.68846, 0.53465],
[0.46384, 0.68909, 0.53958],
[0.45839, 0.68969, 0.54455],
[0.45295, 0.69026, 0.54956],
[0.44749, 0.69082, 0.55459],
[0.44206, 0.69136, 0.55965],
[0.43659, 0.69188, 0.56472],
[0.43111, 0.69239, 0.56983],
[0.4256, 0.69287, 0.57495],
[0.42007, 0.69334, 0.5801],
[0.41451, 0.69378, 0.58527],
[0.40893, 0.69421, 0.59044],
[0.40332, 0.69463, 0.59562],
[0.39766, 0.69504, 0.60082],
[0.39195, 0.69544, 0.60603],
[0.38619, 0.69581, 0.61126],
[0.38043, 0.69617, 0.6165],
[0.37463, 0.69651, 0.62174],
[0.3688, 0.69683, 0.62699],
[0.36293, 0.69714, 0.63226],
[0.35707, 0.69743, 0.63753],
[0.35119, 0.69769, 0.64282],
[0.3453, 0.69794, 0.6481],
[0.33944, 0.69817, 0.65339],
[0.33358, 0.69837, 0.65868],
[0.32776, 0.69855, 0.66399],
[0.322, 0.6987, 0.66928],
[0.31632, 0.69883, 0.67456],
[0.31072, 0.69893, 0.67984],
[0.30523, 0.699, 0.68512],
[0.29987, 0.69904, 0.69038],
[0.29467, 0.69905, 0.69563],
[0.28968, 0.69903, 0.70084],
[0.28489, 0.69897, 0.70604],
[0.28037, 0.69887, 0.7112],
[0.27617, 0.69874, 0.71632],
[0.27224, 0.69857, 0.7214],
[0.26874, 0.69837, 0.72643],
[0.2656, 0.69812, 0.7314],
[0.26291, 0.69783, 0.73632],
[0.2607, 0.6975, 0.74116],
[0.25906, 0.69712, 0.74592],
[0.25793, 0.69671, 0.7506],
[0.25738, 0.69625, 0.75518],
[0.2574, 0.69574, 0.75967],
[0.25799, 0.69518, 0.76406],
[0.2592, 0.69459, 0.76833],
[0.26094, 0.69395, 0.7725],
[0.26333, 0.69327, 0.77655],
[0.26626, 0.69255, 0.78046],
[0.2697, 0.69178, 0.78426],
[0.27363, 0.69097, 0.78793],
[0.27801, 0.69013, 0.79147],
[0.28282, 0.68925, 0.79489],
[0.288, 0.68833, 0.79818],
[0.29353, 0.68738, 0.80134],
[0.2994, 0.68639, 0.80439],
[0.30554, 0.68537, 0.8073],
[0.31189, 0.68431, 0.81009],
[0.3185, 0.68324, 0.81276],
[0.32527, 0.68212, 0.81532],
[0.33223, 0.68098, 0.81775],
[0.33931, 0.67981, 0.82008],
[0.34648, 0.67862, 0.82231],
[0.35375, 0.67741, 0.82442],
[0.3611, 0.67616, 0.82644],
[0.36852, 0.67489, 0.82836],
[0.37597, 0.67361, 0.83019],
[0.38345, 0.6723, 0.83193],
[0.39094, 0.67096, 0.83357],
[0.39844, 0.66961, 0.83514],
[0.40595, 0.66824, 0.83662],
[0.41345, 0.66684, 0.83803],
[0.42095, 0.66543, 0.83935],
[0.42845, 0.664, 0.84061],
[0.43591, 0.66255, 0.84179],
[0.44333, 0.66107, 0.8429],
[0.45073, 0.65959, 0.84395],
[0.4581, 0.65809, 0.84493],
[0.46547, 0.65657, 0.84583],
[0.47282, 0.65502, 0.84666],
[0.48015, 0.65346, 0.84742],
[0.48747, 0.65188, 0.84809],
[0.49477, 0.65028, 0.84868],
[0.50206, 0.64866, 0.84919],
[0.50935, 0.64702, 0.8496],
[0.51663, 0.64536, 0.84992],
[0.52391, 0.64368, 0.85013],
[0.53119, 0.64199, 0.85023],
[0.53848, 0.64028, 0.85023],
[0.54575, 0.63854, 0.85012],
[0.55302, 0.63679, 0.84989],
[0.56029, 0.63501, 0.84953],
[0.56756, 0.63323, 0.84905],
[0.57481, 0.63143, 0.84843],
[0.58206, 0.6296, 0.84768],
[0.5893, 0.62777, 0.84679],
[0.59654, 0.62592, 0.84575],
[0.60375, 0.62407, 0.84456],
[0.61093, 0.6222, 0.84321],
[0.61809, 0.62034, 0.84171],
[0.62521, 0.61846, 0.84004],
[0.63229, 0.61658, 0.83821],
[0.63932, 0.61469, 0.83622],
[0.64629, 0.61281, 0.83407],
[0.65319, 0.61095, 0.83175],
[0.66003, 0.6091, 0.82926],
[0.66678, 0.60725, 0.8266],
]
b_circle_mgbm_67_c31_s25 = bokeh_palette('circle_mgbm_67_c31_s25',circle_mgbm_67_c31_s25)
m_circle_mgbm_67_c31_s25 = mpl_cm('circle_mgbm_67_c31_s25',circle_mgbm_67_c31_s25)
m_circle_mgbm_67_c31_s25_r = mpl_cm('circle_mgbm_67_c31_s25_r',list(reversed(circle_mgbm_67_c31_s25)))
CET_C10s = b_circle_mgbm_67_c31_s25
m_CET_C10s = m_circle_mgbm_67_c31_s25
m_CET_C10s_r = m_circle_mgbm_67_c31_s25_r
palette['CET_C10s'] = b_circle_mgbm_67_c31_s25
cm['CET_C10s'] = m_circle_mgbm_67_c31_s25
cm['CET_C10s_r'] = m_circle_mgbm_67_c31_s25_r
register_cmap('cet_CET_C10s',m_circle_mgbm_67_c31_s25)
register_cmap('cet_CET_C10s_r',m_circle_mgbm_67_c31_s25_r)




linear_kbgyw_10_98_c63 = [  # cmap_def
[0.10767, 0.1077, 0.10769],
[0.11459, 0.10755, 0.1249],
[0.12094, 0.10731, 0.14203],
[0.12675, 0.107, 0.15907],
[0.13204, 0.10662, 0.17602],
[0.13678, 0.10617, 0.19288],
[0.1411, 0.10565, 0.20966],
[0.14494, 0.10507, 0.2264],
[0.14831, 0.10448, 0.24301],
[0.15129, 0.10393, 0.2595],
[0.15385, 0.10333, 0.27588],
[0.15607, 0.10267, 0.29214],
[0.15784, 0.102, 0.3083],
[0.15925, 0.10137, 0.32426],
[0.16027, 0.10076, 0.34007],
[0.161, 0.10016, 0.35572],
[0.16135, 0.099599, 0.3712],
[0.16134, 0.099079, 0.3865],
[0.16098, 0.098654, 0.40155],
[0.16027, 0.098298, 0.4164],
[0.1593, 0.09801, 0.43103],
[0.15799, 0.0978, 0.44544],
[0.15637, 0.097689, 0.45963],
[0.15441, 0.097743, 0.47347],
[0.15217, 0.09793, 0.48708],
[0.14969, 0.098248, 0.5004],
[0.14683, 0.098705, 0.51345],
[0.14375, 0.099335, 0.52621],
[0.14038, 0.10016, 0.53865],
[0.13675, 0.1012, 0.55071],
[0.13289, 0.10246, 0.56244],
[0.12874, 0.10392, 0.57382],
[0.12437, 0.10558, 0.58486],
[0.11978, 0.10747, 0.59553],
[0.11494, 0.10961, 0.60582],
[0.10999, 0.11191, 0.61573],
[0.10474, 0.11444, 0.62518],
[0.099437, 0.11728, 0.63421],
[0.09398, 0.12032, 0.6428],
[0.088391, 0.12358, 0.65096],
[0.082705, 0.12707, 0.65866],
[0.076932, 0.13083, 0.6659],
[0.071296, 0.13475, 0.67266],
[0.06558, 0.13885, 0.67892],
[0.060096, 0.1432, 0.68468],
[0.054684, 0.14774, 0.68991],
[0.049724, 0.15244, 0.6946],
[0.04515, 0.15736, 0.69873],
[0.04121, 0.1625, 0.70228],
[0.03795, 0.16777, 0.70525],
[0.035657, 0.17322, 0.70763],
[0.033944, 0.17882, 0.70945],
[0.032827, 0.18442, 0.71103],
[0.031776, 0.18998, 0.71249],
[0.030819, 0.19551, 0.71385],
[0.029941, 0.201, 0.71511],
[0.029145, 0.20651, 0.71626],
[0.028432, 0.21195, 0.7173],
[0.027801, 0.21736, 0.71822],
[0.027251, 0.22277, 0.71902],
[0.02679, 0.22816, 0.71971],
[0.026417, 0.2335, 0.72026],
[0.026123, 0.23887, 0.72069],
[0.025906, 0.24421, 0.72099],
[0.025762, 0.24956, 0.72115],
[0.025686, 0.25492, 0.72117],
[0.025675, 0.26024, 0.72106],
[0.025721, 0.26556, 0.72079],
[0.025819, 0.2709, 0.72038],
[0.025961, 0.27623, 0.71982],
[0.026176, 0.28155, 0.71907],
[0.026426, 0.28687, 0.71816],
[0.026693, 0.29222, 0.71709],
[0.026965, 0.29757, 0.71583],
[0.027228, 0.30291, 0.71439],
[0.027506, 0.30829, 0.71274],
[0.027765, 0.31367, 0.71091],
[0.027966, 0.31905, 0.70887],
[0.028088, 0.32444, 0.70662],
[0.028164, 0.32988, 0.70412],
[0.028129, 0.33531, 0.7014],
[0.02794, 0.34075, 0.69845],
[0.027618, 0.34621, 0.69524],
[0.027101, 0.3517, 0.69176],
[0.026337, 0.35723, 0.68802],
[0.025343, 0.36275, 0.68397],
[0.024014, 0.36832, 0.67963],
[0.022532, 0.37391, 0.67496],
[0.022085, 0.37945, 0.67002],
[0.023618, 0.38493, 0.66481],
[0.027119, 0.39034, 0.65936],
[0.03218, 0.3957, 0.65369],
[0.039183, 0.401, 0.64779],
[0.046906, 0.40623, 0.64167],
[0.054737, 0.41142, 0.63537],
[0.062911, 0.41657, 0.62886],
[0.070851, 0.42167, 0.62215],
[0.078562, 0.42675, 0.61527],
[0.086118, 0.43181, 0.60823],
[0.09353, 0.43681, 0.60099],
[0.10065, 0.4418, 0.5936],
[0.10748, 0.44675, 0.58606],
[0.11398, 0.4517, 0.57835],
[0.12022, 0.45661, 0.5705],
[0.12626, 0.46151, 0.5625],
[0.13197, 0.46639, 0.55434],
[0.1374, 0.47126, 0.54605],
[0.14253, 0.47611, 0.53761],
[0.14739, 0.48095, 0.52904],
[0.15195, 0.48578, 0.52032],
[0.1563, 0.49061, 0.51148],
[0.1603, 0.49542, 0.5025],
[0.16415, 0.50022, 0.49338],
[0.16765, 0.50502, 0.48413],
[0.17092, 0.50981, 0.47475],
[0.17391, 0.5146, 0.46521],
[0.17664, 0.51938, 0.45555],
[0.17913, 0.52416, 0.44574],
[0.18131, 0.52894, 0.4358],
[0.18326, 0.5337, 0.42569],
[0.18497, 0.53848, 0.41543],
[0.18637, 0.54325, 0.40502],
[0.18753, 0.54801, 0.39444],
[0.18843, 0.55277, 0.38372],
[0.18924, 0.55751, 0.373],
[0.19026, 0.56219, 0.36258],
[0.19155, 0.56678, 0.35246],
[0.19313, 0.57131, 0.34263],
[0.19502, 0.57579, 0.33309],
[0.19724, 0.58019, 0.32381],
[0.19974, 0.58454, 0.31477],
[0.20257, 0.58883, 0.30599],
[0.2057, 0.59309, 0.29739],
[0.2091, 0.59729, 0.28898],
[0.21279, 0.60144, 0.28074],
[0.21679, 0.60554, 0.2727],
[0.22112, 0.6096, 0.26485],
[0.22573, 0.61361, 0.25718],
[0.23057, 0.61759, 0.24961],
[0.23568, 0.62152, 0.24219],
[0.24099, 0.62542, 0.23491],
[0.24661, 0.62927, 0.22774],
[0.25248, 0.63309, 0.22073],
[0.25852, 0.63687, 0.21376],
[0.26474, 0.64061, 0.20691],
[0.27119, 0.64431, 0.20012],
[0.27787, 0.64797, 0.19352],
[0.28468, 0.65162, 0.18693],
[0.29167, 0.65521, 0.18038],
[0.29881, 0.65878, 0.17396],
[0.30616, 0.66232, 0.16762],
[0.31359, 0.66581, 0.16131],
[0.32116, 0.66929, 0.155],
[0.32891, 0.67272, 0.14881],
[0.33675, 0.67611, 0.14267],
[0.3447, 0.67948, 0.13649],
[0.3528, 0.68282, 0.13048],
[0.36098, 0.68612, 0.12438],
[0.36927, 0.6894, 0.11838],
[0.37766, 0.69265, 0.11238],
[0.38616, 0.69585, 0.10644],
[0.39472, 0.69903, 0.1004],
[0.4034, 0.70217, 0.094522],
[0.41216, 0.70529, 0.088596],
[0.42097, 0.70837, 0.08266],
[0.42989, 0.71143, 0.076741],
[0.43887, 0.71444, 0.070937],
[0.44792, 0.71744, 0.065037],
[0.45706, 0.72039, 0.05931],
[0.46625, 0.72331, 0.053366],
[0.47551, 0.72621, 0.047605],
[0.48484, 0.72907, 0.042087],
[0.49423, 0.7319, 0.036467],
[0.50368, 0.7347, 0.031269],
[0.51319, 0.73746, 0.026763],
[0.52275, 0.74019, 0.022795],
[0.53239, 0.74289, 0.019434],
[0.54205, 0.74555, 0.016511],
[0.5518, 0.74819, 0.014179],
[0.56158, 0.75079, 0.012354],
[0.57141, 0.75335, 0.010861],
[0.58128, 0.75589, 0.010058],
[0.59112, 0.75841, 0.00961],
[0.60091, 0.76092, 0.009424],
[0.61061, 0.76343, 0.009392],
[0.62023, 0.76594, 0.009555],
[0.62974, 0.76845, 0.009921],
[0.63918, 0.77096, 0.010449],
[0.64854, 0.77347, 0.011384],
[0.65782, 0.77599, 0.012461],
[0.66702, 0.77851, 0.013656],
[0.67614, 0.78103, 0.015136],
[0.68519, 0.78356, 0.016841],
[0.69415, 0.78609, 0.018809],
[0.70304, 0.78863, 0.021091],
[0.71186, 0.79117, 0.023623],
[0.7206, 0.79372, 0.026458],
[0.72926, 0.79627, 0.029655],
[0.73785, 0.79884, 0.033158],
[0.74637, 0.80141, 0.037185],
[0.7548, 0.804, 0.04144],
[0.76315, 0.80659, 0.045703],
[0.77144, 0.8092, 0.050197],
[0.77963, 0.81181, 0.054739],
[0.78774, 0.81445, 0.05955],
[0.79576, 0.8171, 0.064271],
[0.80372, 0.81976, 0.069198],
[0.81158, 0.82245, 0.074106],
[0.81932, 0.82515, 0.079138],
[0.82699, 0.82788, 0.084403],
[0.83456, 0.83063, 0.089647],
[0.84203, 0.8334, 0.095041],
[0.8494, 0.83619, 0.10046],
[0.85665, 0.83902, 0.10612],
[0.86378, 0.84188, 0.11185],
[0.87078, 0.84476, 0.11768],
[0.87767, 0.84769, 0.12364],
[0.88442, 0.85064, 0.12982],
[0.89102, 0.85364, 0.13603],
[0.89748, 0.85669, 0.14249],
[0.90377, 0.85979, 0.14907],
[0.90988, 0.86294, 0.15586],
[0.9158, 0.86613, 0.16282],
[0.92153, 0.8694, 0.16999],
[0.92704, 0.87272, 0.17739],
[0.93231, 0.87612, 0.18503],
[0.93732, 0.8796, 0.19292],
[0.94205, 0.88317, 0.20109],
[0.94642, 0.88684, 0.20966],
[0.95044, 0.89062, 0.21859],
[0.95407, 0.89452, 0.22788],
[0.95722, 0.89856, 0.23768],
[0.95989, 0.90275, 0.24812],
[0.96239, 0.90696, 0.25972],
[0.96496, 0.9111, 0.27302],
[0.96762, 0.91517, 0.28779],
[0.97033, 0.91916, 0.30418],
[0.97312, 0.92308, 0.32203],
[0.97594, 0.92691, 0.34135],
[0.97881, 0.93067, 0.36224],
[0.98168, 0.93433, 0.38454],
[0.98453, 0.93791, 0.40845],
[0.98734, 0.94139, 0.43399],
[0.99005, 0.94479, 0.46118],
[0.99262, 0.94808, 0.4901],
[0.995, 0.95128, 0.52094],
[0.9971, 0.95437, 0.55384],
[0.99884, 0.95735, 0.5889],
[1, 0.96022, 0.62632],
[1, 0.96299, 0.66629],
[1, 0.96564, 0.7091],
[0.99937, 0.96819, 0.75498],
[0.99686, 0.97063, 0.80429],
[0.99268, 0.97297, 0.85751],
[0.98632, 0.9752, 0.91507],
[0.97718, 0.97737, 0.97732],
]
b_linear_kbgyw_10_98_c63 = bokeh_palette('linear_kbgyw_10_98_c63',linear_kbgyw_10_98_c63)
m_linear_kbgyw_10_98_c63 = mpl_cm('linear_kbgyw_10_98_c63',linear_kbgyw_10_98_c63)
m_linear_kbgyw_10_98_c63_r = mpl_cm('linear_kbgyw_10_98_c63_r',list(reversed(linear_kbgyw_10_98_c63)))
kbgyw = b_linear_kbgyw_10_98_c63
m_kbgyw = m_linear_kbgyw_10_98_c63
m_kbgyw_r = m_linear_kbgyw_10_98_c63_r
palette['kbgyw'] = b_linear_kbgyw_10_98_c63
palette_n['kbgyw'] = b_linear_kbgyw_10_98_c63
cm['kbgyw'] = m_linear_kbgyw_10_98_c63
cm['kbgyw_r'] = m_linear_kbgyw_10_98_c63_r
cm_n['kbgyw'] = mpl_cm('kbgyw',linear_kbgyw_10_98_c63)
cm_n['kbgyw_r'] = mpl_cm('kbgyw_r',list(reversed(linear_kbgyw_10_98_c63)))




cyclic_ymcgy_60_90_c67_s25 = [  # cmap_def
[0.15215, 0.68561, 0.078354],
[0.16719, 0.68634, 0.062319],
[0.18301, 0.68773, 0.046282],
[0.19934, 0.6897, 0.030835],
[0.21593, 0.69218, 0.018538],
[0.23246, 0.69508, 0.008874],
[0.24886, 0.69831, 0.001713],
[0.26497, 0.70181, 0],
[0.28077, 0.7055, 0],
[0.29623, 0.70932, 0],
[0.3113, 0.71322, 0],
[0.326, 0.71718, 0],
[0.34038, 0.72116, 0],
[0.35442, 0.72518, 0],
[0.36816, 0.72918, 0],
[0.38163, 0.7332, 0],
[0.39485, 0.7372, 0],
[0.40784, 0.74119, 0],
[0.42063, 0.74518, 0],
[0.43324, 0.74916, 0],
[0.44569, 0.75313, 0],
[0.45799, 0.7571, 0],
[0.47015, 0.76106, 0],
[0.48219, 0.76501, 0],
[0.49411, 0.76894, 0],
[0.50591, 0.77288, 0],
[0.51764, 0.77681, 0],
[0.52927, 0.78073, 0],
[0.5408, 0.78464, 0],
[0.55228, 0.78855, 0],
[0.56368, 0.79244, 0],
[0.575, 0.79633, 0],
[0.58627, 0.80022, 0],
[0.59749, 0.80409, 0],
[0.60865, 0.80795, 0],
[0.61976, 0.81181, 0],
[0.63083, 0.81567, 0],
[0.64185, 0.81951, 0],
[0.65283, 0.82335, 0],
[0.66378, 0.82718, 0],
[0.67469, 0.83101, 0],
[0.68557, 0.83482, 0],
[0.69642, 0.83863, 0],
[0.70724, 0.84243, 0],
[0.71804, 0.84622, 0],
[0.72881, 0.85001, 0],
[0.73956, 0.85379, 0],
[0.75028, 0.85756, 0],
[0.76099, 0.86133, 0],
[0.77167, 0.86508, 0],
[0.78232, 0.86883, 0],
[0.79293, 0.87253, 0],
[0.8035, 0.87619, 0],
[0.814, 0.87977, 0],
[0.82441, 0.88325, 0],
[0.8347, 0.88658, 0],
[0.84482, 0.88967, 0],
[0.85473, 0.89247, 0],
[0.86437, 0.89485, 0],
[0.87367, 0.89671, 0],
[0.88259, 0.89793, 0],
[0.89108, 0.8984, 0.01207],
[0.8991, 0.89801, 0.041917],
[0.90662, 0.89668, 0.071606],
[0.91364, 0.89436, 0.098235],
[0.92016, 0.89105, 0.12325],
[0.92621, 0.88677, 0.14733],
[0.93184, 0.88159, 0.17055],
[0.93708, 0.87559, 0.19302],
[0.94198, 0.86888, 0.21474],
[0.94659, 0.86158, 0.23574],
[0.95094, 0.8538, 0.25594],
[0.95507, 0.84565, 0.27543],
[0.95901, 0.83722, 0.29417],
[0.96279, 0.82857, 0.31222],
[0.96641, 0.81976, 0.32966],
[0.9699, 0.81085, 0.34647],
[0.97326, 0.80183, 0.36274],
[0.97649, 0.79275, 0.37853],
[0.9796, 0.78361, 0.39387],
[0.9826, 0.77441, 0.40882],
[0.98549, 0.76518, 0.4234],
[0.98826, 0.75588, 0.43769],
[0.99092, 0.74654, 0.45167],
[0.99347, 0.73715, 0.46538],
[0.9959, 0.72769, 0.47889],
[0.99823, 0.71819, 0.49218],
[1, 0.70862, 0.50527],
[1, 0.69898, 0.51821],
[1, 0.68929, 0.53098],
[1, 0.67952, 0.54361],
[1, 0.66969, 0.5561],
[1, 0.65977, 0.56849],
[1, 0.64977, 0.58076],
[1, 0.63969, 0.59293],
[1, 0.62952, 0.605],
[1, 0.61926, 0.61699],
[1, 0.6089, 0.62889],
[1, 0.59842, 0.64072],
[1, 0.58784, 0.65249],
[1, 0.57715, 0.66419],
[1, 0.56633, 0.67584],
[1, 0.55536, 0.68743],
[1, 0.54427, 0.69897],
[1, 0.53302, 0.71047],
[1, 0.5216, 0.72192],
[1, 0.51002, 0.73334],
[1, 0.49826, 0.74471],
[1, 0.48626, 0.75606],
[1, 0.47407, 0.76737],
[1, 0.46163, 0.77866],
[1, 0.44893, 0.78992],
[1, 0.43595, 0.80115],
[1, 0.42263, 0.81238],
[1, 0.409, 0.82359],
[1, 0.39503, 0.83479],
[1, 0.38072, 0.84596],
[1, 0.36605, 0.85713],
[1, 0.35109, 0.86827],
[1, 0.3359, 0.87936],
[1, 0.32065, 0.89038],
[1, 0.30555, 0.90129],
[1, 0.29093, 0.91203],
[1, 0.2773, 0.9225],
[1, 0.26521, 0.9326],
[0.99941, 0.25538, 0.94224],
[0.99574, 0.24849, 0.95125],
[0.99172, 0.24513, 0.95954],
[0.98735, 0.24573, 0.967],
[0.98268, 0.25032, 0.97354],
[0.97773, 0.25863, 0.97914],
[0.97256, 0.27007, 0.9838],
[0.96718, 0.28393, 0.98756],
[0.96164, 0.29956, 0.99049],
[0.95598, 0.31629, 0.99271],
[0.95017, 0.33361, 0.99432],
[0.94427, 0.35113, 0.99545],
[0.93826, 0.36855, 0.99621],
[0.93214, 0.38569, 0.99669],
[0.92593, 0.40246, 0.99697],
[0.91961, 0.41878, 0.99713],
[0.91319, 0.43464, 0.99719],
[0.90666, 0.45004, 0.99719],
[0.90002, 0.46499, 0.99716],
[0.89326, 0.47954, 0.9971],
[0.8864, 0.49368, 0.99703],
[0.8794, 0.50748, 0.99695],
[0.87228, 0.52093, 0.99687],
[0.86502, 0.53409, 0.99678],
[0.85763, 0.54698, 0.99669],
[0.8501, 0.55961, 0.99659],
[0.84243, 0.572, 0.99648],
[0.8346, 0.58417, 0.99637],
[0.82662, 0.59614, 0.99625],
[0.81847, 0.60793, 0.99613],
[0.81015, 0.61953, 0.996],
[0.80166, 0.63097, 0.99587],
[0.79299, 0.64224, 0.99572],
[0.78413, 0.65337, 0.99558],
[0.77506, 0.66437, 0.99543],
[0.7658, 0.67523, 0.99527],
[0.7563, 0.68597, 0.9951],
[0.74658, 0.69659, 0.99493],
[0.73662, 0.7071, 0.99475],
[0.72642, 0.71752, 0.99457],
[0.71594, 0.72781, 0.99438],
[0.70518, 0.73802, 0.99418],
[0.69412, 0.74815, 0.99398],
[0.68276, 0.75818, 0.99377],
[0.67105, 0.76813, 0.99355],
[0.65899, 0.778, 0.99333],
[0.64655, 0.7878, 0.9931],
[0.6337, 0.79752, 0.99286],
[0.62042, 0.80718, 0.99262],
[0.60664, 0.81677, 0.99237],
[0.59237, 0.8263, 0.99211],
[0.57752, 0.83577, 0.99185],
[0.56207, 0.84518, 0.99156],
[0.54593, 0.85452, 0.99125],
[0.52906, 0.86379, 0.99088],
[0.51136, 0.87297, 0.99045],
[0.4928, 0.88204, 0.9899],
[0.47328, 0.89096, 0.98919],
[0.45275, 0.8997, 0.98824],
[0.43114, 0.90817, 0.98697],
[0.40843, 0.91628, 0.98526],
[0.38467, 0.92395, 0.98299],
[0.35995, 0.93103, 0.98003],
[0.33446, 0.9374, 0.97626],
[0.30845, 0.94293, 0.97156],
[0.28235, 0.94749, 0.96585],
[0.25667, 0.951, 0.95907],
[0.2321, 0.95339, 0.95123],
[0.20936, 0.95464, 0.94235],
[0.18932, 0.95479, 0.93251],
[0.17275, 0.9539, 0.92182],
[0.16023, 0.95208, 0.91041],
[0.15207, 0.94946, 0.8984],
[0.14798, 0.94616, 0.88593],
[0.14731, 0.94234, 0.87312],
[0.14921, 0.9381, 0.86006],
[0.15278, 0.93357, 0.84684],
[0.15734, 0.92883, 0.83352],
[0.16233, 0.92395, 0.82014],
[0.16734, 0.91898, 0.80674],
[0.17216, 0.91396, 0.79332],
[0.17668, 0.9089, 0.77991],
[0.18084, 0.90383, 0.76651],
[0.18465, 0.89876, 0.75313],
[0.18802, 0.89369, 0.73978],
[0.19095, 0.88864, 0.72643],
[0.19355, 0.88358, 0.71311],
[0.19578, 0.87854, 0.6998],
[0.19769, 0.8735, 0.68651],
[0.19927, 0.86847, 0.67324],
[0.20056, 0.86345, 0.65998],
[0.20158, 0.85842, 0.64673],
[0.20232, 0.85341, 0.63349],
[0.20279, 0.84841, 0.62027],
[0.203, 0.84341, 0.60705],
[0.20295, 0.83842, 0.59384],
[0.20266, 0.83344, 0.58064],
[0.20212, 0.82847, 0.56744],
[0.20135, 0.82349, 0.55424],
[0.20034, 0.81853, 0.54104],
[0.19911, 0.81358, 0.52784],
[0.19765, 0.80863, 0.51464],
[0.19596, 0.80369, 0.50143],
[0.19405, 0.79875, 0.48821],
[0.19186, 0.79383, 0.47497],
[0.18947, 0.78891, 0.4617],
[0.18684, 0.78399, 0.44841],
[0.18396, 0.77908, 0.4351],
[0.18083, 0.77418, 0.42173],
[0.17749, 0.76928, 0.40834],
[0.17383, 0.7644, 0.39489],
[0.16993, 0.75952, 0.38139],
[0.1657, 0.75465, 0.3678],
[0.16126, 0.74977, 0.35414],
[0.15644, 0.74492, 0.34038],
[0.1513, 0.74007, 0.32648],
[0.14591, 0.73523, 0.31245],
[0.14019, 0.73041, 0.29827],
[0.1343, 0.72562, 0.28391],
[0.12822, 0.72087, 0.26936],
[0.12209, 0.71621, 0.25456],
[0.1163, 0.71163, 0.2395],
[0.11098, 0.70721, 0.22417],
[0.10666, 0.70299, 0.20861],
[0.10379, 0.69904, 0.19277],
[0.10298, 0.69545, 0.17668],
[0.10462, 0.6923, 0.1604],
[0.10917, 0.68966, 0.14403],
[0.11647, 0.68763, 0.12754],
[0.12634, 0.68625, 0.11109],
[0.13841, 0.68558, 0.094713],
]
b_cyclic_ymcgy_60_90_c67_s25 = bokeh_palette('cyclic_ymcgy_60_90_c67_s25',cyclic_ymcgy_60_90_c67_s25)
m_cyclic_ymcgy_60_90_c67_s25 = mpl_cm('cyclic_ymcgy_60_90_c67_s25',cyclic_ymcgy_60_90_c67_s25)
m_cyclic_ymcgy_60_90_c67_s25_r = mpl_cm('cyclic_ymcgy_60_90_c67_s25_r',list(reversed(cyclic_ymcgy_60_90_c67_s25)))
CET_C7s = b_cyclic_ymcgy_60_90_c67_s25
m_CET_C7s = m_cyclic_ymcgy_60_90_c67_s25
m_CET_C7s_r = m_cyclic_ymcgy_60_90_c67_s25_r
palette['CET_C7s'] = b_cyclic_ymcgy_60_90_c67_s25
cm['CET_C7s'] = m_cyclic_ymcgy_60_90_c67_s25
cm['CET_C7s_r'] = m_cyclic_ymcgy_60_90_c67_s25_r
register_cmap('cet_CET_C7s',m_cyclic_ymcgy_60_90_c67_s25)
register_cmap('cet_CET_C7s_r',m_cyclic_ymcgy_60_90_c67_s25_r)




circle_mgbm_67_c31 = [  # cmap_def
[0.85258, 0.56704, 0.52104],
[0.8511, 0.56817, 0.51664],
[0.84946, 0.56937, 0.51234],
[0.84766, 0.57064, 0.50816],
[0.84571, 0.57196, 0.50407],
[0.8436, 0.57334, 0.50009],
[0.84135, 0.57476, 0.49623],
[0.83896, 0.57624, 0.4925],
[0.83643, 0.57775, 0.48888],
[0.83377, 0.57932, 0.4854],
[0.83098, 0.5809, 0.48204],
[0.82807, 0.58253, 0.47879],
[0.82505, 0.58418, 0.47567],
[0.82193, 0.58587, 0.47266],
[0.8187, 0.58756, 0.46978],
[0.81539, 0.58928, 0.467],
[0.81198, 0.59102, 0.46435],
[0.80848, 0.59278, 0.46179],
[0.8049, 0.59454, 0.45935],
[0.80125, 0.59631, 0.457],
[0.79752, 0.59808, 0.45477],
[0.79374, 0.59987, 0.45264],
[0.78987, 0.60165, 0.45059],
[0.78595, 0.60344, 0.44864],
[0.78197, 0.60522, 0.44677],
[0.77793, 0.60702, 0.44501],
[0.77383, 0.60881, 0.44332],
[0.76968, 0.61058, 0.44172],
[0.76549, 0.61235, 0.44019],
[0.76119, 0.61414, 0.43872],
[0.75677, 0.61597, 0.4373],
[0.75316, 0.61743, 0.4362],
[0.748, 0.61951, 0.43474],
[0.7438, 0.62116, 0.43364],
[0.73934, 0.6229, 0.43255],
[0.73482, 0.62463, 0.43152],
[0.73026, 0.62635, 0.43056],
[0.72565, 0.62807, 0.42968],
[0.72099, 0.62977, 0.4289],
[0.71629, 0.63148, 0.4282],
[0.71153, 0.63317, 0.42759],
[0.70673, 0.63485, 0.42707],
[0.70186, 0.63653, 0.42665],
[0.69695, 0.63819, 0.42633],
[0.69199, 0.63985, 0.42612],
[0.68697, 0.64149, 0.42602],
[0.68189, 0.64313, 0.42604],
[0.67677, 0.64476, 0.42619],
[0.67158, 0.64637, 0.42646],
[0.66634, 0.64797, 0.42688],
[0.66105, 0.64957, 0.42743],
[0.65571, 0.65114, 0.42812],
[0.65031, 0.65269, 0.42895],
[0.64486, 0.65424, 0.42994],
[0.63937, 0.65577, 0.4311],
[0.63382, 0.65728, 0.43241],
[0.62823, 0.65875, 0.43386],
[0.6226, 0.66022, 0.43552],
[0.61693, 0.66167, 0.43733],
[0.61123, 0.66309, 0.43931],
[0.60549, 0.66448, 0.44148],
[0.59974, 0.66585, 0.44382],
[0.59396, 0.66719, 0.44632],
[0.58816, 0.6685, 0.449],
[0.58235, 0.66978, 0.45184],
[0.57655, 0.67102, 0.45485],
[0.57073, 0.67224, 0.45801],
[0.56493, 0.67342, 0.46134],
[0.55914, 0.67456, 0.46481],
[0.55336, 0.67568, 0.46842],
[0.5476, 0.67676, 0.47217],
[0.54185, 0.6778, 0.47604],
[0.53612, 0.67881, 0.48003],
[0.53042, 0.67978, 0.48412],
[0.52475, 0.68072, 0.48834],
[0.5191, 0.68163, 0.49263],
[0.51348, 0.68251, 0.49703],
[0.50789, 0.68336, 0.50152],
[0.50231, 0.68416, 0.50606],
[0.49676, 0.68495, 0.51069],
[0.49125, 0.68571, 0.51537],
[0.48573, 0.68643, 0.52011],
[0.48024, 0.68714, 0.52492],
[0.47477, 0.68781, 0.52977],
[0.46931, 0.68846, 0.53465],
[0.46384, 0.68909, 0.53958],
[0.45839, 0.68969, 0.54455],
[0.45295, 0.69026, 0.54956],
[0.44749, 0.69082, 0.55459],
[0.44206, 0.69136, 0.55965],
[0.43659, 0.69188, 0.56472],
[0.43111, 0.69239, 0.56983],
[0.4256, 0.69287, 0.57495],
[0.42007, 0.69334, 0.5801],
[0.41451, 0.69378, 0.58527],
[0.40893, 0.69421, 0.59044],
[0.40332, 0.69463, 0.59562],
[0.39766, 0.69504, 0.60082],
[0.39195, 0.69544, 0.60603],
[0.38619, 0.69581, 0.61126],
[0.38043, 0.69617, 0.6165],
[0.37463, 0.69651, 0.62174],
[0.3688, 0.69683, 0.62699],
[0.36293, 0.69714, 0.63226],
[0.35707, 0.69743, 0.63753],
[0.35119, 0.69769, 0.64282],
[0.3453, 0.69794, 0.6481],
[0.33944, 0.69817, 0.65339],
[0.33358, 0.69837, 0.65868],
[0.32776, 0.69855, 0.66399],
[0.322, 0.6987, 0.66928],
[0.31632, 0.69883, 0.67456],
[0.31072, 0.69893, 0.67984],
[0.30523, 0.699, 0.68512],
[0.29987, 0.69904, 0.69038],
[0.29467, 0.69905, 0.69563],
[0.28968, 0.69903, 0.70084],
[0.28489, 0.69897, 0.70604],
[0.28037, 0.69887, 0.7112],
[0.27617, 0.69874, 0.71632],
[0.27224, 0.69857, 0.7214],
[0.26874, 0.69837, 0.72643],
[0.2656, 0.69812, 0.7314],
[0.26291, 0.69783, 0.73632],
[0.2607, 0.6975, 0.74116],
[0.25906, 0.69712, 0.74592],
[0.25793, 0.69671, 0.7506],
[0.25738, 0.69625, 0.75518],
[0.2574, 0.69574, 0.75967],
[0.25799, 0.69518, 0.76406],
[0.2592, 0.69459, 0.76833],
[0.26094, 0.69395, 0.7725],
[0.26333, 0.69327, 0.77655],
[0.26626, 0.69255, 0.78046],
[0.2697, 0.69178, 0.78426],
[0.27363, 0.69097, 0.78793],
[0.27801, 0.69013, 0.79147],
[0.28282, 0.68925, 0.79489],
[0.288, 0.68833, 0.79818],
[0.29353, 0.68738, 0.80134],
[0.2994, 0.68639, 0.80439],
[0.30554, 0.68537, 0.8073],
[0.31189, 0.68431, 0.81009],
[0.3185, 0.68324, 0.81276],
[0.32527, 0.68212, 0.81532],
[0.33223, 0.68098, 0.81775],
[0.33931, 0.67981, 0.82008],
[0.34648, 0.67862, 0.82231],
[0.35375, 0.67741, 0.82442],
[0.3611, 0.67616, 0.82644],
[0.36852, 0.67489, 0.82836],
[0.37597, 0.67361, 0.83019],
[0.38345, 0.6723, 0.83193],
[0.39094, 0.67096, 0.83357],
[0.39844, 0.66961, 0.83514],
[0.40595, 0.66824, 0.83662],
[0.41345, 0.66684, 0.83803],
[0.42095, 0.66543, 0.83935],
[0.42845, 0.664, 0.84061],
[0.43591, 0.66255, 0.84179],
[0.44333, 0.66107, 0.8429],
[0.45073, 0.65959, 0.84395],
[0.4581, 0.65809, 0.84493],
[0.46547, 0.65657, 0.84583],
[0.47282, 0.65502, 0.84666],
[0.48015, 0.65346, 0.84742],
[0.48747, 0.65188, 0.84809],
[0.49477, 0.65028, 0.84868],
[0.50206, 0.64866, 0.84919],
[0.50935, 0.64702, 0.8496],
[0.51663, 0.64536, 0.84992],
[0.52391, 0.64368, 0.85013],
[0.53119, 0.64199, 0.85023],
[0.53848, 0.64028, 0.85023],
[0.54575, 0.63854, 0.85012],
[0.55302, 0.63679, 0.84989],
[0.56029, 0.63501, 0.84953],
[0.56756, 0.63323, 0.84905],
[0.57481, 0.63143, 0.84843],
[0.58206, 0.6296, 0.84768],
[0.5893, 0.62777, 0.84679],
[0.59654, 0.62592, 0.84575],
[0.60375, 0.62407, 0.84456],
[0.61093, 0.6222, 0.84321],
[0.61809, 0.62034, 0.84171],
[0.62521, 0.61846, 0.84004],
[0.63229, 0.61658, 0.83821],
[0.63932, 0.61469, 0.83622],
[0.64629, 0.61281, 0.83407],
[0.65319, 0.61095, 0.83175],
[0.66003, 0.6091, 0.82926],
[0.66678, 0.60725, 0.8266],
[0.67345, 0.60541, 0.82378],
[0.68002, 0.6036, 0.82081],
[0.68649, 0.60181, 0.81769],
[0.69285, 0.60004, 0.81442],
[0.69908, 0.5983, 0.81102],
[0.70519, 0.5966, 0.80747],
[0.71118, 0.59492, 0.80379],
[0.71703, 0.59329, 0.79998],
[0.72274, 0.59168, 0.79606],
[0.72832, 0.59011, 0.79203],
[0.73376, 0.58858, 0.7879],
[0.73907, 0.58709, 0.78369],
[0.74423, 0.58565, 0.77937],
[0.74924, 0.58424, 0.77498],
[0.75413, 0.58287, 0.77052],
[0.75888, 0.58155, 0.76599],
[0.76349, 0.58027, 0.76138],
[0.76797, 0.57902, 0.75673],
[0.77232, 0.57781, 0.75202],
[0.77655, 0.57666, 0.74727],
[0.78064, 0.57553, 0.74247],
[0.78462, 0.57444, 0.73762],
[0.78848, 0.5734, 0.73275],
[0.79223, 0.57239, 0.72783],
[0.79586, 0.57141, 0.72289],
[0.79939, 0.57047, 0.71793],
[0.80281, 0.56956, 0.71293],
[0.80613, 0.56869, 0.70792],
[0.80935, 0.56786, 0.70288],
[0.81246, 0.56706, 0.69782],
[0.81549, 0.56628, 0.69275],
[0.81843, 0.56553, 0.68766],
[0.82128, 0.56482, 0.68256],
[0.82405, 0.56415, 0.67746],
[0.82672, 0.5635, 0.67233],
[0.82931, 0.56288, 0.66719],
[0.83181, 0.5623, 0.66204],
[0.83421, 0.56174, 0.65688],
[0.83652, 0.56122, 0.65171],
[0.83874, 0.56074, 0.64653],
[0.84085, 0.56031, 0.64133],
[0.84286, 0.55992, 0.63614],
[0.84476, 0.55958, 0.63094],
[0.84654, 0.55927, 0.62572],
[0.84823, 0.55902, 0.62051],
[0.84979, 0.55882, 0.61529],
[0.85123, 0.55867, 0.61007],
[0.85256, 0.55857, 0.60486],
[0.85375, 0.55854, 0.59965],
[0.85481, 0.55856, 0.59444],
[0.85574, 0.55865, 0.58925],
[0.85653, 0.5588, 0.58408],
[0.85717, 0.55901, 0.57892],
[0.85767, 0.5593, 0.57378],
[0.85801, 0.55965, 0.56867],
[0.8582, 0.56007, 0.56361],
[0.85822, 0.56057, 0.55856],
[0.85808, 0.56114, 0.55358],
[0.85777, 0.56179, 0.54866],
[0.8573, 0.56251, 0.54377],
[0.85666, 0.5633, 0.53896],
[0.85585, 0.56416, 0.53421],
[0.85487, 0.56509, 0.52955],
[0.85372, 0.5661, 0.52497],
]
b_circle_mgbm_67_c31 = bokeh_palette('circle_mgbm_67_c31',circle_mgbm_67_c31)
m_circle_mgbm_67_c31 = mpl_cm('circle_mgbm_67_c31',circle_mgbm_67_c31)
m_circle_mgbm_67_c31_r = mpl_cm('circle_mgbm_67_c31_r',list(reversed(circle_mgbm_67_c31)))
cyclic_isoluminant = b_circle_mgbm_67_c31
m_cyclic_isoluminant = m_circle_mgbm_67_c31
m_cyclic_isoluminant_r = m_circle_mgbm_67_c31_r
palette['cyclic_isoluminant'] = b_circle_mgbm_67_c31
palette_n['cyclic_isoluminant'] = b_circle_mgbm_67_c31
cm['cyclic_isoluminant'] = m_circle_mgbm_67_c31
cm['cyclic_isoluminant_r'] = m_circle_mgbm_67_c31_r
cm_n['cyclic_isoluminant'] = mpl_cm('cyclic_isoluminant',circle_mgbm_67_c31)
cm_n['cyclic_isoluminant_r'] = mpl_cm('cyclic_isoluminant_r',list(reversed(circle_mgbm_67_c31)))
CET_C10 = b_circle_mgbm_67_c31
m_CET_C10 = m_circle_mgbm_67_c31
m_CET_C10_r = m_circle_mgbm_67_c31_r
palette['CET_C10'] = b_circle_mgbm_67_c31
cm['CET_C10'] = m_circle_mgbm_67_c31
cm['CET_C10_r'] = m_circle_mgbm_67_c31_r
register_cmap('cet_CET_C10',m_circle_mgbm_67_c31)
register_cmap('cet_CET_C10_r',m_circle_mgbm_67_c31_r)




linear_tritanopic_kcw_5_95_c22 = [  # cmap_def
[0.066001, 0.066024, 0.066019],
[0.067935, 0.069914, 0.070403],
[0.069815, 0.073652, 0.0746],
[0.07164, 0.077107, 0.078563],
[0.073462, 0.080576, 0.082476],
[0.075146, 0.083929, 0.08611],
[0.076843, 0.087146, 0.08973],
[0.078539, 0.090221, 0.093128],
[0.080189, 0.093191, 0.096541],
[0.081756, 0.096155, 0.10003],
[0.083005, 0.099185, 0.10353],
[0.084333, 0.10221, 0.10701],
[0.085504, 0.10522, 0.11049],
[0.086815, 0.10826, 0.11396],
[0.088036, 0.11136, 0.11748],
[0.089245, 0.11437, 0.121],
[0.090448, 0.11746, 0.12455],
[0.091666, 0.12054, 0.12815],
[0.092764, 0.12362, 0.13171],
[0.093966, 0.12674, 0.13529],
[0.095144, 0.1299, 0.13888],
[0.096204, 0.133, 0.14251],
[0.097359, 0.13609, 0.14613],
[0.098483, 0.13924, 0.14979],
[0.099554, 0.14242, 0.1534],
[0.10063, 0.14561, 0.15707],
[0.1017, 0.14873, 0.16075],
[0.10278, 0.15191, 0.16446],
[0.10382, 0.15511, 0.16813],
[0.10479, 0.15832, 0.17185],
[0.10587, 0.16154, 0.17556],
[0.10689, 0.16473, 0.17931],
[0.10783, 0.16794, 0.18304],
[0.10881, 0.17118, 0.18681],
[0.10983, 0.17444, 0.19058],
[0.11075, 0.1777, 0.19439],
[0.11168, 0.18092, 0.19815],
[0.11262, 0.1842, 0.20194],
[0.11352, 0.1875, 0.20578],
[0.11436, 0.19076, 0.20959],
[0.11526, 0.19408, 0.21342],
[0.11617, 0.19736, 0.21728],
[0.11699, 0.20065, 0.22115],
[0.11785, 0.20401, 0.22498],
[0.11871, 0.20732, 0.22888],
[0.11949, 0.21065, 0.23275],
[0.12028, 0.21399, 0.23667],
[0.12106, 0.21734, 0.24054],
[0.12184, 0.22072, 0.24446],
[0.1226, 0.22405, 0.24842],
[0.12336, 0.22744, 0.25234],
[0.1241, 0.23081, 0.25627],
[0.12484, 0.23421, 0.26024],
[0.12558, 0.23762, 0.26421],
[0.12628, 0.241, 0.26817],
[0.12693, 0.2444, 0.27214],
[0.12762, 0.24785, 0.27617],
[0.12829, 0.25127, 0.28015],
[0.12891, 0.25471, 0.28416],
[0.12957, 0.25812, 0.28818],
[0.13022, 0.26158, 0.29221],
[0.13081, 0.26502, 0.29625],
[0.1314, 0.2685, 0.30029],
[0.13197, 0.27195, 0.30435],
[0.13254, 0.27545, 0.30842],
[0.13309, 0.27893, 0.31248],
[0.13363, 0.28243, 0.31657],
[0.13417, 0.2859, 0.32067],
[0.13469, 0.28942, 0.32475],
[0.13517, 0.29292, 0.32887],
[0.13563, 0.29644, 0.33299],
[0.13609, 0.29996, 0.3371],
[0.13656, 0.30349, 0.34123],
[0.13703, 0.30703, 0.34537],
[0.13748, 0.31059, 0.34952],
[0.1379, 0.31413, 0.35367],
[0.1383, 0.31769, 0.35784],
[0.13867, 0.32125, 0.36201],
[0.13903, 0.32481, 0.36619],
[0.13938, 0.3284, 0.37036],
[0.13973, 0.33198, 0.37457],
[0.14008, 0.33558, 0.37876],
[0.14043, 0.33918, 0.38298],
[0.14075, 0.34278, 0.38718],
[0.14105, 0.34637, 0.39142],
[0.14133, 0.35, 0.39565],
[0.14159, 0.35361, 0.39988],
[0.14183, 0.35725, 0.40413],
[0.14207, 0.36086, 0.40838],
[0.14229, 0.36451, 0.41265],
[0.14251, 0.36816, 0.41691],
[0.1427, 0.3718, 0.42118],
[0.14289, 0.37546, 0.42547],
[0.14305, 0.37912, 0.42975],
[0.14321, 0.38279, 0.43404],
[0.14335, 0.38646, 0.43835],
[0.14348, 0.39014, 0.44267],
[0.14359, 0.39383, 0.44698],
[0.14369, 0.39753, 0.4513],
[0.14377, 0.40123, 0.45563],
[0.14384, 0.40492, 0.45996],
[0.1439, 0.40864, 0.46432],
[0.14394, 0.41236, 0.46866],
[0.14396, 0.41607, 0.473],
[0.14397, 0.4198, 0.47737],
[0.14397, 0.42352, 0.48174],
[0.14395, 0.42727, 0.48611],
[0.14392, 0.43101, 0.4905],
[0.14388, 0.43476, 0.49489],
[0.14382, 0.43851, 0.49927],
[0.14374, 0.44227, 0.50367],
[0.14366, 0.44602, 0.50809],
[0.14355, 0.4498, 0.51249],
[0.14344, 0.45357, 0.51691],
[0.1433, 0.45735, 0.52133],
[0.14316, 0.46114, 0.52576],
[0.14299, 0.46493, 0.53019],
[0.14282, 0.46873, 0.53463],
[0.14263, 0.47252, 0.53908],
[0.14242, 0.47633, 0.54353],
[0.14221, 0.48014, 0.54799],
[0.14197, 0.48395, 0.55245],
[0.14173, 0.48778, 0.55692],
[0.14148, 0.49161, 0.56139],
[0.14121, 0.49543, 0.56586],
[0.14092, 0.49926, 0.57035],
[0.14061, 0.5031, 0.57483],
[0.14028, 0.50695, 0.57934],
[0.13993, 0.5108, 0.58383],
[0.13957, 0.51466, 0.58833],
[0.13922, 0.51852, 0.59285],
[0.13887, 0.52238, 0.59737],
[0.13851, 0.52626, 0.60188],
[0.13813, 0.53013, 0.6064],
[0.13772, 0.534, 0.61094],
[0.13729, 0.53789, 0.61547],
[0.13683, 0.54177, 0.62001],
[0.13636, 0.54567, 0.62455],
[0.1359, 0.54956, 0.6291],
[0.13545, 0.55346, 0.63366],
[0.135, 0.55738, 0.63821],
[0.13451, 0.56128, 0.64277],
[0.134, 0.56519, 0.64734],
[0.13348, 0.56911, 0.65191],
[0.13295, 0.57304, 0.65649],
[0.13242, 0.57697, 0.66106],
[0.13188, 0.58089, 0.66564],
[0.13133, 0.58483, 0.67024],
[0.13079, 0.58876, 0.67482],
[0.13024, 0.59272, 0.67941],
[0.12964, 0.59667, 0.68401],
[0.12903, 0.60062, 0.68862],
[0.12847, 0.60458, 0.69322],
[0.12789, 0.60854, 0.69782],
[0.12727, 0.6125, 0.70244],
[0.12668, 0.61647, 0.70706],
[0.12612, 0.62045, 0.71168],
[0.12552, 0.62442, 0.7163],
[0.1249, 0.6284, 0.72092],
[0.12429, 0.63238, 0.72555],
[0.1237, 0.63637, 0.73018],
[0.12311, 0.64036, 0.73481],
[0.12253, 0.64435, 0.73945],
[0.12196, 0.64835, 0.74408],
[0.1214, 0.65236, 0.74872],
[0.12086, 0.65637, 0.75336],
[0.12033, 0.66037, 0.75801],
[0.11982, 0.66438, 0.76266],
[0.11934, 0.6684, 0.76731],
[0.11888, 0.67242, 0.77196],
[0.11844, 0.67644, 0.77661],
[0.118, 0.68046, 0.78126],
[0.11757, 0.68449, 0.78592],
[0.11721, 0.68853, 0.79057],
[0.1169, 0.69257, 0.79522],
[0.11664, 0.6966, 0.79989],
[0.11641, 0.70064, 0.80454],
[0.11623, 0.70468, 0.8092],
[0.11609, 0.70873, 0.81385],
[0.11602, 0.71277, 0.8185],
[0.11601, 0.71683, 0.82316],
[0.11607, 0.72087, 0.82781],
[0.11621, 0.72493, 0.83246],
[0.11643, 0.72899, 0.83711],
[0.11673, 0.73305, 0.84176],
[0.11713, 0.73711, 0.8464],
[0.11766, 0.74116, 0.85103],
[0.11834, 0.74523, 0.85567],
[0.11908, 0.74929, 0.8603],
[0.11996, 0.75335, 0.86492],
[0.12099, 0.75742, 0.86954],
[0.12218, 0.76148, 0.87415],
[0.12355, 0.76555, 0.87875],
[0.12515, 0.76961, 0.88334],
[0.12689, 0.77367, 0.88792],
[0.12887, 0.77773, 0.89249],
[0.13114, 0.78179, 0.89704],
[0.13361, 0.78584, 0.90157],
[0.13637, 0.78989, 0.90608],
[0.13948, 0.79394, 0.91056],
[0.14302, 0.79797, 0.91501],
[0.14694, 0.802, 0.91943],
[0.15138, 0.80601, 0.9238],
[0.15647, 0.81, 0.9281],
[0.16229, 0.81396, 0.93231],
[0.1692, 0.8179, 0.93644],
[0.17755, 0.8218, 0.9404],
[0.18745, 0.82562, 0.94415],
[0.19867, 0.82939, 0.94771],
[0.21106, 0.83308, 0.95106],
[0.22431, 0.83672, 0.95422],
[0.2383, 0.8403, 0.9572],
[0.25286, 0.84381, 0.95999],
[0.26783, 0.84727, 0.9626],
[0.28318, 0.85067, 0.96504],
[0.29879, 0.85401, 0.9673],
[0.31461, 0.8573, 0.96939],
[0.33061, 0.86053, 0.97131],
[0.34673, 0.86371, 0.97308],
[0.36293, 0.86682, 0.97467],
[0.37921, 0.86988, 0.9761],
[0.39554, 0.8729, 0.97739],
[0.41189, 0.87585, 0.97851],
[0.42827, 0.87875, 0.9795],
[0.44466, 0.8816, 0.98033],
[0.46103, 0.8844, 0.98101],
[0.47742, 0.88714, 0.98155],
[0.49379, 0.88983, 0.98195],
[0.51015, 0.89247, 0.98221],
[0.52649, 0.89506, 0.98233],
[0.54281, 0.89759, 0.98232],
[0.5591, 0.90006, 0.98218],
[0.57537, 0.90249, 0.9819],
[0.59162, 0.90487, 0.9815],
[0.60785, 0.90719, 0.98097],
[0.62404, 0.90946, 0.98031],
[0.64022, 0.91167, 0.97954],
[0.65636, 0.91384, 0.97863],
[0.67249, 0.91594, 0.97762],
[0.68859, 0.91801, 0.97648],
[0.70466, 0.92001, 0.97523],
[0.72071, 0.92196, 0.97386],
[0.73673, 0.92386, 0.97238],
[0.75274, 0.9257, 0.97079],
[0.76872, 0.92749, 0.96909],
[0.78469, 0.92922, 0.96728],
[0.80063, 0.93091, 0.96535],
[0.81655, 0.93253, 0.96333],
[0.83245, 0.9341, 0.96121],
[0.84834, 0.93562, 0.95897],
[0.86422, 0.93708, 0.95664],
[0.88006, 0.93848, 0.95421],
[0.89591, 0.93983, 0.95167],
[0.91173, 0.94112, 0.94903],
[0.92754, 0.94235, 0.9463],
[0.94334, 0.94352, 0.94348],
]
b_linear_tritanopic_kcw_5_95_c22 = bokeh_palette('linear_tritanopic_kcw_5_95_c22',linear_tritanopic_kcw_5_95_c22)
m_linear_tritanopic_kcw_5_95_c22 = mpl_cm('linear_tritanopic_kcw_5_95_c22',linear_tritanopic_kcw_5_95_c22)
m_linear_tritanopic_kcw_5_95_c22_r = mpl_cm('linear_tritanopic_kcw_5_95_c22_r',list(reversed(linear_tritanopic_kcw_5_95_c22)))
CET_CBTL3 = b_linear_tritanopic_kcw_5_95_c22
m_CET_CBTL3 = m_linear_tritanopic_kcw_5_95_c22
m_CET_CBTL3_r = m_linear_tritanopic_kcw_5_95_c22_r
palette['CET_CBTL3'] = b_linear_tritanopic_kcw_5_95_c22
cm['CET_CBTL3'] = m_linear_tritanopic_kcw_5_95_c22
cm['CET_CBTL3_r'] = m_linear_tritanopic_kcw_5_95_c22_r
register_cmap('cet_CET_CBTL3',m_linear_tritanopic_kcw_5_95_c22)
register_cmap('cet_CET_CBTL3_r',m_linear_tritanopic_kcw_5_95_c22_r)




cyclic_wrkbw_10_90_c43_s25 = [  # cmap_def
[0.20021, 0.43981, 0.96906],
[0.21551, 0.44588, 0.97457],
[0.23159, 0.45204, 0.97914],
[0.24818, 0.45825, 0.98281],
[0.26504, 0.46453, 0.98564],
[0.28205, 0.47086, 0.9877],
[0.29898, 0.47725, 0.98908],
[0.31574, 0.48369, 0.98987],
[0.33223, 0.4902, 0.99013],
[0.34838, 0.49674, 0.98997],
[0.36416, 0.50334, 0.98944],
[0.37954, 0.50998, 0.98862],
[0.3945, 0.51666, 0.98757],
[0.40905, 0.52338, 0.98633],
[0.42318, 0.53013, 0.98496],
[0.43695, 0.53692, 0.98347],
[0.45032, 0.54373, 0.98191],
[0.46333, 0.55057, 0.98029],
[0.47603, 0.55745, 0.97861],
[0.48841, 0.56434, 0.97691],
[0.5005, 0.57126, 0.97517],
[0.51231, 0.5782, 0.97342],
[0.52387, 0.58518, 0.97166],
[0.53518, 0.59218, 0.96988],
[0.54629, 0.59919, 0.96809],
[0.55717, 0.60622, 0.96629],
[0.56786, 0.61328, 0.96448],
[0.57837, 0.62038, 0.96266],
[0.58871, 0.62748, 0.96082],
[0.59889, 0.63461, 0.95897],
[0.60893, 0.64175, 0.95711],
[0.61881, 0.64892, 0.95524],
[0.62855, 0.65611, 0.95336],
[0.63816, 0.66332, 0.95146],
[0.64766, 0.67054, 0.94955],
[0.65704, 0.67779, 0.94763],
[0.6663, 0.68505, 0.94569],
[0.67546, 0.69234, 0.94374],
[0.68452, 0.69963, 0.94177],
[0.69348, 0.70696, 0.93979],
[0.70235, 0.71429, 0.9378],
[0.71114, 0.72164, 0.93579],
[0.71983, 0.72902, 0.93377],
[0.72843, 0.73641, 0.93173],
[0.73696, 0.7438, 0.92968],
[0.7454, 0.75119, 0.92759],
[0.75376, 0.75857, 0.92549],
[0.76204, 0.76591, 0.92334],
[0.77023, 0.77321, 0.92114],
[0.77834, 0.78046, 0.91889],
[0.78635, 0.7876, 0.91657],
[0.79425, 0.7946, 0.91414],
[0.80203, 0.80142, 0.9116],
[0.8097, 0.80801, 0.9089],
[0.81723, 0.81429, 0.90602],
[0.8246, 0.8202, 0.9029],
[0.83181, 0.82566, 0.8995],
[0.83882, 0.83059, 0.89578],
[0.84564, 0.83488, 0.89168],
[0.85223, 0.83847, 0.88714],
[0.85859, 0.84127, 0.88212],
[0.8647, 0.84321, 0.87657],
[0.87054, 0.84425, 0.87046],
[0.87611, 0.84432, 0.86377],
[0.88141, 0.84341, 0.85645],
[0.88642, 0.84153, 0.84851],
[0.89115, 0.83869, 0.83998],
[0.8956, 0.83491, 0.83084],
[0.89977, 0.83027, 0.82115],
[0.90367, 0.82481, 0.81095],
[0.90731, 0.81863, 0.80029],
[0.9107, 0.8118, 0.7892],
[0.91385, 0.80441, 0.77777],
[0.91677, 0.79653, 0.76602],
[0.91948, 0.78825, 0.75403],
[0.92199, 0.77963, 0.74184],
[0.9243, 0.77076, 0.72951],
[0.92644, 0.76166, 0.71706],
[0.9284, 0.75241, 0.70452],
[0.93021, 0.74303, 0.69193],
[0.93185, 0.73355, 0.67931],
[0.93335, 0.72399, 0.66667],
[0.93471, 0.71438, 0.65405],
[0.93593, 0.70473, 0.64142],
[0.93702, 0.69504, 0.62882],
[0.93798, 0.68533, 0.61624],
[0.93881, 0.67559, 0.60369],
[0.93952, 0.66583, 0.59117],
[0.94012, 0.65606, 0.57868],
[0.94059, 0.64626, 0.56623],
[0.94095, 0.63645, 0.55381],
[0.94119, 0.6266, 0.54142],
[0.94132, 0.61675, 0.52908],
[0.94134, 0.60686, 0.51676],
[0.94125, 0.59694, 0.50448],
[0.94106, 0.58698, 0.49222],
[0.94076, 0.57701, 0.48],
[0.94037, 0.56699, 0.4678],
[0.93986, 0.55693, 0.45565],
[0.93925, 0.54684, 0.44352],
[0.93855, 0.5367, 0.43143],
[0.93775, 0.52651, 0.41936],
[0.93686, 0.51626, 0.40732],
[0.93587, 0.50596, 0.39531],
[0.93478, 0.49561, 0.38333],
[0.93361, 0.48519, 0.37136],
[0.93235, 0.47471, 0.35943],
[0.931, 0.46415, 0.34753],
[0.92955, 0.4535, 0.33565],
[0.928, 0.44279, 0.32378],
[0.92636, 0.43199, 0.31194],
[0.9246, 0.42108, 0.30017],
[0.92274, 0.41012, 0.28842],
[0.92074, 0.39905, 0.27674],
[0.91861, 0.38791, 0.26509],
[0.91631, 0.37671, 0.25356],
[0.91384, 0.36544, 0.24213],
[0.91115, 0.35415, 0.23083],
[0.90822, 0.34286, 0.21971],
[0.905, 0.33158, 0.20879],
[0.90147, 0.32042, 0.19814],
[0.89757, 0.30939, 0.18785],
[0.89326, 0.29854, 0.1779],
[0.88851, 0.28799, 0.16836],
[0.88326, 0.27783, 0.15935],
[0.87749, 0.26809, 0.15092],
[0.87116, 0.25891, 0.14313],
[0.86426, 0.25031, 0.13596],
[0.85677, 0.24243, 0.12959],
[0.84871, 0.23529, 0.12389],
[0.84007, 0.22889, 0.11901],
[0.83088, 0.22328, 0.1148],
[0.82118, 0.21843, 0.11142],
[0.811, 0.21431, 0.10862],
[0.80039, 0.21086, 0.10655],
[0.7894, 0.20802, 0.10489],
[0.77807, 0.2057, 0.10386],
[0.76647, 0.20383, 0.10319],
[0.75465, 0.20228, 0.10284],
[0.74264, 0.20102, 0.10278],
[0.73048, 0.19998, 0.10293],
[0.71822, 0.1991, 0.10324],
[0.70589, 0.19832, 0.10365],
[0.6935, 0.19759, 0.10411],
[0.68109, 0.19691, 0.10462],
[0.66868, 0.1962, 0.10519],
[0.65627, 0.19546, 0.10581],
[0.64385, 0.19473, 0.1064],
[0.63148, 0.19394, 0.10695],
[0.61912, 0.19309, 0.10746],
[0.60679, 0.19219, 0.10794],
[0.59449, 0.19125, 0.10842],
[0.58222, 0.19025, 0.1089],
[0.57, 0.1892, 0.10936],
[0.55781, 0.18812, 0.10977],
[0.54567, 0.18693, 0.11013],
[0.53354, 0.18573, 0.11046],
[0.52147, 0.18443, 0.11077],
[0.50943, 0.18309, 0.11106],
[0.49743, 0.18171, 0.11132],
[0.48545, 0.18029, 0.11156],
[0.47351, 0.17882, 0.11177],
[0.46162, 0.1773, 0.11196],
[0.44976, 0.17568, 0.11212],
[0.43793, 0.17406, 0.11226],
[0.42612, 0.17237, 0.11238],
[0.41437, 0.17064, 0.11248],
[0.40264, 0.16888, 0.11255],
[0.39096, 0.16705, 0.1126],
[0.37929, 0.16512, 0.11263],
[0.36765, 0.16323, 0.11264],
[0.35607, 0.16126, 0.11263],
[0.3445, 0.15922, 0.11262],
[0.33301, 0.15716, 0.11261],
[0.32154, 0.15507, 0.1126],
[0.31017, 0.15294, 0.11262],
[0.29885, 0.1508, 0.11268],
[0.28762, 0.14863, 0.11279],
[0.27655, 0.14648, 0.11299],
[0.26561, 0.14436, 0.1133],
[0.25488, 0.14225, 0.11373],
[0.24434, 0.14019, 0.11434],
[0.23414, 0.13827, 0.11526],
[0.22424, 0.13639, 0.11649],
[0.21479, 0.13476, 0.11804],
[0.20581, 0.13325, 0.11997],
[0.19735, 0.132, 0.12239],
[0.18949, 0.13103, 0.12541],
[0.18227, 0.13038, 0.12894],
[0.17577, 0.13006, 0.13316],
[0.17005, 0.13011, 0.13804],
[0.16504, 0.13055, 0.14357],
[0.16085, 0.13138, 0.14984],
[0.15741, 0.13265, 0.15679],
[0.15475, 0.13435, 0.16446],
[0.15281, 0.13638, 0.17277],
[0.15159, 0.13886, 0.18173],
[0.151, 0.1417, 0.19131],
[0.151, 0.14485, 0.20141],
[0.1515, 0.14824, 0.21205],
[0.15245, 0.15192, 0.22308],
[0.15379, 0.15585, 0.2345],
[0.15544, 0.15988, 0.24624],
[0.15729, 0.16417, 0.25828],
[0.15933, 0.16851, 0.27059],
[0.16152, 0.17298, 0.28308],
[0.16373, 0.17756, 0.29575],
[0.16592, 0.18214, 0.3086],
[0.16815, 0.18684, 0.32156],
[0.17034, 0.19156, 0.33467],
[0.17243, 0.19636, 0.34786],
[0.17447, 0.20112, 0.36115],
[0.17635, 0.20599, 0.37455],
[0.17822, 0.21083, 0.38802],
[0.17987, 0.21572, 0.40157],
[0.1814, 0.22063, 0.41521],
[0.18281, 0.22554, 0.42891],
[0.18408, 0.23047, 0.4427],
[0.18522, 0.23545, 0.45655],
[0.18618, 0.24041, 0.47048],
[0.18698, 0.24541, 0.48447],
[0.18766, 0.25044, 0.49855],
[0.18815, 0.2555, 0.51268],
[0.18846, 0.26055, 0.5269],
[0.18861, 0.26564, 0.54116],
[0.18858, 0.27075, 0.5555],
[0.18837, 0.27587, 0.5699],
[0.18798, 0.28097, 0.58438],
[0.18736, 0.28613, 0.59891],
[0.18653, 0.29133, 0.61351],
[0.18553, 0.29651, 0.62818],
[0.18424, 0.30171, 0.6429],
[0.18274, 0.30695, 0.65768],
[0.18099, 0.31217, 0.67253],
[0.179, 0.31745, 0.68742],
[0.17672, 0.32274, 0.70237],
[0.17419, 0.32802, 0.71736],
[0.17136, 0.33335, 0.73238],
[0.16826, 0.33867, 0.74741],
[0.16491, 0.34402, 0.76244],
[0.16137, 0.3494, 0.77746],
[0.15757, 0.3548, 0.79243],
[0.15368, 0.3602, 0.8073],
[0.14983, 0.36564, 0.82205],
[0.14607, 0.3711, 0.8366],
[0.14266, 0.37661, 0.8509],
[0.13982, 0.38211, 0.86489],
[0.13797, 0.38767, 0.87847],
[0.1373, 0.39327, 0.89156],
[0.13822, 0.39891, 0.90407],
[0.14103, 0.40458, 0.91592],
[0.14591, 0.41033, 0.92702],
[0.15293, 0.4161, 0.93729],
[0.16214, 0.42193, 0.94667],
[0.17323, 0.42784, 0.95511],
[0.18606, 0.43378, 0.96258],
]
b_cyclic_wrkbw_10_90_c43_s25 = bokeh_palette('cyclic_wrkbw_10_90_c43_s25',cyclic_wrkbw_10_90_c43_s25)
m_cyclic_wrkbw_10_90_c43_s25 = mpl_cm('cyclic_wrkbw_10_90_c43_s25',cyclic_wrkbw_10_90_c43_s25)
m_cyclic_wrkbw_10_90_c43_s25_r = mpl_cm('cyclic_wrkbw_10_90_c43_s25_r',list(reversed(cyclic_wrkbw_10_90_c43_s25)))
CET_C3s = b_cyclic_wrkbw_10_90_c43_s25
m_CET_C3s = m_cyclic_wrkbw_10_90_c43_s25
m_CET_C3s_r = m_cyclic_wrkbw_10_90_c43_s25_r
palette['CET_C3s'] = b_cyclic_wrkbw_10_90_c43_s25
cm['CET_C3s'] = m_cyclic_wrkbw_10_90_c43_s25
cm['CET_C3s_r'] = m_cyclic_wrkbw_10_90_c43_s25_r
register_cmap('cet_CET_C3s',m_cyclic_wrkbw_10_90_c43_s25)
register_cmap('cet_CET_C3s_r',m_cyclic_wrkbw_10_90_c43_s25_r)




linear_protanopic_deuteranopic_kbw_5_95_c34 = [  # cmap_def
[0.066001, 0.066024, 0.066019],
[0.068745, 0.069522, 0.072449],
[0.071332, 0.07276, 0.078437],
[0.073863, 0.075905, 0.084184],
[0.07624, 0.078975, 0.089508],
[0.07863, 0.08205, 0.094631],
[0.080915, 0.084906, 0.099619],
[0.082936, 0.087763, 0.10467],
[0.084808, 0.090516, 0.10987],
[0.086573, 0.093153, 0.11495],
[0.088351, 0.095862, 0.12015],
[0.090095, 0.098575, 0.12539],
[0.091845, 0.10126, 0.13065],
[0.093475, 0.10398, 0.13585],
[0.095207, 0.10675, 0.1412],
[0.096775, 0.10948, 0.14648],
[0.09846, 0.11221, 0.15182],
[0.10003, 0.11491, 0.15719],
[0.10161, 0.11771, 0.16261],
[0.1032, 0.12047, 0.168],
[0.10466, 0.12324, 0.17344],
[0.10624, 0.12609, 0.17892],
[0.10768, 0.12886, 0.1844],
[0.10917, 0.1317, 0.18992],
[0.1106, 0.13454, 0.19545],
[0.11201, 0.13737, 0.20099],
[0.11341, 0.14018, 0.20662],
[0.1147, 0.14305, 0.21221],
[0.11609, 0.14592, 0.21783],
[0.11736, 0.14876, 0.22348],
[0.1187, 0.15164, 0.22915],
[0.11992, 0.15454, 0.23486],
[0.12114, 0.15743, 0.24054],
[0.12235, 0.16032, 0.24628],
[0.12355, 0.16328, 0.25205],
[0.12473, 0.16618, 0.25781],
[0.1259, 0.16914, 0.26363],
[0.12697, 0.17205, 0.26945],
[0.1281, 0.17502, 0.27528],
[0.12913, 0.17799, 0.28112],
[0.13023, 0.18091, 0.28699],
[0.13122, 0.1839, 0.2929],
[0.1322, 0.18688, 0.29882],
[0.13316, 0.18987, 0.30475],
[0.13411, 0.19287, 0.31072],
[0.13502, 0.19588, 0.31668],
[0.13586, 0.19887, 0.32268],
[0.13673, 0.20189, 0.32869],
[0.1376, 0.20494, 0.33472],
[0.13839, 0.20798, 0.34076],
[0.13913, 0.21102, 0.34682],
[0.13988, 0.21408, 0.35291],
[0.14064, 0.21714, 0.35899],
[0.14134, 0.22021, 0.36512],
[0.14199, 0.22328, 0.37124],
[0.14263, 0.22637, 0.37741],
[0.14324, 0.22943, 0.38358],
[0.14382, 0.23253, 0.38976],
[0.14437, 0.23565, 0.39597],
[0.14491, 0.23874, 0.40217],
[0.14541, 0.24187, 0.40842],
[0.14586, 0.24497, 0.41467],
[0.14626, 0.24813, 0.42094],
[0.14665, 0.25127, 0.42724],
[0.14703, 0.25441, 0.43352],
[0.14738, 0.25755, 0.43985],
[0.14768, 0.26071, 0.44618],
[0.14795, 0.26389, 0.45254],
[0.14817, 0.26703, 0.4589],
[0.14837, 0.27023, 0.46527],
[0.14854, 0.27341, 0.47168],
[0.14867, 0.2766, 0.47808],
[0.14878, 0.2798, 0.48451],
[0.14885, 0.28301, 0.49097],
[0.14888, 0.28619, 0.49742],
[0.14887, 0.28943, 0.50389],
[0.14882, 0.29264, 0.51037],
[0.14874, 0.29587, 0.51687],
[0.14861, 0.29912, 0.52339],
[0.14845, 0.30234, 0.52992],
[0.14825, 0.30561, 0.53647],
[0.14802, 0.30887, 0.54302],
[0.14775, 0.3121, 0.5496],
[0.14742, 0.31538, 0.55617],
[0.14703, 0.31865, 0.56279],
[0.14661, 0.32193, 0.56939],
[0.14616, 0.32521, 0.57603],
[0.14568, 0.32851, 0.58267],
[0.14512, 0.3318, 0.58932],
[0.1445, 0.33512, 0.596],
[0.14384, 0.33841, 0.60268],
[0.14313, 0.34173, 0.60938],
[0.14236, 0.34505, 0.61609],
[0.14155, 0.34837, 0.62281],
[0.14069, 0.35169, 0.62955],
[0.1397, 0.35505, 0.63631],
[0.13874, 0.35838, 0.64306],
[0.13771, 0.36173, 0.64984],
[0.13655, 0.36508, 0.65663],
[0.13538, 0.36844, 0.66344],
[0.13415, 0.3718, 0.67025],
[0.13281, 0.37517, 0.67708],
[0.13142, 0.37855, 0.68391],
[0.12996, 0.38193, 0.69076],
[0.12837, 0.38531, 0.69763],
[0.12671, 0.38871, 0.70451],
[0.12499, 0.3921, 0.71141],
[0.12314, 0.39551, 0.7183],
[0.12122, 0.39891, 0.72523],
[0.11923, 0.40232, 0.73215],
[0.11708, 0.40575, 0.73909],
[0.11482, 0.40917, 0.74604],
[0.11251, 0.41261, 0.753],
[0.11003, 0.41603, 0.75997],
[0.10737, 0.41948, 0.76696],
[0.10453, 0.42291, 0.77396],
[0.10161, 0.42636, 0.78097],
[0.098517, 0.42983, 0.78799],
[0.095238, 0.43328, 0.79502],
[0.091759, 0.43675, 0.80206],
[0.088046, 0.44022, 0.80911],
[0.084173, 0.4437, 0.81616],
[0.079994, 0.44717, 0.82321],
[0.075648, 0.45066, 0.83025],
[0.07114, 0.45415, 0.83729],
[0.066311, 0.45764, 0.84431],
[0.061257, 0.46114, 0.85131],
[0.056168, 0.46464, 0.85829],
[0.050831, 0.46814, 0.86523],
[0.045506, 0.47166, 0.87211],
[0.040376, 0.47517, 0.87893],
[0.035673, 0.47867, 0.88567],
[0.031614, 0.48219, 0.89232],
[0.02889, 0.4857, 0.89884],
[0.027504, 0.48922, 0.90523],
[0.027708, 0.49274, 0.91146],
[0.029767, 0.49625, 0.91751],
[0.033929, 0.49977, 0.92335],
[0.040741, 0.50329, 0.92897],
[0.049364, 0.50681, 0.93433],
[0.059373, 0.51032, 0.93944],
[0.070404, 0.51384, 0.94425],
[0.082375, 0.51735, 0.94875],
[0.094921, 0.52085, 0.95294],
[0.10784, 0.52435, 0.95681],
[0.12113, 0.52785, 0.96035],
[0.13469, 0.53134, 0.96355],
[0.14827, 0.53484, 0.96642],
[0.16196, 0.53834, 0.96898],
[0.17556, 0.54182, 0.97123],
[0.18913, 0.54532, 0.97318],
[0.20256, 0.54882, 0.97486],
[0.21586, 0.55231, 0.97627],
[0.22893, 0.55579, 0.97745],
[0.24179, 0.5593, 0.97841],
[0.25441, 0.5628, 0.97918],
[0.26676, 0.56631, 0.97977],
[0.27887, 0.56981, 0.98022],
[0.29069, 0.57333, 0.98053],
[0.30223, 0.57686, 0.98073],
[0.31353, 0.58038, 0.98084],
[0.32454, 0.58391, 0.98087],
[0.33531, 0.58745, 0.98083],
[0.3458, 0.591, 0.98074],
[0.35607, 0.59455, 0.9806],
[0.36609, 0.59811, 0.98043],
[0.37589, 0.60168, 0.98023],
[0.38547, 0.60525, 0.98001],
[0.39487, 0.60885, 0.97977],
[0.40407, 0.61242, 0.97952],
[0.41309, 0.61602, 0.97925],
[0.42193, 0.61963, 0.97898],
[0.43063, 0.62324, 0.9787],
[0.43916, 0.62686, 0.97841],
[0.44755, 0.63049, 0.97812],
[0.45581, 0.63412, 0.97783],
[0.46394, 0.63775, 0.97754],
[0.47195, 0.64139, 0.97724],
[0.47984, 0.64505, 0.97694],
[0.48763, 0.64871, 0.97664],
[0.49531, 0.65238, 0.97633],
[0.50289, 0.65605, 0.97602],
[0.51038, 0.65973, 0.97571],
[0.5178, 0.66342, 0.97539],
[0.52512, 0.6671, 0.97507],
[0.53236, 0.67081, 0.97475],
[0.53952, 0.67451, 0.97442],
[0.54662, 0.67822, 0.9741],
[0.55364, 0.68194, 0.97377],
[0.5606, 0.68567, 0.97343],
[0.5675, 0.6894, 0.9731],
[0.57432, 0.69313, 0.97276],
[0.5811, 0.69687, 0.97241],
[0.58781, 0.70063, 0.97206],
[0.59448, 0.70438, 0.97171],
[0.60108, 0.70814, 0.97136],
[0.60765, 0.71191, 0.971],
[0.61415, 0.71568, 0.97064],
[0.62062, 0.71946, 0.97028],
[0.62704, 0.72324, 0.96991],
[0.63341, 0.72703, 0.96954],
[0.63975, 0.73083, 0.96917],
[0.64603, 0.73463, 0.96879],
[0.65229, 0.73844, 0.96841],
[0.6585, 0.74225, 0.96803],
[0.66468, 0.74607, 0.96764],
[0.67082, 0.74989, 0.96725],
[0.67693, 0.75372, 0.96686],
[0.683, 0.75756, 0.96646],
[0.68905, 0.7614, 0.96605],
[0.69505, 0.76525, 0.96565],
[0.70103, 0.7691, 0.96524],
[0.70698, 0.77296, 0.96483],
[0.71289, 0.77682, 0.96441],
[0.71879, 0.78069, 0.964],
[0.72465, 0.78457, 0.96357],
[0.73049, 0.78844, 0.96315],
[0.7363, 0.79232, 0.96272],
[0.74209, 0.79621, 0.96228],
[0.74786, 0.80011, 0.96184],
[0.75359, 0.80401, 0.96141],
[0.75931, 0.80791, 0.96096],
[0.76501, 0.81182, 0.96052],
[0.77068, 0.81574, 0.96006],
[0.77633, 0.81965, 0.95961],
[0.78196, 0.82358, 0.95915],
[0.78756, 0.82751, 0.95869],
[0.79316, 0.83145, 0.95822],
[0.79873, 0.83539, 0.95775],
[0.80428, 0.83933, 0.95727],
[0.80981, 0.84328, 0.95679],
[0.81533, 0.84723, 0.95631],
[0.82083, 0.85119, 0.95582],
[0.82631, 0.85516, 0.95533],
[0.83178, 0.85912, 0.95483],
[0.83723, 0.8631, 0.95434],
[0.84266, 0.86707, 0.95383],
[0.84808, 0.87106, 0.95333],
[0.85348, 0.87505, 0.95281],
[0.85887, 0.87904, 0.9523],
[0.86425, 0.88303, 0.95178],
[0.8696, 0.88704, 0.95126],
[0.87495, 0.89104, 0.95073],
[0.88028, 0.89506, 0.9502],
[0.8856, 0.89907, 0.94966],
[0.89091, 0.90309, 0.94912],
[0.89621, 0.90711, 0.94857],
[0.90149, 0.91114, 0.94803],
[0.90676, 0.91517, 0.94747],
[0.91202, 0.91921, 0.94691],
[0.91727, 0.92325, 0.94635],
[0.9225, 0.92729, 0.94579],
[0.92772, 0.93135, 0.94522],
[0.93293, 0.9354, 0.94464],
[0.93814, 0.93946, 0.94406],
[0.94334, 0.94352, 0.94348],
]
b_linear_protanopic_deuteranopic_kbw_5_95_c34 = bokeh_palette('linear_protanopic_deuteranopic_kbw_5_95_c34',linear_protanopic_deuteranopic_kbw_5_95_c34)
m_linear_protanopic_deuteranopic_kbw_5_95_c34 = mpl_cm('linear_protanopic_deuteranopic_kbw_5_95_c34',linear_protanopic_deuteranopic_kbw_5_95_c34)
m_linear_protanopic_deuteranopic_kbw_5_95_c34_r = mpl_cm('linear_protanopic_deuteranopic_kbw_5_95_c34_r',list(reversed(linear_protanopic_deuteranopic_kbw_5_95_c34)))
CET_CBL3 = b_linear_protanopic_deuteranopic_kbw_5_95_c34
m_CET_CBL3 = m_linear_protanopic_deuteranopic_kbw_5_95_c34
m_CET_CBL3_r = m_linear_protanopic_deuteranopic_kbw_5_95_c34_r
palette['CET_CBL3'] = b_linear_protanopic_deuteranopic_kbw_5_95_c34
cm['CET_CBL3'] = m_linear_protanopic_deuteranopic_kbw_5_95_c34
cm['CET_CBL3_r'] = m_linear_protanopic_deuteranopic_kbw_5_95_c34_r
register_cmap('cet_CET_CBL3',m_linear_protanopic_deuteranopic_kbw_5_95_c34)
register_cmap('cet_CET_CBL3_r',m_linear_protanopic_deuteranopic_kbw_5_95_c34_r)




cyclic_rygcbmr_50_90_c64 = [  # cmap_def
[0.96726, 0.21404, 0.10285],
[0.96657, 0.21983, 0.0919],
[0.96633, 0.22834, 0.081637],
[0.96652, 0.23917, 0.071996],
[0.9671, 0.25198, 0.063086],
[0.96804, 0.26633, 0.054632],
[0.96927, 0.28187, 0.046867],
[0.97077, 0.29827, 0.039573],
[0.97247, 0.31531, 0.032902],
[0.97433, 0.33274, 0.027416],
[0.9763, 0.35036, 0.022767],
[0.97835, 0.36807, 0.018802],
[0.98044, 0.38577, 0.015391],
[0.98254, 0.40339, 0.012442],
[0.98463, 0.42085, 0.009625],
[0.98667, 0.43815, 0.007288],
[0.98867, 0.45524, 0.005171],
[0.99061, 0.47213, 0.003222],
[0.99248, 0.4888, 0.001403],
[0.99427, 0.50528, 0],
[0.99598, 0.52156, 0],
[0.99761, 0.53767, 0],
[0.99914, 0.55359, 0],
[1, 0.56934, 0],
[1, 0.58494, 0],
[1, 0.60037, 0],
[1, 0.61565, 0],
[1, 0.63077, 0],
[1, 0.64572, 0],
[1, 0.66048, 0],
[1, 0.67504, 0],
[1, 0.68935, 0],
[1, 0.70336, 0],
[1, 0.71701, 0],
[1, 0.73023, 0],
[1, 0.74293, 0],
[0.99701, 0.75501, 0],
[0.99313, 0.76637, 0],
[0.98833, 0.77691, 0],
[0.98256, 0.78651, 0],
[0.97574, 0.79507, 0],
[0.96785, 0.80251, 0],
[0.95886, 0.80875, 0],
[0.94877, 0.81373, 0],
[0.93761, 0.81743, 0],
[0.92543, 0.81985, 0],
[0.91229, 0.821, 0],
[0.89826, 0.82095, 0],
[0.88344, 0.81974, 0],
[0.86793, 0.8175, 0],
[0.8518, 0.8143, 0],
[0.83517, 0.81025, 0],
[0.81812, 0.80549, 0],
[0.80073, 0.8001, 0],
[0.78306, 0.7942, 0],
[0.76518, 0.78786, 0],
[0.74713, 0.7812, 0],
[0.72895, 0.77426, 0],
[0.71067, 0.76712, 0],
[0.69231, 0.75982, 0],
[0.67387, 0.7524, 0],
[0.65537, 0.74489, 0],
[0.63682, 0.73731, 0],
[0.61821, 0.72968, 0],
[0.59952, 0.722, 0],
[0.58077, 0.7143, 8.8e-05],
[0.56197, 0.7066, 0.001364],
[0.54309, 0.69889, 0.002744],
[0.52413, 0.69121, 0.004275],
[0.50511, 0.68355, 0.006019],
[0.48603, 0.67593, 0.008055],
[0.46689, 0.66839, 0.010459],
[0.4477, 0.66094, 0.013625],
[0.42851, 0.65363, 0.017232],
[0.40933, 0.6465, 0.021673],
[0.39021, 0.63962, 0.027147],
[0.37122, 0.63303, 0.033854],
[0.35244, 0.62683, 0.042288],
[0.33395, 0.62109, 0.05131],
[0.31583, 0.61589, 0.06103],
[0.29817, 0.61133, 0.071492],
[0.28114, 0.60749, 0.082491],
[0.26486, 0.60444, 0.094141],
[0.24945, 0.60226, 0.10646],
[0.23509, 0.60102, 0.11935],
[0.22182, 0.60075, 0.13293],
[0.20988, 0.60148, 0.14712],
[0.19934, 0.60321, 0.16196],
[0.19034, 0.60592, 0.17737],
[0.18291, 0.60959, 0.19335],
[0.17711, 0.61414, 0.20988],
[0.17287, 0.61953, 0.22692],
[0.17015, 0.62567, 0.24439],
[0.16879, 0.63248, 0.2623],
[0.16861, 0.63987, 0.28053],
[0.16945, 0.64776, 0.2991],
[0.17103, 0.65608, 0.31789],
[0.17321, 0.66474, 0.3369],
[0.17578, 0.67369, 0.35607],
[0.17862, 0.68287, 0.37537],
[0.18147, 0.69224, 0.39479],
[0.18433, 0.70173, 0.4143],
[0.18705, 0.71136, 0.43387],
[0.18954, 0.72106, 0.45351],
[0.19176, 0.73084, 0.4732],
[0.19369, 0.74067, 0.49294],
[0.19522, 0.75056, 0.51274],
[0.19644, 0.76048, 0.53259],
[0.1972, 0.77042, 0.55248],
[0.19751, 0.78036, 0.5724],
[0.19739, 0.7903, 0.59236],
[0.19681, 0.80022, 0.61235],
[0.19573, 0.81009, 0.63237],
[0.19423, 0.81989, 0.65241],
[0.1922, 0.82959, 0.67244],
[0.18974, 0.83913, 0.69245],
[0.18683, 0.84847, 0.71238],
[0.1835, 0.85752, 0.73222],
[0.1798, 0.86622, 0.75189],
[0.17577, 0.87446, 0.77133],
[0.17154, 0.88215, 0.79046],
[0.1672, 0.88917, 0.8092],
[0.16285, 0.89541, 0.82742],
[0.15866, 0.90075, 0.84505],
[0.15479, 0.90508, 0.86196],
[0.15145, 0.9083, 0.87804],
[0.1488, 0.91033, 0.89319],
[0.147, 0.9111, 0.90734],
[0.14614, 0.91058, 0.9204],
[0.14627, 0.90875, 0.93232],
[0.1474, 0.90562, 0.94309],
[0.14944, 0.90123, 0.9527],
[0.15222, 0.89566, 0.96117],
[0.15567, 0.88898, 0.96855],
[0.15952, 0.8813, 0.97492],
[0.16369, 0.87272, 0.98035],
[0.16789, 0.86336, 0.98494],
[0.17206, 0.85333, 0.9888],
[0.17603, 0.84276, 0.99201],
[0.17975, 0.83174, 0.99468],
[0.18308, 0.82034, 0.99691],
[0.18603, 0.80867, 0.99878],
[0.18848, 0.79679, 1],
[0.19041, 0.78475, 1],
[0.19184, 0.7726, 1],
[0.19274, 0.76037, 1],
[0.1931, 0.74809, 1],
[0.1929, 0.73578, 1],
[0.19214, 0.72346, 1],
[0.19086, 0.71113, 1],
[0.18909, 0.6988, 1],
[0.18679, 0.68652, 1],
[0.18401, 0.67427, 1],
[0.18079, 0.66207, 1],
[0.17723, 0.64994, 1],
[0.17333, 0.63789, 1],
[0.16932, 0.62596, 1],
[0.16523, 0.61417, 1],
[0.16146, 0.60259, 1],
[0.15814, 0.59124, 1],
[0.1557, 0.58022, 1],
[0.1545, 0.56956, 1],
[0.15501, 0.55939, 1],
[0.15757, 0.54976, 1],
[0.16255, 0.54079, 1],
[0.17005, 0.53257, 1],
[0.18013, 0.52519, 1],
[0.19269, 0.51875, 1],
[0.20747, 0.51331, 1],
[0.22414, 0.50895, 1],
[0.24245, 0.50571, 1],
[0.26202, 0.50364, 1],
[0.28258, 0.50272, 1],
[0.30381, 0.50294, 1],
[0.32553, 0.50429, 1],
[0.34755, 0.50666, 1],
[0.36963, 0.51003, 1],
[0.39169, 0.51429, 1],
[0.41358, 0.51935, 1],
[0.43525, 0.52511, 1],
[0.45659, 0.53148, 1],
[0.4776, 0.53838, 1],
[0.49823, 0.5457, 1],
[0.51845, 0.55337, 1],
[0.53827, 0.56134, 1],
[0.5577, 0.56954, 1],
[0.57675, 0.57793, 1],
[0.59542, 0.58647, 1],
[0.61374, 0.59512, 1],
[0.63175, 0.60388, 1],
[0.64945, 0.61271, 1],
[0.66687, 0.6216, 1],
[0.68404, 0.63055, 1],
[0.70098, 0.6395, 1],
[0.7177, 0.64845, 1],
[0.73421, 0.65739, 1],
[0.75055, 0.66627, 1],
[0.76671, 0.67509, 1],
[0.78273, 0.68379, 1],
[0.7986, 0.69233, 1],
[0.81433, 0.70062, 1],
[0.8299, 0.70861, 1],
[0.8453, 0.7162, 1],
[0.86052, 0.72328, 1],
[0.8755, 0.72977, 1],
[0.89021, 0.73553, 0.99966],
[0.90458, 0.74046, 0.99387],
[0.91855, 0.74442, 0.987],
[0.93205, 0.74732, 0.97894],
[0.94498, 0.74904, 0.96964],
[0.95728, 0.74952, 0.95905],
[0.96886, 0.74869, 0.94717],
[0.97965, 0.74653, 0.93399],
[0.9896, 0.74302, 0.91958],
[0.99865, 0.73818, 0.90399],
[1, 0.73207, 0.88731],
[1, 0.72474, 0.86966],
[1, 0.7163, 0.85114],
[1, 0.70684, 0.83189],
[1, 0.69646, 0.81201],
[1, 0.68529, 0.79164],
[1, 0.67343, 0.77088],
[1, 0.66099, 0.74982],
[1, 0.64806, 0.72855],
[1, 0.63473, 0.70715],
[1, 0.62107, 0.68567],
[1, 0.60713, 0.66415],
[1, 0.59297, 0.64264],
[1, 0.5786, 0.62116],
[1, 0.56409, 0.59973],
[1, 0.54941, 0.57836],
[1, 0.53459, 0.55708],
[1, 0.51964, 0.53586],
[1, 0.50455, 0.51475],
[1, 0.48929, 0.49372],
[1, 0.47391, 0.4728],
[1, 0.45838, 0.452],
[1, 0.4427, 0.4313],
[1, 0.42686, 0.41072],
[1, 0.41086, 0.39028],
[1, 0.39472, 0.37],
[1, 0.37845, 0.34992],
[1, 0.36209, 0.33004],
[1, 0.34568, 0.31042],
[1, 0.32933, 0.29109],
[0.99872, 0.31308, 0.2721],
[0.99487, 0.29714, 0.25356],
[0.99105, 0.28163, 0.23549],
[0.98733, 0.26683, 0.21793],
[0.98375, 0.25301, 0.20097],
[0.98037, 0.24049, 0.18472],
[0.97725, 0.22969, 0.16917],
[0.97445, 0.22104, 0.15432],
[0.97202, 0.21483, 0.14028],
[0.96999, 0.21152, 0.127],
[0.9684, 0.21123, 0.11451],
]
b_cyclic_rygcbmr_50_90_c64 = bokeh_palette('cyclic_rygcbmr_50_90_c64',cyclic_rygcbmr_50_90_c64)
m_cyclic_rygcbmr_50_90_c64 = mpl_cm('cyclic_rygcbmr_50_90_c64',cyclic_rygcbmr_50_90_c64)
m_cyclic_rygcbmr_50_90_c64_r = mpl_cm('cyclic_rygcbmr_50_90_c64_r',list(reversed(cyclic_rygcbmr_50_90_c64)))
CET_C6 = b_cyclic_rygcbmr_50_90_c64
m_CET_C6 = m_cyclic_rygcbmr_50_90_c64
m_CET_C6_r = m_cyclic_rygcbmr_50_90_c64_r
palette['CET_C6'] = b_cyclic_rygcbmr_50_90_c64
cm['CET_C6'] = m_cyclic_rygcbmr_50_90_c64
cm['CET_C6_r'] = m_cyclic_rygcbmr_50_90_c64_r
register_cmap('cet_CET_C6',m_cyclic_rygcbmr_50_90_c64)
register_cmap('cet_CET_C6_r',m_cyclic_rygcbmr_50_90_c64_r)




cyclic_ymcgy_60_90_c67 = [  # cmap_def
[0.91364, 0.89436, 0.098235],
[0.92016, 0.89105, 0.12325],
[0.92621, 0.88677, 0.14733],
[0.93184, 0.88159, 0.17055],
[0.93708, 0.87559, 0.19302],
[0.94198, 0.86888, 0.21474],
[0.94659, 0.86158, 0.23574],
[0.95094, 0.8538, 0.25594],
[0.95507, 0.84565, 0.27543],
[0.95901, 0.83722, 0.29417],
[0.96279, 0.82857, 0.31222],
[0.96641, 0.81976, 0.32966],
[0.9699, 0.81085, 0.34647],
[0.97326, 0.80183, 0.36274],
[0.97649, 0.79275, 0.37853],
[0.9796, 0.78361, 0.39387],
[0.9826, 0.77441, 0.40882],
[0.98549, 0.76518, 0.4234],
[0.98826, 0.75588, 0.43769],
[0.99092, 0.74654, 0.45167],
[0.99347, 0.73715, 0.46538],
[0.9959, 0.72769, 0.47889],
[0.99823, 0.71819, 0.49218],
[1, 0.70862, 0.50527],
[1, 0.69898, 0.51821],
[1, 0.68929, 0.53098],
[1, 0.67952, 0.54361],
[1, 0.66969, 0.5561],
[1, 0.65977, 0.56849],
[1, 0.64977, 0.58076],
[1, 0.63969, 0.59293],
[1, 0.62952, 0.605],
[1, 0.61926, 0.61699],
[1, 0.6089, 0.62889],
[1, 0.59842, 0.64072],
[1, 0.58784, 0.65249],
[1, 0.57715, 0.66419],
[1, 0.56633, 0.67584],
[1, 0.55536, 0.68743],
[1, 0.54427, 0.69897],
[1, 0.53302, 0.71047],
[1, 0.5216, 0.72192],
[1, 0.51002, 0.73334],
[1, 0.49826, 0.74471],
[1, 0.48626, 0.75606],
[1, 0.47407, 0.76737],
[1, 0.46163, 0.77866],
[1, 0.44893, 0.78992],
[1, 0.43595, 0.80115],
[1, 0.42263, 0.81238],
[1, 0.409, 0.82359],
[1, 0.39503, 0.83479],
[1, 0.38072, 0.84596],
[1, 0.36605, 0.85713],
[1, 0.35109, 0.86827],
[1, 0.3359, 0.87936],
[1, 0.32065, 0.89038],
[1, 0.30555, 0.90129],
[1, 0.29093, 0.91203],
[1, 0.2773, 0.9225],
[1, 0.26521, 0.9326],
[0.99941, 0.25538, 0.94224],
[0.99574, 0.24849, 0.95125],
[0.99172, 0.24513, 0.95954],
[0.98735, 0.24573, 0.967],
[0.98268, 0.25032, 0.97354],
[0.97773, 0.25863, 0.97914],
[0.97256, 0.27007, 0.9838],
[0.96718, 0.28393, 0.98756],
[0.96164, 0.29956, 0.99049],
[0.95598, 0.31629, 0.99271],
[0.95017, 0.33361, 0.99432],
[0.94427, 0.35113, 0.99545],
[0.93826, 0.36855, 0.99621],
[0.93214, 0.38569, 0.99669],
[0.92593, 0.40246, 0.99697],
[0.91961, 0.41878, 0.99713],
[0.91319, 0.43464, 0.99719],
[0.90666, 0.45004, 0.99719],
[0.90002, 0.46499, 0.99716],
[0.89326, 0.47954, 0.9971],
[0.8864, 0.49368, 0.99703],
[0.8794, 0.50748, 0.99695],
[0.87228, 0.52093, 0.99687],
[0.86502, 0.53409, 0.99678],
[0.85763, 0.54698, 0.99669],
[0.8501, 0.55961, 0.99659],
[0.84243, 0.572, 0.99648],
[0.8346, 0.58417, 0.99637],
[0.82662, 0.59614, 0.99625],
[0.81847, 0.60793, 0.99613],
[0.81015, 0.61953, 0.996],
[0.80166, 0.63097, 0.99587],
[0.79299, 0.64224, 0.99572],
[0.78413, 0.65337, 0.99558],
[0.77506, 0.66437, 0.99543],
[0.7658, 0.67523, 0.99527],
[0.7563, 0.68597, 0.9951],
[0.74658, 0.69659, 0.99493],
[0.73662, 0.7071, 0.99475],
[0.72642, 0.71752, 0.99457],
[0.71594, 0.72781, 0.99438],
[0.70518, 0.73802, 0.99418],
[0.69412, 0.74815, 0.99398],
[0.68276, 0.75818, 0.99377],
[0.67105, 0.76813, 0.99355],
[0.65899, 0.778, 0.99333],
[0.64655, 0.7878, 0.9931],
[0.6337, 0.79752, 0.99286],
[0.62042, 0.80718, 0.99262],
[0.60664, 0.81677, 0.99237],
[0.59237, 0.8263, 0.99211],
[0.57752, 0.83577, 0.99185],
[0.56207, 0.84518, 0.99156],
[0.54593, 0.85452, 0.99125],
[0.52906, 0.86379, 0.99088],
[0.51136, 0.87297, 0.99045],
[0.4928, 0.88204, 0.9899],
[0.47328, 0.89096, 0.98919],
[0.45275, 0.8997, 0.98824],
[0.43114, 0.90817, 0.98697],
[0.40843, 0.91628, 0.98526],
[0.38467, 0.92395, 0.98299],
[0.35995, 0.93103, 0.98003],
[0.33446, 0.9374, 0.97626],
[0.30845, 0.94293, 0.97156],
[0.28235, 0.94749, 0.96585],
[0.25667, 0.951, 0.95907],
[0.2321, 0.95339, 0.95123],
[0.20936, 0.95464, 0.94235],
[0.18932, 0.95479, 0.93251],
[0.17275, 0.9539, 0.92182],
[0.16023, 0.95208, 0.91041],
[0.15207, 0.94946, 0.8984],
[0.14798, 0.94616, 0.88593],
[0.14731, 0.94234, 0.87312],
[0.14921, 0.9381, 0.86006],
[0.15278, 0.93357, 0.84684],
[0.15734, 0.92883, 0.83352],
[0.16233, 0.92395, 0.82014],
[0.16734, 0.91898, 0.80674],
[0.17216, 0.91396, 0.79332],
[0.17668, 0.9089, 0.77991],
[0.18084, 0.90383, 0.76651],
[0.18465, 0.89876, 0.75313],
[0.18802, 0.89369, 0.73978],
[0.19095, 0.88864, 0.72643],
[0.19355, 0.88358, 0.71311],
[0.19578, 0.87854, 0.6998],
[0.19769, 0.8735, 0.68651],
[0.19927, 0.86847, 0.67324],
[0.20056, 0.86345, 0.65998],
[0.20158, 0.85842, 0.64673],
[0.20232, 0.85341, 0.63349],
[0.20279, 0.84841, 0.62027],
[0.203, 0.84341, 0.60705],
[0.20295, 0.83842, 0.59384],
[0.20266, 0.83344, 0.58064],
[0.20212, 0.82847, 0.56744],
[0.20135, 0.82349, 0.55424],
[0.20034, 0.81853, 0.54104],
[0.19911, 0.81358, 0.52784],
[0.19765, 0.80863, 0.51464],
[0.19596, 0.80369, 0.50143],
[0.19405, 0.79875, 0.48821],
[0.19186, 0.79383, 0.47497],
[0.18947, 0.78891, 0.4617],
[0.18684, 0.78399, 0.44841],
[0.18396, 0.77908, 0.4351],
[0.18083, 0.77418, 0.42173],
[0.17749, 0.76928, 0.40834],
[0.17383, 0.7644, 0.39489],
[0.16993, 0.75952, 0.38139],
[0.1657, 0.75465, 0.3678],
[0.16126, 0.74977, 0.35414],
[0.15644, 0.74492, 0.34038],
[0.1513, 0.74007, 0.32648],
[0.14591, 0.73523, 0.31245],
[0.14019, 0.73041, 0.29827],
[0.1343, 0.72562, 0.28391],
[0.12822, 0.72087, 0.26936],
[0.12209, 0.71621, 0.25456],
[0.1163, 0.71163, 0.2395],
[0.11098, 0.70721, 0.22417],
[0.10666, 0.70299, 0.20861],
[0.10379, 0.69904, 0.19277],
[0.10298, 0.69545, 0.17668],
[0.10462, 0.6923, 0.1604],
[0.10917, 0.68966, 0.14403],
[0.11647, 0.68763, 0.12754],
[0.12634, 0.68625, 0.11109],
[0.13841, 0.68558, 0.094713],
[0.15215, 0.68561, 0.078354],
[0.16719, 0.68634, 0.062319],
[0.18301, 0.68773, 0.046282],
[0.19934, 0.6897, 0.030835],
[0.21593, 0.69218, 0.018538],
[0.23246, 0.69508, 0.008874],
[0.24886, 0.69831, 0.001713],
[0.26497, 0.70181, 0],
[0.28077, 0.7055, 0],
[0.29623, 0.70932, 0],
[0.3113, 0.71322, 0],
[0.326, 0.71718, 0],
[0.34038, 0.72116, 0],
[0.35442, 0.72518, 0],
[0.36816, 0.72918, 0],
[0.38163, 0.7332, 0],
[0.39485, 0.7372, 0],
[0.40784, 0.74119, 0],
[0.42063, 0.74518, 0],
[0.43324, 0.74916, 0],
[0.44569, 0.75313, 0],
[0.45799, 0.7571, 0],
[0.47015, 0.76106, 0],
[0.48219, 0.76501, 0],
[0.49411, 0.76894, 0],
[0.50591, 0.77288, 0],
[0.51764, 0.77681, 0],
[0.52927, 0.78073, 0],
[0.5408, 0.78464, 0],
[0.55228, 0.78855, 0],
[0.56368, 0.79244, 0],
[0.575, 0.79633, 0],
[0.58627, 0.80022, 0],
[0.59749, 0.80409, 0],
[0.60865, 0.80795, 0],
[0.61976, 0.81181, 0],
[0.63083, 0.81567, 0],
[0.64185, 0.81951, 0],
[0.65283, 0.82335, 0],
[0.66378, 0.82718, 0],
[0.67469, 0.83101, 0],
[0.68557, 0.83482, 0],
[0.69642, 0.83863, 0],
[0.70724, 0.84243, 0],
[0.71804, 0.84622, 0],
[0.72881, 0.85001, 0],
[0.73956, 0.85379, 0],
[0.75028, 0.85756, 0],
[0.76099, 0.86133, 0],
[0.77167, 0.86508, 0],
[0.78232, 0.86883, 0],
[0.79293, 0.87253, 0],
[0.8035, 0.87619, 0],
[0.814, 0.87977, 0],
[0.82441, 0.88325, 0],
[0.8347, 0.88658, 0],
[0.84482, 0.88967, 0],
[0.85473, 0.89247, 0],
[0.86437, 0.89485, 0],
[0.87367, 0.89671, 0],
[0.88259, 0.89793, 0],
[0.89108, 0.8984, 0.01207],
[0.8991, 0.89801, 0.041917],
[0.90662, 0.89668, 0.071606],
]
b_cyclic_ymcgy_60_90_c67 = bokeh_palette('cyclic_ymcgy_60_90_c67',cyclic_ymcgy_60_90_c67)
m_cyclic_ymcgy_60_90_c67 = mpl_cm('cyclic_ymcgy_60_90_c67',cyclic_ymcgy_60_90_c67)
m_cyclic_ymcgy_60_90_c67_r = mpl_cm('cyclic_ymcgy_60_90_c67_r',list(reversed(cyclic_ymcgy_60_90_c67)))
CET_C7 = b_cyclic_ymcgy_60_90_c67
m_CET_C7 = m_cyclic_ymcgy_60_90_c67
m_CET_C7_r = m_cyclic_ymcgy_60_90_c67_r
palette['CET_C7'] = b_cyclic_ymcgy_60_90_c67
cm['CET_C7'] = m_cyclic_ymcgy_60_90_c67
cm['CET_C7_r'] = m_cyclic_ymcgy_60_90_c67_r
register_cmap('cet_CET_C7',m_cyclic_ymcgy_60_90_c67)
register_cmap('cet_CET_C7_r',m_cyclic_ymcgy_60_90_c67_r)




cyclic_bgrmb_35_70_c75_s25 = [  # cmap_def
[0.97668, 0.38763, 0.95518],
[0.9715, 0.38749, 0.9631],
[0.96562, 0.38671, 0.97004],
[0.9591, 0.38535, 0.97603],
[0.95198, 0.38345, 0.98112],
[0.94434, 0.38103, 0.98535],
[0.93623, 0.37815, 0.98879],
[0.92772, 0.37488, 0.99154],
[0.91888, 0.37127, 0.99367],
[0.90976, 0.36739, 0.99527],
[0.90041, 0.36329, 0.99643],
[0.89086, 0.35899, 0.99723],
[0.88116, 0.35457, 0.99773],
[0.87133, 0.35003, 0.998],
[0.86139, 0.3454, 0.99808],
[0.85136, 0.34074, 0.99803],
[0.84126, 0.33602, 0.99788],
[0.83108, 0.33129, 0.99765],
[0.82084, 0.32655, 0.99736],
[0.81054, 0.32182, 0.99704],
[0.80018, 0.3171, 0.99669],
[0.78976, 0.31236, 0.9963],
[0.77928, 0.30766, 0.99592],
[0.76874, 0.30297, 0.99553],
[0.75814, 0.29831, 0.99513],
[0.74746, 0.29366, 0.99474],
[0.7367, 0.28905, 0.99433],
[0.72588, 0.28443, 0.99393],
[0.71496, 0.27985, 0.99352],
[0.70396, 0.27529, 0.99311],
[0.69288, 0.27076, 0.99269],
[0.68168, 0.26623, 0.99226],
[0.67041, 0.26173, 0.99183],
[0.659, 0.25727, 0.9914],
[0.64749, 0.25282, 0.99097],
[0.63586, 0.24842, 0.99053],
[0.6241, 0.244, 0.99009],
[0.61219, 0.23965, 0.98964],
[0.60014, 0.23533, 0.98919],
[0.58793, 0.23102, 0.98874],
[0.57556, 0.22675, 0.98828],
[0.563, 0.2225, 0.98782],
[0.55024, 0.2183, 0.98735],
[0.53728, 0.21413, 0.98689],
[0.5241, 0.21, 0.9864],
[0.5107, 0.20596, 0.98591],
[0.49707, 0.20195, 0.9854],
[0.48321, 0.19806, 0.98486],
[0.46912, 0.19431, 0.98429],
[0.4548, 0.19064, 0.98367],
[0.44027, 0.18719, 0.98299],
[0.42558, 0.18393, 0.98222],
[0.41078, 0.18095, 0.98134],
[0.39592, 0.17835, 0.98032],
[0.38113, 0.17608, 0.97912],
[0.36652, 0.17438, 0.9777],
[0.35226, 0.17319, 0.97601],
[0.33858, 0.17267, 0.97401],
[0.32567, 0.17287, 0.97165],
[0.31386, 0.17387, 0.96886],
[0.30335, 0.17569, 0.9656],
[0.29447, 0.17844, 0.96183],
[0.28745, 0.18201, 0.95752],
[0.28249, 0.18651, 0.95262],
[0.27965, 0.19184, 0.94712],
[0.27896, 0.19798, 0.94102],
[0.28028, 0.20489, 0.93432],
[0.28345, 0.21245, 0.92705],
[0.28818, 0.22063, 0.91923],
[0.29417, 0.22927, 0.9109],
[0.30112, 0.23836, 0.9021],
[0.30875, 0.24782, 0.89288],
[0.31674, 0.25752, 0.8833],
[0.32491, 0.26742, 0.8734],
[0.33309, 0.27749, 0.86325],
[0.3411, 0.28761, 0.85287],
[0.34885, 0.29779, 0.84232],
[0.35627, 0.30796, 0.83163],
[0.36329, 0.31812, 0.82083],
[0.3699, 0.32822, 0.80995],
[0.37608, 0.33826, 0.79901],
[0.3818, 0.34824, 0.78801],
[0.38708, 0.35813, 0.77699],
[0.39193, 0.36793, 0.76593],
[0.39635, 0.37766, 0.75486],
[0.40035, 0.38731, 0.74376],
[0.40393, 0.39688, 0.73265],
[0.40712, 0.40639, 0.72152],
[0.40992, 0.4158, 0.71039],
[0.41234, 0.42516, 0.69924],
[0.41438, 0.43445, 0.68808],
[0.41607, 0.44369, 0.6769],
[0.41741, 0.45287, 0.66569],
[0.41838, 0.46199, 0.65446],
[0.41902, 0.47107, 0.6432],
[0.41932, 0.4801, 0.63192],
[0.41927, 0.48908, 0.62061],
[0.41888, 0.49805, 0.60925],
[0.41815, 0.50695, 0.59785],
[0.41709, 0.51582, 0.58642],
[0.41568, 0.52468, 0.57493],
[0.41392, 0.53349, 0.56339],
[0.4118, 0.54228, 0.55179],
[0.40934, 0.55104, 0.54012],
[0.40651, 0.55977, 0.52839],
[0.4033, 0.56847, 0.51658],
[0.3997, 0.57715, 0.50468],
[0.39574, 0.58582, 0.49266],
[0.39136, 0.59445, 0.48055],
[0.38657, 0.60305, 0.46831],
[0.38139, 0.61161, 0.45594],
[0.37578, 0.62013, 0.44344],
[0.36976, 0.62857, 0.43077],
[0.36334, 0.63694, 0.41792],
[0.35653, 0.6452, 0.40489],
[0.34935, 0.65334, 0.39167],
[0.34186, 0.66131, 0.37823],
[0.33413, 0.66906, 0.36458],
[0.32623, 0.67654, 0.35072],
[0.31833, 0.68369, 0.33663],
[0.31058, 0.69043, 0.32235],
[0.30313, 0.69669, 0.30787],
[0.2963, 0.70239, 0.29327],
[0.2903, 0.70744, 0.27858],
[0.28545, 0.71177, 0.26385],
[0.28203, 0.71529, 0.24913],
[0.28024, 0.71794, 0.23456],
[0.28032, 0.71968, 0.22018],
[0.28239, 0.72046, 0.20611],
[0.28637, 0.72027, 0.19243],
[0.29226, 0.71911, 0.17929],
[0.29984, 0.71701, 0.16677],
[0.30887, 0.71399, 0.1549],
[0.31906, 0.71013, 0.14384],
[0.33015, 0.70547, 0.13357],
[0.34187, 0.70011, 0.12411],
[0.35397, 0.6941, 0.11555],
[0.36627, 0.68756, 0.10776],
[0.37856, 0.68054, 0.10074],
[0.39074, 0.67313, 0.094469],
[0.40269, 0.66539, 0.088741],
[0.41435, 0.65739, 0.083562],
[0.42567, 0.64917, 0.078779],
[0.43662, 0.64077, 0.074448],
[0.44718, 0.63224, 0.070334],
[0.45735, 0.6236, 0.066462],
[0.46716, 0.61486, 0.0628],
[0.47658, 0.60606, 0.059179],
[0.48564, 0.5972, 0.0556],
[0.49438, 0.58826, 0.05209],
[0.50277, 0.57928, 0.048789],
[0.51086, 0.57025, 0.045323],
[0.51866, 0.56117, 0.041976],
[0.52617, 0.55206, 0.038511],
[0.53341, 0.54288, 0.035214],
[0.54041, 0.53365, 0.031867],
[0.54716, 0.52436, 0.028877],
[0.55369, 0.51501, 0.026036],
[0.56001, 0.50559, 0.023343],
[0.56611, 0.4961, 0.020796],
[0.57202, 0.48654, 0.018393],
[0.57773, 0.4769, 0.016132],
[0.58327, 0.46717, 0.014011],
[0.58864, 0.45733, 0.012051],
[0.59384, 0.4474, 0.009965],
[0.59888, 0.43738, 0.008255],
[0.60376, 0.42723, 0.006667],
[0.60849, 0.41694, 0.005204],
[0.61307, 0.40652, 0.003866],
[0.61752, 0.39596, 0.002651],
[0.62182, 0.38521, 0.001556],
[0.62601, 0.37431, 0.000601],
[0.63008, 0.3632, 0],
[0.63404, 0.35189, 0],
[0.63787, 0.34038, 0],
[0.64161, 0.32863, 0],
[0.64526, 0.31662, 0],
[0.64884, 0.30433, 0],
[0.65235, 0.29178, 0],
[0.65582, 0.27893, 0],
[0.65925, 0.26576, 0.000716],
[0.66268, 0.25228, 0.002273],
[0.66612, 0.23847, 0.004328],
[0.66962, 0.22433, 0.006956],
[0.67318, 0.20991, 0.010213],
[0.67685, 0.19515, 0.014464],
[0.68065, 0.18008, 0.019341],
[0.68462, 0.16474, 0.025185],
[0.68878, 0.14912, 0.032125],
[0.69315, 0.13324, 0.040511],
[0.69773, 0.11711, 0.049454],
[0.70256, 0.10078, 0.058636],
[0.70762, 0.084326, 0.068259],
[0.71291, 0.067632, 0.078128],
[0.71843, 0.050818, 0.088512],
[0.72414, 0.034099, 0.099205],
[0.73004, 0.020769, 0.11034],
[0.73608, 0.011276, 0.12174],
[0.74226, 0.005477, 0.13361],
[0.74853, 0.003229, 0.14578],
[0.75487, 0.004142, 0.15823],
[0.76125, 0.007923, 0.17094],
[0.76766, 0.014473, 0.1839],
[0.77407, 0.02311, 0.19709],
[0.78047, 0.033735, 0.21042],
[0.78684, 0.045986, 0.2239],
[0.79318, 0.057778, 0.23755],
[0.79947, 0.069155, 0.25126],
[0.80571, 0.07976, 0.26505],
[0.8119, 0.090028, 0.27896],
[0.81803, 0.099812, 0.29291],
[0.8241, 0.10928, 0.30694],
[0.83011, 0.11837, 0.32103],
[0.83606, 0.12712, 0.33519],
[0.84195, 0.13567, 0.34937],
[0.84776, 0.144, 0.36361],
[0.85352, 0.15207, 0.37792],
[0.85921, 0.15997, 0.39227],
[0.86483, 0.16775, 0.40669],
[0.87039, 0.17537, 0.42116],
[0.87589, 0.18285, 0.4357],
[0.88131, 0.19026, 0.45028],
[0.88667, 0.19756, 0.46493],
[0.89195, 0.20479, 0.47964],
[0.89717, 0.21195, 0.4944],
[0.90231, 0.21901, 0.50922],
[0.90739, 0.22604, 0.5241],
[0.91239, 0.23298, 0.53904],
[0.91733, 0.2399, 0.55404],
[0.92219, 0.24678, 0.5691],
[0.92697, 0.25362, 0.58422],
[0.93168, 0.26042, 0.59939],
[0.93632, 0.26717, 0.61462],
[0.94088, 0.27393, 0.62991],
[0.94537, 0.28062, 0.64527],
[0.94976, 0.28731, 0.66069],
[0.95407, 0.29396, 0.67616],
[0.95827, 0.30058, 0.69167],
[0.96235, 0.30716, 0.70723],
[0.96631, 0.31369, 0.72282],
[0.97012, 0.32016, 0.73844],
[0.97376, 0.32652, 0.75407],
[0.9772, 0.33283, 0.76967],
[0.98041, 0.33898, 0.78524],
[0.98332, 0.34498, 0.80073],
[0.98591, 0.35079, 0.81609],
[0.9881, 0.35635, 0.83126],
[0.98986, 0.36162, 0.84617],
[0.9911, 0.36656, 0.86076],
[0.99177, 0.37108, 0.87494],
[0.99181, 0.37516, 0.88861],
[0.99117, 0.37874, 0.90168],
[0.98981, 0.38176, 0.91407],
[0.98769, 0.38418, 0.92568],
[0.98479, 0.38597, 0.93644],
[0.98112, 0.38713, 0.94628],
]
b_cyclic_bgrmb_35_70_c75_s25 = bokeh_palette('cyclic_bgrmb_35_70_c75_s25',cyclic_bgrmb_35_70_c75_s25)
m_cyclic_bgrmb_35_70_c75_s25 = mpl_cm('cyclic_bgrmb_35_70_c75_s25',cyclic_bgrmb_35_70_c75_s25)
m_cyclic_bgrmb_35_70_c75_s25_r = mpl_cm('cyclic_bgrmb_35_70_c75_s25_r',list(reversed(cyclic_bgrmb_35_70_c75_s25)))
CET_C11s = b_cyclic_bgrmb_35_70_c75_s25
m_CET_C11s = m_cyclic_bgrmb_35_70_c75_s25
m_CET_C11s_r = m_cyclic_bgrmb_35_70_c75_s25_r
palette['CET_C11s'] = b_cyclic_bgrmb_35_70_c75_s25
cm['CET_C11s'] = m_cyclic_bgrmb_35_70_c75_s25
cm['CET_C11s_r'] = m_cyclic_bgrmb_35_70_c75_s25_r
register_cmap('cet_CET_C11s',m_cyclic_bgrmb_35_70_c75_s25)
register_cmap('cet_CET_C11s_r',m_cyclic_bgrmb_35_70_c75_s25_r)




cyclic_wrkbw_10_90_c43 = [  # cmap_def
[0.88141, 0.84341, 0.85645],
[0.88642, 0.84153, 0.84851],
[0.89115, 0.83869, 0.83998],
[0.8956, 0.83491, 0.83084],
[0.89977, 0.83027, 0.82115],
[0.90367, 0.82481, 0.81095],
[0.90731, 0.81863, 0.80029],
[0.9107, 0.8118, 0.7892],
[0.91385, 0.80441, 0.77777],
[0.91677, 0.79653, 0.76602],
[0.91948, 0.78825, 0.75403],
[0.92199, 0.77963, 0.74184],
[0.9243, 0.77076, 0.72951],
[0.92644, 0.76166, 0.71706],
[0.9284, 0.75241, 0.70452],
[0.93021, 0.74303, 0.69193],
[0.93185, 0.73355, 0.67931],
[0.93335, 0.72399, 0.66667],
[0.93471, 0.71438, 0.65405],
[0.93593, 0.70473, 0.64142],
[0.93702, 0.69504, 0.62882],
[0.93798, 0.68533, 0.61624],
[0.93881, 0.67559, 0.60369],
[0.93952, 0.66583, 0.59117],
[0.94012, 0.65606, 0.57868],
[0.94059, 0.64626, 0.56623],
[0.94095, 0.63645, 0.55381],
[0.94119, 0.6266, 0.54142],
[0.94132, 0.61675, 0.52908],
[0.94134, 0.60686, 0.51676],
[0.94125, 0.59694, 0.50448],
[0.94106, 0.58698, 0.49222],
[0.94076, 0.57701, 0.48],
[0.94037, 0.56699, 0.4678],
[0.93986, 0.55693, 0.45565],
[0.93925, 0.54684, 0.44352],
[0.93855, 0.5367, 0.43143],
[0.93775, 0.52651, 0.41936],
[0.93686, 0.51626, 0.40732],
[0.93587, 0.50596, 0.39531],
[0.93478, 0.49561, 0.38333],
[0.93361, 0.48519, 0.37136],
[0.93235, 0.47471, 0.35943],
[0.931, 0.46415, 0.34753],
[0.92955, 0.4535, 0.33565],
[0.928, 0.44279, 0.32378],
[0.92636, 0.43199, 0.31194],
[0.9246, 0.42108, 0.30017],
[0.92274, 0.41012, 0.28842],
[0.92074, 0.39905, 0.27674],
[0.91861, 0.38791, 0.26509],
[0.91631, 0.37671, 0.25356],
[0.91384, 0.36544, 0.24213],
[0.91115, 0.35415, 0.23083],
[0.90822, 0.34286, 0.21971],
[0.905, 0.33158, 0.20879],
[0.90147, 0.32042, 0.19814],
[0.89757, 0.30939, 0.18785],
[0.89326, 0.29854, 0.1779],
[0.88851, 0.28799, 0.16836],
[0.88326, 0.27783, 0.15935],
[0.87749, 0.26809, 0.15092],
[0.87116, 0.25891, 0.14313],
[0.86426, 0.25031, 0.13596],
[0.85677, 0.24243, 0.12959],
[0.84871, 0.23529, 0.12389],
[0.84007, 0.22889, 0.11901],
[0.83088, 0.22328, 0.1148],
[0.82118, 0.21843, 0.11142],
[0.811, 0.21431, 0.10862],
[0.80039, 0.21086, 0.10655],
[0.7894, 0.20802, 0.10489],
[0.77807, 0.2057, 0.10386],
[0.76647, 0.20383, 0.10319],
[0.75465, 0.20228, 0.10284],
[0.74264, 0.20102, 0.10278],
[0.73048, 0.19998, 0.10293],
[0.71822, 0.1991, 0.10324],
[0.70589, 0.19832, 0.10365],
[0.6935, 0.19759, 0.10411],
[0.68109, 0.19691, 0.10462],
[0.66868, 0.1962, 0.10519],
[0.65627, 0.19546, 0.10581],
[0.64385, 0.19473, 0.1064],
[0.63148, 0.19394, 0.10695],
[0.61912, 0.19309, 0.10746],
[0.60679, 0.19219, 0.10794],
[0.59449, 0.19125, 0.10842],
[0.58222, 0.19025, 0.1089],
[0.57, 0.1892, 0.10936],
[0.55781, 0.18812, 0.10977],
[0.54567, 0.18693, 0.11013],
[0.53354, 0.18573, 0.11046],
[0.52147, 0.18443, 0.11077],
[0.50943, 0.18309, 0.11106],
[0.49743, 0.18171, 0.11132],
[0.48545, 0.18029, 0.11156],
[0.47351, 0.17882, 0.11177],
[0.46162, 0.1773, 0.11196],
[0.44976, 0.17568, 0.11212],
[0.43793, 0.17406, 0.11226],
[0.42612, 0.17237, 0.11238],
[0.41437, 0.17064, 0.11248],
[0.40264, 0.16888, 0.11255],
[0.39096, 0.16705, 0.1126],
[0.37929, 0.16512, 0.11263],
[0.36765, 0.16323, 0.11264],
[0.35607, 0.16126, 0.11263],
[0.3445, 0.15922, 0.11262],
[0.33301, 0.15716, 0.11261],
[0.32154, 0.15507, 0.1126],
[0.31017, 0.15294, 0.11262],
[0.29885, 0.1508, 0.11268],
[0.28762, 0.14863, 0.11279],
[0.27655, 0.14648, 0.11299],
[0.26561, 0.14436, 0.1133],
[0.25488, 0.14225, 0.11373],
[0.24434, 0.14019, 0.11434],
[0.23414, 0.13827, 0.11526],
[0.22424, 0.13639, 0.11649],
[0.21479, 0.13476, 0.11804],
[0.20581, 0.13325, 0.11997],
[0.19735, 0.132, 0.12239],
[0.18949, 0.13103, 0.12541],
[0.18227, 0.13038, 0.12894],
[0.17577, 0.13006, 0.13316],
[0.17005, 0.13011, 0.13804],
[0.16504, 0.13055, 0.14357],
[0.16085, 0.13138, 0.14984],
[0.15741, 0.13265, 0.15679],
[0.15475, 0.13435, 0.16446],
[0.15281, 0.13638, 0.17277],
[0.15159, 0.13886, 0.18173],
[0.151, 0.1417, 0.19131],
[0.151, 0.14485, 0.20141],
[0.1515, 0.14824, 0.21205],
[0.15245, 0.15192, 0.22308],
[0.15379, 0.15585, 0.2345],
[0.15544, 0.15988, 0.24624],
[0.15729, 0.16417, 0.25828],
[0.15933, 0.16851, 0.27059],
[0.16152, 0.17298, 0.28308],
[0.16373, 0.17756, 0.29575],
[0.16592, 0.18214, 0.3086],
[0.16815, 0.18684, 0.32156],
[0.17034, 0.19156, 0.33467],
[0.17243, 0.19636, 0.34786],
[0.17447, 0.20112, 0.36115],
[0.17635, 0.20599, 0.37455],
[0.17822, 0.21083, 0.38802],
[0.17987, 0.21572, 0.40157],
[0.1814, 0.22063, 0.41521],
[0.18281, 0.22554, 0.42891],
[0.18408, 0.23047, 0.4427],
[0.18522, 0.23545, 0.45655],
[0.18618, 0.24041, 0.47048],
[0.18698, 0.24541, 0.48447],
[0.18766, 0.25044, 0.49855],
[0.18815, 0.2555, 0.51268],
[0.18846, 0.26055, 0.5269],
[0.18861, 0.26564, 0.54116],
[0.18858, 0.27075, 0.5555],
[0.18837, 0.27587, 0.5699],
[0.18798, 0.28097, 0.58438],
[0.18736, 0.28613, 0.59891],
[0.18653, 0.29133, 0.61351],
[0.18553, 0.29651, 0.62818],
[0.18424, 0.30171, 0.6429],
[0.18274, 0.30695, 0.65768],
[0.18099, 0.31217, 0.67253],
[0.179, 0.31745, 0.68742],
[0.17672, 0.32274, 0.70237],
[0.17419, 0.32802, 0.71736],
[0.17136, 0.33335, 0.73238],
[0.16826, 0.33867, 0.74741],
[0.16491, 0.34402, 0.76244],
[0.16137, 0.3494, 0.77746],
[0.15757, 0.3548, 0.79243],
[0.15368, 0.3602, 0.8073],
[0.14983, 0.36564, 0.82205],
[0.14607, 0.3711, 0.8366],
[0.14266, 0.37661, 0.8509],
[0.13982, 0.38211, 0.86489],
[0.13797, 0.38767, 0.87847],
[0.1373, 0.39327, 0.89156],
[0.13822, 0.39891, 0.90407],
[0.14103, 0.40458, 0.91592],
[0.14591, 0.41033, 0.92702],
[0.15293, 0.4161, 0.93729],
[0.16214, 0.42193, 0.94667],
[0.17323, 0.42784, 0.95511],
[0.18606, 0.43378, 0.96258],
[0.20021, 0.43981, 0.96906],
[0.21551, 0.44588, 0.97457],
[0.23159, 0.45204, 0.97914],
[0.24818, 0.45825, 0.98281],
[0.26504, 0.46453, 0.98564],
[0.28205, 0.47086, 0.9877],
[0.29898, 0.47725, 0.98908],
[0.31574, 0.48369, 0.98987],
[0.33223, 0.4902, 0.99013],
[0.34838, 0.49674, 0.98997],
[0.36416, 0.50334, 0.98944],
[0.37954, 0.50998, 0.98862],
[0.3945, 0.51666, 0.98757],
[0.40905, 0.52338, 0.98633],
[0.42318, 0.53013, 0.98496],
[0.43695, 0.53692, 0.98347],
[0.45032, 0.54373, 0.98191],
[0.46333, 0.55057, 0.98029],
[0.47603, 0.55745, 0.97861],
[0.48841, 0.56434, 0.97691],
[0.5005, 0.57126, 0.97517],
[0.51231, 0.5782, 0.97342],
[0.52387, 0.58518, 0.97166],
[0.53518, 0.59218, 0.96988],
[0.54629, 0.59919, 0.96809],
[0.55717, 0.60622, 0.96629],
[0.56786, 0.61328, 0.96448],
[0.57837, 0.62038, 0.96266],
[0.58871, 0.62748, 0.96082],
[0.59889, 0.63461, 0.95897],
[0.60893, 0.64175, 0.95711],
[0.61881, 0.64892, 0.95524],
[0.62855, 0.65611, 0.95336],
[0.63816, 0.66332, 0.95146],
[0.64766, 0.67054, 0.94955],
[0.65704, 0.67779, 0.94763],
[0.6663, 0.68505, 0.94569],
[0.67546, 0.69234, 0.94374],
[0.68452, 0.69963, 0.94177],
[0.69348, 0.70696, 0.93979],
[0.70235, 0.71429, 0.9378],
[0.71114, 0.72164, 0.93579],
[0.71983, 0.72902, 0.93377],
[0.72843, 0.73641, 0.93173],
[0.73696, 0.7438, 0.92968],
[0.7454, 0.75119, 0.92759],
[0.75376, 0.75857, 0.92549],
[0.76204, 0.76591, 0.92334],
[0.77023, 0.77321, 0.92114],
[0.77834, 0.78046, 0.91889],
[0.78635, 0.7876, 0.91657],
[0.79425, 0.7946, 0.91414],
[0.80203, 0.80142, 0.9116],
[0.8097, 0.80801, 0.9089],
[0.81723, 0.81429, 0.90602],
[0.8246, 0.8202, 0.9029],
[0.83181, 0.82566, 0.8995],
[0.83882, 0.83059, 0.89578],
[0.84564, 0.83488, 0.89168],
[0.85223, 0.83847, 0.88714],
[0.85859, 0.84127, 0.88212],
[0.8647, 0.84321, 0.87657],
[0.87054, 0.84425, 0.87046],
[0.87611, 0.84432, 0.86377],
]
b_cyclic_wrkbw_10_90_c43 = bokeh_palette('cyclic_wrkbw_10_90_c43',cyclic_wrkbw_10_90_c43)
m_cyclic_wrkbw_10_90_c43 = mpl_cm('cyclic_wrkbw_10_90_c43',cyclic_wrkbw_10_90_c43)
m_cyclic_wrkbw_10_90_c43_r = mpl_cm('cyclic_wrkbw_10_90_c43_r',list(reversed(cyclic_wrkbw_10_90_c43)))
CET_C3 = b_cyclic_wrkbw_10_90_c43
m_CET_C3 = m_cyclic_wrkbw_10_90_c43
m_CET_C3_r = m_cyclic_wrkbw_10_90_c43_r
palette['CET_C3'] = b_cyclic_wrkbw_10_90_c43
cm['CET_C3'] = m_cyclic_wrkbw_10_90_c43
cm['CET_C3_r'] = m_cyclic_wrkbw_10_90_c43_r
register_cmap('cet_CET_C3',m_cyclic_wrkbw_10_90_c43)
register_cmap('cet_CET_C3_r',m_cyclic_wrkbw_10_90_c43_r)




cyclic_mygbm_50_90_c46 = [  # cmap_def
[0.91298, 0.5809, 0.58565],
[0.91561, 0.585, 0.58093],
[0.91812, 0.58914, 0.57627],
[0.92049, 0.59335, 0.57161],
[0.92273, 0.59761, 0.567],
[0.92482, 0.60191, 0.56242],
[0.92679, 0.60626, 0.55785],
[0.92862, 0.61066, 0.55333],
[0.93032, 0.61509, 0.54885],
[0.93188, 0.61957, 0.54438],
[0.93331, 0.62409, 0.53995],
[0.93459, 0.62864, 0.53556],
[0.93575, 0.63323, 0.53121],
[0.93678, 0.63785, 0.5269],
[0.93767, 0.6425, 0.52261],
[0.93843, 0.64718, 0.51836],
[0.93907, 0.65188, 0.51414],
[0.93959, 0.65661, 0.50995],
[0.93997, 0.66135, 0.50579],
[0.94023, 0.66612, 0.50168],
[0.94036, 0.67091, 0.49761],
[0.94036, 0.67571, 0.49356],
[0.94024, 0.68052, 0.48955],
[0.94, 0.68536, 0.48557],
[0.93965, 0.69019, 0.48163],
[0.93917, 0.69504, 0.47772],
[0.93858, 0.69989, 0.47384],
[0.93786, 0.70476, 0.47001],
[0.93703, 0.70963, 0.4662],
[0.93609, 0.71449, 0.46242],
[0.93503, 0.71937, 0.45868],
[0.93387, 0.72424, 0.45497],
[0.93259, 0.72911, 0.45129],
[0.93119, 0.73399, 0.44763],
[0.92963, 0.73903, 0.4439],
[0.9282, 0.74337, 0.44069],
[0.92636, 0.74862, 0.43685],
[0.92454, 0.75348, 0.4333],
[0.92264, 0.75826, 0.42987],
[0.9206, 0.76309, 0.4264],
[0.91845, 0.76791, 0.42298],
[0.91618, 0.77273, 0.41961],
[0.91378, 0.77755, 0.41627],
[0.91124, 0.78234, 0.413],
[0.90856, 0.78713, 0.40979],
[0.90573, 0.7919, 0.40665],
[0.90272, 0.79665, 0.40361],
[0.89953, 0.80139, 0.40066],
[0.89616, 0.80611, 0.3978],
[0.89258, 0.8108, 0.39508],
[0.88877, 0.81546, 0.39251],
[0.88473, 0.82009, 0.3901],
[0.88043, 0.82467, 0.38788],
[0.87587, 0.82922, 0.38586],
[0.87101, 0.8337, 0.3841],
[0.86581, 0.8381, 0.38262],
[0.86026, 0.84242, 0.38148],
[0.85434, 0.84662, 0.38068],
[0.848, 0.85069, 0.38032],
[0.84126, 0.85461, 0.38043],
[0.83406, 0.85832, 0.38108],
[0.82643, 0.8618, 0.38228],
[0.81836, 0.86499, 0.38414],
[0.80989, 0.86787, 0.38663],
[0.80109, 0.8704, 0.38976],
[0.792, 0.87253, 0.39354],
[0.78272, 0.87425, 0.39789],
[0.77333, 0.87559, 0.40273],
[0.7639, 0.87653, 0.40802],
[0.75448, 0.87712, 0.41364],
[0.74512, 0.87737, 0.41956],
[0.73584, 0.87733, 0.42568],
[0.72669, 0.87701, 0.43197],
[0.71765, 0.87645, 0.43837],
[0.70871, 0.8757, 0.44485],
[0.69988, 0.87475, 0.45137],
[0.69117, 0.87363, 0.45793],
[0.68257, 0.87237, 0.46452],
[0.67405, 0.87097, 0.47109],
[0.66562, 0.86947, 0.47765],
[0.65726, 0.86786, 0.4842],
[0.64895, 0.86615, 0.49075],
[0.64069, 0.86436, 0.49724],
[0.63248, 0.86249, 0.50372],
[0.62431, 0.86055, 0.51016],
[0.61616, 0.85854, 0.51657],
[0.60804, 0.85649, 0.52295],
[0.59991, 0.85438, 0.5293],
[0.59178, 0.85222, 0.5356],
[0.58365, 0.85001, 0.54188],
[0.57552, 0.84776, 0.54813],
[0.56737, 0.84547, 0.55433],
[0.55917, 0.84316, 0.56049],
[0.55093, 0.84081, 0.56663],
[0.54268, 0.83842, 0.57274],
[0.53439, 0.83599, 0.5788],
[0.52607, 0.83354, 0.58484],
[0.5177, 0.83105, 0.59084],
[0.50929, 0.82852, 0.59682],
[0.50086, 0.82596, 0.60275],
[0.4924, 0.82336, 0.60866],
[0.4839, 0.82073, 0.61452],
[0.47537, 0.81807, 0.62037],
[0.46679, 0.81537, 0.62616],
[0.45816, 0.81263, 0.63193],
[0.4495, 0.80987, 0.63766],
[0.44083, 0.80707, 0.64335],
[0.43213, 0.80422, 0.64901],
[0.42337, 0.80135, 0.65462],
[0.41458, 0.79844, 0.6602],
[0.40576, 0.7955, 0.66575],
[0.39688, 0.79253, 0.67125],
[0.38797, 0.78953, 0.67671],
[0.37903, 0.78649, 0.68212],
[0.37008, 0.78341, 0.6875],
[0.36108, 0.78031, 0.69284],
[0.35206, 0.77717, 0.69812],
[0.343, 0.774, 0.70336],
[0.33389, 0.7708, 0.70856],
[0.32473, 0.76757, 0.7137],
[0.31557, 0.76431, 0.7188],
[0.30635, 0.76103, 0.72385],
[0.29709, 0.75771, 0.72886],
[0.28781, 0.75437, 0.7338],
[0.27853, 0.75099, 0.7387],
[0.26922, 0.74759, 0.74354],
[0.25985, 0.74416, 0.74832],
[0.25044, 0.7407, 0.75305],
[0.24101, 0.73723, 0.75773],
[0.23156, 0.73372, 0.76235],
[0.22204, 0.7302, 0.76691],
[0.2125, 0.72665, 0.77142],
[0.20291, 0.72307, 0.77586],
[0.19332, 0.71949, 0.78024],
[0.18372, 0.71587, 0.78456],
[0.17411, 0.71223, 0.78882],
[0.16448, 0.70858, 0.79302],
[0.15477, 0.7049, 0.79715],
[0.14509, 0.70121, 0.80123],
[0.13531, 0.6975, 0.80524],
[0.12553, 0.69378, 0.8092],
[0.1157, 0.69004, 0.81308],
[0.10584, 0.68628, 0.81691],
[0.096034, 0.68251, 0.82066],
[0.086314, 0.67872, 0.82435],
[0.076594, 0.67492, 0.82798],
[0.066987, 0.6711, 0.83156],
[0.057363, 0.66727, 0.83506],
[0.047913, 0.66344, 0.8385],
[0.038783, 0.65958, 0.84189],
[0.030423, 0.65572, 0.84521],
[0.024144, 0.65184, 0.84846],
[0.019383, 0.64794, 0.85165],
[0.016091, 0.64404, 0.85478],
[0.014241, 0.64014, 0.85785],
[0.013802, 0.63622, 0.86086],
[0.014758, 0.63229, 0.86382],
[0.017308, 0.62835, 0.8667],
[0.021411, 0.62439, 0.86953],
[0.026822, 0.62043, 0.8723],
[0.033498, 0.61646, 0.87502],
[0.041669, 0.61247, 0.87768],
[0.050034, 0.60848, 0.88028],
[0.058959, 0.60447, 0.88282],
[0.067796, 0.60045, 0.88532],
[0.076491, 0.59643, 0.88775],
[0.085201, 0.5924, 0.89013],
[0.093901, 0.58834, 0.89246],
[0.10275, 0.58429, 0.89473],
[0.11172, 0.58022, 0.89692],
[0.12071, 0.57613, 0.89905],
[0.12993, 0.57203, 0.90111],
[0.13919, 0.56791, 0.90309],
[0.1487, 0.56378, 0.90498],
[0.15846, 0.55963, 0.90678],
[0.1684, 0.55546, 0.90848],
[0.17864, 0.55128, 0.91008],
[0.18914, 0.54708, 0.91155],
[0.19994, 0.54287, 0.91289],
[0.21112, 0.53864, 0.91409],
[0.22262, 0.53439, 0.91514],
[0.23456, 0.53013, 0.916],
[0.24696, 0.52586, 0.91667],
[0.2598, 0.52158, 0.91712],
[0.27304, 0.51733, 0.91735],
[0.28691, 0.51307, 0.91727],
[0.30126, 0.50885, 0.91691],
[0.31619, 0.50467, 0.91621],
[0.33157, 0.50055, 0.91515],
[0.34752, 0.49655, 0.91368],
[0.36381, 0.49269, 0.9118],
[0.38047, 0.48901, 0.90948],
[0.39731, 0.48556, 0.90673],
[0.41416, 0.48237, 0.90357],
[0.43092, 0.47948, 0.90001],
[0.44739, 0.47689, 0.89611],
[0.46349, 0.47462, 0.89193],
[0.47914, 0.47267, 0.88749],
[0.49428, 0.47102, 0.88287],
[0.50887, 0.46966, 0.87809],
[0.52293, 0.46856, 0.87318],
[0.53648, 0.46769, 0.86819],
[0.54951, 0.46708, 0.86311],
[0.56208, 0.46665, 0.85797],
[0.57418, 0.46642, 0.85281],
[0.58588, 0.46635, 0.8476],
[0.59718, 0.46644, 0.84238],
[0.60812, 0.46669, 0.83713],
[0.61872, 0.46707, 0.83188],
[0.62898, 0.46758, 0.82661],
[0.63895, 0.46819, 0.82133],
[0.64864, 0.46894, 0.81606],
[0.65807, 0.46978, 0.81078],
[0.66725, 0.47071, 0.8055],
[0.6762, 0.47174, 0.80023],
[0.68493, 0.47284, 0.79496],
[0.69346, 0.47404, 0.78969],
[0.70181, 0.47531, 0.78442],
[0.70998, 0.47664, 0.77914],
[0.71795, 0.47804, 0.77389],
[0.72576, 0.47953, 0.76863],
[0.73341, 0.48106, 0.76338],
[0.74091, 0.48267, 0.75813],
[0.74827, 0.48433, 0.75287],
[0.75547, 0.48607, 0.74764],
[0.76251, 0.4879, 0.7424],
[0.76941, 0.48977, 0.73717],
[0.77618, 0.49174, 0.73194],
[0.7828, 0.49375, 0.72672],
[0.78929, 0.49584, 0.72151],
[0.79562, 0.49803, 0.71631],
[0.80181, 0.50026, 0.71112],
[0.80788, 0.50259, 0.70593],
[0.8138, 0.50499, 0.70075],
[0.81959, 0.50746, 0.69559],
[0.82525, 0.51, 0.69043],
[0.83078, 0.51262, 0.68529],
[0.83616, 0.51531, 0.68015],
[0.8414, 0.5181, 0.67504],
[0.8465, 0.52095, 0.66994],
[0.85148, 0.52388, 0.66485],
[0.85633, 0.52689, 0.65978],
[0.86104, 0.52996, 0.65472],
[0.86562, 0.53311, 0.64969],
[0.87006, 0.53633, 0.64466],
[0.87436, 0.53962, 0.63966],
[0.87853, 0.54301, 0.63468],
[0.88255, 0.54646, 0.62972],
[0.88645, 0.54997, 0.62478],
[0.8902, 0.55355, 0.61987],
[0.89382, 0.55722, 0.61497],
[0.89731, 0.56093, 0.6101],
[0.90066, 0.56471, 0.60525],
[0.90387, 0.56856, 0.60043],
[0.90695, 0.57248, 0.59564],
[0.9099, 0.57645, 0.59087],
]
b_cyclic_mygbm_50_90_c46 = bokeh_palette('cyclic_mygbm_50_90_c46',cyclic_mygbm_50_90_c46)
m_cyclic_mygbm_50_90_c46 = mpl_cm('cyclic_mygbm_50_90_c46',cyclic_mygbm_50_90_c46)
m_cyclic_mygbm_50_90_c46_r = mpl_cm('cyclic_mygbm_50_90_c46_r',list(reversed(cyclic_mygbm_50_90_c46)))
CET_C8 = b_cyclic_mygbm_50_90_c46
m_CET_C8 = m_cyclic_mygbm_50_90_c46
m_CET_C8_r = m_cyclic_mygbm_50_90_c46_r
palette['CET_C8'] = b_cyclic_mygbm_50_90_c46
cm['CET_C8'] = m_cyclic_mygbm_50_90_c46
cm['CET_C8_r'] = m_cyclic_mygbm_50_90_c46_r
register_cmap('cet_CET_C8',m_cyclic_mygbm_50_90_c46)
register_cmap('cet_CET_C8_r',m_cyclic_mygbm_50_90_c46_r)




linear_tritanopic_krw_5_95_c46 = [  # cmap_def
[0.066001, 0.066024, 0.066019],
[0.075827, 0.067863, 0.0665],
[0.084776, 0.069673, 0.066985],
[0.092877, 0.071413, 0.067479],
[0.10046, 0.073132, 0.067977],
[0.10754, 0.074773, 0.06849],
[0.11418, 0.076399, 0.069022],
[0.1208, 0.077883, 0.069527],
[0.1275, 0.079183, 0.070005],
[0.13427, 0.080312, 0.070528],
[0.141, 0.081408, 0.071116],
[0.14768, 0.082432, 0.071653],
[0.15433, 0.083387, 0.072143],
[0.161, 0.084395, 0.072694],
[0.16764, 0.085256, 0.073275],
[0.17428, 0.086146, 0.0738],
[0.18088, 0.087094, 0.074287],
[0.18754, 0.087912, 0.074785],
[0.19415, 0.088708, 0.075294],
[0.20071, 0.089485, 0.075795],
[0.20735, 0.090234, 0.076298],
[0.21395, 0.090962, 0.076799],
[0.22057, 0.09167, 0.077301],
[0.22716, 0.092294, 0.077803],
[0.23375, 0.092873, 0.078302],
[0.24035, 0.093487, 0.078804],
[0.24697, 0.094096, 0.079305],
[0.25359, 0.094661, 0.079804],
[0.2602, 0.095169, 0.080304],
[0.26681, 0.095621, 0.080799],
[0.27345, 0.096042, 0.081306],
[0.28006, 0.096446, 0.081812],
[0.28668, 0.096845, 0.082279],
[0.29331, 0.097246, 0.082715],
[0.29996, 0.097617, 0.083173],
[0.30661, 0.097941, 0.083688],
[0.31324, 0.098225, 0.084207],
[0.31989, 0.098472, 0.084684],
[0.32652, 0.098695, 0.085147],
[0.33319, 0.098897, 0.085628],
[0.33984, 0.099072, 0.086157],
[0.34648, 0.099226, 0.08674],
[0.35314, 0.099355, 0.0873],
[0.3598, 0.099457, 0.087822],
[0.36647, 0.099539, 0.088358],
[0.37312, 0.099595, 0.088917],
[0.37979, 0.099624, 0.089483],
[0.38645, 0.099635, 0.09006],
[0.39313, 0.099622, 0.090649],
[0.3998, 0.099583, 0.09126],
[0.40648, 0.099528, 0.091872],
[0.41315, 0.09945, 0.092439],
[0.41983, 0.099348, 0.093027],
[0.4265, 0.099232, 0.09369],
[0.43317, 0.099097, 0.094378],
[0.43985, 0.098942, 0.095041],
[0.44652, 0.098773, 0.09567],
[0.4532, 0.098593, 0.096302],
[0.45987, 0.098395, 0.096999],
[0.46655, 0.098181, 0.097756],
[0.47321, 0.097954, 0.09846],
[0.47989, 0.097701, 0.099162],
[0.48655, 0.09742, 0.099893],
[0.49321, 0.097131, 0.10064],
[0.49987, 0.096837, 0.10141],
[0.50653, 0.096547, 0.10219],
[0.51319, 0.096264, 0.103],
[0.51983, 0.095994, 0.10381],
[0.52648, 0.095723, 0.10457],
[0.53312, 0.095445, 0.10543],
[0.53975, 0.095166, 0.10633],
[0.54639, 0.094886, 0.10718],
[0.553, 0.094598, 0.10801],
[0.55962, 0.094305, 0.10893],
[0.56623, 0.09402, 0.10989],
[0.57283, 0.093751, 0.1108],
[0.57942, 0.093497, 0.11174],
[0.58601, 0.09326, 0.11271],
[0.59259, 0.09305, 0.11368],
[0.59915, 0.092879, 0.11463],
[0.60569, 0.092737, 0.1157],
[0.61224, 0.092623, 0.11673],
[0.61878, 0.092537, 0.11779],
[0.6253, 0.092488, 0.11888],
[0.63181, 0.092481, 0.11995],
[0.6383, 0.092511, 0.12106],
[0.64478, 0.092582, 0.12219],
[0.65125, 0.092696, 0.12336],
[0.65771, 0.092865, 0.12456],
[0.66414, 0.09311, 0.12581],
[0.67056, 0.093429, 0.127],
[0.67696, 0.093825, 0.1283],
[0.68334, 0.094292, 0.12958],
[0.68971, 0.094817, 0.1309],
[0.69605, 0.095383, 0.13222],
[0.70237, 0.096007, 0.13359],
[0.70868, 0.096744, 0.13499],
[0.71496, 0.097658, 0.13636],
[0.72121, 0.098584, 0.13786],
[0.72745, 0.099597, 0.13929],
[0.73366, 0.10072, 0.14084],
[0.73985, 0.10195, 0.14236],
[0.746, 0.10332, 0.14394],
[0.75213, 0.10469, 0.14556],
[0.75823, 0.10631, 0.14716],
[0.76429, 0.10792, 0.14882],
[0.77032, 0.10976, 0.15055],
[0.77633, 0.11163, 0.15227],
[0.78229, 0.11363, 0.15405],
[0.78822, 0.11575, 0.1559],
[0.79412, 0.11799, 0.15774],
[0.79998, 0.12032, 0.15962],
[0.80578, 0.12279, 0.16161],
[0.81156, 0.12543, 0.16359],
[0.81729, 0.12817, 0.16557],
[0.82297, 0.13102, 0.16767],
[0.82861, 0.13399, 0.16981],
[0.8342, 0.13708, 0.17195],
[0.83973, 0.14028, 0.17418],
[0.84521, 0.14364, 0.17643],
[0.85063, 0.14709, 0.17878],
[0.856, 0.15069, 0.18114],
[0.8613, 0.15441, 0.18359],
[0.86654, 0.15828, 0.18611],
[0.87171, 0.16226, 0.18867],
[0.8768, 0.16634, 0.19127],
[0.88184, 0.17056, 0.19399],
[0.88679, 0.17492, 0.19675],
[0.89166, 0.17939, 0.19954],
[0.89644, 0.184, 0.20245],
[0.90113, 0.18876, 0.20545],
[0.90573, 0.19363, 0.20851],
[0.91023, 0.19861, 0.21167],
[0.91462, 0.20379, 0.21488],
[0.9189, 0.20906, 0.21823],
[0.92308, 0.21447, 0.22165],
[0.92712, 0.22006, 0.22517],
[0.93103, 0.22577, 0.22883],
[0.9348, 0.23163, 0.23257],
[0.93842, 0.23765, 0.23645],
[0.94189, 0.24379, 0.24042],
[0.94519, 0.25012, 0.24455],
[0.9483, 0.25664, 0.24885],
[0.95122, 0.26334, 0.25327],
[0.95392, 0.27021, 0.25784],
[0.9564, 0.27725, 0.26262],
[0.95863, 0.2845, 0.26755],
[0.9606, 0.29197, 0.2727],
[0.96227, 0.29966, 0.27809],
[0.96366, 0.30754, 0.28366],
[0.96492, 0.31543, 0.28935],
[0.96615, 0.3232, 0.29502],
[0.96737, 0.33083, 0.30072],
[0.96856, 0.33837, 0.30645],
[0.96972, 0.34581, 0.31215],
[0.97086, 0.35316, 0.3179],
[0.97199, 0.3604, 0.32364],
[0.97309, 0.36757, 0.32941],
[0.97417, 0.37467, 0.33518],
[0.97522, 0.38168, 0.34095],
[0.97626, 0.38864, 0.34674],
[0.97727, 0.39552, 0.35253],
[0.97826, 0.40233, 0.35834],
[0.97923, 0.4091, 0.36416],
[0.98018, 0.41581, 0.36999],
[0.9811, 0.42246, 0.37584],
[0.982, 0.42907, 0.38169],
[0.98287, 0.43564, 0.38755],
[0.98372, 0.44216, 0.39343],
[0.98455, 0.44863, 0.39931],
[0.98535, 0.45506, 0.40521],
[0.98613, 0.46145, 0.41113],
[0.98689, 0.46781, 0.41705],
[0.98762, 0.47413, 0.42297],
[0.98832, 0.48042, 0.42892],
[0.98901, 0.48668, 0.43487],
[0.98967, 0.4929, 0.44083],
[0.9903, 0.4991, 0.44681],
[0.9909, 0.50526, 0.4528],
[0.99148, 0.51141, 0.45879],
[0.99204, 0.51753, 0.4648],
[0.99257, 0.52362, 0.47081],
[0.99308, 0.52969, 0.47684],
[0.99355, 0.53573, 0.48287],
[0.99401, 0.54175, 0.48892],
[0.99443, 0.54776, 0.49499],
[0.99483, 0.55375, 0.50106],
[0.9952, 0.55971, 0.50714],
[0.99554, 0.56565, 0.51323],
[0.99586, 0.57158, 0.51933],
[0.99615, 0.57749, 0.52544],
[0.99642, 0.58339, 0.53156],
[0.99665, 0.58927, 0.5377],
[0.99686, 0.59513, 0.54384],
[0.99704, 0.60098, 0.54999],
[0.99719, 0.60682, 0.55615],
[0.99731, 0.61264, 0.56234],
[0.9974, 0.61845, 0.56851],
[0.99746, 0.62425, 0.5747],
[0.9975, 0.63003, 0.58091],
[0.9975, 0.6358, 0.58713],
[0.99748, 0.64156, 0.59335],
[0.99742, 0.64731, 0.59959],
[0.99734, 0.65305, 0.60583],
[0.99722, 0.65878, 0.61208],
[0.99707, 0.6645, 0.61835],
[0.99689, 0.67022, 0.62463],
[0.99668, 0.67591, 0.63091],
[0.99644, 0.6816, 0.63721],
[0.99617, 0.68729, 0.6435],
[0.99587, 0.69297, 0.64982],
[0.99553, 0.69863, 0.65615],
[0.99516, 0.7043, 0.66248],
[0.99475, 0.70995, 0.66882],
[0.99432, 0.7156, 0.67517],
[0.99385, 0.72124, 0.68153],
[0.99335, 0.72687, 0.68791],
[0.99281, 0.73251, 0.69428],
[0.99224, 0.73812, 0.70068],
[0.99163, 0.74374, 0.70708],
[0.99099, 0.74935, 0.71349],
[0.99031, 0.75496, 0.71991],
[0.9896, 0.76056, 0.72633],
[0.98885, 0.76615, 0.73277],
[0.98806, 0.77175, 0.73922],
[0.98724, 0.77733, 0.74567],
[0.98638, 0.78291, 0.75213],
[0.98548, 0.78849, 0.75861],
[0.98455, 0.79407, 0.76509],
[0.98357, 0.79964, 0.77159],
[0.98256, 0.8052, 0.77809],
[0.98151, 0.81076, 0.7846],
[0.98041, 0.81632, 0.79111],
[0.97928, 0.82188, 0.79764],
[0.97811, 0.82743, 0.80418],
[0.9769, 0.83298, 0.81073],
[0.97564, 0.83852, 0.81728],
[0.97435, 0.84407, 0.82384],
[0.97302, 0.84961, 0.83042],
[0.97163, 0.85515, 0.837],
[0.97021, 0.86069, 0.84359],
[0.96874, 0.86622, 0.85019],
[0.96724, 0.87176, 0.8568],
[0.96568, 0.87728, 0.86342],
[0.96408, 0.88281, 0.87004],
[0.96243, 0.88834, 0.87667],
[0.96074, 0.89386, 0.88332],
[0.959, 0.89939, 0.88997],
[0.95722, 0.90491, 0.89663],
[0.95539, 0.91043, 0.9033],
[0.9535, 0.91594, 0.90997],
[0.95157, 0.92146, 0.91666],
[0.94959, 0.92698, 0.92335],
[0.94756, 0.9325, 0.93005],
[0.94547, 0.93801, 0.93676],
[0.94334, 0.94352, 0.94348],
]
b_linear_tritanopic_krw_5_95_c46 = bokeh_palette('linear_tritanopic_krw_5_95_c46',linear_tritanopic_krw_5_95_c46)
m_linear_tritanopic_krw_5_95_c46 = mpl_cm('linear_tritanopic_krw_5_95_c46',linear_tritanopic_krw_5_95_c46)
m_linear_tritanopic_krw_5_95_c46_r = mpl_cm('linear_tritanopic_krw_5_95_c46_r',list(reversed(linear_tritanopic_krw_5_95_c46)))
CET_CBTL4 = b_linear_tritanopic_krw_5_95_c46
m_CET_CBTL4 = m_linear_tritanopic_krw_5_95_c46
m_CET_CBTL4_r = m_linear_tritanopic_krw_5_95_c46_r
palette['CET_CBTL4'] = b_linear_tritanopic_krw_5_95_c46
cm['CET_CBTL4'] = m_linear_tritanopic_krw_5_95_c46
cm['CET_CBTL4_r'] = m_linear_tritanopic_krw_5_95_c46_r
register_cmap('cet_CET_CBTL4',m_linear_tritanopic_krw_5_95_c46)
register_cmap('cet_CET_CBTL4_r',m_linear_tritanopic_krw_5_95_c46_r)




cyclic_mybm_20_100_c48 = [  # cmap_def
[0.76163, 0.49843, 0.45807],
[0.76517, 0.50567, 0.45582],
[0.76859, 0.51296, 0.45358],
[0.77192, 0.52026, 0.45138],
[0.77514, 0.52758, 0.4492],
[0.77826, 0.53492, 0.44703],
[0.78127, 0.54228, 0.4449],
[0.78417, 0.54965, 0.44279],
[0.78697, 0.55704, 0.4407],
[0.78966, 0.56444, 0.43864],
[0.79225, 0.57185, 0.43661],
[0.79473, 0.57927, 0.4346],
[0.79709, 0.58671, 0.43262],
[0.79937, 0.59415, 0.43065],
[0.80152, 0.60161, 0.42871],
[0.80357, 0.60909, 0.42679],
[0.80551, 0.61656, 0.4249],
[0.80736, 0.62405, 0.42303],
[0.80909, 0.63155, 0.42118],
[0.81071, 0.63905, 0.41937],
[0.81223, 0.64657, 0.41758],
[0.81363, 0.65409, 0.4158],
[0.81493, 0.66163, 0.41405],
[0.81613, 0.66916, 0.41233],
[0.8172, 0.6767, 0.41063],
[0.81817, 0.68424, 0.40895],
[0.81903, 0.69179, 0.4073],
[0.81978, 0.69934, 0.40568],
[0.82042, 0.7069, 0.40407],
[0.82094, 0.71445, 0.40249],
[0.82135, 0.722, 0.40096],
[0.82165, 0.72956, 0.39942],
[0.82183, 0.73712, 0.39794],
[0.82189, 0.74467, 0.39647],
[0.82183, 0.75221, 0.39504],
[0.82165, 0.75976, 0.39364],
[0.82134, 0.7673, 0.39227],
[0.8209, 0.77483, 0.39095],
[0.82033, 0.78236, 0.38966],
[0.81961, 0.78988, 0.38842],
[0.81876, 0.79738, 0.38722],
[0.81775, 0.80488, 0.38609],
[0.81657, 0.81235, 0.38502],
[0.81522, 0.81981, 0.38404],
[0.81368, 0.82724, 0.38313],
[0.81194, 0.83464, 0.38231],
[0.80998, 0.84199, 0.38162],
[0.80779, 0.84929, 0.38105],
[0.80534, 0.85652, 0.38063],
[0.80261, 0.86368, 0.38038],
[0.79959, 0.87072, 0.38032],
[0.79623, 0.87765, 0.38048],
[0.79253, 0.88443, 0.38089],
[0.78846, 0.89103, 0.38157],
[0.78399, 0.89742, 0.38254],
[0.7791, 0.90355, 0.38386],
[0.77378, 0.90939, 0.38552],
[0.76801, 0.91491, 0.38757],
[0.76179, 0.92004, 0.39002],
[0.75513, 0.92476, 0.39287],
[0.74801, 0.92901, 0.39614],
[0.74046, 0.93277, 0.3998],
[0.73251, 0.936, 0.40388],
[0.72418, 0.93865, 0.40833],
[0.71551, 0.94073, 0.41314],
[0.70655, 0.94221, 0.41825],
[0.69733, 0.94309, 0.42366],
[0.68793, 0.94337, 0.42931],
[0.67838, 0.94306, 0.43516],
[0.66874, 0.94219, 0.44114],
[0.65905, 0.94078, 0.44724],
[0.64938, 0.93886, 0.45341],
[0.63973, 0.93647, 0.45961],
[0.63016, 0.93363, 0.4658],
[0.62069, 0.93042, 0.47197],
[0.61133, 0.92685, 0.47807],
[0.60211, 0.92297, 0.48409],
[0.59303, 0.9188, 0.49003],
[0.58409, 0.91441, 0.49587],
[0.5753, 0.9098, 0.50161],
[0.56666, 0.90501, 0.50723],
[0.55813, 0.90007, 0.51273],
[0.54973, 0.89499, 0.51813],
[0.54144, 0.8898, 0.52341],
[0.53327, 0.88451, 0.52859],
[0.52518, 0.87914, 0.53365],
[0.51718, 0.8737, 0.53862],
[0.50924, 0.8682, 0.54348],
[0.50137, 0.86264, 0.54826],
[0.49355, 0.85703, 0.55293],
[0.48578, 0.85138, 0.55753],
[0.47806, 0.84569, 0.56204],
[0.47037, 0.83998, 0.56645],
[0.4627, 0.83423, 0.57079],
[0.45508, 0.82844, 0.57505],
[0.44746, 0.82264, 0.57924],
[0.43989, 0.8168, 0.58334],
[0.43233, 0.81094, 0.58737],
[0.42477, 0.80506, 0.59133],
[0.41724, 0.79915, 0.59521],
[0.40973, 0.79323, 0.59903],
[0.40222, 0.78727, 0.60276],
[0.39474, 0.7813, 0.60643],
[0.38727, 0.77531, 0.61003],
[0.37983, 0.76929, 0.61355],
[0.37239, 0.76326, 0.61702],
[0.36497, 0.75722, 0.62041],
[0.35758, 0.75115, 0.62373],
[0.35019, 0.74506, 0.62697],
[0.34283, 0.73896, 0.63015],
[0.33548, 0.73284, 0.63326],
[0.32813, 0.7267, 0.6363],
[0.32083, 0.72054, 0.63927],
[0.31353, 0.71437, 0.64217],
[0.30626, 0.70819, 0.645],
[0.29899, 0.70199, 0.64777],
[0.29175, 0.69578, 0.65047],
[0.28452, 0.68955, 0.65309],
[0.27734, 0.68331, 0.65566],
[0.27017, 0.67706, 0.65816],
[0.26301, 0.6708, 0.66058],
[0.25586, 0.66451, 0.66294],
[0.24877, 0.65823, 0.66523],
[0.24168, 0.65193, 0.66746],
[0.23463, 0.64562, 0.66961],
[0.22758, 0.6393, 0.6717],
[0.22058, 0.63297, 0.67372],
[0.21357, 0.62664, 0.67568],
[0.20663, 0.6203, 0.67758],
[0.19966, 0.61394, 0.67939],
[0.19277, 0.60759, 0.68115],
[0.18592, 0.60122, 0.68284],
[0.17905, 0.59484, 0.68447],
[0.17223, 0.58846, 0.68603],
[0.16543, 0.58208, 0.68753],
[0.15873, 0.5757, 0.68897],
[0.15199, 0.5693, 0.69033],
[0.14536, 0.56291, 0.69163],
[0.1387, 0.5565, 0.69288],
[0.13213, 0.5501, 0.69405],
[0.1256, 0.54369, 0.69517],
[0.1191, 0.53729, 0.69622],
[0.1127, 0.53087, 0.6972],
[0.10633, 0.52445, 0.69813],
[0.099987, 0.51803, 0.699],
[0.093771, 0.5116, 0.6998],
[0.087677, 0.50518, 0.70055],
[0.081652, 0.49876, 0.70123],
[0.075664, 0.49233, 0.70186],
[0.069923, 0.4859, 0.70242],
[0.06427, 0.47947, 0.70294],
[0.058824, 0.47302, 0.70338],
[0.053521, 0.46659, 0.70377],
[0.048609, 0.46015, 0.70411],
[0.04384, 0.45371, 0.70439],
[0.039571, 0.44726, 0.70461],
[0.035783, 0.44081, 0.70477],
[0.032406, 0.43436, 0.70488],
[0.029883, 0.42791, 0.70493],
[0.027994, 0.42144, 0.70493],
[0.02673, 0.41498, 0.70488],
[0.02609, 0.40851, 0.70477],
[0.026078, 0.40202, 0.7046],
[0.026701, 0.39555, 0.70438],
[0.027974, 0.38905, 0.7041],
[0.02992, 0.38254, 0.70376],
[0.032576, 0.37603, 0.70337],
[0.036166, 0.3695, 0.70292],
[0.040364, 0.36296, 0.70239],
[0.045202, 0.35642, 0.70181],
[0.050548, 0.34985, 0.70115],
[0.056478, 0.34327, 0.70042],
[0.062852, 0.33666, 0.69961],
[0.069668, 0.33006, 0.69871],
[0.076852, 0.32343, 0.69773],
[0.084667, 0.31679, 0.69664],
[0.092813, 0.31016, 0.69544],
[0.10151, 0.30349, 0.69412],
[0.11069, 0.29685, 0.69269],
[0.12026, 0.2902, 0.6911],
[0.1304, 0.28359, 0.68939],
[0.14094, 0.27701, 0.6875],
[0.1519, 0.27048, 0.68545],
[0.16336, 0.26402, 0.68323],
[0.17519, 0.25762, 0.68082],
[0.18742, 0.25139, 0.67824],
[0.19994, 0.24526, 0.67546],
[0.21282, 0.23935, 0.67251],
[0.22595, 0.23362, 0.66937],
[0.23925, 0.2282, 0.66606],
[0.25271, 0.22304, 0.6626],
[0.26625, 0.21822, 0.65898],
[0.27982, 0.21379, 0.65524],
[0.29335, 0.20977, 0.6514],
[0.30684, 0.20625, 0.64745],
[0.32019, 0.20318, 0.64344],
[0.33337, 0.20063, 0.63939],
[0.34632, 0.19866, 0.63531],
[0.35904, 0.19726, 0.63122],
[0.37149, 0.19643, 0.62714],
[0.38367, 0.19613, 0.62309],
[0.39552, 0.1964, 0.61908],
[0.40706, 0.19719, 0.6151],
[0.41827, 0.19847, 0.61119],
[0.42918, 0.20024, 0.60734],
[0.43978, 0.20246, 0.60355],
[0.45006, 0.20508, 0.59983],
[0.46005, 0.20806, 0.59617],
[0.46977, 0.21138, 0.59259],
[0.4792, 0.21495, 0.58904],
[0.4884, 0.21883, 0.58558],
[0.49735, 0.22292, 0.58215],
[0.50606, 0.22721, 0.57879],
[0.51459, 0.2317, 0.57548],
[0.5229, 0.23634, 0.5722],
[0.53103, 0.24111, 0.56896],
[0.53898, 0.24602, 0.56577],
[0.54677, 0.25107, 0.56262],
[0.5544, 0.2562, 0.55948],
[0.56188, 0.26145, 0.55638],
[0.56921, 0.26678, 0.55331],
[0.57643, 0.27219, 0.55028],
[0.58349, 0.27771, 0.54727],
[0.59044, 0.28328, 0.54428],
[0.59728, 0.28893, 0.54131],
[0.60399, 0.29463, 0.53838],
[0.61059, 0.30042, 0.53545],
[0.61707, 0.30629, 0.53257],
[0.62345, 0.31217, 0.5297],
[0.62971, 0.31816, 0.52685],
[0.63588, 0.32419, 0.52402],
[0.64194, 0.33028, 0.52121],
[0.64789, 0.33642, 0.51844],
[0.65375, 0.34263, 0.51566],
[0.6595, 0.34888, 0.51293],
[0.66515, 0.35519, 0.51021],
[0.67071, 0.36154, 0.50752],
[0.67617, 0.36794, 0.50485],
[0.68152, 0.37441, 0.5022],
[0.68678, 0.38091, 0.49956],
[0.69195, 0.38745, 0.49696],
[0.69702, 0.39406, 0.49438],
[0.70199, 0.40072, 0.49182],
[0.70687, 0.40741, 0.48926],
[0.71165, 0.41415, 0.48675],
[0.71634, 0.42093, 0.48424],
[0.72093, 0.42778, 0.48177],
[0.72544, 0.43466, 0.47931],
[0.72984, 0.44158, 0.47686],
[0.73415, 0.44855, 0.47445],
[0.73836, 0.45556, 0.47205],
[0.74249, 0.4626, 0.46967],
[0.74651, 0.46971, 0.46731],
[0.75044, 0.47683, 0.46496],
[0.75427, 0.48399, 0.46264],
[0.758, 0.49121, 0.46034],
]
b_cyclic_mybm_20_100_c48 = bokeh_palette('cyclic_mybm_20_100_c48',cyclic_mybm_20_100_c48)
m_cyclic_mybm_20_100_c48 = mpl_cm('cyclic_mybm_20_100_c48',cyclic_mybm_20_100_c48)
m_cyclic_mybm_20_100_c48_r = mpl_cm('cyclic_mybm_20_100_c48_r',list(reversed(cyclic_mybm_20_100_c48)))
CET_C9 = b_cyclic_mybm_20_100_c48
m_CET_C9 = m_cyclic_mybm_20_100_c48
m_CET_C9_r = m_cyclic_mybm_20_100_c48_r
palette['CET_C9'] = b_cyclic_mybm_20_100_c48
cm['CET_C9'] = m_cyclic_mybm_20_100_c48
cm['CET_C9_r'] = m_cyclic_mybm_20_100_c48_r
register_cmap('cet_CET_C9',m_cyclic_mybm_20_100_c48)
register_cmap('cet_CET_C9_r',m_cyclic_mybm_20_100_c48_r)




linear_kbgoy_20_95_c57 = [  # cmap_def
[0.18934, 0.18939, 0.18938],
[0.19313, 0.19071, 0.1994],
[0.19684, 0.19204, 0.20948],
[0.20035, 0.19338, 0.2195],
[0.20385, 0.19473, 0.22951],
[0.20717, 0.19606, 0.23951],
[0.21038, 0.1974, 0.2495],
[0.21349, 0.19874, 0.25947],
[0.21652, 0.2001, 0.26943],
[0.2194, 0.20147, 0.27936],
[0.22219, 0.20287, 0.28928],
[0.22486, 0.20429, 0.29917],
[0.22748, 0.20567, 0.30906],
[0.22994, 0.2071, 0.31889],
[0.23236, 0.20852, 0.32873],
[0.23466, 0.20996, 0.33853],
[0.23685, 0.21143, 0.34832],
[0.23893, 0.21289, 0.35807],
[0.24093, 0.21439, 0.36779],
[0.24283, 0.21591, 0.37748],
[0.24465, 0.21743, 0.38715],
[0.24638, 0.21898, 0.39676],
[0.24804, 0.22056, 0.40637],
[0.24955, 0.22213, 0.4159],
[0.25103, 0.22374, 0.42543],
[0.25238, 0.22538, 0.43487],
[0.25366, 0.22705, 0.44431],
[0.25486, 0.22874, 0.45365],
[0.25594, 0.23043, 0.46297],
[0.25697, 0.23219, 0.47224],
[0.25788, 0.23394, 0.48144],
[0.25875, 0.23575, 0.49061],
[0.25952, 0.23758, 0.49967],
[0.26019, 0.23942, 0.50874],
[0.26079, 0.24131, 0.51767],
[0.26131, 0.24321, 0.52658],
[0.26175, 0.24516, 0.53538],
[0.26211, 0.24716, 0.54413],
[0.26239, 0.24918, 0.5528],
[0.2626, 0.25126, 0.56138],
[0.26273, 0.25333, 0.5699],
[0.26278, 0.25547, 0.5783],
[0.26276, 0.25763, 0.58663],
[0.26266, 0.25986, 0.59487],
[0.26249, 0.2621, 0.60299],
[0.26225, 0.26439, 0.61106],
[0.26193, 0.26673, 0.61897],
[0.26155, 0.26913, 0.62678],
[0.2611, 0.27152, 0.63451],
[0.26058, 0.27401, 0.64208],
[0.26, 0.27653, 0.64957],
[0.25936, 0.27909, 0.65692],
[0.25863, 0.2817, 0.66412],
[0.25784, 0.28435, 0.67122],
[0.25702, 0.28706, 0.67816],
[0.2561, 0.28983, 0.68494],
[0.25517, 0.29264, 0.69161],
[0.25415, 0.29551, 0.6981],
[0.25307, 0.29844, 0.70442],
[0.25195, 0.30141, 0.71059],
[0.25077, 0.30445, 0.7166],
[0.24953, 0.30754, 0.72239],
[0.24827, 0.31072, 0.72801],
[0.24693, 0.31391, 0.73347],
[0.24554, 0.31719, 0.7387],
[0.24411, 0.32054, 0.7437],
[0.24265, 0.32392, 0.7485],
[0.24115, 0.32738, 0.7531],
[0.2396, 0.33091, 0.75744],
[0.23801, 0.33454, 0.76152],
[0.23638, 0.3382, 0.76536],
[0.23471, 0.34194, 0.76894],
[0.23296, 0.34574, 0.77226],
[0.23122, 0.34964, 0.77525],
[0.2294, 0.35361, 0.77794],
[0.22757, 0.35767, 0.78031],
[0.22568, 0.36179, 0.78236],
[0.22373, 0.366, 0.78407],
[0.22174, 0.37027, 0.78542],
[0.2197, 0.37466, 0.78639],
[0.21759, 0.37912, 0.78693],
[0.2154, 0.38369, 0.78703],
[0.21312, 0.38835, 0.78669],
[0.21078, 0.39311, 0.78586],
[0.20831, 0.39798, 0.78454],
[0.20572, 0.40294, 0.78268],
[0.20296, 0.40802, 0.78026],
[0.20001, 0.41321, 0.77724],
[0.19691, 0.41851, 0.77356],
[0.19349, 0.42395, 0.76918],
[0.18976, 0.42953, 0.76402],
[0.18568, 0.43525, 0.75805],
[0.18106, 0.4411, 0.75122],
[0.1759, 0.44712, 0.74345],
[0.17032, 0.45321, 0.73494],
[0.16477, 0.45925, 0.72618],
[0.15956, 0.46512, 0.71747],
[0.1547, 0.47088, 0.7088],
[0.15023, 0.47651, 0.70018],
[0.14621, 0.48202, 0.69161],
[0.1427, 0.4874, 0.6831],
[0.13962, 0.49267, 0.67463],
[0.13713, 0.49786, 0.6662],
[0.13522, 0.50291, 0.65784],
[0.13395, 0.50789, 0.64951],
[0.13324, 0.51276, 0.64122],
[0.13315, 0.51756, 0.63299],
[0.13376, 0.52225, 0.6248],
[0.13503, 0.52688, 0.61666],
[0.13687, 0.53141, 0.60856],
[0.13929, 0.53587, 0.60048],
[0.14241, 0.54025, 0.59247],
[0.1461, 0.54456, 0.58448],
[0.15027, 0.5488, 0.57654],
[0.15491, 0.55295, 0.56863],
[0.16006, 0.55705, 0.56076],
[0.16569, 0.56107, 0.55293],
[0.17171, 0.56502, 0.54513],
[0.1781, 0.56892, 0.53737],
[0.18481, 0.57276, 0.52964],
[0.19187, 0.57652, 0.52193],
[0.19921, 0.58021, 0.51428],
[0.20684, 0.58385, 0.50662],
[0.2147, 0.58742, 0.49902],
[0.22286, 0.59093, 0.49145],
[0.2312, 0.59438, 0.48387],
[0.23972, 0.59777, 0.47634],
[0.2485, 0.60109, 0.46883],
[0.25745, 0.60436, 0.46134],
[0.26657, 0.60756, 0.45387],
[0.27587, 0.61069, 0.44641],
[0.28531, 0.61376, 0.43897],
[0.29496, 0.61678, 0.43156],
[0.30475, 0.61972, 0.42414],
[0.31466, 0.6226, 0.41675],
[0.32476, 0.62541, 0.40936],
[0.33503, 0.62816, 0.40197],
[0.3454, 0.63084, 0.3946],
[0.35591, 0.63345, 0.38722],
[0.36658, 0.63599, 0.37985],
[0.3774, 0.63846, 0.37248],
[0.38835, 0.64085, 0.3651],
[0.39942, 0.64318, 0.35772],
[0.41067, 0.64542, 0.35032],
[0.42203, 0.64759, 0.34291],
[0.43344, 0.6497, 0.33549],
[0.44472, 0.65177, 0.32806],
[0.45578, 0.65383, 0.32069],
[0.46667, 0.65587, 0.31331],
[0.47738, 0.6579, 0.30596],
[0.48795, 0.65991, 0.2986],
[0.49838, 0.6619, 0.29128],
[0.50869, 0.66387, 0.28395],
[0.51889, 0.66582, 0.27666],
[0.52899, 0.66775, 0.26937],
[0.53901, 0.66966, 0.26207],
[0.54896, 0.67154, 0.25481],
[0.55883, 0.67341, 0.24753],
[0.56866, 0.67524, 0.24024],
[0.57844, 0.67706, 0.23298],
[0.58818, 0.67884, 0.22575],
[0.59789, 0.68059, 0.21849],
[0.60758, 0.68232, 0.21124],
[0.61725, 0.68402, 0.20401],
[0.6269, 0.68569, 0.19677],
[0.63656, 0.68732, 0.18952],
[0.6462, 0.68893, 0.18228],
[0.65587, 0.69048, 0.1751],
[0.66554, 0.69202, 0.16788],
[0.67522, 0.6935, 0.16071],
[0.68493, 0.69494, 0.15356],
[0.69467, 0.69635, 0.14646],
[0.70444, 0.69771, 0.13939],
[0.71425, 0.69902, 0.13246],
[0.72411, 0.70029, 0.12558],
[0.73401, 0.70151, 0.11883],
[0.74396, 0.70268, 0.11222],
[0.75397, 0.70378, 0.1058],
[0.76404, 0.70483, 0.099637],
[0.77418, 0.70583, 0.093802],
[0.7844, 0.70676, 0.088385],
[0.79468, 0.70761, 0.083462],
[0.80505, 0.70841, 0.079113],
[0.81551, 0.70913, 0.075535],
[0.82606, 0.70977, 0.07286],
[0.83671, 0.71033, 0.071189],
[0.84745, 0.71081, 0.070548],
[0.85824, 0.71123, 0.071057],
[0.86861, 0.71177, 0.07205],
[0.87811, 0.71261, 0.073094],
[0.88682, 0.71374, 0.073971],
[0.89485, 0.71512, 0.074707],
[0.90229, 0.71674, 0.075383],
[0.90918, 0.71856, 0.075978],
[0.91561, 0.72056, 0.076505],
[0.92162, 0.72274, 0.076968],
[0.92722, 0.7251, 0.077371],
[0.93249, 0.72757, 0.07772],
[0.93743, 0.7302, 0.078016],
[0.94205, 0.73296, 0.078259],
[0.9464, 0.73583, 0.078457],
[0.95049, 0.73882, 0.078611],
[0.95435, 0.7419, 0.078722],
[0.95798, 0.74508, 0.078793],
[0.96139, 0.74836, 0.078824],
[0.9646, 0.75173, 0.078816],
[0.96761, 0.75518, 0.078772],
[0.97044, 0.75871, 0.078692],
[0.9731, 0.7623, 0.078579],
[0.97559, 0.76598, 0.078434],
[0.97793, 0.76971, 0.078256],
[0.98013, 0.77351, 0.078048],
[0.98217, 0.77737, 0.077809],
[0.98406, 0.78129, 0.077541],
[0.98582, 0.78526, 0.077244],
[0.98746, 0.78929, 0.076919],
[0.98897, 0.79336, 0.076568],
[0.99036, 0.79748, 0.076191],
[0.99164, 0.80164, 0.075789],
[0.99281, 0.80585, 0.075362],
[0.99386, 0.81009, 0.074907],
[0.9948, 0.81438, 0.074429],
[0.99563, 0.8187, 0.073943],
[0.99636, 0.82307, 0.073414],
[0.997, 0.82746, 0.072791],
[0.99755, 0.83189, 0.072165],
[0.998, 0.83635, 0.071588],
[0.99834, 0.84084, 0.070913],
[0.9986, 0.84536, 0.070203],
[0.99878, 0.8499, 0.069562],
[0.99887, 0.85447, 0.068822],
[0.99886, 0.85907, 0.068012],
[0.99877, 0.8637, 0.0672],
[0.9986, 0.86834, 0.066361],
[0.99835, 0.873, 0.065502],
[0.998, 0.8777, 0.064618],
[0.99759, 0.8824, 0.06371],
[0.9971, 0.88713, 0.062807],
[0.99652, 0.89189, 0.061722],
[0.99587, 0.89666, 0.060732],
[0.99514, 0.90144, 0.059763],
[0.99434, 0.90624, 0.058599],
[0.99346, 0.91106, 0.057436],
[0.99251, 0.91589, 0.056362],
[0.99149, 0.92073, 0.055104],
[0.99038, 0.92561, 0.053841],
[0.98921, 0.93048, 0.052556],
[0.98796, 0.93537, 0.051252],
[0.98664, 0.94028, 0.049917],
[0.98525, 0.94519, 0.048509],
[0.98379, 0.95012, 0.047046],
[0.98225, 0.95506, 0.045473],
[0.98065, 0.96001, 0.043886],
[0.97897, 0.96497, 0.042319],
[0.97722, 0.96994, 0.04062],
[0.97542, 0.97491, 0.03886],
]
b_linear_kbgoy_20_95_c57 = bokeh_palette('linear_kbgoy_20_95_c57',linear_kbgoy_20_95_c57)
m_linear_kbgoy_20_95_c57 = mpl_cm('linear_kbgoy_20_95_c57',linear_kbgoy_20_95_c57)
m_linear_kbgoy_20_95_c57_r = mpl_cm('linear_kbgoy_20_95_c57_r',list(reversed(linear_kbgoy_20_95_c57)))
gouldian = b_linear_kbgoy_20_95_c57
m_gouldian = m_linear_kbgoy_20_95_c57
m_gouldian_r = m_linear_kbgoy_20_95_c57_r
palette['gouldian'] = b_linear_kbgoy_20_95_c57
palette_n['gouldian'] = b_linear_kbgoy_20_95_c57
cm['gouldian'] = m_linear_kbgoy_20_95_c57
cm['gouldian_r'] = m_linear_kbgoy_20_95_c57_r
cm_n['gouldian'] = mpl_cm('gouldian',linear_kbgoy_20_95_c57)
cm_n['gouldian_r'] = mpl_cm('gouldian_r',list(reversed(linear_kbgoy_20_95_c57)))
CET_L20 = b_linear_kbgoy_20_95_c57
m_CET_L20 = m_linear_kbgoy_20_95_c57
m_CET_L20_r = m_linear_kbgoy_20_95_c57_r
palette['CET_L20'] = b_linear_kbgoy_20_95_c57
cm['CET_L20'] = m_linear_kbgoy_20_95_c57
cm['CET_L20_r'] = m_linear_kbgoy_20_95_c57_r
register_cmap('cet_CET_L20',m_linear_kbgoy_20_95_c57)
register_cmap('cet_CET_L20_r',m_linear_kbgoy_20_95_c57_r)




cyclic_rygcbmr_50_90_c64_s25 = [  # cmap_def
[0.68404, 0.63055, 1],
[0.70098, 0.6395, 1],
[0.7177, 0.64845, 1],
[0.73421, 0.65739, 1],
[0.75055, 0.66627, 1],
[0.76671, 0.67509, 1],
[0.78273, 0.68379, 1],
[0.7986, 0.69233, 1],
[0.81433, 0.70062, 1],
[0.8299, 0.70861, 1],
[0.8453, 0.7162, 1],
[0.86052, 0.72328, 1],
[0.8755, 0.72977, 1],
[0.89021, 0.73553, 0.99966],
[0.90458, 0.74046, 0.99387],
[0.91855, 0.74442, 0.987],
[0.93205, 0.74732, 0.97894],
[0.94498, 0.74904, 0.96964],
[0.95728, 0.74952, 0.95905],
[0.96886, 0.74869, 0.94717],
[0.97965, 0.74653, 0.93399],
[0.9896, 0.74302, 0.91958],
[0.99865, 0.73818, 0.90399],
[1, 0.73207, 0.88731],
[1, 0.72474, 0.86966],
[1, 0.7163, 0.85114],
[1, 0.70684, 0.83189],
[1, 0.69646, 0.81201],
[1, 0.68529, 0.79164],
[1, 0.67343, 0.77088],
[1, 0.66099, 0.74982],
[1, 0.64806, 0.72855],
[1, 0.63473, 0.70715],
[1, 0.62107, 0.68567],
[1, 0.60713, 0.66415],
[1, 0.59297, 0.64264],
[1, 0.5786, 0.62116],
[1, 0.56409, 0.59973],
[1, 0.54941, 0.57836],
[1, 0.53459, 0.55708],
[1, 0.51964, 0.53586],
[1, 0.50455, 0.51475],
[1, 0.48929, 0.49372],
[1, 0.47391, 0.4728],
[1, 0.45838, 0.452],
[1, 0.4427, 0.4313],
[1, 0.42686, 0.41072],
[1, 0.41086, 0.39028],
[1, 0.39472, 0.37],
[1, 0.37845, 0.34992],
[1, 0.36209, 0.33004],
[1, 0.34568, 0.31042],
[1, 0.32933, 0.29109],
[0.99872, 0.31308, 0.2721],
[0.99487, 0.29714, 0.25356],
[0.99105, 0.28163, 0.23549],
[0.98733, 0.26683, 0.21793],
[0.98375, 0.25301, 0.20097],
[0.98037, 0.24049, 0.18472],
[0.97725, 0.22969, 0.16917],
[0.97445, 0.22104, 0.15432],
[0.97202, 0.21483, 0.14028],
[0.96999, 0.21152, 0.127],
[0.9684, 0.21123, 0.11451],
[0.96726, 0.21404, 0.10285],
[0.96657, 0.21983, 0.0919],
[0.96633, 0.22834, 0.081637],
[0.96652, 0.23917, 0.071996],
[0.9671, 0.25198, 0.063086],
[0.96804, 0.26633, 0.054632],
[0.96927, 0.28187, 0.046867],
[0.97077, 0.29827, 0.039573],
[0.97247, 0.31531, 0.032902],
[0.97433, 0.33274, 0.027416],
[0.9763, 0.35036, 0.022767],
[0.97835, 0.36807, 0.018802],
[0.98044, 0.38577, 0.015391],
[0.98254, 0.40339, 0.012442],
[0.98463, 0.42085, 0.009625],
[0.98667, 0.43815, 0.007288],
[0.98867, 0.45524, 0.005171],
[0.99061, 0.47213, 0.003222],
[0.99248, 0.4888, 0.001403],
[0.99427, 0.50528, 0],
[0.99598, 0.52156, 0],
[0.99761, 0.53767, 0],
[0.99914, 0.55359, 0],
[1, 0.56934, 0],
[1, 0.58494, 0],
[1, 0.60037, 0],
[1, 0.61565, 0],
[1, 0.63077, 0],
[1, 0.64572, 0],
[1, 0.66048, 0],
[1, 0.67504, 0],
[1, 0.68935, 0],
[1, 0.70336, 0],
[1, 0.71701, 0],
[1, 0.73023, 0],
[1, 0.74293, 0],
[0.99701, 0.75501, 0],
[0.99313, 0.76637, 0],
[0.98833, 0.77691, 0],
[0.98256, 0.78651, 0],
[0.97574, 0.79507, 0],
[0.96785, 0.80251, 0],
[0.95886, 0.80875, 0],
[0.94877, 0.81373, 0],
[0.93761, 0.81743, 0],
[0.92543, 0.81985, 0],
[0.91229, 0.821, 0],
[0.89826, 0.82095, 0],
[0.88344, 0.81974, 0],
[0.86793, 0.8175, 0],
[0.8518, 0.8143, 0],
[0.83517, 0.81025, 0],
[0.81812, 0.80549, 0],
[0.80073, 0.8001, 0],
[0.78306, 0.7942, 0],
[0.76518, 0.78786, 0],
[0.74713, 0.7812, 0],
[0.72895, 0.77426, 0],
[0.71067, 0.76712, 0],
[0.69231, 0.75982, 0],
[0.67387, 0.7524, 0],
[0.65537, 0.74489, 0],
[0.63682, 0.73731, 0],
[0.61821, 0.72968, 0],
[0.59952, 0.722, 0],
[0.58077, 0.7143, 8.8e-05],
[0.56197, 0.7066, 0.001364],
[0.54309, 0.69889, 0.002744],
[0.52413, 0.69121, 0.004275],
[0.50511, 0.68355, 0.006019],
[0.48603, 0.67593, 0.008055],
[0.46689, 0.66839, 0.010459],
[0.4477, 0.66094, 0.013625],
[0.42851, 0.65363, 0.017232],
[0.40933, 0.6465, 0.021673],
[0.39021, 0.63962, 0.027147],
[0.37122, 0.63303, 0.033854],
[0.35244, 0.62683, 0.042288],
[0.33395, 0.62109, 0.05131],
[0.31583, 0.61589, 0.06103],
[0.29817, 0.61133, 0.071492],
[0.28114, 0.60749, 0.082491],
[0.26486, 0.60444, 0.094141],
[0.24945, 0.60226, 0.10646],
[0.23509, 0.60102, 0.11935],
[0.22182, 0.60075, 0.13293],
[0.20988, 0.60148, 0.14712],
[0.19934, 0.60321, 0.16196],
[0.19034, 0.60592, 0.17737],
[0.18291, 0.60959, 0.19335],
[0.17711, 0.61414, 0.20988],
[0.17287, 0.61953, 0.22692],
[0.17015, 0.62567, 0.24439],
[0.16879, 0.63248, 0.2623],
[0.16861, 0.63987, 0.28053],
[0.16945, 0.64776, 0.2991],
[0.17103, 0.65608, 0.31789],
[0.17321, 0.66474, 0.3369],
[0.17578, 0.67369, 0.35607],
[0.17862, 0.68287, 0.37537],
[0.18147, 0.69224, 0.39479],
[0.18433, 0.70173, 0.4143],
[0.18705, 0.71136, 0.43387],
[0.18954, 0.72106, 0.45351],
[0.19176, 0.73084, 0.4732],
[0.19369, 0.74067, 0.49294],
[0.19522, 0.75056, 0.51274],
[0.19644, 0.76048, 0.53259],
[0.1972, 0.77042, 0.55248],
[0.19751, 0.78036, 0.5724],
[0.19739, 0.7903, 0.59236],
[0.19681, 0.80022, 0.61235],
[0.19573, 0.81009, 0.63237],
[0.19423, 0.81989, 0.65241],
[0.1922, 0.82959, 0.67244],
[0.18974, 0.83913, 0.69245],
[0.18683, 0.84847, 0.71238],
[0.1835, 0.85752, 0.73222],
[0.1798, 0.86622, 0.75189],
[0.17577, 0.87446, 0.77133],
[0.17154, 0.88215, 0.79046],
[0.1672, 0.88917, 0.8092],
[0.16285, 0.89541, 0.82742],
[0.15866, 0.90075, 0.84505],
[0.15479, 0.90508, 0.86196],
[0.15145, 0.9083, 0.87804],
[0.1488, 0.91033, 0.89319],
[0.147, 0.9111, 0.90734],
[0.14614, 0.91058, 0.9204],
[0.14627, 0.90875, 0.93232],
[0.1474, 0.90562, 0.94309],
[0.14944, 0.90123, 0.9527],
[0.15222, 0.89566, 0.96117],
[0.15567, 0.88898, 0.96855],
[0.15952, 0.8813, 0.97492],
[0.16369, 0.87272, 0.98035],
[0.16789, 0.86336, 0.98494],
[0.17206, 0.85333, 0.9888],
[0.17603, 0.84276, 0.99201],
[0.17975, 0.83174, 0.99468],
[0.18308, 0.82034, 0.99691],
[0.18603, 0.80867, 0.99878],
[0.18848, 0.79679, 1],
[0.19041, 0.78475, 1],
[0.19184, 0.7726, 1],
[0.19274, 0.76037, 1],
[0.1931, 0.74809, 1],
[0.1929, 0.73578, 1],
[0.19214, 0.72346, 1],
[0.19086, 0.71113, 1],
[0.18909, 0.6988, 1],
[0.18679, 0.68652, 1],
[0.18401, 0.67427, 1],
[0.18079, 0.66207, 1],
[0.17723, 0.64994, 1],
[0.17333, 0.63789, 1],
[0.16932, 0.62596, 1],
[0.16523, 0.61417, 1],
[0.16146, 0.60259, 1],
[0.15814, 0.59124, 1],
[0.1557, 0.58022, 1],
[0.1545, 0.56956, 1],
[0.15501, 0.55939, 1],
[0.15757, 0.54976, 1],
[0.16255, 0.54079, 1],
[0.17005, 0.53257, 1],
[0.18013, 0.52519, 1],
[0.19269, 0.51875, 1],
[0.20747, 0.51331, 1],
[0.22414, 0.50895, 1],
[0.24245, 0.50571, 1],
[0.26202, 0.50364, 1],
[0.28258, 0.50272, 1],
[0.30381, 0.50294, 1],
[0.32553, 0.50429, 1],
[0.34755, 0.50666, 1],
[0.36963, 0.51003, 1],
[0.39169, 0.51429, 1],
[0.41358, 0.51935, 1],
[0.43525, 0.52511, 1],
[0.45659, 0.53148, 1],
[0.4776, 0.53838, 1],
[0.49823, 0.5457, 1],
[0.51845, 0.55337, 1],
[0.53827, 0.56134, 1],
[0.5577, 0.56954, 1],
[0.57675, 0.57793, 1],
[0.59542, 0.58647, 1],
[0.61374, 0.59512, 1],
[0.63175, 0.60388, 1],
[0.64945, 0.61271, 1],
[0.66687, 0.6216, 1],
]
b_cyclic_rygcbmr_50_90_c64_s25 = bokeh_palette('cyclic_rygcbmr_50_90_c64_s25',cyclic_rygcbmr_50_90_c64_s25)
m_cyclic_rygcbmr_50_90_c64_s25 = mpl_cm('cyclic_rygcbmr_50_90_c64_s25',cyclic_rygcbmr_50_90_c64_s25)
m_cyclic_rygcbmr_50_90_c64_s25_r = mpl_cm('cyclic_rygcbmr_50_90_c64_s25_r',list(reversed(cyclic_rygcbmr_50_90_c64_s25)))
CET_C6s = b_cyclic_rygcbmr_50_90_c64_s25
m_CET_C6s = m_cyclic_rygcbmr_50_90_c64_s25
m_CET_C6s_r = m_cyclic_rygcbmr_50_90_c64_s25_r
palette['CET_C6s'] = b_cyclic_rygcbmr_50_90_c64_s25
cm['CET_C6s'] = m_cyclic_rygcbmr_50_90_c64_s25
cm['CET_C6s_r'] = m_cyclic_rygcbmr_50_90_c64_s25_r
register_cmap('cet_CET_C6s',m_cyclic_rygcbmr_50_90_c64_s25)
register_cmap('cet_CET_C6s_r',m_cyclic_rygcbmr_50_90_c64_s25_r)




diverging_linear_protanopic_deuteranopic_bjy_57_89_c34 = [  # cmap_def
[0.018344, 0.53654, 0.99004],
[0.062847, 0.5377, 0.98783],
[0.092838, 0.53887, 0.98562],
[0.11618, 0.54003, 0.98341],
[0.13563, 0.5412, 0.9812],
[0.15266, 0.54238, 0.97899],
[0.16791, 0.54355, 0.97678],
[0.1818, 0.54473, 0.97457],
[0.19466, 0.54592, 0.97236],
[0.20661, 0.5471, 0.97015],
[0.21782, 0.54829, 0.96794],
[0.22842, 0.54948, 0.96572],
[0.23846, 0.55067, 0.96351],
[0.24807, 0.55187, 0.9613],
[0.25722, 0.55306, 0.95908],
[0.26603, 0.55427, 0.95687],
[0.27448, 0.55546, 0.95466],
[0.28265, 0.55668, 0.95244],
[0.29052, 0.55788, 0.95023],
[0.29815, 0.55909, 0.94801],
[0.30557, 0.56031, 0.9458],
[0.31273, 0.56152, 0.94358],
[0.31973, 0.56275, 0.94137],
[0.32652, 0.56397, 0.93915],
[0.33317, 0.56518, 0.93693],
[0.33963, 0.56641, 0.93471],
[0.34593, 0.56764, 0.93249],
[0.3521, 0.56886, 0.93028],
[0.35815, 0.57009, 0.92806],
[0.36405, 0.57133, 0.92584],
[0.36984, 0.57257, 0.92362],
[0.37551, 0.57381, 0.9214],
[0.38109, 0.57505, 0.91918],
[0.38653, 0.5763, 0.91696],
[0.39191, 0.57754, 0.91474],
[0.39717, 0.57879, 0.91251],
[0.40235, 0.58005, 0.91029],
[0.40744, 0.58129, 0.90807],
[0.41246, 0.58255, 0.90585],
[0.41739, 0.58381, 0.90362],
[0.42223, 0.58507, 0.90139],
[0.42703, 0.58633, 0.89917],
[0.43174, 0.58759, 0.89694],
[0.43638, 0.58885, 0.89472],
[0.44095, 0.59013, 0.89249],
[0.44546, 0.5914, 0.89026],
[0.44992, 0.59268, 0.88803],
[0.45432, 0.59395, 0.8858],
[0.45865, 0.59522, 0.88357],
[0.46293, 0.59651, 0.88134],
[0.46717, 0.59778, 0.8791],
[0.47135, 0.59907, 0.87687],
[0.47548, 0.60035, 0.87464],
[0.47955, 0.60164, 0.8724],
[0.48357, 0.60293, 0.87016],
[0.48757, 0.60423, 0.86793],
[0.49151, 0.60551, 0.86569],
[0.4954, 0.60682, 0.86346],
[0.49925, 0.60811, 0.86122],
[0.50306, 0.60941, 0.85898],
[0.50684, 0.61071, 0.85674],
[0.51058, 0.61201, 0.8545],
[0.51428, 0.61332, 0.85226],
[0.51794, 0.61462, 0.85001],
[0.52156, 0.61594, 0.84777],
[0.52515, 0.61725, 0.84552],
[0.52871, 0.61857, 0.84328],
[0.53223, 0.61988, 0.84103],
[0.53571, 0.62119, 0.83878],
[0.53917, 0.62251, 0.83654],
[0.5426, 0.62384, 0.83429],
[0.546, 0.62516, 0.83204],
[0.54936, 0.62648, 0.82979],
[0.55269, 0.62781, 0.82753],
[0.55599, 0.62913, 0.82528],
[0.55927, 0.63047, 0.82303],
[0.56253, 0.6318, 0.82077],
[0.56574, 0.63313, 0.81851],
[0.56894, 0.63447, 0.81626],
[0.57212, 0.6358, 0.81399],
[0.57526, 0.63714, 0.81173],
[0.57838, 0.63848, 0.80947],
[0.58148, 0.63983, 0.80721],
[0.58456, 0.64116, 0.80494],
[0.5876, 0.64252, 0.80268],
[0.59063, 0.64386, 0.80041],
[0.59364, 0.64521, 0.79814],
[0.59662, 0.64656, 0.79587],
[0.59958, 0.64791, 0.79361],
[0.60252, 0.64927, 0.79133],
[0.60543, 0.65063, 0.78906],
[0.60834, 0.65199, 0.78679],
[0.61121, 0.65334, 0.78451],
[0.61406, 0.6547, 0.78223],
[0.61691, 0.65607, 0.77995],
[0.61973, 0.65743, 0.77767],
[0.62252, 0.65879, 0.77538],
[0.6253, 0.66016, 0.7731],
[0.62806, 0.66153, 0.77082],
[0.63081, 0.66291, 0.76853],
[0.63353, 0.66427, 0.76624],
[0.63624, 0.66565, 0.76395],
[0.63893, 0.66702, 0.76166],
[0.6416, 0.66841, 0.75936],
[0.64426, 0.66978, 0.75707],
[0.64691, 0.67116, 0.75478],
[0.64953, 0.67255, 0.75247],
[0.65214, 0.67392, 0.75017],
[0.65473, 0.67531, 0.74788],
[0.65732, 0.6767, 0.74557],
[0.65988, 0.67809, 0.74326],
[0.66243, 0.67947, 0.74095],
[0.66495, 0.68086, 0.73864],
[0.66748, 0.68225, 0.73633],
[0.66998, 0.68365, 0.73402],
[0.67247, 0.68505, 0.7317],
[0.67494, 0.68644, 0.72938],
[0.67741, 0.68784, 0.72706],
[0.67985, 0.68924, 0.72474],
[0.68228, 0.69063, 0.72241],
[0.68471, 0.69205, 0.72009],
[0.68712, 0.69345, 0.71776],
[0.68951, 0.69485, 0.71544],
[0.6919, 0.69626, 0.7131],
[0.69426, 0.69766, 0.71076],
[0.69662, 0.69907, 0.70843],
[0.69896, 0.70049, 0.70609],
[0.7013, 0.70189, 0.70374],
[0.70391, 0.70323, 0.70137],
[0.70681, 0.70447, 0.69897],
[0.70969, 0.70572, 0.69657],
[0.71256, 0.70697, 0.69416],
[0.71542, 0.70821, 0.69176],
[0.71825, 0.70947, 0.68935],
[0.72107, 0.71072, 0.68694],
[0.72389, 0.71197, 0.68452],
[0.72669, 0.71322, 0.6821],
[0.72947, 0.71447, 0.67968],
[0.73225, 0.71574, 0.67727],
[0.73501, 0.717, 0.67484],
[0.73775, 0.71825, 0.67241],
[0.74048, 0.71952, 0.66998],
[0.74321, 0.72077, 0.66755],
[0.74591, 0.72203, 0.6651],
[0.74861, 0.7233, 0.66267],
[0.7513, 0.72457, 0.66022],
[0.75397, 0.72584, 0.65778],
[0.75663, 0.7271, 0.65533],
[0.75928, 0.72837, 0.65287],
[0.76192, 0.72965, 0.65042],
[0.76456, 0.73092, 0.64795],
[0.76717, 0.7322, 0.64549],
[0.76978, 0.73347, 0.64303],
[0.77238, 0.73474, 0.64056],
[0.77496, 0.73601, 0.63809],
[0.77754, 0.7373, 0.63561],
[0.7801, 0.73858, 0.63313],
[0.78266, 0.73986, 0.63065],
[0.78521, 0.74114, 0.62816],
[0.78774, 0.74243, 0.62566],
[0.79027, 0.74371, 0.62317],
[0.79279, 0.74499, 0.62067],
[0.79529, 0.74628, 0.61816],
[0.79779, 0.74757, 0.61565],
[0.80028, 0.74886, 0.61314],
[0.80276, 0.75015, 0.61063],
[0.80523, 0.75145, 0.60811],
[0.80769, 0.75273, 0.60558],
[0.81014, 0.75403, 0.60305],
[0.81259, 0.75533, 0.60052],
[0.81502, 0.75662, 0.59798],
[0.81745, 0.75792, 0.59543],
[0.81987, 0.75921, 0.59289],
[0.82228, 0.76052, 0.59033],
[0.82468, 0.76181, 0.58777],
[0.82708, 0.76311, 0.58522],
[0.82947, 0.76442, 0.58264],
[0.83184, 0.76573, 0.58008],
[0.83422, 0.76703, 0.57749],
[0.83657, 0.76833, 0.5749],
[0.83893, 0.76964, 0.57232],
[0.84128, 0.77096, 0.56972],
[0.84361, 0.77226, 0.56713],
[0.84595, 0.77357, 0.56452],
[0.84827, 0.77488, 0.56191],
[0.85059, 0.7762, 0.55929],
[0.85291, 0.77752, 0.55666],
[0.85521, 0.77883, 0.55403],
[0.8575, 0.78014, 0.5514],
[0.8598, 0.78146, 0.54876],
[0.86208, 0.78278, 0.54611],
[0.86436, 0.7841, 0.54345],
[0.86662, 0.78542, 0.54078],
[0.86889, 0.78675, 0.53812],
[0.87115, 0.78806, 0.53543],
[0.8734, 0.78939, 0.53276],
[0.87564, 0.79071, 0.53007],
[0.87788, 0.79204, 0.52737],
[0.88011, 0.79337, 0.52466],
[0.88233, 0.79469, 0.52194],
[0.88455, 0.79602, 0.51923],
[0.88677, 0.79735, 0.51649],
[0.88898, 0.79868, 0.51376],
[0.89118, 0.80001, 0.51101],
[0.89337, 0.80134, 0.50827],
[0.89556, 0.80268, 0.50549],
[0.89774, 0.80401, 0.50272],
[0.89992, 0.80534, 0.49994],
[0.90209, 0.80669, 0.49716],
[0.90426, 0.80802, 0.49436],
[0.90642, 0.80936, 0.49156],
[0.90858, 0.8107, 0.48873],
[0.91073, 0.81204, 0.4859],
[0.91287, 0.81338, 0.48306],
[0.91501, 0.81472, 0.48022],
[0.91715, 0.81607, 0.47735],
[0.91927, 0.81741, 0.47449],
[0.9214, 0.81875, 0.47161],
[0.92352, 0.8201, 0.46871],
[0.92563, 0.82145, 0.4658],
[0.92774, 0.8228, 0.46289],
[0.92984, 0.82414, 0.45996],
[0.93194, 0.82549, 0.45703],
[0.93403, 0.82685, 0.45408],
[0.93613, 0.8282, 0.45111],
[0.93821, 0.82955, 0.44813],
[0.94029, 0.83091, 0.44514],
[0.94237, 0.83226, 0.44215],
[0.94444, 0.83362, 0.43911],
[0.9465, 0.83497, 0.43609],
[0.94856, 0.83633, 0.43304],
[0.95062, 0.83769, 0.42998],
[0.95267, 0.83904, 0.4269],
[0.95472, 0.84041, 0.4238],
[0.95676, 0.84177, 0.42069],
[0.95881, 0.84313, 0.41757],
[0.96084, 0.84449, 0.41442],
[0.96287, 0.84585, 0.41126],
[0.9649, 0.84721, 0.40808],
[0.96692, 0.84858, 0.40488],
[0.96894, 0.84995, 0.40166],
[0.97095, 0.85131, 0.39843],
[0.97297, 0.85268, 0.39517],
[0.97497, 0.85405, 0.39189],
[0.97697, 0.85542, 0.3886],
[0.97897, 0.85679, 0.38527],
[0.98097, 0.85816, 0.38193],
[0.98296, 0.85953, 0.37857],
[0.98495, 0.8609, 0.37518],
[0.98693, 0.86228, 0.37176],
[0.98891, 0.86366, 0.36833],
[0.99088, 0.86502, 0.36486],
[0.99286, 0.8664, 0.36137],
[0.99482, 0.86778, 0.35786],
[0.99679, 0.86916, 0.35431],
[0.99875, 0.87053, 0.35074],
]
b_diverging_linear_protanopic_deuteranopic_bjy_57_89_c34 = bokeh_palette('diverging_linear_protanopic_deuteranopic_bjy_57_89_c34',diverging_linear_protanopic_deuteranopic_bjy_57_89_c34)
m_diverging_linear_protanopic_deuteranopic_bjy_57_89_c34 = mpl_cm('diverging_linear_protanopic_deuteranopic_bjy_57_89_c34',diverging_linear_protanopic_deuteranopic_bjy_57_89_c34)
m_diverging_linear_protanopic_deuteranopic_bjy_57_89_c34_r = mpl_cm('diverging_linear_protanopic_deuteranopic_bjy_57_89_c34_r',list(reversed(diverging_linear_protanopic_deuteranopic_bjy_57_89_c34)))
CET_CBD2 = b_diverging_linear_protanopic_deuteranopic_bjy_57_89_c34
m_CET_CBD2 = m_diverging_linear_protanopic_deuteranopic_bjy_57_89_c34
m_CET_CBD2_r = m_diverging_linear_protanopic_deuteranopic_bjy_57_89_c34_r
palette['CET_CBD2'] = b_diverging_linear_protanopic_deuteranopic_bjy_57_89_c34
cm['CET_CBD2'] = m_diverging_linear_protanopic_deuteranopic_bjy_57_89_c34
cm['CET_CBD2_r'] = m_diverging_linear_protanopic_deuteranopic_bjy_57_89_c34_r
register_cmap('cet_CET_CBD2',m_diverging_linear_protanopic_deuteranopic_bjy_57_89_c34)
register_cmap('cet_CET_CBD2_r',m_diverging_linear_protanopic_deuteranopic_bjy_57_89_c34_r)




cyclic_mybm_20_100_c48_s25 = [  # cmap_def
[0.27982, 0.21379, 0.65524],
[0.29335, 0.20977, 0.6514],
[0.30684, 0.20625, 0.64745],
[0.32019, 0.20318, 0.64344],
[0.33337, 0.20063, 0.63939],
[0.34632, 0.19866, 0.63531],
[0.35904, 0.19726, 0.63122],
[0.37149, 0.19643, 0.62714],
[0.38367, 0.19613, 0.62309],
[0.39552, 0.1964, 0.61908],
[0.40706, 0.19719, 0.6151],
[0.41827, 0.19847, 0.61119],
[0.42918, 0.20024, 0.60734],
[0.43978, 0.20246, 0.60355],
[0.45006, 0.20508, 0.59983],
[0.46005, 0.20806, 0.59617],
[0.46977, 0.21138, 0.59259],
[0.4792, 0.21495, 0.58904],
[0.4884, 0.21883, 0.58558],
[0.49735, 0.22292, 0.58215],
[0.50606, 0.22721, 0.57879],
[0.51459, 0.2317, 0.57548],
[0.5229, 0.23634, 0.5722],
[0.53103, 0.24111, 0.56896],
[0.53898, 0.24602, 0.56577],
[0.54677, 0.25107, 0.56262],
[0.5544, 0.2562, 0.55948],
[0.56188, 0.26145, 0.55638],
[0.56921, 0.26678, 0.55331],
[0.57643, 0.27219, 0.55028],
[0.58349, 0.27771, 0.54727],
[0.59044, 0.28328, 0.54428],
[0.59728, 0.28893, 0.54131],
[0.60399, 0.29463, 0.53838],
[0.61059, 0.30042, 0.53545],
[0.61707, 0.30629, 0.53257],
[0.62345, 0.31217, 0.5297],
[0.62971, 0.31816, 0.52685],
[0.63588, 0.32419, 0.52402],
[0.64194, 0.33028, 0.52121],
[0.64789, 0.33642, 0.51844],
[0.65375, 0.34263, 0.51566],
[0.6595, 0.34888, 0.51293],
[0.66515, 0.35519, 0.51021],
[0.67071, 0.36154, 0.50752],
[0.67617, 0.36794, 0.50485],
[0.68152, 0.37441, 0.5022],
[0.68678, 0.38091, 0.49956],
[0.69195, 0.38745, 0.49696],
[0.69702, 0.39406, 0.49438],
[0.70199, 0.40072, 0.49182],
[0.70687, 0.40741, 0.48926],
[0.71165, 0.41415, 0.48675],
[0.71634, 0.42093, 0.48424],
[0.72093, 0.42778, 0.48177],
[0.72544, 0.43466, 0.47931],
[0.72984, 0.44158, 0.47686],
[0.73415, 0.44855, 0.47445],
[0.73836, 0.45556, 0.47205],
[0.74249, 0.4626, 0.46967],
[0.74651, 0.46971, 0.46731],
[0.75044, 0.47683, 0.46496],
[0.75427, 0.48399, 0.46264],
[0.758, 0.49121, 0.46034],
[0.76163, 0.49843, 0.45807],
[0.76517, 0.50567, 0.45582],
[0.76859, 0.51296, 0.45358],
[0.77192, 0.52026, 0.45138],
[0.77514, 0.52758, 0.4492],
[0.77826, 0.53492, 0.44703],
[0.78127, 0.54228, 0.4449],
[0.78417, 0.54965, 0.44279],
[0.78697, 0.55704, 0.4407],
[0.78966, 0.56444, 0.43864],
[0.79225, 0.57185, 0.43661],
[0.79473, 0.57927, 0.4346],
[0.79709, 0.58671, 0.43262],
[0.79937, 0.59415, 0.43065],
[0.80152, 0.60161, 0.42871],
[0.80357, 0.60909, 0.42679],
[0.80551, 0.61656, 0.4249],
[0.80736, 0.62405, 0.42303],
[0.80909, 0.63155, 0.42118],
[0.81071, 0.63905, 0.41937],
[0.81223, 0.64657, 0.41758],
[0.81363, 0.65409, 0.4158],
[0.81493, 0.66163, 0.41405],
[0.81613, 0.66916, 0.41233],
[0.8172, 0.6767, 0.41063],
[0.81817, 0.68424, 0.40895],
[0.81903, 0.69179, 0.4073],
[0.81978, 0.69934, 0.40568],
[0.82042, 0.7069, 0.40407],
[0.82094, 0.71445, 0.40249],
[0.82135, 0.722, 0.40096],
[0.82165, 0.72956, 0.39942],
[0.82183, 0.73712, 0.39794],
[0.82189, 0.74467, 0.39647],
[0.82183, 0.75221, 0.39504],
[0.82165, 0.75976, 0.39364],
[0.82134, 0.7673, 0.39227],
[0.8209, 0.77483, 0.39095],
[0.82033, 0.78236, 0.38966],
[0.81961, 0.78988, 0.38842],
[0.81876, 0.79738, 0.38722],
[0.81775, 0.80488, 0.38609],
[0.81657, 0.81235, 0.38502],
[0.81522, 0.81981, 0.38404],
[0.81368, 0.82724, 0.38313],
[0.81194, 0.83464, 0.38231],
[0.80998, 0.84199, 0.38162],
[0.80779, 0.84929, 0.38105],
[0.80534, 0.85652, 0.38063],
[0.80261, 0.86368, 0.38038],
[0.79959, 0.87072, 0.38032],
[0.79623, 0.87765, 0.38048],
[0.79253, 0.88443, 0.38089],
[0.78846, 0.89103, 0.38157],
[0.78399, 0.89742, 0.38254],
[0.7791, 0.90355, 0.38386],
[0.77378, 0.90939, 0.38552],
[0.76801, 0.91491, 0.38757],
[0.76179, 0.92004, 0.39002],
[0.75513, 0.92476, 0.39287],
[0.74801, 0.92901, 0.39614],
[0.74046, 0.93277, 0.3998],
[0.73251, 0.936, 0.40388],
[0.72418, 0.93865, 0.40833],
[0.71551, 0.94073, 0.41314],
[0.70655, 0.94221, 0.41825],
[0.69733, 0.94309, 0.42366],
[0.68793, 0.94337, 0.42931],
[0.67838, 0.94306, 0.43516],
[0.66874, 0.94219, 0.44114],
[0.65905, 0.94078, 0.44724],
[0.64938, 0.93886, 0.45341],
[0.63973, 0.93647, 0.45961],
[0.63016, 0.93363, 0.4658],
[0.62069, 0.93042, 0.47197],
[0.61133, 0.92685, 0.47807],
[0.60211, 0.92297, 0.48409],
[0.59303, 0.9188, 0.49003],
[0.58409, 0.91441, 0.49587],
[0.5753, 0.9098, 0.50161],
[0.56666, 0.90501, 0.50723],
[0.55813, 0.90007, 0.51273],
[0.54973, 0.89499, 0.51813],
[0.54144, 0.8898, 0.52341],
[0.53327, 0.88451, 0.52859],
[0.52518, 0.87914, 0.53365],
[0.51718, 0.8737, 0.53862],
[0.50924, 0.8682, 0.54348],
[0.50137, 0.86264, 0.54826],
[0.49355, 0.85703, 0.55293],
[0.48578, 0.85138, 0.55753],
[0.47806, 0.84569, 0.56204],
[0.47037, 0.83998, 0.56645],
[0.4627, 0.83423, 0.57079],
[0.45508, 0.82844, 0.57505],
[0.44746, 0.82264, 0.57924],
[0.43989, 0.8168, 0.58334],
[0.43233, 0.81094, 0.58737],
[0.42477, 0.80506, 0.59133],
[0.41724, 0.79915, 0.59521],
[0.40973, 0.79323, 0.59903],
[0.40222, 0.78727, 0.60276],
[0.39474, 0.7813, 0.60643],
[0.38727, 0.77531, 0.61003],
[0.37983, 0.76929, 0.61355],
[0.37239, 0.76326, 0.61702],
[0.36497, 0.75722, 0.62041],
[0.35758, 0.75115, 0.62373],
[0.35019, 0.74506, 0.62697],
[0.34283, 0.73896, 0.63015],
[0.33548, 0.73284, 0.63326],
[0.32813, 0.7267, 0.6363],
[0.32083, 0.72054, 0.63927],
[0.31353, 0.71437, 0.64217],
[0.30626, 0.70819, 0.645],
[0.29899, 0.70199, 0.64777],
[0.29175, 0.69578, 0.65047],
[0.28452, 0.68955, 0.65309],
[0.27734, 0.68331, 0.65566],
[0.27017, 0.67706, 0.65816],
[0.26301, 0.6708, 0.66058],
[0.25586, 0.66451, 0.66294],
[0.24877, 0.65823, 0.66523],
[0.24168, 0.65193, 0.66746],
[0.23463, 0.64562, 0.66961],
[0.22758, 0.6393, 0.6717],
[0.22058, 0.63297, 0.67372],
[0.21357, 0.62664, 0.67568],
[0.20663, 0.6203, 0.67758],
[0.19966, 0.61394, 0.67939],
[0.19277, 0.60759, 0.68115],
[0.18592, 0.60122, 0.68284],
[0.17905, 0.59484, 0.68447],
[0.17223, 0.58846, 0.68603],
[0.16543, 0.58208, 0.68753],
[0.15873, 0.5757, 0.68897],
[0.15199, 0.5693, 0.69033],
[0.14536, 0.56291, 0.69163],
[0.1387, 0.5565, 0.69288],
[0.13213, 0.5501, 0.69405],
[0.1256, 0.54369, 0.69517],
[0.1191, 0.53729, 0.69622],
[0.1127, 0.53087, 0.6972],
[0.10633, 0.52445, 0.69813],
[0.099987, 0.51803, 0.699],
[0.093771, 0.5116, 0.6998],
[0.087677, 0.50518, 0.70055],
[0.081652, 0.49876, 0.70123],
[0.075664, 0.49233, 0.70186],
[0.069923, 0.4859, 0.70242],
[0.06427, 0.47947, 0.70294],
[0.058824, 0.47302, 0.70338],
[0.053521, 0.46659, 0.70377],
[0.048609, 0.46015, 0.70411],
[0.04384, 0.45371, 0.70439],
[0.039571, 0.44726, 0.70461],
[0.035783, 0.44081, 0.70477],
[0.032406, 0.43436, 0.70488],
[0.029883, 0.42791, 0.70493],
[0.027994, 0.42144, 0.70493],
[0.02673, 0.41498, 0.70488],
[0.02609, 0.40851, 0.70477],
[0.026078, 0.40202, 0.7046],
[0.026701, 0.39555, 0.70438],
[0.027974, 0.38905, 0.7041],
[0.02992, 0.38254, 0.70376],
[0.032576, 0.37603, 0.70337],
[0.036166, 0.3695, 0.70292],
[0.040364, 0.36296, 0.70239],
[0.045202, 0.35642, 0.70181],
[0.050548, 0.34985, 0.70115],
[0.056478, 0.34327, 0.70042],
[0.062852, 0.33666, 0.69961],
[0.069668, 0.33006, 0.69871],
[0.076852, 0.32343, 0.69773],
[0.084667, 0.31679, 0.69664],
[0.092813, 0.31016, 0.69544],
[0.10151, 0.30349, 0.69412],
[0.11069, 0.29685, 0.69269],
[0.12026, 0.2902, 0.6911],
[0.1304, 0.28359, 0.68939],
[0.14094, 0.27701, 0.6875],
[0.1519, 0.27048, 0.68545],
[0.16336, 0.26402, 0.68323],
[0.17519, 0.25762, 0.68082],
[0.18742, 0.25139, 0.67824],
[0.19994, 0.24526, 0.67546],
[0.21282, 0.23935, 0.67251],
[0.22595, 0.23362, 0.66937],
[0.23925, 0.2282, 0.66606],
[0.25271, 0.22304, 0.6626],
[0.26625, 0.21822, 0.65898],
]
b_cyclic_mybm_20_100_c48_s25 = bokeh_palette('cyclic_mybm_20_100_c48_s25',cyclic_mybm_20_100_c48_s25)
m_cyclic_mybm_20_100_c48_s25 = mpl_cm('cyclic_mybm_20_100_c48_s25',cyclic_mybm_20_100_c48_s25)
m_cyclic_mybm_20_100_c48_s25_r = mpl_cm('cyclic_mybm_20_100_c48_s25_r',list(reversed(cyclic_mybm_20_100_c48_s25)))
CET_C9s = b_cyclic_mybm_20_100_c48_s25
m_CET_C9s = m_cyclic_mybm_20_100_c48_s25
m_CET_C9s_r = m_cyclic_mybm_20_100_c48_s25_r
palette['CET_C9s'] = b_cyclic_mybm_20_100_c48_s25
cm['CET_C9s'] = m_cyclic_mybm_20_100_c48_s25
cm['CET_C9s_r'] = m_cyclic_mybm_20_100_c48_s25_r
register_cmap('cet_CET_C9s',m_cyclic_mybm_20_100_c48_s25)
register_cmap('cet_CET_C9s_r',m_cyclic_mybm_20_100_c48_s25_r)




linear_protanopic_deuteranopic_kyw_5_95_c49 = [  # cmap_def
[0.066001, 0.066024, 0.066019],
[0.071156, 0.069492, 0.065537],
[0.075927, 0.072692, 0.065057],
[0.080551, 0.075818, 0.064574],
[0.084952, 0.078858, 0.064088],
[0.089171, 0.08191, 0.063608],
[0.093163, 0.084749, 0.063143],
[0.097068, 0.08756, 0.062655],
[0.10084, 0.09025, 0.0621],
[0.10445, 0.092863, 0.061536],
[0.10805, 0.095582, 0.061023],
[0.11168, 0.098275, 0.060531],
[0.11521, 0.10093, 0.060012],
[0.11882, 0.10367, 0.059425],
[0.12232, 0.1064, 0.058759],
[0.12593, 0.10911, 0.058058],
[0.12945, 0.11185, 0.05741],
[0.133, 0.11453, 0.056801],
[0.13652, 0.11731, 0.05608],
[0.14006, 0.12008, 0.055301],
[0.14362, 0.12284, 0.054519],
[0.14715, 0.12568, 0.05371],
[0.15068, 0.12846, 0.052879],
[0.1542, 0.13128, 0.052025],
[0.15774, 0.1341, 0.051151],
[0.16131, 0.1369, 0.050244],
[0.16481, 0.13971, 0.049351],
[0.16835, 0.14259, 0.048275],
[0.17188, 0.14547, 0.047257],
[0.17543, 0.14828, 0.046115],
[0.17898, 0.15116, 0.045077],
[0.1825, 0.15403, 0.043837],
[0.18608, 0.15693, 0.042637],
[0.18961, 0.1598, 0.041469],
[0.19317, 0.16275, 0.040124],
[0.19674, 0.16562, 0.038766],
[0.20025, 0.16857, 0.037386],
[0.20385, 0.1715, 0.03599],
[0.20741, 0.17445, 0.034389],
[0.21097, 0.1774, 0.032923],
[0.21453, 0.18033, 0.031457],
[0.21811, 0.18329, 0.02999],
[0.22167, 0.18628, 0.028517],
[0.22524, 0.18925, 0.027043],
[0.22884, 0.19223, 0.025575],
[0.23241, 0.19523, 0.024119],
[0.236, 0.19822, 0.022683],
[0.23957, 0.20122, 0.021274],
[0.24313, 0.20428, 0.019899],
[0.24672, 0.20729, 0.018568],
[0.25029, 0.21031, 0.017288],
[0.2539, 0.21334, 0.016067],
[0.25748, 0.21642, 0.014911],
[0.26106, 0.21946, 0.01383],
[0.26464, 0.22251, 0.012819],
[0.26822, 0.22558, 0.011922],
[0.2718, 0.22866, 0.01089],
[0.27541, 0.23174, 0.010095],
[0.279, 0.23483, 0.009458],
[0.28259, 0.23791, 0.00887],
[0.28615, 0.241, 0.008369],
[0.28976, 0.24409, 0.007948],
[0.29335, 0.24722, 0.007599],
[0.29697, 0.25032, 0.007314],
[0.30056, 0.25346, 0.007086],
[0.30418, 0.25658, 0.006907],
[0.30779, 0.25973, 0.006768],
[0.31142, 0.26287, 0.006662],
[0.31505, 0.26602, 0.006578],
[0.31869, 0.26917, 0.006509],
[0.32234, 0.2723, 0.006447],
[0.32598, 0.27549, 0.006384],
[0.32966, 0.27866, 0.006315],
[0.33333, 0.28183, 0.006263],
[0.337, 0.285, 0.006231],
[0.34068, 0.28819, 0.006218],
[0.34435, 0.2914, 0.006218],
[0.34806, 0.29458, 0.006231],
[0.35174, 0.29778, 0.006253],
[0.35547, 0.30099, 0.006282],
[0.35916, 0.30421, 0.006316],
[0.36288, 0.30743, 0.006354],
[0.36662, 0.31068, 0.006396],
[0.37033, 0.3139, 0.006439],
[0.37408, 0.31714, 0.006484],
[0.37782, 0.32039, 0.006529],
[0.38157, 0.32363, 0.006574],
[0.38532, 0.32688, 0.006619],
[0.3891, 0.33015, 0.006662],
[0.39286, 0.33343, 0.006705],
[0.39663, 0.33669, 0.006748],
[0.40043, 0.33998, 0.00679],
[0.40421, 0.34326, 0.006831],
[0.408, 0.34654, 0.006871],
[0.4118, 0.34984, 0.006911],
[0.41561, 0.35314, 0.00695],
[0.41943, 0.35644, 0.006988],
[0.42324, 0.35975, 0.007026],
[0.42708, 0.36306, 0.007063],
[0.43091, 0.36639, 0.007099],
[0.43474, 0.36971, 0.007134],
[0.43858, 0.37304, 0.007169],
[0.44244, 0.37639, 0.007203],
[0.44629, 0.37972, 0.007235],
[0.45016, 0.38307, 0.007268],
[0.45402, 0.38641, 0.007299],
[0.45789, 0.38977, 0.007329],
[0.46178, 0.39313, 0.007359],
[0.46565, 0.3965, 0.007388],
[0.46956, 0.39987, 0.007416],
[0.47344, 0.40326, 0.007443],
[0.47735, 0.40664, 0.007469],
[0.48126, 0.41003, 0.007494],
[0.48517, 0.41342, 0.007518],
[0.48909, 0.41681, 0.007541],
[0.49301, 0.42021, 0.007563],
[0.49695, 0.42361, 0.007585],
[0.50089, 0.42704, 0.007605],
[0.50483, 0.43045, 0.007625],
[0.50878, 0.43386, 0.007643],
[0.51273, 0.4373, 0.00766],
[0.51669, 0.44072, 0.007677],
[0.52065, 0.44417, 0.007692],
[0.52463, 0.44759, 0.007706],
[0.5286, 0.45105, 0.007719],
[0.53258, 0.4545, 0.007731],
[0.53657, 0.45795, 0.007742],
[0.54056, 0.46141, 0.007752],
[0.54456, 0.46488, 0.007761],
[0.54856, 0.46834, 0.007769],
[0.55257, 0.47182, 0.007775],
[0.55658, 0.4753, 0.007781],
[0.5606, 0.47877, 0.007785],
[0.56462, 0.48226, 0.007788],
[0.56864, 0.48575, 0.00779],
[0.57269, 0.48924, 0.007791],
[0.57673, 0.49274, 0.00779],
[0.58076, 0.49625, 0.007788],
[0.58482, 0.49976, 0.007785],
[0.58886, 0.50327, 0.007781],
[0.59293, 0.50679, 0.007776],
[0.597, 0.51031, 0.007769],
[0.60106, 0.51385, 0.007761],
[0.60514, 0.51738, 0.007752],
[0.60922, 0.52091, 0.007741],
[0.6133, 0.52445, 0.007729],
[0.61739, 0.52799, 0.007716],
[0.62148, 0.53154, 0.007701],
[0.62559, 0.53509, 0.007685],
[0.62969, 0.53866, 0.007668],
[0.6338, 0.54221, 0.007649],
[0.63792, 0.54578, 0.007629],
[0.64204, 0.54935, 0.007608],
[0.64616, 0.55292, 0.007585],
[0.65029, 0.5565, 0.00756],
[0.65442, 0.56009, 0.007534],
[0.65856, 0.56368, 0.007507],
[0.66271, 0.56727, 0.007478],
[0.66685, 0.57085, 0.007448],
[0.67101, 0.57445, 0.007416],
[0.67516, 0.57805, 0.007383],
[0.67932, 0.58166, 0.007348],
[0.68349, 0.58528, 0.007312],
[0.68767, 0.58889, 0.007274],
[0.69185, 0.59252, 0.007235],
[0.69603, 0.59613, 0.007194],
[0.70021, 0.59976, 0.007151],
[0.7044, 0.6034, 0.007107],
[0.7086, 0.60704, 0.007061],
[0.71279, 0.61067, 0.007014],
[0.71701, 0.61431, 0.006964],
[0.72121, 0.61796, 0.006914],
[0.72543, 0.62161, 0.006861],
[0.72964, 0.62527, 0.006807],
[0.73387, 0.62893, 0.006752],
[0.73809, 0.63259, 0.006694],
[0.74233, 0.63626, 0.006635],
[0.74656, 0.63993, 0.006574],
[0.75081, 0.6436, 0.006512],
[0.75505, 0.64729, 0.006448],
[0.7593, 0.65097, 0.006382],
[0.76355, 0.65465, 0.006314],
[0.76781, 0.65834, 0.006245],
[0.77207, 0.66204, 0.006173],
[0.77634, 0.66574, 0.0061],
[0.78061, 0.66944, 0.006025],
[0.78489, 0.67315, 0.005948],
[0.78917, 0.67686, 0.00587],
[0.79346, 0.68056, 0.005789],
[0.79774, 0.68428, 0.005707],
[0.80204, 0.68801, 0.005623],
[0.80634, 0.69173, 0.005537],
[0.81064, 0.69546, 0.005449],
[0.81494, 0.69919, 0.005359],
[0.81925, 0.70293, 0.005267],
[0.82357, 0.70667, 0.005173],
[0.82789, 0.71041, 0.005078],
[0.83222, 0.71415, 0.00498],
[0.83654, 0.71791, 0.00488],
[0.84088, 0.72165, 0.004779],
[0.84521, 0.72542, 0.004675],
[0.84955, 0.72918, 0.004569],
[0.85389, 0.73294, 0.004462],
[0.85824, 0.73671, 0.004352],
[0.8626, 0.74048, 0.00424],
[0.86695, 0.74426, 0.004126],
[0.87132, 0.74804, 0.004011],
[0.87568, 0.75182, 0.003893],
[0.88005, 0.75561, 0.003778],
[0.88442, 0.75939, 0.003729],
[0.8888, 0.76319, 0.003794],
[0.89317, 0.76698, 0.004017],
[0.89756, 0.77079, 0.004459],
[0.90193, 0.77458, 0.005199],
[0.90631, 0.77839, 0.006337],
[0.91069, 0.78221, 0.008002],
[0.91506, 0.78602, 0.010329],
[0.91943, 0.78983, 0.013798],
[0.9238, 0.79365, 0.018176],
[0.92814, 0.79747, 0.023988],
[0.93248, 0.80129, 0.031587],
[0.9368, 0.80512, 0.041609],
[0.94109, 0.80895, 0.052744],
[0.94535, 0.81277, 0.065002],
[0.94957, 0.8166, 0.078113],
[0.95375, 0.82043, 0.092278],
[0.95788, 0.82426, 0.10734],
[0.96193, 0.82809, 0.12331],
[0.96591, 0.83193, 0.14032],
[0.96981, 0.83576, 0.15828],
[0.97359, 0.83961, 0.17718],
[0.97726, 0.84344, 0.19707],
[0.98078, 0.84728, 0.21791],
[0.98414, 0.85113, 0.23971],
[0.98731, 0.855, 0.26248],
[0.99027, 0.85886, 0.28619],
[0.99299, 0.86274, 0.31086],
[0.99544, 0.86663, 0.33641],
[0.9976, 0.87054, 0.36287],
[0.99943, 0.87447, 0.39018],
[1, 0.87843, 0.41832],
[1, 0.88241, 0.44725],
[1, 0.88643, 0.47692],
[1, 0.89047, 0.50728],
[1, 0.89456, 0.53829],
[1, 0.89868, 0.56985],
[1, 0.90285, 0.60197],
[0.9983, 0.90707, 0.63457],
[0.99572, 0.91133, 0.66758],
[0.99251, 0.91564, 0.70097],
[0.98863, 0.92001, 0.7347],
[0.98406, 0.92443, 0.76871],
[0.97875, 0.9289, 0.80298],
[0.97269, 0.93343, 0.83748],
[0.96583, 0.93802, 0.87217],
[0.95814, 0.94266, 0.90703],
]
b_linear_protanopic_deuteranopic_kyw_5_95_c49 = bokeh_palette('linear_protanopic_deuteranopic_kyw_5_95_c49',linear_protanopic_deuteranopic_kyw_5_95_c49)
m_linear_protanopic_deuteranopic_kyw_5_95_c49 = mpl_cm('linear_protanopic_deuteranopic_kyw_5_95_c49',linear_protanopic_deuteranopic_kyw_5_95_c49)
m_linear_protanopic_deuteranopic_kyw_5_95_c49_r = mpl_cm('linear_protanopic_deuteranopic_kyw_5_95_c49_r',list(reversed(linear_protanopic_deuteranopic_kyw_5_95_c49)))
CET_CBL4 = b_linear_protanopic_deuteranopic_kyw_5_95_c49
m_CET_CBL4 = m_linear_protanopic_deuteranopic_kyw_5_95_c49
m_CET_CBL4_r = m_linear_protanopic_deuteranopic_kyw_5_95_c49_r
palette['CET_CBL4'] = b_linear_protanopic_deuteranopic_kyw_5_95_c49
cm['CET_CBL4'] = m_linear_protanopic_deuteranopic_kyw_5_95_c49
cm['CET_CBL4_r'] = m_linear_protanopic_deuteranopic_kyw_5_95_c49_r
register_cmap('cet_CET_CBL4',m_linear_protanopic_deuteranopic_kyw_5_95_c49)
register_cmap('cet_CET_CBL4_r',m_linear_protanopic_deuteranopic_kyw_5_95_c49_r)




cyclic_bgrmb_35_70_c75 = [  # cmap_def
[0.27965, 0.19184, 0.94712],
[0.27896, 0.19798, 0.94102],
[0.28028, 0.20489, 0.93432],
[0.28345, 0.21245, 0.92705],
[0.28818, 0.22063, 0.91923],
[0.29417, 0.22927, 0.9109],
[0.30112, 0.23836, 0.9021],
[0.30875, 0.24782, 0.89288],
[0.31674, 0.25752, 0.8833],
[0.32491, 0.26742, 0.8734],
[0.33309, 0.27749, 0.86325],
[0.3411, 0.28761, 0.85287],
[0.34885, 0.29779, 0.84232],
[0.35627, 0.30796, 0.83163],
[0.36329, 0.31812, 0.82083],
[0.3699, 0.32822, 0.80995],
[0.37608, 0.33826, 0.79901],
[0.3818, 0.34824, 0.78801],
[0.38708, 0.35813, 0.77699],
[0.39193, 0.36793, 0.76593],
[0.39635, 0.37766, 0.75486],
[0.40035, 0.38731, 0.74376],
[0.40393, 0.39688, 0.73265],
[0.40712, 0.40639, 0.72152],
[0.40992, 0.4158, 0.71039],
[0.41234, 0.42516, 0.69924],
[0.41438, 0.43445, 0.68808],
[0.41607, 0.44369, 0.6769],
[0.41741, 0.45287, 0.66569],
[0.41838, 0.46199, 0.65446],
[0.41902, 0.47107, 0.6432],
[0.41932, 0.4801, 0.63192],
[0.41927, 0.48908, 0.62061],
[0.41888, 0.49805, 0.60925],
[0.41815, 0.50695, 0.59785],
[0.41709, 0.51582, 0.58642],
[0.41568, 0.52468, 0.57493],
[0.41392, 0.53349, 0.56339],
[0.4118, 0.54228, 0.55179],
[0.40934, 0.55104, 0.54012],
[0.40651, 0.55977, 0.52839],
[0.4033, 0.56847, 0.51658],
[0.3997, 0.57715, 0.50468],
[0.39574, 0.58582, 0.49266],
[0.39136, 0.59445, 0.48055],
[0.38657, 0.60305, 0.46831],
[0.38139, 0.61161, 0.45594],
[0.37578, 0.62013, 0.44344],
[0.36976, 0.62857, 0.43077],
[0.36334, 0.63694, 0.41792],
[0.35653, 0.6452, 0.40489],
[0.34935, 0.65334, 0.39167],
[0.34186, 0.66131, 0.37823],
[0.33413, 0.66906, 0.36458],
[0.32623, 0.67654, 0.35072],
[0.31833, 0.68369, 0.33663],
[0.31058, 0.69043, 0.32235],
[0.30313, 0.69669, 0.30787],
[0.2963, 0.70239, 0.29327],
[0.2903, 0.70744, 0.27858],
[0.28545, 0.71177, 0.26385],
[0.28203, 0.71529, 0.24913],
[0.28024, 0.71794, 0.23456],
[0.28032, 0.71968, 0.22018],
[0.28239, 0.72046, 0.20611],
[0.28637, 0.72027, 0.19243],
[0.29226, 0.71911, 0.17929],
[0.29984, 0.71701, 0.16677],
[0.30887, 0.71399, 0.1549],
[0.31906, 0.71013, 0.14384],
[0.33015, 0.70547, 0.13357],
[0.34187, 0.70011, 0.12411],
[0.35397, 0.6941, 0.11555],
[0.36627, 0.68756, 0.10776],
[0.37856, 0.68054, 0.10074],
[0.39074, 0.67313, 0.094469],
[0.40269, 0.66539, 0.088741],
[0.41435, 0.65739, 0.083562],
[0.42567, 0.64917, 0.078779],
[0.43662, 0.64077, 0.074448],
[0.44718, 0.63224, 0.070334],
[0.45735, 0.6236, 0.066462],
[0.46716, 0.61486, 0.0628],
[0.47658, 0.60606, 0.059179],
[0.48564, 0.5972, 0.0556],
[0.49438, 0.58826, 0.05209],
[0.50277, 0.57928, 0.048789],
[0.51086, 0.57025, 0.045323],
[0.51866, 0.56117, 0.041976],
[0.52617, 0.55206, 0.038511],
[0.53341, 0.54288, 0.035214],
[0.54041, 0.53365, 0.031867],
[0.54716, 0.52436, 0.028877],
[0.55369, 0.51501, 0.026036],
[0.56001, 0.50559, 0.023343],
[0.56611, 0.4961, 0.020796],
[0.57202, 0.48654, 0.018393],
[0.57773, 0.4769, 0.016132],
[0.58327, 0.46717, 0.014011],
[0.58864, 0.45733, 0.012051],
[0.59384, 0.4474, 0.009965],
[0.59888, 0.43738, 0.008255],
[0.60376, 0.42723, 0.006667],
[0.60849, 0.41694, 0.005204],
[0.61307, 0.40652, 0.003866],
[0.61752, 0.39596, 0.002651],
[0.62182, 0.38521, 0.001556],
[0.62601, 0.37431, 0.000601],
[0.63008, 0.3632, 0],
[0.63404, 0.35189, 0],
[0.63787, 0.34038, 0],
[0.64161, 0.32863, 0],
[0.64526, 0.31662, 0],
[0.64884, 0.30433, 0],
[0.65235, 0.29178, 0],
[0.65582, 0.27893, 0],
[0.65925, 0.26576, 0.000716],
[0.66268, 0.25228, 0.002273],
[0.66612, 0.23847, 0.004328],
[0.66962, 0.22433, 0.006956],
[0.67318, 0.20991, 0.010213],
[0.67685, 0.19515, 0.014464],
[0.68065, 0.18008, 0.019341],
[0.68462, 0.16474, 0.025185],
[0.68878, 0.14912, 0.032125],
[0.69315, 0.13324, 0.040511],
[0.69773, 0.11711, 0.049454],
[0.70256, 0.10078, 0.058636],
[0.70762, 0.084326, 0.068259],
[0.71291, 0.067632, 0.078128],
[0.71843, 0.050818, 0.088512],
[0.72414, 0.034099, 0.099205],
[0.73004, 0.020769, 0.11034],
[0.73608, 0.011276, 0.12174],
[0.74226, 0.005477, 0.13361],
[0.74853, 0.003229, 0.14578],
[0.75487, 0.004142, 0.15823],
[0.76125, 0.007923, 0.17094],
[0.76766, 0.014473, 0.1839],
[0.77407, 0.02311, 0.19709],
[0.78047, 0.033735, 0.21042],
[0.78684, 0.045986, 0.2239],
[0.79318, 0.057778, 0.23755],
[0.79947, 0.069155, 0.25126],
[0.80571, 0.07976, 0.26505],
[0.8119, 0.090028, 0.27896],
[0.81803, 0.099812, 0.29291],
[0.8241, 0.10928, 0.30694],
[0.83011, 0.11837, 0.32103],
[0.83606, 0.12712, 0.33519],
[0.84195, 0.13567, 0.34937],
[0.84776, 0.144, 0.36361],
[0.85352, 0.15207, 0.37792],
[0.85921, 0.15997, 0.39227],
[0.86483, 0.16775, 0.40669],
[0.87039, 0.17537, 0.42116],
[0.87589, 0.18285, 0.4357],
[0.88131, 0.19026, 0.45028],
[0.88667, 0.19756, 0.46493],
[0.89195, 0.20479, 0.47964],
[0.89717, 0.21195, 0.4944],
[0.90231, 0.21901, 0.50922],
[0.90739, 0.22604, 0.5241],
[0.91239, 0.23298, 0.53904],
[0.91733, 0.2399, 0.55404],
[0.92219, 0.24678, 0.5691],
[0.92697, 0.25362, 0.58422],
[0.93168, 0.26042, 0.59939],
[0.93632, 0.26717, 0.61462],
[0.94088, 0.27393, 0.62991],
[0.94537, 0.28062, 0.64527],
[0.94976, 0.28731, 0.66069],
[0.95407, 0.29396, 0.67616],
[0.95827, 0.30058, 0.69167],
[0.96235, 0.30716, 0.70723],
[0.96631, 0.31369, 0.72282],
[0.97012, 0.32016, 0.73844],
[0.97376, 0.32652, 0.75407],
[0.9772, 0.33283, 0.76967],
[0.98041, 0.33898, 0.78524],
[0.98332, 0.34498, 0.80073],
[0.98591, 0.35079, 0.81609],
[0.9881, 0.35635, 0.83126],
[0.98986, 0.36162, 0.84617],
[0.9911, 0.36656, 0.86076],
[0.99177, 0.37108, 0.87494],
[0.99181, 0.37516, 0.88861],
[0.99117, 0.37874, 0.90168],
[0.98981, 0.38176, 0.91407],
[0.98769, 0.38418, 0.92568],
[0.98479, 0.38597, 0.93644],
[0.98112, 0.38713, 0.94628],
[0.97668, 0.38763, 0.95518],
[0.9715, 0.38749, 0.9631],
[0.96562, 0.38671, 0.97004],
[0.9591, 0.38535, 0.97603],
[0.95198, 0.38345, 0.98112],
[0.94434, 0.38103, 0.98535],
[0.93623, 0.37815, 0.98879],
[0.92772, 0.37488, 0.99154],
[0.91888, 0.37127, 0.99367],
[0.90976, 0.36739, 0.99527],
[0.90041, 0.36329, 0.99643],
[0.89086, 0.35899, 0.99723],
[0.88116, 0.35457, 0.99773],
[0.87133, 0.35003, 0.998],
[0.86139, 0.3454, 0.99808],
[0.85136, 0.34074, 0.99803],
[0.84126, 0.33602, 0.99788],
[0.83108, 0.33129, 0.99765],
[0.82084, 0.32655, 0.99736],
[0.81054, 0.32182, 0.99704],
[0.80018, 0.3171, 0.99669],
[0.78976, 0.31236, 0.9963],
[0.77928, 0.30766, 0.99592],
[0.76874, 0.30297, 0.99553],
[0.75814, 0.29831, 0.99513],
[0.74746, 0.29366, 0.99474],
[0.7367, 0.28905, 0.99433],
[0.72588, 0.28443, 0.99393],
[0.71496, 0.27985, 0.99352],
[0.70396, 0.27529, 0.99311],
[0.69288, 0.27076, 0.99269],
[0.68168, 0.26623, 0.99226],
[0.67041, 0.26173, 0.99183],
[0.659, 0.25727, 0.9914],
[0.64749, 0.25282, 0.99097],
[0.63586, 0.24842, 0.99053],
[0.6241, 0.244, 0.99009],
[0.61219, 0.23965, 0.98964],
[0.60014, 0.23533, 0.98919],
[0.58793, 0.23102, 0.98874],
[0.57556, 0.22675, 0.98828],
[0.563, 0.2225, 0.98782],
[0.55024, 0.2183, 0.98735],
[0.53728, 0.21413, 0.98689],
[0.5241, 0.21, 0.9864],
[0.5107, 0.20596, 0.98591],
[0.49707, 0.20195, 0.9854],
[0.48321, 0.19806, 0.98486],
[0.46912, 0.19431, 0.98429],
[0.4548, 0.19064, 0.98367],
[0.44027, 0.18719, 0.98299],
[0.42558, 0.18393, 0.98222],
[0.41078, 0.18095, 0.98134],
[0.39592, 0.17835, 0.98032],
[0.38113, 0.17608, 0.97912],
[0.36652, 0.17438, 0.9777],
[0.35226, 0.17319, 0.97601],
[0.33858, 0.17267, 0.97401],
[0.32567, 0.17287, 0.97165],
[0.31386, 0.17387, 0.96886],
[0.30335, 0.17569, 0.9656],
[0.29447, 0.17844, 0.96183],
[0.28745, 0.18201, 0.95752],
[0.28249, 0.18651, 0.95262],
]
b_cyclic_bgrmb_35_70_c75 = bokeh_palette('cyclic_bgrmb_35_70_c75',cyclic_bgrmb_35_70_c75)
m_cyclic_bgrmb_35_70_c75 = mpl_cm('cyclic_bgrmb_35_70_c75',cyclic_bgrmb_35_70_c75)
m_cyclic_bgrmb_35_70_c75_r = mpl_cm('cyclic_bgrmb_35_70_c75_r',list(reversed(cyclic_bgrmb_35_70_c75)))
CET_C11 = b_cyclic_bgrmb_35_70_c75
m_CET_C11 = m_cyclic_bgrmb_35_70_c75
m_CET_C11_r = m_cyclic_bgrmb_35_70_c75_r
palette['CET_C11'] = b_cyclic_bgrmb_35_70_c75
cm['CET_C11'] = m_cyclic_bgrmb_35_70_c75
cm['CET_C11_r'] = m_cyclic_bgrmb_35_70_c75_r
register_cmap('cet_CET_C11',m_cyclic_bgrmb_35_70_c75)
register_cmap('cet_CET_C11_r',m_cyclic_bgrmb_35_70_c75_r)




rainbow_bgyr_10_90_c83 = [  # cmap_def
[0.014954, 0.001872, 0.42514],
[0.015287, 0.001815, 0.43669],
[0.015613, 0.001749, 0.44829],
[0.015931, 0.001675, 0.45994],
[0.016241, 0.001592, 0.47166],
[0.016542, 0.0015, 0.48341],
[0.016834, 0.001401, 0.49523],
[0.017118, 0.0013, 0.50708],
[0.017394, 0.001201, 0.51898],
[0.017665, 0.001114, 0.5309],
[0.017934, 0.001055, 0.54286],
[0.018205, 0.001046, 0.5548],
[0.018487, 0.00112, 0.56674],
[0.018789, 0.001324, 0.57862],
[0.019126, 0.001716, 0.59042],
[0.019516, 0.002372, 0.6021],
[0.019978, 0.003382, 0.6136],
[0.020536, 0.004846, 0.62487],
[0.021211, 0.006869, 0.63587],
[0.022024, 0.009561, 0.64653],
[0.02299, 0.013186, 0.65685],
[0.024118, 0.017469, 0.66677],
[0.025409, 0.022632, 0.67632],
[0.026855, 0.028707, 0.6855],
[0.028441, 0.035905, 0.69432],
[0.030147, 0.043546, 0.70284],
[0.031948, 0.051388, 0.71108],
[0.033796, 0.059154, 0.71911],
[0.035956, 0.066658, 0.72697],
[0.037895, 0.074065, 0.7347],
[0.039858, 0.081258, 0.74234],
[0.041831, 0.088282, 0.74992],
[0.043551, 0.09517, 0.75747],
[0.045404, 0.10183, 0.765],
[0.047166, 0.10838, 0.77251],
[0.048879, 0.11481, 0.78003],
[0.050405, 0.12113, 0.78756],
[0.051912, 0.12736, 0.7951],
[0.053383, 0.1335, 0.80264],
[0.05482, 0.1395, 0.8102],
[0.056238, 0.14552, 0.81778],
[0.057458, 0.15138, 0.82537],
[0.058764, 0.15721, 0.83297],
[0.060009, 0.16303, 0.84059],
[0.061046, 0.16876, 0.84821],
[0.062195, 0.17446, 0.85585],
[0.063259, 0.1801, 0.8635],
[0.064218, 0.18577, 0.87115],
[0.06518, 0.19135, 0.87879],
[0.066141, 0.19698, 0.8864],
[0.067137, 0.20259, 0.89393],
[0.068205, 0.20827, 0.90135],
[0.069419, 0.21403, 0.90858],
[0.07072, 0.21991, 0.91551],
[0.072366, 0.22598, 0.92203],
[0.074465, 0.2323, 0.92793],
[0.077006, 0.23896, 0.93304],
[0.080229, 0.24607, 0.93714],
[0.084124, 0.2537, 0.93995],
[0.088557, 0.2619, 0.94127],
[0.093514, 0.27074, 0.94087],
[0.098878, 0.28019, 0.9386],
[0.10431, 0.29028, 0.93437],
[0.10967, 0.30093, 0.92814],
[0.11441, 0.31206, 0.91999],
[0.11855, 0.32358, 0.91003],
[0.12154, 0.33537, 0.89846],
[0.12334, 0.3473, 0.8855],
[0.12373, 0.35927, 0.87138],
[0.12255, 0.37121, 0.85635],
[0.11974, 0.38307, 0.8406],
[0.11516, 0.39475, 0.82432],
[0.10879, 0.40625, 0.80766],
[0.10058, 0.41751, 0.7907],
[0.090594, 0.42851, 0.7735],
[0.078794, 0.43924, 0.75611],
[0.065569, 0.44969, 0.7385],
[0.051213, 0.45983, 0.72067],
[0.03689, 0.46967, 0.70259],
[0.025775, 0.47916, 0.68419],
[0.020504, 0.48833, 0.66546],
[0.021772, 0.49716, 0.64633],
[0.030045, 0.50566, 0.62677],
[0.045253, 0.51388, 0.60676],
[0.063127, 0.52179, 0.58629],
[0.081174, 0.52948, 0.56537],
[0.098549, 0.53695, 0.54402],
[0.11466, 0.54423, 0.52225],
[0.12932, 0.55139, 0.50011],
[0.14228, 0.55843, 0.47763],
[0.15344, 0.56541, 0.45484],
[0.16287, 0.57235, 0.43176],
[0.17053, 0.57926, 0.40841],
[0.17656, 0.58613, 0.38484],
[0.18111, 0.59298, 0.36108],
[0.18438, 0.5998, 0.33721],
[0.18661, 0.60658, 0.31324],
[0.18815, 0.61328, 0.28931],
[0.18932, 0.61991, 0.26549],
[0.19062, 0.62641, 0.24193],
[0.19252, 0.63277, 0.21878],
[0.19546, 0.63895, 0.19619],
[0.19984, 0.64494, 0.17432],
[0.206, 0.65073, 0.15334],
[0.21396, 0.6563, 0.13348],
[0.22373, 0.66165, 0.11484],
[0.23514, 0.66678, 0.09774],
[0.24786, 0.67173, 0.082245],
[0.26161, 0.67651, 0.068538],
[0.27612, 0.68112, 0.056756],
[0.29104, 0.68563, 0.046981],
[0.3062, 0.69003, 0.039197],
[0.32139, 0.69434, 0.033329],
[0.33652, 0.69861, 0.029483],
[0.35149, 0.70283, 0.026897],
[0.36627, 0.707, 0.025161],
[0.38082, 0.71116, 0.023976],
[0.39513, 0.71529, 0.023134],
[0.40922, 0.71941, 0.022502],
[0.42308, 0.7235, 0.021972],
[0.43677, 0.72759, 0.021474],
[0.45025, 0.73167, 0.020982],
[0.46357, 0.73574, 0.020492],
[0.47673, 0.73981, 0.020003],
[0.48975, 0.74385, 0.019515],
[0.50265, 0.74789, 0.01903],
[0.51543, 0.75191, 0.018545],
[0.52811, 0.75592, 0.018063],
[0.54069, 0.75992, 0.017582],
[0.55317, 0.76392, 0.017104],
[0.56558, 0.7679, 0.016628],
[0.57791, 0.77187, 0.016155],
[0.59017, 0.77583, 0.015685],
[0.60237, 0.77978, 0.015217],
[0.6145, 0.78373, 0.014752],
[0.62659, 0.78765, 0.014293],
[0.63862, 0.79157, 0.013835],
[0.65061, 0.79548, 0.013374],
[0.66255, 0.79938, 0.012922],
[0.67445, 0.80327, 0.012494],
[0.68631, 0.80715, 0.012066],
[0.69814, 0.81101, 0.011573],
[0.70994, 0.81487, 0.01102],
[0.7217, 0.81871, 0.010517],
[0.73344, 0.82256, 0.010097],
[0.74515, 0.82638, 0.0097],
[0.75684, 0.8302, 0.009289],
[0.7685, 0.834, 0.008873],
[0.78016, 0.8378, 0.008466],
[0.79181, 0.84158, 0.008072],
[0.80344, 0.84533, 0.007691],
[0.81506, 0.84905, 0.007325],
[0.82666, 0.85271, 0.006983],
[0.83824, 0.85628, 0.006675],
[0.84979, 0.85973, 0.006419],
[0.86127, 0.863, 0.006233],
[0.87266, 0.86602, 0.006146],
[0.8839, 0.8687, 0.006189],
[0.89492, 0.87094, 0.0064],
[0.90561, 0.87262, 0.006818],
[0.91587, 0.87364, 0.007481],
[0.9256, 0.8739, 0.008416],
[0.93465, 0.8733, 0.00966],
[0.94294, 0.8718, 0.011317],
[0.95036, 0.86937, 0.013241],
[0.95688, 0.86602, 0.015382],
[0.96247, 0.8618, 0.01777],
[0.96716, 0.8568, 0.020368],
[0.97102, 0.85111, 0.023131],
[0.97413, 0.84486, 0.026014],
[0.9766, 0.83814, 0.028975],
[0.97854, 0.83106, 0.031977],
[0.98006, 0.82371, 0.035191],
[0.98125, 0.81618, 0.038199],
[0.9822, 0.8085, 0.041189],
[0.98297, 0.80073, 0.04394],
[0.9836, 0.7929, 0.046668],
[0.98415, 0.78503, 0.049284],
[0.98462, 0.77713, 0.051617],
[0.98502, 0.76919, 0.053935],
[0.98538, 0.76124, 0.056227],
[0.98571, 0.75328, 0.058279],
[0.98599, 0.7453, 0.060384],
[0.98625, 0.73731, 0.062343],
[0.98647, 0.7293, 0.064193],
[0.98665, 0.72126, 0.065991],
[0.98681, 0.71321, 0.067753],
[0.98692, 0.70514, 0.06948],
[0.98701, 0.69705, 0.071055],
[0.98706, 0.68893, 0.072572],
[0.98708, 0.68078, 0.074106],
[0.98707, 0.67262, 0.075494],
[0.98703, 0.66442, 0.076846],
[0.98698, 0.65619, 0.078152],
[0.9869, 0.64792, 0.079399],
[0.98682, 0.63961, 0.08057],
[0.98673, 0.63126, 0.081665],
[0.98665, 0.62284, 0.082563],
[0.9866, 0.61435, 0.08332],
[0.98658, 0.6058, 0.083988],
[0.98661, 0.59716, 0.084427],
[0.98671, 0.58841, 0.084648],
[0.98688, 0.57957, 0.084663],
[0.98713, 0.57058, 0.084465],
[0.98746, 0.56148, 0.08403],
[0.98787, 0.55224, 0.083325],
[0.98836, 0.54285, 0.082497],
[0.98892, 0.5333, 0.081463],
[0.98954, 0.52362, 0.080216],
[0.99019, 0.51379, 0.078866],
[0.99086, 0.5038, 0.077437],
[0.99155, 0.49367, 0.075965],
[0.99223, 0.48339, 0.07447],
[0.99291, 0.47295, 0.072943],
[0.99356, 0.46237, 0.071411],
[0.99418, 0.45162, 0.06986],
[0.99477, 0.44068, 0.068354],
[0.99532, 0.42957, 0.066843],
[0.99584, 0.41824, 0.065393],
[0.99632, 0.40672, 0.064005],
[0.99676, 0.39495, 0.062706],
[0.99715, 0.38293, 0.06129],
[0.99748, 0.37064, 0.060151],
[0.99775, 0.35808, 0.05894],
[0.99793, 0.34521, 0.057722],
[0.99799, 0.33205, 0.056764],
[0.99791, 0.31858, 0.055743],
[0.99764, 0.30484, 0.054807],
[0.99713, 0.29087, 0.05397],
[0.99631, 0.27673, 0.053231],
[0.99511, 0.2625, 0.052596],
[0.99347, 0.24832, 0.052064],
[0.99132, 0.23427, 0.051634],
[0.9886, 0.22056, 0.051303],
[0.98528, 0.20728, 0.051062],
[0.98135, 0.19464, 0.050904],
[0.9768, 0.18266, 0.050819],
[0.97167, 0.17154, 0.050796],
[0.96602, 0.16125, 0.050825],
[0.9599, 0.15172, 0.050893],
[0.95339, 0.14297, 0.050989],
[0.94656, 0.13485, 0.051103],
[0.93949, 0.12717, 0.051227],
[0.93223, 0.11991, 0.051354],
[0.92484, 0.11292, 0.051478],
[0.91737, 0.10597, 0.051595],
[0.90984, 0.099007, 0.051703],
[0.90228, 0.092003, 0.051798],
[0.89471, 0.08475, 0.051881],
[0.88713, 0.077159, 0.051949],
[0.87955, 0.069372, 0.052003],
[0.87199, 0.060843, 0.052042],
[0.86443, 0.051718, 0.052067],
[0.85688, 0.041794, 0.052077],
[0.84935, 0.0308, 0.052072],
[0.84183, 0.020321, 0.052053],
]
b_rainbow_bgyr_10_90_c83 = bokeh_palette('rainbow_bgyr_10_90_c83',rainbow_bgyr_10_90_c83)
m_rainbow_bgyr_10_90_c83 = mpl_cm('rainbow_bgyr_10_90_c83',rainbow_bgyr_10_90_c83)
m_rainbow_bgyr_10_90_c83_r = mpl_cm('rainbow_bgyr_10_90_c83_r',list(reversed(rainbow_bgyr_10_90_c83)))
rainbow4 = b_rainbow_bgyr_10_90_c83
m_rainbow4 = m_rainbow_bgyr_10_90_c83
m_rainbow4_r = m_rainbow_bgyr_10_90_c83_r
palette['rainbow4'] = b_rainbow_bgyr_10_90_c83
palette_n['rainbow4'] = b_rainbow_bgyr_10_90_c83
cm['rainbow4'] = m_rainbow_bgyr_10_90_c83
cm['rainbow4_r'] = m_rainbow_bgyr_10_90_c83_r
cm_n['rainbow4'] = mpl_cm('rainbow4',rainbow_bgyr_10_90_c83)
cm_n['rainbow4_r'] = mpl_cm('rainbow4_r',list(reversed(rainbow_bgyr_10_90_c83)))
CET_R4 = b_rainbow_bgyr_10_90_c83
m_CET_R4 = m_rainbow_bgyr_10_90_c83
m_CET_R4_r = m_rainbow_bgyr_10_90_c83_r
palette['CET_R4'] = b_rainbow_bgyr_10_90_c83
cm['CET_R4'] = m_rainbow_bgyr_10_90_c83
cm['CET_R4_r'] = m_rainbow_bgyr_10_90_c83_r
register_cmap('cet_CET_R4',m_rainbow_bgyr_10_90_c83)
register_cmap('cet_CET_R4_r',m_rainbow_bgyr_10_90_c83_r)




cyclic_mygbm_50_90_c46_s25 = [  # cmap_def
[0.39731, 0.48556, 0.90673],
[0.41416, 0.48237, 0.90357],
[0.43092, 0.47948, 0.90001],
[0.44739, 0.47689, 0.89611],
[0.46349, 0.47462, 0.89193],
[0.47914, 0.47267, 0.88749],
[0.49428, 0.47102, 0.88287],
[0.50887, 0.46966, 0.87809],
[0.52293, 0.46856, 0.87318],
[0.53648, 0.46769, 0.86819],
[0.54951, 0.46708, 0.86311],
[0.56208, 0.46665, 0.85797],
[0.57418, 0.46642, 0.85281],
[0.58588, 0.46635, 0.8476],
[0.59718, 0.46644, 0.84238],
[0.60812, 0.46669, 0.83713],
[0.61872, 0.46707, 0.83188],
[0.62898, 0.46758, 0.82661],
[0.63895, 0.46819, 0.82133],
[0.64864, 0.46894, 0.81606],
[0.65807, 0.46978, 0.81078],
[0.66725, 0.47071, 0.8055],
[0.6762, 0.47174, 0.80023],
[0.68493, 0.47284, 0.79496],
[0.69346, 0.47404, 0.78969],
[0.70181, 0.47531, 0.78442],
[0.70998, 0.47664, 0.77914],
[0.71795, 0.47804, 0.77389],
[0.72576, 0.47953, 0.76863],
[0.73341, 0.48106, 0.76338],
[0.74091, 0.48267, 0.75813],
[0.74827, 0.48433, 0.75287],
[0.75547, 0.48607, 0.74764],
[0.76251, 0.4879, 0.7424],
[0.76941, 0.48977, 0.73717],
[0.77618, 0.49174, 0.73194],
[0.7828, 0.49375, 0.72672],
[0.78929, 0.49584, 0.72151],
[0.79562, 0.49803, 0.71631],
[0.80181, 0.50026, 0.71112],
[0.80788, 0.50259, 0.70593],
[0.8138, 0.50499, 0.70075],
[0.81959, 0.50746, 0.69559],
[0.82525, 0.51, 0.69043],
[0.83078, 0.51262, 0.68529],
[0.83616, 0.51531, 0.68015],
[0.8414, 0.5181, 0.67504],
[0.8465, 0.52095, 0.66994],
[0.85148, 0.52388, 0.66485],
[0.85633, 0.52689, 0.65978],
[0.86104, 0.52996, 0.65472],
[0.86562, 0.53311, 0.64969],
[0.87006, 0.53633, 0.64466],
[0.87436, 0.53962, 0.63966],
[0.87853, 0.54301, 0.63468],
[0.88255, 0.54646, 0.62972],
[0.88645, 0.54997, 0.62478],
[0.8902, 0.55355, 0.61987],
[0.89382, 0.55722, 0.61497],
[0.89731, 0.56093, 0.6101],
[0.90066, 0.56471, 0.60525],
[0.90387, 0.56856, 0.60043],
[0.90695, 0.57248, 0.59564],
[0.9099, 0.57645, 0.59087],
[0.91298, 0.5809, 0.58565],
[0.91561, 0.585, 0.58093],
[0.91812, 0.58914, 0.57627],
[0.92049, 0.59335, 0.57161],
[0.92273, 0.59761, 0.567],
[0.92482, 0.60191, 0.56242],
[0.92679, 0.60626, 0.55785],
[0.92862, 0.61066, 0.55333],
[0.93032, 0.61509, 0.54885],
[0.93188, 0.61957, 0.54438],
[0.93331, 0.62409, 0.53995],
[0.93459, 0.62864, 0.53556],
[0.93575, 0.63323, 0.53121],
[0.93678, 0.63785, 0.5269],
[0.93767, 0.6425, 0.52261],
[0.93843, 0.64718, 0.51836],
[0.93907, 0.65188, 0.51414],
[0.93959, 0.65661, 0.50995],
[0.93997, 0.66135, 0.50579],
[0.94023, 0.66612, 0.50168],
[0.94036, 0.67091, 0.49761],
[0.94036, 0.67571, 0.49356],
[0.94024, 0.68052, 0.48955],
[0.94, 0.68536, 0.48557],
[0.93965, 0.69019, 0.48163],
[0.93917, 0.69504, 0.47772],
[0.93858, 0.69989, 0.47384],
[0.93786, 0.70476, 0.47001],
[0.93703, 0.70963, 0.4662],
[0.93609, 0.71449, 0.46242],
[0.93503, 0.71937, 0.45868],
[0.93387, 0.72424, 0.45497],
[0.93259, 0.72911, 0.45129],
[0.93119, 0.73399, 0.44763],
[0.92963, 0.73903, 0.4439],
[0.9282, 0.74337, 0.44069],
[0.92636, 0.74862, 0.43685],
[0.92454, 0.75348, 0.4333],
[0.92264, 0.75826, 0.42987],
[0.9206, 0.76309, 0.4264],
[0.91845, 0.76791, 0.42298],
[0.91618, 0.77273, 0.41961],
[0.91378, 0.77755, 0.41627],
[0.91124, 0.78234, 0.413],
[0.90856, 0.78713, 0.40979],
[0.90573, 0.7919, 0.40665],
[0.90272, 0.79665, 0.40361],
[0.89953, 0.80139, 0.40066],
[0.89616, 0.80611, 0.3978],
[0.89258, 0.8108, 0.39508],
[0.88877, 0.81546, 0.39251],
[0.88473, 0.82009, 0.3901],
[0.88043, 0.82467, 0.38788],
[0.87587, 0.82922, 0.38586],
[0.87101, 0.8337, 0.3841],
[0.86581, 0.8381, 0.38262],
[0.86026, 0.84242, 0.38148],
[0.85434, 0.84662, 0.38068],
[0.848, 0.85069, 0.38032],
[0.84126, 0.85461, 0.38043],
[0.83406, 0.85832, 0.38108],
[0.82643, 0.8618, 0.38228],
[0.81836, 0.86499, 0.38414],
[0.80989, 0.86787, 0.38663],
[0.80109, 0.8704, 0.38976],
[0.792, 0.87253, 0.39354],
[0.78272, 0.87425, 0.39789],
[0.77333, 0.87559, 0.40273],
[0.7639, 0.87653, 0.40802],
[0.75448, 0.87712, 0.41364],
[0.74512, 0.87737, 0.41956],
[0.73584, 0.87733, 0.42568],
[0.72669, 0.87701, 0.43197],
[0.71765, 0.87645, 0.43837],
[0.70871, 0.8757, 0.44485],
[0.69988, 0.87475, 0.45137],
[0.69117, 0.87363, 0.45793],
[0.68257, 0.87237, 0.46452],
[0.67405, 0.87097, 0.47109],
[0.66562, 0.86947, 0.47765],
[0.65726, 0.86786, 0.4842],
[0.64895, 0.86615, 0.49075],
[0.64069, 0.86436, 0.49724],
[0.63248, 0.86249, 0.50372],
[0.62431, 0.86055, 0.51016],
[0.61616, 0.85854, 0.51657],
[0.60804, 0.85649, 0.52295],
[0.59991, 0.85438, 0.5293],
[0.59178, 0.85222, 0.5356],
[0.58365, 0.85001, 0.54188],
[0.57552, 0.84776, 0.54813],
[0.56737, 0.84547, 0.55433],
[0.55917, 0.84316, 0.56049],
[0.55093, 0.84081, 0.56663],
[0.54268, 0.83842, 0.57274],
[0.53439, 0.83599, 0.5788],
[0.52607, 0.83354, 0.58484],
[0.5177, 0.83105, 0.59084],
[0.50929, 0.82852, 0.59682],
[0.50086, 0.82596, 0.60275],
[0.4924, 0.82336, 0.60866],
[0.4839, 0.82073, 0.61452],
[0.47537, 0.81807, 0.62037],
[0.46679, 0.81537, 0.62616],
[0.45816, 0.81263, 0.63193],
[0.4495, 0.80987, 0.63766],
[0.44083, 0.80707, 0.64335],
[0.43213, 0.80422, 0.64901],
[0.42337, 0.80135, 0.65462],
[0.41458, 0.79844, 0.6602],
[0.40576, 0.7955, 0.66575],
[0.39688, 0.79253, 0.67125],
[0.38797, 0.78953, 0.67671],
[0.37903, 0.78649, 0.68212],
[0.37008, 0.78341, 0.6875],
[0.36108, 0.78031, 0.69284],
[0.35206, 0.77717, 0.69812],
[0.343, 0.774, 0.70336],
[0.33389, 0.7708, 0.70856],
[0.32473, 0.76757, 0.7137],
[0.31557, 0.76431, 0.7188],
[0.30635, 0.76103, 0.72385],
[0.29709, 0.75771, 0.72886],
[0.28781, 0.75437, 0.7338],
[0.27853, 0.75099, 0.7387],
[0.26922, 0.74759, 0.74354],
[0.25985, 0.74416, 0.74832],
[0.25044, 0.7407, 0.75305],
[0.24101, 0.73723, 0.75773],
[0.23156, 0.73372, 0.76235],
[0.22204, 0.7302, 0.76691],
[0.2125, 0.72665, 0.77142],
[0.20291, 0.72307, 0.77586],
[0.19332, 0.71949, 0.78024],
[0.18372, 0.71587, 0.78456],
[0.17411, 0.71223, 0.78882],
[0.16448, 0.70858, 0.79302],
[0.15477, 0.7049, 0.79715],
[0.14509, 0.70121, 0.80123],
[0.13531, 0.6975, 0.80524],
[0.12553, 0.69378, 0.8092],
[0.1157, 0.69004, 0.81308],
[0.10584, 0.68628, 0.81691],
[0.096034, 0.68251, 0.82066],
[0.086314, 0.67872, 0.82435],
[0.076594, 0.67492, 0.82798],
[0.066987, 0.6711, 0.83156],
[0.057363, 0.66727, 0.83506],
[0.047913, 0.66344, 0.8385],
[0.038783, 0.65958, 0.84189],
[0.030423, 0.65572, 0.84521],
[0.024144, 0.65184, 0.84846],
[0.019383, 0.64794, 0.85165],
[0.016091, 0.64404, 0.85478],
[0.014241, 0.64014, 0.85785],
[0.013802, 0.63622, 0.86086],
[0.014758, 0.63229, 0.86382],
[0.017308, 0.62835, 0.8667],
[0.021411, 0.62439, 0.86953],
[0.026822, 0.62043, 0.8723],
[0.033498, 0.61646, 0.87502],
[0.041669, 0.61247, 0.87768],
[0.050034, 0.60848, 0.88028],
[0.058959, 0.60447, 0.88282],
[0.067796, 0.60045, 0.88532],
[0.076491, 0.59643, 0.88775],
[0.085201, 0.5924, 0.89013],
[0.093901, 0.58834, 0.89246],
[0.10275, 0.58429, 0.89473],
[0.11172, 0.58022, 0.89692],
[0.12071, 0.57613, 0.89905],
[0.12993, 0.57203, 0.90111],
[0.13919, 0.56791, 0.90309],
[0.1487, 0.56378, 0.90498],
[0.15846, 0.55963, 0.90678],
[0.1684, 0.55546, 0.90848],
[0.17864, 0.55128, 0.91008],
[0.18914, 0.54708, 0.91155],
[0.19994, 0.54287, 0.91289],
[0.21112, 0.53864, 0.91409],
[0.22262, 0.53439, 0.91514],
[0.23456, 0.53013, 0.916],
[0.24696, 0.52586, 0.91667],
[0.2598, 0.52158, 0.91712],
[0.27304, 0.51733, 0.91735],
[0.28691, 0.51307, 0.91727],
[0.30126, 0.50885, 0.91691],
[0.31619, 0.50467, 0.91621],
[0.33157, 0.50055, 0.91515],
[0.34752, 0.49655, 0.91368],
[0.36381, 0.49269, 0.9118],
[0.38047, 0.48901, 0.90948],
]
b_cyclic_mygbm_50_90_c46_s25 = bokeh_palette('cyclic_mygbm_50_90_c46_s25',cyclic_mygbm_50_90_c46_s25)
m_cyclic_mygbm_50_90_c46_s25 = mpl_cm('cyclic_mygbm_50_90_c46_s25',cyclic_mygbm_50_90_c46_s25)
m_cyclic_mygbm_50_90_c46_s25_r = mpl_cm('cyclic_mygbm_50_90_c46_s25_r',list(reversed(cyclic_mygbm_50_90_c46_s25)))
CET_C8s = b_cyclic_mygbm_50_90_c46_s25
m_CET_C8s = m_cyclic_mygbm_50_90_c46_s25
m_CET_C8s_r = m_cyclic_mygbm_50_90_c46_s25_r
palette['CET_C8s'] = b_cyclic_mygbm_50_90_c46_s25
cm['CET_C8s'] = m_cyclic_mygbm_50_90_c46_s25
cm['CET_C8s_r'] = m_cyclic_mygbm_50_90_c46_s25_r
register_cmap('cet_CET_C8s',m_cyclic_mygbm_50_90_c46_s25)
register_cmap('cet_CET_C8s_r',m_cyclic_mygbm_50_90_c46_s25_r)



palette_n = AttrODict(sorted(palette_n.items()))
cm_n = AttrODict(sorted(cm_n.items()))
