"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service cloudfront
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudfront.html
 */
class Cloudfront extends shared_1.PolicyStatement {
    /**
     * Action provider for service cloudfront
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudfront.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cloudfront';
        this.actions = {
            "CreateCloudFrontOriginAccessIdentity": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_CreateCloudFrontOriginAccessIdentity.html",
                "description": "This action creates a new CloudFront origin access identity (POST /2019-03-26/origin-access-identity/cloudfront).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "origin-access-identity": {
                        "required": true
                    }
                }
            },
            "CreateDistribution": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_CreateDistribution.html",
                "description": "This action creates a new web distribution (POST /2019-03-26/distribution).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "distribution": {
                        "required": true
                    }
                }
            },
            "CreateDistributionWithTags": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_CreateDistributionWithTags.html",
                "description": "This action creates a new web distribution with tags (POST /2019-03-26/distribution?WithTags).",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "distribution": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateFieldLevelEncryptionConfig": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_CreateFieldLevelEncryptionConfig.html",
                "description": "This action creates a new field-level encryption configuration. (POST /2019-03-26/field-level-encryption).",
                "accessLevel": "Write"
            },
            "CreateFieldLevelEncryptionProfile": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_CreateFieldLevelEncryptionProfile.html",
                "description": "This action creates a field-level encryption profile. (POST /2019-03-26/field-level-encryption-profile).",
                "accessLevel": "Write"
            },
            "CreateInvalidation": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_CreateInvalidation.html",
                "description": "This action creates a new invalidation batch request (POST /2019-03-26/distribution/<DISTRIBUTION_ID>/invalidation).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "distribution": {
                        "required": true
                    }
                }
            },
            "CreatePublicKey": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_CreatePublicKey.html",
                "description": "This action adds a new public key to CloudFront. (POST /2019-03-26/public-key).",
                "accessLevel": "Write"
            },
            "CreateStreamingDistribution": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_CreateStreamingDistribution.html",
                "description": "This action creates a new RTMP distribution (POST /2019-03-26/streaming-distribution).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "streaming-distribution": {
                        "required": true
                    }
                }
            },
            "CreateStreamingDistributionWithTags": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_CreateStreamingDistributionWithTags.html",
                "description": "This action creates a new RTMP distribution with tags (POST /2019-03-26/streaming-distribution?WithTags).",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "streaming-distribution": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteCloudFrontOriginAccessIdentity": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_DeleteCloudFrontOriginAccessIdentity.html",
                "description": "This action deletes a CloudFront origin access identity (DELETE /2019-03-26/origin-access-identity/cloudfront/<OAI_ID>).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "origin-access-identity": {
                        "required": true
                    }
                }
            },
            "DeleteDistribution": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_DeleteDistribution.html",
                "description": "This action deletes a web distribution (DELETE /2019-03-26/distribution/<DISTRIBUTION_ID>).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "distribution": {
                        "required": true
                    }
                }
            },
            "DeleteFieldLevelEncryptionConfig": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_DeleteFieldLevelEncryptionConfig.html",
                "description": "This action deletes a field-level encryption configuration. (DELETE /2019-03-26/field-level-encryption/Id/<Id>).",
                "accessLevel": "Write"
            },
            "DeleteFieldLevelEncryptionProfile": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_DeleteFieldLevelEncryptionProfile.html",
                "description": "This action deletes a field-level encryption profile. (DELETE /2019-03-26/field-level-encryption-profile/<Id>).",
                "accessLevel": "Write"
            },
            "DeletePublicKey": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_DeletePublicKey.html",
                "description": "This action deletes a public key from CloudFront. (DELETE /2019-03-26/public-key/<Id>).",
                "accessLevel": "Write"
            },
            "DeleteStreamingDistribution": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_DeleteStreamingDistribution.html",
                "description": "This action deletes an RTMP distribution (DELETE /2019-03-26/streaming-distribution/<DISTRIBUTION_ID>).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "streaming-distribution": {
                        "required": true
                    }
                }
            },
            "GetCloudFrontOriginAccessIdentity": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetCloudFrontOriginAccessIdentity.html",
                "description": "Get the information about a CloudFront origin access identity (GET /2019-03-26/origin-access-identity/cloudfront/<OAI_ID>).",
                "accessLevel": "Read",
                "resourceTypes": {
                    "origin-access-identity": {
                        "required": true
                    }
                }
            },
            "GetCloudFrontOriginAccessIdentityConfig": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetCloudFrontOriginAccessIdentityConfig.html",
                "description": "Get the configuration information about a Cloudfront origin access identity (GET /2019-03-26/origin-access-identity/cloudfront/<OAI_ID>/config).",
                "accessLevel": "Read",
                "resourceTypes": {
                    "origin-access-identity": {
                        "required": true
                    }
                }
            },
            "GetDistribution": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetDistribution.html",
                "description": "Get the information about a web distribution (GET /2019-03-26/distribution/<DISTRIBUTION_ID>).",
                "accessLevel": "Read",
                "resourceTypes": {
                    "distribution": {
                        "required": true
                    }
                }
            },
            "GetDistributionConfig": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetDistributionConfig.html",
                "description": "Get the configuration information about a distribution (GET /2019-03-26/distribution/<DISTRIBUTION_ID>/config).",
                "accessLevel": "Read",
                "resourceTypes": {
                    "distribution": {
                        "required": true
                    }
                }
            },
            "GetFieldLevelEncryption": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetFieldLevelEncryption.html",
                "description": "Get the field-level encryption configuration information. (GET /2019-03-26/field-level-encryption/<Id>).",
                "accessLevel": "Read"
            },
            "GetFieldLevelEncryptionConfig": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetFieldLevelEncryptionConfig.html",
                "description": "Get the field-level encryption configuration information (GET /2019-03-26/field-level-encryption/<Id>/config).",
                "accessLevel": "Read"
            },
            "GetFieldLevelEncryptionProfile": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetFieldLevelEncryptionProfile.html",
                "description": "Get the field-level encryption configuration information (GET /2019-03-26/field-level-encryption/<Id>/config).",
                "accessLevel": "Read"
            },
            "GetFieldLevelEncryptionProfileConfig": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetFieldLevelEncryptionProfileConfig.html",
                "description": "Get the field-level encryption profile configuration information (GET /2019-03-26/field-level-encryption-profile/<Id>/config).",
                "accessLevel": "Read"
            },
            "GetInvalidation": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetInvalidation.html",
                "description": "Get the information about an invalidation (GET /2019-03-26/distribution/<DISTRIBUTION_ID>/invalidation/<INVALIDATION_ID>).",
                "accessLevel": "Read",
                "resourceTypes": {
                    "distribution": {
                        "required": true
                    }
                }
            },
            "GetPublicKey": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetPublicKey.html",
                "description": "Get the public key information (GET /2019-03-26/public-key/<Id>).",
                "accessLevel": "Read"
            },
            "GetPublicKeyConfig": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetPublicKeyConfig.html",
                "description": "Get the public key configuration information (GET /2019-03-26/public-key/<Id>/config).",
                "accessLevel": "Read"
            },
            "GetStreamingDistribution": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetStreamingDistribution.html",
                "description": "Get the information about an RTMP distribution (GET /2019-03-26/streaming-distribution/<DISTRIBUTION_ID>).",
                "accessLevel": "Read",
                "resourceTypes": {
                    "streaming-distribution": {
                        "required": true
                    }
                }
            },
            "GetStreamingDistributionConfig": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetStreamingDistributionConfig.html",
                "description": "Get the configuration information about a streaming distribution (GET /2019-03-26/streaming-distribution/<DISTRIBUTION_ID>/config).",
                "accessLevel": "Read",
                "resourceTypes": {
                    "streaming-distribution": {
                        "required": true
                    }
                }
            },
            "ListCloudFrontOriginAccessIdentities": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_ListCloudFrontOriginAccessIdentities.html",
                "description": "List your CloudFront origin access identities (GET /2019-03-26/origin-access-identity/cloudfront?Marker=<MARKER>&MaxItems=<MAX_ITEMS>).",
                "accessLevel": "List"
            },
            "ListDistributions": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_ListDistributions.html",
                "description": "List the distributions associated with your AWS account (GET /2019-03-26/distribution?Marker=<MARKER>&MaxItems=<MAX_ITEMS>).",
                "accessLevel": "List"
            },
            "ListDistributionsByLambdaFunction": {
                "url": "",
                "description": "List the distributions associated a Lambda function.",
                "accessLevel": "List"
            },
            "ListDistributionsByWebACLId": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_ListDistributionsByWebACLId.html",
                "description": "List the distributions associated with your AWS account with given AWS WAF web ACL (GET /2019-03-26/distributionsByWebACLId/<WEB_ACL_ID>?Marker=<MARKER>&MaxItems=<MAX_ITEMS>).",
                "accessLevel": "List"
            },
            "ListFieldLevelEncryptionConfigs": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_ListFieldLevelEncryptionConfigs.html",
                "description": "List all field-level encryption configurations that have been created in CloudFront for this account (GET /2019-03-26/field-level-encryption?Marker=<MARKER>&MaxItems=<MAX_ITEMS>).",
                "accessLevel": "List"
            },
            "ListFieldLevelEncryptionProfiles": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_ListFieldLevelEncryptionProfiles.html",
                "description": "List all field-level encryption profiles that have been created in CloudFront for this account. (GET /2019-03-26/field-level-encryption-profile?Marker=<MARKER>&MaxItems=<MAX_ITEMS>).",
                "accessLevel": "List"
            },
            "ListInvalidations": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_ListInvalidations.html",
                "description": "List your invalidation batches (GET /2019-03-26/distribution/<DISTRIBUTION_ID>/invalidation?Marker=<MARKER>&MaxItems=<MAX_ITEMS>).",
                "accessLevel": "List",
                "resourceTypes": {
                    "distribution": {
                        "required": true
                    }
                }
            },
            "ListPublicKeys": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_ListPublicKeys.html",
                "description": "List all public keys that have been added to CloudFront for this account. (GET /2019-03-26/public-key?Marker=<MARKER>&MaxItems=<MAX_ITEMS>).",
                "accessLevel": "List"
            },
            "ListStreamingDistributions": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_ListStreamingDistributions.html",
                "description": "List your RTMP distributions (GET /2019-03-26/streaming-distribution?Marker=<MARKER>&MaxItems=<MAX_ITEMS>).",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_ListTagsForResource.html",
                "description": "List tags for a CloudFront resource (GET /2019-03-26/tagging?Resource=<RESOURCE>).",
                "accessLevel": "Read",
                "resourceTypes": {
                    "distribution": {
                        "required": false
                    },
                    "streaming-distribution": {
                        "required": false
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_TagResource.html",
                "description": "Add tags to a CloudFront resource (POST /2019-03-26/tagging?Operation=Tag?Resource=<RESOURCE>).",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "distribution": {
                        "required": false
                    },
                    "streaming-distribution": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_UntagResource.html",
                "description": "Remove tags from a CloudFront resource (POST /2019-03-26/tagging?Operation=Untag?Resource=<RESOURCE>).",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "distribution": {
                        "required": false
                    },
                    "streaming-distribution": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateCloudFrontOriginAccessIdentity": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_UpdateCloudFrontOriginAccessIdentity.html",
                "description": "This action sets the configuration for a CloudFront origin access identity (PUT /2019-03-26/origin-access-identity/cloudfront/<OAI_ID>/config).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "origin-access-identity": {
                        "required": true
                    }
                }
            },
            "UpdateDistribution": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_UpdateDistribution.html",
                "description": "This action updates the configuration for a web distribution (PUT /2019-03-26/distribution/<DISTRIBUTION_ID>/config).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "distribution": {
                        "required": true
                    }
                }
            },
            "UpdateFieldLevelEncryptionConfig": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_UpdateFieldLevelEncryptionConfig.html",
                "description": "This action updates a field-level encryption configuration. (PUT /2019-03-26/field-level-encryption/<Id>/config).",
                "accessLevel": "Write"
            },
            "UpdateFieldLevelEncryptionProfile": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_UpdateFieldLevelEncryptionProfile.html",
                "description": "This action updates a field-level encryption profile. (PUT /2019-03-26/field-level-encryption-profile/<Id>/config).",
                "accessLevel": "Write"
            },
            "UpdatePublicKey": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_UpdatePublicKey.html",
                "description": "This action updates public key information. (PUT /2019-03-26/public-key/<Id>/config).",
                "accessLevel": "Write"
            },
            "UpdateStreamingDistribution": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_UpdateStreamingDistribution.html",
                "description": "This action updates the configuration for an RTMP distribution (PUT /2019-03-26/streaming-distribution/<DISTRIBUTION_ID>/config).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "streaming-distribution": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "distribution": {
                "name": "distribution",
                "url": "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/AmazonCloudFront/latest/DeveloperGuide/distribution-working-with.html",
                "arn": "arn:${Partition}:cloudfront::${Account}:distribution/${DistributionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "streaming-distribution": {
                "name": "streaming-distribution",
                "url": "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/AmazonCloudFront/latest/DeveloperGuide/distribution-working-with.html",
                "arn": "arn:${Partition}:cloudfront::${Account}:streaming-distribution/${DistributionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "origin-access-identity": {
                "name": "origin-access-identity",
                "url": "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html#private-content-restricting-access-to-s3-overview",
                "arn": "arn:${Partition}:cloudfront::${Account}:origin-access-identity/${Id}",
                "conditionKeys": []
            }
        };
    }
    /**
     * This action creates a new CloudFront origin access identity (POST /2019-03-26/origin-access-identity/cloudfront).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_CreateCloudFrontOriginAccessIdentity.html
     */
    createCloudFrontOriginAccessIdentity() {
        this.add('cloudfront:CreateCloudFrontOriginAccessIdentity');
        return this;
    }
    /**
     * This action creates a new web distribution (POST /2019-03-26/distribution).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_CreateDistribution.html
     */
    createDistribution() {
        this.add('cloudfront:CreateDistribution');
        return this;
    }
    /**
     * This action creates a new web distribution with tags (POST /2019-03-26/distribution?WithTags).
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_CreateDistributionWithTags.html
     */
    createDistributionWithTags() {
        this.add('cloudfront:CreateDistributionWithTags');
        return this;
    }
    /**
     * This action creates a new field-level encryption configuration. (POST /2019-03-26/field-level-encryption).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_CreateFieldLevelEncryptionConfig.html
     */
    createFieldLevelEncryptionConfig() {
        this.add('cloudfront:CreateFieldLevelEncryptionConfig');
        return this;
    }
    /**
     * This action creates a field-level encryption profile. (POST /2019-03-26/field-level-encryption-profile).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_CreateFieldLevelEncryptionProfile.html
     */
    createFieldLevelEncryptionProfile() {
        this.add('cloudfront:CreateFieldLevelEncryptionProfile');
        return this;
    }
    /**
     * This action creates a new invalidation batch request (POST /2019-03-26/distribution/<DISTRIBUTION_ID>/invalidation).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_CreateInvalidation.html
     */
    createInvalidation() {
        this.add('cloudfront:CreateInvalidation');
        return this;
    }
    /**
     * This action adds a new public key to CloudFront. (POST /2019-03-26/public-key).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_CreatePublicKey.html
     */
    createPublicKey() {
        this.add('cloudfront:CreatePublicKey');
        return this;
    }
    /**
     * This action creates a new RTMP distribution (POST /2019-03-26/streaming-distribution).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_CreateStreamingDistribution.html
     */
    createStreamingDistribution() {
        this.add('cloudfront:CreateStreamingDistribution');
        return this;
    }
    /**
     * This action creates a new RTMP distribution with tags (POST /2019-03-26/streaming-distribution?WithTags).
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_CreateStreamingDistributionWithTags.html
     */
    createStreamingDistributionWithTags() {
        this.add('cloudfront:CreateStreamingDistributionWithTags');
        return this;
    }
    /**
     * This action deletes a CloudFront origin access identity (DELETE /2019-03-26/origin-access-identity/cloudfront/<OAI_ID>).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_DeleteCloudFrontOriginAccessIdentity.html
     */
    deleteCloudFrontOriginAccessIdentity() {
        this.add('cloudfront:DeleteCloudFrontOriginAccessIdentity');
        return this;
    }
    /**
     * This action deletes a web distribution (DELETE /2019-03-26/distribution/<DISTRIBUTION_ID>).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_DeleteDistribution.html
     */
    deleteDistribution() {
        this.add('cloudfront:DeleteDistribution');
        return this;
    }
    /**
     * This action deletes a field-level encryption configuration. (DELETE /2019-03-26/field-level-encryption/Id/<Id>).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_DeleteFieldLevelEncryptionConfig.html
     */
    deleteFieldLevelEncryptionConfig() {
        this.add('cloudfront:DeleteFieldLevelEncryptionConfig');
        return this;
    }
    /**
     * This action deletes a field-level encryption profile. (DELETE /2019-03-26/field-level-encryption-profile/<Id>).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_DeleteFieldLevelEncryptionProfile.html
     */
    deleteFieldLevelEncryptionProfile() {
        this.add('cloudfront:DeleteFieldLevelEncryptionProfile');
        return this;
    }
    /**
     * This action deletes a public key from CloudFront. (DELETE /2019-03-26/public-key/<Id>).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_DeletePublicKey.html
     */
    deletePublicKey() {
        this.add('cloudfront:DeletePublicKey');
        return this;
    }
    /**
     * This action deletes an RTMP distribution (DELETE /2019-03-26/streaming-distribution/<DISTRIBUTION_ID>).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_DeleteStreamingDistribution.html
     */
    deleteStreamingDistribution() {
        this.add('cloudfront:DeleteStreamingDistribution');
        return this;
    }
    /**
     * Get the information about a CloudFront origin access identity (GET /2019-03-26/origin-access-identity/cloudfront/<OAI_ID>).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetCloudFrontOriginAccessIdentity.html
     */
    getCloudFrontOriginAccessIdentity() {
        this.add('cloudfront:GetCloudFrontOriginAccessIdentity');
        return this;
    }
    /**
     * Get the configuration information about a Cloudfront origin access identity (GET /2019-03-26/origin-access-identity/cloudfront/<OAI_ID>/config).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetCloudFrontOriginAccessIdentityConfig.html
     */
    getCloudFrontOriginAccessIdentityConfig() {
        this.add('cloudfront:GetCloudFrontOriginAccessIdentityConfig');
        return this;
    }
    /**
     * Get the information about a web distribution (GET /2019-03-26/distribution/<DISTRIBUTION_ID>).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetDistribution.html
     */
    getDistribution() {
        this.add('cloudfront:GetDistribution');
        return this;
    }
    /**
     * Get the configuration information about a distribution (GET /2019-03-26/distribution/<DISTRIBUTION_ID>/config).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetDistributionConfig.html
     */
    getDistributionConfig() {
        this.add('cloudfront:GetDistributionConfig');
        return this;
    }
    /**
     * Get the field-level encryption configuration information. (GET /2019-03-26/field-level-encryption/<Id>).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetFieldLevelEncryption.html
     */
    getFieldLevelEncryption() {
        this.add('cloudfront:GetFieldLevelEncryption');
        return this;
    }
    /**
     * Get the field-level encryption configuration information (GET /2019-03-26/field-level-encryption/<Id>/config).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetFieldLevelEncryptionConfig.html
     */
    getFieldLevelEncryptionConfig() {
        this.add('cloudfront:GetFieldLevelEncryptionConfig');
        return this;
    }
    /**
     * Get the field-level encryption configuration information (GET /2019-03-26/field-level-encryption/<Id>/config).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetFieldLevelEncryptionProfile.html
     */
    getFieldLevelEncryptionProfile() {
        this.add('cloudfront:GetFieldLevelEncryptionProfile');
        return this;
    }
    /**
     * Get the field-level encryption profile configuration information (GET /2019-03-26/field-level-encryption-profile/<Id>/config).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetFieldLevelEncryptionProfileConfig.html
     */
    getFieldLevelEncryptionProfileConfig() {
        this.add('cloudfront:GetFieldLevelEncryptionProfileConfig');
        return this;
    }
    /**
     * Get the information about an invalidation (GET /2019-03-26/distribution/<DISTRIBUTION_ID>/invalidation/<INVALIDATION_ID>).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetInvalidation.html
     */
    getInvalidation() {
        this.add('cloudfront:GetInvalidation');
        return this;
    }
    /**
     * Get the public key information (GET /2019-03-26/public-key/<Id>).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetPublicKey.html
     */
    getPublicKey() {
        this.add('cloudfront:GetPublicKey');
        return this;
    }
    /**
     * Get the public key configuration information (GET /2019-03-26/public-key/<Id>/config).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetPublicKeyConfig.html
     */
    getPublicKeyConfig() {
        this.add('cloudfront:GetPublicKeyConfig');
        return this;
    }
    /**
     * Get the information about an RTMP distribution (GET /2019-03-26/streaming-distribution/<DISTRIBUTION_ID>).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetStreamingDistribution.html
     */
    getStreamingDistribution() {
        this.add('cloudfront:GetStreamingDistribution');
        return this;
    }
    /**
     * Get the configuration information about a streaming distribution (GET /2019-03-26/streaming-distribution/<DISTRIBUTION_ID>/config).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_GetStreamingDistributionConfig.html
     */
    getStreamingDistributionConfig() {
        this.add('cloudfront:GetStreamingDistributionConfig');
        return this;
    }
    /**
     * List your CloudFront origin access identities (GET /2019-03-26/origin-access-identity/cloudfront?Marker=<MARKER>&MaxItems=<MAX_ITEMS>).
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_ListCloudFrontOriginAccessIdentities.html
     */
    listCloudFrontOriginAccessIdentities() {
        this.add('cloudfront:ListCloudFrontOriginAccessIdentities');
        return this;
    }
    /**
     * List the distributions associated with your AWS account (GET /2019-03-26/distribution?Marker=<MARKER>&MaxItems=<MAX_ITEMS>).
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_ListDistributions.html
     */
    listDistributions() {
        this.add('cloudfront:ListDistributions');
        return this;
    }
    /**
     * List the distributions associated a Lambda function.
     *
     * Access Level: List
     */
    listDistributionsByLambdaFunction() {
        this.add('cloudfront:ListDistributionsByLambdaFunction');
        return this;
    }
    /**
     * List the distributions associated with your AWS account with given AWS WAF web ACL (GET /2019-03-26/distributionsByWebACLId/<WEB_ACL_ID>?Marker=<MARKER>&MaxItems=<MAX_ITEMS>).
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_ListDistributionsByWebACLId.html
     */
    listDistributionsByWebACLId() {
        this.add('cloudfront:ListDistributionsByWebACLId');
        return this;
    }
    /**
     * List all field-level encryption configurations that have been created in CloudFront for this account (GET /2019-03-26/field-level-encryption?Marker=<MARKER>&MaxItems=<MAX_ITEMS>).
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_ListFieldLevelEncryptionConfigs.html
     */
    listFieldLevelEncryptionConfigs() {
        this.add('cloudfront:ListFieldLevelEncryptionConfigs');
        return this;
    }
    /**
     * List all field-level encryption profiles that have been created in CloudFront for this account. (GET /2019-03-26/field-level-encryption-profile?Marker=<MARKER>&MaxItems=<MAX_ITEMS>).
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_ListFieldLevelEncryptionProfiles.html
     */
    listFieldLevelEncryptionProfiles() {
        this.add('cloudfront:ListFieldLevelEncryptionProfiles');
        return this;
    }
    /**
     * List your invalidation batches (GET /2019-03-26/distribution/<DISTRIBUTION_ID>/invalidation?Marker=<MARKER>&MaxItems=<MAX_ITEMS>).
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_ListInvalidations.html
     */
    listInvalidations() {
        this.add('cloudfront:ListInvalidations');
        return this;
    }
    /**
     * List all public keys that have been added to CloudFront for this account. (GET /2019-03-26/public-key?Marker=<MARKER>&MaxItems=<MAX_ITEMS>).
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_ListPublicKeys.html
     */
    listPublicKeys() {
        this.add('cloudfront:ListPublicKeys');
        return this;
    }
    /**
     * List your RTMP distributions (GET /2019-03-26/streaming-distribution?Marker=<MARKER>&MaxItems=<MAX_ITEMS>).
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_ListStreamingDistributions.html
     */
    listStreamingDistributions() {
        this.add('cloudfront:ListStreamingDistributions');
        return this;
    }
    /**
     * List tags for a CloudFront resource (GET /2019-03-26/tagging?Resource=<RESOURCE>).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('cloudfront:ListTagsForResource');
        return this;
    }
    /**
     * Add tags to a CloudFront resource (POST /2019-03-26/tagging?Operation=Tag?Resource=<RESOURCE>).
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_TagResource.html
     */
    tagResource() {
        this.add('cloudfront:TagResource');
        return this;
    }
    /**
     * Remove tags from a CloudFront resource (POST /2019-03-26/tagging?Operation=Untag?Resource=<RESOURCE>).
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_UntagResource.html
     */
    untagResource() {
        this.add('cloudfront:UntagResource');
        return this;
    }
    /**
     * This action sets the configuration for a CloudFront origin access identity (PUT /2019-03-26/origin-access-identity/cloudfront/<OAI_ID>/config).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_UpdateCloudFrontOriginAccessIdentity.html
     */
    updateCloudFrontOriginAccessIdentity() {
        this.add('cloudfront:UpdateCloudFrontOriginAccessIdentity');
        return this;
    }
    /**
     * This action updates the configuration for a web distribution (PUT /2019-03-26/distribution/<DISTRIBUTION_ID>/config).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_UpdateDistribution.html
     */
    updateDistribution() {
        this.add('cloudfront:UpdateDistribution');
        return this;
    }
    /**
     * This action updates a field-level encryption configuration. (PUT /2019-03-26/field-level-encryption/<Id>/config).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_UpdateFieldLevelEncryptionConfig.html
     */
    updateFieldLevelEncryptionConfig() {
        this.add('cloudfront:UpdateFieldLevelEncryptionConfig');
        return this;
    }
    /**
     * This action updates a field-level encryption profile. (PUT /2019-03-26/field-level-encryption-profile/<Id>/config).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_UpdateFieldLevelEncryptionProfile.html
     */
    updateFieldLevelEncryptionProfile() {
        this.add('cloudfront:UpdateFieldLevelEncryptionProfile');
        return this;
    }
    /**
     * This action updates public key information. (PUT /2019-03-26/public-key/<Id>/config).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_UpdatePublicKey.html
     */
    updatePublicKey() {
        this.add('cloudfront:UpdatePublicKey');
        return this;
    }
    /**
     * This action updates the configuration for an RTMP distribution (PUT /2019-03-26/streaming-distribution/<DISTRIBUTION_ID>/config).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_Operations.html/API_UpdateStreamingDistribution.html
     */
    updateStreamingDistribution() {
        this.add('cloudfront:UpdateStreamingDistribution');
        return this;
    }
    /**
     * Adds a resource of type distribution to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/AmazonCloudFront/latest/DeveloperGuide/distribution-working-with.html
     *
     * @param distributionId - Identifier for the distributionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDistribution(distributionId, account, partition) {
        var arn = 'arn:${Partition}:cloudfront::${Account}:distribution/${DistributionId}';
        arn = arn.replace('${DistributionId}', distributionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type streaming-distribution to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/AmazonCloudFront/latest/DeveloperGuide/distribution-working-with.html
     *
     * @param distributionId - Identifier for the distributionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onStreamingDistribution(distributionId, account, partition) {
        var arn = 'arn:${Partition}:cloudfront::${Account}:streaming-distribution/${DistributionId}';
        arn = arn.replace('${DistributionId}', distributionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type origin-access-identity to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html#private-content-restricting-access-to-s3-overview
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOriginAccessIdentity(id, account, partition) {
        var arn = 'arn:${Partition}:cloudfront::${Account}:origin-access-identity/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Cloudfront = Cloudfront;
//# sourceMappingURL=data:application/json;base64,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