"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service cognito-identity
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitoidentity.html
 */
class CognitoIdentity extends shared_1.PolicyStatement {
    /**
     * Action provider for service cognito-identity
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitoidentity.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cognito-identity';
        this.actions = {
            "CreateIdentityPool": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_CreateIdentityPool.html",
                "description": "Creates a new identity pool.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "DeleteIdentities": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DeleteIdentities.html",
                "description": "Deletes identities from an identity pool. You can specify a list of 1-60 identities that you want to delete.",
                "accessLevel": "Write"
            },
            "DeleteIdentityPool": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DeleteIdentityPool.html",
                "description": "Deletes a user pool. Once a pool is deleted, users will not be able to authenticate with the pool.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "DescribeIdentity": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DescribeIdentity.html",
                "description": "Returns metadata related to the given identity, including when the identity was created and any associated linked logins.",
                "accessLevel": "Read"
            },
            "DescribeIdentityPool": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DescribeIdentityPool.html",
                "description": "Gets details about a particular identity pool, including the pool name, ID description, creation date, and current number of users.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "GetCredentialsForIdentity": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetCredentialsForIdentity.html",
                "description": "Returns credentials for the provided identity ID.",
                "accessLevel": "Read"
            },
            "GetId": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetId.html",
                "description": "Generates (or retrieves) a Cognito ID. Supplying multiple logins will create an implicit linked account.",
                "accessLevel": "Write"
            },
            "GetIdentityPoolRoles": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetIdentityPoolRoles.html",
                "description": "Gets the roles for an identity pool.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "GetOpenIdToken": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetOpenIdToken.html",
                "description": "Gets an OpenID token, using a known Cognito ID.",
                "accessLevel": "Read"
            },
            "GetOpenIdTokenForDeveloperIdentity": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetOpenIdTokenForDeveloperIdentity.html",
                "description": "Registers (or retrieves) a Cognito IdentityId and an OpenID Connect token for a user authenticated by your backend authentication process.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "ListIdentities": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_ListIdentities.html",
                "description": "Lists the identities in a pool.",
                "accessLevel": "List",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "ListIdentityPools": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_ListIdentityPools.html",
                "description": "Lists all of the Cognito identity pools registered for your account.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_ListTagsForResource.html",
                "description": "Lists the tags that are assigned to an Amazon Cognito identity pool.",
                "accessLevel": "List",
                "resourceTypes": {
                    "identitypool": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "LookupDeveloperIdentity": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_LookupDeveloperIdentity.html",
                "description": "Retrieves the IdentityID associated with a DeveloperUserIdentifier or the list of DeveloperUserIdentifiers associated with an IdentityId for an existing identity.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "MergeDeveloperIdentities": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_MergeDeveloperIdentities.html",
                "description": "Merges two users having different IdentityIds, existing in the same identity pool, and identified by the same developer provider.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "SetIdentityPoolRoles": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_SetIdentityPoolRoles.html",
                "description": "Sets the roles for an identity pool. These roles are used when making calls to GetCredentialsForIdentity action.",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_TagResource.html",
                "description": "Assigns a set of tags to an Amazon Cognito identity pool.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "identitypool": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "UnlinkDeveloperIdentity": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UnlinkDeveloperIdentity.html",
                "description": "Unlinks a DeveloperUserIdentifier from an existing identity.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "UnlinkIdentity": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UnlinkIdentity.html",
                "description": "Unlinks a federated identity from an existing account.",
                "accessLevel": "Write"
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UntagResource.html",
                "description": "Removes the specified tags from an Amazon Cognito identity pool.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "identitypool": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "UpdateIdentityPool": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UpdateIdentityPool.html",
                "description": "Updates a user pool.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "identitypool": {
                "name": "identitypool",
                "url": "https://docs.aws.amazon.com/cognito/latest/developerguide/identity-pools.html",
                "arn": "arn:${Partition}:cognito-identity:${Region}:${Account}:identitypool/${IdentityPoolId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates a new identity pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_CreateIdentityPool.html
     */
    createIdentityPool() {
        this.add('cognito-identity:CreateIdentityPool');
        return this;
    }
    /**
     * Deletes identities from an identity pool. You can specify a list of 1-60 identities that you want to delete.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DeleteIdentities.html
     */
    deleteIdentities() {
        this.add('cognito-identity:DeleteIdentities');
        return this;
    }
    /**
     * Deletes a user pool. Once a pool is deleted, users will not be able to authenticate with the pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DeleteIdentityPool.html
     */
    deleteIdentityPool() {
        this.add('cognito-identity:DeleteIdentityPool');
        return this;
    }
    /**
     * Returns metadata related to the given identity, including when the identity was created and any associated linked logins.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DescribeIdentity.html
     */
    describeIdentity() {
        this.add('cognito-identity:DescribeIdentity');
        return this;
    }
    /**
     * Gets details about a particular identity pool, including the pool name, ID description, creation date, and current number of users.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DescribeIdentityPool.html
     */
    describeIdentityPool() {
        this.add('cognito-identity:DescribeIdentityPool');
        return this;
    }
    /**
     * Returns credentials for the provided identity ID.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetCredentialsForIdentity.html
     */
    getCredentialsForIdentity() {
        this.add('cognito-identity:GetCredentialsForIdentity');
        return this;
    }
    /**
     * Generates (or retrieves) a Cognito ID. Supplying multiple logins will create an implicit linked account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetId.html
     */
    getId() {
        this.add('cognito-identity:GetId');
        return this;
    }
    /**
     * Gets the roles for an identity pool.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetIdentityPoolRoles.html
     */
    getIdentityPoolRoles() {
        this.add('cognito-identity:GetIdentityPoolRoles');
        return this;
    }
    /**
     * Gets an OpenID token, using a known Cognito ID.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetOpenIdToken.html
     */
    getOpenIdToken() {
        this.add('cognito-identity:GetOpenIdToken');
        return this;
    }
    /**
     * Registers (or retrieves) a Cognito IdentityId and an OpenID Connect token for a user authenticated by your backend authentication process.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetOpenIdTokenForDeveloperIdentity.html
     */
    getOpenIdTokenForDeveloperIdentity() {
        this.add('cognito-identity:GetOpenIdTokenForDeveloperIdentity');
        return this;
    }
    /**
     * Lists the identities in a pool.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_ListIdentities.html
     */
    listIdentities() {
        this.add('cognito-identity:ListIdentities');
        return this;
    }
    /**
     * Lists all of the Cognito identity pools registered for your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_ListIdentityPools.html
     */
    listIdentityPools() {
        this.add('cognito-identity:ListIdentityPools');
        return this;
    }
    /**
     * Lists the tags that are assigned to an Amazon Cognito identity pool.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('cognito-identity:ListTagsForResource');
        return this;
    }
    /**
     * Retrieves the IdentityID associated with a DeveloperUserIdentifier or the list of DeveloperUserIdentifiers associated with an IdentityId for an existing identity.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_LookupDeveloperIdentity.html
     */
    lookupDeveloperIdentity() {
        this.add('cognito-identity:LookupDeveloperIdentity');
        return this;
    }
    /**
     * Merges two users having different IdentityIds, existing in the same identity pool, and identified by the same developer provider.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_MergeDeveloperIdentities.html
     */
    mergeDeveloperIdentities() {
        this.add('cognito-identity:MergeDeveloperIdentities');
        return this;
    }
    /**
     * Sets the roles for an identity pool. These roles are used when making calls to GetCredentialsForIdentity action.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_SetIdentityPoolRoles.html
     */
    setIdentityPoolRoles() {
        this.add('cognito-identity:SetIdentityPoolRoles');
        return this;
    }
    /**
     * Assigns a set of tags to an Amazon Cognito identity pool.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('cognito-identity:TagResource');
        return this;
    }
    /**
     * Unlinks a DeveloperUserIdentifier from an existing identity.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UnlinkDeveloperIdentity.html
     */
    unlinkDeveloperIdentity() {
        this.add('cognito-identity:UnlinkDeveloperIdentity');
        return this;
    }
    /**
     * Unlinks a federated identity from an existing account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UnlinkIdentity.html
     */
    unlinkIdentity() {
        this.add('cognito-identity:UnlinkIdentity');
        return this;
    }
    /**
     * Removes the specified tags from an Amazon Cognito identity pool.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('cognito-identity:UntagResource');
        return this;
    }
    /**
     * Updates a user pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UpdateIdentityPool.html
     */
    updateIdentityPool() {
        this.add('cognito-identity:UpdateIdentityPool');
        return this;
    }
    /**
     * Adds a resource of type identitypool to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/identity-pools.html
     *
     * @param identityPoolId - Identifier for the identityPoolId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onIdentitypool(identityPoolId, account, region, partition) {
        var arn = 'arn:${Partition}:cognito-identity:${Region}:${Account}:identitypool/${IdentityPoolId}';
        arn = arn.replace('${IdentityPoolId}', identityPoolId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.CognitoIdentity = CognitoIdentity;
//# sourceMappingURL=data:application/json;base64,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