"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service detective
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazondetective.html
 */
class Detective extends shared_1.PolicyStatement {
    /**
     * Action provider for service detective
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazondetective.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'detective';
        this.actions = {
            "AcceptInvitation": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_AcceptInvitation.html",
                "description": "Grants permission to accept an invitation to become a member of a behavior graph",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "CreateGraph": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_CreateGraph.html",
                "description": "Grants permission to create a behavior graph and begin to aggregate security information",
                "accessLevel": "Write"
            },
            "CreateMembers": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_CreateMembers.html",
                "description": "Grants permission to request the membership of one or more accounts in a behavior graph managed by this account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "DeleteGraph": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_DeleteGraph.html",
                "description": "Grants permission to delete a behavior graph and stop aggregating security information",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "DeleteMembers": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_DeleteMembers.html",
                "description": "Grants permission to remove member accounts from a behavior graph managed by this account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "DisassociateMembership": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_DisassociateMembership.html",
                "description": "Grants permission to remove the association of this account with a behavior graph",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "GetFreeTrialEligibility": {
                "url": "",
                "description": "Grants permission to retrieve a behavior graph's eligibility for a free trial period",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "GetGraphIngestState": {
                "url": "",
                "description": "Grants permission to retrieve the data ingestion state of a behavior graph",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "GetMembers": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_GetMembers.html",
                "description": "Grants permission to retrieve details on specified members of a behavior graph",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "GetPricingInformation": {
                "url": "",
                "description": "Grants permission to retrieve information about Amazon Detective's pricing",
                "accessLevel": "Read"
            },
            "GetUsageInformation": {
                "url": "",
                "description": "Grants permission to list usage information of a behavior graph",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "ListGraphs": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_ListGraphs.html",
                "description": "Grants permission to list behavior graphs managed by this account",
                "accessLevel": "List"
            },
            "ListInvitations": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_ListInvitations.html",
                "description": "Grants permission to retrieve details on the behavior graphs to which this account has been invited to join",
                "accessLevel": "List"
            },
            "ListMembers": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_ListMembers.html",
                "description": "Grants permission to retrieve details on all members of a behavior graph",
                "accessLevel": "List",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "RejectInvitation": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_RejectInvitation.html",
                "description": "Grants permission to reject an invitation to become a member of a behavior graph",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "SearchGraph": {
                "url": "",
                "description": "Grants permission to search the data stored in a behavior graph",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "StartMonitoringMember": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_StartMonitoringMember.html",
                "description": "Grants permission to start data ingest for a member account that has a status of ACCEPTED_BUT_DISABLED.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "Graph": {
                "name": "Graph",
                "url": "https://docs.aws.amazon.com/detective/latest/adminguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources",
                "arn": "arn:${Partition}:detective:${Region}:${Account}:graph:${ResourceId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to accept an invitation to become a member of a behavior graph
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_AcceptInvitation.html
     */
    acceptInvitation() {
        this.add('detective:AcceptInvitation');
        return this;
    }
    /**
     * Grants permission to create a behavior graph and begin to aggregate security information
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_CreateGraph.html
     */
    createGraph() {
        this.add('detective:CreateGraph');
        return this;
    }
    /**
     * Grants permission to request the membership of one or more accounts in a behavior graph managed by this account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_CreateMembers.html
     */
    createMembers() {
        this.add('detective:CreateMembers');
        return this;
    }
    /**
     * Grants permission to delete a behavior graph and stop aggregating security information
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_DeleteGraph.html
     */
    deleteGraph() {
        this.add('detective:DeleteGraph');
        return this;
    }
    /**
     * Grants permission to remove member accounts from a behavior graph managed by this account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_DeleteMembers.html
     */
    deleteMembers() {
        this.add('detective:DeleteMembers');
        return this;
    }
    /**
     * Grants permission to remove the association of this account with a behavior graph
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_DisassociateMembership.html
     */
    disassociateMembership() {
        this.add('detective:DisassociateMembership');
        return this;
    }
    /**
     * Grants permission to retrieve a behavior graph's eligibility for a free trial period
     *
     * Access Level: Read
     */
    getFreeTrialEligibility() {
        this.add('detective:GetFreeTrialEligibility');
        return this;
    }
    /**
     * Grants permission to retrieve the data ingestion state of a behavior graph
     *
     * Access Level: Read
     */
    getGraphIngestState() {
        this.add('detective:GetGraphIngestState');
        return this;
    }
    /**
     * Grants permission to retrieve details on specified members of a behavior graph
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_GetMembers.html
     */
    getMembers() {
        this.add('detective:GetMembers');
        return this;
    }
    /**
     * Grants permission to retrieve information about Amazon Detective's pricing
     *
     * Access Level: Read
     */
    getPricingInformation() {
        this.add('detective:GetPricingInformation');
        return this;
    }
    /**
     * Grants permission to list usage information of a behavior graph
     *
     * Access Level: Read
     */
    getUsageInformation() {
        this.add('detective:GetUsageInformation');
        return this;
    }
    /**
     * Grants permission to list behavior graphs managed by this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_ListGraphs.html
     */
    listGraphs() {
        this.add('detective:ListGraphs');
        return this;
    }
    /**
     * Grants permission to retrieve details on the behavior graphs to which this account has been invited to join
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_ListInvitations.html
     */
    listInvitations() {
        this.add('detective:ListInvitations');
        return this;
    }
    /**
     * Grants permission to retrieve details on all members of a behavior graph
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_ListMembers.html
     */
    listMembers() {
        this.add('detective:ListMembers');
        return this;
    }
    /**
     * Grants permission to reject an invitation to become a member of a behavior graph
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_RejectInvitation.html
     */
    rejectInvitation() {
        this.add('detective:RejectInvitation');
        return this;
    }
    /**
     * Grants permission to search the data stored in a behavior graph
     *
     * Access Level: Read
     */
    searchGraph() {
        this.add('detective:SearchGraph');
        return this;
    }
    /**
     * Grants permission to start data ingest for a member account that has a status of ACCEPTED_BUT_DISABLED.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_StartMonitoringMember.html
     */
    startMonitoringMember() {
        this.add('detective:StartMonitoringMember');
        return this;
    }
    /**
     * Adds a resource of type Graph to the statement
     *
     * https://docs.aws.amazon.com/detective/latest/adminguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGraph(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:detective:${Region}:${Account}:graph:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Detective = Detective;
//# sourceMappingURL=data:application/json;base64,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