"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service eks
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticcontainerserviceforkubernetes.html
 */
class Eks extends shared_1.PolicyStatement {
    /**
     * Action provider for service eks
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticcontainerserviceforkubernetes.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'eks';
        this.actions = {
            "CreateCluster": {
                "url": "https://docs.aws.amazon.com/eks/latest/APIReference/API_CreateCluster.html",
                "description": "Creates an Amazon EKS cluster.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateFargateProfile": {
                "url": "https://docs.aws.amazon.com/eks/latest/APIReference/API_CreateFargateProfile.html",
                "description": "Creates an AWS Fargate profile.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateNodegroup": {
                "url": "https://docs.aws.amazon.com/eks/latest/APIReference/API_CreateNodegroup.html",
                "description": "Creates an Amazon EKS Nodegroup.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteCluster": {
                "url": "https://docs.aws.amazon.com/eks/latest/APIReference/API_DeleteCluster.html",
                "description": "Deletes an Amazon EKS cluster.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "DeleteFargateProfile": {
                "url": "https://docs.aws.amazon.com/eks/latest/APIReference/API_DeleteFargateProfile.html",
                "description": "Deletes an AWS Fargate profile.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fargateprofile": {
                        "required": true
                    }
                }
            },
            "DeleteNodegroup": {
                "url": "https://docs.aws.amazon.com/eks/latest/APIReference/API_DeleteNodegroup.html",
                "description": "Deletes an Amazon EKS Nodegroup.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "nodegroup": {
                        "required": true
                    }
                }
            },
            "DescribeCluster": {
                "url": "https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeCluster.html",
                "description": "Returns descriptive information about an Amazon EKS cluster.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "DescribeFargateProfile": {
                "url": "https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeFargateProfile.html",
                "description": "Returns descriptive information about an AWS Fargate profile associated with a cluster.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fargateprofile": {
                        "required": true
                    }
                }
            },
            "DescribeNodegroup": {
                "url": "https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeNodegroup.html",
                "description": "Returns descriptive information about an Amazon EKS nodegroup.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "nodegroup": {
                        "required": true
                    }
                }
            },
            "DescribeUpdate": {
                "url": "https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeUpdate.html",
                "description": "Describes a given update for a given Amazon EKS cluster/nodegroup (in the specified or default region).",
                "accessLevel": "Read",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    },
                    "nodegroup": {
                        "required": false
                    }
                }
            },
            "ListClusters": {
                "url": "https://docs.aws.amazon.com/eks/latest/APIReference/API_ListClusters.html",
                "description": "Lists the Amazon EKS clusters in your AWS account (in the specified or default region).",
                "accessLevel": "List"
            },
            "ListFargateProfiles": {
                "url": "https://docs.aws.amazon.com/eks/latest/APIReference/API_ListFargateProfiles.html",
                "description": "Lists the AWS Fargate profiles in your AWS account (in the specified or default region) associated with a given cluster.",
                "accessLevel": "List",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "ListNodegroups": {
                "url": "https://docs.aws.amazon.com/eks/latest/APIReference/API_ListNodegroups.html",
                "description": "Lists the Amazon EKS nodegroups in your AWS account (in the specified or default region) attached to given cluster.",
                "accessLevel": "List",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/eks/latest/APIReference/API_ListTagsForResource.html",
                "description": "List tags for the specified resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "cluster": {
                        "required": false
                    },
                    "fargateprofile": {
                        "required": false
                    },
                    "nodegroup": {
                        "required": false
                    }
                }
            },
            "ListUpdates": {
                "url": "https://docs.aws.amazon.com/eks/latest/APIReference/API_ListUpdates.html",
                "description": "Lists the updates for a given Amazon EKS cluster/nodegroup (in the specified or default region).",
                "accessLevel": "List",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    },
                    "nodegroup": {
                        "required": false
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/eks/latest/APIReference/API_TagResource.html",
                "description": "Tags the specified resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "cluster": {
                        "required": false
                    },
                    "fargateprofile": {
                        "required": false
                    },
                    "nodegroup": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/eks/latest/APIReference/API_UntagResource.html",
                "description": "Untags the specified resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "cluster": {
                        "required": false
                    },
                    "fargateprofile": {
                        "required": false
                    },
                    "nodegroup": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateClusterConfig": {
                "url": "https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateClusterConfig.html",
                "description": "Update Amazon EKS cluster configurations (eg: API server endpoint access).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "UpdateClusterVersion": {
                "url": "https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateClusterVersion.html",
                "description": "Update the Kubernetes version of an Amazon EKS cluster.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "UpdateNodegroupConfig": {
                "url": "https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateNodegroupConfig.html",
                "description": "Update Amazon EKS nodegroup configurations (eg: min/max/desired capacity or labels).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "nodegroup": {
                        "required": true
                    }
                }
            },
            "UpdateNodegroupVersion": {
                "url": "https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateNodegroupVersion.html",
                "description": "Update the Kubernetes version of an Amazon EKS nodegroup.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "nodegroup": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "cluster": {
                "name": "cluster",
                "url": "https://docs.aws.amazon.com/eks/latest/userguide/clusters.html",
                "arn": "arn:${Partition}:eks:${Region}:${Account}:cluster/${ClusterName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "nodegroup": {
                "name": "nodegroup",
                "url": "https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html",
                "arn": "arn:${Partition}:eks:${Region}:${Account}:nodegroup/${ClusterName}/${NodegroupName}/${UUID}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "fargateprofile": {
                "name": "fargateprofile",
                "url": "https://docs.aws.amazon.com/eks/latest/userguide/fargate-profile.html",
                "arn": "arn:${Partition}:eks:${Region}:${Account}:fargateprofile/${ClusterName}/${FargateProfileName}/${UUID}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates an Amazon EKS cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_CreateCluster.html
     */
    createCluster() {
        this.add('eks:CreateCluster');
        return this;
    }
    /**
     * Creates an AWS Fargate profile.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_CreateFargateProfile.html
     */
    createFargateProfile() {
        this.add('eks:CreateFargateProfile');
        return this;
    }
    /**
     * Creates an Amazon EKS Nodegroup.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_CreateNodegroup.html
     */
    createNodegroup() {
        this.add('eks:CreateNodegroup');
        return this;
    }
    /**
     * Deletes an Amazon EKS cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DeleteCluster.html
     */
    deleteCluster() {
        this.add('eks:DeleteCluster');
        return this;
    }
    /**
     * Deletes an AWS Fargate profile.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DeleteFargateProfile.html
     */
    deleteFargateProfile() {
        this.add('eks:DeleteFargateProfile');
        return this;
    }
    /**
     * Deletes an Amazon EKS Nodegroup.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DeleteNodegroup.html
     */
    deleteNodegroup() {
        this.add('eks:DeleteNodegroup');
        return this;
    }
    /**
     * Returns descriptive information about an Amazon EKS cluster.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeCluster.html
     */
    describeCluster() {
        this.add('eks:DescribeCluster');
        return this;
    }
    /**
     * Returns descriptive information about an AWS Fargate profile associated with a cluster.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeFargateProfile.html
     */
    describeFargateProfile() {
        this.add('eks:DescribeFargateProfile');
        return this;
    }
    /**
     * Returns descriptive information about an Amazon EKS nodegroup.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeNodegroup.html
     */
    describeNodegroup() {
        this.add('eks:DescribeNodegroup');
        return this;
    }
    /**
     * Describes a given update for a given Amazon EKS cluster/nodegroup (in the specified or default region).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeUpdate.html
     */
    describeUpdate() {
        this.add('eks:DescribeUpdate');
        return this;
    }
    /**
     * Lists the Amazon EKS clusters in your AWS account (in the specified or default region).
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListClusters.html
     */
    listClusters() {
        this.add('eks:ListClusters');
        return this;
    }
    /**
     * Lists the AWS Fargate profiles in your AWS account (in the specified or default region) associated with a given cluster.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListFargateProfiles.html
     */
    listFargateProfiles() {
        this.add('eks:ListFargateProfiles');
        return this;
    }
    /**
     * Lists the Amazon EKS nodegroups in your AWS account (in the specified or default region) attached to given cluster.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListNodegroups.html
     */
    listNodegroups() {
        this.add('eks:ListNodegroups');
        return this;
    }
    /**
     * List tags for the specified resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('eks:ListTagsForResource');
        return this;
    }
    /**
     * Lists the updates for a given Amazon EKS cluster/nodegroup (in the specified or default region).
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListUpdates.html
     */
    listUpdates() {
        this.add('eks:ListUpdates');
        return this;
    }
    /**
     * Tags the specified resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('eks:TagResource');
        return this;
    }
    /**
     * Untags the specified resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('eks:UntagResource');
        return this;
    }
    /**
     * Update Amazon EKS cluster configurations (eg: API server endpoint access).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateClusterConfig.html
     */
    updateClusterConfig() {
        this.add('eks:UpdateClusterConfig');
        return this;
    }
    /**
     * Update the Kubernetes version of an Amazon EKS cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateClusterVersion.html
     */
    updateClusterVersion() {
        this.add('eks:UpdateClusterVersion');
        return this;
    }
    /**
     * Update Amazon EKS nodegroup configurations (eg: min/max/desired capacity or labels).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateNodegroupConfig.html
     */
    updateNodegroupConfig() {
        this.add('eks:UpdateNodegroupConfig');
        return this;
    }
    /**
     * Update the Kubernetes version of an Amazon EKS nodegroup.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateNodegroupVersion.html
     */
    updateNodegroupVersion() {
        this.add('eks:UpdateNodegroupVersion');
        return this;
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/clusters.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onCluster(clusterName, account, region, partition) {
        var arn = 'arn:${Partition}:eks:${Region}:${Account}:cluster/${ClusterName}';
        arn = arn.replace('${ClusterName}', clusterName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type nodegroup to the statement
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param nodegroupName - Identifier for the nodegroupName.
     * @param uUID - Identifier for the uUID.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onNodegroup(clusterName, nodegroupName, uUID, account, region, partition) {
        var arn = 'arn:${Partition}:eks:${Region}:${Account}:nodegroup/${ClusterName}/${NodegroupName}/${UUID}';
        arn = arn.replace('${ClusterName}', clusterName);
        arn = arn.replace('${NodegroupName}', nodegroupName);
        arn = arn.replace('${UUID}', uUID);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type fargateprofile to the statement
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/fargate-profile.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param fargateProfileName - Identifier for the fargateProfileName.
     * @param uUID - Identifier for the uUID.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onFargateprofile(clusterName, fargateProfileName, uUID, account, region, partition) {
        var arn = 'arn:${Partition}:eks:${Region}:${Account}:fargateprofile/${ClusterName}/${FargateProfileName}/${UUID}';
        arn = arn.replace('${ClusterName}', clusterName);
        arn = arn.replace('${FargateProfileName}', fargateProfileName);
        arn = arn.replace('${UUID}', uUID);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Eks = Eks;
//# sourceMappingURL=data:application/json;base64,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