"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service elastictranscoder
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelastictranscoder.html
 */
class Elastictranscoder extends shared_1.PolicyStatement {
    /**
     * Action provider for service elastictranscoder
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelastictranscoder.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'elastictranscoder';
        this.actions = {
            "CancelJob": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/cancel-job.html",
                "description": "Cancel a job that Elastic Transcoder has not begun to process",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                }
            },
            "CreateJob": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/create-job.html",
                "description": "Create a job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    },
                    "preset": {
                        "required": true
                    }
                }
            },
            "CreatePipeline": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/create-pipeline.html",
                "description": "Create a pipeline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "CreatePreset": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/create-preset.html",
                "description": "Create a preset.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "preset": {
                        "required": true
                    }
                }
            },
            "DeletePipeline": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/delete-pipeline.html",
                "description": "Delete a pipeline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "DeletePreset": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/delete-preset.html",
                "description": "Delete a preset",
                "accessLevel": "Write",
                "resourceTypes": {
                    "preset": {
                        "required": true
                    }
                }
            },
            "ListJobsByPipeline": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-jobs-by-pipeline.html",
                "description": "Get a list of the jobs that you assigned to a pipeline",
                "accessLevel": "List",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "ListJobsByStatus": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-jobs-by-status.html",
                "description": "Get information about all of the jobs associated with the current AWS account that have a specified status",
                "accessLevel": "List"
            },
            "ListPipelines": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-pipelines.html",
                "description": "Get a list of the pipelines associated with the current AWS account",
                "accessLevel": "List"
            },
            "ListPresets": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-presets.html",
                "description": "Get a list of all presets associated with the current AWS account.",
                "accessLevel": "List"
            },
            "ReadJob": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/get-job.html",
                "description": "Get detailed information about a job",
                "accessLevel": "Read",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                }
            },
            "ReadPipeline": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/get-pipeline.html",
                "description": "Get detailed information about a pipeline",
                "accessLevel": "Read",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "ReadPreset": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/get-preset.html",
                "description": "Get detailed information about a preset.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "preset": {
                        "required": true
                    }
                }
            },
            "TestRole": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/test-pipeline-role.html",
                "description": "Test the settings for a pipeline to ensure that Elastic Transcoder can create and process jobs",
                "accessLevel": "Write"
            },
            "UpdatePipeline": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/update-pipeline.html",
                "description": "Update settings for a pipeline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "UpdatePipelineNotifications": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/update-pipeline-notifications.html",
                "description": "Update only Amazon Simple Notification Service (Amazon SNS) notifications for a pipeline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "UpdatePipelineStatus": {
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/update-pipeline-status.html",
                "description": "Pause or reactivate a pipeline, so the pipeline stops or restarts processing jobs, update the status for the pipeline.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "job": {
                "name": "job",
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/operations-jobs.html",
                "arn": "arn:${Partition}:elastictranscoder:${Region}:${Account}:job/${JobId}",
                "conditionKeys": []
            },
            "pipeline": {
                "name": "pipeline",
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/operations-pipelines.html",
                "arn": "arn:${Partition}:elastictranscoder:${Region}:${Account}:pipeline/${PipelineId}",
                "conditionKeys": []
            },
            "preset": {
                "name": "preset",
                "url": "https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/operations-presets.html",
                "arn": "arn:${Partition}:elastictranscoder:${Region}:${Account}:preset/${PresetId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Cancel a job that Elastic Transcoder has not begun to process
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/cancel-job.html
     */
    cancelJob() {
        this.add('elastictranscoder:CancelJob');
        return this;
    }
    /**
     * Create a job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/create-job.html
     */
    createJob() {
        this.add('elastictranscoder:CreateJob');
        return this;
    }
    /**
     * Create a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/create-pipeline.html
     */
    createPipeline() {
        this.add('elastictranscoder:CreatePipeline');
        return this;
    }
    /**
     * Create a preset.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/create-preset.html
     */
    createPreset() {
        this.add('elastictranscoder:CreatePreset');
        return this;
    }
    /**
     * Delete a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/delete-pipeline.html
     */
    deletePipeline() {
        this.add('elastictranscoder:DeletePipeline');
        return this;
    }
    /**
     * Delete a preset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/delete-preset.html
     */
    deletePreset() {
        this.add('elastictranscoder:DeletePreset');
        return this;
    }
    /**
     * Get a list of the jobs that you assigned to a pipeline
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-jobs-by-pipeline.html
     */
    listJobsByPipeline() {
        this.add('elastictranscoder:ListJobsByPipeline');
        return this;
    }
    /**
     * Get information about all of the jobs associated with the current AWS account that have a specified status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-jobs-by-status.html
     */
    listJobsByStatus() {
        this.add('elastictranscoder:ListJobsByStatus');
        return this;
    }
    /**
     * Get a list of the pipelines associated with the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-pipelines.html
     */
    listPipelines() {
        this.add('elastictranscoder:ListPipelines');
        return this;
    }
    /**
     * Get a list of all presets associated with the current AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-presets.html
     */
    listPresets() {
        this.add('elastictranscoder:ListPresets');
        return this;
    }
    /**
     * Get detailed information about a job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/get-job.html
     */
    readJob() {
        this.add('elastictranscoder:ReadJob');
        return this;
    }
    /**
     * Get detailed information about a pipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/get-pipeline.html
     */
    readPipeline() {
        this.add('elastictranscoder:ReadPipeline');
        return this;
    }
    /**
     * Get detailed information about a preset.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/get-preset.html
     */
    readPreset() {
        this.add('elastictranscoder:ReadPreset');
        return this;
    }
    /**
     * Test the settings for a pipeline to ensure that Elastic Transcoder can create and process jobs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/test-pipeline-role.html
     */
    testRole() {
        this.add('elastictranscoder:TestRole');
        return this;
    }
    /**
     * Update settings for a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/update-pipeline.html
     */
    updatePipeline() {
        this.add('elastictranscoder:UpdatePipeline');
        return this;
    }
    /**
     * Update only Amazon Simple Notification Service (Amazon SNS) notifications for a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/update-pipeline-notifications.html
     */
    updatePipelineNotifications() {
        this.add('elastictranscoder:UpdatePipelineNotifications');
        return this;
    }
    /**
     * Pause or reactivate a pipeline, so the pipeline stops or restarts processing jobs, update the status for the pipeline.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/update-pipeline-status.html
     */
    updatePipelineStatus() {
        this.add('elastictranscoder:UpdatePipelineStatus');
        return this;
    }
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/operations-jobs.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onJob(jobId, account, region, partition) {
        var arn = 'arn:${Partition}:elastictranscoder:${Region}:${Account}:job/${JobId}';
        arn = arn.replace('${JobId}', jobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type pipeline to the statement
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/operations-pipelines.html
     *
     * @param pipelineId - Identifier for the pipelineId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPipeline(pipelineId, account, region, partition) {
        var arn = 'arn:${Partition}:elastictranscoder:${Region}:${Account}:pipeline/${PipelineId}';
        arn = arn.replace('${PipelineId}', pipelineId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type preset to the statement
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/operations-presets.html
     *
     * @param presetId - Identifier for the presetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPreset(presetId, account, region, partition) {
        var arn = 'arn:${Partition}:elastictranscoder:${Region}:${Account}:preset/${PresetId}';
        arn = arn.replace('${PresetId}', presetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Elastictranscoder = Elastictranscoder;
//# sourceMappingURL=data:application/json;base64,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