"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service schemas
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoneventbridgeschemas.html
 */
class Schemas extends shared_1.PolicyStatement {
    /**
     * Action provider for service schemas
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoneventbridgeschemas.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'schemas';
        this.actions = {
            "CreateDiscoverer": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#CreateDiscoverer",
                "description": "Creates an event schema discoverer. Once created, your events will be automatically map into corresponding schema documents",
                "accessLevel": "Write",
                "resourceTypes": {
                    "discoverer": {
                        "required": true
                    }
                }
            },
            "CreateRegistry": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#CreateRegistry",
                "description": "Create a new schema registry in your account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "registry": {
                        "required": true
                    }
                }
            },
            "CreateSchema": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#CreateSchema",
                "description": "Create a new schema in your account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "schema": {
                        "required": true
                    }
                }
            },
            "DeleteDiscoverer": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#DeleteDiscoverer",
                "description": "Deletes discoverer in your account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "discoverer": {
                        "required": true
                    }
                }
            },
            "DeleteRegistry": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#DeleteRegistry",
                "description": "Deletes an existing registry in your account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "registry": {
                        "required": true
                    }
                }
            },
            "DeleteResourcePolicy": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-policy.html#DeleteResourcePolicy",
                "description": "Delete the resource-based policy attached to a given registry.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "registry": {
                        "required": true
                    }
                }
            },
            "DeleteSchema": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#DeleteSchema",
                "description": "Deletes an existing schema in your account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "schema": {
                        "required": true
                    }
                }
            },
            "DeleteSchemaVersion": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-version-schemaversion.html#DeleteSchemaVersion",
                "description": "Deletes a specific version of schema in your account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "schema": {
                        "required": true
                    }
                }
            },
            "DescribeCodeBinding": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-language-language.html#DescribeCodeBinding",
                "description": "Retrieves metadata for generated code for specific schema in your account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "schema": {
                        "required": true
                    }
                }
            },
            "DescribeDiscoverer": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#DescribeDiscoverer",
                "description": "Retrieves discoverer metadata in your account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "discoverer": {
                        "required": true
                    }
                }
            },
            "DescribeRegistry": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#DescribeRegistry",
                "description": "Describes an existing registry metadata in your account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "registry": {
                        "required": true
                    }
                }
            },
            "DescribeSchema": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#DescribeSchema",
                "description": "Retrieves an existing schema in your account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "schema": {
                        "required": true
                    }
                }
            },
            "GetCodeBindingSource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-language-language-source.html#GetCodeBindingSource",
                "description": "Retrieves metadata for generated code for specific schema in your account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "schema": {
                        "required": true
                    }
                }
            },
            "GetDiscoveredSchema": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discover.html#GetDiscoveredSchema",
                "description": "Retrieves schema for the provided list of sample events.",
                "accessLevel": "Read"
            },
            "GetResourcePolicy": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-policy.html#GetResourcePolicy",
                "description": "Retrieves the resource-based policy attached to a given registry.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "registry": {
                        "required": true
                    }
                }
            },
            "ListDiscoverers": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers.html#ListDiscoverers",
                "description": "Lists all the discoverers in your account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "discoverer": {
                        "required": true
                    }
                }
            },
            "ListRegistries": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries.html#ListRegistries",
                "description": "List all discoverers in your account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "registry": {
                        "required": true
                    }
                }
            },
            "ListSchemaVersions": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-versions.html#ListSchemaVersions",
                "description": "List all versions of a schema.",
                "accessLevel": "List",
                "resourceTypes": {
                    "schema": {
                        "required": true
                    }
                }
            },
            "ListSchemas": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas.html#ListSchemas",
                "description": "List all schemas.",
                "accessLevel": "List",
                "resourceTypes": {
                    "schema": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/tags-resource-arn.html#ListTagsForResource",
                "description": "This action lists tags for a resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "discoverer": {
                        "required": true
                    },
                    "registry": {
                        "required": true
                    },
                    "schema": {
                        "required": true
                    }
                }
            },
            "PutCodeBinding": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-language-language.html#PutCodeBinding",
                "description": "Generates code for specific schema in your account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "schema": {
                        "required": true
                    }
                }
            },
            "PutResourcePolicy": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-policy.html#PutResourcePolicy",
                "description": "Attach resource-based policy to the specific registry.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "registry": {
                        "required": true
                    }
                }
            },
            "SearchSchemas": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-search.html#SearchSchemas",
                "description": "Searches schemas based on specified keywords in your account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "schema": {
                        "required": true
                    }
                }
            },
            "StartDiscoverer": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#StartDiscoverer",
                "description": "Starts the specified discoverer. Once started the discoverer will automatically register schemas for published events to configured source in your account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "discoverer": {
                        "required": true
                    }
                }
            },
            "StopDiscoverer": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#StopDiscoverer",
                "description": "Starts the specified discoverer. Once started the discoverer will automatically register schemas for published events to configured source in your account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "discoverer": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/tags-resource-arn.html#TagResource",
                "description": "This action tags an resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "discoverer": {
                        "required": true
                    },
                    "registry": {
                        "required": true
                    },
                    "schema": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/tags-resource-arn.html#UntagResource",
                "description": "This action removes a tag from on a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "discoverer": {
                        "required": true
                    },
                    "registry": {
                        "required": true
                    },
                    "schema": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateDiscoverer": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#UpdateDiscoverer",
                "description": "Updates an existing discoverer in your account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "discoverer": {
                        "required": true
                    }
                }
            },
            "UpdateRegistry": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#UpdateRegistry",
                "description": "Updates an existing registry metadata in your account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "registry": {
                        "required": true
                    }
                }
            },
            "UpdateSchema": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#UpdateSchema",
                "description": "Updates an existing schema in your account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "schema": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "discoverer": {
                "name": "discoverer",
                "url": "https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-identity-based-access-control-eventbridge.html",
                "arn": "arn:${Partition}:schemas:${Region}:${Account}:discoverer/${DiscovererId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "registry": {
                "name": "registry",
                "url": "https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-identity-based-access-control-eventbridge.html",
                "arn": "arn:${Partition}:schemas:${Region}:${Account}:registry/${RegistryName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "schema": {
                "name": "schema",
                "url": "https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-identity-based-access-control-eventbridge.html",
                "arn": "arn:${Partition}:schemas:${Region}:${Account}:schema/${RegistryName}/${SchemaName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates an event schema discoverer. Once created, your events will be automatically map into corresponding schema documents
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#CreateDiscoverer
     */
    createDiscoverer() {
        this.add('schemas:CreateDiscoverer');
        return this;
    }
    /**
     * Create a new schema registry in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#CreateRegistry
     */
    createRegistry() {
        this.add('schemas:CreateRegistry');
        return this;
    }
    /**
     * Create a new schema in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#CreateSchema
     */
    createSchema() {
        this.add('schemas:CreateSchema');
        return this;
    }
    /**
     * Deletes discoverer in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#DeleteDiscoverer
     */
    deleteDiscoverer() {
        this.add('schemas:DeleteDiscoverer');
        return this;
    }
    /**
     * Deletes an existing registry in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#DeleteRegistry
     */
    deleteRegistry() {
        this.add('schemas:DeleteRegistry');
        return this;
    }
    /**
     * Delete the resource-based policy attached to a given registry.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-policy.html#DeleteResourcePolicy
     */
    deleteResourcePolicy() {
        this.add('schemas:DeleteResourcePolicy');
        return this;
    }
    /**
     * Deletes an existing schema in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#DeleteSchema
     */
    deleteSchema() {
        this.add('schemas:DeleteSchema');
        return this;
    }
    /**
     * Deletes a specific version of schema in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-version-schemaversion.html#DeleteSchemaVersion
     */
    deleteSchemaVersion() {
        this.add('schemas:DeleteSchemaVersion');
        return this;
    }
    /**
     * Retrieves metadata for generated code for specific schema in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-language-language.html#DescribeCodeBinding
     */
    describeCodeBinding() {
        this.add('schemas:DescribeCodeBinding');
        return this;
    }
    /**
     * Retrieves discoverer metadata in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#DescribeDiscoverer
     */
    describeDiscoverer() {
        this.add('schemas:DescribeDiscoverer');
        return this;
    }
    /**
     * Describes an existing registry metadata in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#DescribeRegistry
     */
    describeRegistry() {
        this.add('schemas:DescribeRegistry');
        return this;
    }
    /**
     * Retrieves an existing schema in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#DescribeSchema
     */
    describeSchema() {
        this.add('schemas:DescribeSchema');
        return this;
    }
    /**
     * Retrieves metadata for generated code for specific schema in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-language-language-source.html#GetCodeBindingSource
     */
    getCodeBindingSource() {
        this.add('schemas:GetCodeBindingSource');
        return this;
    }
    /**
     * Retrieves schema for the provided list of sample events.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discover.html#GetDiscoveredSchema
     */
    getDiscoveredSchema() {
        this.add('schemas:GetDiscoveredSchema');
        return this;
    }
    /**
     * Retrieves the resource-based policy attached to a given registry.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-policy.html#GetResourcePolicy
     */
    getResourcePolicy() {
        this.add('schemas:GetResourcePolicy');
        return this;
    }
    /**
     * Lists all the discoverers in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers.html#ListDiscoverers
     */
    listDiscoverers() {
        this.add('schemas:ListDiscoverers');
        return this;
    }
    /**
     * List all discoverers in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries.html#ListRegistries
     */
    listRegistries() {
        this.add('schemas:ListRegistries');
        return this;
    }
    /**
     * List all versions of a schema.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-versions.html#ListSchemaVersions
     */
    listSchemaVersions() {
        this.add('schemas:ListSchemaVersions');
        return this;
    }
    /**
     * List all schemas.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas.html#ListSchemas
     */
    listSchemas() {
        this.add('schemas:ListSchemas');
        return this;
    }
    /**
     * This action lists tags for a resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/tags-resource-arn.html#ListTagsForResource
     */
    listTagsForResource() {
        this.add('schemas:ListTagsForResource');
        return this;
    }
    /**
     * Generates code for specific schema in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-language-language.html#PutCodeBinding
     */
    putCodeBinding() {
        this.add('schemas:PutCodeBinding');
        return this;
    }
    /**
     * Attach resource-based policy to the specific registry.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-policy.html#PutResourcePolicy
     */
    putResourcePolicy() {
        this.add('schemas:PutResourcePolicy');
        return this;
    }
    /**
     * Searches schemas based on specified keywords in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-search.html#SearchSchemas
     */
    searchSchemas() {
        this.add('schemas:SearchSchemas');
        return this;
    }
    /**
     * Starts the specified discoverer. Once started the discoverer will automatically register schemas for published events to configured source in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#StartDiscoverer
     */
    startDiscoverer() {
        this.add('schemas:StartDiscoverer');
        return this;
    }
    /**
     * Starts the specified discoverer. Once started the discoverer will automatically register schemas for published events to configured source in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#StopDiscoverer
     */
    stopDiscoverer() {
        this.add('schemas:StopDiscoverer');
        return this;
    }
    /**
     * This action tags an resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/tags-resource-arn.html#TagResource
     */
    tagResource() {
        this.add('schemas:TagResource');
        return this;
    }
    /**
     * This action removes a tag from on a resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/tags-resource-arn.html#UntagResource
     */
    untagResource() {
        this.add('schemas:UntagResource');
        return this;
    }
    /**
     * Updates an existing discoverer in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#UpdateDiscoverer
     */
    updateDiscoverer() {
        this.add('schemas:UpdateDiscoverer');
        return this;
    }
    /**
     * Updates an existing registry metadata in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#UpdateRegistry
     */
    updateRegistry() {
        this.add('schemas:UpdateRegistry');
        return this;
    }
    /**
     * Updates an existing schema in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#UpdateSchema
     */
    updateSchema() {
        this.add('schemas:UpdateSchema');
        return this;
    }
    /**
     * Adds a resource of type discoverer to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-identity-based-access-control-eventbridge.html
     *
     * @param discovererId - Identifier for the discovererId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDiscoverer(discovererId, account, region, partition) {
        var arn = 'arn:${Partition}:schemas:${Region}:${Account}:discoverer/${DiscovererId}';
        arn = arn.replace('${DiscovererId}', discovererId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type registry to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-identity-based-access-control-eventbridge.html
     *
     * @param registryName - Identifier for the registryName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onRegistry(registryName, account, region, partition) {
        var arn = 'arn:${Partition}:schemas:${Region}:${Account}:registry/${RegistryName}';
        arn = arn.replace('${RegistryName}', registryName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type schema to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-identity-based-access-control-eventbridge.html
     *
     * @param registryName - Identifier for the registryName.
     * @param schemaName - Identifier for the schemaName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onSchema(registryName, schemaName, account, region, partition) {
        var arn = 'arn:${Partition}:schemas:${Region}:${Account}:schema/${RegistryName}/${SchemaName}';
        arn = arn.replace('${RegistryName}', registryName);
        arn = arn.replace('${SchemaName}', schemaName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Schemas = Schemas;
//# sourceMappingURL=data:application/json;base64,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