"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service lex
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonlex.html
 */
class Lex extends shared_1.PolicyStatement {
    /**
     * Action provider for service lex
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonlex.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'lex';
        this.actions = {
            "CreateBotVersion": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_CreateBotVersion.html",
                "description": "Creates a new version based on the $LATEST version of the specified bot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                }
            },
            "CreateIntentVersion": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_CreateIntentVersion.html",
                "description": "Creates a new version based on the $LATEST version of the specified intent.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "intent version": {
                        "required": true
                    }
                }
            },
            "CreateSlotTypeVersion": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_CreateSlotTypeVersion.html",
                "description": "Creates a new version based on the $LATEST version of the specified slot type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "slottype version": {
                        "required": true
                    }
                }
            },
            "DeleteBot": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBot.html",
                "description": "Deletes all versions of a bot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                }
            },
            "DeleteBotAlias": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBotAlias.html",
                "description": "Deletes an alias for a specific bot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot alias": {
                        "required": true
                    }
                }
            },
            "DeleteBotChannelAssociation": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBotChannelAssociation.html",
                "description": "Deletes the association between a Amazon Lex bot alias and a messaging platform.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "DeleteBotVersion": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBotVersion.html",
                "description": "Deletes a specific version of a bot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                }
            },
            "DeleteIntent": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteIntent.html",
                "description": "Deletes all versions of an intent.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "intent version": {
                        "required": true
                    }
                }
            },
            "DeleteIntentVersion": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteIntentVersion.html",
                "description": "Deletes a specific version of an intent.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "intent version": {
                        "required": true
                    }
                }
            },
            "DeleteSession": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_runtime_DeleteSession.html",
                "description": "Removes session information for a specified bot, alias, and user ID.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot alias": {
                        "required": false
                    },
                    "bot version": {
                        "required": false
                    }
                }
            },
            "DeleteSlotType": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteSlotType.html",
                "description": "Deletes all versions of a slot type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "slottype version": {
                        "required": true
                    }
                }
            },
            "DeleteSlotTypeVersion": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteSlotTypeVersion.html",
                "description": "Deletes a specific version of a slot type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "slottype version": {
                        "required": true
                    }
                }
            },
            "DeleteUtterances": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteUtterances.html",
                "description": "Deletes the information Amazon Lex maintains for utterances on a specific bot and userId.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                }
            },
            "GetBot": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBot.html",
                "description": "Returns information for a specific bot. In addition to the bot name, the bot version or alias is required.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bot alias": {
                        "required": false
                    },
                    "bot version": {
                        "required": false
                    }
                }
            },
            "GetBotAlias": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBotAlias.html",
                "description": "Returns information about a Amazon Lex bot alias.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bot alias": {
                        "required": true
                    }
                }
            },
            "GetBotAliases": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBotAliases.html",
                "description": "Returns a list of aliases for a given Amazon Lex bot.",
                "accessLevel": "List"
            },
            "GetBotChannelAssociation": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBotChannelAssociation.html",
                "description": "Returns information about the association between a Amazon Lex bot and a messaging platform.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "GetBotChannelAssociations": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBotChannelAssociations.html",
                "description": "Returns a list of all of the channels associated with a single bot.",
                "accessLevel": "List",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "GetBotVersions": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBotVersions.html",
                "description": "Returns information for all versions of a specific bot.",
                "accessLevel": "List",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                }
            },
            "GetBots": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBots.html",
                "description": "Returns information for the $LATEST version of all bots, subject to filters provided by the client.",
                "accessLevel": "List"
            },
            "GetBuiltinIntent": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBuiltinIntent.html",
                "description": "Returns information about a built-in intent.",
                "accessLevel": "Read"
            },
            "GetBuiltinIntents": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBuiltinIntents.html",
                "description": "Gets a list of built-in intents that meet the specified criteria.",
                "accessLevel": "Read"
            },
            "GetBuiltinSlotTypes": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBuiltinSlotTypes.html",
                "description": "Gets a list of built-in slot types that meet the specified criteria.",
                "accessLevel": "Read"
            },
            "GetExport": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetExport.html",
                "description": "Exports Amazon Lex Resource in a requested format.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                }
            },
            "GetImport": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetImport.html",
                "description": "Gets information about an import job started with StartImport.",
                "accessLevel": "Read"
            },
            "GetIntent": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetIntent.html",
                "description": "Returns information for a specific intent. In addition to the intent name, you must also specify the intent version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "intent version": {
                        "required": true
                    }
                }
            },
            "GetIntentVersions": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetIntentVersions.html",
                "description": "Returns information for all versions of a specific intent.",
                "accessLevel": "List",
                "resourceTypes": {
                    "intent version": {
                        "required": true
                    }
                }
            },
            "GetIntents": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetIntents.html",
                "description": "Returns information for the $LATEST version of all intents, subject to filters provided by the client.",
                "accessLevel": "List"
            },
            "GetSession": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_runtime_GetSession.html",
                "description": "Returns session information for a specified bot, alias, and user ID.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bot alias": {
                        "required": false
                    },
                    "bot version": {
                        "required": false
                    }
                }
            },
            "GetSlotType": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetSlotType.html",
                "description": "Returns information about a specific version of a slot type. In addition to specifying the slot type name, you must also specify the slot type version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "slottype version": {
                        "required": true
                    }
                }
            },
            "GetSlotTypeVersions": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetSlotTypeVersions.html",
                "description": "Returns information for all versions of a specific slot type.",
                "accessLevel": "List",
                "resourceTypes": {
                    "slottype version": {
                        "required": true
                    }
                }
            },
            "GetSlotTypes": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetSlotTypes.html",
                "description": "Returns information for the $LATEST version of all slot types, subject to filters provided by the client.",
                "accessLevel": "List"
            },
            "GetUtterancesView": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetUtterancesView.html",
                "description": "Returns a view of aggregate utterance data for versions of a bot for a recent time period.",
                "accessLevel": "List",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_ListTagsForResource.html",
                "description": "Lists tags for a Lex resource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bot": {
                        "required": false
                    },
                    "bot alias": {
                        "required": false
                    },
                    "channel": {
                        "required": false
                    }
                }
            },
            "PostContent": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_runtime_PostContent.html",
                "description": "Sends user input (text or speech) to Amazon Lex.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot alias": {
                        "required": false
                    },
                    "bot version": {
                        "required": false
                    }
                }
            },
            "PostText": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_runtime_PostText.html",
                "description": "Sends user input (text-only) to Amazon Lex.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot alias": {
                        "required": false
                    },
                    "bot version": {
                        "required": false
                    }
                }
            },
            "PutBot": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_PutBot.html",
                "description": "Creates or updates the $LATEST version of a Amazon Lex conversational bot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "PutBotAlias": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_PutBotAlias.html",
                "description": "Creates or updates an alias for the specific bot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot alias": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "PutIntent": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_PutIntent.html",
                "description": "Creates or updates the $LATEST version of an intent.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "intent version": {
                        "required": true
                    }
                }
            },
            "PutSession": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_runtime_PutSession.html",
                "description": "Creates a new session or modifies an existing session with an Amazon Lex bot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot alias": {
                        "required": false
                    },
                    "bot version": {
                        "required": false
                    }
                }
            },
            "PutSlotType": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_PutSlotType.html",
                "description": "Creates or updates the $LATEST version of a slot type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "slottype version": {
                        "required": true
                    }
                }
            },
            "StartImport": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_StartImport.html",
                "description": "Starts a job to import a resource to Amazon Lex.",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_TagResource.html",
                "description": "Adds or overwrites tags to a Lex resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "bot": {
                        "required": false
                    },
                    "bot alias": {
                        "required": false
                    },
                    "channel": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_UntagResource.html",
                "description": "Removes tags from a Lex resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "bot": {
                        "required": false
                    },
                    "bot alias": {
                        "required": false
                    },
                    "channel": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            }
        };
        this.resourceTypes = {
            "bot": {
                "name": "bot",
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_BotMetadata.html",
                "arn": "arn:${Partition}:lex:${Region}:${Account}:bot:${BotName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "bot version": {
                "name": "bot version",
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_BotMetadata.html",
                "arn": "arn:${Partition}:lex:${Region}:${Account}:bot:${BotName}:${BotVersion}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "bot alias": {
                "name": "bot alias",
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_BotAliasMetadata.html",
                "arn": "arn:${Partition}:lex:${Region}:${Account}:bot:${BotName}:${BotAlias}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "channel": {
                "name": "channel",
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_BotChannelAssociation.html",
                "arn": "arn:${Partition}:lex:${Region}:${Account}:bot-channel:${BotName}:${BotAlias}:${ChannelName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "intent version": {
                "name": "intent version",
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_Intent.html",
                "arn": "arn:${Partition}:lex:${Region}:${Account}:intent:${IntentName}:${IntentVersion}",
                "conditionKeys": []
            },
            "slottype version": {
                "name": "slottype version",
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_SlotTypeMetadata.html",
                "arn": "arn:${Partition}:lex:${Region}:${Account}:slottype:${SlotName}:${SlotVersion}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Creates a new version based on the $LATEST version of the specified bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_CreateBotVersion.html
     */
    createBotVersion() {
        this.add('lex:CreateBotVersion');
        return this;
    }
    /**
     * Creates a new version based on the $LATEST version of the specified intent.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_CreateIntentVersion.html
     */
    createIntentVersion() {
        this.add('lex:CreateIntentVersion');
        return this;
    }
    /**
     * Creates a new version based on the $LATEST version of the specified slot type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_CreateSlotTypeVersion.html
     */
    createSlotTypeVersion() {
        this.add('lex:CreateSlotTypeVersion');
        return this;
    }
    /**
     * Deletes all versions of a bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBot.html
     */
    deleteBot() {
        this.add('lex:DeleteBot');
        return this;
    }
    /**
     * Deletes an alias for a specific bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBotAlias.html
     */
    deleteBotAlias() {
        this.add('lex:DeleteBotAlias');
        return this;
    }
    /**
     * Deletes the association between a Amazon Lex bot alias and a messaging platform.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBotChannelAssociation.html
     */
    deleteBotChannelAssociation() {
        this.add('lex:DeleteBotChannelAssociation');
        return this;
    }
    /**
     * Deletes a specific version of a bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBotVersion.html
     */
    deleteBotVersion() {
        this.add('lex:DeleteBotVersion');
        return this;
    }
    /**
     * Deletes all versions of an intent.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteIntent.html
     */
    deleteIntent() {
        this.add('lex:DeleteIntent');
        return this;
    }
    /**
     * Deletes a specific version of an intent.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteIntentVersion.html
     */
    deleteIntentVersion() {
        this.add('lex:DeleteIntentVersion');
        return this;
    }
    /**
     * Removes session information for a specified bot, alias, and user ID.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_runtime_DeleteSession.html
     */
    deleteSession() {
        this.add('lex:DeleteSession');
        return this;
    }
    /**
     * Deletes all versions of a slot type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteSlotType.html
     */
    deleteSlotType() {
        this.add('lex:DeleteSlotType');
        return this;
    }
    /**
     * Deletes a specific version of a slot type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteSlotTypeVersion.html
     */
    deleteSlotTypeVersion() {
        this.add('lex:DeleteSlotTypeVersion');
        return this;
    }
    /**
     * Deletes the information Amazon Lex maintains for utterances on a specific bot and userId.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteUtterances.html
     */
    deleteUtterances() {
        this.add('lex:DeleteUtterances');
        return this;
    }
    /**
     * Returns information for a specific bot. In addition to the bot name, the bot version or alias is required.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBot.html
     */
    getBot() {
        this.add('lex:GetBot');
        return this;
    }
    /**
     * Returns information about a Amazon Lex bot alias.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBotAlias.html
     */
    getBotAlias() {
        this.add('lex:GetBotAlias');
        return this;
    }
    /**
     * Returns a list of aliases for a given Amazon Lex bot.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBotAliases.html
     */
    getBotAliases() {
        this.add('lex:GetBotAliases');
        return this;
    }
    /**
     * Returns information about the association between a Amazon Lex bot and a messaging platform.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBotChannelAssociation.html
     */
    getBotChannelAssociation() {
        this.add('lex:GetBotChannelAssociation');
        return this;
    }
    /**
     * Returns a list of all of the channels associated with a single bot.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBotChannelAssociations.html
     */
    getBotChannelAssociations() {
        this.add('lex:GetBotChannelAssociations');
        return this;
    }
    /**
     * Returns information for all versions of a specific bot.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBotVersions.html
     */
    getBotVersions() {
        this.add('lex:GetBotVersions');
        return this;
    }
    /**
     * Returns information for the $LATEST version of all bots, subject to filters provided by the client.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBots.html
     */
    getBots() {
        this.add('lex:GetBots');
        return this;
    }
    /**
     * Returns information about a built-in intent.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBuiltinIntent.html
     */
    getBuiltinIntent() {
        this.add('lex:GetBuiltinIntent');
        return this;
    }
    /**
     * Gets a list of built-in intents that meet the specified criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBuiltinIntents.html
     */
    getBuiltinIntents() {
        this.add('lex:GetBuiltinIntents');
        return this;
    }
    /**
     * Gets a list of built-in slot types that meet the specified criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBuiltinSlotTypes.html
     */
    getBuiltinSlotTypes() {
        this.add('lex:GetBuiltinSlotTypes');
        return this;
    }
    /**
     * Exports Amazon Lex Resource in a requested format.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetExport.html
     */
    getExport() {
        this.add('lex:GetExport');
        return this;
    }
    /**
     * Gets information about an import job started with StartImport.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetImport.html
     */
    getImport() {
        this.add('lex:GetImport');
        return this;
    }
    /**
     * Returns information for a specific intent. In addition to the intent name, you must also specify the intent version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetIntent.html
     */
    getIntent() {
        this.add('lex:GetIntent');
        return this;
    }
    /**
     * Returns information for all versions of a specific intent.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetIntentVersions.html
     */
    getIntentVersions() {
        this.add('lex:GetIntentVersions');
        return this;
    }
    /**
     * Returns information for the $LATEST version of all intents, subject to filters provided by the client.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetIntents.html
     */
    getIntents() {
        this.add('lex:GetIntents');
        return this;
    }
    /**
     * Returns session information for a specified bot, alias, and user ID.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_runtime_GetSession.html
     */
    getSession() {
        this.add('lex:GetSession');
        return this;
    }
    /**
     * Returns information about a specific version of a slot type. In addition to specifying the slot type name, you must also specify the slot type version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetSlotType.html
     */
    getSlotType() {
        this.add('lex:GetSlotType');
        return this;
    }
    /**
     * Returns information for all versions of a specific slot type.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetSlotTypeVersions.html
     */
    getSlotTypeVersions() {
        this.add('lex:GetSlotTypeVersions');
        return this;
    }
    /**
     * Returns information for the $LATEST version of all slot types, subject to filters provided by the client.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetSlotTypes.html
     */
    getSlotTypes() {
        this.add('lex:GetSlotTypes');
        return this;
    }
    /**
     * Returns a view of aggregate utterance data for versions of a bot for a recent time period.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetUtterancesView.html
     */
    getUtterancesView() {
        this.add('lex:GetUtterancesView');
        return this;
    }
    /**
     * Lists tags for a Lex resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('lex:ListTagsForResource');
        return this;
    }
    /**
     * Sends user input (text or speech) to Amazon Lex.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_runtime_PostContent.html
     */
    postContent() {
        this.add('lex:PostContent');
        return this;
    }
    /**
     * Sends user input (text-only) to Amazon Lex.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_runtime_PostText.html
     */
    postText() {
        this.add('lex:PostText');
        return this;
    }
    /**
     * Creates or updates the $LATEST version of a Amazon Lex conversational bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_PutBot.html
     */
    putBot() {
        this.add('lex:PutBot');
        return this;
    }
    /**
     * Creates or updates an alias for the specific bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_PutBotAlias.html
     */
    putBotAlias() {
        this.add('lex:PutBotAlias');
        return this;
    }
    /**
     * Creates or updates the $LATEST version of an intent.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_PutIntent.html
     */
    putIntent() {
        this.add('lex:PutIntent');
        return this;
    }
    /**
     * Creates a new session or modifies an existing session with an Amazon Lex bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_runtime_PutSession.html
     */
    putSession() {
        this.add('lex:PutSession');
        return this;
    }
    /**
     * Creates or updates the $LATEST version of a slot type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_PutSlotType.html
     */
    putSlotType() {
        this.add('lex:PutSlotType');
        return this;
    }
    /**
     * Starts a job to import a resource to Amazon Lex.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_StartImport.html
     */
    startImport() {
        this.add('lex:StartImport');
        return this;
    }
    /**
     * Adds or overwrites tags to a Lex resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_TagResource.html
     */
    tagResource() {
        this.add('lex:TagResource');
        return this;
    }
    /**
     * Removes tags from a Lex resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_UntagResource.html
     */
    untagResource() {
        this.add('lex:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type bot to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_BotMetadata.html
     *
     * @param botName - Identifier for the botName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onBot(botName, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:bot:${BotName}';
        arn = arn.replace('${BotName}', botName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type bot version to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_BotMetadata.html
     *
     * @param botName - Identifier for the botName.
     * @param botVersion - Identifier for the botVersion.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onBotVersion(botName, botVersion, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:bot:${BotName}:${BotVersion}';
        arn = arn.replace('${BotName}', botName);
        arn = arn.replace('${BotVersion}', botVersion);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type bot alias to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_BotAliasMetadata.html
     *
     * @param botName - Identifier for the botName.
     * @param botAlias - Identifier for the botAlias.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onBotAlias(botName, botAlias, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:bot:${BotName}:${BotAlias}';
        arn = arn.replace('${BotName}', botName);
        arn = arn.replace('${BotAlias}', botAlias);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type channel to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_BotChannelAssociation.html
     *
     * @param botName - Identifier for the botName.
     * @param botAlias - Identifier for the botAlias.
     * @param channelName - Identifier for the channelName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onChannel(botName, botAlias, channelName, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:bot-channel:${BotName}:${BotAlias}:${ChannelName}';
        arn = arn.replace('${BotName}', botName);
        arn = arn.replace('${BotAlias}', botAlias);
        arn = arn.replace('${ChannelName}', channelName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type intent version to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_Intent.html
     *
     * @param intentName - Identifier for the intentName.
     * @param intentVersion - Identifier for the intentVersion.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIntentVersion(intentName, intentVersion, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:intent:${IntentName}:${IntentVersion}';
        arn = arn.replace('${IntentName}', intentName);
        arn = arn.replace('${IntentVersion}', intentVersion);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type slottype version to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_SlotTypeMetadata.html
     *
     * @param slotName - Identifier for the slotName.
     * @param slotVersion - Identifier for the slotVersion.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSlottypeVersion(slotName, slotVersion, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:slottype:${SlotName}:${SlotVersion}';
        arn = arn.replace('${SlotName}', slotName);
        arn = arn.replace('${SlotVersion}', slotVersion);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Enables you to control access based on the intents included in the request.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAssociatedIntents(value, operator) {
        return this.if(`lex:associatedIntents`, value, operator || 'StringLike');
    }
    /**
     * Enables you to control access based on the slot types included in the request.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAssociatedSlotTypes(value, operator) {
        return this.if(`lex:associatedSlotTypes`, value, operator || 'StringLike');
    }
    /**
     * Enables you to control access based on the channel type included in the request.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifChannelType(value, operator) {
        return this.if(`lex:channelType`, value, operator || 'StringLike');
    }
}
exports.Lex = Lex;
//# sourceMappingURL=data:application/json;base64,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