"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service kafka
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmanagedstreamingforkafka.html
 */
class Kafka extends shared_1.PolicyStatement {
    /**
     * Action provider for service kafka
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmanagedstreamingforkafka.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'kafka';
        this.actions = {
            "CreateCluster": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters.html#CreateCluster",
                "description": "Grants permission to create a cluster.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateConfiguration": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/configurations.html#CreateConfiguration",
                "description": "Grants permission to create a configuration.",
                "accessLevel": "Write"
            },
            "DeleteCluster": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn.html#DeleteCluster",
                "description": "Grants permission to delete a cluster.",
                "accessLevel": "Write"
            },
            "DescribeCluster": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn.html#DescribeCluster",
                "description": "Grants permission to describe a cluster.",
                "accessLevel": "Read"
            },
            "DescribeClusterOperation": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/operations-clusteroperationarn.html#DescribeClusterOperation",
                "description": "Returns a description of the cluster operation specified by the ARN.",
                "accessLevel": "Read"
            },
            "DescribeConfiguration": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/configurations-configurationarn.html#DescribeConfiguration",
                "description": "Grants permission to describe a configuration.",
                "accessLevel": "Read"
            },
            "DescribeConfigurationRevision": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/configurations-configurationarn-revision.html#DescribeConfigurationRevision",
                "description": "Grants permission to describe a configuration revision.",
                "accessLevel": "Read"
            },
            "GetBootstrapBrokers": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-bootstrap-brokers.html#GetBootstrapBrokers",
                "description": "Grants permission to get connection details for the broker nodes in a cluster.",
                "accessLevel": "Read"
            },
            "GetCompatibleKafkaVersions": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/compatible-kafka-versions.html#GetCompatibleKafkaVersions",
                "description": "Returns a list of the Apache Kafka versions to which you can update this cluster.",
                "accessLevel": "List"
            },
            "ListClusterOperations": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-operations.html#ListClusterOperations",
                "description": "Returns a list of all the operations that have been performed on the specified MSK cluster.",
                "accessLevel": "Read"
            },
            "ListClusters": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters.html#ListClusters",
                "description": "Grants permission to return a list of all clusters in the current account.",
                "accessLevel": "List"
            },
            "ListConfigurations": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/configurations.html#CreateConfiguration",
                "description": "Grants permission to return a list of all configurations in the current account.",
                "accessLevel": "List"
            },
            "ListNodes": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes.html#ListNodes",
                "description": "Grants permission to return a list of nodes in a cluster.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#ListTagsForResource",
                "description": "Grants permission to list tags of a MSK resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "cluster": {
                        "required": false
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#TagResource",
                "description": "Grants permission to tag a MSK resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "cluster": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#UntagResource",
                "description": "Grants permission to remove tags from a MSK resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "cluster": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateBrokerCount": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes-count.html#UpdateBrokerCount",
                "description": "Updates the number of broker nodes of the cluster.",
                "accessLevel": "Write"
            },
            "UpdateBrokerStorage": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes-storage.html#UpdateBrokerStorage",
                "description": "Updates the storage size of the broker nodes of the cluster",
                "accessLevel": "Write"
            },
            "UpdateClusterConfiguration": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-configuration.html#UpdateClusterConfiguration",
                "description": "Update Kafka configuration running on a cluster.",
                "accessLevel": "Write"
            },
            "UpdateClusterKafkaVersion": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-version.html#UpdateClusterKafkaVersion",
                "description": "Updates the cluster to the specified Apache Kafka version.",
                "accessLevel": "Write"
            },
            "UpdateMonitoring": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-monitoring.html#UpdateMonitoring",
                "description": "Updates the monitoring settings for the cluster.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "cluster": {
                "name": "cluster",
                "url": "",
                "arn": "arn:${Partition}:kafka:${Region}:${Account}:cluster/${ClusterName}/${UUID}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to create a cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters.html#CreateCluster
     */
    createCluster() {
        this.add('kafka:CreateCluster');
        return this;
    }
    /**
     * Grants permission to create a configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations.html#CreateConfiguration
     */
    createConfiguration() {
        this.add('kafka:CreateConfiguration');
        return this;
    }
    /**
     * Grants permission to delete a cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn.html#DeleteCluster
     */
    deleteCluster() {
        this.add('kafka:DeleteCluster');
        return this;
    }
    /**
     * Grants permission to describe a cluster.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn.html#DescribeCluster
     */
    describeCluster() {
        this.add('kafka:DescribeCluster');
        return this;
    }
    /**
     * Returns a description of the cluster operation specified by the ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/operations-clusteroperationarn.html#DescribeClusterOperation
     */
    describeClusterOperation() {
        this.add('kafka:DescribeClusterOperation');
        return this;
    }
    /**
     * Grants permission to describe a configuration.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations-configurationarn.html#DescribeConfiguration
     */
    describeConfiguration() {
        this.add('kafka:DescribeConfiguration');
        return this;
    }
    /**
     * Grants permission to describe a configuration revision.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations-configurationarn-revision.html#DescribeConfigurationRevision
     */
    describeConfigurationRevision() {
        this.add('kafka:DescribeConfigurationRevision');
        return this;
    }
    /**
     * Grants permission to get connection details for the broker nodes in a cluster.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-bootstrap-brokers.html#GetBootstrapBrokers
     */
    getBootstrapBrokers() {
        this.add('kafka:GetBootstrapBrokers');
        return this;
    }
    /**
     * Returns a list of the Apache Kafka versions to which you can update this cluster.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/compatible-kafka-versions.html#GetCompatibleKafkaVersions
     */
    getCompatibleKafkaVersions() {
        this.add('kafka:GetCompatibleKafkaVersions');
        return this;
    }
    /**
     * Returns a list of all the operations that have been performed on the specified MSK cluster.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-operations.html#ListClusterOperations
     */
    listClusterOperations() {
        this.add('kafka:ListClusterOperations');
        return this;
    }
    /**
     * Grants permission to return a list of all clusters in the current account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters.html#ListClusters
     */
    listClusters() {
        this.add('kafka:ListClusters');
        return this;
    }
    /**
     * Grants permission to return a list of all configurations in the current account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations.html#CreateConfiguration
     */
    listConfigurations() {
        this.add('kafka:ListConfigurations');
        return this;
    }
    /**
     * Grants permission to return a list of nodes in a cluster.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes.html#ListNodes
     */
    listNodes() {
        this.add('kafka:ListNodes');
        return this;
    }
    /**
     * Grants permission to list tags of a MSK resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#ListTagsForResource
     */
    listTagsForResource() {
        this.add('kafka:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to tag a MSK resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#TagResource
     */
    tagResource() {
        this.add('kafka:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from a MSK resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#UntagResource
     */
    untagResource() {
        this.add('kafka:UntagResource');
        return this;
    }
    /**
     * Updates the number of broker nodes of the cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes-count.html#UpdateBrokerCount
     */
    updateBrokerCount() {
        this.add('kafka:UpdateBrokerCount');
        return this;
    }
    /**
     * Updates the storage size of the broker nodes of the cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes-storage.html#UpdateBrokerStorage
     */
    updateBrokerStorage() {
        this.add('kafka:UpdateBrokerStorage');
        return this;
    }
    /**
     * Update Kafka configuration running on a cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-configuration.html#UpdateClusterConfiguration
     */
    updateClusterConfiguration() {
        this.add('kafka:UpdateClusterConfiguration');
        return this;
    }
    /**
     * Updates the cluster to the specified Apache Kafka version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-version.html#UpdateClusterKafkaVersion
     */
    updateClusterKafkaVersion() {
        this.add('kafka:UpdateClusterKafkaVersion');
        return this;
    }
    /**
     * Updates the monitoring settings for the cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-monitoring.html#UpdateMonitoring
     */
    updateMonitoring() {
        this.add('kafka:UpdateMonitoring');
        return this;
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * @param clusterName - Identifier for the clusterName.
     * @param uUID - Identifier for the uUID.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onCluster(clusterName, uUID, account, region, partition) {
        var arn = 'arn:${Partition}:kafka:${Region}:${Account}:cluster/${ClusterName}/${UUID}';
        arn = arn.replace('${ClusterName}', clusterName);
        arn = arn.replace('${UUID}', uUID);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Kafka = Kafka;
//# sourceMappingURL=data:application/json;base64,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