"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service neptune-db
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonneptune.html
 */
class NeptuneDb extends shared_1.PolicyStatement {
    /**
     * Action provider for service neptune-db
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonneptune.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'neptune-db';
        this.actions = {
            "connect": {
                "url": "https://docs.aws.amazon.com/neptune/latest/userguide/get-started.html",
                "description": "Connect to database",
                "accessLevel": "Write",
                "resourceTypes": {
                    "database": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "database": {
                "name": "database",
                "url": "https://docs.aws.amazon.com/neptune/latest/userguide/get-started.html",
                "arn": "arn:${Partition}:neptune-db:${Region}:${Account}:${Cluster}/${Database}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Connect to database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/neptune/latest/userguide/get-started.html
     */
    connect() {
        this.add('neptune-db:connect');
        return this;
    }
    /**
     * Adds a resource of type database to the statement
     *
     * https://docs.aws.amazon.com/neptune/latest/userguide/get-started.html
     *
     * @param cluster - Identifier for the cluster.
     * @param database - Identifier for the database.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDatabase(cluster, database, account, region, partition) {
        var arn = 'arn:${Partition}:neptune-db:${Region}:${Account}:${Cluster}/${Database}';
        arn = arn.replace('${Cluster}', cluster);
        arn = arn.replace('${Database}', database);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.NeptuneDb = NeptuneDb;
//# sourceMappingURL=data:application/json;base64,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