"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service polly
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonpolly.html
 */
class Polly extends shared_1.PolicyStatement {
    /**
     * Action provider for service polly
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonpolly.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'polly';
        this.actions = {
            "DeleteLexicon": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_DeleteLexicon.html",
                "description": "Deletes the specified pronunciation lexicon stored in an AWS Region",
                "accessLevel": "Write",
                "resourceTypes": {
                    "lexicon": {
                        "required": true
                    }
                }
            },
            "DescribeVoices": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_DescribeVoices.html",
                "description": "Returns the list of voices that are available for use when requesting speech synthesis.",
                "accessLevel": "List"
            },
            "GetLexicon": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_GetLexicon.html",
                "description": "Returns the content of the specified pronunciation lexicon stored in an AWS Region.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "lexicon": {
                        "required": true
                    }
                }
            },
            "GetSpeechSynthesisTask": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_GetSpeechSynthesisTask.html",
                "description": "Enables the user to get information about specific speech synthesis task.",
                "accessLevel": "Read"
            },
            "ListLexicons": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_ListLexicons.html",
                "description": "Returns a list of pronunciation lexicons stored in an AWS Region.",
                "accessLevel": "List"
            },
            "ListSpeechSynthesisTasks": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_ListSpeechSynthesisTasks.html",
                "description": "Enables the user to list requested speech synthesis tasks.",
                "accessLevel": "List"
            },
            "PutLexicon": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_PutLexicon.html",
                "description": "Stores a pronunciation lexicon in an AWS Region.",
                "accessLevel": "Write"
            },
            "StartSpeechSynthesisTask": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_StartSpeechSynthesisTask.html",
                "description": "Enables the user to synthesize long inputs to provided S3 location.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "lexicon": {
                        "required": false
                    }
                }
            },
            "SynthesizeSpeech": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_SynthesizeSpeech.html",
                "description": "Synthesizes UTF-8 input, plain text or SSML, to a stream of bytes.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "lexicon": {
                        "required": false
                    }
                }
            }
        };
        this.resourceTypes = {
            "lexicon": {
                "name": "lexicon",
                "url": "https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html",
                "arn": "arn:${Partition}:polly:${Region}:${Account}:lexicon/${LexiconName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Deletes the specified pronunciation lexicon stored in an AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_DeleteLexicon.html
     */
    deleteLexicon() {
        this.add('polly:DeleteLexicon');
        return this;
    }
    /**
     * Returns the list of voices that are available for use when requesting speech synthesis.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_DescribeVoices.html
     */
    describeVoices() {
        this.add('polly:DescribeVoices');
        return this;
    }
    /**
     * Returns the content of the specified pronunciation lexicon stored in an AWS Region.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_GetLexicon.html
     */
    getLexicon() {
        this.add('polly:GetLexicon');
        return this;
    }
    /**
     * Enables the user to get information about specific speech synthesis task.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_GetSpeechSynthesisTask.html
     */
    getSpeechSynthesisTask() {
        this.add('polly:GetSpeechSynthesisTask');
        return this;
    }
    /**
     * Returns a list of pronunciation lexicons stored in an AWS Region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_ListLexicons.html
     */
    listLexicons() {
        this.add('polly:ListLexicons');
        return this;
    }
    /**
     * Enables the user to list requested speech synthesis tasks.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_ListSpeechSynthesisTasks.html
     */
    listSpeechSynthesisTasks() {
        this.add('polly:ListSpeechSynthesisTasks');
        return this;
    }
    /**
     * Stores a pronunciation lexicon in an AWS Region.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_PutLexicon.html
     */
    putLexicon() {
        this.add('polly:PutLexicon');
        return this;
    }
    /**
     * Enables the user to synthesize long inputs to provided S3 location.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_StartSpeechSynthesisTask.html
     */
    startSpeechSynthesisTask() {
        this.add('polly:StartSpeechSynthesisTask');
        return this;
    }
    /**
     * Synthesizes UTF-8 input, plain text or SSML, to a stream of bytes.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_SynthesizeSpeech.html
     */
    synthesizeSpeech() {
        this.add('polly:SynthesizeSpeech');
        return this;
    }
    /**
     * Adds a resource of type lexicon to the statement
     *
     * https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html
     *
     * @param lexiconName - Identifier for the lexiconName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLexicon(lexiconName, account, region, partition) {
        var arn = 'arn:${Partition}:polly:${Region}:${Account}:lexicon/${LexiconName}';
        arn = arn.replace('${LexiconName}', lexiconName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Polly = Polly;
//# sourceMappingURL=data:application/json;base64,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