"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service quicksight
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonquicksight.html
 */
class Quicksight extends shared_1.PolicyStatement {
    /**
     * Action provider for service quicksight
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonquicksight.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'quicksight';
        this.actions = {
            "CreateAdmin": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html",
                "description": "CreateAdmin enables the user to provision Amazon QuickSight administrators, authors, and readers.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "CreateDashboard": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateDashboard.html",
                "description": "Creates a dashboard from a template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateGroup": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateGroup.html",
                "description": "Create a QuickSight group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "CreateGroupMembership": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateGroupMembership.html",
                "description": "Add a QuickSight user to a QuickSight group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true,
                        "conditions": [
                            "quicksight:UserName"
                        ]
                    }
                }
            },
            "CreateIAMPolicyAssignment": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateIAMPolicyAssignment.html",
                "description": "Creates an assignment with one specified IAM Policy ARN that will be assigned to specified groups or users of QuickSight.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "assignment": {
                        "required": true
                    }
                }
            },
            "CreateReader": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html",
                "description": "CreateReader enables the user to provision Amazon QuickSight readers.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "CreateTemplate": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateTemplate.html",
                "description": "Creates a template from an existing QuickSight analysis or template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateTemplateAlias": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateTemplateAlias.html",
                "description": "Creates a template alias for a template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "CreateUser": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html",
                "description": "CreateUser enables the user to provision Amazon QuickSight authors and readers.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DeleteDashboard": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteDashboard.html",
                "description": "Deletes a dashboard",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                }
            },
            "DeleteGroup": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteGroup.html",
                "description": "Remove a user group from QuickSight.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "DeleteGroupMembership": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteGroupMembership.html",
                "description": "Remove a user from a group so that he/she is no longer a member of the group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true,
                        "conditions": [
                            "quicksight:UserName"
                        ]
                    }
                }
            },
            "DeleteIAMPolicyAssignment": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteIAMPolicyAssignment.html",
                "description": "Update an existing assignment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "assignment": {
                        "required": true
                    }
                }
            },
            "DeleteTemplate": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteTemplate.html",
                "description": "Deletes a template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "DeleteTemplateAlias": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteTemplateAlias.html",
                "description": "Deletes the item that the specified template alias points to",
                "accessLevel": "Write",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "DeleteUser": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteUser.html",
                "description": "Delete the QuickSight user that is associated with the identity of the IAM user/role making the call. The IAM user is not deleted as a result of this call.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DeleteUserByPrincipalId": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteUserByPrincipalId.html",
                "description": "Deletes a user identified by its principal ID.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DescribeDashboard": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDashboard.html",
                "description": "Provides a summary for a dashboard",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                }
            },
            "DescribeDashboardPermissions": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDashboardPermissions.html",
                "description": "Describes read and write permissions for a dashboard",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                }
            },
            "DescribeGroup": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeGroup.html",
                "description": "Return a QuickSight group’s description and ARN.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "DescribeIAMPolicyAssignment": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeIAMPolicyAssignment.html",
                "description": "Describe an existing assignment.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "assignment": {
                        "required": true
                    }
                }
            },
            "DescribeTemplate": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTemplate.html",
                "description": "Describes a template's metadata",
                "accessLevel": "Read",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "DescribeTemplateAlias": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTemplateAlias.html",
                "description": "Describes the template alias for a template",
                "accessLevel": "Read",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "DescribeTemplatePermissions": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTemplatePermissions.html",
                "description": "Describes read and write permissions on a template",
                "accessLevel": "Read",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "DescribeUser": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeUser.html",
                "description": "Return information about a user, given the user name.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "GetAuthCode": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html",
                "description": "Return an auth code representing a QuickSight user.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "GetDashboardEmbedUrl": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_GetDashboardEmbedUrl.html",
                "description": "Return a QuickSight dashboard embedding URL.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                }
            },
            "GetGroupMapping": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html",
                "description": "GetGroupMapping is used only in Amazon QuickSight Enterprise edition accounts. It enables the user to use Amazon QuickSight to identify and display the Microsoft Active Directory (Microsoft Active Directory) directory groups that are mapped to roles in Amazon QuickSight.",
                "accessLevel": "Read"
            },
            "ListDashboardVersions": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListDashboardVersions.html",
                "description": "Lists all the versions of the dashboards in the QuickSight subscription",
                "accessLevel": "List",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                }
            },
            "ListDashboards": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListDashboards.html",
                "description": "Lists dashboards in an AWS account",
                "accessLevel": "List",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                }
            },
            "ListGroupMemberships": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListGroupMemberships.html",
                "description": "Return a list of member users in a group.",
                "accessLevel": "List",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "ListGroups": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListGroups.html",
                "description": "Get a list of all user groups in QuickSight.",
                "accessLevel": "List",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "ListIAMPolicyAssignments": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListIAMPolicyAssignments.html",
                "description": "List all assignments in the current Amazon QuickSight account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "assignment": {
                        "required": true
                    }
                }
            },
            "ListIAMPolicyAssignmentsForUser": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListIAMPolicyAssignmentsForUser.html",
                "description": "List all assignments assigned to a user and the groups it belongs",
                "accessLevel": "List",
                "resourceTypes": {
                    "assignment": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTagsForResource.html",
                "description": "List tags of a QuickSight resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "dashboard": {
                        "required": false
                    },
                    "template": {
                        "required": false
                    }
                }
            },
            "ListTemplateAliases": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTemplateAliases.html",
                "description": "Lists all the aliases of a template",
                "accessLevel": "List",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "ListTemplateVersions": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTemplateVersions.html",
                "description": "Lists all the versions of the templates in the current Amazon QuickSight account",
                "accessLevel": "List",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "ListTemplates": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTemplates.html",
                "description": "Lists all the templates in the current Amazon QuickSight account",
                "accessLevel": "List",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "ListUserGroups": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListUserGroups.html",
                "description": "Return a list of groups that a given user is a member of.",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ListUsers": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListUsers.html",
                "description": "Return a list of all of the QuickSight users belonging to this account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "RegisterUser": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_RegisterUser.html",
                "description": "Create a QuickSight user, whose identity is associated with the IAM identity/role specified in the request.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true,
                        "conditions": [
                            "quicksight:IamArn",
                            "quicksight:SessionName"
                        ]
                    }
                }
            },
            "SearchDirectoryGroups": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html",
                "description": "SearchDirectoryGroups is used only in Amazon QuickSight Enterprise edition accounts. It enables the user to use Amazon QuickSight to display your Microsoft Active Directory directory groups so that you can choose which ones to map to roles in Amazon QuickSight.",
                "accessLevel": "Write"
            },
            "SetGroupMapping": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html",
                "description": "SearchDirectoryGroups is used only in Amazon QuickSight Enterprise edition accounts. It enables the user to use Amazon QuickSight to display your Microsoft Active Directory directory groups so that you can choose which ones to map to roles in Amazon QuickSight.",
                "accessLevel": "Write"
            },
            "Subscribe": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html",
                "description": "Subscribe enables the user to subscribe to Amazon QuickSight. Enabling this action also allows the user to upgrade the subscription to Enterprise edition.",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_TagResource.html",
                "description": "Add tags to a QuickSight resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "dashboard": {
                        "required": false
                    },
                    "template": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "Unsubscribe": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html",
                "description": "Unsubscribe enables the user to unsubscribe from Amazon QuickSight, which permanently deletes all users and their resources from Amazon QuickSight.",
                "accessLevel": "Write"
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UntagResource.html",
                "description": "Remove tags from a QuickSight resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "dashboard": {
                        "required": false
                    },
                    "template": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateDashboard": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboard.html",
                "description": "Updates a dashboard in an AWS account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                }
            },
            "UpdateDashboardPermissions": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboardPermissions.html",
                "description": "Updates read and write permissions on a dashboard",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                }
            },
            "UpdateDashboardPublishedVersion": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboardPublishedVersion.html",
                "description": "Updates the published version of a dashboard",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                }
            },
            "UpdateGroup": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateGroup.html",
                "description": "Change group description.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "UpdateIAMPolicyAssignment": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateIAMPolicyAssignment.html",
                "description": "Update an existing assignment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "assignment": {
                        "required": true
                    }
                }
            },
            "UpdateTemplate": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTemplate.html",
                "description": "Updates a template from an existing Amazon QuickSight analysis or another template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "UpdateTemplateAlias": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTemplateAlias.html",
                "description": "Updates the template alias of a template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "UpdateTemplatePermissions": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTemplatePermissions.html",
                "description": "Updates the resource permissions for a template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "UpdateUser": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateUser.html",
                "description": "Updates an Amazon QuickSight user.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "user": {
                "name": "user",
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_User.html",
                "arn": "arn:${Partition}:quicksight:${Region}:${Account}:user/${ResourceId}",
                "conditionKeys": []
            },
            "group": {
                "name": "group",
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Group.html",
                "arn": "arn:${Partition}:quicksight:${Region}:${Account}:group/${ResourceId}",
                "conditionKeys": []
            },
            "dashboard": {
                "name": "dashboard",
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Dashboard.html",
                "arn": "arn:${Partition}:quicksight:${Region}:${Account}:dashboard/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "template": {
                "name": "template",
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Template.html",
                "arn": "arn:${Partition}:quicksight:${Region}:${Account}:template/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "assignment": {
                "name": "assignment",
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_IAMPolicyAssignment.html",
                "arn": "arn:${Partition}:quicksight::${Account}:assignment/${ResourceId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * CreateAdmin enables the user to provision Amazon QuickSight administrators, authors, and readers.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    createAdmin() {
        this.add('quicksight:CreateAdmin');
        return this;
    }
    /**
     * Creates a dashboard from a template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateDashboard.html
     */
    createDashboard() {
        this.add('quicksight:CreateDashboard');
        return this;
    }
    /**
     * Create a QuickSight group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateGroup.html
     */
    createGroup() {
        this.add('quicksight:CreateGroup');
        return this;
    }
    /**
     * Add a QuickSight user to a QuickSight group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateGroupMembership.html
     */
    createGroupMembership() {
        this.add('quicksight:CreateGroupMembership');
        return this;
    }
    /**
     * Creates an assignment with one specified IAM Policy ARN that will be assigned to specified groups or users of QuickSight.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateIAMPolicyAssignment.html
     */
    createIAMPolicyAssignment() {
        this.add('quicksight:CreateIAMPolicyAssignment');
        return this;
    }
    /**
     * CreateReader enables the user to provision Amazon QuickSight readers.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    createReader() {
        this.add('quicksight:CreateReader');
        return this;
    }
    /**
     * Creates a template from an existing QuickSight analysis or template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateTemplate.html
     */
    createTemplate() {
        this.add('quicksight:CreateTemplate');
        return this;
    }
    /**
     * Creates a template alias for a template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateTemplateAlias.html
     */
    createTemplateAlias() {
        this.add('quicksight:CreateTemplateAlias');
        return this;
    }
    /**
     * CreateUser enables the user to provision Amazon QuickSight authors and readers.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    createUser() {
        this.add('quicksight:CreateUser');
        return this;
    }
    /**
     * Deletes a dashboard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteDashboard.html
     */
    deleteDashboard() {
        this.add('quicksight:DeleteDashboard');
        return this;
    }
    /**
     * Remove a user group from QuickSight.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteGroup.html
     */
    deleteGroup() {
        this.add('quicksight:DeleteGroup');
        return this;
    }
    /**
     * Remove a user from a group so that he/she is no longer a member of the group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteGroupMembership.html
     */
    deleteGroupMembership() {
        this.add('quicksight:DeleteGroupMembership');
        return this;
    }
    /**
     * Update an existing assignment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteIAMPolicyAssignment.html
     */
    deleteIAMPolicyAssignment() {
        this.add('quicksight:DeleteIAMPolicyAssignment');
        return this;
    }
    /**
     * Deletes a template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteTemplate.html
     */
    deleteTemplate() {
        this.add('quicksight:DeleteTemplate');
        return this;
    }
    /**
     * Deletes the item that the specified template alias points to
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteTemplateAlias.html
     */
    deleteTemplateAlias() {
        this.add('quicksight:DeleteTemplateAlias');
        return this;
    }
    /**
     * Delete the QuickSight user that is associated with the identity of the IAM user/role making the call. The IAM user is not deleted as a result of this call.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteUser.html
     */
    deleteUser() {
        this.add('quicksight:DeleteUser');
        return this;
    }
    /**
     * Deletes a user identified by its principal ID.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteUserByPrincipalId.html
     */
    deleteUserByPrincipalId() {
        this.add('quicksight:DeleteUserByPrincipalId');
        return this;
    }
    /**
     * Provides a summary for a dashboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDashboard.html
     */
    describeDashboard() {
        this.add('quicksight:DescribeDashboard');
        return this;
    }
    /**
     * Describes read and write permissions for a dashboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDashboardPermissions.html
     */
    describeDashboardPermissions() {
        this.add('quicksight:DescribeDashboardPermissions');
        return this;
    }
    /**
     * Return a QuickSight group’s description and ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeGroup.html
     */
    describeGroup() {
        this.add('quicksight:DescribeGroup');
        return this;
    }
    /**
     * Describe an existing assignment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeIAMPolicyAssignment.html
     */
    describeIAMPolicyAssignment() {
        this.add('quicksight:DescribeIAMPolicyAssignment');
        return this;
    }
    /**
     * Describes a template's metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTemplate.html
     */
    describeTemplate() {
        this.add('quicksight:DescribeTemplate');
        return this;
    }
    /**
     * Describes the template alias for a template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTemplateAlias.html
     */
    describeTemplateAlias() {
        this.add('quicksight:DescribeTemplateAlias');
        return this;
    }
    /**
     * Describes read and write permissions on a template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTemplatePermissions.html
     */
    describeTemplatePermissions() {
        this.add('quicksight:DescribeTemplatePermissions');
        return this;
    }
    /**
     * Return information about a user, given the user name.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeUser.html
     */
    describeUser() {
        this.add('quicksight:DescribeUser');
        return this;
    }
    /**
     * Return an auth code representing a QuickSight user.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    getAuthCode() {
        this.add('quicksight:GetAuthCode');
        return this;
    }
    /**
     * Return a QuickSight dashboard embedding URL.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_GetDashboardEmbedUrl.html
     */
    getDashboardEmbedUrl() {
        this.add('quicksight:GetDashboardEmbedUrl');
        return this;
    }
    /**
     * GetGroupMapping is used only in Amazon QuickSight Enterprise edition accounts. It enables the user to use Amazon QuickSight to identify and display the Microsoft Active Directory (Microsoft Active Directory) directory groups that are mapped to roles in Amazon QuickSight.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    getGroupMapping() {
        this.add('quicksight:GetGroupMapping');
        return this;
    }
    /**
     * Lists all the versions of the dashboards in the QuickSight subscription
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListDashboardVersions.html
     */
    listDashboardVersions() {
        this.add('quicksight:ListDashboardVersions');
        return this;
    }
    /**
     * Lists dashboards in an AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListDashboards.html
     */
    listDashboards() {
        this.add('quicksight:ListDashboards');
        return this;
    }
    /**
     * Return a list of member users in a group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListGroupMemberships.html
     */
    listGroupMemberships() {
        this.add('quicksight:ListGroupMemberships');
        return this;
    }
    /**
     * Get a list of all user groups in QuickSight.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListGroups.html
     */
    listGroups() {
        this.add('quicksight:ListGroups');
        return this;
    }
    /**
     * List all assignments in the current Amazon QuickSight account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListIAMPolicyAssignments.html
     */
    listIAMPolicyAssignments() {
        this.add('quicksight:ListIAMPolicyAssignments');
        return this;
    }
    /**
     * List all assignments assigned to a user and the groups it belongs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListIAMPolicyAssignmentsForUser.html
     */
    listIAMPolicyAssignmentsForUser() {
        this.add('quicksight:ListIAMPolicyAssignmentsForUser');
        return this;
    }
    /**
     * List tags of a QuickSight resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('quicksight:ListTagsForResource');
        return this;
    }
    /**
     * Lists all the aliases of a template
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTemplateAliases.html
     */
    listTemplateAliases() {
        this.add('quicksight:ListTemplateAliases');
        return this;
    }
    /**
     * Lists all the versions of the templates in the current Amazon QuickSight account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTemplateVersions.html
     */
    listTemplateVersions() {
        this.add('quicksight:ListTemplateVersions');
        return this;
    }
    /**
     * Lists all the templates in the current Amazon QuickSight account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTemplates.html
     */
    listTemplates() {
        this.add('quicksight:ListTemplates');
        return this;
    }
    /**
     * Return a list of groups that a given user is a member of.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListUserGroups.html
     */
    listUserGroups() {
        this.add('quicksight:ListUserGroups');
        return this;
    }
    /**
     * Return a list of all of the QuickSight users belonging to this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListUsers.html
     */
    listUsers() {
        this.add('quicksight:ListUsers');
        return this;
    }
    /**
     * Create a QuickSight user, whose identity is associated with the IAM identity/role specified in the request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_RegisterUser.html
     */
    registerUser() {
        this.add('quicksight:RegisterUser');
        return this;
    }
    /**
     * SearchDirectoryGroups is used only in Amazon QuickSight Enterprise edition accounts. It enables the user to use Amazon QuickSight to display your Microsoft Active Directory directory groups so that you can choose which ones to map to roles in Amazon QuickSight.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    searchDirectoryGroups() {
        this.add('quicksight:SearchDirectoryGroups');
        return this;
    }
    /**
     * SearchDirectoryGroups is used only in Amazon QuickSight Enterprise edition accounts. It enables the user to use Amazon QuickSight to display your Microsoft Active Directory directory groups so that you can choose which ones to map to roles in Amazon QuickSight.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    setGroupMapping() {
        this.add('quicksight:SetGroupMapping');
        return this;
    }
    /**
     * Subscribe enables the user to subscribe to Amazon QuickSight. Enabling this action also allows the user to upgrade the subscription to Enterprise edition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    subscribe() {
        this.add('quicksight:Subscribe');
        return this;
    }
    /**
     * Add tags to a QuickSight resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('quicksight:TagResource');
        return this;
    }
    /**
     * Unsubscribe enables the user to unsubscribe from Amazon QuickSight, which permanently deletes all users and their resources from Amazon QuickSight.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    unsubscribe() {
        this.add('quicksight:Unsubscribe');
        return this;
    }
    /**
     * Remove tags from a QuickSight resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('quicksight:UntagResource');
        return this;
    }
    /**
     * Updates a dashboard in an AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboard.html
     */
    updateDashboard() {
        this.add('quicksight:UpdateDashboard');
        return this;
    }
    /**
     * Updates read and write permissions on a dashboard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboardPermissions.html
     */
    updateDashboardPermissions() {
        this.add('quicksight:UpdateDashboardPermissions');
        return this;
    }
    /**
     * Updates the published version of a dashboard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboardPublishedVersion.html
     */
    updateDashboardPublishedVersion() {
        this.add('quicksight:UpdateDashboardPublishedVersion');
        return this;
    }
    /**
     * Change group description.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateGroup.html
     */
    updateGroup() {
        this.add('quicksight:UpdateGroup');
        return this;
    }
    /**
     * Update an existing assignment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateIAMPolicyAssignment.html
     */
    updateIAMPolicyAssignment() {
        this.add('quicksight:UpdateIAMPolicyAssignment');
        return this;
    }
    /**
     * Updates a template from an existing Amazon QuickSight analysis or another template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTemplate.html
     */
    updateTemplate() {
        this.add('quicksight:UpdateTemplate');
        return this;
    }
    /**
     * Updates the template alias of a template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTemplateAlias.html
     */
    updateTemplateAlias() {
        this.add('quicksight:UpdateTemplateAlias');
        return this;
    }
    /**
     * Updates the resource permissions for a template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTemplatePermissions.html
     */
    updateTemplatePermissions() {
        this.add('quicksight:UpdateTemplatePermissions');
        return this;
    }
    /**
     * Updates an Amazon QuickSight user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateUser.html
     */
    updateUser() {
        this.add('quicksight:UpdateUser');
        return this;
    }
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_User.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onUser(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:quicksight:${Region}:${Account}:user/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type group to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Group.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGroup(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:quicksight:${Region}:${Account}:group/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dashboard to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Dashboard.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDashboard(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:quicksight:${Region}:${Account}:dashboard/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type template to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Template.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onTemplate(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:quicksight:${Region}:${Account}:template/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type assignment to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_IAMPolicyAssignment.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAssignment(resourceId, account, partition) {
        var arn = 'arn:${Partition}:quicksight::${Account}:assignment/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * IAM user ARN or role ARN.
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIamArn(value, operator) {
        return this.if(`quicksight:IamArn`, value, operator || 'StringLike');
    }
    /**
     * The session name.
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSessionName(value, operator) {
        return this.if(`quicksight:SessionName`, value, operator || 'StringLike');
    }
    /**
     * The user name.
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserName(value, operator) {
        return this.if(`quicksight:UserName`, value, operator || 'StringLike');
    }
}
exports.Quicksight = Quicksight;
//# sourceMappingURL=data:application/json;base64,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