"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service swf
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsimpleworkflowservice.html
 */
class Swf extends shared_1.PolicyStatement {
    /**
     * Action provider for service swf
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsimpleworkflowservice.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'swf';
        this.actions = {
            "CancelTimer": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CancelTimer.html",
                "description": "Description for CancelTimer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "CancelWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CancelWorkflowExecution.html",
                "description": "Description for CancelWorkflowExecution",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "CompleteWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CompleteWorkflowExecution.html",
                "description": "Description for CompleteWorkflowExecution",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "ContinueAsNewWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ContinueAsNewWorkflowExecution.html",
                "description": "Description for ContinueAsNewWorkflowExecution",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "CountClosedWorkflowExecutions": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountClosedWorkflowExecutions.html",
                "description": "Returns the number of closed workflow executions within the given domain that meet the specified filtering criteria.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:tagFilter.tag",
                    "swf:typeFilter.name",
                    "swf:typeFilter.version"
                ]
            },
            "CountOpenWorkflowExecutions": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountOpenWorkflowExecutions.html",
                "description": "Returns the number of open workflow executions within the given domain that meet the specified filtering criteria.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:tagFilter.tag",
                    "swf:typeFilter.name",
                    "swf:typeFilter.version"
                ]
            },
            "CountPendingActivityTasks": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountPendingActivityTasks.html",
                "description": "Returns the estimated number of activity tasks in the specified task list.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:taskList.name"
                ]
            },
            "CountPendingDecisionTasks": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountPendingDecisionTasks.html",
                "description": "Returns the estimated number of decision tasks in the specified task list.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:taskList.name"
                ]
            },
            "DeprecateActivityType": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateActivityType.html",
                "description": "Deprecates the specified activity type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:activityType.name",
                    "swf:activityType.version"
                ]
            },
            "DeprecateDomain": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateDomain.html",
                "description": "Deprecates the specified domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeprecateWorkflowType": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateWorkflowType.html",
                "description": "Deprecates the specified workflow type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:workflowType.name",
                    "swf:workflowType.version"
                ]
            },
            "DescribeActivityType": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeActivityType.html",
                "description": "Returns information about the specified activity type.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:activityType.name",
                    "swf:activityType.version"
                ]
            },
            "DescribeDomain": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeDomain.html",
                "description": "Returns information about the specified domain, including description and status.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeWorkflowExecution.html",
                "description": "Returns information about the specified workflow execution including its type and some statistics.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeWorkflowType": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeWorkflowType.html",
                "description": "Returns information about the specified workflow type.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:workflowType.name",
                    "swf:workflowType.version"
                ]
            },
            "FailWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_FailWorkflowExecution.html",
                "description": "Description for FailWorkflowExecution",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "GetWorkflowExecutionHistory": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_GetWorkflowExecutionHistory.html",
                "description": "Returns the history of the specified workflow execution.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "ListActivityTypes": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListActivityTypes.html",
                "description": "Returns information about all activities registered in the specified domain that match the specified name and registration status.",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "ListClosedWorkflowExecutions": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListClosedWorkflowExecutions.html",
                "description": "Returns a list of closed workflow executions in the specified domain that meet the filtering criteria.",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:tagFilter.tag",
                    "swf:typeFilter.name",
                    "swf:typeFilter.version"
                ]
            },
            "ListDomains": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListDomains.html",
                "description": "Returns the list of domains registered in the account.",
                "accessLevel": "List"
            },
            "ListOpenWorkflowExecutions": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListOpenWorkflowExecutions.html",
                "description": "Returns a list of open workflow executions in the specified domain that meet the filtering criteria.",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:tagFilter.tag",
                    "swf:typeFilter.name",
                    "swf:typeFilter.version"
                ]
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListTagsForResource.html",
                "description": "This action lists tags for an AWS SWF resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "ListWorkflowTypes": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListWorkflowTypes.html",
                "description": "Returns information about workflow types in the specified domain.",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "PollForActivityTask": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_PollForActivityTask.html",
                "description": "Used by workers to get an ActivityTask from the specified activity taskList.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:taskList.name"
                ]
            },
            "PollForDecisionTask": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_PollForDecisionTask.html",
                "description": "Used by deciders to get a DecisionTask from the specified decision taskList.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:taskList.name"
                ]
            },
            "RecordActivityTaskHeartbeat": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RecordActivityTaskHeartbeat.html",
                "description": "Used by activity workers to report to the service that the ActivityTask represented by the specified taskToken is still making progress.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RecordMarker": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RecordMarker.html",
                "description": "Description for RecordMarker",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RegisterActivityType": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterActivityType.html",
                "description": "Registers a new activity type along with its configuration settings in the specified domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:defaultTaskList.name",
                    "swf:name",
                    "swf:version"
                ]
            },
            "RegisterDomain": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterDomain.html",
                "description": "Registers a new domain.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "RegisterWorkflowType": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterWorkflowType.html",
                "description": "Registers a new workflow type and its configuration settings in the specified domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:defaultTaskList.name",
                    "swf:name",
                    "swf:version"
                ]
            },
            "RequestCancelActivityTask": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RequestCancelActivityTask.html",
                "description": "Description for RequestCancelActivityTask",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RequestCancelExternalWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RequestCancelExternalWorkflowExecution.html",
                "description": "Description for RequestCancelExternalWorkflowExecution",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RequestCancelWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RequestCancelWorkflowExecution.html",
                "description": "Records a WorkflowExecutionCancelRequested event in the currently running workflow execution identified by the given domain, workflowId, and runId.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RespondActivityTaskCanceled": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskCanceled.html",
                "description": "Used by workers to tell the service that the ActivityTask identified by the taskToken was successfully canceled.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RespondActivityTaskCompleted": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskCompleted.html",
                "description": "Used by workers to tell the service that the ActivityTask identified by the taskToken completed successfully with a result (if provided).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:activityType.name",
                    "swf:activityType.version",
                    "swf:tagList.member.0",
                    "swf:tagList.member.1",
                    "swf:tagList.member.2",
                    "swf:tagList.member.3",
                    "swf:tagList.member.4",
                    "swf:taskList.name",
                    "swf:workflowType.name",
                    "swf:workflowType.version"
                ]
            },
            "RespondActivityTaskFailed": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskFailed.html",
                "description": "Used by workers to tell the service that the ActivityTask identified by the taskToken has failed with reason (if specified).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RespondDecisionTaskCompleted": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondDecisionTaskCompleted.html",
                "description": "Used by deciders to tell the service that the DecisionTask identified by the taskToken has successfully completed.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "ScheduleActivityTask": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ScheduleActivityTask.html",
                "description": "Description for ScheduleActivityTask",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "SignalExternalWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_SignalExternalWorkflowExecution.html",
                "description": "Description for SignalExternalWorkflowExecution",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "SignalWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_SignalWorkflowExecution.html",
                "description": "Records a WorkflowExecutionSignaled event in the workflow execution history and creates a decision task for the workflow execution identified by the given domain, workflowId and runId.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "StartChildWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_StartChildWorkflowExecution.html",
                "description": "Description for StartChildWorkflowExecution",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "StartTimer": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_StartTimer.html",
                "description": "Description for StartTimer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "StartWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_StartWorkflowExecution.html",
                "description": "Starts an execution of the workflow type in the specified domain using the provided workflowId and input data.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:tagList.member.0",
                    "swf:tagList.member.1",
                    "swf:tagList.member.2",
                    "swf:tagList.member.3",
                    "swf:tagList.member.4",
                    "swf:taskList.name",
                    "swf:workflowType.name",
                    "swf:workflowType.version"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_TagResource.html",
                "description": "This action tags an AWS SWF resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "TerminateWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_TerminateWorkflowExecution.html",
                "description": "Records a WorkflowExecutionTerminated event and forces closure of the workflow execution identified by the given domain, runId, and workflowId.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_UntagResource.html",
                "description": "This action removes a tag from an AWS SWF resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            }
        };
        this.resourceTypes = {
            "domain": {
                "name": "domain",
                "url": "https://docs.aws.amazon.com/swf/latest/developerguide/swf-dev-domains.html",
                "arn": "arn:${Partition}:swf::${Account}:domain/${DomainName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Description for CancelTimer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CancelTimer.html
     */
    cancelTimer() {
        this.add('swf:CancelTimer');
        return this;
    }
    /**
     * Description for CancelWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CancelWorkflowExecution.html
     */
    cancelWorkflowExecution() {
        this.add('swf:CancelWorkflowExecution');
        return this;
    }
    /**
     * Description for CompleteWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CompleteWorkflowExecution.html
     */
    completeWorkflowExecution() {
        this.add('swf:CompleteWorkflowExecution');
        return this;
    }
    /**
     * Description for ContinueAsNewWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ContinueAsNewWorkflowExecution.html
     */
    continueAsNewWorkflowExecution() {
        this.add('swf:ContinueAsNewWorkflowExecution');
        return this;
    }
    /**
     * Returns the number of closed workflow executions within the given domain that meet the specified filtering criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountClosedWorkflowExecutions.html
     */
    countClosedWorkflowExecutions() {
        this.add('swf:CountClosedWorkflowExecutions');
        return this;
    }
    /**
     * Returns the number of open workflow executions within the given domain that meet the specified filtering criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountOpenWorkflowExecutions.html
     */
    countOpenWorkflowExecutions() {
        this.add('swf:CountOpenWorkflowExecutions');
        return this;
    }
    /**
     * Returns the estimated number of activity tasks in the specified task list.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountPendingActivityTasks.html
     */
    countPendingActivityTasks() {
        this.add('swf:CountPendingActivityTasks');
        return this;
    }
    /**
     * Returns the estimated number of decision tasks in the specified task list.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountPendingDecisionTasks.html
     */
    countPendingDecisionTasks() {
        this.add('swf:CountPendingDecisionTasks');
        return this;
    }
    /**
     * Deprecates the specified activity type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateActivityType.html
     */
    deprecateActivityType() {
        this.add('swf:DeprecateActivityType');
        return this;
    }
    /**
     * Deprecates the specified domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateDomain.html
     */
    deprecateDomain() {
        this.add('swf:DeprecateDomain');
        return this;
    }
    /**
     * Deprecates the specified workflow type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateWorkflowType.html
     */
    deprecateWorkflowType() {
        this.add('swf:DeprecateWorkflowType');
        return this;
    }
    /**
     * Returns information about the specified activity type.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeActivityType.html
     */
    describeActivityType() {
        this.add('swf:DescribeActivityType');
        return this;
    }
    /**
     * Returns information about the specified domain, including description and status.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeDomain.html
     */
    describeDomain() {
        this.add('swf:DescribeDomain');
        return this;
    }
    /**
     * Returns information about the specified workflow execution including its type and some statistics.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeWorkflowExecution.html
     */
    describeWorkflowExecution() {
        this.add('swf:DescribeWorkflowExecution');
        return this;
    }
    /**
     * Returns information about the specified workflow type.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeWorkflowType.html
     */
    describeWorkflowType() {
        this.add('swf:DescribeWorkflowType');
        return this;
    }
    /**
     * Description for FailWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_FailWorkflowExecution.html
     */
    failWorkflowExecution() {
        this.add('swf:FailWorkflowExecution');
        return this;
    }
    /**
     * Returns the history of the specified workflow execution.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_GetWorkflowExecutionHistory.html
     */
    getWorkflowExecutionHistory() {
        this.add('swf:GetWorkflowExecutionHistory');
        return this;
    }
    /**
     * Returns information about all activities registered in the specified domain that match the specified name and registration status.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListActivityTypes.html
     */
    listActivityTypes() {
        this.add('swf:ListActivityTypes');
        return this;
    }
    /**
     * Returns a list of closed workflow executions in the specified domain that meet the filtering criteria.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListClosedWorkflowExecutions.html
     */
    listClosedWorkflowExecutions() {
        this.add('swf:ListClosedWorkflowExecutions');
        return this;
    }
    /**
     * Returns the list of domains registered in the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListDomains.html
     */
    listDomains() {
        this.add('swf:ListDomains');
        return this;
    }
    /**
     * Returns a list of open workflow executions in the specified domain that meet the filtering criteria.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListOpenWorkflowExecutions.html
     */
    listOpenWorkflowExecutions() {
        this.add('swf:ListOpenWorkflowExecutions');
        return this;
    }
    /**
     * This action lists tags for an AWS SWF resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('swf:ListTagsForResource');
        return this;
    }
    /**
     * Returns information about workflow types in the specified domain.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListWorkflowTypes.html
     */
    listWorkflowTypes() {
        this.add('swf:ListWorkflowTypes');
        return this;
    }
    /**
     * Used by workers to get an ActivityTask from the specified activity taskList.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_PollForActivityTask.html
     */
    pollForActivityTask() {
        this.add('swf:PollForActivityTask');
        return this;
    }
    /**
     * Used by deciders to get a DecisionTask from the specified decision taskList.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_PollForDecisionTask.html
     */
    pollForDecisionTask() {
        this.add('swf:PollForDecisionTask');
        return this;
    }
    /**
     * Used by activity workers to report to the service that the ActivityTask represented by the specified taskToken is still making progress.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RecordActivityTaskHeartbeat.html
     */
    recordActivityTaskHeartbeat() {
        this.add('swf:RecordActivityTaskHeartbeat');
        return this;
    }
    /**
     * Description for RecordMarker
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RecordMarker.html
     */
    recordMarker() {
        this.add('swf:RecordMarker');
        return this;
    }
    /**
     * Registers a new activity type along with its configuration settings in the specified domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterActivityType.html
     */
    registerActivityType() {
        this.add('swf:RegisterActivityType');
        return this;
    }
    /**
     * Registers a new domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterDomain.html
     */
    registerDomain() {
        this.add('swf:RegisterDomain');
        return this;
    }
    /**
     * Registers a new workflow type and its configuration settings in the specified domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterWorkflowType.html
     */
    registerWorkflowType() {
        this.add('swf:RegisterWorkflowType');
        return this;
    }
    /**
     * Description for RequestCancelActivityTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RequestCancelActivityTask.html
     */
    requestCancelActivityTask() {
        this.add('swf:RequestCancelActivityTask');
        return this;
    }
    /**
     * Description for RequestCancelExternalWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RequestCancelExternalWorkflowExecution.html
     */
    requestCancelExternalWorkflowExecution() {
        this.add('swf:RequestCancelExternalWorkflowExecution');
        return this;
    }
    /**
     * Records a WorkflowExecutionCancelRequested event in the currently running workflow execution identified by the given domain, workflowId, and runId.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RequestCancelWorkflowExecution.html
     */
    requestCancelWorkflowExecution() {
        this.add('swf:RequestCancelWorkflowExecution');
        return this;
    }
    /**
     * Used by workers to tell the service that the ActivityTask identified by the taskToken was successfully canceled.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskCanceled.html
     */
    respondActivityTaskCanceled() {
        this.add('swf:RespondActivityTaskCanceled');
        return this;
    }
    /**
     * Used by workers to tell the service that the ActivityTask identified by the taskToken completed successfully with a result (if provided).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskCompleted.html
     */
    respondActivityTaskCompleted() {
        this.add('swf:RespondActivityTaskCompleted');
        return this;
    }
    /**
     * Used by workers to tell the service that the ActivityTask identified by the taskToken has failed with reason (if specified).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskFailed.html
     */
    respondActivityTaskFailed() {
        this.add('swf:RespondActivityTaskFailed');
        return this;
    }
    /**
     * Used by deciders to tell the service that the DecisionTask identified by the taskToken has successfully completed.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondDecisionTaskCompleted.html
     */
    respondDecisionTaskCompleted() {
        this.add('swf:RespondDecisionTaskCompleted');
        return this;
    }
    /**
     * Description for ScheduleActivityTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ScheduleActivityTask.html
     */
    scheduleActivityTask() {
        this.add('swf:ScheduleActivityTask');
        return this;
    }
    /**
     * Description for SignalExternalWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_SignalExternalWorkflowExecution.html
     */
    signalExternalWorkflowExecution() {
        this.add('swf:SignalExternalWorkflowExecution');
        return this;
    }
    /**
     * Records a WorkflowExecutionSignaled event in the workflow execution history and creates a decision task for the workflow execution identified by the given domain, workflowId and runId.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_SignalWorkflowExecution.html
     */
    signalWorkflowExecution() {
        this.add('swf:SignalWorkflowExecution');
        return this;
    }
    /**
     * Description for StartChildWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_StartChildWorkflowExecution.html
     */
    startChildWorkflowExecution() {
        this.add('swf:StartChildWorkflowExecution');
        return this;
    }
    /**
     * Description for StartTimer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_StartTimer.html
     */
    startTimer() {
        this.add('swf:StartTimer');
        return this;
    }
    /**
     * Starts an execution of the workflow type in the specified domain using the provided workflowId and input data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_StartWorkflowExecution.html
     */
    startWorkflowExecution() {
        this.add('swf:StartWorkflowExecution');
        return this;
    }
    /**
     * This action tags an AWS SWF resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_TagResource.html
     */
    tagResource() {
        this.add('swf:TagResource');
        return this;
    }
    /**
     * Records a WorkflowExecutionTerminated event and forces closure of the workflow execution identified by the given domain, runId, and workflowId.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_TerminateWorkflowExecution.html
     */
    terminateWorkflowExecution() {
        this.add('swf:TerminateWorkflowExecution');
        return this;
    }
    /**
     * This action removes a tag from an AWS SWF resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_UntagResource.html
     */
    untagResource() {
        this.add('swf:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * https://docs.aws.amazon.com/swf/latest/developerguide/swf-dev-domains.html
     *
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDomain(domainName, account, partition) {
        var arn = 'arn:${Partition}:swf::${Account}:domain/${DomainName}';
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Constrains the policy statement to only an activity type of the specified name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifActivityTypeName(value, operator) {
        return this.if(`swf:activityType.name`, value, operator || 'StringLike');
    }
    /**
     * Contstrains the policy statement to only an activity type of the specified version.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifActivityTypeVersion(value, operator) {
        return this.if(`swf:activityType.version`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a matching defaultTaskList name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDefaultTaskListName(value, operator) {
        return this.if(`swf:defaultTaskList.name`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only activities or workflows with the specified name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifName(value, operator) {
        return this.if(`swf:name`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a matching tagFilter.tag value.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagFilterTag(value, operator) {
        return this.if(`swf:tagFilter.tag`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that contain the specified tag.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember0(value, operator) {
        return this.if(`swf:tagList.member.0`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that contain the specified tag.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember1(value, operator) {
        return this.if(`swf:tagList.member.1`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that contain the specified tag.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember2(value, operator) {
        return this.if(`swf:tagList.member.2`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that contain the specified tag.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember3(value, operator) {
        return this.if(`swf:tagList.member.3`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that contain the specified tag.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember4(value, operator) {
        return this.if(`swf:tagList.member.4`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a tasklist with the specified name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTaskListName(value, operator) {
        return this.if(`swf:taskList.name`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a type filter with the specified name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTypeFilterName(value, operator) {
        return this.if(`swf:typeFilter.name`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a type filter with the specified version.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTypeFilterVersion(value, operator) {
        return this.if(`swf:typeFilter.version`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only activities or workflows with the specified version.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersion(value, operator) {
        return this.if(`swf:version`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a workflow type of the specified name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWorkflowTypeName(value, operator) {
        return this.if(`swf:workflowType.name`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a workflow type of the specified version.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWorkflowTypeVersion(value, operator) {
        return this.if(`swf:workflowType.version`, value, operator || 'StringLike');
    }
}
exports.Swf = Swf;
//# sourceMappingURL=data:application/json;base64,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