"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service appmesh-preview
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsappmeshpreview.html
 */
class AppmeshPreview extends shared_1.PolicyStatement {
    /**
     * Action provider for service appmesh-preview
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsappmeshpreview.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'appmesh-preview';
        this.actions = {
            "CreateGatewayRoute": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateGatewayRoute.html",
                "description": "Creates a gateway route that is associated with a virtual gateway.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "gatewayRoute": {
                        "required": true
                    },
                    "virtualService": {
                        "required": false
                    }
                }
            },
            "CreateMesh": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateMesh.html",
                "description": "Creates a service mesh.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mesh": {
                        "required": true
                    }
                }
            },
            "CreateRoute": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateRoute.html",
                "description": "Creates a route that is associated with a virtual router.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "route": {
                        "required": true
                    },
                    "virtualNode": {
                        "required": false
                    }
                }
            },
            "CreateVirtualGateway": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualGateway.html",
                "description": "Creates a virtual gateway within a service mesh.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "virtualGateway": {
                        "required": true
                    }
                }
            },
            "CreateVirtualNode": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualNode.html",
                "description": "Creates a virtual node within a service mesh.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "virtualNode": {
                        "required": true
                    },
                    "virtualService": {
                        "required": false
                    }
                }
            },
            "CreateVirtualRouter": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualRouter.html",
                "description": "Creates a virtual router within a service mesh.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "virtualRouter": {
                        "required": true
                    }
                }
            },
            "CreateVirtualService": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualService.html",
                "description": "Creates a virtual service within a service mesh.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "virtualService": {
                        "required": true
                    },
                    "virtualNode": {
                        "required": false
                    },
                    "virtualRouter": {
                        "required": false
                    }
                }
            },
            "DeleteGatewayRoute": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteGatewayRoute.html",
                "description": "Deletes an existing gateway route.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "gatewayRoute": {
                        "required": true
                    }
                }
            },
            "DeleteMesh": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteMesh.html",
                "description": "Deletes an existing service mesh.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mesh": {
                        "required": true
                    }
                }
            },
            "DeleteRoute": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteRoute.html",
                "description": "Deletes an existing route.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "route": {
                        "required": true
                    }
                }
            },
            "DeleteVirtualGateway": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualGateway.html",
                "description": "Deletes an existing virtual gateway.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "virtualGateway": {
                        "required": true
                    }
                }
            },
            "DeleteVirtualNode": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualNode.html",
                "description": "Deletes an existing virtual node.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "virtualNode": {
                        "required": true
                    }
                }
            },
            "DeleteVirtualRouter": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualRouter.html",
                "description": "Deletes an existing virtual router.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "virtualRouter": {
                        "required": true
                    }
                }
            },
            "DeleteVirtualService": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualService.html",
                "description": "Deletes an existing virtual service.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "virtualService": {
                        "required": true
                    }
                }
            },
            "DescribeGatewayRoute": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeGatewayRoute.html",
                "description": "Describes an existing gateway route.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "gatewayRoute": {
                        "required": true
                    }
                }
            },
            "DescribeMesh": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeMesh.html",
                "description": "Describes an existing service mesh.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "mesh": {
                        "required": true
                    }
                }
            },
            "DescribeRoute": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeRoute.html",
                "description": "Describes an existing route.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "route": {
                        "required": true
                    }
                }
            },
            "DescribeVirtualGateway": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualGateway.html",
                "description": "Describes an existing virtual gateway.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "virtualGateway": {
                        "required": true
                    }
                }
            },
            "DescribeVirtualNode": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualNode.html",
                "description": "Describes an existing virtual node.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "virtualNode": {
                        "required": true
                    }
                }
            },
            "DescribeVirtualRouter": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualRouter.html",
                "description": "Describes an existing virtual router.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "virtualRouter": {
                        "required": true
                    }
                }
            },
            "DescribeVirtualService": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualService.html",
                "description": "Describes an existing virtual service.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "virtualService": {
                        "required": true
                    }
                }
            },
            "ListGatewayRoutes": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListGatewayRoutes.html",
                "description": "Returns a list of existing gateway routes in a service mesh.",
                "accessLevel": "List",
                "resourceTypes": {
                    "virtualGateway": {
                        "required": true
                    }
                }
            },
            "ListMeshes": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListMeshes.html",
                "description": "Returns a list of existing service meshes.",
                "accessLevel": "List"
            },
            "ListRoutes": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListRoutes.html",
                "description": "Returns a list of existing routes in a service mesh.",
                "accessLevel": "List",
                "resourceTypes": {
                    "virtualRouter": {
                        "required": true
                    }
                }
            },
            "ListVirtualGateways": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualGateways.html",
                "description": "Returns a list of existing virtual gateways in a service mesh.",
                "accessLevel": "List",
                "resourceTypes": {
                    "mesh": {
                        "required": true
                    }
                }
            },
            "ListVirtualNodes": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualNodes.html",
                "description": "Returns a list of existing virtual nodes.",
                "accessLevel": "List",
                "resourceTypes": {
                    "mesh": {
                        "required": true
                    }
                }
            },
            "ListVirtualRouters": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualRouters.html",
                "description": "Returns a list of existing virtual routers in a service mesh.",
                "accessLevel": "List",
                "resourceTypes": {
                    "mesh": {
                        "required": true
                    }
                }
            },
            "ListVirtualServices": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualServices.html",
                "description": "Returns a list of existing virtual services in a service mesh.",
                "accessLevel": "List",
                "resourceTypes": {
                    "mesh": {
                        "required": true
                    }
                }
            },
            "StreamAggregatedResources": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/userguide/envoy.html",
                "description": "Allows an Envoy Proxy to receive streamed resources for an App Mesh endpoint (VirtualNode/VirtualGateway).",
                "accessLevel": "Read",
                "resourceTypes": {
                    "virtualGateway": {
                        "required": false
                    },
                    "virtualNode": {
                        "required": false
                    }
                }
            },
            "UpdateGatewayRoute": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateGatewayRoute.html",
                "description": "Updates an existing gateway route for a specified service mesh and virtual gateway.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "gatewayRoute": {
                        "required": true
                    },
                    "virtualService": {
                        "required": false
                    }
                }
            },
            "UpdateMesh": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateMesh.html",
                "description": "Updates an existing service mesh.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mesh": {
                        "required": true
                    }
                }
            },
            "UpdateRoute": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateRoute.html",
                "description": "Updates an existing route for a specified service mesh and virtual router.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "route": {
                        "required": true
                    },
                    "virtualNode": {
                        "required": false
                    }
                }
            },
            "UpdateVirtualGateway": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualGateway.html",
                "description": "Updates an existing virtual gateway in a specified service mesh.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "virtualGateway": {
                        "required": true
                    }
                }
            },
            "UpdateVirtualNode": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualNode.html",
                "description": "Updates an existing virtual node in a specified service mesh.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "virtualNode": {
                        "required": true
                    }
                }
            },
            "UpdateVirtualRouter": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualRouter.html",
                "description": "Updates an existing virtual router in a specified service mesh.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "virtualRouter": {
                        "required": true
                    }
                }
            },
            "UpdateVirtualService": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualService.html",
                "description": "Updates an existing virtual service in a specified service mesh.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mesh": {
                        "required": true
                    },
                    "virtualNode": {
                        "required": false
                    },
                    "virtualRouter": {
                        "required": false
                    }
                }
            }
        };
        this.resourceTypes = {
            "mesh": {
                "name": "mesh",
                "url": "https://docs.aws.amazon.com/app-mesh/latest/userguide/meshes.html",
                "arn": "arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}",
                "conditionKeys": []
            },
            "virtualService": {
                "name": "virtualService",
                "url": "https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html",
                "arn": "arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualService/${VirtualServiceName}",
                "conditionKeys": []
            },
            "virtualNode": {
                "name": "virtualNode",
                "url": "https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html",
                "arn": "arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualNode/${VirtualNodeName}",
                "conditionKeys": []
            },
            "virtualRouter": {
                "name": "virtualRouter",
                "url": "https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_routers.html",
                "arn": "arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualRouter/${VirtualRouterName}",
                "conditionKeys": []
            },
            "route": {
                "name": "route",
                "url": "https://docs.aws.amazon.com/app-mesh/latest/userguide/routes.html",
                "arn": "arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualRouter/${VirtualRouterName}/route/${RouteName}",
                "conditionKeys": []
            },
            "virtualGateway": {
                "name": "virtualGateway",
                "url": "https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html",
                "arn": "arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualGateway/${VirtualGatewayName}",
                "conditionKeys": []
            },
            "gatewayRoute": {
                "name": "gatewayRoute",
                "url": "https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html",
                "arn": "arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualGateway/${VirtualGatewayName}/gatewayRoute/${GatewayRouteName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Creates a gateway route that is associated with a virtual gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateGatewayRoute.html
     */
    createGatewayRoute() {
        this.add('appmesh-preview:CreateGatewayRoute');
        return this;
    }
    /**
     * Creates a service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateMesh.html
     */
    createMesh() {
        this.add('appmesh-preview:CreateMesh');
        return this;
    }
    /**
     * Creates a route that is associated with a virtual router.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateRoute.html
     */
    createRoute() {
        this.add('appmesh-preview:CreateRoute');
        return this;
    }
    /**
     * Creates a virtual gateway within a service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualGateway.html
     */
    createVirtualGateway() {
        this.add('appmesh-preview:CreateVirtualGateway');
        return this;
    }
    /**
     * Creates a virtual node within a service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualNode.html
     */
    createVirtualNode() {
        this.add('appmesh-preview:CreateVirtualNode');
        return this;
    }
    /**
     * Creates a virtual router within a service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualRouter.html
     */
    createVirtualRouter() {
        this.add('appmesh-preview:CreateVirtualRouter');
        return this;
    }
    /**
     * Creates a virtual service within a service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualService.html
     */
    createVirtualService() {
        this.add('appmesh-preview:CreateVirtualService');
        return this;
    }
    /**
     * Deletes an existing gateway route.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteGatewayRoute.html
     */
    deleteGatewayRoute() {
        this.add('appmesh-preview:DeleteGatewayRoute');
        return this;
    }
    /**
     * Deletes an existing service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteMesh.html
     */
    deleteMesh() {
        this.add('appmesh-preview:DeleteMesh');
        return this;
    }
    /**
     * Deletes an existing route.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteRoute.html
     */
    deleteRoute() {
        this.add('appmesh-preview:DeleteRoute');
        return this;
    }
    /**
     * Deletes an existing virtual gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualGateway.html
     */
    deleteVirtualGateway() {
        this.add('appmesh-preview:DeleteVirtualGateway');
        return this;
    }
    /**
     * Deletes an existing virtual node.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualNode.html
     */
    deleteVirtualNode() {
        this.add('appmesh-preview:DeleteVirtualNode');
        return this;
    }
    /**
     * Deletes an existing virtual router.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualRouter.html
     */
    deleteVirtualRouter() {
        this.add('appmesh-preview:DeleteVirtualRouter');
        return this;
    }
    /**
     * Deletes an existing virtual service.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualService.html
     */
    deleteVirtualService() {
        this.add('appmesh-preview:DeleteVirtualService');
        return this;
    }
    /**
     * Describes an existing gateway route.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeGatewayRoute.html
     */
    describeGatewayRoute() {
        this.add('appmesh-preview:DescribeGatewayRoute');
        return this;
    }
    /**
     * Describes an existing service mesh.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeMesh.html
     */
    describeMesh() {
        this.add('appmesh-preview:DescribeMesh');
        return this;
    }
    /**
     * Describes an existing route.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeRoute.html
     */
    describeRoute() {
        this.add('appmesh-preview:DescribeRoute');
        return this;
    }
    /**
     * Describes an existing virtual gateway.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualGateway.html
     */
    describeVirtualGateway() {
        this.add('appmesh-preview:DescribeVirtualGateway');
        return this;
    }
    /**
     * Describes an existing virtual node.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualNode.html
     */
    describeVirtualNode() {
        this.add('appmesh-preview:DescribeVirtualNode');
        return this;
    }
    /**
     * Describes an existing virtual router.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualRouter.html
     */
    describeVirtualRouter() {
        this.add('appmesh-preview:DescribeVirtualRouter');
        return this;
    }
    /**
     * Describes an existing virtual service.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualService.html
     */
    describeVirtualService() {
        this.add('appmesh-preview:DescribeVirtualService');
        return this;
    }
    /**
     * Returns a list of existing gateway routes in a service mesh.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListGatewayRoutes.html
     */
    listGatewayRoutes() {
        this.add('appmesh-preview:ListGatewayRoutes');
        return this;
    }
    /**
     * Returns a list of existing service meshes.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListMeshes.html
     */
    listMeshes() {
        this.add('appmesh-preview:ListMeshes');
        return this;
    }
    /**
     * Returns a list of existing routes in a service mesh.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListRoutes.html
     */
    listRoutes() {
        this.add('appmesh-preview:ListRoutes');
        return this;
    }
    /**
     * Returns a list of existing virtual gateways in a service mesh.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualGateways.html
     */
    listVirtualGateways() {
        this.add('appmesh-preview:ListVirtualGateways');
        return this;
    }
    /**
     * Returns a list of existing virtual nodes.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualNodes.html
     */
    listVirtualNodes() {
        this.add('appmesh-preview:ListVirtualNodes');
        return this;
    }
    /**
     * Returns a list of existing virtual routers in a service mesh.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualRouters.html
     */
    listVirtualRouters() {
        this.add('appmesh-preview:ListVirtualRouters');
        return this;
    }
    /**
     * Returns a list of existing virtual services in a service mesh.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualServices.html
     */
    listVirtualServices() {
        this.add('appmesh-preview:ListVirtualServices');
        return this;
    }
    /**
     * Allows an Envoy Proxy to receive streamed resources for an App Mesh endpoint (VirtualNode/VirtualGateway).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/envoy.html
     */
    streamAggregatedResources() {
        this.add('appmesh-preview:StreamAggregatedResources');
        return this;
    }
    /**
     * Updates an existing gateway route for a specified service mesh and virtual gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateGatewayRoute.html
     */
    updateGatewayRoute() {
        this.add('appmesh-preview:UpdateGatewayRoute');
        return this;
    }
    /**
     * Updates an existing service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateMesh.html
     */
    updateMesh() {
        this.add('appmesh-preview:UpdateMesh');
        return this;
    }
    /**
     * Updates an existing route for a specified service mesh and virtual router.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateRoute.html
     */
    updateRoute() {
        this.add('appmesh-preview:UpdateRoute');
        return this;
    }
    /**
     * Updates an existing virtual gateway in a specified service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualGateway.html
     */
    updateVirtualGateway() {
        this.add('appmesh-preview:UpdateVirtualGateway');
        return this;
    }
    /**
     * Updates an existing virtual node in a specified service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualNode.html
     */
    updateVirtualNode() {
        this.add('appmesh-preview:UpdateVirtualNode');
        return this;
    }
    /**
     * Updates an existing virtual router in a specified service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualRouter.html
     */
    updateVirtualRouter() {
        this.add('appmesh-preview:UpdateVirtualRouter');
        return this;
    }
    /**
     * Updates an existing virtual service in a specified service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualService.html
     */
    updateVirtualService() {
        this.add('appmesh-preview:UpdateVirtualService');
        return this;
    }
    /**
     * Adds a resource of type mesh to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/meshes.html
     *
     * @param meshName - Identifier for the meshName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onMesh(meshName, account, region, partition) {
        var arn = 'arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}';
        arn = arn.replace('${MeshName}', meshName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type virtualService to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualServiceName - Identifier for the virtualServiceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onVirtualService(meshName, virtualServiceName, account, region, partition) {
        var arn = 'arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualService/${VirtualServiceName}';
        arn = arn.replace('${MeshName}', meshName);
        arn = arn.replace('${VirtualServiceName}', virtualServiceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type virtualNode to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualNodeName - Identifier for the virtualNodeName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onVirtualNode(meshName, virtualNodeName, account, region, partition) {
        var arn = 'arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualNode/${VirtualNodeName}';
        arn = arn.replace('${MeshName}', meshName);
        arn = arn.replace('${VirtualNodeName}', virtualNodeName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type virtualRouter to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_routers.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualRouterName - Identifier for the virtualRouterName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onVirtualRouter(meshName, virtualRouterName, account, region, partition) {
        var arn = 'arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualRouter/${VirtualRouterName}';
        arn = arn.replace('${MeshName}', meshName);
        arn = arn.replace('${VirtualRouterName}', virtualRouterName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type route to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/routes.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualRouterName - Identifier for the virtualRouterName.
     * @param routeName - Identifier for the routeName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRoute(meshName, virtualRouterName, routeName, account, region, partition) {
        var arn = 'arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualRouter/${VirtualRouterName}/route/${RouteName}';
        arn = arn.replace('${MeshName}', meshName);
        arn = arn.replace('${VirtualRouterName}', virtualRouterName);
        arn = arn.replace('${RouteName}', routeName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type virtualGateway to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualGatewayName - Identifier for the virtualGatewayName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onVirtualGateway(meshName, virtualGatewayName, account, region, partition) {
        var arn = 'arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualGateway/${VirtualGatewayName}';
        arn = arn.replace('${MeshName}', meshName);
        arn = arn.replace('${VirtualGatewayName}', virtualGatewayName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type gatewayRoute to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualGatewayName - Identifier for the virtualGatewayName.
     * @param gatewayRouteName - Identifier for the gatewayRouteName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGatewayRoute(meshName, virtualGatewayName, gatewayRouteName, account, region, partition) {
        var arn = 'arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualGateway/${VirtualGatewayName}/gatewayRoute/${GatewayRouteName}';
        arn = arn.replace('${MeshName}', meshName);
        arn = arn.replace('${VirtualGatewayName}', virtualGatewayName);
        arn = arn.replace('${GatewayRouteName}', gatewayRouteName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.AppmeshPreview = AppmeshPreview;
//# sourceMappingURL=data:application/json;base64,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