"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service acm
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscertificatemanager.html
 */
class Acm extends shared_1.PolicyStatement {
    /**
     * Action provider for service acm
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscertificatemanager.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'acm';
        this.actions = {
            "AddTagsToCertificate": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_AddTagsToCertificate.html",
                "description": "Adds one or more tags to a certificate.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "certificate": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteCertificate": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_DeleteCertificate.html",
                "description": "Deletes a certificate and its associated private key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate": {
                        "required": true
                    }
                }
            },
            "DescribeCertificate": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_DescribeCertificate.html",
                "description": "Returns a list of the fields contained in the specified certificate.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate": {
                        "required": true
                    }
                }
            },
            "ExportCertificate": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_ExportCertificate.html",
                "description": "Exports a private certificate issued by a private certificate authority (CA) for use anywhere.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate": {
                        "required": true
                    }
                }
            },
            "GetCertificate": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_GetCertificate.html",
                "description": "Retrieves a certificate and certificate chain for the certificate specified by an ARN.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate": {
                        "required": true
                    }
                }
            },
            "ImportCertificate": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_ImportCertificate.html",
                "description": "Imports a 3rd party SSL/TLS certificate into AWS Certificate Manager (ACM).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "ListCertificates": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_ListCertificates.html",
                "description": "Retrieves a list of the certificate ARNs and the domain name for each ARN.",
                "accessLevel": "List"
            },
            "ListTagsForCertificate": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_ListTagsForCertificate.html",
                "description": "Lists the tags that have been applied to the certificate.",
                "accessLevel": "Read"
            },
            "RemoveTagsFromCertificate": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_RemoveTagsFromCertificate.html",
                "description": "Remove one or more tags from a certificate. A tag consists of a key-value pair",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "certificate": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "RenewCertificate": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_RenewCertificate.html",
                "description": "Renews an eligable private certificate.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate": {
                        "required": true
                    }
                }
            },
            "RequestCertificate": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_RequestCertificate.html",
                "description": "Requests a public or private certificate.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "ResendValidationEmail": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_ResendValidationEmail.html",
                "description": "Resends an email to request domain ownership validation.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate": {
                        "required": true
                    }
                }
            },
            "UpdateCertificateOptions": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_UpdateCertificateOptions.html",
                "description": "Updates a certificate. Use to specify whether to opt in to or out of certificate transparency logging.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "certificate": {
                "name": "certificate",
                "url": "https://docs.aws.amazon.com/acm/latest/userguide/authen-overview.html#acm-resources-operations",
                "arn": "arn:${Partition}:acm:${Region}:${Account}:certificate/${CertificateId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Adds one or more tags to a certificate.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_AddTagsToCertificate.html
     */
    addTagsToCertificate() {
        this.add('acm:AddTagsToCertificate');
        return this;
    }
    /**
     * Deletes a certificate and its associated private key.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_DeleteCertificate.html
     */
    deleteCertificate() {
        this.add('acm:DeleteCertificate');
        return this;
    }
    /**
     * Returns a list of the fields contained in the specified certificate.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_DescribeCertificate.html
     */
    describeCertificate() {
        this.add('acm:DescribeCertificate');
        return this;
    }
    /**
     * Exports a private certificate issued by a private certificate authority (CA) for use anywhere.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_ExportCertificate.html
     */
    exportCertificate() {
        this.add('acm:ExportCertificate');
        return this;
    }
    /**
     * Retrieves a certificate and certificate chain for the certificate specified by an ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_GetCertificate.html
     */
    getCertificate() {
        this.add('acm:GetCertificate');
        return this;
    }
    /**
     * Imports a 3rd party SSL/TLS certificate into AWS Certificate Manager (ACM).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_ImportCertificate.html
     */
    importCertificate() {
        this.add('acm:ImportCertificate');
        return this;
    }
    /**
     * Retrieves a list of the certificate ARNs and the domain name for each ARN.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_ListCertificates.html
     */
    listCertificates() {
        this.add('acm:ListCertificates');
        return this;
    }
    /**
     * Lists the tags that have been applied to the certificate.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_ListTagsForCertificate.html
     */
    listTagsForCertificate() {
        this.add('acm:ListTagsForCertificate');
        return this;
    }
    /**
     * Remove one or more tags from a certificate. A tag consists of a key-value pair
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_RemoveTagsFromCertificate.html
     */
    removeTagsFromCertificate() {
        this.add('acm:RemoveTagsFromCertificate');
        return this;
    }
    /**
     * Renews an eligable private certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_RenewCertificate.html
     */
    renewCertificate() {
        this.add('acm:RenewCertificate');
        return this;
    }
    /**
     * Requests a public or private certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_RequestCertificate.html
     */
    requestCertificate() {
        this.add('acm:RequestCertificate');
        return this;
    }
    /**
     * Resends an email to request domain ownership validation.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_ResendValidationEmail.html
     */
    resendValidationEmail() {
        this.add('acm:ResendValidationEmail');
        return this;
    }
    /**
     * Updates a certificate. Use to specify whether to opt in to or out of certificate transparency logging.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_UpdateCertificateOptions.html
     */
    updateCertificateOptions() {
        this.add('acm:UpdateCertificateOptions');
        return this;
    }
    /**
     * Adds a resource of type certificate to the statement
     *
     * https://docs.aws.amazon.com/acm/latest/userguide/authen-overview.html#acm-resources-operations
     *
     * @param certificateId - Identifier for the certificateId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onCertificate(certificateId, account, region, partition) {
        var arn = 'arn:${Partition}:acm:${Region}:${Account}:certificate/${CertificateId}';
        arn = arn.replace('${CertificateId}', certificateId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Acm = Acm;
//# sourceMappingURL=data:application/json;base64,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