"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service cloudtrail
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloudtrail.html
 */
class Cloudtrail extends shared_1.PolicyStatement {
    /**
     * Action provider for service cloudtrail
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloudtrail.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cloudtrail';
        this.actions = {
            "AddTags": {
                "url": "https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_AddTags.html",
                "description": "Grants permission to add one or more tags to a trail, up to a limit of 10",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "trail": {
                        "required": true
                    }
                }
            },
            "CreateTrail": {
                "url": "https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_CreateTrail.html",
                "description": "Grants permission to create a trail that specifies the settings for delivery of log data to an Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "trail": {
                        "required": true
                    }
                }
            },
            "DeleteTrail": {
                "url": "https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DeleteTrail.html",
                "description": "Grants permission to delete a trail",
                "accessLevel": "Write",
                "resourceTypes": {
                    "trail": {
                        "required": true
                    }
                }
            },
            "DescribeTrails": {
                "url": "https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DescribeTrails.html",
                "description": "Grants permission to list settings for the trails associated with the current region for your account",
                "accessLevel": "Read"
            },
            "GetEventSelectors": {
                "url": "https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetEventSelectors.html",
                "description": "Grants permission to list settings for event selectors configured for a trail",
                "accessLevel": "Read",
                "resourceTypes": {
                    "trail": {
                        "required": true
                    }
                }
            },
            "GetInsightSelectors": {
                "url": "https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetInsightSelectors.html",
                "description": "Grants permission to list CloudTrail Insights selectors that are configured for a trail",
                "accessLevel": "Read",
                "resourceTypes": {
                    "trail": {
                        "required": true
                    }
                }
            },
            "GetTrail": {
                "url": "https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetTrail.html",
                "description": "Grants permission to list settings for the trail",
                "accessLevel": "Read"
            },
            "GetTrailStatus": {
                "url": "https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetTrailStatus.html",
                "description": "Grants permission to retrieve a JSON-formatted list of information about the specified trail",
                "accessLevel": "Read",
                "resourceTypes": {
                    "trail": {
                        "required": true
                    }
                }
            },
            "ListPublicKeys": {
                "url": "https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListPublicKeys.html",
                "description": "Grants permission to list the public keys whose private keys were used to sign trail digest files within a specified time range",
                "accessLevel": "Read"
            },
            "ListTags": {
                "url": "https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListTags.html",
                "description": "Grants permission to list the tags for trails in the current region",
                "accessLevel": "Read",
                "resourceTypes": {
                    "trail": {
                        "required": true
                    }
                }
            },
            "ListTrails": {
                "url": "https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListTrails.html",
                "description": "Grants permission to list trails associated with the current region for your account",
                "accessLevel": "List"
            },
            "LookupEvents": {
                "url": "https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_LookupEvents.html",
                "description": "Grants permission to look up API activity events captured by CloudTrail that create, update, or delete resources in your account",
                "accessLevel": "Read"
            },
            "PutEventSelectors": {
                "url": "https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_PutEventSelectors.html",
                "description": "Grants permission to create and update event selectors for a trail",
                "accessLevel": "Write",
                "resourceTypes": {
                    "trail": {
                        "required": true
                    }
                }
            },
            "PutInsightSelectors": {
                "url": "https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_PutInsightSelectors.html",
                "description": "Grants permission to create and update CloudTrail Insights selectors for a trail",
                "accessLevel": "Write",
                "resourceTypes": {
                    "trail": {
                        "required": true
                    }
                }
            },
            "RemoveTags": {
                "url": "https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_RemoveTags.html",
                "description": "Grants permission to remove tags from a trail",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "trail": {
                        "required": true
                    }
                }
            },
            "StartLogging": {
                "url": "https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_StartLogging.html",
                "description": "Grants permission to start the recording of AWS API calls and log file delivery for a trail",
                "accessLevel": "Write",
                "resourceTypes": {
                    "trail": {
                        "required": true
                    }
                }
            },
            "StopLogging": {
                "url": "https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_StopLogging.html",
                "description": "Grants permission to stop the recording of AWS API calls and log file delivery for a trail",
                "accessLevel": "Write",
                "resourceTypes": {
                    "trail": {
                        "required": true
                    }
                }
            },
            "UpdateTrail": {
                "url": "https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_UpdateTrail.html",
                "description": "Grants permission to update the settings that specify delivery of log files",
                "accessLevel": "Write",
                "resourceTypes": {
                    "trail": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "trail": {
                "name": "trail",
                "url": "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/how-cloudtrail-works.html",
                "arn": "arn:${Partition}:cloudtrail:${Region}:${Account}:trail/${TrailName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to add one or more tags to a trail, up to a limit of 10
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_AddTags.html
     */
    addTags() {
        this.add('cloudtrail:AddTags');
        return this;
    }
    /**
     * Grants permission to create a trail that specifies the settings for delivery of log data to an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_CreateTrail.html
     */
    createTrail() {
        this.add('cloudtrail:CreateTrail');
        return this;
    }
    /**
     * Grants permission to delete a trail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DeleteTrail.html
     */
    deleteTrail() {
        this.add('cloudtrail:DeleteTrail');
        return this;
    }
    /**
     * Grants permission to list settings for the trails associated with the current region for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DescribeTrails.html
     */
    describeTrails() {
        this.add('cloudtrail:DescribeTrails');
        return this;
    }
    /**
     * Grants permission to list settings for event selectors configured for a trail
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetEventSelectors.html
     */
    getEventSelectors() {
        this.add('cloudtrail:GetEventSelectors');
        return this;
    }
    /**
     * Grants permission to list CloudTrail Insights selectors that are configured for a trail
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetInsightSelectors.html
     */
    getInsightSelectors() {
        this.add('cloudtrail:GetInsightSelectors');
        return this;
    }
    /**
     * Grants permission to list settings for the trail
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetTrail.html
     */
    getTrail() {
        this.add('cloudtrail:GetTrail');
        return this;
    }
    /**
     * Grants permission to retrieve a JSON-formatted list of information about the specified trail
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetTrailStatus.html
     */
    getTrailStatus() {
        this.add('cloudtrail:GetTrailStatus');
        return this;
    }
    /**
     * Grants permission to list the public keys whose private keys were used to sign trail digest files within a specified time range
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListPublicKeys.html
     */
    listPublicKeys() {
        this.add('cloudtrail:ListPublicKeys');
        return this;
    }
    /**
     * Grants permission to list the tags for trails in the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListTags.html
     */
    listTags() {
        this.add('cloudtrail:ListTags');
        return this;
    }
    /**
     * Grants permission to list trails associated with the current region for your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListTrails.html
     */
    listTrails() {
        this.add('cloudtrail:ListTrails');
        return this;
    }
    /**
     * Grants permission to look up API activity events captured by CloudTrail that create, update, or delete resources in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_LookupEvents.html
     */
    lookupEvents() {
        this.add('cloudtrail:LookupEvents');
        return this;
    }
    /**
     * Grants permission to create and update event selectors for a trail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_PutEventSelectors.html
     */
    putEventSelectors() {
        this.add('cloudtrail:PutEventSelectors');
        return this;
    }
    /**
     * Grants permission to create and update CloudTrail Insights selectors for a trail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_PutInsightSelectors.html
     */
    putInsightSelectors() {
        this.add('cloudtrail:PutInsightSelectors');
        return this;
    }
    /**
     * Grants permission to remove tags from a trail
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_RemoveTags.html
     */
    removeTags() {
        this.add('cloudtrail:RemoveTags');
        return this;
    }
    /**
     * Grants permission to start the recording of AWS API calls and log file delivery for a trail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_StartLogging.html
     */
    startLogging() {
        this.add('cloudtrail:StartLogging');
        return this;
    }
    /**
     * Grants permission to stop the recording of AWS API calls and log file delivery for a trail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_StopLogging.html
     */
    stopLogging() {
        this.add('cloudtrail:StopLogging');
        return this;
    }
    /**
     * Grants permission to update the settings that specify delivery of log files
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_UpdateTrail.html
     */
    updateTrail() {
        this.add('cloudtrail:UpdateTrail');
        return this;
    }
    /**
     * Adds a resource of type trail to the statement
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/how-cloudtrail-works.html
     *
     * @param trailName - Identifier for the trailName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTrail(trailName, account, region, partition) {
        var arn = 'arn:${Partition}:cloudtrail:${Region}:${Account}:trail/${TrailName}';
        arn = arn.replace('${TrailName}', trailName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Cloudtrail = Cloudtrail;
//# sourceMappingURL=data:application/json;base64,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