"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service codebuild
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodebuild.html
 */
class Codebuild extends shared_1.PolicyStatement {
    /**
     * Action provider for service codebuild
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodebuild.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'codebuild';
        this.actions = {
            "BatchDeleteBuilds": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchDeleteBuilds.html",
                "description": "Deletes one or more builds.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "BatchGetBuilds": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetBuilds.html",
                "description": "Gets information about one or more builds.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "BatchGetProjects": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetProjects.html",
                "description": "Gets information about one or more build projects.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "BatchGetReportGroups": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetReportGroups.html",
                "description": "Returns an array of ReportGroup objects that are specified by the input reportGroupArns parameter.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "report-group": {
                        "required": true
                    }
                }
            },
            "BatchGetReports": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetReports.html",
                "description": "Returns an array of the Report objects specified by the input reportArns parameter.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "report-group": {
                        "required": true
                    }
                }
            },
            "BatchPutTestCases": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies",
                "description": "Adds or updates information about a report.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "report-group": {
                        "required": true
                    }
                }
            },
            "CreateProject": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateProject.html",
                "description": "Creates a build project.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateReport": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies",
                "description": "Creates a report. A report is created when tests specified in the buildspec file for a report groups run during the build of a project.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "report-group": {
                        "required": true
                    }
                }
            },
            "CreateReportGroup": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateReportGroup.html",
                "description": "Creates a report group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "report-group": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateWebhook": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateWebhook.html",
                "description": "For an existing AWS CodeBuild build project that has its source code stored in a GitHub or Bitbucket repository, enables AWS CodeBuild to start rebuilding the source code every time a code change is pushed to the repository.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DeleteOAuthToken": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies",
                "description": "Deletes an OAuth token from a connected third-party OAuth provider. Only used in the AWS CodeBuild console.",
                "accessLevel": "Write"
            },
            "DeleteProject": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteProject.html",
                "description": "Deletes a build project.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DeleteReport": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteReport.html",
                "description": "Deletes a report.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "report-group": {
                        "required": true
                    }
                }
            },
            "DeleteReportGroup": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteReportGroup.html",
                "description": "Deletes a report group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "report-group": {
                        "required": true
                    }
                }
            },
            "DeleteResourcePolicy": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteResourcePolicy.html",
                "description": "Deletes a resource policy for the associated project or report group.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "project": {
                        "required": false
                    },
                    "report-group": {
                        "required": false
                    }
                }
            },
            "DeleteSourceCredentials": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteSourceCredentials.html",
                "description": "Deletes a set of GitHub, GitHub Enterprise, or Bitbucket source credentials.",
                "accessLevel": "Write"
            },
            "DeleteWebhook": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteWebhook.html",
                "description": "For an existing AWS CodeBuild build project that has its source code stored in a GitHub or Bitbucket repository, stops AWS CodeBuild from rebuilding the source code every time a code change is pushed to the repository.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DescribeTestCases": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DescribeTestCases.html",
                "description": "Returns an array of TestCase objects.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "report-group": {
                        "required": true
                    }
                }
            },
            "GetResourcePolicy": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_GetResourcePolicy.html",
                "description": "Returns a resource policy for the specified project or report group.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": false
                    },
                    "report-group": {
                        "required": false
                    }
                }
            },
            "ImportSourceCredentials": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ImportSourceCredentials.html",
                "description": "Imports the source repository credentials for an AWS CodeBuild project that has its source code stored in a GitHub, GitHub Enterprise, or Bitbucket repository.",
                "accessLevel": "Write"
            },
            "InvalidateProjectCache": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_InvalidateProjectCache.html",
                "description": "Resets the cache for a project.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "ListBuilds": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListBuilds.html",
                "description": "Gets a list of build IDs, with each build ID representing a single build.",
                "accessLevel": "List"
            },
            "ListBuildsForProject": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListBuildsForProject.html",
                "description": "Gets a list of build IDs for the specified build project, with each build ID representing a single build.",
                "accessLevel": "List",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "ListConnectedOAuthAccounts": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies",
                "description": "Lists connected third-party OAuth providers. Only used in the AWS CodeBuild console.",
                "accessLevel": "List"
            },
            "ListCuratedEnvironmentImages": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListCuratedEnvironmentImages.html",
                "description": "Gets information about Docker images that are managed by AWS CodeBuild.",
                "accessLevel": "List"
            },
            "ListProjects": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListProjects.html",
                "description": "Gets a list of build project names, with each build project name representing a single build project.",
                "accessLevel": "List"
            },
            "ListReportGroups": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReportGroups.html",
                "description": "Returns a list of report group ARNs. Each report group ARN represents one report group.",
                "accessLevel": "List"
            },
            "ListReports": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReports.html",
                "description": "Returns a list of report ARNs. Each report ARN representing one report.",
                "accessLevel": "List"
            },
            "ListReportsForReportGroup": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReportsForReportGroup.html",
                "description": "Returns a list of report ARNs that belong to the specified report group. Each report ARN represents one report.",
                "accessLevel": "List",
                "resourceTypes": {
                    "report-group": {
                        "required": true
                    }
                }
            },
            "ListRepositories": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies",
                "description": "Lists source code repositories from a connected third-party OAuth provider. Only used in the AWS CodeBuild console.",
                "accessLevel": "List"
            },
            "ListSharedProjects": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSharedProjects.html",
                "description": "Returns a list of project ARNs that have been shared with the requester. Each project ARN represents one project.",
                "accessLevel": "List"
            },
            "ListSharedReportGroups": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSharedReportGroups.html",
                "description": "Returns a list of report group ARNs that have been shared with the requester. Each report group ARN represents one report group.",
                "accessLevel": "List"
            },
            "ListSourceCredentials": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSourceCredentials.html",
                "description": "Returns a list of SourceCredentialsInfo objects.",
                "accessLevel": "List"
            },
            "PersistOAuthToken": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies",
                "description": "Saves an OAuth token from a connected third-party OAuth provider. Only used in the AWS CodeBuild console.",
                "accessLevel": "Write"
            },
            "PutResourcePolicy": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_PutResourcePolicy.html",
                "description": "Creates a resource policy for the associated project or report group.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "project": {
                        "required": false
                    },
                    "report-group": {
                        "required": false
                    }
                }
            },
            "StartBuild": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartBuild.html",
                "description": "Starts running a build.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "StopBuild": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StopBuild.html",
                "description": "Attempts to stop running a build.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "UpdateProject": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateProject.html",
                "description": "Changes the settings of an existing build project.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateReport": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies",
                "description": "Updates information about a report.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "report-group": {
                        "required": true
                    }
                }
            },
            "UpdateReportGroup": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateReportGroup.html",
                "description": "Changes the settings of an existing report group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "report-group": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateWebhook": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateWebhook.html",
                "description": "Updates the webhook associated with an AWS CodeBuild build project.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "build": {
                "name": "build",
                "url": "https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats",
                "arn": "arn:${Partition}:codebuild:${Region}:${Account}:build/${BuildId}",
                "conditionKeys": []
            },
            "project": {
                "name": "project",
                "url": "https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats",
                "arn": "arn:${Partition}:codebuild:${Region}:${Account}:project/${ProjectName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "report-group": {
                "name": "report-group",
                "url": "https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats",
                "arn": "arn:${Partition}:codebuild:${Region}:${Account}:report-group/${ReportGroupName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "report": {
                "name": "report",
                "url": "https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats",
                "arn": "arn:${Partition}:codebuild:${Region}:${Account}:report/${ReportGroupName}:${ReportId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Deletes one or more builds.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchDeleteBuilds.html
     */
    batchDeleteBuilds() {
        this.add('codebuild:BatchDeleteBuilds');
        return this;
    }
    /**
     * Gets information about one or more builds.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetBuilds.html
     */
    batchGetBuilds() {
        this.add('codebuild:BatchGetBuilds');
        return this;
    }
    /**
     * Gets information about one or more build projects.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetProjects.html
     */
    batchGetProjects() {
        this.add('codebuild:BatchGetProjects');
        return this;
    }
    /**
     * Returns an array of ReportGroup objects that are specified by the input reportGroupArns parameter.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetReportGroups.html
     */
    batchGetReportGroups() {
        this.add('codebuild:BatchGetReportGroups');
        return this;
    }
    /**
     * Returns an array of the Report objects specified by the input reportArns parameter.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetReports.html
     */
    batchGetReports() {
        this.add('codebuild:BatchGetReports');
        return this;
    }
    /**
     * Adds or updates information about a report.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    batchPutTestCases() {
        this.add('codebuild:BatchPutTestCases');
        return this;
    }
    /**
     * Creates a build project.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateProject.html
     */
    createProject() {
        this.add('codebuild:CreateProject');
        return this;
    }
    /**
     * Creates a report. A report is created when tests specified in the buildspec file for a report groups run during the build of a project.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    createReport() {
        this.add('codebuild:CreateReport');
        return this;
    }
    /**
     * Creates a report group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateReportGroup.html
     */
    createReportGroup() {
        this.add('codebuild:CreateReportGroup');
        return this;
    }
    /**
     * For an existing AWS CodeBuild build project that has its source code stored in a GitHub or Bitbucket repository, enables AWS CodeBuild to start rebuilding the source code every time a code change is pushed to the repository.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateWebhook.html
     */
    createWebhook() {
        this.add('codebuild:CreateWebhook');
        return this;
    }
    /**
     * Deletes an OAuth token from a connected third-party OAuth provider. Only used in the AWS CodeBuild console.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    deleteOAuthToken() {
        this.add('codebuild:DeleteOAuthToken');
        return this;
    }
    /**
     * Deletes a build project.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteProject.html
     */
    deleteProject() {
        this.add('codebuild:DeleteProject');
        return this;
    }
    /**
     * Deletes a report.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteReport.html
     */
    deleteReport() {
        this.add('codebuild:DeleteReport');
        return this;
    }
    /**
     * Deletes a report group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteReportGroup.html
     */
    deleteReportGroup() {
        this.add('codebuild:DeleteReportGroup');
        return this;
    }
    /**
     * Deletes a resource policy for the associated project or report group.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteResourcePolicy.html
     */
    deleteResourcePolicy() {
        this.add('codebuild:DeleteResourcePolicy');
        return this;
    }
    /**
     * Deletes a set of GitHub, GitHub Enterprise, or Bitbucket source credentials.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteSourceCredentials.html
     */
    deleteSourceCredentials() {
        this.add('codebuild:DeleteSourceCredentials');
        return this;
    }
    /**
     * For an existing AWS CodeBuild build project that has its source code stored in a GitHub or Bitbucket repository, stops AWS CodeBuild from rebuilding the source code every time a code change is pushed to the repository.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteWebhook.html
     */
    deleteWebhook() {
        this.add('codebuild:DeleteWebhook');
        return this;
    }
    /**
     * Returns an array of TestCase objects.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DescribeTestCases.html
     */
    describeTestCases() {
        this.add('codebuild:DescribeTestCases');
        return this;
    }
    /**
     * Returns a resource policy for the specified project or report group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_GetResourcePolicy.html
     */
    getResourcePolicy() {
        this.add('codebuild:GetResourcePolicy');
        return this;
    }
    /**
     * Imports the source repository credentials for an AWS CodeBuild project that has its source code stored in a GitHub, GitHub Enterprise, or Bitbucket repository.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ImportSourceCredentials.html
     */
    importSourceCredentials() {
        this.add('codebuild:ImportSourceCredentials');
        return this;
    }
    /**
     * Resets the cache for a project.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_InvalidateProjectCache.html
     */
    invalidateProjectCache() {
        this.add('codebuild:InvalidateProjectCache');
        return this;
    }
    /**
     * Gets a list of build IDs, with each build ID representing a single build.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListBuilds.html
     */
    listBuilds() {
        this.add('codebuild:ListBuilds');
        return this;
    }
    /**
     * Gets a list of build IDs for the specified build project, with each build ID representing a single build.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListBuildsForProject.html
     */
    listBuildsForProject() {
        this.add('codebuild:ListBuildsForProject');
        return this;
    }
    /**
     * Lists connected third-party OAuth providers. Only used in the AWS CodeBuild console.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    listConnectedOAuthAccounts() {
        this.add('codebuild:ListConnectedOAuthAccounts');
        return this;
    }
    /**
     * Gets information about Docker images that are managed by AWS CodeBuild.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListCuratedEnvironmentImages.html
     */
    listCuratedEnvironmentImages() {
        this.add('codebuild:ListCuratedEnvironmentImages');
        return this;
    }
    /**
     * Gets a list of build project names, with each build project name representing a single build project.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListProjects.html
     */
    listProjects() {
        this.add('codebuild:ListProjects');
        return this;
    }
    /**
     * Returns a list of report group ARNs. Each report group ARN represents one report group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReportGroups.html
     */
    listReportGroups() {
        this.add('codebuild:ListReportGroups');
        return this;
    }
    /**
     * Returns a list of report ARNs. Each report ARN representing one report.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReports.html
     */
    listReports() {
        this.add('codebuild:ListReports');
        return this;
    }
    /**
     * Returns a list of report ARNs that belong to the specified report group. Each report ARN represents one report.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReportsForReportGroup.html
     */
    listReportsForReportGroup() {
        this.add('codebuild:ListReportsForReportGroup');
        return this;
    }
    /**
     * Lists source code repositories from a connected third-party OAuth provider. Only used in the AWS CodeBuild console.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    listRepositories() {
        this.add('codebuild:ListRepositories');
        return this;
    }
    /**
     * Returns a list of project ARNs that have been shared with the requester. Each project ARN represents one project.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSharedProjects.html
     */
    listSharedProjects() {
        this.add('codebuild:ListSharedProjects');
        return this;
    }
    /**
     * Returns a list of report group ARNs that have been shared with the requester. Each report group ARN represents one report group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSharedReportGroups.html
     */
    listSharedReportGroups() {
        this.add('codebuild:ListSharedReportGroups');
        return this;
    }
    /**
     * Returns a list of SourceCredentialsInfo objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSourceCredentials.html
     */
    listSourceCredentials() {
        this.add('codebuild:ListSourceCredentials');
        return this;
    }
    /**
     * Saves an OAuth token from a connected third-party OAuth provider. Only used in the AWS CodeBuild console.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    persistOAuthToken() {
        this.add('codebuild:PersistOAuthToken');
        return this;
    }
    /**
     * Creates a resource policy for the associated project or report group.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_PutResourcePolicy.html
     */
    putResourcePolicy() {
        this.add('codebuild:PutResourcePolicy');
        return this;
    }
    /**
     * Starts running a build.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartBuild.html
     */
    startBuild() {
        this.add('codebuild:StartBuild');
        return this;
    }
    /**
     * Attempts to stop running a build.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StopBuild.html
     */
    stopBuild() {
        this.add('codebuild:StopBuild');
        return this;
    }
    /**
     * Changes the settings of an existing build project.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateProject.html
     */
    updateProject() {
        this.add('codebuild:UpdateProject');
        return this;
    }
    /**
     * Updates information about a report.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    updateReport() {
        this.add('codebuild:UpdateReport');
        return this;
    }
    /**
     * Changes the settings of an existing report group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateReportGroup.html
     */
    updateReportGroup() {
        this.add('codebuild:UpdateReportGroup');
        return this;
    }
    /**
     * Updates the webhook associated with an AWS CodeBuild build project.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateWebhook.html
     */
    updateWebhook() {
        this.add('codebuild:UpdateWebhook');
        return this;
    }
    /**
     * Adds a resource of type build to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param buildId - Identifier for the buildId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onBuild(buildId, account, region, partition) {
        var arn = 'arn:${Partition}:codebuild:${Region}:${Account}:build/${BuildId}';
        arn = arn.replace('${BuildId}', buildId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type project to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param projectName - Identifier for the projectName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onProject(projectName, account, region, partition) {
        var arn = 'arn:${Partition}:codebuild:${Region}:${Account}:project/${ProjectName}';
        arn = arn.replace('${ProjectName}', projectName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type report-group to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param reportGroupName - Identifier for the reportGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onReportGroup(reportGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:codebuild:${Region}:${Account}:report-group/${ReportGroupName}';
        arn = arn.replace('${ReportGroupName}', reportGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type report to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param reportGroupName - Identifier for the reportGroupName.
     * @param reportId - Identifier for the reportId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReport(reportGroupName, reportId, account, region, partition) {
        var arn = 'arn:${Partition}:codebuild:${Region}:${Account}:report/${ReportGroupName}:${ReportId}';
        arn = arn.replace('${ReportGroupName}', reportGroupName);
        arn = arn.replace('${ReportId}', reportId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Codebuild = Codebuild;
//# sourceMappingURL=data:application/json;base64,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