"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service codestar-notifications
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodestarnotifications.html
 */
class CodestarNotifications extends shared_1.PolicyStatement {
    /**
     * Action provider for service codestar-notifications
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodestarnotifications.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'codestar-notifications';
        this.actions = {
            "CreateNotificationRule": {
                "url": "https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_CreateNotificationRule.html",
                "description": "Grants permission to create a notification rule for a resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "notificationrule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "codestar-notifications:NotificationsForResource"
                ]
            },
            "DeleteNotificationRule": {
                "url": "https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_DeleteNotificationRule.html",
                "description": "Grants permission to delete a notification rule for a resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "notificationrule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "codestar-notifications:NotificationsForResource"
                ]
            },
            "DeleteTarget": {
                "url": "https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_DeleteTarget.html",
                "description": "Grants permission to delete a target for a notification rule",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DescribeNotificationRule": {
                "url": "https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_DescribeNotificationRule.html",
                "description": "Grants permission to get information about a notification rule",
                "accessLevel": "Read",
                "resourceTypes": {
                    "notificationrule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "codestar-notifications:NotificationsForResource"
                ]
            },
            "ListEventTypes": {
                "url": "https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_ListEventTypes.html",
                "description": "Grants permission to list notifications event types",
                "accessLevel": "List"
            },
            "ListNotificationRules": {
                "url": "https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_ListNotificationRules.html",
                "description": "Grants permission to list notification rules in an AWS account",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to list the tags attached to a notification rule resource ARN",
                "accessLevel": "List",
                "resourceTypes": {
                    "notificationrule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "ListTargets": {
                "url": "https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_ListTargets.html",
                "description": "Grants permission to list the notification rule targets for an AWS account",
                "accessLevel": "List",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "Subscribe": {
                "url": "https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_Subscribe.html",
                "description": "Grants permission to create an association between a notification rule and an Amazon SNS topic",
                "accessLevel": "Write",
                "resourceTypes": {
                    "notificationrule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "codestar-notifications:NotificationsForResource"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_TagResource.html",
                "description": "Grants permission to attach resource tags to a notification rule resource ARN",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "notificationrule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "Unsubscribe": {
                "url": "https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_Unsubscribe.html",
                "description": "Grants permission to remove an association between a notification rule and an Amazon SNS topic",
                "accessLevel": "Write",
                "resourceTypes": {
                    "notificationrule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "codestar-notifications:NotificationsForResource"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_UntagResource.html",
                "description": "Grants permission to disassociate resource tags from a notification rule resource ARN",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "notificationrule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateNotificationRule": {
                "url": "https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_UpdateNotificationRule.html",
                "description": "Grants permission to change a notification rule for a resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "notificationrule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "codestar-notifications:NotificationsForResource"
                ]
            }
        };
        this.resourceTypes = {
            "notificationrule": {
                "name": "notificationrule",
                "url": "https://docs.aws.amazon.com/codestar-notifications/latest/userguide/security_iam_service-with-iam.html",
                "arn": "arn:${Partition}:codestar-notifications:${Region}:${Account}:notificationrule/${NotificationRuleId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to create a notification rule for a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_CreateNotificationRule.html
     */
    createNotificationRule() {
        this.add('codestar-notifications:CreateNotificationRule');
        return this;
    }
    /**
     * Grants permission to delete a notification rule for a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_DeleteNotificationRule.html
     */
    deleteNotificationRule() {
        this.add('codestar-notifications:DeleteNotificationRule');
        return this;
    }
    /**
     * Grants permission to delete a target for a notification rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_DeleteTarget.html
     */
    deleteTarget() {
        this.add('codestar-notifications:DeleteTarget');
        return this;
    }
    /**
     * Grants permission to get information about a notification rule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_DescribeNotificationRule.html
     */
    describeNotificationRule() {
        this.add('codestar-notifications:DescribeNotificationRule');
        return this;
    }
    /**
     * Grants permission to list notifications event types
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_ListEventTypes.html
     */
    listEventTypes() {
        this.add('codestar-notifications:ListEventTypes');
        return this;
    }
    /**
     * Grants permission to list notification rules in an AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_ListNotificationRules.html
     */
    listNotificationRules() {
        this.add('codestar-notifications:ListNotificationRules');
        return this;
    }
    /**
     * Grants permission to list the tags attached to a notification rule resource ARN
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('codestar-notifications:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to list the notification rule targets for an AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_ListTargets.html
     */
    listTargets() {
        this.add('codestar-notifications:ListTargets');
        return this;
    }
    /**
     * Grants permission to create an association between a notification rule and an Amazon SNS topic
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_Subscribe.html
     */
    subscribe() {
        this.add('codestar-notifications:Subscribe');
        return this;
    }
    /**
     * Grants permission to attach resource tags to a notification rule resource ARN
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('codestar-notifications:TagResource');
        return this;
    }
    /**
     * Grants permission to remove an association between a notification rule and an Amazon SNS topic
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_Unsubscribe.html
     */
    unsubscribe() {
        this.add('codestar-notifications:Unsubscribe');
        return this;
    }
    /**
     * Grants permission to disassociate resource tags from a notification rule resource ARN
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('codestar-notifications:UntagResource');
        return this;
    }
    /**
     * Grants permission to change a notification rule for a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_UpdateNotificationRule.html
     */
    updateNotificationRule() {
        this.add('codestar-notifications:UpdateNotificationRule');
        return this;
    }
    /**
     * Adds a resource of type notificationrule to the statement
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/userguide/security_iam_service-with-iam.html
     *
     * @param notificationRuleId - Identifier for the notificationRuleId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onNotificationrule(notificationRuleId, account, region, partition) {
        var arn = 'arn:${Partition}:codestar-notifications:${Region}:${Account}:notificationrule/${NotificationRuleId}';
        arn = arn.replace('${NotificationRuleId}', notificationRuleId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access based on the ARN of the resource for which notifications are configured
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/userguide/security_iam_id-based-policy-examples.html
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifNotificationsForResource(value, operator) {
        return this.if(`codestar-notifications:NotificationsForResource`, value, operator || 'ArnEquals');
    }
}
exports.CodestarNotifications = CodestarNotifications;
//# sourceMappingURL=data:application/json;base64,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