"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service mediapackage
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediapackage.html
 */
class Mediapackage extends shared_1.PolicyStatement {
    /**
     * Action provider for service mediapackage
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediapackage.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'mediapackage';
        this.actions = {
            "CreateChannel": {
                "url": "https://docs.aws.amazon.com/mediapackage/latest/apireference/channels.html#channelspost",
                "description": "Grants permission to create a channel in AWS Elemental MediaPackage.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateOriginEndpoint": {
                "url": "https://docs.aws.amazon.com/mediapackage/latest/apireference/origin_endpoints.html#origin_endpointspost",
                "description": "Grants permission to create an endpoint in AWS Elemental MediaPackage.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteChannel": {
                "url": "https://docs.aws.amazon.com/mediapackage/latest/apireference/channels-id.html#channels-iddelete",
                "description": "Grants permission to delete a channel in AWS Elemental MediaPackage.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channels": {
                        "required": true
                    }
                }
            },
            "DeleteOriginEndpoint": {
                "url": "https://docs.aws.amazon.com/mediapackage/latest/apireference/origin_endpoints-id.html#origin_endpoints-iddelete",
                "description": "Grants permission to delete an endpoint in AWS Elemental MediaPackage.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "origin_endpoints": {
                        "required": true
                    }
                }
            },
            "DescribeChannel": {
                "url": "https://docs.aws.amazon.com/mediapackage/latest/apireference/channels-id.html#channels-idget",
                "description": "Grants permission to view the details of a channel in AWS Elemental MediaPackage.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "channels": {
                        "required": true
                    }
                }
            },
            "DescribeOriginEndpoint": {
                "url": "https://docs.aws.amazon.com/mediapackage/latest/apireference/origin_endpoints-id.html#origin_endpoints-idget",
                "description": "Grants permission to view the details of an endpoint in AWS Elemental MediaPackage.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "origin_endpoints": {
                        "required": true
                    }
                }
            },
            "ListChannels": {
                "url": "https://docs.aws.amazon.com/mediapackage/latest/apireference/channels.html#channelsget",
                "description": "Grants permission to view a list of channels in AWS Elemental MediaPackage.",
                "accessLevel": "Read"
            },
            "ListOriginEndpoints": {
                "url": "https://docs.aws.amazon.com/mediapackage/latest/apireference/origin_endpoints.html#origin_endpointsget",
                "description": "Grants permission to view a list of endpoints in AWS Elemental MediaPackage.",
                "accessLevel": "Read"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/mediapackage/latest/apireference/tags-resource-arn.html#tags-resource-arnget",
                "description": "Grants permission to list the tags assigned to a Channel or OriginEndpoint.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "channels": {
                        "required": false
                    },
                    "origin_endpoints": {
                        "required": false
                    }
                }
            },
            "RotateIngestEndpointCredentials": {
                "url": "https://docs.aws.amazon.com/mediapackage/latest/apireference/channels-id-ingest_endpoints-ingest_endpoint_id-credentials.html#channels-id-ingest_endpoints-ingest_endpoint_id-credentialsput",
                "description": "Grants permission to rotate IngestEndpoint credentials for a Channel in AWS Elemental MediaPackage.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channels": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/mediapackage/latest/apireference/tags-resource-arn.html#tags-resource-arnpost",
                "description": "Grants permission to assign tags to a Channel or OriginEndpoint.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channels": {
                        "required": false
                    },
                    "origin_endpoints": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/mediapackage/latest/apireference/tags-resource-arn.html#tags-resource-arndelete",
                "description": "Grants permission to delete tags to a Channel or OriginEndpoint.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channels": {
                        "required": false
                    },
                    "origin_endpoints": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateChannel": {
                "url": "https://docs.aws.amazon.com/mediapackage/latest/apireference/channels-id.html#channels-idput",
                "description": "Grants permission to make changes to a channel in AWS Elemental MediaPackage.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channels": {
                        "required": true
                    }
                }
            },
            "UpdateOriginEndpoint": {
                "url": "https://docs.aws.amazon.com/mediapackage/latest/apireference/origin_endpoints-id.html#origin_endpoints-idput",
                "description": "Grants permission to make changes to an endpoint in AWS Elemental MediaPackage.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "origin_endpoints": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "channels": {
                "name": "channels",
                "url": "https://docs.aws.amazon.com/mediapackage/latest/ug/channels.html",
                "arn": "arn:${Partition}:mediapackage:${Region}:${Account}:channels/${ChannelIdentifier}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "origin_endpoints": {
                "name": "origin_endpoints",
                "url": "https://docs.aws.amazon.com/mediapackage/latest/ug/endpoints.html",
                "arn": "arn:${Partition}:mediapackage:${Region}:${Account}:origin_endpoints/${OriginEndpointIdentifier}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to create a channel in AWS Elemental MediaPackage.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/channels.html#channelspost
     */
    createChannel() {
        this.add('mediapackage:CreateChannel');
        return this;
    }
    /**
     * Grants permission to create an endpoint in AWS Elemental MediaPackage.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/origin_endpoints.html#origin_endpointspost
     */
    createOriginEndpoint() {
        this.add('mediapackage:CreateOriginEndpoint');
        return this;
    }
    /**
     * Grants permission to delete a channel in AWS Elemental MediaPackage.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/channels-id.html#channels-iddelete
     */
    deleteChannel() {
        this.add('mediapackage:DeleteChannel');
        return this;
    }
    /**
     * Grants permission to delete an endpoint in AWS Elemental MediaPackage.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/origin_endpoints-id.html#origin_endpoints-iddelete
     */
    deleteOriginEndpoint() {
        this.add('mediapackage:DeleteOriginEndpoint');
        return this;
    }
    /**
     * Grants permission to view the details of a channel in AWS Elemental MediaPackage.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/channels-id.html#channels-idget
     */
    describeChannel() {
        this.add('mediapackage:DescribeChannel');
        return this;
    }
    /**
     * Grants permission to view the details of an endpoint in AWS Elemental MediaPackage.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/origin_endpoints-id.html#origin_endpoints-idget
     */
    describeOriginEndpoint() {
        this.add('mediapackage:DescribeOriginEndpoint');
        return this;
    }
    /**
     * Grants permission to view a list of channels in AWS Elemental MediaPackage.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/channels.html#channelsget
     */
    listChannels() {
        this.add('mediapackage:ListChannels');
        return this;
    }
    /**
     * Grants permission to view a list of endpoints in AWS Elemental MediaPackage.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/origin_endpoints.html#origin_endpointsget
     */
    listOriginEndpoints() {
        this.add('mediapackage:ListOriginEndpoints');
        return this;
    }
    /**
     * Grants permission to list the tags assigned to a Channel or OriginEndpoint.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/tags-resource-arn.html#tags-resource-arnget
     */
    listTagsForResource() {
        this.add('mediapackage:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to rotate IngestEndpoint credentials for a Channel in AWS Elemental MediaPackage.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/channels-id-ingest_endpoints-ingest_endpoint_id-credentials.html#channels-id-ingest_endpoints-ingest_endpoint_id-credentialsput
     */
    rotateIngestEndpointCredentials() {
        this.add('mediapackage:RotateIngestEndpointCredentials');
        return this;
    }
    /**
     * Grants permission to assign tags to a Channel or OriginEndpoint.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/tags-resource-arn.html#tags-resource-arnpost
     */
    tagResource() {
        this.add('mediapackage:TagResource');
        return this;
    }
    /**
     * Grants permission to delete tags to a Channel or OriginEndpoint.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/tags-resource-arn.html#tags-resource-arndelete
     */
    untagResource() {
        this.add('mediapackage:UntagResource');
        return this;
    }
    /**
     * Grants permission to make changes to a channel in AWS Elemental MediaPackage.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/channels-id.html#channels-idput
     */
    updateChannel() {
        this.add('mediapackage:UpdateChannel');
        return this;
    }
    /**
     * Grants permission to make changes to an endpoint in AWS Elemental MediaPackage.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/origin_endpoints-id.html#origin_endpoints-idput
     */
    updateOriginEndpoint() {
        this.add('mediapackage:UpdateOriginEndpoint');
        return this;
    }
    /**
     * Adds a resource of type channels to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/ug/channels.html
     *
     * @param channelIdentifier - Identifier for the channelIdentifier.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onChannels(channelIdentifier, account, region, partition) {
        var arn = 'arn:${Partition}:mediapackage:${Region}:${Account}:channels/${ChannelIdentifier}';
        arn = arn.replace('${ChannelIdentifier}', channelIdentifier);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type origin_endpoints to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/ug/endpoints.html
     *
     * @param originEndpointIdentifier - Identifier for the originEndpointIdentifier.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onOriginEndpoints(originEndpointIdentifier, account, region, partition) {
        var arn = 'arn:${Partition}:mediapackage:${Region}:${Account}:origin_endpoints/${OriginEndpointIdentifier}';
        arn = arn.replace('${OriginEndpointIdentifier}', originEndpointIdentifier);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Mediapackage = Mediapackage;
//# sourceMappingURL=data:application/json;base64,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