"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service health
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awshealthapisandnotifications.html
 */
class Health extends shared_1.PolicyStatement {
    /**
     * Action provider for service health
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awshealthapisandnotifications.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'health';
        this.actions = {
            "DescribeAffectedAccountsForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedAccountsForOrganization.html",
                "description": "Gets a list of accounts that have been affected by the specified events in organization.",
                "accessLevel": "Read"
            },
            "DescribeAffectedEntities": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntities.html",
                "description": "Gets a list of entities that have been affected by the specified events.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "event": {
                        "required": true
                    }
                },
                "conditions": [
                    "health:eventTypeCode",
                    "health:service"
                ]
            },
            "DescribeAffectedEntitiesForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntitiesForOrganization.html",
                "description": "Gets a list of entities that have been affected by the specified events and accounts in organization.",
                "accessLevel": "Read"
            },
            "DescribeEntityAggregates": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEntityAggregates.html",
                "description": "Returns the number of entities that are affected by each of the specified events.",
                "accessLevel": "Read"
            },
            "DescribeEventAggregates": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventAggregates.html",
                "description": "Returns the number of events of each event type (issue, scheduled change, and account notification).",
                "accessLevel": "Read"
            },
            "DescribeEventDetails": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetails.html",
                "description": "Returns detailed information about one or more specified events.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "event": {
                        "required": true
                    }
                },
                "conditions": [
                    "health:eventTypeCode",
                    "health:service"
                ]
            },
            "DescribeEventDetailsForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetailsForOrganization.html",
                "description": "Returns detailed information about one or more specified events for provided accounts in organization.",
                "accessLevel": "Read"
            },
            "DescribeEventTypes": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventTypes.html",
                "description": "Returns the event types that meet the specified filter criteria.",
                "accessLevel": "Read"
            },
            "DescribeEvents": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEvents.html",
                "description": "Returns information about events that meet the specified filter criteria.",
                "accessLevel": "Read"
            },
            "DescribeEventsForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventsForOrganization.html",
                "description": "Returns information about events that meet the specified filter criteria in organization.",
                "accessLevel": "Read"
            },
            "DescribeHealthServiceStatusForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeHealthServiceStatusForOrganization.html",
                "description": "Returns the status of enabling or disabling the Organizational View feature",
                "accessLevel": "Permissions management"
            },
            "DisableHealthServiceAccessForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DisableHealthServiceAccessForOrganization.html",
                "description": "Disables the Organizational View feature.",
                "accessLevel": "Permissions management"
            },
            "EnableHealthServiceAccessForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_EnableHealthServiceAccessForOrganization.html",
                "description": "Enables the Organizational View feature.",
                "accessLevel": "Permissions management"
            }
        };
        this.resourceTypes = {
            "event": {
                "name": "event",
                "url": "https://docs.aws.amazon.com/health/latest/ug/supported-operations.html",
                "arn": "arn:${Partition}:health:${Region}:${Account}:event/${Service}/${EventTypeCode}/${EventTypePlusId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Gets a list of accounts that have been affected by the specified events in organization.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedAccountsForOrganization.html
     */
    describeAffectedAccountsForOrganization() {
        this.add('health:DescribeAffectedAccountsForOrganization');
        return this;
    }
    /**
     * Gets a list of entities that have been affected by the specified events.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntities.html
     */
    describeAffectedEntities() {
        this.add('health:DescribeAffectedEntities');
        return this;
    }
    /**
     * Gets a list of entities that have been affected by the specified events and accounts in organization.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntitiesForOrganization.html
     */
    describeAffectedEntitiesForOrganization() {
        this.add('health:DescribeAffectedEntitiesForOrganization');
        return this;
    }
    /**
     * Returns the number of entities that are affected by each of the specified events.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEntityAggregates.html
     */
    describeEntityAggregates() {
        this.add('health:DescribeEntityAggregates');
        return this;
    }
    /**
     * Returns the number of events of each event type (issue, scheduled change, and account notification).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventAggregates.html
     */
    describeEventAggregates() {
        this.add('health:DescribeEventAggregates');
        return this;
    }
    /**
     * Returns detailed information about one or more specified events.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetails.html
     */
    describeEventDetails() {
        this.add('health:DescribeEventDetails');
        return this;
    }
    /**
     * Returns detailed information about one or more specified events for provided accounts in organization.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetailsForOrganization.html
     */
    describeEventDetailsForOrganization() {
        this.add('health:DescribeEventDetailsForOrganization');
        return this;
    }
    /**
     * Returns the event types that meet the specified filter criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventTypes.html
     */
    describeEventTypes() {
        this.add('health:DescribeEventTypes');
        return this;
    }
    /**
     * Returns information about events that meet the specified filter criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEvents.html
     */
    describeEvents() {
        this.add('health:DescribeEvents');
        return this;
    }
    /**
     * Returns information about events that meet the specified filter criteria in organization.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventsForOrganization.html
     */
    describeEventsForOrganization() {
        this.add('health:DescribeEventsForOrganization');
        return this;
    }
    /**
     * Returns the status of enabling or disabling the Organizational View feature
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeHealthServiceStatusForOrganization.html
     */
    describeHealthServiceStatusForOrganization() {
        this.add('health:DescribeHealthServiceStatusForOrganization');
        return this;
    }
    /**
     * Disables the Organizational View feature.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DisableHealthServiceAccessForOrganization.html
     */
    disableHealthServiceAccessForOrganization() {
        this.add('health:DisableHealthServiceAccessForOrganization');
        return this;
    }
    /**
     * Enables the Organizational View feature.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_EnableHealthServiceAccessForOrganization.html
     */
    enableHealthServiceAccessForOrganization() {
        this.add('health:EnableHealthServiceAccessForOrganization');
        return this;
    }
    /**
     * Adds a resource of type event to the statement
     *
     * https://docs.aws.amazon.com/health/latest/ug/supported-operations.html
     *
     * @param service - Identifier for the service.
     * @param eventTypeCode - Identifier for the eventTypeCode.
     * @param eventTypePlusId - Identifier for the eventTypePlusId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEvent(service, eventTypeCode, eventTypePlusId, account, region, partition) {
        var arn = 'arn:${Partition}:health:${Region}:${Account}:event/${Service}/${EventTypeCode}/${EventTypePlusId}';
        arn = arn.replace('${Service}', service);
        arn = arn.replace('${EventTypeCode}', eventTypeCode);
        arn = arn.replace('${EventTypePlusId}', eventTypePlusId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The type of event.
     *
     * https://docs.aws.amazon.com/health/latest/ug/controlling-access.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEventTypeCode(value, operator) {
        return this.if(`health:eventTypeCode`, value, operator || 'StringLike');
    }
    /**
     * The service of the event.
     *
     * https://docs.aws.amazon.com/health/latest/ug/controlling-access.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifService(value, operator) {
        return this.if(`health:service`, value, operator || 'StringLike');
    }
}
exports.Health = Health;
//# sourceMappingURL=data:application/json;base64,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