"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service kms
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awskeymanagementservice.html
 */
class Kms extends shared_1.PolicyStatement {
    /**
     * Action provider for service kms
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awskeymanagementservice.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'kms';
        this.actions = {
            "CancelKeyDeletion": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_CancelKeyDeletion.html",
                "description": "Controls permission to cancel the scheduled deletion of a customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "ConnectCustomKeyStore": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ConnectCustomKeyStore.html",
                "description": "Controls permission to connect or reconnect a custom key store to its associated AWS CloudHSM cluster.",
                "accessLevel": "Write"
            },
            "CreateAlias": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateAlias.html",
                "description": "Controls permission to create an alias for a customer master key (CMK). Aliases are optional friendly names that you can associate with customer master keys.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "alias": {
                        "required": true
                    },
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "CreateCustomKeyStore": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateCustomKeyStore.html",
                "description": "Controls permission to create a custom key store that is associated with an AWS CloudHSM cluster that you own and manage.",
                "accessLevel": "Write"
            },
            "CreateGrant": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateGrant.html",
                "description": "Controls permission to add a grant to a customer master key. You can use grants to add permissions without changing the key policy or IAM policy.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:GrantConstraintType",
                    "kms:GrantIsForAWSResource",
                    "kms:ViaService"
                ]
            },
            "CreateKey": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateKey.html",
                "description": "Controls permission to create a customer master key that can be used to protect data keys and other sensitive information.",
                "accessLevel": "Write",
                "conditions": [
                    "kms:BypassPolicyLockoutSafetyCheck",
                    "kms:CustomerMasterKeySpec",
                    "kms:CustomerMasterKeyUsage",
                    "kms:KeyOrigin"
                ]
            },
            "Decrypt": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_Decrypt.html",
                "description": "Controls permission to decrypt ciphertext that was encrypted under a customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ViaService"
                ]
            },
            "DeleteAlias": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteAlias.html",
                "description": "Controls permission to delete an alias. Aliases are optional friendly names that you can associate with customer master keys.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "alias": {
                        "required": true
                    },
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "DeleteCustomKeyStore": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteCustomKeyStore.html",
                "description": "Controls permission to delete a custom key store.",
                "accessLevel": "Write"
            },
            "DeleteImportedKeyMaterial": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteImportedKeyMaterial.html",
                "description": "Controls permission to delete cryptographic material that you imported into a customer master key. This action makes the key unusable.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "DescribeCustomKeyStores": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeCustomKeyStores.html",
                "description": "Controls permission to view detailed information about custom key stores in the account and region.",
                "accessLevel": "Read"
            },
            "DescribeKey": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html",
                "description": "Controls permission to view detailed information about a customer master key.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "DisableKey": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DisableKey.html",
                "description": "Controls permission to disable a customer master key, which prevents it from being used in cryptographic operations.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "DisableKeyRotation": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DisableKeyRotation.html",
                "description": "Controls permission to disable automatic rotation of a customer managed customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "DisconnectCustomKeyStore": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_DisconnectCustomKeyStore.html",
                "description": "Controls permission to disconnect the custom key store from its associated AWS CloudHSM cluster.",
                "accessLevel": "Write"
            },
            "EnableKey": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_EnableKey.html",
                "description": "Controls permission to change the state of a customer master key (CMK) to enabled. This allows the CMK to be used in cryptographic operations.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "EnableKeyRotation": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_EnableKeyRotation.html",
                "description": "Controls permission to enable automatic rotation of the cryptographic material in a customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "Encrypt": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html",
                "description": "Controls permission to use the specified customer master key to encrypt data and data keys.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ViaService"
                ]
            },
            "GenerateDataKey": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKey.html",
                "description": "Controls permission to use the customer master key to generate data keys. You can use the data keys to encrypt data outside of AWS KMS.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ViaService"
                ]
            },
            "GenerateDataKeyPair": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyPair.html",
                "description": "Controls permission to use the customer master key to generate data key pairs.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:DataKeyPairSpec",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ViaService"
                ]
            },
            "GenerateDataKeyPairWithoutPlaintext": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyPairWithoutPlaintext.html",
                "description": "Controls permission to use the customer master key to generate data key pairs. Unlike the GenerateDataKeyPair operation, this operation returns an encrypted private key without a plaintext copy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:DataKeyPairSpec",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ViaService"
                ]
            },
            "GenerateDataKeyWithoutPlaintext": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyWithoutPlaintext.html",
                "description": "Controls permission to use the customer master key to generate a data key. Unlike the GenerateDataKey operation, this operation returns an encrypted data key without a plaintext version of the data key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ViaService"
                ]
            },
            "GenerateRandom": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateRandom.html",
                "description": "Controls permission to get a cryptographically secure random byte string from AWS KMS.",
                "accessLevel": "Write"
            },
            "GetKeyPolicy": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GetKeyPolicy.html",
                "description": "Controls permission to view the key policy for the specified customer master key.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "GetKeyRotationStatus": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GetKeyRotationStatus.html",
                "description": "Controls permission to determine whether automatic key rotation is enabled on the customer master key.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "GetParametersForImport": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GetParametersForImport.html",
                "description": "Controls permission to get data that is required to import cryptographic material into a customer managed key, including a public key and import token.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService",
                    "kms:WrappingAlgorithm",
                    "kms:WrappingKeySpec"
                ]
            },
            "GetPublicKey": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_GetPublicKey.html",
                "description": "Controls permission to download the public key of an asymmetric customer master key.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "ImportKeyMaterial": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ImportKeyMaterial.html",
                "description": "Controls permission to import cryptographic material into a customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ExpirationModel",
                    "kms:ValidTo",
                    "kms:ViaService"
                ]
            },
            "ListAliases": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ListAliases.html",
                "description": "Controls permission to view the aliases that are defined in the account. Aliases are optional friendly names that you can associate with customer master keys.",
                "accessLevel": "List"
            },
            "ListGrants": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ListGrants.html",
                "description": "Controls permission to view all grants for a customer master key.",
                "accessLevel": "List",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:GrantIsForAWSResource",
                    "kms:ViaService"
                ]
            },
            "ListKeyPolicies": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ListKeyPolicies.html",
                "description": "Controls permission to view the names of key policies for a customer master key.",
                "accessLevel": "List",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "ListKeys": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ListKeys.html",
                "description": "Controls permission to view the key ID and Amazon Resource Name (ARN) of all customer master keys in the account.",
                "accessLevel": "List"
            },
            "ListResourceTags": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ListResourceTags.html",
                "description": "Controls permission to view all tags that are attached to a customer master key.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "ListRetirableGrants": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ListRetirableGrants.html",
                "description": "Controls permission to view grants in which the specified principal is the retiring principal. Other principals might be able to retire the grant and this principal might be able to retire other grants.",
                "accessLevel": "List",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                }
            },
            "PutKeyPolicy": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_PutKeyPolicy.html",
                "description": "Controls permission to replace the key policy for the specified customer master key.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:BypassPolicyLockoutSafetyCheck",
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "ReEncryptFrom": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ReEncrypt.html",
                "description": "Controls permission to decrypt data as part of the process that decrypts and reencrypts the data within AWS KMS.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ReEncryptOnSameKey",
                    "kms:ViaService"
                ]
            },
            "ReEncryptTo": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ReEncrypt.html",
                "description": "Controls permission to encrypt data as part of the process that decrypts and reencrypts the data within AWS KMS.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:EncryptionAlgorithm",
                    "kms:EncryptionContextKeys",
                    "kms:ReEncryptOnSameKey",
                    "kms:ViaService"
                ]
            },
            "RetireGrant": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_RetireGrant.html",
                "description": "Controls permission to retire a grant. The RetireGrant operation is typically called by the grant user after they complete the tasks that the grant allowed them to perform.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                }
            },
            "RevokeGrant": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_RevokeGrant.html",
                "description": "Controls permission to revoke a grant, which denies permission for all operations that depend on the grant.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:GrantIsForAWSResource",
                    "kms:ViaService"
                ]
            },
            "ScheduleKeyDeletion": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_ScheduleKeyDeletion.html",
                "description": "Controls permission to schedule deletion of a customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "Sign": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_Sign.html",
                "description": "Controls permission to produce a digital signature for a message.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:MessageType",
                    "kms:SigningAlgorithm",
                    "kms:ViaService"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_TagResource.html",
                "description": "Controls permission to create or update tags that are attached to a customer master key.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_UntagResource.html",
                "description": "Controls permission to delete tags that are attached to a customer master key.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "UpdateAlias": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateAlias.html",
                "description": "Controls permission to associate an alias with a different customer master key. An alias is an optional friendly name that you can associate with a customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "alias": {
                        "required": true
                    },
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "UpdateCustomKeyStore": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateCustomKeyStore.html",
                "description": "Controls permission to change the properties of a custom key store.",
                "accessLevel": "Write"
            },
            "UpdateKeyDescription": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateKeyDescription.html",
                "description": "Controls permission to delete or change the description of a customer master key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:ViaService"
                ]
            },
            "Verify": {
                "url": "https://docs.aws.amazon.com/kms/latest/APIReference/API_Verify.html",
                "description": "Controls permission to use the specified customer master key to verify digital signatures.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key": {
                        "required": true
                    }
                },
                "conditions": [
                    "kms:CallerAccount",
                    "kms:MessageType",
                    "kms:SigningAlgorithm",
                    "kms:ViaService"
                ]
            }
        };
        this.resourceTypes = {
            "alias": {
                "name": "alias",
                "url": "https://docs.aws.amazon.com/kms/latest/developerguide/programming-aliases.html",
                "arn": "arn:${Partition}:kms:${Region}:${Account}:alias/${Alias}",
                "conditionKeys": []
            },
            "key": {
                "name": "key",
                "url": "https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys",
                "arn": "arn:${Partition}:kms:${Region}:${Account}:key/${KeyId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Controls permission to cancel the scheduled deletion of a customer master key.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CancelKeyDeletion.html
     */
    cancelKeyDeletion() {
        this.add('kms:CancelKeyDeletion');
        return this;
    }
    /**
     * Controls permission to connect or reconnect a custom key store to its associated AWS CloudHSM cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ConnectCustomKeyStore.html
     */
    connectCustomKeyStore() {
        this.add('kms:ConnectCustomKeyStore');
        return this;
    }
    /**
     * Controls permission to create an alias for a customer master key (CMK). Aliases are optional friendly names that you can associate with customer master keys.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateAlias.html
     */
    createAlias() {
        this.add('kms:CreateAlias');
        return this;
    }
    /**
     * Controls permission to create a custom key store that is associated with an AWS CloudHSM cluster that you own and manage.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateCustomKeyStore.html
     */
    createCustomKeyStore() {
        this.add('kms:CreateCustomKeyStore');
        return this;
    }
    /**
     * Controls permission to add a grant to a customer master key. You can use grants to add permissions without changing the key policy or IAM policy.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateGrant.html
     */
    createGrant() {
        this.add('kms:CreateGrant');
        return this;
    }
    /**
     * Controls permission to create a customer master key that can be used to protect data keys and other sensitive information.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateKey.html
     */
    createKey() {
        this.add('kms:CreateKey');
        return this;
    }
    /**
     * Controls permission to decrypt ciphertext that was encrypted under a customer master key.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_Decrypt.html
     */
    decrypt() {
        this.add('kms:Decrypt');
        return this;
    }
    /**
     * Controls permission to delete an alias. Aliases are optional friendly names that you can associate with customer master keys.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteAlias.html
     */
    deleteAlias() {
        this.add('kms:DeleteAlias');
        return this;
    }
    /**
     * Controls permission to delete a custom key store.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteCustomKeyStore.html
     */
    deleteCustomKeyStore() {
        this.add('kms:DeleteCustomKeyStore');
        return this;
    }
    /**
     * Controls permission to delete cryptographic material that you imported into a customer master key. This action makes the key unusable.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteImportedKeyMaterial.html
     */
    deleteImportedKeyMaterial() {
        this.add('kms:DeleteImportedKeyMaterial');
        return this;
    }
    /**
     * Controls permission to view detailed information about custom key stores in the account and region.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeCustomKeyStores.html
     */
    describeCustomKeyStores() {
        this.add('kms:DescribeCustomKeyStores');
        return this;
    }
    /**
     * Controls permission to view detailed information about a customer master key.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html
     */
    describeKey() {
        this.add('kms:DescribeKey');
        return this;
    }
    /**
     * Controls permission to disable a customer master key, which prevents it from being used in cryptographic operations.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DisableKey.html
     */
    disableKey() {
        this.add('kms:DisableKey');
        return this;
    }
    /**
     * Controls permission to disable automatic rotation of a customer managed customer master key.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DisableKeyRotation.html
     */
    disableKeyRotation() {
        this.add('kms:DisableKeyRotation');
        return this;
    }
    /**
     * Controls permission to disconnect the custom key store from its associated AWS CloudHSM cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DisconnectCustomKeyStore.html
     */
    disconnectCustomKeyStore() {
        this.add('kms:DisconnectCustomKeyStore');
        return this;
    }
    /**
     * Controls permission to change the state of a customer master key (CMK) to enabled. This allows the CMK to be used in cryptographic operations.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_EnableKey.html
     */
    enableKey() {
        this.add('kms:EnableKey');
        return this;
    }
    /**
     * Controls permission to enable automatic rotation of the cryptographic material in a customer master key.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_EnableKeyRotation.html
     */
    enableKeyRotation() {
        this.add('kms:EnableKeyRotation');
        return this;
    }
    /**
     * Controls permission to use the specified customer master key to encrypt data and data keys.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html
     */
    encrypt() {
        this.add('kms:Encrypt');
        return this;
    }
    /**
     * Controls permission to use the customer master key to generate data keys. You can use the data keys to encrypt data outside of AWS KMS.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKey.html
     */
    generateDataKey() {
        this.add('kms:GenerateDataKey');
        return this;
    }
    /**
     * Controls permission to use the customer master key to generate data key pairs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyPair.html
     */
    generateDataKeyPair() {
        this.add('kms:GenerateDataKeyPair');
        return this;
    }
    /**
     * Controls permission to use the customer master key to generate data key pairs. Unlike the GenerateDataKeyPair operation, this operation returns an encrypted private key without a plaintext copy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyPairWithoutPlaintext.html
     */
    generateDataKeyPairWithoutPlaintext() {
        this.add('kms:GenerateDataKeyPairWithoutPlaintext');
        return this;
    }
    /**
     * Controls permission to use the customer master key to generate a data key. Unlike the GenerateDataKey operation, this operation returns an encrypted data key without a plaintext version of the data key.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyWithoutPlaintext.html
     */
    generateDataKeyWithoutPlaintext() {
        this.add('kms:GenerateDataKeyWithoutPlaintext');
        return this;
    }
    /**
     * Controls permission to get a cryptographically secure random byte string from AWS KMS.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateRandom.html
     */
    generateRandom() {
        this.add('kms:GenerateRandom');
        return this;
    }
    /**
     * Controls permission to view the key policy for the specified customer master key.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GetKeyPolicy.html
     */
    getKeyPolicy() {
        this.add('kms:GetKeyPolicy');
        return this;
    }
    /**
     * Controls permission to determine whether automatic key rotation is enabled on the customer master key.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GetKeyRotationStatus.html
     */
    getKeyRotationStatus() {
        this.add('kms:GetKeyRotationStatus');
        return this;
    }
    /**
     * Controls permission to get data that is required to import cryptographic material into a customer managed key, including a public key and import token.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GetParametersForImport.html
     */
    getParametersForImport() {
        this.add('kms:GetParametersForImport');
        return this;
    }
    /**
     * Controls permission to download the public key of an asymmetric customer master key.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GetPublicKey.html
     */
    getPublicKey() {
        this.add('kms:GetPublicKey');
        return this;
    }
    /**
     * Controls permission to import cryptographic material into a customer master key.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ImportKeyMaterial.html
     */
    importKeyMaterial() {
        this.add('kms:ImportKeyMaterial');
        return this;
    }
    /**
     * Controls permission to view the aliases that are defined in the account. Aliases are optional friendly names that you can associate with customer master keys.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListAliases.html
     */
    listAliases() {
        this.add('kms:ListAliases');
        return this;
    }
    /**
     * Controls permission to view all grants for a customer master key.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListGrants.html
     */
    listGrants() {
        this.add('kms:ListGrants');
        return this;
    }
    /**
     * Controls permission to view the names of key policies for a customer master key.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListKeyPolicies.html
     */
    listKeyPolicies() {
        this.add('kms:ListKeyPolicies');
        return this;
    }
    /**
     * Controls permission to view the key ID and Amazon Resource Name (ARN) of all customer master keys in the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListKeys.html
     */
    listKeys() {
        this.add('kms:ListKeys');
        return this;
    }
    /**
     * Controls permission to view all tags that are attached to a customer master key.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListResourceTags.html
     */
    listResourceTags() {
        this.add('kms:ListResourceTags');
        return this;
    }
    /**
     * Controls permission to view grants in which the specified principal is the retiring principal. Other principals might be able to retire the grant and this principal might be able to retire other grants.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListRetirableGrants.html
     */
    listRetirableGrants() {
        this.add('kms:ListRetirableGrants');
        return this;
    }
    /**
     * Controls permission to replace the key policy for the specified customer master key.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_PutKeyPolicy.html
     */
    putKeyPolicy() {
        this.add('kms:PutKeyPolicy');
        return this;
    }
    /**
     * Controls permission to decrypt data as part of the process that decrypts and reencrypts the data within AWS KMS.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ReEncrypt.html
     */
    reEncryptFrom() {
        this.add('kms:ReEncryptFrom');
        return this;
    }
    /**
     * Controls permission to encrypt data as part of the process that decrypts and reencrypts the data within AWS KMS.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ReEncrypt.html
     */
    reEncryptTo() {
        this.add('kms:ReEncryptTo');
        return this;
    }
    /**
     * Controls permission to retire a grant. The RetireGrant operation is typically called by the grant user after they complete the tasks that the grant allowed them to perform.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_RetireGrant.html
     */
    retireGrant() {
        this.add('kms:RetireGrant');
        return this;
    }
    /**
     * Controls permission to revoke a grant, which denies permission for all operations that depend on the grant.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_RevokeGrant.html
     */
    revokeGrant() {
        this.add('kms:RevokeGrant');
        return this;
    }
    /**
     * Controls permission to schedule deletion of a customer master key.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ScheduleKeyDeletion.html
     */
    scheduleKeyDeletion() {
        this.add('kms:ScheduleKeyDeletion');
        return this;
    }
    /**
     * Controls permission to produce a digital signature for a message.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_Sign.html
     */
    sign() {
        this.add('kms:Sign');
        return this;
    }
    /**
     * Controls permission to create or update tags that are attached to a customer master key.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('kms:TagResource');
        return this;
    }
    /**
     * Controls permission to delete tags that are attached to a customer master key.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('kms:UntagResource');
        return this;
    }
    /**
     * Controls permission to associate an alias with a different customer master key. An alias is an optional friendly name that you can associate with a customer master key.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateAlias.html
     */
    updateAlias() {
        this.add('kms:UpdateAlias');
        return this;
    }
    /**
     * Controls permission to change the properties of a custom key store.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateCustomKeyStore.html
     */
    updateCustomKeyStore() {
        this.add('kms:UpdateCustomKeyStore');
        return this;
    }
    /**
     * Controls permission to delete or change the description of a customer master key.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateKeyDescription.html
     */
    updateKeyDescription() {
        this.add('kms:UpdateKeyDescription');
        return this;
    }
    /**
     * Controls permission to use the specified customer master key to verify digital signatures.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_Verify.html
     */
    verify() {
        this.add('kms:Verify');
        return this;
    }
    /**
     * Adds a resource of type alias to the statement
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/programming-aliases.html
     *
     * @param alias - Identifier for the alias.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAlias(alias, account, region, partition) {
        var arn = 'arn:${Partition}:kms:${Region}:${Account}:alias/${Alias}';
        arn = arn.replace('${Alias}', alias);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type key to the statement
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys
     *
     * @param keyId - Identifier for the keyId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onKey(keyId, account, region, partition) {
        var arn = 'arn:${Partition}:kms:${Region}:${Account}:key/${KeyId}';
        arn = arn.replace('${KeyId}', keyId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Controls access to the CreateKey and PutKeyPolicy operations based on the value of the BypassPolicyLockoutSafetyCheck parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-bypass-policy-lockout-safety-check
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifBypassPolicyLockoutSafetyCheck(value) {
        return this.if(`kms:BypassPolicyLockoutSafetyCheck`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Controls access to specified AWS KMS operations based on the AWS account ID of the caller. You can use this condition key to allow or deny access to all IAM users and roles in an AWS account in a single policy statement.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-caller-account
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCallerAccount(value, operator) {
        return this.if(`kms:CallerAccount`, value, operator || 'StringLike');
    }
    /**
     * Controls access to an API operation based on the CustomerMasterKeySpec property of the CMK that is created by or used in the operation. Use it to qualify authorization of the CreateKey operation or any operation that is authorized for a CMK resource.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-customer-master-key-spec
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCustomerMasterKeySpec(value, operator) {
        return this.if(`kms:CustomerMasterKeySpec`, value, operator || 'StringLike');
    }
    /**
     * Controls access to an API operation based on the KeyUsage property of the CMK created by or used in the operation. Use it to qualify authorization of the CreateKey operation or any operation that is authorized for a CMK resource.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-customer-master-key-usage
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCustomerMasterKeyUsage(value, operator) {
        return this.if(`kms:CustomerMasterKeyUsage`, value, operator || 'StringLike');
    }
    /**
     * Controls access to GenerateDataKeyPair and GenerateDataKeyPairWithoutPlaintext operations based on the value of the DataKeyPairSpec parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-data-key-pair-spec
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataKeyPairSpec(value, operator) {
        return this.if(`kms:DataKeyPairSpec`, value, operator || 'StringLike');
    }
    /**
     * Controls access to encryption operations based on the value of the encryption algorithm in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-encryption-algorithm
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEncryptionAlgorithm(value, operator) {
        return this.if(`kms:EncryptionAlgorithm`, value, operator || 'StringLike');
    }
    /**
     * Controls access based on the presence of specified keys in the encryption context. The encryption context is an optional element in a cryptographic operation.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-encryption-context-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEncryptionContextKeys(value, operator) {
        return this.if(`kms:EncryptionContextKeys`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the ImportKeyMaterial operation based on the value of the ExpirationModel parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-expiration-model
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExpirationModel(value, operator) {
        return this.if(`kms:ExpirationModel`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the CreateGrant operation based on the grant constraint in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-grant-constraint-type
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGrantConstraintType(value, operator) {
        return this.if(`kms:GrantConstraintType`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the CreateGrant operation when the request comes from a specified AWS service.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-grant-is-for-aws-resource
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifGrantIsForAWSResource(value) {
        return this.if(`kms:GrantIsForAWSResource`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Controls access to the CreateGrant operation based on the operations in the grant.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-grant-operations
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGrantOperations(value, operator) {
        return this.if(`kms:GrantOperations`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the CreateGrant operation based on the grantee principal in the grant.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-grantee-principal
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGranteePrincipal(value, operator) {
        return this.if(`kms:GranteePrincipal`, value, operator || 'StringLike');
    }
    /**
     * Controls access to an API operation based on the Origin property of the CMK created by or used in the operation. Use it to qualify authorization of the CreateKey operation or any operation that is authorized for a CMK resource.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-key-origin
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifKeyOrigin(value, operator) {
        return this.if(`kms:KeyOrigin`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the Sign and Verify operations based on the value of the MessageType parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-message-type
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMessageType(value, operator) {
        return this.if(`kms:MessageType`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the ReEncrypt operation when it uses the same customer master key that was used for the Encrypt operation.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-reencrypt-on-same-key
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifReEncryptOnSameKey(value) {
        return this.if(`kms:ReEncryptOnSameKey`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Controls access to the CreateGrant operation based on the retiring principal in the grant.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-retiring-principal
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRetiringPrincipal(value, operator) {
        return this.if(`kms:RetiringPrincipal`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the Sign and Verify operations based on the signing algorithm in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-signing-algorithm
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSigningAlgorithm(value, operator) {
        return this.if(`kms:SigningAlgorithm`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the ImportKeyMaterial operation based on the value of the ValidTo parameter in the request. You can use this condition key to allow users to import key material only when it expires by the specified date.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-valid-to
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifValidTo(value, operator) {
        return this.if(`kms:ValidTo`, value, operator || 'NumericEquals');
    }
    /**
     * Controls access when a request made on the principal's behalf comes from a specified AWS service.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-via-service
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifViaService(value, operator) {
        return this.if(`kms:ViaService`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the GetParametersForImport operation based on the value of the WrappingAlgorithm parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-wrapping-algorithm
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWrappingAlgorithm(value, operator) {
        return this.if(`kms:WrappingAlgorithm`, value, operator || 'StringLike');
    }
    /**
     * Controls access to the GetParametersForImport operation based on the value of the WrappingKeySpec parameter in the request.
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-wrapping-key-spec
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWrappingKeySpec(value, operator) {
        return this.if(`kms:WrappingKeySpec`, value, operator || 'StringLike');
    }
}
exports.Kms = Kms;
//# sourceMappingURL=data:application/json;base64,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