"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service lambda
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awslambda.html
 */
class Lambda extends shared_1.PolicyStatement {
    /**
     * Action provider for service lambda
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awslambda.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'lambda';
        this.actions = {
            "AddLayerVersionPermission": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_AddLayerVersionPermission.html",
                "description": "Grants permission to add permissions to the resource-based policy of a version of an AWS Lambda layer",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "layerVersion": {
                        "required": true
                    }
                }
            },
            "AddPermission": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_AddPermission.html",
                "description": "Grants permission to give an AWS service or another account permission to use an AWS Lambda function",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                },
                "conditions": [
                    "lambda:Principal"
                ]
            },
            "CreateAlias": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_CreateAlias.html",
                "description": "Grants permission to create an alias for a Lambda function version",
                "accessLevel": "Write",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "CreateEventSourceMapping": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_CreateEventSourceMapping.html",
                "description": "Grants permission to create a mapping between an event source and an AWS Lambda function",
                "accessLevel": "Write",
                "conditions": [
                    "lambda:FunctionArn"
                ]
            },
            "CreateFunction": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_CreateFunction.html",
                "description": "Grants permission to create an AWS Lambda function",
                "accessLevel": "Write",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                },
                "conditions": [
                    "lambda:Layer"
                ]
            },
            "DeleteAlias": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteAlias.html",
                "description": "Grants permission to delete an AWS Lambda function alias",
                "accessLevel": "Write",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "DeleteEventSourceMapping": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteEventSourceMapping.html",
                "description": "Grants permission to delete an AWS Lambda event source mapping",
                "accessLevel": "Write",
                "resourceTypes": {
                    "eventSourceMapping": {
                        "required": true
                    }
                },
                "conditions": [
                    "lambda:FunctionArn"
                ]
            },
            "DeleteFunction": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteFunction.html",
                "description": "Grants permission to delete an AWS Lambda function",
                "accessLevel": "Write",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "DeleteFunctionConcurrency": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteFunctionConcurrency.html",
                "description": "Grants permission to remove a concurrent execution limit from an AWS Lambda function",
                "accessLevel": "Write",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "DeleteFunctionEventInvokeConfig": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteFunctionEventInvokeConfig.html",
                "description": "Grants permission to delete the configuration for asynchronous invocation for an AWS Lambda function, version, or alias",
                "accessLevel": "Write",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "DeleteLayerVersion": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteLayerVersion.html",
                "description": "Grants permission to delete a version of an AWS Lambda layer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "layerVersion": {
                        "required": true
                    }
                }
            },
            "DeleteProvisionedConcurrencyConfig": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteProvisionedConcurrencyConfig.html",
                "description": "Grants permission to delete the provisioned concurrency configuration for an AWS Lambda function",
                "accessLevel": "Write",
                "resourceTypes": {
                    "function alias": {
                        "required": false
                    },
                    "function version": {
                        "required": false
                    }
                }
            },
            "DisableReplication": {
                "url": "",
                "description": "Grants permission to disable replication for a Lambda@Edge function",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "EnableReplication": {
                "url": "",
                "description": "Grants permission to enable replication for a Lambda@Edge function",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "GetAccountSettings": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_GetAccountSettings.html",
                "description": "Grants permission to view details about an account's limits and usage in an AWS Region",
                "accessLevel": "Read"
            },
            "GetAlias": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_GetAlias.html",
                "description": "Grants permission to view details about an AWS Lambda function alias",
                "accessLevel": "Read",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "GetEventSourceMapping": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_GetEventSourceMapping.html",
                "description": "Grants permission to view details about an AWS Lambda event source mapping",
                "accessLevel": "Read",
                "resourceTypes": {
                    "eventSourceMapping": {
                        "required": true
                    }
                },
                "conditions": [
                    "lambda:FunctionArn"
                ]
            },
            "GetFunction": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_GetFunction.html",
                "description": "Grants permission to view details about an AWS Lambda function",
                "accessLevel": "Read",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "GetFunctionConcurrency": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_GetFunctionConcurrency.html",
                "description": "Grants permission to view details about the reserved concurrency configuration for a function",
                "accessLevel": "Read",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "GetFunctionConfiguration": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_GetFunctionConfiguration.html",
                "description": "Grants permission to view details about the version-specific settings of an AWS Lambda function or version",
                "accessLevel": "Read",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "GetFunctionEventInvokeConfig": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_GetFunctionEventInvokeConfig.html",
                "description": "Grants permission to view the configuration for asynchronous invocation for a function, version, or alias",
                "accessLevel": "Read",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "GetLayerVersion": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_GetLayerVersion.html",
                "description": "Grants permission to view details about a version of an AWS Lambda layer",
                "accessLevel": "Read",
                "resourceTypes": {
                    "layerVersion": {
                        "required": true
                    }
                }
            },
            "GetLayerVersionByArn": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_GetLayerVersionByArn.html",
                "description": "Grants permission to view details about a version of an AWS Lambda layer",
                "accessLevel": "Read",
                "resourceTypes": {
                    "layerVersion": {
                        "required": true
                    }
                }
            },
            "GetLayerVersionPolicy": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_GetLayerVersionPolicy.html",
                "description": "Grants permission to view the resource-based policy for a version of an AWS Lambda layer",
                "accessLevel": "Read",
                "resourceTypes": {
                    "layerVersion": {
                        "required": true
                    }
                }
            },
            "GetPolicy": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_GetPolicy.html",
                "description": "Grants permission to view the resource-based policy for an AWS Lambda function, version, or alias",
                "accessLevel": "Read",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "GetProvisionedConcurrencyConfig": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_GetProvisionedConcurrencyConfig.html",
                "description": "Grants permission to view the provisioned concurrency configuration for an AWS Lambda function's alias or version",
                "accessLevel": "Read",
                "resourceTypes": {
                    "function alias": {
                        "required": false
                    },
                    "function version": {
                        "required": false
                    }
                }
            },
            "InvokeAsync": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_InvokeAsync.html",
                "description": "(Deprecated) Grants permission to invoke a function asynchronously",
                "accessLevel": "Write",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "InvokeFunction": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html",
                "description": "Grants permission to invoke an AWS Lambda function",
                "accessLevel": "Write",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "ListAliases": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_ListAliases.html",
                "description": "Grants permission to retrieve a list of aliases for an AWS Lambda function",
                "accessLevel": "List",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "ListEventSourceMappings": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_ListEventSourceMappings.html",
                "description": "Grants permission to retrieve a list of AWS Lambda event source mappings",
                "accessLevel": "List"
            },
            "ListFunctionEventInvokeConfigs": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_ListFunctionEventInvokeConfigs.html",
                "description": "Grants permission to retrieve a list of configurations for asynchronous invocation for a function",
                "accessLevel": "List",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "ListFunctions": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_ListFunctions.html",
                "description": "Grants permission to retrieve a list of AWS Lambda functions, with the version-specific configuration of each function",
                "accessLevel": "List"
            },
            "ListLayerVersions": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayerVersions.html",
                "description": "Grants permission to retrieve a list of versions of an AWS Lambda layer",
                "accessLevel": "List"
            },
            "ListLayers": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayers.html",
                "description": "Grants permission to retrieve a list of AWS Lambda layers, with details about the latest version of each layer",
                "accessLevel": "List"
            },
            "ListProvisionedConcurrencyConfigs": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_ListProvisionedConcurrencyConfigs.html",
                "description": "Grants permission to retrieve a list of provisioned concurrency configurations for an AWS Lambda function",
                "accessLevel": "List",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "ListTags": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_ListTags.html",
                "description": "Grants permission to retrieve a list of tags for an AWS Lambda function",
                "accessLevel": "Read",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "ListVersionsByFunction": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_ListVersionsByFunction.html",
                "description": "Grants permission to retrieve a list of versions for an AWS Lambda function",
                "accessLevel": "List",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "PublishLayerVersion": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_PublishLayerVersion.html",
                "description": "Grants permission to create an AWS Lambda layer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "layer": {
                        "required": true
                    }
                }
            },
            "PublishVersion": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_PublishVersion.html",
                "description": "Grants permission to create an AWS Lambda function version",
                "accessLevel": "Write",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "PutFunctionConcurrency": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_PutFunctionConcurrency.html",
                "description": "Grants permission to configure reserved concurrency for an AWS Lambda function",
                "accessLevel": "Write",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "PutFunctionEventInvokeConfig": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_PutFunctionEventInvokeConfig.html",
                "description": "Grants permission to configures options for asynchronous invocation on an AWS Lambda function, version, or alias",
                "accessLevel": "Write",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "PutProvisionedConcurrencyConfig": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_PutProvisionedConcurrencyConfig.html",
                "description": "Grants permission to configure provisioned concurrency for an AWS Lambda function's alias or version",
                "accessLevel": "Write",
                "resourceTypes": {
                    "function alias": {
                        "required": false
                    },
                    "function version": {
                        "required": false
                    }
                }
            },
            "RemoveLayerVersionPermission": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_RemoveLayerVersionPermission.html",
                "description": "Grants permission to remove a statement from the permissions policy for a version of an AWS Lambda layer",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "layerVersion": {
                        "required": true
                    }
                }
            },
            "RemovePermission": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_RemovePermission.html",
                "description": "Grants permission to revoke function-use permission from an AWS service or another account",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                },
                "conditions": [
                    "lambda:Principal"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_TagResources.html",
                "description": "Grants permission to add tags to an AWS Lambda function",
                "accessLevel": "Write",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_UntagResource.html",
                "description": "Grants permission to remove tags from an AWS Lambda function",
                "accessLevel": "Write",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "UpdateAlias": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateAlias.html",
                "description": "Grants permission to update the configuration of an AWS Lambda function's alias",
                "accessLevel": "Write",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "UpdateEventSourceMapping": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateEventSourceMapping.html",
                "description": "Grants permission to update the configuration of an AWS Lambda event source mapping",
                "accessLevel": "Write",
                "resourceTypes": {
                    "eventSourceMapping": {
                        "required": true
                    }
                },
                "conditions": [
                    "lambda:FunctionArn"
                ]
            },
            "UpdateFunctionCode": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionCode.html",
                "description": "Grants permission to update the code of an AWS Lambda function",
                "accessLevel": "Write",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            },
            "UpdateFunctionConfiguration": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionConfiguration.html",
                "description": "Grants permission to modify the version-specific settings of an AWS Lambda function",
                "accessLevel": "Write",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                },
                "conditions": [
                    "lambda:Layer"
                ]
            },
            "UpdateFunctionEventInvokeConfig": {
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionEventInvokeConfig.html",
                "description": "Grants permission to modify the configuration for asynchronous invocation for an AWS Lambda function, version, or alias",
                "accessLevel": "Write",
                "resourceTypes": {
                    "function": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "function": {
                "name": "function",
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html",
                "arn": "arn:${Partition}:lambda:${Region}:${Account}:function:${FunctionName}",
                "conditionKeys": []
            },
            "function version": {
                "name": "function version",
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html",
                "arn": "arn:${Partition}:lambda:${Region}:${Account}:function:${FunctionName}:${Version}",
                "conditionKeys": []
            },
            "function alias": {
                "name": "function alias",
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html",
                "arn": "arn:${Partition}:lambda:${Region}:${Account}:function:${FunctionName}:${Alias}",
                "conditionKeys": []
            },
            "layer": {
                "name": "layer",
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html",
                "arn": "arn:${Partition}:lambda:${Region}:${Account}:layer:${LayerName}",
                "conditionKeys": []
            },
            "layerVersion": {
                "name": "layerVersion",
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html",
                "arn": "arn:${Partition}:lambda:${Region}:${Account}:layer:${LayerName}:${LayerVersion}",
                "conditionKeys": []
            },
            "eventSourceMapping": {
                "name": "eventSourceMapping",
                "url": "https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html",
                "arn": "arn:${Partition}:lambda:${Region}:${Account}:event-source-mapping:${UUID}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to add permissions to the resource-based policy of a version of an AWS Lambda layer
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_AddLayerVersionPermission.html
     */
    addLayerVersionPermission() {
        this.add('lambda:AddLayerVersionPermission');
        return this;
    }
    /**
     * Grants permission to give an AWS service or another account permission to use an AWS Lambda function
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_AddPermission.html
     */
    addPermission() {
        this.add('lambda:AddPermission');
        return this;
    }
    /**
     * Grants permission to create an alias for a Lambda function version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_CreateAlias.html
     */
    createAlias() {
        this.add('lambda:CreateAlias');
        return this;
    }
    /**
     * Grants permission to create a mapping between an event source and an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_CreateEventSourceMapping.html
     */
    createEventSourceMapping() {
        this.add('lambda:CreateEventSourceMapping');
        return this;
    }
    /**
     * Grants permission to create an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_CreateFunction.html
     */
    createFunction() {
        this.add('lambda:CreateFunction');
        return this;
    }
    /**
     * Grants permission to delete an AWS Lambda function alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteAlias.html
     */
    deleteAlias() {
        this.add('lambda:DeleteAlias');
        return this;
    }
    /**
     * Grants permission to delete an AWS Lambda event source mapping
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteEventSourceMapping.html
     */
    deleteEventSourceMapping() {
        this.add('lambda:DeleteEventSourceMapping');
        return this;
    }
    /**
     * Grants permission to delete an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteFunction.html
     */
    deleteFunction() {
        this.add('lambda:DeleteFunction');
        return this;
    }
    /**
     * Grants permission to remove a concurrent execution limit from an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteFunctionConcurrency.html
     */
    deleteFunctionConcurrency() {
        this.add('lambda:DeleteFunctionConcurrency');
        return this;
    }
    /**
     * Grants permission to delete the configuration for asynchronous invocation for an AWS Lambda function, version, or alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteFunctionEventInvokeConfig.html
     */
    deleteFunctionEventInvokeConfig() {
        this.add('lambda:DeleteFunctionEventInvokeConfig');
        return this;
    }
    /**
     * Grants permission to delete a version of an AWS Lambda layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteLayerVersion.html
     */
    deleteLayerVersion() {
        this.add('lambda:DeleteLayerVersion');
        return this;
    }
    /**
     * Grants permission to delete the provisioned concurrency configuration for an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteProvisionedConcurrencyConfig.html
     */
    deleteProvisionedConcurrencyConfig() {
        this.add('lambda:DeleteProvisionedConcurrencyConfig');
        return this;
    }
    /**
     * Grants permission to disable replication for a Lambda@Edge function
     *
     * Access Level: Permissions management
     */
    disableReplication() {
        this.add('lambda:DisableReplication');
        return this;
    }
    /**
     * Grants permission to enable replication for a Lambda@Edge function
     *
     * Access Level: Permissions management
     */
    enableReplication() {
        this.add('lambda:EnableReplication');
        return this;
    }
    /**
     * Grants permission to view details about an account's limits and usage in an AWS Region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetAccountSettings.html
     */
    getAccountSettings() {
        this.add('lambda:GetAccountSettings');
        return this;
    }
    /**
     * Grants permission to view details about an AWS Lambda function alias
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetAlias.html
     */
    getAlias() {
        this.add('lambda:GetAlias');
        return this;
    }
    /**
     * Grants permission to view details about an AWS Lambda event source mapping
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetEventSourceMapping.html
     */
    getEventSourceMapping() {
        this.add('lambda:GetEventSourceMapping');
        return this;
    }
    /**
     * Grants permission to view details about an AWS Lambda function
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetFunction.html
     */
    getFunction() {
        this.add('lambda:GetFunction');
        return this;
    }
    /**
     * Grants permission to view details about the reserved concurrency configuration for a function
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetFunctionConcurrency.html
     */
    getFunctionConcurrency() {
        this.add('lambda:GetFunctionConcurrency');
        return this;
    }
    /**
     * Grants permission to view details about the version-specific settings of an AWS Lambda function or version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetFunctionConfiguration.html
     */
    getFunctionConfiguration() {
        this.add('lambda:GetFunctionConfiguration');
        return this;
    }
    /**
     * Grants permission to view the configuration for asynchronous invocation for a function, version, or alias
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetFunctionEventInvokeConfig.html
     */
    getFunctionEventInvokeConfig() {
        this.add('lambda:GetFunctionEventInvokeConfig');
        return this;
    }
    /**
     * Grants permission to view details about a version of an AWS Lambda layer
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetLayerVersion.html
     */
    getLayerVersion() {
        this.add('lambda:GetLayerVersion');
        return this;
    }
    /**
     * Grants permission to view details about a version of an AWS Lambda layer
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetLayerVersionByArn.html
     */
    getLayerVersionByArn() {
        this.add('lambda:GetLayerVersionByArn');
        return this;
    }
    /**
     * Grants permission to view the resource-based policy for a version of an AWS Lambda layer
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetLayerVersionPolicy.html
     */
    getLayerVersionPolicy() {
        this.add('lambda:GetLayerVersionPolicy');
        return this;
    }
    /**
     * Grants permission to view the resource-based policy for an AWS Lambda function, version, or alias
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetPolicy.html
     */
    getPolicy() {
        this.add('lambda:GetPolicy');
        return this;
    }
    /**
     * Grants permission to view the provisioned concurrency configuration for an AWS Lambda function's alias or version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetProvisionedConcurrencyConfig.html
     */
    getProvisionedConcurrencyConfig() {
        this.add('lambda:GetProvisionedConcurrencyConfig');
        return this;
    }
    /**
     * (Deprecated) Grants permission to invoke a function asynchronously
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_InvokeAsync.html
     */
    invokeAsync() {
        this.add('lambda:InvokeAsync');
        return this;
    }
    /**
     * Grants permission to invoke an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html
     */
    invokeFunction() {
        this.add('lambda:InvokeFunction');
        return this;
    }
    /**
     * Grants permission to retrieve a list of aliases for an AWS Lambda function
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListAliases.html
     */
    listAliases() {
        this.add('lambda:ListAliases');
        return this;
    }
    /**
     * Grants permission to retrieve a list of AWS Lambda event source mappings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListEventSourceMappings.html
     */
    listEventSourceMappings() {
        this.add('lambda:ListEventSourceMappings');
        return this;
    }
    /**
     * Grants permission to retrieve a list of configurations for asynchronous invocation for a function
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListFunctionEventInvokeConfigs.html
     */
    listFunctionEventInvokeConfigs() {
        this.add('lambda:ListFunctionEventInvokeConfigs');
        return this;
    }
    /**
     * Grants permission to retrieve a list of AWS Lambda functions, with the version-specific configuration of each function
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListFunctions.html
     */
    listFunctions() {
        this.add('lambda:ListFunctions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of versions of an AWS Lambda layer
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayerVersions.html
     */
    listLayerVersions() {
        this.add('lambda:ListLayerVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of AWS Lambda layers, with details about the latest version of each layer
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayers.html
     */
    listLayers() {
        this.add('lambda:ListLayers');
        return this;
    }
    /**
     * Grants permission to retrieve a list of provisioned concurrency configurations for an AWS Lambda function
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListProvisionedConcurrencyConfigs.html
     */
    listProvisionedConcurrencyConfigs() {
        this.add('lambda:ListProvisionedConcurrencyConfigs');
        return this;
    }
    /**
     * Grants permission to retrieve a list of tags for an AWS Lambda function
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListTags.html
     */
    listTags() {
        this.add('lambda:ListTags');
        return this;
    }
    /**
     * Grants permission to retrieve a list of versions for an AWS Lambda function
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListVersionsByFunction.html
     */
    listVersionsByFunction() {
        this.add('lambda:ListVersionsByFunction');
        return this;
    }
    /**
     * Grants permission to create an AWS Lambda layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_PublishLayerVersion.html
     */
    publishLayerVersion() {
        this.add('lambda:PublishLayerVersion');
        return this;
    }
    /**
     * Grants permission to create an AWS Lambda function version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_PublishVersion.html
     */
    publishVersion() {
        this.add('lambda:PublishVersion');
        return this;
    }
    /**
     * Grants permission to configure reserved concurrency for an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_PutFunctionConcurrency.html
     */
    putFunctionConcurrency() {
        this.add('lambda:PutFunctionConcurrency');
        return this;
    }
    /**
     * Grants permission to configures options for asynchronous invocation on an AWS Lambda function, version, or alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_PutFunctionEventInvokeConfig.html
     */
    putFunctionEventInvokeConfig() {
        this.add('lambda:PutFunctionEventInvokeConfig');
        return this;
    }
    /**
     * Grants permission to configure provisioned concurrency for an AWS Lambda function's alias or version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_PutProvisionedConcurrencyConfig.html
     */
    putProvisionedConcurrencyConfig() {
        this.add('lambda:PutProvisionedConcurrencyConfig');
        return this;
    }
    /**
     * Grants permission to remove a statement from the permissions policy for a version of an AWS Lambda layer
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_RemoveLayerVersionPermission.html
     */
    removeLayerVersionPermission() {
        this.add('lambda:RemoveLayerVersionPermission');
        return this;
    }
    /**
     * Grants permission to revoke function-use permission from an AWS service or another account
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_RemovePermission.html
     */
    removePermission() {
        this.add('lambda:RemovePermission');
        return this;
    }
    /**
     * Grants permission to add tags to an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_TagResources.html
     */
    tagResource() {
        this.add('lambda:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_UntagResource.html
     */
    untagResource() {
        this.add('lambda:UntagResource');
        return this;
    }
    /**
     * Grants permission to update the configuration of an AWS Lambda function's alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateAlias.html
     */
    updateAlias() {
        this.add('lambda:UpdateAlias');
        return this;
    }
    /**
     * Grants permission to update the configuration of an AWS Lambda event source mapping
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateEventSourceMapping.html
     */
    updateEventSourceMapping() {
        this.add('lambda:UpdateEventSourceMapping');
        return this;
    }
    /**
     * Grants permission to update the code of an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionCode.html
     */
    updateFunctionCode() {
        this.add('lambda:UpdateFunctionCode');
        return this;
    }
    /**
     * Grants permission to modify the version-specific settings of an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionConfiguration.html
     */
    updateFunctionConfiguration() {
        this.add('lambda:UpdateFunctionConfiguration');
        return this;
    }
    /**
     * Grants permission to modify the configuration for asynchronous invocation for an AWS Lambda function, version, or alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionEventInvokeConfig.html
     */
    updateFunctionEventInvokeConfig() {
        this.add('lambda:UpdateFunctionEventInvokeConfig');
        return this;
    }
    /**
     * Adds a resource of type function to the statement
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * @param functionName - Identifier for the functionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onFunction(functionName, account, region, partition) {
        var arn = 'arn:${Partition}:lambda:${Region}:${Account}:function:${FunctionName}';
        arn = arn.replace('${FunctionName}', functionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type function version to the statement
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * @param functionName - Identifier for the functionName.
     * @param version - Identifier for the version.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onFunctionVersion(functionName, version, account, region, partition) {
        var arn = 'arn:${Partition}:lambda:${Region}:${Account}:function:${FunctionName}:${Version}';
        arn = arn.replace('${FunctionName}', functionName);
        arn = arn.replace('${Version}', version);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type function alias to the statement
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * @param functionName - Identifier for the functionName.
     * @param alias - Identifier for the alias.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onFunctionAlias(functionName, alias, account, region, partition) {
        var arn = 'arn:${Partition}:lambda:${Region}:${Account}:function:${FunctionName}:${Alias}';
        arn = arn.replace('${FunctionName}', functionName);
        arn = arn.replace('${Alias}', alias);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type layer to the statement
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * @param layerName - Identifier for the layerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLayer(layerName, account, region, partition) {
        var arn = 'arn:${Partition}:lambda:${Region}:${Account}:layer:${LayerName}';
        arn = arn.replace('${LayerName}', layerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type layerVersion to the statement
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * @param layerName - Identifier for the layerName.
     * @param layerVersion - Identifier for the layerVersion.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLayerVersion(layerName, layerVersion, account, region, partition) {
        var arn = 'arn:${Partition}:lambda:${Region}:${Account}:layer:${LayerName}:${LayerVersion}';
        arn = arn.replace('${LayerName}', layerName);
        arn = arn.replace('${LayerVersion}', layerVersion);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type eventSourceMapping to the statement
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * @param uUID - Identifier for the uUID.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEventSourceMapping(uUID, account, region, partition) {
        var arn = 'arn:${Partition}:lambda:${Region}:${Account}:event-source-mapping:${UUID}';
        arn = arn.replace('${UUID}', uUID);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by the ARN of an AWS Lambda function
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifFunctionArn(value, operator) {
        return this.if(`lambda:FunctionArn`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the ARN of an AWS Lambda layer
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLayer(value, operator) {
        return this.if(`lambda:Layer`, value, operator || 'StringLike');
    }
    /**
     * Filters access by restricting the AWS service or account that can invoke a function
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPrincipal(value, operator) {
        return this.if(`lambda:Principal`, value, operator || 'StringLike');
    }
}
exports.Lambda = Lambda;
//# sourceMappingURL=data:application/json;base64,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