"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service aws-marketplace-catalog
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsmarketplacecatalog.html
 */
class AwsMarketplaceCatalog extends shared_1.PolicyStatement {
    /**
     * Action provider for service aws-marketplace-catalog
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsmarketplacecatalog.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'aws-marketplace-catalog';
        this.actions = {
            "CancelChangeSet": {
                "url": "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_CancelChangeSet.html",
                "description": "Cancels a running change set.",
                "accessLevel": "Write"
            },
            "CompleteTask": {
                "url": "",
                "description": "Complete an existing task and submit the content to the associated change.",
                "accessLevel": "Write"
            },
            "DescribeChangeSet": {
                "url": "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_DescribeChangeSet.html",
                "description": "Returns the details of an existing change set.",
                "accessLevel": "Read"
            },
            "DescribeEntity": {
                "url": "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_DescribeEntity.html",
                "description": "Returns the details of an existing entity.",
                "accessLevel": "Read"
            },
            "DescribeTask": {
                "url": "",
                "description": "Returns the details of an existing task.",
                "accessLevel": "Read"
            },
            "ListChangeSets": {
                "url": "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_ListChangeSets.html",
                "description": "Lists existing change sets.",
                "accessLevel": "Read"
            },
            "ListEntities": {
                "url": "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_ListEntities.html",
                "description": "Lists existing entities.",
                "accessLevel": "Read"
            },
            "ListTasks": {
                "url": "",
                "description": "Lists existing tasks.",
                "accessLevel": "List"
            },
            "StartChangeSet": {
                "url": "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_StartChangeSet.html",
                "description": "Requests a new change set.",
                "accessLevel": "Write",
                "conditions": [
                    "catalog:ChangeType"
                ]
            },
            "UpdateTask": {
                "url": "",
                "description": "Update the content of an existing task.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "Entity": {
                "name": "Entity",
                "url": "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_DescribeEntity.html#API_DescribeEntity_ResponseSyntax",
                "arn": "arn:${Partition}:aws-marketplace:${Region}:${Account}:${Catalog}/${EntityType}/${ResourceId}",
                "conditionKeys": []
            },
            "ChangeSet": {
                "name": "ChangeSet",
                "url": "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_StartChangeSet.html#API_StartChangeSet_ResponseSyntax",
                "arn": "arn:${Partition}:aws-marketplace:${Region}:${Account}:${Catalog}/ChangeSet/${ResourceId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Cancels a running change set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_CancelChangeSet.html
     */
    cancelChangeSet() {
        this.add('aws-marketplace-catalog:CancelChangeSet');
        return this;
    }
    /**
     * Complete an existing task and submit the content to the associated change.
     *
     * Access Level: Write
     */
    completeTask() {
        this.add('aws-marketplace-catalog:CompleteTask');
        return this;
    }
    /**
     * Returns the details of an existing change set.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_DescribeChangeSet.html
     */
    describeChangeSet() {
        this.add('aws-marketplace-catalog:DescribeChangeSet');
        return this;
    }
    /**
     * Returns the details of an existing entity.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_DescribeEntity.html
     */
    describeEntity() {
        this.add('aws-marketplace-catalog:DescribeEntity');
        return this;
    }
    /**
     * Returns the details of an existing task.
     *
     * Access Level: Read
     */
    describeTask() {
        this.add('aws-marketplace-catalog:DescribeTask');
        return this;
    }
    /**
     * Lists existing change sets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_ListChangeSets.html
     */
    listChangeSets() {
        this.add('aws-marketplace-catalog:ListChangeSets');
        return this;
    }
    /**
     * Lists existing entities.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_ListEntities.html
     */
    listEntities() {
        this.add('aws-marketplace-catalog:ListEntities');
        return this;
    }
    /**
     * Lists existing tasks.
     *
     * Access Level: List
     */
    listTasks() {
        this.add('aws-marketplace-catalog:ListTasks');
        return this;
    }
    /**
     * Requests a new change set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_Operations.htmlAPI_StartChangeSet.html
     */
    startChangeSet() {
        this.add('aws-marketplace-catalog:StartChangeSet');
        return this;
    }
    /**
     * Update the content of an existing task.
     *
     * Access Level: Write
     */
    updateTask() {
        this.add('aws-marketplace-catalog:UpdateTask');
        return this;
    }
    /**
     * Adds a resource of type Entity to the statement
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_DescribeEntity.html#API_DescribeEntity_ResponseSyntax
     *
     * @param catalog - Identifier for the catalog.
     * @param entityType - Identifier for the entityType.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEntity(catalog, entityType, resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:aws-marketplace:${Region}:${Account}:${Catalog}/${EntityType}/${ResourceId}';
        arn = arn.replace('${Catalog}', catalog);
        arn = arn.replace('${EntityType}', entityType);
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ChangeSet to the statement
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_StartChangeSet.html#API_StartChangeSet_ResponseSyntax
     *
     * @param catalog - Identifier for the catalog.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onChangeSet(catalog, resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:aws-marketplace:${Region}:${Account}:${Catalog}/ChangeSet/${ResourceId}';
        arn = arn.replace('${Catalog}', catalog);
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Enables you to verify change type in the StartChangeSet request.
     *
     * https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/api-access-control.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifChangeType(value, operator) {
        return this.if(`catalog:ChangeType`, value, operator || 'StringLike');
    }
}
exports.AwsMarketplaceCatalog = AwsMarketplaceCatalog;
//# sourceMappingURL=data:application/json;base64,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