"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service outposts
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsoutposts.html
 */
class Outposts extends shared_1.PolicyStatement {
    /**
     * Action provider for service outposts
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsoutposts.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'outposts';
        this.actions = {
            "CreateOutpost": {
                "url": "https://docs.aws.amazon.com/outposts/latest/APIReference/API_CreateOutpost.html",
                "description": "Creates an Outpost",
                "accessLevel": "Write"
            },
            "GetOutpost": {
                "url": "https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetOutpost.html",
                "description": "Gets information about the specified Outpost",
                "accessLevel": "Read"
            },
            "GetOutpostInstanceTypes": {
                "url": "https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetOutpostInstanceTypes.html",
                "description": "Lists the instance types for the specified Outpost",
                "accessLevel": "Read"
            },
            "ListOutposts": {
                "url": "https://docs.aws.amazon.com/outposts/latest/APIReference/API_ListOutposts.html",
                "description": "List the Outposts for your AWS account",
                "accessLevel": "List"
            },
            "ListSites": {
                "url": "https://docs.aws.amazon.com/outposts/latest/APIReference/API_ListSites.html",
                "description": "Lists the sites for the specified AWS account",
                "accessLevel": "List"
            }
        };
        this.resourceTypes = {
            "Outpost": {
                "name": "Outpost",
                "url": "https://docs.aws.amazon.com/outposts/latest/APIReference/API_Outpost.html",
                "arn": "arn:${Partition}:outposts:${Region}:${Account}:outpost/${OutpostId}",
                "conditionKeys": []
            },
            "Site": {
                "name": "Site",
                "url": "https://docs.aws.amazon.com/outposts/latest/APIReference/API_Site.html",
                "arn": "arn:${Partition}:outposts:${Region}:${Account}:site/${SiteId}",
                "conditionKeys": []
            },
            "Order": {
                "name": "Order",
                "url": "https://docs.aws.amazon.com/outposts/latest/APIReference/Welcome.html",
                "arn": "arn:${Partition}:outposts:${Region}:${Account}:order/${OrderId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Creates an Outpost
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_CreateOutpost.html
     */
    createOutpost() {
        this.add('outposts:CreateOutpost');
        return this;
    }
    /**
     * Gets information about the specified Outpost
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetOutpost.html
     */
    getOutpost() {
        this.add('outposts:GetOutpost');
        return this;
    }
    /**
     * Lists the instance types for the specified Outpost
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetOutpostInstanceTypes.html
     */
    getOutpostInstanceTypes() {
        this.add('outposts:GetOutpostInstanceTypes');
        return this;
    }
    /**
     * List the Outposts for your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_ListOutposts.html
     */
    listOutposts() {
        this.add('outposts:ListOutposts');
        return this;
    }
    /**
     * Lists the sites for the specified AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_ListSites.html
     */
    listSites() {
        this.add('outposts:ListSites');
        return this;
    }
    /**
     * Adds a resource of type Outpost to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_Outpost.html
     *
     * @param outpostId - Identifier for the outpostId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOutpost(outpostId, account, region, partition) {
        var arn = 'arn:${Partition}:outposts:${Region}:${Account}:outpost/${OutpostId}';
        arn = arn.replace('${OutpostId}', outpostId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Site to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_Site.html
     *
     * @param siteId - Identifier for the siteId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSite(siteId, account, region, partition) {
        var arn = 'arn:${Partition}:outposts:${Region}:${Account}:site/${SiteId}';
        arn = arn.replace('${SiteId}', siteId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Order to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/Welcome.html
     *
     * @param orderId - Identifier for the orderId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOrder(orderId, account, region, partition) {
        var arn = 'arn:${Partition}:outposts:${Region}:${Account}:order/${OrderId}';
        arn = arn.replace('${OrderId}', orderId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Outposts = Outposts;
//# sourceMappingURL=data:application/json;base64,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