"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service resource-groups
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsresourcegroups.html
 */
class ResourceGroups extends shared_1.PolicyStatement {
    /**
     * Action provider for service resource-groups
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsresourcegroups.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'resource-groups';
        this.actions = {
            "CreateGroup": {
                "url": "https://docs.aws.amazon.com/ARG/latest/APIReference/API_CreateGroup.html",
                "description": "Creates a group with a specified name, description, and resource query.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteGroup": {
                "url": "https://docs.aws.amazon.com/ARG/latest/APIReference/API_DeleteGroup.html",
                "description": "Deletes a specified resource group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "GetGroup": {
                "url": "https://docs.aws.amazon.com/ARG/latest/APIReference/API_GetGroup.html",
                "description": "Gets information of a specified resource group",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "GetGroupQuery": {
                "url": "https://docs.aws.amazon.com/ARG/latest/APIReference/API_GetGroupQuery.html",
                "description": "Gets the query associated with a specified resource group",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "GetTags": {
                "url": "https://docs.aws.amazon.com/ARG/latest/APIReference/API_GetTags.html",
                "description": "Gets the tags associated with a specified resource group",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "ListGroupResources": {
                "url": "https://docs.aws.amazon.com/ARG/latest/APIReference/API_ListGroupResources.html",
                "description": "Lists the resources that are member of a specified resource group",
                "accessLevel": "List",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "ListGroups": {
                "url": "https://docs.aws.amazon.com/ARG/latest/APIReference/API_ListGroups.html",
                "description": "Lists all resource groups",
                "accessLevel": "List",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "SearchResources": {
                "url": "https://docs.aws.amazon.com/ARG/latest/APIReference/API_SearchResources.html",
                "description": "Returns a list of AWS resource identifiers matching the given query",
                "accessLevel": "List"
            },
            "Tag": {
                "url": "https://docs.aws.amazon.com/ARG/latest/APIReference/API_Tag.html",
                "description": "Tags a specified resource group",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "Untag": {
                "url": "https://docs.aws.amazon.com/ARG/latest/APIReference/API_Untag.html",
                "description": "Removes tags associated with a specified resource group",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateGroup": {
                "url": "https://docs.aws.amazon.com/ARG/latest/APIReference/API_UpdateGroup.html",
                "description": "Updates a specified resource group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "UpdateGroupQuery": {
                "url": "https://docs.aws.amazon.com/ARG/latest/APIReference/API_UpdateGroupQuery.html",
                "description": "Updates the query associated with a specified resource group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "group": {
                "name": "group",
                "url": "https://docs.aws.amazon.com/ARG/latest/userguide/welcome.html",
                "arn": "arn:${Partition}:resource-groups:${Region}:${Account}:group/${GroupName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates a group with a specified name, description, and resource query.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_CreateGroup.html
     */
    createGroup() {
        this.add('resource-groups:CreateGroup');
        return this;
    }
    /**
     * Deletes a specified resource group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_DeleteGroup.html
     */
    deleteGroup() {
        this.add('resource-groups:DeleteGroup');
        return this;
    }
    /**
     * Gets information of a specified resource group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_GetGroup.html
     */
    getGroup() {
        this.add('resource-groups:GetGroup');
        return this;
    }
    /**
     * Gets the query associated with a specified resource group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_GetGroupQuery.html
     */
    getGroupQuery() {
        this.add('resource-groups:GetGroupQuery');
        return this;
    }
    /**
     * Gets the tags associated with a specified resource group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_GetTags.html
     */
    getTags() {
        this.add('resource-groups:GetTags');
        return this;
    }
    /**
     * Lists the resources that are member of a specified resource group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_ListGroupResources.html
     */
    listGroupResources() {
        this.add('resource-groups:ListGroupResources');
        return this;
    }
    /**
     * Lists all resource groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_ListGroups.html
     */
    listGroups() {
        this.add('resource-groups:ListGroups');
        return this;
    }
    /**
     * Returns a list of AWS resource identifiers matching the given query
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_SearchResources.html
     */
    searchResources() {
        this.add('resource-groups:SearchResources');
        return this;
    }
    /**
     * Tags a specified resource group
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_Tag.html
     */
    tag() {
        this.add('resource-groups:Tag');
        return this;
    }
    /**
     * Removes tags associated with a specified resource group
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_Untag.html
     */
    untag() {
        this.add('resource-groups:Untag');
        return this;
    }
    /**
     * Updates a specified resource group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_UpdateGroup.html
     */
    updateGroup() {
        this.add('resource-groups:UpdateGroup');
        return this;
    }
    /**
     * Updates the query associated with a specified resource group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_UpdateGroupQuery.html
     */
    updateGroupQuery() {
        this.add('resource-groups:UpdateGroupQuery');
        return this;
    }
    /**
     * Adds a resource of type group to the statement
     *
     * https://docs.aws.amazon.com/ARG/latest/userguide/welcome.html
     *
     * @param groupName - Identifier for the groupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onGroup(groupName, account, region, partition) {
        var arn = 'arn:${Partition}:resource-groups:${Region}:${Account}:group/${GroupName}';
        arn = arn.replace('${GroupName}', groupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.ResourceGroups = ResourceGroups;
//# sourceMappingURL=data:application/json;base64,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