"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service serverlessrepo
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsserverlessapplicationrepository.html
 */
class Serverlessrepo extends shared_1.PolicyStatement {
    /**
     * Action provider for service serverlessrepo
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsserverlessapplicationrepository.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'serverlessrepo';
        this.actions = {
            "CreateApplication": {
                "url": "",
                "description": "Creates an application, optionally including an AWS SAM file to create the first application version in the same call.",
                "accessLevel": "Write"
            },
            "CreateApplicationVersion": {
                "url": "",
                "description": "Creates an application version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "applications": {
                        "required": true
                    }
                }
            },
            "CreateCloudFormationChangeSet": {
                "url": "",
                "description": "Creates an AWS CloudFormation ChangeSet for the given application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "applications": {
                        "required": true
                    }
                },
                "conditions": [
                    "serverlessrepo:applicationType"
                ]
            },
            "CreateCloudFormationTemplate": {
                "url": "",
                "description": "Creates an AWS CloudFormation template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "applications": {
                        "required": true
                    }
                },
                "conditions": [
                    "serverlessrepo:applicationType"
                ]
            },
            "DeleteApplication": {
                "url": "",
                "description": "Deletes the specified application",
                "accessLevel": "Write",
                "resourceTypes": {
                    "applications": {
                        "required": true
                    }
                }
            },
            "GetApplication": {
                "url": "",
                "description": "Gets the specified application.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "applications": {
                        "required": true
                    }
                },
                "conditions": [
                    "serverlessrepo:applicationType"
                ]
            },
            "GetApplicationPolicy": {
                "url": "",
                "description": "Gets the policy for the specified application.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "applications": {
                        "required": true
                    }
                }
            },
            "GetCloudFormationTemplate": {
                "url": "",
                "description": "Gets the specified AWS CloudFormation template",
                "accessLevel": "Read",
                "resourceTypes": {
                    "applications": {
                        "required": true
                    }
                }
            },
            "ListApplicationDependencies": {
                "url": "",
                "description": "Retrieves the list of applications nested in the containing application",
                "accessLevel": "List",
                "resourceTypes": {
                    "applications": {
                        "required": true
                    }
                },
                "conditions": [
                    "serverlessrepo:applicationType"
                ]
            },
            "ListApplicationVersions": {
                "url": "",
                "description": "Lists versions for the specified application owned by the requester.",
                "accessLevel": "List",
                "resourceTypes": {
                    "applications": {
                        "required": true
                    }
                },
                "conditions": [
                    "serverlessrepo:applicationType"
                ]
            },
            "ListApplications": {
                "url": "",
                "description": "Lists applications owned by the requester.",
                "accessLevel": "List"
            },
            "PutApplicationPolicy": {
                "url": "",
                "description": "Puts the policy for the specified application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "applications": {
                        "required": true
                    }
                }
            },
            "SearchApplications": {
                "url": "",
                "description": "Gets all applications authorized for this user",
                "accessLevel": "Read",
                "conditions": [
                    "serverlessrepo:applicationType"
                ]
            },
            "UnshareApplication": {
                "url": "",
                "description": "Unshares the specified application",
                "accessLevel": "Write",
                "resourceTypes": {
                    "applications": {
                        "required": true
                    }
                }
            },
            "UpdateApplication": {
                "url": "",
                "description": "Updates meta-data of the application",
                "accessLevel": "Write",
                "resourceTypes": {
                    "applications": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "applications": {
                "name": "applications",
                "url": "",
                "arn": "arn:${Partition}:serverlessrepo:${Region}:${Account}:applications/${ResourceId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Creates an application, optionally including an AWS SAM file to create the first application version in the same call.
     *
     * Access Level: Write
     */
    createApplication() {
        this.add('serverlessrepo:CreateApplication');
        return this;
    }
    /**
     * Creates an application version.
     *
     * Access Level: Write
     */
    createApplicationVersion() {
        this.add('serverlessrepo:CreateApplicationVersion');
        return this;
    }
    /**
     * Creates an AWS CloudFormation ChangeSet for the given application.
     *
     * Access Level: Write
     */
    createCloudFormationChangeSet() {
        this.add('serverlessrepo:CreateCloudFormationChangeSet');
        return this;
    }
    /**
     * Creates an AWS CloudFormation template
     *
     * Access Level: Write
     */
    createCloudFormationTemplate() {
        this.add('serverlessrepo:CreateCloudFormationTemplate');
        return this;
    }
    /**
     * Deletes the specified application
     *
     * Access Level: Write
     */
    deleteApplication() {
        this.add('serverlessrepo:DeleteApplication');
        return this;
    }
    /**
     * Gets the specified application.
     *
     * Access Level: Read
     */
    getApplication() {
        this.add('serverlessrepo:GetApplication');
        return this;
    }
    /**
     * Gets the policy for the specified application.
     *
     * Access Level: Read
     */
    getApplicationPolicy() {
        this.add('serverlessrepo:GetApplicationPolicy');
        return this;
    }
    /**
     * Gets the specified AWS CloudFormation template
     *
     * Access Level: Read
     */
    getCloudFormationTemplate() {
        this.add('serverlessrepo:GetCloudFormationTemplate');
        return this;
    }
    /**
     * Retrieves the list of applications nested in the containing application
     *
     * Access Level: List
     */
    listApplicationDependencies() {
        this.add('serverlessrepo:ListApplicationDependencies');
        return this;
    }
    /**
     * Lists versions for the specified application owned by the requester.
     *
     * Access Level: List
     */
    listApplicationVersions() {
        this.add('serverlessrepo:ListApplicationVersions');
        return this;
    }
    /**
     * Lists applications owned by the requester.
     *
     * Access Level: List
     */
    listApplications() {
        this.add('serverlessrepo:ListApplications');
        return this;
    }
    /**
     * Puts the policy for the specified application.
     *
     * Access Level: Write
     */
    putApplicationPolicy() {
        this.add('serverlessrepo:PutApplicationPolicy');
        return this;
    }
    /**
     * Gets all applications authorized for this user
     *
     * Access Level: Read
     */
    searchApplications() {
        this.add('serverlessrepo:SearchApplications');
        return this;
    }
    /**
     * Unshares the specified application
     *
     * Access Level: Write
     */
    unshareApplication() {
        this.add('serverlessrepo:UnshareApplication');
        return this;
    }
    /**
     * Updates meta-data of the application
     *
     * Access Level: Write
     */
    updateApplication() {
        this.add('serverlessrepo:UpdateApplication');
        return this;
    }
    /**
     * Adds a resource of type applications to the statement
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onApplications(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:serverlessrepo:${Region}:${Account}:applications/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Application type
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifApplicationType(value, operator) {
        return this.if(`serverlessrepo:applicationType`, value, operator || 'StringLike');
    }
}
exports.Serverlessrepo = Serverlessrepo;
//# sourceMappingURL=data:application/json;base64,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