# -*- coding: utf-8 -*-

# TransportTools, a library for massive analyses of internal voids in biomolecules and ligand transport through them
# Copyright (C) 2021  Jan Brezovsky, Aravind Selvaram Thirunavukarasu, Carlos Eduardo Sequeiros-Borja, Bartlomiej
# Surpeta, Nishita Mandal, Cedrix Jurgal Dongmo Foumthuim, Dheeraj Kumar Sarkar, Nikhil Agrawal  <janbre@amu.edu.pl>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

__version__ = '0.9.0'
__author__ = 'Jan Brezovsky, Aravind Selvaram Thirunavukarasu, Carlos Eduardo Sequeiros-Borja, Bartlomiej Surpeta, ' \
             'Nishita Mandal, Cedrix Jurgal Dongmo Foumthuim, Dheeraj Kumar Sarkar, Nikhil Agrawal'
__mail__ = 'janbre@amu.edu.pl'

import numpy as np
import os
from pathlib import Path
from typing import List, Tuple
from logging import getLogger

logger = getLogger(__name__)


def get_caver_color(color_id: int or None) -> List[float]:
    """
    Converts Pymol color IDs to RGB format, keeping within the set of 'reasonable' colors from CAVER rgb.py
    :param color_id: Pymol color ID
    :return: RBG colors in Pymol format Pymol
    """

    colors = [
        [0.0, 0.0, 1.0], [0.0, 1.0, 0.0], [1.0, 0.0, 0.0], [0.0, 1.0, 1.0], [1.0, 1.0, 0.0], [1.0, 0.0, 1.0],
        [0.71, 0.71, 0.97], [0.5, 0.99, 0.42], [0.99, 0.5, 0.42], [0.21, 0.5, 0.85], [0.5, 0.06, 0.87],
        [0.0, 0.87, 0.5], [0.86, 0.01, 0.5], [0.58, 0.82, 0.0], [0.89, 0.89, 0.6], [1.0, 0.41, 0.84], [0.41, 1.0, 0.85],
        [0.87, 0.49, 0.0], [0.62, 0.42, 0.69], [0.66, 0.67, 0.35], [0.4, 0.71, 0.62], [0.32, 1.0, 0.13],
        [0.72, 0.3, 1.0], [1.0, 0.82, 0.28], [0.96, 0.21, 0.25], [0.24, 0.22, 1.0], [0.0, 0.74, 0.79], [0.4, 0.71, 1.0],
        [0.75, 1.0, 0.2], [0.79, 0.07, 0.79], [0.87, 0.29, 0.6], [0.14, 1.0, 0.73], [0.45, 0.42, 1.0],
        [0.86, 0.59, 0.66], [0.61, 0.89, 0.66], [0.0, 0.36, 0.99], [0.86, 0.77, 0.03], [0.23, 0.99, 0.39],
        [1.0, 0.26, 0.0], [0.0, 0.98, 0.27], [0.15, 0.79, 1.0], [0.53, 0.62, 0.81], [0.88, 0.61, 0.22],
        [0.75, 0.47, 0.47], [0.73, 0.89, 0.42], [0.34, 0.94, 0.6], [0.77, 0.49, 0.87], [0.49, 0.85, 0.22],
        [0.69, 0.0, 1.0], [1.0, 0.1, 0.68], [0.24, 0.78, 0.78], [0.94, 0.23, 0.99], [0.87, 0.7, 0.46], [0.01, 0.6, 1.0],
        [0.33, 0.0, 1.0], [1.0, 0.42, 0.17], [0.61, 0.89, 0.89], [0.99, 0.0, 0.32], [0.22, 0.99, 0.96],
        [0.49, 0.28, 0.83], [0.48, 0.77, 0.43], [1.0, 0.17, 0.47], [0.91, 1.0, 0.36], [0.76, 0.28, 0.78],
        [0.7, 0.79, 0.18], [0.57, 1.0, 0.1], [0.79, 0.97, 0.0], [0.4, 0.47, 0.79], [0.19, 0.81, 0.56],
        [0.88, 0.33, 0.4], [0.62, 0.53, 1.0], [1.0, 0.45, 0.62], [0.43, 0.8, 0.81], [1.0, 0.73, 0.65],
        [0.91, 0.64, 0.84], [1.0, 0.63, 0.06], [0.44, 0.19, 1.0], [0.94, 0.97, 0.18], [0.23, 0.61, 1.0],
        [0.89, 0.41, 1.0], [0.05, 0.18, 0.99], [0.33, 0.64, 0.84], [0.94, 0.24, 0.8], [0.26, 0.4, 1.0],
        [0.01, 0.93, 0.83], [0.13, 0.99, 0.13], [0.42, 0.91, 0.0], [0.66, 0.16, 0.88], [0.78, 0.13, 0.62],
        [0.37, 0.9, 1.0], [0.13, 0.99, 0.54], [0.74, 1.0, 0.59], [0.32, 0.84, 0.45], [1.0, 0.86, 0.46],
        [0.96, 0.0, 0.82], [0.04, 0.57, 0.82], [0.62, 0.4, 0.88], [0.86, 0.42, 0.73], [0.85, 0.08, 0.96],
        [0.82, 0.79, 0.31], [0.05, 0.86, 0.67], [0.37, 0.96, 0.29], [0.61, 0.95, 0.29], [0.48, 0.99, 0.69],
        [0.82, 0.48, 0.31], [1.0, 0.08, 0.15], [0.63, 0.0, 0.78], [0.78, 0.62, 0.07], [1.0, 0.65, 0.34],
        [0.55, 0.77, 1.0], [1.0, 0.82, 0.11], [0.16, 0.05, 0.99], [0.48, 0.88, 0.55], [0.63, 0.8, 0.52],
        [0.98, 0.61, 0.54], [0.34, 0.3, 0.9], [0.14, 0.68, 0.88], [0.27, 0.94, 0.81], [0.11, 0.48, 0.99],
        [0.0, 0.84, 0.97], [0.88, 0.0, 0.66], [0.71, 0.31, 0.63], [1.0, 0.54, 0.75], [0.56, 0.3, 1.0],
        [0.45, 0.57, 0.94], [0.13, 0.69, 0.72], [0.22, 0.98, 0.0], [1.0, 0.33, 0.51], [0.57, 0.11, 1.0],
        [0.78, 0.8, 0.54], [0.67, 0.16, 0.73], [0.05, 1.0, 0.42], [0.75, 0.46, 0.63], [0.6, 0.8, 0.33],
        [0.14, 0.32, 0.94], [0.59, 1.0, 0.55], [0.85, 0.85, 0.17], [0.28, 0.79, 0.92], [0.78, 0.58, 1.0],
        [0.14, 0.88, 0.88], [1.0, 0.36, 0.32], [0.89, 0.44, 0.51], [0.85, 0.18, 0.5], [0.73, 0.65, 0.21],
        [1.0, 0.3, 0.67], [0.61, 0.99, 0.77], [0.7, 0.72, 0.0], [0.0, 1.0, 0.63], [0.23, 0.94, 0.24],
        [0.81, 0.22, 0.91], [0.28, 0.66, 0.7], [0.68, 0.91, 0.08], [0.42, 0.59, 0.72], [1.0, 0.41, 0.02],
        [0.8, 0.62, 0.35], [0.5, 0.8, 0.68], [0.35, 0.83, 0.69], [0.36, 0.99, 0.46], [0.98, 0.13, 0.89],
        [1.0, 0.01, 0.56], [0.67, 0.6, 0.88], [0.99, 0.7, 0.2], [0.86, 0.35, 0.87], [0.14, 0.89, 0.44],
        [0.99, 0.53, 0.27], [0.87, 0.48, 0.15], [0.58, 0.75, 0.86], [0.84, 0.58, 0.52], [0.85, 0.99, 0.49],
        [0.62, 0.29, 0.76], [0.74, 0.44, 1.0], [0.49, 1.0, 0.22], [0.32, 0.52, 0.94], [0.92, 0.51, 0.9],
        [0.93, 0.12, 0.36], [0.47, 0.0, 0.99], [0.95, 0.29, 0.14], [0.91, 0.9, 0.03], [0.21, 0.89, 0.68],
        [0.86, 0.85, 0.43], [0.73, 0.14, 1.0], [0.54, 0.49, 0.8], [0.87, 0.15, 0.71], [0.74, 0.68, 0.47],
        [0.33, 0.13, 0.94], [0.77, 1.0, 0.34], [0.73, 0.01, 0.68], [0.47, 0.89, 0.91], [0.56, 0.65, 0.95],
        [0.01, 0.46, 0.89], [0.99, 0.74, 0.51], [0.34, 1.0, 0.72], [0.64, 1.0, 0.42], [0.02, 0.71, 0.93],
        [0.72, 0.88, 0.28], [0.72, 0.41, 0.78], [0.46, 0.88, 0.36], [0.97, 0.2, 0.59], [0.79, 0.35, 0.51],
        [0.99, 0.13, 0.03], [0.88, 0.7, 0.59], [0.52, 0.89, 0.77], [0.83, 0.72, 0.16], [0.58, 0.82, 0.14],
        [0.12, 0.94, 0.32], [0.86, 0.92, 0.28], [0.89, 0.39, 0.24], [0.1, 1.0, 0.91], [0.0, 0.99, 0.14],
        [0.25, 0.93, 0.51], [0.99, 0.75, 0.0], [0.57, 0.88, 0.43], [0.42, 0.91, 0.13], [0.32, 0.79, 0.57],
        [0.62, 0.04, 0.9], [0.3, 0.42, 0.86], [0.53, 0.19, 0.9], [0.11, 0.81, 0.77], [0.86, 0.54, 0.78],
        [0.76, 0.84, 0.0], [0.48, 0.4, 0.87], [0.89, 0.62, 0.0], [0.71, 0.96, 0.71], [0.46, 0.71, 0.89],
        [0.99, 0.04, 0.44], [0.44, 1.0, 0.55], [0.64, 0.99, 0.0], [0.84, 1.0, 0.11], [0.2, 0.61, 0.79],
        [0.11, 0.91, 1.0], [0.9, 0.09, 0.58], [0.96, 0.76, 0.39], [0.7, 0.77, 0.39], [0.99, 0.25, 0.37],
        [0.76, 0.0, 0.9], [0.98, 0.51, 0.09], [1.0, 0.29, 0.89], [0.12, 0.66, 0.99], [0.62, 0.41, 1.0],
        [0.36, 0.88, 0.87], [0.94, 0.65, 0.73], [0.38, 0.71, 0.74], [0.62, 0.28, 0.9], [0.74, 0.35, 0.89],
        [0.91, 0.6, 0.41], [0.84, 0.3, 1.0], [0.91, 0.71, 0.28], [0.54, 0.49, 0.93], [0.86, 0.29, 0.72],
        [0.01, 0.74, 0.67], [0.24, 0.88, 1.0], [0.7, 0.89, 0.56], [0.71, 0.57, 0.42], [0.23, 1.0, 0.62],
        [0.98, 0.94, 0.29], [0.59, 0.72, 0.43], [0.48, 0.76, 0.55], [0.88, 0.14, 0.84], [0.44, 0.31, 0.96],
        [0.12, 0.57, 0.91], [0.35, 1.0, 0.0], [0.63, 0.21, 0.99], [0.66, 0.51, 0.8], [0.53, 0.37, 0.77],
        [0.51, 0.99, 0.0], [0.88, 0.48, 0.63], [0.21, 0.75, 0.67], [0.8, 0.61, 0.89], [0.33, 0.99, 0.93],
        [0.98, 0.41, 0.73], [0.59, 0.11, 0.8], [0.4, 0.91, 0.77], [0.31, 0.53, 0.79], [0.28, 0.9, 0.35],
        [0.78, 0.23, 0.68], [0.1, 0.88, 0.56], [0.52, 0.89, 0.06], [0.9, 0.8, 0.53], [0.44, 0.89, 0.65],
        [0.53, 0.97, 0.85], [0.65, 0.91, 0.19], [0.64, 0.73, 0.25], [0.97, 0.11, 1.0], [0.85, 0.5, 0.42],
        [0.3, 0.68, 0.94], [0.75, 0.17, 0.81], [0.41, 0.21, 0.89], [0.99, 0.1, 0.26], [0.94, 0.72, 0.1],
        [0.97, 0.55, 0.64], [0.92, 0.39, 0.09], [0.03, 0.99, 0.74], [0.99, 0.19, 0.13], [0.76, 0.5, 0.73],
        [0.24, 0.6, 0.89], [0.92, 0.44, 0.36], [0.65, 0.8, 0.94], [0.0, 0.99, 0.52], [0.9, 0.6, 0.11],
        [0.14, 0.42, 0.9], [0.33, 0.75, 0.83], [0.51, 0.73, 0.77], [0.05, 0.82, 0.87], [0.45, 0.81, 0.99],
        [0.71, 0.69, 0.12], [0.11, 1.0, 0.0], [0.99, 0.58, 0.18], [0.61, 1.0, 0.65], [0.81, 0.37, 0.65],
        [0.9, 0.28, 0.49], [0.36, 0.61, 1.0], [0.44, 0.57, 0.82], [0.97, 0.0, 0.71], [1.0, 0.14, 0.78],
        [0.22, 0.51, 0.99], [0.92, 0.01, 0.92], [0.74, 0.99, 0.48], [0.1, 0.78, 0.62], [1.0, 0.93, 0.09],
        [0.67, 0.49, 0.91], [0.18, 0.78, 0.88], [0.92, 0.78, 0.21], [0.76, 0.11, 0.9], [0.91, 0.3, 0.3],
        [0.79, 0.56, 0.26], [0.18, 1.0, 0.84], [0.42, 0.9, 0.46], [0.67, 0.8, 0.08], [0.79, 0.91, 0.63],
        [0.92, 0.37, 0.65], [0.4, 0.04, 0.92], [0.36, 0.37, 0.99], [0.35, 0.26, 1.0], [0.4, 0.78, 0.49],
        [0.17, 0.91, 0.78], [0.78, 0.9, 0.09], [0.1, 0.96, 0.64], [0.99, 0.5, 0.52], [0.01, 0.48, 1.0],
        [0.93, 0.87, 0.35], [0.75, 0.7, 0.31], [0.97, 0.21, 0.7], [0.9, 0.55, 0.31], [0.41, 0.48, 0.9],
        [0.24, 0.34, 0.91], [0.24, 0.69, 0.84], [0.77, 0.78, 0.1], [0.66, 1.0, 0.14], [0.92, 0.8, 0.63],
        [0.58, 0.0, 1.0], [0.0, 0.81, 0.59], [1.0, 0.39, 0.42], [0.43, 1.0, 0.07], [0.54, 0.92, 0.17],
        [0.68, 0.62, 1.0], [0.9, 1.0, 0.0], [0.87, 0.19, 0.62], [0.57, 0.21, 0.8], [0.69, 0.08, 0.83], [0.8, 0.9, 0.5],
        [0.33, 0.92, 0.2], [1.0, 0.88, 0.2], [0.0, 0.84, 0.76], [0.89, 0.05, 0.76], [0.59, 0.89, 0.54],
        [0.87, 0.45, 0.83], [0.1, 0.99, 0.23], [0.15, 0.16, 0.99], [0.47, 0.1, 0.96], [0.85, 0.18, 1.0],
        [0.85, 0.0, 0.84], [1.0, 0.86, 0.01], [0.92, 0.93, 0.45], [0.89, 0.22, 0.4], [1.0, 0.11, 0.55],
        [0.25, 0.11, 1.0], [0.38, 0.79, 0.91], [0.8, 0.77, 0.43], [0.78, 0.24, 0.57], [0.92, 0.11, 0.48],
        [0.99, 0.65, 0.45], [0.94, 0.34, 0.8], [0.95, 0.33, 0.97], [0.07, 0.66, 0.8], [0.47, 0.97, 0.32],
        [0.99, 0.55, 0.0], [0.99, 0.3, 0.23], [0.81, 0.41, 0.93], [0.22, 0.72, 0.99], [0.53, 0.93, 0.62],
        [0.42, 0.81, 0.6], [0.81, 0.4, 0.43], [0.91, 0.25, 0.9], [0.53, 0.56, 1.0], [0.75, 0.88, 0.19],
        [0.72, 0.24, 0.87], [0.64, 0.9, 0.36], [0.84, 0.23, 0.81], [0.02, 0.93, 0.93], [0.24, 0.98, 0.72],
        [0.15, 0.99, 0.45], [0.59, 0.83, 0.23], [0.85, 1.0, 0.21], [0.26, 0.89, 0.9], [0.81, 0.57, 0.16],
        [0.55, 0.38, 0.94], [0.07, 0.08, 1.0], [0.8, 0.88, 0.35], [0.81, 0.02, 0.59], [0.85, 0.5, 0.98],
        [0.31, 0.85, 0.79], [0.4, 0.51, 1.0], [0.05, 0.28, 0.97], [0.27, 0.87, 0.59], [0.64, 0.69, 0.91],
        [0.98, 0.0, 0.22], [0.79, 0.01, 1.0], [1.0, 0.51, 0.84], [0.79, 0.78, 0.22], [0.39, 1.0, 0.19],
        [0.47, 0.51, 0.73], [0.54, 0.88, 0.3], [0.57, 0.57, 0.89], [0.68, 0.4, 0.62], [0.81, 0.69, 0.0],
        [0.0, 0.55, 0.91], [0.75, 0.1, 0.71], [0.31, 0.74, 0.65], [0.66, 0.93, 0.49], [0.69, 0.35, 0.71],
        [0.69, 1.0, 0.28], [0.81, 0.48, 0.55], [0.92, 0.41, 0.9], [0.34, 0.79, 1.0], [0.01, 0.91, 0.59],
        [0.91, 0.49, 0.24], [0.16, 0.39, 1.0], [0.84, 0.71, 0.36], [0.55, 0.81, 0.6], [0.51, 0.8, 0.91],
        [0.23, 0.98, 0.15], [0.71, 0.99, 0.06], [0.39, 0.37, 0.85], [0.53, 0.2, 1.0], [0.62, 0.71, 1.0],
        [0.37, 0.87, 0.54], [0.09, 0.97, 0.82], [0.58, 1.0, 0.21], [0.41, 0.87, 0.27], [1.0, 0.0, 0.1],
        [0.49, 0.7, 1.0], [0.95, 0.65, 0.63], [0.81, 0.36, 0.79], [0.72, 0.78, 0.27], [0.69, 0.25, 0.7],
        [0.9, 0.16, 0.93], [0.92, 0.5, 0.71], [0.39, 0.08, 1.0], [0.28, 0.99, 0.31], [0.92, 0.36, 0.55],
        [0.96, 0.63, 0.26], [0.84, 0.94, 0.41], [0.39, 0.65, 0.91], [0.16, 0.73, 0.79], [0.93, 1.0, 0.09],
        [0.1, 0.77, 0.92], [0.0, 0.93, 0.36], [0.61, 0.9, 0.77], [0.98, 0.81, 0.57], [0.18, 0.91, 0.59],
        [0.91, 0.82, 0.09], [0.26, 0.31, 1.0], [0.54, 0.81, 0.49], [0.23, 0.87, 0.44], [0.14, 0.84, 0.68],
        [0.67, 0.33, 0.83], [0.6, 0.91, 0.02], [0.66, 0.09, 0.97], [0.1, 0.49, 0.86], [1.0, 0.33, 0.07],
        [0.71, 0.82, 0.48], [0.08, 0.9, 0.75], [0.37, 0.92, 0.38], [0.81, 0.67, 0.52], [0.3, 0.91, 0.7],
        [0.82, 0.68, 0.24], [0.44, 0.66, 0.8], [0.5, 0.8, 0.34], [0.07, 0.37, 0.93], [0.23, 0.44, 0.92],
        [0.91, 0.54, 0.48], [0.4, 1.0, 0.64], [0.05, 0.64, 0.89], [0.07, 0.94, 0.49], [0.73, 0.0, 0.79],
        [0.53, 0.29, 0.91], [0.62, 0.43, 0.79], [0.62, 0.81, 0.42], [0.57, 0.96, 0.37], [0.8, 0.62, 0.44],
        [0.55, 0.81, 0.79], [0.33, 0.61, 0.75], [0.58, 0.33, 0.83], [0.24, 0.0, 0.97], [0.51, 0.96, 0.5],
        [0.9, 0.55, 0.57], [0.9, 0.64, 0.52], [0.9, 0.91, 0.12], [0.56, 0.0, 0.84], [0.74, 0.42, 0.55],
        [0.13, 0.57, 0.82], [0.77, 0.53, 0.38], [0.99, 0.17, 0.33], [0.66, 0.35, 0.95], [0.82, 0.0, 0.73],
        [0.47, 0.99, 0.78], [0.46, 0.72, 0.69], [0.04, 0.76, 1.0], [0.36, 0.94, 0.06], [0.99, 0.57, 0.35],
        [1.0, 0.35, 0.6], [0.66, 0.22, 0.8], [0.3, 0.22, 0.93], [0.93, 0.03, 0.37], [1.0, 0.46, 0.33],
        [0.71, 0.53, 0.99], [0.73, 0.62, 0.3], [0.0, 0.67, 0.74], [0.9, 0.8, 0.3], [0.83, 0.08, 0.68], [0.7, 0.44, 0.7],
        [0.56, 0.85, 0.95], [0.84, 0.4, 0.57], [0.92, 0.37, 0.46], [0.91, 0.87, 0.22], [0.57, 0.94, 0.71],
        [0.14, 0.25, 1.0], [0.84, 0.54, 0.07], [0.91, 0.0, 0.58], [0.4, 0.83, 0.4], [0.26, 1.0, 0.88],
        [0.86, 0.67, 0.08], [0.16, 0.57, 0.99], [0.33, 0.45, 0.99], [0.65, 0.96, 0.59], [0.08, 0.74, 0.84],
        [0.94, 0.88, 0.52], [0.43, 0.83, 0.72], [0.26, 0.83, 0.51], [0.71, 0.92, 0.0], [0.78, 0.23, 0.99],
        [0.93, 0.68, 0.39], [0.78, 0.36, 0.99], [0.32, 0.77, 0.75], [0.33, 1.0, 0.82], [0.99, 0.73, 0.29],
        [0.91, 0.99, 0.28], [1.0, 0.04, 0.89], [0.53, 0.0, 0.92], [0.1, 0.0, 1.0], [0.06, 1.0, 0.07],
        [0.22, 0.85, 0.84], [0.98, 0.61, 0.8], [0.48, 1.0, 0.13], [0.91, 0.7, 0.02], [0.41, 0.94, 0.93],
        [0.21, 0.93, 0.33], [0.26, 0.82, 0.7], [0.37, 0.56, 0.88], [0.3, 1.0, 0.53], [0.91, 0.67, 0.17],
        [0.77, 0.94, 0.27], [0.83, 0.97, 0.58], [0.99, 0.21, 0.86], [0.73, 0.64, 0.93], [0.71, 0.42, 0.87],
        [0.99, 0.25, 0.52], [0.46, 0.92, 0.83], [0.98, 0.42, 0.5], [0.93, 0.81, 0.01], [0.82, 0.93, 0.17],
        [0.76, 0.31, 0.7], [0.3, 0.93, 0.43], [0.59, 0.12, 0.92], [0.17, 1.0, 0.28], [0.94, 0.47, 0.79],
        [0.92, 0.08, 0.68], [0.6, 0.49, 0.87], [0.84, 0.85, 0.04], [0.46, 0.14, 0.88], [0.92, 0.07, 0.86],
        [0.19, 0.84, 0.94], [0.37, 0.65, 0.67], [0.97, 0.5, 0.18], [0.41, 1.0, 0.39], [0.54, 0.86, 0.85],
        [0.94, 0.35, 0.0], [0.67, 0.07, 0.74], [0.41, 0.39, 0.93], [0.8, 0.85, 0.26], [0.66, 0.84, 0.0],
        [0.8, 0.29, 0.86], [0.6, 0.92, 0.11], [0.96, 0.3, 0.44], [1.0, 0.29, 0.76], [0.08, 0.86, 0.93],
        [0.48, 0.51, 0.86], [0.56, 0.85, 0.72], [0.81, 0.51, 0.66], [0.0, 1.0, 0.88], [0.88, 0.59, 0.91],
        [0.76, 0.71, 0.39], [0.88, 0.79, 0.38], [0.88, 0.01, 1.0], [0.68, 0.74, 0.46], [0.49, 0.86, 0.43],
        [0.5, 0.63, 0.89], [0.54, 0.72, 0.93], [0.97, 0.94, 0.38], [0.78, 0.76, 0.0], [0.44, 0.92, 0.21],
        [0.97, 0.42, 0.25], [0.92, 0.27, 0.66], [0.88, 0.64, 0.32], [0.22, 0.69, 0.75], [0.21, 0.27, 0.94],
        [0.6, 0.72, 0.35], [0.05, 1.0, 0.33], [0.69, 0.01, 0.86], [0.31, 1.0, 0.38], [0.26, 0.97, 0.07],
        [0.84, 0.08, 0.88], [0.98, 0.06, 0.76], [0.08, 0.76, 0.71], [0.22, 0.67, 0.92], [0.77, 0.41, 0.72],
        [0.43, 0.3, 0.88], [0.8, 0.16, 0.74], [0.92, 0.16, 0.54], [0.29, 1.0, 0.21], [1.0, 0.12, 0.4],
        [0.07, 0.55, 0.98], [0.73, 0.55, 0.91], [0.92, 0.46, 0.44], [1.0, 0.57, 0.46], [0.98, 0.04, 0.63],
        [0.71, 0.21, 0.95], [0.29, 0.94, 0.99], [0.0, 0.92, 0.71], [0.17, 0.93, 0.94], [0.8, 0.46, 0.79],
        [0.31, 0.69, 0.78], [0.44, 0.64, 0.99], [0.53, 0.44, 0.72], [0.35, 0.86, 0.62], [0.7, 0.96, 0.37],
        [0.76, 0.56, 0.49], [0.73, 0.95, 0.14], [0.75, 0.72, 0.19], [0.91, 0.47, 0.08], [0.78, 0.65, 0.15],
        [0.93, 0.37, 0.17], [0.82, 0.27, 0.5], [0.49, 0.86, 0.14], [0.85, 0.82, 0.59], [0.62, 0.77, 0.18],
        [0.85, 0.09, 0.52], [0.19, 0.88, 0.51], [0.16, 0.5, 0.92], [0.58, 0.99, 0.47], [0.92, 0.56, 0.83],
        [0.0, 0.94, 0.44], [0.77, 0.54, 0.8], [0.93, 0.81, 0.45], [0.32, 0.18, 1.0], [0.25, 0.8, 1.0],
        [1.0, 0.61, 0.69], [0.09, 0.92, 0.39], [0.87, 0.37, 0.33], [0.41, 0.95, 0.7], [0.63, 0.59, 0.95],
        [0.74, 0.06, 0.96], [0.03, 0.68, 1.0], [0.94, 0.04, 0.51], [0.82, 0.58, 0.0], [0.54, 0.44, 0.99],
        [0.95, 0.28, 0.59], [0.08, 0.89, 0.83], [0.07, 0.41, 1.0], [0.58, 0.67, 0.86], [0.77, 0.29, 0.94],
        [0.14, 0.92, 0.7], [0.73, 0.85, 0.12], [0.84, 0.74, 0.52], [0.48, 0.23, 0.94], [0.99, 0.43, 0.09],
        [0.99, 0.15, 0.2], [0.85, 0.63, 0.58], [0.3, 0.6, 0.95], [0.95, 0.74, 0.58], [1.0, 0.17, 0.96],
        [0.39, 0.76, 0.68], [0.85, 0.53, 0.87], [0.87, 0.92, 0.52], [0.91, 0.42, 0.01], [0.38, 0.55, 0.77],
        [0.47, 0.42, 0.78], [0.91, 0.52, 0.38], [0.94, 0.15, 0.65], [0.7, 0.91, 0.64], [0.24, 0.88, 0.76],
        [0.5, 0.36, 1.0], [0.33, 0.86, 0.95], [1.0, 0.83, 0.36], [0.19, 1.0, 0.07], [0.92, 0.55, 0.04],
        [0.68, 0.83, 0.34], [0.26, 1.0, 0.46], [0.99, 0.37, 0.92], [0.43, 0.98, 0.47], [0.25, 0.8, 0.62],
        [0.43, 0.92, 0.58], [0.48, 0.5, 0.98], [0.08, 0.99, 0.99], [0.41, 0.73, 0.83], [0.64, 0.29, 1.0],
        [0.79, 0.32, 0.59], [0.9, 0.58, 0.73], [0.34, 0.48, 0.85], [0.92, 0.21, 0.32], [0.84, 0.94, 0.06],
        [0.0, 0.68, 0.84], [0.06, 0.96, 0.56], [0.37, 0.96, 0.53], [0.56, 0.42, 0.83], [0.99, 0.25, 0.08],
        [0.93, 0.33, 0.71], [0.99, 0.24, 0.19], [0.5, 0.57, 0.77], [0.87, 0.35, 0.95], [0.99, 0.49, 0.69],
        [0.84, 0.01, 0.92], [0.79, 0.7, 0.08], [0.64, 0.87, 0.28], [0.92, 0.13, 0.77], [0.53, 0.94, 0.26],
        [0.73, 0.63, 0.38], [1.0, 0.78, 0.19], [0.21, 1.0, 0.54], [0.93, 0.09, 0.93], [0.3, 0.39, 0.93],
        [0.89, 0.22, 0.74], [0.92, 0.2, 0.47], [0.68, 0.42, 0.94], [0.36, 0.72, 0.9], [0.93, 0.43, 0.59],
        [0.96, 0.48, 0.02], [0.15, 0.71, 0.95], [0.3, 0.92, 0.28], [0.31, 0.99, 0.65], [0.91, 0.56, 0.17],
        [0.58, 0.55, 0.81], [0.86, 0.82, 0.24], [0.83, 0.99, 0.3], [0.73, 0.17, 0.67], [1.0, 0.68, 0.57],
        [0.87, 0.3, 0.8], [0.55, 0.81, 0.41], [0.6, 0.46, 0.94], [0.28, 0.52, 0.87], [0.26, 0.78, 0.85],
        [0.84, 0.51, 0.22], [0.5, 1.0, 0.59], [0.16, 0.78, 0.73], [0.0, 0.12, 0.99], [0.83, 0.55, 0.59],
        [0.74, 0.22, 0.75], [0.53, 0.13, 0.84], [0.87, 0.23, 0.55], [0.46, 0.82, 0.5], [0.97, 0.66, 0.13],
        [0.3, 0.73, 1.0], [0.94, 0.85, 0.15], [0.17, 0.93, 0.39], [0.69, 1.0, 0.53], [0.66, 0.98, 0.22],
        [0.91, 0.0, 0.44], [0.77, 0.49, 0.95], [0.5, 0.77, 0.84], [0.53, 0.81, 0.27], [0.51, 0.92, 0.4],
        [0.99, 0.08, 0.34], [0.94, 0.34, 0.87], [0.77, 0.81, 0.36], [0.85, 0.34, 0.47], [0.3, 0.06, 0.96],
        [0.65, 0.85, 0.14], [0.17, 1.0, 0.2], [0.95, 0.33, 0.36], [0.45, 0.89, 0.98], [0.59, 0.94, 0.83],
        [0.09, 0.34, 1.0], [0.66, 0.14, 0.8], [0.68, 0.66, 0.42], [0.85, 0.78, 0.12], [0.66, 0.87, 0.44],
        [0.51, 0.9, 0.69], [0.78, 0.41, 0.85], [0.46, 0.0, 0.89], [0.12, 0.99, 0.38], [0.76, 0.94, 0.55],
        [0.87, 0.24, 0.96], [0.41, 0.15, 0.94], [0.85, 0.56, 0.37], [0.68, 0.74, 0.33], [0.4, 0.98, 0.77],
        [0.83, 0.14, 0.91], [0.48, 0.21, 0.85], [0.27, 0.59, 0.82], [0.82, 0.15, 0.56], [0.6, 0.22, 0.87],
        [0.75, 0.75, 0.49], [0.92, 0.72, 0.52], [0.82, 0.42, 0.5], [0.47, 0.35, 0.81], [0.83, 0.44, 0.66],
        [0.62, 0.85, 0.07], [0.95, 0.72, 0.45], [0.33, 0.92, 0.77], [0.2, 0.92, 0.86], [0.13, 0.63, 0.77],
        [0.08, 0.83, 1.0], [0.69, 0.99, 0.64], [0.57, 0.24, 0.95], [0.16, 0.63, 0.94], [0.86, 0.65, 0.41],
        [0.41, 0.74, 0.55], [0.68, 0.09, 0.9], [0.97, 0.93, 0.0], [0.46, 0.93, 0.07], [0.82, 0.29, 0.66],
        [0.31, 0.94, 0.87], [1.0, 0.26, 0.3], [0.89, 0.93, 0.35], [0.03, 0.9, 1.0], [0.99, 0.67, 0.0],
        [0.67, 0.83, 0.24], [0.71, 0.37, 1.0], [0.63, 0.98, 0.07], [0.79, 0.85, 0.44], [0.0, 0.76, 0.87],
        [0.86, 0.85, 0.32], [0.43, 0.85, 0.86], [0.86, 0.74, 0.23], [0.16, 0.99, 0.65], [0.93, 0.56, 0.24],
        [0.84, 0.83, 0.49], [0.79, 1.0, 0.43], [0.8, 0.12, 1.0], [0.14, 0.84, 0.61], [0.3, 0.83, 0.86],
        [0.37, 0.82, 0.76], [0.46, 0.97, 0.89], [0.86, 0.46, 0.9], [0.61, 0.82, 0.88], [0.19, 0.32, 1.0],
        [0.73, 0.48, 0.81], [0.61, 0.36, 0.74], [0.12, 0.81, 0.85], [0.64, 1.0, 0.34], [0.17, 0.61, 0.87],
        [0.64, 0.94, 0.71], [0.44, 0.76, 0.75], [0.93, 0.44, 0.16], [0.46, 0.75, 0.62], [0.73, 0.38, 0.66],
        [0.88, 0.38, 0.79], [0.54, 0.99, 0.31], [1.0, 0.35, 0.14], [0.18, 0.83, 0.77], [0.32, 0.92, 0.51],
        [1.0, 0.07, 0.08], [0.54, 0.92, 0.9], [0.52, 0.05, 1.0], [0.4, 0.25, 0.95], [0.47, 0.91, 0.28],
        [0.84, 0.49, 0.73], [0.78, 0.87, 0.57], [0.0, 1.0, 0.81], [0.07, 0.49, 0.93], [0.28, 0.55, 1.0],
        [0.34, 0.45, 0.92], [0.48, 0.84, 0.61], [0.45, 0.76, 0.94], [0.82, 0.35, 0.72], [0.08, 0.69, 0.94],
        [0.82, 0.56, 0.93], [0.92, 0.33, 0.23], [0.43, 1.0, 0.27], [0.94, 0.44, 0.67], [0.8, 0.64, 0.95],
        [0.84, 0.43, 0.37], [0.95, 0.16, 0.83], [0.27, 0.95, 0.57], [1.0, 0.71, 0.06], [0.87, 0.78, 0.46],
        [1.0, 0.7, 0.4], [0.15, 0.93, 0.5], [0.7, 0.54, 0.85], [0.69, 0.67, 0.28], [0.8, 0.01, 0.66],
        [0.24, 0.74, 0.91], [0.93, 0.64, 0.06], [0.96, 0.18, 0.39], [0.81, 0.25, 0.74], [0.66, 0.47, 1.0],
        [0.37, 0.94, 0.83], [0.64, 0.87, 0.59], [0.85, 0.11, 0.78], [0.94, 0.49, 0.3], [0.37, 0.9, 0.67],
        [0.53, 0.89, 0.49], [0.2, 0.56, 0.94], [0.85, 0.59, 0.83], [0.01, 0.88, 0.89], [0.73, 0.9, 0.49],
        [0.38, 0.86, 0.34], [0.73, 0.18, 0.89], [0.26, 0.93, 0.65], [0.82, 0.16, 0.67], [0.19, 0.97, 0.78],
        [0.99, 0.09, 0.83], [0.49, 0.92, 0.0], [1.0, 0.33, 0.83], [0.7, 0.29, 0.93], [0.64, 0.77, 0.01],
        [0.4, 0.86, 0.93], [1.0, 0.38, 0.67], [0.95, 0.5, 0.59], [0.0, 0.23, 0.99], [0.06, 0.62, 0.96],
        [0.57, 0.98, 0.04], [0.09, 0.93, 0.88], [0.85, 0.29, 0.92], [0.97, 0.58, 0.11], [1.0, 0.78, 0.45],
        [0.81, 0.44, 1.0], [0.93, 0.72, 0.65], [0.72, 0.94, 0.23], [1.0, 0.95, 0.22], [0.67, 0.75, 0.13],
        [0.47, 0.44, 0.93], [0.75, 0.67, 0.03], [0.74, 0.35, 0.81], [0.51, 0.69, 0.84], [0.9, 0.13, 1.0],
        [0.61, 0.53, 0.93], [1.0, 0.55, 0.58], [0.0, 0.64, 0.94], [0.9, 0.14, 0.42], [0.85, 0.22, 0.68],
        [0.96, 0.87, 0.08], [0.8, 0.13, 0.84], [0.72, 0.75, 0.07], [0.52, 0.56, 0.93], [0.15, 1.0, 0.97],
        [0.93, 0.41, 0.83], [0.75, 0.05, 0.85], [0.84, 0.51, 0.49], [0.14, 0.74, 0.67], [0.07, 0.99, 0.16],
        [0.79, 1.0, 0.53], [0.0, 0.81, 0.69], [0.62, 0.05, 0.83], [0.59, 0.9, 0.24], [0.85, 0.64, 0.15],
        [1.0, 0.68, 0.7], [0.78, 0.82, 0.16], [0.61, 0.75, 0.93], [0.54, 0.07, 0.93], [1.0, 0.74, 0.13],
        [0.4, 0.01, 0.99], [0.81, 0.62, 0.28], [0.86, 0.58, 0.45], [0.72, 0.13, 0.77], [0.45, 0.66, 0.73],
        [0.39, 0.61, 0.8], [0.86, 0.26, 0.44], [0.8, 0.7, 0.45], [0.72, 0.89, 0.35], [0.98, 0.24, 0.44],
        [0.59, 0.77, 0.27], [1.0, 0.11, 0.48], [0.46, 0.83, 0.3], [0.18, 0.67, 0.81], [0.54, 0.98, 0.77],
        [0.99, 0.23, 0.93], [0.91, 0.46, 0.95], [0.26, 0.16, 0.95], [0.69, 0.28, 0.77], [0.49, 0.35, 0.91],
        [0.63, 0.22, 0.74], [0.96, 0.78, 0.07], [0.94, 0.04, 0.99], [0.54, 1.0, 0.65], [0.28, 0.29, 0.92]
    ]

    if color_id is None:
        return [1.0, 1.0, 1.0]

    color_id = color_id % len(colors)
    return colors[color_id]


def convert_coords2cgo(coords: np.array, color_id: int or None) -> List[float]:
    """
    Converts xyz-coordinates of sequence of points such as trace and tunnel to Pymol compiled graphics object(CGO)
    :param coords: xyz-coordinates of points
    :param color_id: Pymol color id
    :return: CGO of the points
    """

    # using CGO codes defined in Pymol2
    line_strip = 3.0
    begin = 2.0
    end = 3.0
    vertex = 4.0
    color = 6.0

    cgo = [begin, line_strip, color, *get_caver_color(color_id)]
    for xyz in coords:
        cgo.extend([vertex, xyz[0], xyz[1], xyz[2]])

    cgo.append(end)

    return cgo


def _get_boundaries(spheres: List[Tuple[np.array, float]]) -> Tuple[Tuple[float, float], Tuple[float, float],
                                                                    Tuple[float, float]]:
    """
    Calculates the boundaries for the box enclosing the set of spheres
    :param spheres: list of spheres to generate their enclosing box
    :return: ranges of boundary coordinates
    """

    max_x, max_y, max_z = -999.0, -999.0, -999.0
    min_x, min_y, min_z = 999.0, 999.0, 999.0
    max_r = 0.0
    for sphere in spheres:
        if sphere[0][0] > max_x:
            max_x = sphere[0][0]
            if sphere[1] > max_r:
                max_r = sphere[1]
        if sphere[0][0] < min_x:
            min_x = sphere[0][0]
            if sphere[1] > max_r:
                max_r = sphere[1]
        if sphere[0][1] > max_y:
            max_y = sphere[0][1]
            if sphere[1] > max_r:
                max_r = sphere[1]
        if sphere[0][1] < min_y:
            min_y = sphere[0][1]
            if sphere[1] > max_r:
                max_r = sphere[1]
        if sphere[0][2] > max_z:
            max_z = sphere[0][2]
            if sphere[1] > max_r:
                max_r = sphere[1]
        if sphere[0][2] < min_z:
            min_z = sphere[0][2]
            if sphere[1] > max_r:
                max_r = sphere[1]
    # For the limits is necessary to take into account the sphere radius
    return ((min_x - max_r - 0.1, max_x + max_r),
            (min_y - max_r - 0.1, max_y + max_r),
            (min_z - max_r - 0.1, max_z + max_r))


def _build_grid(spheres: List[Tuple[np.array, float]], x_points: np.array, y_points: np.array
                , z_points: np.array) -> np.array:
    """
    For each point in the grid we check if is inside the sphere and if so, store the data in the grid
    :param spheres: list of spheres to generate their grid
    :param x_points: evenly spaced x-coordinates in the box encasing the spheres
    :param y_points: evenly spaced y-coordinates in the box encasing the spheres
    :param z_points: evenly spaced z-coordinates in the box encasing the spheres
    :return: grid of points inside spheres
    """

    grid = np.zeros([x_points.shape[0], y_points.shape[0], z_points.shape[0]])
    for i, xp in enumerate(x_points):
        for j, yp in enumerate(y_points):
            for k, zp in enumerate(z_points):
                for sphere in spheres:
                    if abs(sphere[0][0] - xp) > sphere[1]:
                        continue
                    if abs(sphere[0][1] - yp) > sphere[1]:
                        continue
                    if abs(sphere[0][2] - zp) > sphere[1]:
                        continue
                    dist = np.linalg.norm(sphere[0] - [xp, yp, zp])
                    if dist < sphere[1]:
                        grid[i, j, k] = sphere[1] / dist
    return grid


def _get_mesh(grid: np.array, x_points: np.array, y_points: np.array, z_points: np.array,
              isolevel: float = 0.5) -> Tuple[np.array, np.array, np.array]:
    """
    Use marching cubes to get the triangles and coordinates of the surface.
    :param grid: grid of points inside spheres
    :param x_points: evenly spaced x-coordinates in the box encasing the spheres
    :param y_points: evenly spaced y-coordinates in the box encasing the spheres
    :param z_points: evenly spaced z-coordinates in the box encasing the spheres
    :param isolevel: for the mesh generation, values between 0.0 and 1.0
    :return: arrays of vertices, normals and triangles for the mesh
    """

    import mcubes
    grid_vertices, triangles = mcubes.marching_cubes(grid, isolevel)
    grid_vertices = grid_vertices.astype(int)
    vertices = np.zeros(grid_vertices.shape, dtype=float)
    for i, gv in enumerate(grid_vertices):
        vertices[i] = x_points[gv[0]], y_points[gv[1]], z_points[gv[2]]
    normals = np.zeros(grid_vertices.shape, dtype=float)
    normals[:, 2] = 0.01
    for i, t in enumerate(triangles):
        v1 = vertices[t[1]] - vertices[t[0]]
        v2 = vertices[t[2]] - vertices[t[0]]
        n = np.cross(v1, v2)
        normals[t[0]] += n
        normals[t[1]] += n
        normals[t[2]] += n
    mods = np.linalg.norm(normals, axis=1)
    for i in range(normals.shape[0]):
        normals[i] /= mods[i]
    return vertices, normals, triangles


def convert_spheres2cgo_surface(spheres: List[Tuple[np.array, float]], color_id: int or None,
                                resolution: float = 0.5) -> List[float]:
    """
    Converts spheres (tuples that contain XYZ coords and radius) to surface complied graphics object for Pymol.
    :param spheres: list of spheres to generate their approximate surface
    :param color_id: Pymol color id
    :param resolution: surface grid resolution in Angstroms
    :return: CGO of spheres surface
    """

    # First we need to get the boundaries for the box of the grid
    x_lims, y_lims, z_lims = _get_boundaries(spheres)

    # Then build the grid with the desired resolution
    x_points = np.arange(x_lims[0], x_lims[1]+resolution, resolution)
    y_points = np.arange(y_lims[0], y_lims[1]+resolution, resolution)
    z_points = np.arange(z_lims[0], z_lims[1]+resolution, resolution)
    grid = _build_grid(spheres, x_points, y_points, z_points)
    #  Get surface mesh
    vertices, normals, triangles = _get_mesh(grid, x_points, y_points, z_points)

    # Create and populate the cgo to use in pymol
    cgo_obj = [2.0, 4.0]
    cgo_obj.extend([6.0, *get_caver_color(color_id)])
    for triangle in triangles:
        for t in triangle:
            cgo_obj.append(5.0)
            cgo_obj.append(normals[t, 0])
            cgo_obj.append(normals[t, 1])
            cgo_obj.append(normals[t, 2])
            cgo_obj.append(4.0)
            cgo_obj.append(vertices[t, 0])
            cgo_obj.append(vertices[t, 1])
            cgo_obj.append(vertices[t, 2])
    cgo_obj.append(3.0)

    return cgo_obj


def test_file(filename: str):
    """
    Tests if file exists
    :param filename: path to the file to test
    """

    if not os.path.exists(filename):
        raise FileNotFoundError("Input file '{}' does not exist!".format(filename))


def node_labels_split(node_label: str) -> (int, int):
    """
    Returns layer ID and cluster ID, useful for sorting of nodes labels
    :param node_label: node label formatted as {layer}_{cluster_id} or SP for starting point
    :return: sortable tuple for ordering by layer and then by cluster id
    """

    try:
        return int(node_label.split("_")[0]), int(node_label.split("_")[1])
    except ValueError:  # cannot be split
        return -99, -99


def get_filepath(root_folder: str = ".", pattern: str = "*") -> str:
    """
    Get full and unique path to file in root folder specified by file pattern (may include subfolders)
    :param root_folder: folder to search in
    :param pattern: search patter
    :return: file path
    """

    file_path = [*Path(root_folder).glob(pattern)]
    if len(file_path) != 1 or not file_path[0].exists():
        raise RuntimeError("File '{}' does not exist in '{}' or more then 1 file "
                           "can be matched!".format(pattern, root_folder))

    return file_path[0].as_posix()


def reweighting_file_parser(engine: str = "gamd"):
    # TODO - get file, return tunnel weights
    pass


def splitall(path2split: str) -> List[str]:
    """
    Get all sub-parts of a file or directory path. Credited to Trent Mick.
    :param path2split: path to process
    :return: list of subpart of the path
    """

    all_parts = list()

    while True:
        parts = os.path.split(path2split)
        if parts[0] == path2split:  # sentinel for absolute paths
            all_parts.insert(0, parts[0])
            break
        elif parts[1] == path2split:  # sentinel for relative paths
            all_parts.insert(0, parts[1])
            break
        else:
            path2split = parts[0]
            all_parts.insert(0, parts[1])

    return all_parts


def path_loader_string(path: str) -> str:
    """
    Converts a string with a path to a string with an os.path.join loader for a cross-platform compatibility.
    :param path: path to convert
    :return: os path loader
    """

    all_parts = list()
    for path_segment in splitall(path):
        all_parts.append("'{}'".format(path_segment))

    return "os.path.join({})".format(", ".join(all_parts))
