# -*- coding: utf-8 -*-

# TransportTools, a library for massive analyses of internal voids in biomolecules and ligand transport through them
# Copyright (C) 2021  Jan Brezovsky, Aravind Selvaram Thirunavukarasu, Carlos Eduardo Sequeiros-Borja, Bartlomiej
# Surpeta, Nishita Mandal, Cedrix Jurgal Dongmo Foumthuim, Dheeraj Kumar Sarkar, Nikhil Agrawal  <janbre@amu.edu.pl>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

__version__ = '0.9.0'
__author__ = 'Jan Brezovsky, Aravind Selvaram Thirunavukarasu, Carlos Eduardo Sequeiros-Borja, Bartlomiej Surpeta, ' \
             'Nishita Mandal, Cedrix Jurgal Dongmo Foumthuim, Dheeraj Kumar Sarkar, Nikhil Agrawal'
__mail__ = 'janbre@amu.edu.pl'

import numpy as np
from transport_tools.libs.geometry import ClusterInLayer, Layer4Tunnels, Layer4Events
from sys import maxsize
import os
np.set_printoptions(threshold=maxsize)


def set_root_path(*args):
    cwd = os.getcwd()
    allparts = []
    while 1:
        parts = os.path.split(cwd)
        if parts[0] == cwd:  # sentinel for absolute paths
            allparts.insert(0, parts[0])
            break
        elif parts[1] == cwd:  # sentinel for relative paths
            allparts.insert(0, parts[1])
            break
        else:
            cwd = parts[0]
            allparts.insert(0, parts[1])
    if "transport_tools" not in allparts:
        raise RuntimeError("Must be executed from the 'transport_tools' folder")

    root_index = allparts.index("transport_tools")
    return os.path.join(*allparts[:root_index + 1], *args)


test_matrix_input1 = np.array([[10.70641060822775, 0.08880920197130893, 23.298347462871927, 25.640750957938756,
                                1.4727973160443832, -94.0, 5.0],
                               [10.308503726054784, -0.04938839289167829, 23.00783879776422, 25.21167059969137,
                                1.8401117247695595, -93.0, 9.0]])
test_matrix_input2 = np.array([[-0.027795053135731607, -1.168054089495128, -2.939217780131123, 3.1629296672553497,
                                1.52575562678011, 0.0, 11.0],
                               [0.07880534220806723, -1.6266976354556344, -3.106304729040254, 3.507347223873811,
                                1.3658577517702153, 4.0, 11.0],
                               [0.19022297410725386, -2.0829603481772234, -3.2777994653725275, 3.888300133368464,
                                1.293275418473532, 5.0, 11.0]])

test_cluster_point_mat1 = np.array([[-2.72118037e-01, 6.10992422e-01, -1.06760832e-01, 6.77316648e-01, 1.94588348e+00,
                                     0.00000000e+00, 0.00000000e+00],
                                    [-2.37478161e-02, 3.05230346e-01, 2.01170610e-01, 3.66332004e-01, 1.79574575e+00,
                                     1.00000000e+00, 0.00000000e+00],
                                    [2.24622404e-01, -5.31730023e-04, 5.09102053e-01, 5.56453419e-01, 1.72444906e+00,
                                     2.00000000e+00, 0.00000000e+00],
                                    [4.72992625e-01, -3.06293806e-01, 8.17033496e-01, 9.92512797e-01, 1.73773927e+00,
                                     3.00000000e+00, 0.00000000e+00]])

test_cluster_point_mat2 = np.array([[2.0893996864781696, 0.7650542373301019, 5.897799493017896, 6.30356549071666,
                                     1.0452652570298382, 16.0, 0.0],
                                    [2.3652158648159887, 1.0579835486037068, 6.178784605026609, 6.7000712288457915,
                                     1.1505238531478998, 17.0, 0.0],
                                    [1.7499590645224297, 1.2883257363094929, 5.8784012004443795, 6.267195593224097,
                                     1.120130700278908, 16.0, 1.0],
                                    [2.0720861449382335, 1.5812491867661125, 6.073215873106522, 6.608921320763812,
                                     1.1874115790490916, 17.0, 1.0]])

test_cluster_point_mat3 = np.array([[11.795952048071417, -4.678521065465036, 11.191945873604357, 16.920186066296374,
                                     1.761755629647365, -66.0, 1.0],
                                    [11.823663178507395, -4.346610609053391, 11.002907206731464, 16.72590809927731,
                                     1.549220667160456, -58.0, 2.0]])

test_cluster_point_mat4 = np.array([[-0.9642598768465529, 3.5221061895716215, 9.499953945116294, 10.177630081705932,
                                     1.677700491775785, 48.0, 0.0],
                                    [-0.8498587564786195, 3.419485455370014, 9.974541344560652, 10.578592331676285,
                                     1.7364739481938127, 49.0, 0.0],
                                    [-0.795784294049394, 3.0003430672085383, 10.230721933704938, 10.691258225690433,
                                     1.6123684005312884, 50.0, 0.0],
                                    [-0.6289535083318114, 2.5931786956296365, 10.468176456814277, 10.802910561238162,
                                     1.468601367158507, 51.0, 0.0],
                                    [-0.46680418763378206, 2.206210850907758, 10.709475735117042, 10.944320124581347,
                                     1.4087327811487886, 52.0, 0.0],
                                    [-0.2966480127488378, 1.884420075409269, 11.051201029787798, 11.214637009946252,
                                     1.3886010418266237, 53.0, 0.0],
                                    [0.061962559122509475, 1.5955214933403832, 11.24597283566075, 11.358760197084171,
                                     1.2942448304847682, 54.0, 0.0],
                                    [0.42057313099385496, 1.3066229112714973, 11.440744641533703, 11.522794068434779,
                                     1.2983312820869262, 55.0, 0.0],
                                    [0.7812244025421577, 1.031733766552058, 11.648495292463581, 11.720163339762687,
                                     1.389548645640556, 56.0, 0.0],
                                    [1.1485489821232182, 0.8026569928807241, 11.898688166495825, 11.980909944410081,
                                     1.5360119912460204, 57.0, 0.0],
                                    [1.9459066536385814, -0.7272796428106285, 11.720632784120424, 11.903307155732819,
                                     1.4314861955751625, 61.0, 0.0],
                                    [2.099760647750367, -1.1229967184424772, 11.493254791851248, 11.73733454059674,
                                     1.4131905514813659, 62.0, 0.0],
                                    [2.1323142038777396, -1.4788888279003989, 11.14356768662379, 11.441720885238952,
                                     1.313020165893129, 63.0, 0.0],
                                    [2.164867760005115, -1.8347809373583197, 10.793880581396332, 11.160686874554266,
                                     1.3103338124678892, 64.0, 0.0],
                                    [2.1974213161324876, -2.1906730468162405, 10.444193476168874, 10.895360765311493,
                                     1.4054330438246947, 65.0, 0.0],
                                    [2.251241400422055, -2.4783890672475386, 10.085574270126674, 10.626820247367863,
                                     1.5421378450130485, 66.0, 0.0],
                                    [2.6195005103983497, -2.464421388411417, 9.784598433418811, 10.424659332894228,
                                     1.3303985017048243, 67.0, 0.0],
                                    [3.024555711596678, -2.4503967250045107, 9.491795444219122, 10.258974710825823,
                                     1.166290916126279, 68.0, 0.0],
                                    [3.429610912795007, -2.4363720615976043, 9.198992455019434, 10.115315221148595,
                                     1.1012494045742784, 69.0, 0.0],
                                    [3.8677632592940743, -2.444189276605942, 8.97557663189007, 10.07445431399397,
                                     1.087686926087869, 70.0, 0.0],
                                    [4.351920222393898, -2.4845337698271446, 8.857405764433228, 10.176726121464968,
                                     1.099014591302493, 71.0, 0.0],
                                    [4.704662886193828, -2.6728122430645977, 8.628293222969303, 10.184558021875258,
                                     1.1287761294883223, 72.0, 0.0],
                                    [4.9567273307582465, -2.971586462242908, 8.316741751184681, 10.127569564805246,
                                     1.1975549138838202, 73.0, 0.0],
                                    [5.17221267019978, -3.288026316353339, 7.995473297104645, 10.074249064199309,
                                     1.3485747120727558, 74.0, 0.0],
                                    [5.464754510829522, -3.3633790914360175, 7.69987882598412, 10.023172886425694,
                                     1.475724317611373, 75.0, 0.0],
                                    [6.285888548471876, -2.9739821688892083, 7.359225648466074, 10.12497737922025,
                                     1.5374657075464346, 77.0, 0.0],
                                    [6.7344733058183746, -2.8185265867622413, 7.227493606471256, 10.272968736380383,
                                     1.648244903227911, 78.0, 0.0],
                                    [7.0352834680787755, -2.904600920427742, 7.364046021314934, 10.590613475489345,
                                     1.5507507683401338, 79.0, 0.0],
                                    [7.4871505128412315, -3.031275521115514, 7.5365850357434265, 11.04745075064128,
                                     1.4523694362158122, 80.0, 0.0],
                                    [7.715637827716394, -3.377566223395758, 7.675284449655589, 11.395131068357086,
                                     1.4264403967688775, 81.0, 0.0],
                                    [7.922689580615579, -3.7535939014942508, 7.904954838720338, 11.80452406369429,
                                     1.395949034380277, 82.0, 0.0],
                                    [-1.844672216392457, 3.458031685976297, 9.303224692318471, 10.095087340033787,
                                     1.5896383942384793, 43.0, 1.0],
                                    [-1.7330308786236746, 3.1668826706514905, 9.625949839537432, 10.280634814505696,
                                     1.617997507786623, 44.0, 1.0],
                                    [-1.4652698066696797, 2.969664468859941, 9.999324955779823, 10.53339557005376,
                                     1.6173300639589605, 45.0, 1.0],
                                    [-1.203469238548779, 2.680678323676324, 10.278976018911443, 10.69073068038466,
                                     1.652620426759212, 46.0, 1.0],
                                    [-1.0323860590119835, 2.2706928977280265, 10.505444944507735, 10.797510856337118,
                                     1.633754057675245, 47.0, 1.0],
                                    [-0.7267979510200462, 2.1305376737259065, 10.864246564559576, 11.095011467155718,
                                     1.5779360084446068, 48.0, 1.0],
                                    [-0.4814081819687388, 1.905791517026597, 11.214979643069395, 11.385937095316217,
                                     1.6060403331559783, 49.0, 1.0],
                                    [-0.11883711344371406, 1.6835598980081892, 11.390772584776807, 11.515129016551047,
                                     1.5093229057208049, 50.0, 1.0],
                                    [0.3378794466706019, 1.540159582630935, 11.53347089938358, 11.640756205993673,
                                     1.3708630442602627, 51.0, 1.0],
                                    [0.7937133819877129, 1.4003044441226433, 11.683989854821773, 11.794339845729143,
                                     1.3233810815152032, 52.0, 1.0],
                                    [1.1006356259954364, 1.1028677189164962, 11.71738729093965, 11.820527945591621,
                                     1.2458456258784685, 53.0, 1.0],
                                    [1.2569204588466425, 0.631228823919824, 11.693915134974818, 11.778198081710443,
                                     1.0868424034306055, 54.0, 1.0],
                                    [1.4190743384943865, 0.15840392373048662, 11.705861881705358, 11.792627619610016,
                                     1.0123112829143588, 55.0, 1.0],
                                    [1.5812282181421295, -0.3144209764588499, 11.717808628435897, 11.828194379571762,
                                     1.0492208088340467, 56.0, 1.0],
                                    [1.7326827946354246, -0.7905736940689256, 11.722778112300537, 11.876477722911451,
                                     1.1878664626715094, 57.0, 1.0],
                                    [1.864769326144172, -1.2727504785340868, 11.715117246402672, 11.930684411101414,
                                     1.4087211562868924, 58.0, 1.0],
                                    [2.106637150179507, -1.6817962493070762, 11.644078509689724, 11.952025897165628,
                                     1.6118179105757788, 59.0, 1.0],
                                    [2.4203848370364938, -2.0183896166290363, 11.548834016608671, 11.971120521782815,
                                     1.7720774967764712, 60.0, 1.0]])

test_cluster_merged_coors = np.array(
    [[-0.272118037, 0.610992422, -0.106760832], [-0.0237478161, 0.305230346, 0.20117061],
     [0.224622404, -0.000531730023, 0.509102053], [0.472992625, -0.306293806, 0.817033496],
     [11.795952048071417, -4.678521065465036, 11.191945873604357],
     [11.823663178507395, -4.346610609053391, 11.002907206731464]]
    )

test_cluster_point_mat5 = np.array(
    [[-0.2721180365941773, 0.6109924224913796, -0.10676083241216117, 0.6773166478974898, 1.9458834779974528, 0.0, 0.0],
     [-0.023747816079717765, 0.30523034623404044, 0.20117061027419259, 0.3663320044287687, 1.7957457472348868, 1.0,
      0.0],
     [0.22462240443474357, -0.0005317300232978717, 0.5091020529605454, 0.5564534191102658, 1.7244490604239398, 2.0,
      0.0],
     [0.47299262494920313, -0.3062938062806375, 0.8170334956468984, 0.992512796910553, 1.7377392722896674, 3.0, 0.0]])
test_cluster_point_mat6 = np.array(
    [[0.7205957527061262, -0.6137666971104951, 1.1194867985249606, 1.4660213127276496, 1.8304755794480327, 4.0, 0.0],
     [0.9069899647360948, -1.0577512470126798, 0.984820868335526, 1.706265114042364, 1.6787394561220423, 5.0, 0.0],
     [1.0933841767660617, -1.5017357969148644, 0.8501549381460914, 2.0429054751975446, 1.609003288505866, 6.0, 0.0],
     [1.2347390637866855, -1.912787971996765, 0.6212615605843785, 2.3599373525829144, 1.5584245306473605, 7.0, 0.0],
     [1.1581582117973106, -2.3147742970260903, 0.6366620733188065, 2.6654922782346935, 1.4577432069771163, 8.0, 0.0],
     [1.066947939911148, -2.610041218918727, 0.24672193622747285, 2.830470770870321, 1.2746843447623795, 9.0, 0.0]]
    )
test_cluster_point_mat7 = np.array(
    [[0.9562869808370635, -2.913249808480075, -0.1351402136995825, 3.0691647257334407, 1.1667084291638201, 10.0, 0.0],
     [0.988017862670854, -3.358492217345677, -0.3584475805512668, 3.5191098219480477, 1.000050653220504, 11.0, 0.0],
     [1.021672594717078, -3.8056536392087836, -0.5796127203542896, 3.982808734802047, 0.9375596812069988, 12.0, 0.0]]
    )
test_cluster_point_mat8 = np.array(
    [[1.0175934665858843, -4.255929906791911, -0.7958828534958138, 4.447680895843667, 0.9717324479919796, 13.0, 0.0],
     [1.001556840122273, -4.707193239104905, -1.0106018059811102, 4.917529898691305, 1.1103677760994468, 14.0, 0.0],
     [0.9855202136586616, -5.158456571417901, -1.225320758466406, 5.392804024982089, 1.3414910379570826, 15.0, 0.0],
     [1.0018160638316722, -5.556998270932313, -1.4815378915143453, 5.837706718642919, 1.6024959399917267, 16.0, 0.0]]
    )
test_cluster_point_mat9 = np.array(
    [[1.2518678167878008, -5.848624735852448, -1.2202303762378919, 6.104305570876007, 1.516289703126508, 17.0, 0.0],
     [1.3431987297252377, -6.092037929145109, -0.8183398082372277, 6.291803318558508, 1.4364200456569052, 18.0, 0.0],
     [1.4403836604063045, -6.372292331712506, -0.4270357888953136, 6.546997343436525, 1.3679686036597702, 19.0, 0.0],
     [1.7072816171766014, -6.498753179433438, -0.028199282164253603, 6.71933022012313, 1.3705052104932915, 20.0, 0.0],
     [2.0156759733115246, -6.313016735479083, 0.13431372911922668, 6.628361042494159, 1.3734882154761638, 21.0, 0.0],
     [2.04801643079427, -6.380056346490834, 0.5586739688813267, 6.7239576804666825, 1.2787357198379101, 22.0, 0.0]]
    )
test_cluster_point_mat10 = np.array(
    [[2.151015084175354, -6.595569364368496, 0.995643732984961, 7.008545325213954, 1.1064056740157293, 23.0, 0.0],
     [2.195882535434233, -6.842404198119301, 1.4108943599360333, 7.323320163338827, 0.9785556892551994, 24.0, 0.0],
     [2.2498098300275586, -7.183419056673753, 1.7725581541168811, 7.733376754355015, 0.9681063650942232, 25.0, 0.0],
     [2.288802555078969, -7.5866668605920164, 2.0081329497195184, 8.174884044039423, 0.9730187202216347, 26.0, 0.0],
     [2.539512112941826, -7.925450435606944, 2.277150410355998, 8.628284903178907, 0.9563358217864317, 27.0, 0.0],
     [2.8459373498440765, -8.090647932620728, 2.4927302787344683, 8.931497501056429, 0.9857747864311148, 28.0, 0.0],
     [3.296766219337094, -7.940229338754905, 2.57178320256346, 8.973849692187514, 0.943269922765525, 29.0, 0.0]]
    )
test_cluster_point_mat11 = np.array(
    [[5.4308556463079825, 0.2673726879095599, 7.119804306943207, 8.958643567775331, 1.2680277289921056, 64.0, 0.0],
     [5.081815262997466, 0.6227129734046724, 7.076168492689105, 8.73411577387239, 1.1186970937639178, 65.0, 0.0],
     [4.732774879686949, 0.9780532588997852, 7.032532678435002, 8.533010143688832, 1.0719048294733982, 66.0, 0.0],
     [4.383734496376432, 1.333393544394898, 6.9888968641809, 8.357017760966668, 1.1338266636020464, 67.0, 0.0],
     [4.034694113065915, 1.6887338298900108, 6.945261049926797, 8.207754235225005, 1.2963858350213353, 68.0, 0.0],
     [3.911003201737773, 2.0723866311817516, 6.973222579537319, 8.259332027281205, 1.3215750365748853, 69.0, 0.0]]
    )
test_cluster_point_mat12 = np.array(
    [[3.647926066132536, 2.458357603694181, 7.09653040754184, 8.349349107372152, 1.1952397898783638, 70.0, 0.0],
     [3.4883842687122364, 2.8929215256837058, 7.278384260861467, 8.573954584013089, 1.1553615750382564, 71.0, 0.0],
     [3.3405372784455976, 3.3315816446037174, 7.467377817237914, 8.832913280946553, 1.2204395460544044, 72.0, 0.0]]
    )
test_cluster_point_mat13 = np.array(
    [[3.1683623671570516, 3.7720869365129923, 7.616904063355843, 9.071074217345803, 1.3561755157877464, 73.0, 0.0],
     [2.937245049696088, 4.160401750167806, 7.762881288176571, 9.284324267230517, 1.4991237986848025, 74.0, 0.0],
     [2.7727731510286686, 4.226063866201915, 8.211837212388632, 9.642725607993501, 1.464592011393708, 75.0, 0.0],
     [2.5001793199781304, 4.279796040449783, 8.533254947385005, 9.86833272528681, 1.4220418646605772, 76.0, 0.0],
     [2.013840327488416, 4.2684159160297614, 8.647094947202453, 9.851252627091085, 1.456788738504773, 77.0, 0.0],
     [1.5348108717620272, 4.2979750464885536, 8.776307688965705, 9.892007408191512, 1.5472617251252616, 78.0, 0.0],
     [1.0879414687331366, 4.229758841566054, 8.923229721744612, 9.934712132931777, 1.5337079681143615, 79.0, 0.0]]
    )
test_cluster_point_mat14 = np.array(
    [[3.7919198874505824, -7.995841346146248, 2.6133782078515613, 9.22723582243667, 0.9407327683122928, 30.0, 0.0],
     [4.285846466274668, -8.04977230402381, 2.6574332083067613, 9.498908639192159, 1.0503388673321534, 31.0, 0.0],
     [4.639068975926221, -8.000839069467771, 2.9740983488874972, 9.71491882455939, 1.0317722685390527, 32.0, 0.0]]
    )
test_cluster_point_mat15 = np.array(
    [[6.194431161945801, -1.9408403453319933, 7.305049840757944, 9.772491588234358, 1.5075987488533908, 59.0, 0.0],
     [6.0496521312444065, -1.4804793636997258, 7.1747239520439425, 9.50088279284845, 1.3983022894867487, 60.0, 0.0],
     [5.906695541302838, -1.0199403952213495, 7.04256018717977, 9.248080050340214, 1.3818586974675646, 61.0, 0.0],
     [5.798226366276339, -0.5480625702652486, 6.9678860576243, 9.081367611078354, 1.4051485701767568, 62.0, 0.0],
     [5.76427850605093, -0.05168344548305637, 7.017437046744449, 9.081519728485933, 1.4023621711692102, 63.0, 0.0]]
    )
test_cluster_point_mat16 = np.array(
    [[4.991147444389164, -8.202122133562092, 3.266546411228073, 10.141828521633002, 1.019789547141689, 33.0, 0.0],
     [5.385667255541994, -8.301287202164469, 3.5224121151094216, 10.503531220923106, 1.0895004811005584, 34.0, 0.0],
     [5.843109523752366, -8.186745164145497, 3.688624893368014, 10.713107807410484, 1.2006354965038513, 35.0, 0.0],
     [6.260742825679479, -7.964621490939292, 3.773557571705481, 10.81072767995776, 1.3530299004096833, 36.0, 0.0],
     [6.35537385163453, -7.991305251894454, 4.24039784050472, 11.055890297427561, 1.2955550674709648, 37.0, 0.0],
     [6.448816465796128, -7.8862449787022495, 4.718877154246373, 11.227105382532848, 1.2654162280125403, 38.0, 0.0],
     [6.587665115382869, -7.892301473487899, 5.154493925080603, 11.500198339356782, 1.2931663852409232, 39.0, 0.0],
     [6.875328091115669, -8.017087785904883, 5.522456127005502, 11.91811036205363, 1.3216353145000206, 40.0, 0.0]]
    )
test_cluster_point_mat17 = np.array(
    [[0.6371095965700722, 4.111715924203618, 9.095008714767907, 10.001564877612676, 1.5487731652328767, 80.0, 0.0],
     [0.17051947312618232, 4.012236747070574, 9.242317122752292, 10.07708025187747, 1.633734443740462, 81.0, 0.0],
     [-0.2890264586333444, 3.936066371375925, 9.424019096754067, 10.217058809151093, 1.7783329930974643, 82.0, 0.0],
     [-0.7485723903928712, 3.859895995681276, 9.605721070755841, 10.379259848866758, 1.9954473768010537, 83.0, 0.0],
     [-0.9027345530661979, 4.045949805925653, 10.021998448697703, 10.845510242080296, 1.9596697980296172, 84.0, 0.0],
     [-1.1900854081186125, 4.3512900495769635, 9.84802215923679, 10.832062076215367, 1.786503214847995, 85.0, 0.0],
     [-1.3729202934701377, 4.794619769720711, 9.706494237511304, 10.912805287883542, 1.5101163289113126, 86.0, 0.0],
     [-1.555755178821661, 5.237949489864457, 9.564966315785819, 11.015673817652694, 1.3040486377397007, 87.0, 0.0]]
    )
test_cluster_point_mat18 = np.array(
    [[7.546888626252515, -3.5660438741327094, 8.273516493322958, 11.752585758656943, 1.5067918517105074, 54.0, 0.0],
     [7.198210508114398, -3.2368765933822403, 8.131839179855556, 11.332184831132443, 1.4982115757037955, 55.0, 0.0],
     [6.8495323899762806, -2.9077093126317695, 7.990161866388149, 10.918495961403432, 1.5810014897582338, 56.0, 0.0],
     [6.528908922389162, -2.6487242760754275, 7.731819532309654, 10.460565247047082, 1.6283095646181558, 57.0, 0.0],
     [6.290569994154775, -2.393960420146112, 7.409003050118898, 10.009777397188095, 1.655561844621934, 58.0, 0.0]]
    )
test_cluster_point_mat19 = np.array(
    [[-1.7385900641731862, 5.681279210008205, 9.423438394060335, 11.140054759289274, 1.1865851428558696, 88.0, 0.0],
     [-1.9214249495247113, 6.124608930151951, 9.28191047233485, 11.285236834040854, 1.1709318227301877, 89.0, 0.0],
     [-2.107055436489926, 6.56714855481275, 9.144846238648231, 11.45405323376575, 1.2563865699266887, 90.0, 0.0],
     [-2.4217133731230316, 6.924080709156719, 9.284222488271073, 11.832344507332326, 1.2639617038288689, 91.0, 0.0]]
    )
test_cluster_point_mat20 = np.array(
    [[7.2382985302461975, -7.9727517117610684, 5.863103367894922, 12.261065066827355, 1.305795025608954, 41.0, 0.0],
     [7.614044015696955, -7.898210130032606, 6.18444255990666, 12.593757156143083, 1.3671173274185195, 42.0, 0.0],
     [7.900081036113177, -7.802415249020251, 6.521240677354784, 12.876938458627714, 1.4543645453418048, 43.0, 0.0],
     [8.176450265798694, -7.5107282417137995, 6.492563792786422, 12.861522548802611, 1.3928072919871617, 44.0, 0.0],
     [8.263328848695343, -7.06446209738041, 6.699859534710729, 12.770174085417137, 1.1946800387923828, 45.0, 0.0],
     [8.348920535528915, -6.633099898749949, 6.932599896040539, 12.7186253067678, 1.0616857445593022, 46.0, 0.0],
     [8.386552749491319, -6.221197646522716, 7.213527554300182, 12.69143596894174, 0.9875027293495637, 47.0, 0.0],
     [8.386692373854006, -5.800737135463029, 7.480288352952137, 12.646733733723982, 0.9854827369644854, 48.0, 0.0]]
    )
test_cluster_point_mat21 = np.array(
    [[-2.55293899868569, 7.320230259359036, 9.542649214308868, 12.294934835467636, 1.167002042639248, 92.0, 0.0],
     [-2.650025440271877, 7.725490078933078, 9.818943803425865, 12.771745738489189, 1.1378395631198785, 93.0, 0.0]]
    )
test_cluster_point_mat22 = np.array(
    [[8.364147418168258, -5.35837485358714, 7.712252432050898, 12.575769585996744, 1.0561558023529656, 49.0, 0.0],
     [8.384699982478846, -4.915723656889764, 7.931833498896296, 12.545178975259775, 1.1933175603823745, 50.0, 0.0],
     [8.38228065431492, -4.545516924961829, 8.170816993553387, 12.557332655662085, 1.392833887752006, 51.0, 0.0],
     [8.143217053684873, -4.259491163259121, 8.459838766134897, 12.490961568362362, 1.643576407365549, 52.0, 0.0],
     [7.871801595445556, -3.909974608252806, 8.432815109934738, 12.1805390879708, 1.6007531289536059, 53.0, 0.0]]
    )
test_cluster_point_mat23 = np.array(
    [[-2.7471118818580624, 8.13074989850712, 10.09523839254286, 13.250266254147146, 1.214184950644569, 94.0, 0.0],
     [-2.8719921387289595, 8.489915583337527, 10.411531231760687, 13.737794147789758, 1.3635530271563103, 95.0, 0.0],
     [-3.0936555966292083, 8.822660421558787, 10.685595075105429, 14.198309194189726, 1.4897115534087206, 96.0, 0.0],
     [-3.39065475279647, 9.116984603988454, 10.885114227383262, 14.598001906603702, 1.5438424674094875, 97.0, 0.0],
     [-3.820681429709728, 9.09708914857742, 11.13943926660883, 14.880918813667403, 1.4779699379280211, -98.0, 0.0]]
    )

test_cluster_point_mat24 = np.array([[0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
                                     [0.06365023492524813, -0.040474285302703805, -0.060742309782810366,
                                      0.09684600340001047, 0.0, 1.0, 0.0]]
                                    )
test_cluster_point_mat25 = np.array(
    [[0.5766809115273155, -1.080044599868236, -1.1347257202266832, 1.6693291082253607, 0.0, 2.0, 0.0]]
    )
test_cluster_point_mat26 = np.array(
    [[1.8317855854326517, -5.681518382112968, -1.6029221763147845, 6.180974766216743, 0.0, 3.0, 0.0]]
    )
test_cluster_point_mat27 = np.array(
    [[-0.5870649957828888, -7.479598010247797, -3.517457141893324, 8.286225705906924, 0.0, 4.0, 0.0]]
    )
test_cluster_point_mat28 = np.array(
    [[-3.0059155769984187, -9.277677638382618, -5.4319921074718565, 11.163214997187108, 0.0, -5.0, 0.0]]
    )

test_cluster1 = ClusterInLayer(test_cluster_point_mat5, 1.5, 0.9, False, 1, 0)
test_cluster2 = ClusterInLayer(test_cluster_point_mat6, 1.5, 0.9, False, 1, 1)
test_cluster3 = ClusterInLayer(test_cluster_point_mat7, 1.5, 0.9, False, 1, 2)
test_cluster4 = ClusterInLayer(test_cluster_point_mat8, 1.5, 0.9, False, 1, 3)
test_cluster5 = ClusterInLayer(test_cluster_point_mat9, 1.5, 0.9, False, 1, 4)
test_cluster6 = ClusterInLayer(test_cluster_point_mat10, 1.5, 0.9, False, 1, 5)
test_cluster7 = ClusterInLayer(test_cluster_point_mat11, 1.5, 0.9, False, 2, 5)
test_cluster8 = ClusterInLayer(test_cluster_point_mat12, 1.5, 0.9, False, 3, 5)
test_cluster9 = ClusterInLayer(test_cluster_point_mat13, 1.5, 0.9, False, 1, 6)
test_cluster10 = ClusterInLayer(test_cluster_point_mat14, 1.5, 0.9, False, 2, 6)
test_cluster11 = ClusterInLayer(test_cluster_point_mat15, 1.5, 0.9, False, 3, 6)
test_cluster12 = ClusterInLayer(test_cluster_point_mat16, 1.5, 0.9, False, 1, 7)
test_cluster13 = ClusterInLayer(test_cluster_point_mat17, 1.5, 0.9, False, 2, 7)
test_cluster14 = ClusterInLayer(test_cluster_point_mat18, 1.5, 0.9, False, 2, 7)
test_cluster15 = ClusterInLayer(test_cluster_point_mat19, 1.5, 0.9, False, 4, 7)
test_cluster16 = ClusterInLayer(test_cluster_point_mat20, 1.5, 0.9, False, 1, 8)
test_cluster17 = ClusterInLayer(test_cluster_point_mat21, 1.5, 0.9, False, 2, 8)
test_cluster18 = ClusterInLayer(test_cluster_point_mat22, 1.5, 0.9, False, 3, 8)
test_cluster19 = ClusterInLayer(test_cluster_point_mat23, 1.5, 0.9, True, 1, 9)
test_cluster20 = ClusterInLayer(test_cluster_point_mat24, 1.5, 0.9, False, 0, 0)
test_cluster21 = ClusterInLayer(test_cluster_point_mat25, 1.5, 0.9, False, 0, 1)
test_cluster22 = ClusterInLayer(test_cluster_point_mat26, 1.5, 0.9, False, 0, 4)
test_cluster23 = ClusterInLayer(test_cluster_point_mat27, 1.5, 0.9, False, 0, 5)
test_cluster24 = ClusterInLayer(test_cluster_point_mat28, 1.5, 0.9, True, 0, 7)


params = {
    "transformation_folder": set_root_path("tests", "data", "saved_outputs", "transformations"),
    "caver_foldername" : "caver"
}

test_layer0 = Layer4Tunnels(0, 1.5, params, "Cluster_44", "e10s1_e9s3p0f1600")
test_layer0.clusters[1] = test_cluster1
test_layer1 = Layer4Tunnels(1, 1.5, params, "Cluster_44", "e10s1_e9s3p0f1600")
test_layer1.clusters[1] = test_cluster2
test_layer2 = Layer4Tunnels(2, 1.5, params, "Cluster_44", "e10s1_e9s3p0f1600")
test_layer2.clusters[1] = test_cluster3
test_layer3 = Layer4Tunnels(3, 1.5, params, "Cluster_44", "e10s1_e9s3p0f1600")
test_layer3.clusters[1] = test_cluster4
test_layer4 = Layer4Tunnels(4, 1.5, params, "Cluster_44", "e10s1_e9s3p0f1600")
test_layer4.clusters[1] = test_cluster5
test_layer5 = Layer4Tunnels(5, 1.5, params, "Cluster_44", "e10s1_e9s3p0f1600")
test_layer5.clusters = {
    1: test_cluster6,
    2: test_cluster7,
    3: test_cluster8
}
test_layer6 = Layer4Tunnels(6, 1.5, params, "Cluster_44", "e10s1_e9s3p0f1600")
test_layer6.clusters = {
    1: test_cluster9,
    2: test_cluster10,
    3: test_cluster11
}
test_layer7 = Layer4Tunnels(7, 1.5, params, "Cluster_44", "e10s1_e9s3p0f1600")
test_layer7.clusters = {
    1: test_cluster12,
    2: test_cluster13,
    3: test_cluster14,
    4: test_cluster15
}
test_layer8 = Layer4Tunnels(8, 1.5, params, "Cluster_44", "e10s1_e9s3p0f1600")
test_layer8.clusters = {
    1: test_cluster16,
    2: test_cluster17,
    3: test_cluster18
}
test_layer9 = Layer4Tunnels(9, 1.5, params, "Cluster_44", "e10s1_e9s3p0f1600")
test_layer9.clusters[1] = test_cluster19

test_layer10 =  Layer4Events(0, 1.5, params, "2_release", "e10s1_e9s3p0f1600")
test_layer10.clusters[0] = test_cluster20
test_layer11 =  Layer4Events(1, 1.5, params, "2_release", "e10s1_e9s3p0f1600")
test_layer11.clusters[0] = test_cluster21
test_layer12 =  Layer4Events(4, 1.5, params, "2_release", "e10s1_e9s3p0f1600")
test_layer12.clusters[0] = test_cluster22
test_layer13 =  Layer4Events(5, 1.5, params, "2_release", "e10s1_e9s3p0f1600")
test_layer13.clusters[0] = test_cluster23
test_layer14 =  Layer4Events(7, 1.5, params, "2_release", "e10s1_e9s3p0f1600")
test_layer14.clusters[0] = test_cluster24


test_layers_tun = {
    0: test_layer0,
    1: test_layer1,
    2: test_layer2,
    3: test_layer3,
    4: test_layer4,
    5: test_layer5,
    6: test_layer6,
    7: test_layer7,
    8: test_layer8,
    9: test_layer9
}

test_layers_event = {
    0: test_layer10,
    1: test_layer11,
    4: test_layer12,
    5: test_layer13,
    7: test_layer14
}



test_pathset_tun_str = """LayeredPathSet: Cluster_44 of e10s1_e9s3p0f1600
Paths: 
 0: ['SP' '0_1' '1_1' '2_1' '3_1' '4_1' '5_1' '6_2' '7_1' '8_1' '8_3' '7_3'
 '6_3' '5_2' '5_3' '6_1' '7_2' '7_4' '8_2' '9_1']
num nodes = 20
Labels 20:
['SP', '0_1', '1_1', '2_1', '3_1', '4_1', '5_1', '6_2', '7_1', '8_1', '8_3', '7_3', '6_3', '5_2', '5_3', '6_1', '7_2', '7_4', '8_2', '9_1']
Data:
[[ 0.          0.          0.         -1.          0.          0.5
   0.5       ]
 [ 0.10043729  0.15234931  0.35513633  0.          0.          1.90084216
   0.55901698]
 [ 1.03013585 -1.6684762   0.7431847   1.          0.          1.75460752
   0.76766343]
 [ 0.98865915 -3.35913189 -0.3577335   2.          0.          1.13337687
   0.40788472]
 [ 1.00162165 -4.9196445  -1.12833583  3.          0.          1.52419447
   0.54953664]
 [ 1.63440404 -6.25079688 -0.30013626  4.          0.          1.47635487
   0.70778596]
 [ 2.5096751  -7.45205531  1.93269901  5.          0.          1.03402714
   0.85898915]
 [ 4.23894511 -8.01548424  2.74830326  6.          0.          1.04662555
   0.38355875]
 [ 6.09348132 -8.05521444  4.23592075  7.          0.          1.33105369
   0.9823274 ]
 [ 8.03929604 -7.11295026  6.67345322  8.          0.          1.41127447
   0.99683108]
 [ 8.22922934 -4.59781624  8.14151136  8.          0.          1.6264471
   0.61417721]
 [ 6.88282209 -2.9506629   7.90726802  7.          0.          1.64466093
   0.68580067]
 [ 5.94265674 -1.00820122  7.10153142  6.          0.          1.46661868
   0.69622582]
 [ 4.59581293  1.16044215  7.02264766  5.          0.          1.30898044
   0.82296819]
 [ 3.49228254  2.89428692  7.28076416  5.          0.          1.21539959
   0.40714536]
 [ 2.28787894  4.17635691  8.35307284  6.          0.          1.53912947
   0.86976975]
 [-0.65643315  4.29371552  9.5635684   7.          0.          1.97040307
   0.89856267]
 [-2.04719596  6.32427935  9.2836044   7.          0.          1.26168916
   0.53985591]
 [-2.60148222  7.52286017  9.68079651  8.          0.          1.16408579
   0.25      ]
 [-3.18481916  8.73147993 10.64338364  9.          1.          1.5221901
   0.6503509 ]]"""

test_pathset_event_str = """LayeredPathSet: 2_release of e10s1_e9s3p0f1600
Paths: 
 0: ['0_0' '1_0' '4_0' '5_0' '7_0']
num nodes = 5
Labels 5:
['0_0', '1_0', '4_0', '5_0', '7_0']
Data:
[[ 0.03182512 -0.02023714 -0.03037115  0.          0.          0.
   0.048423  ]
 [ 0.57668091 -1.0800446  -1.13472572  1.          0.          0.
   0.        ]
 [ 1.83178559 -5.68151838 -1.60292218  4.          0.          0.
   0.        ]
 [-0.587065   -7.47959801 -3.51745714  5.          0.          0.
   0.        ]
 [-3.00591558 -9.27767764 -5.43199211  7.          1.          0.
   0.        ]]"""

test_merged_pathset_str = """LayeredPathSet: data merged-unique of various cls
Paths: 
 0: ['merged-SP_1', 'merged-0_1', 'merged-1_1', 'merged-2_1', 'merged-3_1', 'merged-4_1', 'merged-5_1', 'merged-6_1', 'merged-7_1', 'merged-8_1', 'merged-8_2', 'merged-7_2', 'merged-6_2', 'merged-5_2', 'merged-5_3', 'merged-6_3', 'merged-7_3', 'merged-7_4', 'merged-8_3', 'merged-9_1']
num nodes = 20
Labels 20:
['merged-SP_1', 'merged-0_1', 'merged-1_1', 'merged-2_1', 'merged-3_1', 'merged-4_1', 'merged-5_1', 'merged-6_1', 'merged-7_1', 'merged-8_1', 'merged-8_2', 'merged-7_2', 'merged-6_2', 'merged-5_2', 'merged-5_3', 'merged-6_3', 'merged-7_3', 'merged-7_4', 'merged-8_3', 'merged-9_1']
Data:
[[ 0.          0.          0.         -1.          0.          0.5
   0.5       ]
 [ 0.10043729  0.15234931  0.35513633  0.          0.          1.90084216
   0.55901698]
 [ 1.03013585 -1.6684762   0.7431847   1.          0.          1.75460752
   0.76766343]
 [ 0.98865915 -3.35913189 -0.3577335   2.          0.          1.13337687
   0.40788472]
 [ 1.00162165 -4.9196445  -1.12833583  3.          0.          1.52419447
   0.54953664]
 [ 1.63440404 -6.25079688 -0.30013626  4.          0.          1.47635487
   0.70778596]
 [ 2.5096751  -7.45205531  1.93269901  5.          0.          1.03402714
   0.85898915]
 [ 4.23894511 -8.01548424  2.74830326  6.          0.          1.04662555
   0.38355875]
 [ 6.09348132 -8.05521444  4.23592075  7.          0.          1.33105369
   0.9823274 ]
 [ 8.03929604 -7.11295026  6.67345322  8.          0.          1.41127447
   0.99683108]
 [ 8.22922934 -4.59781624  8.14151136  8.          0.          1.6264471
   0.61417721]
 [ 6.88282209 -2.9506629   7.90726802  7.          0.          1.64466093
   0.68580067]
 [ 5.94265674 -1.00820122  7.10153142  6.          0.          1.46661868
   0.69622582]
 [ 4.59581293  1.16044215  7.02264766  5.          0.          1.30898044
   0.82296819]
 [ 3.49228254  2.89428692  7.28076416  5.          0.          1.21539959
   0.40714536]
 [ 2.28787894  4.17635691  8.35307284  6.          0.          1.53912947
   0.86976975]
 [-0.65643315  4.29371552  9.5635684   7.          0.          1.97040307
   0.89856267]
 [-2.04719596  6.32427935  9.2836044   7.          0.          1.26168916
   0.53985591]
 [-2.60148222  7.52286017  9.68079651  8.          0.          1.16408579
   0.25      ]
 [-3.18481916  8.73147993 10.64338364  9.          1.          1.5221901
   0.6503509 ]]"""


test_pathset_extended_labels4nodes1 = ['e10s1_e9s3p0f1600-Cluster_44-SP', 'e10s1_e9s3p0f1600-Cluster_44-0_1',
                                       'e10s1_e9s3p0f1600-Cluster_44-1_1', 'e10s1_e9s3p0f1600-Cluster_44-2_1',
                                       'e10s1_e9s3p0f1600-Cluster_44-3_1', 'e10s1_e9s3p0f1600-Cluster_44-4_1',
                                       'e10s1_e9s3p0f1600-Cluster_44-5_1', 'e10s1_e9s3p0f1600-Cluster_44-6_2',
                                       'e10s1_e9s3p0f1600-Cluster_44-7_1', 'e10s1_e9s3p0f1600-Cluster_44-8_1',
                                       'e10s1_e9s3p0f1600-Cluster_44-8_3', 'e10s1_e9s3p0f1600-Cluster_44-7_3',
                                       'e10s1_e9s3p0f1600-Cluster_44-6_3', 'e10s1_e9s3p0f1600-Cluster_44-5_2',
                                       'e10s1_e9s3p0f1600-Cluster_44-5_3', 'e10s1_e9s3p0f1600-Cluster_44-6_1',
                                       'e10s1_e9s3p0f1600-Cluster_44-7_2', 'e10s1_e9s3p0f1600-Cluster_44-7_4',
                                       'e10s1_e9s3p0f1600-Cluster_44-8_2', 'e10s1_e9s3p0f1600-Cluster_44-9_1']

test_pathset_extended_labels4nodes2 = ['e10s1_e9s3p0f1600-2_release-0_0', 'e10s1_e9s3p0f1600-2_release-1_0',
                                       'e10s1_e9s3p0f1600-2_release-4_0', 'e10s1_e9s3p0f1600-2_release-5_0',
                                       'e10s1_e9s3p0f1600-2_release-7_0']

test_pathset_extended_labels4nodes3 = ['merged-SP_1', 'merged-0_1', 'merged-1_1', 'merged-2_1', 'merged-3_1',
                                       'merged-4_1', 'merged-5_1', 'merged-6_1', 'merged-7_1', 'merged-8_1',
                                       'merged-8_2', 'merged-7_2', 'merged-6_2', 'merged-5_2', 'merged-5_3',
                                       'merged-6_3', 'merged-7_3', 'merged-7_4', 'merged-8_3', 'merged-9_1']

test_pathset_extended_labels4paths1 = [test_pathset_extended_labels4nodes1]
test_pathset_extended_labels4paths2 = [test_pathset_extended_labels4nodes2]
test_pathset_extended_labels4paths3 = [test_pathset_extended_labels4nodes3]

test_pathset_adjacent_nodes1 = ['SP', '4_1', '5_1', '6_2', '6_3', '5_2', '5_3', '6_1']

test_pathset_adjacent_nodes2 = ['SP', '4_1', '5_1', '6_2', '7_1', '8_1', '8_3', '7_3', '6_3', '5_2', '5_3', '6_1',
                                '7_2', '7_4', '8_2', '9_1']

test_pathset_adjacent_nodes3 = ['SP', '4_1', '5_1', '6_2', '6_3', '5_2', '5_3', '6_1']

test_pathset_adjacent_nodes4 = ['SP', '8_1', '8_3', '8_2', '9_1']

test_pathset_fragments1 = [
    ['SP', '0_1', '1_1', '2_1', '3_1', '4_1', '5_1', '6_2', '7_1', '8_1', '8_3', '7_3', '6_3', '5_2', '5_3', '6_1',
     '7_2', '7_4', '8_2', '9_1']]
test_pathset_fragments2 = [['SP', '0_1', '1_1', '2_1', '3_1'],
                           ['4_1', '5_1', '6_2', '7_1', '8_1', '8_3', '7_3', '6_3', '5_2', '5_3', '6_1', '7_2', '7_4',
                            '8_2', '9_1']]


test_pathset_event2_nodes_data = np.array(
    [[-0.031825117462624064, -0.020237142651351903, 0.030371154891405183, 0.0, 0.0, 0.0, 0.048423001700005235],
     [-0.5766809115273155, -1.080044599868236, 1.1347257202266832, 1.0, 0.0, 0.0, 0.0],
     [-1.8317855854326517, -5.681518382112968, 1.6029221763147845, 4.0, 0.0, 0.0, 0.0],
     [-0.5870649957828888, -7.479598010247797, 3.517457141893324, 5.0, 0.0, 0.0, 0.0],
     [-3.0059155769984187, -9.277677638382618, 5.4319921074718565, 7.0, 1.0, 0.0, 0.0]])



test_pathset_adjacent_data1 = np.array([[0., 0., 0., -1., 0., 0.5, 0.5],
                                       [1.63440404, -6.25079688, -0.30013626, 4., 0., 1.47635487, 0.70778596],
                                       [2.5096751, -7.45205531, 1.93269901, 5., 0., 1.03402714, 0.85898915],
                                       [4.23894511, -8.01548424, 2.74830326, 6., 0., 1.04662555, 0.38355875],
                                       [5.94265674, -1.00820122, 7.10153142, 6., 0., 1.46661868, 0.69622582],
                                       [4.59581293, 1.16044215, 7.02264766, 5., 0., 1.30898044, 0.82296819],
                                       [3.49228254, 2.89428692, 7.28076416, 5., 0., 1.21539959, 0.40714536],
                                       [2.28787894, 4.17635691, 8.35307284, 6., 0., 1.53912947, 0.86976975]])

test_pathset_adjacent_data2 = np.array([[0.0, 0.0, 0.0, -1.0, 0.0, 0.5, 0.5],
                                        [1.634404038033623, -6.2507968763522355, -0.3001362595890222, 4.0, 0.0,
                                         1.4763548743917065, 0.7077859590281326],
                                        [2.509675098119873, -7.452055312390876, 1.932699012630189, 5.0, 0.0,
                                         1.0340271414649607, 0.8589891546848893],
                                        [4.238945109883823, -8.015484239879276, 2.748303255015273, 6.0, 0.0,
                                         1.0466255475735333, 0.38355875199269285],
                                        [6.093481321661526, -8.055214435100105, 4.235920754781024, 7.0, 0.0,
                                         1.3310536902729193, 0.9823273952114995],
                                        [8.039296044428076, -7.112950263830479, 6.673453216993297, 8.0, 0.0,
                                         1.4112744679935547, 0.9968310847472718],
                                        [8.22922934081849, -4.597816241390133, 8.141511360114043, 8.0, 0.0,
                                         1.6264470960007718, 0.614177205862268],
                                        [6.882822088177425, -2.9506628952736516, 7.907268024399042, 7.0, 0.0,
                                         1.6446609326204227, 0.6858006716310032],
                                        [5.942656741364063, -1.0082012240002745, 7.101531416870081, 6.0, 0.0,
                                         1.4666186773827372, 0.6962258220918012],
                                        [4.595812933362087, 1.1604421542801129, 7.022647661952056, 5.0, 0.0,
                                         1.3089804357981103, 0.8229681892525533],
                                        [3.4922825377634568, 2.8942869246605345, 7.280764161880406, 5.0, 0.0,
                                         1.2153995948191962, 0.40714536059634715],
                                        [2.2878789365490744, 4.1763569139166945, 8.353072838459832, 6.0, 0.0,
                                         1.5391294709187215, 0.8697697492544836],
                                        [-0.6564331516008213, 4.293715519177398, 9.563568395782715, 7.0, 0.0,
                                         1.9704030716610481, 0.8985626746628881],
                                        [-2.0471959558277137, 6.3242793510324065, 9.283604398328622, 7.0, 0.0,
                                         1.2616891636582148, 0.5398559062421392],
                                        [-2.6014822194787834, 7.522860169146057, 9.680796508867367, 8.0, 0.0,
                                         1.164085794687311, 0.24999999923819152],
                                        [-3.1848191599444857, 8.731479931193862, 10.643383638680215, 9.0, 1.0,
                                         1.5221901018091808, 0.650350901934307]]
                                       )

test_pathset_adjacent_data3 = np.array([[0.0, 0.0, 0.0, -1.0, 0.0, 0.5, 0.5],
                                        [1.634404038033623, -6.2507968763522355, -0.3001362595890222, 4.0, 0.0,
                                         1.4763548743917065, 0.7077859590281326],
                                        [2.509675098119873, -7.452055312390876, 1.932699012630189, 5.0, 0.0,
                                         1.0340271414649607, 0.8589891546848893],
                                        [4.238945109883823, -8.015484239879276, 2.748303255015273, 6.0, 0.0,
                                         1.0466255475735333, 0.38355875199269285],
                                        [5.942656741364063, -1.0082012240002745, 7.101531416870081, 6.0, 0.0,
                                         1.4666186773827372, 0.6962258220918012],
                                        [4.595812933362087, 1.1604421542801129, 7.022647661952056, 5.0, 0.0,
                                         1.3089804357981103, 0.8229681892525533],
                                        [3.4922825377634568, 2.8942869246605345, 7.280764161880406, 5.0, 0.0,
                                         1.2153995948191962, 0.40714536059634715],
                                        [2.2878789365490744, 4.1763569139166945, 8.353072838459832, 6.0, 0.0,
                                         1.5391294709187215, 0.8697697492544836]]
                                       )

test_pathset_adjacent_data4 = np.array([[0.0, 0.0, 0.0, -1.0, 0.0, 0.5, 0.5],
                                        [8.039296044428076, -7.112950263830479, 6.673453216993297, 8.0, 0.0,
                                         1.4112744679935547, 0.9968310847472718],
                                        [8.22922934081849, -4.597816241390133, 8.141511360114043, 8.0, 0.0,
                                         1.6264470960007718, 0.614177205862268],
                                        [-2.6014822194787834, 7.522860169146057, 9.680796508867367, 8.0, 0.0,
                                         1.164085794687311, 0.24999999923819152],
                                        [-3.1848191599444857, 8.731479931193862, 10.643383638680215, 9.0, 1.0,
                                         1.5221901018091808, 0.650350901934307]]
                                       )

test_pathset_zipobject1 = [('SP', 8.472644370416788, 6.663663930416788, 6.663663930416788),
                           ('4_1', 10.831415921482082, 8.046080607090374, 8.046080607090374),
                           ('5_1', 10.219327806962285, 7.876320225497325, 7.876320225497325),
                           ('6_2', 10.12891898703616, 7.773312999462627, 7.773312999462627),
                           ('6_3', 2.554060489280336, -0.221538628102401, -0.221538628102401),
                           ('5_2', 5.782172329113071e-09, -2.617960870015938, -2.617960870015938),
                           ('5_3', 2.0713814611872814, -0.45299857363191487, -0.45299857363191487),
                           ('6_1', 4.023969723645657, 1.1758598127269355, 1.1758598127269355)]

test_pathset_zipobject2 = [('4_0', 11.351331467526224, 10.042351027526223, 10.042351027526223),
                           ('5_0', 14.581026260149079, 13.272045820149078, 13.272045820149078)]

test_pathset_zipobject3 = [('4_0', 11.351331467526224, 10.042351027526223, 9.630866932526223),
                           ('5_0', 14.581026260149079, 13.272045820149078, 12.860561725149077)]

test_pathset_dist_mat = {
    'SP': {'SP': (0.0, -1.0, -1.0), '0_1': (0.46666383333252826, -2.1862000941303004, -2.1862000941303004)},
    '0_1': {'SP': (0.3992740609863447, -2.0015680977823385, -2.0015680977823385),
            '0_1': (0.5776708194963596, -3.476035266735152, -3.476035266735152),
            '1_2': (1.7744885789971139, -2.0747862334867238, -2.0747862334867238),
            '1_1': (1.2226097998607124, -2.8130437214741475, -2.8130437214741475)},
    '1_1': {'SP': (2.0969778756643485, -0.15762964212068908, -0.15762964212068908),
            '0_1': (2.543865127949629, -1.3636063172982373, -1.3636063172982373),
            '1_2': (3.7540047096432496, 0.05096453814305768, 0.05096453814305768),
            '2_1': (5.239347160869649, 1.8923971744456702, 1.8923971744456702),
            '1_1': (0.9982933735745952, -2.891125506776619, -2.891125506776619)},
    '2_1': {'SP': (3.519827724607891, 1.886450850632734, 1.886450850632734),
            '1_2': (5.278214047489037, 2.1964045197987256, 2.1964045197987256),
            '2_1': (6.916524199987972, 4.190804857373874, 4.190804857373874),
            '3_1': (8.451402530004664, 6.021612764310747, 6.021612764310747),
            '1_1': (2.826872844952751, -0.44131539158858235, -0.44131539158858235)},
    '3_1': {'SP': (5.145803109196788, 3.121608639815455, 3.121608639815455),
            '2_1': (8.556298822133623, 5.439761884113349, 5.439761884113349),
            '3_1': (10.104921274768934, 7.284313913668841, 7.284313913668841),
            '4_1': (11.649203275543163, 8.990838392722367, 8.990838392722367)},
    '4_1': {'SP': (6.4679069198060954, 4.491552045414389, 4.491552045414389),
            '3_1': (11.37826412374472, 8.605496357634255, 8.605496357634255),
            '4_1': (12.891916614209734, 10.281391326378564, 10.281391326378564),
            '5_2': (14.343635738721531, 11.365735744930928, 11.365735744930928)},
    '5_1': {'SP': (8.097340486263239, 6.563313344798278, 6.563313344798278),
            '4_1': (14.121771561118013, 11.95357400621359, 11.95357400621359),
            '5_2': (15.45479728348115, 12.919225022617294, 12.919225022617294),
            '6_1': (16.644863312027827, 13.874616948046384, 13.874616948046384),
            '6_2': (14.426202833686899, 11.425106103754164, 11.425106103754164)},
    '6_2': {'SP': (9.474693347327321, 7.928067799753787, 7.928067799753787),
            '5_2': (16.43954670361769, 13.891376036645262, 13.891376036645262),
            '6_1': (17.508812147524964, 14.725967377434948, 14.725967377434948),
            '7_2': (17.79657500302104, 15.148125152603969, 15.148125152603969),
            '6_2': (15.125302468635809, 12.1116073325945, 12.1116073325945),
            '7_1': (14.295977183030278, 11.290297933895378, 11.290297933895378)},
    '7_1': {'SP': (10.952626116766135, 9.121572426493216, 9.121572426493216),
            '6_1': (18.12709041886521, 15.059817506075811, 15.059817506075811),
            '7_2': (18.21011570868435, 15.27723771556789, 15.27723771556789),
            '6_2': (15.478789541931782, 12.180666263191087, 12.180666263191087),
            '7_1': (14.362188009928419, 11.072080618094134, 11.072080618094134),
            '8_10': (14.845043857581546, 11.568338644151408, 11.568338644151408),
            '8_5': (13.5005182401195, 10.302853222221056, 10.302853222221056)},
    '8_1': {'SP': (12.63959335521049, 10.728318887216936, 10.728318887216936),
            '7_2': (18.095451317833277, 15.082352546996184, 15.082352546996184),
            '7_1': (13.863264611922725, 10.492936442367805, 10.492936442367805),
            '8_10': (14.079510172461543, 10.722584181310769, 10.722584181310769),
            '9_2': (15.4128472062701, 12.63042875656216, 12.63042875656216),
            '9_4': (13.013827371089606, 9.94977818014985, 9.94977818014985),
            '8_5': (12.536687434984104, 9.258801639365025, 9.258801639365025),
            '9_1': (7.710436956949444, 4.089177902049237, 4.089177902049237)},
    '8_3': {'SP': (12.455694960950385, 10.329247864949615, 10.329247864949615),
            '7_2': (16.15510935450841, 12.926837955664102, 12.926837955664102),
            '7_1': (11.917238467159008, 8.331737669596869, 8.331737669596869),
            '8_10': (12.041684487571297, 8.469585868413308, 8.469585868413308),
            '9_2': (13.08594037281216, 10.088349295097004, 10.088349295097004),
            '9_4': (10.900366481641708, 7.621144662694735, 7.621144662694735),
            '8_5': (10.47411877354723, 6.981060349920933, 6.981060349920933),
            '9_1': (5.802285863622924, 1.9658541807155006, 1.9658541807155006)},
    '7_3': {'SP': (10.89057110663767, 8.745910174017247, 8.745910174017247),
            '6_1': (14.612152385207978, 11.231272230071074, 11.231272230071074),
            '7_2': (14.029542135229327, 10.783056899765366, 10.783056899765366),
            '6_2': (11.345410901178594, 7.733680380090396, 7.733680380090396),
            '7_1': (9.930105971746645, 6.326391337564857, 6.326391337564857),
            '8_10': (10.198917816850503, 6.608605361072862, 6.608605361072862),
            '8_5': (8.740991146639598, 5.229718886393649, 5.229718886393649)},
    '6_3': {'SP': (9.314686646288655, 7.348067968905918, 7.348067968905918),
            '5_2': (11.921488232191416, 8.953324435409783, 8.953324435409783),
            '6_1': (12.316142786011042, 9.113304886111823, 9.113304886111823),
            '7_2': (11.809030513177742, 8.740587532951466, 8.740587532951466),
            '6_2': (9.358737974181738, 5.925049708331225, 5.925049708331225),
            '7_1': (8.395379304642251, 4.969706925698148, 4.969706925698148)},
    '5_2': {'SP': (8.472644374444686, 6.663663938646575, 6.663663938646575),
            '4_1': (9.257344587591474, 6.814193738353901, 6.814193738353901),
            '5_2': (9.61535924685363, 6.804833691656625, 6.804833691656625),
            '6_1': (9.909718525957114, 6.864518867642522, 6.864518867642522),
            '6_2': (6.972227465118697, 3.6961774408528107, 3.6961774408528107)},
    '5_3': {'SP': (8.57802196935579, 6.862622374536594, 6.862622374536594),
            '4_1': (7.93726468812888, 5.587694679870221, 5.587694679870221),
            '5_2': (8.063121754671831, 5.34617704045374, 5.34617704045374),
            '6_1': (8.20082850178872, 5.249209684453042, 5.249209684453042),
            '6_2': (5.106430593182922, 1.9239614098959503, 1.9239614098959503)},
    '6_1': {'SP': (9.615101296675979, 7.575971825757257, 7.575971825757257),
            '5_2': (7.597200533516739, 4.556525943199122, 4.556525943199122),
            '6_1': (7.516601238921816, 4.2412525454866135, 4.2412525454866135),
            '7_2': (5.90473383499579, 2.76378006123353, 2.76378006123353),
            '6_2': (3.4894804686706755, -0.016718590715821602, -0.016718590715821602),
            '7_1': (3.7374721735120997, 0.23928900103201256, 0.23928900103201256)},
    '7_2': {'SP': (10.503748754752248, 8.0333456830912, 8.0333456830912),
            '6_1': (8.037350706266164, 4.330728412088634, 4.330728412088634),
            '7_2': (5.751718452715616, 2.17949107821103, 2.17949107821103),
            '6_2': (1.8979554765479245, -2.039517183580899, -2.039517183580899),
            '7_1': (1.4966750847182602, -2.432781688504154, -2.432781688504154),
            '8_10': (3.0526650326754416, -0.8633895621428249, -0.8633895621428249),
            '8_5': (4.273646405081241, 0.4366320057946673, 0.4366320057946673)},
    '7_4': {'SP': (11.418092275687684, 9.65640311202947, 9.65640311202947),
            '6_1': (7.059009780870155, 4.061101394695458, 4.061101394695458),
            '7_2': (4.387231701700961, 1.5237182351992082, 1.5237182351992082),
            '6_2': (2.044496658026665, -1.1842620940993251, -1.1842620940993251),
            '7_1': (3.73768317000265, 0.516940304783069, 0.516940304783069),
            '8_10': (4.89505289796376, 1.6877122111483267, 1.6877122111483267),
            '8_5': (6.460857755572465, 3.3325572642887247, 3.3325572642887247)},
    '8_2': {'SP': (12.533114373885171, 10.86902857919786, 10.86902857919786),
            '7_2': (4.411423955066746, 1.6455138575358963, 1.6455138575358963),
            '7_1': (4.942353446257283, 1.8192139500086064, 1.8192139500086064),
            '8_10': (5.842370561439914, 2.7326332435953846, 2.7326332435953846),
            '9_2': (5.501585871858355, 2.966356095456658, 2.966356095456658),
            '9_4': (8.538303906755603, 5.721443389122091, 5.721443389122091),
            '8_5': (7.502719336690627, 4.472022214377791, 4.472022214377791),
            '9_1': (13.601244975504168, 10.227174593910204, 10.227174593910204)},
    '9_1': {'SP': (14.130231072084998, 12.108040970275816, 12.108040970275816),
            '8_10': (6.811242858563936, 3.343401233597537, 3.343401233597537),
            '9_2': (6.071017377660385, 3.177683294136818, 3.177683294136818),
            '10_1': (7.848033382877237, 4.728683455961704, 4.728683455961704),
            '11_1': (9.029280635293915, 5.570001183523357, 5.570001183523357),
            '12_1': (10.008249855491282, 6.2974756506378355, 6.2974756506378355),
            '13_1': (11.264066257188375, 7.186725711453629, 7.186725711453629),
            '14_2': (13.745240753207076, 9.906958462400311, 9.906958462400311),
            '9_4': (9.347776588519144, 6.172811763763763, 6.172811763763763),
            '10_5': (10.401447751415382, 7.377886029962475, 7.377886029962475),
            '10_2': (9.464234163453524, 5.426821009750334, 5.426821009750334),
            '11_3': (11.284886649959988, 7.859000606225443, 7.859000606225443),
            '10_4': (11.091189357972244, 7.730731460350437, 7.730731460350437),
            '11_2': (13.23135080914084, 9.287514124033997, 9.287514124033997),
            '12_3': (14.364754343858019, 10.208249213080514, 10.208249213080514),
            '12_4': (13.62854891921571, 10.084382296842126, 10.084382296842126),
            '12_2': (18.18591511023075, 14.322587042092975, 14.322587042092975),
            '13_2': (16.37152056752336, 12.752360334086251, 12.752360334086251),
            '13_9': (19.76413098522307, 16.009325203502986, 16.009325203502986),
            '13_8': (18.11106074739897, 14.969620119144, 14.969620119144),
            '13_7': (15.064501389063695, 11.548767220804029, 11.548767220804029),
            '14_1': (19.1815087858199, 15.914773526379419, 15.914773526379419),
            '14_7': (19.764449225870177, 16.2640228062077, 16.2640228062077),
            '8_5': (8.532705657564032, 5.143904228129325, 5.143904228129325),
            '12_8': (16.35392278885028, 12.137608864917794, 12.137608864917794),
            '9_1': (14.740847218708419, 11.008672529992587, 11.008672529992587),
            '10_3': (15.24733756703669, 11.2472084229751, 11.2472084229751),
            '13_11': (15.764366061127191, 12.213907003008007, 12.213907003008007),
            '12_5': (13.947103553801599, 10.291001257861684, 10.291001257861684),
            '11_4': (12.762739255602348, 9.062625542520433, 9.062625542520433),
            '13_12': (15.65365295654324, 11.724408580459242, 11.724408580459242),
            '12_11': (13.675569827951248, 9.790839934715363, 9.790839934715363)}}

test_pathset_dists2closest_node = [(0, '0_1'), (0, '0_1'), (0, '1_1'), (0, '1_1'), (3.121608639815455, 'SP'),
                                   (4.491552045414389, 'SP'), (6.563313344798278, 'SP'), (7.928067799753787, 'SP'),
                                   (9.121572426493216, 'SP'), (4.089177902049237, '9_1'), (1.9658541807155006, '9_1'),
                                   (5.229718886393649, '8_5'), (4.969706925698148, '7_1'), (3.6961774408528107, '6_2'),
                                   (1.9239614098959503, '6_2'), (0, '6_2'), (0, '7_1'), (0, '6_2'),
                                   (1.6455138575358963, '7_2'), (3.177683294136818, '9_2')]

test_pathset_path1 = np.array(
    ['SP', '0_1', '1_1', '2_1', '3_1', '4_1', '5_2', '6_1', '7_1', '7_3', '7_2', '8_2', '9_2', '10_2', '11_1', '12_2',
     '13_1'])
test_pathset_path2 = np.array(['SP', '1_2', '2_1', '3_1', '4_1', '5_1', '6_1', '7_2', '8_1'])

test_layers_points_mat1 = np.array(
    [[-0.2721180365941773, 0.6109924224913796, -0.10676083241216117, 0.6773166478974898, 1.9458834779974528, 0.0, 0.0],
     [-0.023747816079717765, 0.30523034623404044, 0.20117061027419259, 0.3663320044287687, 1.7957457472348868, 1.0,
      0.0],
     [0.22462240443474357, -0.0005317300232978717, 0.5091020529605454, 0.5564534191102658, 1.7244490604239398, 2.0,
      0.0],
     [0.47299262494920313, -0.3062938062806375, 0.8170334956468984, 0.992512796910553, 1.7377392722896674, 3.0, 0.0]]
    )
test_layers_points_mat2 = np.array(
    [[1.2518678167878008, -5.848624735852448, -1.2202303762378919, 6.104305570876007, 1.516289703126508, 17.0, 0.0],
     [1.3431987297252377, -6.092037929145109, -0.8183398082372277, 6.291803318558508, 1.4364200456569052, 18.0, 0.0],
     [1.4403836604063045, -6.372292331712506, -0.4270357888953136, 6.546997343436525, 1.3679686036597702, 19.0, 0.0],
     [1.7072816171766014, -6.498753179433438, -0.028199282164253603, 6.71933022012313, 1.3705052104932915, 20.0, 0.0],
     [2.0156759733115246, -6.313016735479083, 0.13431372911922668, 6.628361042494159, 1.3734882154761638, 21.0, 0.0],
     [2.04801643079427, -6.380056346490834, 0.5586739688813267, 6.7239576804666825, 1.2787357198379101, 22.0, 0.0]]
    )
test_layers_points_mat3 = np.array(
    [[-2.7471118818580624, 8.13074989850712, 10.09523839254286, 13.250266254147146, 1.214184950644569, 94.0, 0.0],
     [-2.8719921387289595, 8.489915583337527, 10.411531231760687, 13.737794147789758, 1.3635530271563103, 95.0, 0.0],
     [-3.0936555966292083, 8.822660421558787, 10.685595075105429, 14.198309194189726, 1.4897115534087206, 96.0, 0.0],
     [-3.39065475279647, 9.116984603988454, 10.885114227383262, 14.598001906603702, 1.5438424674094875, 97.0, 0.0],
     [ -3.82068143, 9.09708915, 11.13943927, 14.88091881, 1.47796994, -98., 0.0]]
    )

test_layers_points_mat4 = np.array(
    [[0.45182817626410987, 9.496012380868514, 3.9679469703279673, 10.301601962695374, 1.4349103079322572, 25.0, 0.0],
     [0.622702708948415, 9.585357673581438, 4.410512049142433, 10.569742519562567, 1.4062096029039381, 26.0, 0.0],
     [0.8812423727982832, 9.538047835154957, 4.835857805478698, 10.730166137582799, 1.3856082645347898, 27.0, 0.0],
     [0.5910883328514505, 9.490145251180278, 5.231696348997335, 10.852782546142015, 1.2002056791935922, 28.0, 0.0],
     [0.2885264185289511, 9.442229263165698, 5.626867631734074, 10.995479993953351, 1.1036520909803593, 29.0, 0.0],
     [-0.060089537665016124, 9.398732774169133, 5.980922379627682, 11.140521577731855, 1.096666387264024, 30.0, 0.0],
     [-0.42835122758198985, 9.357481383255372, 6.3143266631628485, 11.296754570285712, 1.1884172064036906, 31.0, 0.0],
     [-0.8133432250606525, 9.276537055633112, 6.509104792095366, 11.36151891873464, 1.2679820904911188, 32.0, 0.0],
     [-1.0785244995520245, 8.956898485868106, 6.787511092703626, 11.28979859943724, 1.264738459220274, 33.0, 0.0],
     [-1.3437057740433964, 8.6372599161031, 7.065917393311885, 11.239884015176772, 1.3610419258056183, 34.0, 0.0],
     [-1.569429401560896, 8.29313963135275, 7.342277809435528, 11.186970860001683, 1.517335140037653, 35.0, 0.0],
     [-1.7846772533847748, 7.8594170023918615, 7.458198902244002, 10.980903395516643, 1.5911717153045724, 36.0, 0.0],
     [-1.7866832357963567, 7.3945438410419495, 7.459606743320476, 10.654447351559792, 1.5340376260944322, 37.0, 0.0],
     [-1.667762239614886, 6.909112641793982, 7.444966395462627, 10.292948703576174, 1.4820580824886271, 38.0, 0.0],
     [-0.9642598768465529, 3.5221061895716215, 9.499953945116294, 10.177630081705932, 1.677700491775785, 48.0, 0.0],
     [-0.8498587564786195, 3.419485455370014, 9.974541344560652, 10.578592331676285, 1.7364739481938127, 49.0, 0.0],
     [-0.795784294049394, 3.0003430672085383, 10.230721933704938, 10.691258225690433, 1.6123684005312884, 50.0, 0.0],
     [-0.6289535083318114, 2.5931786956296365, 10.468176456814277, 10.802910561238162, 1.468601367158507, 51.0, 0.0],
     [-0.46680418763378206, 2.206210850907758, 10.709475735117042, 10.944320124581347, 1.4087327811487886, 52.0, 0.0],
     [-0.2966480127488378, 1.884420075409269, 11.051201029787798, 11.214637009946252, 1.3886010418266237, 53.0, 0.0],
     [0.061962559122509475, 1.5955214933403832, 11.24597283566075, 11.358760197084171, 1.2942448304847682, 54.0, 0.0],
     [0.42057313099385496, 1.3066229112714973, 11.440744641533703, 11.522794068434779, 1.2983312820869262, 55.0, 0.0],
     [0.7812244025421577, 1.031733766552058, 11.648495292463581, 11.720163339762687, 1.389548645640556, 56.0, 0.0],
     [1.1485489821232182, 0.8026569928807241, 11.898688166495825, 11.980909944410081, 1.5360119912460204, 57.0, 0.0],
     [1.9459066536385814, -0.7272796428106285, 11.720632784120424, 11.903307155732819, 1.4314861955751625, 61.0, 0.0],
     [2.099760647750367, -1.1229967184424772, 11.493254791851248, 11.73733454059674, 1.4131905514813659, 62.0, 0.0],
     [2.1323142038777396, -1.4788888279003989, 11.14356768662379, 11.441720885238952, 1.313020165893129, 63.0, 0.0],
     [2.164867760005115, -1.8347809373583197, 10.793880581396332, 11.160686874554266, 1.3103338124678892, 64.0, 0.0],
     [2.1974213161324876, -2.1906730468162405, 10.444193476168874, 10.895360765311493, 1.4054330438246947, 65.0, 0.0],
     [2.251241400422055, -2.4783890672475386, 10.085574270126674, 10.626820247367863, 1.5421378450130485, 66.0, 0.0],
     [2.6195005103983497, -2.464421388411417, 9.784598433418811, 10.424659332894228, 1.3303985017048243, 67.0, 0.0],
     [3.024555711596678, -2.4503967250045107, 9.491795444219122, 10.258974710825823, 1.166290916126279, 68.0, 0.0],
     [3.429610912795007, -2.4363720615976043, 9.198992455019434, 10.115315221148595, 1.1012494045742784, 69.0, 0.0],
     [3.8677632592940743, -2.444189276605942, 8.97557663189007, 10.07445431399397, 1.087686926087869, 70.0, 0.0],
     [4.351920222393898, -2.4845337698271446, 8.857405764433228, 10.176726121464968, 1.099014591302493, 71.0, 0.0],
     [4.704662886193828, -2.6728122430645977, 8.628293222969303, 10.184558021875258, 1.1287761294883223, 72.0, 0.0],
     [4.9567273307582465, -2.971586462242908, 8.316741751184681, 10.127569564805246, 1.1975549138838202, 73.0, 0.0],
     [5.17221267019978, -3.288026316353339, 7.995473297104645, 10.074249064199309, 1.3485747120727558, 74.0, 0.0],
     [5.464754510829522, -3.3633790914360175, 7.69987882598412, 10.023172886425694, 1.475724317611373, 75.0, 0.0],
     [6.285888548471876, -2.9739821688892083, 7.359225648466074, 10.12497737922025, 1.5374657075464346, 77.0, 0.0],
     [6.7344733058183746, -2.8185265867622413, 7.227493606471256, 10.272968736380383, 1.648244903227911, 78.0, 0.0],
     [7.0352834680787755, -2.904600920427742, 7.364046021314934, 10.590613475489345, 1.5507507683401338, 79.0, 0.0],
     [7.4871505128412315, -3.031275521115514, 7.5365850357434265, 11.04745075064128, 1.4523694362158122, 80.0, 0.0],
     [7.715637827716394, -3.377566223395758, 7.675284449655589, 11.395131068357086, 1.4264403967688775, 81.0, 0.0],
     [7.922689580615579, -3.7535939014942508, 7.904954838720338, 11.80452406369429, 1.395949034380277, 82.0, 0.0],
     [-0.6961871597453158, 9.042499163988053, 4.401999722772066, 10.081124404122995, 1.1667107460758481, 24.0, 1.0],
     [-0.7306319381830768, 9.130023096964706, 4.893073154004408, 10.384281856278786, 1.101823590569693, 25.0, 1.0],
     [-0.8379817566607972, 9.25657998248013, 5.354900633103348, 10.726669902022659, 1.0626994332404245, 26.0, 1.0],
     [-1.01242908216366, 9.22540405850935, 5.736116027381703, 10.910367535915405, 1.0429034469939353, 27.0, 1.0],
     [-1.1640794656232707, 8.856916279740044, 5.873232258046377, 10.690879484194298, 1.047505657451491, 28.0, 1.0],
     [-1.292161910170746, 8.530780600604018, 6.206105486207623, 10.628247520813312, 0.9695326797063439, 29.0, 1.0],
     [-1.330898959648529, 8.177353114199763, 6.5576569626358925, 10.56613746052516, 0.9357216281823468, 30.0, 1.0],
     [-1.369636009126312, 7.823925627795507, 6.909208439064164, 10.527434458651653, 1.0171471980583335, 31.0, 1.0],
     [-1.4543072361110667, 7.436980922927415, 7.211975212595206, 10.461179725637017, 1.1855733493960827, 32.0, 1.0],
     [-1.636253020111333, 7.015262997597243, 7.367705507823592, 10.304092552053994, 1.3816608490697628, 33.0, 1.0],
     [-1.642222472546024, 6.591621679882293, 7.407705671775423, 10.050894205974073, 1.4487429580291036, 34.0, 1.0],
     [-1.844672216392457, 3.458031685976297, 9.303224692318471, 10.095087340033787, 1.5896383942384793, 43.0, 1.0],
     [-1.7330308786236746, 3.1668826706514905, 9.625949839537432, 10.280634814505696, 1.617997507786623, 44.0, 1.0],
     [-1.4652698066696797, 2.969664468859941, 9.999324955779823, 10.53339557005376, 1.6173300639589605, 45.0, 1.0],
     [-1.203469238548779, 2.680678323676324, 10.278976018911443, 10.69073068038466, 1.652620426759212, 46.0, 1.0],
     [-1.0323860590119835, 2.2706928977280265, 10.505444944507735, 10.797510856337118, 1.633754057675245, 47.0, 1.0],
     [-0.7267979510200462, 2.1305376737259065, 10.864246564559576, 11.095011467155718, 1.5779360084446068, 48.0, 1.0],
     [-0.4814081819687388, 1.905791517026597, 11.214979643069395, 11.385937095316217, 1.6060403331559783, 49.0, 1.0],
     [-0.11883711344371406, 1.6835598980081892, 11.390772584776807, 11.515129016551047, 1.5093229057208049, 50.0, 1.0],
     [0.3378794466706019, 1.540159582630935, 11.53347089938358, 11.640756205993673, 1.3708630442602627, 51.0, 1.0],
     [0.7937133819877129, 1.4003044441226433, 11.683989854821773, 11.794339845729143, 1.3233810815152032, 52.0, 1.0],
     [1.1006356259954364, 1.1028677189164962, 11.71738729093965, 11.820527945591621, 1.2458456258784685, 53.0, 1.0],
     [1.2569204588466425, 0.631228823919824, 11.693915134974818, 11.778198081710443, 1.0868424034306055, 54.0, 1.0],
     [1.4190743384943865, 0.15840392373048662, 11.705861881705358, 11.792627619610016, 1.0123112829143588, 55.0, 1.0],
     [1.5812282181421295, -0.3144209764588499, 11.717808628435897, 11.828194379571762, 1.0492208088340467, 56.0, 1.0],
     [1.7326827946354246, -0.7905736940689256, 11.722778112300537, 11.876477722911451, 1.1878664626715094, 57.0, 1.0],
     [1.864769326144172, -1.2727504785340868, 11.715117246402672, 11.930684411101414, 1.4087211562868924, 58.0, 1.0],
     [2.106637150179507, -1.6817962493070762, 11.644078509689724, 11.952025897165628, 1.6118179105757788, 59.0, 1.0],
     [2.4203848370364938, -2.0183896166290363, 11.548834016608671, 11.971120521782815, 1.7720774967764712, 60.0, 1.0]])

test_layers_ouliers_point_coords = np.array([[10.302134315248715, 2.324024899951086, 7.724033501401122],
                                             [10.589775278293146, 2.548979303796636, 8.065586116190211],
                                             [10.76739398345264, 2.6561156290168997, 8.474175328138802],
                                             [10.475241609588693, 2.398696616451861, 8.729011039786837],
                                             [10.143704409616886, 2.0958422244970905, 8.948192600653764],
                                             [9.977623542124789, 1.706054032475167, 9.19500142033761],
                                             [9.874343844051017, 1.288215264073401, 9.449449954023791],
                                             [9.771064145977247, 0.8703764956716347, 9.703898487709974],
                                             [9.739219943525578, 0.462201954317341, 9.943112936064155],
                                             [10.05657866075525, 0.11030792841139458, 10.102656122535876],
                                             [10.143228768699094, -0.3754394952762863, 10.12035890139868],
                                             [10.144503671998468, -0.8201077327498201, 10.024096502165513],
                                             [10.229641037579208, -1.2804036918359074, 9.939582707225888],
                                             [10.466902658558592, -1.7176392072554485, 9.88926351810748],
                                             [10.704164279537974, -2.1548747226749905, 9.838944328989074],
                                             [10.941425900517357, -2.5921102380945316, 9.788625139870666],
                                             [6.425584721758157, 4.348716238414938, 10.808905368054123],
                                             [6.825835687127907, 4.285848069007885, 11.070758893522614],
                                             [7.185052676214131, 4.295229540471357, 11.028169671439018],
                                             [7.4919577409525395, 4.185461996994838, 10.649012228388164],
                                             [7.798862805690948, 4.07569445351832, 10.26985478533731],
                                             [8.105767870429357, 3.9659269100418015, 9.890697342286456],
                                             [8.412672935167766, 3.856159366565283, 9.511539899235602],
                                             [8.719253783399557, 3.736267903239369, 9.14890162526157],
                                             [9.028090393396726, 3.4699425553294043, 8.876699124182906],
                                             [9.311708176791855, 3.170306104138177, 8.594247448372561],
                                             [9.595325960186983, 2.8706696529469498, 8.311795772562215],
                                             [9.869058319971055, 2.6100787442232707, 8.347609199216535],
                                             [9.971607351083538, 2.32421121804407, 8.698083920141618],
                                             [9.793427838549198, 1.9170882623568355, 8.891894691295482],
                                             [9.725140738658649, 1.5425989813122039, 9.1977482589761],
                                             [9.650439066616897, 1.1771671062434537, 9.52820491227485],
                                             [9.557702871321524, 0.785606267979805, 9.824987823520164],
                                             [9.66876226791946, 0.4268567260039302, 9.983936940647833],
                                             [9.988547653119, 0.07645926075071552, 10.141920841825062],
                                             [10.171209647262422, -0.36501770663873856, 10.211483651534866],
                                             [10.315244112702432, -0.8435440493404744, 10.222589761355982],
                                             [10.501412828082064, -1.301243670438077, 10.25127421462564],
                                             [10.759869276519602, -1.7282096236009963, 10.221268678575218],
                                             [10.460130179314119, 2.6435769051602915, 7.714863980601313],
                                             [10.280925495689528, 2.5182601630155643, 8.139405583753685],
                                             [10.086425885884584, 2.2978718563075677, 8.543879177381223],
                                             [9.888854667504997, 2.0783082501092296, 8.947228809321741],
                                             [9.627590174648578, 1.879652820184312, 9.320629299800013],
                                             [9.41494258672332, 1.5369847522163147, 9.585913985879717],
                                             [9.354638363247048, 1.1590819107669206, 9.903025340088558],
                                             [9.543468306658312, 0.7592280107747147, 10.098468132039713],
                                             [9.782660933771094, 0.3604210117818121, 10.28215224353641],
                                             [10.087244845377752, 0.004133734298432223, 10.426151517669405],
                                             [10.172594029350636, -0.40581091727415064, 10.368459475253747],
                                             [10.40462471969611, -0.7706152600858815, 10.494562034690988],
                                             [10.539439692399952, -1.1521886686003677, 10.384005837178577]]
                                            )

test_layers_ouliers = [-1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 1, 1,
                       1, 1, 1, 1, 1, 1, 1, 1, 1, 1, -1, -1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, -1, 1]


test_layered_repre_tunnel_points_data1 = np.array(
    [[43.342725119880804, 43.41308860474493, 31.56822444037365, 1.5497795059693167, 1.7243465455890299, 0.0],
     [43.29979450447869, 43.04421115481897, 31.417055416976716, 1.1728624014868785, 1.6433032459183152, 1.0],
     [43.03014180641928, 42.674214620367096, 31.216182344736293, 0.6831119214575987, 1.5678964718892254, 2.0],
     [42.77942107905536, 42.25931891873157, 31.11217916674646, 0.3099829596402512, 1.4875798780662215, 3.0],
     [42.534942076427, 41.828725984417, 31.042754803643835, 0.4892382401654554, 1.4652781515785198, 4.0],
     [42.1714979790833, 41.58658993539901, 30.8985082213662, 0.8594461138692063, 1.57145841632708, 5.0],
     [41.98030783227736, 41.166314924579716, 30.866923358366112, 1.2964454208137988, 1.6271923751218484, 6.0],
     [41.686546569151034, 40.79683359291796, 30.93935169284507, 1.771856573822721, 1.5886683145330902, 7.0],
     [41.3701640178157, 40.705380636534734, 30.563516769785274, 2.0602756510086833, 1.3586026886483735, 8.0],
     [41.05463920634171, 40.61593150353198, 30.186100625523597, 2.4170566232431034, 1.2107422790283227, 9.0],
     [40.73911439486771, 40.52648237052924, 29.808684481261913, 2.8177277515380066, 1.1615479993916453, 10.0],
     [40.4235895833937, 40.43703323752649, 29.431268337000233, 3.2460771160912887, 1.2170518345634804, 11.0],
     [40.0973736257855, 40.35655816267835, 29.132798417078373, 3.6589012453487375, 1.3197692345267862, 12.0],
     [39.69127753482394, 40.3431337999908, 29.42418433196132, 3.8592056303333604, 1.129734415653391, 13.0],
     [39.28662976096114, 40.118544732309914, 29.584804222180942, 4.244829986278678, 0.9539958794909384, 14.0],
     [38.87336221617702, 39.87923356883839, 29.723150753583933, 4.669755627112658, 0.8617764955424281, 15.0],
     [38.41021483544881, 39.74889578391602, 29.84613413608668, 5.100156734959158, 0.8234615171391237, 16.0],
     [38.04214127280059, 39.608464935296766, 30.075214130936953, 5.448122943010524, 0.7993971561814845, 17.0],
     [37.6770779243212, 39.42747712959114, 30.341695637496702, 5.824246538161021, 0.7879175436896337, 18.0],
     [37.282097334618925, 39.34186089047744, 30.59393525610782, 6.198413021130953, 0.8418889308210262, 19.0],
     [36.893887052774254, 39.322976802104854, 30.908476458935716, 6.550980647302803, 0.9262392352776538, 20.0],
     [36.497534755695646, 39.26062965936305, 31.194045472860395, 6.945614434656645, 1.110269276253853, 21.0],
     [36.20728620229894, 39.08256879684687, 31.468122091902732, 7.302919781943532, 1.235313010499895, 22.0],
     [36.278389511551964, 38.8227355032019, 31.883854675721395, 7.402376702274705, 1.060931504828768, 23.0],
     [36.31491806001538, 38.4533183667643, 32.21881235133638, 7.600378568133862, 0.9839746066841173, 24.0],
     [36.3514466084788, 38.083901230326695, 32.553770026951376, 7.825364395062872, 1.0198375831926947, 25.0],
     [36.387975156942225, 37.7144840938891, 32.88872770256637, 8.075079036894122, 1.1633900881717667, 26.0],
     [36.33868771148881, 37.5537578052315, 33.320823561663396, 8.323540831984134, 1.2248190972450803, 27.0],
     [36.62853437065723, 37.21845878625265, 33.55226141950157, 8.369810527508912, 1.4180827975367316, 28.0],
     [36.703534845767734, 37.16110988373517, 33.95968616122282, 8.491483526643377, 1.409167644694542, 29.0],
     [36.680086516042266, 36.8368514888939, 34.32010208262649, 8.84013072457245, 1.3300931511814802, 30.0],
     [36.592179603266686, 36.36472904721894, 34.440446366116, 9.238751376065975, 1.2802626408544298, 31.0],
     [36.3783858736921, 36.06071840352514, 34.74177288057273, 9.69297688869275, 1.0969379493909657, 32.0],
     [36.07186169061939, 35.87303415114786, 35.07878980471991, 10.15238613876813, 0.9575963753300176, 33.0],
     [35.71786171511895, 35.706568547432276, 35.38574595836637, 10.620353303775289, 0.8794614174509918, 34.0],
     [35.45962916266057, 35.65964069021049, 35.77191710927496, 10.990949803235878, 0.9087906289458936, 35.0],
     [35.16458697242823, 35.56850783240689, 36.16312217732604, 11.421132795031655, 1.0085782433159844, 36.0],
     [34.83935488013632, 35.330721548266695, 36.42299235791807, 11.897711785553415, 1.1144163565286938, 37.0],
     [34.53423605949165, 35.16703577658176, 36.77550328641762, 12.363156540844283, 1.3255254033083756, 38.0],
     [34.29138405134891, 35.118749768430035, 37.183798395350955, 12.752684050217812, 1.587918300970543, 39.0],
     [33.84664743301765, 35.04299489456489, 37.399368660927955, 13.200595471210173, 1.7499405483890602, 40.0],
     [33.401910814686374, 34.96724002069975, 37.614938926504955, 13.652124051018191, 1.983050427683082, 41.0],
     [33.464837906480696, 34.693499273381654, 37.99804336390399, 13.949338037383708, 1.7607168971548688, 42.0],
     [33.56578203105891, 34.40493210595183, 38.39369380759724, 14.246279617213785, 1.566420424355356, 43.0],
     [33.676045764886574, 34.11787138120542, 38.78494471558066, 14.545258161073027, 1.4451621892525146, 44.0],
     [33.989125054708865, 33.86359422090173, 39.080451879677014, 14.660104652831604, 1.2280468057357992, 45.0],
     [34.302204344531134, 33.609317060598045, 39.37595904377337, 14.790970282293463, 1.1008562697909905, 46.0],
     [34.602691631107284, 33.41303230618481, 39.72163409034843, 14.940023274352308, 1.053013480311675, 47.0],
     [34.88570558950797, 33.27393080430698, 40.10666807841938, 15.10353342288734, 1.091749820017222, 48.0],
     [35.28898365440771, 33.199773562249646, 40.367245803891564, 15.104372297687567, 1.078414652850477, 49.0],
     [35.73842298507455, 33.13513386835505, 40.576592401721456, 15.060251165739635, 1.1233333568317405, 50.0],
     [35.981811010670306, 32.893614794957564, 40.80837593210792, 15.246125219792912, 1.203925264120593, 51.0],
     [36.00676933979842, 32.46459043416155, 41.06394427844743, 15.666202284097455, 1.3128752342367138, 52.0],
     [36.122201389187914, 32.46029787299938, 41.475759222339185, 15.892551870263418, 1.2051480176124636, 53.0],
     [36.25572651166032, 32.440344786932215, 41.95121572412747, 16.17294079761577, 1.124984305835204, 54.0],
     [36.172349536400425, 32.4328493072749, 42.42540280209893, 16.54068730518139, 0.980141296410822, 55.0],
     [36.29244561392302, 32.20168538924314, 42.743978984844894, 16.854630295863753, 0.8588834202858171, 56.0],
     [36.56877061231995, 31.86438468550687, 42.96325934049869, 17.10814833477711, 0.8454566389490445, 57.0],
     [36.6927958292071, 31.495151367891715, 43.27676293336327, 17.510859266902, 0.8288897362462995, 58.0],
     [36.81682104609424, 31.125918050276557, 43.59026652622784, 17.918472174986302, 0.9328136960059228, 59.0],
     [36.98804757964962, 30.800418266716658, 43.91002132873909, 18.2928906546974, 1.10878220667746, 60.0],
     [37.04320752335535, 30.33571033045933, 43.7339420437853, 18.44457174164079, 1.0908583694985798, 61.0],
     [37.13315434929198, 29.895483219506975, 43.53398271743937, 18.565284614900552, 1.1330295123516978, 62.0],
     [37.17212558398819, 29.51359904035209, 43.21359901532977, 18.59324867392127, 1.1964237657241938, 63.0],
     [37.12369216824051, 29.09967859456566, 43.03991370915796, 18.77853666813172, 1.2614592820685722, 64.0],
     [36.93699163167821, 28.639939891434693, 43.10142171262745, 19.19709015842925, 1.270097674379276, 65.0],
     [36.797632103062746, 28.167988825824466, 43.140511838679906, 19.599551006534377, 1.3702305461909823, 66.0],
     [36.828215749161984, 27.700223623872493, 43.10167079819146, 19.90342564557149, 1.5182789518300794, 67.0],
     [37.25696336065339, 27.70827768661922, 43.01311809612267, 19.71844268023249, 1.3878769191506743, 68.0],
     [37.743387232312266, 27.716060615804658, 43.09406324113192, 19.632319405334293, 1.164128976373754, 69.0],
     [38.22200202659206, 27.835978593393364, 43.1749802192487, 19.477343267792598, 0.9824434245715847, 70.0],
     [38.66294410935609, 28.01707123962292, 43.250327754107026, 19.293533181478644, 0.8834342093438063, 71.0],
     [39.040223219402534, 28.007335664045122, 43.31843842119013, 19.26769072785701, 0.8926879364819176, 72.0],
     [39.533234406429735, 27.928599457854965, 43.29122483449502, 19.218433104837455, 0.9143013844199868, 73.0],
     [39.83677286533097, 27.663095840632163, 43.12212188493926, 19.259002064498013, 0.9266292149818847, 74.0],
     [39.953969811003155, 27.262149576025287, 42.847336570879534, 19.373742828189602, 0.9485331231787029, 75.0],
     [40.117540379044804, 26.819729810282865, 42.78098722365924, 19.65305341116658, 0.9229469437462867, 76.0],
     [40.260726015578705, 26.365365273163988, 42.71194930261769, 19.95026786994265, 0.8508897516480853, 77.0],
     [40.344231099505556, 25.878324903687783, 42.635669688966, 20.283698796032457, 0.8342012161204113, 78.0],
     [40.13650216519417, 25.547839750246467, 42.87030074156331, 20.711665703951198, 0.8170173981026372, 79.0],
     [40.08470156211611, 25.168906152525164, 43.10273668386207, 21.15789896171002, 0.7864622140441113, 80.0],
     [40.14428212815837, 24.725167026902188, 43.30558577187666, 21.62538018843983, 0.7644538205184477, 81.0],
     [40.400286095999526, 24.396600665874526, 43.50867814957327, 21.978640212715696, 0.874887686464793, 82.0],
     [40.551476502018566, 23.96652036854425, 43.681484802999066, 22.411125350805726, 1.0361725982732184, 83.0],
     [40.726245256032755, 23.681016159800308, 43.99175820431761, 22.805020749200132, 1.0419077063431958, 84.0],
     [41.05241680380601, 23.604912584120644, 44.3450935147209, 23.045357307437847, 0.9628393895047165, 85.0],
     [41.395951081660364, 23.59741485134881, 44.7083124348536, 23.24335970914712, 0.9622661020250527, 86.0],
     [41.76717111652275, 23.48955381252564, 44.95000465424481, 23.455321828759846, 1.0007853031011071, 87.0],
     [42.17735945188072, 23.204072499441537, 44.965687428745426, 23.678629013893456, 1.0160755600914515, 88.0],
     [42.5875477872387, 22.91859118635744, 44.981370203246044, 23.91030785800334, 1.140625092944932, -89.0]]
)

test_layered_repre_tunnel_points_data2 = np.array(
    [[45.054610009709506, 41.451556604363155, 30.18739112223223, 2.4599490444058967, 2.3466575020955247, 0.0],
     [45.47235490646152, 41.27255372022013, 29.978951277897856, 2.9521213803857376, 2.438975063090094, 1.0],
     [45.776574076198344, 41.15905641363226, 29.748847762479613, 3.3309733193986046, 2.542608921566803, 2.0],
     [45.491425500156836, 41.38563069781623, 29.406277502054067, 3.147889122930239, 2.4337209169613967, 3.0],
     [45.206276924115336, 41.612204982000215, 29.063707241628528, 3.0369439583205438, 2.3917591602074504, 4.0],
     [44.92112834807385, 41.83877926618419, 28.721136981202985, 3.006135606005965, 2.419052365952375, 5.0],
     [44.672343411958806, 42.03060898724461, 28.353908598376734, 3.1009703176875103, 2.5061933838274086, 6.0],
     [44.40527444678243, 42.00614032777669, 28.114379324858007, 3.150370552632932, 2.4212512430002917, 7.0],
     [43.91571332572381, 41.943779147968314, 28.194635115930843, 2.860204476265201, 2.1675801494882636, 8.0],
     [43.43052855967059, 41.83421855256556, 28.227636615387198, 2.68601016994715, 1.941362658609228, 9.0],
     [42.946812169644026, 41.708821378492935, 28.244783123902472, 2.6156814422542207, 1.7713231395242304, 10.0],
     [42.46309577961745, 41.58342420442031, 28.261929632417747, 2.6398727586091573, 1.6791253960256431, 11.0],
     [41.992464784373716, 41.46027030001957, 28.308994610681555, 2.7241657324964264, 1.6618805542238075, 12.0],
     [41.67578522757763, 41.242078843920225, 28.5677838924032, 2.6753811074206437, 1.5281222559126435, 13.0],
     [41.38544788610194, 40.85149258243526, 28.682442518262874, 2.8758344212381703, 1.3343762594607333, 14.0],
     [41.09511054462626, 40.4609063209503, 28.797101144122554, 3.1437530384578984, 1.2297067736314278, 15.0],
     [40.73750992944964, 40.12673146984305, 28.895175374126467, 3.4779532598955143, 1.178467099054685, 16.0],
     [40.37291120310239, 39.79842569556473, 28.991524153680086, 3.84961831077494, 1.2263433434843052, 17.0],
     [40.008312476755144, 39.470119921286404, 29.0878729332337, 4.24770095455054, 1.3723876651188507, 18.0],
     [39.67473142515541, 39.367708610642154, 29.424859585881553, 4.418757262747705, 1.2637641737906902, 19.0],
     [39.40093583940794, 39.411501880910805, 29.782272695055156, 4.494100369626704, 1.1071411299808231, 20.0],
     [38.99357584468922, 39.29553133559201, 30.02541656768483, 4.830765246588952, 1.0096459240360771, 21.0],
     [38.537956187582715, 39.169109655769034, 30.15731418008555, 5.249476997479309, 0.9509636034344029, 22.0],
     [38.074246306071544, 39.060868230038594, 30.30229669346273, 5.6752686131193775, 0.9341267254470542, 23.0],
     [37.69962760081821, 38.78326001470537, 30.482836175166387, 6.126787453982377, 0.8474112672128784, 24.0],
     [37.49783769215179, 38.965895992431115, 30.88583636303426, 6.1904586880289365, 0.9399549906900166, 25.0],
     [37.1662023858723, 39.01113358241043, 31.242944125560133, 6.4677824943731554, 0.9806239063107423, 26.0],
     [36.80781725039421, 38.92142589215629, 31.565257135390095, 6.851876056369666, 1.057265451961589, 27.0],
     [36.46899681499489, 38.70645725439637, 31.827356023196085, 7.281861725126688, 1.12069110198049, 28.0],
     [36.39655013495339, 38.2372919865614, 31.829904358529454, 7.5782106881231455, 0.9595190430259148, 29.0],
     [36.3408636349085, 37.74204071370227, 31.78959070768802, 7.887494583007952, 0.8833251787041323, 30.0],
     [36.299584105237116, 37.264102101640944, 31.794539904663985, 8.199085679786767, 0.9006847052032885, 31.0],
     [36.38253840523006, 36.95947098969504, 32.182249163964414, 8.375746001554747, 0.8222416526039857, 32.0],
     [36.46549270522299, 36.65483987774914, 32.56995842326483, 8.577950581615415, 0.8654409631790434, 33.0],
     [36.602423140588776, 36.33579602267443, 32.925690249801356, 8.767589714621616, 0.9758474238523889, 34.0],
     [36.866202264579236, 35.99885546571184, 33.181567794507984, 8.881066649846804, 1.0690674892811545, 35.0],
     [36.73875187463952, 35.558343219959305, 33.2619369712342, 9.296480970558056, 1.1138331057553839, 36.0],
     [36.66246390486659, 35.11887196658585, 33.25886898960587, 9.66177853500515, 1.1568965383598793, 37.0],
     [36.50071217957223, 34.70379289797471, 33.25998889471215, 10.069564593498379, 1.2963699342991306, 38.0],
     [36.774208281197794, 34.327210816075905, 33.08923082846628, 10.148686416598858, 1.2428226496468435, 39.0],
     [37.061180331979735, 33.957161115955905, 32.91398739346482, 10.23749044978713, 1.280918048093304, 40.0],
     [37.125593154989446, 33.63580974332004, 32.56773840876363, 10.398287161985973, 1.2563839087248294, 41.0],
     [37.32262730383059, 33.35483924417785, 32.20409968310005, 10.47218711401998, 1.2868922787628547, 42.0],
     [37.5354690904755, 33.08293027956132, 31.842745282151913, 10.553789202821898, 1.4092451551242762, 43.0],
     [37.5722989647206, 32.624204569037026, 31.766473077873204, 10.927045662194958, 1.3752904984856376, 44.0],
     [37.38556029075441, 32.19105356793496, 31.838408589286516, 11.401316965527593, 1.189056088721345, 45.0],
     [37.34747695297086, 31.799405985645002, 32.07471076337694, 11.786413202920114, 1.1664264654930434, 46.0],
     [37.24014763719406, 31.339776540085623, 32.2396992197903, 12.259467724717503, 1.1739026222689313, 47.0],
     [37.107461232051875, 30.876084228319478, 32.36641129880062, 12.744744709162086, 1.259482359846163, 48.0],
     [36.894548242206945, 30.439475585823814, 32.34250363985025, 13.224192433776986, 1.3402645986203037, 49.0],
     [36.602036242381516, 30.041967152871575, 32.283521099914, 13.701342346020455, 1.4246980306300834, 50.0],
     [36.30829459323361, 30.070041106482698, 32.54082892309177, 13.841609900114554, 1.2692820699120404, 51.0],
     [36.022504528112236, 30.239325151026215, 32.913361903639846, 13.882775939262437, 1.0732210115568048, 52.0],
     [35.73052492180056, 30.364942995314657, 33.29932571890548, 13.984087334657326, 0.9644272972596288, 53.0],
     [35.43854531548889, 30.490560839603102, 33.68528953417112, 14.10240863777826, 0.9683997055113898, 54.0],
     [35.098487504350146, 30.635983179612996, 33.98819290498135, 14.228368566564091, 1.0037451875112053, 55.0],
     [34.69161912994301, 30.90154180288844, 34.06914062992519, 14.259423862648827, 0.8913579538659173, 56.0],
     [34.33203395491288, 31.079323975530546, 34.354481523965305, 14.396028788594299, 0.9005098679039045, 57.0],
     [33.90400126569155, 31.24472577046169, 34.54701944188849, 14.574819634225914, 0.91950067811493, 58.0],
     [33.58233755079716, 31.414785241119368, 34.710297231751326, 14.690173661815505, 0.9639128739176479, 59.0],
     [33.184031598419715, 31.560103361713043, 34.97531612188526, 14.910497620205234, 1.0886954224590124, 60.0],
     [32.85098845529381, 31.809637572234838, 35.19234342424671, 15.014946942729104, 1.2630491962280368, 61.0],
     [32.78537304211993, 32.131004264673024, 35.54844102464504, 14.94797373329668, 1.173439738865828, 62.0],
     [32.83109119618175, 32.523416901657555, 35.85490827040553, 14.757860368598083, 1.0727164634144601, 63.0],
     [32.87680935024359, 32.915829538642086, 36.16137551616603, 14.582419778169482, 1.0797508750130487, 64.0],
     [32.922527504305414, 33.30824217562662, 36.46784276192653, 14.422187438612044, 1.1935931110300257, 65.0],
     [32.96824565836724, 33.700654812611134, 36.774310007687035, 14.277675386496565, 1.4002522781872615, 66.0],
     [33.01764900148973, 34.040775108245484, 37.12265722760614, 14.195334646019067, 1.6510694025329002, 67.0],
     [33.18233221999665, 34.371305018758655, 37.256493290793564, 13.955394657350334, 1.8171889329327497, 68.0],
     [33.39844052635335, 34.77213076631024, 37.15798606762399, 13.53846993782011, 1.9288362977568407, 69.0],
     [33.292673617357394, 34.96902094108161, 37.56549002315962, 13.702094756710945, 2.0885111562299166, 70.0],
     [33.104433407475405, 34.96050575438267, 37.88026881818594, 13.993962303755255, 2.118523069708112, 71.0],
     [33.062166293540315, 34.75161388605642, 38.309716870834535, 14.35147214503733, 1.924196148775107, 72.0],
     [33.05614739875996, 34.58711664632474, 38.77032951356826, 14.685718278675724, 1.7041176379920573, 73.0],
     [33.157773775115146, 34.33185020570354, 39.18807479950395, 14.982174327281756, 1.4914413561359112, 74.0],
     [33.25940015147033, 34.07658376508233, 39.605820085439646, 15.289237187822389, 1.3600417753897875, 75.0],
     [33.46582949155638, 33.82954791457902, 39.916292142505235, 15.475423804699853, 1.3282115551883689, 76.0],
     [33.67362546375945, 33.5506576706291, 40.239672106416776, 15.696613237613716, 1.3189267869089478, 77.0],
     [33.9684065105434, 33.187765076793575, 40.4169095172427, 15.838639623738969, 1.3323593579002806, 78.0],
     [34.263187557327356, 32.82487248295804, 40.594146928068625, 15.99504120438951, 1.4421006183116536, 79.0],
     [34.65679681825527, 32.69201867790443, 40.86995725088552, 16.039286285144268, 1.4422147217629921, 80.0],
     [35.06501636405833, 32.618247971848696, 41.13258512243042, 16.048117364409297, 1.4755631545540926, 81.0],
     [35.466878396268385, 32.71800156411972, 41.41234918524296, 15.982815383051175, 1.4680105421952423, 82.0],
     [35.634390571007174, 32.53820476839416, 41.19917941179335, 15.874066226621661, 1.479411486918676, 83.0],
     [35.81016041617889, 32.099611643486604, 41.07790873279999, 15.988921668259382, 1.299816706176389, 84.0],
     [35.99665000228916, 31.646889588890573, 40.9765954782142, 16.13499611490412, 1.1908060737895807, 85.0],
     [36.0847134682603, 31.2073362225742, 40.86541522744821, 16.32032824916723, 1.1423317257098942, 86.0],
     [36.36500918261674, 30.924275322879407, 41.1484553757127, 16.575117425624978, 1.0250471894437083, 87.0],
     [36.56760451360249, 30.652963691378126, 41.51634820042513, 16.914270619661583, 0.9007855257969866, 88.0],
     [36.77019984458824, 30.38165205987685, 41.884241025137555, 17.261245132290924, 0.8918328459857598, 89.0],
     [36.97279517557399, 30.110340428375565, 42.252133849849976, 17.61557879942999, 0.9995975735783269, 90.0],
     [37.17539050655974, 29.839028796874285, 42.6200266745624, 17.976836466987667, 1.2084630609246318, 91.0],
     [37.264549973255555, 29.474309418133757, 42.91996789900921, 18.397819622716998, 1.3683084846322677, 92.0],
     [37.32435979560474, 29.030114088903773, 43.14158309397749, 18.8340341606199, 1.5105818212905835, 93.0],
     [37.50920789833025, 28.591215145081744, 43.13186097861151, 19.08478139376967, 1.5626597435109584, 94.0],
     [37.684126881178166, 28.167570723279496, 43.178485406819775, 19.37148982737972, 1.593962734174595, 95.0],
     [37.79677652760515, 27.776488392735537, 43.40800699328668, 19.772643407824628, 1.519402781088106, 96.0],
     [37.68543962489498, 27.392132647906635, 43.70384694259507, 20.26917533006927, 1.466935360724568, 97.0],
     [37.36963843813583, 27.148183011524345, 43.96918767093971, 20.6965582489587, 1.4540991255007267, 98.0],
     [37.27306489503483, 26.878452526451177, 44.23841313115275, 21.08887669401466, 1.4924417937809056, 99.0],
     [37.301026854426084, 26.6522580863413, 44.597867714730384, 21.47524718473735, 1.4804359317738263, 100.0],
     [37.10043511055197, 26.55336919651993, 45.04506335397504, 21.886926337429195, 1.4403219736970805, 101.0],
     [36.89984336667785, 26.45448030669856, 45.4922589932197, 22.30221619827353, 1.4940422166757277, 102.0],
     [36.58199870493549, 26.28096042551593, 45.81538073242104, 22.722480918712048, 1.4322126461245535, 103.0],
     [36.28600173741556, 26.093586102304528, 46.171648478808265, 23.170723110065843, 1.3605474923061738, 104.0],
     [35.9994628369687, 25.901511175125144, 46.53359140370077, 23.625322236303138, 1.3777629028315654, 105.0],
     [35.673876436221164, 25.88383109904572, 46.894954427474865, 23.97396779647451, 1.4178592794715954, 106.0],
     [35.20270991744454, 25.75375432583615, 47.000226193182954, 24.27614175898415, 1.2926909260185433, 107.0],
     [34.7315433986679, 25.623677552626575, 47.10549795889105, 24.58477141439781, 1.2645139273688064, 108.0],
     [34.267686422806825, 25.488580976540664, 47.215748606887374, 24.9037571401273, 1.331152571834682, 109.0],
     [34.02744571129364, 25.055068072162424, 47.24223051157874, 25.295786649379487, 1.1130765161043603, 110.0],
     [33.784039287263084, 24.61860726893775, 47.25820799539606, 25.687821011974222, 0.9837209525360746, 111.0],
     [33.548327495609115, 24.20928018020612, 47.31610428838567, 26.088602062731336, 0.9276467204090342, 112.0],
     [33.39179443618157, 23.74050446678544, 47.39191148997746, 26.516162219013296, 0.9243530146017964, 113.0],
     [33.32581465629319, 23.26580429349607, 47.391217835006124, 26.871140880630172, 1.049971811729043, 114.0],
     [33.3389879712327, 22.93085754745311, 47.722490473438995, 27.30721214217521, 0.9190780221136057, 115.0],
     [33.33797945152649, 22.652326843229993, 48.137724900832715, 27.761797055839047, 0.8152553889884333, 116.0],
     [33.3369709318203, 22.37379613900689, 48.55295932822642, 28.21791835981927, 0.8327363089906503, 117.0],
     [33.335962412114085, 22.09526543478378, 48.968193755620135, 28.675502739328866, 0.9685892928194042, 118.0],
     [33.24334681497826, 21.795641236208105, 49.31978383606257, 29.138604022799036, 1.1423504076118227, 119.0],
     [32.96930209831794, 21.473208801141777, 49.56226012665623, 29.608221588130863, 1.239253963209442, 120.0],
     [32.663144946296015, 21.436216362649827, 49.860263360405284, 29.925740935557354, 1.1750354915618237, -121.0]]
)

test_layered_repre_tunnel_points_data3 = np.array(
    [[42.97003064225104, 41.8133870948386, 30.60492538518701, 0.4571926582314952, 2.064006692600957, 0.0],
     [42.80228753922399, 41.38059039327849, 30.691525425362386, 0.8067603278723796, 1.9080106777581893, 1.0],
     [42.54802929719248, 41.02027147054603, 30.92715865790814, 1.1880777769665165, 1.7206478404218488, 2.0],
     [42.29377105516096, 40.65995254781357, 31.162791890453896, 1.6346851016677955, 1.6108072225719525, 3.0],
     [42.01839770996671, 40.300567175269116, 31.37167629838116, 2.104963178713993, 1.5637576381832863, 4.0],
     [41.68255414189666, 39.98705942998305, 31.439104973585525, 2.5331309530501103, 1.4954753075150282, 5.0],
     [41.28144922231067, 39.8389533520014, 31.17991501786036, 2.8069684946829514, 1.2704905828330426, 6.0],
     [40.88034430272468, 39.690847274019745, 30.920725062135197, 3.13709927082086, 1.1326080481534564, 7.0],
     [40.50577123033512, 39.615551742162744, 30.616779408402383, 3.4378727026395755, 1.0412735319430977, 8.0],
     [40.16554953078789, 39.563243123933994, 30.25419329492621, 3.746092668309998, 1.052021552315139, 9.0],
     [39.817386592584356, 39.52231930980071, 29.89767147245491, 4.089347214148757, 1.1574324643886602, 10.0],
     [39.61676041840182, 39.19599598414278, 29.612845001431054, 4.516398100203399, 1.1862606417715036, 11.0],
     [39.15207584511943, 39.03228756112646, 29.692497037306126, 4.937657445596529, 1.0359768319599925, 12.0],
     [38.702768530239176, 38.909611314940975, 29.733443390883934, 5.342985599738014, 0.9484648191281011, 13.0],
     [38.37116644074896, 38.99350653202313, 30.08498944987499, 5.494602136054132, 0.922120381542155, 14.0],
     [38.0109887431827, 38.91847381338959, 30.387107188355795, 5.799804092691764, 0.8877966821045862, 15.0],
     [37.601410865531065, 38.73158971251909, 30.597091063600192, 6.232012925317738, 0.8367663194096779, 16.0],
     [37.28382732074529, 38.76219917276448, 30.89149317470377, 6.479455700023994, 0.9176963476624624, 17.0],
     [36.929533870129944, 38.87930914987878, 31.155125712851454, 6.733079918095142, 1.0173134409697606, 18.0],
     [36.472902109095585, 38.80138430102159, 31.320521835768186, 7.180177943805988, 1.1573893512609155, 19.0],
     [36.17610664474224, 38.69750331718298, 31.697651953340102, 7.5252611336652535, 1.201098250418312, 20.0],
     [35.877584016385455, 38.400678340724546, 31.96104097215659, 7.959685407560438, 1.247311269812356, 21.0],
     [35.71180475505397, 38.09434797193386, 32.29256150592674, 8.303897731564026, 1.265853827754629, 22.0],
     [35.72230722503535, 37.86223788000066, 32.72901128780133, 8.499269169409414, 1.250463852475699, 23.0],
     [35.832559426943384, 37.66461354402826, 33.174156216416485, 8.62206924476898, 1.2542366469807014, 24.0],
     [35.928022515151326, 37.43102230651061, 33.60189730795194, 8.796978916429403, 1.3558915710607249, 25.0],
     [36.000634743007154, 37.10654432942886, 33.97254395933837, 9.043571470180954, 1.5243499361057269, 26.0],
     [36.05329984589817, 36.872453598371806, 34.32454545239349, 9.265590495259232, 1.5801706391451644, 27.0],
     [35.90032476063657, 36.40511257049214, 34.410310699054634, 9.681038342360807, 1.5152889321606928, 28.0],
     [35.83500047127673, 35.94086337576708, 34.54476328425248, 10.059215160244015, 1.4482283777247622, 29.0],
     [35.57039488170798, 35.77811098434152, 34.872333916609634, 10.466460159437897, 1.3278262478432106, 30.0],
     [35.25805226110717, 35.73139023285175, 35.254720596032385, 10.862990987993543, 1.2486246878159222, 31.0],
     [34.989620423216046, 35.658928587018714, 35.670284860965694, 11.266033900569022, 1.2227361292083885, 32.0],
     [34.73388849591744, 35.64784506110035, 36.09879117777758, 11.639113360467361, 1.2652325629301473, 33.0],
     [34.46991756471726, 35.62109705631366, 36.522176974003315, 12.033136757209096, 1.3813971879746523, 34.0],
     [34.20672889542476, 35.583293883969205, 36.9452207232138, 12.439902332134249, 1.5743904029643765, 35.0],
     [34.15408043361781, 35.4472259665912, 37.4070623324159, 12.78077306642717, 1.6677726783179712, 36.0],
     [34.16158897794524, 35.26781625562221, 37.87370542662399, 13.116309489600717, 1.7807161864123586, 37.0],
     [33.952831339001825, 34.98692316037125, 38.21574989392576, 13.586437400760012, 1.7435275229028637, 38.0],
     [33.70366527373508, 34.659436792623914, 38.491607175475295, 14.072463439011884, 1.7357919375405795, 39.0],
     [33.557789667417204, 34.299703324814075, 38.755613018213374, 14.504102498900842, 1.705607127794882, 40.0],
     [33.61884526830715, 33.97564971343542, 39.13058971567735, 14.851043599461462, 1.5622641430798365, 41.0],
     [33.67234435472905, 33.66020755638602, 39.51482153802948, 15.211463742725323, 1.4964800739242021, 42.0],
     [33.82963518218205, 33.32148037098024, 39.80552507298056, 15.478185107138122, 1.3834812125511398, 43.0],
     [34.018241237783286, 32.928041631168426, 40.04972522409811, 15.741836113206137, 1.2911198850830015, 44.0],
     [34.155658465127246, 32.52092931664154, 40.298321151475065, 16.05506570367182, 1.2565900215501788, 45.0],
     [34.23262148141843, 32.10049598863611, 40.556937406829526, 16.42291394152713, 1.2714565252015657, 46.0],
     [34.34052589324368, 31.63996289102327, 40.45229715641537, 16.593465101738772, 1.1189326626250133, 47.0],
     [34.44813089177542, 31.172740854638104, 40.31043548785378, 16.759690467322457, 1.0416831402431865, 48.0],
     [34.55202700700686, 30.70632158080017, 40.17129716126715, 16.941812102129283, 1.0722124088790521, 49.0],
     [34.26505953206299, 30.32450207078927, 40.319160375419216, 17.422120426626854, 0.9814282200382844, 50.0],
     [33.97809205711912, 29.942682560778373, 40.46702358957128, 17.903507003264004, 1.0032496289106396, 51.0],
     [33.87333113437314, 29.53027300642157, 40.67355461249649, 18.348977063420733, 1.0179587162518595, 52.0],
     [33.901979751170295, 29.096639142435627, 40.91922590985196, 18.76798075922206, 1.0556746991624595, 53.0],
     [34.04211237150324, 28.66792005344822, 41.135010041466025, 19.1199953327922, 1.1032866652769087, 54.0],
     [34.16800616441514, 28.32957477281453, 41.44769496853599, 19.472981314345958, 1.0477884779757056, 55.0],
     [34.463530571494246, 28.062897373604546, 41.75026477795383, 19.702104428398535, 0.8767206337854307, 56.0],
     [34.759054978573346, 27.796219974394567, 42.05283458737165, 19.94113428434239, 0.818163556270896, 57.0],
     [35.10241402406387, 27.619702815395193, 42.3422544104724, 20.098517287241926, 0.8344289313526809, 58.0],
     [35.53380906783812, 27.63709543764239, 42.59443651312517, 20.071789013075133, 0.8723413265720172, 59.0],
     [35.96520411161239, 27.654488059889598, 42.84661861577794, 20.057493144090675, 1.0249260821231212, 60.0],
     [36.39659915538663, 27.671880682136795, 43.09880071843071, 20.055656266128157, 1.2710308715200587, 61.0],
     [36.78421303107518, 27.54425646185479, 43.24920123033144, 20.12115664744247, 1.4874593945886023, 62.0],
     [37.0285168898954, 27.12329955591092, 43.1535789829217, 20.30061400820596, 1.5344298916694556, 63.0],
     [37.449876582188786, 27.21620564226261, 43.13086946062337, 20.101968093743583, 1.3904336775581905, 64.0],
     [37.916774898491454, 27.09440924073334, 43.26190990784581, 20.155050631358066, 1.2144303076505738, 65.0],
     [38.39539924845504, 27.01039942478326, 43.293253579891086, 20.12743790259647, 1.093180131532751, 66.0],
     [38.887149938501246, 26.96868861523341, 43.21299482830736, 20.007610109773285, 1.03053137113555, 67.0],
     [39.37902134353764, 26.928277032352316, 43.134960748190956, 19.899988343592693, 1.0771224111644584, 68.0],
     [39.669429220371846, 26.611637757697913, 43.14159494225012, 20.100604399516012, 1.0047917683282337, 69.0],
     [39.71476975303887, 26.137233496527404, 43.22529039899261, 20.51386557873447, 0.8459509694177576, 70.0],
     [39.687784548799925, 25.64798962305537, 43.324850586930665, 20.961913466930913, 0.7613731586407904, 71.0],
     [39.66451591910105, 25.161682017799066, 43.43298467728509, 21.414436752524498, 0.7942253015163505, 72.0],
     [39.807377854350065, 24.694131125763874, 43.51719670227915, 21.816710981584045, 0.8144117893585803, 73.0],
     [39.95215531157706, 24.257400279640258, 43.70141889345406, 22.255410934204164, 0.824434880079951, 74.0],
     [40.23467377108303, 23.92012226854623, 43.828036557701495, 22.566835814186163, 0.9452421254182577, 75.0],
     [40.441216098447356, 23.55032822439753, 44.0854080478947, 22.99250072359663, 1.1265879268741414, 76.0],
     [40.8375403829497, 23.535330791841677, 44.28349135319411, 23.082617132320895, 0.9996183996718202, 77.0],
     [41.22581429963132, 23.584493020482295, 44.594660826150225, 23.19680333298486, 0.905945893314603, 78.0],
     [41.61408821631295, 23.633655249122928, 44.90583029910636, 23.3211525457264, 0.9287778434967, 79.0],
     [42.04226418846723, 23.393693316791275, 44.92931163821808, 23.50899825611759, 0.8702076002396539, 80.0],
     [42.472467281080206, 23.139043181186178, 44.93817772769086, 23.708594991761068, 0.920617045344746, 81.0],
     [42.90267037369318, 22.884393045581085, 44.94704381716363, 23.91698045240193, 1.0821642793461004, -82.0]]
)

test_layered_repre_tunnel_points_data4 = np.array(
    [[42.98452500287441, 42.792243161867816, 30.8244276516935, 0.6446273691738699, 2.0864352271659232, 0.0],
     [42.53849393425804, 42.57976493124409, 30.747555762553635, 0.47810168389475405, 2.066317720481117, 1.0],
     [42.311423198018296, 42.1781177525967, 30.69948686417618, 0.4911510730557149, 2.00984823087307, 2.0],
     [42.10639911627694, 41.80195773266039, 30.476423106936625, 0.8491746082469213, 1.8591549843889352, 3.0],
     [41.87751141646452, 41.465701187888925, 30.185662110630485, 1.316218421628182, 1.7380904417139031, 4.0],
     [41.522351837122564, 41.20731090955195, 29.992323984544246, 1.7947397259718376, 1.547058450260226, 5.0],
     [41.088153444740435, 40.99766036089813, 29.859966878569832, 2.2793766789238576, 1.3341626619770535, 6.0],
     [40.653955052358306, 40.788009812244304, 29.727609772595414, 2.7694810714035354, 1.2078269592250808, 7.0],
     [40.21975665997617, 40.578359263590485, 29.595252666621, 3.2625898861645104, 1.1818650116628355, 8.0],
     [39.792741084152695, 40.38498762604161, 29.436217422827934, 3.7536621484116077, 1.2525469545634984, 9.0],
     [39.57829614668066, 40.027736041929316, 29.611350173067695, 4.047003716747327, 1.208721614803175, 10.0],
     [39.17007158464778, 39.91941396429746, 29.878211859798053, 4.367354077735335, 0.9724317056817748, 11.0],
     [38.75968090088774, 39.80610873469722, 30.140397693946664, 4.723965823749716, 0.8324044035497036, 12.0],
     [38.33852609814362, 39.653369617342385, 30.359074004829743, 5.137003784513562, 0.8002604334724863, 13.0],
     [37.91290112040791, 39.48425420582711, 30.55968147207734, 5.576663291889123, 0.8867338602988759, 14.0],
     [37.47501873486536, 39.35694689942165, 30.75256606380344, 6.016885226300226, 1.0477745414686959, 15.0],
     [37.125937744612514, 39.32394820736828, 31.021774947137, 6.3452373793909445, 1.1346454302979911, 16.0],
     [36.74258280512776, 39.19896911574286, 31.31388465520821, 6.759237470208776, 1.2156449276204018, 17.0],
     [36.46148316556969, 39.083784020845734, 31.62796947444639, 7.090728886061714, 1.2160599348596624, 18.0],
     [36.52842404224349, 38.89792619329795, 32.08729048089979, 7.181915700309197, 0.9810843274212766, 19.0],
     [36.59536491891729, 38.71206836575016, 32.54661148735318, 7.3062569156213915, 0.8437495144492015, 20.0],
     [36.642601852367164, 38.4834326334989, 32.98594123300418, 7.493443560231329, 0.8055645037054189, 21.0],
     [36.67114415649455, 38.214210250010595, 33.40630374929694, 7.737364968526712, 0.8729377540468708, 22.0],
     [36.699686460621926, 37.944987866522304, 33.826666265589694, 8.005119419596202, 1.0520774599872442, 23.0],
     [36.72822876474932, 37.675765483034006, 34.247028781882456, 8.294399145677053, 1.3192528335624283, 24.0],
     [36.58592353975651, 37.97834055222074, 34.588596994534555, 8.388789096497064, 1.2472901927123072, 25.0],
     [36.50405617357246, 38.29857177951964, 34.95084210752464, 8.466017280769176, 1.1964225881251174, 26.0],
     [36.38179044624583, 38.54428129580805, 35.27616575044111, 8.61262072090385, 1.1569819463803566, 27.0],
     [36.01469916505297, 38.50277143890783, 35.584036044160484, 9.064348945224413, 1.1024378382763829, 28.0],
     [35.717345856036964, 38.446425328395364, 35.90260111850484, 9.47829966675744, 1.0589426913856463, 29.0],
     [35.444938673323136, 38.41807444408635, 36.32091984168707, 9.920211199865703, 1.0322768824074793, 30.0],
     [35.17943411867026, 38.281565145144285, 36.70805118064232, 10.38462422791781, 1.050371975719502, 31.0],
     [34.928083560440555, 38.01547416219468, 37.04866604477852, 10.863021355426064, 1.097068955622704, 32.0],
     [34.67673300221084, 37.74938317924507, 37.38928090891473, 11.343281958138396, 1.247218327952677, 33.0],
     [34.3256585375317, 37.56792385983305, 37.62040695591955, 11.799273708650606, 1.3645880872202263, 34.0],
     [33.97855695144887, 37.327325360891294, 37.72101528767978, 12.20142895225265, 1.3772192588767955, 35.0],
     [33.81071760295348, 36.872814721541324, 37.84450630684442, 12.577674903170395, 1.4434039282277829, 36.0],
     [33.64287825445808, 36.41830408219135, 37.967997326009055, 12.962289283256093, 1.5987692915306086, 37.0],
     [33.47503890596269, 35.96379344284138, 38.091488345173694, 13.354549073472644, 1.8296261760585855, 38.0],
     [33.26370778803383, 35.613433700524084, 38.294767462886085, 13.776898831454112, 2.020965420992529, 39.0],
     [33.06467646345118, 35.3215907194782, 38.51724630287065, 14.17496411089679, 2.00573925148264, 40.0],
     [32.922470766122785, 34.93899909453317, 38.70730217929583, 14.562723503232135, 1.9439571424236977, 41.0],
     [33.13781243096477, 34.68550578246783, 39.07747885158085, 14.75078410444837, 1.7540635591274112, 42.0],
     [33.386475296203244, 34.388517200232855, 39.38929017780294, 14.921030829028203, 1.530880059357117, 43.0],
     [33.62371869355259, 34.091294598465396, 39.71390424929819, 15.120315520178544, 1.3691267594485355, 44.0],
     [33.835578313167176, 33.833316040422034, 40.082830076745324, 15.353425535774063, 1.264632869394142, 45.0],
     [34.03240642206929, 33.598576626284014, 40.47799599505832, 15.610789584713043, 1.2389581568788965, 46.0],
     [34.389718565222594, 33.48061424102435, 40.73630913148603, 15.641752201998155, 1.1633224350457334, 47.0],
     [34.86213835475246, 33.446410523770226, 40.8964642069178, 15.516174149116736, 1.078863744204982, 48.0],
     [35.32779257889077, 33.39364407762396, 41.06343814023731, 15.424180586558112, 1.102960429490705, 49.0],
     [35.75470440174043, 33.23457983519154, 41.26945964952839, 15.453242088481892, 1.2347964056798781, 50.0],
     [36.086466478698284, 33.0088818841111, 41.48206208252143, 15.582823839062966, 1.4263541600860126, 51.0],
     [36.28548780311473, 33.0118807228332, 41.94051605058154, 15.815611742156241, 1.3513459285743414, 52.0],
     [36.44682603301819, 32.83658996686702, 42.35208342716122, 16.144158862381158, 1.1933332057466541, 53.0],
     [36.66672937061138, 32.50269220871408, 42.65234133166302, 16.470029520509698, 0.9961258031700637, 54.0],
     [36.855881183304604, 32.16185175377501, 42.96342241205711, 16.82723308936728, 0.880892923981534, 55.0],
     [37.009660736849575, 31.813025390146677, 43.28695296231372, 17.217161553846015, 0.8582261178008104, 56.0],
     [37.19557138131136, 31.402143786286132, 43.417510201722564, 17.500764532664153, 0.8522152206863118, 57.0],
     [37.41741558874472, 30.96068501502277, 43.451235975445655, 17.731456276528068, 0.895264092449054, 58.0],
     [37.587496789307366, 30.612366167853164, 43.3967636279679, 17.865274770842653, 0.9754147882509632, 59.0],
     [37.56280260229765, 30.198179618635464, 43.172176716875185, 17.988218805364784, 0.9426957793552997, 60.0],
     [37.53214368553514, 29.72455606521099, 43.151385668160614, 18.30180679948638, 0.9822698000536183, 61.0],
     [37.52338486449776, 29.251537146713986, 43.12575702127019, 18.61240783149447, 1.1006844312876216, 62.0],
     [37.53145486173149, 28.77766045829013, 42.96645931471707, 18.83919121663865, 1.31580162863341, 63.0],
     [37.40858783338727, 28.313500673100137, 42.919259218437844, 19.176351302918693, 1.491536399052215, 64.0],
     [37.363837142340955, 27.89362018058055, 42.91921599284871, 19.4944647807317, 1.61438242037413, 65.0],
     [37.057698275573316, 27.69226424313806, 42.74581415063264, 19.623313261951846, 1.4688050042692933, 66.0],
     [36.68219511518338, 27.625913589330267, 42.430227586375054, 19.59628985839095, 1.2737136905487698, 67.0],
     [36.261038479643524, 27.56715364221322, 42.1672174022461, 19.62162030716593, 1.1139875563103911, 68.0],
     [35.83988184410366, 27.508393695096167, 41.904207218117136, 19.6596386218846, 1.0560623658733956, 69.0],
     [35.4187252085638, 27.44963374797912, 41.641197033988185, 19.710271383408372, 1.107617036395159, 70.0],
     [35.02804714502642, 27.25958794122317, 41.57275125384125, 19.964044933841496, 1.0296967752408228, 71.0],
     [34.68900454025005, 26.901379622985978, 41.52405250737211, 20.339894847226283, 0.8765817815183989, 72.0],
     [34.32102266234028, 26.604492303935054, 41.4104511560012, 20.653004533058564, 0.8630835546908842, 73.0],
     [34.14990320118535, 26.91708439315571, 41.05973144487747, 20.31071399360497, 1.0713036805936127, 74.0],
     [33.79340452892839, 26.990413772169433, 40.754615679005624, 20.25881492794961, 1.073264872141842, 75.0],
     [33.38107138639506, 26.97323688294567, 40.48138828162267, 20.326640723989282, 1.0704264594289323, 76.0],
     [33.17098705313201, 26.952242730631227, 40.10357366129159, 20.26439043490224, 1.2226406023051142, 77.0],
     [33.13393727676328, 26.75993468179921, 39.744511065396765, 20.265979875025778, 1.2934778605820962, 78.0],
     [33.31360547509847, 26.390602650668875, 39.45935638932056, 20.3417211587587, 1.2689529788550147, 79.0],
     [33.50528697560199, 26.019664060192895, 39.18507474735519, 20.429682970524006, 1.335875813881232, 80.0],
     [33.748421054484666, 25.715817881675324, 38.89177893063874, 20.445130489737576, 1.4148344323652944, 81.0],
     [34.0521190803665, 25.523305224715013, 38.54480781392549, 20.334551759965468, 1.4659884909240424, -82.0]]
)

test_layered_repre_tunnel_points_data5 = np.array(
    [[41.76751708984376, 46.30036163330078, 32.51044082641601, 4.582988842929305, 0.0, 0.0, 0.0],
     [42.080410003662124, 47.86328887939454, 33.110633850097656, 6.181986028474331, 0.0, 1.0, 0.0],
     [42.393302917480476, 49.42621994018555, 33.710823059082024, 7.825081438824707, 0.0, 2.0, 0.0],
     [42.806346893310554, 48.898094177246094, 34.01490402221679, 7.452254141811291, 0.0, 3.0, 0.0],
     [42.766838073730476, 49.13829803466797, 34.5927619934082, 7.928265803725821, 0.0, 4.0, 0.0],
     [42.564304351806655, 49.40069580078125, 34.382984161376946, 8.068225876764703, 0.0, 5.0, 0.0],
     [42.72606658935548, 48.811378479003906, 34.13966751098633, 7.429252554555418, 0.0, 6.0, 0.0],
     [42.46198272705079, 48.94657516479492, 33.82892990112304, 7.4259883186092415, 0.0, 7.0, 0.0],
     [41.418685913085945, 49.8198356628418, 32.65068817138671, 7.988638727319191, 0.0, 8.0, 0.0],
     [41.26764297485353, 49.773773193359375, 31.981468200683594, 7.845206904992795, 0.0, 9.0, 0.0],
     [41.213058471679695, 50.65064239501954, 31.54781150817871, 8.660534448647995, 0.0, 10.0, 0.0],
     [42.33898162841798, 48.701305389404304, 33.892646789550774, 7.236288641184163, 0.0, 11.0, 0.0],
     [42.97335052490236, 49.17356872558594, 34.11601257324218, 7.746614992481262, 0.0, 12.0, 0.0],
     [44.612270355224624, 50.77322006225586, 35.47158050537109, 9.95219725271135, 0.0, 13.0, 0.0],
     [43.98202896118165, 50.38957214355469, 34.98529052734374, 9.295305063906, 0.0, 14.0, 0.0],
     [43.87643814086915, 50.137550354003906, 34.9856948852539, 9.060065535954974, 0.0, 15.0, 0.0],
     [43.933200836181655, 50.044090270996094, 35.01275634765624, 8.997568760548766, 0.0, 16.0, 0.0],
     [44.46725845336915, 50.07977294921875, 34.8570327758789, 9.041081868882419, 0.0, 17.0, 0.0],
     [43.31948089599611, 49.50358200073242, 34.65526580810546, 8.296819805573385, 0.0, 18.0, 0.0],
     [43.069896697998054, 48.95524978637696, 34.21790313720703, 7.597703883087006, 0.0, 19.0, 0.0],
     [42.628978729248054, 49.04669952392579, 33.9011001586914, 7.541250506849936, 0.0, 20.0, 0.0],
     [42.758823394775405, 49.243770599365234, 34.31572341918945, 7.894588153667878, 0.0, 21.0, 0.0],
     [42.93318176269533, 49.17014312744141, 34.882083892822266, 8.09876214913491, 0.0, 22.0, 0.0],
     [43.62960052490236, 49.78305816650391, 35.000999450683594, 8.729992290776469, 0.0, 23.0, 0.0],
     [42.93603897094728, 48.924110412597656, 33.90078735351562, 7.429046538928816, 0.0, 24.0, 0.0],
     [44.141921997070334, 49.86714172363281, 35.08721923828125, 8.907858598251714, 0.0, 25.0, 0.0],
     [43.87744903564454, 50.172691345214844, 35.16545867919921, 9.174931072502227, 0.0, 26.0, 0.0],
     [43.65211486816408, 49.850822448730476, 35.07431411743164, 8.826472601836407, 0.0, 27.0, 0.0],
     [44.0879135131836, 49.82625961303711, 35.38613128662109, 9.012274116533805, 0.0, 28.0, 0.0],
     [43.91638183593751, 49.69148254394531, 34.61071395874023, 8.501495279490664, 0.0, 29.0, 0.0],
     [43.60285949707033, 49.4387092590332, 34.67058563232422, 8.2696085792721, 0.0, 30.0, 0.0],
     [43.83904266357423, 49.286094665527344, 34.333168029785156, 8.00889108530646, 0.0, 31.0, 0.0],
     [43.7696990966797, 49.57588195800781, 34.82412719726562, 8.479706657147526, 0.0, 32.0, 0.0],
     [43.725387573242195, 50.206993103027344, 35.12910842895508, 9.171046135667419, 0.0, 33.0, 0.0],
     [44.432792663574226, 50.13774871826172, 35.08746337890624, 9.190353631935595, 0.0, 34.0, 0.0],
     [43.566356658935554, 50.27623748779297, 35.22721862792968, 9.263398422263167, 0.0, 35.0, 0.0],
     [43.582576751709, 49.28577423095703, 35.17907333374023, 8.386311566145205, 0.0, 36.0, 0.0],
     [44.23309707641604, 50.4329833984375, 35.507789611816406, 9.61287800163936, 0.0, 37.0, 0.0],
     [44.2927589416504, 54.04015350341797, 35.223030090332024, 12.75304801150288, 0.0, 38.0, 0.0],
     [40.792366027832045, 55.76319122314454, 38.22528076171875, 15.612217239153253, 0.0, 39.0, 0.0],
     [40.00946044921877, 55.91952896118164, 39.22503662109375, 16.35766383542346, 0.0, -40.0, 0.0]]
)

test_layered_repre_entity_points2clusters1 = {
    0: {2: '0_0', 3: '0_0', 4: '0_0', 5: '0_0', 6: '1_0', 7: '1_0', 8: '1_0', 9: '1_0', 10: '1_0', 0: '1_1', 1: '1_1',
        11: '2_0', 12: '2_0', 13: '2_0', 14: '3_0', 15: '3_0', 16: '3_0', 17: '3_0', 18: '3_0', 19: '4_0', 20: '4_0',
        21: '4_0', 22: '5_0', 23: '5_0', 24: '5_0', 25: '5_0', 26: '5_0', 27: '5_0', 28: '5_1', 29: '5_1', 30: '5_1',
        31: '6_0', 32: '6_0', 33: '7_0', 34: '7_0', 35: '7_0', 36: '7_0', 37: '7_0', 38: '8_0', 39: '8_0', 40: '9_0',
        41: '9_0', 42: '9_0', 43: '9_0', 44: '9_0', 45: '9_0', 46: '9_0', 47: '9_0', 48: '10_0', 49: '10_0', 50: '10_0',
        51: '10_0', 52: '10_0', 53: '10_0', 54: '11_0', 55: '11_0', 56: '11_0', 57: '11_0', 58: '11_0', 59: '11_0',
        60: '12_0', 61: '12_0', 62: '12_0', 63: '12_0', 64: '12_0', 71: '13_0', 72: '13_0', 73: '13_0', 74: '13_0',
        75: '13_0', 76: '13_0', 77: '13_0', 78: '13_0', 79: '13_0', 65: '13_1', 66: '13_1', 67: '13_1', 68: '13_1',
        69: '13_1', 70: '13_1', 80: '14_0', 81: '14_0', 82: '14_0', 83: '15_1', 84: '15_1', 85: '15_1', 86: '15_1',
        87: '15_1', 88: '15_1', 89: '15_1'}}

test_layered_repre_entity_points2clusters2 = {
    0: {8: '1_0', 9: '1_0', 10: '1_0', 11: '1_0', 12: '1_0', 13: '1_0', 14: '1_0', 0: '1_1', 1: '1_1', 2: '2_0',
        3: '2_0', 4: '2_0', 5: '2_0', 6: '2_0', 7: '2_0', 15: '2_1', 16: '2_1', 17: '2_1', 18: '3_0', 19: '3_0',
        20: '3_0', 21: '3_0', 22: '3_0', 23: '3_0', 24: '4_0', 25: '4_0', 26: '4_0', 27: '4_0', 28: '5_0', 29: '5_0',
        30: '5_0', 31: '5_0', 32: '5_0', 33: '5_0', 34: '5_0', 35: '5_0', 36: '6_0', 37: '6_0', 38: '7_0', 39: '7_0',
        40: '7_0', 41: '7_0', 42: '7_0', 43: '7_0', 44: '7_0', 45: '7_0', 46: '7_0', 47: '8_0', 48: '8_0', 67: '9_0',
        68: '9_0', 69: '9_0', 70: '9_0', 71: '9_0', 72: '9_0', 73: '9_0', 74: '9_0', 54: '9_1', 55: '9_1', 56: '9_1',
        57: '9_1', 58: '9_1', 59: '9_1', 60: '9_1', 62: '9_2', 63: '9_2', 64: '9_2', 65: '9_2', 66: '9_2', 49: '9_3',
        50: '9_3', 51: '9_3', 52: '9_3', 53: '9_3', 75: '10_0', 76: '10_0', 77: '10_0', 78: '10_0', 79: '10_0',
        82: '10_1', 83: '10_1', 84: '10_1', 61: '10_2', 85: '11_0', 86: '11_0', 87: '11_0', 88: '11_0', 89: '11_0',
        90: '11_0', 91: '11_0', 80: '11_1', 81: '11_1', 92: '12_0', 93: '12_0', 94: '13_0', 95: '13_0', 96: '13_0',
        97: '13_0', 98: '13_0', 99: '14_0', 100: '14_0', 101: '14_0', 102: '15_0', 103: '15_0', 104: '15_0',
        105: '15_0', 106: '15_0', 107: '16_0', 108: '16_0', 109: '16_0', 110: '17_0', 111: '17_0', 112: '17_0',
        113: '17_0', 114: '17_0', 115: '18_0', 116: '18_0', 117: '19_1', 118: '19_1', 119: '19_1', 120: '19_1',
        121: '19_1'}}

test_layered_repre_entity_points2clusters3 = {
    0: {0: '0_0', 1: '0_0', 2: '1_0', 3: '1_0', 4: '1_0', 5: '1_0', 6: '1_0', 7: '2_0', 8: '2_0', 9: '2_0', 10: '3_0',
        11: '3_0', 12: '3_0', 13: '3_0', 14: '3_0', 15: '3_0', 16: '4_0', 17: '4_0', 18: '4_0', 19: '5_0', 20: '5_0',
        21: '5_0', 22: '5_0', 23: '5_0', 24: '5_0', 25: '5_0', 26: '6_0', 27: '6_0', 28: '6_0', 29: '7_0', 30: '7_0',
        31: '7_0', 32: '7_0', 33: '7_0', 34: '8_0', 35: '8_0', 36: '8_0', 37: '9_0', 38: '9_0', 39: '9_0', 40: '9_0',
        41: '9_0', 42: '10_0', 43: '10_0', 44: '10_0', 45: '11_0', 46: '11_0', 47: '11_0', 48: '11_0', 49: '11_0',
        50: '11_0', 51: '11_0', 52: '12_0', 53: '12_0', 54: '13_0', 55: '13_0', 56: '13_0', 57: '13_0', 58: '13_0',
        59: '13_0', 60: '13_0', 61: '13_1', 62: '13_1', 63: '13_1', 64: '13_1', 65: '13_1', 66: '13_1', 67: '13_2',
        68: '13_2', 69: '13_2', 70: '13_2', 71: '13_2', 72: '14_0', 73: '14_0', 74: '15_0', 75: '15_0', 76: '15_0',
        77: '15_0', 78: '15_0', 79: '15_0', 80: '15_0', 81: '15_0', 82: '15_1'}}

test_layered_repre_entity_points2clusters4 = {
    0: {0: '0_0', 1: '0_0', 2: '0_0', 3: '0_0', 4: '1_0', 5: '1_0', 6: '1_0', 7: '1_0', 8: '2_0', 9: '2_0', 10: '3_0',
        11: '3_0', 12: '3_0', 13: '3_0', 14: '3_0', 15: '4_0', 16: '4_0', 17: '4_0', 22: '5_0', 23: '5_0', 24: '5_0',
        25: '5_0', 26: '5_0', 27: '5_0', 18: '5_1', 19: '5_1', 20: '5_1', 21: '5_1', 28: '6_0', 29: '6_0', 30: '6_0',
        31: '7_0', 32: '7_0', 33: '7_0', 34: '7_0', 35: '8_0', 36: '8_0', 37: '8_0', 38: '9_0', 39: '9_0', 40: '9_0',
        41: '9_0', 42: '9_0', 43: '9_0', 48: '10_0', 49: '10_0', 50: '10_0', 51: '10_0', 52: '10_0', 44: '10_1',
        45: '10_1', 46: '10_1', 47: '10_1', 53: '11_0', 54: '11_0', 55: '11_0', 56: '11_0', 57: '11_0', 58: '11_0',
        59: '11_0', 60: '11_0', 61: '12_0', 62: '12_0', 63: '12_0', 69: '13_0', 70: '13_0', 71: '13_0', 72: '13_0',
        73: '13_0', 74: '13_0', 75: '13_0', 76: '13_3', 77: '13_3', 78: '13_3', 79: '13_3', 80: '13_3', 81: '13_3',
        82: '13_3', 64: '13_2', 65: '13_2', 66: '13_2', 67: '13_2', 68: '13_2'}}

test_layered_repre_entity_points2clusters5 = {
    0: {0: '3_0', 1: '4_0', 2: '5_0', 3: '5_0', 4: '5_0', 5: '5_0', 6: '5_0', 7: '5_0', 8: '5_0', 9: '5_0', 10: '5_0',
        11: '5_0', 12: '5_0', 16: '5_0', 18: '5_0', 19: '5_0', 20: '5_0', 21: '5_0', 22: '5_0', 23: '5_0', 24: '5_0',
        25: '5_0', 27: '5_0', 29: '5_0', 30: '5_0', 31: '5_0', 32: '5_0', 36: '5_0', 13: '6_0', 14: '6_0', 15: '6_0',
        17: '6_0', 26: '6_0', 28: '6_0', 33: '6_0', 34: '6_0', 35: '6_0', 37: '6_0', 38: '8_0', 39: '10_0', 40: '11_0'}}

test_layered_repre_entity_points2clusters6 = {
    0: {2: '0_0', 3: '0_0', 4: '0_0', 5: '0_0', 6: '1_0', 7: '1_0', 8: '1_0', 9: '1_0', 10: '1_0', 0: '1_3', 1: '1_3',
        11: '2_0', 12: '2_0', 13: '2_0', 14: '3_0', 15: '3_0', 16: '3_0', 17: '3_0', 18: '3_0', 19: '4_0', 20: '4_0',
        21: '4_0', 22: '5_3', 23: '5_3', 24: '5_3', 25: '5_3', 26: '5_3', 27: '5_3', 28: '5_3', 29: '5_3', 30: '5_3',
        31: '6_0', 32: '6_0', 33: '7_0', 34: '7_0', 35: '7_0', 36: '7_0', 37: '7_0', 38: '8_0', 39: '8_0', 40: '9_0',
        41: '9_0', 42: '9_0', 43: '9_0', 44: '9_0', 45: '9_0', 46: '9_4', 47: '9_4', 48: '10_0', 49: '10_0', 50: '10_0',
        51: '10_0', 52: '10_0', 53: '10_0', 54: '11_2', 55: '11_2', 56: '11_2', 57: '11_2', 58: '11_2', 59: '11_2',
        60: '12_0', 61: '12_0', 62: '12_0', 63: '12_0', 64: '12_0', 65: '13_0', 66: '13_0', 67: '13_0', 68: '13_0',
        69: '13_0', 70: '13_0', 75: '13_1', 76: '13_1', 77: '13_1', 79: '13_1', 80: '14_0', 81: '14_0', 82: '14_0',
        83: '15_2', 84: '15_2', 85: '15_2', 86: '15_2', 87: '15_2', 88: '15_2', 89: '15_2'},
    2: {0: '0_0', 1: '0_0', 2: '1_0', 3: '1_0', 4: '1_0', 5: '1_0', 6: '1_0', 7: '2_0', 8: '2_0', 9: '2_0', 10: '3_0',
        11: '3_0', 12: '3_0', 13: '3_0', 14: '3_0', 15: '3_0', 16: '4_0', 17: '4_0', 18: '4_0', 19: '5_3', 20: '5_3',
        21: '5_3', 22: '5_3', 23: '5_3', 24: '5_3', 25: '5_3', 26: '6_0', 27: '6_0', 28: '6_0', 29: '7_0', 30: '7_0',
        31: '7_0', 32: '7_0', 33: '7_0', 34: '8_0', 35: '8_0', 36: '8_0', 37: '9_0', 38: '9_0', 39: '9_0', 40: '9_0',
        41: '9_0', 42: '10_1', 43: '10_1', 44: '10_1', 45: '11_1', 46: '11_1', 47: '11_1', 48: '11_1', 49: '11_1',
        50: '11_1', 51: '11_1', 52: '12_1', 53: '12_1', 57: '13_0', 58: '13_0', 59: '13_0', 60: '13_0', 61: '13_0',
        62: '13_0', 63: '13_0', 64: '13_0', 65: '13_0', 68: '13_1', 69: '13_1', 70: '13_1', 72: '14_0', 73: '14_0',
        74: '15_2', 75: '15_2', 76: '15_2', 77: '15_2', 78: '15_2', 79: '15_2', 80: '15_2', 81: '15_2', 82: '15_2'},
    3: {0: '0_0', 1: '0_0', 2: '0_0', 3: '0_0', 4: '1_0', 5: '1_0', 6: '1_0', 7: '1_0', 8: '2_0', 9: '2_0', 10: '3_0',
        11: '3_0', 12: '3_0', 13: '3_0', 14: '3_0', 15: '4_0', 16: '4_0', 17: '4_0', 18: '5_3', 19: '5_3', 20: '5_3',
        21: '5_3', 22: '5_3', 23: '5_3', 24: '5_3', 25: '5_3', 26: '5_2', 27: '5_2', 28: '6_1', 29: '6_1', 30: '6_1',
        31: '7_2', 32: '7_2', 33: '7_2', 34: '7_2', 35: '8_0', 36: '8_0', 37: '8_0', 38: '9_0', 39: '9_0', 40: '9_0',
        41: '9_0', 42: '9_0', 43: '9_0', 48: '10_0', 49: '10_0', 50: '10_0', 51: '10_0', 52: '10_0', 44: '10_1',
        45: '10_1', 46: '10_1', 47: '10_1', 53: '11_2', 54: '11_2', 55: '11_2', 56: '11_2', 57: '11_2', 58: '11_2',
        59: '11_2', 60: '11_2', 61: '12_0', 62: '12_0', 63: '12_0', 64: '13_0', 65: '13_0', 66: '13_0', 67: '13_0',
        68: '13_0', 69: '13_0', 70: '13_0', 71: '13_0', 82: '13_2'},
    1: {8: '1_1', 9: '1_1', 10: '1_1', 11: '1_1', 12: '1_1', 13: '1_1', 14: '1_1', 0: '1_2', 1: '1_2', 15: '2_0',
        16: '2_0', 17: '2_0', 2: '2_1', 3: '2_1', 4: '2_1', 5: '2_1', 6: '2_1', 7: '2_1', 18: '3_0', 19: '3_0',
        20: '3_0', 21: '3_0', 22: '3_0', 23: '3_0', 24: '4_0', 25: '4_0', 26: '4_0', 27: '4_0', 28: '5_3', 29: '5_3',
        30: '5_3', 31: '5_3', 32: '5_3', 33: '5_3', 34: '5_3', 35: '5_3', 36: '6_0', 37: '6_0', 38: '7_1', 39: '7_1',
        40: '7_1', 41: '7_1', 42: '7_1', 43: '7_1', 44: '7_1', 45: '7_1', 46: '7_1', 47: '8_1', 48: '8_1', 67: '9_0',
        68: '9_0', 69: '9_0', 70: '9_0', 71: '9_0', 72: '9_0', 73: '9_0', 74: '9_0', 54: '9_1', 55: '9_1', 56: '9_1',
        57: '9_1', 58: '9_1', 59: '9_1', 60: '9_1', 49: '9_2', 50: '9_2', 51: '9_2', 52: '9_2', 53: '9_2', 62: '9_3',
        63: '9_3', 64: '9_3', 65: '9_3', 66: '9_3', 82: '10_0', 83: '10_0', 84: '10_0', 75: '10_1', 76: '10_1',
        77: '10_1', 78: '10_1', 79: '10_1', 61: '10_2', 85: '11_2', 86: '11_2', 87: '11_2', 88: '11_2', 89: '11_2',
        90: '11_2', 91: '11_2', 80: '11_1', 81: '11_1', 92: '12_0', 93: '12_0', 94: '13_0', 95: '13_0', 96: '13_0',
        97: '13_0', 99: '14_1', 100: '14_1', 101: '14_1', 102: '15_1', 103: '15_1', 104: '15_1', 105: '15_1',
        106: '15_1', 107: '16_0', 108: '16_0', 109: '16_0', 110: '17_0', 111: '17_0', 112: '17_0', 113: '17_0',
        114: '17_0', 115: '18_0', 116: '18_0', 117: '19_1', 118: '19_1', 119: '19_1', 120: '19_1', 121: '19_1'}}


test_layered_repre_str1 = """LayeredPathSet: Cluster_34 of e10s1_e9s3p0f1600
Paths: 
 0: ['SP' '0_0' '1_0' '2_0' '3_0' '4_0' '5_0' '5_1' '6_0' '7_0' '8_0' '9_0'
 '10_0' '11_0' '12_0' '13_1' '13_0' '14_0' '15_1']
num nodes = 19
Labels 19:
['SP', '0_0', '1_0', '2_0', '3_0', '4_0', '5_0', '5_1', '6_0', '7_0', '8_0', '9_0', '10_0', '11_0', '12_0', '13_1', '13_0', '14_0', '15_1']
Data:
[[ 0.          0.          0.         -1.          0.          0.5
   0.5       ]
 [ 0.31170683  0.13404055  0.13683421  0.          0.          1.57038983
   0.53539699]
 [ 0.61358494 -0.63775711  1.87281351  1.          0.          1.61178275
   0.65103698]
 [-0.04999095 -0.9193352   3.48883252  2.          0.          1.29922575
   0.33285392]
 [ 0.8851972  -0.44876399  4.98136832  3.          0.          0.91710813
   0.67511403]
 [ 2.05833497  0.32368071  6.26316178  4.          0.          1.07346327
   0.40480464]
 [ 3.89802166  0.46598915  6.69545575  5.          0.          1.23006605
   0.83183639]
 [ 5.83059352  0.09530656  6.32582722  5.          0.          1.41629977
   0.35720565]
 [ 6.84869255 -0.19312229  6.58838701  6.          0.          1.26193017
   0.23923205]
 [ 8.21769013  0.33558873  7.38071688  7.          0.          1.07208111
   0.67501846]
 [ 9.56785712  0.96294667  8.13181662  8.          0.          1.56167901
   0.23875381]
 [11.39210936  1.24397104  8.47659158  9.          0.          1.82741696
   1.04935628]
 [13.83417551  0.43704657  6.66049855 10.          0.          1.25901163
   0.71476525]
 [16.06024952  0.31343379  5.66774122 11.          0.          1.0525628
   0.7649861 ]
 [17.64853432 -1.19599373  5.56168883 12.          0.          1.23544508
   0.68195453]
 [18.37496564 -2.98934242  6.11938166 13.          0.          1.45307794
   0.6299824 ]
 [18.59267313 -4.79525586  4.21890035 13.          0.          0.93101
   1.07691497]
 [20.06932765 -6.61230742  4.46576252 14.          0.          0.85720259
   0.38256158]
 [21.68357896 -7.56587743  3.3999622  15.          1.          1.08139466
   0.90673166]]"""

test_layered_repre_str2 = """LayeredPathSet: Cluster_34 of e10s1_e9s3p0f1600
Paths: 
 0: ['SP' '1_1' '2_0' '1_0' '2_1' '3_0' '4_0' '5_0' '6_0' '7_0' '8_0' '9_3'
 '9_1' '10_2' '9_2' '9_0' '10_0' '11_1' '10_1' '11_0' '12_0' '13_0' '14_0'
 '15_0' '16_0' '17_0' '18_0' '19_1']
num nodes = 28
Labels 28:
['SP', '1_1', '2_0', '1_0', '2_1', '3_0', '4_0', '5_0', '6_0', '7_0', '8_0', '9_3', '9_1', '10_2', '9_2', '9_0', '10_0', '11_1', '10_1', '11_0', '12_0', '13_0', '14_0', '15_0', '16_0', '17_0', '18_0', '19_1']
Data:
[[ 0.          0.          0.         -1.          0.          0.5
   0.5       ]
 [-0.30136707 -2.00146161 -1.73032276  1.          0.          2.42974331
   0.25      ]
 [-1.44625343 -2.23200967 -1.33045666  1.          0.          2.52440115
   0.80466535]
 [-1.64819075 -1.53366228  1.16206487  1.          0.          2.03184965
   0.94865279]
 [-0.3246798  -1.59700042  3.09229153  2.          0.          1.22903409
   0.40789737]
 [ 0.99035421 -1.13933368  4.56998434  3.          0.          1.31807592
   0.79823108]
 [ 2.36202866 -0.08014972  5.98476915  4.          0.          1.03427299
   0.53364394]
 [ 4.34563688 -0.47658416  6.9207968   5.          0.          1.08455457
   1.0376773 ]
 [ 6.19006687 -1.58217165  7.05154718  6.          0.          1.1525902
   0.22302704]
 [ 6.53266603 -3.77351959  7.52142765  7.          0.          1.38208143
   1.11808432]
 [ 7.63596713 -5.51473667  8.2607315   8.          0.          1.25092439
   0.24933521]
 [ 8.4796335  -5.65789786  9.21737401  9.          0.          1.39092466
   0.5825696 ]
 [ 9.56404887 -3.41075874 10.28624877  9.          0.          1.03772528
   0.93737286]
 [ 9.97292192 -1.79744003 11.12280364 10.          0.          1.2630492
   0.        ]
 [10.18697753 -0.49825119 10.46892032  9.          0.          1.31758861
   0.70710678]
 [10.7233918   1.50090555  9.17870113  9.          0.          2.09751473
   0.80155128]
 [13.16394978  1.44848713  8.38922768 10.          0.          1.40927708
   0.67221892]
 [14.24840674  0.71046621  7.39663481 11.          0.          1.47222831
   0.24548865]
 [14.4966618   0.3305081   6.69687624 10.          0.          1.4771313
   0.3258993 ]
 [15.70645631 -1.26079176  6.3272238  11.          0.          1.19786887
   0.9451996 ]
 [17.6178248  -2.01927528  5.72596415 12.          0.          1.49635449
   0.25      ]
 [18.74603346 -3.06087726  5.77988141 13.          0.          1.58144154
   0.62808476]
 [20.3387932  -3.23472327  6.16473513 14.          0.          1.49004062
   0.36777381]
 [22.0133239  -2.56002256  6.75683225 15.          0.          1.46931039
   0.69358873]
 [23.16462711 -1.8613479   8.06109032 16.          0.          1.32346024
   0.40642142]
 [24.19738942 -2.41138869  9.47754968 17.          0.          1.08783463
   0.6847727 ]
 [25.47474516 -3.0876114  10.00337456 18.          0.          0.90869576
   0.25      ]
 [27.11432228 -3.11926102 10.14599203 19.          1.          1.21356657
   0.63708438]]"""

test_layered_repre_str3 = """LayeredPathSet: Cluster_34 of e10s1_e9s3p0f1600
Paths: 
 0: ['SP' '0_0' '1_0' '2_0' '3_0' '4_0' '5_0' '6_0' '7_0' '8_0' '9_0' '10_0'
 '11_0' '12_0' '13_0' '13_1' '13_2' '14_0' '15_0' '15_1']
num nodes = 20
Labels 20:
['SP', '0_0', '1_0', '2_0', '3_0', '4_0', '5_0', '6_0', '7_0', '8_0', '9_0', '10_0', '11_0', '12_0', '13_0', '13_1', '13_2', '14_0', '15_0', '15_1']
Data:
[[ 0.          0.          0.         -1.          0.          0.5
   0.5       ]
 [ 0.20717714 -0.55056023  0.18117141  0.          0.          2.04840709
   0.23608807]
 [ 1.4117338  -0.85684995  1.2520812   1.          0.          1.67671159
   0.64799425]
 [ 1.38939746 -1.10583431  2.97223109  2.          0.          1.11649075
   0.40273951]
 [ 1.19771951 -1.17455374  4.75848729  3.          0.          1.17184655
   0.73176139]
 [ 2.29664153 -0.24747543  6.09174132  4.          0.          0.99739002
   0.36237672]
 [ 4.0088474   0.50295226  7.06016879  5.          0.          1.30186893
   0.9327932 ]
 [ 6.27129203  0.30558107  6.9525831   6.          0.          1.5690065
   0.35345126]
 [ 7.7636117   0.28352589  7.63153811  7.          0.          1.40006753
   0.68723443]
 [ 9.33936219  1.32717759  8.12892788  8.          0.          1.64909622
   0.39401829]
 [11.1503542   1.53415515  8.42235612  9.          0.          1.76584072
   0.67272136]
 [12.95346698  1.08262996  8.45067925 10.          0.          1.4738803
   0.39662725]
 [14.54077522 -0.46883965  8.56789007 11.          0.          1.26253662
   0.89904337]
 [15.94406164 -1.56960141  9.41261877 12.          0.          1.0519031
   0.24960642]
 [17.63249989 -2.45716321  8.62149402 13.          0.          1.06998775
   0.95163091]
 [18.82036392 -3.4909138   6.28610187 13.          0.          1.51094464
   0.72398686]
 [19.12417889 -5.02665594  4.61590945 13.          0.          1.058486
   0.59722238]
 [20.15493997 -6.20540496  4.79275056 14.          0.          0.81239314
   0.24804487]
 [21.54535404 -7.36052217  3.71583647 15.          0.          1.03770926
   1.01263987]
 [22.27455328 -8.4482808   2.17816334 15.          1.          1.08216428
   0.        ]]"""

test_layered_repre_str4 = """LayeredPathSet: Cluster_34 of e10s1_e9s3p0f1600
Paths: 
 0: ['SP' '0_0' '1_0' '2_0' '3_0' '4_0' '5_1' '5_0' '6_0' '7_0' '8_0' '9_0'
 '10_1' '10_0' '11_0' '12_0' '13_2' '13_0' '13_3']
num nodes = 19
Labels 19:
['SP', '0_0', '1_0', '2_0', '3_0', '4_0', '5_1', '5_0', '6_0', '7_0', '8_0', '9_0', '10_1', '10_0', '11_0', '12_0', '13_2', '13_0', '13_3']
Data:
[[ 0.00000000e+00  0.00000000e+00  0.00000000e+00 -1.00000000e+00
   0.00000000e+00  5.00000000e-01  5.00000000e-01]
 [-1.32503899e-01  2.12840879e-01  2.92947855e-01  0.00000000e+00
   0.00000000e+00  2.08039998e+00  5.17348256e-01]
 [-1.63355604e-02 -5.76917812e-01  1.98168911e+00  1.00000000e+00
   0.00000000e+00  1.68078084e+00  5.50265664e-01]
 [ 5.51390574e-02 -7.25566210e-01  3.46133946e+00  2.00000000e+00
   0.00000000e+00  1.24547876e+00  2.47501061e-01]
 [ 1.01739866e+00 -4.65377993e-01  4.65374011e+00  3.00000000e+00
   0.00000000e+00  1.11420565e+00  7.05008276e-01]
 [ 2.16077521e+00  2.78120773e-01  6.02956941e+00  4.00000000e+00
   0.00000000e+00  1.19944503e+00  3.82945698e-01]
 [ 3.54241369e+00  7.22633828e-01  6.33252685e+00  5.00000000e+00
   0.00000000e+00  1.14556725e+00  5.57743443e-01]
 [ 5.64420512e+00  1.13713816e+00  5.93537365e+00  5.00000000e+00
   0.00000000e+00  1.28327151e+00  7.07206712e-01]
 [ 6.81824623e+00  2.49633988e+00  6.17060802e+00  6.00000000e+00
   0.00000000e+00  1.09373881e+00  3.82662589e-01]
 [ 8.23266024e+00  3.05230847e+00  6.85104797e+00  7.00000000e+00
   0.00000000e+00  1.32937716e+00  5.40272764e-01]
 [ 9.40112035e+00  2.96393893e+00  7.87285207e+00  8.00000000e+00
   0.00000000e+00  1.56769622e+00  4.08248290e-01]
 [ 1.10694187e+01  2.26995740e+00  8.73711433e+00  9.00000000e+00
   0.00000000e+00  2.01335234e+00  7.23752812e-01]
 [ 1.30910663e+01  1.59056140e+00  8.05793373e+00  1.00000000e+01
   0.00000000e+00  1.33777859e+00  5.34029039e-01]
 [ 1.41656757e+01  9.62271651e-01  6.39709184e+00  1.00000000e+01
   0.00000000e+00  1.39635087e+00  6.56810466e-01]
 [ 1.64338801e+01 -1.14151020e-01  5.14660911e+00  1.10000000e+01
   0.00000000e+00  1.05528802e+00  1.02798792e+00]
 [ 1.76446501e+01 -2.09578415e+00  5.50026526e+00  1.20000000e+01
   0.00000000e+00  1.27277819e+00  3.95149755e-01]
 [ 1.80831747e+01 -3.17295760e+00  6.60710952e+00  1.30000000e+01
   0.00000000e+00  1.56524401e+00  5.90070327e-01]
 [ 1.75977330e+01 -3.37615454e+00  9.18042653e+00  1.30000000e+01
   0.00000000e+00  1.08700574e+00  8.20185643e-01]
 [ 1.64813947e+01 -4.31802502e+00  1.11194712e+01  1.30000000e+01
   1.00000000e+00  1.43529606e+00  8.89235579e-01]]"""

test_layered_repre_str5 = """LayeredPathSet: 2_release of e10s1_e9s3p0f1600
Paths: 
 0: ['3_0' '4_0' '5_0' '6_0' '8_0' '10_0' '11_0']
num nodes = 7
Labels 7:
['3_0', '4_0', '5_0', '6_0', '8_0', '10_0', '11_0']
Data:
[[-0.69133667  4.44307624 -0.85302905  3.          0.          0.
   0.        ]
 [-1.05206942  5.80420677 -1.80965129  4.          0.          0.
   0.        ]
 [-1.04837082  7.12598363 -3.38927813  5.          0.          0.
   1.26790068]
 [-0.736923    7.76745513 -4.97317254  6.          0.          0.
   0.45382868]
 [-2.76165113 10.65007123 -6.39822539  8.          0.          0.
   0.        ]
 [-0.93645568 14.8655843  -4.64054646 10.          0.          0.
   0.        ]
 [-0.12968873 15.78311426 -4.26601097 11.          1.          0.
   0.        ]]"""

test_layered_repre_str6 = """LayeredPathSet: Cluster_34 of e10s1_e9s3p0f1600
Paths: 
 0: ['SP' '0_0' '1_0' '2_0' '3_0' '4_0' '5_3' '6_0' '7_0' '8_0' '9_0' '9_4'
 '10_0' '11_2' '12_0' '13_0' '13_1' '14_0' '15_2']
 1: ['SP' '1_2' '2_1' '1_1' '2_0' '3_0' '4_0' '5_3' '6_0' '7_1' '8_1' '9_2'
 '9_1' '10_2' '9_3' '9_0' '10_1' '11_1' '10_0' '11_2' '12_0' '13_0' '14_1'
 '15_1' '16_0' '17_0' '18_0' '19_1']
 2: ['SP' '0_0' '1_0' '2_0' '3_0' '4_0' '5_3' '6_0' '7_0' '8_0' '9_0' '10_1'
 '11_1' '12_1' '13_0' '13_1' '14_0' '15_2']
 3: ['SP' '0_0' '1_0' '2_0' '3_0' '4_0' '5_3' '5_2' '6_1' '7_2' '8_0' '9_0'
 '10_1' '10_0' '11_2' '12_0' '13_0' '13_2']
num nodes = 41
Labels 41:
['SP', '0_0', '1_0', '2_0', '3_0', '4_0', '5_3', '6_0', '7_0', '8_0', '9_0', '9_4', '10_0', '11_2', '12_0', '13_0', '13_1', '14_0', '15_2', '1_2', '2_1', '1_1', '7_1', '8_1', '9_2', '9_1', '10_2', '9_3', '10_1', '11_1', '14_1', '15_1', '16_0', '17_0', '18_0', '19_1', '12_1', '5_2', '6_1', '7_2', '13_2']
Data:
[[ 0.          0.          0.         -1.          0.          0.5
   0.5       ]
 [ 0.1131166   0.02864053  0.20814711  0.          0.          2.06832947
   0.60405372]
 [ 0.71866082 -0.6986219   1.68223072  1.          0.          1.6926112
   0.91514816]
 [ 0.28676893 -1.11978565  3.23479494  2.          0.          1.25254695
   0.87128657]
 [ 1.02915553 -0.83881975  4.733926    3.          0.          1.20647552
   0.83303342]
 [ 2.23041306  0.0571061   6.08403801  4.          0.          1.1297702
   0.51958236]
 [ 4.36075392  0.27923341  6.66713433  5.          0.          1.32900912
   1.23087956]
 [ 6.41305642 -0.37626353  6.87680252  6.          0.          1.54667822
   0.90625075]
 [ 7.99065091  0.30955731  7.5061275   7.          0.          1.33986646
   0.72940189]
 [ 9.41964523  1.84990536  8.03362163  8.          0.          1.61947031
   0.96119837]
 [10.95588355  1.66014231  8.79918678  9.          0.          2.01487495
   0.89027054]
 [12.59760191  0.87298763  7.89856941  9.          0.          1.09607199
   0.24915435]
 [14.0945298   0.60179728  6.57421995 10.          0.          1.45551363
   0.74362571]
 [16.08465392 -0.37419751  5.68904223 11.          0.          1.19080607
   1.27805719]
 [17.64122716 -1.63058717  5.57611682 12.          0.          1.38253582
   0.69865968]
 [18.30317215 -3.08401942  6.76657868 13.          0.          1.54572183
   1.15670979]
 [18.85127161 -5.27007406  4.28321539 13.          0.          1.03372403
   0.62949831]
 [20.10357258 -6.44954643  4.59655774 14.          0.          0.85069733
   0.42387858]
 [21.65140239 -7.51835001  3.48153691 15.          1.          1.1043761
   1.05210686]
 [-0.30136707 -2.00146161 -1.73032276  1.          0.          2.42974331
   0.25      ]
 [-1.44625343 -2.23200967 -1.33045666  1.          0.          2.52440115
   0.80466535]
 [-1.64819075 -1.53366228  1.16206487  1.          0.          2.03184965
   0.94865279]
 [ 6.53266603 -3.77351959  7.52142765  7.          0.          1.38208143
   1.11808432]
 [ 7.63596713 -5.51473667  8.2607315   8.          0.          1.25092439
   0.24933521]
 [ 8.4796335  -5.65789786  9.21737401  9.          0.          1.39092466
   0.5825696 ]
 [ 9.56404887 -3.41075874 10.28624877  9.          0.          1.03772528
   0.93737286]
 [ 9.97292192 -1.79744003 11.12280364 10.          0.          1.2630492
   0.        ]
 [10.18697753 -0.49825119 10.46892032  9.          0.          1.31758861
   0.70710678]
 [13.0870346   1.40438093  8.29415926 10.          0.          1.43623868
   0.6296199 ]
 [14.47580445 -0.20677168  8.30761112 11.          0.          1.44888441
   1.06526362]
 [20.3387932  -3.23472327  6.16473513 14.          0.          1.49004062
   0.36777381]
 [22.0133239  -2.56002256  6.75683225 15.          0.          1.46931039
   0.69358873]
 [23.16462711 -1.8613479   8.06109032 16.          0.          1.32346024
   0.40642142]
 [24.19738942 -2.41138869  9.47754968 17.          0.          1.08783463
   0.6847727 ]
 [25.47474516 -3.0876114  10.00337456 18.          0.          0.90869576
   0.25      ]
 [27.11432228 -3.11926102 10.14599203 19.          1.          1.21356657
   0.63708438]
 [15.94406164 -1.56960141  9.41261877 12.          0.          1.0519031
   0.24960642]
 [ 6.0988486   1.78387119  5.76591137  5.          0.          1.19247852
   0.21281303]
 [ 6.81824623  2.49633988  6.17060802  6.          0.          1.09373881
   0.38266259]
 [ 8.23266024  3.05230847  6.85104797  7.          0.          1.32937716
   0.54027276]
 [16.08894333 -5.63110258 11.12327308 13.          1.          1.46598849
   0.        ]]"""

test_helpers_rthetaphis = [
    [0., 0., 0.], [1.73205081, 0.95531662, 0.78539816], [1.41421356, 1.57079633, 2.35619449],
    [1., 1.57079633, -1.57079633], [6.08276253, 0.16514868, 1.57079633],
    [2.9890715, 1.57066251, 0.69278586], [1.54894726, 1.52209886, 1.57079633],
    [6.70820393, 0.46364761, -1.57079633], [14.261136,  0.1916621, 1.46013911],
    [4., 1.57079633, 0.], [4., 1.57079633, 3.14159265], [2., 1.57079633, 1.57079633],
    [4., 0., 0.], [4., 3.14159265, 0.]
]

test_helpers_angles = [
    0.0, 1.5707963267948966, 2.356194490192345, 1.7359450042095235, 1.4654708122246194, 0.8789856083217459,
    1.985746467871314, 0.6544542271623628, 1.5497585826429303, 3.141592653589793, 1.5707963267948966,
    1.5707963267948966, 3.141592653589793
]

test_helpers_layer_ids = [
    np.array([-1.0, 999.0, 12999.0, 1999.0, 999.0, 1999.0, 2999.0, 3999.0, 3999.0, 4999.0, 7999.0, 999.0, 2999.0,
              5999.0, 4999.0, 9999.0, 119999.0, 999.0]),
    np.array([-1.0, 1.0, 25.0, 3.0, 1.0, 3.0, 5.0, 7.0, 7.0, 9.0, 15.0, 1.0, 5.0, 11.0, 9.0, 19.0, 239.0, 1.0]),
    np.array([-1.0, 0.0, 12.0, 1.0, 0.0, 1.0, 2.0, 3.0, 3.0, 4.0, 7.0, 0.0, 2.0, 5.0, 4.0, 9.0, 119.0, 0.0]),
    np.array([-1.0, 0.0, 8.0, 1.0, 0.0, 1.0, 1.0, 2.0, 2.0, 3.0, 5.0, 0.0, 1.0, 3.0, 3.0, 6.0, 79.0, 0.0]),
    np.array([-1.0, 0.0, 6.0, 0.0, 0.0, 0.0, 1.0, 1.0, 1.0, 2.0, 3.0, 0.0, 1.0, 2.0, 2.0, 4.0, 59.0, 0.0]),
    np.array([-1.0, 0.0, 2.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 1.0, 0.0, 1.0, 23.0, 0.0])
]

test_helpers_unique_ids_membership = [
    (np.array([ 0, 999, 1999, 2999, 3999, 4999, 5999, 7999, 9999, 12999, 119999]),
     np.array([ 0., 999., 12999., 1999., 999., 1999., 2999., 3999., 3999., 4999., 7999., 999., 2999., 5999., 4999., 
                9999., 119999., 999.])),
    (np.array([  0,   1,   3,   5,   7,   9,  11,  15,  19,  25, 239]),
     np.array([  0.,   1.,  25.,   3.,   1.,   3.,   5.,   7.,   7.,   9.,  15., 1.,   5.,  11., 9.,  19., 239., 1.])),
    (np.array([  0,   1,   2,   3,   4,   5,   7,   9,  12, 119]),
     np.array([  0.,   0.,  12.,   1.,   0.,   1.,   2.,   3.,   3.,   4., 7., 0., 2.,   5.,   4.,   9., 119.,   0.])),
    (np.array([ 0,  1,  2,  3,  5,  6,  8, 79]),
     np.array([ 0.,  0.,  8.,  1.,  0.,  1.,  1.,  2.,  2.,  3.,  5.,  0.,  1., 3.,  3.,  6., 79.,  0.])),
    (np.array([ 0,  1,  2,  3,  4,  6, 59]),
     np.array([ 0.,  0.,  6.,  0.,  0.,  0.,  1.,  1.,  1.,  2.,  3.,  0.,  1., 2.,  2.,  4., 59.,  0.])),
    (np.array([ 0,  1,  2, 23]),
     np.array([ 0.,  0.,  2.,  0.,  0.,  0.,  0.,  0.,  0.,  0.,  1.,  0.,  0., 1.,  0.,  1., 23.,  0.]))
]

test_helpers_direct_paths = [
    ['0_1', '1_1', '2_1', '3_5', '4_2', '5_2', '6_1', '7_2', '8_2'],
    ['1_1', '2_1', '3_5', '4_2', '5_2', '6_1', '7_2', '8_2'],
    ['0_1', '1_1', '2_1', '3_5', '4_2', '5_2', '6_1', '7_2', '8_2', '9_2'],
    ['0_1', '1_1', '2_1', '7_2', '8_2', '9_2'],
    ['1_2', '2_1', '3_1', '4_1', '5_1', '6_1', '7_1', '8_2', '9_1', '10_2', '11_4']
]

test_helpers_redundant_path_ids = [
    set(),
    {3},
    set(),
    {5, 8, 10, 11, 13, 15, 16, 17, 18, 23}
]

test_helpers_coarse_grained_paths = [
    ['0_1', '1_1', '2_1', '3_1', '5_1', '6_1', '7_1', '9_2', '8_1', '9_1', '10_1', '11_1'],
    ['0_1', '1_1', '2_2', '3_1', '4_1', '5_1', '6_1', '7_1', '8_2', '9_2', '9_1', '10_1'],
    ['0_1', '1_1', '2_2', '3_1', '4_1', '5_1', '6_1', '7_1', '8_2', '9_1', '10_1', '11_1'],
    ['0_1', '1_1', '2_2', '3_1', '4_1', '5_1', '6_1', '7_1', '8_2', '9_2', '8_1', '9_1', '10_3', '11_1'],
    ['0_1', '1_1', '2_2', '3_1', '4_1', '5_1', '6_1', '7_1', '8_2', '9_2', '10_2', '11_2'],
    ['0_1', '1_1', '2_1', '3_1', '4_1', '5_1', '6_1', '7_1', '9_2', '9_1', '10_1', '11_1'],
    ['0_1', '1_1', '2_2', '3_1', '4_1', '5_1', '6_1', '7_1', '8_2', '9_2', '8_1', '9_1'],
    ['0_1', '1_1', '2_2', '3_1', '4_1', '5_1', '7_1', '8_2', '9_2', '10_2', '11_2', '12_1']
]

test_helpers_squeezed_paths = [
    ['1_1', '0_1', '1_1', '2_1', '3_5', '4_2', '5_2', '6_1', '7_2', '8_2'],
    ['1_1', '2_1', '3_5', '4_2', '5_2', '6_1', '7_2', '6_1', '7_2', '8_2'],
    ['0_1', '1_1', '2_1', '3_5', '4_2', '5_2', '6_1', '7_2', '8_2', '9_2'],
    ['0_1', '1_1', '2_1', '3_5', '4_2', '5_2', '2_1', '7_2', '8_2', '9_2'],
    ['1_2', '2_1', '3_1', '4_1', '5_1', '6_1', '7_1', '8_2', '9_1', '10_2', '11_4']
]

test_helpers_xyzs = [
    [0., 0., 0.], [1., 1., 1.], [-1., 1., 0.], [-0., -1., 0.], [0., 1., 6.], [2.3, 1.90907, 0.0004],
    [0.000000001, 1.547111, .0754], [0., -3., 6.], [.3, 2.7, 14.], [4., 0., 0.], [-4., 0., 0.], [0., 2., 0.],
    [0., 0., 4.], [0., 0., -4.]
]

test_helpers_all_paths = [
    {0: ['0_1', '1_1', '2_1', '3_1', '4_1', '5_1', '6_2', '7_1', '8_1', '8_3', '7_3', '6_3', '5_2', '5_3',
         '6_1', '7_2', '7_4', '8_2', '9_1']},
    {0: ['0_1', '1_1', '2_1', '3_1', '4_1', '5_1', '6_1', '7_1', '8_1', '9_2', '10_2', '10_1', '11_2', '11_1',
         '12_2'],
     1: ['1_1', '2_1', '3_1', '4_1', '5_1', '6_1', '7_1', '8_1', '9_2', '10_1', '11_2', '12_3'],
     2: ['0_1', '1_1', '2_1', '3_1', '4_1', '5_1', '6_1', '7_1', '8_1', '9_2', '9_1', '10_1', '11_2'],
     3: ['0_1', '1_1', '2_1', '3_1', '4_1', '5_1', '6_2', '7_1', '8_1', '9_2', '10_1', '11_2', '12_3', '13_1'],
     4: ['0_1', '1_1', '2_1', '3_1', '4_1', '5_1', '6_2', '6_1', '7_1', '8_1', '9_2', '10_1', '11_2',
         '12_3', '13_1'],
     5: ['0_1', '1_1', '2_1', '3_1', '4_1', '5_1', '6_2', '7_1', '8_1', '9_2', '10_1', '11_2', '12_3', '13_1'],
     6: ['0_1', '1_1', '2_1', '3_1', '4_1', '5_1', '6_1', '7_1', '8_1', '9_1', '10_1', '11_2', '12_3'],
     7: ['1_1', '2_1', '3_1', '4_1', '5_1', '6_1', '7_1', '8_1', '9_2', '9_1', '10_1', '11_1', '12_3', '13_1'],
     10: ['0_1', '1_1', '2_1', '3_1', '4_1', '5_1', '6_1', '7_1', '8_1', '9_1', '10_3', '11_1', '12_1', '13_3'],
     11: ['2_2', '3_1', '4_1', '5_1', '6_1', '7_1', '8_1', '9_2', '10_1', '10_3', '11_1', '12_1', '13_2']},
    {0: ['1_1', '2_1', '3_1', '4_1', '5_1', '5_2', '6_1', '7_2', '8_1', '9_1', '10_1', '11_1', '12_1'],
     1: ['0_1', '1_1', '2_1', '3_1', '4_1', '5_1', '5_3', '6_2', '7_1', '8_2', '8_1', '9_1', '10_1', '11_1'],
     2: ['0_1', '1_1', '2_1', '3_1', '4_1', '5_1', '5_2', '6_1', '7_2', '8_1', '9_1', '10_1', '11_1']},
    {0: ['0_1', '1_1', '2_1', '3_1', '4_1', '5_2', '6_2', '7_1', '7_3', '8_2', '9_2', '10_2', '11_2',
         '10_3', '9_1', '10_1', '11_1', '12_1', '13_1', '14_0'],
     3: ['1_1', '2_1', '3_1', '4_1', '5_2', '6_2', '7_1', '7_3', '8_2', '9_1', '10_1', '11_1', '12_2'],
     4: ['0_1', '1_1', '2_1', '3_1', '4_1', '5_2', '6_2', '7_1', '6_3', '7_3', '8_2', '9_1', '10_1', '11_1',
         '12_2', '13_2'],
     5: ['0_1', '1_1', '2_1', '3_1', '4_1', '5_2', '6_2', '7_1', '6_3', '7_3', '8_2', '9_1', '10_1', '11_1'],
     6: ['0_1', '1_1', '2_1', '3_1', '4_1', '5_2', '6_2', '7_1', '7_3', '8_2', '9_1', '10_1'],
     8: ['0_1', '1_1', '2_1', '3_1', '4_1', '5_2', '6_2', '7_1', '6_3', '7_3', '8_2', '9_1', '10_1', '11_1'],
     10: ['0_1', '1_1', '2_1', '3_1', '4_1', '5_2', '6_2', '7_1', '6_3', '7_3', '8_2', '9_1', '10_1', '11_1'],
     11: ['3_1', '4_1', '5_2', '6_2', '7_1', '6_3', '7_3', '8_2', '9_1', '10_1', '11_1'],
     12: ['0_1', '1_1', '2_1', '3_1', '4_1', '5_2', '6_2', '7_1', '6_3', '7_3', '8_2', '9_3', '9_1', '10_1',
          '11_1'],
     13: ['0_1', '1_1', '2_1', '3_1', '4_1', '5_2', '6_2', '7_1', '6_3', '7_3', '8_2', '9_1', '10_1'],
     15: ['0_1', '1_1', '2_1', '3_1', '4_1', '5_2', '6_2', '7_1', '6_3', '7_3', '8_2', '9_1', '10_1', '11_1'],
     16: ['0_1', '1_1', '2_1', '3_1', '4_1', '5_2', '6_2', '7_1', '6_3', '7_3', '8_2', '9_1', '10_1', '11_1'],
     17: ['1_1', '2_1', '3_1', '4_1', '5_2', '6_2', '7_1', '6_3', '7_3', '8_2', '9_1', '10_1', '11_1'],
     18: ['1_1', '2_1', '3_1', '4_1', '5_2', '6_2', '7_1', '6_3', '7_3', '8_2', '9_1', '10_1'],
     22: ['1_1', '2_1', '3_1', '4_1', '5_2', '6_2', '7_1', '6_3', '7_3', '8_2', '9_1', '10_1', '11_1'],
     23: ['3_1', '4_1', '5_2', '6_2', '7_1', '6_3', '7_3', '8_2', '9_1', '10_1', '11_1']}
]

test_helpers_point2cluster_map = {
    1: {0: '0_1', 1: '0_1', 2: '0_1', 3: '1_1', 4: '1_1', 5: '1_1', 6: '1_1', 7: '2_1', 8: '2_1', 12: '3_1',
        13: '3_1', 14: '3_1', 15: '3_1', 25: '5_1', 26: '5_1', 27: '5_1', 28: '5_1', 29: '5_1', 30: '5_1',
        31: '5_1', 32: '5_1', 35: '6_1', 36: '6_1', 38: '7_1', 39: '7_1', 40: '7_1', 41: '7_1', 50: '8_1',
        51: '8_1', 56: '9_1', 57: '9_1', 58: '9_1', 59: '9_1', 47: '9_2', 48: '9_2', 60: '10_1', 61: '10_1',
        62: '11_1'},
    2: {0: '0_1', 1: '0_1', 2: '1_1', 3: '1_1', 4: '1_1', 5: '1_1', 6: '1_1', 7: '2_2', 8: '2_2', 9: '3_1',
        10: '3_1', 11: '3_1', 12: '3_1', 13: '3_1', 14: '4_1', 15: '4_1', 16: '4_1', 17: '4_1', 18: '4_1',
        19: '4_1', 20: '5_1', 21: '5_1', 22: '5_1', 23: '5_1', 24: '5_1', 25: '5_1', 28: '6_1', 29: '6_1',
        30: '6_1', 31: '6_1', 32: '7_1', 33: '7_1', 34: '7_1', 35: '7_1', 36: '7_1', 37: '7_1',
        38: '8_2', 39: '8_2', 40: '8_2', 41: '8_2', 47: '9_1', 48: '9_1', 49: '9_1', 50: '9_1', 51: '9_1',
        42: '9_2', 43: '9_2', 52: '10_1', 53: '10_1'},
    3: {0: '0_1', 1: '0_1', 2: '1_1', 3: '1_1', 4: '1_1', 5: '1_1', 6: '1_1', 7: '2_2', 8: '2_2', 9: '3_1',
        10: '3_1', 11: '3_1', 12: '3_1', 13: '3_1', 14: '4_1',
        15: '4_1', 16: '4_1', 17: '4_1', 18: '4_1', 19: '4_1', 21: '5_1', 22: '5_1',
        23: '5_1', 25: '5_1', 29: '6_1', 30: '6_1', 31: '6_1', 32: '7_1', 33: '7_1',
        34: '7_1', 35: '7_1', 36: '7_1', 37: '7_1', 38: '8_2', 39: '8_2', 40: '8_2',
        50: '9_1', 51: '9_1', 52: '9_1', 53: '9_1', 55: '10_1', 56: '10_1', 57: '11_1',
        58: '11_1', 59: '11_1'},
    5: {0: '0_1', 1: '0_1', 2: '1_1', 3: '1_1', 4: '1_1', 5: '1_1', 6: '1_1', 7: '1_1',
        8: '2_2', 9: '2_2', 10: '3_1', 11: '3_1', 12: '3_1', 13: '3_1', 18: '4_1',
        21: '5_1', 22: '5_1', 23: '5_1', 24: '5_1', 26: '6_1', 27: '6_1', 30: '7_1',
        31: '7_1', 32: '7_1', 33: '7_1', 34: '7_1', 35: '7_1', 36: '7_1', 37: '7_1',
        48: '8_1', 38: '8_2', 39: '8_2', 40: '8_2', 49: '9_1', 50: '9_1', 51: '9_1',
        52: '9_1', 53: '9_1', 41: '9_2', 42: '9_2', 43: '9_2', 57: '10_3', 58: '10_3',
        59: '11_1', 60: '11_1', 61: '11_1'},
    6: {0: '0_1', 1: '1_1', 2: '1_1', 3: '1_1', 4: '1_1', 5: '2_2', 6: '2_2', 7: '3_1',
        8: '3_1', 9: '3_1', 10: '3_1', 11: '3_1', 13: '4_1', 14: '4_1', 15: '4_1',
        16: '4_1', 17: '4_1', 18: '5_1', 19: '5_1', 20: '5_1', 21: '5_1', 22: '5_1',
        23: '5_1', 24: '6_1', 25: '6_1', 26: '6_1', 27: '6_1', 28: '6_1', 29: '6_1',
        30: '7_1', 31: '7_1', 32: '7_1', 33: '7_1', 34: '7_1', 35: '7_1', 36: '8_2',
        37: '8_2', 38: '8_2', 39: '8_2', 40: '9_2', 41: '9_2', 42: '9_2', 43: '9_2',
        44: '10_2', 45: '10_2', 46: '10_2', 47: '11_2', 48: '11_2'},
    7: {0: '0_1', 1: '1_1', 2: '1_1', 3: '1_1', 4: '1_1', 5: '2_1', 6: '2_1', 8: '3_1',
        9: '3_1', 10: '3_1', 11: '3_1', 12: '3_1', 14: '4_1', 15: '4_1', 16: '4_1',
        17: '4_1', 20: '5_1', 21: '5_1', 23: '5_1', 24: '5_1', 25: '5_1', 27: '6_1',
        28: '6_1', 30: '7_1', 31: '7_1', 32: '7_1', 33: '7_1', 34: '7_1', 35: '7_1',
        36: '7_1', 37: '7_1', 45: '9_1', 46: '9_1', 47: '9_1', 48: '9_1', 49: '9_1',
        50: '9_1', 51: '9_1', 52: '9_1', 43: '9_2', 44: '9_2', 53: '10_1', 54: '10_1',
        55: '11_1'},
    11: {0: '0_1', 1: '0_1', 2: '0_1', 3: '1_1', 4: '1_1', 5: '1_1', 6: '1_1', 7: '1_1',
         8: '2_2', 9: '2_2', 10: '2_2', 11: '3_1', 12: '3_1', 13: '3_1', 14: '3_1',
         15: '3_1', 17: '4_1', 18: '4_1', 19: '4_1', 20: '4_1', 21: '4_1', 22: '5_1',
         23: '5_1', 24: '5_1', 25: '5_1', 26: '5_1', 27: '5_1', 28: '5_1', 29: '5_1',
         31: '6_1', 32: '6_1', 33: '6_1', 34: '6_1', 35: '6_1', 36: '7_1', 37: '7_1',
         38: '7_1', 39: '7_1', 40: '7_1', 41: '7_1', 50: '8_1', 46: '8_1', 47: '8_1',
         48: '8_1', 42: '8_2', 43: '8_2', 44: '8_2', 51: '9_1', 52: '9_1', 53: '9_1',
         54: '9_1', 55: '9_1', 45: '9_2'},
    15: {0: '0_1', 1: '0_1', 2: '1_1', 3: '1_1', 4: '1_1', 5: '1_1', 6: '1_1', 7: '2_2',
         8: '2_2', 9: '3_1', 10: '3_1', 11: '3_1', 12: '3_1', 13: '3_1', 14: '4_1',
         15: '4_1', 16: '4_1', 17: '4_1', 18: '4_1', 19: '5_1', 20: '5_1', 21: '5_1',
         22: '5_1', 32: '7_1', 33: '7_1', 34: '7_1', 35: '7_1', 38: '8_2', 39: '8_2',
         40: '8_2', 41: '9_2', 42: '9_2', 43: '9_2', 44: '9_2', 45: '9_2', 46: '10_2',
         47: '10_2', 48: '11_2', 49: '11_2', 50: '11_2', 51: '11_2', 52: '11_2',
         53: '11_2', 54: '12_1'}
}