# -*- coding: utf-8 -*-

# TransportTools, a library for massive analyses of internal voids in biomolecules and ligand transport through them
# Copyright (C) 2021  Jan Brezovsky, Aravind Selvaram Thirunavukarasu, Carlos Eduardo Sequeiros-Borja, Bartlomiej
# Surpeta, Nishita Mandal, Cedrix Jurgal Dongmo Foumthuim, Dheeraj Kumar Sarkar, Nikhil Agrawal  <janbre@amu.edu.pl>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

__version__ = '0.9.0'
__author__ = 'Jan Brezovsky, Aravind Selvaram Thirunavukarasu, Carlos Eduardo Sequeiros-Borja, Bartlomiej Surpeta, ' \
             'Nishita Mandal, Cedrix Jurgal Dongmo Foumthuim, Dheeraj Kumar Sarkar, Nikhil Agrawal'
__mail__ = 'janbre@amu.edu.pl'

import numpy as np
from transport_tools.libs.geometry import Point
from sys import maxsize
np.set_printoptions(threshold=maxsize)

datasection0 = [
    "stripped_system.1.pdb, 1, 2, 0.5662269402609552, 0.5687603266150838, 1.1096497696890928, -, -, -, "
    "1.19923106015622, 13.935477752845646, , X, 3.9231396368321483, 4.409257133292514, 4.89537462975288, "
    "5.382361184870531, 5.839344475388041, 6.184970661388952, 6.515655695208539, 6.988791860273927, "
    "7.471285868236148, 7.867203774242666, 8.299213096978562, 8.732541757245892, 9.165870417513222, "
    "9.603066214174294, 9.961835583879651, 10.13164387613461, 10.293875194817657, 10.450338049067156, "
    "10.605709002184067, 10.811303743207105, 10.96837706784503, 11.114314712886188, 11.260252357927346, "
    "11.356463949875144, 11.512457789759821, 11.678472746970444, 11.876125411810433, 12.075495725928016, "
    "12.246241188364774, 12.41698665080153\n",
    "stripped_system.1.pdb, 1, 2, 0.5662269402609552, 0.5687603266150838, 1.1096497696890928, -, -, -, "
    "1.19923106015622, 13.935477752845646, , Y, 3.8686058563233057, 3.752867518897115, 3.637129181470924, "
    "3.5287962069899437, 3.401352046966683, 3.18336363647974, 3.394821686792071, 3.489670307071897, "
    "3.6102417115136114, 3.692794351918671, 3.6345334925674697, 3.5820909642740975, 3.5296484359807248, "
    "3.595445922301469, 3.5348252846786803, 3.1658798578518517, 2.8293373117486222, 2.4885638009030484,"
    " 2.0996803567254805, 1.7197623469364594, 1.3684849827671266, 0.9830381299608426, 0.5975912771545585, "
    "0.38641732216490915, 0.23321269970391703, 0.10408379991582889, 0.05268244334746751, -0.0448822031970907, "
    "-0.4472807594514884, -0.849679315705886\n",
    "stripped_system.1.pdb, 1, 2, 0.5662269402609552, 0.5687603266150838, 1.1096497696890928, -, -, -, "
    "1.19923106015622, 13.935477752845646, , Z, -7.909170286835895, -7.892011706331463, -7.8748531258270305, "
    "-7.876354124020829, -7.944349902916014, -8.078075541321214, -8.142641644910547, -8.181969814274334, "
    "-8.170216159546937, -8.423263346652227, -8.667697280765092, -8.911574408469024, -9.155451536172954, "
    "-9.375703690279577, -9.471692404259596, -9.650863753497099, -9.98315897449902, -10.312871843695083, "
    "-10.577454951646743, -10.827163906771256, -11.11873393201795, -11.401811531664357, -11.684889131310765, "
    "-12.022069245766284, -12.471729412530719, -12.92333656073672, -13.379726004769536, -13.797450373835208, "
    "-14.040187657707369, -14.28292494157953\n",
    "stripped_system.1.pdb, 1, 2, 0.5662269402609552, 0.5687603266150838, 1.1096497696890928, -, -, -, "
    "1.19923106015622, 13.935477752845646, , distance, 0.7544140306520318, 0.889421773622573, "
    "1.2300413217218367, 1.652024425811616, 2.066379454096729, 2.386876427150538, 2.7216152347220848, "
    "3.200038672825053, 3.6924886385112377, 4.09969639500886, 4.53955725440011, 4.9923858974449775, "
    "5.453462554381235, 5.926835223681634, 6.293101125707901, 6.487471115679011, 6.734772704805368, "
    "7.003248431807964, 7.280800263190667, 7.619889040569415, 7.940440167999238, 8.27270303391884, "
    "8.621173986948888, 8.908236787484757, 9.286434980590938, 9.677071567872883, 10.08040610411876, "
    "10.486966941078906, 10.879500348814005, 11.280539883197358\n",
    "stripped_system.1.pdb, 1, 2, 0.5662269402609552, 0.5687603266150838, 1.1096497696890928, -, -, -, "
    "1.19923106015622, 13.935477752845646, , length, 0.7544140306520318, 1.2544140306520317, "
    "1.7544140306520317, 2.2544140306520317, 2.7544140306520317, 3.2544140306520317, 3.7544140306520317, "
    "4.254414030652032, 4.754414030652032, 5.254414030652032, 5.754414030652032, 6.254414030652032, "
    "6.754414030652032, 7.254414030652032, 7.754414030652032, 8.254414030652033, 8.754414030652033, "
    "9.254414030652033, 9.754414030652033, 10.254414030652033, 10.754414030652033, 11.254414030652033, "
    "11.754414030652033, 12.254414030652033, 12.754414030652033, 13.254414030652033, 13.754414030652033, "
    "14.254414030652033, 14.754414030652033, 15.254414030652033\n",
    "stripped_system.1.pdb, 1, 2, 0.5662269402609552, 0.5687603266150838, 1.1096497696890928, -, -, -, "
    "1.19923106015622, 13.935477752845646, , R, 2.2067557209025823, 2.051383998226272, 1.9681242503950565, "
    "1.9462705626289207, 1.9288729385539092, 1.8715874488243498, 1.952328168820759, 2.0026649487252195, "
    "2.1032299960039804, 2.0034493989089137, 1.8938114672372006, 1.8598527140332897, 1.9070448771051554, "
    "1.9713289263410367, 2.032652730297512, 1.8324575024981373, 1.5922395880362556, 1.4221472678646763, "
    "1.2610425357639368, 1.1629135735312672, 1.1101064017784779, 1.1504492902881023, 1.2925152246062173, "
    "1.3432333318415461, 1.3789987880054542, 1.4844346527273264, 1.601270632999346, 1.7409697455591113, "
    "1.6077959280216805, 1.5600965386261822\n",
    "stripped_system.1.pdb, 1, 2, 0.5662269402609552, 0.5687603266150838, 1.1096497696890928, -, -, -, "
    "1.19923106015622, 13.935477752845646, , Upper limit of R overestimation, -, -, -, -, -, -, -, -, -, -, -, "
    "-, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -\n"
]

datasection1 = [
    "stripped_system.2.pdb, 1, 1, 0.6443086380845451, 0.43957741599811684, 1.594852937295866, -, -, -, "
    "1.1566240802231509, 14.155197558747808, , X, -3.502651221169928, -3.101447141319363, -2.6734538155477936,"
    " -2.2073817490976744, -1.8338839836594196, -1.4589959545490778, -1.0899315964811909, -0.7866209207020652,"
    " -0.4826724600539944, -0.17876761135698832, 0.1531413159834093, 0.24727179721777098, 0.33953953081358124,"
    " 0.4318072644093914, 0.5215138644569995, 0.565745733955586, 0.6118604920640812, 0.6580344005984449,"
    " 0.6762631194084107, 0.6716913569240399, 1.1176779966047772, 1.5532201862375012, 2.008050978402818,"
    " 2.430941142432723, 2.8682244044548764, 3.3055076664770295, 3.711705309676459, 4.012792358562298,"
    " 4.106136164780007, 4.289995013960793\n",
    "stripped_system.2.pdb, 1, 1, 0.6443086380845451, 0.43957741599811684, 1.594852937295866, -, -, -,"
    " 1.1566240802231509, 14.155197558747808, , Y, 2.02209392733789, 1.8027454820581006, 1.6369229002639258,"
    " 1.5485387420583834, 1.543083875436772, 1.5766685482756537, 1.6115982999233562, 1.6607219763055077, "
    "1.712032267855725, 1.611165063333656, 1.4196620489796692, 1.4732005411900868, 1.5280471379700356, "
    "1.5828937347499843, 1.638303007172626, 1.7037029820543104, 1.4264221178354366, 1.1383760296605918, "
    "0.8704235530698239, 0.7080624697545863, 0.7697147677821992, 0.8923856292411831, 0.9112590017960445, "
    "0.8203320269630932, 0.731191765535294, 0.6420515041074949, 0.8689540912206986, 1.0882369179485756, "
    "1.1027130886102559, 1.1215128366245022\n",
    "stripped_system.2.pdb, 1, 1, 0.6443086380845451, 0.43957741599811684, 1.594852937295866, -, -, -,"
    " 1.1566240802231509, 14.155197558747808, , Z, -9.004352202215454, -9.206642948085355, -9.390722973210227,"
    " -9.548287153847854, -9.78809156117294, -10.117099304483755, -10.451903496798746, -10.846351843705044, "
    "-11.239881079026247, -11.567206136477019, -11.843672194975237, -12.330778844190924, -12.819121465880604, "
    "-13.307464087570285, -13.796093790623202, -14.28982082553838, -14.698566137792767, -15.104641772690787, "
    "-15.525704664848544, -15.924560177168821, -16.105806376817135, -16.318538275691854, -16.520125541572973, "
    "-16.73547042960623, -16.960941852513542, -17.186413275420858, -17.209593077003934, -17.467402712316463, "
    "-17.958399020167924, -18.39804187522917\n",
    "stripped_system.2.pdb, 1, 1, 0.6443086380845451, 0.43957741599811684, 1.594852937295866, -, -, -,"
    " 1.1566240802231509, 14.155197558747808, , distance, 0.7469291498912518, 0.3459679240883162, "
    "0.4008037389005893, 0.8192788665456902, 1.2531688600439947, 1.7433620518327133, 2.236973910682348, "
    "2.7217713555487566, 3.2110434697589016, 3.6590900735197063, 4.100339035659819, 4.508613287669967, "
    "4.9333026280971115, 5.3709734558109234, 5.817454853950314, 6.251522287231107, 6.63026297810345, "
    "7.022266734277435, 7.4247901784440336, 7.793989978512233, 8.1523419365139, 8.538482889446815, "
    "8.945981695711266, 9.365750131543875, 9.807707708822882, 10.254998072983414, 10.499079217262677, "
    "10.871957800606364, 11.313311697314498, 11.772409174255642\n",
    "stripped_system.2.pdb, 1, 1, 0.6443086380845451, 0.43957741599811684, 1.594852937295866, -, -, -, "
    "1.1566240802231509, 14.155197558747808, , length, 0.7469291498912518, 1.2469291498912518, "
    "1.7469291498912518, 2.246929149891252, 2.746929149891252, 3.246929149891252, 3.746929149891252, "
    "4.246929149891252, 4.746929149891252, 5.246929149891252, 5.746929149891252, 6.246929149891252, "
    "6.746929149891252, 7.246929149891252, 7.746929149891252, 8.246929149891251, 8.746929149891251, "
    "9.246929149891251, 9.746929149891251, 10.246929149891251, 10.746929149891251, 11.246929149891251, "
    "11.746929149891251, 12.246929149891251, 12.746929149891251, 13.246929149891251, 13.746929149891251, "
    "14.246929149891251, 14.746929149891251, 15.246929149891251\n",
    "stripped_system.2.pdb, 1, 1, 0.6443086380845451, 0.43957741599811684, 1.594852937295866, -, -, -, "
    "1.1566240802231509, 14.155197558747808, , R, 2.0752245485517093, 1.9826909985648882, 1.9397907473173606, "
    "1.9363391887851449, 1.8749674894811956, 1.8044768813097438, 1.8101632904666747, 1.832146992054237, "
    "1.9337334151909287, 2.0455850117490275, 2.1404378846156007, 1.8968762011665399, 1.7165181115093093, "
    "1.6145183574053934, 1.5971849816068457, 1.6286710311191033, 1.633666161718119, 1.7219111087714156, "
    "1.8711786569608198, 1.9894394593559432, 1.895325110072671, 1.816707836872726, 1.7794294448620425, "
    "1.7587777925803187, 1.8143279343341445, 1.9492824207345747, 1.9586100207900883, 1.8952770601334274, "
    "1.841494416604735, 1.8139856105888483\n",
    "stripped_system.2.pdb, 1, 1, 0.6443086380845451, 0.43957741599811684, 1.594852937295866, -, -, -, "
    "1.1566240802231509, 14.155197558747808, , Upper limit of R overestimation, -, -, -, -, -, -, -, -, -, -, "
    "-, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -\n",
]
datasection2 = [
    "stripped_system.3590.pdb, 1, 1, 0.5671650786843087, 0.5671048735645572, 1.173287696427179, -, -, -, "
    "1.1248349566005307, 14.37838512501541, , X, -3.0918719950355924, -2.616332121871792, "
    "-2.1692346442307917, -1.8270707071576164, -1.484906770084441, -1.1389747420959169, -0.7648072913311019, "
    "-0.359571275706977, -0.005528452561227248, 0.3104463600678389, 0.40192529627198964, 0.5104122482254683, "
    "0.618899200178947, 0.7628469801808333, 0.9173559920763892, 1.07006562637637, 1.2052667003302062, "
    "1.38351933999242, 1.6185217398930758, 1.8535241397937319, 2.124685282242875, 2.530466355780498, "
    "2.674680512979762, 2.6593257445548146, 2.6091007747075636, 2.558875804860312, 2.4960241190203725, "
    "2.671364477290327, 2.8954349417048575, 2.879579369235922, 2.7755197461834897\n",
    "stripped_system.3590.pdb, 1, 1, 0.5671650786843087, 0.5671048735645572, 1.173287696427179, -, -, -, "
    "1.1248349566005307, 14.37838512501541, , Y, 1.4961436614049264, 1.4384051716518023, 1.3786360900919394, "
    "1.3113754740177532, 1.244114857943567, 1.2362316419497892, 1.2246660494741974, 1.1010461262996505, "
    "1.0805600984144212, 1.0769245806391856, 1.1390798061385945, 1.2419667181753664, 1.3448536302121383, "
    "1.2354326677683338, 1.389316362252548, 1.0610470755295838, 0.9786869312521009, 0.9014442175506461, "
    "0.5571066680861545, 0.21276911862166292, -0.061708020456717146, -0.07609304987779285, "
    "-0.1717108071605275, -0.30453919595613343, -0.4350591592167458, -0.5655791224773582, -0.679851613712605, "
    "-0.7322982384654073, -0.6741070092472432, -0.7739142591689165, -0.9417058499865076\n",
    "stripped_system.3590.pdb, 1, 1, 0.5671650786843087, 0.5671048735645572, 1.173287696427179, -, -, -, "
    "1.1248349566005307, 14.37838512501541, , Z, -9.076617092258951, -9.21989338420003, -9.409029682096703, "
    "-9.767358929705527, -10.125688177314348, -10.483709563006625, -10.764305465392853, -11.013033336132633, "
    "-11.363052932853883, -11.716558740128383, -12.199756740382512, -12.676878176274236, -13.153999612165961, "
    "-13.498841439294466, -13.924589669281563, -14.269436074201728, -14.65148043196701, -14.998793932079483, "
    "-15.274847412531587, -15.550900892983693, -15.830283846651458, -16.122062514972512, -16.547224768778705, "
    "-17.026678665170454, -17.506722404421673, -17.986766143672888, -18.419503116642016, -18.879777771215206, "
    "-19.281832340273024, -19.73500896416801, -20.168373188094417\n",
    "stripped_system.3590.pdb, 1, 1, 0.5671650786843087, 0.5671048735645572, 1.173287696427179, -, -, -, "
    "1.1248349566005307, 14.37838512501541, , distance, 0.37546544741029936, 0.4072904178857246, "
    "0.8292764068484743, 1.3031063753562226, 1.7912211175710684, 2.273125455106211, 2.732284195419423, "
    "3.218092146470542, 3.7088363281397783, 4.172329168081204, 4.554111346971607, 4.95872754549884, "
    "5.37940773509151, 5.745326406975855, 6.163810989115738, 6.55657005921, 6.9516398890746025, "
    "7.344306845213426, 7.748188677902097, 8.16272090298889, 8.59475293176708, 9.063461353997496, "
    "9.501401149236647, 9.903198523202407, 10.294964459908439, 10.695758293435638, 11.05478979448169, "
    "11.54180374949457, 11.979929972944259, 12.382574319596213, 12.747303931398303\n",
    "stripped_system.3590.pdb, 1, 1, 0.5671650786843087, 0.5671048735645572, 1.173287696427179, -, -, -, "
    "1.1248349566005307, 14.37838512501541, , length, 0.37546544741029936, 0.8754654474102994, "
    "1.3754654474102994, 1.8754654474102994, 2.375465447410299, 2.875465447410299, 3.375465447410299, "
    "3.875465447410299, 4.375465447410299, 4.875465447410299, 5.375465447410299, 5.875465447410299, "
    "6.375465447410299, 6.875465447410299, 7.375465447410299, 7.875465447410299, 8.3754654474103, "
    "8.8754654474103, 9.3754654474103, 9.8754654474103, 10.3754654474103, 10.8754654474103, 11.3754654474103, "
    "11.8754654474103, 12.3754654474103, 12.8754654474103, 13.3754654474103, 13.8754654474103, "
    "14.3754654474103, 14.8754654474103, 15.3754654474103\n",
    "stripped_system.3590.pdb, 1, 1, 0.5671650786843087, 0.5671048735645572, 1.173287696427179, -, -, -, "
    "1.1248349566005307, 14.37838512501541, , R, 2.058332603177, 1.9595026959835977, 1.9089659688329006, "
    "1.8291131985218916, 1.8307398673521875, 1.8645667662334395, 1.8575953778080219, 1.84624025160103, "
    "1.7609328732582423, 1.724638982400699, 1.5379246598681424, 1.414468712393252, 1.38353669473696, "
    "1.356343238115499, 1.3636392232311232, 1.2149238679111323, 1.1952606118078226, 1.1915903204914529, "
    "1.1819141672230191, 1.2753436001830212, 1.420659543623078, 1.5045034225713676, 1.5366910516363192, "
    "1.5728416243101495, 1.6741714021129839, 1.8550076634832322, 2.0444883096317774, 2.0870211054357792, "
    "2.1007660783531588, 2.1763593458251265, 2.186992444956825\n",
    "stripped_system.3590.pdb, 1, 1, 0.5671650786843087, 0.5671048735645572, 1.173287696427179, -, -, -, "
    "1.1248349566005307, 14.37838512501541, , Upper limit of R overestimation, -, -, -, -, -, -, -, -, -, -, "
    "-, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -\n",
]
datasection3 = [
    "stripped_system.9881.pdb, 1, 1, 0.6184593557197385, 0.48052380358577024, 1.5270199148154504, -, -, -, "
    "1.2361080659231054, 14.637354970685315, , X, -2.925182474416817, -2.7330474125501256, -2.53122436827732, "
    "-2.2365242049757508, -1.8977352957681122, -1.4985456072757755, -1.1006941922344662, -0.7239388120171892, "
    "-0.34313818722320444, -0.039617853272864545, 0.2553961840164098, 0.5077610249120005, 0.7540381053446046, "
    "1.013028650240976, 0.9992106551317869, 1.0184554595569055, 1.095466897565474, 1.1501465648911156, "
    "1.2947798558711587, 1.4212548507688245, 1.5274315816899757, 1.6262103107912693, 1.5190731595409128, "
    "1.3952287367509886, 1.2713843139610643, 1.14753989117114, 1.05633670340256, 1.0122794955285683, "
    "0.8584383614534996, 1.0728097500262543, 1.1688194740628604\n",
    "stripped_system.9881.pdb, 1, 1, 0.6184593557197385, 0.48052380358577024, 1.5270199148154504, -, -, -, "
    "1.2361080659231054, 14.637354970685315, , Y, 1.002224147387297, 1.341638428111183, 1.7467122382206768, "
    "1.922970722077312, 1.870888240196416, 1.9164441845029327, 1.9627629753338554, 2.098653726685382, "
    "2.269781389092218, 2.139236043142967, 1.8380121823028932, 1.6346046598796848, 1.4649868970290985, "
    "1.2816140481205465, 1.5031100225384981, 1.735808864925084, 1.7614167452112177, 1.362523043616748, "
    "1.0500471170375372, 0.7003398818834905, 0.3598633253854916, 0.02335234618037761, -0.1587997177068782, "
    "-0.32816388419353515, -0.49752805068019207, -0.666892217166849, -0.7110311304298167, "
    "-0.7310979391859432, -0.7894940887558838, -0.5362769026062449, -0.4970313699008958\n",
    "stripped_system.9881.pdb, 1, 1, 0.6184593557197385, 0.48052380358577024, 1.5270199148154504, -, -, -, "
    "1.2361080659231054, 14.637354970685315, , Z, -8.785059051207206, -9.063677298882233, -9.276239447826573, "
    "-9.590735921712621, -9.954742455435754, -10.252353356680539, -10.551408481214008, -10.844681974469154, "
    "-11.119577507941118, -11.40782468697083, -11.67559601597133, -12.017839256755668, -12.416580961709187, "
    "-12.798276340294272, -13.246326110216994, -13.685338056487062, -14.061769540793383, -14.358238677145584, "
    "-14.713756823788474, -15.04612538719024, -15.396556858787987, -15.752656005852776, -16.19954005150748, "
    "-16.653388416111905, -17.107236780716327, -17.561085145320753, -18.028470837922523, -18.51812316873699, "
    "-18.96994462234458, -19.30953758157295, -19.79469778573809\n",
    "stripped_system.9881.pdb, 1, 1, 0.6184593557197385, 0.48052380358577024, 1.5270199148154504, -, -, -, "
    "1.2361080659231054, 14.637354970685315, , distance, 0.7861699158201562, 0.40826224873053296, "
    "0.37354756315332205, 0.8239847426668793, 1.3070219168769623, 1.802899340179019, 2.3005954417410006, "
    "2.7894981542784296, 3.2774234683344456, 3.674108275760017, 4.051888765923671, 4.46764340620837, "
    "4.9257111266254805, 5.389153918707232, 5.688721829927953, 6.028912409540521, 6.371129182589209, "
    "6.651034821378445, 7.046477618354808, 7.429822816558858, 7.827949516251937, 8.236999248093573, "
    "8.589345085139056, 8.950223698365493, 9.323956215575235, 9.709058387358935, 10.097554181664513, "
    "10.524904285056428, 10.894584614432395, 11.223235092420403, 11.694001693038642\n",
    "stripped_system.9881.pdb, 1, 1, 0.6184593557197385, 0.48052380358577024, 1.5270199148154504, -, -, -, "
    "1.2361080659231054, 14.637354970685315, , length, 0.7861699158201562, 1.2861699158201563, "
    "1.7861699158201563, 2.2861699158201563, 2.7861699158201563, 3.2861699158201563, 3.7861699158201563, "
    "4.286169915820157, 4.786169915820157, 5.286169915820157, 5.786169915820157, 6.286169915820157, "
    "6.786169915820157, 7.286169915820157, 7.786169915820157, 8.286169915820157, 8.786169915820157, "
    "9.286169915820157, 9.786169915820157, 10.286169915820157, 10.786169915820157, 11.286169915820157, "
    "11.786169915820157, 12.286169915820157, 12.786169915820157, 13.286169915820157, 13.786169915820157, "
    "14.286169915820157, 14.786169915820157, 15.286169915820157, 15.786169915820157\n",
    "stripped_system.9881.pdb, 1, 1, 0.6184593557197385, 0.48052380358577024, 1.5270199148154504, -, -, -, "
    "1.2361080659231054, 14.637354970685315, , R, 1.9173052895743654, 1.8449327704692318, 1.8385100531557483, "
    "1.7802378386385607, 1.690055709028946, 1.6437562158560557, 1.683038630064577, 1.7242707438997609, "
    "1.8261889997564682, 1.7946342338204502, 1.7100874579481278, 1.659608941584519, 1.6684520703454646, "
    "1.7639431742386955, 1.7357332854176717, 1.764147822259002, 1.734724939336352, 1.6349683299446867, "
    "1.56936018053242, 1.5281000695451332, 1.5596671637630826, 1.6729335978980024, 1.688380316210157, "
    "1.7742588505169634, 1.9384216684006448, 2.169445863564941, 2.155354874035865, 2.022247822462984, "
    "1.830431014875116, 1.8247777708723392, 1.7131233488584272\n",
    "stripped_system.9881.pdb, 1, 1, 0.6184593557197385, 0.48052380358577024, 1.5270199148154504, -, -, -, "
    "1.2361080659231054, 14.637354970685315, , Upper limit of R overestimation, -, -, -, -, -, -, -, -, -, -, "
    "-, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -\n",
]

test_create_from_data_section_out = "(x, y, z, distance, radius, length) = (-9.12, -31.31, 58.42, 0.75, 2.21, 0.75)-(x, y, z, distance, radius, length) = (-9.08, -31.60, 58.02, 0.89, 2.05, 1.25)-(x, y, z, distance, radius, length) = (-9.03, -31.89, 57.61, 1.23, 1.97, 1.75)-(x, y, z, distance, radius, length) = (-9.01, -32.18, 57.21, 1.65, 1.95, 2.25)-(x, y, z, distance, radius, length) = (-9.03, -32.50, 56.85, 2.07, 1.93, 2.75)-(x, y, z, distance, radius, length) = (-9.04, -32.88, 56.65, 2.39, 1.87, 3.25)-(x, y, z, distance, radius, length) = (-9.24, -32.89, 56.30, 2.72, 1.95, 3.75)-(x, y, z, distance, radius, length) = (-9.35, -33.03, 55.86, 3.20, 2.00, 4.25)-(x, y, z, distance, radius, length) = (-9.44, -33.14, 55.38, 3.69, 2.10, 4.75)-(x, y, z, distance, radius, length) = (-9.72, -33.36, 55.06, 4.10, 2.00, 5.25)-(x, y, z, distance, radius, length) = (-9.93, -33.70, 54.76, 4.54, 1.89, 5.75)-(x, y, z, distance, radius, length) = (-10.13, -34.04, 54.46, 4.99, 1.86, 6.25)-(x, y, z, distance, radius, length) = (-10.34, -34.37, 54.15, 5.45, 1.91, 6.75)-(x, y, z, distance, radius, length) = (-10.59, -34.61, 53.80, 5.93, 1.97, 7.25)-(x, y, z, distance, radius, length) = (-10.66, -34.85, 53.52, 6.29, 2.03, 7.75)-(x, y, z, distance, radius, length) = (-10.62, -35.30, 53.54, 6.49, 1.83, 8.25)-(x, y, z, distance, radius, length) = (-10.73, -35.78, 53.59, 6.73, 1.59, 8.75)-(x, y, z, distance, radius, length) = (-10.83, -36.26, 53.65, 7.00, 1.42, 9.25)-(x, y, z, distance, radius, length) = (-10.86, -36.76, 53.71, 7.28, 1.26, 9.75)-(x, y, z, distance, radius, length) = (-10.87, -37.25, 53.72, 7.62, 1.16, 10.25)-(x, y, z, distance, radius, length) = (-10.94, -37.73, 53.78, 7.94, 1.11, 10.75)-(x, y, z, distance, radius, length) = (-10.98, -38.22, 53.85, 8.27, 1.15, 11.25)-(x, y, z, distance, radius, length) = (-11.02, -38.71, 53.92, 8.62, 1.29, 11.75)-(x, y, z, distance, radius, length) = (-11.19, -39.08, 54.00, 8.91, 1.34, 12.25)-(x, y, z, distance, radius, length) = (-11.50, -39.47, 54.03, 9.29, 1.38, 12.75)-(x, y, z, distance, radius, length) = (-11.82, -39.85, 54.05, 9.68, 1.48, 13.25)-(x, y, z, distance, radius, length) = (-12.19, -40.19, 54.02, 10.08, 1.60, 13.75)-(x, y, z, distance, radius, length) = (-12.50, -40.54, 53.99, 10.49, 1.74, 14.25)-(x, y, z, distance, radius, length) = (-12.50, -41.04, 54.03, 10.88, 1.61, 14.75)-(x, y, z, distance, radius, length) = (-12.49, -41.54, 54.08, 11.28, 1.56, 15.25)-"



test_get_csv_lines_out = """md_label, stripped_system.1.pdb, 1, 2, 0.5662269402609552, 0.5687603266150838, 1.1096497696890928, -, -, -, 1.19923106015622, 13.935477752845646, , X, -9.118836752231328, -9.0751308876552, -9.031425023079075, -9.00740647206823, -9.026769396285047, -9.044838308664206, -9.235407174890145, -9.35166084637227, -9.439608863246228, -9.72308069682723, -9.925890364369554, -10.1314592235335, -10.337028082697445, -10.586793547138628, -10.658953882915165, -10.621784852389869, -10.729883544910285, -10.833145481039788, -10.85582317390352, -10.874283913411146, -10.939962081355985, -10.979349016059885, -11.018735950763787, -11.194012513190541, -11.49885802254482, -11.818993352845993, -12.187182756964264, -12.498205615113775, -12.496337328623227, -12.494469042132677
md_label, stripped_system.1.pdb, 1, 2, 0.5662269402609552, 0.5687603266150838, 1.1096497696890928, -, -, -, 1.19923106015622, 13.935477752845646, , Y, -31.31334549939548, -31.600923148643805, -31.888500797892128, -32.17936879603873, -32.50331487745197, -32.88107417093713, -32.88667427331918, -33.03140109193857, -33.136346626607136, -33.35734344125713, -33.699013851852776, -34.036462584012135, -34.373911316171494, -34.61013774294749, -34.853634173766466, -35.2954483554302, -35.780083039321234, -36.264364328771954, -36.7552798232639, -37.25355735645561, -37.72852194647673, -38.22137685879521, -38.71423177111368, -39.075800190756375, -39.46991755422525, -39.850472360176475, -40.18623938293266, -40.540622587785066, -41.03838443360603, -41.536146279427
md_label, stripped_system.1.pdb, 1, 2, 0.5662269402609552, 0.5687603266150838, 1.1096497696890928, -, -, -, 1.19923106015622, 13.935477752845646, , Z, 58.42423123723586, 58.01781190121856, 57.61139256520126, 57.20705853817803, 56.85474190017667, 56.6507623408354, 56.30206044625762, 55.85527292861378, 55.377321745156095, 55.064502679945065, 54.762146888504816, 54.45656339493469, 54.15097990136456, 53.797050616356074, 53.519731158806664, 53.53608996425048, 53.59200557819357, 53.65376849392268, 53.71369031216983, 53.721169703533, 53.775017827131215, 53.847563350215765, 53.920108873300315, 53.99652378550957, 54.031950608176174, 54.05110095224063, 54.01792505727895, 53.987205316541186, 54.031448811682914, 54.07569230682464
md_label, stripped_system.1.pdb, 1, 2, 0.5662269402609552, 0.5687603266150838, 1.1096497696890928, -, -, -, 1.19923106015622, 13.935477752845646, , distance, 0.7544140306520318, 0.889421773622573, 1.2300413217218367, 1.652024425811616, 2.066379454096729, 2.386876427150538, 2.7216152347220848, 3.200038672825053, 3.6924886385112377, 4.09969639500886, 4.53955725440011, 4.9923858974449775, 5.453462554381235, 5.926835223681634, 6.293101125707901, 6.487471115679011, 6.734772704805368, 7.003248431807964, 7.280800263190667, 7.619889040569415, 7.940440167999238, 8.27270303391884, 8.621173986948888, 8.908236787484757, 9.286434980590938, 9.677071567872883, 10.08040610411876, 10.486966941078906, 10.879500348814005, 11.280539883197358
md_label, stripped_system.1.pdb, 1, 2, 0.5662269402609552, 0.5687603266150838, 1.1096497696890928, -, -, -, 1.19923106015622, 13.935477752845646, , length, 0.7544140306520318, 1.2544140306520317, 1.7544140306520317, 2.2544140306520317, 2.7544140306520317, 3.2544140306520317, 3.7544140306520317, 4.254414030652032, 4.754414030652032, 5.254414030652032, 5.754414030652032, 6.254414030652032, 6.754414030652032, 7.254414030652032, 7.754414030652032, 8.254414030652033, 8.754414030652033, 9.254414030652033, 9.754414030652033, 10.254414030652033, 10.754414030652033, 11.254414030652033, 11.754414030652033, 12.254414030652033, 12.754414030652033, 13.254414030652033, 13.754414030652033, 14.254414030652033, 14.754414030652033, 15.254414030652033
md_label, stripped_system.1.pdb, 1, 2, 0.5662269402609552, 0.5687603266150838, 1.1096497696890928, -, -, -, 1.19923106015622, 13.935477752845646, , R, 2.2067557209025823, 2.051383998226272, 1.9681242503950565, 1.9462705626289207, 1.9288729385539092, 1.8715874488243498, 1.952328168820759, 2.0026649487252195, 2.1032299960039804, 2.0034493989089137, 1.8938114672372006, 1.8598527140332897, 1.9070448771051554, 1.9713289263410367, 2.032652730297512, 1.8324575024981373, 1.5922395880362556, 1.4221472678646763, 1.2610425357639368, 1.1629135735312672, 1.1101064017784779, 1.1504492902881023, 1.2925152246062173, 1.3432333318415461, 1.3789987880054542, 1.4844346527273264, 1.601270632999346, 1.7409697455591113, 1.6077959280216805, 1.5600965386261822
md_label, stripped_system.1.pdb, 1, 2, 0.5662269402609552, 0.5687603266150838, 1.1096497696890928, -, -, -, 1.19923106015622, 13.935477752845646, , Upper limit of R overestimation, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -, -
"""

test_get_center_line_out = np.array([[-9.118836752231328, -31.31334549939548, 58.42423123723586],
                                     [-9.0751308876552, -31.600923148643805, 58.01781190121856],
                                     [-9.031425023079075, -31.888500797892128, 57.61139256520126],
                                     [-9.00740647206823, -32.17936879603873, 57.20705853817803],
                                     [-9.026769396285047, -32.50331487745197, 56.85474190017667],
                                     [-9.044838308664206, -32.88107417093713, 56.6507623408354],
                                     [-9.235407174890145, -32.88667427331918, 56.30206044625762],
                                     [-9.35166084637227, -33.03140109193857, 55.85527292861378],
                                     [-9.439608863246228, -33.136346626607136, 55.377321745156095],
                                     [-9.72308069682723, -33.35734344125713, 55.064502679945065],
                                     [-9.925890364369554, -33.699013851852776, 54.762146888504816],
                                     [-10.1314592235335, -34.036462584012135, 54.45656339493469],
                                     [-10.337028082697445, -34.373911316171494, 54.15097990136456],
                                     [-10.586793547138628, -34.61013774294749, 53.797050616356074],
                                     [-10.658953882915165, -34.853634173766466, 53.519731158806664],
                                     [-10.621784852389869, -35.2954483554302, 53.53608996425048],
                                     [-10.729883544910285, -35.780083039321234, 53.59200557819357],
                                     [-10.833145481039788, -36.264364328771954, 53.65376849392268],
                                     [-10.85582317390352, -36.7552798232639, 53.71369031216983],
                                     [-10.874283913411146, -37.25355735645561, 53.721169703533],
                                     [-10.939962081355985, -37.72852194647673, 53.775017827131215],
                                     [-10.979349016059885, -38.22137685879521, 53.847563350215765],
                                     [-11.018735950763787, -38.71423177111368, 53.920108873300315],
                                     [-11.194012513190541, -39.075800190756375, 53.99652378550957],
                                     [-11.49885802254482, -39.46991755422525, 54.031950608176174],
                                     [-11.818993352845993, -39.850472360176475, 54.05110095224063],
                                     [-12.187182756964264, -40.18623938293266, 54.01792505727895],
                                     [-12.498205615113775, -40.540622587785066, 53.987205316541186],
                                     [-12.496337328623227, -41.03838443360603, 54.031448811682914],
                                     [-12.494469042132677, -41.536146279427, 54.07569230682464]]
                                    )

test_tunnel_get_points_data_out = np.array(
    [[3.9231396368321483, 3.8686058563233057, -7.909170286835895, 0.7544140306520318, 2.2067557209025823, 0.0],
     [4.409257133292514, 3.752867518897115, -7.892011706331463, 0.889421773622573, 2.051383998226272, 1.0],
     [4.89537462975288, 3.637129181470924, -7.8748531258270305, 1.2300413217218367, 1.9681242503950565, 2.0],
     [5.382361184870531, 3.5287962069899437, -7.876354124020829, 1.652024425811616, 1.9462705626289207, 3.0],
     [5.839344475388041, 3.401352046966683, -7.944349902916014, 2.066379454096729, 1.9288729385539092, 4.0],
     [6.184970661388952, 3.18336363647974, -8.078075541321214, 2.386876427150538, 1.8715874488243498, 5.0],
     [6.515655695208539, 3.394821686792071, -8.142641644910547, 2.7216152347220848, 1.952328168820759, 6.0],
     [6.988791860273927, 3.489670307071897, -8.181969814274334, 3.200038672825053, 2.0026649487252195, 7.0],
     [7.471285868236148, 3.6102417115136114, -8.170216159546937, 3.6924886385112377, 2.1032299960039804, 8.0],
     [7.867203774242666, 3.692794351918671, -8.423263346652227, 4.09969639500886, 2.0034493989089137, 9.0],
     [8.299213096978562, 3.6345334925674697, -8.667697280765092, 4.53955725440011, 1.8938114672372006, 10.0],
     [8.732541757245892, 3.5820909642740975, -8.911574408469024, 4.9923858974449775, 1.8598527140332897, 11.0],
     [9.165870417513222, 3.5296484359807248, -9.155451536172954, 5.453462554381235, 1.9070448771051554, 12.0],
     [9.603066214174294, 3.595445922301469, -9.375703690279577, 5.926835223681634, 1.9713289263410367, 13.0],
     [9.961835583879651, 3.5348252846786803, -9.471692404259596, 6.293101125707901, 2.032652730297512, 14.0],
     [10.13164387613461, 3.1658798578518517, -9.650863753497099, 6.487471115679011, 1.8324575024981373, 15.0],
     [10.293875194817657, 2.8293373117486222, -9.98315897449902, 6.734772704805368, 1.5922395880362556, 16.0],
     [10.450338049067156, 2.4885638009030484, -10.312871843695083, 7.003248431807964, 1.4221472678646763, 17.0],
     [10.605709002184067, 2.0996803567254805, -10.577454951646743, 7.280800263190667, 1.2610425357639368, 18.0],
     [10.811303743207105, 1.7197623469364594, -10.827163906771256, 7.619889040569415, 1.1629135735312672, 19.0],
     [10.96837706784503, 1.3684849827671266, -11.11873393201795, 7.940440167999238, 1.1101064017784779, 20.0],
     [11.114314712886188, 0.9830381299608426, -11.401811531664357, 8.27270303391884, 1.1504492902881023, 21.0],
     [11.260252357927346, 0.5975912771545585, -11.684889131310765, 8.621173986948888, 1.2925152246062173, 22.0],
     [11.356463949875144, 0.38641732216490915, -12.022069245766284, 8.908236787484757, 1.3432333318415461, 23.0],
     [11.512457789759821, 0.23321269970391703, -12.471729412530719, 9.286434980590938, 1.3789987880054542, 24.0],
     [11.678472746970444, 0.10408379991582889, -12.92333656073672, 9.677071567872883, 1.4844346527273264, 25.0],
     [11.876125411810433, 0.05268244334746751, -13.379726004769536, 10.08040610411876, 1.601270632999346, 26.0],
     [12.075495725928016, -0.0448822031970907, -13.797450373835208, 10.486966941078906, 1.7409697455591113, 27.0],
     [12.246241188364774, -0.4472807594514884, -14.040187657707369, 10.879500348814005, 1.6077959280216805, 28.0],
     [12.41698665080153, -0.849679315705886, -14.28292494157953, 11.280539883197358, 1.5600965386261822, -29.0]])

test_get_closest_sphere2coords1 = (-1.536708417, np.array([-9.439608863, -33.136346626, 55.37732174515,
                                                           3.692488638, 2.103229996, 4.754414030]))
test_get_closest_sphere2coords2 = (11.629743707, np.array([-11.018735950, -38.7142317711, 53.920108873, 8.621173986,
                                                           1.292515224, 11.754414030]))
test_get_closest_sphere2coords3 = (1.906485405, np.array([-9.439608863, -33.136346626, 55.377321745, 3.692488638,
                                                          2.103229996, 4.754414030]))

test_get_pdb_file_format_out = ['ATOM      1  H   FIL T   1      -9.119 -31.313  58.424        2.21\n',
                                'ATOM      2  H   FIL T   1      -9.075 -31.601  58.018        2.05\n',
                                'CONECT    1    2\n',
                                'ATOM      3  H   FIL T   1      -9.031 -31.889  57.611        1.97\n',
                                'CONECT    2    3\n',
                                'ATOM      4  H   FIL T   1      -9.007 -32.179  57.207        1.95\n',
                                'CONECT    3    4\n',
                                'ATOM      5  H   FIL T   1      -9.027 -32.503  56.855        1.93\n',
                                'CONECT    4    5\n',
                                'ATOM      6  H   FIL T   1      -9.045 -32.881  56.651        1.87\n',
                                'CONECT    5    6\n',
                                'ATOM      7  H   FIL T   1      -9.235 -32.887  56.302        1.95\n',
                                'CONECT    6    7\n',
                                'ATOM      8  H   FIL T   1      -9.352 -33.031  55.855        2.00\n',
                                'CONECT    7    8\n',
                                'ATOM      9  H   FIL T   1      -9.440 -33.136  55.377        2.10\n',
                                'CONECT    8    9\n',
                                'ATOM     10  H   FIL T   1      -9.723 -33.357  55.065        2.00\n',
                                'CONECT    9   10\n',
                                'ATOM     11  H   FIL T   1      -9.926 -33.699  54.762        1.89\n',
                                'CONECT   10   11\n',
                                'ATOM     12  H   FIL T   1     -10.131 -34.036  54.457        1.86\n',
                                'CONECT   11   12\n',
                                'ATOM     13  H   FIL T   1     -10.337 -34.374  54.151        1.91\n',
                                'CONECT   12   13\n',
                                'ATOM     14  H   FIL T   1     -10.587 -34.610  53.797        1.97\n',
                                'CONECT   13   14\n',
                                'ATOM     15  H   FIL T   1     -10.659 -34.854  53.520        2.03\n',
                                'CONECT   14   15\n',
                                'ATOM     16  H   FIL T   1     -10.622 -35.295  53.536        1.83\n',
                                'CONECT   15   16\n',
                                'ATOM     17  H   FIL T   1     -10.730 -35.780  53.592        1.59\n',
                                'CONECT   16   17\n',
                                'ATOM     18  H   FIL T   1     -10.833 -36.264  53.654        1.42\n',
                                'CONECT   17   18\n',
                                'ATOM     19  H   FIL T   1     -10.856 -36.755  53.714        1.26\n',
                                'CONECT   18   19\n',
                                'ATOM     20  H   FIL T   1     -10.874 -37.254  53.721        1.16\n',
                                'CONECT   19   20\n',
                                'ATOM     21  H   FIL T   1     -10.940 -37.729  53.775        1.11\n',
                                'CONECT   20   21\n',
                                'ATOM     22  H   FIL T   1     -10.979 -38.221  53.848        1.15\n',
                                'CONECT   21   22\n',
                                'ATOM     23  H   FIL T   1     -11.019 -38.714  53.920        1.29\n',
                                'CONECT   22   23\n',
                                'ATOM     24  H   FIL T   1     -11.194 -39.076  53.997        1.34\n',
                                'CONECT   23   24\n',
                                'ATOM     25  H   FIL T   1     -11.499 -39.470  54.032        1.38\n',
                                'CONECT   24   25\n',
                                'ATOM     26  H   FIL T   1     -11.819 -39.850  54.051        1.48\n',
                                'CONECT   25   26\n',
                                'ATOM     27  H   FIL T   1     -12.187 -40.186  54.018        1.60\n',
                                'CONECT   26   27\n',
                                'ATOM     28  H   FIL T   1     -12.498 -40.541  53.987        1.74\n',
                                'CONECT   27   28\n',
                                'ATOM     29  H   FIL T   1     -12.496 -41.038  54.031        1.61\n',
                                'CONECT   28   29\n',
                                'ATOM     30  H   FIL T   1     -12.494 -41.536  54.076        1.56\n',
                                'CONECT   29   30\n']


test_create_layered_cluster_pathset = """LayeredPathSet: Cluster_1 of md_label
Paths: 
 0: ['SP' '6_1' '7_1' '8_1' '9_1' '10_1' '11_1' '12_2']
 1: ['SP' '6_1' '7_1' '8_1' '9_1' '10_1' '11_1' '12_1' '13_1']
num nodes = 10
Labels 10:
['SP', '6_1', '7_1', '8_1', '9_1', '10_1', '11_1', '12_2', '12_1', '13_1']
Data:
[[  0.           0.           0.          -1.           0.
    0.5          0.5       ]
 [ -2.75518269   1.54140608  -9.19805212   6.           0.
    2.0600218    0.52088423]
 [ -0.84240646   1.61039093 -10.7493106    7.           0.
    1.92785682   1.07040831]
 [  0.53913952   1.39478569 -12.46560454   8.           0.
    1.81711639   0.40993798]
 [  0.87595866   1.41239745 -13.97540866   9.           0.
    1.73543078   0.6433359 ]
 [  1.35611361   0.54100296 -15.4405071   10.           0.
    1.88300474   0.69633191]
 [  2.30093247   0.25533373 -16.82676439  11.           0.
    1.95394622   1.17612067]
 [  4.19806559   1.11211296 -18.17822045  12.           1.
    1.83874354   0.2384551 ]
 [  1.78087903  -0.67188995 -18.23821582  12.           0.
    2.1220949    0.78537482]
 [  2.04599516  -0.70640396 -19.44845318  13.           1.
    2.18061259   0.99481158]]"""

test_event_get_points_data_out1 = np.array([[48.436911316, 40.073945941, 28.523349441, 6.41, 0.0, 0.0, 0.0],
                                            [48.261014456, 40.604420197, 28.337149724, 6.16, 0.0, 1.0, 0.0],
                                            [50.049451279, 39.493132374, 27.003772120, 8.58, 0.0, -2.0, 0.0]])
test_event_get_points_data_out2 = np.array([[51.833651922, 38.389620817, 25.675052447, 11.02, 0.0, 0.0, 0.0],
                                            [51.933849410, 38.036954378, 24.871623114, 11.61, 0.0, 1.0, 0.0],
                                            [52.346492650, 39.812218024, 24.495640830, 11.64, 0.0, 2.0, 0.0],
                                            [54.206242159, 38.963708905, 28.874061535, 11.98, 0.0, 3.0, 0.0],
                                            [56.004645608, 38.634699053, 30.371621350, 13.65, 0.0, 4.0, 0.0],
                                            [55.516719365, 38.446934292, 28.886676014, 13.36, 0.0, 5.0, 0.0],
                                            [54.217056108, 38.704428815, 28.994566762, 12.04, 0.0, 6.0, 0.0],
                                            [55.238327136, 38.242146055, 29.126161061, 13.12, 0.0, 7.0, 0.0],
                                            [51.281674137, 38.781589159, 25.894308351, 10.33, 0.0, 8.0, 0.0],
                                            [52.062629043, 38.232191276, 25.866837557, 11.17, 0.0, 9.0, 0.0],
                                            [55.015853888, 38.606293121, 28.960465461, 12.83, 0.0, 10.0, 0.0],
                                            [56.030032289, 36.533217831, 28.938480437, 14.47, 0.0, 11.0, 0.0],
                                            [56.966220933, 36.582335333, 30.955348172, 15.2, 0.0, 12.0, 0.0],
                                            [56.550850792, 36.551724217, 29.660903588, 14.87, 0.0, -13.0, 0.0]])
test_event_get_points_data_out3 = np.array([[53.600714083, 38.756084976, 24.455105441, 12.95, 0.0, -3.0, 0.0],
                                            [52.994253355, 39.192799059, 25.152181290, 11.99, 0.0, 2.0, 0.0],
                                            [49.705671324, 41.167422199, 29.484756820, 7.06, 0.0, 1.0, 0.0],
                                            [49.067117790, 40.621768126, 28.998445142, 6.67, 0.0, 0.0, 0.0]])

test_create_layered_event_pathset1 = """LayeredPathSet: 1_inside of e10s1_e9s3p0f1600
Paths: 
 0: ['4_0' '5_0']
num nodes = 2
Labels 2:
['4_0', '5_0']
Data:
[[-1.345      -4.87       -3.72        4.          0.          0.
   0.29453904]
 [-2.2        -6.91       -4.58        5.          1.          0.
   0.        ]]"""

test_create_layered_event_pathset2 = """LayeredPathSet: 1_release of e10s1_e9s3p0f1600
Paths: 
 0: ['7_0' '9_1' '10_0' '9_1']
num nodes = 4
Labels 4:
['7_0', '9_1', '10_0', '9_1']
Data:
[[ -2.81166667  -8.95833333  -6.03833333   7.           0.
    0.           1.78229525]
 [  0.386       -9.666       -9.924        9.           1.
    0.           1.18308079]
 [  2.21       -10.26       -11.          10.           0.
    0.           0.        ]
 [  0.386       -9.666       -9.924        9.           1.
    0.           1.18308079]]"""

test_create_layered_event_pathset3 = """LayeredPathSet: 1_entry of e10s1_e9s3p0f1600
Paths: 
 0: ['4_0' '5_0' '7_0' '8_0']
num nodes = 4
Labels 4:
['4_0', '5_0', '7_0', '8_0']
Data:
[[ -1.07  -4.69  -4.62   4.     0.     0.     0.  ]
 [ -1.    -4.31  -5.51   5.     0.     0.     0.  ]
 [ -3.79  -9.25  -6.62   7.     0.     0.     0.  ]
 [ -4.18 -10.17  -6.83   8.     1.     0.     0.  ]]"""

cgo1 = [2.0, 1.0, 6.0, (1.0, 0.0, 0.0), 4.0, 53.75986862182617, 40.71662902832031, 25.09145736694336, 4.0,
        53.01210403442383, 41.04368591308594, 25.707672119140625, 4.0, 53.01210403442383, 41.04368591308594,
        25.707672119140625, 4.0, 48.98432922363281, 42.46281051635742, 29.608217239379883, 4.0, 48.98432922363281,
        42.46281051635742, 29.608217239379883, 4.0, 48.54757499694824, 41.742483139038086, 29.139482498168945, 6.0,
        (0.0, 0.5, 0.0), 4.0, 48.54757499694824, 41.742483139038086, 29.139482498168945, 4.0, 48.54757499694824,
        41.742483139038086, 29.139482498168945, 4.0, 48.54757499694824, 41.742483139038086, 29.139482498168945, 4.0,
        48.11082077026367, 41.02215576171875, 28.670747756958008, 4.0, 48.11082077026367, 41.02215576171875,
        28.670747756958008, 4.0, 47.82691955566406, 41.47747039794922, 28.426923751831055, 4.0, 47.82691955566406,
        41.47747039794922, 28.426923751831055, 4.0, 49.93216133117676, 40.747554779052734, 27.3231258392334, 6.0,
        (0.0, 0.0, 1.0), 4.0, 49.93216133117676, 40.747554779052734, 27.3231258392334, 4.0, 49.93216133117676,
        40.747554779052734, 27.3231258392334, 4.0, 49.93216133117676, 40.747554779052734, 27.3231258392334, 4.0,
        52.03740310668945, 40.01763916015625, 26.219327926635742, 4.0, 52.03740310668945, 40.01763916015625,
        26.219327926635742, 4.0, 52.28764724731445, 39.64469909667969, 25.468948364257812, 4.0, 52.28764724731445,
        39.64469909667969, 25.468948364257812, 4.0, 52.29142761230469, 41.42877960205078, 24.954166412353516, 4.0,
        52.29142761230469, 41.42877960205078, 24.954166412353516, 4.0, 53.925289154052734, 41.401084899902344,
        29.50446319580078, 4.0, 53.925289154052734, 41.401084899902344, 29.50446319580078, 4.0, 55.60820388793945,
        41.64215087890625, 31.134807586669922, 4.0, 55.60820388793945, 41.64215087890625, 31.134807586669922, 4.0,
        55.31340026855469, 41.22490310668945, 29.652244567871094, 4.0, 55.31340026855469, 41.22490310668945,
        29.652244567871094, 4.0, 53.98349380493164, 41.16704559326172, 29.647794723510742, 4.0, 53.98349380493164,
        41.16704559326172, 29.647794723510742, 4.0, 55.06949996948242, 40.97859191894531, 29.882253646850586, 4.0,
        55.06949996948242, 40.97859191894531, 29.882253646850586, 4.0, 51.3917236328125, 40.27886962890625,
        26.376453399658203, 4.0, 51.3917236328125, 40.27886962890625, 26.376453399658203, 4.0, 52.275333404541016,
        39.941688537597656, 26.44455337524414, 4.0, 52.275333404541016, 39.941688537597656, 26.44455337524414, 4.0,
        54.77970504760742, 41.26239776611328, 29.668624877929688, 4.0, 54.77970504760742, 41.26239776611328,
        29.668624877929688, 4.0, 56.256690979003906, 39.51252746582031, 29.909578323364258, 4.0, 56.256690979003906,
        39.51252746582031, 29.909578323364258, 4.0, 56.98036193847656, 39.94428253173828, 31.968421936035156, 4.0,
        56.98036193847656, 39.94428253173828, 31.968421936035156, 4.0, 56.69823455810547, 39.719154357910156,
        30.666568756103516, 3.0]
cgo2 = [2.0, 1.0, 6.0, (1.0, 0.0, 0.0), 4.0, 52.50569152832031, 40.94850540161133, 24.858413696289062, 4.0,
        52.15513610839844, 40.034751892089844, 26.24872589111328, 4.0, 52.15513610839844, 40.034751892089844,
        26.24872589111328, 4.0, 51.161842346191406, 40.29851531982422, 25.929445266723633, 4.0, 51.161842346191406,
        40.29851531982422, 25.929445266723633, 4.0, 51.22600173950195, 40.19047164916992, 25.48920249938965, 4.0,
        51.22600173950195, 40.19047164916992, 25.48920249938965, 4.0, 52.214012145996094, 40.20823669433594,
        27.270496368408203, 4.0, 52.214012145996094, 40.20823669433594, 27.270496368408203, 4.0, 51.39042663574219,
        38.98044204711914, 24.68115997314453, 4.0, 51.39042663574219, 38.98044204711914, 24.68115997314453, 4.0,
        52.13292694091797, 39.70564651489258, 24.131113052368164, 4.0, 52.13292694091797, 39.70564651489258,
        24.131113052368164, 4.0, 51.737037658691406, 40.60599136352539, 25.62929916381836, 4.0, 51.737037658691406,
        40.60599136352539, 25.62929916381836, 4.0, 52.05986785888672, 40.113407135009766, 25.144298553466797, 4.0,
        52.05986785888672, 40.113407135009766, 25.144298553466797, 4.0, 51.21947479248047, 40.86566162109375,
        24.880842208862305, 4.0, 51.21947479248047, 40.86566162109375, 24.880842208862305, 4.0, 51.037261962890625,
        40.669921875, 25.445384979248047, 4.0, 51.037261962890625, 40.669921875, 25.445384979248047, 4.0,
        50.07574462890625, 40.883766174316406, 27.55447006225586, 4.0, 50.07574462890625, 40.883766174316406,
        27.55447006225586, 4.0, 46.50026893615723, 41.17519187927246, 29.024028778076172, 6.0, (0.0, 0.5, 0.0), 4.0,
        46.50026893615723, 41.17519187927246, 29.024028778076172, 4.0, 46.50026893615723, 41.17519187927246,
        29.024028778076172, 4.0, 46.50026893615723, 41.17519187927246, 29.024028778076172, 4.0, 42.9247932434082,
        41.466617584228516, 30.493587493896484, 4.0, 42.9247932434082, 41.466617584228516, 30.493587493896484, 4.0,
        44.28599166870117, 41.352561950683594, 29.61571502685547, 4.0, 44.28599166870117, 41.352561950683594,
        29.61571502685547, 4.0, 47.548126220703125, 40.74769592285156, 28.91498565673828, 4.0, 47.548126220703125,
        40.74769592285156, 28.91498565673828, 4.0, 46.46072769165039, 43.28129577636719, 29.06939697265625, 4.0,
        46.46072769165039, 43.28129577636719, 29.06939697265625, 4.0, 47.89954948425293, 42.4506778717041,
        27.569915771484375, 6.0, (0.0, 0.0, 1.0), 4.0, 47.89954948425293, 42.4506778717041, 27.569915771484375, 4.0,
        47.89954948425293, 42.4506778717041, 27.569915771484375, 4.0, 47.89954948425293, 42.4506778717041,
        27.569915771484375, 4.0, 49.33837127685547, 41.620059967041016, 26.0704345703125, 4.0, 49.33837127685547,
        41.620059967041016, 26.0704345703125, 4.0, 48.975555419921875, 41.637794494628906, 25.793994903564453, 4.0,
        48.975555419921875, 41.637794494628906, 25.793994903564453, 4.0, 49.93598937988281, 42.06351852416992,
        25.33991813659668, 4.0, 49.93598937988281, 42.06351852416992, 25.33991813659668, 4.0, 51.94810104370117,
        37.85246276855469, 24.12131118774414, 3.0]

test__parse_aquaduct_path_out1 = np.array(
    [[53.759868621, 40.716629028, 25.091457366, -1.0, 0.0, -3.0, 0.0],
     [53.012104034, 41.043685913, 25.707672119, -1.0, 0.0, 2.0, 0.0],
     [48.984329223, 42.462810516, 29.608217239, -1.0, 0.0, 1.0, 0.0],
     [48.547574996, 41.742483139, 29.139482498, -1.0, 0.0, 0.0, 0.0]]
)
test__parse_aquaduct_path_out2 = np.array(
    [[48.110820770, 41.022155761, 28.670747756, -1.0, 0.0, 0.0, 0.0],
     [47.826919555, 41.477470397, 28.426923756, -1.0, 0.0, 1.0, 0.0],
     [49.932161331, 40.747554779, 27.323125832, -1.0, 0.0, -2.0, 0.0]]
)
test__parse_aquaduct_path_out3 = np.array(
    [[52.037403106, 40.017639160, 26.219327926, -1.0, 0.0, 0.0, 0.0],
     [52.287647247, 39.644699096, 25.468948364, -1.0, 0.0, 1.0, 0.0],
     [52.291427612, 41.428779602, 24.954166412, -1.0, 0.0, 2.0, 0.0],
     [53.925289154, 41.401084899, 29.504463195, -1.0, 0.0, 3.0, 0.0],
     [55.608203887, 41.642150878, 31.134807586, -1.0, 0.0, 4.0, 0.0],
     [55.313400268, 41.224903106, 29.652244567, -1.0, 0.0, 5.0, 0.0],
     [53.983493804, 41.167045593, 29.647794723, -1.0, 0.0, 6.0, 0.0],
     [55.069499969, 40.978591918, 29.882253646, -1.0, 0.0, 7.0, 0.0],
     [51.391723632, 40.278869628, 26.376453399, -1.0, 0.0, 8.0, 0.0],
     [52.275333404, 39.941688537, 26.444553375, -1.0, 0.0, 9.0, 0.0],
     [54.779705047, 41.262397766, 29.668624877, -1.0, 0.0, 10.0, 0.0],
     [56.256690979, 39.512527465, 29.909578323, -1.0, 0.0, 11.0, 0.0],
     [56.980361938, 39.944282531, 31.968421936, -1.0, 0.0, 12.0, 0.0],
     [56.698234558, 39.719154357, 30.666568756, -1.0, 0.0, -13.0, 0.0]]
    )
test__parse_aquaduct_path_out4 = np.array(
    [[52.505691528, 40.948505401, 24.858413696, -1.0, 0.0, -12.0, 0.0],
     [52.155136108, 40.034751892, 26.248725891, -1.0, 0.0, 11.0, 0.0],
     [51.161842346, 40.298515319, 25.929445266, -1.0, 0.0, 10.0, 0.0],
     [51.226001739, 40.190471649, 25.489202499, -1.0, 0.0, 9.0, 0.0],
     [52.214012145, 40.208236694, 27.270496368, -1.0, 0.0, 8.0, 0.0],
     [51.390426635, 38.980442047, 24.681159973, -1.0, 0.0, 7.0, 0.0],
     [52.132926940, 39.705646514, 24.131113052, -1.0, 0.0, 6.0, 0.0],
     [51.737037658, 40.605991363, 25.629299163, -1.0, 0.0, 5.0, 0.0],
     [52.059867858, 40.113407135, 25.144298553, -1.0, 0.0, 4.0, 0.0],
     [51.219474792, 40.865661621, 24.880842208, -1.0, 0.0, 3.0, 0.0],
     [51.037261962, 40.669921875, 25.445384979, -1.0, 0.0, 2.0, 0.0],
     [50.075744628, 40.883766174, 27.554470062, -1.0, 0.0, 1.0, 0.0],
     [46.500268936, 41.175191879, 29.024028778, -1.0, 0.0, 0.0, 0.0]]
    )
test__parse_aquaduct_path_out5 = np.array(
    [[42.924793243, 41.466617584, 30.493587493, -1.0, 0.0, 0.0, 0.0],
     [44.285991668, 41.352561950, 29.615715026, -1.0, 0.0, 1.0, 0.0],
     [47.548126220, 40.747695922, 28.914985656, -1.0, 0.0, 2.0, 0.0],
     [46.460727691, 43.281295776, 29.069396972, -1.0, 0.0, 3.0, 0.0],
     [47.899549484, 42.450677871, 27.569915771, -1.0, 0.0, -4.0, 0.0]]
    )
test__parse_aquaduct_path_out6 = np.array(
    [[49.338371276, 41.620059967, 26.070434570, -1.0, 0.0, 0.0, 0.0],
     [48.975555419, 41.637794494, 25.793994903, -1.0, 0.0, 1.0, 0.0],
     [49.935989379, 42.063518524, 25.339918136, -1.0, 0.0, 2.0, 0.0],
     [51.948101043, 37.852462768, 24.121311187, -1.0, 0.0, -3.0, 0.0]]
    )

list_of_points1 = [
    Point([0.30, -0.81, -0.55], 1.03, 0.00),
    Point([-0.43, -2.07, -1.27], 2.47, 0.00),
    Point([-0.78, -4.81, -3.23], 5.85, 0.00),
    Point([-1.84, -2.34, -3.85], 4.87, 0.00),
    Point([-2.78, -4.36, -4.04], 6.56, 0.00)
]

list_of_points2 = [
    Point([-2.78, -4.36, -4.04], 6.56, 0.00),
    Point([-1.84, -2.34, -3.85], 4.87, 0.00),
    Point([-0.78, -4.81, -3.23], 5.85, 0.00),
    Point([-0.43, -2.07, -1.27], 2.47, 0.00),
    Point([0.30, -0.81, -0.55], 1.03, 0.00)
]

list_of_points3 = [
    Point([0.06, -0.04, -0.06], 0.10, 0.00),
    Point([0.00, 0.00, 0.00], 0.00, 0.00)
]

list_of_points4 = [
    Point([1.08, -3.01, 4.46], 5.49, 0.00),
    Point([0.09, -2.88, 4.79], 5.59, 0.00),
    Point([0.67, -2.40, 4.93], 5.52, 0.00)
]

full_scores = {'BP': 5.609217279972872, 'SP': np.inf, 0: 6.488174186054253, 1: 6.585181422829492, 2: 6.521948226554441}

filter_out1 = {'length': (0.0, 999.0), 'radius': (0.0, 999.0), 'curvature': (1.0, 999.0), 'min_sims_num': 0,
               'min_snapshots_num': 0, 'min_avg_snapshots_num': 0.0, 'min_transport_events': 0, 'min_entry_events': 0,
               'min_release_events': 0}
filter_out2 = {'length': (10, 20), 'radius': (2, 999.0), 'curvature': (1.0, 5), 'min_sims_num': 0,
               'min_snapshots_num': 0, 'min_avg_snapshots_num': 500, 'min_transport_events': 0, 'min_entry_events': 0,
               'min_release_events': 0}
