from functools import reduce

from django.db.models import QuerySet
from guardian.shortcuts import get_objects_for_user
from rest_framework.exceptions import ValidationError


from django.conf import settings

from global_requests import get_thread_user
from utils import validate_foreign_keys, assign_default_permissions


class InvalidOperation(Exception):
    pass

PUBLIC_VIEW_MODELS_LOWER = [model_name.lower() for model_name in settings.PUBLIC_VIEW_MODELS]


class GuardianQuerySet(QuerySet):
    def __init__(self, *args, **kwargs):
        self._reset_validation()
        self.user = get_thread_user()
        super().__init__(*args, **kwargs)

    def some_oper_validated(self):
        return reduce(lambda x, value:x or value, self._validated_opers.values(), False)


    def __len__(self):
        if self.some_oper_validated():
            return super().__len__()
        else:
            raise InvalidOperation("No operations have been validated")

    def __iter__(self):
        if not self._validated_opers["view"]:
            return self.can_view().__iter__()
            # raise InvalidOperation("View operations have not been validated")
        return super().__iter__()


    @property
    def model_name(self):
        return self.model.__name__.lower()

    def create(self, with_user=True, **kwargs):
        if with_user:
            obj = super().create(**{
                'user': self.user,
                **kwargs,
            })
        else:
            obj = super().create(**kwargs)
        try:
            validate_foreign_keys(self.user, obj)
        except ValidationError as e:
            obj.delete()
            raise e
        assign_default_permissions(self.user, obj)
        return obj


    def _reset_validation(self):
        self._validated_opers = {
            "view": False,
            "change": False,
            "delete": False
        }

    def _clone(self):
        c = super()._clone()
        c._validated_opers = self._validated_opers
        c.user = self.user
        return c



    def can_do(self, perms):
        if len(perms) == 1 and perms[0] == "view" and self.model_name in PUBLIC_VIEW_MODELS_LOWER:
            self._validated_opers["view"] = True
            return self
        else:
            clone = get_objects_for_user(self.user, [perm + "_" + self.model_name for perm in perms], klass=self)
            for perm in perms:
                if perm == "view":
                   clone._validated_opers["view"] = True
                elif perm == "change":
                    clone._validated_opers["change"] = True
                elif perm == "delete":
                    clone._validated_opers["delete"] = True
            return self

    def can_view(self):
        return self.can_do(["view"])

    def can_change(self):
        return self.can_do(["change"])

    def can_delete(self):
        return self.can_do(["delete"])

    def is_owner(self):
        return self.can_do(["view", "change", "delete"])

    def force_validate(self):
        for oper in self._validated_opers.keys():
            self._validated_opers[oper] = True
        return self


    def values(self, *args, **kwargs):
        if not self._validated_opers["view"]:
            raise InvalidOperation("View operations have not been validated")
        return super().values(*args, **kwargs)

    def _values_list(self, *args, **kwargs):
        return super().values_list(*args, **kwargs)
    def values_list(self, *args, **kwargs):
        if not self._validated_opers["view"]:
            raise InvalidOperation("View operations have not been validated")
        return super().values_list(*args, **kwargs)

    def _get(self, *args, **kwargs):
        return super().get(*args, **kwargs)
    def get(self, *args, **kwargs):
        if not self._validated_opers["view"]:
            raise InvalidOperation("View operations have not been validated")
        return super().get(*args, **kwargs)

    def _first(self):
        return super().first()
    def first(self):
        if not self._validated_opers["view"]:
            raise InvalidOperation("View operations have not been validated")
        return super().first()

    def _update(self, **kwargs):
        return super().update(**kwargs)
    def update(self, **kwargs):
        if not self._validated_opers["change"]:
            raise InvalidOperation("Update operations have not been validated")
        return super().update(**kwargs)

    def _delete(self):
        return super().delete()
    def delete(self):
        if not self._validated_opers["delete"]:
            raise InvalidOperation("Delete operations have not been validated")
        return super().delete()

    def _exists(self):
        return super().exists()
    def exists(self):
        if not self.some_oper_validated():
            q = self.can_view()
            self.__dict__.update(q.__dict__)
        return super().exists()
