"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createGlueDatabase = exports.createGlueTable = exports.createGlueJobRole = exports.deployGlueJob = exports.buildGlueJob = exports.SinkStoreType = void 0;
const glue = require("@aws-cdk/aws-glue");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const defaults = require("../");
const utils_1 = require("./utils");
/**
 * Enumeration of data store types that could include S3, DynamoDB, DocumentDB, RDS or Redshift. Current
 * construct implementation only supports S3, but potential to add other output types in the future
 */
var SinkStoreType;
(function (SinkStoreType) {
    SinkStoreType["S3"] = "S3";
})(SinkStoreType = exports.SinkStoreType || (exports.SinkStoreType = {}));
function buildGlueJob(scope, props) {
    if (!props.existingCfnJob) {
        if (props.glueJobProps) {
            if (props.glueJobProps.glueVersion === '2.0' && props.glueJobProps.maxCapacity) {
                throw Error('Cannot set "MaxCapacity" with GlueVersion 2.0 or higher. Use "NumberOfWorkers" and "WorkerType". ' +
                    'Refer the API documentation https://docs.aws.amazon.com/glue/latest/webapi/API_Job.html for more details');
            }
            if (props.glueJobProps.maxCapacity && (props.glueJobProps.numberOfWorkers || props.glueJobProps.workerType)) {
                throw Error('Cannot set MaxCapacity and "WorkerType" or  "NumberOfWorkers". If using glueVersion 2.0 or beyond, ' +
                    'it is recommended to use "WorkerType" or  "NumberOfWorkers"');
            }
            return deployGlueJob(scope, props.glueJobProps, props.database, props.table, props.outputDataStore, props.etlCodeAsset);
        }
        else {
            throw Error('Either glueJobProps or existingCfnJob is required');
        }
    }
    else {
        return [props.existingCfnJob, aws_iam_1.Role.fromRoleArn(scope, 'ExistingRole', props.existingCfnJob.role)];
    }
}
exports.buildGlueJob = buildGlueJob;
function deployGlueJob(scope, glueJobProps, database, table, outputDataStore, etlCodeAsset) {
    let _glueSecurityConfigName;
    if (glueJobProps.securityConfiguration === undefined) {
        _glueSecurityConfigName = 'ETLJobSecurityConfig';
        const _glueKMSKey = `arn:${core_1.Aws.PARTITION}:kms:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:alias/aws/glue`;
        new glue.CfnSecurityConfiguration(scope, 'GlueSecurityConfig', {
            name: _glueSecurityConfigName,
            encryptionConfiguration: {
                jobBookmarksEncryption: {
                    jobBookmarksEncryptionMode: 'CSE-KMS',
                    kmsKeyArn: _glueKMSKey
                },
                s3Encryptions: [{
                        s3EncryptionMode: 'SSE-S3'
                    }]
            }
        });
    }
    else {
        _glueSecurityConfigName = glueJobProps.securityConfiguration;
    }
    const _glueJobPolicy = new aws_iam_1.Policy(scope, 'LogPolicy', {
        statements: [
            new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
                resources: [`arn:${core_1.Aws.PARTITION}:logs:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:log-group:/aws-glue/*`]
            })
        ]
    });
    let _jobRole;
    if (glueJobProps.role) {
        _jobRole = aws_iam_1.Role.fromRoleArn(scope, 'JobRole', glueJobProps.role);
    }
    else {
        _jobRole = defaults.createGlueJobRole(scope);
    }
    _glueJobPolicy.attachToRole(_jobRole);
    let _outputLocation;
    if (outputDataStore !== undefined && outputDataStore.datastoreType === SinkStoreType.S3) {
        if (outputDataStore.existingS3OutputBucket !== undefined) {
            _outputLocation = [outputDataStore.existingS3OutputBucket, undefined];
        }
        else {
            _outputLocation = defaults.buildS3Bucket(scope, { bucketProps: outputDataStore.outputBucketProps });
        }
    }
    else {
        _outputLocation = defaults.buildS3Bucket(scope, {});
    }
    _outputLocation[0].grantReadWrite(_jobRole);
    const _jobArgumentsList = {
        "--enable-metrics": true,
        "--enable-continuous-cloudwatch-log": true,
        "--database_name": database.ref,
        "--table_name": table.ref,
        ...((outputDataStore === undefined || (outputDataStore && outputDataStore.datastoreType === SinkStoreType.S3)) &&
            { '--output_path': `s3a://${_outputLocation[0].bucketName}/output/` }),
        ...glueJobProps.defaultArguments
    };
    const _newGlueJobProps = utils_1.overrideProps(defaults.DefaultGlueJobProps(_jobRole, glueJobProps, _glueSecurityConfigName, _jobArgumentsList, etlCodeAsset), glueJobProps);
    if (etlCodeAsset) {
        etlCodeAsset.grantRead(_jobRole);
    }
    else {
        // create CDK Bucket instance from S3 url and grant read access to Glue Job's service principal
        if (isJobCommandProperty(_newGlueJobProps.command)) {
            if (!_newGlueJobProps.command.scriptLocation) {
                throw Error('Script location has to be provided as an s3 Url location. Script location cannot be empty');
            }
            const _scriptLocation = _newGlueJobProps.command.scriptLocation;
            const _scriptBucketLocation = aws_s3_1.Bucket.fromBucketArn(scope, 'ScriptLocaiton', getS3ArnfromS3Url(_scriptLocation));
            _scriptBucketLocation.grantRead(_jobRole);
        }
    }
    const _glueJob = new glue.CfnJob(scope, 'KinesisETLJob', _newGlueJobProps);
    return [_glueJob, _jobRole, _outputLocation];
}
exports.deployGlueJob = deployGlueJob;
/**
 * This is a helper method to create the Role required for the Glue Job. If a role is already created then this
 * method is not required to be called.
 *
 * @param scope - The AWS Construct under which the role is to be created
 */
function createGlueJobRole(scope) {
    return new aws_iam_1.Role(scope, 'JobRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('glue.amazonaws.com'),
        description: 'Service role that Glue custom ETL jobs will assume for exeuction',
    });
}
exports.createGlueJobRole = createGlueJobRole;
/**
 * This method creates an AWS Glue table. The method is called when an existing Glue table is not provided
 */
function createGlueTable(scope, database, tableProps, fieldSchema, sourceType, parameters) {
    return defaults.DefaultGlueTable(scope, tableProps !== undefined ? tableProps :
        defaults.DefaultGlueTableProps(database, fieldSchema, sourceType, parameters));
}
exports.createGlueTable = createGlueTable;
/**
 * This method creates an AWS Glue database. The method is only called with an existing Glue database type is not provided.
 * The method uses the user provided props to override the defaul props for the Glue database
 *
 * @param scope
 * @param databaseProps
 */
function createGlueDatabase(scope, databaseProps) {
    const _mergedDBProps = (databaseProps !== undefined) ? utils_1.overrideProps(defaults.DefaultGlueDatabaseProps(), databaseProps) :
        defaults.DefaultGlueDatabaseProps();
    return defaults.DefaultGlueDatabase(scope, _mergedDBProps);
}
exports.createGlueDatabase = createGlueDatabase;
/**
 * A utility method to generate the S3 Arn from an S3 Url.
 *
 * @param s3Url
 */
function getS3ArnfromS3Url(s3Url) {
    if (s3Url && s3Url.startsWith('s3://')) {
        const splitString = s3Url.slice('s3://'.length);
        return `arn:${core_1.Aws.PARTITION}:s3:::${splitString}`;
    }
    else {
        throw Error(`Received S3URL as ${s3Url}. The S3 url string does not begin with s3://. This is not a standard S3 url`);
    }
}
/**
 * A utility method to type check CfnJob.JobCommandProperty type. For the construct to work for streaming ETL from Kinesis Data
 * Streams, all three attributes of the JobCommandProperty are required, even though they may be optional for other use cases.
 *
 * @param command
 */
function isJobCommandProperty(command) {
    if (command.name &&
        command.pythonVersion &&
        command.scriptLocation) {
        return true;
    }
    else {
        defaults.printWarning('command not of type JobCommandProperty type');
        return false;
    }
}
//# sourceMappingURL=data:application/json;base64,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