"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addPermission = exports.deployLambdaFunction = exports.buildLambdaFunction = void 0;
const lambda = require("@aws-cdk/aws-lambda");
const iam = require("@aws-cdk/aws-iam");
const lambda_defaults_1 = require("./lambda-defaults");
const cdk = require("@aws-cdk/core");
const utils_1 = require("./utils");
const security_group_helper_1 = require("./security-group-helper");
function buildLambdaFunction(scope, props) {
    var _a, _b;
    // Conditional lambda function creation
    if (!props.existingLambdaObj) {
        if (props.lambdaFunctionProps) {
            return deployLambdaFunction(scope, props.lambdaFunctionProps, undefined, props.vpc);
        }
        else {
            throw Error('Either existingLambdaObj or lambdaFunctionProps is required');
        }
    }
    else {
        if (props.vpc) {
            const levelOneFunction = props.existingLambdaObj.node.defaultChild;
            if ((_a = props.lambdaFunctionProps) === null || _a === void 0 ? void 0 : _a.securityGroups) {
                let ctr = 20;
                (_b = props.lambdaFunctionProps) === null || _b === void 0 ? void 0 : _b.securityGroups.forEach(sg => {
                    // TODO: Discuss with someone why I can't get R/O access to VpcConfigSecurityGroupIds
                    levelOneFunction.addOverride(`Properties.VpcConfig.SecurityGroupIds.${ctr++}`, sg.securityGroupId);
                });
            }
            if (!props.existingLambdaObj.isBoundToVpc) {
                throw Error('A Lambda function must be bound to a VPC upon creation, it cannot be added to a VPC in a subsequent construct');
            }
        }
        return props.existingLambdaObj;
    }
}
exports.buildLambdaFunction = buildLambdaFunction;
function deployLambdaFunction(scope, lambdaFunctionProps, functionId, vpc) {
    var _a, _b;
    const _functionId = functionId ? functionId : 'LambdaFunction';
    const _functionRoleId = _functionId + 'ServiceRole';
    if (vpc && lambdaFunctionProps.vpc) {
        throw new Error("Cannot provide a VPC in both the lambdaFunctionProps and the function argument");
    }
    // Setup the IAM Role for Lambda Service
    const lambdaServiceRole = new iam.Role(scope, _functionRoleId, {
        assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        inlinePolicies: {
            LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:PutLogEvents'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`]
                    })]
            })
        }
    });
    // If this Lambda function is going to access resoures in a
    // VPC, then it needs privileges to access an ENI in that VPC
    if (lambdaFunctionProps.vpc || vpc) {
        lambdaServiceRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                "ec2:CreateNetworkInterface",
                "ec2:DescribeNetworkInterfaces",
                "ec2:DeleteNetworkInterface",
                "ec2:AssignPrivateIpAddresses",
                "ec2:UnassignPrivateIpAddresses"
            ],
            resources: ["*"]
        }));
    }
    // Override the DefaultFunctionProps with user provided  lambdaFunctionProps
    let finalLambdaFunctionProps = utils_1.overrideProps(lambda_defaults_1.DefaultLambdaFunctionProps(lambdaServiceRole), lambdaFunctionProps);
    if (vpc) {
        // This is literally setting up what would be the default SG, but
        // we need to to it explicitly to disable the cfn_nag error
        const lambdaSecurityGroup = security_group_helper_1.buildSecurityGroup(scope, "ReplaceDefaultSecurityGroup", {
            vpc,
            allowAllOutbound: true,
        }, [], []);
        finalLambdaFunctionProps = utils_1.overrideProps(finalLambdaFunctionProps, {
            securityGroups: [lambdaSecurityGroup],
            vpc,
        }, true);
    }
    const lambdafunction = new lambda.Function(scope, _functionId, finalLambdaFunctionProps);
    if (lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_14_X ||
        lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_14_X ||
        lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_14_X) {
        lambdafunction.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });
    }
    const cfnLambdafunction = lambdafunction.node.findChild('Resource');
    utils_1.addCfnSuppressRules(lambdafunction, [
        {
            id: 'W58',
            reason: `Lambda functions has the required permission to write CloudWatch Logs. It uses custom policy instead of arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole with tighter permissions.`
        },
        {
            id: 'W89',
            reason: `This is not a rule for the general case, just for specific use cases/industries`
        },
        {
            id: 'W92',
            reason: `Impossible for us to define the correct concurrency for clients`
        }
    ]);
    if (cfnLambdafunction.tracingConfig) {
        // Find the X-Ray IAM Policy
        const cfnLambdafunctionDefPolicy = (_b = (_a = lambdafunction.role) === null || _a === void 0 ? void 0 : _a.node.tryFindChild('DefaultPolicy')) === null || _b === void 0 ? void 0 : _b.node.findChild('Resource');
        // Add the CFN NAG suppress to allow for "Resource": "*" for AWS X-Ray
        utils_1.addCfnSuppressRules(cfnLambdafunctionDefPolicy, [
            {
                id: 'W12',
                reason: `Lambda needs the following minimum required permissions to send trace data to X-Ray and access ENIs in a VPC.`
            }
        ]);
    }
    return lambdafunction;
}
exports.deployLambdaFunction = deployLambdaFunction;
// A wrapper above Function.addPermision that
// prevents two different calls to addPermission using
// the same construct id.
function addPermission(targetFunction, name, permission) {
    targetFunction.addPermission(GetNextId(targetFunction.permissionsNode.children, name), permission);
}
exports.addPermission = addPermission;
// Scan the current permissions for any entries with this core name and
// return the first available synthesized name. Names are coreName-suffix.
function GetNextId(children, coreName) {
    let lastSuffix = 0;
    children.forEach(child => {
        // if (compare right side of string)
        if (child.node.id.indexOf(coreName) === 0) {
            const components = child.node.id.split('-');
            if (components.length !== 2) {
                throw new Error("Incorrectly formatted synthesized construct ID");
            }
            const usedSuffix = Number(components[1]);
            if (usedSuffix > lastSuffix) {
                lastSuffix = usedSuffix;
            }
        }
    });
    return `${coreName}-${lastSuffix + 1}`;
}
//# sourceMappingURL=data:application/json;base64,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