"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildDeadLetterQueue = exports.buildQueue = void 0;
// Imports
const sqs = require("@aws-cdk/aws-sqs");
const defaults = require("./sqs-defaults");
const utils_1 = require("./utils");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const kms_helper_1 = require("./kms-helper");
function buildQueue(scope, id, props) {
    // If an existingQueueObj is not specified
    if (!props.existingQueueObj) {
        // Setup the queue
        let queueProps;
        if (props.queueProps) {
            // If property overrides have been provided, incorporate them and deploy
            const checkFifo = props.queueProps.fifo ? true : undefined;
            queueProps = utils_1.overrideProps(defaults.DefaultQueueProps(), { ...props.queueProps, fifo: checkFifo });
        }
        else {
            // If no property overrides, deploy using the default configuration
            queueProps = defaults.DefaultQueueProps();
        }
        // Determine whether a DLQ property should be added
        if (props.deadLetterQueue) {
            queueProps.deadLetterQueue = props.deadLetterQueue;
        }
        // Set encryption properties
        if (props.enableEncryptionWithCustomerManagedKey) {
            // Use the imported Customer Managed KMS key
            if (props.encryptionKey) {
                queueProps.encryptionMasterKey = props.encryptionKey;
            }
            else {
                queueProps.encryptionMasterKey = kms_helper_1.buildEncryptionKey(scope, props.encryptionKeyProps);
            }
        }
        const queue = new sqs.Queue(scope, id, queueProps);
        applySecureQueuePolicy(queue);
        // Return the queue
        return [queue, queue.encryptionMasterKey];
    }
    else {
        // If an existingQueueObj is specified, return that object as the queue to be used
        return [props.existingQueueObj];
    }
}
exports.buildQueue = buildQueue;
function buildDeadLetterQueue(scope, props) {
    if (!props.existingQueueObj && (props.deployDeadLetterQueue || props.deployDeadLetterQueue === undefined)) {
        // Create the Dead Letter Queue
        const [dlq] = buildQueue(scope, 'deadLetterQueue', {
            queueProps: props.deadLetterQueueProps
        });
        const mrc = (props.maxReceiveCount) ? props.maxReceiveCount : defaults.defaultMaxReceiveCount;
        // Setup the Dead Letter Queue interface
        const dlqInterface = {
            maxReceiveCount: mrc,
            queue: dlq
        };
        // Return the dead letter queue interface
        return dlqInterface;
    }
    return;
}
exports.buildDeadLetterQueue = buildDeadLetterQueue;
function applySecureQueuePolicy(queue) {
    // Apply queue policy to enforce only the queue owner can perform operations on queue
    queue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'QueueOwnerOnlyAccess',
        resources: [
            `${queue.queueArn}`
        ],
        actions: [
            "sqs:DeleteMessage",
            "sqs:ReceiveMessage",
            "sqs:SendMessage",
            "sqs:GetQueueAttributes",
            "sqs:RemovePermission",
            "sqs:AddPermission",
            "sqs:SetQueueAttributes"
        ],
        principals: [new aws_iam_1.AccountPrincipal(core_1.Stack.of(queue).account)],
        effect: aws_iam_1.Effect.ALLOW
    }));
    // Apply Topic policy to enforce encryption of data in transit
    queue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${queue.queueArn}`
        ],
        actions: [
            "SQS:*"
        ],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
//# sourceMappingURL=data:application/json;base64,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