"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const test_helper_1 = require("./test-helper");
const cdk = require("@aws-cdk/core");
const ec2 = require("@aws-cdk/aws-ec2");
const elasticache_helper_1 = require("../lib/elasticache-helper");
const elasticache_defaults_1 = require("../lib/elasticache-defaults");
test("Test returning existing Cache", () => {
    const stack = new cdk.Stack();
    const testVpc = test_helper_1.getTestVpc(stack, false);
    const existingCache = test_helper_1.CreateTestCache(stack, 'test', testVpc);
    const securityGroup = new ec2.SecurityGroup(stack, 'test-sg', {
        vpc: testVpc
    });
    const obtainedCache = elasticache_helper_1.obtainMemcachedCluster(stack, 'test-cache', {
        existingCache,
        cacheSecurityGroupId: securityGroup.securityGroupId
    });
    expect(obtainedCache).toBe(existingCache);
});
test("Test create cache with no client props", () => {
    const stack = new cdk.Stack();
    const testVpc = test_helper_1.getTestVpc(stack, false);
    const securityGroup = new ec2.SecurityGroup(stack, 'test-sg', {
        vpc: testVpc
    });
    elasticache_helper_1.obtainMemcachedCluster(stack, 'test-cache', {
        vpc: testVpc,
        cacheSecurityGroupId: securityGroup.securityGroupId,
        cachePort: 11111,
    });
    expect(stack).toHaveResourceLike("AWS::ElastiCache::CacheCluster", {
        Port: 11111,
        AZMode: 'cross-az',
        Engine: 'memcached',
    });
});
test("Test create cache with client props", () => {
    const stack = new cdk.Stack();
    const testVpc = test_helper_1.getTestVpc(stack, false);
    const securityGroup = new ec2.SecurityGroup(stack, 'test-sg', {
        vpc: testVpc
    });
    elasticache_helper_1.obtainMemcachedCluster(stack, 'test-cache', {
        vpc: testVpc,
        cacheSecurityGroupId: securityGroup.securityGroupId,
        cachePort: 12321,
        cacheProps: {
            azMode: 'single-az',
            clusterName: 'test-name'
        }
    });
    expect(stack).toHaveResourceLike("AWS::ElastiCache::CacheCluster", {
        Port: 12321,
        AZMode: 'single-az',
        Engine: 'memcached',
        ClusterName: 'test-name'
    });
});
test("Test GetCachePort() with existing cache", () => {
    const stack = new cdk.Stack();
    const testVpc = test_helper_1.getTestVpc(stack, false);
    const existingCache = test_helper_1.CreateTestCache(stack, 'test', testVpc, 32123);
    const port = elasticache_helper_1.getCachePort(undefined, existingCache);
    // Since the port from the existing cache is a token,
    // we can't check it directly, but we can ensure
    // the default port was replaced
    expect(port).not.toEqual(elasticache_defaults_1.GetDefaultCachePort());
});
test("Test GetCachePort() with clientCacheProps", () => {
    const clientPort = 32123;
    const port = elasticache_helper_1.getCachePort({ port: clientPort });
    expect(port).toEqual(clientPort);
});
test("Test GetCachePort() with default port", () => {
    const port = elasticache_helper_1.getCachePort();
    expect(port).toEqual(elasticache_defaults_1.GetDefaultCachePort());
});
//# sourceMappingURL=data:application/json;base64,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