"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
function deployES(stack, domainName, cfnDomainProps, lambdaRoleARN) {
    const userpool = defaults.buildUserPool(stack);
    const userpoolclient = defaults.buildUserPoolClient(stack, userpool, {
        userPoolClientName: 'test',
        userPool: userpool
    });
    const identitypool = defaults.buildIdentityPool(stack, userpool, userpoolclient);
    const cognitoAuthorizedRole = defaults.setupCognitoForElasticSearch(stack, 'test-domain', {
        userpool,
        userpoolclient,
        identitypool
    });
    if (lambdaRoleARN) {
        return defaults.buildElasticSearch(stack, domainName, {
            userpool,
            identitypool,
            cognitoAuthorizedRoleARN: cognitoAuthorizedRole.roleArn,
            serviceRoleARN: lambdaRoleARN
        }, cfnDomainProps);
    }
    else {
        return defaults.buildElasticSearch(stack, domainName, {
            userpool,
            identitypool,
            cognitoAuthorizedRoleARN: cognitoAuthorizedRole.roleArn
        }, cfnDomainProps);
    }
}
test('Test override SnapshotOptions for buildElasticSearch', () => {
    const stack = new core_1.Stack();
    deployES(stack, 'test-domain', {
        snapshotOptions: {
            automatedSnapshotStartHour: 5
        }
    });
    expect(stack).toHaveResource('AWS::Elasticsearch::Domain', {
        AccessPolicies: {
            Statement: [
                {
                    Action: "es:ESHttp*",
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::GetAtt": [
                                "CognitoAuthorizedRole14E74FE0",
                                "Arn"
                            ]
                        }
                    },
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:aws:es:",
                                {
                                    Ref: "AWS::Region"
                                },
                                ":",
                                {
                                    Ref: "AWS::AccountId"
                                },
                                ":domain/test-domain/*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        CognitoOptions: {
            Enabled: true,
            IdentityPoolId: {
                Ref: "CognitoIdentityPool"
            },
            RoleArn: {
                "Fn::GetAtt": [
                    "CognitoKibanaConfigureRole62CCE76A",
                    "Arn"
                ]
            },
            UserPoolId: {
                Ref: "CognitoUserPool53E37E69"
            }
        },
        DomainName: "test-domain",
        EBSOptions: {
            EBSEnabled: true,
            VolumeSize: 10
        },
        ElasticsearchClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessConfig: {
                AvailabilityZoneCount: 3
            },
            ZoneAwarenessEnabled: true
        },
        ElasticsearchVersion: "6.3",
        EncryptionAtRestOptions: {
            Enabled: true
        },
        NodeToNodeEncryptionOptions: {
            Enabled: true
        },
        SnapshotOptions: {
            AutomatedSnapshotStartHour: 5
        }
    });
});
test('Test override ES version for buildElasticSearch', () => {
    const stack = new core_1.Stack();
    deployES(stack, 'test-domain', {
        elasticsearchVersion: '7.0'
    });
    expect(stack).toHaveResource('AWS::Elasticsearch::Domain', {
        AccessPolicies: {
            Statement: [
                {
                    Action: "es:ESHttp*",
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::GetAtt": [
                                "CognitoAuthorizedRole14E74FE0",
                                "Arn"
                            ]
                        }
                    },
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:aws:es:",
                                {
                                    Ref: "AWS::Region"
                                },
                                ":",
                                {
                                    Ref: "AWS::AccountId"
                                },
                                ":domain/test-domain/*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        CognitoOptions: {
            Enabled: true,
            IdentityPoolId: {
                Ref: "CognitoIdentityPool"
            },
            RoleArn: {
                "Fn::GetAtt": [
                    "CognitoKibanaConfigureRole62CCE76A",
                    "Arn"
                ]
            },
            UserPoolId: {
                Ref: "CognitoUserPool53E37E69"
            }
        },
        DomainName: "test-domain",
        EBSOptions: {
            EBSEnabled: true,
            VolumeSize: 10
        },
        ElasticsearchClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessConfig: {
                AvailabilityZoneCount: 3
            },
            ZoneAwarenessEnabled: true
        },
        ElasticsearchVersion: "7.0",
        EncryptionAtRestOptions: {
            Enabled: true
        },
        NodeToNodeEncryptionOptions: {
            Enabled: true
        },
        SnapshotOptions: {
            AutomatedSnapshotStartHour: 1
        }
    });
});
test('Test ES with lambdaRoleARN', () => {
    const stack = new core_1.Stack();
    deployES(stack, 'test-domain', {}, 'arn:aws:us-east-1:mylambdaRoleARN');
    expect(stack).toHaveResource('AWS::Elasticsearch::Domain', {
        AccessPolicies: {
            Statement: [
                {
                    Action: "es:ESHttp*",
                    Effect: "Allow",
                    Principal: {
                        AWS: [
                            {
                                "Fn::GetAtt": [
                                    "CognitoAuthorizedRole14E74FE0",
                                    "Arn"
                                ]
                            },
                            "arn:aws:us-east-1:mylambdaRoleARN"
                        ]
                    },
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:aws:es:",
                                {
                                    Ref: "AWS::Region"
                                },
                                ":",
                                {
                                    Ref: "AWS::AccountId"
                                },
                                ":domain/test-domain/*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        CognitoOptions: {
            Enabled: true,
            IdentityPoolId: {
                Ref: "CognitoIdentityPool"
            },
            RoleArn: {
                "Fn::GetAtt": [
                    "CognitoKibanaConfigureRole62CCE76A",
                    "Arn"
                ]
            },
            UserPoolId: {
                Ref: "CognitoUserPool53E37E69"
            }
        },
        DomainName: "test-domain",
        EBSOptions: {
            EBSEnabled: true,
            VolumeSize: 10
        },
        ElasticsearchClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessConfig: {
                AvailabilityZoneCount: 3
            },
            ZoneAwarenessEnabled: true
        },
        ElasticsearchVersion: "6.3",
        EncryptionAtRestOptions: {
            Enabled: true
        },
        NodeToNodeEncryptionOptions: {
            Enabled: true
        },
        SnapshotOptions: {
            AutomatedSnapshotStartHour: 1
        }
    });
});
test('Count ES CW Alarms', () => {
    const stack = new core_1.Stack();
    deployES(stack, 'test-domain');
    const cwList = defaults.buildElasticSearchCWAlarms(stack);
    expect(cwList.length).toEqual(9);
});
//# sourceMappingURL=data:application/json;base64,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