"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const ec2 = require("@aws-cdk/aws-ec2");
const defaults = require("../");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test deployment with VPC
// --------------------------------------------------------------
test('Test deployment with VPC', () => {
    // Stack
    const stack = new core_1.Stack();
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    let sagemaker;
    let vpc;
    let sg;
    // Build Sagemaker Notebook Instance
    [sagemaker, vpc, sg] = defaults.buildSagemakerNotebook(stack, {
        role: sagemakerRole,
    });
    // Assertion
    expect(vpc === null || vpc === void 0 ? void 0 : vpc.privateSubnets.length).toEqual(2);
    expect(vpc === null || vpc === void 0 ? void 0 : vpc.publicSubnets.length).toEqual(2);
    expect(sagemaker.instanceType).toEqual('ml.t2.medium');
    expect(sg).toBeInstanceOf(ec2.SecurityGroup);
});
// --------------------------------------------------------------
// Test deployment in existing VPC
// --------------------------------------------------------------
test('Test deployment w/ existing VPC', () => {
    // Stack
    const stack = new core_1.Stack();
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    // Build Sagemaker Notebook Instance
    defaults.buildSagemakerNotebook(stack, {
        role: sagemakerRole,
        deployInsideVpc: true,
        sagemakerNotebookProps: {
            subnetId: 'subnet-deadbeef',
            securityGroupIds: ['sg-deadbeef'],
        },
    });
    expect(stack).toHaveResource('AWS::SageMaker::NotebookInstance', {
        DirectInternetAccess: 'Disabled',
        SecurityGroupIds: ['sg-deadbeef'],
        SubnetId: 'subnet-deadbeef',
    });
});
// --------------------------------------------------------------
// Test deployment with override
// --------------------------------------------------------------
test('Test deployment w/ override', () => {
    // Stack
    const stack = new core_1.Stack();
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    const key = new kms.Key(stack, 'MyEncryptionKey');
    // Build Sagemaker Notebook Instance
    defaults.buildSagemakerNotebook(stack, {
        role: sagemakerRole,
        sagemakerNotebookProps: {
            instanceType: 'ml.c4.2xlarge',
            kmsKeyId: key.keyArn,
        },
    });
    expect(stack).toHaveResource('AWS::SageMaker::NotebookInstance', {
        DirectInternetAccess: 'Disabled',
        InstanceType: 'ml.c4.2xlarge',
        KmsKeyId: {
            'Fn::GetAtt': ['MyEncryptionKeyD795679F', 'Arn'],
        },
    });
});
// --------------------------------------------------------------
// Test exception
// --------------------------------------------------------------
test('Test exception', () => {
    // Stack
    const stack = new core_1.Stack();
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    expect(() => {
        // Build Sagemaker Notebook Instance
        defaults.buildSagemakerNotebook(stack, {
            role: sagemakerRole,
            deployInsideVpc: true,
            sagemakerNotebookProps: {
                subnetId: 'subnet-deadbeef',
            },
        });
    }).toThrowError();
});
// ---------------------------------------------------------------
// Test exception for not providing primaryContainer in modelProps
// ---------------------------------------------------------------
test('Test exception for not providing primaryContainer in modelProps', () => {
    // Stack
    const stack = new core_1.Stack();
    const app = () => {
        // Build Sagemaker Inference Endpoint
        defaults.BuildSagemakerEndpoint(stack, {
            modelProps: {},
        });
    };
    // Assertion 1
    expect(app).toThrowError();
});
// -------------------------------------------------------------------------
// Test exception for not providing modelProps
// -------------------------------------------------------------------------
test('Test exception for not providing modelProps', () => {
    // Stack
    const stack = new core_1.Stack();
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    const app = () => {
        // Build Sagemaker Inference Endpoint
        defaults.deploySagemakerEndpoint(stack, { vpc });
    };
    // Assertion 1
    expect(app).toThrowError();
});
// -------------------------------------------------------------------------
// Test exception for not providing modelProps or existingSagemkaerObj
// -------------------------------------------------------------------------
test('Test exception for not providing modelProps or existingSagemkaerObj', () => {
    // Stack
    const stack = new core_1.Stack();
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    const app = () => {
        // Build Sagemaker Inference Endpoint
        defaults.BuildSagemakerEndpoint(stack, { vpc });
    };
    // Assertion 1
    expect(app).toThrowError();
});
// -----------------------------------------------------------------------------------------
// Test exception for not providing private or isolated subnets in an existing vpc
// -----------------------------------------------------------------------------------------
test('Test exception for not providing private or isolated subnets in an existing vpc', () => {
    // Stack
    const stack = new core_1.Stack();
    // create a VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        userVpcProps: {
            natGateways: 0,
            subnetConfiguration: [
                {
                    cidrMask: 18,
                    name: 'public',
                    subnetType: ec2.SubnetType.PUBLIC,
                },
            ],
        },
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    const app = () => {
        // Build Sagemaker Inference Endpoint
        defaults.deploySagemakerEndpoint(stack, {
            modelProps: {
                primaryContainer: {
                    image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                    modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
                },
            },
            vpc,
        });
    };
    // Assertion 1
    expect(app).toThrowError();
});
//# sourceMappingURL=data:application/json;base64,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