"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const defaults = require("../");
require("@aws-cdk/assert/jest");
const ec2 = require("@aws-cdk/aws-ec2");
// --------------------------------------------------------------
// Test minimal deployment with no properties
// --------------------------------------------------------------
test("Test minimal deployment with no properties", () => {
    // Stack
    const stack = new core_1.Stack();
    const vpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    defaults.buildSecurityGroup(stack, "primary-queue", {
        vpc,
        allowAllOutbound: true,
    }, [], []);
    expect(stack).toHaveResource("AWS::EC2::SecurityGroup", {
        SecurityGroupEgress: [
            {
                CidrIp: "0.0.0.0/0",
                Description: "Allow all outbound traffic by default",
                IpProtocol: "-1",
            },
        ],
    });
});
test("Test deployment with ingress rules", () => {
    // Stack
    const stack = new core_1.Stack();
    const vpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    defaults.buildSecurityGroup(stack, "primary-queue", {
        vpc,
        allowAllOutbound: true,
    }, [{ peer: ec2.Peer.ipv4("1.1.1.1/16"), connection: ec2.Port.tcp(100) }], []);
    expect(stack).toHaveResource("AWS::EC2::SecurityGroup", {
        SecurityGroupIngress: [
            {
                CidrIp: "1.1.1.1/16",
                Description: "from 1.1.1.1/16:100",
                FromPort: 100,
                IpProtocol: "tcp",
                ToPort: 100,
            },
        ],
    });
});
test("Test deployment with egress rule", () => {
    // Stack
    const stack = new core_1.Stack();
    const vpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    defaults.buildSecurityGroup(stack, "primary-queue", {
        vpc,
        allowAllOutbound: false,
    }, [], [
        { peer: ec2.Peer.ipv4("1.1.1.1/16"), connection: ec2.Port.tcp(100) },
        { peer: ec2.Peer.ipv4("2.2.2.2/24"), connection: ec2.Port.tcp(200) },
    ]);
    expect(stack).toHaveResource("AWS::EC2::SecurityGroup", {
        SecurityGroupEgress: [
            {
                CidrIp: "1.1.1.1/16",
                Description: "from 1.1.1.1/16:100",
                FromPort: 100,
                IpProtocol: "tcp",
                ToPort: 100,
            },
            {
                CidrIp: "2.2.2.2/24",
                Description: "from 2.2.2.2/24:200",
                FromPort: 200,
                IpProtocol: "tcp",
                ToPort: 200,
            },
        ],
    });
});
test("Test self referencing security group", () => {
    const testPort = 33333;
    // Stack
    const stack = new core_1.Stack();
    const vpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    defaults.CreateSelfReferencingSecurityGroup(stack, "testsg", vpc, testPort);
    expect(stack).toHaveResourceLike("AWS::EC2::SecurityGroupIngress", {
        IpProtocol: "TCP",
        FromPort: testPort,
        ToPort: testPort,
        GroupId: {
            "Fn::GetAtt": [
                "testsgcachesg72A723EA",
                "GroupId"
            ]
        },
        SourceSecurityGroupId: {
            "Fn::GetAtt": [
                "testsgcachesg72A723EA",
                "GroupId"
            ]
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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