"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const defaults = require("../");
require("@aws-cdk/assert/jest");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cloudwatch_log_group_helper_1 = require("../lib/cloudwatch-log-group-helper");
// --------------------------------------------------------------
// Test deployment w/ custom properties
// --------------------------------------------------------------
test('Test deployment w/ custom properties', () => {
    // Stack
    const stack = new core_1.Stack();
    // Step function definition
    const startState = new sfn.Pass(stack, 'StartState');
    // Build state machine
    defaults.buildStateMachine(stack, {
        definition: startState,
        stateMachineName: 'myStateMachine'
    });
    // Assertion
    expect(stack).toCountResources("AWS::Logs::LogGroup", 1);
    expect(stack).toHaveResource("AWS::StepFunctions::StateMachine", {
        StateMachineName: "myStateMachine"
    });
});
// --------------------------------------------------------------
// Test deployment w/ logging enabled
// --------------------------------------------------------------
test('Test deployment w/ logging enabled', () => {
    // Stack
    const stack = new core_1.Stack();
    // Step function definition
    const startState = new sfn.Pass(stack, 'StartState');
    // Log group
    // const logGroup = new LogGroup(stack, 'myLogGroup', defaults.buildLogGroup(stack));
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(stack, 'StateMachineLogGroup');
    // Build state machine
    defaults.buildStateMachine(stack, {
        definition: startState,
        logs: {
            destination: logGroup,
            level: sfn.LogLevel.FATAL
        }
    });
    // Assertion
    expect(stack).toCountResources("AWS::Logs::LogGroup", 1);
    expect(stack).toHaveResource("AWS::StepFunctions::StateMachine", {
        LoggingConfiguration: {
            Destinations: [{
                    CloudWatchLogsLogGroup: {
                        LogGroupArn: {
                            "Fn::GetAtt": [
                                "StateMachineLogGroup15B91BCB",
                                "Arn"
                            ]
                        }
                    }
                }],
            Level: 'FATAL'
        }
    });
});
// --------------------------------------------------------------
// Check default Cloudwatch permissions
// --------------------------------------------------------------
test('Check default Cloudwatch permissions', () => {
    // Stack
    const stack = new core_1.Stack();
    // Step function definition
    const startState = new sfn.Pass(stack, 'StartState');
    // Build state machine
    defaults.buildStateMachine(stack, {
        definition: startState
    });
    // Assertion
    expect(stack).toHaveResource("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "logs:CreateLogDelivery",
                        "logs:GetLogDelivery",
                        "logs:UpdateLogDelivery",
                        "logs:DeleteLogDelivery",
                        "logs:ListLogDeliveries"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "logs:PutResourcePolicy",
                        "logs:DescribeResourcePolicies",
                        "logs:DescribeLogGroups"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:",
                                {
                                    Ref: "AWS::Partition"
                                },
                                ":logs:",
                                {
                                    Ref: "AWS::Region"
                                },
                                ":",
                                {
                                    Ref: "AWS::AccountId"
                                },
                                ":*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        }
    });
});
// --------------------------------------------------------------
// Check CW Alarms
// --------------------------------------------------------------
test('Count State Machine CW Alarms', () => {
    // Stack
    const stack = new core_1.Stack();
    // Step function definition
    const startState = new sfn.Pass(stack, 'StartState');
    // Build state machine
    const [sm] = defaults.buildStateMachine(stack, {
        definition: startState
    });
    const cwList = defaults.buildStepFunctionCWAlarms(stack, sm);
    expect(cwList.length).toEqual(3);
});
//# sourceMappingURL=data:application/json;base64,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