"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CreateTestCache = exports.suppressAutoDeleteHandlerWarnings = exports.getFakeCertificate = exports.getTestVpc = exports.generateIntegStackName = exports.CreateScrapBucket = exports.fakeEcrRepoArn = void 0;
// Imports
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const vpc_helper_1 = require("../lib/vpc-helper");
const vpc_defaults_1 = require("../lib/vpc-defaults");
const utils_1 = require("../lib/utils");
const elasticache_helper_1 = require("../lib/elasticache-helper");
const path = require("path");
const cache = require("@aws-cdk/aws-elasticache");
const ec2 = require("@aws-cdk/aws-ec2");
const acm = require("@aws-cdk/aws-certificatemanager");
const elasticache_defaults_1 = require("../lib/elasticache-defaults");
exports.fakeEcrRepoArn = 'arn:aws:ecr:us-east-1:123456789012:repository/fake-repo';
// Creates a bucket used for testing - minimal properties, destroyed after test
function CreateScrapBucket(scope, props) {
    const defaultProps = {
        versioned: true,
        removalPolicy: core_1.RemovalPolicy.DESTROY,
        encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
    };
    let synthesizedProps;
    if (props) {
        synthesizedProps = utils_1.overrideProps(defaultProps, props);
    }
    else {
        synthesizedProps = defaultProps;
    }
    const scriptBucket = new aws_s3_1.Bucket(scope, "existingScriptLocation", synthesizedProps);
    utils_1.addCfnSuppressRules(scriptBucket, [
        {
            id: "W51",
            reason: "This S3 bucket is created for unit/ integration testing purposes only and not part of       the actual construct implementation",
        },
        {
            id: "W35",
            reason: "This S3 bucket is created for unit/ integration testing purposes only and not part of       the actual construct implementation",
        },
        {
            id: "W41",
            reason: "This S3 bucket is created for unit/ integration testing purposes only and not part of       the actual construct",
        }
    ]);
    return scriptBucket;
}
exports.CreateScrapBucket = CreateScrapBucket;
/**
 * @summary Creates a stack name for Integration tests
 * @param {string} filename - the filename of the integ test
 * @returns {string} - a string with current filename after removing anything before the prefix '.' and suffix '.js'
 * e.g. 'integ.apigateway-dynamodb-CRUD.js' will return 'apigateway-dynamodb-CRUD'
 */
function generateIntegStackName(filename) {
    const file = path.basename(filename, path.extname(filename));
    const stackname = file.slice(file.lastIndexOf('.') + 1).replace(/_/g, '-');
    return stackname;
}
exports.generateIntegStackName = generateIntegStackName;
// Helper Functions
function getTestVpc(stack, publicFacing = true) {
    return vpc_helper_1.buildVpc(stack, {
        defaultVpcProps: publicFacing ?
            vpc_defaults_1.DefaultPublicPrivateVpcProps() :
            vpc_defaults_1.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
            cidr: '172.168.0.0/16',
        },
    });
}
exports.getTestVpc = getTestVpc;
function getFakeCertificate(scope, id) {
    return acm.Certificate.fromCertificateArn(scope, id, "arn:aws:acm:us-east-1:123456789012:certificate/11112222-3333-1234-1234-123456789012");
}
exports.getFakeCertificate = getFakeCertificate;
function suppressAutoDeleteHandlerWarnings(stack) {
    core_1.Stack.of(stack).node.children.forEach(child => {
        if (child.node.id === 'Custom::S3AutoDeleteObjectsCustomResourceProvider') {
            const handlerFunction = child.node.findChild('Handler');
            utils_1.addCfnSuppressRules(handlerFunction, [{ id: "W58", reason: "CDK generated custom resource" }]);
            utils_1.addCfnSuppressRules(handlerFunction, [{ id: "W89", reason: "CDK generated custom resource" }]);
            utils_1.addCfnSuppressRules(handlerFunction, [{ id: "W92", reason: "CDK generated custom resource" }]);
        }
    });
}
exports.suppressAutoDeleteHandlerWarnings = suppressAutoDeleteHandlerWarnings;
function CreateTestCache(scope, id, vpc, port) {
    const cachePort = port !== null && port !== void 0 ? port : elasticache_defaults_1.GetDefaultCachePort();
    // Create the subnet group from all the isolated subnets in the VPC
    const subnetGroup = elasticache_helper_1.createCacheSubnetGroup(scope, vpc, id);
    const emptySG = new ec2.SecurityGroup(scope, `${id}-cachesg`, {
        vpc,
        allowAllOutbound: true,
    });
    utils_1.addCfnSuppressRules(emptySG, [{ id: "W40", reason: "Test Resource" }]);
    utils_1.addCfnSuppressRules(emptySG, [{ id: "W5", reason: "Test Resource" }]);
    utils_1.addCfnSuppressRules(emptySG, [{ id: "W36", reason: "Test Resource" }]);
    const cacheProps = {
        clusterName: `${id}-cdk-cluster`,
        cacheNodeType: "cache.t3.medium",
        engine: "memcached",
        numCacheNodes: 2,
        port: cachePort,
        azMode: "cross-az",
        vpcSecurityGroupIds: [emptySG.securityGroupId],
        cacheSubnetGroupName: subnetGroup.cacheSubnetGroupName,
    };
    const newCache = new cache.CfnCacheCluster(scope, `${id}-cluster`, cacheProps);
    newCache.addDependsOn(subnetGroup);
    return newCache;
}
exports.CreateTestCache = CreateTestCache;
//# sourceMappingURL=data:application/json;base64,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