import argparse
from abc import ABC, abstractmethod
from collections import Counter
import logging
from pathlib import Path

from amilib.file_lib import FileLib
from amilib.util import Util

"""Base class for specialist sub_parsers .
Also includes ArgParseBuilder
"""

logger = FileLib.get_logger(__file__)

class AbstractArgs(ABC):

    PIPELINE = "Pipeline"
    DEBUG = "debug"
    INDIR = "indir"
    INPUT = "input"
    INFORMAT = "informat"
    KWARGS = "kwords"
    OPERATION = "operation"
    OUTPUT = "output"
    OUTDIR = "outdir"

    DEBUG_HELP = f"will output debugging information (not fully implemented) \n"

    INPUT_HELP = f"input from:\n" \
                 f"   file/s single, multiple, and glob/wildcard (experimental)\n" \
                 f"   directories NYI\n" \
                 f"   URL/s (must start with 'https:'); provide {OUTDIR} for output \n"

    INDIR_HELP = f"Directory containing input files\n"

    OUTPUT_HELP = "output file or similar resource"

    OUTDIR_HELP = "output directory, required for URL input. If not given, autogenerated from file names"

    def get_input_help(self):
        return self.INPUT_HELP

    def get_indir_help(self):
        return self.INDIR_HELP

    def get_output_help(self):
        return self.OUTPUT_HELP

    def get_outdir_help(self):
        return self.OUTDIR_HELP

    def __init__(self):
        self.parser = None
        self.parsed_args = None
        self.ref_counter = Counter()
        self.arg_dict = self.create_default_arg_dict()
        self.subparser_arg = "UNKNOWN"

    def create_arg_dict(self, args=None):
        if args:
            self.parsed_args = args
        # print(f"PARSED_ARGS {type(self.parsed_args)} {self.parsed_args}")
        if not self.parsed_args:
            return None
        try:
            arg_vars = vars(self.parsed_args)
        except TypeError:
            # print(f" type args {type(self.parsed_args)} {self.parsed_args}")
            arg_vars = self.parsed_args
        self.arg_dict = dict()
        for item in arg_vars.items():
            key = item[0]
            if item[1] is None:
                pass
            elif type(item[1]) is list and len(item[1]) == 1:
                self.arg_dict[key] = item[1][0]
            else:
                self.arg_dict[key] = item[1]

        return self.arg_dict

    def parse_and_process(self):
        """Parse args after program name.
        If running in IDE there may be 2 names.
        All names should contain name of module (e.g. ami_dict)

        '/Applications/PyCharm CE.app/Contents/plugins/python-ce/helpers/pycharm/_jb_pytest_runner.py', 'ami_dict.py::test_process_args']
        or
        '/Users/pm286/workspace/pyami/pyamihtmlx/ami_dict.py', '--dict', 'foo', '--words', 'bar'

        """
        # strip all tokens including ".py" (will proably fail on some m/c)
        raise Exception("do we ever get here - test parse_and_process()")
        print(f"parse_and_process() module_stem: {self.module_stem}\n sys.argv {sys.argv}")
        args_store = sys.argv.copy()
        while len(sys.argv) > 0 and self.module_stem not in str(sys.argv[0]):
            print(f"trimming sys.argv {sys.argv}")
            sys.argv = sys.argv[1:]
        if len(sys.argv) == 0:  # must have name of prog
            sys.argv = args_store.copy()
        try:
            print(f"adding args")
            self.add_arguments()
        except Exception as e:
            print(f"failed to add args {e}")
            raise e
        logger.warning(f"AbstractArgs ADDED ARGS {sys.argv}")
        # print(f"argv {sys.argv}")
        if len(sys.argv) == 1:  # no args, print help
            self.parser.print_help()
        else:
            logging.warning(f"sys.argv {sys.argv}")
            argv_ = sys.argv[1:]
            print(f"argv: {argv_}")
            self.parse_and_process1(argv_)

    def parse_and_process1(self, argv_):
        print("running parse_and_process1 in util?")
        logging.warning(f"********** args for parse_and_process1 {argv_}")
        self.parsed_args = argv_ if self.parser is None else self.parser.parse_args(argv_)
        self.arg_dict = self.create_arg_dict()
        self.process_args()

    @property
    # @abstractmethod  # I don't know why this doesn't work
    def subparser_name(self):
        pass

    def add_argumants(self):

        self.parser.add_argument(f"--{self.DEBUG}",
                                 action='store_true',
                                 help=self.DEBUG_HELP)

        self.parser.add_argument(f"--{self.INPUT}", nargs="+",
                                 help=self.INPUT_HELP)

        self.parser.add_argument(f"--{self.INDIR}", nargs="+",
                                 help=self.INDIR_HELP)

        self.parser.add_argument(f"--{self.OUTPUT}", nargs="+",
                                 help=self.OUTPUT_HELP)

        self.parser.add_argument(f"--{self.OUTDIR}", nargs="+",
                                 help=self.OUTDIR_HELP)

        INFORM_HELP = "input format/s; experimental"
        self.parser.add_argument(f"--{self.KWARGS}", nargs="*",
                help="space-separated list of colon_separated keyword-value pairs, "
                     "format kw1:val1 kw2:val2;\nif empty list gives help")


    @abstractmethod
    def process_args(self):
        pass

    @abstractmethod
    def create_default_arg_dict(self):
        pass

    @property
    def module_stem(self):
        """name of module"""
        return Path(__file__).stem

    def get_operation(self):
        """The operation to run (makes this explicit)"""
        operation = self.arg_dict.get(AbstractArgs.OPERATION)
        return operation

    def get_indir(self):
        indir = self.arg_dict.get(AbstractArgs.INDIR)
        return indir

    def get_input(self):
        input = self.arg_dict.get(AbstractArgs.INPUT)
        return input

    def get_outdir(self):
        outdir = self.arg_dict.get(AbstractArgs.OUTDIR)
        return outdir

    def get_output(self):
        output = self.arg_dict.get(AbstractArgs.OUTPUT)
        return output

    def parse_kwargs_to_string(self, kwargs, keys=None):
        kwargs_dict = {}
        logger.info(f"args: {kwargs}")
        if not kwargs:
            if keys:
                print(f"possible keys: {keys}")
        else:
            if type(kwargs) is not list:
                kwargs = [kwargs]
            for arg in kwargs:
                logger.debug(f"pair {arg}")
                argz = arg.split(':')
                key = argz[0].strip()
                value = argz[1].strip()
                kwargs_dict[key] = value
            logger.warning(f"kwargs_dict {kwargs_dict}")
        return kwargs_dict


    def get_kwargs(self):
        kwargs = self.arg_dict.get(AbstractArgs.KWARGS)
        print(f"kwargs {kwargs}")
        if kwargs is None:
            return None
        if len(kwargs) == 0:
            self.kwargs_help()
        else:
            pass

        return

    def kwargs_help(self):
        print(f"key value pairs separated by ':' ; normally explicitly offered by subclass ")


    def make_run_func(self):
        """probably obsolete"""
        func_name = self.module_stem.replace("ami_", "run_")
        print(f"run_func_name {func_name}")
        return func_name

    @classmethod
    def make_sub_parser(cls, subclass, subparsers):
        """make subparser from subparsers
        requires self.subparser_arg (probably should be argument
        ALSO adds arguments through `self.add_arguments`
        :param subparsers: subparser generator
        :return: new subparser"""
        subclass.parser = subparsers.add_parser(subclass.subparser_arg)
        subclass.add_arguments()
        return subclass.parser

    @property
    def module_stem(self):
        """name of module"""
        return Path(__file__).stem


class ArgParseBuilder:
    ARG_LIST = "arg_list"
    DESCRIPTION = "description"

    def __init__(self):
        self.parser = None

    def create_arg_parse(self, arg_dict=None, arg_dict_file=None):
        # arg_dict_file takes precedence
        if arg_dict_file and arg_dict_file.exists():
            with open(arg_dict_file, 'r') as f:
                data = f.read()
                arg_dict = json.loads(data)
                print(f"arg_dict {arg_dict}")

        if arg_dict is not None:
            desc = f'{arg_dict.get(self.DESCRIPTION)}'
            print(f"\ndesc: '{desc}'")
            self.parser = argparse.ArgumentParser(description=desc)
            arg_list = arg_dict.get(self.ARG_LIST)
            if arg_list is None:
                raise ValueError(f"must give arg_list to ArgParseBuilder")
            for arg_dict in arg_list:
                if not type(arg_dict) is dict:
                    raise ValueError(f"arg_list_dict {arg_dict} is not a dict")
                args = arg_dict.keys()
                for arg in args:
                    print(f"\n{arg}:")
                    param_dict = arg_dict.get(arg)
                    self.process_params(param_dict)
                # self.parser.add_argument(f"--{ProjectArgs.PROJECT}", type=str, nargs=1, help="project directory")

    """https://stackoverflow.com/questions/28348117/using-argparse-and-json-together"""

    def process_params(self, param_dict):
        for param, param_val in param_dict.items():
            print(f"  {param}='{param_val}'")


class AmiArgParseException(Exception):
    """
    to capture error messages from AmiArgparser
    """
    pass


class AmiArgParser(argparse.ArgumentParser):
    """
    subclasses ArgumentParser and overrides error()
    """

    def error(self, message):
        """
        raises self.exit(2, error_message) so can be caught
        """
        raise AmiArgParseException(message)


