# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['plafosim', 'plafosim.algorithms', 'plafosim.cli']

package_data = \
{'': ['*'], 'plafosim': ['sumocfg/*']}

install_requires = \
['numpy>=1.19,<2.0', 'ortools>=9.0,<9.1', 'pandas>=1.1,<2.0', 'tqdm>=4.61,<5.0']

entry_points = \
{'console_scripts': ['plafosim = plafosim.cli.plafosim:main',
                     'plafosim-replay = plafosim.cli.trace_replay:main']}

setup_kwargs = {
    'name': 'plafosim',
    'version': '0.14.4',
    'description': 'A simple and scalable simulator for platoon formation.',
    'long_description': '# Platoon Formation Simulator (PlaFoSim)\n\n[![Code Version](https://img.shields.io/badge/code-v0.14.4-blue)](CHANGELOG.md)\n[![PyPI Version](https://img.shields.io/pypi/v/plafosim)](https://pypi.org/project/plafosim/)\n![Python Version](https://img.shields.io/badge/python-3.7-blue)\n![License](https://img.shields.io/github/license/heinovski/plafosim?color=green)\n\n[PlaFoSim](https://www.plafosim.de) - A simple and scalable simulator for platoon formation.\n\nPlaFoSim aims to facilitate and accelerate the research of platoon maneuvers and formation for individually driven vehicles.\nWhile the main focus of the simulator is on the assignment process, simulation of advertisements and maneuvers is implemented in a more abstract way.\n\n| Scenario | Advertisement | Assignment | Maneuver |\n| -------- | ------------- | ---------- | -------- |\n![Scenario](doc/scenario.png) | ![Advertisement](doc/advertisement.png) | ![Assignment](doc/assignment.png) | ![Maneuver](doc/maneuver.png) |\n\nPlaFoSim is published here:\n\n> Julian Heinovski, Dominik S. Buse and Falko Dressler, "Scalable Simulation of Platoon Formation Maneuvers with PlaFoSim," Proceedings of 13th IEEE Vehicular Networking Conference (VNC 2021), Poster Session, Virtual Conference, November 2021.\n\nPlease note that PlaFoSim is still under heavy development.\n\n---\n\n## Installation\n\n- Install Python (>=3.7.0)\n- Optionally install SUMO (>=1.6.0)\n- Install via pypi:\n```pip install plafosim```\n\nNOTE: The project is currently only tested under Linux.\n\n## Running a Simulation\n\nYou can use the simulator as module as well as from the command-line.\nCurrently, only command-line is thoroughly tested and thus completely available though.\n\n### Quickstart\n\nUse PlaFoSim\'s binary to run a simulation with the default configuration:\n\n```plafosim```\n\n### Advanced Simulation Control\n\nYou can use a variety of different parameters to customize the scenario and the simulation itself.\nE.g., use the parameter `vehicles` to configure the number of vehicles in the simulation:\n\n```plafosim --vehicles 1000```\n\nThe available parameters are grouped into different categories:\n\n```\n- road network properties\n- vehicle properties\n- trip properties\n- communication properties\n- platoon properties\n- formation properties\n- infrastructure properties\n- simulation properties\n- gui properties\n- result recording properties\n```\n\nYou can see the complete list of available parameters in the help:\n\n```plafosim -h, --help```\n\n### Examples\n\n```\n# Configure a 100km freeway with ramps at every 10km\nplafosim --road-length 100 --ramp-interval 10\n\n# Configure random (normally distributed) desired driving speed of 130km/h\nplafosim --random-driving-speed true --desired-speed 36\n\n# Configure random trips for 500 vehicles\nplafosim --vehicles 500 --random-depart-position true --random-arrival-position true --depart-desired true\n\n# Pre fill the freeway with 1000 vehicles\nplafosim --vehicles 1000 --pre-fill true\n\n# Configure 50% of the vehicles with Advanced Cruise Control (ACC) and a headway time of 1.5s\nplafosim --penetration 0.5 --acc-headway-time 1.5\n\n# Enable a simple, distributed platoon formation algorithm [1] in order to form platoons every 30s\nplafosim --formation-algorithm speedposition --formation-strategy distributed --execution-interval 30\n```\n\n### Live GUI\n\nYou can get a very simple live GUI based on SUMO by using the parameter `gui`:\n\n```plafosim --gui```\n\n![](doc/gui.png)\n\nMore options for the live GUI can be found within the ``gui properties`` section of the help.\n\n### Faster Simulation\n\nYou can speed up the simulation performance by enabling Python\'s optimization ```PYTHONOPTIMIZE```, e.g., in order to disable assertion:\n\n```PYTHONOPTIMIZE=1 plafosim```\n\nSee the Python [documention](https://docs.python.org/3/using/cmdline.html#envvar-PYTHONOPTIMIZE) for more details.\n\n## Re-Playing a Simulation\n\nThe simulation can write a trace file for every simulated vehicle (default `results_vehicle_traces.csv`).\nYou can replay the simulation based on the trace file by using a corresponding binary:\n\n```plafosim-replay results_vehicle_traces.csv```\n\nTo see all options of this script, run:\n\n```plafosim-replay -h, --help```\n\n## Extending\n\n- Clone the repository\n- Install poetry\n```pip install poetry```\n- Install PlaFoSim from source in editable mode\n```poetry install```\n- Run PlaFoSim in the virtual environment with\n```poetry run plafosim```\nor activate the virtual enviroment first with\n```poetry shell```\nand run the commands as usual (see above)\n\nNOTE: The project is currently only tested under Linux.\n\nIn order to add a new formation algorithm, you need to follow these steps:\n- Create a new sub-class of `FormationAlgorithm` (see `formation_algorithm.py`)\n- Add the name of your algorithm to the list of available algorithms within `plafosim`\n- Add the argument parser group of your new algorithm to `plafosim`\n- Add parsing of the algorithm name to `__init__` within `PlatooningVehicle` (see `platooning_vehicle.py`) and/or `Infrastructure` (see `infrastructure.py`)\n\n## Contributing\n\nIn order to contribute, please follow these steps:\n- Install PlaFoSim from source (see above)\n- Make desired changes\n- Run the tests located in `scripts` (see `.drone.yml`)\n- Submit a Pull Request (PR)\n\n### Testing\n\nWhen adding methods and functions, make sure to add corresponding unit tests for `py.test`.\nThe tests are located under `tests` and can be executed with `./scripts/run-pytest.sh`.\nThis will also generate a test coverage report.\n\n### Validation\n\nTo validate the behavior of PlaFoSim, it is compared to SUMO 1.6.0 by means of simulation results (e.g., vehicle traces).\nThe corresponding scripts are located under `scripts` and executed withn CI/CD pipelines.\nYou can have a look at `.drone.yml` for details regarding the execution.\n\n## Citing\n\nIf you are working with `PlaFoSim`, please cite the following paper:\n\n> Julian Heinovski, Dominik S. Buse and Falko Dressler, "Scalable Simulation of Platoon Formation Maneuvers with PlaFoSim," Proceedings of 13th IEEE Vehicular Networking Conference (VNC 2021), Poster Session, Virtual Conference, November 2021.\n\n```bibtex\n@inproceedings{heinovski2021scalable,\n    author = {Heinovski, Julian and Buse, Dominik S. and Dressler, Falko},\n    title = {{Scalable Simulation of Platoon Formation Maneuvers with PlaFoSim}},\n    publisher = {IEEE},\n    issn = {2157-9865},\n    isbn = {978-1-66544-450-7},\n    address = {Virtual Conference},\n    booktitle = {13th IEEE Vehicular Networking Conference (VNC 2021), Poster Session},\n    month = {11},\n    year = {2021},\n}\n```\n\n## References\n\n[1] Julian Heinovski and Falko Dressler, "Platoon Formation: Optimized Car to Platoon Assignment Strategies and Protocols," Proceedings of 10th IEEE Vehicular Networking Conference (VNC 2018), Taipei, Taiwan, December 2018.\n\n## License\n```\n# Copyright (c) 2020-2021 Julian Heinovski <heinovski@ccs-labs.org>\n#\n# This program is free software: you can redistribute it and/or modify\n# it under the terms of the GNU General Public License as published by\n# the Free Software Foundation, either version 3 of the License, or\n# any later version.\n#\n# This program is distributed in the hope that it will be useful,\n# but WITHOUT ANY WARRANTY; without even the implied warranty of\n# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n# GNU General Public License for more details.\n#\n# You should have received a copy of the GNU General Public License\n# along with this program.  If not, see <https://www.gnu.org/licenses/>.\n```\n',
    'author': 'Julian Heinovski',
    'author_email': 'heinovski@ccs-labs.org',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://www.plafosim.de',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
