import * as kms from '@aws-cdk/aws-kms';
import { Duration, RemovalPolicy } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { IQueue, QueueAttributes, QueueBase } from './queue-base';
/**
 * Properties for creating a new Queue.
 *
 * @stability stable
 */
export interface QueueProps {
    /**
     * A name for the queue.
     *
     * If specified and this is a FIFO queue, must end in the string '.fifo'.
     *
     * @default CloudFormation-generated name
     * @stability stable
     */
    readonly queueName?: string;
    /**
     * The number of seconds that Amazon SQS retains a message.
     *
     * You can specify an integer value from 60 seconds (1 minute) to 1209600
     * seconds (14 days). The default value is 345600 seconds (4 days).
     *
     * @default Duration.days(4)
     * @stability stable
     */
    readonly retentionPeriod?: Duration;
    /**
     * The time in seconds that the delivery of all messages in the queue is delayed.
     *
     * You can specify an integer value of 0 to 900 (15 minutes). The default
     * value is 0.
     *
     * @default 0
     * @stability stable
     */
    readonly deliveryDelay?: Duration;
    /**
     * The limit of how many bytes that a message can contain before Amazon SQS rejects it.
     *
     * You can specify an integer value from 1024 bytes (1 KiB) to 262144 bytes
     * (256 KiB). The default value is 262144 (256 KiB).
     *
     * @default 256KiB
     * @stability stable
     */
    readonly maxMessageSizeBytes?: number;
    /**
     * Default wait time for ReceiveMessage calls.
     *
     * Does not wait if set to 0, otherwise waits this amount of seconds
     * by default for messages to arrive.
     *
     * For more information, see Amazon SQS Long Poll.
     *
     * @default 0
     * @stability stable
     */
    readonly receiveMessageWaitTime?: Duration;
    /**
     * Timeout of processing a single message.
     *
     * After dequeuing, the processor has this much time to handle the message
     * and delete it from the queue before it becomes visible again for dequeueing
     * by another processor.
     *
     * Values must be from 0 to 43200 seconds (12 hours). If you don't specify
     * a value, AWS CloudFormation uses the default value of 30 seconds.
     *
     * @default Duration.seconds(30)
     * @stability stable
     */
    readonly visibilityTimeout?: Duration;
    /**
     * Send messages to this queue if they were unsuccessfully dequeued a number of times.
     *
     * @default no dead-letter queue
     * @stability stable
     */
    readonly deadLetterQueue?: DeadLetterQueue;
    /**
     * Whether the contents of the queue are encrypted, and by what type of key.
     *
     * Be aware that encryption is not available in all regions, please see the docs
     * for current availability details.
     *
     * @default Unencrypted
     * @stability stable
     */
    readonly encryption?: QueueEncryption;
    /**
     * External KMS master key to use for queue encryption.
     *
     * Individual messages will be encrypted using data keys. The data keys in
     * turn will be encrypted using this key, and reused for a maximum of
     * `dataKeyReuseSecs` seconds.
     *
     * If the 'encryptionMasterKey' property is set, 'encryption' type will be
     * implicitly set to "KMS".
     *
     * @default If encryption is set to KMS and not specified, a key will be created.
     * @stability stable
     */
    readonly encryptionMasterKey?: kms.IKey;
    /**
     * The length of time that Amazon SQS reuses a data key before calling KMS again.
     *
     * The value must be an integer between 60 (1 minute) and 86,400 (24
     * hours). The default is 300 (5 minutes).
     *
     * @default Duration.minutes(5)
     * @stability stable
     */
    readonly dataKeyReuse?: Duration;
    /**
     * Whether this a first-in-first-out (FIFO) queue.
     *
     * @default false, unless queueName ends in '.fifo' or 'contentBasedDeduplication' is true.
     * @stability stable
     */
    readonly fifo?: boolean;
    /**
     * Specifies whether to enable content-based deduplication.
     *
     * During the deduplication interval (5 minutes), Amazon SQS treats
     * messages that are sent with identical content (excluding attributes) as
     * duplicates and delivers only one copy of the message.
     *
     * If you don't enable content-based deduplication and you want to deduplicate
     * messages, provide an explicit deduplication ID in your SendMessage() call.
     *
     * (Only applies to FIFO queues.)
     *
     * @default false
     * @stability stable
     */
    readonly contentBasedDeduplication?: boolean;
    /**
     * Policy to apply when the user pool is removed from the stack.
     *
     * Even though queues are technically stateful, their contents are transient and it
     * is common to add and remove Queues while rearchitecting your application. The
     * default is therefore `DESTROY`. Change it to `RETAIN` if the messages are so
     * valuable that accidentally losing them would be unacceptable.
     *
     * @default RemovalPolicy.DESTROY
     * @stability stable
     */
    readonly removalPolicy?: RemovalPolicy;
}
/**
 * Dead letter queue settings.
 *
 * @stability stable
 */
export interface DeadLetterQueue {
    /**
     * The dead-letter queue to which Amazon SQS moves messages after the value of maxReceiveCount is exceeded.
     *
     * @stability stable
     */
    readonly queue: IQueue;
    /**
     * The number of times a message can be unsuccesfully dequeued before being moved to the dead-letter queue.
     *
     * @stability stable
     */
    readonly maxReceiveCount: number;
}
/**
 * What kind of encryption to apply to this queue.
 *
 * @stability stable
 */
export declare enum QueueEncryption {
    /**
     * Messages in the queue are not encrypted.
     *
     * @stability stable
     */
    UNENCRYPTED = "NONE",
    /**
     * Server-side KMS encryption with a master key managed by SQS.
     *
     * @stability stable
     */
    KMS_MANAGED = "MANAGED",
    /**
     * Server-side encryption with a KMS key managed by the user.
     *
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     *
     * @stability stable
     */
    KMS = "KMS"
}
/**
 * A new Amazon SQS queue.
 *
 * @stability stable
 */
export declare class Queue extends QueueBase {
    /**
     * Import an existing SQS queue provided an ARN.
     *
     * @param scope The parent creating construct.
     * @param id The construct's name.
     * @param queueArn queue ARN (i.e. arn:aws:sqs:us-east-2:444455556666:queue1).
     * @stability stable
     */
    static fromQueueArn(scope: Construct, id: string, queueArn: string): IQueue;
    /**
     * Import an existing queue.
     *
     * @stability stable
     */
    static fromQueueAttributes(scope: Construct, id: string, attrs: QueueAttributes): IQueue;
    /**
     * The ARN of this queue.
     *
     * @stability stable
     */
    readonly queueArn: string;
    /**
     * The name of this queue.
     *
     * @stability stable
     */
    readonly queueName: string;
    /**
     * The URL of this queue.
     *
     * @stability stable
     */
    readonly queueUrl: string;
    /**
     * If this queue is encrypted, this is the KMS key.
     *
     * @stability stable
     */
    readonly encryptionMasterKey?: kms.IKey;
    /**
     * Whether this queue is an Amazon SQS FIFO queue.
     *
     * If false, this is a standard queue.
     *
     * @stability stable
     */
    readonly fifo: boolean;
    /**
     * Controls automatic creation of policy objects.
     *
     * Set by subclasses.
     *
     * @stability stable
     */
    protected readonly autoCreatePolicy = true;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props?: QueueProps);
    /**
     * Look at the props, see if the FIFO props agree, and return the correct subset of props
     */
    private determineFifoProps;
}
