# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['async_search_client',
 'async_search_client.models',
 'async_search_client.paths',
 'async_search_client.utils']

package_data = \
{'': ['*']}

install_requires = \
['httpx>=0.17.0,<0.18.0', 'pydantic>=1.8,<2.0']

setup_kwargs = {
    'name': 'async-search-client',
    'version': '0.1.1',
    'description': 'A Python async client for the MeiliSearch API',
    'long_description': '# Async Search Client\n\n[![Tests Status](https://github.com/sanders41/async-search-client/workflows/Testing/badge.svg?branch=main&event=push)](https://github.com/sanders41/async-search-client/actions?query=workflow%3ATesting+branch%3Amain+event%3Apush)\n[![Lint Status](https://github.com/sanders41/async-search-client/workflows/Linting/badge.svg?branch=main&event=push)](https://github.com/sanders41/async-search-client/actions?query=workflow%3ALinting+branch%3Amain+event%3Apush)\n[![Coverage](https://codecov.io/github/sanders41/async-search-client/coverage.svg?branch=main)](https://codecov.io/gh/sanders41/async-search-client)\n[![PyPI version](https://badge.fury.io/py/async-search-client.svg)](https://badge.fury.io/py/async-search-client)\n\nAsync Serach Client is a Python async client for the [MeiliSearch](https://github.com/meilisearch/MeiliSearch) API. MeiliSearch also has an official [Python client](https://github.com/meilisearch/meilisearch-python).\n\nWhich of the two clients to use comes down to your particular use case. The purpose for this async client is to allow for non-blocking calls when you are working in async frameworks such as [FastAPI](https://fastapi.tiangolo.com/), or if your own code base you are working in is async. If this does not match your use case then the official client will be a better choice.\n\nFor the most part this client mirrors the functionality of the official client and the same [documenation](https://docs.meilisearch.com/reference/) will apply. There are are few exceptions to this to be aware of:\n\n1. The async client runs in a context manager. This means to create a client, instead of:\n\n    ```py\n    client = Client("http://127.0.0.1:7700", "masterKey")\n    ```\n\n    In the async client it would be:\n\n    ```py\n    async with Client("http://127.0.0.1:7700", "masterKey") as client:\n        ...\n    ```\n\n2. Because this client is async you need to await the calls. For example adding documents in the official verison would be:\n\n    ```py\n    index.add_documents(documents)\n    ```\n\n    In the async client it would be:\n\n    ```py\n    await index.add_documents(documents)\n    ```\n\n3. The async client uses [Pydantic](https://pydantic-docs.helpmanual.io/) to serialize/deserialize the JSON from MeiliSearch into python objects wherever possible, and in the process converts the camelCaseNames from JSON into more Pythonic snake_case_names. The official client uses dictionaries to store the return values in most cases.\n\n## Installation\n\nUsing a virtual environmnet is recommended for installing this package. Once the virtual environment is created and activated install the package with:\n\n```sh\npip install async-search-client\n```\n\n## Run MeiliSearch\n\nThere are several ways to [run MeiliSearch](https://docs.meilisearch.com/reference/features/installation.html#download-and-launch). Pick the one that works best for your use case and then start the server.\n\nAs as example to use Docker:\n\n```sh\ndocker pull getmeili/meilisearch:latest\ndocker run -it --rm -p 7700:7700 getmeili/meilisearch:latest ./meilisearch --master-key=masterKey\n```\n\n## Useage\n\n### Add Documents\n\n* Note: `client.index("books") creates an instance of an Index object but does not make a network call to send the data yet so it does not need to be awaited.\n\n```py\nfrom async_search_client import Client\n\nasync with Client(\'http://127.0.0.1:7700\', \'masterKey\') as client:\n    index = client.index("books")\n\n    documents = [\n        {"id": 1, "title": "Ready Player One"},\n        {"id": 42, "title": "The Hitchhiker\'s Guide to the Galaxy"},\n    ]\n\n    await index.add_documents(documents)\n```\n\nThe server will return a update id that can be used to [get the status](https://docs.meilisearch.com/reference/api/updates.html#get-an-update-status) of the updates. To do this you would save the result response from adding the documets to a variable, this will be a UpdateId object, and use it to check the status of the updates.\n\n```py\nupdate = await index.add_documents(documents)\nstatus = await client.index(\'books\').get_update_status(update.update_id)\n```\n\n### Basic Searching\n\n```py\nsearch_result = await index.search("ready player")\n```\n\n### Base Search Results: SearchResults object with values\n\n```py\nSearchResults(\n    hits = [\n        {\n            "id": 1,\n            "title": "Ready Player One",\n        },\n    ],\n    offset = 0,\n    limit = 20,\n    nb_hits = 1,\n    exhaustive_nb_hits = bool,\n    facets_distributionn = None,\n    processing_time_ms = 1,\n    query = "ready player",\n)\n```\n\n### Custom Search\n\nInformation about the parameters can be found in the [search parameters](https://docs.meilisearch.com/reference/features/search_parameters.html) section of the documentation.\n\n```py\nindex.search(\n    "guide",\n    attributes_to_highlight=["title"],\n    filters="book_id > 10"\n)\n```\n\n### Custom Search Results: SearchResults object with values\n\n```py\nSearchResults(\n    hits = [\n        {\n            "id": 42,\n            "title": "The Hitchhiker\'s Guide to the Galaxy",\n            "_formatted": {\n                "id": 42,\n                "title": "The Hitchhiker\'s Guide to the <em>Galaxy</em>"\n            }\n        },\n    ],\n    offset = 0,\n    limit = 20,\n    nb_hits = 1,\n    exhaustive_nb_hits = bool,\n    facets_distributionn = None,\n    processing_time_ms = 5,\n    query = "galaxy",\n)\n```\n\n## Learn More\n\nFor more see the [API Reference](https://docs.meilisearch.com/reference/api/) in the MeiliSearch documentation. Keep in mind you will need to await the examples shown in the documentation, and that you will be getting python objects instead of JSON for you results.\n\n## Contributing\n\nContributions to this project are welcome. If you are interesting in contributing please see our [contributing guide](CONTRIBUTING.md)\n',
    'author': 'Paul Sanders',
    'author_email': 'psanders1@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/sanders41/async-search-client',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
