"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SqsSubscription = void 0;
const iam = require("@aws-cdk/aws-iam");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
/**
 * Use an SQS queue as a subscription target
 */
class SqsSubscription {
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
    }
    /**
     * Returns a configuration for an SQS queue to subscribe to an SNS topic
     */
    bind(topic) {
        // Create subscription under *consuming* construct to make sure it ends up
        // in the correct stack in cases of cross-stack subscriptions.
        if (!core_1.Construct.isConstruct(this.queue)) {
            throw new Error('The supplied Queue object must be an instance of Construct');
        }
        // add a statement to the queue resource policy which allows this topic
        // to send messages to the queue.
        this.queue.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.queue.queueArn],
            actions: ['sqs:SendMessage'],
            principals: [new iam.ServicePrincipal('sns.amazonaws.com')],
            conditions: {
                ArnEquals: { 'aws:SourceArn': topic.topicArn },
            },
        }));
        return {
            subscriberScope: this.queue,
            subscriberId: topic.node.uniqueId,
            endpoint: this.queue.queueArn,
            protocol: sns.SubscriptionProtocol.SQS,
            rawMessageDelivery: this.props.rawMessageDelivery,
            filterPolicy: this.props.filterPolicy,
            region: this.regionFromArn(topic),
            deadLetterQueue: this.props.deadLetterQueue,
        };
    }
    regionFromArn(topic) {
        // no need to specify `region` for topics defined within the same stack
        if (topic instanceof sns.Topic) {
            return undefined;
        }
        return core_1.Stack.of(topic).parseArn(topic.topicArn).region;
    }
}
exports.SqsSubscription = SqsSubscription;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3FzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsic3FzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHdDQUF3QztBQUN4Qyx3Q0FBd0M7QUFFeEMsd0NBQWlEO0FBaUJqRDs7R0FFRztBQUNILE1BQWEsZUFBZTtJQUMxQixZQUE2QixLQUFpQixFQUFtQixRQUE4QixFQUFFO1FBQXBFLFVBQUssR0FBTCxLQUFLLENBQVk7UUFBbUIsVUFBSyxHQUFMLEtBQUssQ0FBMkI7SUFDakcsQ0FBQztJQUVEOztPQUVHO0lBQ0ksSUFBSSxDQUFDLEtBQWlCO1FBQzNCLDBFQUEwRTtRQUMxRSw4REFBOEQ7UUFDOUQsSUFBSSxDQUFDLGdCQUFTLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsRUFBRTtZQUN0QyxNQUFNLElBQUksS0FBSyxDQUFDLDREQUE0RCxDQUFDLENBQUM7U0FDL0U7UUFFRCx1RUFBdUU7UUFDdkUsaUNBQWlDO1FBQ2pDLElBQUksQ0FBQyxLQUFLLENBQUMsbUJBQW1CLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3JELFNBQVMsRUFBRSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDO1lBQ2hDLE9BQU8sRUFBRSxDQUFDLGlCQUFpQixDQUFDO1lBQzVCLFVBQVUsRUFBRSxDQUFDLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLG1CQUFtQixDQUFDLENBQUM7WUFDM0QsVUFBVSxFQUFFO2dCQUNWLFNBQVMsRUFBRSxFQUFFLGVBQWUsRUFBRSxLQUFLLENBQUMsUUFBUSxFQUFFO2FBQy9DO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixPQUFPO1lBQ0wsZUFBZSxFQUFFLElBQUksQ0FBQyxLQUFLO1lBQzNCLFlBQVksRUFBRSxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVE7WUFDakMsUUFBUSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUTtZQUM3QixRQUFRLEVBQUUsR0FBRyxDQUFDLG9CQUFvQixDQUFDLEdBQUc7WUFDdEMsa0JBQWtCLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxrQkFBa0I7WUFDakQsWUFBWSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWTtZQUNyQyxNQUFNLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxLQUFLLENBQUM7WUFDakMsZUFBZSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsZUFBZTtTQUM1QyxDQUFDO0lBQ0osQ0FBQztJQUVPLGFBQWEsQ0FBQyxLQUFpQjtRQUNyQyx1RUFBdUU7UUFDdkUsSUFBSSxLQUFLLFlBQVksR0FBRyxDQUFDLEtBQUssRUFBRTtZQUM5QixPQUFPLFNBQVMsQ0FBQztTQUNsQjtRQUNELE9BQU8sWUFBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDLE1BQU0sQ0FBQztJQUN6RCxDQUFDO0NBQ0Y7QUE1Q0QsMENBNENDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgc25zIGZyb20gJ0Bhd3MtY2RrL2F3cy1zbnMnO1xuaW1wb3J0ICogYXMgc3FzIGZyb20gJ0Bhd3MtY2RrL2F3cy1zcXMnO1xuaW1wb3J0IHsgQ29uc3RydWN0LCBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgU3Vic2NyaXB0aW9uUHJvcHMgfSBmcm9tICcuL3N1YnNjcmlwdGlvbic7XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYW4gU1FTIHN1YnNjcmlwdGlvblxuICovXG5leHBvcnQgaW50ZXJmYWNlIFNxc1N1YnNjcmlwdGlvblByb3BzIGV4dGVuZHMgU3Vic2NyaXB0aW9uUHJvcHMge1xuICAvKipcbiAgICogVGhlIG1lc3NhZ2UgdG8gdGhlIHF1ZXVlIGlzIHRoZSBzYW1lIGFzIGl0IHdhcyBzZW50IHRvIHRoZSB0b3BpY1xuICAgKlxuICAgKiBJZiBmYWxzZSwgdGhlIG1lc3NhZ2Ugd2lsbCBiZSB3cmFwcGVkIGluIGFuIFNOUyBlbnZlbG9wZS5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHJhd01lc3NhZ2VEZWxpdmVyeT86IGJvb2xlYW47XG59XG5cbi8qKlxuICogVXNlIGFuIFNRUyBxdWV1ZSBhcyBhIHN1YnNjcmlwdGlvbiB0YXJnZXRcbiAqL1xuZXhwb3J0IGNsYXNzIFNxc1N1YnNjcmlwdGlvbiBpbXBsZW1lbnRzIHNucy5JVG9waWNTdWJzY3JpcHRpb24ge1xuICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IHF1ZXVlOiBzcXMuSVF1ZXVlLCBwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBTcXNTdWJzY3JpcHRpb25Qcm9wcyA9IHt9KSB7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhIGNvbmZpZ3VyYXRpb24gZm9yIGFuIFNRUyBxdWV1ZSB0byBzdWJzY3JpYmUgdG8gYW4gU05TIHRvcGljXG4gICAqL1xuICBwdWJsaWMgYmluZCh0b3BpYzogc25zLklUb3BpYyk6IHNucy5Ub3BpY1N1YnNjcmlwdGlvbkNvbmZpZyB7XG4gICAgLy8gQ3JlYXRlIHN1YnNjcmlwdGlvbiB1bmRlciAqY29uc3VtaW5nKiBjb25zdHJ1Y3QgdG8gbWFrZSBzdXJlIGl0IGVuZHMgdXBcbiAgICAvLyBpbiB0aGUgY29ycmVjdCBzdGFjayBpbiBjYXNlcyBvZiBjcm9zcy1zdGFjayBzdWJzY3JpcHRpb25zLlxuICAgIGlmICghQ29uc3RydWN0LmlzQ29uc3RydWN0KHRoaXMucXVldWUpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1RoZSBzdXBwbGllZCBRdWV1ZSBvYmplY3QgbXVzdCBiZSBhbiBpbnN0YW5jZSBvZiBDb25zdHJ1Y3QnKTtcbiAgICB9XG5cbiAgICAvLyBhZGQgYSBzdGF0ZW1lbnQgdG8gdGhlIHF1ZXVlIHJlc291cmNlIHBvbGljeSB3aGljaCBhbGxvd3MgdGhpcyB0b3BpY1xuICAgIC8vIHRvIHNlbmQgbWVzc2FnZXMgdG8gdGhlIHF1ZXVlLlxuICAgIHRoaXMucXVldWUuYWRkVG9SZXNvdXJjZVBvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICByZXNvdXJjZXM6IFt0aGlzLnF1ZXVlLnF1ZXVlQXJuXSxcbiAgICAgIGFjdGlvbnM6IFsnc3FzOlNlbmRNZXNzYWdlJ10sXG4gICAgICBwcmluY2lwYWxzOiBbbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdzbnMuYW1hem9uYXdzLmNvbScpXSxcbiAgICAgIGNvbmRpdGlvbnM6IHtcbiAgICAgICAgQXJuRXF1YWxzOiB7ICdhd3M6U291cmNlQXJuJzogdG9waWMudG9waWNBcm4gfSxcbiAgICAgIH0sXG4gICAgfSkpO1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIHN1YnNjcmliZXJTY29wZTogdGhpcy5xdWV1ZSxcbiAgICAgIHN1YnNjcmliZXJJZDogdG9waWMubm9kZS51bmlxdWVJZCxcbiAgICAgIGVuZHBvaW50OiB0aGlzLnF1ZXVlLnF1ZXVlQXJuLFxuICAgICAgcHJvdG9jb2w6IHNucy5TdWJzY3JpcHRpb25Qcm90b2NvbC5TUVMsXG4gICAgICByYXdNZXNzYWdlRGVsaXZlcnk6IHRoaXMucHJvcHMucmF3TWVzc2FnZURlbGl2ZXJ5LFxuICAgICAgZmlsdGVyUG9saWN5OiB0aGlzLnByb3BzLmZpbHRlclBvbGljeSxcbiAgICAgIHJlZ2lvbjogdGhpcy5yZWdpb25Gcm9tQXJuKHRvcGljKSxcbiAgICAgIGRlYWRMZXR0ZXJRdWV1ZTogdGhpcy5wcm9wcy5kZWFkTGV0dGVyUXVldWUsXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgcmVnaW9uRnJvbUFybih0b3BpYzogc25zLklUb3BpYyk6IHN0cmluZyB8IHVuZGVmaW5lZCB7XG4gICAgLy8gbm8gbmVlZCB0byBzcGVjaWZ5IGByZWdpb25gIGZvciB0b3BpY3MgZGVmaW5lZCB3aXRoaW4gdGhlIHNhbWUgc3RhY2tcbiAgICBpZiAodG9waWMgaW5zdGFuY2VvZiBzbnMuVG9waWMpIHtcbiAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuICAgIHJldHVybiBTdGFjay5vZih0b3BpYykucGFyc2VBcm4odG9waWMudG9waWNBcm4pLnJlZ2lvbjtcbiAgfVxufVxuIl19