"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReceiptRuleSet = void 0;
const core_1 = require("@aws-cdk/core");
const receipt_rule_1 = require("./receipt-rule");
const ses_generated_1 = require("./ses.generated");
/**
 * A new or imported receipt rule set.
 */
class ReceiptRuleSetBase extends core_1.Resource {
    /**
     * Adds a new receipt rule in this rule set. The new rule is added after
     * the last added rule unless `after` is specified.
     */
    addRule(id, options) {
        this.lastAddedRule = new receipt_rule_1.ReceiptRule(this, id, {
            after: this.lastAddedRule ? this.lastAddedRule : undefined,
            ruleSet: this,
            ...options,
        });
        return this.lastAddedRule;
    }
    /**
     * Adds a drop spam rule
     */
    addDropSpamRule() {
        const dropSpam = new receipt_rule_1.DropSpamReceiptRule(this, 'DropSpam', {
            ruleSet: this,
        });
        this.lastAddedRule = dropSpam.rule;
    }
}
/**
 * A new receipt rule set.
 */
class ReceiptRuleSet extends ReceiptRuleSetBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.receiptRuleSetName,
        });
        const resource = new ses_generated_1.CfnReceiptRuleSet(this, 'Resource', {
            ruleSetName: this.physicalName,
        });
        this.receiptRuleSetName = resource.ref;
        if (props) {
            const rules = props.rules || [];
            rules.forEach((ruleOption, idx) => this.addRule(`Rule${idx}`, ruleOption));
            if (props.dropSpam) {
                this.addDropSpamRule();
            }
        }
    }
    /**
     * Import an exported receipt rule set.
     */
    static fromReceiptRuleSetName(scope, id, receiptRuleSetName) {
        class Import extends ReceiptRuleSetBase {
            constructor() {
                super(...arguments);
                this.receiptRuleSetName = receiptRuleSetName;
            }
        }
        return new Import(scope, id);
    }
}
exports.ReceiptRuleSet = ReceiptRuleSet;
//# sourceMappingURL=data:application/json;base64,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