"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DropSpamReceiptRule = exports.ReceiptRule = exports.TlsPolicy = void 0;
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const ses_generated_1 = require("./ses.generated");
/**
 * The type of TLS policy for a receipt rule.
 */
var TlsPolicy;
(function (TlsPolicy) {
    /**
     * Do not check for TLS.
     */
    TlsPolicy["OPTIONAL"] = "Optional";
    /**
     * Bounce emails that are not received over TLS.
     */
    TlsPolicy["REQUIRE"] = "Require";
})(TlsPolicy = exports.TlsPolicy || (exports.TlsPolicy = {}));
/**
 * A new receipt rule.
 */
class ReceiptRule extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.receiptRuleName,
        });
        this.actions = new Array();
        const resource = new ses_generated_1.CfnReceiptRule(this, 'Resource', {
            after: props.after ? props.after.receiptRuleName : undefined,
            rule: {
                actions: core_1.Lazy.anyValue({ produce: () => this.renderActions() }),
                enabled: props.enabled === undefined ? true : props.enabled,
                name: this.physicalName,
                recipients: props.recipients,
                scanEnabled: props.scanEnabled,
                tlsPolicy: props.tlsPolicy,
            },
            ruleSetName: props.ruleSet.receiptRuleSetName,
        });
        this.receiptRuleName = resource.ref;
        for (const action of props.actions || []) {
            this.addAction(action);
        }
    }
    static fromReceiptRuleName(scope, id, receiptRuleName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.receiptRuleName = receiptRuleName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds an action to this receipt rule.
     */
    addAction(action) {
        this.actions.push(action.bind(this));
    }
    renderActions() {
        if (this.actions.length === 0) {
            return undefined;
        }
        return this.actions;
    }
}
exports.ReceiptRule = ReceiptRule;
/**
 * A rule added at the top of the rule set to drop spam/virus.
 *
 * @see https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html
 */
class DropSpamReceiptRule extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const fn = new lambda.SingletonFunction(this, 'Function', {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromInline(`exports.handler = ${dropSpamCode}`),
            uuid: '224e77f9-a32e-4b4d-ac32-983477abba16',
        });
        fn.addPermission('AllowSes', {
            action: 'lambda:InvokeFunction',
            principal: new iam.ServicePrincipal('ses.amazonaws.com'),
            sourceAccount: core_1.Aws.ACCOUNT_ID,
        });
        this.rule = new ReceiptRule(this, 'Rule', {
            actions: [
                {
                    bind: () => ({
                        lambdaAction: {
                            functionArn: fn.functionArn,
                            invocationType: 'RequestResponse',
                        },
                    }),
                },
            ],
            scanEnabled: true,
            ruleSet: props.ruleSet,
        });
    }
}
exports.DropSpamReceiptRule = DropSpamReceiptRule;
// Adapted from https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html
/* eslint-disable no-console */
function dropSpamCode(event, _, callback) {
    console.log('Spam filter');
    const sesNotification = event.Records[0].ses;
    console.log('SES Notification:\n', JSON.stringify(sesNotification, null, 2));
    // Check if any spam check failed
    if (sesNotification.receipt.spfVerdict.status === 'FAIL'
        || sesNotification.receipt.dkimVerdict.status === 'FAIL'
        || sesNotification.receipt.spamVerdict.status === 'FAIL'
        || sesNotification.receipt.virusVerdict.status === 'FAIL') {
        console.log('Dropping spam');
        // Stop processing rule set, dropping message
        callback(null, { disposition: 'STOP_RULE_SET' });
    }
    else {
        callback(null, null);
    }
}
//# sourceMappingURL=data:application/json;base64,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