"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EventBus = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const archive_1 = require("./archive");
const events_generated_1 = require("./events.generated");
class EventBusBase extends core_1.Resource {
    /**
     * Create an EventBridge archive to send events to.
     *
     * When you create an archive, incoming events might not immediately start being sent to the archive.
     * Allow a short period of time for changes to take effect.
     *
     * @stability stable
     */
    archive(id, props) {
        return new archive_1.Archive(this, id, {
            sourceEventBus: this,
            description: props.description || `Event Archive for ${this.eventBusName} Event Bus`,
            eventPattern: props.eventPattern,
            retention: props.retention,
            archiveName: props.archiveName,
        });
    }
    /**
     * Grants an IAM Principal to send custom events to the eventBus so that they can be matched to rules.
     *
     * @stability stable
     */
    grantPutEventsTo(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['events:PutEvents'],
            resourceArns: [this.eventBusArn],
        });
    }
}
/**
 * Define an EventBridge EventBus.
 *
 * @stability stable
 * @resource AWS::Events::EventBus
 */
class EventBus extends EventBusBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        const { eventBusName, eventSourceName } = EventBus.eventBusProps(core_1.Lazy.string({ produce: () => core_1.Names.uniqueId(this) }), props);
        super(scope, id, { physicalName: eventBusName });
        const eventBus = new events_generated_1.CfnEventBus(this, 'Resource', {
            name: eventBusName,
            eventSourceName,
        });
        this.eventBusArn = this.getResourceArnAttribute(eventBus.attrArn, {
            service: 'events',
            resource: 'event-bus',
            resourceName: eventBus.name,
        });
        this.eventBusName = this.getResourceNameAttribute(eventBus.ref);
        this.eventBusPolicy = eventBus.attrPolicy;
        this.eventSourceName = eventBus.eventSourceName;
    }
    /**
     * Import an existing event bus resource.
     *
     * @param scope Parent construct.
     * @param id Construct ID.
     * @param eventBusArn ARN of imported event bus.
     * @stability stable
     */
    static fromEventBusArn(scope, id, eventBusArn) {
        const parts = core_1.Stack.of(scope).parseArn(eventBusArn);
        return new ImportedEventBus(scope, id, {
            eventBusArn: eventBusArn,
            eventBusName: parts.resourceName || '',
            eventBusPolicy: '',
        });
    }
    /**
     * Import an existing event bus resource.
     *
     * @param scope Parent construct.
     * @param id Construct ID.
     * @param eventBusName Name of imported event bus.
     * @stability stable
     */
    static fromEventBusName(scope, id, eventBusName) {
        const eventBusArn = core_1.Stack.of(scope).formatArn({
            resource: 'event-bus',
            service: 'events',
            resourceName: eventBusName,
        });
        return EventBus.fromEventBusAttributes(scope, id, {
            eventBusName: eventBusName,
            eventBusArn: eventBusArn,
            eventBusPolicy: '',
        });
    }
    /**
     * Import an existing event bus resource.
     *
     * @param scope Parent construct.
     * @param id Construct ID.
     * @param attrs Imported event bus properties.
     * @stability stable
     */
    static fromEventBusAttributes(scope, id, attrs) {
        return new ImportedEventBus(scope, id, attrs);
    }
    /**
     * (deprecated) Permits an IAM Principal to send custom events to EventBridge so that they can be matched to rules.
     *
     * @param grantee The principal (no-op if undefined).
     * @deprecated use grantAllPutEvents instead
     */
    static grantPutEvents(grantee) {
        // It's currently not possible to restrict PutEvents to specific resources.
        // See https://docs.aws.amazon.com/eventbridge/latest/userguide/permissions-reference-eventbridge.html
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['events:PutEvents'],
            resourceArns: ['*'],
        });
    }
    /**
     * Permits an IAM Principal to send custom events to EventBridge so that they can be matched to rules.
     *
     * @param grantee The principal (no-op if undefined).
     * @stability stable
     */
    static grantAllPutEvents(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['events:PutEvents'],
            resourceArns: ['*'],
        });
    }
    static eventBusProps(defaultEventBusName, props) {
        if (props) {
            const { eventBusName, eventSourceName } = props;
            const eventBusNameRegex = /^[\/\.\-_A-Za-z0-9]{1,256}$/;
            if (eventBusName !== undefined && eventSourceName !== undefined) {
                throw new Error('\'eventBusName\' and \'eventSourceName\' cannot both be provided');
            }
            else if (eventBusName !== undefined && !core_1.Token.isUnresolved(eventBusName)) {
                if (eventBusName === 'default') {
                    throw new Error('\'eventBusName\' must not be \'default\'');
                }
                else if (eventBusName.indexOf('/') > -1) {
                    throw new Error('\'eventBusName\' must not contain \'/\'');
                }
                else if (!eventBusNameRegex.test(eventBusName)) {
                    throw new Error(`'eventBusName' must satisfy: ${eventBusNameRegex}`);
                }
                return { eventBusName };
            }
            else if (eventSourceName !== undefined) {
                // Ex: aws.partner/PartnerName/acct1/repo1
                const eventSourceNameRegex = /^aws\.partner(\/[\.\-_A-Za-z0-9]+){2,}$/;
                if (!eventSourceNameRegex.test(eventSourceName)) {
                    throw new Error(`'eventSourceName' must satisfy: ${eventSourceNameRegex}`);
                }
                else if (!eventBusNameRegex.test(eventSourceName)) {
                    throw new Error(`'eventSourceName' must satisfy: ${eventBusNameRegex}`);
                }
                return { eventBusName: eventSourceName, eventSourceName };
            }
        }
        return { eventBusName: defaultEventBusName };
    }
}
exports.EventBus = EventBus;
_a = JSII_RTTI_SYMBOL_1;
EventBus[_a] = { fqn: "@aws-cdk/aws-events.EventBus", version: "1.124.0" };
class ImportedEventBus extends EventBusBase {
    constructor(scope, id, attrs) {
        const arnParts = core_1.Stack.of(scope).parseArn(attrs.eventBusArn);
        super(scope, id, {
            account: arnParts.account,
            region: arnParts.region,
        });
        this.eventBusArn = attrs.eventBusArn;
        this.eventBusName = attrs.eventBusName;
        this.eventBusPolicy = attrs.eventBusPolicy;
        this.eventSourceName = attrs.eventSourceName;
    }
}
//# sourceMappingURL=data:application/json;base64,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